# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pipeline_views']

package_data = \
{'': ['*']}

install_requires = \
['Django>=1.10', 'djangorestframework>=3.7.0']

setup_kwargs = {
    'name': 'drf-pipeline-views',
    'version': '0.1.3',
    'description': 'Django REST framework views using the pipeline pattern.',
    'long_description': '# Django REST Framework Pipeline Views\n\n[![Coverage Status](https://coveralls.io/repos/github/MrThearMan/drf-pipeline-views/badge.svg?branch=main)](https://coveralls.io/github/MrThearMan/drf-pipeline-views?branch=main)\n[![GitHub Workflow Status](https://img.shields.io/github/workflow/status/MrThearMan/drf-pipeline-views/Tests)](https://github.com/MrThearMan/drf-pipeline-views/actions/workflows/main.yml)\n[![PyPI](https://img.shields.io/pypi/v/drf-pipeline-views)](https://pypi.org/project/drf-pipeline-views)\n[![PyPI - Python Version](https://img.shields.io/pypi/pyversions/drf-pipeline-views)](https://pypi.org/project/drf-pipeline-views)\n[![PyPI - Django Version](https://img.shields.io/pypi/djversions/drf-pipeline-views)](https://pypi.org/project/drf-pipeline-views)\n[![GitHub](https://img.shields.io/github/license/MrThearMan/drf-pipeline-views)](https://github.com/MrThearMan/drf-pipeline-views/blob/main/LICENSE)\n[![GitHub last commit](https://img.shields.io/github/last-commit/MrThearMan/drf-pipeline-views)](https://github.com/MrThearMan/drf-pipeline-views/commits/main)\n\n```\npip install drf-pipeline-views\n```\n\nInspired by a talk on [The Clean Architecture in Python](https://archive.org/details/pyvideo_2840___The_Clean_Architecture_in_Python)\nby Brandon Rhodes, `drf-pipeline-views` aims to simplify writing testable API endpoints with\n[Django REST framework](https://www.django-rest-framework.org/) using the\n*[Pipeline Design Pattern](https://java-design-patterns.com/patterns/pipeline/)*.\n\nThe main idea behind the pipeline pattern is to process data in steps. Input from the previous step\nis passed to the next, resulting in a collection of data-in, data-out functions. These functions\ncan be easily unit tested, since none of the functions depend on the state of the objects in the other parts\nof the pipeline. Furthermore, IO can be separated into its own step, making the other parts of the\nlogic simpler and faster to test by not having to mock or do any other special setup around the IO.\nThis also means that the IO block, or in fact any other part of the application, can be replaced as long as the\ndata flowing through the pipeline remains the same.\n\n\n## Basic Usage:\n\nLet\'s create a basic pipeline:\n\n```python\ndef step1(step1_input1, step1_input2):\n    # Process the data...\n    return {"step2_input1": ..., "step2_input2": ...}\n\ndef step2(step2_input1, step2_input2):\n    # Maybe do some IO...\n    return {"step3_input1": ..., "step3_input2": ...}\n\ndef step3(step3_input1, step3_input2):\n    # Process the data, but do not pass on anything...\n    return\n\ndef step4():\n    # Build some response...\n    return {"end_result1": ..., "end_result2": ...}\n```\n\nNext, we\'ll create input and output serializers for our endpoint:\n\n```python\nfrom rest_framework import fields\nfrom rest_framework.serializers import Serializer\n\nclass InputSerializer(Serializer):\n    step1_input1 = fields.CharField()\n    step1_input2 = fields.DateField()\n\nclass OutputSerializer(Serializer):\n    end_result1 = fields.CharField()\n    end_result2 = fields.FloatField()\n```\n\nFinally, we can create our view:\n\n```python\nfrom pipeline_views import BaseAPIView, GetMixin\n\n\nclass SomeView(GetMixin, BaseAPIView):\n\n  pipelines = {\n    "GET": [\n        InputSerializer,\n        [\n            step1,\n            step2,\n            step3,\n            step4,\n        ],\n        OutputSerializer,\n    ],\n  }\n```\n\nUsing input and output serializers like this forces verification of the incoming and outcoming data,\nso that if something changes in the logic, or some unexpected values are returned,\nthe endpoint will break instead of creating side effects in the application using the API.\n\n---\n\nUsing serializers is totally optional. A pipeline like this will work just as well:\n\n```python\nclass SomeView(GetMixin, BaseAPIView):\n\n  pipelines = {\n    "GET": [\n        step1,\n        step2,\n        step3,\n        step4,\n    ],\n  }\n```\n\nBaseAPIView will try to infer a serializer with the correct serializer fields for\nbased on the type hints to the first function `step1`.\n\n```python\nfrom rest_framework.fields import CharField, IntegerField\nfrom pipeline_views import MockSerializer\n\n# Callable\ndef logic_callable(name: str, age: int):\n    ...\n\n# Inferred Serializer\nclass LogicCallableSerializer(MockSerializer):\n    name = CharField()\n    age = IntegerField()\n```\n\nThis is only used by the Django REST Framework Browsable API to create forms.\n`MockSerializer` makes sure the fields are only used for input and not validation.\n\n---\n\nPipeline logic can be grouped into blocks:\n\n```python\nclass SomeView(GetMixin, BaseAPIView):\n\n  pipelines = {\n    "GET": [\n        [\n            block1_step1,\n            block1_step2,\n        ],\n        [\n            block2_step1,\n            block2_step2,\n        ],\n    ],\n  }\n```\n\nLogic blocks are useful if you want to skip some logic methods under certain conditions, e.g., to return a cached result.\nThis can be accomplished by raising a `NextLogicBlock` exception. The exception can be initialized with\nany number of keyword arguments that will be passed to the next step in the logic, or to the response if it\'s\nthe last step in the logic.\n\n```python\nfrom pipeline_views import NextLogicBlock\n\n\ndef block1_step1(step1_input1, step1_input2):\n    if condition:\n        raise NextLogicBlock(step3_input1=..., step3_input2=...)\n    ...\n\ndef block1_step2(step2_input1, step2_input2):\n    ...\n\ndef block2_step1(step3_input1, step3_input2):\n    ...\n\ndef block2_step2():\n    ...\n```\n\n---\n\nIf you wish to add data to a request, you can do that on the endpoint level by overriding\n`_process_request`, or on the endpoint HTTP method level by overriding the spesific method, like `get`.\n\n```python\nfrom rest_framework.exceptions import NotAuthenticated\nfrom rest_framework.authentication import get_authorization_header\nfrom pipeline_views import BaseAPIView, GetMixin\n\n\nclass BasicView(GetMixin, BaseAPIView):\n\n    pipelines = {"GET": ...}\n\n    def get(self, request, *args, **kwargs):\n        # Add language to every get request for this endpoint\n        kwargs["lang"] = request.LANGUAGE_CODE\n        return super().get(request, *args, **kwargs)\n\n    def _process_request(self, data):\n        # Add authorization token to every http method\n        data["token"] = self._token_from_headers()\n        return super()._process_request(data)\n\n    def _token_from_headers(self):\n        auth_header = get_authorization_header(self.request)\n        if not auth_header:\n            raise NotAuthenticated("You must be logged in for this endpoint.")\n        return auth_header.split()[1].decode()\n```\n',
    'author': 'Matti Lamppu',
    'author_email': 'lamppu.matti.akseli@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/MrThearMan/drf-pipeline-views',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6.1,<4',
}


setup(**setup_kwargs)
