"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnGuardValidator = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const os = require("os");
const path = require("path");
const check_1 = require("./check");
const utils_1 = require("./utils");
/**
 * A validation plugin using CFN Guard
 */
class CfnGuardValidator {
    constructor(props = {}) {
        this.rulesPaths = [];
        this.executionConfig = [];
        this.name = 'cdk-validator-cfnguard';
        this.disabledRules = props.disabledRules ?? [];
        if (props.controlTowerRulesEnabled ?? true) {
            this.rulesPaths.push(path.join(__dirname, '..', 'rules', 'control-tower'));
        }
        this.rulesPaths.push(...props.rules ?? []);
        const osPlatform = os.platform();
        // guard calls it ubuntu but seems to apply to all linux
        // https://github.com/aws-cloudformation/cloudformation-guard/blob/184002cdfc0ae9e29c61995aae41b7d1f1d3b26c/install-guard.sh#L43-L46
        const platform = osPlatform === 'linux'
            ? 'ubuntu'
            : osPlatform === 'darwin' ? 'macos' : undefined;
        if (!platform) {
            throw new Error(`${os.platform()} not supported, must be either 'darwin' or 'linux'`);
        }
        this.guard = path.join(__dirname, '..', 'bin', platform, 'cfn-guard');
    }
    /**
     * This is (hopefully) a temporary solution to https://github.com/aws-cloudformation/cloudformation-guard/issues/180
     * Rather than try and parse the output and split out the JSON entries we'll just
     * invoke guard separately for each rule.
     */
    generateGuardExecutionConfig(filePath, templatePaths) {
        const stat = fs.statSync(filePath);
        if (stat.isDirectory()) {
            const dir = fs.readdirSync(filePath);
            dir.forEach(d => this.generateGuardExecutionConfig(path.join(filePath, d), templatePaths));
        }
        else {
            templatePaths.forEach(template => {
                if (!this.disabledRules.includes(path.parse(filePath).name)) {
                    this.executionConfig.push({ rulePath: filePath, templatePath: template });
                }
            });
        }
    }
    validate(context) {
        const templatePaths = context.templatePaths;
        this.rulesPaths.forEach(rule => this.generateGuardExecutionConfig(rule, templatePaths));
        const result = this.executionConfig.reduce((acc, config) => {
            const report = this.execGuard(config);
            return {
                violations: [...acc.violations, ...report.violations],
                success: acc.success === false ? false : report.success,
            };
        }, { violations: [], success: true });
        return {
            ...result,
        };
    }
    execGuard(config) {
        const flags = [
            'validate',
            '--rules',
            config.rulePath,
            '--data',
            config.templatePath,
            '--output-format',
            'json',
            '--show-summary',
            'none',
        ];
        const violations = [];
        let success;
        try {
            const result = (0, utils_1.exec)([this.guard, ...flags], {
                json: true,
            });
            const guardResult = JSON.parse(JSON.stringify(result), reviver);
            if (!guardResult.not_compliant || guardResult.not_compliant.length === 0) {
                return { success: true, violations: [] };
            }
            success = false;
            guardResult.not_compliant.forEach((check) => {
                const violationCheck = new check_1.ViolationCheck(check, config.templatePath, config.rulePath);
                const violation = violationCheck.processCheck();
                violations.push(...violation);
            });
        }
        catch (e) {
            success = false;
            throw new Error(`
        CfnGuardValidator plugin failed processing cfn-guard results.
        Please create an issue https://github.com/cdklabs/cdk-validator-cfnguard/issues/new
        Rule: ${path.basename(config.rulePath)}
        Error: ${e}`);
        }
        return {
            success,
            violations: violations,
        };
    }
}
_a = JSII_RTTI_SYMBOL_1;
CfnGuardValidator[_a] = { fqn: "@cdklabs/cdk-validator-cfnguard.CfnGuardValidator", version: "0.0.1" };
exports.CfnGuardValidator = CfnGuardValidator;
/**
 * Guard does not have a standard JSON schema and the schema
 * that is returned can be dependent on the type of rule or type
 * of check that was executed. The results are very much an attempt to
 * display the internals of guard to the user. Trying to make sense of that
 * can be difficult.
 *
 * The result structure can depend on the way that the rule was written. For example
 * I could write a rule like this:
 *
 *     rule MY_RULE {
 *       # This is a "check" and is a `Clause` type check
 *       Properties.SomeProp == true
 *     }
 *
 * Or I could write a rule like this:
 *
 *     rule MY_RULE {
 *       #  This is a "check" and is a `Rule` type check
 *       check(Properties)
 *     }
 *     rule check(properties) {
 *       properties.SomeProp == true
 *     }
 *
 * Both of the above examples are checking the same thing
 * but the schema that is returned is different because the
 * way the rule was written is different
 *
 * This reviver function is executed bottom up and is essentially
 * creating a new object with a well known schema that the rest of the
 * plugin can work with. It looks for certain fields that always appear in
 * the guard results, but appear in different locations. It finds those fields
 * and then pulls them up the object, dropping any fields that we don't
 * care about. For example guard may return
 *
 * {
 *   Clause: {
 *     Unary: {
 *       check: {
 *         UnResolved: {
 *           value: {
 *             traversed_to: {...} // we only care about this!!!
 *           }
 *         }
 *       }
 *     }
 *   }
 * }
 *
 * Or it may return
 *
 * {
 *   Rule: {
 *     checks: [{
 *       Block: {
 *         unresolved: {
 *           traversed_to: {...} // we only care about this!!!
 *         }
 *       }
 *     }]
 *   }
 * }
 *
 * In the above example we only care about the 'traversed_to' field,
 * so this reviver function will grab that field and pull it up the object, dropping
 * the fields we don't care about, ending with something like
 * {
 *   checks: [{
 *     resolved: false,
 *     traversed: {...}
 *   }]
 * }
 *
 */
function reviver(key, value) {
    if (key === 'not_compliant') {
        // not_compliant can sometimes be an empty object (but not an Array), so we
        // process this value before diving into other object values to ensure this
        // one is always made into an Array
        return Object.values(value).map((v) => v.Rule);
    }
    else if (value !== null && typeof value === 'object' && !Array.isArray(value)) {
        return extractNestedObject(value);
    }
    else if (key === 'checks' && Array.isArray(value)) {
        return extractNestedChecks(value.flatMap(v => v));
    }
    return value;
}
/**
 * Extract a nested 'checks' object. This also handles checks
 * nested within checks. It will grab the checks at the level below
 * and pull it up to the next level.
 */
function extractNestedChecks(checks) {
    const containsNestedChecks = checks.some(check => Object.values(check).some((value) => {
        return typeof value === 'object' && value.hasOwnProperty('checks');
    }));
    if (containsNestedChecks) {
        return checks.flatMap(check => {
            if (Object.keys(check).includes('traversed')) {
                return check;
            }
            return Object.values(check).flatMap((checkValue) => {
                return Object.values(checkValue.checks ?? checkValue).flatMap((nestedCheckValue) => {
                    return {
                        ...nestedCheckValue,
                        name: checkValue.name,
                        messages: checkValue.messages ?? nestedCheckValue.messages,
                    };
                });
            });
        });
    }
    return checks.flatMap(check => {
        if (Object.keys(check).includes('traversed')) {
            return check;
        }
        return Object.values(check);
    });
}
/**
 * Extract a nested object and pull it up a level
 */
function extractNestedObject(object) {
    let newObject = object;
    Object.entries(object).forEach(([level1NestedKey, level1NestedValue]) => {
        const nestedValue = level1NestedValue;
        switch (level1NestedKey.toLowerCase()) {
            // this should always be found earlier than the rest since it appears
            // within the 'unresolved' and 'resolved' objects. The object
            // is slightly different for each case so here we create
            // a new object with the key 'traversed' with a consistent value
            case 'traversed_to':
                newObject = {
                    traversed: {
                        to: {
                            path: nestedValue.path,
                            value: nestedValue.value,
                        },
                        from: nestedValue.from ? {
                            path: nestedValue.from.path,
                            value: undefined,
                        } : undefined,
                    },
                    messages: nestedValue.messages,
                };
                break;
            // This should be found in the "second" pass after the above
            // 'traversed_to' case has been executed. We take the new
            // object that was created in the `traversed_to` case and
            // a couple other fields, dropping the rest that we don't care about
            case 'unresolved':
                newObject = {
                    resolved: false,
                    traversed: nestedValue.traversed,
                    messages: nestedValue.messages ?? object.messages,
                };
                break;
            // This should be found in the "second" pass after the above
            // 'traversed_to' case has been executed. We take the new
            // object that was created in the `traversed_to` case and
            // a couple other fields, dropping the rest that we don't care about
            // A check can either be resolved or unresolved
            case 'resolved':
            case 'inresolved':
                newObject = {
                    resolved: true,
                    traversed: {
                        from: nestedValue.from,
                        to: {
                            path: nestedValue.from.path,
                            value: nestedValue.to.value,
                        },
                    },
                    messages: nestedValue.messages,
                };
                break;
        }
        // this check will be evaluated _after_ the 'traversed_to' check and _before_ the 'resolved'
        // and 'unresolved' checks above. There may be a case where 'traversed' is nested 2 (or 3 or 4) below
        // 'unresolved' or 'resolved' and this will keep pulling it up until it is just one below
        // and the above checks can work
        if (level1NestedValue !== null && typeof level1NestedValue === 'object' && !Array.isArray(level1NestedValue)) {
            Object.entries(level1NestedValue).forEach(([level2NestedKey, level2NestedValue]) => {
                switch (level2NestedKey.toLowerCase()) {
                    case 'traversed':
                        newObject = {
                            traversed: nestedValue.traversed,
                            resolved: nestedValue.resolved,
                            messages: nestedValue.messages ?? level2NestedValue.messages ?? object.messages,
                        };
                        break;
                }
            });
        }
    });
    return newObject;
}
//# sourceMappingURL=data:application/json;base64,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