# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pelc']

package_data = \
{'': ['*'], 'pelc': ['data/*']}

install_requires = \
['openpyxl>=3.0.10,<4.0.0', 'pandas>=1.5.1,<2.0.0']

setup_kwargs = {
    'name': 'pelc',
    'version': '0.4.1',
    'description': 'Python Eplet Load Calculator',
    'long_description': '[![DOI](https://zenodo.org/badge/555576588.svg)](https://zenodo.org/badge/latestdoi/555576588)\n[![Downloads](https://pepy.tech/badge/pelc)](https://pepy.tech/project/pelc)\n# PELC (Python Eplet Load Calculator)\n\n### Overview\nPELC is a Python package designed to calculate efficiently the HLA Eplet Load (based on the\n[EpRegistry database](https://www.epregistry.com.br/)) between donors and recipients by loading in a pandas.DataFrame\nin `eplet_comparison.compute_epletic_load` the recipients\' and donors\' typings.  See minimal reproducible example for\nmore details.\n\n\n### Getting started\n#### Install from PyPI (recommended)\nTo use `pelc`, run `pip install pelc` in your terminal.\n\n\n#### Usage\n\n##### a. Comparing two alleles\nHere is a minimal example of how to use `pelc` to compare two alleles:\n```py\nfrom pelc.simple_comparison import simple_comparison\n\nsimple_comparison(\n    "A*68:01",\n    "A*68:02",\n    "output",  # file will be saved as output.csv in the current directory\n    verifiedonly=False,  # if True, only verified eplets will be considered, otherwise all eplets will be considered\n    interlocus2=True  # doesn\'t matter for class I alleles\n)\n```\nIn the `output.csv` file created in the current directory, you will find two rows: "In A\\*68:02 but not in A\\*68:01" and\n"In A\\*68:01 but not in A\\*68:02"\n\n##### b. Batch mode\nHere is a minimal example with the file [Template.xlsx](https://github.com/MICS-Lab/pelc/raw/main/Template.xlsx)\n(click to download):\n```py\nimport pandas as pd\n\nfrom pelc import batch_eplet_comp, batch_eplet_comp_aux, output_type\n\nif __name__ == "__main__":\n    input_path: str = "Template.xlsx"\n\n    output_path: str = "MyOutput"\n    input_df: pd.DataFrame = pd.read_excel(\n        input_path,\n        sheet_name="My Sheet",\n        index_col="Index"\n    )\n\n    donordf: pd.DataFrame\n    recipientdf: pd.DataFrame\n    donordf, recipientdf = batch_eplet_comp_aux.split_dataframe(input_df)\n\n    batch_eplet_comp.compute_epletic_load(\n        donordf,\n        recipientdf,\n        output_path,\n        output_type.OutputType.DETAILS_AND_COUNT,\n        class_i=True,  # Compute class I eplets comparison?\n        class_ii=True,  # Compute class II eplets comparison?\n        verifiedonly=False,  # How should the epletic charge be computed? Verified eplets only? Or all eplets?\n        exclude=None,  # list of indices to exclude\n        interlocus2=True  # whether or not to take into account interlocus eplets for HLA of class II\n    )\n```\n\n#### Advanced usage:\n##### a. Not taking into account all loci (if they are not typed for example)\nIf one wants to determine the eplet mismatches between a donor and a recipient but without taking into account\na certain locus, one can use `A*`, `B*`, ..., `DPB1*` as the allele name for both recipients and donors on this locus\nand the eplet mismatch computation will only take into account the loci filled in.\n\n##### b. Not creating a file but generating a pandas.DataFrame\nIf one wants to generate a `pandas.DataFrame` directly, the `output_path` argument of `simple_comparison` can be \nset to `None`. The `pandas.DataFrame` will be returned by the function. Same goes for `compute_epletic_load`.\n\n\n#### Exit codes:\n```\n- 55: an eplet did not match the regular expression \'^\\d+\' (ABC, DR, DQ or DP) and it also did not match the regular\nexpression \'^.[pqr]*(\\d+)\' (interlocus2) either.\n```\n\n\n#### Unit tests\nTested on `Python 3.10.2` & `Python 3.11.0`.\n```\nplatform win32 -- Python 3.10.2, pytest-7.2.0, pluggy-1.0.0\nplugins: mypy-0.10.0\ncollected 19 items                                                                                                                                     \n\nunit_tests_mypy.py ..                                                               [ 10%]\nunit_tests_simple.py .                                                              [ 15%] \npelc\\__init__.py .                                                                  [ 21%] \npelc\\_unexpected_alleles.py .                                                       [ 26%] \npelc\\epitope_comparison.py .                                                        [ 31%] \npelc\\epitope_comparison_aux.py .                                                    [ 36%] \npelc\\output_type.py .                                                               [ 42%] \ntests\\__init__.py .                                                                 [ 47%] \ntests\\base_loading_for_tests.py .                                                   [ 52%] \ntests\\test_epitope_mismatches.py ......                                             [ 78%]\ntests\\test_pelc.py ..                                                               [ 89%] \ntests\\test_unexpected_alleles.py ..                                                 [100%]\n```\n```\nplatform win32 -- Python 3.11.0, pytest-7.2.0, pluggy-1.0.0\nplugins: mypy-0.10.0\ncollected 19 items\n\nunit_tests_mypy.py ..                                                               [ 10%]\nunit_tests_simple.py .                                                              [ 15%]\npelc\\__init__.py .                                                                  [ 21%]\npelc\\_unexpected_alleles.py .                                                       [ 26%]\npelc\\epitope_comparison.py .                                                        [ 31%]\npelc\\epitope_comparison_aux.py .                                                    [ 36%]\npelc\\output_type.py .                                                               [ 42%]\ntests\\__init__.py .                                                                 [ 47%]\ntests\\base_loading_for_tests.py .                                                   [ 52%]\ntests\\test_epitope_mismatches.py ......                                             [ 78%]\ntests\\test_pelc.py ..                                                               [ 89%]\ntests\\test_unexpected_alleles.py ..                                                 [100%]\n```\n\n\n\n### About the source code\n- Follows [PEP8](https://peps.python.org/pep-0008/) Style Guidelines.\n- All functions are unit-tested with [pytest](https://docs.pytest.org/en/stable/).\n- All variables are correctly type-hinted, reviewed with [static type checker](https://mypy.readthedocs.io/en/stable/)\n`mypy`.\n- All functions are documented with [docstrings](https://www.python.org/dev/peps/pep-0257/).\n\n\n\n### Useful links:\n- [Corresponding GitHub repository](https://github.com/MICS-Lab/pelc)\n- [Corresponding PyPI page](https://pypi.org/project/pelc)\n\n\n\n### Citation\nIf you use this software, please cite it as below.\n\n- APA:\n```\nIf you use this software, please cite it as below. Lhotte, R., Clichet, V., Usureau, C. & Taupin, J. (2022). \nPython Eplet Load Calculator (PELC) package (Version 0.4.1) [Computer software]. https://doi.org/10.5281/zenodo.7254809\n```\n\n- BibTeX:\n```\n@software{lhotte_romain_2022_7526198,\n  author       = {Lhotte, Romain and\n                  Clichet, Valentin and\n                  Usureau, Cédric and\n                  Taupin, Jean-Luc},\n  title        = {Python Eplet Load Calculator},\n  month        = oct,\n  year         = 2022,\n  publisher    = {Zenodo},\n  version      = {0.4.1},\n  doi          = {10.5281/zenodo.7526198},\n}\n```\n',
    'author': 'JasonMendoza2008',
    'author_email': 'lhotteromain@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
