"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Chime = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [chime](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonchime.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Chime extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [chime](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonchime.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'chime';
        this.actionList = {
            "AcceptDelegate": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to accept the delegate invitation to share management of an Amazon Chime account with another AWS Account",
                "accessLevel": "Write"
            },
            "ActivateUsers": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-access.html",
                "description": "Grants permission to activate users in an Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "AddDomain": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html",
                "description": "Grants permission to add a domain to your Amazon Chime account",
                "accessLevel": "Write"
            },
            "AddOrUpdateGroups": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html",
                "description": "Grants permission to add new or update existing Active Directory or Okta user groups associated with your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "AssociatePhoneNumberWithUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumberWithUser.html",
                "description": "Grants permission to associate a phone number with an Amazon Chime user",
                "accessLevel": "Write"
            },
            "AssociatePhoneNumbersWithVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumbersWithVoiceConnector.html",
                "description": "Grants permission to associate multiple phone numbers with an Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "AssociatePhoneNumbersWithVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumbersWithVoiceConnectorGroup.html",
                "description": "Grants permission to associate multiple phone numbers with an Amazon Chime Voice Connector Group",
                "accessLevel": "Write"
            },
            "AssociateSigninDelegateGroupsWithAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociateSigninDelegateGroupsWithAccount.html",
                "description": "Grants permission to associate the specified sign-in delegate groups with the specified Amazon Chime account.",
                "accessLevel": "Write"
            },
            "AuthorizeDirectory": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to authorize an Active Directory for your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "BatchCreateAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchCreateAttendee.html",
                "description": "Grants permission to create new attendees for an active Amazon Chime SDK meeting",
                "accessLevel": "Write",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "BatchCreateRoomMembership": {
                "url": "",
                "description": "Grants permission to batch add room members",
                "accessLevel": "Write"
            },
            "BatchDeletePhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchDeletePhoneNumber.html",
                "description": "Grants permission to move up to 50 phone numbers to the deletion queue",
                "accessLevel": "Write"
            },
            "BatchSuspendUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchSuspendUser.html",
                "description": "Grants permission to suspend up to 50 users from a Team or EnterpriseLWA Amazon Chime account",
                "accessLevel": "Write"
            },
            "BatchUnsuspendUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUnsuspendUser.html",
                "description": "Grants permission to remove the suspension from up to 50 previously suspended users for the specified Amazon Chime EnterpriseLWA account",
                "accessLevel": "Write"
            },
            "BatchUpdatePhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUpdatePhoneNumber.html",
                "description": "Grants permission to update phone number details within the UpdatePhoneNumberRequestItem object for up to 50 phone numbers",
                "accessLevel": "Write"
            },
            "BatchUpdateUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUpdateUser.html",
                "description": "Grants permission to update user details within the UpdateUserRequestItem object for up to 20 users for the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "ConnectDirectory": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/active_directory.html",
                "description": "Grants permission to connect an Active Directory to your Amazon Chime Enterprise account",
                "accessLevel": "Write",
                "dependentActions": [
                    "ds:ConnectDirectory"
                ]
            },
            "CreateAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateAccount.html",
                "description": "Grants permission to create an Amazon Chime account under the administrator's AWS account",
                "accessLevel": "Write"
            },
            "CreateApiKey": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to create a new SCIM access key for your Amazon Chime account and Okta configuration",
                "accessLevel": "Write"
            },
            "CreateAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateAttendee.html",
                "description": "Grants permission to create a new attendee for an active Amazon Chime SDK meeting",
                "accessLevel": "Write",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "CreateBot": {
                "url": "",
                "description": "Grants permission to create a bot for an Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "CreateBotMembership": {
                "url": "",
                "description": "Grants permission to add a bot to a chat room in your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "CreateCDRBucket": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-access.html",
                "description": "Grants permission to create a new Call Detail Record S3 bucket",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:CreateBucket",
                    "s3:ListAllMyBuckets"
                ]
            },
            "CreateMeeting": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateMeeting.html",
                "description": "Grants permission to create a new Amazon Chime SDK meeting in the specified media Region, with no initial attendees",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePhoneNumberOrder": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreatePhoneNumberOrder.html",
                "description": "Grants permission to create a phone number order with the Carriers",
                "accessLevel": "Write"
            },
            "CreateProxySession": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateProxySession.html",
                "description": "Grants permission to create a proxy session for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "CreateRoom": {
                "url": "",
                "description": "Grants permission to create a room",
                "accessLevel": "Write"
            },
            "CreateRoomMembership": {
                "url": "",
                "description": "Grants permission to add a room member",
                "accessLevel": "Write"
            },
            "CreateUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateUser.html",
                "description": "Grants permission to create a user under the specified Amazon Chime account.",
                "accessLevel": "Write"
            },
            "CreateVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateVoiceConnector.html",
                "description": "Grants permission to create a Amazon Chime Voice Connector under the administrator's AWS account",
                "accessLevel": "Write"
            },
            "CreateVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateVoiceConnectorGroup.html",
                "description": "Grants permission to create a Amazon Chime Voice Connector Group under the administrator's AWS account",
                "accessLevel": "Write"
            },
            "DeleteAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteAccount.html",
                "description": "Grants permission to delete the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "DeleteAccountOpenIdConfig": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to delete the OpenIdConfig attributes from your Amazon Chime account",
                "accessLevel": "Write"
            },
            "DeleteApiKey": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to delete the specified SCIM access key associated with your Amazon Chime account and Okta configuration",
                "accessLevel": "Write"
            },
            "DeleteAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteAttendee.html",
                "description": "Grants permission to delete the specified attendee from an Amazon Chime SDK meeting",
                "accessLevel": "Write",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "DeleteCDRBucket": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to delete a Call Detail Record S3 bucket from your Amazon Chime account",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:DeleteBucket"
                ]
            },
            "DeleteDelegate": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to delete delegated AWS account management from your Amazon Chime account",
                "accessLevel": "Write"
            },
            "DeleteDomain": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html",
                "description": "Grants permission to delete a domain from your Amazon Chime account",
                "accessLevel": "Write"
            },
            "DeleteEventsConfiguration": {
                "url": "",
                "description": "Grants permission to delete an events configuration for a bot to receive outgoing events",
                "accessLevel": "Write"
            },
            "DeleteGroups": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to delete Active Directory or Okta user groups from your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "DeleteMeeting": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteMeeting.html",
                "description": "Grants permission to delete the specified Amazon Chime SDK meeting",
                "accessLevel": "Write",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "DeletePhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeletePhoneNumber.html",
                "description": "Grants permission to move a phone number to the deletion queue",
                "accessLevel": "Write"
            },
            "DeleteProxySession": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteProxySession.html",
                "description": "Grants permission to delete a proxy session for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteRoom": {
                "url": "",
                "description": "Grants permission to delete a room",
                "accessLevel": "Write"
            },
            "DeleteRoomMembership": {
                "url": "",
                "description": "Grants permission to remove a room member",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnector.html",
                "description": "Grants permission to delete the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorEmergencyCallingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorEmergencyCallingConfiguration.html",
                "description": "Grants permission to delete emergency calling configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorGroup.html",
                "description": "Grants permission to delete the specified Amazon Chime Voice Connector Group",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorOrigination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorOrigination.html",
                "description": "Grants permission to delete the origination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorProxy": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorProxy.html",
                "description": "Grants permission to delete proxy configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorStreamingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorStreamingConfiguration.html",
                "description": "Grants permission to delete streaming configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorTermination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorTermination.html",
                "description": "Grants permission to delete the termination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorTerminationCredentials": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorTerminationCredentials.html",
                "description": "Grants permission to delete SIP termination credentials for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DisassociatePhoneNumberFromUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumberFromUser.html",
                "description": "Grants permission to disassociate the primary provisioned number from the specified Amazon Chime user",
                "accessLevel": "Write"
            },
            "DisassociatePhoneNumbersFromVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumbersFromVoiceConnector.html",
                "description": "Grants permission to disassociate multiple phone numbers from the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DisassociatePhoneNumbersFromVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumbersFromVoiceConnectorGroup.html",
                "description": "Grants permission to disassociate multiple phone numbers from the specified Amazon Chime Voice Connector Group",
                "accessLevel": "Write"
            },
            "DisassociateSigninDelegateGroupsFromAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociateSigninDelegateGroupsFromAccount.html",
                "description": "Grants permission to disassociate the specified sign-in delegate groups from the specified Amazon Chime account.",
                "accessLevel": "Write"
            },
            "DisconnectDirectory": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to disconnect the Active Directory from your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "GetAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAccount.html",
                "description": "Grants permission to get details for the specified Amazon Chime account",
                "accessLevel": "Read"
            },
            "GetAccountResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to get details for the account resource associated with your Amazon Chime account",
                "accessLevel": "Read"
            },
            "GetAccountSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAccountSettings.html",
                "description": "Grants permission to get account settings for the specified Amazon Chime account ID",
                "accessLevel": "Read"
            },
            "GetAccountWithOpenIdConfig": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to get the account details and OpenIdConfig attributes for your Amazon Chime account",
                "accessLevel": "Read"
            },
            "GetAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAttendee.html",
                "description": "Grants permission to get attendee details for a specified meeting ID and attendee ID",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "GetBot": {
                "url": "",
                "description": "Grants permission to retrieve details for the specified bot",
                "accessLevel": "Read"
            },
            "GetCDRBucket": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to get details of a Call Detail Record S3 bucket associated with your Amazon Chime account",
                "accessLevel": "Read",
                "dependentActions": [
                    "s3:GetBucketAcl",
                    "s3:GetBucketLocation",
                    "s3:GetBucketLogging",
                    "s3:GetBucketVersioning",
                    "s3:GetBucketWebsite"
                ]
            },
            "GetDomain": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html",
                "description": "Grants permission to get domain details for a domain associated with your Amazon Chime account",
                "accessLevel": "Read"
            },
            "GetEventsConfiguration": {
                "url": "",
                "description": "Grants permission to retrieve details for an events configuration for a bot to receive outgoing events",
                "accessLevel": "Read"
            },
            "GetGlobalSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetGlobalSettings.html",
                "description": "Grants permission to get global settings related to Amazon Chime for the AWS account",
                "accessLevel": "Read"
            },
            "GetMeeting": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetMeeting.html",
                "description": "Grants permission to get the meeting record for a specified meeting ID",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "GetMeetingDetail": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to get attendee, connection, and other details for a meeting",
                "accessLevel": "Read"
            },
            "GetPhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumber.html",
                "description": "Grants permission to get details for the specified phone number",
                "accessLevel": "Read"
            },
            "GetPhoneNumberOrder": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumberOrder.html",
                "description": "Grants permission to get details for the specified phone number order",
                "accessLevel": "Read"
            },
            "GetPhoneNumberSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumberSettings.html",
                "description": "Grants permission to get phone number settings related to Amazon Chime for the AWS account",
                "accessLevel": "Read"
            },
            "GetProxySession": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetProxySession.html",
                "description": "Grants permission to get details of the specified proxy session for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetRetentionSettings": {
                "url": "",
                "description": "Gets the retention settings for the specified Amazon Chime account.",
                "accessLevel": "Read"
            },
            "GetRoom": {
                "url": "",
                "description": "Grants permission to retrieve a room",
                "accessLevel": "Read"
            },
            "GetTelephonyLimits": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/phone-numbers.html",
                "description": "Grants permission to get telephony limits for the AWS account",
                "accessLevel": "Read"
            },
            "GetUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetUser.html",
                "description": "Grants permission to get details for the specified user ID",
                "accessLevel": "Read"
            },
            "GetUserActivityReportData": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/user-details.html",
                "description": "Grants permission to get a summary of user activity on the user details page",
                "accessLevel": "Read"
            },
            "GetUserByEmail": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/user-details.html",
                "description": "Grants permission to get user details for an Amazon Chime user based on the email address in an Amazon Chime Enterprise or Team account",
                "accessLevel": "Read"
            },
            "GetUserSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetUserSettings.html",
                "description": "Grants permission to get user settings related to the specified Amazon Chime user",
                "accessLevel": "Read"
            },
            "GetVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnector.html",
                "description": "Grants permission to get details for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorEmergencyCallingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorEmergencyCallingConfiguration.html",
                "description": "Grants permission to get details of the emergency calling configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorGroup.html",
                "description": "Grants permission to get details for the specified Amazon Chime Voice Connector Group",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorLoggingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorLoggingConfiguration.html",
                "description": "Grants permission to get details of the logging configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorOrigination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorOrigination.html",
                "description": "Grants permission to get details of the origination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorProxy": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorProxy.html",
                "description": "Grants permission to get details of the proxy configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorStreamingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorStreamingConfiguration.html",
                "description": "Grants permission to get details of the streaming configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorTermination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorTermination.html",
                "description": "Grants permission to get details of the termination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorTerminationHealth": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorTerminationHealth.html",
                "description": "Grants permission to get details of the termination health for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "InviteDelegate": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to send an invitation to accept a request for AWS account delegation for an Amazon Chime account",
                "accessLevel": "Write"
            },
            "InviteUsers": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_InviteUsers.html",
                "description": "Grants permission to invite as many as 50 users to the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "InviteUsersFromProvider": {
                "url": "",
                "description": "Grants permission to invite users from a third party provider to your Amazon Chime account",
                "accessLevel": "Write"
            },
            "ListAccountUsageReportData": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/view-reports.html",
                "description": "Grants permission to list Amazon Chime account usage reporting data",
                "accessLevel": "List"
            },
            "ListAccounts": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAccounts.html",
                "description": "Grants permission to list the Amazon Chime accounts under the administrator's AWS account",
                "accessLevel": "List"
            },
            "ListApiKeys": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to list the SCIM access keys defined for your Amazon Chime account and Okta configuration",
                "accessLevel": "List"
            },
            "ListAttendeeTags": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAttendeeTags.html",
                "description": "Grants permission to list the tags applied to an Amazon Chime SDK attendee resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "ListAttendees": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAttendees.html",
                "description": "Grants permission to list up to 100 attendees for a specified Amazon Chime SDK meeting",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "ListBots": {
                "url": "",
                "description": "Grants permission to list the bots associated with the administrator's Amazon Chime Enterprise account",
                "accessLevel": "List"
            },
            "ListCDRBucket": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to list Call Detail Record S3 buckets",
                "accessLevel": "List",
                "dependentActions": [
                    "s3:ListAllMyBuckets",
                    "s3:ListBucket"
                ]
            },
            "ListCallingRegions": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/phone-numbers.html",
                "description": "Grants permission to list the calling regions available for the administrator's AWS account",
                "accessLevel": "List"
            },
            "ListDelegates": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to list account delegate information associated with your Amazon Chime account",
                "accessLevel": "List"
            },
            "ListDirectories": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to list active Active Directories hosted in the Directory Service of your AWS account",
                "accessLevel": "List"
            },
            "ListDomains": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html",
                "description": "Grants permission to list domains associated with your Amazon Chime account",
                "accessLevel": "List"
            },
            "ListGroups": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to list Active Directory or Okta user groups associated with your Amazon Chime Enterprise account",
                "accessLevel": "List"
            },
            "ListMeetingEvents": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/view-reports.html",
                "description": "Grants permission to list all events that occurred for a specified meeting",
                "accessLevel": "List"
            },
            "ListMeetingTags": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListMeetingTags.html",
                "description": "Grants permission to list the tags applied to an Amazon Chime SDK meeting resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "ListMeetings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListMeetings.html",
                "description": "Grants permission to list up to 100 active Amazon Chime SDK meetings",
                "accessLevel": "Read"
            },
            "ListMeetingsReportData": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/view-reports.html",
                "description": "Grants permission to list meetings ended during the specified date range",
                "accessLevel": "List"
            },
            "ListPhoneNumberOrders": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListPhoneNumberOrders.html",
                "description": "Grants permission to list the phone number orders under the administrator's AWS account",
                "accessLevel": "List"
            },
            "ListPhoneNumbers": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListPhoneNumbers.html",
                "description": "Grants permission to list the phone numbers under the administrator's AWS account",
                "accessLevel": "List"
            },
            "ListProxySessions": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListProxySessions.html",
                "description": "Grants permission to list proxy sessions for the specified Amazon Chime Voice Connector",
                "accessLevel": "List"
            },
            "ListRoomMemberships": {
                "url": "",
                "description": "Grants permission to list all room members",
                "accessLevel": "Read"
            },
            "ListRooms": {
                "url": "",
                "description": "Grants permission to list rooms",
                "accessLevel": "Read"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to list the tags applied to an Amazon Chime SDK meeting resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": false
                    }
                }
            },
            "ListUsers": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListUsers.html",
                "description": "Grants permission to list the users that belong to the specified Amazon Chime account",
                "accessLevel": "List"
            },
            "ListVoiceConnectorGroups": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectorGroups.html",
                "description": "Grants permission to list the Amazon Chime Voice Connector Groups under the administrator's AWS account",
                "accessLevel": "List"
            },
            "ListVoiceConnectorTerminationCredentials": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectorTerminationCredentials.html",
                "description": "Grants permission to list the SIP termination credentials for the specified Amazon Chime Voice Connector",
                "accessLevel": "List"
            },
            "ListVoiceConnectors": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectors.html",
                "description": "Grants permission to list the Amazon Chime Voice Connectors under the administrator's AWS account",
                "accessLevel": "List"
            },
            "LogoutUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_LogoutUser.html",
                "description": "Grants permission to log out the specified user from all of the devices they are currently logged into",
                "accessLevel": "Write"
            },
            "PutEventsConfiguration": {
                "url": "",
                "description": "Grants permission to update details for an events configuration for a bot to receive outgoing events",
                "accessLevel": "Write"
            },
            "PutRetentionSettings": {
                "url": "",
                "description": "Puts retention settings for the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorEmergencyCallingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorEmergencyCallingConfiguration.html",
                "description": "Grants permission to add emergency calling configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorLoggingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorLoggingConfiguration.html",
                "description": "Grants permission to add logging configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write",
                "dependentActions": [
                    "logs:CreateLogDelivery",
                    "logs:CreateLogGroup",
                    "logs:DeleteLogDelivery",
                    "logs:DescribeLogGroups",
                    "logs:GetLogDelivery",
                    "logs:ListLogDeliveries"
                ]
            },
            "PutVoiceConnectorOrigination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorOrigination.html",
                "description": "Grants permission to update the origination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorProxy": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorProxy.html",
                "description": "Grants permission to add proxy configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorStreamingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorStreamingConfiguration.html",
                "description": "Grants permission to add streaming configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorTermination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorTermination.html",
                "description": "Grants permission to update the termination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorTerminationCredentials": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorTerminationCredentials.html",
                "description": "Grants permission to add SIP termination credentials for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "RedactConversationMessage": {
                "url": "",
                "description": "Redacts the specified Chime conversation Message",
                "accessLevel": "Write"
            },
            "RedactRoomMessage": {
                "url": "",
                "description": "Redacts the specified Chime room Message",
                "accessLevel": "Write"
            },
            "RegenerateSecurityToken": {
                "url": "",
                "description": "Grants permission to regenerate the security token for the specified bot",
                "accessLevel": "Write"
            },
            "RenameAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/rename-account.html",
                "description": "Grants permission to modify the account name for your Amazon Chime Enterprise or Team account",
                "accessLevel": "Write"
            },
            "RenewDelegate": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to renew the delegation request associated with an Amazon Chime account",
                "accessLevel": "Write"
            },
            "ResetAccountResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to reset the account resource in your Amazon Chime account",
                "accessLevel": "Write"
            },
            "ResetPersonalPIN": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ResetPersonalPIN.html",
                "description": "Grants permission to reset the personal meeting PIN for the specified user on an Amazon Chime account",
                "accessLevel": "Write"
            },
            "RestorePhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_RestorePhoneNumber.html",
                "description": "Grants permission to restore the specified phone number from the deltion queue back to the phone number inventory",
                "accessLevel": "Write"
            },
            "RetrieveDataExports": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/request-attachments.html",
                "description": "Grants permission to download the file containing links to all user attachments returned as part of the \"Request attachments\" action",
                "accessLevel": "List"
            },
            "SearchAvailablePhoneNumbers": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_SearchAvailablePhoneNumbers.html",
                "description": "Grants permission to search phone numbers that can be ordered from the carrier",
                "accessLevel": "Read"
            },
            "StartDataExport": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/request-attachments.html",
                "description": "Grants permission to submit the \"Request attachments\" request",
                "accessLevel": "Write"
            },
            "SubmitSupportRequest": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/chime-getting-admin-support.html",
                "description": "Grants permission to submit a customer service support request",
                "accessLevel": "Write"
            },
            "SuspendUsers": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-access.html",
                "description": "Grants permission to suspend users from an Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "TagAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_TagAttendee.html",
                "description": "Grants permission to apply the specified tags to the specified Amazon Chime SDK attendee",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "TagMeeting": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_TagMeeting.html",
                "description": "Grants permission to apply the specified tags to the specified Amazon Chime SDK meeting.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_TagResource.html",
                "description": "Grants permission to apply the specified tags to the specified Amazon Chime SDK meeting resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UnauthorizeDirectory": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to unauthorize an Active Directory from your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "UntagAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagAttendee.html",
                "description": "Grants permission to untag the specified tags from the specified Amazon Chime SDK attendee.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "UntagMeeting": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagMeeting.html",
                "description": "Grants permission to untag the specified tags from the specified Amazon Chime SDK meeting.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagResource.html",
                "description": "Grants permission to untag the specified tags from the specified Amazon Chime SDK meeting resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": false
                    }
                }
            },
            "UpdateAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateAccount.html",
                "description": "Grants permission to update account details for the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "UpdateAccountOpenIdConfig": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to update the OpenIdConfig attributes for your Amazon Chime account",
                "accessLevel": "Write"
            },
            "UpdateAccountResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to update the account resource in your Amazon Chime account",
                "accessLevel": "Write"
            },
            "UpdateAccountSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateAccountSettings.html",
                "description": "Grants permission to update the settings for the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "UpdateBot": {
                "url": "",
                "description": "Grants permission to update the status of the specified bot",
                "accessLevel": "Write"
            },
            "UpdateCDRSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to update your Call Detail Record S3 bucket",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:CreateBucket",
                    "s3:DeleteBucket",
                    "s3:ListAllMyBuckets"
                ]
            },
            "UpdateGlobalSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateGlobalSettings.html",
                "description": "Grants permission to update the global settings related to Amazon Chime for the AWS account",
                "accessLevel": "Write"
            },
            "UpdatePhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdatePhoneNumber.html",
                "description": "Grants permission to update phone number details for the specified phone number",
                "accessLevel": "Write"
            },
            "UpdatePhoneNumberSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdatePhoneNumberSettings.html",
                "description": "Grants permission to update phone number settings related to Amazon Chime for the AWS account",
                "accessLevel": "Write"
            },
            "UpdateProxySession": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateProxySession.html",
                "description": "Grants permission to update a proxy session for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "UpdateRoom": {
                "url": "",
                "description": "Grants permission to update a room",
                "accessLevel": "Write"
            },
            "UpdateRoomMembership": {
                "url": "",
                "description": "Grants permission to update room membership role",
                "accessLevel": "Write"
            },
            "UpdateSupportedLicenses": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-access.html",
                "description": "Grants permission to update the supported license tiers available for users in your Amazon Chime account",
                "accessLevel": "Write"
            },
            "UpdateUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateUser.html",
                "description": "Grants permission to update user details for a specified user ID",
                "accessLevel": "Write"
            },
            "UpdateUserLicenses": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-access.html",
                "description": "Grants permission to update the licenses for your Amazon Chime users",
                "accessLevel": "Write"
            },
            "UpdateUserSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateUserSettings.html",
                "description": "Grants permission to update user settings related to the specified Amazon Chime user",
                "accessLevel": "Write"
            },
            "UpdateVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateVoiceConnector.html",
                "description": "Grants permission to update Amazon Chime Voice Connector details for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "UpdateVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateVoiceConnectorGroup.html",
                "description": "Grants permission to update Amazon Chime Voice Connector Group details for the specified Amazon Chime Voice Connector Group",
                "accessLevel": "Write"
            },
            "ValidateAccountResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to validate the account resource in your Amazon Chime account",
                "accessLevel": "Read"
            }
        };
        this.resourceTypes = {
            "meeting": {
                "name": "meeting",
                "url": "",
                "arn": "arn:${Partition}:chime::${AccountId}:meeting/${MeetingId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to accept the delegate invitation to share management of an Amazon Chime account with another AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    acceptDelegate() {
        this.add('chime:AcceptDelegate');
        return this;
    }
    /**
     * Grants permission to activate users in an Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    activateUsers() {
        this.add('chime:ActivateUsers');
        return this;
    }
    /**
     * Grants permission to add a domain to your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html
     */
    addDomain() {
        this.add('chime:AddDomain');
        return this;
    }
    /**
     * Grants permission to add new or update existing Active Directory or Okta user groups associated with your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html
     */
    addOrUpdateGroups() {
        this.add('chime:AddOrUpdateGroups');
        return this;
    }
    /**
     * Grants permission to associate a phone number with an Amazon Chime user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumberWithUser.html
     */
    associatePhoneNumberWithUser() {
        this.add('chime:AssociatePhoneNumberWithUser');
        return this;
    }
    /**
     * Grants permission to associate multiple phone numbers with an Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumbersWithVoiceConnector.html
     */
    associatePhoneNumbersWithVoiceConnector() {
        this.add('chime:AssociatePhoneNumbersWithVoiceConnector');
        return this;
    }
    /**
     * Grants permission to associate multiple phone numbers with an Amazon Chime Voice Connector Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumbersWithVoiceConnectorGroup.html
     */
    associatePhoneNumbersWithVoiceConnectorGroup() {
        this.add('chime:AssociatePhoneNumbersWithVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to associate the specified sign-in delegate groups with the specified Amazon Chime account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociateSigninDelegateGroupsWithAccount.html
     */
    associateSigninDelegateGroupsWithAccount() {
        this.add('chime:AssociateSigninDelegateGroupsWithAccount');
        return this;
    }
    /**
     * Grants permission to authorize an Active Directory for your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    authorizeDirectory() {
        this.add('chime:AuthorizeDirectory');
        return this;
    }
    /**
     * Grants permission to create new attendees for an active Amazon Chime SDK meeting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchCreateAttendee.html
     */
    batchCreateAttendee() {
        this.add('chime:BatchCreateAttendee');
        return this;
    }
    /**
     * Grants permission to batch add room members
     *
     * Access Level: Write
     */
    batchCreateRoomMembership() {
        this.add('chime:BatchCreateRoomMembership');
        return this;
    }
    /**
     * Grants permission to move up to 50 phone numbers to the deletion queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchDeletePhoneNumber.html
     */
    batchDeletePhoneNumber() {
        this.add('chime:BatchDeletePhoneNumber');
        return this;
    }
    /**
     * Grants permission to suspend up to 50 users from a Team or EnterpriseLWA Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchSuspendUser.html
     */
    batchSuspendUser() {
        this.add('chime:BatchSuspendUser');
        return this;
    }
    /**
     * Grants permission to remove the suspension from up to 50 previously suspended users for the specified Amazon Chime EnterpriseLWA account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUnsuspendUser.html
     */
    batchUnsuspendUser() {
        this.add('chime:BatchUnsuspendUser');
        return this;
    }
    /**
     * Grants permission to update phone number details within the UpdatePhoneNumberRequestItem object for up to 50 phone numbers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUpdatePhoneNumber.html
     */
    batchUpdatePhoneNumber() {
        this.add('chime:BatchUpdatePhoneNumber');
        return this;
    }
    /**
     * Grants permission to update user details within the UpdateUserRequestItem object for up to 20 users for the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUpdateUser.html
     */
    batchUpdateUser() {
        this.add('chime:BatchUpdateUser');
        return this;
    }
    /**
     * Grants permission to connect an Active Directory to your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ds:ConnectDirectory
     *
     * https://docs.aws.amazon.com/chime/latest/ag/active_directory.html
     */
    connectDirectory() {
        this.add('chime:ConnectDirectory');
        return this;
    }
    /**
     * Grants permission to create an Amazon Chime account under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateAccount.html
     */
    createAccount() {
        this.add('chime:CreateAccount');
        return this;
    }
    /**
     * Grants permission to create a new SCIM access key for your Amazon Chime account and Okta configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    createApiKey() {
        this.add('chime:CreateApiKey');
        return this;
    }
    /**
     * Grants permission to create a new attendee for an active Amazon Chime SDK meeting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateAttendee.html
     */
    createAttendee() {
        this.add('chime:CreateAttendee');
        return this;
    }
    /**
     * Grants permission to create a bot for an Amazon Chime Enterprise account
     *
     * Access Level: Write
     */
    createBot() {
        this.add('chime:CreateBot');
        return this;
    }
    /**
     * Grants permission to add a bot to a chat room in your Amazon Chime Enterprise account
     *
     * Access Level: Write
     */
    createBotMembership() {
        this.add('chime:CreateBotMembership');
        return this;
    }
    /**
     * Grants permission to create a new Call Detail Record S3 bucket
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:CreateBucket
     * - s3:ListAllMyBuckets
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    createCDRBucket() {
        this.add('chime:CreateCDRBucket');
        return this;
    }
    /**
     * Grants permission to create a new Amazon Chime SDK meeting in the specified media Region, with no initial attendees
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateMeeting.html
     */
    createMeeting() {
        this.add('chime:CreateMeeting');
        return this;
    }
    /**
     * Grants permission to create a phone number order with the Carriers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreatePhoneNumberOrder.html
     */
    createPhoneNumberOrder() {
        this.add('chime:CreatePhoneNumberOrder');
        return this;
    }
    /**
     * Grants permission to create a proxy session for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateProxySession.html
     */
    createProxySession() {
        this.add('chime:CreateProxySession');
        return this;
    }
    /**
     * Grants permission to create a room
     *
     * Access Level: Write
     */
    createRoom() {
        this.add('chime:CreateRoom');
        return this;
    }
    /**
     * Grants permission to add a room member
     *
     * Access Level: Write
     */
    createRoomMembership() {
        this.add('chime:CreateRoomMembership');
        return this;
    }
    /**
     * Grants permission to create a user under the specified Amazon Chime account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateUser.html
     */
    createUser() {
        this.add('chime:CreateUser');
        return this;
    }
    /**
     * Grants permission to create a Amazon Chime Voice Connector under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateVoiceConnector.html
     */
    createVoiceConnector() {
        this.add('chime:CreateVoiceConnector');
        return this;
    }
    /**
     * Grants permission to create a Amazon Chime Voice Connector Group under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateVoiceConnectorGroup.html
     */
    createVoiceConnectorGroup() {
        this.add('chime:CreateVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to delete the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteAccount.html
     */
    deleteAccount() {
        this.add('chime:DeleteAccount');
        return this;
    }
    /**
     * Grants permission to delete the OpenIdConfig attributes from your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    deleteAccountOpenIdConfig() {
        this.add('chime:DeleteAccountOpenIdConfig');
        return this;
    }
    /**
     * Grants permission to delete the specified SCIM access key associated with your Amazon Chime account and Okta configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    deleteApiKey() {
        this.add('chime:DeleteApiKey');
        return this;
    }
    /**
     * Grants permission to delete the specified attendee from an Amazon Chime SDK meeting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteAttendee.html
     */
    deleteAttendee() {
        this.add('chime:DeleteAttendee');
        return this;
    }
    /**
     * Grants permission to delete a Call Detail Record S3 bucket from your Amazon Chime account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:DeleteBucket
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    deleteCDRBucket() {
        this.add('chime:DeleteCDRBucket');
        return this;
    }
    /**
     * Grants permission to delete delegated AWS account management from your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    deleteDelegate() {
        this.add('chime:DeleteDelegate');
        return this;
    }
    /**
     * Grants permission to delete a domain from your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html
     */
    deleteDomain() {
        this.add('chime:DeleteDomain');
        return this;
    }
    /**
     * Grants permission to delete an events configuration for a bot to receive outgoing events
     *
     * Access Level: Write
     */
    deleteEventsConfiguration() {
        this.add('chime:DeleteEventsConfiguration');
        return this;
    }
    /**
     * Grants permission to delete Active Directory or Okta user groups from your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    deleteGroups() {
        this.add('chime:DeleteGroups');
        return this;
    }
    /**
     * Grants permission to delete the specified Amazon Chime SDK meeting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteMeeting.html
     */
    deleteMeeting() {
        this.add('chime:DeleteMeeting');
        return this;
    }
    /**
     * Grants permission to move a phone number to the deletion queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeletePhoneNumber.html
     */
    deletePhoneNumber() {
        this.add('chime:DeletePhoneNumber');
        return this;
    }
    /**
     * Grants permission to delete a proxy session for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteProxySession.html
     */
    deleteProxySession() {
        this.add('chime:DeleteProxySession');
        return this;
    }
    /**
     * Grants permission to delete a room
     *
     * Access Level: Write
     */
    deleteRoom() {
        this.add('chime:DeleteRoom');
        return this;
    }
    /**
     * Grants permission to remove a room member
     *
     * Access Level: Write
     */
    deleteRoomMembership() {
        this.add('chime:DeleteRoomMembership');
        return this;
    }
    /**
     * Grants permission to delete the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnector.html
     */
    deleteVoiceConnector() {
        this.add('chime:DeleteVoiceConnector');
        return this;
    }
    /**
     * Grants permission to delete emergency calling configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorEmergencyCallingConfiguration.html
     */
    deleteVoiceConnectorEmergencyCallingConfiguration() {
        this.add('chime:DeleteVoiceConnectorEmergencyCallingConfiguration');
        return this;
    }
    /**
     * Grants permission to delete the specified Amazon Chime Voice Connector Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorGroup.html
     */
    deleteVoiceConnectorGroup() {
        this.add('chime:DeleteVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to delete the origination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorOrigination.html
     */
    deleteVoiceConnectorOrigination() {
        this.add('chime:DeleteVoiceConnectorOrigination');
        return this;
    }
    /**
     * Grants permission to delete proxy configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorProxy.html
     */
    deleteVoiceConnectorProxy() {
        this.add('chime:DeleteVoiceConnectorProxy');
        return this;
    }
    /**
     * Grants permission to delete streaming configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorStreamingConfiguration.html
     */
    deleteVoiceConnectorStreamingConfiguration() {
        this.add('chime:DeleteVoiceConnectorStreamingConfiguration');
        return this;
    }
    /**
     * Grants permission to delete the termination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorTermination.html
     */
    deleteVoiceConnectorTermination() {
        this.add('chime:DeleteVoiceConnectorTermination');
        return this;
    }
    /**
     * Grants permission to delete SIP termination credentials for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorTerminationCredentials.html
     */
    deleteVoiceConnectorTerminationCredentials() {
        this.add('chime:DeleteVoiceConnectorTerminationCredentials');
        return this;
    }
    /**
     * Grants permission to disassociate the primary provisioned number from the specified Amazon Chime user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumberFromUser.html
     */
    disassociatePhoneNumberFromUser() {
        this.add('chime:DisassociatePhoneNumberFromUser');
        return this;
    }
    /**
     * Grants permission to disassociate multiple phone numbers from the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumbersFromVoiceConnector.html
     */
    disassociatePhoneNumbersFromVoiceConnector() {
        this.add('chime:DisassociatePhoneNumbersFromVoiceConnector');
        return this;
    }
    /**
     * Grants permission to disassociate multiple phone numbers from the specified Amazon Chime Voice Connector Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumbersFromVoiceConnectorGroup.html
     */
    disassociatePhoneNumbersFromVoiceConnectorGroup() {
        this.add('chime:DisassociatePhoneNumbersFromVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to disassociate the specified sign-in delegate groups from the specified Amazon Chime account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociateSigninDelegateGroupsFromAccount.html
     */
    disassociateSigninDelegateGroupsFromAccount() {
        this.add('chime:DisassociateSigninDelegateGroupsFromAccount');
        return this;
    }
    /**
     * Grants permission to disconnect the Active Directory from your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    disconnectDirectory() {
        this.add('chime:DisconnectDirectory');
        return this;
    }
    /**
     * Grants permission to get details for the specified Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAccount.html
     */
    getAccount() {
        this.add('chime:GetAccount');
        return this;
    }
    /**
     * Grants permission to get details for the account resource associated with your Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    getAccountResource() {
        this.add('chime:GetAccountResource');
        return this;
    }
    /**
     * Grants permission to get account settings for the specified Amazon Chime account ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAccountSettings.html
     */
    getAccountSettings() {
        this.add('chime:GetAccountSettings');
        return this;
    }
    /**
     * Grants permission to get the account details and OpenIdConfig attributes for your Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    getAccountWithOpenIdConfig() {
        this.add('chime:GetAccountWithOpenIdConfig');
        return this;
    }
    /**
     * Grants permission to get attendee details for a specified meeting ID and attendee ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAttendee.html
     */
    getAttendee() {
        this.add('chime:GetAttendee');
        return this;
    }
    /**
     * Grants permission to retrieve details for the specified bot
     *
     * Access Level: Read
     */
    getBot() {
        this.add('chime:GetBot');
        return this;
    }
    /**
     * Grants permission to get details of a Call Detail Record S3 bucket associated with your Amazon Chime account
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - s3:GetBucketAcl
     * - s3:GetBucketLocation
     * - s3:GetBucketLogging
     * - s3:GetBucketVersioning
     * - s3:GetBucketWebsite
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    getCDRBucket() {
        this.add('chime:GetCDRBucket');
        return this;
    }
    /**
     * Grants permission to get domain details for a domain associated with your Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html
     */
    getDomain() {
        this.add('chime:GetDomain');
        return this;
    }
    /**
     * Grants permission to retrieve details for an events configuration for a bot to receive outgoing events
     *
     * Access Level: Read
     */
    getEventsConfiguration() {
        this.add('chime:GetEventsConfiguration');
        return this;
    }
    /**
     * Grants permission to get global settings related to Amazon Chime for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetGlobalSettings.html
     */
    getGlobalSettings() {
        this.add('chime:GetGlobalSettings');
        return this;
    }
    /**
     * Grants permission to get the meeting record for a specified meeting ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetMeeting.html
     */
    getMeeting() {
        this.add('chime:GetMeeting');
        return this;
    }
    /**
     * Grants permission to get attendee, connection, and other details for a meeting
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    getMeetingDetail() {
        this.add('chime:GetMeetingDetail');
        return this;
    }
    /**
     * Grants permission to get details for the specified phone number
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumber.html
     */
    getPhoneNumber() {
        this.add('chime:GetPhoneNumber');
        return this;
    }
    /**
     * Grants permission to get details for the specified phone number order
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumberOrder.html
     */
    getPhoneNumberOrder() {
        this.add('chime:GetPhoneNumberOrder');
        return this;
    }
    /**
     * Grants permission to get phone number settings related to Amazon Chime for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumberSettings.html
     */
    getPhoneNumberSettings() {
        this.add('chime:GetPhoneNumberSettings');
        return this;
    }
    /**
     * Grants permission to get details of the specified proxy session for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetProxySession.html
     */
    getProxySession() {
        this.add('chime:GetProxySession');
        return this;
    }
    /**
     * Gets the retention settings for the specified Amazon Chime account.
     *
     * Access Level: Read
     */
    getRetentionSettings() {
        this.add('chime:GetRetentionSettings');
        return this;
    }
    /**
     * Grants permission to retrieve a room
     *
     * Access Level: Read
     */
    getRoom() {
        this.add('chime:GetRoom');
        return this;
    }
    /**
     * Grants permission to get telephony limits for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/phone-numbers.html
     */
    getTelephonyLimits() {
        this.add('chime:GetTelephonyLimits');
        return this;
    }
    /**
     * Grants permission to get details for the specified user ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetUser.html
     */
    getUser() {
        this.add('chime:GetUser');
        return this;
    }
    /**
     * Grants permission to get a summary of user activity on the user details page
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/user-details.html
     */
    getUserActivityReportData() {
        this.add('chime:GetUserActivityReportData');
        return this;
    }
    /**
     * Grants permission to get user details for an Amazon Chime user based on the email address in an Amazon Chime Enterprise or Team account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/user-details.html
     */
    getUserByEmail() {
        this.add('chime:GetUserByEmail');
        return this;
    }
    /**
     * Grants permission to get user settings related to the specified Amazon Chime user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetUserSettings.html
     */
    getUserSettings() {
        this.add('chime:GetUserSettings');
        return this;
    }
    /**
     * Grants permission to get details for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnector.html
     */
    getVoiceConnector() {
        this.add('chime:GetVoiceConnector');
        return this;
    }
    /**
     * Grants permission to get details of the emergency calling configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorEmergencyCallingConfiguration.html
     */
    getVoiceConnectorEmergencyCallingConfiguration() {
        this.add('chime:GetVoiceConnectorEmergencyCallingConfiguration');
        return this;
    }
    /**
     * Grants permission to get details for the specified Amazon Chime Voice Connector Group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorGroup.html
     */
    getVoiceConnectorGroup() {
        this.add('chime:GetVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to get details of the logging configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorLoggingConfiguration.html
     */
    getVoiceConnectorLoggingConfiguration() {
        this.add('chime:GetVoiceConnectorLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to get details of the origination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorOrigination.html
     */
    getVoiceConnectorOrigination() {
        this.add('chime:GetVoiceConnectorOrigination');
        return this;
    }
    /**
     * Grants permission to get details of the proxy configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorProxy.html
     */
    getVoiceConnectorProxy() {
        this.add('chime:GetVoiceConnectorProxy');
        return this;
    }
    /**
     * Grants permission to get details of the streaming configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorStreamingConfiguration.html
     */
    getVoiceConnectorStreamingConfiguration() {
        this.add('chime:GetVoiceConnectorStreamingConfiguration');
        return this;
    }
    /**
     * Grants permission to get details of the termination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorTermination.html
     */
    getVoiceConnectorTermination() {
        this.add('chime:GetVoiceConnectorTermination');
        return this;
    }
    /**
     * Grants permission to get details of the termination health for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorTerminationHealth.html
     */
    getVoiceConnectorTerminationHealth() {
        this.add('chime:GetVoiceConnectorTerminationHealth');
        return this;
    }
    /**
     * Grants permission to send an invitation to accept a request for AWS account delegation for an Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    inviteDelegate() {
        this.add('chime:InviteDelegate');
        return this;
    }
    /**
     * Grants permission to invite as many as 50 users to the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_InviteUsers.html
     */
    inviteUsers() {
        this.add('chime:InviteUsers');
        return this;
    }
    /**
     * Grants permission to invite users from a third party provider to your Amazon Chime account
     *
     * Access Level: Write
     */
    inviteUsersFromProvider() {
        this.add('chime:InviteUsersFromProvider');
        return this;
    }
    /**
     * Grants permission to list Amazon Chime account usage reporting data
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/view-reports.html
     */
    listAccountUsageReportData() {
        this.add('chime:ListAccountUsageReportData');
        return this;
    }
    /**
     * Grants permission to list the Amazon Chime accounts under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAccounts.html
     */
    listAccounts() {
        this.add('chime:ListAccounts');
        return this;
    }
    /**
     * Grants permission to list the SCIM access keys defined for your Amazon Chime account and Okta configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    listApiKeys() {
        this.add('chime:ListApiKeys');
        return this;
    }
    /**
     * Grants permission to list the tags applied to an Amazon Chime SDK attendee resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAttendeeTags.html
     */
    listAttendeeTags() {
        this.add('chime:ListAttendeeTags');
        return this;
    }
    /**
     * Grants permission to list up to 100 attendees for a specified Amazon Chime SDK meeting
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAttendees.html
     */
    listAttendees() {
        this.add('chime:ListAttendees');
        return this;
    }
    /**
     * Grants permission to list the bots associated with the administrator's Amazon Chime Enterprise account
     *
     * Access Level: List
     */
    listBots() {
        this.add('chime:ListBots');
        return this;
    }
    /**
     * Grants permission to list Call Detail Record S3 buckets
     *
     * Access Level: List
     *
     * Dependent actions:
     * - s3:ListAllMyBuckets
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    listCDRBucket() {
        this.add('chime:ListCDRBucket');
        return this;
    }
    /**
     * Grants permission to list the calling regions available for the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/phone-numbers.html
     */
    listCallingRegions() {
        this.add('chime:ListCallingRegions');
        return this;
    }
    /**
     * Grants permission to list account delegate information associated with your Amazon Chime account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    listDelegates() {
        this.add('chime:ListDelegates');
        return this;
    }
    /**
     * Grants permission to list active Active Directories hosted in the Directory Service of your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    listDirectories() {
        this.add('chime:ListDirectories');
        return this;
    }
    /**
     * Grants permission to list domains associated with your Amazon Chime account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html
     */
    listDomains() {
        this.add('chime:ListDomains');
        return this;
    }
    /**
     * Grants permission to list Active Directory or Okta user groups associated with your Amazon Chime Enterprise account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    listGroups() {
        this.add('chime:ListGroups');
        return this;
    }
    /**
     * Grants permission to list all events that occurred for a specified meeting
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/view-reports.html
     */
    listMeetingEvents() {
        this.add('chime:ListMeetingEvents');
        return this;
    }
    /**
     * Grants permission to list the tags applied to an Amazon Chime SDK meeting resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListMeetingTags.html
     */
    listMeetingTags() {
        this.add('chime:ListMeetingTags');
        return this;
    }
    /**
     * Grants permission to list up to 100 active Amazon Chime SDK meetings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListMeetings.html
     */
    listMeetings() {
        this.add('chime:ListMeetings');
        return this;
    }
    /**
     * Grants permission to list meetings ended during the specified date range
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/view-reports.html
     */
    listMeetingsReportData() {
        this.add('chime:ListMeetingsReportData');
        return this;
    }
    /**
     * Grants permission to list the phone number orders under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListPhoneNumberOrders.html
     */
    listPhoneNumberOrders() {
        this.add('chime:ListPhoneNumberOrders');
        return this;
    }
    /**
     * Grants permission to list the phone numbers under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListPhoneNumbers.html
     */
    listPhoneNumbers() {
        this.add('chime:ListPhoneNumbers');
        return this;
    }
    /**
     * Grants permission to list proxy sessions for the specified Amazon Chime Voice Connector
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListProxySessions.html
     */
    listProxySessions() {
        this.add('chime:ListProxySessions');
        return this;
    }
    /**
     * Grants permission to list all room members
     *
     * Access Level: Read
     */
    listRoomMemberships() {
        this.add('chime:ListRoomMemberships');
        return this;
    }
    /**
     * Grants permission to list rooms
     *
     * Access Level: Read
     */
    listRooms() {
        this.add('chime:ListRooms');
        return this;
    }
    /**
     * Grants permission to list the tags applied to an Amazon Chime SDK meeting resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('chime:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to list the users that belong to the specified Amazon Chime account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListUsers.html
     */
    listUsers() {
        this.add('chime:ListUsers');
        return this;
    }
    /**
     * Grants permission to list the Amazon Chime Voice Connector Groups under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectorGroups.html
     */
    listVoiceConnectorGroups() {
        this.add('chime:ListVoiceConnectorGroups');
        return this;
    }
    /**
     * Grants permission to list the SIP termination credentials for the specified Amazon Chime Voice Connector
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectorTerminationCredentials.html
     */
    listVoiceConnectorTerminationCredentials() {
        this.add('chime:ListVoiceConnectorTerminationCredentials');
        return this;
    }
    /**
     * Grants permission to list the Amazon Chime Voice Connectors under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectors.html
     */
    listVoiceConnectors() {
        this.add('chime:ListVoiceConnectors');
        return this;
    }
    /**
     * Grants permission to log out the specified user from all of the devices they are currently logged into
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_LogoutUser.html
     */
    logoutUser() {
        this.add('chime:LogoutUser');
        return this;
    }
    /**
     * Grants permission to update details for an events configuration for a bot to receive outgoing events
     *
     * Access Level: Write
     */
    putEventsConfiguration() {
        this.add('chime:PutEventsConfiguration');
        return this;
    }
    /**
     * Puts retention settings for the specified Amazon Chime account
     *
     * Access Level: Write
     */
    putRetentionSettings() {
        this.add('chime:PutRetentionSettings');
        return this;
    }
    /**
     * Grants permission to add emergency calling configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorEmergencyCallingConfiguration.html
     */
    putVoiceConnectorEmergencyCallingConfiguration() {
        this.add('chime:PutVoiceConnectorEmergencyCallingConfiguration');
        return this;
    }
    /**
     * Grants permission to add logging configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - logs:CreateLogDelivery
     * - logs:CreateLogGroup
     * - logs:DeleteLogDelivery
     * - logs:DescribeLogGroups
     * - logs:GetLogDelivery
     * - logs:ListLogDeliveries
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorLoggingConfiguration.html
     */
    putVoiceConnectorLoggingConfiguration() {
        this.add('chime:PutVoiceConnectorLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to update the origination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorOrigination.html
     */
    putVoiceConnectorOrigination() {
        this.add('chime:PutVoiceConnectorOrigination');
        return this;
    }
    /**
     * Grants permission to add proxy configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorProxy.html
     */
    putVoiceConnectorProxy() {
        this.add('chime:PutVoiceConnectorProxy');
        return this;
    }
    /**
     * Grants permission to add streaming configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorStreamingConfiguration.html
     */
    putVoiceConnectorStreamingConfiguration() {
        this.add('chime:PutVoiceConnectorStreamingConfiguration');
        return this;
    }
    /**
     * Grants permission to update the termination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorTermination.html
     */
    putVoiceConnectorTermination() {
        this.add('chime:PutVoiceConnectorTermination');
        return this;
    }
    /**
     * Grants permission to add SIP termination credentials for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorTerminationCredentials.html
     */
    putVoiceConnectorTerminationCredentials() {
        this.add('chime:PutVoiceConnectorTerminationCredentials');
        return this;
    }
    /**
     * Redacts the specified Chime conversation Message
     *
     * Access Level: Write
     */
    redactConversationMessage() {
        this.add('chime:RedactConversationMessage');
        return this;
    }
    /**
     * Redacts the specified Chime room Message
     *
     * Access Level: Write
     */
    redactRoomMessage() {
        this.add('chime:RedactRoomMessage');
        return this;
    }
    /**
     * Grants permission to regenerate the security token for the specified bot
     *
     * Access Level: Write
     */
    regenerateSecurityToken() {
        this.add('chime:RegenerateSecurityToken');
        return this;
    }
    /**
     * Grants permission to modify the account name for your Amazon Chime Enterprise or Team account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/rename-account.html
     */
    renameAccount() {
        this.add('chime:RenameAccount');
        return this;
    }
    /**
     * Grants permission to renew the delegation request associated with an Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    renewDelegate() {
        this.add('chime:RenewDelegate');
        return this;
    }
    /**
     * Grants permission to reset the account resource in your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    resetAccountResource() {
        this.add('chime:ResetAccountResource');
        return this;
    }
    /**
     * Grants permission to reset the personal meeting PIN for the specified user on an Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ResetPersonalPIN.html
     */
    resetPersonalPIN() {
        this.add('chime:ResetPersonalPIN');
        return this;
    }
    /**
     * Grants permission to restore the specified phone number from the deltion queue back to the phone number inventory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_RestorePhoneNumber.html
     */
    restorePhoneNumber() {
        this.add('chime:RestorePhoneNumber');
        return this;
    }
    /**
     * Grants permission to download the file containing links to all user attachments returned as part of the "Request attachments" action
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/request-attachments.html
     */
    retrieveDataExports() {
        this.add('chime:RetrieveDataExports');
        return this;
    }
    /**
     * Grants permission to search phone numbers that can be ordered from the carrier
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_SearchAvailablePhoneNumbers.html
     */
    searchAvailablePhoneNumbers() {
        this.add('chime:SearchAvailablePhoneNumbers');
        return this;
    }
    /**
     * Grants permission to submit the "Request attachments" request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/request-attachments.html
     */
    startDataExport() {
        this.add('chime:StartDataExport');
        return this;
    }
    /**
     * Grants permission to submit a customer service support request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/chime-getting-admin-support.html
     */
    submitSupportRequest() {
        this.add('chime:SubmitSupportRequest');
        return this;
    }
    /**
     * Grants permission to suspend users from an Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    suspendUsers() {
        this.add('chime:SuspendUsers');
        return this;
    }
    /**
     * Grants permission to apply the specified tags to the specified Amazon Chime SDK attendee
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_TagAttendee.html
     */
    tagAttendee() {
        this.add('chime:TagAttendee');
        return this;
    }
    /**
     * Grants permission to apply the specified tags to the specified Amazon Chime SDK meeting.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_TagMeeting.html
     */
    tagMeeting() {
        this.add('chime:TagMeeting');
        return this;
    }
    /**
     * Grants permission to apply the specified tags to the specified Amazon Chime SDK meeting resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('chime:TagResource');
        return this;
    }
    /**
     * Grants permission to unauthorize an Active Directory from your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    unauthorizeDirectory() {
        this.add('chime:UnauthorizeDirectory');
        return this;
    }
    /**
     * Grants permission to untag the specified tags from the specified Amazon Chime SDK attendee.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagAttendee.html
     */
    untagAttendee() {
        this.add('chime:UntagAttendee');
        return this;
    }
    /**
     * Grants permission to untag the specified tags from the specified Amazon Chime SDK meeting.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagMeeting.html
     */
    untagMeeting() {
        this.add('chime:UntagMeeting');
        return this;
    }
    /**
     * Grants permission to untag the specified tags from the specified Amazon Chime SDK meeting resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('chime:UntagResource');
        return this;
    }
    /**
     * Grants permission to update account details for the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateAccount.html
     */
    updateAccount() {
        this.add('chime:UpdateAccount');
        return this;
    }
    /**
     * Grants permission to update the OpenIdConfig attributes for your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    updateAccountOpenIdConfig() {
        this.add('chime:UpdateAccountOpenIdConfig');
        return this;
    }
    /**
     * Grants permission to update the account resource in your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    updateAccountResource() {
        this.add('chime:UpdateAccountResource');
        return this;
    }
    /**
     * Grants permission to update the settings for the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateAccountSettings.html
     */
    updateAccountSettings() {
        this.add('chime:UpdateAccountSettings');
        return this;
    }
    /**
     * Grants permission to update the status of the specified bot
     *
     * Access Level: Write
     */
    updateBot() {
        this.add('chime:UpdateBot');
        return this;
    }
    /**
     * Grants permission to update your Call Detail Record S3 bucket
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:CreateBucket
     * - s3:DeleteBucket
     * - s3:ListAllMyBuckets
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    updateCDRSettings() {
        this.add('chime:UpdateCDRSettings');
        return this;
    }
    /**
     * Grants permission to update the global settings related to Amazon Chime for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateGlobalSettings.html
     */
    updateGlobalSettings() {
        this.add('chime:UpdateGlobalSettings');
        return this;
    }
    /**
     * Grants permission to update phone number details for the specified phone number
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdatePhoneNumber.html
     */
    updatePhoneNumber() {
        this.add('chime:UpdatePhoneNumber');
        return this;
    }
    /**
     * Grants permission to update phone number settings related to Amazon Chime for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdatePhoneNumberSettings.html
     */
    updatePhoneNumberSettings() {
        this.add('chime:UpdatePhoneNumberSettings');
        return this;
    }
    /**
     * Grants permission to update a proxy session for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateProxySession.html
     */
    updateProxySession() {
        this.add('chime:UpdateProxySession');
        return this;
    }
    /**
     * Grants permission to update a room
     *
     * Access Level: Write
     */
    updateRoom() {
        this.add('chime:UpdateRoom');
        return this;
    }
    /**
     * Grants permission to update room membership role
     *
     * Access Level: Write
     */
    updateRoomMembership() {
        this.add('chime:UpdateRoomMembership');
        return this;
    }
    /**
     * Grants permission to update the supported license tiers available for users in your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    updateSupportedLicenses() {
        this.add('chime:UpdateSupportedLicenses');
        return this;
    }
    /**
     * Grants permission to update user details for a specified user ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateUser.html
     */
    updateUser() {
        this.add('chime:UpdateUser');
        return this;
    }
    /**
     * Grants permission to update the licenses for your Amazon Chime users
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    updateUserLicenses() {
        this.add('chime:UpdateUserLicenses');
        return this;
    }
    /**
     * Grants permission to update user settings related to the specified Amazon Chime user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateUserSettings.html
     */
    updateUserSettings() {
        this.add('chime:UpdateUserSettings');
        return this;
    }
    /**
     * Grants permission to update Amazon Chime Voice Connector details for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateVoiceConnector.html
     */
    updateVoiceConnector() {
        this.add('chime:UpdateVoiceConnector');
        return this;
    }
    /**
     * Grants permission to update Amazon Chime Voice Connector Group details for the specified Amazon Chime Voice Connector Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateVoiceConnectorGroup.html
     */
    updateVoiceConnectorGroup() {
        this.add('chime:UpdateVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to validate the account resource in your Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    validateAccountResource() {
        this.add('chime:ValidateAccountResource');
        return this;
    }
    /**
     * Adds a resource of type meeting to the statement
     *
     * @param meetingId - Identifier for the meetingId.
     * @param accountId - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMeeting(meetingId, accountId, partition) {
        var arn = 'arn:${Partition}:chime::${AccountId}:meeting/${MeetingId}';
        arn = arn.replace('${MeetingId}', meetingId);
        arn = arn.replace('${AccountId}', accountId || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .createMeeting()
     * - .tagMeeting()
     * - .tagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .tagMeeting()
     * - .tagResource()
     *
     * Applies to resource types:
     * - meeting
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .createMeeting()
     * - .tagMeeting()
     * - .tagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Chime = Chime;
//# sourceMappingURL=data:application/json;base64,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