"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dynamodb = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [dynamodb](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazondynamodb.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Dynamodb extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [dynamodb](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazondynamodb.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'dynamodb';
        this.actionList = {
            "BatchGetItem": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchGetItem.html",
                "description": "Returns the attributes of one or more items from one or more tables",
                "accessLevel": "Read",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                },
                "conditions": [
                    "dynamodb:Attributes",
                    "dynamodb:LeadingKeys",
                    "dynamodb:ReturnConsumedCapacity",
                    "dynamodb:Select"
                ]
            },
            "BatchWriteItem": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchWriteItem.html",
                "description": "Puts or deletes multiple items in one or more tables",
                "accessLevel": "Write",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                },
                "conditions": [
                    "dynamodb:Attributes",
                    "dynamodb:LeadingKeys",
                    "dynamodb:ReturnConsumedCapacity"
                ]
            },
            "ConditionCheckItem": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ConditionCheckItem.html",
                "description": "The ConditionCheckItem operation checks the existence of a set of attributes for the item with the given primary key",
                "accessLevel": "Read",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                },
                "conditions": [
                    "dynamodb:Attributes",
                    "dynamodb:LeadingKeys",
                    "dynamodb:ReturnConsumedCapacity",
                    "dynamodb:ReturnValues"
                ]
            },
            "CreateBackup": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_CreateBackup.html",
                "description": "Creates a backup for an existing table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                }
            },
            "CreateGlobalTable": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_CreateGlobalTable.html",
                "description": "Enables the user to create a global table from an existing table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "global-table": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "CreateTable": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_CreateTable.html",
                "description": "The CreateTable operation adds a new table to your account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                }
            },
            "CreateTableReplica": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/V2gt_IAM.html",
                "description": "Adds a new replica table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                }
            },
            "DeleteBackup": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteBackup.html",
                "description": "Deletes an existing backup of a table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backup": {
                        "required": true
                    }
                }
            },
            "DeleteItem": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html",
                "description": "Deletes a single item in a table by primary key",
                "accessLevel": "Write",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                },
                "conditions": [
                    "dynamodb:Attributes",
                    "dynamodb:EnclosingOperation",
                    "dynamodb:LeadingKeys",
                    "dynamodb:ReturnConsumedCapacity",
                    "dynamodb:ReturnValues"
                ]
            },
            "DeleteTable": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteTable.html",
                "description": "The DeleteTable operation deletes a table and all of its items",
                "accessLevel": "Write",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                }
            },
            "DeleteTableReplica": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/V2gt_IAM.html",
                "description": "Deletes a replica table and all of its items",
                "accessLevel": "Write",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                }
            },
            "DescribeBackup": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeBackup.html",
                "description": "Describes an existing backup of a table",
                "accessLevel": "Read",
                "resourceTypes": {
                    "backup": {
                        "required": true
                    }
                }
            },
            "DescribeContinuousBackups": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeContinuousBackups.html",
                "description": "Checks the status of the backup restore settings on the specified table",
                "accessLevel": "Read",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                }
            },
            "DescribeContributorInsights": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeContributorInsights.html",
                "description": "Describes the contributor insights status and related details for a given table or global secondary index",
                "accessLevel": "Read",
                "resourceTypes": {
                    "table": {
                        "required": true
                    },
                    "index": {
                        "required": false
                    }
                }
            },
            "DescribeGlobalTable": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeGlobalTable.html",
                "description": "Returns information about the specified global table",
                "accessLevel": "Read",
                "resourceTypes": {
                    "global-table": {
                        "required": true
                    }
                }
            },
            "DescribeGlobalTableSettings": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeGlobalTableSettings.html",
                "description": "Returns settings information about the specified global table",
                "accessLevel": "Read",
                "resourceTypes": {
                    "global-table": {
                        "required": true
                    }
                }
            },
            "DescribeLimits": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeLimits.html",
                "description": "Returns the current provisioned-capacity limits for your AWS account in a region, both for the region as a whole and for any one DynamoDB table that you create there",
                "accessLevel": "Read"
            },
            "DescribeReservedCapacity": {
                "url": "",
                "description": "Describes one or more of the Reserved Capacity purchased",
                "accessLevel": "Read"
            },
            "DescribeReservedCapacityOfferings": {
                "url": "",
                "description": "Describes Reserved Capacity offerings that are available for purchase",
                "accessLevel": "Read"
            },
            "DescribeStream": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeStream.html",
                "description": "Returns information about a stream, including the current status of the stream, its Amazon Resource Name (ARN), the composition of its shards, and its corresponding DynamoDB table",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "DescribeTable": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeTable.html",
                "description": "Returns information about the table",
                "accessLevel": "Read",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                }
            },
            "DescribeTableReplicaAutoScaling": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeTableReplicaAutoScaling.html",
                "description": "Describes the auto scaling settings across all replicas of the global table",
                "accessLevel": "Read",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                }
            },
            "DescribeTimeToLive": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeTimeToLive.html",
                "description": "Gives a description of the Time to Live (TTL) status on the specified table.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                }
            },
            "GetItem": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html",
                "description": "The GetItem operation returns a set of attributes for the item with the given primary key",
                "accessLevel": "Read",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                },
                "conditions": [
                    "dynamodb:Attributes",
                    "dynamodb:EnclosingOperation",
                    "dynamodb:LeadingKeys",
                    "dynamodb:ReturnConsumedCapacity",
                    "dynamodb:Select"
                ]
            },
            "GetRecords": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetRecords.html",
                "description": "Retrieves the stream records from a given shard",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "GetShardIterator": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetShardIterator.html",
                "description": "Returns a shard iterator",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "ListBackups": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListBackups.html",
                "description": "List backups associated with the account and endpoint",
                "accessLevel": "List"
            },
            "ListContributorInsights": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListContributorInsights.html",
                "description": "Lists the ContributorInsightsSummary for all tables and global secondary indexes associated with the current account and endpoint",
                "accessLevel": "List"
            },
            "ListGlobalTables": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListGlobalTables.html",
                "description": "Lists all global tables that have a replica in the specified region",
                "accessLevel": "List"
            },
            "ListStreams": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListStreams.html",
                "description": "Returns an array of stream ARNs associated with the current account and endpoint",
                "accessLevel": "Read"
            },
            "ListTables": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListTables.html",
                "description": "Returns an array of table names associated with the current account and endpoint",
                "accessLevel": "List"
            },
            "ListTagsOfResource": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListTagsOfResource.html",
                "description": "List all tags on an Amazon DynamoDB resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                }
            },
            "PurchaseReservedCapacityOfferings": {
                "url": "",
                "description": "Purchases Reserved Capacity for use with your account",
                "accessLevel": "Write"
            },
            "PutItem": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html",
                "description": "Creates a new item, or replaces an old item with a new item",
                "accessLevel": "Write",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                },
                "conditions": [
                    "dynamodb:Attributes",
                    "dynamodb:EnclosingOperation",
                    "dynamodb:LeadingKeys",
                    "dynamodb:ReturnConsumedCapacity",
                    "dynamodb:ReturnValues"
                ]
            },
            "Query": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Query.html",
                "description": "Uses the primary key of a table or a secondary index to directly access items from that table or index",
                "accessLevel": "Read",
                "resourceTypes": {
                    "table": {
                        "required": true
                    },
                    "index": {
                        "required": false
                    }
                },
                "conditions": [
                    "dynamodb:Attributes",
                    "dynamodb:LeadingKeys",
                    "dynamodb:ReturnConsumedCapacity",
                    "dynamodb:ReturnValues",
                    "dynamodb:Select"
                ]
            },
            "RestoreTableFromBackup": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_RestoreTableFromBackup.html",
                "description": "Creates a new table from an existing backup",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backup": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "RestoreTableToPointInTime": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_RestoreTableToPointInTime.html",
                "description": "Restores a table to a point in time",
                "accessLevel": "Write",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                }
            },
            "Scan": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Scan.html",
                "description": "Returns one or more items and item attributes by accessing every item in a table or a secondary index",
                "accessLevel": "Read",
                "resourceTypes": {
                    "table": {
                        "required": true
                    },
                    "index": {
                        "required": false
                    }
                },
                "conditions": [
                    "dynamodb:Attributes",
                    "dynamodb:ReturnConsumedCapacity",
                    "dynamodb:ReturnValues",
                    "dynamodb:Select"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_TagResource.html",
                "description": "Associate a set of tags with an Amazon DynamoDB resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UntagResource.html",
                "description": "Removes the association of tags from an Amazon DynamoDB resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                }
            },
            "UpdateContinuousBackups": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateContinuousBackups.html",
                "description": "Enables or disables continuous backups",
                "accessLevel": "Write",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                }
            },
            "UpdateContributorInsights": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateContributorInsights.html",
                "description": "Updates the status for contributor insights for a specific table or global secondary index",
                "accessLevel": "Write",
                "resourceTypes": {
                    "table": {
                        "required": true
                    },
                    "index": {
                        "required": false
                    }
                }
            },
            "UpdateGlobalTable": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateGlobalTable.html",
                "description": "Enables the user to add or remove replicas in the specified global table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "global-table": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "UpdateGlobalTableSettings": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateGlobalTableSettings.html",
                "description": "Enables the user to update settings of the specified global table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "global-table": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "UpdateItem": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html",
                "description": "Edits an existing item's attributes, or adds a new item to the table if it does not already exist",
                "accessLevel": "Write",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                },
                "conditions": [
                    "dynamodb:Attributes",
                    "dynamodb:EnclosingOperation",
                    "dynamodb:LeadingKeys",
                    "dynamodb:ReturnConsumedCapacity",
                    "dynamodb:ReturnValues"
                ]
            },
            "UpdateTable": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateTable.html",
                "description": "Modifies the provisioned throughput settings, global secondary indexes, or DynamoDB Streams settings for a given table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                }
            },
            "UpdateTableReplicaAutoScaling": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateTableReplicaAutoScaling.html",
                "description": "Updates auto scaling settings on your replica table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                }
            },
            "UpdateTimeToLive": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateTimeToLive.html",
                "description": "Enables or disables TTL for the specified table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "table": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "index": {
                "name": "index",
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.CoreComponents.htmlHowItWorks.CoreComponents.html#HowItWorks.CoreComponents.PrimaryKey",
                "arn": "arn:${Partition}:dynamodb:${Region}:${Account}:table/${TableName}/index/${IndexName}",
                "conditionKeys": []
            },
            "stream": {
                "name": "stream",
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.CoreComponents.htmlHowItWorks.CoreComponents.html#HowItWorks.CoreComponents.Streams",
                "arn": "arn:${Partition}:dynamodb:${Region}:${Account}:table/${TableName}/stream/${StreamLabel}",
                "conditionKeys": []
            },
            "table": {
                "name": "table",
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.CoreComponents.htmlHowItWorks.CoreComponents.html#HowItWorks.CoreComponents.TablesItemsAttributes",
                "arn": "arn:${Partition}:dynamodb:${Region}:${Account}:table/${TableName}",
                "conditionKeys": []
            },
            "backup": {
                "name": "backup",
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/backuprestore_HowItWorks.html",
                "arn": "arn:${Partition}:dynamodb:${Region}:${Account}:table/${TableName}/backup/${BackupName}",
                "conditionKeys": []
            },
            "global-table": {
                "name": "global-table",
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables_HowItWorks.html",
                "arn": "arn:${Partition}:dynamodb::${Account}:global-table/${GlobalTableName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Returns the attributes of one or more items from one or more tables
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchGetItem.html
     */
    batchGetItem() {
        this.add('dynamodb:BatchGetItem');
        return this;
    }
    /**
     * Puts or deletes multiple items in one or more tables
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchWriteItem.html
     */
    batchWriteItem() {
        this.add('dynamodb:BatchWriteItem');
        return this;
    }
    /**
     * The ConditionCheckItem operation checks the existence of a set of attributes for the item with the given primary key
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ConditionCheckItem.html
     */
    conditionCheckItem() {
        this.add('dynamodb:ConditionCheckItem');
        return this;
    }
    /**
     * Creates a backup for an existing table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_CreateBackup.html
     */
    createBackup() {
        this.add('dynamodb:CreateBackup');
        return this;
    }
    /**
     * Enables the user to create a global table from an existing table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_CreateGlobalTable.html
     */
    createGlobalTable() {
        this.add('dynamodb:CreateGlobalTable');
        return this;
    }
    /**
     * The CreateTable operation adds a new table to your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_CreateTable.html
     */
    createTable() {
        this.add('dynamodb:CreateTable');
        return this;
    }
    /**
     * Adds a new replica table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/V2gt_IAM.html
     */
    createTableReplica() {
        this.add('dynamodb:CreateTableReplica');
        return this;
    }
    /**
     * Deletes an existing backup of a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteBackup.html
     */
    deleteBackup() {
        this.add('dynamodb:DeleteBackup');
        return this;
    }
    /**
     * Deletes a single item in a table by primary key
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html
     */
    deleteItem() {
        this.add('dynamodb:DeleteItem');
        return this;
    }
    /**
     * The DeleteTable operation deletes a table and all of its items
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteTable.html
     */
    deleteTable() {
        this.add('dynamodb:DeleteTable');
        return this;
    }
    /**
     * Deletes a replica table and all of its items
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/V2gt_IAM.html
     */
    deleteTableReplica() {
        this.add('dynamodb:DeleteTableReplica');
        return this;
    }
    /**
     * Describes an existing backup of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeBackup.html
     */
    describeBackup() {
        this.add('dynamodb:DescribeBackup');
        return this;
    }
    /**
     * Checks the status of the backup restore settings on the specified table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeContinuousBackups.html
     */
    describeContinuousBackups() {
        this.add('dynamodb:DescribeContinuousBackups');
        return this;
    }
    /**
     * Describes the contributor insights status and related details for a given table or global secondary index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeContributorInsights.html
     */
    describeContributorInsights() {
        this.add('dynamodb:DescribeContributorInsights');
        return this;
    }
    /**
     * Returns information about the specified global table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeGlobalTable.html
     */
    describeGlobalTable() {
        this.add('dynamodb:DescribeGlobalTable');
        return this;
    }
    /**
     * Returns settings information about the specified global table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeGlobalTableSettings.html
     */
    describeGlobalTableSettings() {
        this.add('dynamodb:DescribeGlobalTableSettings');
        return this;
    }
    /**
     * Returns the current provisioned-capacity limits for your AWS account in a region, both for the region as a whole and for any one DynamoDB table that you create there
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeLimits.html
     */
    describeLimits() {
        this.add('dynamodb:DescribeLimits');
        return this;
    }
    /**
     * Describes one or more of the Reserved Capacity purchased
     *
     * Access Level: Read
     */
    describeReservedCapacity() {
        this.add('dynamodb:DescribeReservedCapacity');
        return this;
    }
    /**
     * Describes Reserved Capacity offerings that are available for purchase
     *
     * Access Level: Read
     */
    describeReservedCapacityOfferings() {
        this.add('dynamodb:DescribeReservedCapacityOfferings');
        return this;
    }
    /**
     * Returns information about a stream, including the current status of the stream, its Amazon Resource Name (ARN), the composition of its shards, and its corresponding DynamoDB table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeStream.html
     */
    describeStream() {
        this.add('dynamodb:DescribeStream');
        return this;
    }
    /**
     * Returns information about the table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeTable.html
     */
    describeTable() {
        this.add('dynamodb:DescribeTable');
        return this;
    }
    /**
     * Describes the auto scaling settings across all replicas of the global table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeTableReplicaAutoScaling.html
     */
    describeTableReplicaAutoScaling() {
        this.add('dynamodb:DescribeTableReplicaAutoScaling');
        return this;
    }
    /**
     * Gives a description of the Time to Live (TTL) status on the specified table.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeTimeToLive.html
     */
    describeTimeToLive() {
        this.add('dynamodb:DescribeTimeToLive');
        return this;
    }
    /**
     * The GetItem operation returns a set of attributes for the item with the given primary key
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html
     */
    getItem() {
        this.add('dynamodb:GetItem');
        return this;
    }
    /**
     * Retrieves the stream records from a given shard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetRecords.html
     */
    getRecords() {
        this.add('dynamodb:GetRecords');
        return this;
    }
    /**
     * Returns a shard iterator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetShardIterator.html
     */
    getShardIterator() {
        this.add('dynamodb:GetShardIterator');
        return this;
    }
    /**
     * List backups associated with the account and endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListBackups.html
     */
    listBackups() {
        this.add('dynamodb:ListBackups');
        return this;
    }
    /**
     * Lists the ContributorInsightsSummary for all tables and global secondary indexes associated with the current account and endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListContributorInsights.html
     */
    listContributorInsights() {
        this.add('dynamodb:ListContributorInsights');
        return this;
    }
    /**
     * Lists all global tables that have a replica in the specified region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListGlobalTables.html
     */
    listGlobalTables() {
        this.add('dynamodb:ListGlobalTables');
        return this;
    }
    /**
     * Returns an array of stream ARNs associated with the current account and endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListStreams.html
     */
    listStreams() {
        this.add('dynamodb:ListStreams');
        return this;
    }
    /**
     * Returns an array of table names associated with the current account and endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListTables.html
     */
    listTables() {
        this.add('dynamodb:ListTables');
        return this;
    }
    /**
     * List all tags on an Amazon DynamoDB resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListTagsOfResource.html
     */
    listTagsOfResource() {
        this.add('dynamodb:ListTagsOfResource');
        return this;
    }
    /**
     * Purchases Reserved Capacity for use with your account
     *
     * Access Level: Write
     */
    purchaseReservedCapacityOfferings() {
        this.add('dynamodb:PurchaseReservedCapacityOfferings');
        return this;
    }
    /**
     * Creates a new item, or replaces an old item with a new item
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html
     */
    putItem() {
        this.add('dynamodb:PutItem');
        return this;
    }
    /**
     * Uses the primary key of a table or a secondary index to directly access items from that table or index
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Query.html
     */
    query() {
        this.add('dynamodb:Query');
        return this;
    }
    /**
     * Creates a new table from an existing backup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_RestoreTableFromBackup.html
     */
    restoreTableFromBackup() {
        this.add('dynamodb:RestoreTableFromBackup');
        return this;
    }
    /**
     * Restores a table to a point in time
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_RestoreTableToPointInTime.html
     */
    restoreTableToPointInTime() {
        this.add('dynamodb:RestoreTableToPointInTime');
        return this;
    }
    /**
     * Returns one or more items and item attributes by accessing every item in a table or a secondary index
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Scan.html
     */
    scan() {
        this.add('dynamodb:Scan');
        return this;
    }
    /**
     * Associate a set of tags with an Amazon DynamoDB resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('dynamodb:TagResource');
        return this;
    }
    /**
     * Removes the association of tags from an Amazon DynamoDB resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('dynamodb:UntagResource');
        return this;
    }
    /**
     * Enables or disables continuous backups
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateContinuousBackups.html
     */
    updateContinuousBackups() {
        this.add('dynamodb:UpdateContinuousBackups');
        return this;
    }
    /**
     * Updates the status for contributor insights for a specific table or global secondary index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateContributorInsights.html
     */
    updateContributorInsights() {
        this.add('dynamodb:UpdateContributorInsights');
        return this;
    }
    /**
     * Enables the user to add or remove replicas in the specified global table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateGlobalTable.html
     */
    updateGlobalTable() {
        this.add('dynamodb:UpdateGlobalTable');
        return this;
    }
    /**
     * Enables the user to update settings of the specified global table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateGlobalTableSettings.html
     */
    updateGlobalTableSettings() {
        this.add('dynamodb:UpdateGlobalTableSettings');
        return this;
    }
    /**
     * Edits an existing item's attributes, or adds a new item to the table if it does not already exist
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html
     */
    updateItem() {
        this.add('dynamodb:UpdateItem');
        return this;
    }
    /**
     * Modifies the provisioned throughput settings, global secondary indexes, or DynamoDB Streams settings for a given table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateTable.html
     */
    updateTable() {
        this.add('dynamodb:UpdateTable');
        return this;
    }
    /**
     * Updates auto scaling settings on your replica table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateTableReplicaAutoScaling.html
     */
    updateTableReplicaAutoScaling() {
        this.add('dynamodb:UpdateTableReplicaAutoScaling');
        return this;
    }
    /**
     * Enables or disables TTL for the specified table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateTimeToLive.html
     */
    updateTimeToLive() {
        this.add('dynamodb:UpdateTimeToLive');
        return this;
    }
    /**
     * Adds a resource of type index to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.CoreComponents.htmlHowItWorks.CoreComponents.html#HowItWorks.CoreComponents.PrimaryKey
     *
     * @param tableName - Identifier for the tableName.
     * @param indexName - Identifier for the indexName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIndex(tableName, indexName, account, region, partition) {
        var arn = 'arn:${Partition}:dynamodb:${Region}:${Account}:table/${TableName}/index/${IndexName}';
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${IndexName}', indexName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type stream to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.CoreComponents.htmlHowItWorks.CoreComponents.html#HowItWorks.CoreComponents.Streams
     *
     * @param tableName - Identifier for the tableName.
     * @param streamLabel - Identifier for the streamLabel.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onStream(tableName, streamLabel, account, region, partition) {
        var arn = 'arn:${Partition}:dynamodb:${Region}:${Account}:table/${TableName}/stream/${StreamLabel}';
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${StreamLabel}', streamLabel);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type table to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.CoreComponents.htmlHowItWorks.CoreComponents.html#HowItWorks.CoreComponents.TablesItemsAttributes
     *
     * @param tableName - Identifier for the tableName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTable(tableName, account, region, partition) {
        var arn = 'arn:${Partition}:dynamodb:${Region}:${Account}:table/${TableName}';
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type backup to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/backuprestore_HowItWorks.html
     *
     * @param tableName - Identifier for the tableName.
     * @param backupName - Identifier for the backupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBackup(tableName, backupName, account, region, partition) {
        var arn = 'arn:${Partition}:dynamodb:${Region}:${Account}:table/${TableName}/backup/${BackupName}';
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${BackupName}', backupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type global-table to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables_HowItWorks.html
     *
     * @param globalTableName - Identifier for the globalTableName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGlobalTable(globalTableName, account, partition) {
        var arn = 'arn:${Partition}:dynamodb::${Account}:global-table/${GlobalTableName}';
        arn = arn.replace('${GlobalTableName}', globalTableName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filter based on the attribute (field or column) names of the table.
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .batchGetItem()
     * - .batchWriteItem()
     * - .conditionCheckItem()
     * - .deleteItem()
     * - .getItem()
     * - .putItem()
     * - .query()
     * - .scan()
     * - .updateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAttributes(value, operator) {
        return this.if(`dynamodb:Attributes`, value, operator || 'StringLike');
    }
    /**
     * Used to block Transactions APIs calls and allow the non-Transaction APIs calls and vice-versa.
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .deleteItem()
     * - .getItem()
     * - .putItem()
     * - .updateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnclosingOperation(value, operator) {
        return this.if(`dynamodb:EnclosingOperation`, value, operator || 'StringLike');
    }
    /**
     * Filters based on the partition key of the table.
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .batchGetItem()
     * - .batchWriteItem()
     * - .conditionCheckItem()
     * - .deleteItem()
     * - .getItem()
     * - .putItem()
     * - .query()
     * - .updateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLeadingKeys(value, operator) {
        return this.if(`dynamodb:LeadingKeys`, value, operator || 'StringLike');
    }
    /**
     * Filter based on the ReturnConsumedCapacity parameter of a request. Contains either "TOTAL" or "NONE".
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .batchGetItem()
     * - .batchWriteItem()
     * - .conditionCheckItem()
     * - .deleteItem()
     * - .getItem()
     * - .putItem()
     * - .query()
     * - .scan()
     * - .updateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReturnConsumedCapacity(value, operator) {
        return this.if(`dynamodb:ReturnConsumedCapacity`, value, operator || 'StringLike');
    }
    /**
     * Filter based on the ReturnValues parameter of request. Contains one of the following: "ALL_OLD", "UPDATED_OLD","ALL_NEW","UPDATED_NEW", or "NONE".
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .conditionCheckItem()
     * - .deleteItem()
     * - .putItem()
     * - .query()
     * - .scan()
     * - .updateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReturnValues(value, operator) {
        return this.if(`dynamodb:ReturnValues`, value, operator || 'StringLike');
    }
    /**
     * Filter based on the Select parameter of a Query or Scan request.
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .batchGetItem()
     * - .getItem()
     * - .query()
     * - .scan()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSelect(value, operator) {
        return this.if(`dynamodb:Select`, value, operator || 'StringLike');
    }
}
exports.Dynamodb = Dynamodb;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYW1hem9uZHluYW1vZGIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhbWF6b25keW5hbW9kYi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxzQ0FBb0U7QUFFcEU7Ozs7R0FJRztBQUNILE1BQWEsUUFBUyxTQUFRLHdCQUFlO0lBMGlCM0M7Ozs7T0FJRztJQUNILFlBQWEsR0FBWTtRQUN2QixLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUEvaUJOLGtCQUFhLEdBQUcsVUFBVSxDQUFDO1FBQ3hCLGVBQVUsR0FBWTtZQUM5QixjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLHFFQUFxRTtnQkFDcEYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixxQkFBcUI7b0JBQ3JCLHNCQUFzQjtvQkFDdEIsaUNBQWlDO29CQUNqQyxpQkFBaUI7aUJBQ2xCO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLHdGQUF3RjtnQkFDL0YsYUFBYSxFQUFFLHNEQUFzRDtnQkFDckUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixxQkFBcUI7b0JBQ3JCLHNCQUFzQjtvQkFDdEIsaUNBQWlDO2lCQUNsQzthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSw0RkFBNEY7Z0JBQ25HLGFBQWEsRUFBRSxzSEFBc0g7Z0JBQ3JJLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1oscUJBQXFCO29CQUNyQixzQkFBc0I7b0JBQ3RCLGlDQUFpQztvQkFDakMsdUJBQXVCO2lCQUN4QjthQUNGO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLEtBQUssRUFBRSxzRkFBc0Y7Z0JBQzdGLGFBQWEsRUFBRSx3Q0FBd0M7Z0JBQ3ZELGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSwyRkFBMkY7Z0JBQ2xHLGFBQWEsRUFBRSxrRUFBa0U7Z0JBQ2pGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsY0FBYyxFQUFFO3dCQUNkLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtvQkFDRCxPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLDREQUE0RDtnQkFDM0UsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLDhFQUE4RTtnQkFDckYsYUFBYSxFQUFFLDBCQUEwQjtnQkFDekMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLHVDQUF1QztnQkFDdEQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxZQUFZLEVBQUU7Z0JBQ1osS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLGlEQUFpRDtnQkFDaEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixxQkFBcUI7b0JBQ3JCLDZCQUE2QjtvQkFDN0Isc0JBQXNCO29CQUN0QixpQ0FBaUM7b0JBQ2pDLHVCQUF1QjtpQkFDeEI7YUFDRjtZQUNELGFBQWEsRUFBRTtnQkFDYixLQUFLLEVBQUUscUZBQXFGO2dCQUM1RixhQUFhLEVBQUUsZ0VBQWdFO2dCQUMvRSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELG9CQUFvQixFQUFFO2dCQUNwQixLQUFLLEVBQUUsOEVBQThFO2dCQUNyRixhQUFhLEVBQUUsOENBQThDO2dCQUM3RCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELGdCQUFnQixFQUFFO2dCQUNoQixLQUFLLEVBQUUsd0ZBQXdGO2dCQUMvRixhQUFhLEVBQUUseUNBQXlDO2dCQUN4RCxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELDJCQUEyQixFQUFFO2dCQUMzQixLQUFLLEVBQUUsbUdBQW1HO2dCQUMxRyxhQUFhLEVBQUUseUVBQXlFO2dCQUN4RixhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELDZCQUE2QixFQUFFO2dCQUM3QixLQUFLLEVBQUUscUdBQXFHO2dCQUM1RyxhQUFhLEVBQUUsMkdBQTJHO2dCQUMxSCxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7b0JBQ0QsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtpQkFDRjthQUNGO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSw2RkFBNkY7Z0JBQ3BHLGFBQWEsRUFBRSxzREFBc0Q7Z0JBQ3JFLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsY0FBYyxFQUFFO3dCQUNkLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsNkJBQTZCLEVBQUU7Z0JBQzdCLEtBQUssRUFBRSxxR0FBcUc7Z0JBQzVHLGFBQWEsRUFBRSwrREFBK0Q7Z0JBQzlFLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsY0FBYyxFQUFFO3dCQUNkLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSx3RkFBd0Y7Z0JBQy9GLGFBQWEsRUFBRSx1S0FBdUs7Z0JBQ3RMLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsMEJBQTBCLEVBQUU7Z0JBQzFCLEtBQUssRUFBRSxFQUFFO2dCQUNULGFBQWEsRUFBRSwwREFBMEQ7Z0JBQ3pFLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsbUNBQW1DLEVBQUU7Z0JBQ25DLEtBQUssRUFBRSxFQUFFO2dCQUNULGFBQWEsRUFBRSx1RUFBdUU7Z0JBQ3RGLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSx3RkFBd0Y7Z0JBQy9GLGFBQWEsRUFBRSxxTEFBcUw7Z0JBQ3BNLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLEtBQUssRUFBRSx1RkFBdUY7Z0JBQzlGLGFBQWEsRUFBRSxxQ0FBcUM7Z0JBQ3BELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsaUNBQWlDLEVBQUU7Z0JBQ2pDLEtBQUssRUFBRSx5R0FBeUc7Z0JBQ2hILGFBQWEsRUFBRSw2RUFBNkU7Z0JBQzVGLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSw0RkFBNEY7Z0JBQ25HLGFBQWEsRUFBRSw4RUFBOEU7Z0JBQzdGLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsU0FBUyxFQUFFO2dCQUNULEtBQUssRUFBRSxpRkFBaUY7Z0JBQ3hGLGFBQWEsRUFBRSwyRkFBMkY7Z0JBQzFHLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1oscUJBQXFCO29CQUNyQiw2QkFBNkI7b0JBQzdCLHNCQUFzQjtvQkFDdEIsaUNBQWlDO29CQUNqQyxpQkFBaUI7aUJBQ2xCO2FBQ0Y7WUFDRCxZQUFZLEVBQUU7Z0JBQ1osS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLGlEQUFpRDtnQkFDaEUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLDBGQUEwRjtnQkFDakcsYUFBYSxFQUFFLDBCQUEwQjtnQkFDekMsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLHVEQUF1RDtnQkFDdEUsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCx5QkFBeUIsRUFBRTtnQkFDekIsS0FBSyxFQUFFLGlHQUFpRztnQkFDeEcsYUFBYSxFQUFFLG1JQUFtSTtnQkFDbEosYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLDBGQUEwRjtnQkFDakcsYUFBYSxFQUFFLHFFQUFxRTtnQkFDcEYsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLGtGQUFrRjtnQkFDakcsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxZQUFZLEVBQUU7Z0JBQ1osS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLGtGQUFrRjtnQkFDakcsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLDRGQUE0RjtnQkFDbkcsYUFBYSxFQUFFLDhDQUE4QztnQkFDN0QsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxtQ0FBbUMsRUFBRTtnQkFDbkMsS0FBSyxFQUFFLEVBQUU7Z0JBQ1QsYUFBYSxFQUFFLHVEQUF1RDtnQkFDdEUsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxTQUFTLEVBQUU7Z0JBQ1QsS0FBSyxFQUFFLGlGQUFpRjtnQkFDeEYsYUFBYSxFQUFFLDZEQUE2RDtnQkFDNUUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixxQkFBcUI7b0JBQ3JCLDZCQUE2QjtvQkFDN0Isc0JBQXNCO29CQUN0QixpQ0FBaUM7b0JBQ2pDLHVCQUF1QjtpQkFDeEI7YUFDRjtZQUNELE9BQU8sRUFBRTtnQkFDUCxLQUFLLEVBQUUsK0VBQStFO2dCQUN0RixhQUFhLEVBQUUsd0dBQXdHO2dCQUN2SCxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7b0JBQ0QsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1oscUJBQXFCO29CQUNyQixzQkFBc0I7b0JBQ3RCLGlDQUFpQztvQkFDakMsdUJBQXVCO29CQUN2QixpQkFBaUI7aUJBQ2xCO2FBQ0Y7WUFDRCx3QkFBd0IsRUFBRTtnQkFDeEIsS0FBSyxFQUFFLGdHQUFnRztnQkFDdkcsYUFBYSxFQUFFLDZDQUE2QztnQkFDNUQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO29CQUNELE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELDJCQUEyQixFQUFFO2dCQUMzQixLQUFLLEVBQUUsbUdBQW1HO2dCQUMxRyxhQUFhLEVBQUUscUNBQXFDO2dCQUNwRCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELE1BQU0sRUFBRTtnQkFDTixLQUFLLEVBQUUsOEVBQThFO2dCQUNyRixhQUFhLEVBQUUsdUdBQXVHO2dCQUN0SCxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7b0JBQ0QsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1oscUJBQXFCO29CQUNyQixpQ0FBaUM7b0JBQ2pDLHVCQUF1QjtvQkFDdkIsaUJBQWlCO2lCQUNsQjthQUNGO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLEtBQUssRUFBRSxxRkFBcUY7Z0JBQzVGLGFBQWEsRUFBRSwwREFBMEQ7Z0JBQ3pFLGFBQWEsRUFBRSxTQUFTO2dCQUN4QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLEtBQUssRUFBRSx1RkFBdUY7Z0JBQzlGLGFBQWEsRUFBRSxtRUFBbUU7Z0JBQ2xGLGFBQWEsRUFBRSxTQUFTO2dCQUN4QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QseUJBQXlCLEVBQUU7Z0JBQ3pCLEtBQUssRUFBRSxpR0FBaUc7Z0JBQ3hHLGFBQWEsRUFBRSx3Q0FBd0M7Z0JBQ3ZELGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsMkJBQTJCLEVBQUU7Z0JBQzNCLEtBQUssRUFBRSxtR0FBbUc7Z0JBQzFHLGFBQWEsRUFBRSw0RkFBNEY7Z0JBQzNHLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtvQkFDRCxPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLDJGQUEyRjtnQkFDbEcsYUFBYSxFQUFFLDBFQUEwRTtnQkFDekYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixjQUFjLEVBQUU7d0JBQ2QsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO29CQUNELE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELDJCQUEyQixFQUFFO2dCQUMzQixLQUFLLEVBQUUsbUdBQW1HO2dCQUMxRyxhQUFhLEVBQUUsbUVBQW1FO2dCQUNsRixhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLGNBQWMsRUFBRTt3QkFDZCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7b0JBQ0QsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsWUFBWSxFQUFFO2dCQUNaLEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLGFBQWEsRUFBRSxtR0FBbUc7Z0JBQ2xILGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1oscUJBQXFCO29CQUNyQiw2QkFBNkI7b0JBQzdCLHNCQUFzQjtvQkFDdEIsaUNBQWlDO29CQUNqQyx1QkFBdUI7aUJBQ3hCO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLHdIQUF3SDtnQkFDdkksYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCwrQkFBK0IsRUFBRTtnQkFDL0IsS0FBSyxFQUFFLHVHQUF1RztnQkFDOUcsYUFBYSxFQUFFLHFEQUFxRDtnQkFDcEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLDBGQUEwRjtnQkFDakcsYUFBYSxFQUFFLGlEQUFpRDtnQkFDaEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7U0FDRixDQUFDO1FBQ1Esa0JBQWEsR0FBa0I7WUFDdkMsT0FBTyxFQUFFO2dCQUNQLE1BQU0sRUFBRSxPQUFPO2dCQUNmLEtBQUssRUFBRSxvS0FBb0s7Z0JBQzNLLEtBQUssRUFBRSxzRkFBc0Y7Z0JBQzdGLGVBQWUsRUFBRSxFQUFFO2FBQ3BCO1lBQ0QsUUFBUSxFQUFFO2dCQUNSLE1BQU0sRUFBRSxRQUFRO2dCQUNoQixLQUFLLEVBQUUsaUtBQWlLO2dCQUN4SyxLQUFLLEVBQUUseUZBQXlGO2dCQUNoRyxlQUFlLEVBQUUsRUFBRTthQUNwQjtZQUNELE9BQU8sRUFBRTtnQkFDUCxNQUFNLEVBQUUsT0FBTztnQkFDZixLQUFLLEVBQUUsK0tBQStLO2dCQUN0TCxLQUFLLEVBQUUsbUVBQW1FO2dCQUMxRSxlQUFlLEVBQUUsRUFBRTthQUNwQjtZQUNELFFBQVEsRUFBRTtnQkFDUixNQUFNLEVBQUUsUUFBUTtnQkFDaEIsS0FBSyxFQUFFLGdHQUFnRztnQkFDdkcsS0FBSyxFQUFFLHdGQUF3RjtnQkFDL0YsZUFBZSxFQUFFLEVBQUU7YUFDcEI7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsTUFBTSxFQUFFLGNBQWM7Z0JBQ3RCLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLEtBQUssRUFBRSx1RUFBdUU7Z0JBQzlFLGVBQWUsRUFBRSxFQUFFO2FBQ3BCO1NBQ0YsQ0FBQztJQVNGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxZQUFZO1FBQ2pCLElBQUksQ0FBQyxHQUFHLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixJQUFJLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxHQUFHLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXO1FBQ2hCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxVQUFVO1FBQ2YsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLElBQUksQ0FBQyxHQUFHLENBQUMsb0NBQW9DLENBQUMsQ0FBQztRQUMvQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO1FBQ2pELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMkJBQTJCO1FBQ2hDLElBQUksQ0FBQyxHQUFHLENBQUMsc0NBQXNDLENBQUMsQ0FBQztRQUNqRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksd0JBQXdCO1FBQzdCLElBQUksQ0FBQyxHQUFHLENBQUMsbUNBQW1DLENBQUMsQ0FBQztRQUM5QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksaUNBQWlDO1FBQ3RDLElBQUksQ0FBQyxHQUFHLENBQUMsNENBQTRDLENBQUMsQ0FBQztRQUN2RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwrQkFBK0I7UUFDcEMsSUFBSSxDQUFDLEdBQUcsQ0FBQywwQ0FBMEMsQ0FBQyxDQUFDO1FBQ3JELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLE9BQU87UUFDWixJQUFJLENBQUMsR0FBRyxDQUFDLGtCQUFrQixDQUFDLENBQUM7UUFDN0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksVUFBVTtRQUNmLElBQUksQ0FBQyxHQUFHLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsR0FBRyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxHQUFHLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXO1FBQ2hCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxVQUFVO1FBQ2YsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLGlDQUFpQztRQUN0QyxJQUFJLENBQUMsR0FBRyxDQUFDLDRDQUE0QyxDQUFDLENBQUM7UUFDdkQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLE9BQU87UUFDWixJQUFJLENBQUMsR0FBRyxDQUFDLGtCQUFrQixDQUFDLENBQUM7UUFDN0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLEtBQUs7UUFDVixJQUFJLENBQUMsR0FBRyxDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFDM0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLElBQUksQ0FBQyxHQUFHLENBQUMsaUNBQWlDLENBQUMsQ0FBQztRQUM1QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO1FBQy9DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLElBQUk7UUFDVCxJQUFJLENBQUMsR0FBRyxDQUFDLGVBQWUsQ0FBQyxDQUFDO1FBQzFCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsR0FBRyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLElBQUksQ0FBQyxHQUFHLENBQUMsb0NBQW9DLENBQUMsQ0FBQztRQUMvQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsR0FBRyxDQUFDLG9DQUFvQyxDQUFDLENBQUM7UUFDL0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLFVBQVU7UUFDZixJQUFJLENBQUMsR0FBRyxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNkJBQTZCO1FBQ2xDLElBQUksQ0FBQyxHQUFHLENBQUMsd0NBQXdDLENBQUMsQ0FBQztRQUNuRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxPQUFPLENBQUMsU0FBaUIsRUFBRSxTQUFpQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3hHLElBQUksR0FBRyxHQUFHLHNGQUFzRixDQUFDO1FBQ2pHLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUM3QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDN0MsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQzlDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksUUFBUSxDQUFDLFNBQWlCLEVBQUUsV0FBbUIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUMzRyxJQUFJLEdBQUcsR0FBRyx5RkFBeUYsQ0FBQztRQUNwRyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDN0MsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDakQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQzlDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxPQUFPLENBQUMsU0FBaUIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNyRixJQUFJLEdBQUcsR0FBRyxtRUFBbUUsQ0FBQztRQUM5RSxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDN0MsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQzlDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksUUFBUSxDQUFDLFNBQWlCLEVBQUUsVUFBa0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUMxRyxJQUFJLEdBQUcsR0FBRyx3RkFBd0YsQ0FBQztRQUNuRyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDN0MsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsZUFBZSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQy9DLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSSxhQUFhLENBQUMsZUFBdUIsRUFBRSxPQUFnQixFQUFFLFNBQWtCO1FBQ2hGLElBQUksR0FBRyxHQUFHLHVFQUF1RSxDQUFDO1FBQ2xGLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLG9CQUFvQixFQUFFLGVBQWUsQ0FBQyxDQUFDO1FBQ3pELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FrQkc7SUFDSSxZQUFZLENBQUMsS0FBd0IsRUFBRSxRQUFpQjtRQUM3RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUN6RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLG9CQUFvQixDQUFDLEtBQXdCLEVBQUUsUUFBaUI7UUFDckUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7OztPQWlCRztJQUNJLGFBQWEsQ0FBQyxLQUF3QixFQUFFLFFBQWlCO1FBQzlELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzFFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Ba0JHO0lBQ0ksd0JBQXdCLENBQUMsS0FBd0IsRUFBRSxRQUFpQjtRQUN6RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUNBQWlDLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNyRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0ksY0FBYyxDQUFDLEtBQXdCLEVBQUUsUUFBaUI7UUFDL0QsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDM0UsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxRQUFRLENBQUMsS0FBd0IsRUFBRSxRQUFpQjtRQUN6RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNyRSxDQUFDO0NBQ0Y7QUFyNENELDRCQXE0Q0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY3Rpb25zLCBQb2xpY3lTdGF0ZW1lbnQsIFJlc291cmNlVHlwZXMgfSBmcm9tIFwiLi4vc2hhcmVkXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtkeW5hbW9kYl0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2xpc3RfYW1hem9uZHluYW1vZGIuaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBEeW5hbW9kYiBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ2R5bmFtb2RiJztcbiAgcHJvdGVjdGVkIGFjdGlvbkxpc3Q6IEFjdGlvbnMgPSB7XG4gICAgXCJCYXRjaEdldEl0ZW1cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQmF0Y2hHZXRJdGVtLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIHRoZSBhdHRyaWJ1dGVzIG9mIG9uZSBvciBtb3JlIGl0ZW1zIGZyb20gb25lIG9yIG1vcmUgdGFibGVzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJ0YWJsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImR5bmFtb2RiOkF0dHJpYnV0ZXNcIixcbiAgICAgICAgXCJkeW5hbW9kYjpMZWFkaW5nS2V5c1wiLFxuICAgICAgICBcImR5bmFtb2RiOlJldHVybkNvbnN1bWVkQ2FwYWNpdHlcIixcbiAgICAgICAgXCJkeW5hbW9kYjpTZWxlY3RcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJCYXRjaFdyaXRlSXRlbVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9CYXRjaFdyaXRlSXRlbS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUHV0cyBvciBkZWxldGVzIG11bHRpcGxlIGl0ZW1zIGluIG9uZSBvciBtb3JlIHRhYmxlc1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInRhYmxlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiZHluYW1vZGI6QXR0cmlidXRlc1wiLFxuICAgICAgICBcImR5bmFtb2RiOkxlYWRpbmdLZXlzXCIsXG4gICAgICAgIFwiZHluYW1vZGI6UmV0dXJuQ29uc3VtZWRDYXBhY2l0eVwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkNvbmRpdGlvbkNoZWNrSXRlbVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Db25kaXRpb25DaGVja0l0ZW0uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlRoZSBDb25kaXRpb25DaGVja0l0ZW0gb3BlcmF0aW9uIGNoZWNrcyB0aGUgZXhpc3RlbmNlIG9mIGEgc2V0IG9mIGF0dHJpYnV0ZXMgZm9yIHRoZSBpdGVtIHdpdGggdGhlIGdpdmVuIHByaW1hcnkga2V5XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJ0YWJsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImR5bmFtb2RiOkF0dHJpYnV0ZXNcIixcbiAgICAgICAgXCJkeW5hbW9kYjpMZWFkaW5nS2V5c1wiLFxuICAgICAgICBcImR5bmFtb2RiOlJldHVybkNvbnN1bWVkQ2FwYWNpdHlcIixcbiAgICAgICAgXCJkeW5hbW9kYjpSZXR1cm5WYWx1ZXNcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJDcmVhdGVCYWNrdXBcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlQmFja3VwLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIGEgYmFja3VwIGZvciBhbiBleGlzdGluZyB0YWJsZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInRhYmxlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJDcmVhdGVHbG9iYWxUYWJsZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVHbG9iYWxUYWJsZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRW5hYmxlcyB0aGUgdXNlciB0byBjcmVhdGUgYSBnbG9iYWwgdGFibGUgZnJvbSBhbiBleGlzdGluZyB0YWJsZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImdsb2JhbC10YWJsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH0sXG4gICAgICAgIFwidGFibGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkNyZWF0ZVRhYmxlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZVRhYmxlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJUaGUgQ3JlYXRlVGFibGUgb3BlcmF0aW9uIGFkZHMgYSBuZXcgdGFibGUgdG8geW91ciBhY2NvdW50XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwidGFibGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkNyZWF0ZVRhYmxlUmVwbGljYVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL1YyZ3RfSUFNLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJBZGRzIGEgbmV3IHJlcGxpY2EgdGFibGVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJ0YWJsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlQmFja3VwXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUJhY2t1cC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVsZXRlcyBhbiBleGlzdGluZyBiYWNrdXAgb2YgYSB0YWJsZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJhY2t1cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlSXRlbVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVJdGVtLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZWxldGVzIGEgc2luZ2xlIGl0ZW0gaW4gYSB0YWJsZSBieSBwcmltYXJ5IGtleVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInRhYmxlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiZHluYW1vZGI6QXR0cmlidXRlc1wiLFxuICAgICAgICBcImR5bmFtb2RiOkVuY2xvc2luZ09wZXJhdGlvblwiLFxuICAgICAgICBcImR5bmFtb2RiOkxlYWRpbmdLZXlzXCIsXG4gICAgICAgIFwiZHluYW1vZGI6UmV0dXJuQ29uc3VtZWRDYXBhY2l0eVwiLFxuICAgICAgICBcImR5bmFtb2RiOlJldHVyblZhbHVlc1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkRlbGV0ZVRhYmxlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVRhYmxlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJUaGUgRGVsZXRlVGFibGUgb3BlcmF0aW9uIGRlbGV0ZXMgYSB0YWJsZSBhbmQgYWxsIG9mIGl0cyBpdGVtc1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInRhYmxlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVUYWJsZVJlcGxpY2FcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9WMmd0X0lBTS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVsZXRlcyBhIHJlcGxpY2EgdGFibGUgYW5kIGFsbCBvZiBpdHMgaXRlbXNcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJ0YWJsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVCYWNrdXBcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVCYWNrdXAuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlc2NyaWJlcyBhbiBleGlzdGluZyBiYWNrdXAgb2YgYSB0YWJsZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiYmFja3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXNjcmliZUNvbnRpbnVvdXNCYWNrdXBzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlQ29udGludW91c0JhY2t1cHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNoZWNrcyB0aGUgc3RhdHVzIG9mIHRoZSBiYWNrdXAgcmVzdG9yZSBzZXR0aW5ncyBvbiB0aGUgc3BlY2lmaWVkIHRhYmxlXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJ0YWJsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVDb250cmlidXRvckluc2lnaHRzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlQ29udHJpYnV0b3JJbnNpZ2h0cy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVzY3JpYmVzIHRoZSBjb250cmlidXRvciBpbnNpZ2h0cyBzdGF0dXMgYW5kIHJlbGF0ZWQgZGV0YWlscyBmb3IgYSBnaXZlbiB0YWJsZSBvciBnbG9iYWwgc2Vjb25kYXJ5IGluZGV4XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJ0YWJsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH0sXG4gICAgICAgIFwiaW5kZXhcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXNjcmliZUdsb2JhbFRhYmxlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlR2xvYmFsVGFibGUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgaW5mb3JtYXRpb24gYWJvdXQgdGhlIHNwZWNpZmllZCBnbG9iYWwgdGFibGVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImdsb2JhbC10YWJsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVHbG9iYWxUYWJsZVNldHRpbmdzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlR2xvYmFsVGFibGVTZXR0aW5ncy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBzZXR0aW5ncyBpbmZvcm1hdGlvbiBhYm91dCB0aGUgc3BlY2lmaWVkIGdsb2JhbCB0YWJsZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZ2xvYmFsLXRhYmxlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXNjcmliZUxpbWl0c1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUxpbWl0cy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyB0aGUgY3VycmVudCBwcm92aXNpb25lZC1jYXBhY2l0eSBsaW1pdHMgZm9yIHlvdXIgQVdTIGFjY291bnQgaW4gYSByZWdpb24sIGJvdGggZm9yIHRoZSByZWdpb24gYXMgYSB3aG9sZSBhbmQgZm9yIGFueSBvbmUgRHluYW1vREIgdGFibGUgdGhhdCB5b3UgY3JlYXRlIHRoZXJlXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiXG4gICAgfSxcbiAgICBcIkRlc2NyaWJlUmVzZXJ2ZWRDYXBhY2l0eVwiOiB7XG4gICAgICBcInVybFwiOiBcIlwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlc2NyaWJlcyBvbmUgb3IgbW9yZSBvZiB0aGUgUmVzZXJ2ZWQgQ2FwYWNpdHkgcHVyY2hhc2VkXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiXG4gICAgfSxcbiAgICBcIkRlc2NyaWJlUmVzZXJ2ZWRDYXBhY2l0eU9mZmVyaW5nc1wiOiB7XG4gICAgICBcInVybFwiOiBcIlwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlc2NyaWJlcyBSZXNlcnZlZCBDYXBhY2l0eSBvZmZlcmluZ3MgdGhhdCBhcmUgYXZhaWxhYmxlIGZvciBwdXJjaGFzZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIlxuICAgIH0sXG4gICAgXCJEZXNjcmliZVN0cmVhbVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVN0cmVhbS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBpbmZvcm1hdGlvbiBhYm91dCBhIHN0cmVhbSwgaW5jbHVkaW5nIHRoZSBjdXJyZW50IHN0YXR1cyBvZiB0aGUgc3RyZWFtLCBpdHMgQW1hem9uIFJlc291cmNlIE5hbWUgKEFSTiksIHRoZSBjb21wb3NpdGlvbiBvZiBpdHMgc2hhcmRzLCBhbmQgaXRzIGNvcnJlc3BvbmRpbmcgRHluYW1vREIgdGFibGVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0cmVhbVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVUYWJsZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVRhYmxlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGluZm9ybWF0aW9uIGFib3V0IHRoZSB0YWJsZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwidGFibGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlVGFibGVSZXBsaWNhQXV0b1NjYWxpbmdcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVUYWJsZVJlcGxpY2FBdXRvU2NhbGluZy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVzY3JpYmVzIHRoZSBhdXRvIHNjYWxpbmcgc2V0dGluZ3MgYWNyb3NzIGFsbCByZXBsaWNhcyBvZiB0aGUgZ2xvYmFsIHRhYmxlXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJ0YWJsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVUaW1lVG9MaXZlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlVGltZVRvTGl2ZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR2l2ZXMgYSBkZXNjcmlwdGlvbiBvZiB0aGUgVGltZSB0byBMaXZlIChUVEwpIHN0YXR1cyBvbiB0aGUgc3BlY2lmaWVkIHRhYmxlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwidGFibGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldEl0ZW1cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0SXRlbS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVGhlIEdldEl0ZW0gb3BlcmF0aW9uIHJldHVybnMgYSBzZXQgb2YgYXR0cmlidXRlcyBmb3IgdGhlIGl0ZW0gd2l0aCB0aGUgZ2l2ZW4gcHJpbWFyeSBrZXlcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInRhYmxlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiZHluYW1vZGI6QXR0cmlidXRlc1wiLFxuICAgICAgICBcImR5bmFtb2RiOkVuY2xvc2luZ09wZXJhdGlvblwiLFxuICAgICAgICBcImR5bmFtb2RiOkxlYWRpbmdLZXlzXCIsXG4gICAgICAgIFwiZHluYW1vZGI6UmV0dXJuQ29uc3VtZWRDYXBhY2l0eVwiLFxuICAgICAgICBcImR5bmFtb2RiOlNlbGVjdFwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkdldFJlY29yZHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0UmVjb3Jkcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0cmlldmVzIHRoZSBzdHJlYW0gcmVjb3JkcyBmcm9tIGEgZ2l2ZW4gc2hhcmRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0cmVhbVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0U2hhcmRJdGVyYXRvclwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRTaGFyZEl0ZXJhdG9yLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGEgc2hhcmQgaXRlcmF0b3JcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0cmVhbVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiTGlzdEJhY2t1cHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdEJhY2t1cHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkxpc3QgYmFja3VwcyBhc3NvY2lhdGVkIHdpdGggdGhlIGFjY291bnQgYW5kIGVuZHBvaW50XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkxpc3RDb250cmlidXRvckluc2lnaHRzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RDb250cmlidXRvckluc2lnaHRzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJMaXN0cyB0aGUgQ29udHJpYnV0b3JJbnNpZ2h0c1N1bW1hcnkgZm9yIGFsbCB0YWJsZXMgYW5kIGdsb2JhbCBzZWNvbmRhcnkgaW5kZXhlcyBhc3NvY2lhdGVkIHdpdGggdGhlIGN1cnJlbnQgYWNjb3VudCBhbmQgZW5kcG9pbnRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiTGlzdEdsb2JhbFRhYmxlc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0R2xvYmFsVGFibGVzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJMaXN0cyBhbGwgZ2xvYmFsIHRhYmxlcyB0aGF0IGhhdmUgYSByZXBsaWNhIGluIHRoZSBzcGVjaWZpZWQgcmVnaW9uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkxpc3RTdHJlYW1zXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RTdHJlYW1zLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGFuIGFycmF5IG9mIHN0cmVhbSBBUk5zIGFzc29jaWF0ZWQgd2l0aCB0aGUgY3VycmVudCBhY2NvdW50IGFuZCBlbmRwb2ludFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIlxuICAgIH0sXG4gICAgXCJMaXN0VGFibGVzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RUYWJsZXMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgYW4gYXJyYXkgb2YgdGFibGUgbmFtZXMgYXNzb2NpYXRlZCB3aXRoIHRoZSBjdXJyZW50IGFjY291bnQgYW5kIGVuZHBvaW50XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkxpc3RUYWdzT2ZSZXNvdXJjZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VGFnc09mUmVzb3VyY2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkxpc3QgYWxsIHRhZ3Mgb24gYW4gQW1hem9uIER5bmFtb0RCIHJlc291cmNlXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJ0YWJsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiUHVyY2hhc2VSZXNlcnZlZENhcGFjaXR5T2ZmZXJpbmdzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUHVyY2hhc2VzIFJlc2VydmVkIENhcGFjaXR5IGZvciB1c2Ugd2l0aCB5b3VyIGFjY291bnRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiXG4gICAgfSxcbiAgICBcIlB1dEl0ZW1cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUHV0SXRlbS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhIG5ldyBpdGVtLCBvciByZXBsYWNlcyBhbiBvbGQgaXRlbSB3aXRoIGEgbmV3IGl0ZW1cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJ0YWJsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImR5bmFtb2RiOkF0dHJpYnV0ZXNcIixcbiAgICAgICAgXCJkeW5hbW9kYjpFbmNsb3NpbmdPcGVyYXRpb25cIixcbiAgICAgICAgXCJkeW5hbW9kYjpMZWFkaW5nS2V5c1wiLFxuICAgICAgICBcImR5bmFtb2RiOlJldHVybkNvbnN1bWVkQ2FwYWNpdHlcIixcbiAgICAgICAgXCJkeW5hbW9kYjpSZXR1cm5WYWx1ZXNcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJRdWVyeVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9RdWVyeS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVXNlcyB0aGUgcHJpbWFyeSBrZXkgb2YgYSB0YWJsZSBvciBhIHNlY29uZGFyeSBpbmRleCB0byBkaXJlY3RseSBhY2Nlc3MgaXRlbXMgZnJvbSB0aGF0IHRhYmxlIG9yIGluZGV4XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJ0YWJsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH0sXG4gICAgICAgIFwiaW5kZXhcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiZHluYW1vZGI6QXR0cmlidXRlc1wiLFxuICAgICAgICBcImR5bmFtb2RiOkxlYWRpbmdLZXlzXCIsXG4gICAgICAgIFwiZHluYW1vZGI6UmV0dXJuQ29uc3VtZWRDYXBhY2l0eVwiLFxuICAgICAgICBcImR5bmFtb2RiOlJldHVyblZhbHVlc1wiLFxuICAgICAgICBcImR5bmFtb2RiOlNlbGVjdFwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlJlc3RvcmVUYWJsZUZyb21CYWNrdXBcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVzdG9yZVRhYmxlRnJvbUJhY2t1cC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhIG5ldyB0YWJsZSBmcm9tIGFuIGV4aXN0aW5nIGJhY2t1cFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJhY2t1cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH0sXG4gICAgICAgIFwidGFibGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlJlc3RvcmVUYWJsZVRvUG9pbnRJblRpbWVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVzdG9yZVRhYmxlVG9Qb2ludEluVGltZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmVzdG9yZXMgYSB0YWJsZSB0byBhIHBvaW50IGluIHRpbWVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJ0YWJsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiU2NhblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TY2FuLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIG9uZSBvciBtb3JlIGl0ZW1zIGFuZCBpdGVtIGF0dHJpYnV0ZXMgYnkgYWNjZXNzaW5nIGV2ZXJ5IGl0ZW0gaW4gYSB0YWJsZSBvciBhIHNlY29uZGFyeSBpbmRleFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwidGFibGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9LFxuICAgICAgICBcImluZGV4XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImR5bmFtb2RiOkF0dHJpYnV0ZXNcIixcbiAgICAgICAgXCJkeW5hbW9kYjpSZXR1cm5Db25zdW1lZENhcGFjaXR5XCIsXG4gICAgICAgIFwiZHluYW1vZGI6UmV0dXJuVmFsdWVzXCIsXG4gICAgICAgIFwiZHluYW1vZGI6U2VsZWN0XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiVGFnUmVzb3VyY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVGFnUmVzb3VyY2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkFzc29jaWF0ZSBhIHNldCBvZiB0YWdzIHdpdGggYW4gQW1hem9uIER5bmFtb0RCIHJlc291cmNlXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiVGFnZ2luZ1wiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJ0YWJsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVW50YWdSZXNvdXJjZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VbnRhZ1Jlc291cmNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZW1vdmVzIHRoZSBhc3NvY2lhdGlvbiBvZiB0YWdzIGZyb20gYW4gQW1hem9uIER5bmFtb0RCIHJlc291cmNlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlRhZ2dpbmdcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwidGFibGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVwZGF0ZUNvbnRpbnVvdXNCYWNrdXBzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUNvbnRpbnVvdXNCYWNrdXBzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJFbmFibGVzIG9yIGRpc2FibGVzIGNvbnRpbnVvdXMgYmFja3Vwc1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInRhYmxlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVDb250cmlidXRvckluc2lnaHRzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUNvbnRyaWJ1dG9ySW5zaWdodHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlVwZGF0ZXMgdGhlIHN0YXR1cyBmb3IgY29udHJpYnV0b3IgaW5zaWdodHMgZm9yIGEgc3BlY2lmaWMgdGFibGUgb3IgZ2xvYmFsIHNlY29uZGFyeSBpbmRleFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInRhYmxlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfSxcbiAgICAgICAgXCJpbmRleFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVwZGF0ZUdsb2JhbFRhYmxlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUdsb2JhbFRhYmxlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJFbmFibGVzIHRoZSB1c2VyIHRvIGFkZCBvciByZW1vdmUgcmVwbGljYXMgaW4gdGhlIHNwZWNpZmllZCBnbG9iYWwgdGFibGVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJnbG9iYWwtdGFibGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9LFxuICAgICAgICBcInRhYmxlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVHbG9iYWxUYWJsZVNldHRpbmdzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUdsb2JhbFRhYmxlU2V0dGluZ3MuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkVuYWJsZXMgdGhlIHVzZXIgdG8gdXBkYXRlIHNldHRpbmdzIG9mIHRoZSBzcGVjaWZpZWQgZ2xvYmFsIHRhYmxlXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZ2xvYmFsLXRhYmxlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfSxcbiAgICAgICAgXCJ0YWJsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVXBkYXRlSXRlbVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVJdGVtLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJFZGl0cyBhbiBleGlzdGluZyBpdGVtJ3MgYXR0cmlidXRlcywgb3IgYWRkcyBhIG5ldyBpdGVtIHRvIHRoZSB0YWJsZSBpZiBpdCBkb2VzIG5vdCBhbHJlYWR5IGV4aXN0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwidGFibGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJkeW5hbW9kYjpBdHRyaWJ1dGVzXCIsXG4gICAgICAgIFwiZHluYW1vZGI6RW5jbG9zaW5nT3BlcmF0aW9uXCIsXG4gICAgICAgIFwiZHluYW1vZGI6TGVhZGluZ0tleXNcIixcbiAgICAgICAgXCJkeW5hbW9kYjpSZXR1cm5Db25zdW1lZENhcGFjaXR5XCIsXG4gICAgICAgIFwiZHluYW1vZGI6UmV0dXJuVmFsdWVzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiVXBkYXRlVGFibGVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlVGFibGUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIk1vZGlmaWVzIHRoZSBwcm92aXNpb25lZCB0aHJvdWdocHV0IHNldHRpbmdzLCBnbG9iYWwgc2Vjb25kYXJ5IGluZGV4ZXMsIG9yIER5bmFtb0RCIFN0cmVhbXMgc2V0dGluZ3MgZm9yIGEgZ2l2ZW4gdGFibGVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJ0YWJsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVXBkYXRlVGFibGVSZXBsaWNhQXV0b1NjYWxpbmdcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlVGFibGVSZXBsaWNhQXV0b1NjYWxpbmcuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlVwZGF0ZXMgYXV0byBzY2FsaW5nIHNldHRpbmdzIG9uIHlvdXIgcmVwbGljYSB0YWJsZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInRhYmxlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVUaW1lVG9MaXZlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZVRpbWVUb0xpdmUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkVuYWJsZXMgb3IgZGlzYWJsZXMgVFRMIGZvciB0aGUgc3BlY2lmaWVkIHRhYmxlXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwidGFibGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICB9O1xuICBwcm90ZWN0ZWQgcmVzb3VyY2VUeXBlczogUmVzb3VyY2VUeXBlcyA9IHtcbiAgICBcImluZGV4XCI6IHtcbiAgICAgIFwibmFtZVwiOiBcImluZGV4XCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvSG93SXRXb3Jrcy5Db3JlQ29tcG9uZW50cy5odG1sSG93SXRXb3Jrcy5Db3JlQ29tcG9uZW50cy5odG1sI0hvd0l0V29ya3MuQ29yZUNvbXBvbmVudHMuUHJpbWFyeUtleVwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmR5bmFtb2RiOiR7UmVnaW9ufToke0FjY291bnR9OnRhYmxlLyR7VGFibGVOYW1lfS9pbmRleC8ke0luZGV4TmFtZX1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXVxuICAgIH0sXG4gICAgXCJzdHJlYW1cIjoge1xuICAgICAgXCJuYW1lXCI6IFwic3RyZWFtXCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvSG93SXRXb3Jrcy5Db3JlQ29tcG9uZW50cy5odG1sSG93SXRXb3Jrcy5Db3JlQ29tcG9uZW50cy5odG1sI0hvd0l0V29ya3MuQ29yZUNvbXBvbmVudHMuU3RyZWFtc1wiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmR5bmFtb2RiOiR7UmVnaW9ufToke0FjY291bnR9OnRhYmxlLyR7VGFibGVOYW1lfS9zdHJlYW0vJHtTdHJlYW1MYWJlbH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXVxuICAgIH0sXG4gICAgXCJ0YWJsZVwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJ0YWJsZVwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL0hvd0l0V29ya3MuQ29yZUNvbXBvbmVudHMuaHRtbEhvd0l0V29ya3MuQ29yZUNvbXBvbmVudHMuaHRtbCNIb3dJdFdvcmtzLkNvcmVDb21wb25lbnRzLlRhYmxlc0l0ZW1zQXR0cmlidXRlc1wiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmR5bmFtb2RiOiR7UmVnaW9ufToke0FjY291bnR9OnRhYmxlLyR7VGFibGVOYW1lfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtdXG4gICAgfSxcbiAgICBcImJhY2t1cFwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJiYWNrdXBcIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9iYWNrdXByZXN0b3JlX0hvd0l0V29ya3MuaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmR5bmFtb2RiOiR7UmVnaW9ufToke0FjY291bnR9OnRhYmxlLyR7VGFibGVOYW1lfS9iYWNrdXAvJHtCYWNrdXBOYW1lfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtdXG4gICAgfSxcbiAgICBcImdsb2JhbC10YWJsZVwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJnbG9iYWwtdGFibGVcIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9nbG9iYWx0YWJsZXNfSG93SXRXb3Jrcy5odG1sXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06ZHluYW1vZGI6OiR7QWNjb3VudH06Z2xvYmFsLXRhYmxlLyR7R2xvYmFsVGFibGVOYW1lfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtdXG4gICAgfVxuICB9O1xuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2R5bmFtb2RiXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b25keW5hbW9kYi5odG1sKS5cbiAgICpcbiAgICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gICAqL1xuICBjb25zdHJ1Y3RvciAoc2lkPzogc3RyaW5nKSB7XG4gICAgc3VwZXIoc2lkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBhdHRyaWJ1dGVzIG9mIG9uZSBvciBtb3JlIGl0ZW1zIGZyb20gb25lIG9yIG1vcmUgdGFibGVzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF0dHJpYnV0ZXMoKVxuICAgKiAtIC5pZkxlYWRpbmdLZXlzKClcbiAgICogLSAuaWZSZXR1cm5Db25zdW1lZENhcGFjaXR5KClcbiAgICogLSAuaWZTZWxlY3QoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQmF0Y2hHZXRJdGVtLmh0bWxcbiAgICovXG4gIHB1YmxpYyBiYXRjaEdldEl0ZW0oKSB7XG4gICAgdGhpcy5hZGQoJ2R5bmFtb2RiOkJhdGNoR2V0SXRlbScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFB1dHMgb3IgZGVsZXRlcyBtdWx0aXBsZSBpdGVtcyBpbiBvbmUgb3IgbW9yZSB0YWJsZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF0dHJpYnV0ZXMoKVxuICAgKiAtIC5pZkxlYWRpbmdLZXlzKClcbiAgICogLSAuaWZSZXR1cm5Db25zdW1lZENhcGFjaXR5KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0JhdGNoV3JpdGVJdGVtLmh0bWxcbiAgICovXG4gIHB1YmxpYyBiYXRjaFdyaXRlSXRlbSgpIHtcbiAgICB0aGlzLmFkZCgnZHluYW1vZGI6QmF0Y2hXcml0ZUl0ZW0nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgQ29uZGl0aW9uQ2hlY2tJdGVtIG9wZXJhdGlvbiBjaGVja3MgdGhlIGV4aXN0ZW5jZSBvZiBhIHNldCBvZiBhdHRyaWJ1dGVzIGZvciB0aGUgaXRlbSB3aXRoIHRoZSBnaXZlbiBwcmltYXJ5IGtleVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdHRyaWJ1dGVzKClcbiAgICogLSAuaWZMZWFkaW5nS2V5cygpXG4gICAqIC0gLmlmUmV0dXJuQ29uc3VtZWRDYXBhY2l0eSgpXG4gICAqIC0gLmlmUmV0dXJuVmFsdWVzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NvbmRpdGlvbkNoZWNrSXRlbS5odG1sXG4gICAqL1xuICBwdWJsaWMgY29uZGl0aW9uQ2hlY2tJdGVtKCkge1xuICAgIHRoaXMuYWRkKCdkeW5hbW9kYjpDb25kaXRpb25DaGVja0l0ZW0nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgYmFja3VwIGZvciBhbiBleGlzdGluZyB0YWJsZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVCYWNrdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIGNyZWF0ZUJhY2t1cCgpIHtcbiAgICB0aGlzLmFkZCgnZHluYW1vZGI6Q3JlYXRlQmFja3VwJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRW5hYmxlcyB0aGUgdXNlciB0byBjcmVhdGUgYSBnbG9iYWwgdGFibGUgZnJvbSBhbiBleGlzdGluZyB0YWJsZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVHbG9iYWxUYWJsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgY3JlYXRlR2xvYmFsVGFibGUoKSB7XG4gICAgdGhpcy5hZGQoJ2R5bmFtb2RiOkNyZWF0ZUdsb2JhbFRhYmxlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVGhlIENyZWF0ZVRhYmxlIG9wZXJhdGlvbiBhZGRzIGEgbmV3IHRhYmxlIHRvIHlvdXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVUYWJsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgY3JlYXRlVGFibGUoKSB7XG4gICAgdGhpcy5hZGQoJ2R5bmFtb2RiOkNyZWF0ZVRhYmxlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIG5ldyByZXBsaWNhIHRhYmxlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvVjJndF9JQU0uaHRtbFxuICAgKi9cbiAgcHVibGljIGNyZWF0ZVRhYmxlUmVwbGljYSgpIHtcbiAgICB0aGlzLmFkZCgnZHluYW1vZGI6Q3JlYXRlVGFibGVSZXBsaWNhJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBhbiBleGlzdGluZyBiYWNrdXAgb2YgYSB0YWJsZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVCYWNrdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlbGV0ZUJhY2t1cCgpIHtcbiAgICB0aGlzLmFkZCgnZHluYW1vZGI6RGVsZXRlQmFja3VwJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBhIHNpbmdsZSBpdGVtIGluIGEgdGFibGUgYnkgcHJpbWFyeSBrZXlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF0dHJpYnV0ZXMoKVxuICAgKiAtIC5pZkVuY2xvc2luZ09wZXJhdGlvbigpXG4gICAqIC0gLmlmTGVhZGluZ0tleXMoKVxuICAgKiAtIC5pZlJldHVybkNvbnN1bWVkQ2FwYWNpdHkoKVxuICAgKiAtIC5pZlJldHVyblZhbHVlcygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVJdGVtLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVJdGVtKCkge1xuICAgIHRoaXMuYWRkKCdkeW5hbW9kYjpEZWxldGVJdGVtJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVGhlIERlbGV0ZVRhYmxlIG9wZXJhdGlvbiBkZWxldGVzIGEgdGFibGUgYW5kIGFsbCBvZiBpdHMgaXRlbXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlVGFibGUuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlbGV0ZVRhYmxlKCkge1xuICAgIHRoaXMuYWRkKCdkeW5hbW9kYjpEZWxldGVUYWJsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYSByZXBsaWNhIHRhYmxlIGFuZCBhbGwgb2YgaXRzIGl0ZW1zXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvVjJndF9JQU0uaHRtbFxuICAgKi9cbiAgcHVibGljIGRlbGV0ZVRhYmxlUmVwbGljYSgpIHtcbiAgICB0aGlzLmFkZCgnZHluYW1vZGI6RGVsZXRlVGFibGVSZXBsaWNhJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIGFuIGV4aXN0aW5nIGJhY2t1cCBvZiBhIHRhYmxlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVCYWNrdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlQmFja3VwKCkge1xuICAgIHRoaXMuYWRkKCdkeW5hbW9kYjpEZXNjcmliZUJhY2t1cCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrcyB0aGUgc3RhdHVzIG9mIHRoZSBiYWNrdXAgcmVzdG9yZSBzZXR0aW5ncyBvbiB0aGUgc3BlY2lmaWVkIHRhYmxlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVDb250aW51b3VzQmFja3Vwcy5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVDb250aW51b3VzQmFja3VwcygpIHtcbiAgICB0aGlzLmFkZCgnZHluYW1vZGI6RGVzY3JpYmVDb250aW51b3VzQmFja3VwcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlcyB0aGUgY29udHJpYnV0b3IgaW5zaWdodHMgc3RhdHVzIGFuZCByZWxhdGVkIGRldGFpbHMgZm9yIGEgZ2l2ZW4gdGFibGUgb3IgZ2xvYmFsIHNlY29uZGFyeSBpbmRleFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlQ29udHJpYnV0b3JJbnNpZ2h0cy5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVDb250cmlidXRvckluc2lnaHRzKCkge1xuICAgIHRoaXMuYWRkKCdkeW5hbW9kYjpEZXNjcmliZUNvbnRyaWJ1dG9ySW5zaWdodHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGluZm9ybWF0aW9uIGFib3V0IHRoZSBzcGVjaWZpZWQgZ2xvYmFsIHRhYmxlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVHbG9iYWxUYWJsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVHbG9iYWxUYWJsZSgpIHtcbiAgICB0aGlzLmFkZCgnZHluYW1vZGI6RGVzY3JpYmVHbG9iYWxUYWJsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgc2V0dGluZ3MgaW5mb3JtYXRpb24gYWJvdXQgdGhlIHNwZWNpZmllZCBnbG9iYWwgdGFibGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUdsb2JhbFRhYmxlU2V0dGluZ3MuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlR2xvYmFsVGFibGVTZXR0aW5ncygpIHtcbiAgICB0aGlzLmFkZCgnZHluYW1vZGI6RGVzY3JpYmVHbG9iYWxUYWJsZVNldHRpbmdzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgY3VycmVudCBwcm92aXNpb25lZC1jYXBhY2l0eSBsaW1pdHMgZm9yIHlvdXIgQVdTIGFjY291bnQgaW4gYSByZWdpb24sIGJvdGggZm9yIHRoZSByZWdpb24gYXMgYSB3aG9sZSBhbmQgZm9yIGFueSBvbmUgRHluYW1vREIgdGFibGUgdGhhdCB5b3UgY3JlYXRlIHRoZXJlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVMaW1pdHMuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlTGltaXRzKCkge1xuICAgIHRoaXMuYWRkKCdkeW5hbW9kYjpEZXNjcmliZUxpbWl0cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlcyBvbmUgb3IgbW9yZSBvZiB0aGUgUmVzZXJ2ZWQgQ2FwYWNpdHkgcHVyY2hhc2VkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlUmVzZXJ2ZWRDYXBhY2l0eSgpIHtcbiAgICB0aGlzLmFkZCgnZHluYW1vZGI6RGVzY3JpYmVSZXNlcnZlZENhcGFjaXR5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIFJlc2VydmVkIENhcGFjaXR5IG9mZmVyaW5ncyB0aGF0IGFyZSBhdmFpbGFibGUgZm9yIHB1cmNoYXNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlUmVzZXJ2ZWRDYXBhY2l0eU9mZmVyaW5ncygpIHtcbiAgICB0aGlzLmFkZCgnZHluYW1vZGI6RGVzY3JpYmVSZXNlcnZlZENhcGFjaXR5T2ZmZXJpbmdzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBpbmZvcm1hdGlvbiBhYm91dCBhIHN0cmVhbSwgaW5jbHVkaW5nIHRoZSBjdXJyZW50IHN0YXR1cyBvZiB0aGUgc3RyZWFtLCBpdHMgQW1hem9uIFJlc291cmNlIE5hbWUgKEFSTiksIHRoZSBjb21wb3NpdGlvbiBvZiBpdHMgc2hhcmRzLCBhbmQgaXRzIGNvcnJlc3BvbmRpbmcgRHluYW1vREIgdGFibGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVN0cmVhbS5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVTdHJlYW0oKSB7XG4gICAgdGhpcy5hZGQoJ2R5bmFtb2RiOkRlc2NyaWJlU3RyZWFtJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBpbmZvcm1hdGlvbiBhYm91dCB0aGUgdGFibGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVRhYmxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZXNjcmliZVRhYmxlKCkge1xuICAgIHRoaXMuYWRkKCdkeW5hbW9kYjpEZXNjcmliZVRhYmxlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIHRoZSBhdXRvIHNjYWxpbmcgc2V0dGluZ3MgYWNyb3NzIGFsbCByZXBsaWNhcyBvZiB0aGUgZ2xvYmFsIHRhYmxlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVUYWJsZVJlcGxpY2FBdXRvU2NhbGluZy5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVUYWJsZVJlcGxpY2FBdXRvU2NhbGluZygpIHtcbiAgICB0aGlzLmFkZCgnZHluYW1vZGI6RGVzY3JpYmVUYWJsZVJlcGxpY2FBdXRvU2NhbGluZycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdpdmVzIGEgZGVzY3JpcHRpb24gb2YgdGhlIFRpbWUgdG8gTGl2ZSAoVFRMKSBzdGF0dXMgb24gdGhlIHNwZWNpZmllZCB0YWJsZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVRpbWVUb0xpdmUuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlVGltZVRvTGl2ZSgpIHtcbiAgICB0aGlzLmFkZCgnZHluYW1vZGI6RGVzY3JpYmVUaW1lVG9MaXZlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVGhlIEdldEl0ZW0gb3BlcmF0aW9uIHJldHVybnMgYSBzZXQgb2YgYXR0cmlidXRlcyBmb3IgdGhlIGl0ZW0gd2l0aCB0aGUgZ2l2ZW4gcHJpbWFyeSBrZXlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXR0cmlidXRlcygpXG4gICAqIC0gLmlmRW5jbG9zaW5nT3BlcmF0aW9uKClcbiAgICogLSAuaWZMZWFkaW5nS2V5cygpXG4gICAqIC0gLmlmUmV0dXJuQ29uc3VtZWRDYXBhY2l0eSgpXG4gICAqIC0gLmlmU2VsZWN0KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldEl0ZW0uaHRtbFxuICAgKi9cbiAgcHVibGljIGdldEl0ZW0oKSB7XG4gICAgdGhpcy5hZGQoJ2R5bmFtb2RiOkdldEl0ZW0nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgdGhlIHN0cmVhbSByZWNvcmRzIGZyb20gYSBnaXZlbiBzaGFyZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldFJlY29yZHMuaHRtbFxuICAgKi9cbiAgcHVibGljIGdldFJlY29yZHMoKSB7XG4gICAgdGhpcy5hZGQoJ2R5bmFtb2RiOkdldFJlY29yZHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGEgc2hhcmQgaXRlcmF0b3JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRTaGFyZEl0ZXJhdG9yLmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRTaGFyZEl0ZXJhdG9yKCkge1xuICAgIHRoaXMuYWRkKCdkeW5hbW9kYjpHZXRTaGFyZEl0ZXJhdG9yJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdCBiYWNrdXBzIGFzc29jaWF0ZWQgd2l0aCB0aGUgYWNjb3VudCBhbmQgZW5kcG9pbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0QmFja3Vwcy5odG1sXG4gICAqL1xuICBwdWJsaWMgbGlzdEJhY2t1cHMoKSB7XG4gICAgdGhpcy5hZGQoJ2R5bmFtb2RiOkxpc3RCYWNrdXBzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdHMgdGhlIENvbnRyaWJ1dG9ySW5zaWdodHNTdW1tYXJ5IGZvciBhbGwgdGFibGVzIGFuZCBnbG9iYWwgc2Vjb25kYXJ5IGluZGV4ZXMgYXNzb2NpYXRlZCB3aXRoIHRoZSBjdXJyZW50IGFjY291bnQgYW5kIGVuZHBvaW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdENvbnRyaWJ1dG9ySW5zaWdodHMuaHRtbFxuICAgKi9cbiAgcHVibGljIGxpc3RDb250cmlidXRvckluc2lnaHRzKCkge1xuICAgIHRoaXMuYWRkKCdkeW5hbW9kYjpMaXN0Q29udHJpYnV0b3JJbnNpZ2h0cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3RzIGFsbCBnbG9iYWwgdGFibGVzIHRoYXQgaGF2ZSBhIHJlcGxpY2EgaW4gdGhlIHNwZWNpZmllZCByZWdpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0R2xvYmFsVGFibGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0R2xvYmFsVGFibGVzKCkge1xuICAgIHRoaXMuYWRkKCdkeW5hbW9kYjpMaXN0R2xvYmFsVGFibGVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhbiBhcnJheSBvZiBzdHJlYW0gQVJOcyBhc3NvY2lhdGVkIHdpdGggdGhlIGN1cnJlbnQgYWNjb3VudCBhbmQgZW5kcG9pbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0U3RyZWFtcy5odG1sXG4gICAqL1xuICBwdWJsaWMgbGlzdFN0cmVhbXMoKSB7XG4gICAgdGhpcy5hZGQoJ2R5bmFtb2RiOkxpc3RTdHJlYW1zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhbiBhcnJheSBvZiB0YWJsZSBuYW1lcyBhc3NvY2lhdGVkIHdpdGggdGhlIGN1cnJlbnQgYWNjb3VudCBhbmQgZW5kcG9pbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VGFibGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0VGFibGVzKCkge1xuICAgIHRoaXMuYWRkKCdkeW5hbW9kYjpMaXN0VGFibGVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdCBhbGwgdGFncyBvbiBhbiBBbWF6b24gRHluYW1vREIgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VGFnc09mUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIGxpc3RUYWdzT2ZSZXNvdXJjZSgpIHtcbiAgICB0aGlzLmFkZCgnZHluYW1vZGI6TGlzdFRhZ3NPZlJlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUHVyY2hhc2VzIFJlc2VydmVkIENhcGFjaXR5IGZvciB1c2Ugd2l0aCB5b3VyIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKi9cbiAgcHVibGljIHB1cmNoYXNlUmVzZXJ2ZWRDYXBhY2l0eU9mZmVyaW5ncygpIHtcbiAgICB0aGlzLmFkZCgnZHluYW1vZGI6UHVyY2hhc2VSZXNlcnZlZENhcGFjaXR5T2ZmZXJpbmdzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIG5ldyBpdGVtLCBvciByZXBsYWNlcyBhbiBvbGQgaXRlbSB3aXRoIGEgbmV3IGl0ZW1cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF0dHJpYnV0ZXMoKVxuICAgKiAtIC5pZkVuY2xvc2luZ09wZXJhdGlvbigpXG4gICAqIC0gLmlmTGVhZGluZ0tleXMoKVxuICAgKiAtIC5pZlJldHVybkNvbnN1bWVkQ2FwYWNpdHkoKVxuICAgKiAtIC5pZlJldHVyblZhbHVlcygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXRJdGVtLmh0bWxcbiAgICovXG4gIHB1YmxpYyBwdXRJdGVtKCkge1xuICAgIHRoaXMuYWRkKCdkeW5hbW9kYjpQdXRJdGVtJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXNlcyB0aGUgcHJpbWFyeSBrZXkgb2YgYSB0YWJsZSBvciBhIHNlY29uZGFyeSBpbmRleCB0byBkaXJlY3RseSBhY2Nlc3MgaXRlbXMgZnJvbSB0aGF0IHRhYmxlIG9yIGluZGV4XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF0dHJpYnV0ZXMoKVxuICAgKiAtIC5pZkxlYWRpbmdLZXlzKClcbiAgICogLSAuaWZSZXR1cm5Db25zdW1lZENhcGFjaXR5KClcbiAgICogLSAuaWZSZXR1cm5WYWx1ZXMoKVxuICAgKiAtIC5pZlNlbGVjdCgpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9RdWVyeS5odG1sXG4gICAqL1xuICBwdWJsaWMgcXVlcnkoKSB7XG4gICAgdGhpcy5hZGQoJ2R5bmFtb2RiOlF1ZXJ5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIG5ldyB0YWJsZSBmcm9tIGFuIGV4aXN0aW5nIGJhY2t1cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZXN0b3JlVGFibGVGcm9tQmFja3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyByZXN0b3JlVGFibGVGcm9tQmFja3VwKCkge1xuICAgIHRoaXMuYWRkKCdkeW5hbW9kYjpSZXN0b3JlVGFibGVGcm9tQmFja3VwJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVzdG9yZXMgYSB0YWJsZSB0byBhIHBvaW50IGluIHRpbWVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVzdG9yZVRhYmxlVG9Qb2ludEluVGltZS5odG1sXG4gICAqL1xuICBwdWJsaWMgcmVzdG9yZVRhYmxlVG9Qb2ludEluVGltZSgpIHtcbiAgICB0aGlzLmFkZCgnZHluYW1vZGI6UmVzdG9yZVRhYmxlVG9Qb2ludEluVGltZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgb25lIG9yIG1vcmUgaXRlbXMgYW5kIGl0ZW0gYXR0cmlidXRlcyBieSBhY2Nlc3NpbmcgZXZlcnkgaXRlbSBpbiBhIHRhYmxlIG9yIGEgc2Vjb25kYXJ5IGluZGV4XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF0dHJpYnV0ZXMoKVxuICAgKiAtIC5pZlJldHVybkNvbnN1bWVkQ2FwYWNpdHkoKVxuICAgKiAtIC5pZlJldHVyblZhbHVlcygpXG4gICAqIC0gLmlmU2VsZWN0KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1NjYW4uaHRtbFxuICAgKi9cbiAgcHVibGljIHNjYW4oKSB7XG4gICAgdGhpcy5hZGQoJ2R5bmFtb2RiOlNjYW4nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBc3NvY2lhdGUgYSBzZXQgb2YgdGFncyB3aXRoIGFuIEFtYXpvbiBEeW5hbW9EQiByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1RhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0YWdSZXNvdXJjZSgpIHtcbiAgICB0aGlzLmFkZCgnZHluYW1vZGI6VGFnUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZW1vdmVzIHRoZSBhc3NvY2lhdGlvbiBvZiB0YWdzIGZyb20gYW4gQW1hem9uIER5bmFtb0RCIHJlc291cmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VudGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHVudGFnUmVzb3VyY2UoKSB7XG4gICAgdGhpcy5hZGQoJ2R5bmFtb2RiOlVudGFnUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBFbmFibGVzIG9yIGRpc2FibGVzIGNvbnRpbnVvdXMgYmFja3Vwc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVDb250aW51b3VzQmFja3Vwcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdXBkYXRlQ29udGludW91c0JhY2t1cHMoKSB7XG4gICAgdGhpcy5hZGQoJ2R5bmFtb2RiOlVwZGF0ZUNvbnRpbnVvdXNCYWNrdXBzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyB0aGUgc3RhdHVzIGZvciBjb250cmlidXRvciBpbnNpZ2h0cyBmb3IgYSBzcGVjaWZpYyB0YWJsZSBvciBnbG9iYWwgc2Vjb25kYXJ5IGluZGV4XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUNvbnRyaWJ1dG9ySW5zaWdodHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHVwZGF0ZUNvbnRyaWJ1dG9ySW5zaWdodHMoKSB7XG4gICAgdGhpcy5hZGQoJ2R5bmFtb2RiOlVwZGF0ZUNvbnRyaWJ1dG9ySW5zaWdodHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBFbmFibGVzIHRoZSB1c2VyIHRvIGFkZCBvciByZW1vdmUgcmVwbGljYXMgaW4gdGhlIHNwZWNpZmllZCBnbG9iYWwgdGFibGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlR2xvYmFsVGFibGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHVwZGF0ZUdsb2JhbFRhYmxlKCkge1xuICAgIHRoaXMuYWRkKCdkeW5hbW9kYjpVcGRhdGVHbG9iYWxUYWJsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEVuYWJsZXMgdGhlIHVzZXIgdG8gdXBkYXRlIHNldHRpbmdzIG9mIHRoZSBzcGVjaWZpZWQgZ2xvYmFsIHRhYmxlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUdsb2JhbFRhYmxlU2V0dGluZ3MuaHRtbFxuICAgKi9cbiAgcHVibGljIHVwZGF0ZUdsb2JhbFRhYmxlU2V0dGluZ3MoKSB7XG4gICAgdGhpcy5hZGQoJ2R5bmFtb2RiOlVwZGF0ZUdsb2JhbFRhYmxlU2V0dGluZ3MnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBFZGl0cyBhbiBleGlzdGluZyBpdGVtJ3MgYXR0cmlidXRlcywgb3IgYWRkcyBhIG5ldyBpdGVtIHRvIHRoZSB0YWJsZSBpZiBpdCBkb2VzIG5vdCBhbHJlYWR5IGV4aXN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdHRyaWJ1dGVzKClcbiAgICogLSAuaWZFbmNsb3NpbmdPcGVyYXRpb24oKVxuICAgKiAtIC5pZkxlYWRpbmdLZXlzKClcbiAgICogLSAuaWZSZXR1cm5Db25zdW1lZENhcGFjaXR5KClcbiAgICogLSAuaWZSZXR1cm5WYWx1ZXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlSXRlbS5odG1sXG4gICAqL1xuICBwdWJsaWMgdXBkYXRlSXRlbSgpIHtcbiAgICB0aGlzLmFkZCgnZHluYW1vZGI6VXBkYXRlSXRlbScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIE1vZGlmaWVzIHRoZSBwcm92aXNpb25lZCB0aHJvdWdocHV0IHNldHRpbmdzLCBnbG9iYWwgc2Vjb25kYXJ5IGluZGV4ZXMsIG9yIER5bmFtb0RCIFN0cmVhbXMgc2V0dGluZ3MgZm9yIGEgZ2l2ZW4gdGFibGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlVGFibGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHVwZGF0ZVRhYmxlKCkge1xuICAgIHRoaXMuYWRkKCdkeW5hbW9kYjpVcGRhdGVUYWJsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgYXV0byBzY2FsaW5nIHNldHRpbmdzIG9uIHlvdXIgcmVwbGljYSB0YWJsZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVUYWJsZVJlcGxpY2FBdXRvU2NhbGluZy5odG1sXG4gICAqL1xuICBwdWJsaWMgdXBkYXRlVGFibGVSZXBsaWNhQXV0b1NjYWxpbmcoKSB7XG4gICAgdGhpcy5hZGQoJ2R5bmFtb2RiOlVwZGF0ZVRhYmxlUmVwbGljYUF1dG9TY2FsaW5nJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRW5hYmxlcyBvciBkaXNhYmxlcyBUVEwgZm9yIHRoZSBzcGVjaWZpZWQgdGFibGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlVGltZVRvTGl2ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdXBkYXRlVGltZVRvTGl2ZSgpIHtcbiAgICB0aGlzLmFkZCgnZHluYW1vZGI6VXBkYXRlVGltZVRvTGl2ZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGluZGV4IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9Ib3dJdFdvcmtzLkNvcmVDb21wb25lbnRzLmh0bWxIb3dJdFdvcmtzLkNvcmVDb21wb25lbnRzLmh0bWwjSG93SXRXb3Jrcy5Db3JlQ29tcG9uZW50cy5QcmltYXJ5S2V5XG4gICAqXG4gICAqIEBwYXJhbSB0YWJsZU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgdGFibGVOYW1lLlxuICAgKiBAcGFyYW0gaW5kZXhOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGluZGV4TmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICovXG4gIHB1YmxpYyBvbkluZGV4KHRhYmxlTmFtZTogc3RyaW5nLCBpbmRleE5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06ZHluYW1vZGI6JHtSZWdpb259OiR7QWNjb3VudH06dGFibGUvJHtUYWJsZU5hbWV9L2luZGV4LyR7SW5kZXhOYW1lfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7VGFibGVOYW1lfScsIHRhYmxlTmFtZSk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7SW5kZXhOYW1lfScsIGluZGV4TmFtZSk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgc3RyZWFtIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9Ib3dJdFdvcmtzLkNvcmVDb21wb25lbnRzLmh0bWxIb3dJdFdvcmtzLkNvcmVDb21wb25lbnRzLmh0bWwjSG93SXRXb3Jrcy5Db3JlQ29tcG9uZW50cy5TdHJlYW1zXG4gICAqXG4gICAqIEBwYXJhbSB0YWJsZU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgdGFibGVOYW1lLlxuICAgKiBAcGFyYW0gc3RyZWFtTGFiZWwgLSBJZGVudGlmaWVyIGZvciB0aGUgc3RyZWFtTGFiZWwuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25TdHJlYW0odGFibGVOYW1lOiBzdHJpbmcsIHN0cmVhbUxhYmVsOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmR5bmFtb2RiOiR7UmVnaW9ufToke0FjY291bnR9OnRhYmxlLyR7VGFibGVOYW1lfS9zdHJlYW0vJHtTdHJlYW1MYWJlbH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1RhYmxlTmFtZX0nLCB0YWJsZU5hbWUpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1N0cmVhbUxhYmVsfScsIHN0cmVhbUxhYmVsKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSB0YWJsZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvSG93SXRXb3Jrcy5Db3JlQ29tcG9uZW50cy5odG1sSG93SXRXb3Jrcy5Db3JlQ29tcG9uZW50cy5odG1sI0hvd0l0V29ya3MuQ29yZUNvbXBvbmVudHMuVGFibGVzSXRlbXNBdHRyaWJ1dGVzXG4gICAqXG4gICAqIEBwYXJhbSB0YWJsZU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgdGFibGVOYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKi9cbiAgcHVibGljIG9uVGFibGUodGFibGVOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmR5bmFtb2RiOiR7UmVnaW9ufToke0FjY291bnR9OnRhYmxlLyR7VGFibGVOYW1lfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7VGFibGVOYW1lfScsIHRhYmxlTmFtZSk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgYmFja3VwIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9iYWNrdXByZXN0b3JlX0hvd0l0V29ya3MuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gdGFibGVOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIHRhYmxlTmFtZS5cbiAgICogQHBhcmFtIGJhY2t1cE5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgYmFja3VwTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICovXG4gIHB1YmxpYyBvbkJhY2t1cCh0YWJsZU5hbWU6IHN0cmluZywgYmFja3VwTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpkeW5hbW9kYjoke1JlZ2lvbn06JHtBY2NvdW50fTp0YWJsZS8ke1RhYmxlTmFtZX0vYmFja3VwLyR7QmFja3VwTmFtZX0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1RhYmxlTmFtZX0nLCB0YWJsZU5hbWUpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0JhY2t1cE5hbWV9JywgYmFja3VwTmFtZSk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgZ2xvYmFsLXRhYmxlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9nbG9iYWx0YWJsZXNfSG93SXRXb3Jrcy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBnbG9iYWxUYWJsZU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgZ2xvYmFsVGFibGVOYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25HbG9iYWxUYWJsZShnbG9iYWxUYWJsZU5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmR5bmFtb2RiOjoke0FjY291bnR9Omdsb2JhbC10YWJsZS8ke0dsb2JhbFRhYmxlTmFtZX0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0dsb2JhbFRhYmxlTmFtZX0nLCBnbG9iYWxUYWJsZU5hbWUpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVyIGJhc2VkIG9uIHRoZSBhdHRyaWJ1dGUgKGZpZWxkIG9yIGNvbHVtbikgbmFtZXMgb2YgdGhlIHRhYmxlLlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NwZWNpZnlpbmctY29uZGl0aW9ucy5odG1sI0ZHQUNfRERCLkNvbmRpdGlvbktleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC5iYXRjaEdldEl0ZW0oKVxuICAgKiAtIC5iYXRjaFdyaXRlSXRlbSgpXG4gICAqIC0gLmNvbmRpdGlvbkNoZWNrSXRlbSgpXG4gICAqIC0gLmRlbGV0ZUl0ZW0oKVxuICAgKiAtIC5nZXRJdGVtKClcbiAgICogLSAucHV0SXRlbSgpXG4gICAqIC0gLnF1ZXJ5KClcbiAgICogLSAuc2NhbigpXG4gICAqIC0gLnVwZGF0ZUl0ZW0oKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF0dHJpYnV0ZXModmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBkeW5hbW9kYjpBdHRyaWJ1dGVzYCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogVXNlZCB0byBibG9jayBUcmFuc2FjdGlvbnMgQVBJcyBjYWxscyBhbmQgYWxsb3cgdGhlIG5vbi1UcmFuc2FjdGlvbiBBUElzIGNhbGxzIGFuZCB2aWNlLXZlcnNhLlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NwZWNpZnlpbmctY29uZGl0aW9ucy5odG1sI0ZHQUNfRERCLkNvbmRpdGlvbktleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC5kZWxldGVJdGVtKClcbiAgICogLSAuZ2V0SXRlbSgpXG4gICAqIC0gLnB1dEl0ZW0oKVxuICAgKiAtIC51cGRhdGVJdGVtKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZFbmNsb3NpbmdPcGVyYXRpb24odmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBkeW5hbW9kYjpFbmNsb3NpbmdPcGVyYXRpb25gLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGJhc2VkIG9uIHRoZSBwYXJ0aXRpb24ga2V5IG9mIHRoZSB0YWJsZS5cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zcGVjaWZ5aW5nLWNvbmRpdGlvbnMuaHRtbCNGR0FDX0REQi5Db25kaXRpb25LZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAuYmF0Y2hHZXRJdGVtKClcbiAgICogLSAuYmF0Y2hXcml0ZUl0ZW0oKVxuICAgKiAtIC5jb25kaXRpb25DaGVja0l0ZW0oKVxuICAgKiAtIC5kZWxldGVJdGVtKClcbiAgICogLSAuZ2V0SXRlbSgpXG4gICAqIC0gLnB1dEl0ZW0oKVxuICAgKiAtIC5xdWVyeSgpXG4gICAqIC0gLnVwZGF0ZUl0ZW0oKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkxlYWRpbmdLZXlzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgZHluYW1vZGI6TGVhZGluZ0tleXNgLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXIgYmFzZWQgb24gdGhlIFJldHVybkNvbnN1bWVkQ2FwYWNpdHkgcGFyYW1ldGVyIG9mIGEgcmVxdWVzdC4gQ29udGFpbnMgZWl0aGVyIFwiVE9UQUxcIiBvciBcIk5PTkVcIi5cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zcGVjaWZ5aW5nLWNvbmRpdGlvbnMuaHRtbCNGR0FDX0REQi5Db25kaXRpb25LZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAuYmF0Y2hHZXRJdGVtKClcbiAgICogLSAuYmF0Y2hXcml0ZUl0ZW0oKVxuICAgKiAtIC5jb25kaXRpb25DaGVja0l0ZW0oKVxuICAgKiAtIC5kZWxldGVJdGVtKClcbiAgICogLSAuZ2V0SXRlbSgpXG4gICAqIC0gLnB1dEl0ZW0oKVxuICAgKiAtIC5xdWVyeSgpXG4gICAqIC0gLnNjYW4oKVxuICAgKiAtIC51cGRhdGVJdGVtKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZSZXR1cm5Db25zdW1lZENhcGFjaXR5KHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgZHluYW1vZGI6UmV0dXJuQ29uc3VtZWRDYXBhY2l0eWAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlciBiYXNlZCBvbiB0aGUgUmV0dXJuVmFsdWVzIHBhcmFtZXRlciBvZiByZXF1ZXN0LiBDb250YWlucyBvbmUgb2YgdGhlIGZvbGxvd2luZzogXCJBTExfT0xEXCIsIFwiVVBEQVRFRF9PTERcIixcIkFMTF9ORVdcIixcIlVQREFURURfTkVXXCIsIG9yIFwiTk9ORVwiLlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NwZWNpZnlpbmctY29uZGl0aW9ucy5odG1sI0ZHQUNfRERCLkNvbmRpdGlvbktleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC5jb25kaXRpb25DaGVja0l0ZW0oKVxuICAgKiAtIC5kZWxldGVJdGVtKClcbiAgICogLSAucHV0SXRlbSgpXG4gICAqIC0gLnF1ZXJ5KClcbiAgICogLSAuc2NhbigpXG4gICAqIC0gLnVwZGF0ZUl0ZW0oKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJldHVyblZhbHVlcyh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGR5bmFtb2RiOlJldHVyblZhbHVlc2AsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlciBiYXNlZCBvbiB0aGUgU2VsZWN0IHBhcmFtZXRlciBvZiBhIFF1ZXJ5IG9yIFNjYW4gcmVxdWVzdC5cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zcGVjaWZ5aW5nLWNvbmRpdGlvbnMuaHRtbCNGR0FDX0REQi5Db25kaXRpb25LZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAuYmF0Y2hHZXRJdGVtKClcbiAgICogLSAuZ2V0SXRlbSgpXG4gICAqIC0gLnF1ZXJ5KClcbiAgICogLSAuc2NhbigpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmU2VsZWN0KHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgZHluYW1vZGI6U2VsZWN0YCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cbn1cbiJdfQ==