"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ec2 = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ec2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ec2 extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ec2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ec2';
        this.actionList = {
            "AcceptReservedInstancesExchangeQuote": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptReservedInstancesExchangeQuote.html",
                "description": "Grants permission to accept a Convertible Reserved Instance exchange quote",
                "accessLevel": "Write"
            },
            "AcceptTransitGatewayPeeringAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptTransitGatewayPeeringAttachment.html",
                "description": "Grants permission to accept a transit gateway peering attachment request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AcceptTransitGatewayVpcAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptTransitGatewayVpcAttachment.html",
                "description": "Grants permission to accept a request to attach a VPC to a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AcceptVpcEndpointConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptVpcEndpointConnections.html",
                "description": "Grants permission to accept one or more interface VPC endpoint connections to your VPC endpoint service",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AcceptVpcPeeringConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptVpcPeeringConnection.html",
                "description": "Grants permission to accept a VPC peering connection request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    },
                    "vpc-peering-connection": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AccepterVpc",
                            "ec2:Region",
                            "ec2:RequesterVpc",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AdvertiseByoipCidr": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AdvertiseByoipCidr.html",
                "description": "Grants permission to advertise an IP address range that is provisioned for use in AWS through bring your own IP addresses (BYOIP)",
                "accessLevel": "Write"
            },
            "AllocateAddress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AllocateAddress.html",
                "description": "Grants permission to allocate an Elastic IP address (EIP) to your account",
                "accessLevel": "Write"
            },
            "AllocateHosts": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AllocateHosts.html",
                "description": "Grants permission to allocate a Dedicated Host to your account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dedicated-host": {
                        "required": true
                    }
                }
            },
            "ApplySecurityGroupsToClientVpnTargetNetwork": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ApplySecurityGroupsToClientVpnTargetNetwork.html",
                "description": "Grants permission to apply a security group to the association between a Client VPN endpoint and a target network",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AssignIpv6Addresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssignIpv6Addresses.html",
                "description": "Grants permission to assign one or more IPv6 addresses to a network interface",
                "accessLevel": "Write"
            },
            "AssignPrivateIpAddresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssignPrivateIpAddresses.html",
                "description": "Grants permission to assign one or more secondary private IP addresses to a network interface",
                "accessLevel": "Write"
            },
            "AssociateAddress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateAddress.html",
                "description": "Grants permission to associate an Elastic IP address (EIP) with an instance or a network interface",
                "accessLevel": "Write"
            },
            "AssociateClientVpnTargetNetwork": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateClientVpnTargetNetwork.html",
                "description": "Grants permission to associate a target network with a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AssociateDhcpOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateDhcpOptions.html",
                "description": "Grants permission to associate or disassociate a set of DHCP options with a VPC",
                "accessLevel": "Write"
            },
            "AssociateIamInstanceProfile": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateIamInstanceProfile.html",
                "description": "Grants permission to associate an IAM instance profile with a running or stopped instance",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "AssociateRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateRouteTable.html",
                "description": "Grants permission to associate a subnet or gateway with a route table",
                "accessLevel": "Write"
            },
            "AssociateSubnetCidrBlock": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateSubnetCidrBlock.html",
                "description": "Grants permission to associate a CIDR block with a subnet",
                "accessLevel": "Write"
            },
            "AssociateTransitGatewayMulticastDomain": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateTransitGatewayMulticastDomain.html",
                "description": "Grants permission to associate an attachment and list of subnets with a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AssociateTransitGatewayRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateTransitGatewayRouteTable.html",
                "description": "Grants permission to associate an attachment with a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AssociateVpcCidrBlock": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateVpcCidrBlock.html",
                "description": "Grants permission to associate a CIDR block with a VPC",
                "accessLevel": "Write"
            },
            "AttachClassicLinkVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachClassicLinkVpc.html",
                "description": "Grants permission to link an EC2-Classic instance to a ClassicLink-enabled VPC through one or more of the VPC's security groups",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "AttachInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachInternetGateway.html",
                "description": "Grants permission to attach an internet gateway to a VPC",
                "accessLevel": "Write"
            },
            "AttachNetworkInterface": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachNetworkInterface.html",
                "description": "Grants permission to attach a network interface to an instance",
                "accessLevel": "Write"
            },
            "AttachVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachVolume.html",
                "description": "Grants permission to attach an EBS volume to a running or stopped instance and expose it to the instance with the specified device name",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "AttachVpnGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachVpnGateway.html",
                "description": "Grants permission to attach a virtual private gateway to a VPC",
                "accessLevel": "Write"
            },
            "AuthorizeClientVpnIngress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeClientVpnIngress.html",
                "description": "Grants permission to add an inbound authorization rule to a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AuthorizeSecurityGroupEgress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeSecurityGroupEgress.html",
                "description": "Grants permission to add one or more outbound rules to a VPC security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "AuthorizeSecurityGroupIngress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeSecurityGroupIngress.html",
                "description": "Grants permission to add one or more inbound rules to a security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "BundleInstance": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_BundleInstance.html",
                "description": "Grants permission to bundle an instance store-backed Windows instance",
                "accessLevel": "Write"
            },
            "CancelBundleTask": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelBundleTask.html",
                "description": "Grants permission to cancel a bundling operation",
                "accessLevel": "Write"
            },
            "CancelCapacityReservation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelCapacityReservation.html",
                "description": "Grants permission to cancel a Capacity Reservation and release the reserved capacity",
                "accessLevel": "Write",
                "resourceTypes": {
                    "capacity-reservation": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CancelConversionTask": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelConversionTask.html",
                "description": "Grants permission to cancel an active conversion task",
                "accessLevel": "Write"
            },
            "CancelExportTask": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelExportTask.html",
                "description": "Grants permission to cancel an active export task",
                "accessLevel": "Write"
            },
            "CancelImportTask": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelImportTask.html",
                "description": "Grants permission to cancel an in-process import virtual machine or import snapshot task",
                "accessLevel": "Write"
            },
            "CancelReservedInstancesListing": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelReservedInstancesListing.html",
                "description": "Grants permission to cancel a Reserved Instance listing on the Reserved Instance Marketplace",
                "accessLevel": "Write"
            },
            "CancelSpotFleetRequests": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelSpotFleetRequests.html",
                "description": "Grants permission to cancel one or more Spot Fleet requests",
                "accessLevel": "Write"
            },
            "CancelSpotInstanceRequests": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelSpotInstanceRequests.html",
                "description": "Grants permission to cancel one or more Spot Instance requests",
                "accessLevel": "Write"
            },
            "ConfirmProductInstance": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ConfirmProductInstance.html",
                "description": "Grants permission to determine whether an owned product code is associated with an instance",
                "accessLevel": "Write"
            },
            "CopyFpgaImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopyFpgaImage.html",
                "description": "Grants permission to copy a source Amazon FPGA image (AFI) to the current Region",
                "accessLevel": "Write"
            },
            "CopyImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopyImage.html",
                "description": "Grants permission to copy an Amazon Machine Image (AMI) from a source Region to the current Region",
                "accessLevel": "Write"
            },
            "CopySnapshot": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopySnapshot.html",
                "description": "Grants permission to copy a point-in-time snapshot of an EBS volume and store it in Amazon S3",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:TagKeys",
                            "aws:RequestTag/${TagKey}",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateCapacityReservation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateCapacityReservation.html",
                "description": "Grants permission to create a Capacity Reservation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "capacity-reservation": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateClientVpnEndpoint": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateClientVpnEndpoint.html",
                "description": "Grants permission to create a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateClientVpnRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateClientVpnRoute.html",
                "description": "Grants permission to add a network route to a Client VPN endpoint's route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateCustomerGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateCustomerGateway.html",
                "description": "Grants permission to create a customer gateway, which provides information to AWS about your customer gateway device",
                "accessLevel": "Write"
            },
            "CreateDefaultSubnet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDefaultSubnet.html",
                "description": "Grants permission to create a default subnet in a specified Availability Zone in a default VPC",
                "accessLevel": "Write"
            },
            "CreateDefaultVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDefaultVpc.html",
                "description": "Grants permission to create a default VPC with a default subnet in each Availability Zone",
                "accessLevel": "Write"
            },
            "CreateDhcpOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDhcpOptions.html",
                "description": "Grants permission to create a set of DHCP options for a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dhcp-options": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateEgressOnlyInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateEgressOnlyInternetGateway.html",
                "description": "Grants permission to create an egress-only internet gateway for a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "egress-only-internet-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateFleet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet.html",
                "description": "Grants permission to launch an EC2 Fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateFlowLogs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFlowLogs.html",
                "description": "Grants permission to create one or more flow logs to capture IP traffic for a network interface",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "vpc-flow-log": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    },
                    "network-interface": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Subnet",
                            "ec2:Vpc"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "vpc": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "CreateFpgaImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFpgaImage.html",
                "description": "Grants permission to create an Amazon FPGA Image (AFI) from a design checkpoint (DCP)",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fpga-image": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateImage.html",
                "description": "Grants permission to create an Amazon EBS-backed AMI from a stopped or running Amazon EBS-backed instance",
                "accessLevel": "Write"
            },
            "CreateInstanceExportTask": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateInstanceExportTask.html",
                "description": "Grants permission to export a running or stopped instance to an Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "export-instance-task": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateInternetGateway.html",
                "description": "Grants permission to create an internet gateway for a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "internet-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateKeyPair": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateKeyPair.html",
                "description": "Grants permission to create a 2048-bit RSA key pair",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key-pair": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateLaunchTemplate": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLaunchTemplate.html",
                "description": "Grants permission to create a launch template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launch-template": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateLaunchTemplateVersion": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLaunchTemplateVersion.html",
                "description": "Grants permission to create a new version of a launch template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launch-template": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateLocalGatewayRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLocalGatewayRoute.html",
                "description": "Grants permission to create a static route for a local gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "local-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "local-gateway-virtual-interface-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateLocalGatewayRouteTableVpcAssociation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLocalGatewayRouteTableVpcAssociation.html",
                "description": "Grants permission to associate a VPC with a local gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "local-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "local-gateway-route-table-vpc-association": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    },
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "CreateManagedPrefixList": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateManagedPrefixList.html",
                "description": "Grants permission to create a managed prefix list",
                "accessLevel": "Write",
                "resourceTypes": {
                    "prefix-list": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateNatGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNatGateway.html",
                "description": "Grants permission to create a NAT gateway in a subnet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "natgateway": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateNetworkAcl": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkAcl.html",
                "description": "Grants permission to create a network ACL in a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-acl": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateNetworkAclEntry": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkAclEntry.html",
                "description": "Grants permission to create a numbered entry (a rule) in a network ACL",
                "accessLevel": "Write"
            },
            "CreateNetworkInterface": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkInterface.html",
                "description": "Grants permission to create a network interface in a subnet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateNetworkInterfacePermission": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkInterfacePermission.html",
                "description": "Grants permission to create a permission for an AWS-authorized user to perform certain operations on a network interface",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AuthorizedUser",
                            "ec2:AvailabilityZone",
                            "ec2:Permission",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Subnet",
                            "ec2:Vpc",
                            "ec2:AuthorizedService"
                        ]
                    }
                }
            },
            "CreatePlacementGroup": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreatePlacementGroup.html",
                "description": "Grants permission to create a placement group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "placement-group": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateReservedInstancesListing": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateReservedInstancesListing.html",
                "description": "Grants permission to create a listing for Standard Reserved Instances to be sold in the Reserved Instance Marketplace",
                "accessLevel": "Write"
            },
            "CreateRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateRoute.html",
                "description": "Grants permission to create a route in a VPC route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "CreateRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateRouteTable.html",
                "description": "Grants permission to create a route table for a VPC",
                "accessLevel": "Write"
            },
            "CreateSecurityGroup": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSecurityGroup.html",
                "description": "Grants permission to create a security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateSnapshot": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSnapshot.html",
                "description": "Grants permission to create a snapshot of an EBS volume and store it in Amazon S3",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:TagKeys",
                            "aws:RequestTag/${TagKey}",
                            "ec2:ParentVolume",
                            "ec2:Region"
                        ]
                    },
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Encrypted",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "CreateSnapshots": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSnapshots.html",
                "description": "Grants permission to create crash-consistent snapshots of multiple EBS volumes and store them in Amazon S3",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:TagKeys",
                            "aws:RequestTag/${TagKey}",
                            "ec2:ParentVolume",
                            "ec2:Region"
                        ]
                    },
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Encrypted",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "CreateSpotDatafeedSubscription": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSpotDatafeedSubscription.html",
                "description": "Grants permission to create a data feed for Spot Instances to view Spot Instance usage logs",
                "accessLevel": "Write"
            },
            "CreateSubnet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSubnet.html",
                "description": "Grants permission to create a subnet in a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTags": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html",
                "description": "Grants permission to add or overwrite one or more tags for Amazon EC2 resources",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "capacity-reservation": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "client-vpn-endpoint": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "dedicated-host": {
                        "required": false
                    },
                    "dhcp-options": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "egress-only-internet-gateway": {
                        "required": false
                    },
                    "export-instance-task": {
                        "required": false
                    },
                    "fleet": {
                        "required": false
                    },
                    "fpga-image": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Owner",
                            "ec2:Public",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "image": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:ImageType",
                            "ec2:Owner",
                            "ec2:Public",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType"
                        ]
                    },
                    "instance": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "internet-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "local-gateway": {
                        "required": false
                    },
                    "local-gateway-route-table": {
                        "required": false
                    },
                    "local-gateway-route-table-virtual-interface-group-association": {
                        "required": false
                    },
                    "local-gateway-route-table-vpc-association": {
                        "required": false
                    },
                    "local-gateway-virtual-interface": {
                        "required": false
                    },
                    "local-gateway-virtual-interface-group": {
                        "required": false
                    },
                    "natgateway": {
                        "required": false
                    },
                    "network-acl": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "network-interface": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Subnet",
                            "ec2:Vpc",
                            "ec2:AssociatePublicIpAddress"
                        ]
                    },
                    "reserved-instances": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:InstanceType",
                            "ec2:Region",
                            "ec2:ReservedInstancesOfferingType",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    },
                    "route-table": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "security-group": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "snapshot": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:SnapshotTime",
                            "ec2:VolumeSize"
                        ]
                    },
                    "spot-instance-request": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "traffic-mirror-filter": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-session": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-target": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "volume": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    },
                    "vpc": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    },
                    "vpc-endpoint": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-endpoint-service": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-flow-log": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpn-connection": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpn-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "ec2:CreateAction"
                ]
            },
            "CreateTrafficMirrorFilter": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorFilter.html",
                "description": "Grants permission to create a traffic mirror filter",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTrafficMirrorFilterRule": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorFilterRule.html",
                "description": "Grants permission to create a traffic mirror filter rule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-filter-rule": {
                        "required": true,
                        "conditions": [
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTrafficMirrorSession": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorSession.html",
                "description": "Grants permission to create a traffic mirror session",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-session": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    },
                    "traffic-mirror-target": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateTrafficMirrorTarget": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorTarget.html",
                "description": "Grants permission to create a traffic mirror target",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-target": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    },
                    "network-interface": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateTransitGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGateway.html",
                "description": "Grants permission to create a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTransitGatewayMulticastDomain": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayMulticastDomain.html",
                "description": "Grants permission to create a multicast domain for a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTransitGatewayPeeringAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayPeeringAttachment.html",
                "description": "Grants permission to request a transit gateway peering attachment between a requester and accepter transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTransitGatewayRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayRoute.html",
                "description": "Grants permission to create a static route for a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateTransitGatewayRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayRouteTable.html",
                "description": "Grants permission to create a route table for a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTransitGatewayVpcAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayVpcAttachment.html",
                "description": "Grants permission to attach a VPC to a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    },
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html",
                "description": "Grants permission to create an EBS volume",
                "accessLevel": "Write",
                "resourceTypes": {
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "CreateVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpc.html",
                "description": "Grants permission to create a VPC with a specified CIDR block",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateVpcEndpoint": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpoint.html",
                "description": "Grants permission to create a VPC endpoint for an AWS service",
                "accessLevel": "Write",
                "dependentActions": [
                    "route53:AssociateVPCWithHostedZone"
                ],
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:VpceServiceName",
                            "ec2:VpceServiceOwner"
                        ]
                    },
                    "route-table": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "security-group": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateVpcEndpointConnectionNotification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpointConnectionNotification.html",
                "description": "Grants permission to create a connection notification for a VPC endpoint or VPC endpoint service",
                "accessLevel": "Write"
            },
            "CreateVpcEndpointServiceConfiguration": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpointServiceConfiguration.html",
                "description": "Grants permission to create a VPC endpoint service configuration to which service consumers (AWS accounts, IAM users, and IAM roles) can connect",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:VpceServicePrivateDnsName"
                        ]
                    }
                }
            },
            "CreateVpcPeeringConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcPeeringConnection.html",
                "description": "Grants permission to request a VPC peering connection between two VPCs",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    },
                    "vpc-peering-connection": {
                        "required": true,
                        "conditions": [
                            "ec2:AccepterVpc",
                            "ec2:Region",
                            "ec2:RequesterVpc"
                        ]
                    }
                }
            },
            "CreateVpnConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnConnection.html",
                "description": "Grants permission to create a VPN connection between a virtual private gateway or transit gateway and a customer gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpn-connection": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:AuthenticationType",
                            "ec2:DPDTimeoutSeconds",
                            "ec2:GatewayType",
                            "ec2:IKEVersions",
                            "ec2:InsideTunnelCidr",
                            "ec2:Phase1DHGroupNumbers",
                            "ec2:Phase2DHGroupNumbers",
                            "ec2:Phase1EncryptionAlgorithms",
                            "ec2:Phase2EncryptionAlgorithms",
                            "ec2:Phase1IntegrityAlgorithms",
                            "ec2:Phase2IntegrityAlgorithms",
                            "ec2:Phase1LifetimeSeconds",
                            "ec2:Phase2LifetimeSeconds",
                            "ec2:PresharedKeys",
                            "ec2:RekeyFuzzPercentage",
                            "ec2:RekeyMarginTimeSeconds",
                            "ec2:RoutingType"
                        ]
                    }
                }
            },
            "CreateVpnConnectionRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnConnectionRoute.html",
                "description": "Grants permission to create a static route for a VPN connection between a virtual private gateway and a customer gateway",
                "accessLevel": "Write"
            },
            "CreateVpnGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnGateway.html",
                "description": "Grants permission to create a virtual private gateway",
                "accessLevel": "Write"
            },
            "DeleteClientVpnEndpoint": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteClientVpnEndpoint.html",
                "description": "Grants permission to delete a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteClientVpnRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteClientVpnRoute.html",
                "description": "Grants permission to delete a route from a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteCustomerGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteCustomerGateway.html",
                "description": "Grants permission to delete a customer gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "customer-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteDhcpOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteDhcpOptions.html",
                "description": "Grants permission to delete a set of DHCP options",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dhcp-options": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteEgressOnlyInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteEgressOnlyInternetGateway.html",
                "description": "Grants permission to delete an egress-only internet gateway",
                "accessLevel": "Write"
            },
            "DeleteFleets": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFleets.html",
                "description": "Grants permission to delete one or more EC2 Fleets",
                "accessLevel": "Write"
            },
            "DeleteFlowLogs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFlowLogs.html",
                "description": "Grants permission to delete one or more flow logs",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-flow-log": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteFpgaImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFpgaImage.html",
                "description": "Grants permission to delete an Amazon FPGA Image (AFI)",
                "accessLevel": "Write"
            },
            "DeleteInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteInternetGateway.html",
                "description": "Grants permission to delete an internet gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "internet-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteKeyPair": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteKeyPair.html",
                "description": "Grants permission to delete a key pair by removing the public key from Amazon EC2",
                "accessLevel": "Write"
            },
            "DeleteLaunchTemplate": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLaunchTemplate.html",
                "description": "Grants permission to delete a launch template and its associated versions",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launch-template": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteLaunchTemplateVersions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLaunchTemplateVersions.html",
                "description": "Grants permission to delete one or more versions of a launch template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launch-template": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteLocalGatewayRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLocalGatewayRoute.html",
                "description": "Grants permission to delete a route from a local gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "local-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteLocalGatewayRouteTableVpcAssociation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLocalGatewayRouteTableVpcAssociation.html",
                "description": "Grants permission to delete an association between a VPC and local gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "local-gateway-route-table-vpc-association": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteManagedPrefixList": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteManagedPrefixList.html",
                "description": "Grants permission to delete a managed prefix list",
                "accessLevel": "Write",
                "resourceTypes": {
                    "prefix-list": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteNatGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNatGateway.html",
                "description": "Grants permission to delete a NAT gateway",
                "accessLevel": "Write"
            },
            "DeleteNetworkAcl": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkAcl.html",
                "description": "Grants permission to delete a network ACL",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-acl": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "DeleteNetworkAclEntry": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkAclEntry.html",
                "description": "Grants permission to delete an inbound or outbound entry (rule) from a network ACL",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-acl": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "DeleteNetworkInterface": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkInterface.html",
                "description": "Grants permission to delete a detached network interface",
                "accessLevel": "Write"
            },
            "DeleteNetworkInterfacePermission": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkInterfacePermission.html",
                "description": "Grants permission to delete a permission that is associated with a network interface",
                "accessLevel": "Permissions management"
            },
            "DeletePlacementGroup": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeletePlacementGroup.html",
                "description": "Grants permission to delete a placement group",
                "accessLevel": "Write"
            },
            "DeleteRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteRoute.html",
                "description": "Grants permission to delete a route from a route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "DeleteRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteRouteTable.html",
                "description": "Grants permission to delete a route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "DeleteSecurityGroup": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSecurityGroup.html",
                "description": "Grants permission to delete a security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "DeleteSnapshot": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSnapshot.html",
                "description": "Grants permission to delete a snapshot of an EBS volume",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:SnapshotTime",
                            "ec2:VolumeSize"
                        ]
                    }
                }
            },
            "DeleteSpotDatafeedSubscription": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSpotDatafeedSubscription.html",
                "description": "Grants permission to delete a data feed for Spot Instances",
                "accessLevel": "Write"
            },
            "DeleteSubnet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSubnet.html",
                "description": "Grants permission to delete a subnet",
                "accessLevel": "Write"
            },
            "DeleteTags": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTags.html",
                "description": "Grants permission to delete one or more tags from Amazon EC2 resources",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "capacity-reservation": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "client-vpn-endpoint": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "dedicated-host": {
                        "required": false
                    },
                    "dhcp-options": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "egress-only-internet-gateway": {
                        "required": false
                    },
                    "export-instance-task": {
                        "required": false
                    },
                    "fleet": {
                        "required": false
                    },
                    "fpga-image": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "image": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "instance": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "internet-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "local-gateway": {
                        "required": false
                    },
                    "local-gateway-route-table": {
                        "required": false
                    },
                    "local-gateway-route-table-virtual-interface-group-association": {
                        "required": false
                    },
                    "local-gateway-route-table-vpc-association": {
                        "required": false
                    },
                    "local-gateway-virtual-interface": {
                        "required": false
                    },
                    "local-gateway-virtual-interface-group": {
                        "required": false
                    },
                    "natgateway": {
                        "required": false
                    },
                    "network-acl": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "network-interface": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "reserved-instances": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "route-table": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "security-group": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "snapshot": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "spot-instance-request": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "volume": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-endpoint": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-endpoint-service": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-flow-log": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpn-connection": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpn-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTrafficMirrorFilter": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorFilter.html",
                "description": "Grants permission to delete a traffic mirror filter",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTrafficMirrorFilterRule": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorFilterRule.html",
                "description": "Grants permission to delete a traffic mirror filter rule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-filter-rule": {
                        "required": true,
                        "conditions": [
                            "ec2:Region"
                        ]
                    }
                }
            },
            "DeleteTrafficMirrorSession": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorSession.html",
                "description": "Grants permission to delete a traffic mirror session",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-session": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTrafficMirrorTarget": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorTarget.html",
                "description": "Grants permission to delete a traffic mirror target",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-target": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGateway.html",
                "description": "Grants permission to delete a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGatewayMulticastDomain": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayMulticastDomain.html",
                "description": "Grants permissions to delete a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGatewayPeeringAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayPeeringAttachment.html",
                "description": "Grants permission to delete a peering attachment from a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGatewayRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayRoute.html",
                "description": "Grants permission to delete a route from a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGatewayRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayRouteTable.html",
                "description": "Grants permission to delete a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGatewayVpcAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayVpcAttachment.html",
                "description": "Grants permission to delete a VPC attachment from a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVolume.html",
                "description": "Grants permission to delete an EBS volume",
                "accessLevel": "Write",
                "resourceTypes": {
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "DeleteVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpc.html",
                "description": "Grants permission to delete a VPC",
                "accessLevel": "Write"
            },
            "DeleteVpcEndpointConnectionNotifications": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpointConnectionNotifications.html",
                "description": "Grants permission to delete one or more VPC endpoint connection notifications",
                "accessLevel": "Write"
            },
            "DeleteVpcEndpointServiceConfigurations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpointServiceConfigurations.html",
                "description": "Grants permission to delete one or more VPC endpoint service configurations",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteVpcEndpoints": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpoints.html",
                "description": "Grants permission to delete one or more VPC endpoints",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteVpcPeeringConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcPeeringConnection.html",
                "description": "Grants permission to delete a VPC peering connection",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-peering-connection": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AccepterVpc",
                            "ec2:Region",
                            "ec2:RequesterVpc",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteVpnConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnConnection.html",
                "description": "Grants permission to delete a VPN connection",
                "accessLevel": "Write"
            },
            "DeleteVpnConnectionRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnConnectionRoute.html",
                "description": "Grants permission to delete a static route for a VPN connection between a virtual private gateway and a customer gateway",
                "accessLevel": "Write"
            },
            "DeleteVpnGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnGateway.html",
                "description": "Grants permission to delete a virtual private gateway",
                "accessLevel": "Write"
            },
            "DeprovisionByoipCidr": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeprovisionByoipCidr.html",
                "description": "Grants permission to release an IP address range that was provisioned through bring your own IP addresses (BYOIP), and to delete the corresponding address pool",
                "accessLevel": "Write"
            },
            "DeregisterImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterImage.html",
                "description": "Grants permission to deregister an Amazon Machine Image (AMI)",
                "accessLevel": "Write"
            },
            "DeregisterInstanceEventNotificationAttributes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterInstanceEventNotificationAttributes.html",
                "description": "Grants permission to remove tags from the set of tags to include in notifications about scheduled events for your instances",
                "accessLevel": "Write"
            },
            "DeregisterTransitGatewayMulticastGroupMembers": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterTransitGatewayMulticastGroupMembers.html",
                "description": "Grants permission to deregister one or more network interface members from a group IP address in a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeregisterTransitGatewayMulticastGroupSources": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterTransitGatewayMulticastGroupSources.html",
                "description": "Grants permission to deregister one or more network interface sources from a group IP address in a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DescribeAccountAttributes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAccountAttributes.html",
                "description": "Grants permission to describe the attributes of the AWS account",
                "accessLevel": "List"
            },
            "DescribeAddresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAddresses.html",
                "description": "Grants permission to describe one or more Elastic IP addresses",
                "accessLevel": "List"
            },
            "DescribeAggregateIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAggregateIdFormat.html",
                "description": "Grants permission to describe the longer ID format settings for all resource types",
                "accessLevel": "List"
            },
            "DescribeAvailabilityZones": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAvailabilityZones.html",
                "description": "Grants permission to describe one or more of the Availability Zones that are available to you",
                "accessLevel": "List"
            },
            "DescribeBundleTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeBundleTasks.html",
                "description": "Grants permission to describe one or more bundling tasks",
                "accessLevel": "List"
            },
            "DescribeByoipCidrs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeByoipCidrs.html",
                "description": "Grants permission to describe the IP address ranges that were provisioned through bring your own IP addresses (BYOIP)",
                "accessLevel": "List"
            },
            "DescribeCapacityReservations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCapacityReservations.html",
                "description": "Grants permission to describe one or more Capacity Reservations",
                "accessLevel": "List"
            },
            "DescribeClassicLinkInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClassicLinkInstances.html",
                "description": "Grants permission to describe one or more linked EC2-Classic instances",
                "accessLevel": "List"
            },
            "DescribeClientVpnAuthorizationRules": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnAuthorizationRules.html",
                "description": "Grants permission to describe the authorization rules for a Client VPN endpoint",
                "accessLevel": "List"
            },
            "DescribeClientVpnConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnConnections.html",
                "description": "Grants permission to describe active client connections and connections that have been terminated within the last 60 minutes for a Client VPN endpoint",
                "accessLevel": "List"
            },
            "DescribeClientVpnEndpoints": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnEndpoints.html",
                "description": "Grants permission to describe one or more Client VPN endpoints",
                "accessLevel": "List"
            },
            "DescribeClientVpnRoutes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnRoutes.html",
                "description": "Grants permission to describe the routes for a Client VPN endpoint",
                "accessLevel": "List"
            },
            "DescribeClientVpnTargetNetworks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnTargetNetworks.html",
                "description": "Grants permission to describe the target networks that are associated with a Client VPN endpoint",
                "accessLevel": "List"
            },
            "DescribeCoipPools": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCoipPools.html",
                "description": "Grants permission to describe the specified customer-owned address pools or all of your customer-owned address pools",
                "accessLevel": "List"
            },
            "DescribeConversionTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeConversionTasks.html",
                "description": "Grants permission to describe one or more conversion tasks",
                "accessLevel": "List"
            },
            "DescribeCustomerGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCustomerGateways.html",
                "description": "Grants permission to describe one or more customer gateways",
                "accessLevel": "List"
            },
            "DescribeDhcpOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeDhcpOptions.html",
                "description": "Grants permission to describe one or more DHCP options sets",
                "accessLevel": "List"
            },
            "DescribeEgressOnlyInternetGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeEgressOnlyInternetGateways.html",
                "description": "Grants permission to describe one or more egress-only internet gateways",
                "accessLevel": "List"
            },
            "DescribeElasticGpus": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeElasticGpus.html",
                "description": "Grants permission to describe an Elastic Graphics accelerator that is associated with an instance",
                "accessLevel": "Read"
            },
            "DescribeExportImageTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeExportImageTasks.html",
                "description": "Grants permission to describe one or more export image tasks",
                "accessLevel": "List"
            },
            "DescribeExportTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeExportTasks.html",
                "description": "Grants permission to describe one or more export instance tasks",
                "accessLevel": "List"
            },
            "DescribeFastSnapshotRestores": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFastSnapshotRestores.html",
                "description": "Grants permission to describe the state of fast snapshot restores for snapshots",
                "accessLevel": "Read"
            },
            "DescribeFleetHistory": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleetHistory.html",
                "description": "Grants permission to describe the events for an EC2 Fleet during a specified time",
                "accessLevel": "List"
            },
            "DescribeFleetInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleetInstances.html",
                "description": "Grants permission to describe the running instances for an EC2 Fleet",
                "accessLevel": "List"
            },
            "DescribeFleets": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleets.html",
                "description": "Grants permission to describe one or more EC2 Fleets",
                "accessLevel": "List"
            },
            "DescribeFlowLogs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFlowLogs.html",
                "description": "Grants permission to describe one or more flow logs",
                "accessLevel": "List"
            },
            "DescribeFpgaImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFpgaImageAttribute.html",
                "description": "Grants permission to describe the attributes of an Amazon FPGA Image (AFI)",
                "accessLevel": "List"
            },
            "DescribeFpgaImages": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFpgaImages.html",
                "description": "Grants permission to describe one or more Amazon FPGA Images (AFIs)",
                "accessLevel": "List"
            },
            "DescribeHostReservationOfferings": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHostReservationOfferings.html",
                "description": "Grants permission to describe the Dedicated Host Reservations that are available to purchase",
                "accessLevel": "List"
            },
            "DescribeHostReservations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHostReservations.html",
                "description": "Grants permission to describe the Dedicated Host Reservations that are associated with Dedicated Hosts in the AWS account",
                "accessLevel": "List"
            },
            "DescribeHosts": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHosts.html",
                "description": "Grants permission to describe one or more Dedicated Hosts",
                "accessLevel": "List"
            },
            "DescribeIamInstanceProfileAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIamInstanceProfileAssociations.html",
                "description": "Grants permission to describe the IAM instance profile associations",
                "accessLevel": "List"
            },
            "DescribeIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIdFormat.html",
                "description": "Grants permission to describe the ID format settings for resources",
                "accessLevel": "List"
            },
            "DescribeIdentityIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIdentityIdFormat.html",
                "description": "Grants permission to describe the ID format settings for resources for an IAM user, IAM role, or root user",
                "accessLevel": "List"
            },
            "DescribeImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImageAttribute.html",
                "description": "Grants permission to describe an attribute of an Amazon Machine Image (AMI)",
                "accessLevel": "List"
            },
            "DescribeImages": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImages.html",
                "description": "Grants permission to describe one or more images (AMIs, AKIs, and ARIs)",
                "accessLevel": "List"
            },
            "DescribeImportImageTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImportImageTasks.html",
                "description": "Grants permission to describe import virtual machine or import snapshot tasks",
                "accessLevel": "List"
            },
            "DescribeImportSnapshotTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImportSnapshotTasks.html",
                "description": "Grants permission to describe import snapshot tasks",
                "accessLevel": "List"
            },
            "DescribeInstanceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceAttribute.html",
                "description": "Grants permission to describe the attributes of an instance",
                "accessLevel": "List"
            },
            "DescribeInstanceCreditSpecifications": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceCreditSpecifications.html",
                "description": "Grants permission to describe the credit option for CPU usage of one or more burstable performance instances",
                "accessLevel": "List"
            },
            "DescribeInstanceEventNotificationAttributes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceEventNotificationAttributes.html",
                "description": "Grants permission to describe the set of tags to include in notifications about scheduled events for your instances",
                "accessLevel": "List"
            },
            "DescribeInstanceStatus": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceStatus.html",
                "description": "Grants permission to describe the status of one or more instances",
                "accessLevel": "List"
            },
            "DescribeInstanceTypeOfferings": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceTypeOfferings.html",
                "description": "Grants permission to describe the set of instance types that are offered in a location",
                "accessLevel": "List"
            },
            "DescribeInstanceTypes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceTypes.html",
                "description": "Grants permission to describe the details of instance types that are offered in a location",
                "accessLevel": "List"
            },
            "DescribeInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstances.html",
                "description": "Grants permission to describe one or more instances",
                "accessLevel": "List"
            },
            "DescribeInternetGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInternetGateways.html",
                "description": "Grants permission to describe one or more internet gateways",
                "accessLevel": "List"
            },
            "DescribeKeyPairs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeKeyPairs.html",
                "description": "Grants permission to describe one or more key pairs",
                "accessLevel": "List"
            },
            "DescribeLaunchTemplateVersions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLaunchTemplateVersions.html",
                "description": "Grants permission to describe one or more launch template versions",
                "accessLevel": "List"
            },
            "DescribeLaunchTemplates": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLaunchTemplates.html",
                "description": "Grants permission to describe one or more launch templates",
                "accessLevel": "List"
            },
            "DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations.html",
                "description": "Grants permission to describe the associations between virtual interface groups and local gateway route tables",
                "accessLevel": "List"
            },
            "DescribeLocalGatewayRouteTableVpcAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTableVpcAssociations.html",
                "description": "Grants permission to describe an association between VPCs and local gateway route tables",
                "accessLevel": "List"
            },
            "DescribeLocalGatewayRouteTables": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTables.html",
                "description": "Grants permission to describe one or more local gateway route tables",
                "accessLevel": "List"
            },
            "DescribeLocalGatewayVirtualInterfaceGroups": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayVirtualInterfaceGroups.html",
                "description": "Grants permission to describe local gateway virtual interface groups",
                "accessLevel": "List"
            },
            "DescribeLocalGatewayVirtualInterfaces": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayVirtualInterfaces.html",
                "description": "Grants permission to describe local gateway virtual interfaces",
                "accessLevel": "List"
            },
            "DescribeLocalGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGateways.html",
                "description": "Grants permission to describe one or more local gateways",
                "accessLevel": "List"
            },
            "DescribeManagedPrefixLists": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeManagedPrefixLists.html",
                "description": "Grants permission to describe your managed prefix lists and any AWS-managed prefix lists",
                "accessLevel": "List"
            },
            "DescribeMovingAddresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeMovingAddresses.html",
                "description": "Grants permission to describe Elastic IP addresses that are being moved to the EC2-VPC platform",
                "accessLevel": "List"
            },
            "DescribeNatGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNatGateways.html",
                "description": "Grants permission to describe one or more NAT gateways",
                "accessLevel": "List"
            },
            "DescribeNetworkAcls": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkAcls.html",
                "description": "Grants permission to describe one or more network ACLs",
                "accessLevel": "List"
            },
            "DescribeNetworkInterfaceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfaceAttribute.html",
                "description": "Grants permission to describe a network interface attribute",
                "accessLevel": "List"
            },
            "DescribeNetworkInterfacePermissions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfacePermissions.html",
                "description": "Grants permission to describe the permissions that are associated with a network interface",
                "accessLevel": "List"
            },
            "DescribeNetworkInterfaces": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfaces.html",
                "description": "Grants permission to describe one or more network interfaces",
                "accessLevel": "List"
            },
            "DescribePlacementGroups": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePlacementGroups.html",
                "description": "Grants permission to describe one or more placement groups",
                "accessLevel": "List"
            },
            "DescribePrefixLists": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePrefixLists.html",
                "description": "Grants permission to describe available AWS services in a prefix list format",
                "accessLevel": "List"
            },
            "DescribePrincipalIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePrincipalIdFormat.html",
                "description": "Grants permission to describe the ID format settings for the root user and all IAM roles and IAM users that have explicitly specified a longer ID (17-character ID) preference",
                "accessLevel": "List"
            },
            "DescribePublicIpv4Pools": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePublicIpv4Pools.html",
                "description": "Grants permission to describe one or more IPv4 address pools",
                "accessLevel": "List"
            },
            "DescribeRegions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeRegions.html",
                "description": "Grants permission to describe one or more AWS Regions that are currently available in your account",
                "accessLevel": "List"
            },
            "DescribeReservedInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstances.html",
                "description": "Grants permission to describe one or more purchased Reserved Instances in your account",
                "accessLevel": "List"
            },
            "DescribeReservedInstancesListings": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesListings.html",
                "description": "Grants permission to describe your account's Reserved Instance listings in the Reserved Instance Marketplace",
                "accessLevel": "List"
            },
            "DescribeReservedInstancesModifications": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesModifications.html",
                "description": "Grants permission to describe the modifications made to one or more Reserved Instances",
                "accessLevel": "List"
            },
            "DescribeReservedInstancesOfferings": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesOfferings.html",
                "description": "Grants permission to describe the Reserved Instance offerings that are available for purchase",
                "accessLevel": "List"
            },
            "DescribeRouteTables": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeRouteTables.html",
                "description": "Grants permission to describe one or more route tables",
                "accessLevel": "List"
            },
            "DescribeScheduledInstanceAvailability": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeScheduledInstanceAvailability.html",
                "description": "Grants permission to find available schedules for Scheduled Instances",
                "accessLevel": "Read"
            },
            "DescribeScheduledInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeScheduledInstances.html",
                "description": "Grants permission to describe one or more Scheduled Instances in your account",
                "accessLevel": "Read"
            },
            "DescribeSecurityGroupReferences": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSecurityGroupReferences.html",
                "description": "Grants permission to describe the VPCs on the other side of a VPC peering connection that are referencing specified VPC security groups",
                "accessLevel": "List"
            },
            "DescribeSecurityGroups": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSecurityGroups.html",
                "description": "Grants permission to describe one or more security groups",
                "accessLevel": "List"
            },
            "DescribeSnapshotAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSnapshotAttribute.html",
                "description": "Grants permission to describe an attribute of a snapshot",
                "accessLevel": "List"
            },
            "DescribeSnapshots": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSnapshots.html",
                "description": "Grants permission to describe one or more EBS snapshots",
                "accessLevel": "List"
            },
            "DescribeSpotDatafeedSubscription": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotDatafeedSubscription.html",
                "description": "Grants permission to describe the data feed for Spot Instances",
                "accessLevel": "List"
            },
            "DescribeSpotFleetInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetInstances.html",
                "description": "Grants permission to describe the running instances for a Spot Fleet",
                "accessLevel": "List"
            },
            "DescribeSpotFleetRequestHistory": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetRequestHistory.html",
                "description": "Grants permission to describe the events for a Spot Fleet request during a specified time",
                "accessLevel": "List"
            },
            "DescribeSpotFleetRequests": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetRequests.html",
                "description": "Grants permission to describe one or more Spot Fleet requests",
                "accessLevel": "List"
            },
            "DescribeSpotInstanceRequests": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotInstanceRequests.html",
                "description": "Grants permission to describe one or more Spot Instance requests",
                "accessLevel": "List"
            },
            "DescribeSpotPriceHistory": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotPriceHistory.html",
                "description": "Grants permission to describe the Spot Instance price history",
                "accessLevel": "List"
            },
            "DescribeStaleSecurityGroups": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeStaleSecurityGroups.html",
                "description": "Grants permission to describe the stale security group rules for security groups in a specified VPC",
                "accessLevel": "List"
            },
            "DescribeSubnets": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSubnets.html",
                "description": "Grants permission to describe one or more subnets",
                "accessLevel": "List"
            },
            "DescribeTags": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTags.html",
                "description": "Grants permission to describe one or more tags for an Amazon EC2 resource",
                "accessLevel": "Read"
            },
            "DescribeTrafficMirrorFilters": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorFilters.html",
                "description": "Grants permission to describe one or more traffic mirror filters",
                "accessLevel": "List"
            },
            "DescribeTrafficMirrorSessions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorSessions.html",
                "description": "Grants permission to describe one or more traffic mirror sessions",
                "accessLevel": "List"
            },
            "DescribeTrafficMirrorTargets": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorTargets.html",
                "description": "Grants permission to describe one or more traffic mirror targets",
                "accessLevel": "List"
            },
            "DescribeTransitGatewayAttachments": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayAttachments.html",
                "description": "Grants permission to describe one or more attachments between resources and transit gateways",
                "accessLevel": "List"
            },
            "DescribeTransitGatewayMulticastDomains": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayMulticastDomains.html",
                "description": "Grants permission to describe one or more transit gateway multicast domains",
                "accessLevel": "List"
            },
            "DescribeTransitGatewayPeeringAttachments": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayPeeringAttachments.html",
                "description": "Grants permission to describe one or more transit gateway peering attachments",
                "accessLevel": "List"
            },
            "DescribeTransitGatewayRouteTables": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayRouteTables.html",
                "description": "Grants permission to describe one or more transit gateway route tables",
                "accessLevel": "List"
            },
            "DescribeTransitGatewayVpcAttachments": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayVpcAttachments.html",
                "description": "Grants permission to describe one or more VPC attachments on a transit gateway",
                "accessLevel": "List"
            },
            "DescribeTransitGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGateways.html",
                "description": "Grants permission to describe one or more transit gateways",
                "accessLevel": "List"
            },
            "DescribeVolumeAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumeAttribute.html",
                "description": "Grants permission to describe an attribute of an EBS volume",
                "accessLevel": "List"
            },
            "DescribeVolumeStatus": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumeStatus.html",
                "description": "Grants permission to describe the status of one or more EBS volumes",
                "accessLevel": "List"
            },
            "DescribeVolumes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumes.html",
                "description": "Grants permission to describe one or more EBS volumes",
                "accessLevel": "List"
            },
            "DescribeVolumesModifications": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumesModifications.html",
                "description": "Grants permission to describe the current modification status of one or more EBS volumes",
                "accessLevel": "Read"
            },
            "DescribeVpcAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcAttribute.html",
                "description": "Grants permission to describe an attribute of a VPC",
                "accessLevel": "List"
            },
            "DescribeVpcClassicLink": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcClassicLink.html",
                "description": "Grants permission to describe the ClassicLink status of one or more VPCs",
                "accessLevel": "List"
            },
            "DescribeVpcClassicLinkDnsSupport": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcClassicLinkDnsSupport.html",
                "description": "Grants permission to describe the ClassicLink DNS support status of one or more VPCs",
                "accessLevel": "List"
            },
            "DescribeVpcEndpointConnectionNotifications": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointConnectionNotifications.html",
                "description": "Grants permission to describe the connection notifications for VPC endpoints and VPC endpoint services",
                "accessLevel": "List"
            },
            "DescribeVpcEndpointConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointConnections.html",
                "description": "Grants permission to describe the VPC endpoint connections to your VPC endpoint services",
                "accessLevel": "List"
            },
            "DescribeVpcEndpointServiceConfigurations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServiceConfigurations.html",
                "description": "Grants permission to describe VPC endpoint service configurations (your services)",
                "accessLevel": "List"
            },
            "DescribeVpcEndpointServicePermissions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServicePermissions.html",
                "description": "Grants permission to describe the principals (service consumers) that are permitted to discover your VPC endpoint service",
                "accessLevel": "List"
            },
            "DescribeVpcEndpointServices": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServices.html",
                "description": "Grants permission to describe all supported AWS services that can be specified when creating a VPC endpoint",
                "accessLevel": "List"
            },
            "DescribeVpcEndpoints": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpoints.html",
                "description": "Grants permission to describe one or more VPC endpoints",
                "accessLevel": "List"
            },
            "DescribeVpcPeeringConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcPeeringConnections.html",
                "description": "Grants permission to describe one or more VPC peering connections",
                "accessLevel": "List"
            },
            "DescribeVpcs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcs.html",
                "description": "Grants permission to describe one or more VPCs",
                "accessLevel": "List"
            },
            "DescribeVpnConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpnConnections.html",
                "description": "Grants permission to describe one or more VPN connections",
                "accessLevel": "Read"
            },
            "DescribeVpnGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpnGateways.html",
                "description": "Grants permission to describe one or more virtual private gateways",
                "accessLevel": "List"
            },
            "DetachClassicLinkVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachClassicLinkVpc.html",
                "description": "Grants permission to unlink (detach) a linked EC2-Classic instance from a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "DetachInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachInternetGateway.html",
                "description": "Grants permission to detach an internet gateway from a VPC",
                "accessLevel": "Write"
            },
            "DetachNetworkInterface": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachNetworkInterface.html",
                "description": "Grants permission to detach a network interface from an instance",
                "accessLevel": "Write"
            },
            "DetachVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachVolume.html",
                "description": "Grants permission to detach an EBS volume from an instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "DetachVpnGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachVpnGateway.html",
                "description": "Grants permission to detach a virtual private gateway from a VPC",
                "accessLevel": "Write"
            },
            "DisableEbsEncryptionByDefault": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableEbsEncryptionByDefault.html",
                "description": "Grants permission to disable EBS encryption by default for your account",
                "accessLevel": "Write"
            },
            "DisableFastSnapshotRestores": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableFastSnapshotRestores.html",
                "description": "Grants permission to disable fast snapshot restores for one or more snapshots in specified Availability Zones",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:AvailabilityZone",
                            "ec2:SnapshotTime",
                            "ec2:Encrypted",
                            "ec2:VolumeSize",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DisableTransitGatewayRouteTablePropagation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableTransitGatewayRouteTablePropagation.html",
                "description": "Grants permission to disable a resource attachment from propagating routes to the specified propagation route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DisableVgwRoutePropagation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVgwRoutePropagation.html",
                "description": "Grants permission to disable a virtual private gateway from propagating routes to a specified route table of a VPC",
                "accessLevel": "Write"
            },
            "DisableVpcClassicLink": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVpcClassicLink.html",
                "description": "Grants permission to disable ClassicLink for a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "DisableVpcClassicLinkDnsSupport": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVpcClassicLinkDnsSupport.html",
                "description": "Grants permission to disable ClassicLink DNS support for a VPC",
                "accessLevel": "Write"
            },
            "DisassociateAddress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateAddress.html",
                "description": "Grants permission to disassociate an Elastic IP address from an instance or network interface",
                "accessLevel": "Write"
            },
            "DisassociateClientVpnTargetNetwork": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateClientVpnTargetNetwork.html",
                "description": "Grants permission to disassociate a target network from a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DisassociateIamInstanceProfile": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateIamInstanceProfile.html",
                "description": "Grants permission to disassociate an IAM instance profile from a running or stopped instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "DisassociateRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateRouteTable.html",
                "description": "Grants permission to disassociate a subnet from a route table",
                "accessLevel": "Write"
            },
            "DisassociateSubnetCidrBlock": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateSubnetCidrBlock.html",
                "description": "Grants permission to disassociate a CIDR block from a subnet",
                "accessLevel": "Write"
            },
            "DisassociateTransitGatewayMulticastDomain": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateTransitGatewayMulticastDomain.html",
                "description": "Grants permission to disassociate one or more subnets from a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DisassociateTransitGatewayRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateTransitGatewayRouteTable.html",
                "description": "Grants permission to disassociate a resource attachment from a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DisassociateVpcCidrBlock": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateVpcCidrBlock.html",
                "description": "Grants permission to disassociate a CIDR block from a VPC",
                "accessLevel": "Write"
            },
            "EnableEbsEncryptionByDefault": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableEbsEncryptionByDefault.html",
                "description": "Grants permission to enable EBS encryption by default for your account",
                "accessLevel": "Write"
            },
            "EnableFastSnapshotRestores": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableFastSnapshotRestores.html",
                "description": "Grants permission to enable fast snapshot restores for one or more snapshots in specified Availability Zones",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:AvailabilityZone",
                            "ec2:SnapshotTime",
                            "ec2:Encrypted",
                            "ec2:VolumeSize",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "EnableTransitGatewayRouteTablePropagation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableTransitGatewayRouteTablePropagation.html",
                "description": "Grants permission to enable an attachment to propagate routes to a propagation route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "EnableVgwRoutePropagation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVgwRoutePropagation.html",
                "description": "Grants permission to enable a virtual private gateway to propagate routes to a VPC route table",
                "accessLevel": "Write"
            },
            "EnableVolumeIO": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVolumeIO.html",
                "description": "Grants permission to enable I/O operations for a volume that had I/O operations disabled",
                "accessLevel": "Write"
            },
            "EnableVpcClassicLink": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVpcClassicLink.html",
                "description": "Grants permission to enable a VPC for ClassicLink",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "EnableVpcClassicLinkDnsSupport": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVpcClassicLinkDnsSupport.html",
                "description": "Grants permission to enable a VPC to support DNS hostname resolution for ClassicLink",
                "accessLevel": "Write"
            },
            "ExportClientVpnClientCertificateRevocationList": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportClientVpnClientCertificateRevocationList.html",
                "description": "Grants permission to download the client certificate revocation list for a Client VPN endpoint",
                "accessLevel": "List"
            },
            "ExportClientVpnClientConfiguration": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportClientVpnClientConfiguration.html",
                "description": "Grants permission to download the contents of the Client VPN endpoint configuration file for a Client VPN endpoint",
                "accessLevel": "List"
            },
            "ExportImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportImage.html",
                "description": "Grants permission to export an Amazon Machine Image (AMI) to a VM file",
                "accessLevel": "Write"
            },
            "ExportTransitGatewayRoutes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportTransitGatewayRoutes.html",
                "description": "Grants permission to export routes from a transit gateway route table to an Amazon S3 bucket",
                "accessLevel": "Write"
            },
            "GetCapacityReservationUsage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetCapacityReservationUsage.html",
                "description": "Grants permission to get usage information about a Capacity Reservation",
                "accessLevel": "Read"
            },
            "GetCoipPoolUsage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetCoipPoolUsage.html",
                "description": "Grants permission to describe the allocations from the specified customer-owned address pool",
                "accessLevel": "Read"
            },
            "GetConsoleOutput": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetConsoleOutput.html",
                "description": "Grants permission to get the console output for an instance",
                "accessLevel": "Read"
            },
            "GetConsoleScreenshot": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetConsoleScreenshot.html",
                "description": "Grants permission to retrieve a JPG-format screenshot of a running instance",
                "accessLevel": "Read",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "GetDefaultCreditSpecification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetDefaultCreditSpecification.html",
                "description": "Grants permission to get the default credit option for CPU usage of a burstable performance instance family",
                "accessLevel": "Read"
            },
            "GetEbsDefaultKmsKeyId": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetEbsDefaultKmsKeyId.html",
                "description": "Grants permission to get the ID of the default customer master key (CMK) for EBS encryption by default",
                "accessLevel": "Read"
            },
            "GetEbsEncryptionByDefault": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetEbsEncryptionByDefault.html",
                "description": "Grants permission to describe whether EBS encryption by default is enabled for your account",
                "accessLevel": "Read"
            },
            "GetHostReservationPurchasePreview": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetHostReservationPurchasePreview.html",
                "description": "Grants permission to preview a reservation purchase with configurations that match those of a Dedicated Host",
                "accessLevel": "Read"
            },
            "GetLaunchTemplateData": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetLaunchTemplateData.html",
                "description": "Grants permission to get the configuration data of the specified instance for use with a new launch template or launch template version",
                "accessLevel": "Read"
            },
            "GetManagedPrefixListAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetManagedPrefixListAssociations.html",
                "description": "Grants permission to get information about the resources that are associated with the specified managed prefix list",
                "accessLevel": "Read"
            },
            "GetManagedPrefixListEntries": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetManagedPrefixListEntries.html",
                "description": "Grants permission to get information about the entries for a specified managed prefix list",
                "accessLevel": "Read"
            },
            "GetPasswordData": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetPasswordData.html",
                "description": "Grants permission to retrieve the encrypted administrator password for a running Windows instance",
                "accessLevel": "Read"
            },
            "GetReservedInstancesExchangeQuote": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetReservedInstancesExchangeQuote.html",
                "description": "Grants permission to return a quote and exchange information for exchanging one or more Convertible Reserved Instances for a new Convertible Reserved Instance",
                "accessLevel": "Read"
            },
            "GetTransitGatewayAttachmentPropagations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayAttachmentPropagations.html",
                "description": "Grants permission to list the route tables to which a resource attachment propagates routes",
                "accessLevel": "List"
            },
            "GetTransitGatewayMulticastDomainAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayMulticastDomainAssociations.html",
                "description": "Grants permission to get information about the associations for a transit gateway multicast domain",
                "accessLevel": "List"
            },
            "GetTransitGatewayRouteTableAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayRouteTableAssociations.html",
                "description": "Grants permission to get information about associations for a transit gateway route table",
                "accessLevel": "List"
            },
            "GetTransitGatewayRouteTablePropagations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayRouteTablePropagations.html",
                "description": "Grants permission to get information about the route table propagations for a transit gateway route table",
                "accessLevel": "List"
            },
            "ImportClientVpnClientCertificateRevocationList": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportClientVpnClientCertificateRevocationList.html",
                "description": "Grants permission to upload a client certificate revocation list to a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ImportImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportImage.html",
                "description": "Grants permission to import single or multi-volume disk images or EBS snapshots into an Amazon Machine Image (AMI)",
                "accessLevel": "Write"
            },
            "ImportInstance": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportInstance.html",
                "description": "Grants permission to create an import instance task using metadata from a disk image",
                "accessLevel": "Write"
            },
            "ImportKeyPair": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportKeyPair.html",
                "description": "Grants permission to import a public key from an RSA key pair that was created with a third-party tool",
                "accessLevel": "Write"
            },
            "ImportSnapshot": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportSnapshot.html",
                "description": "Grants permission to import a disk into an EBS snapshot",
                "accessLevel": "Write"
            },
            "ImportVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportVolume.html",
                "description": "Grants permission to create an import volume task using metadata from a disk image",
                "accessLevel": "Write"
            },
            "ModifyCapacityReservation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyCapacityReservation.html",
                "description": "Grants permission to modify a Capacity Reservation's capacity and the conditions under which it is to be released",
                "accessLevel": "Write",
                "resourceTypes": {
                    "capacity-reservation": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyClientVpnEndpoint": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyClientVpnEndpoint.html",
                "description": "Grants permission to modify a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyDefaultCreditSpecification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyDefaultCreditSpecification.html",
                "description": "Grants permission to change the account level default credit option for CPU usage of burstable performance instances",
                "accessLevel": "Write"
            },
            "ModifyEbsDefaultKmsKeyId": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyEbsDefaultKmsKeyId.html",
                "description": "Grants permission to change the default customer master key (CMK) for EBS encryption by default for your account",
                "accessLevel": "Write"
            },
            "ModifyFleet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyFleet.html",
                "description": "Grants permission to modify an EC2 Fleet",
                "accessLevel": "Write"
            },
            "ModifyFpgaImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyFpgaImageAttribute.html",
                "description": "Grants permission to modify an attribute of an Amazon FPGA Image (AFI)",
                "accessLevel": "Write"
            },
            "ModifyHosts": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyHosts.html",
                "description": "Grants permission to modify a Dedicated Host",
                "accessLevel": "Write"
            },
            "ModifyIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyIdFormat.html",
                "description": "Grants permission to modify the ID format for a resource",
                "accessLevel": "Write"
            },
            "ModifyIdentityIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyIdentityIdFormat.html",
                "description": "Grants permission to modify the ID format of a resource for a specific principal in your account",
                "accessLevel": "Write"
            },
            "ModifyImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyImageAttribute.html",
                "description": "Grants permission to modify an attribute of an Amazon Machine Image (AMI)",
                "accessLevel": "Write"
            },
            "ModifyInstanceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceAttribute.html",
                "description": "Grants permission to modify an attribute of an instance",
                "accessLevel": "Write"
            },
            "ModifyInstanceCapacityReservationAttributes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCapacityReservationAttributes.html",
                "description": "Grants permission to modify the Capacity Reservation settings for a stopped instance",
                "accessLevel": "Write"
            },
            "ModifyInstanceCreditSpecification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCreditSpecification.html",
                "description": "Grants permission to modify the credit option for CPU usage on an instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true
                    }
                }
            },
            "ModifyInstanceEventStartTime": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceEventStartTime.html",
                "description": "Grants permission to modify the start time for a scheduled EC2 instance event",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "ec2:Region"
                        ]
                    }
                }
            },
            "ModifyInstanceMetadataOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceMetadataOptions.html",
                "description": "Grants permission to modify the metadata options for an instance",
                "accessLevel": "Write"
            },
            "ModifyInstancePlacement": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstancePlacement.html",
                "description": "Grants permission to modify the placement attributes for an instance",
                "accessLevel": "Write"
            },
            "ModifyLaunchTemplate": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyLaunchTemplate.html",
                "description": "Grants permission to modify a launch template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launch-template": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyManagedPrefixList": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyManagedPrefixList.html",
                "description": "Grants permission to modify a managed prefix list",
                "accessLevel": "Write",
                "resourceTypes": {
                    "prefix-list": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyNetworkInterfaceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyNetworkInterfaceAttribute.html",
                "description": "Grants permission to modify an attribute of a network interface",
                "accessLevel": "Write"
            },
            "ModifyReservedInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyReservedInstances.html",
                "description": "Grants permission to modify attributes of one or more Reserved Instances",
                "accessLevel": "Write"
            },
            "ModifySnapshotAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySnapshotAttribute.html",
                "description": "Grants permission to add or remove permission settings for a snapshot",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:SnapshotTime",
                            "ec2:VolumeSize"
                        ]
                    }
                }
            },
            "ModifySpotFleetRequest": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySpotFleetRequest.html",
                "description": "Grants permission to modify a Spot Fleet request",
                "accessLevel": "Write"
            },
            "ModifySubnetAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySubnetAttribute.html",
                "description": "Grants permission to modify an attribute of a subnet",
                "accessLevel": "Write"
            },
            "ModifyTrafficMirrorFilterNetworkServices": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorFilterNetworkServices.html",
                "description": "Grants permission to allow or restrict mirroring network services",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyTrafficMirrorFilterRule": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorFilterRule.html",
                "description": "Grants permission to modify a traffic mirror rule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-filter-rule": {
                        "required": true,
                        "conditions": [
                            "ec2:Region"
                        ]
                    }
                }
            },
            "ModifyTrafficMirrorSession": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorSession.html",
                "description": "Grants permission to modify a traffic mirror session",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-session": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-filter": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-target": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyTransitGatewayVpcAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTransitGatewayVpcAttachment.html",
                "description": "Grants permission to modify a VPC attachment on a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVolume.html",
                "description": "Grants permission to modify the parameters of an EBS volume",
                "accessLevel": "Write"
            },
            "ModifyVolumeAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVolumeAttribute.html",
                "description": "Grants permission to modify an attribute of a volume",
                "accessLevel": "Write"
            },
            "ModifyVpcAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcAttribute.html",
                "description": "Grants permission to modify an attribute of a VPC",
                "accessLevel": "Write"
            },
            "ModifyVpcEndpoint": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpoint.html",
                "description": "Grants permission to modify an attribute of a VPC endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "route-table": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "security-group": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyVpcEndpointConnectionNotification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointConnectionNotification.html",
                "description": "Grants permission to modify a connection notification for a VPC endpoint or VPC endpoint service",
                "accessLevel": "Write"
            },
            "ModifyVpcEndpointServiceConfiguration": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointServiceConfiguration.html",
                "description": "Grants permission to modify the attributes of a VPC endpoint service configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:VpceServicePrivateDnsName",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyVpcEndpointServicePermissions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointServicePermissions.html",
                "description": "Grants permission to modify the permissions for a VPC endpoint service",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyVpcPeeringConnectionOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcPeeringConnectionOptions.html",
                "description": "Grants permission to modify the VPC peering connection options on one side of a VPC peering connection",
                "accessLevel": "Write"
            },
            "ModifyVpcTenancy": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcTenancy.html",
                "description": "Grants permission to modify the instance tenancy attribute of a VPC",
                "accessLevel": "Write"
            },
            "ModifyVpnConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnConnection.html",
                "description": "Grants permission to modify the target gateway of a Site-to-Site VPN connection",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpn-connection": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:GatewayType"
                        ]
                    }
                }
            },
            "ModifyVpnTunnelCertificate": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnTunnelCertificate",
                "description": "Grants permission to modify the certificate for a Site-to-Site VPN connection",
                "accessLevel": "Write"
            },
            "ModifyVpnTunnelOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnTunnelOptions.html",
                "description": "Grants permission to modify the options for a Site-to-Site VPN connection",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpn-connection": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:AuthenticationType",
                            "ec2:DPDTimeoutSeconds",
                            "ec2:IKEVersions",
                            "ec2:InsideTunnelCidr",
                            "ec2:Phase1DHGroupNumbers",
                            "ec2:Phase2DHGroupNumbers",
                            "ec2:Phase1EncryptionAlgorithms",
                            "ec2:Phase2EncryptionAlgorithms",
                            "ec2:Phase1IntegrityAlgorithms",
                            "ec2:Phase2IntegrityAlgorithms",
                            "ec2:Phase1LifetimeSeconds",
                            "ec2:Phase2LifetimeSeconds",
                            "ec2:PresharedKeys",
                            "ec2:RekeyFuzzPercentage",
                            "ec2:RekeyMarginTimeSeconds",
                            "ec2:RoutingType"
                        ]
                    }
                }
            },
            "MonitorInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_MonitorInstances.html",
                "description": "Grants permission to enable detailed monitoring for a running instance",
                "accessLevel": "Write"
            },
            "MoveAddressToVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_MoveAddressToVpc.html",
                "description": "Grants permission to move an Elastic IP address from the EC2-Classic platform to the EC2-VPC platform",
                "accessLevel": "Write"
            },
            "ProvisionByoipCidr": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ProvisionByoipCidr.html",
                "description": "Grants permission to provision an address range for use in AWS through bring your own IP addresses (BYOIP), and to create a corresponding address pool",
                "accessLevel": "Write"
            },
            "PurchaseHostReservation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseHostReservation.html",
                "description": "Grants permission to purchase a reservation with configurations that match those of a Dedicated Host",
                "accessLevel": "Write"
            },
            "PurchaseReservedInstancesOffering": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseReservedInstancesOffering.html",
                "description": "Grants permission to purchase a Reserved Instance offering",
                "accessLevel": "Write"
            },
            "PurchaseScheduledInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseScheduledInstances.html",
                "description": "Grants permission to purchase one or more Scheduled Instances with a specified schedule",
                "accessLevel": "Write"
            },
            "RebootInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RebootInstances.html",
                "description": "Grants permission to request a reboot of one or more instances",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "RegisterImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterImage.html",
                "description": "Grants permission to register an Amazon Machine Image (AMI)",
                "accessLevel": "Write"
            },
            "RegisterInstanceEventNotificationAttributes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterInstanceEventNotificationAttributes.html",
                "description": "Grants permission to add tags to the set of tags to include in notifications about scheduled events for your instances",
                "accessLevel": "Write"
            },
            "RegisterTransitGatewayMulticastGroupMembers": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterTransitGatewayMulticastGroupMembers.html",
                "description": "Grants permission to register one or more network interfaces as a member of a group IP address in a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RegisterTransitGatewayMulticastGroupSources": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterTransitGatewayMulticastGroupSources.html",
                "description": "Grants permission to register one or more network interfaces as a source of a group IP address in a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RejectTransitGatewayPeeringAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectTransitGatewayPeeringAttachment.html",
                "description": "Grants permission to reject a transit gateway peering attachment request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RejectTransitGatewayVpcAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectTransitGatewayVpcAttachment.html",
                "description": "Grants permission to reject a request to attach a VPC to a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RejectVpcEndpointConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectVpcEndpointConnections.html",
                "description": "Grants permission to reject one or more VPC endpoint connection requests to a VPC endpoint service",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RejectVpcPeeringConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectVpcPeeringConnection.html",
                "description": "Grants permission to reject a VPC peering connection request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-peering-connection": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AccepterVpc",
                            "ec2:Region",
                            "ec2:RequesterVpc",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ReleaseAddress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReleaseAddress.html",
                "description": "Grants permission to release an Elastic IP address",
                "accessLevel": "Write"
            },
            "ReleaseHosts": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReleaseHosts.html",
                "description": "Grants permission to release one or more On-Demand Dedicated Hosts",
                "accessLevel": "Write"
            },
            "ReplaceIamInstanceProfileAssociation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceIamInstanceProfileAssociation.html",
                "description": "Grants permission to replace an IAM instance profile for an instance",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "ReplaceNetworkAclAssociation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceNetworkAclAssociation.html",
                "description": "Grants permission to change which network ACL a subnet is associated with",
                "accessLevel": "Write"
            },
            "ReplaceNetworkAclEntry": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceNetworkAclEntry.html",
                "description": "Grants permission to replace an entry (rule) in a network ACL",
                "accessLevel": "Write"
            },
            "ReplaceRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceRoute.html",
                "description": "Grants permission to replace a route within a route table in a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "ReplaceRouteTableAssociation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceRouteTableAssociation.html",
                "description": "Grants permission to change the route table that is associated with a subnet",
                "accessLevel": "Write"
            },
            "ReplaceTransitGatewayRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceTransitGatewayRoute.html",
                "description": "Grants permission to replace a route in a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ReportInstanceStatus": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReportInstanceStatus.html",
                "description": "Grants permission to submit feedback about the status of an instance",
                "accessLevel": "Write"
            },
            "RequestSpotFleet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotFleet.html",
                "description": "Grants permission to create a Spot Fleet request",
                "accessLevel": "Write"
            },
            "RequestSpotInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotInstances.html",
                "description": "Grants permission to create a Spot Instance request",
                "accessLevel": "Write"
            },
            "ResetEbsDefaultKmsKeyId": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetEbsDefaultKmsKeyId.html",
                "description": "Grants permission to reset the default customer master key (CMK) for EBS encryption for your account to use the AWS-managed CMK for EBS",
                "accessLevel": "Write"
            },
            "ResetFpgaImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetFpgaImageAttribute.html",
                "description": "Grants permission to reset an attribute of an Amazon FPGA Image (AFI) to its default value",
                "accessLevel": "Write"
            },
            "ResetImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetImageAttribute.html",
                "description": "Grants permission to reset an attribute of an Amazon Machine Image (AMI) to its default value",
                "accessLevel": "Write"
            },
            "ResetInstanceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetInstanceAttribute.html",
                "description": "Grants permission to reset an attribute of an instance to its default value",
                "accessLevel": "Write"
            },
            "ResetNetworkInterfaceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetNetworkInterfaceAttribute.html",
                "description": "Grants permission to reset an attribute of a network interface",
                "accessLevel": "Write"
            },
            "ResetSnapshotAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetSnapshotAttribute.html",
                "description": "Grants permission to reset permission settings for a snapshot",
                "accessLevel": "Permissions management"
            },
            "RestoreAddressToClassic": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RestoreAddressToClassic.html",
                "description": "Grants permission to restore an Elastic IP address that was previously moved to the EC2-VPC platform back to the EC2-Classic platform",
                "accessLevel": "Write"
            },
            "RestoreManagedPrefixListVersion": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RestoreManagedPrefixListVersion.html",
                "description": "Grants permission to restore the entries from a previous version of a managed prefix list to a new version of the prefix list",
                "accessLevel": "Write",
                "resourceTypes": {
                    "prefix-list": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RevokeClientVpnIngress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeClientVpnIngress.html",
                "description": "Grants permission to remove an inbound authorization rule from a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RevokeSecurityGroupEgress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeSecurityGroupEgress.html",
                "description": "Grants permission to remove one or more outbound rules from a VPC security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "RevokeSecurityGroupIngress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeSecurityGroupIngress.html",
                "description": "Grants permission to remove one or more inbound rules from a security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "RunInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html",
                "description": "Grants permission to launch one or more instances",
                "accessLevel": "Write",
                "resourceTypes": {
                    "image": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:ImageType",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Owner",
                            "ec2:Public",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType"
                        ]
                    },
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy",
                            "ec2:MetadataHttpEndpoint",
                            "ec2:MetadataHttpTokens",
                            "ec2:MetadataHttpPutResponseHopLimit"
                        ]
                    },
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/",
                            "ec2:AvailabilityZone",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Region",
                            "ec2:ResourceTag/",
                            "ec2:Subnet",
                            "ec2:Vpc",
                            "ec2:AssociatePublicIpAddress"
                        ]
                    },
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    },
                    "capacity-reservation": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "elastic-gpu": {
                        "required": false,
                        "conditions": [
                            "ec2:ElasticGpuType"
                        ]
                    },
                    "elastic-inference": {
                        "required": false
                    },
                    "key-pair": {
                        "required": false,
                        "conditions": [
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Region"
                        ]
                    },
                    "launch-template": {
                        "required": false,
                        "conditions": [
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Region"
                        ]
                    },
                    "placement-group": {
                        "required": false,
                        "conditions": [
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:PlacementGroupStrategy",
                            "ec2:Region"
                        ]
                    },
                    "snapshot": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:SnapshotTime",
                            "ec2:VolumeSize"
                        ]
                    }
                }
            },
            "RunScheduledInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunScheduledInstances.html",
                "description": "Grants permission to launch one or more Scheduled Instances",
                "accessLevel": "Write"
            },
            "SearchLocalGatewayRoutes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchLocalGatewayRoutes.html",
                "description": "Grants permission to search for routes in a local gateway route table",
                "accessLevel": "List"
            },
            "SearchTransitGatewayMulticastGroups": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchTransitGatewayMulticastGroups.html",
                "description": "Grants permission to search for groups, sources, and members in a transit gateway multicast domain",
                "accessLevel": "List"
            },
            "SearchTransitGatewayRoutes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchTransitGatewayRoutes.html",
                "description": "Grants permission to search for routes in a transit gateway route table",
                "accessLevel": "List"
            },
            "SendDiagnosticInterrupt": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SendDiagnosticInterrupt.html",
                "description": "Grants permission to send a diagnostic interrupt to an Amazon EC2 instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "StartInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StartInstances.html",
                "description": "Grants permission to start a stopped instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "StartVpcEndpointServicePrivateDnsVerification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StartVpcEndpointServicePrivateDnsVerification.html",
                "description": "Grants permission to start the private DNS verification process for a VPC endpoint service",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "StopInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StopInstances.html",
                "description": "Grants permission to stop an Amazon EBS-backed instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "TerminateClientVpnConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_TerminateClientVpnConnections.html",
                "description": "Grants permission to terminate active Client VPN endpoint connections",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "TerminateInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_TerminateInstances.html",
                "description": "Grants permission to shut down one or more instances",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "UnassignIpv6Addresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnassignIpv6Addresses.html",
                "description": "Grants permission to unassign one or more IPv6 addresses from a network interface",
                "accessLevel": "Write"
            },
            "UnassignPrivateIpAddresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnassignPrivateIpAddresses.html",
                "description": "Grants permission to unassign one or more secondary private IP addresses from a network interface",
                "accessLevel": "Write"
            },
            "UnmonitorInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnmonitorInstances.html",
                "description": "Grants permission to disable detailed monitoring for a running instance",
                "accessLevel": "Write"
            },
            "UpdateSecurityGroupRuleDescriptionsEgress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UpdateSecurityGroupRuleDescriptionsEgress.html",
                "description": "Grants permission to update descriptions for one or more outbound rules in a VPC security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "UpdateSecurityGroupRuleDescriptionsIngress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UpdateSecurityGroupRuleDescriptionsIngress.html",
                "description": "Grants permission to update descriptions for one or more inbound rules in a security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "WithdrawByoipCidr": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_WithdrawByoipCidr.html",
                "description": "Grants permission to stop advertising an address range that was provisioned for use in AWS through bring your own IP addresses (BYOIP)",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "capacity-reservation": {
                "name": "capacity-reservation",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-capacity-reservations.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:capacity-reservation/${CapacityReservationId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "client-vpn-endpoint": {
                "name": "client-vpn-endpoint",
                "url": "https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/what-is.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:client-vpn-endpoint/${ClientVpnEndpointId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "customer-gateway": {
                "name": "customer-gateway",
                "url": "https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:customer-gateway/${CustomerGatewayId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "dedicated-host": {
                "name": "dedicated-host",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/dedicated-hosts-overview.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:dedicated-host/${HostId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AutoPlacement",
                    "ec2:AvailabilityZone",
                    "ec2:HostRecovery",
                    "ec2:InstanceType",
                    "ec2:Quantity",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "dhcp-options": {
                "name": "dhcp-options",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/VPC_DHCP_Options.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:dhcp-options/${DhcpOptionsId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "egress-only-internet-gateway": {
                "name": "egress-only-internet-gateway",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/egress-only-internet-gateway.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:egress-only-internet-gateway/${EgressOnlyInternetGatewayId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "elastic-gpu": {
                "name": "elastic-gpu",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/elastic-gpus.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:elastic-gpu/${ElasticGpuId}",
                "conditionKeys": [
                    "ec2:ElasticGpuType"
                ]
            },
            "elastic-inference": {
                "name": "elastic-inference",
                "url": "https://docs.aws.amazon.com/elastic-inference/latest/developerguide/what-is-ei.html",
                "arn": "arn:${Partition}:elastic-inference:${Region}:${Account}:elastic-inference-accelerator/${ElasticInferenceAcceleratorId}",
                "conditionKeys": []
            },
            "export-instance-task": {
                "name": "export-instance-task",
                "url": "https://docs.aws.amazon.com/vm-import/latest/userguide/vmexport.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:export-instance-task/${ExportTaskId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "fleet": {
                "name": "fleet",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:fleet/${FleetId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "fpga-image": {
                "name": "fpga-image",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#EC2_ARN_Format",
                "arn": "arn:${Partition}:ec2:${Region}::fpga-image/${FpgaImageId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Owner",
                    "ec2:Public",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "image": {
                "name": "image",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/AMIs.html",
                "arn": "arn:${Partition}:ec2:${Region}::image/${ImageId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:ImageType",
                    "ec2:Owner",
                    "ec2:Public",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:RootDeviceType"
                ]
            },
            "instance": {
                "name": "instance",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Instances.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:instance/${InstanceId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AvailabilityZone",
                    "ec2:EbsOptimized",
                    "ec2:InstanceProfile",
                    "ec2:InstanceType",
                    "ec2:PlacementGroup",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:RootDeviceType",
                    "ec2:Tenancy"
                ]
            },
            "internet-gateway": {
                "name": "internet-gateway",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Internet_Gateway.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:internet-gateway/${InternetGatewayId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "key-pair": {
                "name": "key-pair",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-key-pairs.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:key-pair/${KeyPairName}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "launch-template": {
                "name": "launch-template",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:launch-template/${LaunchTemplateId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway": {
                "name": "local-gateway",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway/${LocalGatewayId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway-route-table": {
                "name": "local-gateway-route-table",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table/${LocalGatewayRouteTableId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway-route-table-virtual-interface-group-association": {
                "name": "local-gateway-route-table-virtual-interface-group-association",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table-virtual-interface-group-association/${LocalGatewayRouteTableVirtualInterfaceGroupAssociationId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway-route-table-vpc-association": {
                "name": "local-gateway-route-table-vpc-association",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table-vpc-association/${LocalGatewayRouteTableVpcAssociationId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway-virtual-interface": {
                "name": "local-gateway-virtual-interface",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway-virtual-interface/${LocalGatewayVirtualInterfaceId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway-virtual-interface-group": {
                "name": "local-gateway-virtual-interface-group",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway-virtual-interface-group/${LocalGatewayVirtualInterfaceGroupId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "natgateway": {
                "name": "natgateway",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/vpc-nat-gateway.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:natgateway/${NatGatewayId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "network-acl": {
                "name": "network-acl",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/vpc-network-acls.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:network-acl/${NaclId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Vpc"
                ]
            },
            "network-interface": {
                "name": "network-interface",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:network-interface/${NetworkInterfaceId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AssociatePublicIpAddress",
                    "ec2:AuthorizedService",
                    "ec2:AvailabilityZone",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Subnet",
                    "ec2:Vpc"
                ]
            },
            "placement-group": {
                "name": "placement-group",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:placement-group/${PlacementGroupName}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:PlacementGroupStrategy",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "prefix-list": {
                "name": "prefix-list",
                "url": "",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:prefix-list/${PrefixListId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "reserved-instances": {
                "name": "reserved-instances",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-reserved-instances.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:reserved-instances/${ReservationId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AvailabilityZone",
                    "ec2:InstanceType",
                    "ec2:Region",
                    "ec2:ReservedInstancesOfferingType",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Tenancy"
                ]
            },
            "route-table": {
                "name": "route-table",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Route_Tables.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:route-table/${RouteTableId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Vpc"
                ]
            },
            "security-group": {
                "name": "security-group",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-security-groups.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:security-group/${SecurityGroupId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Vpc"
                ]
            },
            "snapshot": {
                "name": "snapshot",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSSnapshots.html",
                "arn": "arn:${Partition}:ec2:${Region}::snapshot/${SnapshotId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Owner",
                    "ec2:ParentVolume",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:SnapshotTime",
                    "ec2:VolumeSize"
                ]
            },
            "spot-instance-request": {
                "name": "spot-instance-request",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-spot-instances.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:spot-instances-request/${SpotInstanceRequestId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "subnet": {
                "name": "subnet",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:subnet/${SubnetId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AvailabilityZone",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Vpc"
                ]
            },
            "traffic-mirror-session": {
                "name": "traffic-mirror-session",
                "url": "https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-session.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-session/${TrafficMirrorSessionId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "traffic-mirror-target": {
                "name": "traffic-mirror-target",
                "url": "https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-target.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-target/${TrafficMirrorTargetId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "traffic-mirror-filter": {
                "name": "traffic-mirror-filter",
                "url": "https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-filter.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-filter/${TrafficMirrorFilterId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "traffic-mirror-filter-rule": {
                "name": "traffic-mirror-filter-rule",
                "url": "https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-filter.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-filter-rule/${TrafficMirrorFilterRuleId}",
                "conditionKeys": [
                    "ec2:Region"
                ]
            },
            "transit-gateway-attachment": {
                "name": "transit-gateway-attachment",
                "url": "https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-attachment/${TransitGatewayAttachmentId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "transit-gateway-multicast-domain": {
                "name": "transit-gateway-multicast-domain",
                "url": "https://docs.aws.amazon.com/vpc/latest/tgw/tgw-multicast-overview.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-multicast-domain/${TransitGatewayMulticastDomainId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "transit-gateway-route-table": {
                "name": "transit-gateway-route-table",
                "url": "https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-route-table/${TransitGatewayRouteTableId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "transit-gateway": {
                "name": "transit-gateway",
                "url": "https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:transit-gateway/${TransitGatewayId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "volume": {
                "name": "volume",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volumes.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:volume/${VolumeId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AvailabilityZone",
                    "ec2:Encrypted",
                    "ec2:ParentSnapshot",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:VolumeIops",
                    "ec2:VolumeSize",
                    "ec2:VolumeType"
                ]
            },
            "vpc": {
                "name": "vpc",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpc/${VpcId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Tenancy"
                ]
            },
            "vpc-endpoint": {
                "name": "vpc-endpoint",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-services-overview.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpc-endpoint/${VpceId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:VpceServiceName",
                    "ec2:VpceServiceOwner"
                ]
            },
            "vpc-endpoint-service": {
                "name": "vpc-endpoint-service",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-services-overview.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpc-endpoint-service/${VpceServiceId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:VpceServicePrivateDnsName"
                ]
            },
            "vpc-flow-log": {
                "name": "vpc-flow-log",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpc-flow-log/${VpcFlowLogId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "vpc-peering-connection": {
                "name": "vpc-peering-connection",
                "url": "https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpc-peering-connection/${VpcPeeringConnectionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "ec2:AccepterVpc",
                    "ec2:Region",
                    "ec2:RequesterVpc",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "vpn-connection": {
                "name": "vpn-connection",
                "url": "https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpn-connection/${VpnConnectionId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AuthenticationType",
                    "ec2:DPDTimeoutSeconds",
                    "ec2:GatewayType",
                    "ec2:IKEVersions",
                    "ec2:InsideTunnelCidr",
                    "ec2:Phase1DHGroupNumbers",
                    "ec2:Phase1EncryptionAlgorithms",
                    "ec2:Phase1IntegrityAlgorithms",
                    "ec2:Phase1LifetimeSeconds",
                    "ec2:Phase2DHGroupNumbers",
                    "ec2:Phase2EncryptionAlgorithms",
                    "ec2:Phase2IntegrityAlgorithms",
                    "ec2:Phase2LifetimeSeconds",
                    "ec2:PresharedKeys",
                    "ec2:Region",
                    "ec2:RekeyFuzzPercentage",
                    "ec2:RekeyMarginTimeSeconds",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:RoutingType"
                ]
            },
            "vpn-gateway": {
                "name": "vpn-gateway",
                "url": "https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpn-gateway/${VpnGatewayId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to accept a Convertible Reserved Instance exchange quote
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptReservedInstancesExchangeQuote.html
     */
    acceptReservedInstancesExchangeQuote() {
        this.add('ec2:AcceptReservedInstancesExchangeQuote');
        return this;
    }
    /**
     * Grants permission to accept a transit gateway peering attachment request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptTransitGatewayPeeringAttachment.html
     */
    acceptTransitGatewayPeeringAttachment() {
        this.add('ec2:AcceptTransitGatewayPeeringAttachment');
        return this;
    }
    /**
     * Grants permission to accept a request to attach a VPC to a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptTransitGatewayVpcAttachment.html
     */
    acceptTransitGatewayVpcAttachment() {
        this.add('ec2:AcceptTransitGatewayVpcAttachment');
        return this;
    }
    /**
     * Grants permission to accept one or more interface VPC endpoint connections to your VPC endpoint service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptVpcEndpointConnections.html
     */
    acceptVpcEndpointConnections() {
        this.add('ec2:AcceptVpcEndpointConnections');
        return this;
    }
    /**
     * Grants permission to accept a VPC peering connection request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptVpcPeeringConnection.html
     */
    acceptVpcPeeringConnection() {
        this.add('ec2:AcceptVpcPeeringConnection');
        return this;
    }
    /**
     * Grants permission to advertise an IP address range that is provisioned for use in AWS through bring your own IP addresses (BYOIP)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AdvertiseByoipCidr.html
     */
    advertiseByoipCidr() {
        this.add('ec2:AdvertiseByoipCidr');
        return this;
    }
    /**
     * Grants permission to allocate an Elastic IP address (EIP) to your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AllocateAddress.html
     */
    allocateAddress() {
        this.add('ec2:AllocateAddress');
        return this;
    }
    /**
     * Grants permission to allocate a Dedicated Host to your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AllocateHosts.html
     */
    allocateHosts() {
        this.add('ec2:AllocateHosts');
        return this;
    }
    /**
     * Grants permission to apply a security group to the association between a Client VPN endpoint and a target network
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ApplySecurityGroupsToClientVpnTargetNetwork.html
     */
    applySecurityGroupsToClientVpnTargetNetwork() {
        this.add('ec2:ApplySecurityGroupsToClientVpnTargetNetwork');
        return this;
    }
    /**
     * Grants permission to assign one or more IPv6 addresses to a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssignIpv6Addresses.html
     */
    assignIpv6Addresses() {
        this.add('ec2:AssignIpv6Addresses');
        return this;
    }
    /**
     * Grants permission to assign one or more secondary private IP addresses to a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssignPrivateIpAddresses.html
     */
    assignPrivateIpAddresses() {
        this.add('ec2:AssignPrivateIpAddresses');
        return this;
    }
    /**
     * Grants permission to associate an Elastic IP address (EIP) with an instance or a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateAddress.html
     */
    associateAddress() {
        this.add('ec2:AssociateAddress');
        return this;
    }
    /**
     * Grants permission to associate a target network with a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateClientVpnTargetNetwork.html
     */
    associateClientVpnTargetNetwork() {
        this.add('ec2:AssociateClientVpnTargetNetwork');
        return this;
    }
    /**
     * Grants permission to associate or disassociate a set of DHCP options with a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateDhcpOptions.html
     */
    associateDhcpOptions() {
        this.add('ec2:AssociateDhcpOptions');
        return this;
    }
    /**
     * Grants permission to associate an IAM instance profile with a running or stopped instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateIamInstanceProfile.html
     */
    associateIamInstanceProfile() {
        this.add('ec2:AssociateIamInstanceProfile');
        return this;
    }
    /**
     * Grants permission to associate a subnet or gateway with a route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateRouteTable.html
     */
    associateRouteTable() {
        this.add('ec2:AssociateRouteTable');
        return this;
    }
    /**
     * Grants permission to associate a CIDR block with a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateSubnetCidrBlock.html
     */
    associateSubnetCidrBlock() {
        this.add('ec2:AssociateSubnetCidrBlock');
        return this;
    }
    /**
     * Grants permission to associate an attachment and list of subnets with a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateTransitGatewayMulticastDomain.html
     */
    associateTransitGatewayMulticastDomain() {
        this.add('ec2:AssociateTransitGatewayMulticastDomain');
        return this;
    }
    /**
     * Grants permission to associate an attachment with a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateTransitGatewayRouteTable.html
     */
    associateTransitGatewayRouteTable() {
        this.add('ec2:AssociateTransitGatewayRouteTable');
        return this;
    }
    /**
     * Grants permission to associate a CIDR block with a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateVpcCidrBlock.html
     */
    associateVpcCidrBlock() {
        this.add('ec2:AssociateVpcCidrBlock');
        return this;
    }
    /**
     * Grants permission to link an EC2-Classic instance to a ClassicLink-enabled VPC through one or more of the VPC's security groups
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachClassicLinkVpc.html
     */
    attachClassicLinkVpc() {
        this.add('ec2:AttachClassicLinkVpc');
        return this;
    }
    /**
     * Grants permission to attach an internet gateway to a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachInternetGateway.html
     */
    attachInternetGateway() {
        this.add('ec2:AttachInternetGateway');
        return this;
    }
    /**
     * Grants permission to attach a network interface to an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachNetworkInterface.html
     */
    attachNetworkInterface() {
        this.add('ec2:AttachNetworkInterface');
        return this;
    }
    /**
     * Grants permission to attach an EBS volume to a running or stopped instance and expose it to the instance with the specified device name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachVolume.html
     */
    attachVolume() {
        this.add('ec2:AttachVolume');
        return this;
    }
    /**
     * Grants permission to attach a virtual private gateway to a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachVpnGateway.html
     */
    attachVpnGateway() {
        this.add('ec2:AttachVpnGateway');
        return this;
    }
    /**
     * Grants permission to add an inbound authorization rule to a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeClientVpnIngress.html
     */
    authorizeClientVpnIngress() {
        this.add('ec2:AuthorizeClientVpnIngress');
        return this;
    }
    /**
     * Grants permission to add one or more outbound rules to a VPC security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeSecurityGroupEgress.html
     */
    authorizeSecurityGroupEgress() {
        this.add('ec2:AuthorizeSecurityGroupEgress');
        return this;
    }
    /**
     * Grants permission to add one or more inbound rules to a security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeSecurityGroupIngress.html
     */
    authorizeSecurityGroupIngress() {
        this.add('ec2:AuthorizeSecurityGroupIngress');
        return this;
    }
    /**
     * Grants permission to bundle an instance store-backed Windows instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_BundleInstance.html
     */
    bundleInstance() {
        this.add('ec2:BundleInstance');
        return this;
    }
    /**
     * Grants permission to cancel a bundling operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelBundleTask.html
     */
    cancelBundleTask() {
        this.add('ec2:CancelBundleTask');
        return this;
    }
    /**
     * Grants permission to cancel a Capacity Reservation and release the reserved capacity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelCapacityReservation.html
     */
    cancelCapacityReservation() {
        this.add('ec2:CancelCapacityReservation');
        return this;
    }
    /**
     * Grants permission to cancel an active conversion task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelConversionTask.html
     */
    cancelConversionTask() {
        this.add('ec2:CancelConversionTask');
        return this;
    }
    /**
     * Grants permission to cancel an active export task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelExportTask.html
     */
    cancelExportTask() {
        this.add('ec2:CancelExportTask');
        return this;
    }
    /**
     * Grants permission to cancel an in-process import virtual machine or import snapshot task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelImportTask.html
     */
    cancelImportTask() {
        this.add('ec2:CancelImportTask');
        return this;
    }
    /**
     * Grants permission to cancel a Reserved Instance listing on the Reserved Instance Marketplace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelReservedInstancesListing.html
     */
    cancelReservedInstancesListing() {
        this.add('ec2:CancelReservedInstancesListing');
        return this;
    }
    /**
     * Grants permission to cancel one or more Spot Fleet requests
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelSpotFleetRequests.html
     */
    cancelSpotFleetRequests() {
        this.add('ec2:CancelSpotFleetRequests');
        return this;
    }
    /**
     * Grants permission to cancel one or more Spot Instance requests
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelSpotInstanceRequests.html
     */
    cancelSpotInstanceRequests() {
        this.add('ec2:CancelSpotInstanceRequests');
        return this;
    }
    /**
     * Grants permission to determine whether an owned product code is associated with an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ConfirmProductInstance.html
     */
    confirmProductInstance() {
        this.add('ec2:ConfirmProductInstance');
        return this;
    }
    /**
     * Grants permission to copy a source Amazon FPGA image (AFI) to the current Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopyFpgaImage.html
     */
    copyFpgaImage() {
        this.add('ec2:CopyFpgaImage');
        return this;
    }
    /**
     * Grants permission to copy an Amazon Machine Image (AMI) from a source Region to the current Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopyImage.html
     */
    copyImage() {
        this.add('ec2:CopyImage');
        return this;
    }
    /**
     * Grants permission to copy a point-in-time snapshot of an EBS volume and store it in Amazon S3
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopySnapshot.html
     */
    copySnapshot() {
        this.add('ec2:CopySnapshot');
        return this;
    }
    /**
     * Grants permission to create a Capacity Reservation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateCapacityReservation.html
     */
    createCapacityReservation() {
        this.add('ec2:CreateCapacityReservation');
        return this;
    }
    /**
     * Grants permission to create a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateClientVpnEndpoint.html
     */
    createClientVpnEndpoint() {
        this.add('ec2:CreateClientVpnEndpoint');
        return this;
    }
    /**
     * Grants permission to add a network route to a Client VPN endpoint's route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateClientVpnRoute.html
     */
    createClientVpnRoute() {
        this.add('ec2:CreateClientVpnRoute');
        return this;
    }
    /**
     * Grants permission to create a customer gateway, which provides information to AWS about your customer gateway device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateCustomerGateway.html
     */
    createCustomerGateway() {
        this.add('ec2:CreateCustomerGateway');
        return this;
    }
    /**
     * Grants permission to create a default subnet in a specified Availability Zone in a default VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDefaultSubnet.html
     */
    createDefaultSubnet() {
        this.add('ec2:CreateDefaultSubnet');
        return this;
    }
    /**
     * Grants permission to create a default VPC with a default subnet in each Availability Zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDefaultVpc.html
     */
    createDefaultVpc() {
        this.add('ec2:CreateDefaultVpc');
        return this;
    }
    /**
     * Grants permission to create a set of DHCP options for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDhcpOptions.html
     */
    createDhcpOptions() {
        this.add('ec2:CreateDhcpOptions');
        return this;
    }
    /**
     * Grants permission to create an egress-only internet gateway for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateEgressOnlyInternetGateway.html
     */
    createEgressOnlyInternetGateway() {
        this.add('ec2:CreateEgressOnlyInternetGateway');
        return this;
    }
    /**
     * Grants permission to launch an EC2 Fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet.html
     */
    createFleet() {
        this.add('ec2:CreateFleet');
        return this;
    }
    /**
     * Grants permission to create one or more flow logs to capture IP traffic for a network interface
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFlowLogs.html
     */
    createFlowLogs() {
        this.add('ec2:CreateFlowLogs');
        return this;
    }
    /**
     * Grants permission to create an Amazon FPGA Image (AFI) from a design checkpoint (DCP)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFpgaImage.html
     */
    createFpgaImage() {
        this.add('ec2:CreateFpgaImage');
        return this;
    }
    /**
     * Grants permission to create an Amazon EBS-backed AMI from a stopped or running Amazon EBS-backed instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateImage.html
     */
    createImage() {
        this.add('ec2:CreateImage');
        return this;
    }
    /**
     * Grants permission to export a running or stopped instance to an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateInstanceExportTask.html
     */
    createInstanceExportTask() {
        this.add('ec2:CreateInstanceExportTask');
        return this;
    }
    /**
     * Grants permission to create an internet gateway for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateInternetGateway.html
     */
    createInternetGateway() {
        this.add('ec2:CreateInternetGateway');
        return this;
    }
    /**
     * Grants permission to create a 2048-bit RSA key pair
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateKeyPair.html
     */
    createKeyPair() {
        this.add('ec2:CreateKeyPair');
        return this;
    }
    /**
     * Grants permission to create a launch template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLaunchTemplate.html
     */
    createLaunchTemplate() {
        this.add('ec2:CreateLaunchTemplate');
        return this;
    }
    /**
     * Grants permission to create a new version of a launch template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLaunchTemplateVersion.html
     */
    createLaunchTemplateVersion() {
        this.add('ec2:CreateLaunchTemplateVersion');
        return this;
    }
    /**
     * Grants permission to create a static route for a local gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLocalGatewayRoute.html
     */
    createLocalGatewayRoute() {
        this.add('ec2:CreateLocalGatewayRoute');
        return this;
    }
    /**
     * Grants permission to associate a VPC with a local gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLocalGatewayRouteTableVpcAssociation.html
     */
    createLocalGatewayRouteTableVpcAssociation() {
        this.add('ec2:CreateLocalGatewayRouteTableVpcAssociation');
        return this;
    }
    /**
     * Grants permission to create a managed prefix list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateManagedPrefixList.html
     */
    createManagedPrefixList() {
        this.add('ec2:CreateManagedPrefixList');
        return this;
    }
    /**
     * Grants permission to create a NAT gateway in a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNatGateway.html
     */
    createNatGateway() {
        this.add('ec2:CreateNatGateway');
        return this;
    }
    /**
     * Grants permission to create a network ACL in a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkAcl.html
     */
    createNetworkAcl() {
        this.add('ec2:CreateNetworkAcl');
        return this;
    }
    /**
     * Grants permission to create a numbered entry (a rule) in a network ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkAclEntry.html
     */
    createNetworkAclEntry() {
        this.add('ec2:CreateNetworkAclEntry');
        return this;
    }
    /**
     * Grants permission to create a network interface in a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkInterface.html
     */
    createNetworkInterface() {
        this.add('ec2:CreateNetworkInterface');
        return this;
    }
    /**
     * Grants permission to create a permission for an AWS-authorized user to perform certain operations on a network interface
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkInterfacePermission.html
     */
    createNetworkInterfacePermission() {
        this.add('ec2:CreateNetworkInterfacePermission');
        return this;
    }
    /**
     * Grants permission to create a placement group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreatePlacementGroup.html
     */
    createPlacementGroup() {
        this.add('ec2:CreatePlacementGroup');
        return this;
    }
    /**
     * Grants permission to create a listing for Standard Reserved Instances to be sold in the Reserved Instance Marketplace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateReservedInstancesListing.html
     */
    createReservedInstancesListing() {
        this.add('ec2:CreateReservedInstancesListing');
        return this;
    }
    /**
     * Grants permission to create a route in a VPC route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateRoute.html
     */
    createRoute() {
        this.add('ec2:CreateRoute');
        return this;
    }
    /**
     * Grants permission to create a route table for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateRouteTable.html
     */
    createRouteTable() {
        this.add('ec2:CreateRouteTable');
        return this;
    }
    /**
     * Grants permission to create a security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSecurityGroup.html
     */
    createSecurityGroup() {
        this.add('ec2:CreateSecurityGroup');
        return this;
    }
    /**
     * Grants permission to create a snapshot of an EBS volume and store it in Amazon S3
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSnapshot.html
     */
    createSnapshot() {
        this.add('ec2:CreateSnapshot');
        return this;
    }
    /**
     * Grants permission to create crash-consistent snapshots of multiple EBS volumes and store them in Amazon S3
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSnapshots.html
     */
    createSnapshots() {
        this.add('ec2:CreateSnapshots');
        return this;
    }
    /**
     * Grants permission to create a data feed for Spot Instances to view Spot Instance usage logs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSpotDatafeedSubscription.html
     */
    createSpotDatafeedSubscription() {
        this.add('ec2:CreateSpotDatafeedSubscription');
        return this;
    }
    /**
     * Grants permission to create a subnet in a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSubnet.html
     */
    createSubnet() {
        this.add('ec2:CreateSubnet');
        return this;
    }
    /**
     * Grants permission to add or overwrite one or more tags for Amazon EC2 resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifCreateAction()
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html
     */
    createTags() {
        this.add('ec2:CreateTags');
        return this;
    }
    /**
     * Grants permission to create a traffic mirror filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorFilter.html
     */
    createTrafficMirrorFilter() {
        this.add('ec2:CreateTrafficMirrorFilter');
        return this;
    }
    /**
     * Grants permission to create a traffic mirror filter rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorFilterRule.html
     */
    createTrafficMirrorFilterRule() {
        this.add('ec2:CreateTrafficMirrorFilterRule');
        return this;
    }
    /**
     * Grants permission to create a traffic mirror session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorSession.html
     */
    createTrafficMirrorSession() {
        this.add('ec2:CreateTrafficMirrorSession');
        return this;
    }
    /**
     * Grants permission to create a traffic mirror target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorTarget.html
     */
    createTrafficMirrorTarget() {
        this.add('ec2:CreateTrafficMirrorTarget');
        return this;
    }
    /**
     * Grants permission to create a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGateway.html
     */
    createTransitGateway() {
        this.add('ec2:CreateTransitGateway');
        return this;
    }
    /**
     * Grants permission to create a multicast domain for a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayMulticastDomain.html
     */
    createTransitGatewayMulticastDomain() {
        this.add('ec2:CreateTransitGatewayMulticastDomain');
        return this;
    }
    /**
     * Grants permission to request a transit gateway peering attachment between a requester and accepter transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayPeeringAttachment.html
     */
    createTransitGatewayPeeringAttachment() {
        this.add('ec2:CreateTransitGatewayPeeringAttachment');
        return this;
    }
    /**
     * Grants permission to create a static route for a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayRoute.html
     */
    createTransitGatewayRoute() {
        this.add('ec2:CreateTransitGatewayRoute');
        return this;
    }
    /**
     * Grants permission to create a route table for a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayRouteTable.html
     */
    createTransitGatewayRouteTable() {
        this.add('ec2:CreateTransitGatewayRouteTable');
        return this;
    }
    /**
     * Grants permission to attach a VPC to a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayVpcAttachment.html
     */
    createTransitGatewayVpcAttachment() {
        this.add('ec2:CreateTransitGatewayVpcAttachment');
        return this;
    }
    /**
     * Grants permission to create an EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html
     */
    createVolume() {
        this.add('ec2:CreateVolume');
        return this;
    }
    /**
     * Grants permission to create a VPC with a specified CIDR block
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpc.html
     */
    createVpc() {
        this.add('ec2:CreateVpc');
        return this;
    }
    /**
     * Grants permission to create a VPC endpoint for an AWS service
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - route53:AssociateVPCWithHostedZone
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpoint.html
     */
    createVpcEndpoint() {
        this.add('ec2:CreateVpcEndpoint');
        return this;
    }
    /**
     * Grants permission to create a connection notification for a VPC endpoint or VPC endpoint service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpointConnectionNotification.html
     */
    createVpcEndpointConnectionNotification() {
        this.add('ec2:CreateVpcEndpointConnectionNotification');
        return this;
    }
    /**
     * Grants permission to create a VPC endpoint service configuration to which service consumers (AWS accounts, IAM users, and IAM roles) can connect
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpointServiceConfiguration.html
     */
    createVpcEndpointServiceConfiguration() {
        this.add('ec2:CreateVpcEndpointServiceConfiguration');
        return this;
    }
    /**
     * Grants permission to request a VPC peering connection between two VPCs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcPeeringConnection.html
     */
    createVpcPeeringConnection() {
        this.add('ec2:CreateVpcPeeringConnection');
        return this;
    }
    /**
     * Grants permission to create a VPN connection between a virtual private gateway or transit gateway and a customer gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnConnection.html
     */
    createVpnConnection() {
        this.add('ec2:CreateVpnConnection');
        return this;
    }
    /**
     * Grants permission to create a static route for a VPN connection between a virtual private gateway and a customer gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnConnectionRoute.html
     */
    createVpnConnectionRoute() {
        this.add('ec2:CreateVpnConnectionRoute');
        return this;
    }
    /**
     * Grants permission to create a virtual private gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnGateway.html
     */
    createVpnGateway() {
        this.add('ec2:CreateVpnGateway');
        return this;
    }
    /**
     * Grants permission to delete a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteClientVpnEndpoint.html
     */
    deleteClientVpnEndpoint() {
        this.add('ec2:DeleteClientVpnEndpoint');
        return this;
    }
    /**
     * Grants permission to delete a route from a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteClientVpnRoute.html
     */
    deleteClientVpnRoute() {
        this.add('ec2:DeleteClientVpnRoute');
        return this;
    }
    /**
     * Grants permission to delete a customer gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteCustomerGateway.html
     */
    deleteCustomerGateway() {
        this.add('ec2:DeleteCustomerGateway');
        return this;
    }
    /**
     * Grants permission to delete a set of DHCP options
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteDhcpOptions.html
     */
    deleteDhcpOptions() {
        this.add('ec2:DeleteDhcpOptions');
        return this;
    }
    /**
     * Grants permission to delete an egress-only internet gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteEgressOnlyInternetGateway.html
     */
    deleteEgressOnlyInternetGateway() {
        this.add('ec2:DeleteEgressOnlyInternetGateway');
        return this;
    }
    /**
     * Grants permission to delete one or more EC2 Fleets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFleets.html
     */
    deleteFleets() {
        this.add('ec2:DeleteFleets');
        return this;
    }
    /**
     * Grants permission to delete one or more flow logs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFlowLogs.html
     */
    deleteFlowLogs() {
        this.add('ec2:DeleteFlowLogs');
        return this;
    }
    /**
     * Grants permission to delete an Amazon FPGA Image (AFI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFpgaImage.html
     */
    deleteFpgaImage() {
        this.add('ec2:DeleteFpgaImage');
        return this;
    }
    /**
     * Grants permission to delete an internet gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteInternetGateway.html
     */
    deleteInternetGateway() {
        this.add('ec2:DeleteInternetGateway');
        return this;
    }
    /**
     * Grants permission to delete a key pair by removing the public key from Amazon EC2
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteKeyPair.html
     */
    deleteKeyPair() {
        this.add('ec2:DeleteKeyPair');
        return this;
    }
    /**
     * Grants permission to delete a launch template and its associated versions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLaunchTemplate.html
     */
    deleteLaunchTemplate() {
        this.add('ec2:DeleteLaunchTemplate');
        return this;
    }
    /**
     * Grants permission to delete one or more versions of a launch template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLaunchTemplateVersions.html
     */
    deleteLaunchTemplateVersions() {
        this.add('ec2:DeleteLaunchTemplateVersions');
        return this;
    }
    /**
     * Grants permission to delete a route from a local gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLocalGatewayRoute.html
     */
    deleteLocalGatewayRoute() {
        this.add('ec2:DeleteLocalGatewayRoute');
        return this;
    }
    /**
     * Grants permission to delete an association between a VPC and local gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLocalGatewayRouteTableVpcAssociation.html
     */
    deleteLocalGatewayRouteTableVpcAssociation() {
        this.add('ec2:DeleteLocalGatewayRouteTableVpcAssociation');
        return this;
    }
    /**
     * Grants permission to delete a managed prefix list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteManagedPrefixList.html
     */
    deleteManagedPrefixList() {
        this.add('ec2:DeleteManagedPrefixList');
        return this;
    }
    /**
     * Grants permission to delete a NAT gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNatGateway.html
     */
    deleteNatGateway() {
        this.add('ec2:DeleteNatGateway');
        return this;
    }
    /**
     * Grants permission to delete a network ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkAcl.html
     */
    deleteNetworkAcl() {
        this.add('ec2:DeleteNetworkAcl');
        return this;
    }
    /**
     * Grants permission to delete an inbound or outbound entry (rule) from a network ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkAclEntry.html
     */
    deleteNetworkAclEntry() {
        this.add('ec2:DeleteNetworkAclEntry');
        return this;
    }
    /**
     * Grants permission to delete a detached network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkInterface.html
     */
    deleteNetworkInterface() {
        this.add('ec2:DeleteNetworkInterface');
        return this;
    }
    /**
     * Grants permission to delete a permission that is associated with a network interface
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkInterfacePermission.html
     */
    deleteNetworkInterfacePermission() {
        this.add('ec2:DeleteNetworkInterfacePermission');
        return this;
    }
    /**
     * Grants permission to delete a placement group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeletePlacementGroup.html
     */
    deletePlacementGroup() {
        this.add('ec2:DeletePlacementGroup');
        return this;
    }
    /**
     * Grants permission to delete a route from a route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteRoute.html
     */
    deleteRoute() {
        this.add('ec2:DeleteRoute');
        return this;
    }
    /**
     * Grants permission to delete a route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteRouteTable.html
     */
    deleteRouteTable() {
        this.add('ec2:DeleteRouteTable');
        return this;
    }
    /**
     * Grants permission to delete a security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSecurityGroup.html
     */
    deleteSecurityGroup() {
        this.add('ec2:DeleteSecurityGroup');
        return this;
    }
    /**
     * Grants permission to delete a snapshot of an EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSnapshot.html
     */
    deleteSnapshot() {
        this.add('ec2:DeleteSnapshot');
        return this;
    }
    /**
     * Grants permission to delete a data feed for Spot Instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSpotDatafeedSubscription.html
     */
    deleteSpotDatafeedSubscription() {
        this.add('ec2:DeleteSpotDatafeedSubscription');
        return this;
    }
    /**
     * Grants permission to delete a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSubnet.html
     */
    deleteSubnet() {
        this.add('ec2:DeleteSubnet');
        return this;
    }
    /**
     * Grants permission to delete one or more tags from Amazon EC2 resources
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTags.html
     */
    deleteTags() {
        this.add('ec2:DeleteTags');
        return this;
    }
    /**
     * Grants permission to delete a traffic mirror filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorFilter.html
     */
    deleteTrafficMirrorFilter() {
        this.add('ec2:DeleteTrafficMirrorFilter');
        return this;
    }
    /**
     * Grants permission to delete a traffic mirror filter rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorFilterRule.html
     */
    deleteTrafficMirrorFilterRule() {
        this.add('ec2:DeleteTrafficMirrorFilterRule');
        return this;
    }
    /**
     * Grants permission to delete a traffic mirror session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorSession.html
     */
    deleteTrafficMirrorSession() {
        this.add('ec2:DeleteTrafficMirrorSession');
        return this;
    }
    /**
     * Grants permission to delete a traffic mirror target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorTarget.html
     */
    deleteTrafficMirrorTarget() {
        this.add('ec2:DeleteTrafficMirrorTarget');
        return this;
    }
    /**
     * Grants permission to delete a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGateway.html
     */
    deleteTransitGateway() {
        this.add('ec2:DeleteTransitGateway');
        return this;
    }
    /**
     * Grants permissions to delete a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayMulticastDomain.html
     */
    deleteTransitGatewayMulticastDomain() {
        this.add('ec2:DeleteTransitGatewayMulticastDomain');
        return this;
    }
    /**
     * Grants permission to delete a peering attachment from a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayPeeringAttachment.html
     */
    deleteTransitGatewayPeeringAttachment() {
        this.add('ec2:DeleteTransitGatewayPeeringAttachment');
        return this;
    }
    /**
     * Grants permission to delete a route from a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayRoute.html
     */
    deleteTransitGatewayRoute() {
        this.add('ec2:DeleteTransitGatewayRoute');
        return this;
    }
    /**
     * Grants permission to delete a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayRouteTable.html
     */
    deleteTransitGatewayRouteTable() {
        this.add('ec2:DeleteTransitGatewayRouteTable');
        return this;
    }
    /**
     * Grants permission to delete a VPC attachment from a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayVpcAttachment.html
     */
    deleteTransitGatewayVpcAttachment() {
        this.add('ec2:DeleteTransitGatewayVpcAttachment');
        return this;
    }
    /**
     * Grants permission to delete an EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVolume.html
     */
    deleteVolume() {
        this.add('ec2:DeleteVolume');
        return this;
    }
    /**
     * Grants permission to delete a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpc.html
     */
    deleteVpc() {
        this.add('ec2:DeleteVpc');
        return this;
    }
    /**
     * Grants permission to delete one or more VPC endpoint connection notifications
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpointConnectionNotifications.html
     */
    deleteVpcEndpointConnectionNotifications() {
        this.add('ec2:DeleteVpcEndpointConnectionNotifications');
        return this;
    }
    /**
     * Grants permission to delete one or more VPC endpoint service configurations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpointServiceConfigurations.html
     */
    deleteVpcEndpointServiceConfigurations() {
        this.add('ec2:DeleteVpcEndpointServiceConfigurations');
        return this;
    }
    /**
     * Grants permission to delete one or more VPC endpoints
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpoints.html
     */
    deleteVpcEndpoints() {
        this.add('ec2:DeleteVpcEndpoints');
        return this;
    }
    /**
     * Grants permission to delete a VPC peering connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcPeeringConnection.html
     */
    deleteVpcPeeringConnection() {
        this.add('ec2:DeleteVpcPeeringConnection');
        return this;
    }
    /**
     * Grants permission to delete a VPN connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnConnection.html
     */
    deleteVpnConnection() {
        this.add('ec2:DeleteVpnConnection');
        return this;
    }
    /**
     * Grants permission to delete a static route for a VPN connection between a virtual private gateway and a customer gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnConnectionRoute.html
     */
    deleteVpnConnectionRoute() {
        this.add('ec2:DeleteVpnConnectionRoute');
        return this;
    }
    /**
     * Grants permission to delete a virtual private gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnGateway.html
     */
    deleteVpnGateway() {
        this.add('ec2:DeleteVpnGateway');
        return this;
    }
    /**
     * Grants permission to release an IP address range that was provisioned through bring your own IP addresses (BYOIP), and to delete the corresponding address pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeprovisionByoipCidr.html
     */
    deprovisionByoipCidr() {
        this.add('ec2:DeprovisionByoipCidr');
        return this;
    }
    /**
     * Grants permission to deregister an Amazon Machine Image (AMI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterImage.html
     */
    deregisterImage() {
        this.add('ec2:DeregisterImage');
        return this;
    }
    /**
     * Grants permission to remove tags from the set of tags to include in notifications about scheduled events for your instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterInstanceEventNotificationAttributes.html
     */
    deregisterInstanceEventNotificationAttributes() {
        this.add('ec2:DeregisterInstanceEventNotificationAttributes');
        return this;
    }
    /**
     * Grants permission to deregister one or more network interface members from a group IP address in a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterTransitGatewayMulticastGroupMembers.html
     */
    deregisterTransitGatewayMulticastGroupMembers() {
        this.add('ec2:DeregisterTransitGatewayMulticastGroupMembers');
        return this;
    }
    /**
     * Grants permission to deregister one or more network interface sources from a group IP address in a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterTransitGatewayMulticastGroupSources.html
     */
    deregisterTransitGatewayMulticastGroupSources() {
        this.add('ec2:DeregisterTransitGatewayMulticastGroupSources');
        return this;
    }
    /**
     * Grants permission to describe the attributes of the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAccountAttributes.html
     */
    describeAccountAttributes() {
        this.add('ec2:DescribeAccountAttributes');
        return this;
    }
    /**
     * Grants permission to describe one or more Elastic IP addresses
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAddresses.html
     */
    describeAddresses() {
        this.add('ec2:DescribeAddresses');
        return this;
    }
    /**
     * Grants permission to describe the longer ID format settings for all resource types
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAggregateIdFormat.html
     */
    describeAggregateIdFormat() {
        this.add('ec2:DescribeAggregateIdFormat');
        return this;
    }
    /**
     * Grants permission to describe one or more of the Availability Zones that are available to you
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAvailabilityZones.html
     */
    describeAvailabilityZones() {
        this.add('ec2:DescribeAvailabilityZones');
        return this;
    }
    /**
     * Grants permission to describe one or more bundling tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeBundleTasks.html
     */
    describeBundleTasks() {
        this.add('ec2:DescribeBundleTasks');
        return this;
    }
    /**
     * Grants permission to describe the IP address ranges that were provisioned through bring your own IP addresses (BYOIP)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeByoipCidrs.html
     */
    describeByoipCidrs() {
        this.add('ec2:DescribeByoipCidrs');
        return this;
    }
    /**
     * Grants permission to describe one or more Capacity Reservations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCapacityReservations.html
     */
    describeCapacityReservations() {
        this.add('ec2:DescribeCapacityReservations');
        return this;
    }
    /**
     * Grants permission to describe one or more linked EC2-Classic instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClassicLinkInstances.html
     */
    describeClassicLinkInstances() {
        this.add('ec2:DescribeClassicLinkInstances');
        return this;
    }
    /**
     * Grants permission to describe the authorization rules for a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnAuthorizationRules.html
     */
    describeClientVpnAuthorizationRules() {
        this.add('ec2:DescribeClientVpnAuthorizationRules');
        return this;
    }
    /**
     * Grants permission to describe active client connections and connections that have been terminated within the last 60 minutes for a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnConnections.html
     */
    describeClientVpnConnections() {
        this.add('ec2:DescribeClientVpnConnections');
        return this;
    }
    /**
     * Grants permission to describe one or more Client VPN endpoints
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnEndpoints.html
     */
    describeClientVpnEndpoints() {
        this.add('ec2:DescribeClientVpnEndpoints');
        return this;
    }
    /**
     * Grants permission to describe the routes for a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnRoutes.html
     */
    describeClientVpnRoutes() {
        this.add('ec2:DescribeClientVpnRoutes');
        return this;
    }
    /**
     * Grants permission to describe the target networks that are associated with a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnTargetNetworks.html
     */
    describeClientVpnTargetNetworks() {
        this.add('ec2:DescribeClientVpnTargetNetworks');
        return this;
    }
    /**
     * Grants permission to describe the specified customer-owned address pools or all of your customer-owned address pools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCoipPools.html
     */
    describeCoipPools() {
        this.add('ec2:DescribeCoipPools');
        return this;
    }
    /**
     * Grants permission to describe one or more conversion tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeConversionTasks.html
     */
    describeConversionTasks() {
        this.add('ec2:DescribeConversionTasks');
        return this;
    }
    /**
     * Grants permission to describe one or more customer gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCustomerGateways.html
     */
    describeCustomerGateways() {
        this.add('ec2:DescribeCustomerGateways');
        return this;
    }
    /**
     * Grants permission to describe one or more DHCP options sets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeDhcpOptions.html
     */
    describeDhcpOptions() {
        this.add('ec2:DescribeDhcpOptions');
        return this;
    }
    /**
     * Grants permission to describe one or more egress-only internet gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeEgressOnlyInternetGateways.html
     */
    describeEgressOnlyInternetGateways() {
        this.add('ec2:DescribeEgressOnlyInternetGateways');
        return this;
    }
    /**
     * Grants permission to describe an Elastic Graphics accelerator that is associated with an instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeElasticGpus.html
     */
    describeElasticGpus() {
        this.add('ec2:DescribeElasticGpus');
        return this;
    }
    /**
     * Grants permission to describe one or more export image tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeExportImageTasks.html
     */
    describeExportImageTasks() {
        this.add('ec2:DescribeExportImageTasks');
        return this;
    }
    /**
     * Grants permission to describe one or more export instance tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeExportTasks.html
     */
    describeExportTasks() {
        this.add('ec2:DescribeExportTasks');
        return this;
    }
    /**
     * Grants permission to describe the state of fast snapshot restores for snapshots
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFastSnapshotRestores.html
     */
    describeFastSnapshotRestores() {
        this.add('ec2:DescribeFastSnapshotRestores');
        return this;
    }
    /**
     * Grants permission to describe the events for an EC2 Fleet during a specified time
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleetHistory.html
     */
    describeFleetHistory() {
        this.add('ec2:DescribeFleetHistory');
        return this;
    }
    /**
     * Grants permission to describe the running instances for an EC2 Fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleetInstances.html
     */
    describeFleetInstances() {
        this.add('ec2:DescribeFleetInstances');
        return this;
    }
    /**
     * Grants permission to describe one or more EC2 Fleets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleets.html
     */
    describeFleets() {
        this.add('ec2:DescribeFleets');
        return this;
    }
    /**
     * Grants permission to describe one or more flow logs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFlowLogs.html
     */
    describeFlowLogs() {
        this.add('ec2:DescribeFlowLogs');
        return this;
    }
    /**
     * Grants permission to describe the attributes of an Amazon FPGA Image (AFI)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFpgaImageAttribute.html
     */
    describeFpgaImageAttribute() {
        this.add('ec2:DescribeFpgaImageAttribute');
        return this;
    }
    /**
     * Grants permission to describe one or more Amazon FPGA Images (AFIs)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFpgaImages.html
     */
    describeFpgaImages() {
        this.add('ec2:DescribeFpgaImages');
        return this;
    }
    /**
     * Grants permission to describe the Dedicated Host Reservations that are available to purchase
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHostReservationOfferings.html
     */
    describeHostReservationOfferings() {
        this.add('ec2:DescribeHostReservationOfferings');
        return this;
    }
    /**
     * Grants permission to describe the Dedicated Host Reservations that are associated with Dedicated Hosts in the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHostReservations.html
     */
    describeHostReservations() {
        this.add('ec2:DescribeHostReservations');
        return this;
    }
    /**
     * Grants permission to describe one or more Dedicated Hosts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHosts.html
     */
    describeHosts() {
        this.add('ec2:DescribeHosts');
        return this;
    }
    /**
     * Grants permission to describe the IAM instance profile associations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIamInstanceProfileAssociations.html
     */
    describeIamInstanceProfileAssociations() {
        this.add('ec2:DescribeIamInstanceProfileAssociations');
        return this;
    }
    /**
     * Grants permission to describe the ID format settings for resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIdFormat.html
     */
    describeIdFormat() {
        this.add('ec2:DescribeIdFormat');
        return this;
    }
    /**
     * Grants permission to describe the ID format settings for resources for an IAM user, IAM role, or root user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIdentityIdFormat.html
     */
    describeIdentityIdFormat() {
        this.add('ec2:DescribeIdentityIdFormat');
        return this;
    }
    /**
     * Grants permission to describe an attribute of an Amazon Machine Image (AMI)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImageAttribute.html
     */
    describeImageAttribute() {
        this.add('ec2:DescribeImageAttribute');
        return this;
    }
    /**
     * Grants permission to describe one or more images (AMIs, AKIs, and ARIs)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImages.html
     */
    describeImages() {
        this.add('ec2:DescribeImages');
        return this;
    }
    /**
     * Grants permission to describe import virtual machine or import snapshot tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImportImageTasks.html
     */
    describeImportImageTasks() {
        this.add('ec2:DescribeImportImageTasks');
        return this;
    }
    /**
     * Grants permission to describe import snapshot tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImportSnapshotTasks.html
     */
    describeImportSnapshotTasks() {
        this.add('ec2:DescribeImportSnapshotTasks');
        return this;
    }
    /**
     * Grants permission to describe the attributes of an instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceAttribute.html
     */
    describeInstanceAttribute() {
        this.add('ec2:DescribeInstanceAttribute');
        return this;
    }
    /**
     * Grants permission to describe the credit option for CPU usage of one or more burstable performance instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceCreditSpecifications.html
     */
    describeInstanceCreditSpecifications() {
        this.add('ec2:DescribeInstanceCreditSpecifications');
        return this;
    }
    /**
     * Grants permission to describe the set of tags to include in notifications about scheduled events for your instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceEventNotificationAttributes.html
     */
    describeInstanceEventNotificationAttributes() {
        this.add('ec2:DescribeInstanceEventNotificationAttributes');
        return this;
    }
    /**
     * Grants permission to describe the status of one or more instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceStatus.html
     */
    describeInstanceStatus() {
        this.add('ec2:DescribeInstanceStatus');
        return this;
    }
    /**
     * Grants permission to describe the set of instance types that are offered in a location
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceTypeOfferings.html
     */
    describeInstanceTypeOfferings() {
        this.add('ec2:DescribeInstanceTypeOfferings');
        return this;
    }
    /**
     * Grants permission to describe the details of instance types that are offered in a location
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceTypes.html
     */
    describeInstanceTypes() {
        this.add('ec2:DescribeInstanceTypes');
        return this;
    }
    /**
     * Grants permission to describe one or more instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstances.html
     */
    describeInstances() {
        this.add('ec2:DescribeInstances');
        return this;
    }
    /**
     * Grants permission to describe one or more internet gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInternetGateways.html
     */
    describeInternetGateways() {
        this.add('ec2:DescribeInternetGateways');
        return this;
    }
    /**
     * Grants permission to describe one or more key pairs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeKeyPairs.html
     */
    describeKeyPairs() {
        this.add('ec2:DescribeKeyPairs');
        return this;
    }
    /**
     * Grants permission to describe one or more launch template versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLaunchTemplateVersions.html
     */
    describeLaunchTemplateVersions() {
        this.add('ec2:DescribeLaunchTemplateVersions');
        return this;
    }
    /**
     * Grants permission to describe one or more launch templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLaunchTemplates.html
     */
    describeLaunchTemplates() {
        this.add('ec2:DescribeLaunchTemplates');
        return this;
    }
    /**
     * Grants permission to describe the associations between virtual interface groups and local gateway route tables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations.html
     */
    describeLocalGatewayRouteTableVirtualInterfaceGroupAssociations() {
        this.add('ec2:DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations');
        return this;
    }
    /**
     * Grants permission to describe an association between VPCs and local gateway route tables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTableVpcAssociations.html
     */
    describeLocalGatewayRouteTableVpcAssociations() {
        this.add('ec2:DescribeLocalGatewayRouteTableVpcAssociations');
        return this;
    }
    /**
     * Grants permission to describe one or more local gateway route tables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTables.html
     */
    describeLocalGatewayRouteTables() {
        this.add('ec2:DescribeLocalGatewayRouteTables');
        return this;
    }
    /**
     * Grants permission to describe local gateway virtual interface groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayVirtualInterfaceGroups.html
     */
    describeLocalGatewayVirtualInterfaceGroups() {
        this.add('ec2:DescribeLocalGatewayVirtualInterfaceGroups');
        return this;
    }
    /**
     * Grants permission to describe local gateway virtual interfaces
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayVirtualInterfaces.html
     */
    describeLocalGatewayVirtualInterfaces() {
        this.add('ec2:DescribeLocalGatewayVirtualInterfaces');
        return this;
    }
    /**
     * Grants permission to describe one or more local gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGateways.html
     */
    describeLocalGateways() {
        this.add('ec2:DescribeLocalGateways');
        return this;
    }
    /**
     * Grants permission to describe your managed prefix lists and any AWS-managed prefix lists
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeManagedPrefixLists.html
     */
    describeManagedPrefixLists() {
        this.add('ec2:DescribeManagedPrefixLists');
        return this;
    }
    /**
     * Grants permission to describe Elastic IP addresses that are being moved to the EC2-VPC platform
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeMovingAddresses.html
     */
    describeMovingAddresses() {
        this.add('ec2:DescribeMovingAddresses');
        return this;
    }
    /**
     * Grants permission to describe one or more NAT gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNatGateways.html
     */
    describeNatGateways() {
        this.add('ec2:DescribeNatGateways');
        return this;
    }
    /**
     * Grants permission to describe one or more network ACLs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkAcls.html
     */
    describeNetworkAcls() {
        this.add('ec2:DescribeNetworkAcls');
        return this;
    }
    /**
     * Grants permission to describe a network interface attribute
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfaceAttribute.html
     */
    describeNetworkInterfaceAttribute() {
        this.add('ec2:DescribeNetworkInterfaceAttribute');
        return this;
    }
    /**
     * Grants permission to describe the permissions that are associated with a network interface
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfacePermissions.html
     */
    describeNetworkInterfacePermissions() {
        this.add('ec2:DescribeNetworkInterfacePermissions');
        return this;
    }
    /**
     * Grants permission to describe one or more network interfaces
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfaces.html
     */
    describeNetworkInterfaces() {
        this.add('ec2:DescribeNetworkInterfaces');
        return this;
    }
    /**
     * Grants permission to describe one or more placement groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePlacementGroups.html
     */
    describePlacementGroups() {
        this.add('ec2:DescribePlacementGroups');
        return this;
    }
    /**
     * Grants permission to describe available AWS services in a prefix list format
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePrefixLists.html
     */
    describePrefixLists() {
        this.add('ec2:DescribePrefixLists');
        return this;
    }
    /**
     * Grants permission to describe the ID format settings for the root user and all IAM roles and IAM users that have explicitly specified a longer ID (17-character ID) preference
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePrincipalIdFormat.html
     */
    describePrincipalIdFormat() {
        this.add('ec2:DescribePrincipalIdFormat');
        return this;
    }
    /**
     * Grants permission to describe one or more IPv4 address pools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePublicIpv4Pools.html
     */
    describePublicIpv4Pools() {
        this.add('ec2:DescribePublicIpv4Pools');
        return this;
    }
    /**
     * Grants permission to describe one or more AWS Regions that are currently available in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeRegions.html
     */
    describeRegions() {
        this.add('ec2:DescribeRegions');
        return this;
    }
    /**
     * Grants permission to describe one or more purchased Reserved Instances in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstances.html
     */
    describeReservedInstances() {
        this.add('ec2:DescribeReservedInstances');
        return this;
    }
    /**
     * Grants permission to describe your account's Reserved Instance listings in the Reserved Instance Marketplace
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesListings.html
     */
    describeReservedInstancesListings() {
        this.add('ec2:DescribeReservedInstancesListings');
        return this;
    }
    /**
     * Grants permission to describe the modifications made to one or more Reserved Instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesModifications.html
     */
    describeReservedInstancesModifications() {
        this.add('ec2:DescribeReservedInstancesModifications');
        return this;
    }
    /**
     * Grants permission to describe the Reserved Instance offerings that are available for purchase
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesOfferings.html
     */
    describeReservedInstancesOfferings() {
        this.add('ec2:DescribeReservedInstancesOfferings');
        return this;
    }
    /**
     * Grants permission to describe one or more route tables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeRouteTables.html
     */
    describeRouteTables() {
        this.add('ec2:DescribeRouteTables');
        return this;
    }
    /**
     * Grants permission to find available schedules for Scheduled Instances
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeScheduledInstanceAvailability.html
     */
    describeScheduledInstanceAvailability() {
        this.add('ec2:DescribeScheduledInstanceAvailability');
        return this;
    }
    /**
     * Grants permission to describe one or more Scheduled Instances in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeScheduledInstances.html
     */
    describeScheduledInstances() {
        this.add('ec2:DescribeScheduledInstances');
        return this;
    }
    /**
     * Grants permission to describe the VPCs on the other side of a VPC peering connection that are referencing specified VPC security groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSecurityGroupReferences.html
     */
    describeSecurityGroupReferences() {
        this.add('ec2:DescribeSecurityGroupReferences');
        return this;
    }
    /**
     * Grants permission to describe one or more security groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSecurityGroups.html
     */
    describeSecurityGroups() {
        this.add('ec2:DescribeSecurityGroups');
        return this;
    }
    /**
     * Grants permission to describe an attribute of a snapshot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSnapshotAttribute.html
     */
    describeSnapshotAttribute() {
        this.add('ec2:DescribeSnapshotAttribute');
        return this;
    }
    /**
     * Grants permission to describe one or more EBS snapshots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSnapshots.html
     */
    describeSnapshots() {
        this.add('ec2:DescribeSnapshots');
        return this;
    }
    /**
     * Grants permission to describe the data feed for Spot Instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotDatafeedSubscription.html
     */
    describeSpotDatafeedSubscription() {
        this.add('ec2:DescribeSpotDatafeedSubscription');
        return this;
    }
    /**
     * Grants permission to describe the running instances for a Spot Fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetInstances.html
     */
    describeSpotFleetInstances() {
        this.add('ec2:DescribeSpotFleetInstances');
        return this;
    }
    /**
     * Grants permission to describe the events for a Spot Fleet request during a specified time
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetRequestHistory.html
     */
    describeSpotFleetRequestHistory() {
        this.add('ec2:DescribeSpotFleetRequestHistory');
        return this;
    }
    /**
     * Grants permission to describe one or more Spot Fleet requests
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetRequests.html
     */
    describeSpotFleetRequests() {
        this.add('ec2:DescribeSpotFleetRequests');
        return this;
    }
    /**
     * Grants permission to describe one or more Spot Instance requests
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotInstanceRequests.html
     */
    describeSpotInstanceRequests() {
        this.add('ec2:DescribeSpotInstanceRequests');
        return this;
    }
    /**
     * Grants permission to describe the Spot Instance price history
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotPriceHistory.html
     */
    describeSpotPriceHistory() {
        this.add('ec2:DescribeSpotPriceHistory');
        return this;
    }
    /**
     * Grants permission to describe the stale security group rules for security groups in a specified VPC
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeStaleSecurityGroups.html
     */
    describeStaleSecurityGroups() {
        this.add('ec2:DescribeStaleSecurityGroups');
        return this;
    }
    /**
     * Grants permission to describe one or more subnets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSubnets.html
     */
    describeSubnets() {
        this.add('ec2:DescribeSubnets');
        return this;
    }
    /**
     * Grants permission to describe one or more tags for an Amazon EC2 resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTags.html
     */
    describeTags() {
        this.add('ec2:DescribeTags');
        return this;
    }
    /**
     * Grants permission to describe one or more traffic mirror filters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorFilters.html
     */
    describeTrafficMirrorFilters() {
        this.add('ec2:DescribeTrafficMirrorFilters');
        return this;
    }
    /**
     * Grants permission to describe one or more traffic mirror sessions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorSessions.html
     */
    describeTrafficMirrorSessions() {
        this.add('ec2:DescribeTrafficMirrorSessions');
        return this;
    }
    /**
     * Grants permission to describe one or more traffic mirror targets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorTargets.html
     */
    describeTrafficMirrorTargets() {
        this.add('ec2:DescribeTrafficMirrorTargets');
        return this;
    }
    /**
     * Grants permission to describe one or more attachments between resources and transit gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayAttachments.html
     */
    describeTransitGatewayAttachments() {
        this.add('ec2:DescribeTransitGatewayAttachments');
        return this;
    }
    /**
     * Grants permission to describe one or more transit gateway multicast domains
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayMulticastDomains.html
     */
    describeTransitGatewayMulticastDomains() {
        this.add('ec2:DescribeTransitGatewayMulticastDomains');
        return this;
    }
    /**
     * Grants permission to describe one or more transit gateway peering attachments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayPeeringAttachments.html
     */
    describeTransitGatewayPeeringAttachments() {
        this.add('ec2:DescribeTransitGatewayPeeringAttachments');
        return this;
    }
    /**
     * Grants permission to describe one or more transit gateway route tables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayRouteTables.html
     */
    describeTransitGatewayRouteTables() {
        this.add('ec2:DescribeTransitGatewayRouteTables');
        return this;
    }
    /**
     * Grants permission to describe one or more VPC attachments on a transit gateway
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayVpcAttachments.html
     */
    describeTransitGatewayVpcAttachments() {
        this.add('ec2:DescribeTransitGatewayVpcAttachments');
        return this;
    }
    /**
     * Grants permission to describe one or more transit gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGateways.html
     */
    describeTransitGateways() {
        this.add('ec2:DescribeTransitGateways');
        return this;
    }
    /**
     * Grants permission to describe an attribute of an EBS volume
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumeAttribute.html
     */
    describeVolumeAttribute() {
        this.add('ec2:DescribeVolumeAttribute');
        return this;
    }
    /**
     * Grants permission to describe the status of one or more EBS volumes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumeStatus.html
     */
    describeVolumeStatus() {
        this.add('ec2:DescribeVolumeStatus');
        return this;
    }
    /**
     * Grants permission to describe one or more EBS volumes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumes.html
     */
    describeVolumes() {
        this.add('ec2:DescribeVolumes');
        return this;
    }
    /**
     * Grants permission to describe the current modification status of one or more EBS volumes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumesModifications.html
     */
    describeVolumesModifications() {
        this.add('ec2:DescribeVolumesModifications');
        return this;
    }
    /**
     * Grants permission to describe an attribute of a VPC
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcAttribute.html
     */
    describeVpcAttribute() {
        this.add('ec2:DescribeVpcAttribute');
        return this;
    }
    /**
     * Grants permission to describe the ClassicLink status of one or more VPCs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcClassicLink.html
     */
    describeVpcClassicLink() {
        this.add('ec2:DescribeVpcClassicLink');
        return this;
    }
    /**
     * Grants permission to describe the ClassicLink DNS support status of one or more VPCs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcClassicLinkDnsSupport.html
     */
    describeVpcClassicLinkDnsSupport() {
        this.add('ec2:DescribeVpcClassicLinkDnsSupport');
        return this;
    }
    /**
     * Grants permission to describe the connection notifications for VPC endpoints and VPC endpoint services
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointConnectionNotifications.html
     */
    describeVpcEndpointConnectionNotifications() {
        this.add('ec2:DescribeVpcEndpointConnectionNotifications');
        return this;
    }
    /**
     * Grants permission to describe the VPC endpoint connections to your VPC endpoint services
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointConnections.html
     */
    describeVpcEndpointConnections() {
        this.add('ec2:DescribeVpcEndpointConnections');
        return this;
    }
    /**
     * Grants permission to describe VPC endpoint service configurations (your services)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServiceConfigurations.html
     */
    describeVpcEndpointServiceConfigurations() {
        this.add('ec2:DescribeVpcEndpointServiceConfigurations');
        return this;
    }
    /**
     * Grants permission to describe the principals (service consumers) that are permitted to discover your VPC endpoint service
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServicePermissions.html
     */
    describeVpcEndpointServicePermissions() {
        this.add('ec2:DescribeVpcEndpointServicePermissions');
        return this;
    }
    /**
     * Grants permission to describe all supported AWS services that can be specified when creating a VPC endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServices.html
     */
    describeVpcEndpointServices() {
        this.add('ec2:DescribeVpcEndpointServices');
        return this;
    }
    /**
     * Grants permission to describe one or more VPC endpoints
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpoints.html
     */
    describeVpcEndpoints() {
        this.add('ec2:DescribeVpcEndpoints');
        return this;
    }
    /**
     * Grants permission to describe one or more VPC peering connections
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcPeeringConnections.html
     */
    describeVpcPeeringConnections() {
        this.add('ec2:DescribeVpcPeeringConnections');
        return this;
    }
    /**
     * Grants permission to describe one or more VPCs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcs.html
     */
    describeVpcs() {
        this.add('ec2:DescribeVpcs');
        return this;
    }
    /**
     * Grants permission to describe one or more VPN connections
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpnConnections.html
     */
    describeVpnConnections() {
        this.add('ec2:DescribeVpnConnections');
        return this;
    }
    /**
     * Grants permission to describe one or more virtual private gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpnGateways.html
     */
    describeVpnGateways() {
        this.add('ec2:DescribeVpnGateways');
        return this;
    }
    /**
     * Grants permission to unlink (detach) a linked EC2-Classic instance from a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachClassicLinkVpc.html
     */
    detachClassicLinkVpc() {
        this.add('ec2:DetachClassicLinkVpc');
        return this;
    }
    /**
     * Grants permission to detach an internet gateway from a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachInternetGateway.html
     */
    detachInternetGateway() {
        this.add('ec2:DetachInternetGateway');
        return this;
    }
    /**
     * Grants permission to detach a network interface from an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachNetworkInterface.html
     */
    detachNetworkInterface() {
        this.add('ec2:DetachNetworkInterface');
        return this;
    }
    /**
     * Grants permission to detach an EBS volume from an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachVolume.html
     */
    detachVolume() {
        this.add('ec2:DetachVolume');
        return this;
    }
    /**
     * Grants permission to detach a virtual private gateway from a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachVpnGateway.html
     */
    detachVpnGateway() {
        this.add('ec2:DetachVpnGateway');
        return this;
    }
    /**
     * Grants permission to disable EBS encryption by default for your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableEbsEncryptionByDefault.html
     */
    disableEbsEncryptionByDefault() {
        this.add('ec2:DisableEbsEncryptionByDefault');
        return this;
    }
    /**
     * Grants permission to disable fast snapshot restores for one or more snapshots in specified Availability Zones
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableFastSnapshotRestores.html
     */
    disableFastSnapshotRestores() {
        this.add('ec2:DisableFastSnapshotRestores');
        return this;
    }
    /**
     * Grants permission to disable a resource attachment from propagating routes to the specified propagation route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableTransitGatewayRouteTablePropagation.html
     */
    disableTransitGatewayRouteTablePropagation() {
        this.add('ec2:DisableTransitGatewayRouteTablePropagation');
        return this;
    }
    /**
     * Grants permission to disable a virtual private gateway from propagating routes to a specified route table of a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVgwRoutePropagation.html
     */
    disableVgwRoutePropagation() {
        this.add('ec2:DisableVgwRoutePropagation');
        return this;
    }
    /**
     * Grants permission to disable ClassicLink for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVpcClassicLink.html
     */
    disableVpcClassicLink() {
        this.add('ec2:DisableVpcClassicLink');
        return this;
    }
    /**
     * Grants permission to disable ClassicLink DNS support for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVpcClassicLinkDnsSupport.html
     */
    disableVpcClassicLinkDnsSupport() {
        this.add('ec2:DisableVpcClassicLinkDnsSupport');
        return this;
    }
    /**
     * Grants permission to disassociate an Elastic IP address from an instance or network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateAddress.html
     */
    disassociateAddress() {
        this.add('ec2:DisassociateAddress');
        return this;
    }
    /**
     * Grants permission to disassociate a target network from a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateClientVpnTargetNetwork.html
     */
    disassociateClientVpnTargetNetwork() {
        this.add('ec2:DisassociateClientVpnTargetNetwork');
        return this;
    }
    /**
     * Grants permission to disassociate an IAM instance profile from a running or stopped instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateIamInstanceProfile.html
     */
    disassociateIamInstanceProfile() {
        this.add('ec2:DisassociateIamInstanceProfile');
        return this;
    }
    /**
     * Grants permission to disassociate a subnet from a route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateRouteTable.html
     */
    disassociateRouteTable() {
        this.add('ec2:DisassociateRouteTable');
        return this;
    }
    /**
     * Grants permission to disassociate a CIDR block from a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateSubnetCidrBlock.html
     */
    disassociateSubnetCidrBlock() {
        this.add('ec2:DisassociateSubnetCidrBlock');
        return this;
    }
    /**
     * Grants permission to disassociate one or more subnets from a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateTransitGatewayMulticastDomain.html
     */
    disassociateTransitGatewayMulticastDomain() {
        this.add('ec2:DisassociateTransitGatewayMulticastDomain');
        return this;
    }
    /**
     * Grants permission to disassociate a resource attachment from a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateTransitGatewayRouteTable.html
     */
    disassociateTransitGatewayRouteTable() {
        this.add('ec2:DisassociateTransitGatewayRouteTable');
        return this;
    }
    /**
     * Grants permission to disassociate a CIDR block from a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateVpcCidrBlock.html
     */
    disassociateVpcCidrBlock() {
        this.add('ec2:DisassociateVpcCidrBlock');
        return this;
    }
    /**
     * Grants permission to enable EBS encryption by default for your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableEbsEncryptionByDefault.html
     */
    enableEbsEncryptionByDefault() {
        this.add('ec2:EnableEbsEncryptionByDefault');
        return this;
    }
    /**
     * Grants permission to enable fast snapshot restores for one or more snapshots in specified Availability Zones
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableFastSnapshotRestores.html
     */
    enableFastSnapshotRestores() {
        this.add('ec2:EnableFastSnapshotRestores');
        return this;
    }
    /**
     * Grants permission to enable an attachment to propagate routes to a propagation route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableTransitGatewayRouteTablePropagation.html
     */
    enableTransitGatewayRouteTablePropagation() {
        this.add('ec2:EnableTransitGatewayRouteTablePropagation');
        return this;
    }
    /**
     * Grants permission to enable a virtual private gateway to propagate routes to a VPC route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVgwRoutePropagation.html
     */
    enableVgwRoutePropagation() {
        this.add('ec2:EnableVgwRoutePropagation');
        return this;
    }
    /**
     * Grants permission to enable I/O operations for a volume that had I/O operations disabled
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVolumeIO.html
     */
    enableVolumeIO() {
        this.add('ec2:EnableVolumeIO');
        return this;
    }
    /**
     * Grants permission to enable a VPC for ClassicLink
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVpcClassicLink.html
     */
    enableVpcClassicLink() {
        this.add('ec2:EnableVpcClassicLink');
        return this;
    }
    /**
     * Grants permission to enable a VPC to support DNS hostname resolution for ClassicLink
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVpcClassicLinkDnsSupport.html
     */
    enableVpcClassicLinkDnsSupport() {
        this.add('ec2:EnableVpcClassicLinkDnsSupport');
        return this;
    }
    /**
     * Grants permission to download the client certificate revocation list for a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportClientVpnClientCertificateRevocationList.html
     */
    exportClientVpnClientCertificateRevocationList() {
        this.add('ec2:ExportClientVpnClientCertificateRevocationList');
        return this;
    }
    /**
     * Grants permission to download the contents of the Client VPN endpoint configuration file for a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportClientVpnClientConfiguration.html
     */
    exportClientVpnClientConfiguration() {
        this.add('ec2:ExportClientVpnClientConfiguration');
        return this;
    }
    /**
     * Grants permission to export an Amazon Machine Image (AMI) to a VM file
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportImage.html
     */
    exportImage() {
        this.add('ec2:ExportImage');
        return this;
    }
    /**
     * Grants permission to export routes from a transit gateway route table to an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportTransitGatewayRoutes.html
     */
    exportTransitGatewayRoutes() {
        this.add('ec2:ExportTransitGatewayRoutes');
        return this;
    }
    /**
     * Grants permission to get usage information about a Capacity Reservation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetCapacityReservationUsage.html
     */
    getCapacityReservationUsage() {
        this.add('ec2:GetCapacityReservationUsage');
        return this;
    }
    /**
     * Grants permission to describe the allocations from the specified customer-owned address pool
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetCoipPoolUsage.html
     */
    getCoipPoolUsage() {
        this.add('ec2:GetCoipPoolUsage');
        return this;
    }
    /**
     * Grants permission to get the console output for an instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetConsoleOutput.html
     */
    getConsoleOutput() {
        this.add('ec2:GetConsoleOutput');
        return this;
    }
    /**
     * Grants permission to retrieve a JPG-format screenshot of a running instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetConsoleScreenshot.html
     */
    getConsoleScreenshot() {
        this.add('ec2:GetConsoleScreenshot');
        return this;
    }
    /**
     * Grants permission to get the default credit option for CPU usage of a burstable performance instance family
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetDefaultCreditSpecification.html
     */
    getDefaultCreditSpecification() {
        this.add('ec2:GetDefaultCreditSpecification');
        return this;
    }
    /**
     * Grants permission to get the ID of the default customer master key (CMK) for EBS encryption by default
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetEbsDefaultKmsKeyId.html
     */
    getEbsDefaultKmsKeyId() {
        this.add('ec2:GetEbsDefaultKmsKeyId');
        return this;
    }
    /**
     * Grants permission to describe whether EBS encryption by default is enabled for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetEbsEncryptionByDefault.html
     */
    getEbsEncryptionByDefault() {
        this.add('ec2:GetEbsEncryptionByDefault');
        return this;
    }
    /**
     * Grants permission to preview a reservation purchase with configurations that match those of a Dedicated Host
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetHostReservationPurchasePreview.html
     */
    getHostReservationPurchasePreview() {
        this.add('ec2:GetHostReservationPurchasePreview');
        return this;
    }
    /**
     * Grants permission to get the configuration data of the specified instance for use with a new launch template or launch template version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetLaunchTemplateData.html
     */
    getLaunchTemplateData() {
        this.add('ec2:GetLaunchTemplateData');
        return this;
    }
    /**
     * Grants permission to get information about the resources that are associated with the specified managed prefix list
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetManagedPrefixListAssociations.html
     */
    getManagedPrefixListAssociations() {
        this.add('ec2:GetManagedPrefixListAssociations');
        return this;
    }
    /**
     * Grants permission to get information about the entries for a specified managed prefix list
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetManagedPrefixListEntries.html
     */
    getManagedPrefixListEntries() {
        this.add('ec2:GetManagedPrefixListEntries');
        return this;
    }
    /**
     * Grants permission to retrieve the encrypted administrator password for a running Windows instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetPasswordData.html
     */
    getPasswordData() {
        this.add('ec2:GetPasswordData');
        return this;
    }
    /**
     * Grants permission to return a quote and exchange information for exchanging one or more Convertible Reserved Instances for a new Convertible Reserved Instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetReservedInstancesExchangeQuote.html
     */
    getReservedInstancesExchangeQuote() {
        this.add('ec2:GetReservedInstancesExchangeQuote');
        return this;
    }
    /**
     * Grants permission to list the route tables to which a resource attachment propagates routes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayAttachmentPropagations.html
     */
    getTransitGatewayAttachmentPropagations() {
        this.add('ec2:GetTransitGatewayAttachmentPropagations');
        return this;
    }
    /**
     * Grants permission to get information about the associations for a transit gateway multicast domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayMulticastDomainAssociations.html
     */
    getTransitGatewayMulticastDomainAssociations() {
        this.add('ec2:GetTransitGatewayMulticastDomainAssociations');
        return this;
    }
    /**
     * Grants permission to get information about associations for a transit gateway route table
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayRouteTableAssociations.html
     */
    getTransitGatewayRouteTableAssociations() {
        this.add('ec2:GetTransitGatewayRouteTableAssociations');
        return this;
    }
    /**
     * Grants permission to get information about the route table propagations for a transit gateway route table
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayRouteTablePropagations.html
     */
    getTransitGatewayRouteTablePropagations() {
        this.add('ec2:GetTransitGatewayRouteTablePropagations');
        return this;
    }
    /**
     * Grants permission to upload a client certificate revocation list to a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportClientVpnClientCertificateRevocationList.html
     */
    importClientVpnClientCertificateRevocationList() {
        this.add('ec2:ImportClientVpnClientCertificateRevocationList');
        return this;
    }
    /**
     * Grants permission to import single or multi-volume disk images or EBS snapshots into an Amazon Machine Image (AMI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportImage.html
     */
    importImage() {
        this.add('ec2:ImportImage');
        return this;
    }
    /**
     * Grants permission to create an import instance task using metadata from a disk image
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportInstance.html
     */
    importInstance() {
        this.add('ec2:ImportInstance');
        return this;
    }
    /**
     * Grants permission to import a public key from an RSA key pair that was created with a third-party tool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportKeyPair.html
     */
    importKeyPair() {
        this.add('ec2:ImportKeyPair');
        return this;
    }
    /**
     * Grants permission to import a disk into an EBS snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportSnapshot.html
     */
    importSnapshot() {
        this.add('ec2:ImportSnapshot');
        return this;
    }
    /**
     * Grants permission to create an import volume task using metadata from a disk image
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportVolume.html
     */
    importVolume() {
        this.add('ec2:ImportVolume');
        return this;
    }
    /**
     * Grants permission to modify a Capacity Reservation's capacity and the conditions under which it is to be released
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyCapacityReservation.html
     */
    modifyCapacityReservation() {
        this.add('ec2:ModifyCapacityReservation');
        return this;
    }
    /**
     * Grants permission to modify a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyClientVpnEndpoint.html
     */
    modifyClientVpnEndpoint() {
        this.add('ec2:ModifyClientVpnEndpoint');
        return this;
    }
    /**
     * Grants permission to change the account level default credit option for CPU usage of burstable performance instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyDefaultCreditSpecification.html
     */
    modifyDefaultCreditSpecification() {
        this.add('ec2:ModifyDefaultCreditSpecification');
        return this;
    }
    /**
     * Grants permission to change the default customer master key (CMK) for EBS encryption by default for your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyEbsDefaultKmsKeyId.html
     */
    modifyEbsDefaultKmsKeyId() {
        this.add('ec2:ModifyEbsDefaultKmsKeyId');
        return this;
    }
    /**
     * Grants permission to modify an EC2 Fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyFleet.html
     */
    modifyFleet() {
        this.add('ec2:ModifyFleet');
        return this;
    }
    /**
     * Grants permission to modify an attribute of an Amazon FPGA Image (AFI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyFpgaImageAttribute.html
     */
    modifyFpgaImageAttribute() {
        this.add('ec2:ModifyFpgaImageAttribute');
        return this;
    }
    /**
     * Grants permission to modify a Dedicated Host
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyHosts.html
     */
    modifyHosts() {
        this.add('ec2:ModifyHosts');
        return this;
    }
    /**
     * Grants permission to modify the ID format for a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyIdFormat.html
     */
    modifyIdFormat() {
        this.add('ec2:ModifyIdFormat');
        return this;
    }
    /**
     * Grants permission to modify the ID format of a resource for a specific principal in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyIdentityIdFormat.html
     */
    modifyIdentityIdFormat() {
        this.add('ec2:ModifyIdentityIdFormat');
        return this;
    }
    /**
     * Grants permission to modify an attribute of an Amazon Machine Image (AMI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyImageAttribute.html
     */
    modifyImageAttribute() {
        this.add('ec2:ModifyImageAttribute');
        return this;
    }
    /**
     * Grants permission to modify an attribute of an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceAttribute.html
     */
    modifyInstanceAttribute() {
        this.add('ec2:ModifyInstanceAttribute');
        return this;
    }
    /**
     * Grants permission to modify the Capacity Reservation settings for a stopped instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCapacityReservationAttributes.html
     */
    modifyInstanceCapacityReservationAttributes() {
        this.add('ec2:ModifyInstanceCapacityReservationAttributes');
        return this;
    }
    /**
     * Grants permission to modify the credit option for CPU usage on an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCreditSpecification.html
     */
    modifyInstanceCreditSpecification() {
        this.add('ec2:ModifyInstanceCreditSpecification');
        return this;
    }
    /**
     * Grants permission to modify the start time for a scheduled EC2 instance event
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceEventStartTime.html
     */
    modifyInstanceEventStartTime() {
        this.add('ec2:ModifyInstanceEventStartTime');
        return this;
    }
    /**
     * Grants permission to modify the metadata options for an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceMetadataOptions.html
     */
    modifyInstanceMetadataOptions() {
        this.add('ec2:ModifyInstanceMetadataOptions');
        return this;
    }
    /**
     * Grants permission to modify the placement attributes for an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstancePlacement.html
     */
    modifyInstancePlacement() {
        this.add('ec2:ModifyInstancePlacement');
        return this;
    }
    /**
     * Grants permission to modify a launch template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyLaunchTemplate.html
     */
    modifyLaunchTemplate() {
        this.add('ec2:ModifyLaunchTemplate');
        return this;
    }
    /**
     * Grants permission to modify a managed prefix list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyManagedPrefixList.html
     */
    modifyManagedPrefixList() {
        this.add('ec2:ModifyManagedPrefixList');
        return this;
    }
    /**
     * Grants permission to modify an attribute of a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyNetworkInterfaceAttribute.html
     */
    modifyNetworkInterfaceAttribute() {
        this.add('ec2:ModifyNetworkInterfaceAttribute');
        return this;
    }
    /**
     * Grants permission to modify attributes of one or more Reserved Instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyReservedInstances.html
     */
    modifyReservedInstances() {
        this.add('ec2:ModifyReservedInstances');
        return this;
    }
    /**
     * Grants permission to add or remove permission settings for a snapshot
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySnapshotAttribute.html
     */
    modifySnapshotAttribute() {
        this.add('ec2:ModifySnapshotAttribute');
        return this;
    }
    /**
     * Grants permission to modify a Spot Fleet request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySpotFleetRequest.html
     */
    modifySpotFleetRequest() {
        this.add('ec2:ModifySpotFleetRequest');
        return this;
    }
    /**
     * Grants permission to modify an attribute of a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySubnetAttribute.html
     */
    modifySubnetAttribute() {
        this.add('ec2:ModifySubnetAttribute');
        return this;
    }
    /**
     * Grants permission to allow or restrict mirroring network services
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorFilterNetworkServices.html
     */
    modifyTrafficMirrorFilterNetworkServices() {
        this.add('ec2:ModifyTrafficMirrorFilterNetworkServices');
        return this;
    }
    /**
     * Grants permission to modify a traffic mirror rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorFilterRule.html
     */
    modifyTrafficMirrorFilterRule() {
        this.add('ec2:ModifyTrafficMirrorFilterRule');
        return this;
    }
    /**
     * Grants permission to modify a traffic mirror session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorSession.html
     */
    modifyTrafficMirrorSession() {
        this.add('ec2:ModifyTrafficMirrorSession');
        return this;
    }
    /**
     * Grants permission to modify a VPC attachment on a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTransitGatewayVpcAttachment.html
     */
    modifyTransitGatewayVpcAttachment() {
        this.add('ec2:ModifyTransitGatewayVpcAttachment');
        return this;
    }
    /**
     * Grants permission to modify the parameters of an EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVolume.html
     */
    modifyVolume() {
        this.add('ec2:ModifyVolume');
        return this;
    }
    /**
     * Grants permission to modify an attribute of a volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVolumeAttribute.html
     */
    modifyVolumeAttribute() {
        this.add('ec2:ModifyVolumeAttribute');
        return this;
    }
    /**
     * Grants permission to modify an attribute of a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcAttribute.html
     */
    modifyVpcAttribute() {
        this.add('ec2:ModifyVpcAttribute');
        return this;
    }
    /**
     * Grants permission to modify an attribute of a VPC endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpoint.html
     */
    modifyVpcEndpoint() {
        this.add('ec2:ModifyVpcEndpoint');
        return this;
    }
    /**
     * Grants permission to modify a connection notification for a VPC endpoint or VPC endpoint service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointConnectionNotification.html
     */
    modifyVpcEndpointConnectionNotification() {
        this.add('ec2:ModifyVpcEndpointConnectionNotification');
        return this;
    }
    /**
     * Grants permission to modify the attributes of a VPC endpoint service configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointServiceConfiguration.html
     */
    modifyVpcEndpointServiceConfiguration() {
        this.add('ec2:ModifyVpcEndpointServiceConfiguration');
        return this;
    }
    /**
     * Grants permission to modify the permissions for a VPC endpoint service
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointServicePermissions.html
     */
    modifyVpcEndpointServicePermissions() {
        this.add('ec2:ModifyVpcEndpointServicePermissions');
        return this;
    }
    /**
     * Grants permission to modify the VPC peering connection options on one side of a VPC peering connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcPeeringConnectionOptions.html
     */
    modifyVpcPeeringConnectionOptions() {
        this.add('ec2:ModifyVpcPeeringConnectionOptions');
        return this;
    }
    /**
     * Grants permission to modify the instance tenancy attribute of a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcTenancy.html
     */
    modifyVpcTenancy() {
        this.add('ec2:ModifyVpcTenancy');
        return this;
    }
    /**
     * Grants permission to modify the target gateway of a Site-to-Site VPN connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnConnection.html
     */
    modifyVpnConnection() {
        this.add('ec2:ModifyVpnConnection');
        return this;
    }
    /**
     * Grants permission to modify the certificate for a Site-to-Site VPN connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnTunnelCertificate
     */
    modifyVpnTunnelCertificate() {
        this.add('ec2:ModifyVpnTunnelCertificate');
        return this;
    }
    /**
     * Grants permission to modify the options for a Site-to-Site VPN connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnTunnelOptions.html
     */
    modifyVpnTunnelOptions() {
        this.add('ec2:ModifyVpnTunnelOptions');
        return this;
    }
    /**
     * Grants permission to enable detailed monitoring for a running instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_MonitorInstances.html
     */
    monitorInstances() {
        this.add('ec2:MonitorInstances');
        return this;
    }
    /**
     * Grants permission to move an Elastic IP address from the EC2-Classic platform to the EC2-VPC platform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_MoveAddressToVpc.html
     */
    moveAddressToVpc() {
        this.add('ec2:MoveAddressToVpc');
        return this;
    }
    /**
     * Grants permission to provision an address range for use in AWS through bring your own IP addresses (BYOIP), and to create a corresponding address pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ProvisionByoipCidr.html
     */
    provisionByoipCidr() {
        this.add('ec2:ProvisionByoipCidr');
        return this;
    }
    /**
     * Grants permission to purchase a reservation with configurations that match those of a Dedicated Host
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseHostReservation.html
     */
    purchaseHostReservation() {
        this.add('ec2:PurchaseHostReservation');
        return this;
    }
    /**
     * Grants permission to purchase a Reserved Instance offering
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseReservedInstancesOffering.html
     */
    purchaseReservedInstancesOffering() {
        this.add('ec2:PurchaseReservedInstancesOffering');
        return this;
    }
    /**
     * Grants permission to purchase one or more Scheduled Instances with a specified schedule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseScheduledInstances.html
     */
    purchaseScheduledInstances() {
        this.add('ec2:PurchaseScheduledInstances');
        return this;
    }
    /**
     * Grants permission to request a reboot of one or more instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RebootInstances.html
     */
    rebootInstances() {
        this.add('ec2:RebootInstances');
        return this;
    }
    /**
     * Grants permission to register an Amazon Machine Image (AMI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterImage.html
     */
    registerImage() {
        this.add('ec2:RegisterImage');
        return this;
    }
    /**
     * Grants permission to add tags to the set of tags to include in notifications about scheduled events for your instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterInstanceEventNotificationAttributes.html
     */
    registerInstanceEventNotificationAttributes() {
        this.add('ec2:RegisterInstanceEventNotificationAttributes');
        return this;
    }
    /**
     * Grants permission to register one or more network interfaces as a member of a group IP address in a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterTransitGatewayMulticastGroupMembers.html
     */
    registerTransitGatewayMulticastGroupMembers() {
        this.add('ec2:RegisterTransitGatewayMulticastGroupMembers');
        return this;
    }
    /**
     * Grants permission to register one or more network interfaces as a source of a group IP address in a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterTransitGatewayMulticastGroupSources.html
     */
    registerTransitGatewayMulticastGroupSources() {
        this.add('ec2:RegisterTransitGatewayMulticastGroupSources');
        return this;
    }
    /**
     * Grants permission to reject a transit gateway peering attachment request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectTransitGatewayPeeringAttachment.html
     */
    rejectTransitGatewayPeeringAttachment() {
        this.add('ec2:RejectTransitGatewayPeeringAttachment');
        return this;
    }
    /**
     * Grants permission to reject a request to attach a VPC to a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectTransitGatewayVpcAttachment.html
     */
    rejectTransitGatewayVpcAttachment() {
        this.add('ec2:RejectTransitGatewayVpcAttachment');
        return this;
    }
    /**
     * Grants permission to reject one or more VPC endpoint connection requests to a VPC endpoint service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectVpcEndpointConnections.html
     */
    rejectVpcEndpointConnections() {
        this.add('ec2:RejectVpcEndpointConnections');
        return this;
    }
    /**
     * Grants permission to reject a VPC peering connection request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectVpcPeeringConnection.html
     */
    rejectVpcPeeringConnection() {
        this.add('ec2:RejectVpcPeeringConnection');
        return this;
    }
    /**
     * Grants permission to release an Elastic IP address
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReleaseAddress.html
     */
    releaseAddress() {
        this.add('ec2:ReleaseAddress');
        return this;
    }
    /**
     * Grants permission to release one or more On-Demand Dedicated Hosts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReleaseHosts.html
     */
    releaseHosts() {
        this.add('ec2:ReleaseHosts');
        return this;
    }
    /**
     * Grants permission to replace an IAM instance profile for an instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceIamInstanceProfileAssociation.html
     */
    replaceIamInstanceProfileAssociation() {
        this.add('ec2:ReplaceIamInstanceProfileAssociation');
        return this;
    }
    /**
     * Grants permission to change which network ACL a subnet is associated with
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceNetworkAclAssociation.html
     */
    replaceNetworkAclAssociation() {
        this.add('ec2:ReplaceNetworkAclAssociation');
        return this;
    }
    /**
     * Grants permission to replace an entry (rule) in a network ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceNetworkAclEntry.html
     */
    replaceNetworkAclEntry() {
        this.add('ec2:ReplaceNetworkAclEntry');
        return this;
    }
    /**
     * Grants permission to replace a route within a route table in a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceRoute.html
     */
    replaceRoute() {
        this.add('ec2:ReplaceRoute');
        return this;
    }
    /**
     * Grants permission to change the route table that is associated with a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceRouteTableAssociation.html
     */
    replaceRouteTableAssociation() {
        this.add('ec2:ReplaceRouteTableAssociation');
        return this;
    }
    /**
     * Grants permission to replace a route in a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceTransitGatewayRoute.html
     */
    replaceTransitGatewayRoute() {
        this.add('ec2:ReplaceTransitGatewayRoute');
        return this;
    }
    /**
     * Grants permission to submit feedback about the status of an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReportInstanceStatus.html
     */
    reportInstanceStatus() {
        this.add('ec2:ReportInstanceStatus');
        return this;
    }
    /**
     * Grants permission to create a Spot Fleet request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotFleet.html
     */
    requestSpotFleet() {
        this.add('ec2:RequestSpotFleet');
        return this;
    }
    /**
     * Grants permission to create a Spot Instance request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotInstances.html
     */
    requestSpotInstances() {
        this.add('ec2:RequestSpotInstances');
        return this;
    }
    /**
     * Grants permission to reset the default customer master key (CMK) for EBS encryption for your account to use the AWS-managed CMK for EBS
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetEbsDefaultKmsKeyId.html
     */
    resetEbsDefaultKmsKeyId() {
        this.add('ec2:ResetEbsDefaultKmsKeyId');
        return this;
    }
    /**
     * Grants permission to reset an attribute of an Amazon FPGA Image (AFI) to its default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetFpgaImageAttribute.html
     */
    resetFpgaImageAttribute() {
        this.add('ec2:ResetFpgaImageAttribute');
        return this;
    }
    /**
     * Grants permission to reset an attribute of an Amazon Machine Image (AMI) to its default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetImageAttribute.html
     */
    resetImageAttribute() {
        this.add('ec2:ResetImageAttribute');
        return this;
    }
    /**
     * Grants permission to reset an attribute of an instance to its default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetInstanceAttribute.html
     */
    resetInstanceAttribute() {
        this.add('ec2:ResetInstanceAttribute');
        return this;
    }
    /**
     * Grants permission to reset an attribute of a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetNetworkInterfaceAttribute.html
     */
    resetNetworkInterfaceAttribute() {
        this.add('ec2:ResetNetworkInterfaceAttribute');
        return this;
    }
    /**
     * Grants permission to reset permission settings for a snapshot
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetSnapshotAttribute.html
     */
    resetSnapshotAttribute() {
        this.add('ec2:ResetSnapshotAttribute');
        return this;
    }
    /**
     * Grants permission to restore an Elastic IP address that was previously moved to the EC2-VPC platform back to the EC2-Classic platform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RestoreAddressToClassic.html
     */
    restoreAddressToClassic() {
        this.add('ec2:RestoreAddressToClassic');
        return this;
    }
    /**
     * Grants permission to restore the entries from a previous version of a managed prefix list to a new version of the prefix list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RestoreManagedPrefixListVersion.html
     */
    restoreManagedPrefixListVersion() {
        this.add('ec2:RestoreManagedPrefixListVersion');
        return this;
    }
    /**
     * Grants permission to remove an inbound authorization rule from a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeClientVpnIngress.html
     */
    revokeClientVpnIngress() {
        this.add('ec2:RevokeClientVpnIngress');
        return this;
    }
    /**
     * Grants permission to remove one or more outbound rules from a VPC security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeSecurityGroupEgress.html
     */
    revokeSecurityGroupEgress() {
        this.add('ec2:RevokeSecurityGroupEgress');
        return this;
    }
    /**
     * Grants permission to remove one or more inbound rules from a security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeSecurityGroupIngress.html
     */
    revokeSecurityGroupIngress() {
        this.add('ec2:RevokeSecurityGroupIngress');
        return this;
    }
    /**
     * Grants permission to launch one or more instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html
     */
    runInstances() {
        this.add('ec2:RunInstances');
        return this;
    }
    /**
     * Grants permission to launch one or more Scheduled Instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunScheduledInstances.html
     */
    runScheduledInstances() {
        this.add('ec2:RunScheduledInstances');
        return this;
    }
    /**
     * Grants permission to search for routes in a local gateway route table
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchLocalGatewayRoutes.html
     */
    searchLocalGatewayRoutes() {
        this.add('ec2:SearchLocalGatewayRoutes');
        return this;
    }
    /**
     * Grants permission to search for groups, sources, and members in a transit gateway multicast domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchTransitGatewayMulticastGroups.html
     */
    searchTransitGatewayMulticastGroups() {
        this.add('ec2:SearchTransitGatewayMulticastGroups');
        return this;
    }
    /**
     * Grants permission to search for routes in a transit gateway route table
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchTransitGatewayRoutes.html
     */
    searchTransitGatewayRoutes() {
        this.add('ec2:SearchTransitGatewayRoutes');
        return this;
    }
    /**
     * Grants permission to send a diagnostic interrupt to an Amazon EC2 instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SendDiagnosticInterrupt.html
     */
    sendDiagnosticInterrupt() {
        this.add('ec2:SendDiagnosticInterrupt');
        return this;
    }
    /**
     * Grants permission to start a stopped instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StartInstances.html
     */
    startInstances() {
        this.add('ec2:StartInstances');
        return this;
    }
    /**
     * Grants permission to start the private DNS verification process for a VPC endpoint service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StartVpcEndpointServicePrivateDnsVerification.html
     */
    startVpcEndpointServicePrivateDnsVerification() {
        this.add('ec2:StartVpcEndpointServicePrivateDnsVerification');
        return this;
    }
    /**
     * Grants permission to stop an Amazon EBS-backed instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StopInstances.html
     */
    stopInstances() {
        this.add('ec2:StopInstances');
        return this;
    }
    /**
     * Grants permission to terminate active Client VPN endpoint connections
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_TerminateClientVpnConnections.html
     */
    terminateClientVpnConnections() {
        this.add('ec2:TerminateClientVpnConnections');
        return this;
    }
    /**
     * Grants permission to shut down one or more instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_TerminateInstances.html
     */
    terminateInstances() {
        this.add('ec2:TerminateInstances');
        return this;
    }
    /**
     * Grants permission to unassign one or more IPv6 addresses from a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnassignIpv6Addresses.html
     */
    unassignIpv6Addresses() {
        this.add('ec2:UnassignIpv6Addresses');
        return this;
    }
    /**
     * Grants permission to unassign one or more secondary private IP addresses from a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnassignPrivateIpAddresses.html
     */
    unassignPrivateIpAddresses() {
        this.add('ec2:UnassignPrivateIpAddresses');
        return this;
    }
    /**
     * Grants permission to disable detailed monitoring for a running instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnmonitorInstances.html
     */
    unmonitorInstances() {
        this.add('ec2:UnmonitorInstances');
        return this;
    }
    /**
     * Grants permission to update descriptions for one or more outbound rules in a VPC security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UpdateSecurityGroupRuleDescriptionsEgress.html
     */
    updateSecurityGroupRuleDescriptionsEgress() {
        this.add('ec2:UpdateSecurityGroupRuleDescriptionsEgress');
        return this;
    }
    /**
     * Grants permission to update descriptions for one or more inbound rules in a security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UpdateSecurityGroupRuleDescriptionsIngress.html
     */
    updateSecurityGroupRuleDescriptionsIngress() {
        this.add('ec2:UpdateSecurityGroupRuleDescriptionsIngress');
        return this;
    }
    /**
     * Grants permission to stop advertising an address range that was provisioned for use in AWS through bring your own IP addresses (BYOIP)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_WithdrawByoipCidr.html
     */
    withdrawByoipCidr() {
        this.add('ec2:WithdrawByoipCidr');
        return this;
    }
    /**
     * Adds a resource of type capacity-reservation to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-capacity-reservations.html
     *
     * @param capacityReservationId - Identifier for the capacityReservationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onCapacityReservation(capacityReservationId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:capacity-reservation/${CapacityReservationId}';
        arn = arn.replace('${CapacityReservationId}', capacityReservationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type client-vpn-endpoint to the statement
     *
     * https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/what-is.html
     *
     * @param clientVpnEndpointId - Identifier for the clientVpnEndpointId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onClientVpnEndpoint(clientVpnEndpointId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:client-vpn-endpoint/${ClientVpnEndpointId}';
        arn = arn.replace('${ClientVpnEndpointId}', clientVpnEndpointId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type customer-gateway to the statement
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html
     *
     * @param customerGatewayId - Identifier for the customerGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onCustomerGateway(customerGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:customer-gateway/${CustomerGatewayId}';
        arn = arn.replace('${CustomerGatewayId}', customerGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dedicated-host to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/dedicated-hosts-overview.html
     *
     * @param hostId - Identifier for the hostId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAutoPlacement()
     * - .ifAvailabilityZone()
     * - .ifHostRecovery()
     * - .ifInstanceType()
     * - .ifQuantity()
     * - .ifResourceTag()
     */
    onDedicatedHost(hostId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:dedicated-host/${HostId}';
        arn = arn.replace('${HostId}', hostId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dhcp-options to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_DHCP_Options.html
     *
     * @param dhcpOptionsId - Identifier for the dhcpOptionsId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onDhcpOptions(dhcpOptionsId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:dhcp-options/${DhcpOptionsId}';
        arn = arn.replace('${DhcpOptionsId}', dhcpOptionsId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type egress-only-internet-gateway to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/egress-only-internet-gateway.html
     *
     * @param egressOnlyInternetGatewayId - Identifier for the egressOnlyInternetGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onEgressOnlyInternetGateway(egressOnlyInternetGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:egress-only-internet-gateway/${EgressOnlyInternetGatewayId}';
        arn = arn.replace('${EgressOnlyInternetGatewayId}', egressOnlyInternetGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type elastic-gpu to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/elastic-gpus.html
     *
     * @param elasticGpuId - Identifier for the elasticGpuId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifElasticGpuType()
     */
    onElasticGpu(elasticGpuId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:elastic-gpu/${ElasticGpuId}';
        arn = arn.replace('${ElasticGpuId}', elasticGpuId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type elastic-inference to the statement
     *
     * https://docs.aws.amazon.com/elastic-inference/latest/developerguide/what-is-ei.html
     *
     * @param elasticInferenceAcceleratorId - Identifier for the elasticInferenceAcceleratorId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onElasticInference(elasticInferenceAcceleratorId, account, region, partition) {
        var arn = 'arn:${Partition}:elastic-inference:${Region}:${Account}:elastic-inference-accelerator/${ElasticInferenceAcceleratorId}';
        arn = arn.replace('${ElasticInferenceAcceleratorId}', elasticInferenceAcceleratorId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type export-instance-task to the statement
     *
     * https://docs.aws.amazon.com/vm-import/latest/userguide/vmexport.html
     *
     * @param exportTaskId - Identifier for the exportTaskId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onExportInstanceTask(exportTaskId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:export-instance-task/${ExportTaskId}';
        arn = arn.replace('${ExportTaskId}', exportTaskId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type fleet to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet.html
     *
     * @param fleetId - Identifier for the fleetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onFleet(fleetId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:fleet/${FleetId}';
        arn = arn.replace('${FleetId}', fleetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type fpga-image to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#EC2_ARN_Format
     *
     * @param fpgaImageId - Identifier for the fpgaImageId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifOwner()
     * - .ifPublic()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onFpgaImage(fpgaImageId, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}::fpga-image/${FpgaImageId}';
        arn = arn.replace('${FpgaImageId}', fpgaImageId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type image to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/AMIs.html
     *
     * @param imageId - Identifier for the imageId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifImageType()
     * - .ifOwner()
     * - .ifPublic()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifRootDeviceType()
     */
    onImage(imageId, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}::image/${ImageId}';
        arn = arn.replace('${ImageId}', imageId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type instance to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Instances.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAvailabilityZone()
     * - .ifEbsOptimized()
     * - .ifInstanceProfile()
     * - .ifInstanceType()
     * - .ifPlacementGroup()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifRootDeviceType()
     * - .ifTenancy()
     */
    onInstance(instanceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:instance/${InstanceId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type internet-gateway to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Internet_Gateway.html
     *
     * @param internetGatewayId - Identifier for the internetGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onInternetGateway(internetGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:internet-gateway/${InternetGatewayId}';
        arn = arn.replace('${InternetGatewayId}', internetGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type key-pair to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-key-pairs.html
     *
     * @param keyPairName - Identifier for the keyPairName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onKeyPair(keyPairName, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:key-pair/${KeyPairName}';
        arn = arn.replace('${KeyPairName}', keyPairName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type launch-template to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html
     *
     * @param launchTemplateId - Identifier for the launchTemplateId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onLaunchTemplate(launchTemplateId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:launch-template/${LaunchTemplateId}';
        arn = arn.replace('${LaunchTemplateId}', launchTemplateId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayId - Identifier for the localGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onLocalGateway(localGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway/${LocalGatewayId}';
        arn = arn.replace('${LocalGatewayId}', localGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway-route-table to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayRouteTableId - Identifier for the localGatewayRouteTableId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onLocalGatewayRouteTable(localGatewayRouteTableId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table/${LocalGatewayRouteTableId}';
        arn = arn.replace('${LocalGatewayRouteTableId}', localGatewayRouteTableId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway-route-table-virtual-interface-group-association to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayRouteTableVirtualInterfaceGroupAssociationId - Identifier for the localGatewayRouteTableVirtualInterfaceGroupAssociationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onLocalGatewayRouteTableVirtualInterfaceGroupAssociation(localGatewayRouteTableVirtualInterfaceGroupAssociationId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table-virtual-interface-group-association/${LocalGatewayRouteTableVirtualInterfaceGroupAssociationId}';
        arn = arn.replace('${LocalGatewayRouteTableVirtualInterfaceGroupAssociationId}', localGatewayRouteTableVirtualInterfaceGroupAssociationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway-route-table-vpc-association to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayRouteTableVpcAssociationId - Identifier for the localGatewayRouteTableVpcAssociationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onLocalGatewayRouteTableVpcAssociation(localGatewayRouteTableVpcAssociationId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table-vpc-association/${LocalGatewayRouteTableVpcAssociationId}';
        arn = arn.replace('${LocalGatewayRouteTableVpcAssociationId}', localGatewayRouteTableVpcAssociationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway-virtual-interface to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayVirtualInterfaceId - Identifier for the localGatewayVirtualInterfaceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onLocalGatewayVirtualInterface(localGatewayVirtualInterfaceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway-virtual-interface/${LocalGatewayVirtualInterfaceId}';
        arn = arn.replace('${LocalGatewayVirtualInterfaceId}', localGatewayVirtualInterfaceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway-virtual-interface-group to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayVirtualInterfaceGroupId - Identifier for the localGatewayVirtualInterfaceGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onLocalGatewayVirtualInterfaceGroup(localGatewayVirtualInterfaceGroupId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway-virtual-interface-group/${LocalGatewayVirtualInterfaceGroupId}';
        arn = arn.replace('${LocalGatewayVirtualInterfaceGroupId}', localGatewayVirtualInterfaceGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type natgateway to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/vpc-nat-gateway.html
     *
     * @param natGatewayId - Identifier for the natGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onNatgateway(natGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:natgateway/${NatGatewayId}';
        arn = arn.replace('${NatGatewayId}', natGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type network-acl to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/vpc-network-acls.html
     *
     * @param naclId - Identifier for the naclId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifVpc()
     */
    onNetworkAcl(naclId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:network-acl/${NaclId}';
        arn = arn.replace('${NaclId}', naclId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type network-interface to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html
     *
     * @param networkInterfaceId - Identifier for the networkInterfaceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAssociatePublicIpAddress()
     * - .ifAuthorizedService()
     * - .ifAvailabilityZone()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifSubnet()
     * - .ifVpc()
     */
    onNetworkInterface(networkInterfaceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:network-interface/${NetworkInterfaceId}';
        arn = arn.replace('${NetworkInterfaceId}', networkInterfaceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type placement-group to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html
     *
     * @param placementGroupName - Identifier for the placementGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifPlacementGroupStrategy()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onPlacementGroup(placementGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:placement-group/${PlacementGroupName}';
        arn = arn.replace('${PlacementGroupName}', placementGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type prefix-list to the statement
     *
     * @param prefixListId - Identifier for the prefixListId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onPrefixList(prefixListId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:prefix-list/${PrefixListId}';
        arn = arn.replace('${PrefixListId}', prefixListId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type reserved-instances to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-reserved-instances.html
     *
     * @param reservationId - Identifier for the reservationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAvailabilityZone()
     * - .ifInstanceType()
     * - .ifRegion()
     * - .ifReservedInstancesOfferingType()
     * - .ifResourceTag()
     * - .ifTenancy()
     */
    onReservedInstances(reservationId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:reserved-instances/${ReservationId}';
        arn = arn.replace('${ReservationId}', reservationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type route-table to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Route_Tables.html
     *
     * @param routeTableId - Identifier for the routeTableId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifVpc()
     */
    onRouteTable(routeTableId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:route-table/${RouteTableId}';
        arn = arn.replace('${RouteTableId}', routeTableId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type security-group to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-security-groups.html
     *
     * @param securityGroupId - Identifier for the securityGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifVpc()
     */
    onSecurityGroup(securityGroupId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:security-group/${SecurityGroupId}';
        arn = arn.replace('${SecurityGroupId}', securityGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSSnapshots.html
     *
     * @param snapshotId - Identifier for the snapshotId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifOwner()
     * - .ifParentVolume()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifSnapshotTime()
     * - .ifVolumeSize()
     */
    onSnapshot(snapshotId, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}::snapshot/${SnapshotId}';
        arn = arn.replace('${SnapshotId}', snapshotId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type spot-instance-request to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-spot-instances.html
     *
     * @param spotInstanceRequestId - Identifier for the spotInstanceRequestId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onSpotInstanceRequest(spotInstanceRequestId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:spot-instances-request/${SpotInstanceRequestId}';
        arn = arn.replace('${SpotInstanceRequestId}', spotInstanceRequestId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type subnet to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html
     *
     * @param subnetId - Identifier for the subnetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAvailabilityZone()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifVpc()
     */
    onSubnet(subnetId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:subnet/${SubnetId}';
        arn = arn.replace('${SubnetId}', subnetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type traffic-mirror-session to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-session.html
     *
     * @param trafficMirrorSessionId - Identifier for the trafficMirrorSessionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onTrafficMirrorSession(trafficMirrorSessionId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-session/${TrafficMirrorSessionId}';
        arn = arn.replace('${TrafficMirrorSessionId}', trafficMirrorSessionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type traffic-mirror-target to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-target.html
     *
     * @param trafficMirrorTargetId - Identifier for the trafficMirrorTargetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onTrafficMirrorTarget(trafficMirrorTargetId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-target/${TrafficMirrorTargetId}';
        arn = arn.replace('${TrafficMirrorTargetId}', trafficMirrorTargetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type traffic-mirror-filter to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-filter.html
     *
     * @param trafficMirrorFilterId - Identifier for the trafficMirrorFilterId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onTrafficMirrorFilter(trafficMirrorFilterId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-filter/${TrafficMirrorFilterId}';
        arn = arn.replace('${TrafficMirrorFilterId}', trafficMirrorFilterId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type traffic-mirror-filter-rule to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-filter.html
     *
     * @param trafficMirrorFilterRuleId - Identifier for the trafficMirrorFilterRuleId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifRegion()
     */
    onTrafficMirrorFilterRule(trafficMirrorFilterRuleId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-filter-rule/${TrafficMirrorFilterRuleId}';
        arn = arn.replace('${TrafficMirrorFilterRuleId}', trafficMirrorFilterRuleId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type transit-gateway-attachment to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html
     *
     * @param transitGatewayAttachmentId - Identifier for the transitGatewayAttachmentId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onTransitGatewayAttachment(transitGatewayAttachmentId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-attachment/${TransitGatewayAttachmentId}';
        arn = arn.replace('${TransitGatewayAttachmentId}', transitGatewayAttachmentId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type transit-gateway-multicast-domain to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/tgw-multicast-overview.html
     *
     * @param transitGatewayMulticastDomainId - Identifier for the transitGatewayMulticastDomainId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onTransitGatewayMulticastDomain(transitGatewayMulticastDomainId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-multicast-domain/${TransitGatewayMulticastDomainId}';
        arn = arn.replace('${TransitGatewayMulticastDomainId}', transitGatewayMulticastDomainId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type transit-gateway-route-table to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html
     *
     * @param transitGatewayRouteTableId - Identifier for the transitGatewayRouteTableId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onTransitGatewayRouteTable(transitGatewayRouteTableId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-route-table/${TransitGatewayRouteTableId}';
        arn = arn.replace('${TransitGatewayRouteTableId}', transitGatewayRouteTableId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type transit-gateway to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html
     *
     * @param transitGatewayId - Identifier for the transitGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onTransitGateway(transitGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:transit-gateway/${TransitGatewayId}';
        arn = arn.replace('${TransitGatewayId}', transitGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type volume to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volumes.html
     *
     * @param volumeId - Identifier for the volumeId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAvailabilityZone()
     * - .ifEncrypted()
     * - .ifParentSnapshot()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifVolumeIops()
     * - .ifVolumeSize()
     * - .ifVolumeType()
     */
    onVolume(volumeId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:volume/${VolumeId}';
        arn = arn.replace('${VolumeId}', volumeId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpc to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html
     *
     * @param vpcId - Identifier for the vpcId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifTenancy()
     */
    onVpc(vpcId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpc/${VpcId}';
        arn = arn.replace('${VpcId}', vpcId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpc-endpoint to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-services-overview.html
     *
     * @param vpceId - Identifier for the vpceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifVpceServiceName()
     * - .ifVpceServiceOwner()
     */
    onVpcEndpoint(vpceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpc-endpoint/${VpceId}';
        arn = arn.replace('${VpceId}', vpceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpc-endpoint-service to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-services-overview.html
     *
     * @param vpceServiceId - Identifier for the vpceServiceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifVpceServicePrivateDnsName()
     */
    onVpcEndpointService(vpceServiceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpc-endpoint-service/${VpceServiceId}';
        arn = arn.replace('${VpceServiceId}', vpceServiceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpc-flow-log to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html
     *
     * @param vpcFlowLogId - Identifier for the vpcFlowLogId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onVpcFlowLog(vpcFlowLogId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpc-flow-log/${VpcFlowLogId}';
        arn = arn.replace('${VpcFlowLogId}', vpcFlowLogId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpc-peering-connection to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html
     *
     * @param vpcPeeringConnectionId - Identifier for the vpcPeeringConnectionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAccepterVpc()
     * - .ifRegion()
     * - .ifRequesterVpc()
     * - .ifResourceTag()
     */
    onVpcPeeringConnection(vpcPeeringConnectionId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpc-peering-connection/${VpcPeeringConnectionId}';
        arn = arn.replace('${VpcPeeringConnectionId}', vpcPeeringConnectionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpn-connection to the statement
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html
     *
     * @param vpnConnectionId - Identifier for the vpnConnectionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAuthenticationType()
     * - .ifDPDTimeoutSeconds()
     * - .ifGatewayType()
     * - .ifIKEVersions()
     * - .ifInsideTunnelCidr()
     * - .ifPhase1DHGroupNumbers()
     * - .ifPhase1EncryptionAlgorithms()
     * - .ifPhase1IntegrityAlgorithms()
     * - .ifPhase1LifetimeSeconds()
     * - .ifPhase2DHGroupNumbers()
     * - .ifPhase2EncryptionAlgorithms()
     * - .ifPhase2IntegrityAlgorithms()
     * - .ifPhase2LifetimeSeconds()
     * - .ifPresharedKeys()
     * - .ifRegion()
     * - .ifRekeyFuzzPercentage()
     * - .ifRekeyMarginTimeSeconds()
     * - .ifResourceTag()
     * - .ifRoutingType()
     */
    onVpnConnection(vpnConnectionId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpn-connection/${VpnConnectionId}';
        arn = arn.replace('${VpnConnectionId}', vpnConnectionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpn-gateway to the statement
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html
     *
     * @param vpnGatewayId - Identifier for the vpnGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onVpnGateway(vpnGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpn-gateway/${VpnGatewayId}';
        arn = arn.replace('${VpnGatewayId}', vpnGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by a tag key and value pair that is allowed in the request
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/supported-iam-actions-tagging.html#control-tagging
     *
     * Applies to actions:
     * - .copySnapshot()
     * - .createCapacityReservation()
     * - .createClientVpnEndpoint()
     * - .createDhcpOptions()
     * - .createEgressOnlyInternetGateway()
     * - .createFleet()
     * - .createFlowLogs()
     * - .createFpgaImage()
     * - .createInstanceExportTask()
     * - .createInternetGateway()
     * - .createKeyPair()
     * - .createLaunchTemplate()
     * - .createLocalGatewayRouteTableVpcAssociation()
     * - .createManagedPrefixList()
     * - .createNatGateway()
     * - .createNetworkAcl()
     * - .createNetworkInterface()
     * - .createPlacementGroup()
     * - .createSecurityGroup()
     * - .createSnapshot()
     * - .createSnapshots()
     * - .createSubnet()
     * - .createTags()
     * - .createTrafficMirrorFilter()
     * - .createTrafficMirrorSession()
     * - .createTrafficMirrorTarget()
     * - .createTransitGateway()
     * - .createTransitGatewayMulticastDomain()
     * - .createTransitGatewayPeeringAttachment()
     * - .createTransitGatewayRouteTable()
     * - .createTransitGatewayVpcAttachment()
     * - .createVolume()
     * - .createVpc()
     * - .createVpcEndpoint()
     * - .createVpcEndpointServiceConfiguration()
     * - .deleteTags()
     * - .runInstances()
     *
     * Applies to resource types:
     * - capacity-reservation
     * - client-vpn-endpoint
     * - dedicated-host
     * - dhcp-options
     * - egress-only-internet-gateway
     * - export-instance-task
     * - fleet
     * - fpga-image
     * - image
     * - instance
     * - internet-gateway
     * - key-pair
     * - launch-template
     * - local-gateway
     * - local-gateway-route-table
     * - local-gateway-route-table-virtual-interface-group-association
     * - local-gateway-route-table-vpc-association
     * - local-gateway-virtual-interface
     * - local-gateway-virtual-interface-group
     * - natgateway
     * - network-acl
     * - network-interface
     * - placement-group
     * - prefix-list
     * - reserved-instances
     * - route-table
     * - security-group
     * - snapshot
     * - spot-instance-request
     * - subnet
     * - traffic-mirror-session
     * - traffic-mirror-target
     * - traffic-mirror-filter
     * - transit-gateway-attachment
     * - transit-gateway-multicast-domain
     * - transit-gateway-route-table
     * - transit-gateway
     * - volume
     * - vpc
     * - vpc-endpoint
     * - vpc-endpoint-service
     * - vpc-flow-log
     * - vpn-connection
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the preface string for a tag key and value pair that are attached to a resource
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/control-access-with-tags.html
     *
     * Applies to actions:
     * - .runInstances()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTagExists(value, operator) {
        return this.if(`aws:ResourceTag/`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair of a resource
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/control-access-with-tags.html
     *
     * Applies to actions:
     * - .acceptTransitGatewayPeeringAttachment()
     * - .acceptTransitGatewayVpcAttachment()
     * - .acceptVpcEndpointConnections()
     * - .acceptVpcPeeringConnection()
     * - .applySecurityGroupsToClientVpnTargetNetwork()
     * - .associateClientVpnTargetNetwork()
     * - .associateIamInstanceProfile()
     * - .associateTransitGatewayMulticastDomain()
     * - .associateTransitGatewayRouteTable()
     * - .attachClassicLinkVpc()
     * - .attachVolume()
     * - .authorizeClientVpnIngress()
     * - .authorizeSecurityGroupEgress()
     * - .authorizeSecurityGroupIngress()
     * - .cancelCapacityReservation()
     * - .createClientVpnRoute()
     * - .createFlowLogs()
     * - .createLaunchTemplateVersion()
     * - .createLocalGatewayRoute()
     * - .createLocalGatewayRouteTableVpcAssociation()
     * - .createNetworkInterfacePermission()
     * - .createRoute()
     * - .createSnapshot()
     * - .createSnapshots()
     * - .createTags()
     * - .createTrafficMirrorFilterRule()
     * - .createTrafficMirrorSession()
     * - .createTrafficMirrorTarget()
     * - .createTransitGatewayMulticastDomain()
     * - .createTransitGatewayPeeringAttachment()
     * - .createTransitGatewayRoute()
     * - .createTransitGatewayRouteTable()
     * - .createTransitGatewayVpcAttachment()
     * - .createVpcEndpoint()
     * - .createVpcPeeringConnection()
     * - .deleteClientVpnEndpoint()
     * - .deleteClientVpnRoute()
     * - .deleteCustomerGateway()
     * - .deleteDhcpOptions()
     * - .deleteFlowLogs()
     * - .deleteInternetGateway()
     * - .deleteLaunchTemplate()
     * - .deleteLaunchTemplateVersions()
     * - .deleteLocalGatewayRoute()
     * - .deleteLocalGatewayRouteTableVpcAssociation()
     * - .deleteManagedPrefixList()
     * - .deleteNetworkAcl()
     * - .deleteNetworkAclEntry()
     * - .deleteRoute()
     * - .deleteRouteTable()
     * - .deleteSecurityGroup()
     * - .deleteSnapshot()
     * - .deleteTags()
     * - .deleteTrafficMirrorFilter()
     * - .deleteTrafficMirrorFilterRule()
     * - .deleteTrafficMirrorSession()
     * - .deleteTrafficMirrorTarget()
     * - .deleteTransitGateway()
     * - .deleteTransitGatewayMulticastDomain()
     * - .deleteTransitGatewayPeeringAttachment()
     * - .deleteTransitGatewayRoute()
     * - .deleteTransitGatewayRouteTable()
     * - .deleteTransitGatewayVpcAttachment()
     * - .deleteVolume()
     * - .deleteVpcEndpointServiceConfigurations()
     * - .deleteVpcEndpoints()
     * - .deleteVpcPeeringConnection()
     * - .deregisterTransitGatewayMulticastGroupMembers()
     * - .deregisterTransitGatewayMulticastGroupSources()
     * - .detachClassicLinkVpc()
     * - .detachVolume()
     * - .disableFastSnapshotRestores()
     * - .disableTransitGatewayRouteTablePropagation()
     * - .disableVpcClassicLink()
     * - .disassociateClientVpnTargetNetwork()
     * - .disassociateIamInstanceProfile()
     * - .disassociateTransitGatewayMulticastDomain()
     * - .disassociateTransitGatewayRouteTable()
     * - .enableFastSnapshotRestores()
     * - .enableTransitGatewayRouteTablePropagation()
     * - .enableVpcClassicLink()
     * - .getConsoleScreenshot()
     * - .importClientVpnClientCertificateRevocationList()
     * - .modifyCapacityReservation()
     * - .modifyClientVpnEndpoint()
     * - .modifyLaunchTemplate()
     * - .modifyManagedPrefixList()
     * - .modifySnapshotAttribute()
     * - .modifyTrafficMirrorFilterNetworkServices()
     * - .modifyTrafficMirrorFilterRule()
     * - .modifyTrafficMirrorSession()
     * - .modifyTransitGatewayVpcAttachment()
     * - .modifyVpcEndpoint()
     * - .modifyVpcEndpointServiceConfiguration()
     * - .modifyVpcEndpointServicePermissions()
     * - .modifyVpnConnection()
     * - .modifyVpnTunnelOptions()
     * - .rebootInstances()
     * - .registerTransitGatewayMulticastGroupMembers()
     * - .registerTransitGatewayMulticastGroupSources()
     * - .rejectTransitGatewayPeeringAttachment()
     * - .rejectTransitGatewayVpcAttachment()
     * - .rejectVpcEndpointConnections()
     * - .rejectVpcPeeringConnection()
     * - .replaceIamInstanceProfileAssociation()
     * - .replaceRoute()
     * - .replaceTransitGatewayRoute()
     * - .restoreManagedPrefixListVersion()
     * - .revokeClientVpnIngress()
     * - .revokeSecurityGroupEgress()
     * - .revokeSecurityGroupIngress()
     * - .runInstances()
     * - .sendDiagnosticInterrupt()
     * - .startInstances()
     * - .startVpcEndpointServicePrivateDnsVerification()
     * - .stopInstances()
     * - .terminateClientVpnConnections()
     * - .terminateInstances()
     * - .updateSecurityGroupRuleDescriptionsEgress()
     * - .updateSecurityGroupRuleDescriptionsIngress()
     *
     * Applies to resource types:
     * - capacity-reservation
     * - client-vpn-endpoint
     * - customer-gateway
     * - dedicated-host
     * - dhcp-options
     * - egress-only-internet-gateway
     * - export-instance-task
     * - fleet
     * - fpga-image
     * - image
     * - instance
     * - internet-gateway
     * - key-pair
     * - launch-template
     * - local-gateway
     * - local-gateway-route-table
     * - local-gateway-route-table-virtual-interface-group-association
     * - local-gateway-route-table-vpc-association
     * - local-gateway-virtual-interface
     * - local-gateway-virtual-interface-group
     * - natgateway
     * - network-acl
     * - network-interface
     * - placement-group
     * - prefix-list
     * - reserved-instances
     * - route-table
     * - security-group
     * - snapshot
     * - spot-instance-request
     * - subnet
     * - traffic-mirror-session
     * - traffic-mirror-target
     * - traffic-mirror-filter
     * - transit-gateway-attachment
     * - transit-gateway-multicast-domain
     * - transit-gateway-route-table
     * - transit-gateway
     * - volume
     * - vpc
     * - vpc-endpoint
     * - vpc-endpoint-service
     * - vpc-flow-log
     * - vpc-peering-connection
     * - vpn-connection
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a list of tag keys that are allowed in the request
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/supported-iam-actions-tagging.html#control-tagging
     *
     * Applies to actions:
     * - .copySnapshot()
     * - .createCapacityReservation()
     * - .createClientVpnEndpoint()
     * - .createDhcpOptions()
     * - .createEgressOnlyInternetGateway()
     * - .createFleet()
     * - .createFlowLogs()
     * - .createFpgaImage()
     * - .createInstanceExportTask()
     * - .createInternetGateway()
     * - .createKeyPair()
     * - .createLaunchTemplate()
     * - .createLocalGatewayRouteTableVpcAssociation()
     * - .createManagedPrefixList()
     * - .createNatGateway()
     * - .createNetworkAcl()
     * - .createNetworkInterface()
     * - .createPlacementGroup()
     * - .createSecurityGroup()
     * - .createSnapshot()
     * - .createSnapshots()
     * - .createSubnet()
     * - .createTags()
     * - .createTrafficMirrorFilter()
     * - .createTrafficMirrorSession()
     * - .createTrafficMirrorTarget()
     * - .createTransitGateway()
     * - .createTransitGatewayMulticastDomain()
     * - .createTransitGatewayPeeringAttachment()
     * - .createTransitGatewayRouteTable()
     * - .createTransitGatewayVpcAttachment()
     * - .createVolume()
     * - .createVpc()
     * - .createVpcEndpoint()
     * - .createVpcEndpointServiceConfiguration()
     * - .deleteTags()
     * - .runInstances()
     *
     * Applies to resource types:
     * - capacity-reservation
     * - client-vpn-endpoint
     * - dedicated-host
     * - dhcp-options
     * - egress-only-internet-gateway
     * - export-instance-task
     * - fleet
     * - fpga-image
     * - image
     * - instance
     * - internet-gateway
     * - key-pair
     * - launch-template
     * - local-gateway
     * - local-gateway-route-table
     * - local-gateway-route-table-virtual-interface-group-association
     * - local-gateway-route-table-vpc-association
     * - local-gateway-virtual-interface
     * - local-gateway-virtual-interface-group
     * - natgateway
     * - network-acl
     * - network-interface
     * - placement-group
     * - prefix-list
     * - reserved-instances
     * - route-table
     * - security-group
     * - snapshot
     * - spot-instance-request
     * - subnet
     * - traffic-mirror-session
     * - traffic-mirror-target
     * - traffic-mirror-filter
     * - transit-gateway-attachment
     * - transit-gateway-multicast-domain
     * - transit-gateway-route-table
     * - transit-gateway
     * - volume
     * - vpc
     * - vpc-endpoint
     * - vpc-endpoint-service
     * - vpc-flow-log
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of an accepter VPC in a VPC peering connection
     *
     * https://docs.aws.amazon.com/vpc/latest/peering/security-iam.html
     *
     * Applies to actions:
     * - .acceptVpcPeeringConnection()
     * - .createVpcPeeringConnection()
     * - .deleteVpcPeeringConnection()
     * - .rejectVpcPeeringConnection()
     *
     * Applies to resource types:
     * - vpc-peering-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifAccepterVpc(value, operator) {
        return this.if(`ec2:AccepterVpc`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by whether the user wants to associate a public IP address with the instance
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .createTags()
     * - .runInstances()
     *
     * Applies to resource types:
     * - network-interface
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifAssociatePublicIpAddress(value) {
        return this.if(`ec2:AssociatePublicIpAddress`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the authentication type for the VPN tunnel endpoints
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .createVpnConnection()
     * - .modifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthenticationType(value, operator) {
        return this.if(`ec2:AuthenticationType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the AWS service that has permission to use a resource
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .createNetworkInterfacePermission()
     *
     * Applies to resource types:
     * - network-interface
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthorizedService(value, operator) {
        return this.if(`ec2:AuthorizedService`, value, operator || 'StringLike');
    }
    /**
     * Filters access by an IAM principal that has permission to use a resource
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .createNetworkInterfacePermission()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthorizedUser(value, operator) {
        return this.if(`ec2:AuthorizedUser`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the Auto Placement properties of a Dedicated Host
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to resource types:
     * - dedicated-host
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAutoPlacement(value, operator) {
        return this.if(`ec2:AutoPlacement`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the name of an Availability Zone in an AWS Region
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .associateIamInstanceProfile()
     * - .attachClassicLinkVpc()
     * - .attachVolume()
     * - .createFlowLogs()
     * - .createNetworkInterfacePermission()
     * - .createSnapshots()
     * - .createTags()
     * - .createVolume()
     * - .deleteVolume()
     * - .detachClassicLinkVpc()
     * - .detachVolume()
     * - .disableFastSnapshotRestores()
     * - .disassociateIamInstanceProfile()
     * - .enableFastSnapshotRestores()
     * - .getConsoleScreenshot()
     * - .rebootInstances()
     * - .replaceIamInstanceProfileAssociation()
     * - .runInstances()
     * - .sendDiagnosticInterrupt()
     * - .startInstances()
     * - .stopInstances()
     * - .terminateInstances()
     *
     * Applies to resource types:
     * - dedicated-host
     * - instance
     * - network-interface
     * - reserved-instances
     * - subnet
     * - volume
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAvailabilityZone(value, operator) {
        return this.if(`ec2:AvailabilityZone`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the name of a resource-creating API action
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/supported-iam-actions-tagging.html
     *
     * Applies to actions:
     * - .createTags()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCreateAction(value, operator) {
        return this.if(`ec2:CreateAction`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the duration after which DPD timeout occurs on a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .createVpnConnection()
     * - .modifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifDPDTimeoutSeconds(value, operator) {
        return this.if(`ec2:DPDTimeoutSeconds`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by whether the instance is enabled for EBS optimization
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .associateIamInstanceProfile()
     * - .attachClassicLinkVpc()
     * - .attachVolume()
     * - .createSnapshots()
     * - .createTags()
     * - .detachClassicLinkVpc()
     * - .detachVolume()
     * - .disassociateIamInstanceProfile()
     * - .getConsoleScreenshot()
     * - .rebootInstances()
     * - .replaceIamInstanceProfileAssociation()
     * - .runInstances()
     * - .sendDiagnosticInterrupt()
     * - .startInstances()
     * - .stopInstances()
     * - .terminateInstances()
     *
     * Applies to resource types:
     * - instance
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifEbsOptimized(value) {
        return this.if(`ec2:EbsOptimized`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the type of Elastic Graphics accelerator
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .runInstances()
     *
     * Applies to resource types:
     * - elastic-gpu
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifElasticGpuType(value, operator) {
        return this.if(`ec2:ElasticGpuType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether the EBS volume is encrypted
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .attachVolume()
     * - .createSnapshot()
     * - .createSnapshots()
     * - .createTags()
     * - .createVolume()
     * - .deleteVolume()
     * - .detachVolume()
     * - .disableFastSnapshotRestores()
     * - .enableFastSnapshotRestores()
     * - .runInstances()
     *
     * Applies to resource types:
     * - volume
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifEncrypted(value) {
        return this.if(`ec2:Encrypted`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the gateway type for a VPN endpoint on the AWS side of a VPN connection
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .createVpnConnection()
     * - .modifyVpnConnection()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGatewayType(value, operator) {
        return this.if(`ec2:GatewayType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether host recovery is enabled for a Dedicated Host
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to resource types:
     * - dedicated-host
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifHostRecovery(value, operator) {
        return this.if(`ec2:HostRecovery`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the internet key exchange (IKE) versions that are permitted for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .createVpnConnection()
     * - .modifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIKEVersions(value, operator) {
        return this.if(`ec2:IKEVersions`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the type of image (machine, aki, or ari)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .createTags()
     * - .runInstances()
     *
     * Applies to resource types:
     * - image
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifImageType(value, operator) {
        return this.if(`ec2:ImageType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the range of inside IP addresses for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .createVpnConnection()
     * - .modifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInsideTunnelCidr(value, operator) {
        return this.if(`ec2:InsideTunnelCidr`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the market or purchasing option of an instance (on-demand or spot)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceMarketType(value, operator) {
        return this.if(`ec2:InstanceMarketType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of an instance profile
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .associateIamInstanceProfile()
     * - .attachClassicLinkVpc()
     * - .attachVolume()
     * - .createSnapshots()
     * - .createTags()
     * - .detachClassicLinkVpc()
     * - .detachVolume()
     * - .disassociateIamInstanceProfile()
     * - .getConsoleScreenshot()
     * - .rebootInstances()
     * - .replaceIamInstanceProfileAssociation()
     * - .runInstances()
     * - .sendDiagnosticInterrupt()
     * - .startInstances()
     * - .stopInstances()
     * - .terminateInstances()
     *
     * Applies to resource types:
     * - instance
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifInstanceProfile(value, operator) {
        return this.if(`ec2:InstanceProfile`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the type of instance
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .associateIamInstanceProfile()
     * - .attachClassicLinkVpc()
     * - .attachVolume()
     * - .createSnapshots()
     * - .createTags()
     * - .detachClassicLinkVpc()
     * - .detachVolume()
     * - .disassociateIamInstanceProfile()
     * - .getConsoleScreenshot()
     * - .rebootInstances()
     * - .replaceIamInstanceProfileAssociation()
     * - .runInstances()
     * - .sendDiagnosticInterrupt()
     * - .startInstances()
     * - .stopInstances()
     * - .terminateInstances()
     *
     * Applies to resource types:
     * - dedicated-host
     * - instance
     * - reserved-instances
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceType(value, operator) {
        return this.if(`ec2:InstanceType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether users are able to override resources that are specified in the launch template
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .runInstances()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifIsLaunchTemplateResource(value) {
        return this.if(`ec2:IsLaunchTemplateResource`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the ARN of a launch template
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .runInstances()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifLaunchTemplate(value, operator) {
        return this.if(`ec2:LaunchTemplate`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by whether the HTTP endpoint is enabled for the instance metadata service
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .runInstances()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMetadataHttpEndpoint(value, operator) {
        return this.if(`ec2:MetadataHttpEndpoint`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the allowed number of hops when calling the instance metadata service
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .runInstances()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMetadataHttpPutResponseHopLimit(value, operator) {
        return this.if(`ec2:MetadataHttpPutResponseHopLimit`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by whether tokens are required when calling the instance metadata service (optional or required)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .runInstances()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMetadataHttpTokens(value, operator) {
        return this.if(`ec2:MetadataHttpTokens`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the owner of the resource (amazon, aws-marketplace, or an AWS account ID)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .createTags()
     * - .deleteSnapshot()
     * - .disableFastSnapshotRestores()
     * - .enableFastSnapshotRestores()
     * - .modifySnapshotAttribute()
     * - .runInstances()
     *
     * Applies to resource types:
     * - fpga-image
     * - image
     * - snapshot
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOwner(value, operator) {
        return this.if(`ec2:Owner`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of the parent snapshot
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .attachVolume()
     * - .createTags()
     * - .createVolume()
     * - .deleteVolume()
     * - .detachVolume()
     * - .runInstances()
     *
     * Applies to resource types:
     * - volume
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifParentSnapshot(value, operator) {
        return this.if(`ec2:ParentSnapshot`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the ARN of the parent volume from which the snapshot was created
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .createSnapshot()
     * - .createSnapshots()
     * - .createTags()
     * - .deleteSnapshot()
     * - .disableFastSnapshotRestores()
     * - .enableFastSnapshotRestores()
     * - .modifySnapshotAttribute()
     * - .runInstances()
     *
     * Applies to resource types:
     * - snapshot
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifParentVolume(value, operator) {
        return this.if(`ec2:ParentVolume`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the type of permission for a resource (INSTANCE-ATTACH or EIP-ASSOCIATE)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .createNetworkInterfacePermission()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPermission(value, operator) {
        return this.if(`ec2:Permission`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the Diffie-Hellman group numbers that are permitted for a VPN tunnel for the phase 1 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .createVpnConnection()
     * - .modifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifPhase1DHGroupNumbers(value, operator) {
        return this.if(`ec2:Phase1DHGroupNumbers`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the encryption algorithms that are permitted for a VPN tunnel for the phase 1 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.htmls
     *
     * Applies to actions:
     * - .createVpnConnection()
     * - .modifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPhase1EncryptionAlgorithms(value, operator) {
        return this.if(`ec2:Phase1EncryptionAlgorithms`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the integrity algorithms that are permitted for a VPN tunnel for the phase 1 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .createVpnConnection()
     * - .modifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPhase1IntegrityAlgorithms(value, operator) {
        return this.if(`ec2:Phase1IntegrityAlgorithms`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the lifetime in seconds for phase 1 of the IKE negotiations for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .createVpnConnection()
     * - .modifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifPhase1LifetimeSeconds(value, operator) {
        return this.if(`ec2:Phase1LifetimeSeconds`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the Diffie-Hellman group numbers that are permitted for a VPN tunnel for the phase 2 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .createVpnConnection()
     * - .modifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifPhase2DHGroupNumbers(value, operator) {
        return this.if(`ec2:Phase2DHGroupNumbers`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the encryption algorithms that are permitted for a VPN tunnel for the phase 2 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .createVpnConnection()
     * - .modifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPhase2EncryptionAlgorithms(value, operator) {
        return this.if(`ec2:Phase2EncryptionAlgorithms`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the integrity algorithms that are permitted for a VPN tunnel for the phase 2 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .createVpnConnection()
     * - .modifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPhase2IntegrityAlgorithms(value, operator) {
        return this.if(`ec2:Phase2IntegrityAlgorithms`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the lifetime in seconds for phase 2 of the IKE negotiations for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .createVpnConnection()
     * - .modifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifPhase2LifetimeSeconds(value, operator) {
        return this.if(`ec2:Phase2LifetimeSeconds`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the ARN of the placement group
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .associateIamInstanceProfile()
     * - .attachClassicLinkVpc()
     * - .attachVolume()
     * - .createSnapshots()
     * - .createTags()
     * - .detachClassicLinkVpc()
     * - .detachVolume()
     * - .disassociateIamInstanceProfile()
     * - .getConsoleScreenshot()
     * - .rebootInstances()
     * - .replaceIamInstanceProfileAssociation()
     * - .runInstances()
     * - .sendDiagnosticInterrupt()
     * - .startInstances()
     * - .stopInstances()
     * - .terminateInstances()
     *
     * Applies to resource types:
     * - instance
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifPlacementGroup(value, operator) {
        return this.if(`ec2:PlacementGroup`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the instance placement strategy used by the placement group (cluster, spread, or partition)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .runInstances()
     *
     * Applies to resource types:
     * - placement-group
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPlacementGroupStrategy(value, operator) {
        return this.if(`ec2:PlacementGroupStrategy`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the pre-shared key (PSK) used to establish the initial IKE security association between a virtual private gateway and a customer gateway
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .createVpnConnection()
     * - .modifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPresharedKeys(value, operator) {
        return this.if(`ec2:PresharedKeys`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the product code that is associated with the AMI
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProductCode(value, operator) {
        return this.if(`ec2:ProductCode`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether the image has public launch permissions
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .createTags()
     * - .runInstances()
     *
     * Applies to resource types:
     * - fpga-image
     * - image
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifPublic(value) {
        return this.if(`ec2:Public`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the number of Dedicated Hosts in a request
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to resource types:
     * - dedicated-host
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifQuantity(value, operator) {
        return this.if(`ec2:Quantity`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the name of the AWS Region
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .acceptTransitGatewayPeeringAttachment()
     * - .acceptTransitGatewayVpcAttachment()
     * - .acceptVpcEndpointConnections()
     * - .acceptVpcPeeringConnection()
     * - .applySecurityGroupsToClientVpnTargetNetwork()
     * - .associateClientVpnTargetNetwork()
     * - .associateIamInstanceProfile()
     * - .associateTransitGatewayMulticastDomain()
     * - .associateTransitGatewayRouteTable()
     * - .attachClassicLinkVpc()
     * - .attachVolume()
     * - .authorizeClientVpnIngress()
     * - .authorizeSecurityGroupEgress()
     * - .authorizeSecurityGroupIngress()
     * - .cancelCapacityReservation()
     * - .copySnapshot()
     * - .createCapacityReservation()
     * - .createClientVpnEndpoint()
     * - .createClientVpnRoute()
     * - .createDhcpOptions()
     * - .createEgressOnlyInternetGateway()
     * - .createFleet()
     * - .createFlowLogs()
     * - .createFpgaImage()
     * - .createInstanceExportTask()
     * - .createInternetGateway()
     * - .createKeyPair()
     * - .createLaunchTemplate()
     * - .createLaunchTemplateVersion()
     * - .createLocalGatewayRoute()
     * - .createLocalGatewayRouteTableVpcAssociation()
     * - .createManagedPrefixList()
     * - .createNatGateway()
     * - .createNetworkAcl()
     * - .createNetworkInterface()
     * - .createNetworkInterfacePermission()
     * - .createPlacementGroup()
     * - .createRoute()
     * - .createSecurityGroup()
     * - .createSnapshot()
     * - .createSnapshots()
     * - .createSubnet()
     * - .createTags()
     * - .createTrafficMirrorFilter()
     * - .createTrafficMirrorFilterRule()
     * - .createTrafficMirrorSession()
     * - .createTrafficMirrorTarget()
     * - .createTransitGateway()
     * - .createTransitGatewayMulticastDomain()
     * - .createTransitGatewayPeeringAttachment()
     * - .createTransitGatewayRoute()
     * - .createTransitGatewayRouteTable()
     * - .createTransitGatewayVpcAttachment()
     * - .createVolume()
     * - .createVpc()
     * - .createVpcEndpoint()
     * - .createVpcEndpointServiceConfiguration()
     * - .createVpcPeeringConnection()
     * - .createVpnConnection()
     * - .deleteClientVpnEndpoint()
     * - .deleteClientVpnRoute()
     * - .deleteCustomerGateway()
     * - .deleteDhcpOptions()
     * - .deleteFlowLogs()
     * - .deleteInternetGateway()
     * - .deleteLaunchTemplate()
     * - .deleteLaunchTemplateVersions()
     * - .deleteLocalGatewayRoute()
     * - .deleteLocalGatewayRouteTableVpcAssociation()
     * - .deleteManagedPrefixList()
     * - .deleteNetworkAcl()
     * - .deleteNetworkAclEntry()
     * - .deleteRoute()
     * - .deleteRouteTable()
     * - .deleteSecurityGroup()
     * - .deleteSnapshot()
     * - .deleteTags()
     * - .deleteTrafficMirrorFilter()
     * - .deleteTrafficMirrorFilterRule()
     * - .deleteTrafficMirrorSession()
     * - .deleteTrafficMirrorTarget()
     * - .deleteTransitGateway()
     * - .deleteTransitGatewayMulticastDomain()
     * - .deleteTransitGatewayPeeringAttachment()
     * - .deleteTransitGatewayRoute()
     * - .deleteTransitGatewayRouteTable()
     * - .deleteTransitGatewayVpcAttachment()
     * - .deleteVolume()
     * - .deleteVpcEndpointServiceConfigurations()
     * - .deleteVpcEndpoints()
     * - .deleteVpcPeeringConnection()
     * - .deregisterTransitGatewayMulticastGroupMembers()
     * - .deregisterTransitGatewayMulticastGroupSources()
     * - .detachClassicLinkVpc()
     * - .detachVolume()
     * - .disableFastSnapshotRestores()
     * - .disableTransitGatewayRouteTablePropagation()
     * - .disableVpcClassicLink()
     * - .disassociateClientVpnTargetNetwork()
     * - .disassociateIamInstanceProfile()
     * - .disassociateTransitGatewayMulticastDomain()
     * - .disassociateTransitGatewayRouteTable()
     * - .enableFastSnapshotRestores()
     * - .enableTransitGatewayRouteTablePropagation()
     * - .enableVpcClassicLink()
     * - .getConsoleScreenshot()
     * - .importClientVpnClientCertificateRevocationList()
     * - .modifyCapacityReservation()
     * - .modifyClientVpnEndpoint()
     * - .modifyInstanceEventStartTime()
     * - .modifyLaunchTemplate()
     * - .modifyManagedPrefixList()
     * - .modifySnapshotAttribute()
     * - .modifyTrafficMirrorFilterNetworkServices()
     * - .modifyTrafficMirrorFilterRule()
     * - .modifyTrafficMirrorSession()
     * - .modifyTransitGatewayVpcAttachment()
     * - .modifyVpcEndpoint()
     * - .modifyVpcEndpointServiceConfiguration()
     * - .modifyVpcEndpointServicePermissions()
     * - .modifyVpnConnection()
     * - .modifyVpnTunnelOptions()
     * - .rebootInstances()
     * - .registerTransitGatewayMulticastGroupMembers()
     * - .registerTransitGatewayMulticastGroupSources()
     * - .rejectTransitGatewayPeeringAttachment()
     * - .rejectTransitGatewayVpcAttachment()
     * - .rejectVpcEndpointConnections()
     * - .rejectVpcPeeringConnection()
     * - .replaceIamInstanceProfileAssociation()
     * - .replaceRoute()
     * - .replaceTransitGatewayRoute()
     * - .restoreManagedPrefixListVersion()
     * - .revokeClientVpnIngress()
     * - .revokeSecurityGroupEgress()
     * - .revokeSecurityGroupIngress()
     * - .runInstances()
     * - .sendDiagnosticInterrupt()
     * - .startInstances()
     * - .startVpcEndpointServicePrivateDnsVerification()
     * - .stopInstances()
     * - .terminateClientVpnConnections()
     * - .terminateInstances()
     * - .updateSecurityGroupRuleDescriptionsEgress()
     * - .updateSecurityGroupRuleDescriptionsIngress()
     *
     * Applies to resource types:
     * - capacity-reservation
     * - client-vpn-endpoint
     * - customer-gateway
     * - dhcp-options
     * - egress-only-internet-gateway
     * - export-instance-task
     * - fleet
     * - fpga-image
     * - image
     * - instance
     * - internet-gateway
     * - key-pair
     * - launch-template
     * - local-gateway
     * - local-gateway-route-table
     * - local-gateway-route-table-virtual-interface-group-association
     * - local-gateway-route-table-vpc-association
     * - local-gateway-virtual-interface
     * - local-gateway-virtual-interface-group
     * - natgateway
     * - network-acl
     * - network-interface
     * - placement-group
     * - prefix-list
     * - reserved-instances
     * - route-table
     * - security-group
     * - snapshot
     * - spot-instance-request
     * - subnet
     * - traffic-mirror-session
     * - traffic-mirror-target
     * - traffic-mirror-filter
     * - traffic-mirror-filter-rule
     * - transit-gateway-attachment
     * - transit-gateway-multicast-domain
     * - transit-gateway-route-table
     * - transit-gateway
     * - volume
     * - vpc
     * - vpc-endpoint
     * - vpc-endpoint-service
     * - vpc-flow-log
     * - vpc-peering-connection
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRegion(value, operator) {
        return this.if(`ec2:Region`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the percentage of increase of the rekey window (determined by the rekey margin time) within which the rekey time is randomly selected for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .createVpnConnection()
     * - .modifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRekeyFuzzPercentage(value, operator) {
        return this.if(`ec2:RekeyFuzzPercentage`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the margin time before the phase 2 lifetime expires for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .createVpnConnection()
     * - .modifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRekeyMarginTimeSeconds(value, operator) {
        return this.if(`ec2:RekeyMarginTimeSeconds`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the ARN of a requester VPC in a VPC peering connection
     *
     * https://docs.aws.amazon.com/vpc/latest/peering/security-iam.html
     *
     * Applies to actions:
     * - .acceptVpcPeeringConnection()
     * - .createVpcPeeringConnection()
     * - .deleteVpcPeeringConnection()
     * - .rejectVpcPeeringConnection()
     *
     * Applies to resource types:
     * - vpc-peering-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifRequesterVpc(value, operator) {
        return this.if(`ec2:RequesterVpc`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the payment option of the Reserved Instance offering (No Upfront, Partial Upfront, or All Upfront)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-reserved-instances.html#ri-payment-options
     *
     * Applies to actions:
     * - .createTags()
     *
     * Applies to resource types:
     * - reserved-instances
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReservedInstancesOfferingType(value, operator) {
        return this.if(`ec2:ReservedInstancesOfferingType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the preface string for a tag key and value pair that are attached to a resource
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/control-access-with-tags.html
     *
     * Applies to actions:
     * - .runInstances()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTagExists(value, operator) {
        return this.if(`ec2:ResourceTag/`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair of a resource
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/control-access-with-tags.html
     *
     * Applies to actions:
     * - .acceptTransitGatewayPeeringAttachment()
     * - .acceptTransitGatewayVpcAttachment()
     * - .acceptVpcEndpointConnections()
     * - .acceptVpcPeeringConnection()
     * - .applySecurityGroupsToClientVpnTargetNetwork()
     * - .associateClientVpnTargetNetwork()
     * - .associateIamInstanceProfile()
     * - .associateTransitGatewayMulticastDomain()
     * - .associateTransitGatewayRouteTable()
     * - .attachClassicLinkVpc()
     * - .attachVolume()
     * - .authorizeClientVpnIngress()
     * - .authorizeSecurityGroupEgress()
     * - .authorizeSecurityGroupIngress()
     * - .cancelCapacityReservation()
     * - .createClientVpnRoute()
     * - .createFlowLogs()
     * - .createLaunchTemplateVersion()
     * - .createLocalGatewayRoute()
     * - .createLocalGatewayRouteTableVpcAssociation()
     * - .createNetworkInterfacePermission()
     * - .createRoute()
     * - .createSnapshot()
     * - .createSnapshots()
     * - .createTags()
     * - .createTrafficMirrorFilterRule()
     * - .createTrafficMirrorSession()
     * - .createTrafficMirrorTarget()
     * - .createTransitGatewayMulticastDomain()
     * - .createTransitGatewayPeeringAttachment()
     * - .createTransitGatewayRoute()
     * - .createTransitGatewayRouteTable()
     * - .createTransitGatewayVpcAttachment()
     * - .createVpcEndpoint()
     * - .createVpcPeeringConnection()
     * - .deleteClientVpnEndpoint()
     * - .deleteClientVpnRoute()
     * - .deleteCustomerGateway()
     * - .deleteDhcpOptions()
     * - .deleteFlowLogs()
     * - .deleteInternetGateway()
     * - .deleteLaunchTemplate()
     * - .deleteLaunchTemplateVersions()
     * - .deleteLocalGatewayRoute()
     * - .deleteLocalGatewayRouteTableVpcAssociation()
     * - .deleteManagedPrefixList()
     * - .deleteNetworkAcl()
     * - .deleteNetworkAclEntry()
     * - .deleteRoute()
     * - .deleteRouteTable()
     * - .deleteSecurityGroup()
     * - .deleteSnapshot()
     * - .deleteTags()
     * - .deleteTrafficMirrorFilter()
     * - .deleteTrafficMirrorFilterRule()
     * - .deleteTrafficMirrorSession()
     * - .deleteTrafficMirrorTarget()
     * - .deleteTransitGateway()
     * - .deleteTransitGatewayMulticastDomain()
     * - .deleteTransitGatewayPeeringAttachment()
     * - .deleteTransitGatewayRoute()
     * - .deleteTransitGatewayRouteTable()
     * - .deleteTransitGatewayVpcAttachment()
     * - .deleteVolume()
     * - .deleteVpcEndpointServiceConfigurations()
     * - .deleteVpcEndpoints()
     * - .deleteVpcPeeringConnection()
     * - .deregisterTransitGatewayMulticastGroupMembers()
     * - .deregisterTransitGatewayMulticastGroupSources()
     * - .detachClassicLinkVpc()
     * - .detachVolume()
     * - .disableFastSnapshotRestores()
     * - .disableTransitGatewayRouteTablePropagation()
     * - .disableVpcClassicLink()
     * - .disassociateClientVpnTargetNetwork()
     * - .disassociateIamInstanceProfile()
     * - .disassociateTransitGatewayMulticastDomain()
     * - .disassociateTransitGatewayRouteTable()
     * - .enableFastSnapshotRestores()
     * - .enableTransitGatewayRouteTablePropagation()
     * - .enableVpcClassicLink()
     * - .getConsoleScreenshot()
     * - .importClientVpnClientCertificateRevocationList()
     * - .modifyCapacityReservation()
     * - .modifyClientVpnEndpoint()
     * - .modifyLaunchTemplate()
     * - .modifyManagedPrefixList()
     * - .modifySnapshotAttribute()
     * - .modifyTrafficMirrorFilterNetworkServices()
     * - .modifyTrafficMirrorFilterRule()
     * - .modifyTrafficMirrorSession()
     * - .modifyTransitGatewayVpcAttachment()
     * - .modifyVpcEndpoint()
     * - .modifyVpcEndpointServiceConfiguration()
     * - .modifyVpcEndpointServicePermissions()
     * - .modifyVpnConnection()
     * - .modifyVpnTunnelOptions()
     * - .rebootInstances()
     * - .registerTransitGatewayMulticastGroupMembers()
     * - .registerTransitGatewayMulticastGroupSources()
     * - .rejectTransitGatewayPeeringAttachment()
     * - .rejectTransitGatewayVpcAttachment()
     * - .rejectVpcEndpointConnections()
     * - .rejectVpcPeeringConnection()
     * - .replaceIamInstanceProfileAssociation()
     * - .replaceRoute()
     * - .replaceTransitGatewayRoute()
     * - .restoreManagedPrefixListVersion()
     * - .revokeClientVpnIngress()
     * - .revokeSecurityGroupEgress()
     * - .revokeSecurityGroupIngress()
     * - .runInstances()
     * - .sendDiagnosticInterrupt()
     * - .startInstances()
     * - .startVpcEndpointServicePrivateDnsVerification()
     * - .stopInstances()
     * - .terminateClientVpnConnections()
     * - .terminateInstances()
     * - .updateSecurityGroupRuleDescriptionsEgress()
     * - .updateSecurityGroupRuleDescriptionsIngress()
     *
     * Applies to resource types:
     * - capacity-reservation
     * - client-vpn-endpoint
     * - customer-gateway
     * - dedicated-host
     * - dhcp-options
     * - egress-only-internet-gateway
     * - export-instance-task
     * - fleet
     * - fpga-image
     * - image
     * - instance
     * - internet-gateway
     * - key-pair
     * - launch-template
     * - local-gateway
     * - local-gateway-route-table
     * - local-gateway-route-table-virtual-interface-group-association
     * - local-gateway-route-table-vpc-association
     * - local-gateway-virtual-interface
     * - local-gateway-virtual-interface-group
     * - natgateway
     * - network-acl
     * - network-interface
     * - placement-group
     * - prefix-list
     * - reserved-instances
     * - route-table
     * - security-group
     * - snapshot
     * - spot-instance-request
     * - subnet
     * - traffic-mirror-session
     * - traffic-mirror-target
     * - traffic-mirror-filter
     * - transit-gateway-attachment
     * - transit-gateway-multicast-domain
     * - transit-gateway-route-table
     * - transit-gateway
     * - volume
     * - vpc
     * - vpc-endpoint
     * - vpc-endpoint-service
     * - vpc-flow-log
     * - vpc-peering-connection
     * - vpn-connection
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`ec2:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the version of the instance metadata service for retrieving IAM role credentials for EC2
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRoleDelivery(value, operator) {
        return this.if(`ec2:RoleDelivery`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the root device type of the instance (ebs or instance-store)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .associateIamInstanceProfile()
     * - .attachClassicLinkVpc()
     * - .attachVolume()
     * - .createSnapshots()
     * - .createTags()
     * - .detachClassicLinkVpc()
     * - .detachVolume()
     * - .disassociateIamInstanceProfile()
     * - .getConsoleScreenshot()
     * - .rebootInstances()
     * - .replaceIamInstanceProfileAssociation()
     * - .runInstances()
     * - .sendDiagnosticInterrupt()
     * - .startInstances()
     * - .stopInstances()
     * - .terminateInstances()
     *
     * Applies to resource types:
     * - image
     * - instance
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRootDeviceType(value, operator) {
        return this.if(`ec2:RootDeviceType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the routing type for the VPN connection
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .createVpnConnection()
     * - .modifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRoutingType(value, operator) {
        return this.if(`ec2:RoutingType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the initiation time of a snapshot
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .createTags()
     * - .deleteSnapshot()
     * - .disableFastSnapshotRestores()
     * - .enableFastSnapshotRestores()
     * - .modifySnapshotAttribute()
     * - .runInstances()
     *
     * Applies to resource types:
     * - snapshot
     *
     * @param value The value(s) to check
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateEquals`
     */
    ifSnapshotTime(value, operator) {
        if (typeof value.getMonth === "function") {
            value = value.toISOString();
        }
        else if (Array.isArray(value)) {
            value = value.map((item) => {
                if (typeof item.getMonth === "function") {
                    item = item.toISOString();
                }
                return item;
            });
        }
        return this.if(`ec2:SnapshotTime`, value, operator || 'DateEquals');
    }
    /**
     * Filters access by the ARN of the instance from which the request originated
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifSourceInstanceARN(value, operator) {
        return this.if(`ec2:SourceInstanceARN`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the ARN of the subnet
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .createFlowLogs()
     * - .createNetworkInterfacePermission()
     * - .createTags()
     * - .runInstances()
     *
     * Applies to resource types:
     * - network-interface
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifSubnet(value, operator) {
        return this.if(`ec2:Subnet`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the tenancy of the VPC or instance (default, dedicated, or host)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .acceptVpcPeeringConnection()
     * - .associateIamInstanceProfile()
     * - .attachClassicLinkVpc()
     * - .attachVolume()
     * - .createFlowLogs()
     * - .createLocalGatewayRouteTableVpcAssociation()
     * - .createSnapshots()
     * - .createTags()
     * - .createVpcPeeringConnection()
     * - .detachClassicLinkVpc()
     * - .detachVolume()
     * - .disableVpcClassicLink()
     * - .disassociateIamInstanceProfile()
     * - .enableVpcClassicLink()
     * - .getConsoleScreenshot()
     * - .rebootInstances()
     * - .replaceIamInstanceProfileAssociation()
     * - .runInstances()
     * - .sendDiagnosticInterrupt()
     * - .startInstances()
     * - .stopInstances()
     * - .terminateInstances()
     *
     * Applies to resource types:
     * - instance
     * - reserved-instances
     * - vpc
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTenancy(value, operator) {
        return this.if(`ec2:Tenancy`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the the number of input/output operations per second (IOPS) provisioned for the volume
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .attachVolume()
     * - .createSnapshot()
     * - .createSnapshots()
     * - .createTags()
     * - .createVolume()
     * - .deleteVolume()
     * - .detachVolume()
     * - .runInstances()
     *
     * Applies to resource types:
     * - volume
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifVolumeIops(value, operator) {
        return this.if(`ec2:VolumeIops`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the size of the volume, in GiB.
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .attachVolume()
     * - .createSnapshot()
     * - .createSnapshots()
     * - .createTags()
     * - .createVolume()
     * - .deleteSnapshot()
     * - .deleteVolume()
     * - .detachVolume()
     * - .disableFastSnapshotRestores()
     * - .enableFastSnapshotRestores()
     * - .modifySnapshotAttribute()
     * - .runInstances()
     *
     * Applies to resource types:
     * - snapshot
     * - volume
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifVolumeSize(value, operator) {
        return this.if(`ec2:VolumeSize`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the type of volume (gp2, io1, st1, sc1, or standard)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .attachVolume()
     * - .createSnapshot()
     * - .createSnapshots()
     * - .createTags()
     * - .createVolume()
     * - .deleteVolume()
     * - .detachVolume()
     * - .runInstances()
     *
     * Applies to resource types:
     * - volume
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVolumeType(value, operator) {
        return this.if(`ec2:VolumeType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of the VPC
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .attachClassicLinkVpc()
     * - .authorizeSecurityGroupEgress()
     * - .authorizeSecurityGroupIngress()
     * - .createFlowLogs()
     * - .createNetworkInterfacePermission()
     * - .createRoute()
     * - .createTags()
     * - .deleteNetworkAcl()
     * - .deleteNetworkAclEntry()
     * - .deleteRoute()
     * - .deleteRouteTable()
     * - .deleteSecurityGroup()
     * - .replaceRoute()
     * - .revokeSecurityGroupEgress()
     * - .revokeSecurityGroupIngress()
     * - .runInstances()
     * - .updateSecurityGroupRuleDescriptionsEgress()
     * - .updateSecurityGroupRuleDescriptionsIngress()
     *
     * Applies to resource types:
     * - network-acl
     * - network-interface
     * - route-table
     * - security-group
     * - subnet
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifVpc(value, operator) {
        return this.if(`ec2:Vpc`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the name of the VPC endpoint service
     *
     * Applies to actions:
     * - .createVpcEndpoint()
     *
     * Applies to resource types:
     * - vpc-endpoint
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpceServiceName(value, operator) {
        return this.if(`ec2:VpceServiceName`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the service owner of the VPC endpoint service (amazon, aws-marketplace, or an AWS account ID)
     *
     * Applies to actions:
     * - .createVpcEndpoint()
     *
     * Applies to resource types:
     * - vpc-endpoint
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpceServiceOwner(value, operator) {
        return this.if(`ec2:VpceServiceOwner`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the private DNS name of the VPC endpoint service
     *
     * Applies to actions:
     * - .createVpcEndpointServiceConfiguration()
     * - .modifyVpcEndpointServiceConfiguration()
     *
     * Applies to resource types:
     * - vpc-endpoint-service
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpceServicePrivateDnsName(value, operator) {
        return this.if(`ec2:VpceServicePrivateDnsName`, value, operator || 'StringLike');
    }
}
exports.Ec2 = Ec2;
//# sourceMappingURL=data:application/json;base64,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