"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Sagemaker = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [sagemaker](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Sagemaker extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [sagemaker](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'sagemaker';
        this.actionList = {
            "AddTags": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_AddTags.html",
                "description": "Adds or overwrites one or more tags for the specified Amazon SageMaker resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "app": {
                        "required": false
                    },
                    "automl-job": {
                        "required": false
                    },
                    "domain": {
                        "required": false
                    },
                    "endpoint": {
                        "required": false
                    },
                    "endpoint-config": {
                        "required": false
                    },
                    "experiment": {
                        "required": false
                    },
                    "experiment-trial": {
                        "required": false
                    },
                    "experiment-trial-component": {
                        "required": false
                    },
                    "flow-definition": {
                        "required": false
                    },
                    "human-task-ui": {
                        "required": false
                    },
                    "hyper-parameter-tuning-job": {
                        "required": false
                    },
                    "labeling-job": {
                        "required": false
                    },
                    "model": {
                        "required": false
                    },
                    "monitoring-schedule": {
                        "required": false
                    },
                    "notebook-instance": {
                        "required": false
                    },
                    "processing-job": {
                        "required": false
                    },
                    "training-job": {
                        "required": false
                    },
                    "transform-job": {
                        "required": false
                    },
                    "user-profile": {
                        "required": false
                    },
                    "workteam": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "AssociateTrialComponent": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_AssociateTrialComponent.html",
                "description": "Associate a trial component with a trial.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment-trial": {
                        "required": true
                    },
                    "experiment-trial-component": {
                        "required": true
                    }
                }
            },
            "BatchGetMetrics": {
                "url": "",
                "description": "Retrieve metrics associated with SageMaker Resources such as Training Jobs. This API is not publicly exposed at this point, however admins can control this action",
                "accessLevel": "Read",
                "resourceTypes": {
                    "training-job": {
                        "required": true
                    }
                }
            },
            "BatchPutMetrics": {
                "url": "",
                "description": "Publish metrics associated with a SageMaker Resource such as a Training Job. This API is not publicly exposed at this point, however admins can control this action",
                "accessLevel": "Write",
                "resourceTypes": {
                    "training-job": {
                        "required": true
                    }
                }
            },
            "CreateAlgorithm": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateAlgorithm.html",
                "description": "Create an algorithm.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "algorithm": {
                        "required": true
                    }
                }
            },
            "CreateApp": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateApp.html",
                "description": "Grants permission to create an App for a SageMaker Studio UserProfile",
                "accessLevel": "Write",
                "resourceTypes": {
                    "app": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:InstanceTypes"
                ]
            },
            "CreateAutoMLJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateAutoMLJob.html",
                "description": "Creates automl job.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "automl-job": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:InterContainerTrafficEncryption",
                    "sagemaker:OutputKmsKey",
                    "sagemaker:VolumeKmsKey",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:VpcSubnets"
                ]
            },
            "CreateCodeRepository": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateCodeRepository.html",
                "description": "Create a code repository.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "code-repository": {
                        "required": true
                    }
                }
            },
            "CreateCompilationJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateCompilationJob.html",
                "description": "Create a compilation job.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "compilation-job": {
                        "required": true
                    }
                }
            },
            "CreateDomain": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateDomain.html",
                "description": "Grants permission to create a Domain for SageMaker Studio",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:CreateServiceLinkedRole",
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:AppNetworkAccess",
                    "sagemaker:InstanceTypes",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:VpcSubnets",
                    "sagemaker:DomainSharingOutputKmsKey",
                    "sagemaker:HomeEfsFileSystemKmsKey"
                ]
            },
            "CreateEndpoint": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateEndpoint.html",
                "description": "Creates an endpoint using the endpoint configuration specified in the request.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "endpoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateEndpointConfig": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateEndpointConfig.html",
                "description": "Creates an endpoint configuration that can be deployed using Amazon SageMaker hosting services.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "endpoint-config": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:AcceleratorTypes",
                    "sagemaker:InstanceTypes",
                    "sagemaker:ModelArn",
                    "sagemaker:VolumeKmsKey"
                ]
            },
            "CreateExperiment": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateExperiment.html",
                "description": "Create an experiment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateFlowDefinition": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateFlowDefinition.html",
                "description": "Creates a flow definition, which defines settings for a human workflow.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "flow-definition": {
                        "required": true
                    }
                },
                "conditions": [
                    "sagemaker:WorkteamArn",
                    "sagemaker:WorkteamType",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateHumanTaskUi": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateHumanTaskUi.html",
                "description": "Defines the settings you will use for the human review workflow user interface.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "human-task-ui": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateHyperParameterTuningJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateHyperParameterTuningJob.html",
                "description": "Creates hyper parameter tuning job that can be deployed using Amazon SageMaker.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "hyper-parameter-tuning-job": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:FileSystemAccessMode",
                    "sagemaker:FileSystemDirectoryPath",
                    "sagemaker:FileSystemId",
                    "sagemaker:FileSystemType",
                    "sagemaker:InstanceTypes",
                    "sagemaker:InterContainerTrafficEncryption",
                    "sagemaker:MaxRuntimeInSeconds",
                    "sagemaker:NetworkIsolation",
                    "sagemaker:OutputKmsKey",
                    "sagemaker:VolumeKmsKey",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:VpcSubnets"
                ]
            },
            "CreateLabelingJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateLabelingJob.html",
                "description": "Starts a labeling job. A labeling job takes unlabeled data in and produces labeled data as output, which can be used for training SageMaker models.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "labeling-job": {
                        "required": true
                    }
                },
                "conditions": [
                    "sagemaker:WorkteamArn",
                    "sagemaker:WorkteamType",
                    "sagemaker:VolumeKmsKey",
                    "sagemaker:OutputKmsKey",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateModel": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateModel.html",
                "description": "Creates a model in Amazon SageMaker. In the request, you specify a name for the model and describe one or more containers.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:NetworkIsolation",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:VpcSubnets"
                ]
            },
            "CreateModelPackage": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateModelPackage.html",
                "description": "Create a model package.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "model-package": {
                        "required": true
                    }
                }
            },
            "CreateMonitoringSchedule": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateMonitoringSchedule.html",
                "description": "Creates a monitoring schedule.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "monitoring-schedule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:InstanceTypes",
                    "sagemaker:MaxRuntimeInSeconds",
                    "sagemaker:NetworkIsolation",
                    "sagemaker:OutputKmsKey",
                    "sagemaker:VolumeKmsKey",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:VpcSubnets"
                ]
            },
            "CreateNotebookInstance": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateNotebookInstance.html",
                "description": "Creates an Amazon SageMaker notebook instance. A notebook instance is an Amazon EC2 instance running on a Jupyter Notebook.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "notebook-instance": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:AcceleratorTypes",
                    "sagemaker:DirectInternetAccess",
                    "sagemaker:InstanceTypes",
                    "sagemaker:RootAccess",
                    "sagemaker:VolumeKmsKey",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:VpcSubnets"
                ]
            },
            "CreateNotebookInstanceLifecycleConfig": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateNotebookInstanceLifecycleConfig.html",
                "description": "Creates an notebook instance lifecycle configuration that can be deployed using Amazon SageMaker.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "notebook-instance-lifecycle-config": {
                        "required": true
                    }
                }
            },
            "CreatePresignedDomainUrl": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreatePresignedDomainUrl.html",
                "description": "Grants permission to return a URL that you can use from your browser to connect to the Domain as a specified UserProfile when AuthMode is 'IAM'",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user-profile": {
                        "required": true
                    }
                }
            },
            "CreatePresignedNotebookInstanceUrl": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreatePresignedNotebookInstanceUrl.html",
                "description": "Returns a URL that you can use from your browser to connect to the Notebook Instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "notebook-instance": {
                        "required": true
                    }
                }
            },
            "CreateProcessingJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateProcessingJob.html",
                "description": "Starts a processing job. After processing completes, Amazon SageMaker saves the resulting artifacts and other optional output to an Amazon S3 location that you specify.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "processing-job": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:InstanceTypes",
                    "sagemaker:MaxRuntimeInSeconds",
                    "sagemaker:NetworkIsolation",
                    "sagemaker:OutputKmsKey",
                    "sagemaker:VolumeKmsKey",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:VpcSubnets",
                    "sagemaker:InterContainerTrafficEncryption"
                ]
            },
            "CreateTrainingJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTrainingJob.html",
                "description": "Starts a model training job. After training completes, Amazon SageMaker saves the resulting model artifacts and other optional output to an Amazon S3 location that you specify.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "training-job": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:FileSystemAccessMode",
                    "sagemaker:FileSystemDirectoryPath",
                    "sagemaker:FileSystemId",
                    "sagemaker:FileSystemType",
                    "sagemaker:InstanceTypes",
                    "sagemaker:InterContainerTrafficEncryption",
                    "sagemaker:MaxRuntimeInSeconds",
                    "sagemaker:NetworkIsolation",
                    "sagemaker:OutputKmsKey",
                    "sagemaker:VolumeKmsKey",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:VpcSubnets"
                ]
            },
            "CreateTransformJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTransformJob.html",
                "description": "Starts a transform job. After the results are obtained, Amazon SageMaker saves them to an Amazon S3 location that you specify.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transform-job": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:InstanceTypes",
                    "sagemaker:ModelArn",
                    "sagemaker:OutputKmsKey",
                    "sagemaker:VolumeKmsKey"
                ]
            },
            "CreateTrial": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTrial.html",
                "description": "Create a trial.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment-trial": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateTrialComponent": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTrialComponent.html",
                "description": "Create a trial component.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment-trial-component": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateUserProfile": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateUserProfile.html",
                "description": "Grants permission to create a UserProfile for a SageMaker Studio Domain",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "user-profile": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:InstanceTypes",
                    "sagemaker:DomainSharingOutputKmsKey"
                ]
            },
            "CreateWorkforce": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateWorkforce.html",
                "description": "Create a workforce.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workforce": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateWorkteam": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateWorkteam.html",
                "description": "Create a workteam.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workteam": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteAlgorithm": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteAlgorithm.html",
                "description": "Deletes an algorithm.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "algorithm": {
                        "required": true
                    }
                }
            },
            "DeleteApp": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteApp.html",
                "description": "Grants permission to delete an App",
                "accessLevel": "Write",
                "resourceTypes": {
                    "app": {
                        "required": true
                    }
                }
            },
            "DeleteCodeRepository": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteCodeRepository.html",
                "description": "Deletes a code repository.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "code-repository": {
                        "required": true
                    }
                }
            },
            "DeleteDomain": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteDomain.html",
                "description": "Grants permission to delete a Domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteEndpoint": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteEndpoint.html",
                "description": "Deletes an endpoint. Amazon SageMaker frees up all the resources that were deployed when the endpoint was created.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "endpoint": {
                        "required": true
                    }
                }
            },
            "DeleteEndpointConfig": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteEndpointConfig.html",
                "description": "Deletes the endpoint configuration created using the CreateEndpointConfig API. The DeleteEndpointConfig API deletes only the specified configuration. It does not delete any endpoints created using the configuration.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "endpoint-config": {
                        "required": true
                    }
                }
            },
            "DeleteExperiment": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteExperiment.html",
                "description": "Deletes an experiment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment": {
                        "required": true
                    }
                }
            },
            "DeleteFlowDefinition": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteFlowDefinition.html",
                "description": "Deltes the specified flow definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "flow-definition": {
                        "required": true
                    }
                }
            },
            "DeleteHumanLoop": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteHumanLoop.html",
                "description": "Deletes the specified human loop.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "human-loop": {
                        "required": true
                    }
                }
            },
            "DeleteModel": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteModel.html",
                "description": "Deletes a model created using the CreateModel API. The DeleteModel API deletes only the model entry in Amazon SageMaker that you created by calling the CreateModel API. It does not delete model artifacts, inference code, or the IAM role that you specified when creating the model.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                }
            },
            "DeleteModelPackage": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteModelPackage.html",
                "description": "Deletes a model package.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "model-package": {
                        "required": true
                    }
                }
            },
            "DeleteMonitoringSchedule": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteMonitoringSchedule.html",
                "description": "Deletes a monitoring schedule. Amazon SageMaker will no longer run the scheduled monitoring.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "monitoring-schedule": {
                        "required": true
                    }
                }
            },
            "DeleteNotebookInstance": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteNotebookInstance.html",
                "description": "Deletes an Amazon SageMaker notebook instance. Before you can delete a notebook instance, you must call the StopNotebookInstance API.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "notebook-instance": {
                        "required": true
                    }
                }
            },
            "DeleteNotebookInstanceLifecycleConfig": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteNotebookInstanceLifecycleConfig.html",
                "description": "Deletes an notebook instance lifecycle configuration that can be deployed using Amazon SageMaker.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "notebook-instance-lifecycle-config": {
                        "required": true
                    }
                }
            },
            "DeleteTags": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteTags.html",
                "description": "Deletes the specified set of tags from an Amazon SageMaker resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "app": {
                        "required": false
                    },
                    "automl-job": {
                        "required": false
                    },
                    "compilation-job": {
                        "required": false
                    },
                    "domain": {
                        "required": false
                    },
                    "endpoint": {
                        "required": false
                    },
                    "endpoint-config": {
                        "required": false
                    },
                    "experiment": {
                        "required": false
                    },
                    "experiment-trial": {
                        "required": false
                    },
                    "experiment-trial-component": {
                        "required": false
                    },
                    "flow-definition": {
                        "required": false
                    },
                    "human-task-ui": {
                        "required": false
                    },
                    "hyper-parameter-tuning-job": {
                        "required": false
                    },
                    "labeling-job": {
                        "required": false
                    },
                    "model": {
                        "required": false
                    },
                    "monitoring-schedule": {
                        "required": false
                    },
                    "notebook-instance": {
                        "required": false
                    },
                    "processing-job": {
                        "required": false
                    },
                    "training-job": {
                        "required": false
                    },
                    "transform-job": {
                        "required": false
                    },
                    "user-profile": {
                        "required": false
                    },
                    "workteam": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "DeleteTrial": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteTrial.html",
                "description": "Deletes a trial.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment-trial": {
                        "required": true
                    }
                }
            },
            "DeleteTrialComponent": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteTrialComponent.html",
                "description": "Deletes a trial component.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment-trial-component": {
                        "required": true
                    }
                }
            },
            "DeleteUserProfile": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteUserProfile.html",
                "description": "Grants permission to delete a UserProfile",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user-profile": {
                        "required": true
                    }
                }
            },
            "DeleteWorkforce": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteWorkforce.html",
                "description": "Deletes a workforce.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workforce": {
                        "required": true
                    }
                }
            },
            "DeleteWorkteam": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteWorkteam.html",
                "description": "Deletes a workteam.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workteam": {
                        "required": true
                    }
                }
            },
            "DescribeAlgorithm": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeAlgorithm.html",
                "description": "Returns information about an algorithm.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "algorithm": {
                        "required": true
                    }
                }
            },
            "DescribeApp": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeApp.html",
                "description": "Grants permission to describe an App",
                "accessLevel": "Read",
                "resourceTypes": {
                    "app": {
                        "required": true
                    }
                }
            },
            "DescribeAutoMLJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeAutoMLJob.html",
                "description": "Describes an automl job that was created via CreateAutoMLJob API.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "automl-job": {
                        "required": true
                    }
                }
            },
            "DescribeCodeRepository": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeCodeRepository.html",
                "description": "Returns information about a code repository.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "code-repository": {
                        "required": true
                    }
                }
            },
            "DescribeCompilationJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeCompilationJob.html",
                "description": "Returns information about a compilation job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "compilation-job": {
                        "required": true
                    }
                }
            },
            "DescribeDomain": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeDomain.html",
                "description": "Grants permission to describe a Domain",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeEndpoint": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeEndpoint.html",
                "description": "Returns the description of an endpoint.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "endpoint": {
                        "required": true
                    }
                }
            },
            "DescribeEndpointConfig": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeEndpointConfig.html",
                "description": "Returns the description of an endpoint configuration, which was created using the CreateEndpointConfig API.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "endpoint-config": {
                        "required": true
                    }
                }
            },
            "DescribeExperiment": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeExperiment.html",
                "description": "Returns information about an experiment.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "experiment": {
                        "required": true
                    }
                }
            },
            "DescribeFlowDefinition": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeFlowDefinition.html",
                "description": "Returns detailed information about the specified flow definition.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "flow-definition": {
                        "required": true
                    }
                }
            },
            "DescribeHumanLoop": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeHumanLoop.html",
                "description": "Returns detailed information about the specified human loop.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "human-loop": {
                        "required": true
                    }
                }
            },
            "DescribeHumanTaskUi": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeHumanTaskUi.html",
                "description": "Returns detailed information about the specified human review workflow user interface.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "human-task-ui": {
                        "required": true
                    }
                }
            },
            "DescribeHyperParameterTuningJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeHyperParameterTuningJob.html",
                "description": "Describes a hyper parameter tuning job that was created via CreateHyperParameterTuningJob API.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hyper-parameter-tuning-job": {
                        "required": true
                    }
                }
            },
            "DescribeLabelingJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeLabelingJob.html",
                "description": "Returns information about a labeling job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "labeling-job": {
                        "required": true
                    }
                }
            },
            "DescribeModel": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeModel.html",
                "description": "Describes a model that you created using the CreateModel API.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                }
            },
            "DescribeModelPackage": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeModelPackage.html",
                "description": "Returns information about a model package.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "model-package": {
                        "required": true
                    }
                }
            },
            "DescribeMonitoringSchedule": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeMonitoringSchedule.html",
                "description": "Returns information about a monitoring schedule.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "monitoring-schedule": {
                        "required": true
                    }
                }
            },
            "DescribeNotebookInstance": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeNotebookInstance.html",
                "description": "Returns information about a notebook instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "notebook-instance": {
                        "required": true
                    }
                }
            },
            "DescribeNotebookInstanceLifecycleConfig": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeNotebookInstanceLifecycleConfig.html",
                "description": "Describes an notebook instance lifecycle configuration that was created via CreateNotebookInstanceLifecycleConfig API.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "notebook-instance-lifecycle-config": {
                        "required": true
                    }
                }
            },
            "DescribeProcessingJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeProcessingJob.html",
                "description": "Returns information about a processing job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "processing-job": {
                        "required": true
                    }
                }
            },
            "DescribeSubscribedWorkteam": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeSubscribedWorkteam.html",
                "description": "Returns information about a subscribed workteam.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "workteam": {
                        "required": true
                    }
                }
            },
            "DescribeTrainingJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeTrainingJob.html",
                "description": "Returns information about a training job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "training-job": {
                        "required": true
                    }
                }
            },
            "DescribeTransformJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeTransformJob.html",
                "description": "Returns information about a transform job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "transform-job": {
                        "required": true
                    }
                }
            },
            "DescribeTrial": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeTrial.html",
                "description": "Returns information about a trial.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "experiment-trial": {
                        "required": true
                    }
                }
            },
            "DescribeTrialComponent": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeTrialComponent.html",
                "description": "Returns information about a trial component.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "experiment-trial-component": {
                        "required": true
                    }
                }
            },
            "DescribeUserProfile": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeUserProfile.html",
                "description": "Grants permission to describe a UserProfile",
                "accessLevel": "Read",
                "resourceTypes": {
                    "user-profile": {
                        "required": true
                    }
                }
            },
            "DescribeWorkforce": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeWorkforce.html",
                "description": "Returns information about a workforce.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "workforce": {
                        "required": true
                    }
                }
            },
            "DescribeWorkteam": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeWorkteam.html",
                "description": "Returns information about a workteam.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "workteam": {
                        "required": true
                    }
                }
            },
            "DisassociateTrialComponent": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DisassociateTrialComponent.html",
                "description": "Disassociate a trial component with a trial.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment-trial": {
                        "required": true
                    },
                    "experiment-trial-component": {
                        "required": true
                    },
                    "processing-job": {
                        "required": true
                    }
                }
            },
            "GetSearchSuggestions": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_GetSearchSuggestions.html",
                "description": "Get search suggestions when provided with keyword.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "training-job": {
                        "required": true
                    }
                }
            },
            "InvokeEndpoint": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_InvokeEndpoint.html",
                "description": "After you deploy a model into production using Amazon SageMaker hosting services, your client applications use this API to get inferences from the model hosted at the specified endpoint.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "endpoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "sagemaker:TargetModel"
                ]
            },
            "ListAlgorithms": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListAlgorithms.html",
                "description": "Lists algorithms.",
                "accessLevel": "List"
            },
            "ListApps": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListApps.html",
                "description": "Grants permission to list the Apps in your account",
                "accessLevel": "List"
            },
            "ListAutoMLJobs": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListAutoMLJobs.html",
                "description": "Lists automl jobs created via the CreateAutoMLJob.",
                "accessLevel": "List"
            },
            "ListCandidatesForAutoMLJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListCandidatesForAutoMLJob.html",
                "description": "Lists candidates for automl job created via the CreateAutoMLJob.",
                "accessLevel": "List"
            },
            "ListCodeRepositories": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListCodeRepositories.html",
                "description": "Lists code repositories.",
                "accessLevel": "List"
            },
            "ListCompilationJobs": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListCompilationJobs.html",
                "description": "Lists compilation jobs.",
                "accessLevel": "List"
            },
            "ListDomains": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListDomains.html",
                "description": "Grants permission to list the Domains in your account",
                "accessLevel": "List"
            },
            "ListEndpointConfigs": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListEndpointConfigs.html",
                "description": "Lists endpoint configurations.",
                "accessLevel": "List"
            },
            "ListEndpoints": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListEndpoints.html",
                "description": "Lists endpoints.",
                "accessLevel": "List"
            },
            "ListExperiments": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListExperiments.html",
                "description": "Lists experiments.",
                "accessLevel": "List"
            },
            "ListFlowDefinitions": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListFlowDefinitions.html",
                "description": "Returns summary information about flow definitions, given the specified parameters.",
                "accessLevel": "List"
            },
            "ListHumanLoops": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListHumanLoops.html",
                "description": "Returns summary information about human loops, given the specified parameters.",
                "accessLevel": "List"
            },
            "ListHumanTaskUis": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListHumanTaskUis.html",
                "description": "Returns summary information about human review workflow user interfaces, given the specified parameters.",
                "accessLevel": "List"
            },
            "ListHyperParameterTuningJobs": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListHyperParameterTuningJobs.html",
                "description": "Lists hyper parameter tuning jobs that was created using Amazon SageMaker.",
                "accessLevel": "List"
            },
            "ListLabelingJobs": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListLabelingJobs.html",
                "description": "Lists labeling jobs.",
                "accessLevel": "List"
            },
            "ListLabelingJobsForWorkteam": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListLabelingJobs.html",
                "description": "Lists labeling jobs for workteam.",
                "accessLevel": "List",
                "resourceTypes": {
                    "workteam": {
                        "required": true
                    }
                }
            },
            "ListModelPackages": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListModelPackages.html",
                "description": "Lists model packages.",
                "accessLevel": "List"
            },
            "ListModels": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListModels.html",
                "description": "Lists the models created with the CreateModel API.",
                "accessLevel": "List"
            },
            "ListMonitoringExecutions": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListMonitoringExecutions.html",
                "description": "Lists monitoring executions.",
                "accessLevel": "List"
            },
            "ListMonitoringSchedules": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListMonitoringSchedules.html",
                "description": "Lists monitoring schedules.",
                "accessLevel": "List"
            },
            "ListNotebookInstanceLifecycleConfigs": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListNotebookInstanceLifecycleConfigs.html",
                "description": "Lists notebook instance lifecycle configurations that can be deployed using Amazon SageMaker.",
                "accessLevel": "List"
            },
            "ListNotebookInstances": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListNotebookInstances.html",
                "description": "Returns a list of the Amazon SageMaker notebook instances in the requester's account in an AWS Region.",
                "accessLevel": "List"
            },
            "ListProcessingJobs": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListProcessingJobs.html",
                "description": "Lists processing jobs.",
                "accessLevel": "List"
            },
            "ListSubscribedWorkteams": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListSubscribedWorkteams.html",
                "description": "Lists subscribed workteams.",
                "accessLevel": "List"
            },
            "ListTags": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListTags.html",
                "description": "Returns the tag set associated with the specified resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "app": {
                        "required": false
                    },
                    "automl-job": {
                        "required": false
                    },
                    "domain": {
                        "required": false
                    },
                    "endpoint": {
                        "required": false
                    },
                    "endpoint-config": {
                        "required": false
                    },
                    "experiment": {
                        "required": false
                    },
                    "experiment-trial": {
                        "required": false
                    },
                    "experiment-trial-component": {
                        "required": false
                    },
                    "flow-definition": {
                        "required": false
                    },
                    "human-task-ui": {
                        "required": false
                    },
                    "hyper-parameter-tuning-job": {
                        "required": false
                    },
                    "labeling-job": {
                        "required": false
                    },
                    "model": {
                        "required": false
                    },
                    "monitoring-schedule": {
                        "required": false
                    },
                    "notebook-instance": {
                        "required": false
                    },
                    "training-job": {
                        "required": false
                    },
                    "transform-job": {
                        "required": false
                    },
                    "user-profile": {
                        "required": false
                    },
                    "workteam": {
                        "required": false
                    }
                }
            },
            "ListTrainingJobs": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListTrainingJobs.html",
                "description": "Lists training jobs.",
                "accessLevel": "List"
            },
            "ListTrainingJobsForHyperParameterTuningJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ ListTrainingJobsForHyperParameterTuningJob.html",
                "description": "Lists training jobs for a hyper parameter tuning job that was created using Amazon SageMaker.",
                "accessLevel": "List",
                "resourceTypes": {
                    "hyper-parameter-tuning-job": {
                        "required": true
                    }
                }
            },
            "ListTransformJobs": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListTransformJobs.html",
                "description": "Lists transform jobs.",
                "accessLevel": "List"
            },
            "ListTrialComponents": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListTrialComponents.html",
                "description": "Lists trial components.",
                "accessLevel": "List"
            },
            "ListTrials": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListTrials.html",
                "description": "Lists trials.",
                "accessLevel": "List"
            },
            "ListUserProfiles": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListUserProfiles.html",
                "description": "Grants permission to list the UserProfiles in your account",
                "accessLevel": "List"
            },
            "ListWorkforces": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListWorkforces.html",
                "description": "Lists workforces.",
                "accessLevel": "List"
            },
            "ListWorkteams": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListWorkteams.html",
                "description": "Lists workteams.",
                "accessLevel": "List"
            },
            "RenderUiTemplate": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_RenderUiTemplate.html",
                "description": "Render a UI template used for a human annotation task.",
                "accessLevel": "Read",
                "dependentActions": [
                    "iam:PassRole"
                ]
            },
            "Search": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_Search.html",
                "description": "Search for a training job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "training-job": {
                        "required": true
                    }
                }
            },
            "StartHumanLoop": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StartHumanLoop.html",
                "description": "Starts a human loop.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "flow-definition": {
                        "required": true
                    }
                }
            },
            "StartMonitoringSchedule": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StartMonitoringSchedule.html",
                "description": "Starts a monitoring schedule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "monitoring-schedule": {
                        "required": true
                    }
                }
            },
            "StartNotebookInstance": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StartNotebookInstance.html",
                "description": "Launches an EC2 instance with the latest version of the libraries and attaches your EBS volume.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "notebook-instance": {
                        "required": true
                    }
                }
            },
            "StopAutoMLJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopAutoMLJob.html",
                "description": "Stops a running automl job created via the CreateAutoMLJob.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "automl-job": {
                        "required": true
                    }
                }
            },
            "StopCompilationJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopCompilationJob.html",
                "description": "Stops a compilation job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "compilation-job": {
                        "required": true
                    }
                }
            },
            "StopHumanLoop": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopHumanLoop.html",
                "description": "Stops the specified human loop.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "human-loop": {
                        "required": true
                    }
                }
            },
            "StopHyperParameterTuningJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopHyperParameterTuningJob.html",
                "description": "Stops a running hyper parameter tuning job create via the CreateHyperParameterTuningJob.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hyper-parameter-tuning-job": {
                        "required": true
                    }
                }
            },
            "StopLabelingJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopLabelingJob.html",
                "description": "Stops a labeling job. Any labels already generated will be exported before stopping.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "labeling-job": {
                        "required": true
                    }
                }
            },
            "StopMonitoringSchedule": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopMonitoringSchedule.html",
                "description": "Stops a monitoring schedule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "monitoring-schedule": {
                        "required": true
                    }
                }
            },
            "StopNotebookInstance": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopNotebookInstance.html",
                "description": "Terminates the EC2 instance. Before terminating the instance, Amazon SageMaker disconnects the EBS volume from it. Amazon SageMaker preserves the EBS volume.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "notebook-instance": {
                        "required": true
                    }
                }
            },
            "StopProcessingJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopProcessingJob.html",
                "description": "Stops a processing job. To stop a job, Amazon SageMaker sends the algorithm the SIGTERM signal, which delays job termination for 120 seconds.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "processing-job": {
                        "required": true
                    }
                }
            },
            "StopTrainingJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopTrainingJob.html",
                "description": "Stops a training job. To stop a job, Amazon SageMaker sends the algorithm the SIGTERM signal, which delays job termination for 120 seconds.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "training-job": {
                        "required": true
                    }
                }
            },
            "StopTransformJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopTransformJob.html",
                "description": "Stops a transform job. When Amazon SageMaker receives a StopTransformJob request, the status of the job changes to Stopping. After Amazon SageMaker stops the job, the status is set to Stopped",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transform-job": {
                        "required": true
                    }
                }
            },
            "UpdateCodeRepository": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateCodeRepository.html",
                "description": "Updates a code repository.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "code-repository": {
                        "required": true
                    }
                }
            },
            "UpdateDomain": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateDomain.html",
                "description": "Grants permission to update a Domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:InstanceTypes",
                    "sagemaker:DomainSharingOutputKmsKey"
                ]
            },
            "UpdateEndpoint": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateEndpoint.html",
                "description": "Updates an endpoint to use the endpoint configuration specified in the request.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "endpoint": {
                        "required": true
                    }
                }
            },
            "UpdateEndpointWeightsAndCapacities": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateEndpointWeightsAndCapacities.html",
                "description": "Updates variant weight, capacity, or both of one or more variants associated with an endpoint.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "endpoint": {
                        "required": true
                    }
                }
            },
            "UpdateExperiment": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateExperiment.html",
                "description": "Updates an experiment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment": {
                        "required": true
                    }
                }
            },
            "UpdateMonitoringSchedule": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateMonitoringSchedule.html",
                "description": "Updates a monitoring schedule.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "monitoring-schedule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:InstanceTypes",
                    "sagemaker:MaxRuntimeInSeconds",
                    "sagemaker:NetworkIsolation",
                    "sagemaker:OutputKmsKey",
                    "sagemaker:VolumeKmsKey",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:VpcSubnets"
                ]
            },
            "UpdateNotebookInstance": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateNotebookInstance.html",
                "description": "Updates a notebook instance. Notebook instance updates include upgrading or downgrading the EC2 instance used for your notebook instance to accommodate changes in your workload requirements. You can also update the VPC security groups.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "notebook-instance": {
                        "required": true
                    }
                },
                "conditions": [
                    "sagemaker:AcceleratorTypes",
                    "sagemaker:InstanceTypes",
                    "sagemaker:RootAccess"
                ]
            },
            "UpdateNotebookInstanceLifecycleConfig": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ UpdateNotebookInstanceLifecycleConfig.html",
                "description": "Updates a notebook instance lifecycle configuration created with the CreateNotebookInstanceLifecycleConfig API.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "notebook-instance-lifecycle-config": {
                        "required": true
                    }
                }
            },
            "UpdateTrial": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateTrial.html",
                "description": "Updates a trial.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment-trial": {
                        "required": true
                    }
                }
            },
            "UpdateTrialComponent": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateTrialComponent.html",
                "description": "Updates a trial component.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment-trial-component": {
                        "required": true
                    }
                }
            },
            "UpdateUserProfile": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateUserProfile.html",
                "description": "Grants permission to update a UserProfile",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user-profile": {
                        "required": true
                    }
                },
                "conditions": [
                    "sagemaker:InstanceTypes",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:InstanceTypes",
                    "sagemaker:DomainSharingOutputKmsKey"
                ]
            },
            "UpdateWorkforce": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateWorkforce.html",
                "description": "Updates a workforce.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workforce": {
                        "required": true
                    }
                }
            },
            "UpdateWorkteam": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateWorkteam.html",
                "description": "Updates a workteam.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workteam": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "human-loop": {
                "name": "human-loop",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:human-loop/${HumanLoopName}",
                "conditionKeys": []
            },
            "flow-definition": {
                "name": "flow-definition",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:flow-definition/${FlowDefinitionName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "human-task-ui": {
                "name": "human-task-ui",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:human-task-ui/${HumanTaskUiName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "labeling-job": {
                "name": "labeling-job",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:labeling-job/${LabelingJobName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "workteam": {
                "name": "workteam",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:workteam/${WorkteamName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "workforce": {
                "name": "workforce",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:workforce/${WorkforceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "domain": {
                "name": "domain",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:domain/${DomainId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "user-profile": {
                "name": "user-profile",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:user-profile/${DomainId}/${UserProfileName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "app": {
                "name": "app",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:app/${DomainId}/${UserProfileName}/${AppType}/${AppName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "notebook-instance": {
                "name": "notebook-instance",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:notebook-instance/${NotebookInstanceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "notebook-instance-lifecycle-config": {
                "name": "notebook-instance-lifecycle-config",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:notebook-instance-lifecycle-config/${NotebookInstanceLifecycleConfigName}",
                "conditionKeys": []
            },
            "code-repository": {
                "name": "code-repository",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:code-repository/${CodeRepositoryName}",
                "conditionKeys": []
            },
            "algorithm": {
                "name": "algorithm",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:algorithm/${AlgorithmName}",
                "conditionKeys": []
            },
            "training-job": {
                "name": "training-job",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:training-job/${TrainingJobName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "processing-job": {
                "name": "processing-job",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:processing-job/${ProcessingJobName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "hyper-parameter-tuning-job": {
                "name": "hyper-parameter-tuning-job",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:hyper-parameter-tuning-job/${HyperParameterTuningJobName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "model-package": {
                "name": "model-package",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:model-package/${ModelPackageName}",
                "conditionKeys": []
            },
            "model": {
                "name": "model",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:model/${ModelName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "endpoint-config": {
                "name": "endpoint-config",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:endpoint-config/${EndpointConfigName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "endpoint": {
                "name": "endpoint",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:endpoint/${EndpointName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "transform-job": {
                "name": "transform-job",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:transform-job/${TransformJobName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "compilation-job": {
                "name": "compilation-job",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:compilation-job/${CompilationJobName}",
                "conditionKeys": []
            },
            "automl-job": {
                "name": "automl-job",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:automl-job/${AutoMLJobJobName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "monitoring-schedule": {
                "name": "monitoring-schedule",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:monitoring-schedule/${MonitoringScheduleName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "experiment": {
                "name": "experiment",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:experiment/${ExperimentName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "experiment-trial": {
                "name": "experiment-trial",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:experiment-trial/${TrialName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "experiment-trial-component": {
                "name": "experiment-trial-component",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:experiment-trial-component/${TrialComponentName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Adds or overwrites one or more tags for the specified Amazon SageMaker resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_AddTags.html
     */
    addTags() {
        this.add('sagemaker:AddTags');
        return this;
    }
    /**
     * Associate a trial component with a trial.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_AssociateTrialComponent.html
     */
    associateTrialComponent() {
        this.add('sagemaker:AssociateTrialComponent');
        return this;
    }
    /**
     * Retrieve metrics associated with SageMaker Resources such as Training Jobs. This API is not publicly exposed at this point, however admins can control this action
     *
     * Access Level: Read
     */
    batchGetMetrics() {
        this.add('sagemaker:BatchGetMetrics');
        return this;
    }
    /**
     * Publish metrics associated with a SageMaker Resource such as a Training Job. This API is not publicly exposed at this point, however admins can control this action
     *
     * Access Level: Write
     */
    batchPutMetrics() {
        this.add('sagemaker:BatchPutMetrics');
        return this;
    }
    /**
     * Create an algorithm.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateAlgorithm.html
     */
    createAlgorithm() {
        this.add('sagemaker:CreateAlgorithm');
        return this;
    }
    /**
     * Grants permission to create an App for a SageMaker Studio UserProfile
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceTypes()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateApp.html
     */
    createApp() {
        this.add('sagemaker:CreateApp');
        return this;
    }
    /**
     * Creates automl job.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInterContainerTrafficEncryption()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateAutoMLJob.html
     */
    createAutoMLJob() {
        this.add('sagemaker:CreateAutoMLJob');
        return this;
    }
    /**
     * Create a code repository.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateCodeRepository.html
     */
    createCodeRepository() {
        this.add('sagemaker:CreateCodeRepository');
        return this;
    }
    /**
     * Create a compilation job.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateCompilationJob.html
     */
    createCompilationJob() {
        this.add('sagemaker:CreateCompilationJob');
        return this;
    }
    /**
     * Grants permission to create a Domain for SageMaker Studio
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAppNetworkAccess()
     * - .ifInstanceTypes()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     * - .ifDomainSharingOutputKmsKey()
     * - .ifHomeEfsFileSystemKmsKey()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateDomain.html
     */
    createDomain() {
        this.add('sagemaker:CreateDomain');
        return this;
    }
    /**
     * Creates an endpoint using the endpoint configuration specified in the request.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateEndpoint.html
     */
    createEndpoint() {
        this.add('sagemaker:CreateEndpoint');
        return this;
    }
    /**
     * Creates an endpoint configuration that can be deployed using Amazon SageMaker hosting services.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAcceleratorTypes()
     * - .ifInstanceTypes()
     * - .ifModelArn()
     * - .ifVolumeKmsKey()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateEndpointConfig.html
     */
    createEndpointConfig() {
        this.add('sagemaker:CreateEndpointConfig');
        return this;
    }
    /**
     * Create an experiment.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateExperiment.html
     */
    createExperiment() {
        this.add('sagemaker:CreateExperiment');
        return this;
    }
    /**
     * Creates a flow definition, which defines settings for a human workflow.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifWorkteamArn()
     * - .ifWorkteamType()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateFlowDefinition.html
     */
    createFlowDefinition() {
        this.add('sagemaker:CreateFlowDefinition');
        return this;
    }
    /**
     * Defines the settings you will use for the human review workflow user interface.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateHumanTaskUi.html
     */
    createHumanTaskUi() {
        this.add('sagemaker:CreateHumanTaskUi');
        return this;
    }
    /**
     * Creates hyper parameter tuning job that can be deployed using Amazon SageMaker.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifFileSystemAccessMode()
     * - .ifFileSystemDirectoryPath()
     * - .ifFileSystemId()
     * - .ifFileSystemType()
     * - .ifInstanceTypes()
     * - .ifInterContainerTrafficEncryption()
     * - .ifMaxRuntimeInSeconds()
     * - .ifNetworkIsolation()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateHyperParameterTuningJob.html
     */
    createHyperParameterTuningJob() {
        this.add('sagemaker:CreateHyperParameterTuningJob');
        return this;
    }
    /**
     * Starts a labeling job. A labeling job takes unlabeled data in and produces labeled data as output, which can be used for training SageMaker models.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifWorkteamArn()
     * - .ifWorkteamType()
     * - .ifVolumeKmsKey()
     * - .ifOutputKmsKey()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateLabelingJob.html
     */
    createLabelingJob() {
        this.add('sagemaker:CreateLabelingJob');
        return this;
    }
    /**
     * Creates a model in Amazon SageMaker. In the request, you specify a name for the model and describe one or more containers.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifNetworkIsolation()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateModel.html
     */
    createModel() {
        this.add('sagemaker:CreateModel');
        return this;
    }
    /**
     * Create a model package.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateModelPackage.html
     */
    createModelPackage() {
        this.add('sagemaker:CreateModelPackage');
        return this;
    }
    /**
     * Creates a monitoring schedule.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceTypes()
     * - .ifMaxRuntimeInSeconds()
     * - .ifNetworkIsolation()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateMonitoringSchedule.html
     */
    createMonitoringSchedule() {
        this.add('sagemaker:CreateMonitoringSchedule');
        return this;
    }
    /**
     * Creates an Amazon SageMaker notebook instance. A notebook instance is an Amazon EC2 instance running on a Jupyter Notebook.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAcceleratorTypes()
     * - .ifDirectInternetAccess()
     * - .ifInstanceTypes()
     * - .ifRootAccess()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateNotebookInstance.html
     */
    createNotebookInstance() {
        this.add('sagemaker:CreateNotebookInstance');
        return this;
    }
    /**
     * Creates an notebook instance lifecycle configuration that can be deployed using Amazon SageMaker.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateNotebookInstanceLifecycleConfig.html
     */
    createNotebookInstanceLifecycleConfig() {
        this.add('sagemaker:CreateNotebookInstanceLifecycleConfig');
        return this;
    }
    /**
     * Grants permission to return a URL that you can use from your browser to connect to the Domain as a specified UserProfile when AuthMode is 'IAM'
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreatePresignedDomainUrl.html
     */
    createPresignedDomainUrl() {
        this.add('sagemaker:CreatePresignedDomainUrl');
        return this;
    }
    /**
     * Returns a URL that you can use from your browser to connect to the Notebook Instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreatePresignedNotebookInstanceUrl.html
     */
    createPresignedNotebookInstanceUrl() {
        this.add('sagemaker:CreatePresignedNotebookInstanceUrl');
        return this;
    }
    /**
     * Starts a processing job. After processing completes, Amazon SageMaker saves the resulting artifacts and other optional output to an Amazon S3 location that you specify.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceTypes()
     * - .ifMaxRuntimeInSeconds()
     * - .ifNetworkIsolation()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     * - .ifInterContainerTrafficEncryption()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateProcessingJob.html
     */
    createProcessingJob() {
        this.add('sagemaker:CreateProcessingJob');
        return this;
    }
    /**
     * Starts a model training job. After training completes, Amazon SageMaker saves the resulting model artifacts and other optional output to an Amazon S3 location that you specify.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifFileSystemAccessMode()
     * - .ifFileSystemDirectoryPath()
     * - .ifFileSystemId()
     * - .ifFileSystemType()
     * - .ifInstanceTypes()
     * - .ifInterContainerTrafficEncryption()
     * - .ifMaxRuntimeInSeconds()
     * - .ifNetworkIsolation()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTrainingJob.html
     */
    createTrainingJob() {
        this.add('sagemaker:CreateTrainingJob');
        return this;
    }
    /**
     * Starts a transform job. After the results are obtained, Amazon SageMaker saves them to an Amazon S3 location that you specify.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceTypes()
     * - .ifModelArn()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTransformJob.html
     */
    createTransformJob() {
        this.add('sagemaker:CreateTransformJob');
        return this;
    }
    /**
     * Create a trial.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTrial.html
     */
    createTrial() {
        this.add('sagemaker:CreateTrial');
        return this;
    }
    /**
     * Create a trial component.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTrialComponent.html
     */
    createTrialComponent() {
        this.add('sagemaker:CreateTrialComponent');
        return this;
    }
    /**
     * Grants permission to create a UserProfile for a SageMaker Studio Domain
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifVpcSecurityGroupIds()
     * - .ifInstanceTypes()
     * - .ifDomainSharingOutputKmsKey()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateUserProfile.html
     */
    createUserProfile() {
        this.add('sagemaker:CreateUserProfile');
        return this;
    }
    /**
     * Create a workforce.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateWorkforce.html
     */
    createWorkforce() {
        this.add('sagemaker:CreateWorkforce');
        return this;
    }
    /**
     * Create a workteam.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateWorkteam.html
     */
    createWorkteam() {
        this.add('sagemaker:CreateWorkteam');
        return this;
    }
    /**
     * Deletes an algorithm.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteAlgorithm.html
     */
    deleteAlgorithm() {
        this.add('sagemaker:DeleteAlgorithm');
        return this;
    }
    /**
     * Grants permission to delete an App
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteApp.html
     */
    deleteApp() {
        this.add('sagemaker:DeleteApp');
        return this;
    }
    /**
     * Deletes a code repository.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteCodeRepository.html
     */
    deleteCodeRepository() {
        this.add('sagemaker:DeleteCodeRepository');
        return this;
    }
    /**
     * Grants permission to delete a Domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteDomain.html
     */
    deleteDomain() {
        this.add('sagemaker:DeleteDomain');
        return this;
    }
    /**
     * Deletes an endpoint. Amazon SageMaker frees up all the resources that were deployed when the endpoint was created.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteEndpoint.html
     */
    deleteEndpoint() {
        this.add('sagemaker:DeleteEndpoint');
        return this;
    }
    /**
     * Deletes the endpoint configuration created using the CreateEndpointConfig API. The DeleteEndpointConfig API deletes only the specified configuration. It does not delete any endpoints created using the configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteEndpointConfig.html
     */
    deleteEndpointConfig() {
        this.add('sagemaker:DeleteEndpointConfig');
        return this;
    }
    /**
     * Deletes an experiment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteExperiment.html
     */
    deleteExperiment() {
        this.add('sagemaker:DeleteExperiment');
        return this;
    }
    /**
     * Deltes the specified flow definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteFlowDefinition.html
     */
    deleteFlowDefinition() {
        this.add('sagemaker:DeleteFlowDefinition');
        return this;
    }
    /**
     * Deletes the specified human loop.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteHumanLoop.html
     */
    deleteHumanLoop() {
        this.add('sagemaker:DeleteHumanLoop');
        return this;
    }
    /**
     * Deletes a model created using the CreateModel API. The DeleteModel API deletes only the model entry in Amazon SageMaker that you created by calling the CreateModel API. It does not delete model artifacts, inference code, or the IAM role that you specified when creating the model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteModel.html
     */
    deleteModel() {
        this.add('sagemaker:DeleteModel');
        return this;
    }
    /**
     * Deletes a model package.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteModelPackage.html
     */
    deleteModelPackage() {
        this.add('sagemaker:DeleteModelPackage');
        return this;
    }
    /**
     * Deletes a monitoring schedule. Amazon SageMaker will no longer run the scheduled monitoring.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteMonitoringSchedule.html
     */
    deleteMonitoringSchedule() {
        this.add('sagemaker:DeleteMonitoringSchedule');
        return this;
    }
    /**
     * Deletes an Amazon SageMaker notebook instance. Before you can delete a notebook instance, you must call the StopNotebookInstance API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteNotebookInstance.html
     */
    deleteNotebookInstance() {
        this.add('sagemaker:DeleteNotebookInstance');
        return this;
    }
    /**
     * Deletes an notebook instance lifecycle configuration that can be deployed using Amazon SageMaker.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteNotebookInstanceLifecycleConfig.html
     */
    deleteNotebookInstanceLifecycleConfig() {
        this.add('sagemaker:DeleteNotebookInstanceLifecycleConfig');
        return this;
    }
    /**
     * Deletes the specified set of tags from an Amazon SageMaker resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteTags.html
     */
    deleteTags() {
        this.add('sagemaker:DeleteTags');
        return this;
    }
    /**
     * Deletes a trial.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteTrial.html
     */
    deleteTrial() {
        this.add('sagemaker:DeleteTrial');
        return this;
    }
    /**
     * Deletes a trial component.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteTrialComponent.html
     */
    deleteTrialComponent() {
        this.add('sagemaker:DeleteTrialComponent');
        return this;
    }
    /**
     * Grants permission to delete a UserProfile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteUserProfile.html
     */
    deleteUserProfile() {
        this.add('sagemaker:DeleteUserProfile');
        return this;
    }
    /**
     * Deletes a workforce.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteWorkforce.html
     */
    deleteWorkforce() {
        this.add('sagemaker:DeleteWorkforce');
        return this;
    }
    /**
     * Deletes a workteam.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteWorkteam.html
     */
    deleteWorkteam() {
        this.add('sagemaker:DeleteWorkteam');
        return this;
    }
    /**
     * Returns information about an algorithm.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeAlgorithm.html
     */
    describeAlgorithm() {
        this.add('sagemaker:DescribeAlgorithm');
        return this;
    }
    /**
     * Grants permission to describe an App
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeApp.html
     */
    describeApp() {
        this.add('sagemaker:DescribeApp');
        return this;
    }
    /**
     * Describes an automl job that was created via CreateAutoMLJob API.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeAutoMLJob.html
     */
    describeAutoMLJob() {
        this.add('sagemaker:DescribeAutoMLJob');
        return this;
    }
    /**
     * Returns information about a code repository.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeCodeRepository.html
     */
    describeCodeRepository() {
        this.add('sagemaker:DescribeCodeRepository');
        return this;
    }
    /**
     * Returns information about a compilation job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeCompilationJob.html
     */
    describeCompilationJob() {
        this.add('sagemaker:DescribeCompilationJob');
        return this;
    }
    /**
     * Grants permission to describe a Domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeDomain.html
     */
    describeDomain() {
        this.add('sagemaker:DescribeDomain');
        return this;
    }
    /**
     * Returns the description of an endpoint.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeEndpoint.html
     */
    describeEndpoint() {
        this.add('sagemaker:DescribeEndpoint');
        return this;
    }
    /**
     * Returns the description of an endpoint configuration, which was created using the CreateEndpointConfig API.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeEndpointConfig.html
     */
    describeEndpointConfig() {
        this.add('sagemaker:DescribeEndpointConfig');
        return this;
    }
    /**
     * Returns information about an experiment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeExperiment.html
     */
    describeExperiment() {
        this.add('sagemaker:DescribeExperiment');
        return this;
    }
    /**
     * Returns detailed information about the specified flow definition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeFlowDefinition.html
     */
    describeFlowDefinition() {
        this.add('sagemaker:DescribeFlowDefinition');
        return this;
    }
    /**
     * Returns detailed information about the specified human loop.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeHumanLoop.html
     */
    describeHumanLoop() {
        this.add('sagemaker:DescribeHumanLoop');
        return this;
    }
    /**
     * Returns detailed information about the specified human review workflow user interface.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeHumanTaskUi.html
     */
    describeHumanTaskUi() {
        this.add('sagemaker:DescribeHumanTaskUi');
        return this;
    }
    /**
     * Describes a hyper parameter tuning job that was created via CreateHyperParameterTuningJob API.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeHyperParameterTuningJob.html
     */
    describeHyperParameterTuningJob() {
        this.add('sagemaker:DescribeHyperParameterTuningJob');
        return this;
    }
    /**
     * Returns information about a labeling job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeLabelingJob.html
     */
    describeLabelingJob() {
        this.add('sagemaker:DescribeLabelingJob');
        return this;
    }
    /**
     * Describes a model that you created using the CreateModel API.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeModel.html
     */
    describeModel() {
        this.add('sagemaker:DescribeModel');
        return this;
    }
    /**
     * Returns information about a model package.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeModelPackage.html
     */
    describeModelPackage() {
        this.add('sagemaker:DescribeModelPackage');
        return this;
    }
    /**
     * Returns information about a monitoring schedule.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeMonitoringSchedule.html
     */
    describeMonitoringSchedule() {
        this.add('sagemaker:DescribeMonitoringSchedule');
        return this;
    }
    /**
     * Returns information about a notebook instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeNotebookInstance.html
     */
    describeNotebookInstance() {
        this.add('sagemaker:DescribeNotebookInstance');
        return this;
    }
    /**
     * Describes an notebook instance lifecycle configuration that was created via CreateNotebookInstanceLifecycleConfig API.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeNotebookInstanceLifecycleConfig.html
     */
    describeNotebookInstanceLifecycleConfig() {
        this.add('sagemaker:DescribeNotebookInstanceLifecycleConfig');
        return this;
    }
    /**
     * Returns information about a processing job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeProcessingJob.html
     */
    describeProcessingJob() {
        this.add('sagemaker:DescribeProcessingJob');
        return this;
    }
    /**
     * Returns information about a subscribed workteam.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeSubscribedWorkteam.html
     */
    describeSubscribedWorkteam() {
        this.add('sagemaker:DescribeSubscribedWorkteam');
        return this;
    }
    /**
     * Returns information about a training job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeTrainingJob.html
     */
    describeTrainingJob() {
        this.add('sagemaker:DescribeTrainingJob');
        return this;
    }
    /**
     * Returns information about a transform job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeTransformJob.html
     */
    describeTransformJob() {
        this.add('sagemaker:DescribeTransformJob');
        return this;
    }
    /**
     * Returns information about a trial.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeTrial.html
     */
    describeTrial() {
        this.add('sagemaker:DescribeTrial');
        return this;
    }
    /**
     * Returns information about a trial component.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeTrialComponent.html
     */
    describeTrialComponent() {
        this.add('sagemaker:DescribeTrialComponent');
        return this;
    }
    /**
     * Grants permission to describe a UserProfile
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeUserProfile.html
     */
    describeUserProfile() {
        this.add('sagemaker:DescribeUserProfile');
        return this;
    }
    /**
     * Returns information about a workforce.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeWorkforce.html
     */
    describeWorkforce() {
        this.add('sagemaker:DescribeWorkforce');
        return this;
    }
    /**
     * Returns information about a workteam.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeWorkteam.html
     */
    describeWorkteam() {
        this.add('sagemaker:DescribeWorkteam');
        return this;
    }
    /**
     * Disassociate a trial component with a trial.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DisassociateTrialComponent.html
     */
    disassociateTrialComponent() {
        this.add('sagemaker:DisassociateTrialComponent');
        return this;
    }
    /**
     * Get search suggestions when provided with keyword.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_GetSearchSuggestions.html
     */
    getSearchSuggestions() {
        this.add('sagemaker:GetSearchSuggestions');
        return this;
    }
    /**
     * After you deploy a model into production using Amazon SageMaker hosting services, your client applications use this API to get inferences from the model hosted at the specified endpoint.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTargetModel()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_InvokeEndpoint.html
     */
    invokeEndpoint() {
        this.add('sagemaker:InvokeEndpoint');
        return this;
    }
    /**
     * Lists algorithms.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListAlgorithms.html
     */
    listAlgorithms() {
        this.add('sagemaker:ListAlgorithms');
        return this;
    }
    /**
     * Grants permission to list the Apps in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListApps.html
     */
    listApps() {
        this.add('sagemaker:ListApps');
        return this;
    }
    /**
     * Lists automl jobs created via the CreateAutoMLJob.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListAutoMLJobs.html
     */
    listAutoMLJobs() {
        this.add('sagemaker:ListAutoMLJobs');
        return this;
    }
    /**
     * Lists candidates for automl job created via the CreateAutoMLJob.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListCandidatesForAutoMLJob.html
     */
    listCandidatesForAutoMLJob() {
        this.add('sagemaker:ListCandidatesForAutoMLJob');
        return this;
    }
    /**
     * Lists code repositories.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListCodeRepositories.html
     */
    listCodeRepositories() {
        this.add('sagemaker:ListCodeRepositories');
        return this;
    }
    /**
     * Lists compilation jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListCompilationJobs.html
     */
    listCompilationJobs() {
        this.add('sagemaker:ListCompilationJobs');
        return this;
    }
    /**
     * Grants permission to list the Domains in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListDomains.html
     */
    listDomains() {
        this.add('sagemaker:ListDomains');
        return this;
    }
    /**
     * Lists endpoint configurations.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListEndpointConfigs.html
     */
    listEndpointConfigs() {
        this.add('sagemaker:ListEndpointConfigs');
        return this;
    }
    /**
     * Lists endpoints.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListEndpoints.html
     */
    listEndpoints() {
        this.add('sagemaker:ListEndpoints');
        return this;
    }
    /**
     * Lists experiments.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListExperiments.html
     */
    listExperiments() {
        this.add('sagemaker:ListExperiments');
        return this;
    }
    /**
     * Returns summary information about flow definitions, given the specified parameters.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListFlowDefinitions.html
     */
    listFlowDefinitions() {
        this.add('sagemaker:ListFlowDefinitions');
        return this;
    }
    /**
     * Returns summary information about human loops, given the specified parameters.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListHumanLoops.html
     */
    listHumanLoops() {
        this.add('sagemaker:ListHumanLoops');
        return this;
    }
    /**
     * Returns summary information about human review workflow user interfaces, given the specified parameters.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListHumanTaskUis.html
     */
    listHumanTaskUis() {
        this.add('sagemaker:ListHumanTaskUis');
        return this;
    }
    /**
     * Lists hyper parameter tuning jobs that was created using Amazon SageMaker.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListHyperParameterTuningJobs.html
     */
    listHyperParameterTuningJobs() {
        this.add('sagemaker:ListHyperParameterTuningJobs');
        return this;
    }
    /**
     * Lists labeling jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListLabelingJobs.html
     */
    listLabelingJobs() {
        this.add('sagemaker:ListLabelingJobs');
        return this;
    }
    /**
     * Lists labeling jobs for workteam.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListLabelingJobs.html
     */
    listLabelingJobsForWorkteam() {
        this.add('sagemaker:ListLabelingJobsForWorkteam');
        return this;
    }
    /**
     * Lists model packages.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListModelPackages.html
     */
    listModelPackages() {
        this.add('sagemaker:ListModelPackages');
        return this;
    }
    /**
     * Lists the models created with the CreateModel API.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListModels.html
     */
    listModels() {
        this.add('sagemaker:ListModels');
        return this;
    }
    /**
     * Lists monitoring executions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListMonitoringExecutions.html
     */
    listMonitoringExecutions() {
        this.add('sagemaker:ListMonitoringExecutions');
        return this;
    }
    /**
     * Lists monitoring schedules.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListMonitoringSchedules.html
     */
    listMonitoringSchedules() {
        this.add('sagemaker:ListMonitoringSchedules');
        return this;
    }
    /**
     * Lists notebook instance lifecycle configurations that can be deployed using Amazon SageMaker.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListNotebookInstanceLifecycleConfigs.html
     */
    listNotebookInstanceLifecycleConfigs() {
        this.add('sagemaker:ListNotebookInstanceLifecycleConfigs');
        return this;
    }
    /**
     * Returns a list of the Amazon SageMaker notebook instances in the requester's account in an AWS Region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListNotebookInstances.html
     */
    listNotebookInstances() {
        this.add('sagemaker:ListNotebookInstances');
        return this;
    }
    /**
     * Lists processing jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListProcessingJobs.html
     */
    listProcessingJobs() {
        this.add('sagemaker:ListProcessingJobs');
        return this;
    }
    /**
     * Lists subscribed workteams.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListSubscribedWorkteams.html
     */
    listSubscribedWorkteams() {
        this.add('sagemaker:ListSubscribedWorkteams');
        return this;
    }
    /**
     * Returns the tag set associated with the specified resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListTags.html
     */
    listTags() {
        this.add('sagemaker:ListTags');
        return this;
    }
    /**
     * Lists training jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListTrainingJobs.html
     */
    listTrainingJobs() {
        this.add('sagemaker:ListTrainingJobs');
        return this;
    }
    /**
     * Lists training jobs for a hyper parameter tuning job that was created using Amazon SageMaker.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ ListTrainingJobsForHyperParameterTuningJob.html
     */
    listTrainingJobsForHyperParameterTuningJob() {
        this.add('sagemaker:ListTrainingJobsForHyperParameterTuningJob');
        return this;
    }
    /**
     * Lists transform jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListTransformJobs.html
     */
    listTransformJobs() {
        this.add('sagemaker:ListTransformJobs');
        return this;
    }
    /**
     * Lists trial components.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListTrialComponents.html
     */
    listTrialComponents() {
        this.add('sagemaker:ListTrialComponents');
        return this;
    }
    /**
     * Lists trials.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListTrials.html
     */
    listTrials() {
        this.add('sagemaker:ListTrials');
        return this;
    }
    /**
     * Grants permission to list the UserProfiles in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListUserProfiles.html
     */
    listUserProfiles() {
        this.add('sagemaker:ListUserProfiles');
        return this;
    }
    /**
     * Lists workforces.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListWorkforces.html
     */
    listWorkforces() {
        this.add('sagemaker:ListWorkforces');
        return this;
    }
    /**
     * Lists workteams.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListWorkteams.html
     */
    listWorkteams() {
        this.add('sagemaker:ListWorkteams');
        return this;
    }
    /**
     * Render a UI template used for a human annotation task.
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_RenderUiTemplate.html
     */
    renderUiTemplate() {
        this.add('sagemaker:RenderUiTemplate');
        return this;
    }
    /**
     * Search for a training job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_Search.html
     */
    search() {
        this.add('sagemaker:Search');
        return this;
    }
    /**
     * Starts a human loop.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StartHumanLoop.html
     */
    startHumanLoop() {
        this.add('sagemaker:StartHumanLoop');
        return this;
    }
    /**
     * Starts a monitoring schedule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StartMonitoringSchedule.html
     */
    startMonitoringSchedule() {
        this.add('sagemaker:StartMonitoringSchedule');
        return this;
    }
    /**
     * Launches an EC2 instance with the latest version of the libraries and attaches your EBS volume.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StartNotebookInstance.html
     */
    startNotebookInstance() {
        this.add('sagemaker:StartNotebookInstance');
        return this;
    }
    /**
     * Stops a running automl job created via the CreateAutoMLJob.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopAutoMLJob.html
     */
    stopAutoMLJob() {
        this.add('sagemaker:StopAutoMLJob');
        return this;
    }
    /**
     * Stops a compilation job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopCompilationJob.html
     */
    stopCompilationJob() {
        this.add('sagemaker:StopCompilationJob');
        return this;
    }
    /**
     * Stops the specified human loop.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopHumanLoop.html
     */
    stopHumanLoop() {
        this.add('sagemaker:StopHumanLoop');
        return this;
    }
    /**
     * Stops a running hyper parameter tuning job create via the CreateHyperParameterTuningJob.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopHyperParameterTuningJob.html
     */
    stopHyperParameterTuningJob() {
        this.add('sagemaker:StopHyperParameterTuningJob');
        return this;
    }
    /**
     * Stops a labeling job. Any labels already generated will be exported before stopping.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopLabelingJob.html
     */
    stopLabelingJob() {
        this.add('sagemaker:StopLabelingJob');
        return this;
    }
    /**
     * Stops a monitoring schedule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopMonitoringSchedule.html
     */
    stopMonitoringSchedule() {
        this.add('sagemaker:StopMonitoringSchedule');
        return this;
    }
    /**
     * Terminates the EC2 instance. Before terminating the instance, Amazon SageMaker disconnects the EBS volume from it. Amazon SageMaker preserves the EBS volume.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopNotebookInstance.html
     */
    stopNotebookInstance() {
        this.add('sagemaker:StopNotebookInstance');
        return this;
    }
    /**
     * Stops a processing job. To stop a job, Amazon SageMaker sends the algorithm the SIGTERM signal, which delays job termination for 120 seconds.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopProcessingJob.html
     */
    stopProcessingJob() {
        this.add('sagemaker:StopProcessingJob');
        return this;
    }
    /**
     * Stops a training job. To stop a job, Amazon SageMaker sends the algorithm the SIGTERM signal, which delays job termination for 120 seconds.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopTrainingJob.html
     */
    stopTrainingJob() {
        this.add('sagemaker:StopTrainingJob');
        return this;
    }
    /**
     * Stops a transform job. When Amazon SageMaker receives a StopTransformJob request, the status of the job changes to Stopping. After Amazon SageMaker stops the job, the status is set to Stopped
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopTransformJob.html
     */
    stopTransformJob() {
        this.add('sagemaker:StopTransformJob');
        return this;
    }
    /**
     * Updates a code repository.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateCodeRepository.html
     */
    updateCodeRepository() {
        this.add('sagemaker:UpdateCodeRepository');
        return this;
    }
    /**
     * Grants permission to update a Domain
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifVpcSecurityGroupIds()
     * - .ifInstanceTypes()
     * - .ifDomainSharingOutputKmsKey()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateDomain.html
     */
    updateDomain() {
        this.add('sagemaker:UpdateDomain');
        return this;
    }
    /**
     * Updates an endpoint to use the endpoint configuration specified in the request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateEndpoint.html
     */
    updateEndpoint() {
        this.add('sagemaker:UpdateEndpoint');
        return this;
    }
    /**
     * Updates variant weight, capacity, or both of one or more variants associated with an endpoint.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateEndpointWeightsAndCapacities.html
     */
    updateEndpointWeightsAndCapacities() {
        this.add('sagemaker:UpdateEndpointWeightsAndCapacities');
        return this;
    }
    /**
     * Updates an experiment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateExperiment.html
     */
    updateExperiment() {
        this.add('sagemaker:UpdateExperiment');
        return this;
    }
    /**
     * Updates a monitoring schedule.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceTypes()
     * - .ifMaxRuntimeInSeconds()
     * - .ifNetworkIsolation()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateMonitoringSchedule.html
     */
    updateMonitoringSchedule() {
        this.add('sagemaker:UpdateMonitoringSchedule');
        return this;
    }
    /**
     * Updates a notebook instance. Notebook instance updates include upgrading or downgrading the EC2 instance used for your notebook instance to accommodate changes in your workload requirements. You can also update the VPC security groups.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAcceleratorTypes()
     * - .ifInstanceTypes()
     * - .ifRootAccess()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateNotebookInstance.html
     */
    updateNotebookInstance() {
        this.add('sagemaker:UpdateNotebookInstance');
        return this;
    }
    /**
     * Updates a notebook instance lifecycle configuration created with the CreateNotebookInstanceLifecycleConfig API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ UpdateNotebookInstanceLifecycleConfig.html
     */
    updateNotebookInstanceLifecycleConfig() {
        this.add('sagemaker:UpdateNotebookInstanceLifecycleConfig');
        return this;
    }
    /**
     * Updates a trial.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateTrial.html
     */
    updateTrial() {
        this.add('sagemaker:UpdateTrial');
        return this;
    }
    /**
     * Updates a trial component.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateTrialComponent.html
     */
    updateTrialComponent() {
        this.add('sagemaker:UpdateTrialComponent');
        return this;
    }
    /**
     * Grants permission to update a UserProfile
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceTypes()
     * - .ifVpcSecurityGroupIds()
     * - .ifInstanceTypes()
     * - .ifDomainSharingOutputKmsKey()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateUserProfile.html
     */
    updateUserProfile() {
        this.add('sagemaker:UpdateUserProfile');
        return this;
    }
    /**
     * Updates a workforce.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateWorkforce.html
     */
    updateWorkforce() {
        this.add('sagemaker:UpdateWorkforce');
        return this;
    }
    /**
     * Updates a workteam.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateWorkteam.html
     */
    updateWorkteam() {
        this.add('sagemaker:UpdateWorkteam');
        return this;
    }
    /**
     * Adds a resource of type human-loop to the statement
     *
     * @param humanLoopName - Identifier for the humanLoopName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onHumanLoop(humanLoopName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:human-loop/${HumanLoopName}';
        arn = arn.replace('${HumanLoopName}', humanLoopName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type flow-definition to the statement
     *
     * @param flowDefinitionName - Identifier for the flowDefinitionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onFlowDefinition(flowDefinitionName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:flow-definition/${FlowDefinitionName}';
        arn = arn.replace('${FlowDefinitionName}', flowDefinitionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type human-task-ui to the statement
     *
     * @param humanTaskUiName - Identifier for the humanTaskUiName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onHumanTaskUi(humanTaskUiName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:human-task-ui/${HumanTaskUiName}';
        arn = arn.replace('${HumanTaskUiName}', humanTaskUiName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type labeling-job to the statement
     *
     * @param labelingJobName - Identifier for the labelingJobName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onLabelingJob(labelingJobName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:labeling-job/${LabelingJobName}';
        arn = arn.replace('${LabelingJobName}', labelingJobName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type workteam to the statement
     *
     * @param workteamName - Identifier for the workteamName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onWorkteam(workteamName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:workteam/${WorkteamName}';
        arn = arn.replace('${WorkteamName}', workteamName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type workforce to the statement
     *
     * @param workforceName - Identifier for the workforceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onWorkforce(workforceName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:workforce/${WorkforceName}';
        arn = arn.replace('${WorkforceName}', workforceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * @param domainId - Identifier for the domainId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onDomain(domainId, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:domain/${DomainId}';
        arn = arn.replace('${DomainId}', domainId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type user-profile to the statement
     *
     * @param domainId - Identifier for the domainId.
     * @param userProfileName - Identifier for the userProfileName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onUserProfile(domainId, userProfileName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:user-profile/${DomainId}/${UserProfileName}';
        arn = arn.replace('${DomainId}', domainId);
        arn = arn.replace('${UserProfileName}', userProfileName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type app to the statement
     *
     * @param domainId - Identifier for the domainId.
     * @param userProfileName - Identifier for the userProfileName.
     * @param appType - Identifier for the appType.
     * @param appName - Identifier for the appName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onApp(domainId, userProfileName, appType, appName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:app/${DomainId}/${UserProfileName}/${AppType}/${AppName}';
        arn = arn.replace('${DomainId}', domainId);
        arn = arn.replace('${UserProfileName}', userProfileName);
        arn = arn.replace('${AppType}', appType);
        arn = arn.replace('${AppName}', appName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type notebook-instance to the statement
     *
     * @param notebookInstanceName - Identifier for the notebookInstanceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onNotebookInstance(notebookInstanceName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:notebook-instance/${NotebookInstanceName}';
        arn = arn.replace('${NotebookInstanceName}', notebookInstanceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type notebook-instance-lifecycle-config to the statement
     *
     * @param notebookInstanceLifecycleConfigName - Identifier for the notebookInstanceLifecycleConfigName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onNotebookInstanceLifecycleConfig(notebookInstanceLifecycleConfigName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:notebook-instance-lifecycle-config/${NotebookInstanceLifecycleConfigName}';
        arn = arn.replace('${NotebookInstanceLifecycleConfigName}', notebookInstanceLifecycleConfigName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type code-repository to the statement
     *
     * @param codeRepositoryName - Identifier for the codeRepositoryName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCodeRepository(codeRepositoryName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:code-repository/${CodeRepositoryName}';
        arn = arn.replace('${CodeRepositoryName}', codeRepositoryName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type algorithm to the statement
     *
     * @param algorithmName - Identifier for the algorithmName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAlgorithm(algorithmName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:algorithm/${AlgorithmName}';
        arn = arn.replace('${AlgorithmName}', algorithmName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type training-job to the statement
     *
     * @param trainingJobName - Identifier for the trainingJobName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onTrainingJob(trainingJobName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:training-job/${TrainingJobName}';
        arn = arn.replace('${TrainingJobName}', trainingJobName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type processing-job to the statement
     *
     * @param processingJobName - Identifier for the processingJobName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onProcessingJob(processingJobName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:processing-job/${ProcessingJobName}';
        arn = arn.replace('${ProcessingJobName}', processingJobName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type hyper-parameter-tuning-job to the statement
     *
     * @param hyperParameterTuningJobName - Identifier for the hyperParameterTuningJobName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onHyperParameterTuningJob(hyperParameterTuningJobName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:hyper-parameter-tuning-job/${HyperParameterTuningJobName}';
        arn = arn.replace('${HyperParameterTuningJobName}', hyperParameterTuningJobName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type model-package to the statement
     *
     * @param modelPackageName - Identifier for the modelPackageName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onModelPackage(modelPackageName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:model-package/${ModelPackageName}';
        arn = arn.replace('${ModelPackageName}', modelPackageName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type model to the statement
     *
     * @param modelName - Identifier for the modelName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onModel(modelName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:model/${ModelName}';
        arn = arn.replace('${ModelName}', modelName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type endpoint-config to the statement
     *
     * @param endpointConfigName - Identifier for the endpointConfigName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onEndpointConfig(endpointConfigName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:endpoint-config/${EndpointConfigName}';
        arn = arn.replace('${EndpointConfigName}', endpointConfigName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type endpoint to the statement
     *
     * @param endpointName - Identifier for the endpointName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onEndpoint(endpointName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:endpoint/${EndpointName}';
        arn = arn.replace('${EndpointName}', endpointName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type transform-job to the statement
     *
     * @param transformJobName - Identifier for the transformJobName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onTransformJob(transformJobName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:transform-job/${TransformJobName}';
        arn = arn.replace('${TransformJobName}', transformJobName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type compilation-job to the statement
     *
     * @param compilationJobName - Identifier for the compilationJobName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCompilationJob(compilationJobName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:compilation-job/${CompilationJobName}';
        arn = arn.replace('${CompilationJobName}', compilationJobName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type automl-job to the statement
     *
     * @param autoMLJobJobName - Identifier for the autoMLJobJobName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onAutomlJob(autoMLJobJobName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:automl-job/${AutoMLJobJobName}';
        arn = arn.replace('${AutoMLJobJobName}', autoMLJobJobName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type monitoring-schedule to the statement
     *
     * @param monitoringScheduleName - Identifier for the monitoringScheduleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onMonitoringSchedule(monitoringScheduleName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:monitoring-schedule/${MonitoringScheduleName}';
        arn = arn.replace('${MonitoringScheduleName}', monitoringScheduleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type experiment to the statement
     *
     * @param experimentName - Identifier for the experimentName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onExperiment(experimentName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:experiment/${ExperimentName}';
        arn = arn.replace('${ExperimentName}', experimentName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type experiment-trial to the statement
     *
     * @param trialName - Identifier for the trialName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onExperimentTrial(trialName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:experiment-trial/${TrialName}';
        arn = arn.replace('${TrialName}', trialName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type experiment-trial-component to the statement
     *
     * @param trialComponentName - Identifier for the trialComponentName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onExperimentTrialComponent(trialComponentName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:experiment-trial-component/${TrialComponentName}';
        arn = arn.replace('${TrialComponentName}', trialComponentName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * A key that is present in the request the user makes to the SageMaker service.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .addTags()
     * - .createApp()
     * - .createAutoMLJob()
     * - .createDomain()
     * - .createEndpoint()
     * - .createEndpointConfig()
     * - .createExperiment()
     * - .createFlowDefinition()
     * - .createHumanTaskUi()
     * - .createHyperParameterTuningJob()
     * - .createLabelingJob()
     * - .createModel()
     * - .createMonitoringSchedule()
     * - .createNotebookInstance()
     * - .createProcessingJob()
     * - .createTrainingJob()
     * - .createTransformJob()
     * - .createTrial()
     * - .createTrialComponent()
     * - .createUserProfile()
     * - .createWorkforce()
     * - .createWorkteam()
     * - .updateMonitoringSchedule()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * A tag key and value pair.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to resource types:
     * - flow-definition
     * - human-task-ui
     * - labeling-job
     * - workteam
     * - workforce
     * - domain
     * - user-profile
     * - app
     * - notebook-instance
     * - training-job
     * - processing-job
     * - hyper-parameter-tuning-job
     * - model
     * - endpoint-config
     * - endpoint
     * - transform-job
     * - automl-job
     * - monitoring-schedule
     * - experiment
     * - experiment-trial
     * - experiment-trial-component
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * The list of all the tag key names associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .addTags()
     * - .createApp()
     * - .createAutoMLJob()
     * - .createDomain()
     * - .createEndpoint()
     * - .createEndpointConfig()
     * - .createExperiment()
     * - .createFlowDefinition()
     * - .createHumanTaskUi()
     * - .createHyperParameterTuningJob()
     * - .createLabelingJob()
     * - .createModel()
     * - .createMonitoringSchedule()
     * - .createNotebookInstance()
     * - .createProcessingJob()
     * - .createTrainingJob()
     * - .createTransformJob()
     * - .createTrial()
     * - .createTrialComponent()
     * - .createUserProfile()
     * - .createWorkforce()
     * - .createWorkteam()
     * - .deleteTags()
     * - .updateMonitoringSchedule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
    /**
     * The list of all accelerator types associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .createEndpointConfig()
     * - .createNotebookInstance()
     * - .updateNotebookInstance()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAcceleratorTypes(value, operator) {
        return this.if(`sagemaker:AcceleratorTypes`, value, operator || 'StringLike');
    }
    /**
     * App network access associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .createDomain()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAppNetworkAccess(value, operator) {
        return this.if(`sagemaker:AppNetworkAccess`, value, operator || 'StringLike');
    }
    /**
     * The direct internet access associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .createNotebookInstance()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDirectInternetAccess(value, operator) {
        return this.if(`sagemaker:DirectInternetAccess`, value, operator || 'StringLike');
    }
    /**
     * The Domain sharing output KMS key associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .createDomain()
     * - .createUserProfile()
     * - .updateDomain()
     * - .updateUserProfile()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifDomainSharingOutputKmsKey(value, operator) {
        return this.if(`sagemaker:DomainSharingOutputKmsKey`, value, operator || 'ArnEquals');
    }
    /**
     * File system access mode associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .createHyperParameterTuningJob()
     * - .createTrainingJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemAccessMode(value, operator) {
        return this.if(`sagemaker:FileSystemAccessMode`, value, operator || 'StringLike');
    }
    /**
     * File system directory path associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .createHyperParameterTuningJob()
     * - .createTrainingJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemDirectoryPath(value, operator) {
        return this.if(`sagemaker:FileSystemDirectoryPath`, value, operator || 'StringLike');
    }
    /**
     * A file system ID associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .createHyperParameterTuningJob()
     * - .createTrainingJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemId(value, operator) {
        return this.if(`sagemaker:FileSystemId`, value, operator || 'StringLike');
    }
    /**
     * File system type associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .createHyperParameterTuningJob()
     * - .createTrainingJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemType(value, operator) {
        return this.if(`sagemaker:FileSystemType`, value, operator || 'StringLike');
    }
    /**
     * The KMS Key Id of the EFS File System used for UserProfile home directories, which is associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .createDomain()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifHomeEfsFileSystemKmsKey(value, operator) {
        return this.if(`sagemaker:HomeEfsFileSystemKmsKey`, value, operator || 'ArnEquals');
    }
    /**
     * The list of all instance types associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .createApp()
     * - .createDomain()
     * - .createEndpointConfig()
     * - .createHyperParameterTuningJob()
     * - .createMonitoringSchedule()
     * - .createNotebookInstance()
     * - .createProcessingJob()
     * - .createTrainingJob()
     * - .createTransformJob()
     * - .createUserProfile()
     * - .updateDomain()
     * - .updateMonitoringSchedule()
     * - .updateNotebookInstance()
     * - .updateUserProfile()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceTypes(value, operator) {
        return this.if(`sagemaker:InstanceTypes`, value, operator || 'StringLike');
    }
    /**
     * The inter container traffic encryption associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .createAutoMLJob()
     * - .createHyperParameterTuningJob()
     * - .createProcessingJob()
     * - .createTrainingJob()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifInterContainerTrafficEncryption(value) {
        return this.if(`sagemaker:InterContainerTrafficEncryption`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * The max runtime in seconds associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .createHyperParameterTuningJob()
     * - .createMonitoringSchedule()
     * - .createProcessingJob()
     * - .createTrainingJob()
     * - .updateMonitoringSchedule()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMaxRuntimeInSeconds(value, operator) {
        return this.if(`sagemaker:MaxRuntimeInSeconds`, value, operator || 'NumericEquals');
    }
    /**
     * The model arn associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .createEndpointConfig()
     * - .createTransformJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifModelArn(value, operator) {
        return this.if(`sagemaker:ModelArn`, value, operator || 'ArnEquals');
    }
    /**
     * The network isolation associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .createHyperParameterTuningJob()
     * - .createModel()
     * - .createMonitoringSchedule()
     * - .createProcessingJob()
     * - .createTrainingJob()
     * - .updateMonitoringSchedule()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifNetworkIsolation(value) {
        return this.if(`sagemaker:NetworkIsolation`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * The output kms key associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .createAutoMLJob()
     * - .createHyperParameterTuningJob()
     * - .createLabelingJob()
     * - .createMonitoringSchedule()
     * - .createProcessingJob()
     * - .createTrainingJob()
     * - .createTransformJob()
     * - .updateMonitoringSchedule()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifOutputKmsKey(value, operator) {
        return this.if(`sagemaker:OutputKmsKey`, value, operator || 'ArnEquals');
    }
    /**
     * The preface string for a tag key and value pair attached to a resource.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTagExists(value, operator) {
        return this.if(`sagemaker:ResourceTag/`, value, operator || 'StringLike');
    }
    /**
     * A tag key and value pair.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to resource types:
     * - flow-definition
     * - human-task-ui
     * - labeling-job
     * - workteam
     * - workforce
     * - domain
     * - user-profile
     * - app
     * - notebook-instance
     * - training-job
     * - processing-job
     * - hyper-parameter-tuning-job
     * - model
     * - endpoint-config
     * - endpoint
     * - transform-job
     * - automl-job
     * - monitoring-schedule
     * - experiment
     * - experiment-trial
     * - experiment-trial-component
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`sagemaker:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * The root access associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .createNotebookInstance()
     * - .updateNotebookInstance()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRootAccess(value, operator) {
        return this.if(`sagemaker:RootAccess`, value, operator || 'StringLike');
    }
    /**
     * The target model associated with the Multi-Model Endpoint in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .invokeEndpoint()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTargetModel(value, operator) {
        return this.if(`sagemaker:TargetModel`, value, operator || 'StringLike');
    }
    /**
     * The volume kms key associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .createAutoMLJob()
     * - .createEndpointConfig()
     * - .createHyperParameterTuningJob()
     * - .createLabelingJob()
     * - .createMonitoringSchedule()
     * - .createNotebookInstance()
     * - .createProcessingJob()
     * - .createTrainingJob()
     * - .createTransformJob()
     * - .updateMonitoringSchedule()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifVolumeKmsKey(value, operator) {
        return this.if(`sagemaker:VolumeKmsKey`, value, operator || 'ArnEquals');
    }
    /**
     * The list of all vpc security group ids associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .createAutoMLJob()
     * - .createDomain()
     * - .createHyperParameterTuningJob()
     * - .createModel()
     * - .createMonitoringSchedule()
     * - .createNotebookInstance()
     * - .createProcessingJob()
     * - .createTrainingJob()
     * - .createUserProfile()
     * - .updateDomain()
     * - .updateMonitoringSchedule()
     * - .updateUserProfile()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcSecurityGroupIds(value, operator) {
        return this.if(`sagemaker:VpcSecurityGroupIds`, value, operator || 'StringLike');
    }
    /**
     * The list of all vpc subnets associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .createAutoMLJob()
     * - .createDomain()
     * - .createHyperParameterTuningJob()
     * - .createModel()
     * - .createMonitoringSchedule()
     * - .createNotebookInstance()
     * - .createProcessingJob()
     * - .createTrainingJob()
     * - .updateMonitoringSchedule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcSubnets(value, operator) {
        return this.if(`sagemaker:VpcSubnets`, value, operator || 'StringLike');
    }
    /**
     * The workteam arn associated to the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .createFlowDefinition()
     * - .createLabelingJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifWorkteamArn(value, operator) {
        return this.if(`sagemaker:WorkteamArn`, value, operator || 'ArnEquals');
    }
    /**
     * The workteam type associated to the request. This can be public-crowd, private-crowd or vendor-crowd.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .createFlowDefinition()
     * - .createLabelingJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWorkteamType(value, operator) {
        return this.if(`sagemaker:WorkteamType`, value, operator || 'StringLike');
    }
}
exports.Sagemaker = Sagemaker;
//# sourceMappingURL=data:application/json;base64,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