"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Swf = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [swf](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsimpleworkflowservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Swf extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [swf](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsimpleworkflowservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'swf';
        this.actionList = {
            "CancelTimer": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CancelTimer.html",
                "description": "Description for CancelTimer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "CancelWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CancelWorkflowExecution.html",
                "description": "Description for CancelWorkflowExecution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "CompleteWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CompleteWorkflowExecution.html",
                "description": "Description for CompleteWorkflowExecution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "ContinueAsNewWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ContinueAsNewWorkflowExecution.html",
                "description": "Description for ContinueAsNewWorkflowExecution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "CountClosedWorkflowExecutions": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountClosedWorkflowExecutions.html",
                "description": "Returns the number of closed workflow executions within the given domain that meet the specified filtering criteria.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:tagFilter.tag",
                    "swf:typeFilter.name",
                    "swf:typeFilter.version"
                ]
            },
            "CountOpenWorkflowExecutions": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountOpenWorkflowExecutions.html",
                "description": "Returns the number of open workflow executions within the given domain that meet the specified filtering criteria.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:tagFilter.tag",
                    "swf:typeFilter.name",
                    "swf:typeFilter.version"
                ]
            },
            "CountPendingActivityTasks": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountPendingActivityTasks.html",
                "description": "Returns the estimated number of activity tasks in the specified task list.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:taskList.name"
                ]
            },
            "CountPendingDecisionTasks": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountPendingDecisionTasks.html",
                "description": "Returns the estimated number of decision tasks in the specified task list.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:taskList.name"
                ]
            },
            "DeprecateActivityType": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateActivityType.html",
                "description": "Deprecates the specified activity type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:activityType.name",
                    "swf:activityType.version"
                ]
            },
            "DeprecateDomain": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateDomain.html",
                "description": "Deprecates the specified domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeprecateWorkflowType": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateWorkflowType.html",
                "description": "Deprecates the specified workflow type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:workflowType.name",
                    "swf:workflowType.version"
                ]
            },
            "DescribeActivityType": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeActivityType.html",
                "description": "Returns information about the specified activity type.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:activityType.name",
                    "swf:activityType.version"
                ]
            },
            "DescribeDomain": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeDomain.html",
                "description": "Returns information about the specified domain, including description and status.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeWorkflowExecution.html",
                "description": "Returns information about the specified workflow execution including its type and some statistics.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeWorkflowType": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeWorkflowType.html",
                "description": "Returns information about the specified workflow type.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:workflowType.name",
                    "swf:workflowType.version"
                ]
            },
            "FailWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_FailWorkflowExecution.html",
                "description": "Description for FailWorkflowExecution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "GetWorkflowExecutionHistory": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_GetWorkflowExecutionHistory.html",
                "description": "Returns the history of the specified workflow execution.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "ListActivityTypes": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListActivityTypes.html",
                "description": "Returns information about all activities registered in the specified domain that match the specified name and registration status.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "ListClosedWorkflowExecutions": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListClosedWorkflowExecutions.html",
                "description": "Returns a list of closed workflow executions in the specified domain that meet the filtering criteria.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:tagFilter.tag",
                    "swf:typeFilter.name",
                    "swf:typeFilter.version"
                ]
            },
            "ListDomains": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListDomains.html",
                "description": "Returns the list of domains registered in the account.",
                "accessLevel": "List"
            },
            "ListOpenWorkflowExecutions": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListOpenWorkflowExecutions.html",
                "description": "Returns a list of open workflow executions in the specified domain that meet the filtering criteria.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:tagFilter.tag",
                    "swf:typeFilter.name",
                    "swf:typeFilter.version"
                ]
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListTagsForResource.html",
                "description": "This action lists tags for an AWS SWF resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "ListWorkflowTypes": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListWorkflowTypes.html",
                "description": "Returns information about workflow types in the specified domain.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "PollForActivityTask": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_PollForActivityTask.html",
                "description": "Used by workers to get an ActivityTask from the specified activity taskList.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:taskList.name"
                ]
            },
            "PollForDecisionTask": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_PollForDecisionTask.html",
                "description": "Used by deciders to get a DecisionTask from the specified decision taskList.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:taskList.name"
                ]
            },
            "RecordActivityTaskHeartbeat": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RecordActivityTaskHeartbeat.html",
                "description": "Used by activity workers to report to the service that the ActivityTask represented by the specified taskToken is still making progress.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RecordMarker": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RecordMarker.html",
                "description": "Description for RecordMarker",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RegisterActivityType": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterActivityType.html",
                "description": "Registers a new activity type along with its configuration settings in the specified domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:defaultTaskList.name",
                    "swf:name",
                    "swf:version"
                ]
            },
            "RegisterDomain": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterDomain.html",
                "description": "Registers a new domain.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "RegisterWorkflowType": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterWorkflowType.html",
                "description": "Registers a new workflow type and its configuration settings in the specified domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:defaultTaskList.name",
                    "swf:name",
                    "swf:version"
                ]
            },
            "RequestCancelActivityTask": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelActivityTask.html",
                "description": "Description for RequestCancelActivityTask",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RequestCancelExternalWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelExternalWorkflowExecution.html",
                "description": "Description for RequestCancelExternalWorkflowExecution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RequestCancelWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelWorkflowExecution.html",
                "description": "Records a WorkflowExecutionCancelRequested event in the currently running workflow execution identified by the given domain, workflowId, and runId.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RespondActivityTaskCanceled": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskCanceled.html",
                "description": "Used by workers to tell the service that the ActivityTask identified by the taskToken was successfully canceled.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RespondActivityTaskCompleted": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskCompleted.html",
                "description": "Used by workers to tell the service that the ActivityTask identified by the taskToken completed successfully with a result (if provided).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:activityType.name",
                    "swf:activityType.version",
                    "swf:tagList.member.0",
                    "swf:tagList.member.1",
                    "swf:tagList.member.2",
                    "swf:tagList.member.3",
                    "swf:tagList.member.4",
                    "swf:taskList.name",
                    "swf:workflowType.name",
                    "swf:workflowType.version"
                ]
            },
            "RespondActivityTaskFailed": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskFailed.html",
                "description": "Used by workers to tell the service that the ActivityTask identified by the taskToken has failed with reason (if specified).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RespondDecisionTaskCompleted": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondDecisionTaskCompleted.html",
                "description": "Used by deciders to tell the service that the DecisionTask identified by the taskToken has successfully completed.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "ScheduleActivityTask": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ScheduleActivityTask.html",
                "description": "Description for ScheduleActivityTask",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "SignalExternalWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_SignalExternalWorkflowExecution.html",
                "description": "Description for SignalExternalWorkflowExecution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "SignalWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_SignalWorkflowExecution.html",
                "description": "Records a WorkflowExecutionSignaled event in the workflow execution history and creates a decision task for the workflow execution identified by the given domain, workflowId and runId.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "StartChildWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartChildWorkflowExecution.html",
                "description": "Description for StartChildWorkflowExecution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "StartTimer": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartTimer.html",
                "description": "Description for StartTimer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "StartWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartWorkflowExecution.html",
                "description": "Starts an execution of the workflow type in the specified domain using the provided workflowId and input data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:tagList.member.0",
                    "swf:tagList.member.1",
                    "swf:tagList.member.2",
                    "swf:tagList.member.3",
                    "swf:tagList.member.4",
                    "swf:taskList.name",
                    "swf:workflowType.name",
                    "swf:workflowType.version"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_TagResource.html",
                "description": "This action tags an AWS SWF resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "TerminateWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_TerminateWorkflowExecution.html",
                "description": "Records a WorkflowExecutionTerminated event and forces closure of the workflow execution identified by the given domain, runId, and workflowId.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_UntagResource.html",
                "description": "This action removes a tag from an AWS SWF resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            }
        };
        this.resourceTypes = {
            "domain": {
                "name": "domain",
                "url": "https://docs.aws.amazon.com/swf/latest/developerguide/swf-dev-domains.html",
                "arn": "arn:${Partition}:swf::${Account}:domain/${DomainName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Description for CancelTimer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CancelTimer.html
     */
    cancelTimer() {
        this.add('swf:CancelTimer');
        return this;
    }
    /**
     * Description for CancelWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CancelWorkflowExecution.html
     */
    cancelWorkflowExecution() {
        this.add('swf:CancelWorkflowExecution');
        return this;
    }
    /**
     * Description for CompleteWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CompleteWorkflowExecution.html
     */
    completeWorkflowExecution() {
        this.add('swf:CompleteWorkflowExecution');
        return this;
    }
    /**
     * Description for ContinueAsNewWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ContinueAsNewWorkflowExecution.html
     */
    continueAsNewWorkflowExecution() {
        this.add('swf:ContinueAsNewWorkflowExecution');
        return this;
    }
    /**
     * Returns the number of closed workflow executions within the given domain that meet the specified filtering criteria.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTagFilterTag()
     * - .ifTypeFilterName()
     * - .ifTypeFilterVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountClosedWorkflowExecutions.html
     */
    countClosedWorkflowExecutions() {
        this.add('swf:CountClosedWorkflowExecutions');
        return this;
    }
    /**
     * Returns the number of open workflow executions within the given domain that meet the specified filtering criteria.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTagFilterTag()
     * - .ifTypeFilterName()
     * - .ifTypeFilterVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountOpenWorkflowExecutions.html
     */
    countOpenWorkflowExecutions() {
        this.add('swf:CountOpenWorkflowExecutions');
        return this;
    }
    /**
     * Returns the estimated number of activity tasks in the specified task list.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTaskListName()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountPendingActivityTasks.html
     */
    countPendingActivityTasks() {
        this.add('swf:CountPendingActivityTasks');
        return this;
    }
    /**
     * Returns the estimated number of decision tasks in the specified task list.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTaskListName()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountPendingDecisionTasks.html
     */
    countPendingDecisionTasks() {
        this.add('swf:CountPendingDecisionTasks');
        return this;
    }
    /**
     * Deprecates the specified activity type.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifActivityTypeName()
     * - .ifActivityTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateActivityType.html
     */
    deprecateActivityType() {
        this.add('swf:DeprecateActivityType');
        return this;
    }
    /**
     * Deprecates the specified domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateDomain.html
     */
    deprecateDomain() {
        this.add('swf:DeprecateDomain');
        return this;
    }
    /**
     * Deprecates the specified workflow type.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifWorkflowTypeName()
     * - .ifWorkflowTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateWorkflowType.html
     */
    deprecateWorkflowType() {
        this.add('swf:DeprecateWorkflowType');
        return this;
    }
    /**
     * Returns information about the specified activity type.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifActivityTypeName()
     * - .ifActivityTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeActivityType.html
     */
    describeActivityType() {
        this.add('swf:DescribeActivityType');
        return this;
    }
    /**
     * Returns information about the specified domain, including description and status.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeDomain.html
     */
    describeDomain() {
        this.add('swf:DescribeDomain');
        return this;
    }
    /**
     * Returns information about the specified workflow execution including its type and some statistics.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeWorkflowExecution.html
     */
    describeWorkflowExecution() {
        this.add('swf:DescribeWorkflowExecution');
        return this;
    }
    /**
     * Returns information about the specified workflow type.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifWorkflowTypeName()
     * - .ifWorkflowTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeWorkflowType.html
     */
    describeWorkflowType() {
        this.add('swf:DescribeWorkflowType');
        return this;
    }
    /**
     * Description for FailWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_FailWorkflowExecution.html
     */
    failWorkflowExecution() {
        this.add('swf:FailWorkflowExecution');
        return this;
    }
    /**
     * Returns the history of the specified workflow execution.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_GetWorkflowExecutionHistory.html
     */
    getWorkflowExecutionHistory() {
        this.add('swf:GetWorkflowExecutionHistory');
        return this;
    }
    /**
     * Returns information about all activities registered in the specified domain that match the specified name and registration status.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListActivityTypes.html
     */
    listActivityTypes() {
        this.add('swf:ListActivityTypes');
        return this;
    }
    /**
     * Returns a list of closed workflow executions in the specified domain that meet the filtering criteria.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifTagFilterTag()
     * - .ifTypeFilterName()
     * - .ifTypeFilterVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListClosedWorkflowExecutions.html
     */
    listClosedWorkflowExecutions() {
        this.add('swf:ListClosedWorkflowExecutions');
        return this;
    }
    /**
     * Returns the list of domains registered in the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListDomains.html
     */
    listDomains() {
        this.add('swf:ListDomains');
        return this;
    }
    /**
     * Returns a list of open workflow executions in the specified domain that meet the filtering criteria.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifTagFilterTag()
     * - .ifTypeFilterName()
     * - .ifTypeFilterVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListOpenWorkflowExecutions.html
     */
    listOpenWorkflowExecutions() {
        this.add('swf:ListOpenWorkflowExecutions');
        return this;
    }
    /**
     * This action lists tags for an AWS SWF resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('swf:ListTagsForResource');
        return this;
    }
    /**
     * Returns information about workflow types in the specified domain.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListWorkflowTypes.html
     */
    listWorkflowTypes() {
        this.add('swf:ListWorkflowTypes');
        return this;
    }
    /**
     * Used by workers to get an ActivityTask from the specified activity taskList.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTaskListName()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_PollForActivityTask.html
     */
    pollForActivityTask() {
        this.add('swf:PollForActivityTask');
        return this;
    }
    /**
     * Used by deciders to get a DecisionTask from the specified decision taskList.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTaskListName()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_PollForDecisionTask.html
     */
    pollForDecisionTask() {
        this.add('swf:PollForDecisionTask');
        return this;
    }
    /**
     * Used by activity workers to report to the service that the ActivityTask represented by the specified taskToken is still making progress.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RecordActivityTaskHeartbeat.html
     */
    recordActivityTaskHeartbeat() {
        this.add('swf:RecordActivityTaskHeartbeat');
        return this;
    }
    /**
     * Description for RecordMarker
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RecordMarker.html
     */
    recordMarker() {
        this.add('swf:RecordMarker');
        return this;
    }
    /**
     * Registers a new activity type along with its configuration settings in the specified domain.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDefaultTaskListName()
     * - .ifName()
     * - .ifVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterActivityType.html
     */
    registerActivityType() {
        this.add('swf:RegisterActivityType');
        return this;
    }
    /**
     * Registers a new domain.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterDomain.html
     */
    registerDomain() {
        this.add('swf:RegisterDomain');
        return this;
    }
    /**
     * Registers a new workflow type and its configuration settings in the specified domain.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDefaultTaskListName()
     * - .ifName()
     * - .ifVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterWorkflowType.html
     */
    registerWorkflowType() {
        this.add('swf:RegisterWorkflowType');
        return this;
    }
    /**
     * Description for RequestCancelActivityTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelActivityTask.html
     */
    requestCancelActivityTask() {
        this.add('swf:RequestCancelActivityTask');
        return this;
    }
    /**
     * Description for RequestCancelExternalWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelExternalWorkflowExecution.html
     */
    requestCancelExternalWorkflowExecution() {
        this.add('swf:RequestCancelExternalWorkflowExecution');
        return this;
    }
    /**
     * Records a WorkflowExecutionCancelRequested event in the currently running workflow execution identified by the given domain, workflowId, and runId.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelWorkflowExecution.html
     */
    requestCancelWorkflowExecution() {
        this.add('swf:RequestCancelWorkflowExecution');
        return this;
    }
    /**
     * Used by workers to tell the service that the ActivityTask identified by the taskToken was successfully canceled.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskCanceled.html
     */
    respondActivityTaskCanceled() {
        this.add('swf:RespondActivityTaskCanceled');
        return this;
    }
    /**
     * Used by workers to tell the service that the ActivityTask identified by the taskToken completed successfully with a result (if provided).
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifActivityTypeName()
     * - .ifActivityTypeVersion()
     * - .ifTagListMember0()
     * - .ifTagListMember1()
     * - .ifTagListMember2()
     * - .ifTagListMember3()
     * - .ifTagListMember4()
     * - .ifTaskListName()
     * - .ifWorkflowTypeName()
     * - .ifWorkflowTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskCompleted.html
     */
    respondActivityTaskCompleted() {
        this.add('swf:RespondActivityTaskCompleted');
        return this;
    }
    /**
     * Used by workers to tell the service that the ActivityTask identified by the taskToken has failed with reason (if specified).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskFailed.html
     */
    respondActivityTaskFailed() {
        this.add('swf:RespondActivityTaskFailed');
        return this;
    }
    /**
     * Used by deciders to tell the service that the DecisionTask identified by the taskToken has successfully completed.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondDecisionTaskCompleted.html
     */
    respondDecisionTaskCompleted() {
        this.add('swf:RespondDecisionTaskCompleted');
        return this;
    }
    /**
     * Description for ScheduleActivityTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ScheduleActivityTask.html
     */
    scheduleActivityTask() {
        this.add('swf:ScheduleActivityTask');
        return this;
    }
    /**
     * Description for SignalExternalWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_SignalExternalWorkflowExecution.html
     */
    signalExternalWorkflowExecution() {
        this.add('swf:SignalExternalWorkflowExecution');
        return this;
    }
    /**
     * Records a WorkflowExecutionSignaled event in the workflow execution history and creates a decision task for the workflow execution identified by the given domain, workflowId and runId.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_SignalWorkflowExecution.html
     */
    signalWorkflowExecution() {
        this.add('swf:SignalWorkflowExecution');
        return this;
    }
    /**
     * Description for StartChildWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartChildWorkflowExecution.html
     */
    startChildWorkflowExecution() {
        this.add('swf:StartChildWorkflowExecution');
        return this;
    }
    /**
     * Description for StartTimer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartTimer.html
     */
    startTimer() {
        this.add('swf:StartTimer');
        return this;
    }
    /**
     * Starts an execution of the workflow type in the specified domain using the provided workflowId and input data.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTagListMember0()
     * - .ifTagListMember1()
     * - .ifTagListMember2()
     * - .ifTagListMember3()
     * - .ifTagListMember4()
     * - .ifTaskListName()
     * - .ifWorkflowTypeName()
     * - .ifWorkflowTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartWorkflowExecution.html
     */
    startWorkflowExecution() {
        this.add('swf:StartWorkflowExecution');
        return this;
    }
    /**
     * This action tags an AWS SWF resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_TagResource.html
     */
    tagResource() {
        this.add('swf:TagResource');
        return this;
    }
    /**
     * Records a WorkflowExecutionTerminated event and forces closure of the workflow execution identified by the given domain, runId, and workflowId.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_TerminateWorkflowExecution.html
     */
    terminateWorkflowExecution() {
        this.add('swf:TerminateWorkflowExecution');
        return this;
    }
    /**
     * This action removes a tag from an AWS SWF resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_UntagResource.html
     */
    untagResource() {
        this.add('swf:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * https://docs.aws.amazon.com/swf/latest/developerguide/swf-dev-domains.html
     *
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDomain(domainName, account, partition) {
        var arn = 'arn:${Partition}:swf::${Account}:domain/${DomainName}';
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Tag for request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .registerDomain()
     * - .tagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Tag for resource.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - domain
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Tag for key.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .registerDomain()
     * - .tagResource()
     * - .untagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only an activity type of the specified name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .deprecateActivityType()
     * - .describeActivityType()
     * - .respondActivityTaskCompleted()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifActivityTypeName(value, operator) {
        return this.if(`swf:activityType.name`, value, operator || 'StringLike');
    }
    /**
     * Contstrains the policy statement to only an activity type of the specified version.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .deprecateActivityType()
     * - .describeActivityType()
     * - .respondActivityTaskCompleted()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifActivityTypeVersion(value, operator) {
        return this.if(`swf:activityType.version`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a matching defaultTaskList name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .registerActivityType()
     * - .registerWorkflowType()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDefaultTaskListName(value, operator) {
        return this.if(`swf:defaultTaskList.name`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only activities or workflows with the specified name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .registerActivityType()
     * - .registerWorkflowType()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifName(value, operator) {
        return this.if(`swf:name`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a matching tagFilter.tag value.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .countClosedWorkflowExecutions()
     * - .countOpenWorkflowExecutions()
     * - .listClosedWorkflowExecutions()
     * - .listOpenWorkflowExecutions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagFilterTag(value, operator) {
        return this.if(`swf:tagFilter.tag`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that contain the specified tag.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .respondActivityTaskCompleted()
     * - .startWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember0(value, operator) {
        return this.if(`swf:tagList.member.0`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that contain the specified tag.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .respondActivityTaskCompleted()
     * - .startWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember1(value, operator) {
        return this.if(`swf:tagList.member.1`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that contain the specified tag.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .respondActivityTaskCompleted()
     * - .startWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember2(value, operator) {
        return this.if(`swf:tagList.member.2`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that contain the specified tag.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .respondActivityTaskCompleted()
     * - .startWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember3(value, operator) {
        return this.if(`swf:tagList.member.3`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that contain the specified tag.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .respondActivityTaskCompleted()
     * - .startWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember4(value, operator) {
        return this.if(`swf:tagList.member.4`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a tasklist with the specified name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .countPendingActivityTasks()
     * - .countPendingDecisionTasks()
     * - .pollForActivityTask()
     * - .pollForDecisionTask()
     * - .respondActivityTaskCompleted()
     * - .startWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTaskListName(value, operator) {
        return this.if(`swf:taskList.name`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a type filter with the specified name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .countClosedWorkflowExecutions()
     * - .countOpenWorkflowExecutions()
     * - .listClosedWorkflowExecutions()
     * - .listOpenWorkflowExecutions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTypeFilterName(value, operator) {
        return this.if(`swf:typeFilter.name`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a type filter with the specified version.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .countClosedWorkflowExecutions()
     * - .countOpenWorkflowExecutions()
     * - .listClosedWorkflowExecutions()
     * - .listOpenWorkflowExecutions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTypeFilterVersion(value, operator) {
        return this.if(`swf:typeFilter.version`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only activities or workflows with the specified version.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .registerActivityType()
     * - .registerWorkflowType()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersion(value, operator) {
        return this.if(`swf:version`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a workflow type of the specified name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .deprecateWorkflowType()
     * - .describeWorkflowType()
     * - .respondActivityTaskCompleted()
     * - .startWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWorkflowTypeName(value, operator) {
        return this.if(`swf:workflowType.name`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a workflow type of the specified version.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .deprecateWorkflowType()
     * - .describeWorkflowType()
     * - .respondActivityTaskCompleted()
     * - .startWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWorkflowTypeVersion(value, operator) {
        return this.if(`swf:workflowType.version`, value, operator || 'StringLike');
    }
}
exports.Swf = Swf;
//# sourceMappingURL=data:application/json;base64,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