"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Sns = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [sns](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsns.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Sns extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [sns](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsns.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'sns';
        this.actionList = {
            "AddPermission": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_AddPermission.html",
                "description": "Adds a statement to a topic's access control policy, granting access for the specified AWS accounts to the specified actions.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "CheckIfPhoneNumberIsOptedOut": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_CheckIfPhoneNumberIsOptedOut.html",
                "description": "Accepts a phone number and indicates whether the phone holder has opted out of receiving SMS messages from your account.",
                "accessLevel": "Read"
            },
            "ConfirmSubscription": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_ConfirmSubscription.html",
                "description": "Verifies an endpoint owner's intent to receive messages by validating the token sent to the endpoint by an earlier Subscribe action.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "CreatePlatformApplication": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_CreatePlatformApplication.html",
                "description": "Creates a platform application object for one of the supported push notification services, such as APNS and GCM, to which devices and mobile apps may register.",
                "accessLevel": "Write"
            },
            "CreatePlatformEndpoint": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_CreatePlatformEndpoint.html",
                "description": "Creates an endpoint for a device and mobile app on one of the supported push notification services, such as GCM and APNS.",
                "accessLevel": "Write"
            },
            "CreateTopic": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_CreateTopic.html",
                "description": "Creates a topic to which notifications can be published.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "DeleteEndpoint": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_DeleteEndpoint.html",
                "description": "Deletes the endpoint for a device and mobile app from Amazon SNS.",
                "accessLevel": "Write"
            },
            "DeletePlatformApplication": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_DeletePlatformApplication.html",
                "description": "Deletes a platform application object for one of the supported push notification services, such as APNS and GCM.",
                "accessLevel": "Write"
            },
            "DeleteTopic": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_DeleteTopic.html",
                "description": "Deletes a topic and all its subscriptions.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "GetEndpointAttributes": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_GetEndpointAttributes.html",
                "description": "Retrieves the endpoint attributes for a device on one of the supported push notification services, such as GCM and APNS.",
                "accessLevel": "Read"
            },
            "GetPlatformApplicationAttributes": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_GetPlatformApplicationAttributes.html",
                "description": "Retrieves the attributes of the platform application object for the supported push notification services, such as APNS and GCM.",
                "accessLevel": "Read"
            },
            "GetSMSAttributes": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_GetSMSAttributes.html",
                "description": "Returns the settings for sending SMS messages from your account.",
                "accessLevel": "Read"
            },
            "GetSubscriptionAttributes": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_GetSubscriptionAttributes.html",
                "description": "Returns all of the properties of a subscription.",
                "accessLevel": "Read"
            },
            "GetTopicAttributes": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_GetTopicAttributes.html",
                "description": "Returns all of the properties of a topic. Topic properties returned might differ based on the authorization of the user.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "ListEndpointsByPlatformApplication": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_ListEndpointsByPlatformApplication.html",
                "description": "Lists the endpoints and endpoint attributes for devices in a supported push notification service, such as GCM and APNS.",
                "accessLevel": "List"
            },
            "ListPhoneNumbersOptedOut": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_ListPhoneNumbersOptedOut.html",
                "description": "Returns a list of phone numbers that are opted out, meaning you cannot send SMS messages to them.",
                "accessLevel": "Read"
            },
            "ListPlatformApplications": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_ListPlatformApplications.html",
                "description": "Lists the platform application objects for the supported push notification services, such as APNS and GCM.",
                "accessLevel": "List"
            },
            "ListSubscriptions": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_ListSubscriptions.html",
                "description": "Returns a list of the requester's subscriptions.",
                "accessLevel": "List"
            },
            "ListSubscriptionsByTopic": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_ListSubscriptionsByTopic.html",
                "description": "Returns a list of the subscriptions to a specific topic.",
                "accessLevel": "List",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_ListTagsForResource.html",
                "description": "List all tags added to the specified Amazon SNS topic.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "topic": {
                        "required": false
                    }
                }
            },
            "ListTopics": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_ListTopics.html",
                "description": "Returns a list of the requester's topics. Each call returns a limited list of topics, up to 100.",
                "accessLevel": "List"
            },
            "OptInPhoneNumber": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_OptInPhoneNumber.html",
                "description": "Opts in a phone number that is currently opted out, which enables you to resume sending SMS messages to the number.",
                "accessLevel": "Write"
            },
            "Publish": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_Publish.html",
                "description": "Sends a message to all of a topic's subscribed endpoints.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "RemovePermission": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_RemovePermission.html",
                "description": "Removes a statement from a topic's access control policy.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "SetEndpointAttributes": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_SetEndpointAttributes.html",
                "description": "Sets the attributes for an endpoint for a device on one of the supported push notification services, such as GCM and APNS.",
                "accessLevel": "Write"
            },
            "SetPlatformApplicationAttributes": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_SetPlatformApplicationAttributes.html",
                "description": "Sets the attributes of the platform application object for the supported push notification services, such as APNS and GCM.",
                "accessLevel": "Write"
            },
            "SetSMSAttributes": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_SetSMSAttributes.html",
                "description": "Set the default settings for sending SMS messages and receiving daily SMS usage reports.",
                "accessLevel": "Write"
            },
            "SetSubscriptionAttributes": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_SetSubscriptionAttributes.html",
                "description": "Allows a subscription owner to set an attribute of the topic to a new value.",
                "accessLevel": "Write"
            },
            "SetTopicAttributes": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_SetTopicAttributes.html",
                "description": "Allows a topic owner to set an attribute of the topic to a new value.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "Subscribe": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_Subscribe.html",
                "description": "Prepares to subscribe an endpoint by sending the endpoint a confirmation message.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                },
                "conditions": [
                    "sns:Endpoint",
                    "sns:Protocol"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_TagResource.html",
                "description": "Add tags to the specified Amazon SNS topic.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "topic": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "Unsubscribe": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_Unsubscribe.html",
                "description": "Deletes a subscription. If the subscription requires authentication for deletion, only the owner of the subscription or the topic's owner can unsubscribe, and an AWS signature is required.",
                "accessLevel": "Write"
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_UntagResource.html",
                "description": "Remove tags from the specified Amazon SNS topic.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "topic": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            }
        };
        this.resourceTypes = {
            "topic": {
                "name": "topic",
                "url": "https://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html",
                "arn": "arn:${Partition}:sns:${Region}:${Account}:${TopicName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Adds a statement to a topic's access control policy, granting access for the specified AWS accounts to the specified actions.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_AddPermission.html
     */
    addPermission() {
        this.add('sns:AddPermission');
        return this;
    }
    /**
     * Accepts a phone number and indicates whether the phone holder has opted out of receiving SMS messages from your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_CheckIfPhoneNumberIsOptedOut.html
     */
    checkIfPhoneNumberIsOptedOut() {
        this.add('sns:CheckIfPhoneNumberIsOptedOut');
        return this;
    }
    /**
     * Verifies an endpoint owner's intent to receive messages by validating the token sent to the endpoint by an earlier Subscribe action.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ConfirmSubscription.html
     */
    confirmSubscription() {
        this.add('sns:ConfirmSubscription');
        return this;
    }
    /**
     * Creates a platform application object for one of the supported push notification services, such as APNS and GCM, to which devices and mobile apps may register.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_CreatePlatformApplication.html
     */
    createPlatformApplication() {
        this.add('sns:CreatePlatformApplication');
        return this;
    }
    /**
     * Creates an endpoint for a device and mobile app on one of the supported push notification services, such as GCM and APNS.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_CreatePlatformEndpoint.html
     */
    createPlatformEndpoint() {
        this.add('sns:CreatePlatformEndpoint');
        return this;
    }
    /**
     * Creates a topic to which notifications can be published.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_CreateTopic.html
     */
    createTopic() {
        this.add('sns:CreateTopic');
        return this;
    }
    /**
     * Deletes the endpoint for a device and mobile app from Amazon SNS.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_DeleteEndpoint.html
     */
    deleteEndpoint() {
        this.add('sns:DeleteEndpoint');
        return this;
    }
    /**
     * Deletes a platform application object for one of the supported push notification services, such as APNS and GCM.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_DeletePlatformApplication.html
     */
    deletePlatformApplication() {
        this.add('sns:DeletePlatformApplication');
        return this;
    }
    /**
     * Deletes a topic and all its subscriptions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_DeleteTopic.html
     */
    deleteTopic() {
        this.add('sns:DeleteTopic');
        return this;
    }
    /**
     * Retrieves the endpoint attributes for a device on one of the supported push notification services, such as GCM and APNS.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetEndpointAttributes.html
     */
    getEndpointAttributes() {
        this.add('sns:GetEndpointAttributes');
        return this;
    }
    /**
     * Retrieves the attributes of the platform application object for the supported push notification services, such as APNS and GCM.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetPlatformApplicationAttributes.html
     */
    getPlatformApplicationAttributes() {
        this.add('sns:GetPlatformApplicationAttributes');
        return this;
    }
    /**
     * Returns the settings for sending SMS messages from your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetSMSAttributes.html
     */
    getSMSAttributes() {
        this.add('sns:GetSMSAttributes');
        return this;
    }
    /**
     * Returns all of the properties of a subscription.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetSubscriptionAttributes.html
     */
    getSubscriptionAttributes() {
        this.add('sns:GetSubscriptionAttributes');
        return this;
    }
    /**
     * Returns all of the properties of a topic. Topic properties returned might differ based on the authorization of the user.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetTopicAttributes.html
     */
    getTopicAttributes() {
        this.add('sns:GetTopicAttributes');
        return this;
    }
    /**
     * Lists the endpoints and endpoint attributes for devices in a supported push notification service, such as GCM and APNS.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListEndpointsByPlatformApplication.html
     */
    listEndpointsByPlatformApplication() {
        this.add('sns:ListEndpointsByPlatformApplication');
        return this;
    }
    /**
     * Returns a list of phone numbers that are opted out, meaning you cannot send SMS messages to them.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListPhoneNumbersOptedOut.html
     */
    listPhoneNumbersOptedOut() {
        this.add('sns:ListPhoneNumbersOptedOut');
        return this;
    }
    /**
     * Lists the platform application objects for the supported push notification services, such as APNS and GCM.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListPlatformApplications.html
     */
    listPlatformApplications() {
        this.add('sns:ListPlatformApplications');
        return this;
    }
    /**
     * Returns a list of the requester's subscriptions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListSubscriptions.html
     */
    listSubscriptions() {
        this.add('sns:ListSubscriptions');
        return this;
    }
    /**
     * Returns a list of the subscriptions to a specific topic.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListSubscriptionsByTopic.html
     */
    listSubscriptionsByTopic() {
        this.add('sns:ListSubscriptionsByTopic');
        return this;
    }
    /**
     * List all tags added to the specified Amazon SNS topic.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('sns:ListTagsForResource');
        return this;
    }
    /**
     * Returns a list of the requester's topics. Each call returns a limited list of topics, up to 100.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListTopics.html
     */
    listTopics() {
        this.add('sns:ListTopics');
        return this;
    }
    /**
     * Opts in a phone number that is currently opted out, which enables you to resume sending SMS messages to the number.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_OptInPhoneNumber.html
     */
    optInPhoneNumber() {
        this.add('sns:OptInPhoneNumber');
        return this;
    }
    /**
     * Sends a message to all of a topic's subscribed endpoints.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_Publish.html
     */
    publish() {
        this.add('sns:Publish');
        return this;
    }
    /**
     * Removes a statement from a topic's access control policy.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_RemovePermission.html
     */
    removePermission() {
        this.add('sns:RemovePermission');
        return this;
    }
    /**
     * Sets the attributes for an endpoint for a device on one of the supported push notification services, such as GCM and APNS.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetEndpointAttributes.html
     */
    setEndpointAttributes() {
        this.add('sns:SetEndpointAttributes');
        return this;
    }
    /**
     * Sets the attributes of the platform application object for the supported push notification services, such as APNS and GCM.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetPlatformApplicationAttributes.html
     */
    setPlatformApplicationAttributes() {
        this.add('sns:SetPlatformApplicationAttributes');
        return this;
    }
    /**
     * Set the default settings for sending SMS messages and receiving daily SMS usage reports.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetSMSAttributes.html
     */
    setSMSAttributes() {
        this.add('sns:SetSMSAttributes');
        return this;
    }
    /**
     * Allows a subscription owner to set an attribute of the topic to a new value.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetSubscriptionAttributes.html
     */
    setSubscriptionAttributes() {
        this.add('sns:SetSubscriptionAttributes');
        return this;
    }
    /**
     * Allows a topic owner to set an attribute of the topic to a new value.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetTopicAttributes.html
     */
    setTopicAttributes() {
        this.add('sns:SetTopicAttributes');
        return this;
    }
    /**
     * Prepares to subscribe an endpoint by sending the endpoint a confirmation message.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEndpoint()
     * - .ifProtocol()
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_Subscribe.html
     */
    subscribe() {
        this.add('sns:Subscribe');
        return this;
    }
    /**
     * Add tags to the specified Amazon SNS topic.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_TagResource.html
     */
    tagResource() {
        this.add('sns:TagResource');
        return this;
    }
    /**
     * Deletes a subscription. If the subscription requires authentication for deletion, only the owner of the subscription or the topic's owner can unsubscribe, and an AWS signature is required.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_Unsubscribe.html
     */
    unsubscribe() {
        this.add('sns:Unsubscribe');
        return this;
    }
    /**
     * Remove tags from the specified Amazon SNS topic.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_UntagResource.html
     */
    untagResource() {
        this.add('sns:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type topic to the statement
     *
     * https://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html
     *
     * @param topicName - Identifier for the topicName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTopic(topicName, account, region, partition) {
        var arn = 'arn:${Partition}:sns:${Region}:${Account}:${TopicName}';
        arn = arn.replace('${TopicName}', topicName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Tags from request
     *
     * Applies to actions:
     * - .tagResource()
     * - .untagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Tag keys from request
     *
     * Applies to actions:
     * - .tagResource()
     * - .untagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
    /**
     * The URL, email address, or ARN from a Subscribe request or a previously confirmed subscription.
     *
     * https://docs.aws.amazon.com/sns/latest/dg/UsingIAMwithSNS.html#w2ab1c11c23c19
     *
     * Applies to actions:
     * - .subscribe()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEndpoint(value, operator) {
        return this.if(`sns:Endpoint`, value, operator || 'StringLike');
    }
    /**
     * The protocol value from a Subscribe request or a previously confirmed subscription.
     *
     * https://docs.aws.amazon.com/sns/latest/dg/UsingIAMwithSNS.html#w2ab1c11c23c19
     *
     * Applies to actions:
     * - .subscribe()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProtocol(value, operator) {
        return this.if(`sns:Protocol`, value, operator || 'StringLike');
    }
}
exports.Sns = Sns;
//# sourceMappingURL=data:application/json;base64,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