"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workmail = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [workmail](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonworkmail.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Workmail extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [workmail](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonworkmail.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'workmail';
        this.actionList = {
            "AddMembersToGroup": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html",
                "description": "Adds a list of members (users or groups) to a group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "AssociateDelegateToResource": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_AssociateDelegateToResource.html",
                "description": "Adds a member (user or group) to the resource's set of delegates.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "AssociateMemberToGroup": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_AssociateMemberToGroup.html",
                "description": "Adds a member (user or group) to the group's set.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "CreateAlias": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateAlias.html",
                "description": "Adds an alias to the set of a given member (user or group) of WorkMail.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "CreateGroup": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateGroup.html",
                "description": "Creates a group that can be used in WorkMail by calling the RegisterToWorkMail operation.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "CreateInboundMailFlowRule": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/create-email-rules.html",
                "description": "Create an inbound email flow rule which will apply to all email sent to an organization",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "CreateMailDomain": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/add_domain.html",
                "description": "Creates a mail domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "CreateMailUser": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/manage-users.html",
                "description": "Creates a user in the directory and the WorkMail storage but does not enable the user for mail.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "CreateOrganization": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/add_new_organization.html",
                "description": "Creates an organization, either using an existing directory or creates a new directory on-the-fly. Also creates and enables the complementary mail domain. Optionally creates KMS key",
                "accessLevel": "Write"
            },
            "CreateOutboundMailFlowRule": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/create-email-rules.html",
                "description": "Create an outbound email flow rule which will apply to all email sent from an organization",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "CreateResource": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateResource.html",
                "description": "Creates a new WorkMail resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "CreateSmtpGateway": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html",
                "description": "Register an SMTP device against a WorkMail organization",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "CreateUser": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateUser.html",
                "description": "Creates a user who can be used in WorkMail by calling the RegisterToWorkMail operation.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DeleteAccessControlRule": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteAccessControlRule.html",
                "description": "Deletes an access control rule for the specified WorkMail organization.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DeleteAlias": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteAlias.html",
                "description": "Remove one or more specified aliases from a set of aliases for a given user.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DeleteGroup": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteGroup.html",
                "description": "Deletes a group from WorkMail.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DeleteInboundMailFlowRule": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/remove-email-flow-rule.html",
                "description": "Remove an inbound email flow rule to no longer apply to emails sent to an organization",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DeleteMailDomain": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/remove_domain.html",
                "description": "Removes an unused mail domain from an organization",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DeleteMailboxPermissions": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteMailboxPermissions.html",
                "description": "Deletes permissions granted to a member (user or group).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DeleteMobileDevice": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/manage-devices.html#remove_mobile_device",
                "description": "Removes a mobile device from a user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DeleteOrganization": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/remove_organization.html",
                "description": "Removes an organization from an account, either removing the directory from directory services or leaving it available for re-use",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DeleteOutboundMailFlowRule": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/remove-email-flow-rule.html",
                "description": "Remove an outbound email flow rule to no longer apply to emails sent from an organization",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DeleteResource": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteResource.html",
                "description": "Deletes the specified resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DeleteRetentionPolicy": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteRetentionPolicy.html",
                "description": "Deletes the retention policy based on the supplied organization and policy identifiers.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DeleteSmtpGateway": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html",
                "description": "Remove an SMTP device from an organization",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DeleteUser": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteUser.html",
                "description": "Deletes a user from WorkMail and all subsequent systems. The action cannot be undone.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DeregisterFromWorkMail": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeregisterFromWorkMail.html",
                "description": "Mark a user, group, or resource as no longer used in WorkMail.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DescribeDirectories": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/add_new_organization.html",
                "description": "Shows a list of directories available for use in creating an organization",
                "accessLevel": "List"
            },
            "DescribeGroup": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeGroup.html",
                "description": "Returns the data available for the group.",
                "accessLevel": "List",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DescribeInboundMailFlowRule": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/email-flows.html#email-flows-rule-actions",
                "description": "Returns the details of an inbound mail flow rule configured for an organization",
                "accessLevel": "Read",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DescribeKmsKeys": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/add_new_organization.html",
                "description": "Shows a list of KMS Keys available for use in creating an organization",
                "accessLevel": "List"
            },
            "DescribeMailDomains": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/domains_overview.html",
                "description": "Shows the details of all mail domains associated with the organization",
                "accessLevel": "List",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DescribeMailGroups": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html",
                "description": "Shows the details of all groups associated with the organization",
                "accessLevel": "List",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DescribeMailUsers": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/users_overview.html",
                "description": "Shows the details of all users associated with the orgaization",
                "accessLevel": "List",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DescribeOrganization": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeOrganization.html",
                "description": "Provides more information regarding a given organization based on its identifier.",
                "accessLevel": "List",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DescribeOrganizations": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/organizations_overview.html",
                "description": "Shows a summary of all organizations associated with the account",
                "accessLevel": "List"
            },
            "DescribeOutboundMailFlowRule": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/email-flows.html#email-flows-rule-outbound",
                "description": "Returns the details of an outbound mail flow rule configured for an organization",
                "accessLevel": "Read",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DescribeResource": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeResource.html",
                "description": "Returns the data available for the resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DescribeSmtpGateway": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html",
                "description": "Returns the details of an SMTP device registered against an organization",
                "accessLevel": "Read",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DescribeUser": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeUser.html",
                "description": "Provides information regarding the user.",
                "accessLevel": "List",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DisableMailGroups": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/remove_group.html",
                "description": "Disable a mail group when it is not being used and, to allow it to be deleted",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DisableMailUsers": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/manage-mailboxes.html#delete_user_mailbox",
                "description": "Disable a user mailbox when it is no longer being used, and to allow it to be deleted",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DisassociateDelegateFromResource": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_DisassociateDelegateFromResource.html",
                "description": "Removes a member from the resource's set of delegates.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "DisassociateMemberFromGroup": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_DisassociateMemberFromGroup.html",
                "description": "Removes a member from a group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "EnableMailDomain": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/add_domain.html",
                "description": "Enable a mail domain in the organization",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "EnableMailGroups": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/enable_existing_group.html",
                "description": "Enable a mail group after it has been created to allow it to receive mail",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "EnableMailUsers": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/manage-users.html#enable_existing_user",
                "description": "Enable a user's mailbox after it has been created to allow it to receive mail",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "GetAccessControlEffect": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_GetAccessControlEffect.html",
                "description": "Gets the effects of an organization's access control rules as they apply to a specified IPv4 address, access protocol action, or user ID.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "GetDefaultRetentionPolicy": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_GetDefaultRetentionPolicy.html",
                "description": "Retrieves the retention policy associated at an organizational level.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "GetJournalingRules": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/journaling_overview.html",
                "description": "Returns journaling and fallback email addresses configured for email journaling",
                "accessLevel": "Read",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "GetMailDomainDetails": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/domains_overview.html",
                "description": "Get the details of the mail domain",
                "accessLevel": "Read",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "GetMailGroupDetails": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html",
                "description": "Get the details of the mail group",
                "accessLevel": "Read",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "GetMailUserDetails": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/users_overview.html",
                "description": "Get the details of the user's mailbox and account",
                "accessLevel": "Read",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "GetMailboxDetails": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_GetMailboxDetails.html",
                "description": "Returns the details of the user's mailbox.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "GetMobileDeviceDetails": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/manage-devices.html",
                "description": "Get the details of the mobile device",
                "accessLevel": "Read",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "GetMobileDevicesForUser": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/manage-devices.html",
                "description": "Get a list of the mobile devices associated with the user",
                "accessLevel": "Read",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "GetMobilePolicyDetails": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/edit_organization_mobile_policy.html",
                "description": "Get the details of the mobile device policy associated with the organization",
                "accessLevel": "Read",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "ListAccessControlRules": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListAccessControlRules.html",
                "description": "Lists the access control rules for the specified organization.",
                "accessLevel": "List",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "ListAliases": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListAliases.html",
                "description": "Creates a paginated call to list the aliases associated with a given entity.",
                "accessLevel": "List",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "ListGroupMembers": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListGroupMembers.html",
                "description": "Returns an overview of the members of a group. Users and groups can be members of a group.",
                "accessLevel": "List",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "ListGroups": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListGroups.html",
                "description": "Returns summaries of the organization's groups.",
                "accessLevel": "List",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "ListInboundMailFlowRules": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/email-flows.html#email-flows-rule-actions",
                "description": "Returns a list of inbound mail flow rules configured for an organization",
                "accessLevel": "List",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "ListMailboxPermissions": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListMailboxPermissions.html",
                "description": "Lists the mailbox permissions associated with a user, group, or resource mailbox.",
                "accessLevel": "List",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "ListMembersInMailGroup": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html",
                "description": "Get a list of all the members in a mail group",
                "accessLevel": "Read",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "ListOrganizations": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListOrganizations.html",
                "description": "Returns summaries of the customer's non-deleted organizations.",
                "accessLevel": "List"
            },
            "ListOutboundMailFlowRules": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/email-flows.html#email-flows-rule-outbound",
                "description": "Returns a list of outbound mail flow rules configured for an organization",
                "accessLevel": "List",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "ListResourceDelegates": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListResourceDelegates.html",
                "description": "Lists the delegates associated with a resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "ListResources": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListResources.html",
                "description": "Returns summaries of the organization's resources.",
                "accessLevel": "List",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "ListSmtpGateways": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html",
                "description": "Returns a list of SMTP devices registered against the organization",
                "accessLevel": "List",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to list the tags applied to an Amazon WorkMail organization resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "ListUsers": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListUsers.html",
                "description": "Returns summaries of the organization's users.",
                "accessLevel": "List",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "PutAccessControlRule": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_PutAccessControlRule.html",
                "description": "Adds a new access control rule for the specified organization. The rule allows or denies access to the organization for the specified IPv4 addresses, access protocol actions, and user IDs. Adding a new rule with the same name as an existing rule replaces the older rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "PutMailboxPermissions": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_PutMailboxPermissions.html",
                "description": "Sets permissions for a user, group, or resource. This replaces any pre-existing permissions.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "PutRetentionPolicy": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_PutRetentionPolicy.html",
                "description": "Adds or updates the retention policy for the specified organization.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "RegisterToWorkMail": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_RegisterToWorkMail.html",
                "description": "Registers an existing and disabled user, group, or resource for use by associating a mailbox and calendaring capabilities.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "RemoveMembersFromGroup": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html",
                "description": "Remove members from a mail group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "ResetPassword": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_ResetPassword.html",
                "description": "Allows the administrator to reset the password for a user.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "ResetUserPassword": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/manage-users.html#reset_user_password",
                "description": "Reset the password for a user's account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "SearchMembers": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html",
                "description": "Prefix search to find a specific user in a mail group",
                "accessLevel": "Read",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "SetAdmin": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/users_overview.html",
                "description": "Mark a user as being an administrator",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "SetDefaultMailDomain": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/default_domain.html",
                "description": "Set the default mail domain for the organization",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "SetJournalingRules": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/journaling_overview.html",
                "description": "Set journaling and fallback email addresses for email journaling",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "SetMailGroupDetails": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/add_new_group.html",
                "description": "Set the details of the mail group which has just been created",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "SetMailUserDetails": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/manage-users.html",
                "description": "Set the details for the user account which has just been created",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "SetMobilePolicyDetails": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/edit_organization_mobile_policy.html",
                "description": "Set the details of a mobile policy associated with the organization",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_TagResource.html",
                "description": "Grants permission to tag the specified Amazon WorkMail organization resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "TestInboundMailFlowRules": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/test-email-flow-rule.html",
                "description": "Test what inbound rules will apply to an email with a given sender and recipient",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "TestOutboundMailFlowRules": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/test-email-flow-rule.html",
                "description": "Test what outbound rules will apply to an email with a given sender and recipient",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_UntagResource.html",
                "description": "Grants permission to untag the specified Amazon WorkMail organization resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "UpdateInboundMailFlowRule": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/modify-email-flow-rule.html",
                "description": "Update the details of an inbound email flow rule which will apply to all email sent to an organization",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "UpdateMailboxQuota": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_UpdateMailboxQuota.html",
                "description": "Updates the maximum size (in MB) of the user's mailbox.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "UpdateOutboundMailFlowRule": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/modify-email-flow-rule.html",
                "description": "Update the details of an outbound email flow rule which will apply to all email sent from an organization",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "UpdatePrimaryEmailAddress": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_UpdatePrimaryEmailAddress.html",
                "description": "Updates the primary email for a user, group, or resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "UpdateResource": {
                "url": "https://docs.aws.amazon.com/workmail/latest/APIReference/API_UpdateResource.html",
                "description": "Updates data for the resource. To retrieve the latest information, it must be preceded by a DescribeResource call.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "UpdateSmtpGateway": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html",
                "description": "Update the details of an existing SMTP device registered against an organization",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            },
            "WipeMobileDevice": {
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/manage-devices.html#remote_wipe_device",
                "description": "Remotely wipe the mobile device associated with a user's account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organization": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "organization": {
                "name": "organization",
                "url": "https://docs.aws.amazon.com/workmail/latest/adminguide/organizations_overview.html",
                "arn": "arn:${Partition}:workmail:${Region}:${Account}:organization/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Adds a list of members (users or groups) to a group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html
     */
    addMembersToGroup() {
        this.add('workmail:AddMembersToGroup');
        return this;
    }
    /**
     * Adds a member (user or group) to the resource's set of delegates.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_AssociateDelegateToResource.html
     */
    associateDelegateToResource() {
        this.add('workmail:AssociateDelegateToResource');
        return this;
    }
    /**
     * Adds a member (user or group) to the group's set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_AssociateMemberToGroup.html
     */
    associateMemberToGroup() {
        this.add('workmail:AssociateMemberToGroup');
        return this;
    }
    /**
     * Adds an alias to the set of a given member (user or group) of WorkMail.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateAlias.html
     */
    createAlias() {
        this.add('workmail:CreateAlias');
        return this;
    }
    /**
     * Creates a group that can be used in WorkMail by calling the RegisterToWorkMail operation.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateGroup.html
     */
    createGroup() {
        this.add('workmail:CreateGroup');
        return this;
    }
    /**
     * Create an inbound email flow rule which will apply to all email sent to an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/create-email-rules.html
     */
    createInboundMailFlowRule() {
        this.add('workmail:CreateInboundMailFlowRule');
        return this;
    }
    /**
     * Creates a mail domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/add_domain.html
     */
    createMailDomain() {
        this.add('workmail:CreateMailDomain');
        return this;
    }
    /**
     * Creates a user in the directory and the WorkMail storage but does not enable the user for mail.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-users.html
     */
    createMailUser() {
        this.add('workmail:CreateMailUser');
        return this;
    }
    /**
     * Creates an organization, either using an existing directory or creates a new directory on-the-fly. Also creates and enables the complementary mail domain. Optionally creates KMS key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/add_new_organization.html
     */
    createOrganization() {
        this.add('workmail:CreateOrganization');
        return this;
    }
    /**
     * Create an outbound email flow rule which will apply to all email sent from an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/create-email-rules.html
     */
    createOutboundMailFlowRule() {
        this.add('workmail:CreateOutboundMailFlowRule');
        return this;
    }
    /**
     * Creates a new WorkMail resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateResource.html
     */
    createResource() {
        this.add('workmail:CreateResource');
        return this;
    }
    /**
     * Register an SMTP device against a WorkMail organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html
     */
    createSmtpGateway() {
        this.add('workmail:CreateSmtpGateway');
        return this;
    }
    /**
     * Creates a user who can be used in WorkMail by calling the RegisterToWorkMail operation.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateUser.html
     */
    createUser() {
        this.add('workmail:CreateUser');
        return this;
    }
    /**
     * Deletes an access control rule for the specified WorkMail organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteAccessControlRule.html
     */
    deleteAccessControlRule() {
        this.add('workmail:DeleteAccessControlRule');
        return this;
    }
    /**
     * Remove one or more specified aliases from a set of aliases for a given user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteAlias.html
     */
    deleteAlias() {
        this.add('workmail:DeleteAlias');
        return this;
    }
    /**
     * Deletes a group from WorkMail.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteGroup.html
     */
    deleteGroup() {
        this.add('workmail:DeleteGroup');
        return this;
    }
    /**
     * Remove an inbound email flow rule to no longer apply to emails sent to an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/remove-email-flow-rule.html
     */
    deleteInboundMailFlowRule() {
        this.add('workmail:DeleteInboundMailFlowRule');
        return this;
    }
    /**
     * Removes an unused mail domain from an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/remove_domain.html
     */
    deleteMailDomain() {
        this.add('workmail:DeleteMailDomain');
        return this;
    }
    /**
     * Deletes permissions granted to a member (user or group).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteMailboxPermissions.html
     */
    deleteMailboxPermissions() {
        this.add('workmail:DeleteMailboxPermissions');
        return this;
    }
    /**
     * Removes a mobile device from a user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-devices.html#remove_mobile_device
     */
    deleteMobileDevice() {
        this.add('workmail:DeleteMobileDevice');
        return this;
    }
    /**
     * Removes an organization from an account, either removing the directory from directory services or leaving it available for re-use
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/remove_organization.html
     */
    deleteOrganization() {
        this.add('workmail:DeleteOrganization');
        return this;
    }
    /**
     * Remove an outbound email flow rule to no longer apply to emails sent from an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/remove-email-flow-rule.html
     */
    deleteOutboundMailFlowRule() {
        this.add('workmail:DeleteOutboundMailFlowRule');
        return this;
    }
    /**
     * Deletes the specified resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteResource.html
     */
    deleteResource() {
        this.add('workmail:DeleteResource');
        return this;
    }
    /**
     * Deletes the retention policy based on the supplied organization and policy identifiers.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteRetentionPolicy.html
     */
    deleteRetentionPolicy() {
        this.add('workmail:DeleteRetentionPolicy');
        return this;
    }
    /**
     * Remove an SMTP device from an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html
     */
    deleteSmtpGateway() {
        this.add('workmail:DeleteSmtpGateway');
        return this;
    }
    /**
     * Deletes a user from WorkMail and all subsequent systems. The action cannot be undone.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteUser.html
     */
    deleteUser() {
        this.add('workmail:DeleteUser');
        return this;
    }
    /**
     * Mark a user, group, or resource as no longer used in WorkMail.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeregisterFromWorkMail.html
     */
    deregisterFromWorkMail() {
        this.add('workmail:DeregisterFromWorkMail');
        return this;
    }
    /**
     * Shows a list of directories available for use in creating an organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/add_new_organization.html
     */
    describeDirectories() {
        this.add('workmail:DescribeDirectories');
        return this;
    }
    /**
     * Returns the data available for the group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeGroup.html
     */
    describeGroup() {
        this.add('workmail:DescribeGroup');
        return this;
    }
    /**
     * Returns the details of an inbound mail flow rule configured for an organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/email-flows.html#email-flows-rule-actions
     */
    describeInboundMailFlowRule() {
        this.add('workmail:DescribeInboundMailFlowRule');
        return this;
    }
    /**
     * Shows a list of KMS Keys available for use in creating an organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/add_new_organization.html
     */
    describeKmsKeys() {
        this.add('workmail:DescribeKmsKeys');
        return this;
    }
    /**
     * Shows the details of all mail domains associated with the organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/domains_overview.html
     */
    describeMailDomains() {
        this.add('workmail:DescribeMailDomains');
        return this;
    }
    /**
     * Shows the details of all groups associated with the organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html
     */
    describeMailGroups() {
        this.add('workmail:DescribeMailGroups');
        return this;
    }
    /**
     * Shows the details of all users associated with the orgaization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/users_overview.html
     */
    describeMailUsers() {
        this.add('workmail:DescribeMailUsers');
        return this;
    }
    /**
     * Provides more information regarding a given organization based on its identifier.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeOrganization.html
     */
    describeOrganization() {
        this.add('workmail:DescribeOrganization');
        return this;
    }
    /**
     * Shows a summary of all organizations associated with the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/organizations_overview.html
     */
    describeOrganizations() {
        this.add('workmail:DescribeOrganizations');
        return this;
    }
    /**
     * Returns the details of an outbound mail flow rule configured for an organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/email-flows.html#email-flows-rule-outbound
     */
    describeOutboundMailFlowRule() {
        this.add('workmail:DescribeOutboundMailFlowRule');
        return this;
    }
    /**
     * Returns the data available for the resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeResource.html
     */
    describeResource() {
        this.add('workmail:DescribeResource');
        return this;
    }
    /**
     * Returns the details of an SMTP device registered against an organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html
     */
    describeSmtpGateway() {
        this.add('workmail:DescribeSmtpGateway');
        return this;
    }
    /**
     * Provides information regarding the user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeUser.html
     */
    describeUser() {
        this.add('workmail:DescribeUser');
        return this;
    }
    /**
     * Disable a mail group when it is not being used and, to allow it to be deleted
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/remove_group.html
     */
    disableMailGroups() {
        this.add('workmail:DisableMailGroups');
        return this;
    }
    /**
     * Disable a user mailbox when it is no longer being used, and to allow it to be deleted
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-mailboxes.html#delete_user_mailbox
     */
    disableMailUsers() {
        this.add('workmail:DisableMailUsers');
        return this;
    }
    /**
     * Removes a member from the resource's set of delegates.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DisassociateDelegateFromResource.html
     */
    disassociateDelegateFromResource() {
        this.add('workmail:DisassociateDelegateFromResource');
        return this;
    }
    /**
     * Removes a member from a group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DisassociateMemberFromGroup.html
     */
    disassociateMemberFromGroup() {
        this.add('workmail:DisassociateMemberFromGroup');
        return this;
    }
    /**
     * Enable a mail domain in the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/add_domain.html
     */
    enableMailDomain() {
        this.add('workmail:EnableMailDomain');
        return this;
    }
    /**
     * Enable a mail group after it has been created to allow it to receive mail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/enable_existing_group.html
     */
    enableMailGroups() {
        this.add('workmail:EnableMailGroups');
        return this;
    }
    /**
     * Enable a user's mailbox after it has been created to allow it to receive mail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-users.html#enable_existing_user
     */
    enableMailUsers() {
        this.add('workmail:EnableMailUsers');
        return this;
    }
    /**
     * Gets the effects of an organization's access control rules as they apply to a specified IPv4 address, access protocol action, or user ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_GetAccessControlEffect.html
     */
    getAccessControlEffect() {
        this.add('workmail:GetAccessControlEffect');
        return this;
    }
    /**
     * Retrieves the retention policy associated at an organizational level.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_GetDefaultRetentionPolicy.html
     */
    getDefaultRetentionPolicy() {
        this.add('workmail:GetDefaultRetentionPolicy');
        return this;
    }
    /**
     * Returns journaling and fallback email addresses configured for email journaling
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/journaling_overview.html
     */
    getJournalingRules() {
        this.add('workmail:GetJournalingRules');
        return this;
    }
    /**
     * Get the details of the mail domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/domains_overview.html
     */
    getMailDomainDetails() {
        this.add('workmail:GetMailDomainDetails');
        return this;
    }
    /**
     * Get the details of the mail group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html
     */
    getMailGroupDetails() {
        this.add('workmail:GetMailGroupDetails');
        return this;
    }
    /**
     * Get the details of the user's mailbox and account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/users_overview.html
     */
    getMailUserDetails() {
        this.add('workmail:GetMailUserDetails');
        return this;
    }
    /**
     * Returns the details of the user's mailbox.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_GetMailboxDetails.html
     */
    getMailboxDetails() {
        this.add('workmail:GetMailboxDetails');
        return this;
    }
    /**
     * Get the details of the mobile device
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-devices.html
     */
    getMobileDeviceDetails() {
        this.add('workmail:GetMobileDeviceDetails');
        return this;
    }
    /**
     * Get a list of the mobile devices associated with the user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-devices.html
     */
    getMobileDevicesForUser() {
        this.add('workmail:GetMobileDevicesForUser');
        return this;
    }
    /**
     * Get the details of the mobile device policy associated with the organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/edit_organization_mobile_policy.html
     */
    getMobilePolicyDetails() {
        this.add('workmail:GetMobilePolicyDetails');
        return this;
    }
    /**
     * Lists the access control rules for the specified organization.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListAccessControlRules.html
     */
    listAccessControlRules() {
        this.add('workmail:ListAccessControlRules');
        return this;
    }
    /**
     * Creates a paginated call to list the aliases associated with a given entity.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListAliases.html
     */
    listAliases() {
        this.add('workmail:ListAliases');
        return this;
    }
    /**
     * Returns an overview of the members of a group. Users and groups can be members of a group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListGroupMembers.html
     */
    listGroupMembers() {
        this.add('workmail:ListGroupMembers');
        return this;
    }
    /**
     * Returns summaries of the organization's groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListGroups.html
     */
    listGroups() {
        this.add('workmail:ListGroups');
        return this;
    }
    /**
     * Returns a list of inbound mail flow rules configured for an organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/email-flows.html#email-flows-rule-actions
     */
    listInboundMailFlowRules() {
        this.add('workmail:ListInboundMailFlowRules');
        return this;
    }
    /**
     * Lists the mailbox permissions associated with a user, group, or resource mailbox.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListMailboxPermissions.html
     */
    listMailboxPermissions() {
        this.add('workmail:ListMailboxPermissions');
        return this;
    }
    /**
     * Get a list of all the members in a mail group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html
     */
    listMembersInMailGroup() {
        this.add('workmail:ListMembersInMailGroup');
        return this;
    }
    /**
     * Returns summaries of the customer's non-deleted organizations.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListOrganizations.html
     */
    listOrganizations() {
        this.add('workmail:ListOrganizations');
        return this;
    }
    /**
     * Returns a list of outbound mail flow rules configured for an organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/email-flows.html#email-flows-rule-outbound
     */
    listOutboundMailFlowRules() {
        this.add('workmail:ListOutboundMailFlowRules');
        return this;
    }
    /**
     * Lists the delegates associated with a resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListResourceDelegates.html
     */
    listResourceDelegates() {
        this.add('workmail:ListResourceDelegates');
        return this;
    }
    /**
     * Returns summaries of the organization's resources.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListResources.html
     */
    listResources() {
        this.add('workmail:ListResources');
        return this;
    }
    /**
     * Returns a list of SMTP devices registered against the organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html
     */
    listSmtpGateways() {
        this.add('workmail:ListSmtpGateways');
        return this;
    }
    /**
     * Grants permission to list the tags applied to an Amazon WorkMail organization resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('workmail:ListTagsForResource');
        return this;
    }
    /**
     * Returns summaries of the organization's users.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListUsers.html
     */
    listUsers() {
        this.add('workmail:ListUsers');
        return this;
    }
    /**
     * Adds a new access control rule for the specified organization. The rule allows or denies access to the organization for the specified IPv4 addresses, access protocol actions, and user IDs. Adding a new rule with the same name as an existing rule replaces the older rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_PutAccessControlRule.html
     */
    putAccessControlRule() {
        this.add('workmail:PutAccessControlRule');
        return this;
    }
    /**
     * Sets permissions for a user, group, or resource. This replaces any pre-existing permissions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_PutMailboxPermissions.html
     */
    putMailboxPermissions() {
        this.add('workmail:PutMailboxPermissions');
        return this;
    }
    /**
     * Adds or updates the retention policy for the specified organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_PutRetentionPolicy.html
     */
    putRetentionPolicy() {
        this.add('workmail:PutRetentionPolicy');
        return this;
    }
    /**
     * Registers an existing and disabled user, group, or resource for use by associating a mailbox and calendaring capabilities.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_RegisterToWorkMail.html
     */
    registerToWorkMail() {
        this.add('workmail:RegisterToWorkMail');
        return this;
    }
    /**
     * Remove members from a mail group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html
     */
    removeMembersFromGroup() {
        this.add('workmail:RemoveMembersFromGroup');
        return this;
    }
    /**
     * Allows the administrator to reset the password for a user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ResetPassword.html
     */
    resetPassword() {
        this.add('workmail:ResetPassword');
        return this;
    }
    /**
     * Reset the password for a user's account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-users.html#reset_user_password
     */
    resetUserPassword() {
        this.add('workmail:ResetUserPassword');
        return this;
    }
    /**
     * Prefix search to find a specific user in a mail group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html
     */
    searchMembers() {
        this.add('workmail:SearchMembers');
        return this;
    }
    /**
     * Mark a user as being an administrator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/users_overview.html
     */
    setAdmin() {
        this.add('workmail:SetAdmin');
        return this;
    }
    /**
     * Set the default mail domain for the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/default_domain.html
     */
    setDefaultMailDomain() {
        this.add('workmail:SetDefaultMailDomain');
        return this;
    }
    /**
     * Set journaling and fallback email addresses for email journaling
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/journaling_overview.html
     */
    setJournalingRules() {
        this.add('workmail:SetJournalingRules');
        return this;
    }
    /**
     * Set the details of the mail group which has just been created
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/add_new_group.html
     */
    setMailGroupDetails() {
        this.add('workmail:SetMailGroupDetails');
        return this;
    }
    /**
     * Set the details for the user account which has just been created
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-users.html
     */
    setMailUserDetails() {
        this.add('workmail:SetMailUserDetails');
        return this;
    }
    /**
     * Set the details of a mobile policy associated with the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/edit_organization_mobile_policy.html
     */
    setMobilePolicyDetails() {
        this.add('workmail:SetMobilePolicyDetails');
        return this;
    }
    /**
     * Grants permission to tag the specified Amazon WorkMail organization resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('workmail:TagResource');
        return this;
    }
    /**
     * Test what inbound rules will apply to an email with a given sender and recipient
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/test-email-flow-rule.html
     */
    testInboundMailFlowRules() {
        this.add('workmail:TestInboundMailFlowRules');
        return this;
    }
    /**
     * Test what outbound rules will apply to an email with a given sender and recipient
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/test-email-flow-rule.html
     */
    testOutboundMailFlowRules() {
        this.add('workmail:TestOutboundMailFlowRules');
        return this;
    }
    /**
     * Grants permission to untag the specified Amazon WorkMail organization resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('workmail:UntagResource');
        return this;
    }
    /**
     * Update the details of an inbound email flow rule which will apply to all email sent to an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/modify-email-flow-rule.html
     */
    updateInboundMailFlowRule() {
        this.add('workmail:UpdateInboundMailFlowRule');
        return this;
    }
    /**
     * Updates the maximum size (in MB) of the user's mailbox.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UpdateMailboxQuota.html
     */
    updateMailboxQuota() {
        this.add('workmail:UpdateMailboxQuota');
        return this;
    }
    /**
     * Update the details of an outbound email flow rule which will apply to all email sent from an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/modify-email-flow-rule.html
     */
    updateOutboundMailFlowRule() {
        this.add('workmail:UpdateOutboundMailFlowRule');
        return this;
    }
    /**
     * Updates the primary email for a user, group, or resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UpdatePrimaryEmailAddress.html
     */
    updatePrimaryEmailAddress() {
        this.add('workmail:UpdatePrimaryEmailAddress');
        return this;
    }
    /**
     * Updates data for the resource. To retrieve the latest information, it must be preceded by a DescribeResource call.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UpdateResource.html
     */
    updateResource() {
        this.add('workmail:UpdateResource');
        return this;
    }
    /**
     * Update the details of an existing SMTP device registered against an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html
     */
    updateSmtpGateway() {
        this.add('workmail:UpdateSmtpGateway');
        return this;
    }
    /**
     * Remotely wipe the mobile device associated with a user's account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-devices.html#remote_wipe_device
     */
    wipeMobileDevice() {
        this.add('workmail:WipeMobileDevice');
        return this;
    }
    /**
     * Adds a resource of type organization to the statement
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/organizations_overview.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOrganization(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:workmail:${Region}:${Account}:organization/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - organization
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Workmail = Workmail;
//# sourceMappingURL=data:application/json;base64,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