"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Deeplens = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [deeplens](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdeeplens.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Deeplens extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [deeplens](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdeeplens.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'deeplens';
        this.actionList = {
            "AssociateServiceRoleToAccount": {
                "url": "",
                "description": "Associates the user's account with IAM roles controlling various permissions needed by AWS DeepLens for proper functionality.",
                "accessLevel": "Permissions management"
            },
            "BatchGetDevice": {
                "url": "",
                "description": "Retrieves a list of AWS DeepLens devices.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "device": {
                        "required": true
                    }
                }
            },
            "BatchGetModel": {
                "url": "",
                "description": "Retrieves a list of AWS DeepLens Models.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                }
            },
            "BatchGetProject": {
                "url": "",
                "description": "Retrieves a list of AWS DeepLens Projects.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "CreateDeviceCertificates": {
                "url": "",
                "description": "Creates a certificate package that is used to successfully authenticate and Register an AWS DeepLens device.",
                "accessLevel": "Write"
            },
            "CreateModel": {
                "url": "",
                "description": "Creates a new AWS DeepLens Model.",
                "accessLevel": "Write"
            },
            "CreateProject": {
                "url": "",
                "description": "Creates a new AWS DeepLens Project.",
                "accessLevel": "Write"
            },
            "DeleteModel": {
                "url": "",
                "description": "Deletes an AWS DeepLens Model.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                }
            },
            "DeleteProject": {
                "url": "",
                "description": "Deletes an AWS DeepLens Project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DeployProject": {
                "url": "",
                "description": "Deploys an AWS DeepLens project to a registered AWS DeepLens device.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "device": {
                        "required": true
                    },
                    "project": {
                        "required": true
                    }
                }
            },
            "DeregisterDevice": {
                "url": "",
                "description": "Begins a device de-registration workflow for a registered AWS DeepLens device.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "device": {
                        "required": true
                    }
                }
            },
            "GetAssociatedResources": {
                "url": "",
                "description": "Retrieves the account level resources associated with the user's account.",
                "accessLevel": "Read"
            },
            "GetDeploymentStatus": {
                "url": "",
                "description": "Retrieves the the deployment status of a particular AWS DeepLens device, along with any associated metadata.",
                "accessLevel": "Read"
            },
            "GetDevice": {
                "url": "",
                "description": "Retrieves information about an AWS DeepLens device.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "device": {
                        "required": true
                    }
                }
            },
            "GetModel": {
                "url": "",
                "description": "Retrieves an AWS DeepLens Model.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                }
            },
            "GetProject": {
                "url": "",
                "description": "Retrieves an AWS DeepLens Project.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "ImportProjectFromTemplate": {
                "url": "",
                "description": "Creates a new AWS DeepLens project from a sample project template.",
                "accessLevel": "Write"
            },
            "ListDeployments": {
                "url": "",
                "description": "Retrieves a list of AWS DeepLens Deployment identifiers.",
                "accessLevel": "List"
            },
            "ListDevices": {
                "url": "",
                "description": "Retrieves a list of AWS DeepLens device identifiers.",
                "accessLevel": "List"
            },
            "ListModels": {
                "url": "",
                "description": "Retrieves a list of AWS DeepLens Model identifiers.",
                "accessLevel": "List"
            },
            "ListProjects": {
                "url": "",
                "description": "Retrieves a list of AWS DeepLens Project identifiers.",
                "accessLevel": "List"
            },
            "RegisterDevice": {
                "url": "",
                "description": "Begins a device registration workflow for an AWS DeepLens device.",
                "accessLevel": "Write"
            },
            "RemoveProject": {
                "url": "",
                "description": "Removes a deployed AWS DeepLens project from an AWS DeepLens device.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "device": {
                        "required": true
                    }
                }
            },
            "UpdateProject": {
                "url": "",
                "description": "Updates an existing AWS DeepLens Project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "device": {
                "name": "device",
                "url": "",
                "arn": "arn:${Partition}:deeplens:${Region}:${Account}:device/${DeviceName}",
                "conditionKeys": []
            },
            "project": {
                "name": "project",
                "url": "",
                "arn": "arn:${Partition}:deeplens:${Region}:${Account}:project/${ProjectName}",
                "conditionKeys": []
            },
            "model": {
                "name": "model",
                "url": "",
                "arn": "arn:${Partition}:deeplens:${Region}:${Account}:model/${ModelName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Associates the user's account with IAM roles controlling various permissions needed by AWS DeepLens for proper functionality.
     *
     * Access Level: Permissions management
     */
    associateServiceRoleToAccount() {
        this.add('deeplens:AssociateServiceRoleToAccount');
        return this;
    }
    /**
     * Retrieves a list of AWS DeepLens devices.
     *
     * Access Level: Read
     */
    batchGetDevice() {
        this.add('deeplens:BatchGetDevice');
        return this;
    }
    /**
     * Retrieves a list of AWS DeepLens Models.
     *
     * Access Level: Read
     */
    batchGetModel() {
        this.add('deeplens:BatchGetModel');
        return this;
    }
    /**
     * Retrieves a list of AWS DeepLens Projects.
     *
     * Access Level: Read
     */
    batchGetProject() {
        this.add('deeplens:BatchGetProject');
        return this;
    }
    /**
     * Creates a certificate package that is used to successfully authenticate and Register an AWS DeepLens device.
     *
     * Access Level: Write
     */
    createDeviceCertificates() {
        this.add('deeplens:CreateDeviceCertificates');
        return this;
    }
    /**
     * Creates a new AWS DeepLens Model.
     *
     * Access Level: Write
     */
    createModel() {
        this.add('deeplens:CreateModel');
        return this;
    }
    /**
     * Creates a new AWS DeepLens Project.
     *
     * Access Level: Write
     */
    createProject() {
        this.add('deeplens:CreateProject');
        return this;
    }
    /**
     * Deletes an AWS DeepLens Model.
     *
     * Access Level: Write
     */
    deleteModel() {
        this.add('deeplens:DeleteModel');
        return this;
    }
    /**
     * Deletes an AWS DeepLens Project.
     *
     * Access Level: Write
     */
    deleteProject() {
        this.add('deeplens:DeleteProject');
        return this;
    }
    /**
     * Deploys an AWS DeepLens project to a registered AWS DeepLens device.
     *
     * Access Level: Write
     */
    deployProject() {
        this.add('deeplens:DeployProject');
        return this;
    }
    /**
     * Begins a device de-registration workflow for a registered AWS DeepLens device.
     *
     * Access Level: Write
     */
    deregisterDevice() {
        this.add('deeplens:DeregisterDevice');
        return this;
    }
    /**
     * Retrieves the account level resources associated with the user's account.
     *
     * Access Level: Read
     */
    getAssociatedResources() {
        this.add('deeplens:GetAssociatedResources');
        return this;
    }
    /**
     * Retrieves the the deployment status of a particular AWS DeepLens device, along with any associated metadata.
     *
     * Access Level: Read
     */
    getDeploymentStatus() {
        this.add('deeplens:GetDeploymentStatus');
        return this;
    }
    /**
     * Retrieves information about an AWS DeepLens device.
     *
     * Access Level: Read
     */
    getDevice() {
        this.add('deeplens:GetDevice');
        return this;
    }
    /**
     * Retrieves an AWS DeepLens Model.
     *
     * Access Level: Read
     */
    getModel() {
        this.add('deeplens:GetModel');
        return this;
    }
    /**
     * Retrieves an AWS DeepLens Project.
     *
     * Access Level: Read
     */
    getProject() {
        this.add('deeplens:GetProject');
        return this;
    }
    /**
     * Creates a new AWS DeepLens project from a sample project template.
     *
     * Access Level: Write
     */
    importProjectFromTemplate() {
        this.add('deeplens:ImportProjectFromTemplate');
        return this;
    }
    /**
     * Retrieves a list of AWS DeepLens Deployment identifiers.
     *
     * Access Level: List
     */
    listDeployments() {
        this.add('deeplens:ListDeployments');
        return this;
    }
    /**
     * Retrieves a list of AWS DeepLens device identifiers.
     *
     * Access Level: List
     */
    listDevices() {
        this.add('deeplens:ListDevices');
        return this;
    }
    /**
     * Retrieves a list of AWS DeepLens Model identifiers.
     *
     * Access Level: List
     */
    listModels() {
        this.add('deeplens:ListModels');
        return this;
    }
    /**
     * Retrieves a list of AWS DeepLens Project identifiers.
     *
     * Access Level: List
     */
    listProjects() {
        this.add('deeplens:ListProjects');
        return this;
    }
    /**
     * Begins a device registration workflow for an AWS DeepLens device.
     *
     * Access Level: Write
     */
    registerDevice() {
        this.add('deeplens:RegisterDevice');
        return this;
    }
    /**
     * Removes a deployed AWS DeepLens project from an AWS DeepLens device.
     *
     * Access Level: Write
     */
    removeProject() {
        this.add('deeplens:RemoveProject');
        return this;
    }
    /**
     * Updates an existing AWS DeepLens Project.
     *
     * Access Level: Write
     */
    updateProject() {
        this.add('deeplens:UpdateProject');
        return this;
    }
    /**
     * Adds a resource of type device to the statement
     *
     * @param deviceName - Identifier for the deviceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDevice(deviceName, account, region, partition) {
        var arn = 'arn:${Partition}:deeplens:${Region}:${Account}:device/${DeviceName}';
        arn = arn.replace('${DeviceName}', deviceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type project to the statement
     *
     * @param projectName - Identifier for the projectName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProject(projectName, account, region, partition) {
        var arn = 'arn:${Partition}:deeplens:${Region}:${Account}:project/${ProjectName}';
        arn = arn.replace('${ProjectName}', projectName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type model to the statement
     *
     * @param modelName - Identifier for the modelName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onModel(modelName, account, region, partition) {
        var arn = 'arn:${Partition}:deeplens:${Region}:${Account}:model/${ModelName}';
        arn = arn.replace('${ModelName}', modelName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Deeplens = Deeplens;
//# sourceMappingURL=data:application/json;base64,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