"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Fms = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [fms](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsfirewallmanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Fms extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [fms](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsfirewallmanager.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'fms';
        this.actionList = {
            "AssociateAdminAccount": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_AssociateAdminAccount.html",
                "description": "Sets the AWS Firewall Manager administrator account and enables the service in all organization accounts",
                "accessLevel": "Write"
            },
            "DeleteAppsList": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DeleteAppsList.html",
                "description": "Grants permission to permanently deletes an AWS Firewall Manager applications list.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "applications-list": {
                        "required": true
                    }
                }
            },
            "DeleteNotificationChannel": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DeleteNotificationChannel.html",
                "description": "Deletes an AWS Firewall Manager association with the IAM role and the Amazon Simple Notification Service (SNS) topic that is used to notify the FM administrator about major FM events and errors across the organization.",
                "accessLevel": "Write"
            },
            "DeletePolicy": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DeletePolicy.html",
                "description": "Permanently deletes an AWS Firewall Manager policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "DeleteProtocolsList": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DeleteProtocolsList.html",
                "description": "Grants permission to permanently deletes an AWS Firewall Manager protocols list.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "protocols-list": {
                        "required": true
                    }
                }
            },
            "DisassociateAdminAccount": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DisassociateAdminAccount.html",
                "description": "Disassociates the account that has been set as the AWS Firewall Manager administrator account and and disables the service in all organization accounts",
                "accessLevel": "Write"
            },
            "GetAdminAccount": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetAdminAccount.html",
                "description": "Returns the AWS Organizations master account that is associated with AWS Firewall Manager as the AWS Firewall Manager administrator.",
                "accessLevel": "Read"
            },
            "GetAppsList": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetAppsList.html",
                "description": "Grants permission to return information about the specified AWS Firewall Manager applications list.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "applications-list": {
                        "required": true
                    }
                }
            },
            "GetComplianceDetail": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetComplianceDetail.html",
                "description": "Returns detailed compliance information about the specified member account. Details include resources that are in and out of compliance with the specified policy.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "GetNotificationChannel": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetNotificationChannel.html",
                "description": "Returns information about the Amazon Simple Notification Service (SNS) topic that is used to record AWS Firewall Manager SNS logs.",
                "accessLevel": "Read"
            },
            "GetPolicy": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetPolicy.html",
                "description": "Returns information about the specified AWS Firewall Manager policy.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "GetProtectionStatus": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetProtectionStatus.html",
                "description": "Returns policy-level attack summary information in the event of a potential DDoS attack.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "GetProtocolsList": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetProtocolsList.html",
                "description": "Grants permission to return information about the specified AWS Firewall Manager protocols list.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "protocols-list": {
                        "required": true
                    }
                }
            },
            "ListAppsLists": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListAppsLists.html",
                "description": "Grants permission to return an array of AppsListDataSummary objects.",
                "accessLevel": "List"
            },
            "ListComplianceStatus": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListComplianceStatus.html",
                "description": "Returns an array of PolicyComplianceStatus objects in the response. Use PolicyComplianceStatus to get a summary of which member accounts are protected by the specified policy.",
                "accessLevel": "List",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "ListMemberAccounts": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListMemberAccounts.html",
                "description": "Returns an array of member account ids if the caller is FMS admin account.",
                "accessLevel": "List"
            },
            "ListPolicies": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListPolicies.html",
                "description": "Returns an array of PolicySummary objects in the response.",
                "accessLevel": "List"
            },
            "ListProtocolsLists": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListProtocolsLists.html",
                "description": "Grants permission to return an array of ProtocolsListDataSummary objects.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListTagsForResource.html",
                "description": "Lists the Tags for a given resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "PutAppsList": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutAppsList.html",
                "description": "Grants permission to create an AWS Firewall Manager applications list.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "applications-list": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "PutNotificationChannel": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutNotificationChannel.html",
                "description": "Designates the IAM role and Amazon Simple Notification Service (SNS) topic that AWS Firewall Manager (FM) could use to notify the FM administrator about major FM events and errors across the organization.",
                "accessLevel": "Write"
            },
            "PutPolicy": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutPolicy.html",
                "description": "Creates an AWS Firewall Manager policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "PutProtocolsList": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutProtocolsList.html",
                "description": "Grants permission to creates an AWS Firewall Manager protocols list.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "protocols-list": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_TagResource.html",
                "description": "Adds a Tag to a given resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_UntagResource.html",
                "description": "Removes a Tag from a given resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            }
        };
        this.resourceTypes = {
            "policy": {
                "name": "policy",
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_Policy.html",
                "arn": "arn:${Partition}:fms:${Region}:${Account}:policy/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "applications-list": {
                "name": "applications-list",
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_AppsListData.html",
                "arn": "arn:${Partition}:fms:${Region}:${Account}:applications-list/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "protocols-list": {
                "name": "protocols-list",
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ProtocolsListData.html",
                "arn": "arn:${Partition}:fms:${Region}:${Account}:protocols-list/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Sets the AWS Firewall Manager administrator account and enables the service in all organization accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_AssociateAdminAccount.html
     */
    associateAdminAccount() {
        this.add('fms:AssociateAdminAccount');
        return this;
    }
    /**
     * Grants permission to permanently deletes an AWS Firewall Manager applications list.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DeleteAppsList.html
     */
    deleteAppsList() {
        this.add('fms:DeleteAppsList');
        return this;
    }
    /**
     * Deletes an AWS Firewall Manager association with the IAM role and the Amazon Simple Notification Service (SNS) topic that is used to notify the FM administrator about major FM events and errors across the organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DeleteNotificationChannel.html
     */
    deleteNotificationChannel() {
        this.add('fms:DeleteNotificationChannel');
        return this;
    }
    /**
     * Permanently deletes an AWS Firewall Manager policy.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DeletePolicy.html
     */
    deletePolicy() {
        this.add('fms:DeletePolicy');
        return this;
    }
    /**
     * Grants permission to permanently deletes an AWS Firewall Manager protocols list.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DeleteProtocolsList.html
     */
    deleteProtocolsList() {
        this.add('fms:DeleteProtocolsList');
        return this;
    }
    /**
     * Disassociates the account that has been set as the AWS Firewall Manager administrator account and and disables the service in all organization accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DisassociateAdminAccount.html
     */
    disassociateAdminAccount() {
        this.add('fms:DisassociateAdminAccount');
        return this;
    }
    /**
     * Returns the AWS Organizations master account that is associated with AWS Firewall Manager as the AWS Firewall Manager administrator.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetAdminAccount.html
     */
    getAdminAccount() {
        this.add('fms:GetAdminAccount');
        return this;
    }
    /**
     * Grants permission to return information about the specified AWS Firewall Manager applications list.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetAppsList.html
     */
    getAppsList() {
        this.add('fms:GetAppsList');
        return this;
    }
    /**
     * Returns detailed compliance information about the specified member account. Details include resources that are in and out of compliance with the specified policy.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetComplianceDetail.html
     */
    getComplianceDetail() {
        this.add('fms:GetComplianceDetail');
        return this;
    }
    /**
     * Returns information about the Amazon Simple Notification Service (SNS) topic that is used to record AWS Firewall Manager SNS logs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetNotificationChannel.html
     */
    getNotificationChannel() {
        this.add('fms:GetNotificationChannel');
        return this;
    }
    /**
     * Returns information about the specified AWS Firewall Manager policy.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetPolicy.html
     */
    getPolicy() {
        this.add('fms:GetPolicy');
        return this;
    }
    /**
     * Returns policy-level attack summary information in the event of a potential DDoS attack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetProtectionStatus.html
     */
    getProtectionStatus() {
        this.add('fms:GetProtectionStatus');
        return this;
    }
    /**
     * Grants permission to return information about the specified AWS Firewall Manager protocols list.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetProtocolsList.html
     */
    getProtocolsList() {
        this.add('fms:GetProtocolsList');
        return this;
    }
    /**
     * Grants permission to return an array of AppsListDataSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListAppsLists.html
     */
    listAppsLists() {
        this.add('fms:ListAppsLists');
        return this;
    }
    /**
     * Returns an array of PolicyComplianceStatus objects in the response. Use PolicyComplianceStatus to get a summary of which member accounts are protected by the specified policy.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListComplianceStatus.html
     */
    listComplianceStatus() {
        this.add('fms:ListComplianceStatus');
        return this;
    }
    /**
     * Returns an array of member account ids if the caller is FMS admin account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListMemberAccounts.html
     */
    listMemberAccounts() {
        this.add('fms:ListMemberAccounts');
        return this;
    }
    /**
     * Returns an array of PolicySummary objects in the response.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListPolicies.html
     */
    listPolicies() {
        this.add('fms:ListPolicies');
        return this;
    }
    /**
     * Grants permission to return an array of ProtocolsListDataSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListProtocolsLists.html
     */
    listProtocolsLists() {
        this.add('fms:ListProtocolsLists');
        return this;
    }
    /**
     * Lists the Tags for a given resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('fms:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to create an AWS Firewall Manager applications list.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutAppsList.html
     */
    putAppsList() {
        this.add('fms:PutAppsList');
        return this;
    }
    /**
     * Designates the IAM role and Amazon Simple Notification Service (SNS) topic that AWS Firewall Manager (FM) could use to notify the FM administrator about major FM events and errors across the organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutNotificationChannel.html
     */
    putNotificationChannel() {
        this.add('fms:PutNotificationChannel');
        return this;
    }
    /**
     * Creates an AWS Firewall Manager policy.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutPolicy.html
     */
    putPolicy() {
        this.add('fms:PutPolicy');
        return this;
    }
    /**
     * Grants permission to creates an AWS Firewall Manager protocols list.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutProtocolsList.html
     */
    putProtocolsList() {
        this.add('fms:PutProtocolsList');
        return this;
    }
    /**
     * Adds a Tag to a given resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('fms:TagResource');
        return this;
    }
    /**
     * Removes a Tag from a given resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('fms:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type policy to the statement
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_Policy.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPolicy(id, account, region, partition) {
        var arn = 'arn:${Partition}:fms:${Region}:${Account}:policy/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type applications-list to the statement
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_AppsListData.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApplicationsList(id, account, region, partition) {
        var arn = 'arn:${Partition}:fms:${Region}:${Account}:applications-list/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type protocols-list to the statement
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ProtocolsListData.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProtocolsList(id, account, region, partition) {
        var arn = 'arn:${Partition}:fms:${Region}:${Account}:protocols-list/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the allowed set of values for each of the tags
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .putAppsList()
     * - .putPolicy()
     * - .putProtocolsList()
     * - .tagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on tag-value assoicated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .deletePolicy()
     *
     * Applies to resource types:
     * - policy
     * - applications-list
     * - protocols-list
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of mandatory tags in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .putAppsList()
     * - .putPolicy()
     * - .putProtocolsList()
     * - .tagResource()
     * - .untagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Fms = Fms;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzZmlyZXdhbGxtYW5hZ2VyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiYXdzZmlyZXdhbGxtYW5hZ2VyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHNDQUFvRTtBQUVwRTs7OztHQUlHO0FBQ0gsTUFBYSxHQUFJLFNBQVEsd0JBQWU7SUE2UHRDOzs7O09BSUc7SUFDSCxZQUFhLEdBQVk7UUFDdkIsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBbFFOLGtCQUFhLEdBQUcsS0FBSyxDQUFDO1FBQ25CLGVBQVUsR0FBWTtZQUM5Qix1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLHdGQUF3RjtnQkFDL0YsYUFBYSxFQUFFLDBHQUEwRztnQkFDekgsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLGlGQUFpRjtnQkFDeEYsYUFBYSxFQUFFLHFGQUFxRjtnQkFDcEcsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixtQkFBbUIsRUFBRTt3QkFDbkIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCwyQkFBMkIsRUFBRTtnQkFDM0IsS0FBSyxFQUFFLDRGQUE0RjtnQkFDbkcsYUFBYSxFQUFFLDROQUE0TjtnQkFDM08sYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLCtFQUErRTtnQkFDdEYsYUFBYSxFQUFFLHFEQUFxRDtnQkFDcEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwyQkFBMkI7aUJBQzVCO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLGtGQUFrRjtnQkFDakcsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixnQkFBZ0IsRUFBRTt3QkFDaEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCwwQkFBMEIsRUFBRTtnQkFDMUIsS0FBSyxFQUFFLDJGQUEyRjtnQkFDbEcsYUFBYSxFQUFFLHlKQUF5SjtnQkFDeEssYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxpQkFBaUIsRUFBRTtnQkFDakIsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLHNJQUFzSTtnQkFDckosYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLDhFQUE4RTtnQkFDckYsYUFBYSxFQUFFLHFHQUFxRztnQkFDcEgsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixtQkFBbUIsRUFBRTt3QkFDbkIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLG9LQUFvSztnQkFDbkwsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCx3QkFBd0IsRUFBRTtnQkFDeEIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLG9JQUFvSTtnQkFDbkosYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxXQUFXLEVBQUU7Z0JBQ1gsS0FBSyxFQUFFLDRFQUE0RTtnQkFDbkYsYUFBYSxFQUFFLHNFQUFzRTtnQkFDckYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLDBGQUEwRjtnQkFDekcsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLG1GQUFtRjtnQkFDMUYsYUFBYSxFQUFFLGtHQUFrRztnQkFDakgsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixnQkFBZ0IsRUFBRTt3QkFDaEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLGdGQUFnRjtnQkFDdkYsYUFBYSxFQUFFLHNFQUFzRTtnQkFDckYsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxzQkFBc0IsRUFBRTtnQkFDdEIsS0FBSyxFQUFFLHVGQUF1RjtnQkFDOUYsYUFBYSxFQUFFLGlMQUFpTDtnQkFDaE0sYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLDRFQUE0RTtnQkFDM0YsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLCtFQUErRTtnQkFDdEYsYUFBYSxFQUFFLDREQUE0RDtnQkFDM0UsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLDJFQUEyRTtnQkFDMUYsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLHNDQUFzQztnQkFDckQsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLDhFQUE4RTtnQkFDckYsYUFBYSxFQUFFLHdFQUF3RTtnQkFDdkYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixtQkFBbUIsRUFBRTt3QkFDbkIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELHdCQUF3QixFQUFFO2dCQUN4QixLQUFLLEVBQUUseUZBQXlGO2dCQUNoRyxhQUFhLEVBQUUsOE1BQThNO2dCQUM3TixhQUFhLEVBQUUsT0FBTzthQUN2QjtZQUNELFdBQVcsRUFBRTtnQkFDWCxLQUFLLEVBQUUsNEVBQTRFO2dCQUNuRixhQUFhLEVBQUUseUNBQXlDO2dCQUN4RCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDBCQUEwQjtvQkFDMUIsYUFBYTtpQkFDZDthQUNGO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLEtBQUssRUFBRSxtRkFBbUY7Z0JBQzFGLGFBQWEsRUFBRSxzRUFBc0U7Z0JBQ3JGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsZ0JBQWdCLEVBQUU7d0JBQ2hCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osMEJBQTBCO29CQUMxQixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLDhFQUE4RTtnQkFDckYsYUFBYSxFQUFFLGlDQUFpQztnQkFDaEQsYUFBYSxFQUFFLFNBQVM7Z0JBQ3hCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELGVBQWUsRUFBRTtnQkFDZixLQUFLLEVBQUUsZ0ZBQWdGO2dCQUN2RixhQUFhLEVBQUUsc0NBQXNDO2dCQUNyRCxhQUFhLEVBQUUsU0FBUztnQkFDeEIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7aUJBQ2Q7YUFDRjtTQUNGLENBQUM7UUFDUSxrQkFBYSxHQUFrQjtZQUN2QyxRQUFRLEVBQUU7Z0JBQ1IsTUFBTSxFQUFFLFFBQVE7Z0JBQ2hCLEtBQUssRUFBRSx5RUFBeUU7Z0JBQ2hGLEtBQUssRUFBRSx3REFBd0Q7Z0JBQy9ELGVBQWUsRUFBRTtvQkFDZiwyQkFBMkI7aUJBQzVCO2FBQ0Y7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsTUFBTSxFQUFFLG1CQUFtQjtnQkFDM0IsS0FBSyxFQUFFLCtFQUErRTtnQkFDdEYsS0FBSyxFQUFFLG1FQUFtRTtnQkFDMUUsZUFBZSxFQUFFO29CQUNmLDJCQUEyQjtpQkFDNUI7YUFDRjtZQUNELGdCQUFnQixFQUFFO2dCQUNoQixNQUFNLEVBQUUsZ0JBQWdCO2dCQUN4QixLQUFLLEVBQUUsb0ZBQW9GO2dCQUMzRixLQUFLLEVBQUUsZ0VBQWdFO2dCQUN2RSxlQUFlLEVBQUU7b0JBQ2YsMkJBQTJCO2lCQUM1QjthQUNGO1NBQ0YsQ0FBQztJQVNGLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBQy9CLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsR0FBRyxDQUFDLCtCQUErQixDQUFDLENBQUM7UUFDMUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksWUFBWTtRQUNqQixJQUFJLENBQUMsR0FBRyxDQUFDLGtCQUFrQixDQUFDLENBQUM7UUFDN0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBQzVCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLElBQUksQ0FBQyxHQUFHLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxTQUFTO1FBQ2QsSUFBSSxDQUFDLEdBQUcsQ0FBQyxlQUFlLENBQUMsQ0FBQztRQUMxQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFDOUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQzdCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsR0FBRyxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFDNUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLElBQUksQ0FBQyxHQUFHLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksU0FBUztRQUNkLElBQUksQ0FBQyxHQUFHLENBQUMsZUFBZSxDQUFDLENBQUM7UUFDMUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBQzVCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBQzlCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFFBQVEsQ0FBQyxFQUFVLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDL0UsSUFBSSxHQUFHLEdBQUcsd0RBQXdELENBQUM7UUFDbkUsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQy9CLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksa0JBQWtCLENBQUMsRUFBVSxFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3pGLElBQUksR0FBRyxHQUFHLG1FQUFtRSxDQUFDO1FBQzlFLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQztRQUMvQixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGVBQWUsQ0FBQyxFQUFVLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDdEYsSUFBSSxHQUFHLEdBQUcsZ0VBQWdFLENBQUM7UUFDM0UsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQy9CLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxlQUFlLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBaUI7UUFDaEYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFtQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2hGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7OztPQWdCRztJQUNJLGdCQUFnQixDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQWlCO1FBQ2pGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBb0IsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxZQUFZLENBQUMsS0FBd0IsRUFBRSxRQUFpQjtRQUM3RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakUsQ0FBQztDQUNGO0FBcHNCRCxrQkFvc0JDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWN0aW9ucywgUG9saWN5U3RhdGVtZW50LCBSZXNvdXJjZVR5cGVzIH0gZnJvbSBcIi4uL3NoYXJlZFwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbZm1zXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hd3NmaXJld2FsbG1hbmFnZXIuaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBGbXMgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdmbXMnO1xuICBwcm90ZWN0ZWQgYWN0aW9uTGlzdDogQWN0aW9ucyA9IHtcbiAgICBcIkFzc29jaWF0ZUFkbWluQWNjb3VudFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mbXMvMjAxOC0wMS0wMS9BUElSZWZlcmVuY2UvQVBJX0Fzc29jaWF0ZUFkbWluQWNjb3VudC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiU2V0cyB0aGUgQVdTIEZpcmV3YWxsIE1hbmFnZXIgYWRtaW5pc3RyYXRvciBhY2NvdW50IGFuZCBlbmFibGVzIHRoZSBzZXJ2aWNlIGluIGFsbCBvcmdhbml6YXRpb24gYWNjb3VudHNcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiXG4gICAgfSxcbiAgICBcIkRlbGV0ZUFwcHNMaXN0XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Ztcy8yMDE4LTAxLTAxL0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlQXBwc0xpc3QuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHBlcm1hbmVudGx5IGRlbGV0ZXMgYW4gQVdTIEZpcmV3YWxsIE1hbmFnZXIgYXBwbGljYXRpb25zIGxpc3QuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiYXBwbGljYXRpb25zLWxpc3RcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZU5vdGlmaWNhdGlvbkNoYW5uZWxcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZm1zLzIwMTgtMDEtMDEvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVOb3RpZmljYXRpb25DaGFubmVsLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZWxldGVzIGFuIEFXUyBGaXJld2FsbCBNYW5hZ2VyIGFzc29jaWF0aW9uIHdpdGggdGhlIElBTSByb2xlIGFuZCB0aGUgQW1hem9uIFNpbXBsZSBOb3RpZmljYXRpb24gU2VydmljZSAoU05TKSB0b3BpYyB0aGF0IGlzIHVzZWQgdG8gbm90aWZ5IHRoZSBGTSBhZG1pbmlzdHJhdG9yIGFib3V0IG1ham9yIEZNIGV2ZW50cyBhbmQgZXJyb3JzIGFjcm9zcyB0aGUgb3JnYW5pemF0aW9uLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiRGVsZXRlUG9saWN5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Ztcy8yMDE4LTAxLTAxL0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlUG9saWN5Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJQZXJtYW5lbnRseSBkZWxldGVzIGFuIEFXUyBGaXJld2FsbCBNYW5hZ2VyIHBvbGljeS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJwb2xpY3lcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiRGVsZXRlUHJvdG9jb2xzTGlzdFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mbXMvMjAxOC0wMS0wMS9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVByb3RvY29sc0xpc3QuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHBlcm1hbmVudGx5IGRlbGV0ZXMgYW4gQVdTIEZpcmV3YWxsIE1hbmFnZXIgcHJvdG9jb2xzIGxpc3QuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicHJvdG9jb2xzLWxpc3RcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRpc2Fzc29jaWF0ZUFkbWluQWNjb3VudFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mbXMvMjAxOC0wMS0wMS9BUElSZWZlcmVuY2UvQVBJX0Rpc2Fzc29jaWF0ZUFkbWluQWNjb3VudC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGlzYXNzb2NpYXRlcyB0aGUgYWNjb3VudCB0aGF0IGhhcyBiZWVuIHNldCBhcyB0aGUgQVdTIEZpcmV3YWxsIE1hbmFnZXIgYWRtaW5pc3RyYXRvciBhY2NvdW50IGFuZCBhbmQgZGlzYWJsZXMgdGhlIHNlcnZpY2UgaW4gYWxsIG9yZ2FuaXphdGlvbiBhY2NvdW50c1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiR2V0QWRtaW5BY2NvdW50XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Ztcy8yMDE4LTAxLTAxL0FQSVJlZmVyZW5jZS9BUElfR2V0QWRtaW5BY2NvdW50Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIHRoZSBBV1MgT3JnYW5pemF0aW9ucyBtYXN0ZXIgYWNjb3VudCB0aGF0IGlzIGFzc29jaWF0ZWQgd2l0aCBBV1MgRmlyZXdhbGwgTWFuYWdlciBhcyB0aGUgQVdTIEZpcmV3YWxsIE1hbmFnZXIgYWRtaW5pc3RyYXRvci5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCJcbiAgICB9LFxuICAgIFwiR2V0QXBwc0xpc3RcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZm1zLzIwMTgtMDEtMDEvQVBJUmVmZXJlbmNlL0FQSV9HZXRBcHBzTGlzdC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIGluZm9ybWF0aW9uIGFib3V0IHRoZSBzcGVjaWZpZWQgQVdTIEZpcmV3YWxsIE1hbmFnZXIgYXBwbGljYXRpb25zIGxpc3QuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJhcHBsaWNhdGlvbnMtbGlzdFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0Q29tcGxpYW5jZURldGFpbFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mbXMvMjAxOC0wMS0wMS9BUElSZWZlcmVuY2UvQVBJX0dldENvbXBsaWFuY2VEZXRhaWwuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgZGV0YWlsZWQgY29tcGxpYW5jZSBpbmZvcm1hdGlvbiBhYm91dCB0aGUgc3BlY2lmaWVkIG1lbWJlciBhY2NvdW50LiBEZXRhaWxzIGluY2x1ZGUgcmVzb3VyY2VzIHRoYXQgYXJlIGluIGFuZCBvdXQgb2YgY29tcGxpYW5jZSB3aXRoIHRoZSBzcGVjaWZpZWQgcG9saWN5LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicG9saWN5XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXROb3RpZmljYXRpb25DaGFubmVsXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Ztcy8yMDE4LTAxLTAxL0FQSVJlZmVyZW5jZS9BUElfR2V0Tm90aWZpY2F0aW9uQ2hhbm5lbC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBpbmZvcm1hdGlvbiBhYm91dCB0aGUgQW1hem9uIFNpbXBsZSBOb3RpZmljYXRpb24gU2VydmljZSAoU05TKSB0b3BpYyB0aGF0IGlzIHVzZWQgdG8gcmVjb3JkIEFXUyBGaXJld2FsbCBNYW5hZ2VyIFNOUyBsb2dzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIlxuICAgIH0sXG4gICAgXCJHZXRQb2xpY3lcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZm1zLzIwMTgtMDEtMDEvQVBJUmVmZXJlbmNlL0FQSV9HZXRQb2xpY3kuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgaW5mb3JtYXRpb24gYWJvdXQgdGhlIHNwZWNpZmllZCBBV1MgRmlyZXdhbGwgTWFuYWdlciBwb2xpY3kuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJwb2xpY3lcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldFByb3RlY3Rpb25TdGF0dXNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZm1zLzIwMTgtMDEtMDEvQVBJUmVmZXJlbmNlL0FQSV9HZXRQcm90ZWN0aW9uU3RhdHVzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIHBvbGljeS1sZXZlbCBhdHRhY2sgc3VtbWFyeSBpbmZvcm1hdGlvbiBpbiB0aGUgZXZlbnQgb2YgYSBwb3RlbnRpYWwgRERvUyBhdHRhY2suXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJwb2xpY3lcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldFByb3RvY29sc0xpc3RcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZm1zLzIwMTgtMDEtMDEvQVBJUmVmZXJlbmNlL0FQSV9HZXRQcm90b2NvbHNMaXN0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gaW5mb3JtYXRpb24gYWJvdXQgdGhlIHNwZWNpZmllZCBBV1MgRmlyZXdhbGwgTWFuYWdlciBwcm90b2NvbHMgbGlzdC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInByb3RvY29scy1saXN0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJMaXN0QXBwc0xpc3RzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Ztcy8yMDE4LTAxLTAxL0FQSVJlZmVyZW5jZS9BUElfTGlzdEFwcHNMaXN0cy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIGFuIGFycmF5IG9mIEFwcHNMaXN0RGF0YVN1bW1hcnkgb2JqZWN0cy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiTGlzdENvbXBsaWFuY2VTdGF0dXNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZm1zLzIwMTgtMDEtMDEvQVBJUmVmZXJlbmNlL0FQSV9MaXN0Q29tcGxpYW5jZVN0YXR1cy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBhbiBhcnJheSBvZiBQb2xpY3lDb21wbGlhbmNlU3RhdHVzIG9iamVjdHMgaW4gdGhlIHJlc3BvbnNlLiBVc2UgUG9saWN5Q29tcGxpYW5jZVN0YXR1cyB0byBnZXQgYSBzdW1tYXJ5IG9mIHdoaWNoIG1lbWJlciBhY2NvdW50cyBhcmUgcHJvdGVjdGVkIGJ5IHRoZSBzcGVjaWZpZWQgcG9saWN5LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicG9saWN5XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJMaXN0TWVtYmVyQWNjb3VudHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZm1zLzIwMTgtMDEtMDEvQVBJUmVmZXJlbmNlL0FQSV9MaXN0TWVtYmVyQWNjb3VudHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgYW4gYXJyYXkgb2YgbWVtYmVyIGFjY291bnQgaWRzIGlmIHRoZSBjYWxsZXIgaXMgRk1TIGFkbWluIGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkxpc3RQb2xpY2llc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mbXMvMjAxOC0wMS0wMS9BUElSZWZlcmVuY2UvQVBJX0xpc3RQb2xpY2llcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBhbiBhcnJheSBvZiBQb2xpY3lTdW1tYXJ5IG9iamVjdHMgaW4gdGhlIHJlc3BvbnNlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJMaXN0UHJvdG9jb2xzTGlzdHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZm1zLzIwMTgtMDEtMDEvQVBJUmVmZXJlbmNlL0FQSV9MaXN0UHJvdG9jb2xzTGlzdHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiBhbiBhcnJheSBvZiBQcm90b2NvbHNMaXN0RGF0YVN1bW1hcnkgb2JqZWN0cy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiTGlzdFRhZ3NGb3JSZXNvdXJjZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mbXMvMjAxOC0wMS0wMS9BUElSZWZlcmVuY2UvQVBJX0xpc3RUYWdzRm9yUmVzb3VyY2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkxpc3RzIHRoZSBUYWdzIGZvciBhIGdpdmVuIHJlc291cmNlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicG9saWN5XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJQdXRBcHBzTGlzdFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mbXMvMjAxOC0wMS0wMS9BUElSZWZlcmVuY2UvQVBJX1B1dEFwcHNMaXN0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW4gQVdTIEZpcmV3YWxsIE1hbmFnZXIgYXBwbGljYXRpb25zIGxpc3QuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiYXBwbGljYXRpb25zLWxpc3RcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlB1dE5vdGlmaWNhdGlvbkNoYW5uZWxcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZm1zLzIwMTgtMDEtMDEvQVBJUmVmZXJlbmNlL0FQSV9QdXROb3RpZmljYXRpb25DaGFubmVsLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZXNpZ25hdGVzIHRoZSBJQU0gcm9sZSBhbmQgQW1hem9uIFNpbXBsZSBOb3RpZmljYXRpb24gU2VydmljZSAoU05TKSB0b3BpYyB0aGF0IEFXUyBGaXJld2FsbCBNYW5hZ2VyIChGTSkgY291bGQgdXNlIHRvIG5vdGlmeSB0aGUgRk0gYWRtaW5pc3RyYXRvciBhYm91dCBtYWpvciBGTSBldmVudHMgYW5kIGVycm9ycyBhY3Jvc3MgdGhlIG9yZ2FuaXphdGlvbi5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiXG4gICAgfSxcbiAgICBcIlB1dFBvbGljeVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mbXMvMjAxOC0wMS0wMS9BUElSZWZlcmVuY2UvQVBJX1B1dFBvbGljeS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhbiBBV1MgRmlyZXdhbGwgTWFuYWdlciBwb2xpY3kuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicG9saWN5XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiYXdzOlJlcXVlc3RUYWcvJHtUYWdLZXl9XCIsXG4gICAgICAgIFwiYXdzOlRhZ0tleXNcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJQdXRQcm90b2NvbHNMaXN0XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Ztcy8yMDE4LTAxLTAxL0FQSVJlZmVyZW5jZS9BUElfUHV0UHJvdG9jb2xzTGlzdC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlcyBhbiBBV1MgRmlyZXdhbGwgTWFuYWdlciBwcm90b2NvbHMgbGlzdC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJwcm90b2NvbHMtbGlzdFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiVGFnUmVzb3VyY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZm1zLzIwMTgtMDEtMDEvQVBJUmVmZXJlbmNlL0FQSV9UYWdSZXNvdXJjZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQWRkcyBhIFRhZyB0byBhIGdpdmVuIHJlc291cmNlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlRhZ2dpbmdcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicG9saWN5XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiYXdzOlJlcXVlc3RUYWcvJHtUYWdLZXl9XCIsXG4gICAgICAgIFwiYXdzOlRhZ0tleXNcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJVbnRhZ1Jlc291cmNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Ztcy8yMDE4LTAxLTAxL0FQSVJlZmVyZW5jZS9BUElfVW50YWdSZXNvdXJjZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmVtb3ZlcyBhIFRhZyBmcm9tIGEgZ2l2ZW4gcmVzb3VyY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiVGFnZ2luZ1wiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJwb2xpY3lcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfVxuICB9O1xuICBwcm90ZWN0ZWQgcmVzb3VyY2VUeXBlczogUmVzb3VyY2VUeXBlcyA9IHtcbiAgICBcInBvbGljeVwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJwb2xpY3lcIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Ztcy8yMDE4LTAxLTAxL0FQSVJlZmVyZW5jZS9BUElfUG9saWN5Lmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpmbXM6JHtSZWdpb259OiR7QWNjb3VudH06cG9saWN5LyR7SWR9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW1xuICAgICAgICBcImF3czpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJhcHBsaWNhdGlvbnMtbGlzdFwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJhcHBsaWNhdGlvbnMtbGlzdFwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZm1zLzIwMTgtMDEtMDEvQVBJUmVmZXJlbmNlL0FQSV9BcHBzTGlzdERhdGEuaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmZtczoke1JlZ2lvbn06JHtBY2NvdW50fTphcHBsaWNhdGlvbnMtbGlzdC8ke0lkfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtcbiAgICAgICAgXCJhd3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwicHJvdG9jb2xzLWxpc3RcIjoge1xuICAgICAgXCJuYW1lXCI6IFwicHJvdG9jb2xzLWxpc3RcIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Ztcy8yMDE4LTAxLTAxL0FQSVJlZmVyZW5jZS9BUElfUHJvdG9jb2xzTGlzdERhdGEuaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmZtczoke1JlZ2lvbn06JHtBY2NvdW50fTpwcm90b2NvbHMtbGlzdC8ke0lkfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtcbiAgICAgICAgXCJhd3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XCJcbiAgICAgIF1cbiAgICB9XG4gIH07XG5cbiAgLyoqXG4gICAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbZm1zXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hd3NmaXJld2FsbG1hbmFnZXIuaHRtbCkuXG4gICAqXG4gICAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICAgKi9cbiAgY29uc3RydWN0b3IgKHNpZD86IHN0cmluZykge1xuICAgIHN1cGVyKHNpZCk7XG4gIH1cblxuICAvKipcbiAgICogU2V0cyB0aGUgQVdTIEZpcmV3YWxsIE1hbmFnZXIgYWRtaW5pc3RyYXRvciBhY2NvdW50IGFuZCBlbmFibGVzIHRoZSBzZXJ2aWNlIGluIGFsbCBvcmdhbml6YXRpb24gYWNjb3VudHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZm1zLzIwMTgtMDEtMDEvQVBJUmVmZXJlbmNlL0FQSV9Bc3NvY2lhdGVBZG1pbkFjY291bnQuaHRtbFxuICAgKi9cbiAgcHVibGljIGFzc29jaWF0ZUFkbWluQWNjb3VudCgpIHtcbiAgICB0aGlzLmFkZCgnZm1zOkFzc29jaWF0ZUFkbWluQWNjb3VudCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHBlcm1hbmVudGx5IGRlbGV0ZXMgYW4gQVdTIEZpcmV3YWxsIE1hbmFnZXIgYXBwbGljYXRpb25zIGxpc3QuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Ztcy8yMDE4LTAxLTAxL0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlQXBwc0xpc3QuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlbGV0ZUFwcHNMaXN0KCkge1xuICAgIHRoaXMuYWRkKCdmbXM6RGVsZXRlQXBwc0xpc3QnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGFuIEFXUyBGaXJld2FsbCBNYW5hZ2VyIGFzc29jaWF0aW9uIHdpdGggdGhlIElBTSByb2xlIGFuZCB0aGUgQW1hem9uIFNpbXBsZSBOb3RpZmljYXRpb24gU2VydmljZSAoU05TKSB0b3BpYyB0aGF0IGlzIHVzZWQgdG8gbm90aWZ5IHRoZSBGTSBhZG1pbmlzdHJhdG9yIGFib3V0IG1ham9yIEZNIGV2ZW50cyBhbmQgZXJyb3JzIGFjcm9zcyB0aGUgb3JnYW5pemF0aW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mbXMvMjAxOC0wMS0wMS9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZU5vdGlmaWNhdGlvbkNoYW5uZWwuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlbGV0ZU5vdGlmaWNhdGlvbkNoYW5uZWwoKSB7XG4gICAgdGhpcy5hZGQoJ2ZtczpEZWxldGVOb3RpZmljYXRpb25DaGFubmVsJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUGVybWFuZW50bHkgZGVsZXRlcyBhbiBBV1MgRmlyZXdhbGwgTWFuYWdlciBwb2xpY3kuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mbXMvMjAxOC0wMS0wMS9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVBvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVsZXRlUG9saWN5KCkge1xuICAgIHRoaXMuYWRkKCdmbXM6RGVsZXRlUG9saWN5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcGVybWFuZW50bHkgZGVsZXRlcyBhbiBBV1MgRmlyZXdhbGwgTWFuYWdlciBwcm90b2NvbHMgbGlzdC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZm1zLzIwMTgtMDEtMDEvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVQcm90b2NvbHNMaXN0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVQcm90b2NvbHNMaXN0KCkge1xuICAgIHRoaXMuYWRkKCdmbXM6RGVsZXRlUHJvdG9jb2xzTGlzdCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERpc2Fzc29jaWF0ZXMgdGhlIGFjY291bnQgdGhhdCBoYXMgYmVlbiBzZXQgYXMgdGhlIEFXUyBGaXJld2FsbCBNYW5hZ2VyIGFkbWluaXN0cmF0b3IgYWNjb3VudCBhbmQgYW5kIGRpc2FibGVzIHRoZSBzZXJ2aWNlIGluIGFsbCBvcmdhbml6YXRpb24gYWNjb3VudHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZm1zLzIwMTgtMDEtMDEvQVBJUmVmZXJlbmNlL0FQSV9EaXNhc3NvY2lhdGVBZG1pbkFjY291bnQuaHRtbFxuICAgKi9cbiAgcHVibGljIGRpc2Fzc29jaWF0ZUFkbWluQWNjb3VudCgpIHtcbiAgICB0aGlzLmFkZCgnZm1zOkRpc2Fzc29jaWF0ZUFkbWluQWNjb3VudCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIEFXUyBPcmdhbml6YXRpb25zIG1hc3RlciBhY2NvdW50IHRoYXQgaXMgYXNzb2NpYXRlZCB3aXRoIEFXUyBGaXJld2FsbCBNYW5hZ2VyIGFzIHRoZSBBV1MgRmlyZXdhbGwgTWFuYWdlciBhZG1pbmlzdHJhdG9yLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Ztcy8yMDE4LTAxLTAxL0FQSVJlZmVyZW5jZS9BUElfR2V0QWRtaW5BY2NvdW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRBZG1pbkFjY291bnQoKSB7XG4gICAgdGhpcy5hZGQoJ2ZtczpHZXRBZG1pbkFjY291bnQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gaW5mb3JtYXRpb24gYWJvdXQgdGhlIHNwZWNpZmllZCBBV1MgRmlyZXdhbGwgTWFuYWdlciBhcHBsaWNhdGlvbnMgbGlzdC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mbXMvMjAxOC0wMS0wMS9BUElSZWZlcmVuY2UvQVBJX0dldEFwcHNMaXN0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRBcHBzTGlzdCgpIHtcbiAgICB0aGlzLmFkZCgnZm1zOkdldEFwcHNMaXN0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBkZXRhaWxlZCBjb21wbGlhbmNlIGluZm9ybWF0aW9uIGFib3V0IHRoZSBzcGVjaWZpZWQgbWVtYmVyIGFjY291bnQuIERldGFpbHMgaW5jbHVkZSByZXNvdXJjZXMgdGhhdCBhcmUgaW4gYW5kIG91dCBvZiBjb21wbGlhbmNlIHdpdGggdGhlIHNwZWNpZmllZCBwb2xpY3kuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZm1zLzIwMTgtMDEtMDEvQVBJUmVmZXJlbmNlL0FQSV9HZXRDb21wbGlhbmNlRGV0YWlsLmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRDb21wbGlhbmNlRGV0YWlsKCkge1xuICAgIHRoaXMuYWRkKCdmbXM6R2V0Q29tcGxpYW5jZURldGFpbCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgaW5mb3JtYXRpb24gYWJvdXQgdGhlIEFtYXpvbiBTaW1wbGUgTm90aWZpY2F0aW9uIFNlcnZpY2UgKFNOUykgdG9waWMgdGhhdCBpcyB1c2VkIHRvIHJlY29yZCBBV1MgRmlyZXdhbGwgTWFuYWdlciBTTlMgbG9ncy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mbXMvMjAxOC0wMS0wMS9BUElSZWZlcmVuY2UvQVBJX0dldE5vdGlmaWNhdGlvbkNoYW5uZWwuaHRtbFxuICAgKi9cbiAgcHVibGljIGdldE5vdGlmaWNhdGlvbkNoYW5uZWwoKSB7XG4gICAgdGhpcy5hZGQoJ2ZtczpHZXROb3RpZmljYXRpb25DaGFubmVsJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBpbmZvcm1hdGlvbiBhYm91dCB0aGUgc3BlY2lmaWVkIEFXUyBGaXJld2FsbCBNYW5hZ2VyIHBvbGljeS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mbXMvMjAxOC0wMS0wMS9BUElSZWZlcmVuY2UvQVBJX0dldFBvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgZ2V0UG9saWN5KCkge1xuICAgIHRoaXMuYWRkKCdmbXM6R2V0UG9saWN5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBwb2xpY3ktbGV2ZWwgYXR0YWNrIHN1bW1hcnkgaW5mb3JtYXRpb24gaW4gdGhlIGV2ZW50IG9mIGEgcG90ZW50aWFsIEREb1MgYXR0YWNrLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Ztcy8yMDE4LTAxLTAxL0FQSVJlZmVyZW5jZS9BUElfR2V0UHJvdGVjdGlvblN0YXR1cy5odG1sXG4gICAqL1xuICBwdWJsaWMgZ2V0UHJvdGVjdGlvblN0YXR1cygpIHtcbiAgICB0aGlzLmFkZCgnZm1zOkdldFByb3RlY3Rpb25TdGF0dXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gaW5mb3JtYXRpb24gYWJvdXQgdGhlIHNwZWNpZmllZCBBV1MgRmlyZXdhbGwgTWFuYWdlciBwcm90b2NvbHMgbGlzdC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mbXMvMjAxOC0wMS0wMS9BUElSZWZlcmVuY2UvQVBJX0dldFByb3RvY29sc0xpc3QuaHRtbFxuICAgKi9cbiAgcHVibGljIGdldFByb3RvY29sc0xpc3QoKSB7XG4gICAgdGhpcy5hZGQoJ2ZtczpHZXRQcm90b2NvbHNMaXN0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIGFuIGFycmF5IG9mIEFwcHNMaXN0RGF0YVN1bW1hcnkgb2JqZWN0cy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mbXMvMjAxOC0wMS0wMS9BUElSZWZlcmVuY2UvQVBJX0xpc3RBcHBzTGlzdHMuaHRtbFxuICAgKi9cbiAgcHVibGljIGxpc3RBcHBzTGlzdHMoKSB7XG4gICAgdGhpcy5hZGQoJ2ZtczpMaXN0QXBwc0xpc3RzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhbiBhcnJheSBvZiBQb2xpY3lDb21wbGlhbmNlU3RhdHVzIG9iamVjdHMgaW4gdGhlIHJlc3BvbnNlLiBVc2UgUG9saWN5Q29tcGxpYW5jZVN0YXR1cyB0byBnZXQgYSBzdW1tYXJ5IG9mIHdoaWNoIG1lbWJlciBhY2NvdW50cyBhcmUgcHJvdGVjdGVkIGJ5IHRoZSBzcGVjaWZpZWQgcG9saWN5LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Ztcy8yMDE4LTAxLTAxL0FQSVJlZmVyZW5jZS9BUElfTGlzdENvbXBsaWFuY2VTdGF0dXMuaHRtbFxuICAgKi9cbiAgcHVibGljIGxpc3RDb21wbGlhbmNlU3RhdHVzKCkge1xuICAgIHRoaXMuYWRkKCdmbXM6TGlzdENvbXBsaWFuY2VTdGF0dXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFuIGFycmF5IG9mIG1lbWJlciBhY2NvdW50IGlkcyBpZiB0aGUgY2FsbGVyIGlzIEZNUyBhZG1pbiBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Ztcy8yMDE4LTAxLTAxL0FQSVJlZmVyZW5jZS9BUElfTGlzdE1lbWJlckFjY291bnRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0TWVtYmVyQWNjb3VudHMoKSB7XG4gICAgdGhpcy5hZGQoJ2ZtczpMaXN0TWVtYmVyQWNjb3VudHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFuIGFycmF5IG9mIFBvbGljeVN1bW1hcnkgb2JqZWN0cyBpbiB0aGUgcmVzcG9uc2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZm1zLzIwMTgtMDEtMDEvQVBJUmVmZXJlbmNlL0FQSV9MaXN0UG9saWNpZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIGxpc3RQb2xpY2llcygpIHtcbiAgICB0aGlzLmFkZCgnZm1zOkxpc3RQb2xpY2llcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiBhbiBhcnJheSBvZiBQcm90b2NvbHNMaXN0RGF0YVN1bW1hcnkgb2JqZWN0cy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mbXMvMjAxOC0wMS0wMS9BUElSZWZlcmVuY2UvQVBJX0xpc3RQcm90b2NvbHNMaXN0cy5odG1sXG4gICAqL1xuICBwdWJsaWMgbGlzdFByb3RvY29sc0xpc3RzKCkge1xuICAgIHRoaXMuYWRkKCdmbXM6TGlzdFByb3RvY29sc0xpc3RzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdHMgdGhlIFRhZ3MgZm9yIGEgZ2l2ZW4gcmVzb3VyY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZm1zLzIwMTgtMDEtMDEvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VGFnc0ZvclJlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0VGFnc0ZvclJlc291cmNlKCkge1xuICAgIHRoaXMuYWRkKCdmbXM6TGlzdFRhZ3NGb3JSZXNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhbiBBV1MgRmlyZXdhbGwgTWFuYWdlciBhcHBsaWNhdGlvbnMgbGlzdC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZm1zLzIwMTgtMDEtMDEvQVBJUmVmZXJlbmNlL0FQSV9QdXRBcHBzTGlzdC5odG1sXG4gICAqL1xuICBwdWJsaWMgcHV0QXBwc0xpc3QoKSB7XG4gICAgdGhpcy5hZGQoJ2ZtczpQdXRBcHBzTGlzdCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2lnbmF0ZXMgdGhlIElBTSByb2xlIGFuZCBBbWF6b24gU2ltcGxlIE5vdGlmaWNhdGlvbiBTZXJ2aWNlIChTTlMpIHRvcGljIHRoYXQgQVdTIEZpcmV3YWxsIE1hbmFnZXIgKEZNKSBjb3VsZCB1c2UgdG8gbm90aWZ5IHRoZSBGTSBhZG1pbmlzdHJhdG9yIGFib3V0IG1ham9yIEZNIGV2ZW50cyBhbmQgZXJyb3JzIGFjcm9zcyB0aGUgb3JnYW5pemF0aW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mbXMvMjAxOC0wMS0wMS9BUElSZWZlcmVuY2UvQVBJX1B1dE5vdGlmaWNhdGlvbkNoYW5uZWwuaHRtbFxuICAgKi9cbiAgcHVibGljIHB1dE5vdGlmaWNhdGlvbkNoYW5uZWwoKSB7XG4gICAgdGhpcy5hZGQoJ2ZtczpQdXROb3RpZmljYXRpb25DaGFubmVsJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhbiBBV1MgRmlyZXdhbGwgTWFuYWdlciBwb2xpY3kuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Ztcy8yMDE4LTAxLTAxL0FQSVJlZmVyZW5jZS9BUElfUHV0UG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBwdXRQb2xpY3koKSB7XG4gICAgdGhpcy5hZGQoJ2ZtczpQdXRQb2xpY3knKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGVzIGFuIEFXUyBGaXJld2FsbCBNYW5hZ2VyIHByb3RvY29scyBsaXN0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mbXMvMjAxOC0wMS0wMS9BUElSZWZlcmVuY2UvQVBJX1B1dFByb3RvY29sc0xpc3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHB1dFByb3RvY29sc0xpc3QoKSB7XG4gICAgdGhpcy5hZGQoJ2ZtczpQdXRQcm90b2NvbHNMaXN0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIFRhZyB0byBhIGdpdmVuIHJlc291cmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Ztcy8yMDE4LTAxLTAxL0FQSVJlZmVyZW5jZS9BUElfVGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRhZ1Jlc291cmNlKCkge1xuICAgIHRoaXMuYWRkKCdmbXM6VGFnUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZW1vdmVzIGEgVGFnIGZyb20gYSBnaXZlbiByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mbXMvMjAxOC0wMS0wMS9BUElSZWZlcmVuY2UvQVBJX1VudGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHVudGFnUmVzb3VyY2UoKSB7XG4gICAgdGhpcy5hZGQoJ2ZtczpVbnRhZ1Jlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgcG9saWN5IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Ztcy8yMDE4LTAxLTAxL0FQSVJlZmVyZW5jZS9BUElfUG9saWN5Lmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblBvbGljeShpZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpmbXM6JHtSZWdpb259OiR7QWNjb3VudH06cG9saWN5LyR7SWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtJZH0nLCBpZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgYXBwbGljYXRpb25zLWxpc3QgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZm1zLzIwMTgtMDEtMDEvQVBJUmVmZXJlbmNlL0FQSV9BcHBzTGlzdERhdGEuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gaWQgLSBJZGVudGlmaWVyIGZvciB0aGUgaWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uQXBwbGljYXRpb25zTGlzdChpZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpmbXM6JHtSZWdpb259OiR7QWNjb3VudH06YXBwbGljYXRpb25zLWxpc3QvJHtJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0lkfScsIGlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBwcm90b2NvbHMtbGlzdCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mbXMvMjAxOC0wMS0wMS9BUElSZWZlcmVuY2UvQVBJX1Byb3RvY29sc0xpc3REYXRhLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblByb3RvY29sc0xpc3QoaWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06Zm1zOiR7UmVnaW9ufToke0FjY291bnR9OnByb3RvY29scy1saXN0LyR7SWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtJZH0nLCBpZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY3Rpb25zIGJhc2VkIG9uIHRoZSBhbGxvd2VkIHNldCBvZiB2YWx1ZXMgZm9yIGVhY2ggb2YgdGhlIHRhZ3NcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXJlcXVlc3R0YWdcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC5wdXRBcHBzTGlzdCgpXG4gICAqIC0gLnB1dFBvbGljeSgpXG4gICAqIC0gLnB1dFByb3RvY29sc0xpc3QoKVxuICAgKiAtIC50YWdSZXNvdXJjZSgpXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXF1ZXN0VGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXF1ZXN0VGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY3Rpb25zIGJhc2VkIG9uIHRhZy12YWx1ZSBhc3NvaWNhdGVkIHdpdGggdGhlIHJlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1yZXNvdXJjZXRhZ1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLmRlbGV0ZVBvbGljeSgpXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gcG9saWN5XG4gICAqIC0gYXBwbGljYXRpb25zLWxpc3RcbiAgICogLSBwcm90b2NvbHMtbGlzdFxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVzb3VyY2VUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlJlc291cmNlVGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY3Rpb25zIGJhc2VkIG9uIHRoZSBwcmVzZW5jZSBvZiBtYW5kYXRvcnkgdGFncyBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtdGFna2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnB1dEFwcHNMaXN0KClcbiAgICogLSAucHV0UG9saWN5KClcbiAgICogLSAucHV0UHJvdG9jb2xzTGlzdCgpXG4gICAqIC0gLnRhZ1Jlc291cmNlKClcbiAgICogLSAudW50YWdSZXNvdXJjZSgpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzVGFnS2V5cyh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpUYWdLZXlzYCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cbn1cbiJdfQ==