"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Sts = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [sts](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssecuritytokenservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Sts extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [sts](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssecuritytokenservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'sts';
        this.actionList = {
            "AssumeRole": {
                "url": "https://docs.aws.amazon.com/STS/latest/APIReference/API_AssumeRole.html",
                "description": "Returns a set of temporary security credentials that you can use to access AWS resources that you might not normally have access to",
                "accessLevel": "Write",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:PrincipalTag/${TagKey}",
                    "aws:RequestTag/${TagKey}",
                    "sts:TransitiveTagKeys",
                    "sts:ExternalId",
                    "sts:RoleSessionName"
                ]
            },
            "AssumeRoleWithSAML": {
                "url": "https://docs.aws.amazon.com/STS/latest/APIReference/API_AssumeRoleWithSAML.html",
                "description": "Returns a set of temporary security credentials for users who have been authenticated via a SAML authentication response",
                "accessLevel": "Write",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                },
                "conditions": [
                    "saml:namequalifier",
                    "saml:sub",
                    "saml:sub_type",
                    "saml:aud",
                    "saml:iss",
                    "saml:doc",
                    "saml:cn",
                    "saml:commonName",
                    "saml:eduorghomepageuri",
                    "saml:eduorgidentityauthnpolicyuri",
                    "saml:eduorglegalname",
                    "saml:eduorgsuperioruri",
                    "saml:eduorgwhitepagesuri",
                    "saml:edupersonaffiliation",
                    "saml:edupersonassurance",
                    "saml:edupersonentitlement",
                    "saml:edupersonnickname",
                    "saml:edupersonorgdn",
                    "saml:edupersonorgunitdn",
                    "saml:edupersonprimaryaffiliation",
                    "saml:edupersonprimaryorgunitdn",
                    "saml:edupersonprincipalname",
                    "saml:edupersonscopedaffiliation",
                    "saml:edupersontargetedid",
                    "saml:givenName",
                    "saml:mail",
                    "saml:name",
                    "saml:organizationStatus",
                    "saml:primaryGroupSID",
                    "saml:surname",
                    "saml:uid",
                    "saml:x500UniqueIdentifier",
                    "aws:TagKeys",
                    "aws:PrincipalTag/${TagKey}",
                    "aws:RequestTag/${TagKey}",
                    "sts:TransitiveTagKeys"
                ]
            },
            "AssumeRoleWithWebIdentity": {
                "url": "https://docs.aws.amazon.com/STS/latest/APIReference/API_AssumeRoleWithWebIdentity.html",
                "description": "Returns a set of temporary security credentials for users who have been authenticated in a mobile or web application with a web identity provider",
                "accessLevel": "Write",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                },
                "conditions": [
                    "cognito-identity.amazonaws.com:amr",
                    "cognito-identity.amazonaws.com:aud",
                    "cognito-identity.amazonaws.com:sub",
                    "www.amazon.com:app_id",
                    "www.amazon.com:user_id",
                    "graph.facebook.com:app_id",
                    "graph.facebook.com:id",
                    "accounts.google.com:aud",
                    "accounts.google.com:oaud",
                    "accounts.google.com:sub",
                    "aws:TagKeys",
                    "aws:PrincipalTag/${TagKey}",
                    "aws:RequestTag/${TagKey}",
                    "sts:TransitiveTagKeys"
                ]
            },
            "DecodeAuthorizationMessage": {
                "url": "https://docs.aws.amazon.com/STS/latest/APIReference/API_DecodeAuthorizationMessage.html",
                "description": "Decodes additional information about the authorization status of a request from an encoded message returned in response to an AWS request",
                "accessLevel": "Write"
            },
            "GetAccessKeyInfo": {
                "url": "https://docs.aws.amazon.com/STS/latest/APIReference/API_GetAccessKeyInfo.html",
                "description": "Returns details about the access key id passed as a parameter to the request.",
                "accessLevel": "Read"
            },
            "GetCallerIdentity": {
                "url": "https://docs.aws.amazon.com/STS/latest/APIReference/API_GetCallerIdentity.html",
                "description": "Returns details about the IAM identity whose credentials are used to call the API",
                "accessLevel": "Read"
            },
            "GetFederationToken": {
                "url": "https://docs.aws.amazon.com/STS/latest/APIReference/API_GetFederationToken.html",
                "description": "Returns a set of temporary security credentials (consisting of an access key ID, a secret access key, and a security token) for a federated user",
                "accessLevel": "Read",
                "resourceTypes": {
                    "user": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:PrincipalTag/${TagKey}",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "GetServiceBearerToken": {
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_bearer.html",
                "description": "Returns a STS bearer token for an AWS root user, IAM role, or an IAM user",
                "accessLevel": "Read"
            },
            "GetSessionToken": {
                "url": "https://docs.aws.amazon.com/STS/latest/APIReference/API_GetSessionToken.html",
                "description": "Returns a set of temporary security credentials (consisting of an access key ID, a secret access key, and a security token) for an AWS account or IAM user",
                "accessLevel": "Read"
            },
            "TagSession": {
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_session-tags.html",
                "description": "Grants permission to add tags to a STS session",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "role": {
                        "required": false
                    },
                    "user": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:PrincipalTag/${TagKey}",
                    "aws:RequestTag/${TagKey}",
                    "sts:TransitiveTagKeys"
                ]
            }
        };
        this.resourceTypes = {
            "role": {
                "name": "role",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles.html",
                "arn": "arn:${Partition}:iam::${Account}:role/${RoleNameWithPath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "user": {
                "name": "user",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users.html",
                "arn": "arn:${Partition}:iam::${Account}:user/${UserNameWithPath}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Returns a set of temporary security credentials that you can use to access AWS resources that you might not normally have access to
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsPrincipalTag()
     * - .ifAwsRequestTag()
     * - .ifTransitiveTagKeys()
     * - .ifExternalId()
     * - .ifRoleSessionName()
     *
     * https://docs.aws.amazon.com/STS/latest/APIReference/API_AssumeRole.html
     */
    assumeRole() {
        this.add('sts:AssumeRole');
        return this;
    }
    /**
     * Returns a set of temporary security credentials for users who have been authenticated via a SAML authentication response
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifNamequalifier()
     * - .ifSub()
     * - .ifSubType()
     * - .ifAud()
     * - .ifIss()
     * - .ifDoc()
     * - .ifCn()
     * - .ifCommonName()
     * - .ifEduorghomepageuri()
     * - .ifEduorgidentityauthnpolicyuri()
     * - .ifEduorglegalname()
     * - .ifEduorgsuperioruri()
     * - .ifEduorgwhitepagesuri()
     * - .ifEdupersonaffiliation()
     * - .ifEdupersonassurance()
     * - .ifEdupersonentitlement()
     * - .ifEdupersonnickname()
     * - .ifEdupersonorgdn()
     * - .ifEdupersonorgunitdn()
     * - .ifEdupersonprimaryaffiliation()
     * - .ifEdupersonprimaryorgunitdn()
     * - .ifEdupersonprincipalname()
     * - .ifEdupersonscopedaffiliation()
     * - .ifEdupersontargetedid()
     * - .ifGivenName()
     * - .ifMail()
     * - .ifName()
     * - .ifOrganizationStatus()
     * - .ifPrimaryGroupSID()
     * - .ifSurname()
     * - .ifUid()
     * - .ifX500UniqueIdentifier()
     * - .ifAwsTagKeys()
     * - .ifAwsPrincipalTag()
     * - .ifAwsRequestTag()
     * - .ifTransitiveTagKeys()
     *
     * https://docs.aws.amazon.com/STS/latest/APIReference/API_AssumeRoleWithSAML.html
     */
    assumeRoleWithSAML() {
        this.add('sts:AssumeRoleWithSAML');
        return this;
    }
    /**
     * Returns a set of temporary security credentials for users who have been authenticated in a mobile or web application with a web identity provider
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCognitoAmr()
     * - .ifCognitoAud()
     * - .ifCognitoSub()
     * - .ifAmazonAppId()
     * - .ifAmazonUserId()
     * - .ifFacebookAppId()
     * - .ifFacebookId()
     * - .ifGoogleAud()
     * - .ifGoogleOaud()
     * - .ifGoogleSub()
     * - .ifAwsTagKeys()
     * - .ifAwsPrincipalTag()
     * - .ifAwsRequestTag()
     * - .ifTransitiveTagKeys()
     *
     * https://docs.aws.amazon.com/STS/latest/APIReference/API_AssumeRoleWithWebIdentity.html
     */
    assumeRoleWithWebIdentity() {
        this.add('sts:AssumeRoleWithWebIdentity');
        return this;
    }
    /**
     * Decodes additional information about the authorization status of a request from an encoded message returned in response to an AWS request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/STS/latest/APIReference/API_DecodeAuthorizationMessage.html
     */
    decodeAuthorizationMessage() {
        this.add('sts:DecodeAuthorizationMessage');
        return this;
    }
    /**
     * Returns details about the access key id passed as a parameter to the request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/STS/latest/APIReference/API_GetAccessKeyInfo.html
     */
    getAccessKeyInfo() {
        this.add('sts:GetAccessKeyInfo');
        return this;
    }
    /**
     * Returns details about the IAM identity whose credentials are used to call the API
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/STS/latest/APIReference/API_GetCallerIdentity.html
     */
    getCallerIdentity() {
        this.add('sts:GetCallerIdentity');
        return this;
    }
    /**
     * Returns a set of temporary security credentials (consisting of an access key ID, a secret access key, and a security token) for a federated user
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsPrincipalTag()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/STS/latest/APIReference/API_GetFederationToken.html
     */
    getFederationToken() {
        this.add('sts:GetFederationToken');
        return this;
    }
    /**
     * Returns a STS bearer token for an AWS root user, IAM role, or an IAM user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_bearer.html
     */
    getServiceBearerToken() {
        this.add('sts:GetServiceBearerToken');
        return this;
    }
    /**
     * Returns a set of temporary security credentials (consisting of an access key ID, a secret access key, and a security token) for an AWS account or IAM user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/STS/latest/APIReference/API_GetSessionToken.html
     */
    getSessionToken() {
        this.add('sts:GetSessionToken');
        return this;
    }
    /**
     * Grants permission to add tags to a STS session
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsPrincipalTag()
     * - .ifAwsRequestTag()
     * - .ifTransitiveTagKeys()
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_session-tags.html
     */
    tagSession() {
        this.add('sts:TagSession');
        return this;
    }
    /**
     * Adds a resource of type role to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles.html
     *
     * @param roleNameWithPath - Identifier for the roleNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRole(roleNameWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:role/${RoleNameWithPath}';
        arn = arn.replace('${RoleNameWithPath}', roleNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users.html
     *
     * @param userNameWithPath - Identifier for the userNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onUser(userNameWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:user/${UserNameWithPath}';
        arn = arn.replace('${UserNameWithPath}', userNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the Google application ID
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_aud
     *
     * Applies to actions:
     * - .assumeRoleWithWebIdentity()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGoogleAud(value, operator) {
        return this.if(`accounts.google.com:aud`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the Google audience
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_oaud
     *
     * Applies to actions:
     * - .assumeRoleWithWebIdentity()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGoogleOaud(value, operator) {
        return this.if(`accounts.google.com:oaud`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the subject of the claim (the Google user ID)
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_sub
     *
     * Applies to actions:
     * - .assumeRoleWithWebIdentity()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGoogleSub(value, operator) {
        return this.if(`accounts.google.com:sub`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the IdP that was used to authenticate the user
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_federatedprovider
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsFederatedProvider(value, operator) {
        return this.if(`aws:FederatedProvider`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tag associated with the principal that is making the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principaltag
     *
     * Applies to actions:
     * - .assumeRole()
     * - .assumeRoleWithSAML()
     * - .assumeRoleWithWebIdentity()
     * - .getFederationToken()
     * - .tagSession()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsPrincipalTag(tagKey, value, operator) {
        return this.if(`aws:PrincipalTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .assumeRole()
     * - .assumeRoleWithSAML()
     * - .assumeRoleWithWebIdentity()
     * - .getFederationToken()
     * - .tagSession()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - role
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .assumeRole()
     * - .assumeRoleWithSAML()
     * - .assumeRoleWithWebIdentity()
     * - .getFederationToken()
     * - .tagSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the login information for Amazon Cognito
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_amr
     *
     * Applies to actions:
     * - .assumeRoleWithWebIdentity()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCognitoAmr(value, operator) {
        return this.if(`cognito-identity.amazonaws.com:amr`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the Amazon Cognito identity pool ID
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_aud
     *
     * Applies to actions:
     * - .assumeRoleWithWebIdentity()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCognitoAud(value, operator) {
        return this.if(`cognito-identity.amazonaws.com:aud`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the subject of the claim (the Amazon Cognito user ID)
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_sub
     *
     * Applies to actions:
     * - .assumeRoleWithWebIdentity()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCognitoSub(value, operator) {
        return this.if(`cognito-identity.amazonaws.com:sub`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the Facebook application ID
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_id
     *
     * Applies to actions:
     * - .assumeRoleWithWebIdentity()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFacebookAppId(value, operator) {
        return this.if(`graph.facebook.com:app_id`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the Facebook user ID
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_id
     *
     * Applies to actions:
     * - .assumeRoleWithWebIdentity()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFacebookId(value, operator) {
        return this.if(`graph.facebook.com:id`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the endpoint URL to which SAML assertions are presented
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_aud
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAud(value, operator) {
        return this.if(`saml:aud`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the eduOrg attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_cn
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCn(value, operator) {
        return this.if(`saml:cn`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the commonName attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_commonname
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCommonName(value, operator) {
        return this.if(`saml:commonName`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the principal that was used to assume the role
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_doc
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDoc(value, operator) {
        return this.if(`saml:doc`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the eduOrg attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_eduorghomepageuri
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEduorghomepageuri(value, operator) {
        return this.if(`saml:eduorghomepageuri`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the eduOrg attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_aud
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEduorgidentityauthnpolicyuri(value, operator) {
        return this.if(`saml:eduorgidentityauthnpolicyuri`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the eduOrg attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_eduorglegalname
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEduorglegalname(value, operator) {
        return this.if(`saml:eduorglegalname`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the eduOrg attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_eduorgsuperioruri
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEduorgsuperioruri(value, operator) {
        return this.if(`saml:eduorgsuperioruri`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the eduOrg attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_eduorgwhitepagesuri
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEduorgwhitepagesuri(value, operator) {
        return this.if(`saml:eduorgwhitepagesuri`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the eduPerson attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_edupersonaffiliation
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEdupersonaffiliation(value, operator) {
        return this.if(`saml:edupersonaffiliation`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the eduPerson attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_edupersonassurance
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEdupersonassurance(value, operator) {
        return this.if(`saml:edupersonassurance`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the eduPerson attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_edupersonentitlement
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEdupersonentitlement(value, operator) {
        return this.if(`saml:edupersonentitlement`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the eduPerson attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_edupersonnickname
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEdupersonnickname(value, operator) {
        return this.if(`saml:edupersonnickname`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the eduPerson attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_edupersonorgdn
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEdupersonorgdn(value, operator) {
        return this.if(`saml:edupersonorgdn`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the eduPerson attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_edupersonorgunitdn
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEdupersonorgunitdn(value, operator) {
        return this.if(`saml:edupersonorgunitdn`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the eduPerson attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_edupersonprimaryaffiliation
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEdupersonprimaryaffiliation(value, operator) {
        return this.if(`saml:edupersonprimaryaffiliation`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the eduPerson attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_edupersonprimaryorgunitdn
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEdupersonprimaryorgunitdn(value, operator) {
        return this.if(`saml:edupersonprimaryorgunitdn`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the eduPerson attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_edupersonprincipalname
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEdupersonprincipalname(value, operator) {
        return this.if(`saml:edupersonprincipalname`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the eduPerson attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_edupersonscopedaffiliation
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEdupersonscopedaffiliation(value, operator) {
        return this.if(`saml:edupersonscopedaffiliation`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the eduPerson attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_edupersontargetedid
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEdupersontargetedid(value, operator) {
        return this.if(`saml:edupersontargetedid`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the givenName attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_givenname
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGivenName(value, operator) {
        return this.if(`saml:givenName`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the issuer, which is represented by a URN
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_iss
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIss(value, operator) {
        return this.if(`saml:iss`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the mail attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_mail
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMail(value, operator) {
        return this.if(`saml:mail`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the name attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_name
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifName(value, operator) {
        return this.if(`saml:name`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the hash value of the issuer, account ID, and friendly name
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_namequalifier
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifNamequalifier(value, operator) {
        return this.if(`saml:namequalifier`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the organizationStatus attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_organizationstatus
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOrganizationStatus(value, operator) {
        return this.if(`saml:organizationStatus`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the primaryGroupSID attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_primarygroupsid
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPrimaryGroupSID(value, operator) {
        return this.if(`saml:primaryGroupSID`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the subject of the claim (the SAML user ID)
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_sub
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSub(value, operator) {
        return this.if(`saml:sub`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the value persistent, transient, or the full Format URI
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_subtype
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSubType(value, operator) {
        return this.if(`saml:sub_type`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the surname attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_surname
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSurname(value, operator) {
        return this.if(`saml:surname`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the uid attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_uid
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUid(value, operator) {
        return this.if(`saml:uid`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the uid attribute
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_x500uniqueidentifier
     *
     * Applies to actions:
     * - .assumeRoleWithSAML()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifX500UniqueIdentifier(value, operator) {
        return this.if(`saml:x500UniqueIdentifier`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the unique identifier required when you assume a role in another account
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_externalid
     *
     * Applies to actions:
     * - .assumeRole()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExternalId(value, operator) {
        return this.if(`sts:ExternalId`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the role session name required when you assume a role
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_rolesessionname
     *
     * Applies to actions:
     * - .assumeRole()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRoleSessionName(value, operator) {
        return this.if(`sts:RoleSessionName`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the transitive tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_TransitiveTagKeys
     *
     * Applies to actions:
     * - .assumeRole()
     * - .assumeRoleWithSAML()
     * - .assumeRoleWithWebIdentity()
     * - .tagSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTransitiveTagKeys(value, operator) {
        return this.if(`sts:TransitiveTagKeys`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the Login with Amazon application ID
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_id
     *
     * Applies to actions:
     * - .assumeRoleWithWebIdentity()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAmazonAppId(value, operator) {
        return this.if(`www.amazon.com:app_id`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the Login with Amazon user ID
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_id
     *
     * Applies to actions:
     * - .assumeRoleWithWebIdentity()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAmazonUserId(value, operator) {
        return this.if(`www.amazon.com:user_id`, value, operator || 'StringLike');
    }
}
exports.Sts = Sts;
//# sourceMappingURL=data:application/json;base64,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