"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Sms = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [sms](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsservermigrationservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Sms extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [sms](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsservermigrationservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'sms';
        this.actionList = {
            "CreateApp": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_CreateApp.html",
                "description": "Grants permission to create an application configuration to migrate on-premise application onto AWS",
                "accessLevel": "Write"
            },
            "CreateReplicationJob": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_CreateReplicationJob.html",
                "description": "Grants permission to create a job to migrate on-premise server onto AWS",
                "accessLevel": "Write"
            },
            "DeleteApp": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_DeleteApp.html",
                "description": "Grants permission to delete an existing application configuration",
                "accessLevel": "Write"
            },
            "DeleteAppLaunchConfiguration": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_DeleteAppLaunchConfiguration.html",
                "description": "Grants permission to delete launch configuration for an existing application",
                "accessLevel": "Write"
            },
            "DeleteAppReplicationConfiguration": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_DeleteAppReplicationConfiguration.html",
                "description": "Grants permission to delete replication configuration for an existing application",
                "accessLevel": "Write"
            },
            "DeleteAppValidationConfiguration": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_DeleteAppValidationConfiguration.html",
                "description": "Grants permission to delete validation configuration for an existing application",
                "accessLevel": "Write"
            },
            "DeleteReplicationJob": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_DeleteReplicationJob.html",
                "description": "Grants permission to delete an existing job to migrate on-premise server onto AWS",
                "accessLevel": "Write"
            },
            "DeleteServerCatalog": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_DeleteServerCatalog.html",
                "description": "Grants permission to delete the complete list of on-premise servers gathered into AWS",
                "accessLevel": "Write"
            },
            "DisassociateConnector": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_DisassociateConnector.html",
                "description": "Grants permission to disassociate a connector that has been associated",
                "accessLevel": "Write"
            },
            "GenerateChangeSet": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GenerateChangeSet.html",
                "description": "Grants permission to generate a changeSet for the CloudFormation stack of an application",
                "accessLevel": "Write"
            },
            "GenerateTemplate": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GenerateTemplate.html",
                "description": "Grants permission to generate a CloudFormation template for an existing application",
                "accessLevel": "Write"
            },
            "GetApp": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetApp.html",
                "description": "Grants permission to get the configuration and statuses for an existing application",
                "accessLevel": "Read"
            },
            "GetAppLaunchConfiguration": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetAppLaunchConfiguration.html",
                "description": "Grants permission to get launch configuration for an existing application",
                "accessLevel": "Read"
            },
            "GetAppReplicationConfiguration": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetAppReplicationConfiguration.html",
                "description": "Grants permission to get replication configuration for an existing application",
                "accessLevel": "Read"
            },
            "GetAppValidationConfiguration": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetAppValidationConfiguration.html",
                "description": "Grants permission to get validation configuration for an existing application",
                "accessLevel": "Read"
            },
            "GetAppValidationOutput": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetAppValidationOutput.html",
                "description": "Grants permission to get notification sent from application validation script.",
                "accessLevel": "Read"
            },
            "GetConnectors": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetConnectors.html",
                "description": "Grants permission to get all connectors that have been associated",
                "accessLevel": "Read"
            },
            "GetMessages": {
                "url": "",
                "description": "Grants permission to gets messages from AWS Server Migration Service to Server Migration Connector",
                "accessLevel": "Read"
            },
            "GetReplicationJobs": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetReplicationJobs.html",
                "description": "Grants permission to get all existing jobs to migrate on-premise servers onto AWS",
                "accessLevel": "Read"
            },
            "GetReplicationRuns": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetReplicationRuns.html",
                "description": "Grants permission to get all runs for an existing job",
                "accessLevel": "Read"
            },
            "GetServers": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetServers.html",
                "description": "Grants permission to get all servers that have been imported",
                "accessLevel": "Read"
            },
            "ImportAppCatalog": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_ImportAppCatalog.html",
                "description": "Grants permission to import application catalog from AWS Application Discovery Service",
                "accessLevel": "Write"
            },
            "ImportServerCatalog": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_ImportServerCatalog.html",
                "description": "Grants permission to gather a complete list of on-premise servers",
                "accessLevel": "Write"
            },
            "LaunchApp": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_LaunchApp.html",
                "description": "Grants permission to create and launch a CloudFormation stack for an existing application",
                "accessLevel": "Write"
            },
            "ListApps": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_ListAppss.html",
                "description": "Grants permission to get a list of summaries for existing applications",
                "accessLevel": "List"
            },
            "NotifyAppValidationOutput": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_NotifyAppValidationOutput.html",
                "description": "Grants permission to send notification for application validation script",
                "accessLevel": "Write"
            },
            "PutAppLaunchConfiguration": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_PutAppLaunchConfiguration.html",
                "description": "Grants permission to create or update launch configuration for an existing application",
                "accessLevel": "Write"
            },
            "PutAppReplicationConfiguration": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_PutAppReplicationConfiguration.html",
                "description": "Grants permission to create or update replication configuration for an existing application",
                "accessLevel": "Write"
            },
            "PutAppValidationConfiguration": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_PutAppValidationConfiguration.html",
                "description": "Grants permission to put validation configuration for an existing application",
                "accessLevel": "Write"
            },
            "SendMessage": {
                "url": "",
                "description": "Grants permission to send message from Server Migration Connector to AWS Server Migration Service",
                "accessLevel": "Write"
            },
            "StartAppReplication": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_StartAppReplication.html",
                "description": "Grants permission to create and start replication jobs for an existing application",
                "accessLevel": "Write"
            },
            "StartOnDemandAppReplication": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_StartOnDemandAppReplication.html",
                "description": "Grants permission to start a replication run for an existing application",
                "accessLevel": "Write"
            },
            "StartOnDemandReplicationRun": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_StartOnDemandReplicationRun.html",
                "description": "Grants permission to start a replication run for an existing replication job",
                "accessLevel": "Write"
            },
            "StopAppReplication": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_StopAppReplication.html",
                "description": "Grants permission to stop and delete replication jobs for an existing application",
                "accessLevel": "Write"
            },
            "TerminateApp": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_TerminateApp.html",
                "description": "Grants permission to terminate the CloudFormation stack for an existing application",
                "accessLevel": "Write"
            },
            "UpdateApp": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_UpdateApp.html",
                "description": "Grants permission to update an existing application configuration",
                "accessLevel": "Write"
            },
            "UpdateReplicationJob": {
                "url": "https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_UpdateReplicationJob.html",
                "description": "Grants permission to update an existing job to migrate on-premise server onto AWS",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {};
    }
    /**
     * Grants permission to create an application configuration to migrate on-premise application onto AWS
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_CreateApp.html
     */
    createApp() {
        this.add('sms:CreateApp');
        return this;
    }
    /**
     * Grants permission to create a job to migrate on-premise server onto AWS
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_CreateReplicationJob.html
     */
    createReplicationJob() {
        this.add('sms:CreateReplicationJob');
        return this;
    }
    /**
     * Grants permission to delete an existing application configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_DeleteApp.html
     */
    deleteApp() {
        this.add('sms:DeleteApp');
        return this;
    }
    /**
     * Grants permission to delete launch configuration for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_DeleteAppLaunchConfiguration.html
     */
    deleteAppLaunchConfiguration() {
        this.add('sms:DeleteAppLaunchConfiguration');
        return this;
    }
    /**
     * Grants permission to delete replication configuration for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_DeleteAppReplicationConfiguration.html
     */
    deleteAppReplicationConfiguration() {
        this.add('sms:DeleteAppReplicationConfiguration');
        return this;
    }
    /**
     * Grants permission to delete validation configuration for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_DeleteAppValidationConfiguration.html
     */
    deleteAppValidationConfiguration() {
        this.add('sms:DeleteAppValidationConfiguration');
        return this;
    }
    /**
     * Grants permission to delete an existing job to migrate on-premise server onto AWS
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_DeleteReplicationJob.html
     */
    deleteReplicationJob() {
        this.add('sms:DeleteReplicationJob');
        return this;
    }
    /**
     * Grants permission to delete the complete list of on-premise servers gathered into AWS
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_DeleteServerCatalog.html
     */
    deleteServerCatalog() {
        this.add('sms:DeleteServerCatalog');
        return this;
    }
    /**
     * Grants permission to disassociate a connector that has been associated
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_DisassociateConnector.html
     */
    disassociateConnector() {
        this.add('sms:DisassociateConnector');
        return this;
    }
    /**
     * Grants permission to generate a changeSet for the CloudFormation stack of an application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GenerateChangeSet.html
     */
    generateChangeSet() {
        this.add('sms:GenerateChangeSet');
        return this;
    }
    /**
     * Grants permission to generate a CloudFormation template for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GenerateTemplate.html
     */
    generateTemplate() {
        this.add('sms:GenerateTemplate');
        return this;
    }
    /**
     * Grants permission to get the configuration and statuses for an existing application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetApp.html
     */
    getApp() {
        this.add('sms:GetApp');
        return this;
    }
    /**
     * Grants permission to get launch configuration for an existing application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetAppLaunchConfiguration.html
     */
    getAppLaunchConfiguration() {
        this.add('sms:GetAppLaunchConfiguration');
        return this;
    }
    /**
     * Grants permission to get replication configuration for an existing application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetAppReplicationConfiguration.html
     */
    getAppReplicationConfiguration() {
        this.add('sms:GetAppReplicationConfiguration');
        return this;
    }
    /**
     * Grants permission to get validation configuration for an existing application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetAppValidationConfiguration.html
     */
    getAppValidationConfiguration() {
        this.add('sms:GetAppValidationConfiguration');
        return this;
    }
    /**
     * Grants permission to get notification sent from application validation script.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetAppValidationOutput.html
     */
    getAppValidationOutput() {
        this.add('sms:GetAppValidationOutput');
        return this;
    }
    /**
     * Grants permission to get all connectors that have been associated
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetConnectors.html
     */
    getConnectors() {
        this.add('sms:GetConnectors');
        return this;
    }
    /**
     * Grants permission to gets messages from AWS Server Migration Service to Server Migration Connector
     *
     * Access Level: Read
     */
    getMessages() {
        this.add('sms:GetMessages');
        return this;
    }
    /**
     * Grants permission to get all existing jobs to migrate on-premise servers onto AWS
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetReplicationJobs.html
     */
    getReplicationJobs() {
        this.add('sms:GetReplicationJobs');
        return this;
    }
    /**
     * Grants permission to get all runs for an existing job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetReplicationRuns.html
     */
    getReplicationRuns() {
        this.add('sms:GetReplicationRuns');
        return this;
    }
    /**
     * Grants permission to get all servers that have been imported
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetServers.html
     */
    getServers() {
        this.add('sms:GetServers');
        return this;
    }
    /**
     * Grants permission to import application catalog from AWS Application Discovery Service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_ImportAppCatalog.html
     */
    importAppCatalog() {
        this.add('sms:ImportAppCatalog');
        return this;
    }
    /**
     * Grants permission to gather a complete list of on-premise servers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_ImportServerCatalog.html
     */
    importServerCatalog() {
        this.add('sms:ImportServerCatalog');
        return this;
    }
    /**
     * Grants permission to create and launch a CloudFormation stack for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_LaunchApp.html
     */
    launchApp() {
        this.add('sms:LaunchApp');
        return this;
    }
    /**
     * Grants permission to get a list of summaries for existing applications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_ListAppss.html
     */
    listApps() {
        this.add('sms:ListApps');
        return this;
    }
    /**
     * Grants permission to send notification for application validation script
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_NotifyAppValidationOutput.html
     */
    notifyAppValidationOutput() {
        this.add('sms:NotifyAppValidationOutput');
        return this;
    }
    /**
     * Grants permission to create or update launch configuration for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_PutAppLaunchConfiguration.html
     */
    putAppLaunchConfiguration() {
        this.add('sms:PutAppLaunchConfiguration');
        return this;
    }
    /**
     * Grants permission to create or update replication configuration for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_PutAppReplicationConfiguration.html
     */
    putAppReplicationConfiguration() {
        this.add('sms:PutAppReplicationConfiguration');
        return this;
    }
    /**
     * Grants permission to put validation configuration for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_PutAppValidationConfiguration.html
     */
    putAppValidationConfiguration() {
        this.add('sms:PutAppValidationConfiguration');
        return this;
    }
    /**
     * Grants permission to send message from Server Migration Connector to AWS Server Migration Service
     *
     * Access Level: Write
     */
    sendMessage() {
        this.add('sms:SendMessage');
        return this;
    }
    /**
     * Grants permission to create and start replication jobs for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_StartAppReplication.html
     */
    startAppReplication() {
        this.add('sms:StartAppReplication');
        return this;
    }
    /**
     * Grants permission to start a replication run for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_StartOnDemandAppReplication.html
     */
    startOnDemandAppReplication() {
        this.add('sms:StartOnDemandAppReplication');
        return this;
    }
    /**
     * Grants permission to start a replication run for an existing replication job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_StartOnDemandReplicationRun.html
     */
    startOnDemandReplicationRun() {
        this.add('sms:StartOnDemandReplicationRun');
        return this;
    }
    /**
     * Grants permission to stop and delete replication jobs for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_StopAppReplication.html
     */
    stopAppReplication() {
        this.add('sms:StopAppReplication');
        return this;
    }
    /**
     * Grants permission to terminate the CloudFormation stack for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_TerminateApp.html
     */
    terminateApp() {
        this.add('sms:TerminateApp');
        return this;
    }
    /**
     * Grants permission to update an existing application configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_UpdateApp.html
     */
    updateApp() {
        this.add('sms:UpdateApp');
        return this;
    }
    /**
     * Grants permission to update an existing job to migrate on-premise server onto AWS
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_UpdateReplicationJob.html
     */
    updateReplicationJob() {
        this.add('sms:UpdateReplicationJob');
        return this;
    }
}
exports.Sms = Sms;
//# sourceMappingURL=data:application/json;base64,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