"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Trustedadvisor = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [trustedadvisor](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awstrustedadvisor.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Trustedadvisor extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [trustedadvisor](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awstrustedadvisor.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'trustedadvisor';
        this.actionList = {
            "DescribeAccount": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view the AWS Support plan and various AWS Trusted Advisor preferences",
                "accessLevel": "Read"
            },
            "DescribeAccountAccess": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view if the AWS account has enabled or disabled AWS Trusted Advisor",
                "accessLevel": "Read"
            },
            "DescribeCheckItems": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view details for the check items",
                "accessLevel": "Read",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "DescribeCheckRefreshStatuses": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view the refresh statuses for AWS Trusted Advisor checks",
                "accessLevel": "Read",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "DescribeCheckSummaries": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view AWS Trusted Advisor check summaries",
                "accessLevel": "Read",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "DescribeChecks": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view details for AWS Trusted Advisor checks",
                "accessLevel": "Read"
            },
            "DescribeNotificationPreferences": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view the notification preferences for the AWS account",
                "accessLevel": "Read"
            },
            "DescribeOrganization": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view if the AWS account meets the requirements to enable the organizational view feature",
                "accessLevel": "Read"
            },
            "DescribeOrganizationAccounts": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view the linked AWS accounts that are in the organization",
                "accessLevel": "Read"
            },
            "DescribeReports": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view details for organizational view reports, such as the report name, runtime, date created, status, and format",
                "accessLevel": "Read"
            },
            "DescribeServiceMetadata": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to view information about organizational view reports, such as the AWS Regions, check categories, check names, and resource statuses",
                "accessLevel": "Read"
            },
            "ExcludeCheckItems": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to exclude recommendations for AWS Trusted Advisor checks",
                "accessLevel": "Write",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "GenerateReport": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to create a report for AWS Trusted Advisor checks in your organization",
                "accessLevel": "Write"
            },
            "IncludeCheckItems": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to include recommendations for AWS Trusted Advisor checks",
                "accessLevel": "Write",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "RefreshCheck": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to refresh an AWS Trusted Advisor check",
                "accessLevel": "Write",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "SetAccountAccess": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to enable or disable AWS Trusted Advisor for the account",
                "accessLevel": "Write"
            },
            "SetOrganizationAccess": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to enable the organizational view feature for AWS Trusted Advisor",
                "accessLevel": "Write"
            },
            "UpdateNotificationPreferences": {
                "url": "https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations",
                "description": "Grants permission to update notification preferences for AWS Trusted Advisor",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "checks": {
                "name": "checks",
                "url": "https://docs.aws.amazon.com/awssupport/latest/APIReference/API_TrustedAdvisorCheckDescription.html",
                "arn": "arn:${Partition}:trustedadvisor:${Region}:${Account}:checks/${CategoryCode}/${CheckId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to view the AWS Support plan and various AWS Trusted Advisor preferences
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    describeAccount() {
        this.add('trustedadvisor:DescribeAccount');
        return this;
    }
    /**
     * Grants permission to view if the AWS account has enabled or disabled AWS Trusted Advisor
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    describeAccountAccess() {
        this.add('trustedadvisor:DescribeAccountAccess');
        return this;
    }
    /**
     * Grants permission to view details for the check items
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    describeCheckItems() {
        this.add('trustedadvisor:DescribeCheckItems');
        return this;
    }
    /**
     * Grants permission to view the refresh statuses for AWS Trusted Advisor checks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    describeCheckRefreshStatuses() {
        this.add('trustedadvisor:DescribeCheckRefreshStatuses');
        return this;
    }
    /**
     * Grants permission to view AWS Trusted Advisor check summaries
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    describeCheckSummaries() {
        this.add('trustedadvisor:DescribeCheckSummaries');
        return this;
    }
    /**
     * Grants permission to view details for AWS Trusted Advisor checks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    describeChecks() {
        this.add('trustedadvisor:DescribeChecks');
        return this;
    }
    /**
     * Grants permission to view the notification preferences for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    describeNotificationPreferences() {
        this.add('trustedadvisor:DescribeNotificationPreferences');
        return this;
    }
    /**
     * Grants permission to view if the AWS account meets the requirements to enable the organizational view feature
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    describeOrganization() {
        this.add('trustedadvisor:DescribeOrganization');
        return this;
    }
    /**
     * Grants permission to view the linked AWS accounts that are in the organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    describeOrganizationAccounts() {
        this.add('trustedadvisor:DescribeOrganizationAccounts');
        return this;
    }
    /**
     * Grants permission to view details for organizational view reports, such as the report name, runtime, date created, status, and format
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    describeReports() {
        this.add('trustedadvisor:DescribeReports');
        return this;
    }
    /**
     * Grants permission to view information about organizational view reports, such as the AWS Regions, check categories, check names, and resource statuses
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    describeServiceMetadata() {
        this.add('trustedadvisor:DescribeServiceMetadata');
        return this;
    }
    /**
     * Grants permission to exclude recommendations for AWS Trusted Advisor checks
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    excludeCheckItems() {
        this.add('trustedadvisor:ExcludeCheckItems');
        return this;
    }
    /**
     * Grants permission to create a report for AWS Trusted Advisor checks in your organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    generateReport() {
        this.add('trustedadvisor:GenerateReport');
        return this;
    }
    /**
     * Grants permission to include recommendations for AWS Trusted Advisor checks
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    includeCheckItems() {
        this.add('trustedadvisor:IncludeCheckItems');
        return this;
    }
    /**
     * Grants permission to refresh an AWS Trusted Advisor check
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    refreshCheck() {
        this.add('trustedadvisor:RefreshCheck');
        return this;
    }
    /**
     * Grants permission to enable or disable AWS Trusted Advisor for the account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    setAccountAccess() {
        this.add('trustedadvisor:SetAccountAccess');
        return this;
    }
    /**
     * Grants permission to enable the organizational view feature for AWS Trusted Advisor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    setOrganizationAccess() {
        this.add('trustedadvisor:SetOrganizationAccess');
        return this;
    }
    /**
     * Grants permission to update notification preferences for AWS Trusted Advisor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    updateNotificationPreferences() {
        this.add('trustedadvisor:UpdateNotificationPreferences');
        return this;
    }
    /**
     * Adds a resource of type checks to the statement
     *
     * https://docs.aws.amazon.com/awssupport/latest/APIReference/API_TrustedAdvisorCheckDescription.html
     *
     * @param categoryCode - Identifier for the categoryCode.
     * @param checkId - Identifier for the checkId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onChecks(categoryCode, checkId, account, region, partition) {
        var arn = 'arn:${Partition}:trustedadvisor:${Region}:${Account}:checks/${CategoryCode}/${CheckId}';
        arn = arn.replace('${CategoryCode}', categoryCode);
        arn = arn.replace('${CheckId}', checkId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Trustedadvisor = Trustedadvisor;
//# sourceMappingURL=data:application/json;base64,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