"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("../apigateway.generated");
const authorizer_1 = require("../authorizer");
class LambdaAuthorizer extends authorizer_1.Authorizer {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.handler = props.handler;
        this.role = props.assumeRole;
        if (props.resultsCacheTtl && ((_a = props.resultsCacheTtl) === null || _a === void 0 ? void 0 : _a.toSeconds()) > 3600) {
            throw new Error('Lambda authorizer property \'resultsCacheTtl\' must not be greater than 3600 seconds (1 hour)');
        }
    }
    /**
     * Attaches this authorizer to a specific REST API.
     * @internal
     */
    _attachToApi(restApi) {
        if (this.restApiId && this.restApiId !== restApi.restApiId) {
            throw new Error('Cannot attach authorizer to two different rest APIs');
        }
        this.restApiId = restApi.restApiId;
    }
    /**
     * Sets up the permissions necessary for the API Gateway service to invoke the Lambda function.
     */
    setupPermissions() {
        if (!this.role) {
            this.handler.addPermission(`${this.node.uniqueId}:Permissions`, {
                principal: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                sourceArn: this.authorizerArn,
            });
        }
        else if (this.role instanceof iam.Role) { // i.e. not imported
            this.role.attachInlinePolicy(new iam.Policy(this, 'authorizerInvokePolicy', {
                statements: [
                    new iam.PolicyStatement({
                        resources: [this.handler.functionArn],
                        actions: ['lambda:InvokeFunction'],
                    }),
                ],
            }));
        }
    }
    /**
     * Returns a token that resolves to the Rest Api Id at the time of synthesis.
     * Throws an error, during token resolution, if no RestApi is attached to this authorizer.
     */
    lazyRestApiId() {
        return core_1.Lazy.stringValue({
            produce: () => {
                if (!this.restApiId) {
                    throw new Error(`Authorizer (${this.node.path}) must be attached to a RestApi`);
                }
                return this.restApiId;
            },
        });
    }
}
/**
 * Token based lambda authorizer that recognizes the caller's identity as a bearer token,
 * such as a JSON Web Token (JWT) or an OAuth token.
 * Based on the token, authorization is performed by a lambda function.
 *
 * @resource AWS::ApiGateway::Authorizer
 */
class TokenAuthorizer extends LambdaAuthorizer {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, props);
        const restApiId = this.lazyRestApiId();
        const resource = new apigateway_generated_1.CfnAuthorizer(this, 'Resource', {
            name: (_a = props.authorizerName) !== null && _a !== void 0 ? _a : this.node.uniqueId,
            restApiId,
            type: 'TOKEN',
            authorizerUri: `arn:aws:apigateway:${core_1.Stack.of(this).region}:lambda:path/2015-03-31/functions/${props.handler.functionArn}/invocations`,
            authorizerCredentials: (_b = props.assumeRole) === null || _b === void 0 ? void 0 : _b.roleArn,
            authorizerResultTtlInSeconds: (_c = props.resultsCacheTtl) === null || _c === void 0 ? void 0 : _c.toSeconds(),
            identitySource: props.identitySource || 'method.request.header.Authorization',
            identityValidationExpression: props.validationRegex,
        });
        this.authorizerId = resource.ref;
        this.authorizerArn = core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: restApiId,
            resourceName: `authorizers/${this.authorizerId}`,
        });
        this.setupPermissions();
    }
}
exports.TokenAuthorizer = TokenAuthorizer;
/**
 * Request-based lambda authorizer that recognizes the caller's identity via request parameters,
 * such as headers, paths, query strings, stage variables, or context variables.
 * Based on the request, authorization is performed by a lambda function.
 *
 * @resource AWS::ApiGateway::Authorizer
 */
class RequestAuthorizer extends LambdaAuthorizer {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, props);
        if ((props.resultsCacheTtl === undefined || props.resultsCacheTtl.toSeconds() !== 0) && props.identitySources.length === 0) {
            throw new Error('At least one Identity Source is required for a REQUEST-based Lambda authorizer if caching is enabled.');
        }
        const restApiId = this.lazyRestApiId();
        const resource = new apigateway_generated_1.CfnAuthorizer(this, 'Resource', {
            name: (_a = props.authorizerName) !== null && _a !== void 0 ? _a : this.node.uniqueId,
            restApiId,
            type: 'REQUEST',
            authorizerUri: `arn:aws:apigateway:${core_1.Stack.of(this).region}:lambda:path/2015-03-31/functions/${props.handler.functionArn}/invocations`,
            authorizerCredentials: (_b = props.assumeRole) === null || _b === void 0 ? void 0 : _b.roleArn,
            authorizerResultTtlInSeconds: (_c = props.resultsCacheTtl) === null || _c === void 0 ? void 0 : _c.toSeconds(),
            identitySource: props.identitySources.map(is => is.toString()).join(','),
        });
        this.authorizerId = resource.ref;
        this.authorizerArn = core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: restApiId,
            resourceName: `authorizers/${this.authorizerId}`,
        });
        this.setupPermissions();
    }
}
exports.RequestAuthorizer = RequestAuthorizer;
//# sourceMappingURL=data:application/json;base64,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