"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Base class for backend integrations for an API Gateway method.
 *
 * Use one of the concrete classes such as `MockIntegration`, `AwsIntegration`, `LambdaIntegration`
 * or implement on your own by specifying the set of props.
 */
class Integration {
    constructor(props) {
        this.props = props;
    }
    /**
     * Allows `Method` to access the integration props.
     *
     * @internal
     */
    get _props() {
        return this.props;
    }
    /**
     * Can be overridden by subclasses to allow the integration to interact with the method
     * being integrated, access the REST API object, method ARNs, etc.
     */
    bind(_method) {
        return;
    }
}
exports.Integration = Integration;
var ContentHandling;
(function (ContentHandling) {
    /**
     * Converts a request payload from a base64-encoded string to a binary blob.
     */
    ContentHandling["CONVERT_TO_BINARY"] = "CONVERT_TO_BINARY";
    /**
     * Converts a request payload from a binary blob to a base64-encoded string.
     */
    ContentHandling["CONVERT_TO_TEXT"] = "CONVERT_TO_TEXT";
})(ContentHandling = exports.ContentHandling || (exports.ContentHandling = {}));
var IntegrationType;
(function (IntegrationType) {
    /**
     * For integrating the API method request with an AWS service action,
     * including the Lambda function-invoking action. With the Lambda
     * function-invoking action, this is referred to as the Lambda custom
     * integration. With any other AWS service action, this is known as AWS
     * integration.
     */
    IntegrationType["AWS"] = "AWS";
    /**
     * For integrating the API method request with the Lambda function-invoking
     * action with the client request passed through as-is. This integration is
     * also referred to as the Lambda proxy integration
     */
    IntegrationType["AWS_PROXY"] = "AWS_PROXY";
    /**
     * For integrating the API method request with an HTTP endpoint, including a
     * private HTTP endpoint within a VPC. This integration is also referred to
     * as the HTTP custom integration.
     */
    IntegrationType["HTTP"] = "HTTP";
    /**
     * For integrating the API method request with an HTTP endpoint, including a
     * private HTTP endpoint within a VPC, with the client request passed
     * through as-is. This is also referred to as the HTTP proxy integration
     */
    IntegrationType["HTTP_PROXY"] = "HTTP_PROXY";
    /**
     * For integrating the API method request with API Gateway as a "loop-back"
     * endpoint without invoking any backend.
     */
    IntegrationType["MOCK"] = "MOCK";
})(IntegrationType = exports.IntegrationType || (exports.IntegrationType = {}));
var PassthroughBehavior;
(function (PassthroughBehavior) {
    /**
     * Passes the request body for unmapped content types through to the
     * integration back end without transformation.
     */
    PassthroughBehavior["WHEN_NO_MATCH"] = "WHEN_NO_MATCH";
    /**
     * Rejects unmapped content types with an HTTP 415 'Unsupported Media Type'
     * response
     */
    PassthroughBehavior["NEVER"] = "NEVER";
    /**
     * Allows pass-through when the integration has NO content types mapped to
     * templates. However if there is at least one content type defined,
     * unmapped content types will be rejected with the same 415 response.
     */
    PassthroughBehavior["WHEN_NO_TEMPLATES"] = "WHEN_NO_TEMPLATES";
})(PassthroughBehavior = exports.PassthroughBehavior || (exports.PassthroughBehavior = {}));
var ConnectionType;
(function (ConnectionType) {
    /**
     * For connections through the public routable internet
     */
    ConnectionType["INTERNET"] = "INTERNET";
    /**
     * For private connections between API Gateway and a network load balancer in a VPC
     */
    ConnectionType["VPC_LINK"] = "VPC_LINK";
})(ConnectionType = exports.ConnectionType || (exports.ConnectionType = {}));
//# sourceMappingURL=data:application/json;base64,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