"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const aws_1 = require("./aws");
/**
 * Integrates an AWS Lambda function to an API Gateway method.
 *
 * @example
 *
 *    const handler = new lambda.Function(this, 'MyFunction', ...);
 *    api.addMethod('GET', new LambdaIntegration(handler));
 *
 */
class LambdaIntegration extends aws_1.AwsIntegration {
    constructor(handler, options = {}) {
        const proxy = options.proxy === undefined ? true : options.proxy;
        super({
            proxy,
            service: 'lambda',
            path: `2015-03-31/functions/${handler.functionArn}/invocations`,
            options,
        });
        this.handler = handler;
        this.enableTest = options.allowTestInvoke === undefined ? true : false;
    }
    bind(method) {
        super.bind(method);
        const principal = new iam.ServicePrincipal('apigateway.amazonaws.com');
        const desc = `${method.restApi.node.uniqueId}.${method.httpMethod}.${method.resource.path.replace(/\//g, '.')}`;
        this.handler.addPermission(`ApiPermission.${desc}`, {
            principal,
            scope: method,
            sourceArn: core_1.Lazy.stringValue({ produce: () => method.methodArn }),
        });
        // add permission to invoke from the console
        if (this.enableTest) {
            this.handler.addPermission(`ApiPermission.Test.${desc}`, {
                principal,
                scope: method,
                sourceArn: method.testMethodArn,
            });
        }
    }
}
exports.LambdaIntegration = LambdaIntegration;
//# sourceMappingURL=data:application/json;base64,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