"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const api_key_1 = require("./api-key");
const apigateway_generated_1 = require("./apigateway.generated");
const deployment_1 = require("./deployment");
const domain_name_1 = require("./domain-name");
const gateway_response_1 = require("./gateway-response");
const model_1 = require("./model");
const requestvalidator_1 = require("./requestvalidator");
const resource_1 = require("./resource");
const stage_1 = require("./stage");
const usage_plan_1 = require("./usage-plan");
class RestApiBase extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.restApiName || id,
        });
    }
    /**
     * Returns the URL for an HTTP path.
     *
     * Fails if `deploymentStage` is not set either by `deploy` or explicitly.
     */
    urlForPath(path = '/') {
        if (!this.deploymentStage) {
            throw new Error('Cannot determine deployment stage for API from "deploymentStage". Use "deploy" or explicitly set "deploymentStage"');
        }
        return this.deploymentStage.urlForPath(path);
    }
    /**
     * API Gateway deployment that represents the latest changes of the API.
     * This resource will be automatically updated every time the REST API model changes.
     * This will be undefined if `deploy` is false.
     */
    get latestDeployment() {
        return this._latestDeployment;
    }
    /**
     * Defines an API Gateway domain name and maps it to this API.
     * @param id The construct id
     * @param options custom domain options
     */
    addDomainName(id, options) {
        const domainName = new domain_name_1.DomainName(this, id, {
            ...options,
            mapping: this,
        });
        if (!this._domainName) {
            this._domainName = domainName;
        }
        return domainName;
    }
    /**
     * Adds a usage plan.
     */
    addUsagePlan(id, props = {}) {
        return new usage_plan_1.UsagePlan(this, id, props);
    }
    /**
     * The first domain name mapped to this API, if defined through the `domainName`
     * configuration prop, or added via `addDomainName`
     */
    get domainName() {
        return this._domainName;
    }
    /**
     * Gets the "execute-api" ARN
     * @returns The "execute-api" ARN.
     * @default "*" returns the execute API ARN for all methods/resources in
     * this API.
     * @param method The method (default `*`)
     * @param path The resource path. Must start with '/' (default `*`)
     * @param stage The stage (default `*`)
     */
    arnForExecuteApi(method = '*', path = '/*', stage = '*') {
        if (!path.startsWith('/')) {
            throw new Error(`"path" must begin with a "/": '${path}'`);
        }
        if (method.toUpperCase() === 'ANY') {
            method = '*';
        }
        return core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: this.restApiId,
            sep: '/',
            resourceName: `${stage}/${method}${path}`,
        });
    }
    /**
     * Adds a new gateway response.
     */
    addGatewayResponse(id, options) {
        return new gateway_response_1.GatewayResponse(this, id, {
            restApi: this,
            ...options,
        });
    }
    configureCloudWatchRole(apiResource) {
        const role = new iam.Role(this, 'CloudWatchRole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonAPIGatewayPushToCloudWatchLogs')],
        });
        const resource = new apigateway_generated_1.CfnAccount(this, 'Account', {
            cloudWatchRoleArn: role.roleArn,
        });
        resource.node.addDependency(apiResource);
    }
    configureDeployment(props) {
        const deploy = props.deploy === undefined ? true : props.deploy;
        if (deploy) {
            this._latestDeployment = new deployment_1.Deployment(this, 'Deployment', {
                description: 'Automatically created by the RestApi construct',
                api: this,
                retainDeployments: props.retainDeployments,
            });
            // encode the stage name into the construct id, so if we change the stage name, it will recreate a new stage.
            // stage name is part of the endpoint, so that makes sense.
            const stageName = (props.deployOptions && props.deployOptions.stageName) || 'prod';
            this.deploymentStage = new stage_1.Stage(this, `DeploymentStage.${stageName}`, {
                deployment: this._latestDeployment,
                ...props.deployOptions,
            });
            new core_1.CfnOutput(this, 'Endpoint', { exportName: props.endpointExportName, value: this.urlForPath() });
        }
        else {
            if (props.deployOptions) {
                throw new Error('Cannot set \'deployOptions\' if \'deploy\' is disabled');
            }
        }
    }
}
/**
 * Represents a REST API in Amazon API Gateway, created with an OpenAPI specification.
 *
 * Some properties normally accessible on @see {@link RestApi} - such as the description -
 * must be declared in the specification. All Resources and Methods need to be defined as
 * part of the OpenAPI specification file, and cannot be added via the CDK.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 *
 * @experimental
 *
 * @resource AWS::ApiGateway::RestApi
 */
class SpecRestApi extends RestApiBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        const apiDefConfig = props.apiDefinition.bind(this);
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.physicalName,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            body: apiDefConfig.inlineDefinition ? apiDefConfig.inlineDefinition : undefined,
            bodyS3Location: apiDefConfig.inlineDefinition ? undefined : apiDefConfig.s3Location,
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        this.restApiRootResourceId = resource.attrRootResourceId;
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
    }
}
exports.SpecRestApi = SpecRestApi;
/**
 * Represents a REST API in Amazon API Gateway.
 *
 * Use `addResource` and `addMethod` to configure the API model.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 */
class RestApi extends RestApiBase {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        /**
         * The list of methods bound to this RestApi
         */
        this.methods = new Array();
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.physicalName,
            description: props.description,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            minimumCompressionSize: props.minimumCompressionSize,
            binaryMediaTypes: props.binaryMediaTypes,
            endpointConfiguration: this.configureEndpoints(props),
            apiKeySourceType: props.apiKeySourceType,
            cloneFrom: props.cloneFrom ? props.cloneFrom.restApiId : undefined,
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        this.root = new RootResource(this, props, resource.attrRootResourceId);
        this.restApiRootResourceId = resource.attrRootResourceId;
    }
    static fromRestApiId(scope, id, restApiId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.restApiId = restApiId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The deployed root URL of this REST API.
     */
    get url() {
        return this.urlForPath();
    }
    /**
     * Add an ApiKey
     */
    addApiKey(id, options) {
        return new api_key_1.ApiKey(this, id, {
            resources: [this],
            ...options,
        });
    }
    /**
     * Adds a new model.
     */
    addModel(id, props) {
        return new model_1.Model(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * Adds a new request validator.
     */
    addRequestValidator(id, props) {
        return new requestvalidator_1.RequestValidator(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        this.methods.push(method);
    }
    /**
     * Performs validation of the REST API.
     */
    validate() {
        if (this.methods.length === 0) {
            return ["The REST API doesn't contain any methods"];
        }
        return [];
    }
    configureEndpoints(props) {
        var _a, _b;
        if (props.endpointTypes && props.endpointConfiguration) {
            throw new Error('Only one of the RestApi props, endpointTypes or endpointConfiguration, is allowed');
        }
        if (props.endpointConfiguration) {
            return {
                types: props.endpointConfiguration.types,
                vpcEndpointIds: (_b = (_a = props.endpointConfiguration) === null || _a === void 0 ? void 0 : _a.vpcEndpoints) === null || _b === void 0 ? void 0 : _b.map(vpcEndpoint => vpcEndpoint.vpcEndpointId),
            };
        }
        if (props.endpointTypes) {
            return { types: props.endpointTypes };
        }
        return undefined;
    }
}
exports.RestApi = RestApi;
var ApiKeySourceType;
(function (ApiKeySourceType) {
    /**
     * To read the API key from the `X-API-Key` header of a request.
     */
    ApiKeySourceType["HEADER"] = "HEADER";
    /**
     * To read the API key from the `UsageIdentifierKey` from a custom authorizer.
     */
    ApiKeySourceType["AUTHORIZER"] = "AUTHORIZER";
})(ApiKeySourceType = exports.ApiKeySourceType || (exports.ApiKeySourceType = {}));
var EndpointType;
(function (EndpointType) {
    /**
     * For an edge-optimized API and its custom domain name.
     */
    EndpointType["EDGE"] = "EDGE";
    /**
     * For a regional API and its custom domain name.
     */
    EndpointType["REGIONAL"] = "REGIONAL";
    /**
     * For a private API and its custom domain name.
     */
    EndpointType["PRIVATE"] = "PRIVATE";
})(EndpointType = exports.EndpointType || (exports.EndpointType = {}));
class RootResource extends resource_1.ResourceBase {
    constructor(api, props, resourceId) {
        super(api, 'Default');
        this.parentResource = undefined;
        this.defaultIntegration = props.defaultIntegration;
        this.defaultMethodOptions = props.defaultMethodOptions;
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions;
        this.restApi = api;
        this.resourceId = resourceId;
        this.path = '/';
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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