"use strict";
// tslint:disable:object-literal-key-quotes
const assert_1 = require("@aws-cdk/assert");
const acm = require("@aws-cdk/aws-certificatemanager");
const core_1 = require("@aws-cdk/core");
const apigw = require("../lib");
module.exports = {
    'can define either an EDGE or REGIONAL domain name'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cert = new acm.Certificate(stack, 'Cert', { domainName: 'example.com' });
        // WHEN
        const regionalDomain = new apigw.DomainName(stack, 'my-domain', {
            domainName: 'example.com',
            certificate: cert,
            endpointType: apigw.EndpointType.REGIONAL,
        });
        const edgeDomain = new apigw.DomainName(stack, 'your-domain', {
            domainName: 'example.com',
            certificate: cert,
            endpointType: apigw.EndpointType.EDGE,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': 'example.com',
            'EndpointConfiguration': { 'Types': ['REGIONAL'] },
            'RegionalCertificateArn': { 'Ref': 'Cert5C9FAEC1' },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': 'example.com',
            'EndpointConfiguration': { 'Types': ['EDGE'] },
            'CertificateArn': { 'Ref': 'Cert5C9FAEC1' },
        }));
        test.deepEqual(stack.resolve(regionalDomain.domainNameAliasDomainName), { 'Fn::GetAtt': ['mydomain592C948B', 'RegionalDomainName'] });
        test.deepEqual(stack.resolve(regionalDomain.domainNameAliasHostedZoneId), { 'Fn::GetAtt': ['mydomain592C948B', 'RegionalHostedZoneId'] });
        test.deepEqual(stack.resolve(edgeDomain.domainNameAliasDomainName), { 'Fn::GetAtt': ['yourdomain5FE30C81', 'DistributionDomainName'] });
        test.deepEqual(stack.resolve(edgeDomain.domainNameAliasHostedZoneId), { 'Fn::GetAtt': ['yourdomain5FE30C81', 'DistributionHostedZoneId'] });
        test.done();
    },
    'default endpoint type is REGIONAL'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cert = new acm.Certificate(stack, 'Cert', { domainName: 'example.com' });
        // WHEN
        new apigw.DomainName(stack, 'my-domain', {
            domainName: 'example.com',
            certificate: cert,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': 'example.com',
            'EndpointConfiguration': { 'Types': ['REGIONAL'] },
            'RegionalCertificateArn': { 'Ref': 'Cert5C9FAEC1' },
        }));
        test.done();
    },
    'accepts different security policies'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cert = new acm.Certificate(stack, 'Cert', { domainName: 'example.com' });
        // WHEN
        new apigw.DomainName(stack, 'my-domain', {
            domainName: 'old.example.com',
            certificate: cert,
            securityPolicy: apigw.SecurityPolicy.TLS_1_0,
        });
        new apigw.DomainName(stack, 'your-domain', {
            domainName: 'new.example.com',
            certificate: cert,
            securityPolicy: apigw.SecurityPolicy.TLS_1_2,
        });
        new apigw.DomainName(stack, 'default-domain', {
            domainName: 'default.example.com',
            certificate: cert,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': 'old.example.com',
            'EndpointConfiguration': { 'Types': ['REGIONAL'] },
            'RegionalCertificateArn': { 'Ref': 'Cert5C9FAEC1' },
            'SecurityPolicy': 'TLS_1_0',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': 'new.example.com',
            'EndpointConfiguration': { 'Types': ['REGIONAL'] },
            'RegionalCertificateArn': { 'Ref': 'Cert5C9FAEC1' },
            'SecurityPolicy': 'TLS_1_2',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': 'default.example.com',
            'EndpointConfiguration': { 'Types': ['REGIONAL'] },
            'RegionalCertificateArn': { 'Ref': 'Cert5C9FAEC1' },
            'SecurityPolicy': assert_1.ABSENT,
        }));
        test.done();
    },
    '"mapping" can be used to automatically map this domain to the deployment stage of an API'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        api.root.addMethod('GET');
        // WHEN
        new apigw.DomainName(stack, 'Domain', {
            domainName: 'foo.com',
            certificate: acm.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d'),
            endpointType: apigw.EndpointType.EDGE,
            mapping: api,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            'DomainName': {
                'Ref': 'Domain66AC69E0',
            },
            'RestApiId': {
                'Ref': 'apiC8550315',
            },
            'Stage': {
                'Ref': 'apiDeploymentStageprod896C8101',
            },
        }));
        test.done();
    },
    '"addBasePathMapping" can be used to add base path mapping to the domain'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api1 = new apigw.RestApi(stack, 'api1');
        const api2 = new apigw.RestApi(stack, 'api2');
        const domain = new apigw.DomainName(stack, 'my-domain', {
            domainName: 'example.com',
            certificate: acm.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d'),
            endpointType: apigw.EndpointType.REGIONAL,
        });
        api1.root.addMethod('GET');
        api2.root.addMethod('GET');
        // WHEN
        domain.addBasePathMapping(api1, { basePath: 'api1' });
        domain.addBasePathMapping(api2, { basePath: 'api2' });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            'DomainName': {
                'Ref': 'mydomain592C948B',
            },
            'BasePath': 'api1',
            'RestApiId': {
                'Ref': 'api1A91238E2',
            },
            'Stage': {
                'Ref': 'api1DeploymentStageprod362746F6',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            'DomainName': {
                'Ref': 'mydomain592C948B',
            },
            'BasePath': 'api2',
            'RestApiId': {
                'Ref': 'api2C4850CEA',
            },
            'Stage': {
                'Ref': 'api2DeploymentStageprod4120D74E',
            },
        }));
        test.done();
    },
    'a domain name can be defined with the API'(test) {
        // GIVEN
        const domainName = 'my.domain.com';
        const stack = new core_1.Stack();
        const certificate = new acm.Certificate(stack, 'cert', { domainName: 'my.domain.com' });
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {
            domainName: { domainName, certificate },
        });
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': 'my.domain.com',
            'EndpointConfiguration': {
                'Types': [
                    'REGIONAL',
                ],
            },
            'RegionalCertificateArn': {
                'Ref': 'cert56CA94EB',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            'DomainName': {
                'Ref': 'apiCustomDomain64773C4F',
            },
            'RestApiId': {
                'Ref': 'apiC8550315',
            },
            'Stage': {
                'Ref': 'apiDeploymentStageprod896C8101',
            },
        }));
        test.done();
    },
    'a domain name can be added later'(test) {
        // GIVEN
        const domainName = 'my.domain.com';
        const stack = new core_1.Stack();
        const certificate = new acm.Certificate(stack, 'cert', { domainName: 'my.domain.com' });
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {});
        api.root.addMethod('GET');
        api.addDomainName('domainId', { domainName, certificate });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': domainName,
            'EndpointConfiguration': {
                'Types': [
                    'REGIONAL',
                ],
            },
            'RegionalCertificateArn': {
                'Ref': 'cert56CA94EB',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            'DomainName': {
                'Ref': 'apidomainId102F8DAA',
            },
            'RestApiId': {
                'Ref': 'apiC8550315',
            },
            'Stage': {
                'Ref': 'apiDeploymentStageprod896C8101',
            },
        }));
        test.done();
    },
    'multiple domain names can be added'(test) {
        // GIVEN
        const domainName = 'my.domain.com';
        const stack = new core_1.Stack();
        const certificate = new acm.Certificate(stack, 'cert', { domainName: 'my.domain.com' });
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {});
        api.root.addMethod('GET');
        const domainName1 = api.addDomainName('domainId', { domainName, certificate });
        api.addDomainName('domainId1', { domainName: 'your.domain.com', certificate });
        api.addDomainName('domainId2', { domainName: 'our.domain.com', certificate });
        test.deepEqual(api.domainName, domainName1);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': 'my.domain.com',
            'EndpointConfiguration': {
                'Types': [
                    'REGIONAL',
                ],
            },
            'RegionalCertificateArn': {
                'Ref': 'cert56CA94EB',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': 'your.domain.com',
            'EndpointConfiguration': {
                'Types': [
                    'REGIONAL',
                ],
            },
            'RegionalCertificateArn': {
                'Ref': 'cert56CA94EB',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': 'our.domain.com',
            'EndpointConfiguration': {
                'Types': [
                    'REGIONAL',
                ],
            },
            'RegionalCertificateArn': {
                'Ref': 'cert56CA94EB',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            'DomainName': {
                'Ref': 'apidomainId102F8DAA',
            },
            'RestApiId': {
                'Ref': 'apiC8550315',
            },
            'Stage': {
                'Ref': 'apiDeploymentStageprod896C8101',
            },
        }));
        test.done();
    },
    '"addBasePathMapping" can be used to add base path mapping to the domain with specific stage'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api1 = new apigw.RestApi(stack, 'api1');
        const api2 = new apigw.RestApi(stack, 'api2');
        const domain = new apigw.DomainName(stack, 'my-domain', {
            domainName: 'example.com',
            certificate: acm.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d'),
            endpointType: apigw.EndpointType.REGIONAL,
        });
        api1.root.addMethod('GET');
        api2.root.addMethod('GET');
        const testDeploy = new apigw.Deployment(stack, 'test-deployment', {
            api: api1,
        });
        const testStage = new apigw.Stage(stack, 'test-stage', {
            deployment: testDeploy,
        });
        // WHEN
        domain.addBasePathMapping(api1, { basePath: 'api1', stage: testStage });
        domain.addBasePathMapping(api2, { basePath: 'api2' });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            'DomainName': {
                'Ref': 'mydomain592C948B',
            },
            'BasePath': 'api1',
            'RestApiId': {
                'Ref': 'api1A91238E2',
            },
            'Stage': stack.resolve(testStage.stageName),
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            'DomainName': {
                'Ref': 'mydomain592C948B',
            },
            'BasePath': 'api2',
            'RestApiId': {
                'Ref': 'api2C4850CEA',
            },
            'Stage': {
                'Ref': 'api2DeploymentStageprod4120D74E',
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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