"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const apigateway = require("../lib");
const API_KEY_RESOURCE_TYPE = 'AWS::ApiGateway::ApiKey';
const USAGE_PLAN_RESOURCE_TYPE = 'AWS::ApiGateway::UsagePlan';
const USAGE_PLAN_KEY_RESOURCE_TYPE = 'AWS::ApiGateway::UsagePlanKey';
module.exports = {
    'default setup'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'my-api', { cloudWatchRole: false, deploy: false });
        api.root.addMethod('GET'); // Need at least one method on the api
        // WHEN
        new apigateway.RateLimitedApiKey(stack, 'my-api-key');
        // THEN
        // should have an api key with no props defined.
        assert_1.expect(stack).to(assert_1.haveResource(API_KEY_RESOURCE_TYPE, undefined, assert_1.ResourcePart.CompleteDefinition));
        // should not have a usage plan.
        assert_1.expect(stack).notTo(assert_1.haveResource(USAGE_PLAN_RESOURCE_TYPE));
        // should not have a usage plan key.
        assert_1.expect(stack).notTo(assert_1.haveResource(USAGE_PLAN_KEY_RESOURCE_TYPE));
        test.done();
    },
    'only api key is created when rate limiting properties are not provided'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: true, deployOptions: { stageName: 'test' } });
        api.root.addMethod('GET'); // api must have atleast one method.
        // WHEN
        new apigateway.RateLimitedApiKey(stack, 'test-api-key', {
            customerId: 'test-customer',
            resources: [api],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::ApiKey', {
            CustomerId: 'test-customer',
            StageKeys: [
                {
                    RestApiId: { Ref: 'testapiD6451F70' },
                    StageName: { Ref: 'testapiDeploymentStagetest5869DF71' },
                },
            ],
        }));
        // should not have a usage plan.
        assert_1.expect(stack).notTo(assert_1.haveResource(USAGE_PLAN_RESOURCE_TYPE));
        // should not have a usage plan key.
        assert_1.expect(stack).notTo(assert_1.haveResource(USAGE_PLAN_KEY_RESOURCE_TYPE));
        test.done();
    },
    'api key and usage plan are created and linked when rate limiting properties are provided'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: true, deployOptions: { stageName: 'test' } });
        api.root.addMethod('GET'); // api must have atleast one method.
        // WHEN
        new apigateway.RateLimitedApiKey(stack, 'test-api-key', {
            customerId: 'test-customer',
            resources: [api],
            quota: {
                limit: 10000,
                period: apigateway.Period.MONTH,
            },
        });
        // THEN
        // should have an api key
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::ApiKey', {
            CustomerId: 'test-customer',
            StageKeys: [
                {
                    RestApiId: { Ref: 'testapiD6451F70' },
                    StageName: { Ref: 'testapiDeploymentStagetest5869DF71' },
                },
            ],
        }));
        // should have a usage plan with specified quota.
        assert_1.expect(stack).to(assert_1.haveResource(USAGE_PLAN_RESOURCE_TYPE, {
            Quota: {
                Limit: 10000,
                Period: 'MONTH',
            },
        }, assert_1.ResourcePart.Properties));
        // should have a usage plan key linking the api key and usage plan
        assert_1.expect(stack).to(assert_1.haveResource(USAGE_PLAN_KEY_RESOURCE_TYPE, {
            KeyId: {
                Ref: 'testapikey998028B6',
            },
            KeyType: 'API_KEY',
            UsagePlanId: {
                Ref: 'testapikeyUsagePlanResource66DB63D6',
            },
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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