"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Origin = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const http_origin_1 = require("./http-origin");
/**
 * An Origin that is backed by an S3 bucket.
 *
 * If the bucket is configured for website hosting, this origin will be configured to use the bucket as an
 * HTTP server origin and will use the bucket's configured website redirects and error handling. Otherwise,
 * the origin is created as a bucket origin and will use CloudFront's redirect and error handling.
 */
class S3Origin {
    constructor(bucket, props = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_cloudfront_origins_S3OriginProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.origin = bucket.isWebsite ?
            new http_origin_1.HttpOrigin(bucket.bucketWebsiteDomainName, {
                protocolPolicy: cloudfront.OriginProtocolPolicy.HTTP_ONLY,
                ...props,
            }) :
            new S3BucketOrigin(bucket, props);
    }
    bind(scope, options) {
        return this.origin.bind(scope, options);
    }
}
exports.S3Origin = S3Origin;
_a = JSII_RTTI_SYMBOL_1;
S3Origin[_a] = { fqn: "@aws-cdk/aws-cloudfront-origins.S3Origin", version: "1.154.0" };
/**
 * An Origin specific to a S3 bucket (not configured for website hosting).
 *
 * Contains additional logic around bucket permissions and origin access identities.
 */
class S3BucketOrigin extends cloudfront.OriginBase {
    constructor(bucket, { originAccessIdentity, ...props }) {
        super(bucket.bucketRegionalDomainName, props);
        this.bucket = bucket;
        if (originAccessIdentity) {
            this.originAccessIdentity = originAccessIdentity;
        }
    }
    bind(scope, options) {
        if (!this.originAccessIdentity) {
            // Using a bucket from another stack creates a cyclic reference with
            // the bucket taking a dependency on the generated S3CanonicalUserId for the grant principal,
            // and the distribution having a dependency on the bucket's domain name.
            // Fix this by parenting the OAI in the bucket's stack when cross-stack usage is detected.
            const bucketStack = cdk.Stack.of(this.bucket);
            const bucketInDifferentStack = bucketStack !== cdk.Stack.of(scope);
            const oaiScope = bucketInDifferentStack ? bucketStack : scope;
            const oaiId = bucketInDifferentStack ? `${cdk.Names.uniqueId(scope)}S3Origin` : 'S3Origin';
            this.originAccessIdentity = new cloudfront.OriginAccessIdentity(oaiScope, oaiId, {
                comment: `Identity for ${options.originId}`,
            });
        }
        // Used rather than `grantRead` because `grantRead` will grant overly-permissive policies.
        // Only GetObject is needed to retrieve objects for the distribution.
        // This also excludes KMS permissions; currently, OAI only supports SSE-S3 for buckets.
        // Source: https://aws.amazon.com/blogs/networking-and-content-delivery/serving-sse-kms-encrypted-content-from-s3-using-cloudfront/
        this.bucket.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.bucket.arnForObjects('*')],
            actions: ['s3:GetObject'],
            principals: [this.originAccessIdentity.grantPrincipal],
        }));
        return super.bind(scope, options);
    }
    renderS3OriginConfig() {
        return { originAccessIdentity: `origin-access-identity/cloudfront/${this.originAccessIdentity.originAccessIdentityName}` };
    }
}
//# sourceMappingURL=data:application/json;base64,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