"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const have_resource_1 = require("@aws-cdk/assert/lib/assertions/have-resource");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('User Pool', () => {
    test('default setup', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool');
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPool', {
            AdminCreateUserConfig: {
                AllowAdminCreateUserOnly: true,
                InviteMessageTemplate: have_resource_1.ABSENT,
            },
            EmailVerificationMessage: 'The verification code to your new account is {####}',
            EmailVerificationSubject: 'Verify your new account',
            SmsVerificationMessage: 'The verification code to your new account is {####}',
            VerificationMessageTemplate: {
                DefaultEmailOption: 'CONFIRM_WITH_CODE',
                EmailMessage: 'The verification code to your new account is {####}',
                EmailSubject: 'Verify your new account',
                SmsMessage: 'The verification code to your new account is {####}',
            },
            SmsConfiguration: have_resource_1.ABSENT,
            lambdaTriggers: have_resource_1.ABSENT,
        });
    });
    test('self sign up option is correctly configured', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            selfSignUpEnabled: true,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPool', {
            AdminCreateUserConfig: {
                AllowAdminCreateUserOnly: false,
            },
        });
    });
    test('email verification via link is configured correctly', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            userVerification: {
                emailStyle: lib_1.VerificationEmailStyle.LINK,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            EmailVerificationMessage: have_resource_1.ABSENT,
            EmailVerificationSubject: have_resource_1.ABSENT,
            SmsVerificationMessage: 'The verification code to your new account is {####}',
            VerificationMessageTemplate: {
                DefaultEmailOption: 'CONFIRM_WITH_LINK',
                EmailMessageByLink: 'Verify your account by clicking on {##Verify Email##}',
                EmailSubjectByLink: 'Verify your new account',
                SmsMessage: 'The verification code to your new account is {####}',
            },
        });
    }),
        test('email and sms verification messages are validated', () => {
            const stack = new core_1.Stack();
            expect(() => new lib_1.UserPool(stack, 'Pool1', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.CODE,
                    emailBody: 'invalid email body',
                },
            })).toThrow(/Verification email body/);
            expect(() => new lib_1.UserPool(stack, 'Pool2', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.CODE,
                    emailBody: 'valid email body {####}',
                },
            })).not.toThrow();
            expect(() => new lib_1.UserPool(stack, 'Pool3', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.CODE,
                    smsMessage: 'invalid sms message',
                },
            })).toThrow(/SMS message/);
            expect(() => new lib_1.UserPool(stack, 'Pool4', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.CODE,
                    smsMessage: 'invalid sms message {####}',
                },
            })).not.toThrow();
            expect(() => new lib_1.UserPool(stack, 'Pool5', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.LINK,
                    emailBody: 'invalid email body {####}',
                },
            })).toThrow(/Verification email body/);
            expect(() => new lib_1.UserPool(stack, 'Pool6', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.LINK,
                    emailBody: 'invalid email body {##Verify Email##}',
                },
            })).not.toThrow();
        });
    test('validation is skipped for email and sms messages when tokens', () => {
        const stack = new core_1.Stack();
        const parameter = new core_1.CfnParameter(stack, 'Parameter');
        expect(() => new lib_1.UserPool(stack, 'Pool1', {
            userVerification: {
                emailStyle: lib_1.VerificationEmailStyle.CODE,
                emailBody: parameter.valueAsString,
            },
        })).not.toThrow();
        expect(() => new lib_1.UserPool(stack, 'Pool2', {
            userVerification: {
                emailStyle: lib_1.VerificationEmailStyle.CODE,
                smsMessage: parameter.valueAsString,
            },
        })).not.toThrow();
    });
    test('user invitation messages are configured correctly', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            userInvitation: {
                emailBody: 'invitation email body',
                emailSubject: 'invitation email subject',
                smsMessage: 'invitation sms',
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            AdminCreateUserConfig: {
                InviteMessageTemplate: {
                    EmailMessage: 'invitation email body',
                    EmailSubject: 'invitation email subject',
                    SMSMessage: 'invitation sms',
                },
            },
        });
    });
    test('smsRole property is recognized', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const role = aws_iam_1.Role.fromRoleArn(stack, 'smsRole', 'arn:aws:iam::664773442901:role/sms-role');
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            smsRole: role,
            smsRoleExternalId: 'test-external-id',
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            SmsConfiguration: {
                ExternalId: 'test-external-id',
                SnsCallerArn: role.roleArn,
            },
        });
    });
    test('import using id', () => {
        // GIVEN
        const stack = new core_1.Stack(undefined, undefined, {
            env: { region: 'some-region-1', account: '0123456789012' },
        });
        const userPoolId = 'test-user-pool';
        // WHEN
        const pool = lib_1.UserPool.fromUserPoolId(stack, 'userpool', userPoolId);
        expect(pool.userPoolId).toEqual(userPoolId);
        expect(pool.userPoolArn).toMatch(/cognito-idp:some-region-1:0123456789012:userpool\/test-user-pool/);
    });
    test('import using arn', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const userPoolArn = 'arn:aws:cognito-idp:us-east-1:0123456789012:userpool/test-user-pool';
        // WHEN
        const pool = lib_1.UserPool.fromUserPoolArn(stack, 'userpool', userPoolArn);
        expect(pool.userPoolId).toEqual('test-user-pool');
        expect(stack.resolve(pool.userPoolArn)).toEqual({
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':cognito-idp:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':userpool/test-user-pool',
                ]],
        });
    });
    test('support tags', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const pool = new lib_1.UserPool(stack, 'Pool', {
            userPoolName: 'myPool',
        });
        core_1.Tags.of(pool).add('PoolTag', 'PoolParty');
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'myPool',
            UserPoolTags: {
                PoolTag: 'PoolParty',
            },
        });
    });
    test('lambda triggers via properties are correctly configured', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const fn = fooFunction(stack, 'preSignUp');
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            lambdaTriggers: {
                preSignUp: fn,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            LambdaConfig: {
                PreSignUp: stack.resolve(fn.functionArn),
            },
        });
        expect(stack).toHaveResourceLike('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            FunctionName: stack.resolve(fn.functionArn),
            Principal: 'cognito-idp.amazonaws.com',
        });
    });
    test('add* API correctly appends triggers', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const createAuthChallenge = fooFunction(stack, 'createAuthChallenge');
        const customMessage = fooFunction(stack, 'customMessage');
        const defineAuthChallenge = fooFunction(stack, 'defineAuthChallenge');
        const postAuthentication = fooFunction(stack, 'postAuthentication');
        const postConfirmation = fooFunction(stack, 'postConfirmation');
        const preAuthentication = fooFunction(stack, 'preAuthentication');
        const preSignUp = fooFunction(stack, 'preSignUp');
        const preTokenGeneration = fooFunction(stack, 'preTokenGeneration');
        const userMigration = fooFunction(stack, 'userMigration');
        const verifyAuthChallengeResponse = fooFunction(stack, 'verifyAuthChallengeResponse');
        // WHEN
        const pool = new lib_1.UserPool(stack, 'Pool');
        pool.addTrigger(lib_1.UserPoolOperation.CREATE_AUTH_CHALLENGE, createAuthChallenge);
        pool.addTrigger(lib_1.UserPoolOperation.CUSTOM_MESSAGE, customMessage);
        pool.addTrigger(lib_1.UserPoolOperation.DEFINE_AUTH_CHALLENGE, defineAuthChallenge);
        pool.addTrigger(lib_1.UserPoolOperation.POST_AUTHENTICATION, postAuthentication);
        pool.addTrigger(lib_1.UserPoolOperation.POST_CONFIRMATION, postConfirmation);
        pool.addTrigger(lib_1.UserPoolOperation.PRE_AUTHENTICATION, preAuthentication);
        pool.addTrigger(lib_1.UserPoolOperation.PRE_SIGN_UP, preSignUp);
        pool.addTrigger(lib_1.UserPoolOperation.PRE_TOKEN_GENERATION, preTokenGeneration);
        pool.addTrigger(lib_1.UserPoolOperation.USER_MIGRATION, userMigration);
        pool.addTrigger(lib_1.UserPoolOperation.VERIFY_AUTH_CHALLENGE_RESPONSE, verifyAuthChallengeResponse);
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            LambdaConfig: {
                CreateAuthChallenge: stack.resolve(createAuthChallenge.functionArn),
                CustomMessage: stack.resolve(customMessage.functionArn),
                DefineAuthChallenge: stack.resolve(defineAuthChallenge.functionArn),
                PostAuthentication: stack.resolve(postAuthentication.functionArn),
                PostConfirmation: stack.resolve(postConfirmation.functionArn),
                PreAuthentication: stack.resolve(preAuthentication.functionArn),
                PreSignUp: stack.resolve(preSignUp.functionArn),
                PreTokenGeneration: stack.resolve(preTokenGeneration.functionArn),
                UserMigration: stack.resolve(userMigration.functionArn),
                VerifyAuthChallengeResponse: stack.resolve(verifyAuthChallengeResponse.functionArn),
            },
        });
        [createAuthChallenge, customMessage, defineAuthChallenge, postAuthentication,
            postConfirmation, preAuthentication, preSignUp, preTokenGeneration, userMigration,
            verifyAuthChallengeResponse].forEach((fn) => {
            expect(stack).toHaveResourceLike('AWS::Lambda::Permission', {
                Action: 'lambda:InvokeFunction',
                FunctionName: stack.resolve(fn.functionArn),
                Principal: 'cognito-idp.amazonaws.com',
            });
        });
    });
    test('fails when the same trigger is added twice', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const userpool = new lib_1.UserPool(stack, 'Pool');
        const fn1 = new lambda.Function(stack, 'fn1', {
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
        });
        const fn2 = new lambda.Function(stack, 'fn2', {
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
        });
        // WHEN
        userpool.addTrigger(lib_1.UserPoolOperation.CREATE_AUTH_CHALLENGE, fn1);
        expect(() => userpool.addTrigger(lib_1.UserPoolOperation.CREATE_AUTH_CHALLENGE, fn2)).toThrow(/already exists/);
    });
    test('no username aliases specified', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool');
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameAttributes: have_resource_1.ABSENT,
            AliasAttributes: have_resource_1.ABSENT,
        });
    });
    test('fails when preferredUsername is used without username', () => {
        const stack = new core_1.Stack();
        expect(() => new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { preferredUsername: true },
        })).toThrow(/username/);
    });
    test('username and email are specified as the username aliases', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { username: true, email: true },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameAttributes: have_resource_1.ABSENT,
            AliasAttributes: ['email'],
        });
    });
    test('email and phone number are specified as the username aliases', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { email: true, phone: true },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameAttributes: ['email', 'phone_number'],
            AliasAttributes: have_resource_1.ABSENT,
        });
    });
    test('email and phone number are autoverified, by default, if they are specified as signIn', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool1', {
            userPoolName: 'Pool1',
            signInAliases: { email: true },
        });
        new lib_1.UserPool(stack, 'Pool2', {
            userPoolName: 'Pool2',
            signInAliases: { email: true, phone: true },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool1',
            AutoVerifiedAttributes: ['email'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool2',
            AutoVerifiedAttributes: ['email', 'phone_number'],
        });
    });
    test('explicit autoverify are correctly picked up', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { username: true },
            autoVerify: { email: true, phone: true },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            AutoVerifiedAttributes: ['email', 'phone_number'],
        });
    });
    test('sign in case sensitive is correctly picked up', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInCaseSensitive: false,
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameConfiguration: {
                CaseSensitive: false,
            },
        });
    });
    test('sign in case sensitive is absent by default', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {});
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameConfiguration: have_resource_1.ABSENT,
        });
    });
    test('standard attributes default to mutable', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            standardAttributes: {
                fullname: {
                    required: true,
                },
                timezone: {
                    required: true,
                },
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Schema: [
                {
                    Name: 'name',
                    Required: true,
                    Mutable: true,
                },
                {
                    Name: 'zoneinfo',
                    Required: true,
                    Mutable: true,
                },
            ],
        });
    });
    test('mutable standard attributes', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            userPoolName: 'Pool',
            standardAttributes: {
                fullname: {
                    required: true,
                    mutable: true,
                },
                timezone: {
                    required: true,
                    mutable: true,
                },
            },
        });
        new lib_1.UserPool(stack, 'Pool1', {
            userPoolName: 'Pool1',
            standardAttributes: {
                fullname: {
                    mutable: false,
                },
                timezone: {
                    mutable: false,
                },
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool',
            Schema: [
                {
                    Mutable: true,
                    Name: 'name',
                    Required: true,
                },
                {
                    Mutable: true,
                    Name: 'zoneinfo',
                    Required: true,
                },
            ],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool1',
            Schema: [
                {
                    Name: 'name',
                    Required: false,
                    Mutable: false,
                },
                {
                    Name: 'zoneinfo',
                    Required: false,
                    Mutable: false,
                },
            ],
        });
    });
    test('schema is absent when attributes are not specified', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', { userPoolName: 'Pool' });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool',
            Schema: have_resource_1.ABSENT,
        });
    });
    test('optional mutable standardAttributes', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            userPoolName: 'Pool',
            standardAttributes: {
                timezone: {
                    mutable: true,
                },
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool',
            Schema: [
                {
                    Mutable: true,
                    Required: false,
                    Name: 'zoneinfo',
                },
            ],
        });
    });
    test('custom attributes with default constraints', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            customAttributes: {
                'custom-string-attr': new lib_1.StringAttribute(),
                'custom-number-attr': new lib_1.NumberAttribute(),
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Schema: [
                {
                    Name: 'custom-string-attr',
                    AttributeDataType: 'String',
                    StringAttributeConstraints: have_resource_1.ABSENT,
                    NumberAttributeConstraints: have_resource_1.ABSENT,
                },
                {
                    Name: 'custom-number-attr',
                    AttributeDataType: 'Number',
                    StringAttributeConstraints: have_resource_1.ABSENT,
                    NumberAttributeConstraints: have_resource_1.ABSENT,
                },
            ],
        });
    });
    test('custom attributes with constraints', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            customAttributes: {
                'custom-string-attr': new lib_1.StringAttribute({ minLen: 5, maxLen: 50 }),
                'custom-number-attr': new lib_1.NumberAttribute({ min: 500, max: 2000 }),
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Schema: [
                {
                    AttributeDataType: 'String',
                    Name: 'custom-string-attr',
                    StringAttributeConstraints: {
                        MaxLength: '50',
                        MinLength: '5',
                    },
                },
                {
                    AttributeDataType: 'Number',
                    Name: 'custom-number-attr',
                    NumberAttributeConstraints: {
                        MaxValue: '2000',
                        MinValue: '500',
                    },
                },
            ],
        });
    });
    test('mfaTypes is ignored when mfaEnforcement is undefined or set to OFF', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool1', {
            userPoolName: 'Pool1',
            mfaSecondFactor: {
                sms: true,
                otp: true,
            },
        });
        new lib_1.UserPool(stack, 'Pool2', {
            userPoolName: 'Pool2',
            mfa: lib_1.Mfa.OFF,
            mfaSecondFactor: {
                sms: true,
                otp: true,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool1',
            MfaConfiguration: have_resource_1.ABSENT,
            EnabledMfas: have_resource_1.ABSENT,
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool2',
            MfaConfiguration: 'OFF',
            EnabledMfas: have_resource_1.ABSENT,
        });
    });
    test('sms mfa type is the default when mfaEnforcement is set to REQUIRED or OPTIONAL', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool1', {
            userPoolName: 'Pool1',
            mfa: lib_1.Mfa.OPTIONAL,
        });
        new lib_1.UserPool(stack, 'Pool2', {
            userPoolName: 'Pool2',
            mfa: lib_1.Mfa.REQUIRED,
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool1',
            MfaConfiguration: 'OPTIONAL',
            EnabledMfas: ['SMS_MFA'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool2',
            MfaConfiguration: 'ON',
            EnabledMfas: ['SMS_MFA'],
        });
    });
    test('mfa type is correctly picked up when specified', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            mfa: lib_1.Mfa.REQUIRED,
            mfaSecondFactor: {
                sms: true,
                otp: true,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            EnabledMfas: ['SMS_MFA', 'SOFTWARE_TOKEN_MFA'],
        });
    });
    test('password policy is correctly set', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            passwordPolicy: {
                tempPasswordValidity: core_1.Duration.days(2),
                minLength: 15,
                requireDigits: true,
                requireLowercase: true,
                requireUppercase: true,
                requireSymbols: true,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Policies: {
                PasswordPolicy: {
                    TemporaryPasswordValidityDays: 2,
                    MinimumLength: 15,
                    RequireLowercase: true,
                    RequireUppercase: true,
                    RequireNumbers: true,
                    RequireSymbols: true,
                },
            },
        });
    });
    test('password minimum length is set to the default when other parts of the policy is configured', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            passwordPolicy: {
                tempPasswordValidity: core_1.Duration.days(2),
                requireDigits: true,
            },
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Policies: {
                PasswordPolicy: {
                    MinimumLength: 8,
                },
            },
        });
    });
    test('throws when tempPassword validity is not in round days', () => {
        const stack = new core_1.Stack();
        expect(() => new lib_1.UserPool(stack, 'Pool', {
            passwordPolicy: {
                tempPasswordValidity: core_1.Duration.hours(30),
            },
        })).toThrow();
    });
    test('temp password throws an error when above the max', () => {
        const stack = new core_1.Stack();
        expect(() => new lib_1.UserPool(stack, 'Pool', {
            passwordPolicy: {
                tempPasswordValidity: core_1.Duration.days(400),
            },
        })).toThrow(/tempPasswordValidity cannot be greater than/);
    });
    test('throws when minLength is out of range', () => {
        const stack = new core_1.Stack();
        expect(() => new lib_1.UserPool(stack, 'Pool1', {
            passwordPolicy: {
                minLength: 5,
            },
        })).toThrow(/minLength for password must be between/);
        expect(() => new lib_1.UserPool(stack, 'Pool2', {
            passwordPolicy: {
                minLength: 100,
            },
        })).toThrow(/minLength for password must be between/);
    });
    test('email transmission settings are recognized correctly', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            emailSettings: {
                from: 'from@myawesomeapp.com',
                replyTo: 'replyTo@myawesomeapp.com',
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            EmailConfiguration: {
                From: 'from@myawesomeapp.com',
                ReplyToEmailAddress: 'replyTo@myawesomeapp.com',
            },
        });
    });
    test('addClient', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const userpool = new lib_1.UserPool(stack, 'Pool');
        userpool.addClient('UserPoolClient', {
            userPoolClientName: 'userpoolclient',
        });
        const imported = lib_1.UserPool.fromUserPoolId(stack, 'imported', 'imported-userpool-id');
        imported.addClient('UserPoolImportedClient', {
            userPoolClientName: 'userpoolimportedclient',
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ClientName: 'userpoolclient',
            UserPoolId: stack.resolve(userpool.userPoolId),
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ClientName: 'userpoolimportedclient',
            UserPoolId: stack.resolve(imported.userPoolId),
        });
    });
    test('addDomain', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const userpool = new lib_1.UserPool(stack, 'Pool');
        userpool.addDomain('UserPoolDomain', {
            cognitoDomain: {
                domainPrefix: 'userpooldomain',
            },
        });
        const imported = lib_1.UserPool.fromUserPoolId(stack, 'imported', 'imported-userpool-id');
        imported.addDomain('UserPoolImportedDomain', {
            cognitoDomain: {
                domainPrefix: 'userpoolimporteddomain',
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolDomain', {
            Domain: 'userpooldomain',
            UserPoolId: stack.resolve(userpool.userPoolId),
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolDomain', {
            Domain: 'userpoolimporteddomain',
            UserPoolId: stack.resolve(imported.userPoolId),
        });
    });
    test('registered identity providers', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const userPool = new lib_1.UserPool(stack, 'pool');
        const provider1 = lib_1.UserPoolIdentityProvider.fromProviderName(stack, 'provider1', 'provider1');
        const provider2 = lib_1.UserPoolIdentityProvider.fromProviderName(stack, 'provider2', 'provider2');
        // WHEN
        userPool.registerIdentityProvider(provider1);
        userPool.registerIdentityProvider(provider2);
        // THEN
        expect(userPool.identityProviders).toEqual([provider1, provider2]);
    });
    describe('AccountRecoverySetting should be configured correctly', () => {
        test('EMAIL_AND_PHONE_WITHOUT_MFA', () => {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.UserPool(stack, 'pool', { accountRecovery: lib_1.AccountRecovery.EMAIL_AND_PHONE_WITHOUT_MFA });
            // THEN
            expect(stack).toHaveResource('AWS::Cognito::UserPool', {
                AccountRecoverySetting: {
                    RecoveryMechanisms: [
                        { Name: 'verified_email', Priority: 1 },
                        { Name: 'verified_phone_number', Priority: 2 },
                    ],
                },
            });
        });
        test('PHONE_WITHOUT_MFA_AND_EMAIL', () => {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.UserPool(stack, 'pool', { accountRecovery: lib_1.AccountRecovery.PHONE_WITHOUT_MFA_AND_EMAIL });
            // THEN
            expect(stack).toHaveResource('AWS::Cognito::UserPool', {
                AccountRecoverySetting: {
                    RecoveryMechanisms: [
                        { Name: 'verified_phone_number', Priority: 1 },
                        { Name: 'verified_email', Priority: 2 },
                    ],
                },
            });
        });
        test('EMAIL_ONLY', () => {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.UserPool(stack, 'pool', { accountRecovery: lib_1.AccountRecovery.EMAIL_ONLY });
            // THEN
            expect(stack).toHaveResource('AWS::Cognito::UserPool', {
                AccountRecoverySetting: {
                    RecoveryMechanisms: [
                        { Name: 'verified_email', Priority: 1 },
                    ],
                },
            });
        });
        test('PHONE_ONLY_WITHOUT_MFA', () => {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.UserPool(stack, 'pool', { accountRecovery: lib_1.AccountRecovery.PHONE_ONLY_WITHOUT_MFA });
            // THEN
            expect(stack).toHaveResource('AWS::Cognito::UserPool', {
                AccountRecoverySetting: {
                    RecoveryMechanisms: [
                        { Name: 'verified_phone_number', Priority: 1 },
                    ],
                },
            });
        });
        test('NONE', () => {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.UserPool(stack, 'pool', { accountRecovery: lib_1.AccountRecovery.NONE });
            // THEN
            expect(stack).toHaveResource('AWS::Cognito::UserPool', {
                AccountRecoverySetting: {
                    RecoveryMechanisms: [
                        { Name: 'admin_only', Priority: 1 },
                    ],
                },
            });
        });
        test('PHONE_AND_EMAIL', () => {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.UserPool(stack, 'pool', { accountRecovery: lib_1.AccountRecovery.PHONE_AND_EMAIL });
            // THEN
            expect(stack).toHaveResource('AWS::Cognito::UserPool', {
                AccountRecoverySetting: have_resource_1.ABSENT,
            });
        });
        test('default', () => {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.UserPool(stack, 'pool');
            // THEN
            expect(stack).toHaveResource('AWS::Cognito::UserPool', {
                AccountRecoverySetting: {
                    RecoveryMechanisms: [
                        { Name: 'verified_phone_number', Priority: 1 },
                        { Name: 'verified_email', Priority: 2 },
                    ],
                },
            });
        });
    });
    describe('sms roles', () => {
        test('default', () => {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.UserPool(stack, 'pool');
            // THEN
            expect(stack).toHaveResource('AWS::Cognito::UserPool', {
                SmsConfiguration: have_resource_1.ABSENT,
            });
        });
        test('smsRole and smsExternalId is set', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const smsRole = new aws_iam_1.Role(stack, 'smsRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('service.amazonaws.com'),
            });
            // WHEN
            new lib_1.UserPool(stack, 'pool', {
                smsRole,
                smsRoleExternalId: 'role-external-id',
            });
            // THEN
            expect(stack).toHaveResource('AWS::Cognito::UserPool', {
                SmsConfiguration: {
                    ExternalId: 'role-external-id',
                    SnsCallerArn: { 'Fn::GetAtt': ['smsRoleA4587CE8', 'Arn'] },
                },
            });
        });
        test('setting enableSmsRole creates an sms role', () => {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.UserPool(stack, 'pool', {
                enableSmsRole: true,
            });
            // THEN
            expect(stack).toHaveResource('AWS::Cognito::UserPool', {
                SmsConfiguration: {
                    ExternalId: 'pool',
                    SnsCallerArn: { 'Fn::GetAtt': ['poolsmsRole04048F13', 'Arn'] },
                },
            });
            expect(stack).toHaveResource('AWS::IAM::Role', {
                AssumeRolePolicyDocument: {
                    Statement: [
                        {
                            Action: 'sts:AssumeRole',
                            Condition: {
                                StringEquals: {
                                    'sts:ExternalId': 'pool',
                                },
                            },
                            Effect: 'Allow',
                            Principal: {
                                Service: 'cognito-idp.amazonaws.com',
                            },
                        },
                    ],
                    Version: '2012-10-17',
                },
                Policies: [
                    {
                        PolicyDocument: {
                            Statement: [
                                {
                                    Action: 'sns:Publish',
                                    Effect: 'Allow',
                                    Resource: '*',
                                },
                            ],
                            Version: '2012-10-17',
                        },
                        PolicyName: 'sns-publish',
                    },
                ],
            });
        });
        test('auto sms role is not created when MFA and phoneVerification is off', () => {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.UserPool(stack, 'pool', {
                mfa: lib_1.Mfa.OFF,
                signInAliases: {
                    phone: false,
                },
            });
            // THEN
            expect(stack).toHaveResource('AWS::Cognito::UserPool', {
                SmsConfiguration: have_resource_1.ABSENT,
            });
        });
        test('auto sms role is not created when OTP-based MFA is enabled and phoneVerification is off', () => {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.UserPool(stack, 'pool', {
                mfa: lib_1.Mfa.REQUIRED,
                mfaSecondFactor: {
                    otp: true,
                    sms: false,
                },
                signInAliases: {
                    phone: false,
                },
            });
            // THEN
            expect(stack).toHaveResource('AWS::Cognito::UserPool', {
                SmsConfiguration: have_resource_1.ABSENT,
            });
        });
        test('auto sms role is created when phone verification is turned on', () => {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.UserPool(stack, 'pool', {
                mfa: lib_1.Mfa.OFF,
                signInAliases: { phone: true },
            });
            // THEN
            expect(stack).toHaveResource('AWS::Cognito::UserPool', {
                SmsConfiguration: {
                    ExternalId: 'pool',
                    SnsCallerArn: { 'Fn::GetAtt': ['poolsmsRole04048F13', 'Arn'] },
                },
            });
        });
        test('auto sms role is created when phone auto-verification is set', () => {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.UserPool(stack, 'pool', {
                mfa: lib_1.Mfa.OFF,
                signInAliases: { phone: false },
                autoVerify: { phone: true },
            });
            // THEN
            expect(stack).toHaveResource('AWS::Cognito::UserPool', {
                SmsConfiguration: {
                    ExternalId: 'pool',
                    SnsCallerArn: { 'Fn::GetAtt': ['poolsmsRole04048F13', 'Arn'] },
                },
            });
        });
        test('auto sms role is created when MFA is turned on', () => {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.UserPool(stack, 'pool', {
                mfa: lib_1.Mfa.REQUIRED,
                mfaSecondFactor: {
                    sms: true,
                    otp: false,
                },
                signInAliases: {
                    phone: false,
                },
            });
            // THEN
            expect(stack).toHaveResource('AWS::Cognito::UserPool', {
                SmsConfiguration: {
                    ExternalId: 'pool',
                    SnsCallerArn: { 'Fn::GetAtt': ['poolsmsRole04048F13', 'Arn'] },
                },
            });
        });
        test('auto sms role is not created when enableSmsRole is unset, even when MFA is configured', () => {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.UserPool(stack, 'pool', {
                mfa: lib_1.Mfa.REQUIRED,
                mfaSecondFactor: {
                    sms: true,
                    otp: false,
                },
                enableSmsRole: false,
            });
            // THEN
            expect(stack).toHaveResource('AWS::Cognito::UserPool', {
                SmsConfiguration: have_resource_1.ABSENT,
            });
        });
        test('throws an error when smsRole is specified but enableSmsRole is unset', () => {
            const stack = new core_1.Stack();
            const smsRole = new aws_iam_1.Role(stack, 'smsRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('service.amazonaws.com'),
            });
            expect(() => new lib_1.UserPool(stack, 'pool', {
                smsRole,
                enableSmsRole: false,
            })).toThrow(/enableSmsRole cannot be disabled/);
        });
    });
});
function fooFunction(scope, name) {
    return new lambda.Function(scope, name, {
        functionName: name,
        code: lambda.Code.inline('foo'),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
    });
}
//# sourceMappingURL=data:application/json;base64,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