from __future__ import absolute_import
from __future__ import print_function
from future.utils import python_2_unicode_compatible

import requests
import binascii
import json
from datetime import datetime, date
import logging
import uuid
import hashlib
import base64
import os
from datetime import datetime

from .elements import Value
from .query import Query
from . import reports
from . import utils

@python_2_unicode_compatible
class Account(object):
    """ A wrapper for the Adobe Analytics API. Allows you to query the reporting API """
    DEFAULT_ENDPOINT = 'https://api.omniture.com/admin/1.4/rest/'

    def __init__(self, username, secret, endpoint=DEFAULT_ENDPOINT, cache=False, cache_key=None):
        """Authentication to make requests."""
        self.log = logging.getLogger(__name__)
        self.log.info(datetime.now().strftime("%Y-%m-%d %I%p:%M:%S"))
        self.username = username
        self.secret = secret
        self.endpoint = endpoint
        #Allow someone to set a custom cache key
        self.cache = cache
        if cache_key:
            self.cache_key = cache_key
        else:
            self.cache_key = date.today().isoformat()
        if self.cache:
            data = self.request_cached('Company', 'GetReportSuites')['report_suites']
        else:
            data = self.request('Company', 'GetReportSuites')['report_suites']
        suites = [Suite(suite['site_title'], suite['rsid'], self) for suite in data]
        self.suites = utils.AddressableList(suites)

    def request_cached(self, api, method, query={}, cache_key=None):
        if cache_key:
            key = cache_key
        else:
            key = self.cache_key

        #Generate a shortened hash of the query string so that method don't collide
        query_hash = base64.urlsafe_b64encode(hashlib.md5(query).digest())

        try:
            with open(self.file_path+'/data_'+api+'_'+method+'_'+query_hash+'_'+key+'.txt') as fp:
                for line in fp:
                    if line:
                        data = ast.literal_eval(line)

        except IOError as e:
            data = self.request(api, method, query)

            # Capture all other old text files
            #TODO decide if the query should be included in the file list to be cleared out when the cache key changes
            filelist = [f for f in os.listdir(self.file_path) if f.startswith('data_'+api+'_'+method)]

            # Delete them
            for f in filelist:
                os.remove(self.file_path+'/'+f)

            # Build the new data
            the_file = open(self.file_path+'/data_'+api+'_'+method+'_'+query_hash+'_'+key+'.txt', 'w')
            the_file.write(str(data))
            the_file.close()


    def request(self, api, method, query={}):
        """
        Make a request to the Adobe APIs.

        * api -- the class of APIs you would like to call (e.g. Report,
            ReportSuite, Company, etc.)
        * method -- the method you would like to call inside that class
            of api
        * query -- a python object representing the parameters you would
            like to pass to the API
        """
        self.log.info("Request: %s.%s  Parameters: %s", api, method, query)
        response = requests.post(
            self.endpoint,
            params={'method': api + '.' + method},
            data=json.dumps(query),
            headers=self._build_token()
            )
        self.log.debug("Response for %s.%s:%s", api, method, response.text)
        json_response = response.json()

        if type(json_response) == dict:
            self.log.debug("Error Code %s", json_response.get('error'))
            if json_response.get('error') == 'report_not_ready':
                raise reports.ReportNotReadyError(json_response)
            elif json_response.get('error') != None:
                raise reports.InvalidReportError(json_response)
            else:
                return json_response
        else:
            return json_response

    def jsonReport(self, reportJSON):
        """Generates a Report from the JSON (including selecting the report suite)"""
        if type(reportJSON) == str:
            reportJSON = json.loads(reportJSON)
        suiteID = reportJSON['reportDescription']['reportSuiteID']
        suite = self.suites[suiteID]
        return suite.jsonReport(reportJSON)


    def _serialize_header(self, properties):
        header = []
        for key, value in properties.items():
            header.append('{key}="{value}"'.format(key=key, value=value))
        return ', '.join(header)

    def _build_token(self):
        nonce = str(uuid.uuid4())
        base64nonce = binascii.b2a_base64(binascii.a2b_qp(nonce))
        created_date = datetime.utcnow().isoformat() + 'Z'
        sha = nonce + created_date + self.secret
        sha_object = hashlib.sha1(sha.encode())
        password_64 = binascii.b2a_base64(sha_object.digest())

        properties = {
            "Username": self.username,
            "PasswordDigest": password_64.decode().strip(),
            "Nonce": base64nonce.decode().strip(),
            "Created": created_date,
        }
        header = 'UsernameToken ' + self._serialize_header(properties)

        return {'X-WSSE': header}

    def _repr_html_(self):
        """ Format in HTML for iPython Users """
        html = ""
        html += "<b>{0}</b>: {1}</br>".format("Username", self.username)
        html += "<b>{0}</b>: {1}</br>".format("Secret", "***************")
        html += "<b>{0}</b>: {1}</br>".format("Report Suites", len(self.suites))
        html += "<b>{0}</b>: {1}</br>".format("Endpoint", self.endpoint)
        return html

    def __str__(self):
        return "Analytics Account -------------\n Username: \
            {0} \n Report Suites: {1} \n Endpoint: {2}" \
            .format(self.username, len(self.suites), self.endpoint)


class Suite(Value):
    """Lets you query a specific report suite. """
    def request(self, api, method, query={}):
        raw_query = {}
        raw_query.update(query)
        if method == 'GetMetrics' or method == 'GetElements':
            raw_query['reportSuiteID'] = self.id

        return self.account.request(api, method, raw_query)

    def __init__(self, title, id, account, cache=False):
        self.log = logging.getLogger(__name__)
        super(Suite, self).__init__(title, id, account)
        self.account = account

    @property
    @utils.memoize
    def metrics(self):
        """ Return the list of valid metricsfor the current report suite"""
        if self.account.cache:
            data = self.request_cache('Report', 'GetMetrics')
        else:
            data = self.request('Report', 'GetMetrics')
        return Value.list('metrics', data, self, 'name', 'id')

    @property
    @utils.memoize
    def elements(self):
        """ Return the list of valid elementsfor the current report suite """
        if self.account.cache:
            data = self.request_cached('Report', 'GetElements')
        else:
            data = self.request('Report', 'GetElements')
        return Value.list('elements', data, self, 'name', 'id')

    @property
    @utils.memoize
    def segments(self):
        """ Return the list of valid segments for the current report suite """
        try:
            if self.account.cache:
                data = self.request_cached('Segments', 'Get',{"accessLevel":"shared"})
            else:
                data = self.request('Segments', 'Get',{"accessLevel":"shared"})
            return Value.list('segments', data, self, 'name', 'id',)
        except reports.InvalidReportError:
            data = []
            return Value.list('segments', data, self, 'name', 'id',)

    @property
    def report(self):
        """ Return a report to be run on this report suite """
        return Query(self)

    def jsonReport(self,reportJSON):
        """Creates a report from JSON. Accepts either JSON or a string. Useful for deserializing requests"""
        q = Query(self)
        #TODO: Add a method to the Account Object to populate the report suite this call will ignore it on purpose
        if type(reportJSON) == str:
            reportJSON = json.loads(reportJSON)

        reportJSON = reportJSON['reportDescription']

        if 'dateFrom' in reportJSON and 'dateTo' in reportJSON:
            q = q.range(reportJSON['dateFrom'],reportJSON['dateTo'])
        elif 'dateFrom' in reportJSON:
            q = q.range(reportJSON['dateFrom'])
        elif 'date' in reportJSON:
            q = q.range(reportJSON['date'])
        else:
            q = q

        if 'dateGranularity' in reportJSON:
            q = q.granularity(reportJSON['dateGranularity'])

        if 'source' in reportJSON:
            q = q.set('source',reportJSON['source'])

        if 'metrics' in reportJSON:
            for m in reportJSON['metrics']:
                q = q.metric(m['id'])

        if 'elements' in reportJSON:
            for e in reportJSON['elements']:
                id = e['id']
                del e['id']
                q= q.element(id, **e)

        if 'locale' in reportJSON:
            q = q.set('locale',reportJSON['locale'])

        if 'sortMethod' in reportJSON:
            q = q.set('sortMethod',reportJSON['sortMethod'])

        if 'sortBy' in reportJSON:
            q = q.sortBy(reportJSON['sortBy'])

        #WARNING This doesn't carry over segment IDs meaning you can't manipulate the segments in the new object
        #TODO Loop through and add segment ID with filter method (need to figure out how to handle combined)
        if 'segments' in reportJSON:
            q = q.set('segments', reportJSON['segments'])

        if 'anomalyDetection' in reportJSON:
            q = q.set('anomalyDetection',reportJSON['anomalyDetection'])

        if 'currentData' in reportJSON:
            q = q.set('currentData',reportJSON['currentData'])

        if 'elementDataEncoding' in reportJSON:
            q = q.set('elementDataEncoding',reportJSON['elementDataEncoding'])
        return q

    def _repr_html_(self):
        """ Format in HTML for iPython Users """
        return "<td>{0}</td><td>{1}</td>".format(self.id, self.title)

    def __str__(self):
        return "ID {0:25} | Name: {1} \n".format(self.id, self.title)
