# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['mt3scm']

package_data = \
{'': ['*']}

install_requires = \
['numpy>=1.22.3,<2.0.0', 'pandas>=1.4.1,<2.0.0', 'scikit-learn>=1.0.2,<2.0.0']

setup_kwargs = {
    'name': 'mt3scm',
    'version': '0.4.7',
    'description': "Python package for 'Multivariate Time Series Sub-Sequence CLustering Metric'",
    'long_description': '# Multivariate Time Series Sub-Sequence Clustering Metric\n\n[![DOI](https://zenodo.org/badge/470918479.svg)](https://zenodo.org/badge/latestdoi/470918479)\n![](coverage.svg) [![Python 3.9](https://img.shields.io/badge/Python-3.9-blue)](https://www.python.org/downloads/release/python-390/)\n\nThis repository provides a Python package for computing a multivariate time series subsequence clustering metric[^koehn].\nThe purpose is to have a meaningful metric for comparing time-series clustering algorithms.\n\n## Motivation\n\nTo our knowledge no existing clustering metric exists, that takes the time space variations like curvature, acceleration or torsion in a multidimensional space into consideration. We believe using these curve parameters, is an intuitive method to measure similarities between mechatronic system state changes or subsequences in multivariate time-series data (MTSD) in general.\n\n## Details\n\nOur MT3SCM score consists of three main components\n\n$$\nmt3scm = (cc_w + s_L + s_P) / 3\n$$\n\nThe weighted curvature consistency ( $cc_w$ ), the silhouette location based ( $s_L$ ) and the silhouette curve-parameter based ( $s_P$ ). When making the attempt of clustering TSD, it is subjective and domain specific. Nevertheless, we try to take the intuitive approach of treating MTSD as space curves and use the parameterization as a similarity measure. This is done in two different ways. First we create new features by computing the curve parameters sample by sample (e.g.: curvature, torsion, acceleration) and determine their standard deviation for each cluster. Our hypothesis is, that with a low standard deviation of the curve parameters inside a cluster, the actions of a mechatronic system in this cluster are similar. We call this the curvature consistency ( $cc$ )\n\nThe second procedure is to apply these newly computed features, which are computed to scalar values per subsequence, onto a well established internal clustering metric, the silhouette score[^rous1]\n\nThe computation of the $cc$ comprises the calculation of the curvature $\\kappa$ and the torsion $\\tau$ at every time step $t$ with $\\boldsymbol{x}_{t}$ .\n\nAfterwards the $cc$ is calculated per cluster $i \\in \\mathcal{I}$ , by taking the empirical standard deviation for each curve parameter (exemplarily for $\\kappa$ in with the set of subsequence indexes $\\mathcal{J}_i$ within our cluster $i$ ).\nThe arithmetic mean of the standard deviations for the curvature $\\kappa$, torsion $\\tau$ and the acceleration $a$ results in the final $cc$ per cluster.\n\nThe main idea of this approach is to combine three main parts inside one metric.\nFirst incentive is to reward a **low standard deviation of the curve parameters** in between a cluster (accomplished by $cc$ ).\nSecond, to benchmark the clusters **spatial separation based on the new feature space** (curve parameters, accomplished by $s_P$ ).\nAnd third, to benchmark the clusters **spatial separation based on the median of the subsequence in the original feature space** (accomplished by $s_L$ ).\n\nFor further details on the computation see the pubplication [^koehn]\n\n# Usage\n\nThere are two ways to compute the metric score\n```python\nimport numpy as np\nfrom mt3scm import mt3scm_score\n# Number of datapoints (time-steps)\nn_p = 1000\n# Number of dimensions or features\ndim = 5\nX = np.random.rand(n_p, dim)\n# Number of clusters\nn_c = 5\ny = np.random.randint(n_c, size=n_p)\n\n# Compute mt3scm\nscore = mt3scm_score(X, y)\nprint(score)\n```\n\nWhen using the class you can investigate internal values as well.\n\n```python\nfrom mt3scm import MT3SCM\n\nmetric = MT3SCM()\nkappa, tau, speed, acceleration = metric.compute_curvature(X)\nscore = metric.mt3scm_score(X, y)\nprint(score)\nprint(metric.df_centers)\nprint(metric.df_curve)\n\n```\n\n## Creating plots\n\n```bash\n$ python -m main --plot\n```\n\n<figure>\n    <p style="text-align:center">\n        <img\n        width=80%\n        src="plots/lorenz-attractor-3d.png"\n        alt="lorenz-attractor-3d">\n        <figcaption>Lorenz-attractor dataset. Computed with ̇$ X = s(Y − X)$ ;  ̇$Y = rX − Y − XZ$ ;  ̇$Z = XY − bZ$ and parameters used s = 10, r = 28 and b = 2.667. Color and marker size indicate amount of curvature on a logarithmic scale for better visibility.</figcaption>\n    </p>\n</figure>\n\n<figure>\n    <p style="text-align:center">\n        <img\n        width=80%\n        src="plots/qualitative_curve_parameters.png"\n        alt="qualitative_curve_parameters">\n        <figcaption>Qualitative visualization of the (a) curvature $\\kappa$ , (b) torsion $\\tau$ , (c) speed $v$ and (d) acceleration $a$ computed on part of the thomas-attractor dataset. Color and marker size indicate amount of curve parameter on a logarithmic scale for better visibility (dark and thin means low value, bright and thick means high value).</figcaption>\n    </p>\n</figure>\n\n## Comparison of unsupervised clustering metrics with lorenz attractor data\nThis example shows the effect of different metrics on the lorenz attractor dataset when using different types of label arrays. For the different unsupervised clustering labels we use the AgglomerativeClustering algorithm by varying the connectivity and the linkage as well as the number of clusters (along the lines of the [scikit-learn example](https://scikit-learn.org/stable/auto_examples/cluster/plot_agglomerative_clustering.html#sphx-glr-auto-examples-cluster-plot-agglomerative-clustering-py))\n\n![](ClusterMetricComparisonAgglomerative-lorenz.png)\n\n## References\n[^koehn]: Köhne, J. et al. Autoencoder based iterative modeling and multivariate time-series subsequence clustering algorithm\n\n[^rous1]: "Rousseeuw, P. J. Silhouettes: A graphical aid to the interpretation and validation of cluster analysis. Journal of Computational and Applied Mathematics 20. PII: 0377042787901257, 53–65. ISSN: 03770427 (1987)"\n',
    'author': 'Jonas Köhne',
    'author_email': 'jokohonas@gmail.com',
    'maintainer': 'Jonas Köhne',
    'maintainer_email': 'jokohonas@gmail.com',
    'url': 'https://github.com/Jokonu/mt3scm',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9',
}


setup(**setup_kwargs)
