"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EndpointType = exports.ApiKeySourceType = exports.RestApi = exports.SpecRestApi = exports.RestApiBase = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const api_key_1 = require("./api-key");
const apigateway_generated_1 = require("./apigateway.generated");
const deployment_1 = require("./deployment");
const domain_name_1 = require("./domain-name");
const gateway_response_1 = require("./gateway-response");
const model_1 = require("./model");
const requestvalidator_1 = require("./requestvalidator");
const resource_1 = require("./resource");
const stage_1 = require("./stage");
const usage_plan_1 = require("./usage-plan");
const RESTAPI_SYMBOL = Symbol.for('@aws-cdk/aws-apigateway.RestApiBase');
/**
 * Base implementation that are common to various implementations of IRestApi
 */
class RestApiBase extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.restApiName || id,
        });
        Object.defineProperty(this, RESTAPI_SYMBOL, { value: true });
    }
    /**
     * Checks if the given object is an instance of RestApiBase.
     * @internal
     */
    static _isRestApiBase(x) {
        return x !== null && typeof (x) === 'object' && RESTAPI_SYMBOL in x;
    }
    /**
     * API Gateway deployment that represents the latest changes of the API.
     * This resource will be automatically updated every time the REST API model changes.
     * This will be undefined if `deploy` is false.
     */
    get latestDeployment() {
        return this._latestDeployment;
    }
    /**
     * The first domain name mapped to this API, if defined through the `domainName`
     * configuration prop, or added via `addDomainName`
     */
    get domainName() {
        return this._domainName;
    }
    /**
     * Returns the URL for an HTTP path.
     *
     * Fails if `deploymentStage` is not set either by `deploy` or explicitly.
     */
    urlForPath(path = '/') {
        if (!this.deploymentStage) {
            throw new Error('Cannot determine deployment stage for API from "deploymentStage". Use "deploy" or explicitly set "deploymentStage"');
        }
        return this.deploymentStage.urlForPath(path);
    }
    /**
     * Defines an API Gateway domain name and maps it to this API.
     * @param id The construct id
     * @param options custom domain options
     */
    addDomainName(id, options) {
        const domainName = new domain_name_1.DomainName(this, id, {
            ...options,
            mapping: this,
        });
        if (!this._domainName) {
            this._domainName = domainName;
        }
        return domainName;
    }
    /**
     * Adds a usage plan.
     */
    addUsagePlan(id, props = {}) {
        return new usage_plan_1.UsagePlan(this, id, props);
    }
    arnForExecuteApi(method = '*', path = '/*', stage = '*') {
        if (!path.startsWith('/')) {
            throw new Error(`"path" must begin with a "/": '${path}'`);
        }
        if (method.toUpperCase() === 'ANY') {
            method = '*';
        }
        return core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: this.restApiId,
            sep: '/',
            resourceName: `${stage}/${method}${path}`,
        });
    }
    /**
     * Adds a new gateway response.
     */
    addGatewayResponse(id, options) {
        return new gateway_response_1.GatewayResponse(this, id, {
            restApi: this,
            ...options,
        });
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        ignore(method);
    }
    /**
     * Associates a Deployment resource with this REST API.
     *
     * @internal
     */
    _attachDeployment(deployment) {
        ignore(deployment);
    }
    configureCloudWatchRole(apiResource) {
        const role = new iam.Role(this, 'CloudWatchRole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonAPIGatewayPushToCloudWatchLogs')],
        });
        const resource = new apigateway_generated_1.CfnAccount(this, 'Account', {
            cloudWatchRoleArn: role.roleArn,
        });
        resource.node.addDependency(apiResource);
    }
    configureDeployment(props) {
        const deploy = props.deploy === undefined ? true : props.deploy;
        if (deploy) {
            this._latestDeployment = new deployment_1.Deployment(this, 'Deployment', {
                description: 'Automatically created by the RestApi construct',
                api: this,
                retainDeployments: props.retainDeployments,
            });
            // encode the stage name into the construct id, so if we change the stage name, it will recreate a new stage.
            // stage name is part of the endpoint, so that makes sense.
            const stageName = (props.deployOptions && props.deployOptions.stageName) || 'prod';
            this.deploymentStage = new stage_1.Stage(this, `DeploymentStage.${stageName}`, {
                deployment: this._latestDeployment,
                ...props.deployOptions,
            });
            new core_1.CfnOutput(this, 'Endpoint', { exportName: props.endpointExportName, value: this.urlForPath() });
        }
        else {
            if (props.deployOptions) {
                throw new Error('Cannot set \'deployOptions\' if \'deploy\' is disabled');
            }
        }
    }
    /**
     * @internal
     */
    _configureEndpoints(props) {
        var _a, _b;
        if (props.endpointTypes && props.endpointConfiguration) {
            throw new Error('Only one of the RestApi props, endpointTypes or endpointConfiguration, is allowed');
        }
        if (props.endpointConfiguration) {
            return {
                types: props.endpointConfiguration.types,
                vpcEndpointIds: (_b = (_a = props.endpointConfiguration) === null || _a === void 0 ? void 0 : _a.vpcEndpoints) === null || _b === void 0 ? void 0 : _b.map(vpcEndpoint => vpcEndpoint.vpcEndpointId),
            };
        }
        if (props.endpointTypes) {
            return { types: props.endpointTypes };
        }
        return undefined;
    }
}
exports.RestApiBase = RestApiBase;
/**
 * Represents a REST API in Amazon API Gateway, created with an OpenAPI specification.
 *
 * Some properties normally accessible on @see {@link RestApi} - such as the description -
 * must be declared in the specification. All Resources and Methods need to be defined as
 * part of the OpenAPI specification file, and cannot be added via the CDK.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 *
 * @experimental
 *
 * @resource AWS::ApiGateway::RestApi
 */
class SpecRestApi extends RestApiBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        const apiDefConfig = props.apiDefinition.bind(this);
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.physicalName,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            body: apiDefConfig.inlineDefinition ? apiDefConfig.inlineDefinition : undefined,
            bodyS3Location: apiDefConfig.inlineDefinition ? undefined : apiDefConfig.s3Location,
            endpointConfiguration: this._configureEndpoints(props),
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        this.restApiRootResourceId = resource.attrRootResourceId;
        this.root = new RootResource(this, {}, this.restApiRootResourceId);
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
    }
}
exports.SpecRestApi = SpecRestApi;
/**
 * Represents a REST API in Amazon API Gateway.
 *
 * Use `addResource` and `addMethod` to configure the API model.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 */
class RestApi extends RestApiBase {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        /**
         * The list of methods bound to this RestApi
         */
        this.methods = new Array();
        /**
         * This list of deployments bound to this RestApi
         */
        this.deployments = new Array();
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.physicalName,
            description: props.description,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            minimumCompressionSize: props.minimumCompressionSize,
            binaryMediaTypes: props.binaryMediaTypes,
            endpointConfiguration: this._configureEndpoints(props),
            apiKeySourceType: props.apiKeySourceType,
            cloneFrom: props.cloneFrom ? props.cloneFrom.restApiId : undefined,
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        this.root = new RootResource(this, props, resource.attrRootResourceId);
        this.restApiRootResourceId = resource.attrRootResourceId;
    }
    /**
     * Import an existing RestApi.
     */
    static fromRestApiId(scope, id, restApiId) {
        class Import extends RestApiBase {
            constructor() {
                super(...arguments);
                this.restApiId = restApiId;
            }
            get root() {
                throw new Error('root is not configured when imported using `fromRestApiId()`. Use `fromRestApiAttributes()` API instead.');
            }
            get restApiRootResourceId() {
                throw new Error('restApiRootResourceId is not configured when imported using `fromRestApiId()`. Use `fromRestApiAttributes()` API instead.');
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing RestApi that can be configured with additional Methods and Resources.
     * @experimental
     */
    static fromRestApiAttributes(scope, id, attrs) {
        class Import extends RestApiBase {
            constructor() {
                super(...arguments);
                this.restApiId = attrs.restApiId;
                this.restApiRootResourceId = attrs.rootResourceId;
                this.root = new RootResource(this, {}, this.restApiRootResourceId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * The deployed root URL of this REST API.
     */
    get url() {
        return this.urlForPath();
    }
    /**
     * Add an ApiKey
     */
    addApiKey(id, options) {
        return new api_key_1.ApiKey(this, id, {
            resources: [this],
            ...options,
        });
    }
    /**
     * Adds a new model.
     */
    addModel(id, props) {
        return new model_1.Model(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * Adds a new request validator.
     */
    addRequestValidator(id, props) {
        return new requestvalidator_1.RequestValidator(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        this.methods.push(method);
        // add this method as a dependency to all deployments defined for this api
        // when additional deployments are added, _attachDeployment is called and
        // this method will be added there.
        for (const dep of this.deployments) {
            dep._addMethodDependency(method);
        }
    }
    /**
     * Attaches a deployment to this REST API.
     *
     * @internal
     */
    _attachDeployment(deployment) {
        this.deployments.push(deployment);
        // add all methods that were already defined as dependencies of this
        // deployment when additional methods are added, _attachMethod is called and
        // it will be added as a dependency to this deployment.
        for (const method of this.methods) {
            deployment._addMethodDependency(method);
        }
    }
    /**
     * Performs validation of the REST API.
     */
    validate() {
        if (this.methods.length === 0) {
            return ["The REST API doesn't contain any methods"];
        }
        return [];
    }
}
exports.RestApi = RestApi;
var ApiKeySourceType;
(function (ApiKeySourceType) {
    /**
     * To read the API key from the `X-API-Key` header of a request.
     */
    ApiKeySourceType["HEADER"] = "HEADER";
    /**
     * To read the API key from the `UsageIdentifierKey` from a custom authorizer.
     */
    ApiKeySourceType["AUTHORIZER"] = "AUTHORIZER";
})(ApiKeySourceType = exports.ApiKeySourceType || (exports.ApiKeySourceType = {}));
var EndpointType;
(function (EndpointType) {
    /**
     * For an edge-optimized API and its custom domain name.
     */
    EndpointType["EDGE"] = "EDGE";
    /**
     * For a regional API and its custom domain name.
     */
    EndpointType["REGIONAL"] = "REGIONAL";
    /**
     * For a private API and its custom domain name.
     */
    EndpointType["PRIVATE"] = "PRIVATE";
})(EndpointType = exports.EndpointType || (exports.EndpointType = {}));
class RootResource extends resource_1.ResourceBase {
    constructor(api, props, resourceId) {
        super(api, 'Default');
        this.parentResource = undefined;
        this.defaultIntegration = props.defaultIntegration;
        this.defaultMethodOptions = props.defaultMethodOptions;
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions;
        this.api = api;
        this.resourceId = resourceId;
        this.path = '/';
        if (api instanceof RestApi) {
            this._restApi = api;
        }
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
    /**
     * Get the RestApi associated with this Resource.
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        if (!this._restApi) {
            throw new Error('RestApi is not available on Resource not connected to an instance of RestApi. Use `api` instead');
        }
        return this._restApi;
    }
}
function ignore(_x) {
    return;
}
//# sourceMappingURL=data:application/json;base64,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