"use strict";
const assert_1 = require("@aws-cdk/assert");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const path = require("path");
const apigateway = require("../lib");
module.exports = {
    'minimal setup'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigateway.RestApi(stack, 'api', { deploy: false, cloudWatchRole: false });
        api.root.addMethod('GET');
        // WHEN
        new apigateway.Deployment(stack, 'deployment', { api });
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                apiGETECF0BD67: {
                    Type: 'AWS::ApiGateway::Method',
                    Properties: {
                        HttpMethod: 'GET',
                        ResourceId: {
                            'Fn::GetAtt': [
                                'apiC8550315',
                                'RootResourceId',
                            ],
                        },
                        RestApiId: {
                            Ref: 'apiC8550315',
                        },
                        AuthorizationType: 'NONE',
                        Integration: {
                            Type: 'MOCK',
                        },
                    },
                },
                apiC8550315: {
                    Type: 'AWS::ApiGateway::RestApi',
                    Properties: {
                        Name: 'api',
                    },
                },
                deployment33381975bba46c5132329b81e7befcbbba5a0e75: {
                    Type: 'AWS::ApiGateway::Deployment',
                    Properties: {
                        RestApiId: {
                            Ref: 'apiC8550315',
                        },
                    },
                    DependsOn: [
                        'apiGETECF0BD67',
                    ],
                },
            },
        });
        test.done();
    },
    '"retainDeployments" can be used to control the deletion policy of the resource'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigateway.RestApi(stack, 'api', { deploy: false, cloudWatchRole: false });
        api.root.addMethod('GET');
        // WHEN
        new apigateway.Deployment(stack, 'deployment', { api, retainDeployments: true });
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                apiGETECF0BD67: {
                    Type: 'AWS::ApiGateway::Method',
                    Properties: {
                        HttpMethod: 'GET',
                        ResourceId: {
                            'Fn::GetAtt': [
                                'apiC8550315',
                                'RootResourceId',
                            ],
                        },
                        RestApiId: {
                            Ref: 'apiC8550315',
                        },
                        AuthorizationType: 'NONE',
                        Integration: {
                            Type: 'MOCK',
                        },
                    },
                },
                apiC8550315: {
                    Type: 'AWS::ApiGateway::RestApi',
                    Properties: {
                        Name: 'api',
                    },
                },
                deployment33381975bba46c5132329b81e7befcbbba5a0e75: {
                    Type: 'AWS::ApiGateway::Deployment',
                    Properties: {
                        RestApiId: {
                            Ref: 'apiC8550315',
                        },
                    },
                    DeletionPolicy: 'Retain',
                    UpdateReplacePolicy: 'Retain',
                    DependsOn: [
                        'apiGETECF0BD67',
                    ],
                },
            },
        });
        test.done();
    },
    '"description" can be set on the deployment'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigateway.RestApi(stack, 'api', { deploy: false, cloudWatchRole: false });
        api.root.addMethod('GET');
        // WHEN
        new apigateway.Deployment(stack, 'deployment', { api, description: 'this is my deployment' });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Deployment', {
            Description: 'this is my deployment',
        }));
        test.done();
    },
    'logical ID of the deployment resource is salted'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigateway.RestApi(stack, 'api', { deploy: false, cloudWatchRole: false });
        const deployment = new apigateway.Deployment(stack, 'deployment', { api });
        api.root.addMethod('GET');
        const resources = synthesize().Resources;
        test.ok(resources.deployment33381975bba46c5132329b81e7befcbbba5a0e75, `resource deployment33381975bba46c5132329b81e7befcbbba5a0e75 not found, instead found ${Object.keys(resources)}`);
        // adding some salt
        deployment.addToLogicalId({ foo: 123 }); // add some data to the logical ID
        // the logical ID changed
        const template = synthesize();
        test.ok(!template.Resources.deployment33381975bba46c5132329b81e7befcbbba5a0e75, 'old resource id is not deleted');
        test.ok(template.Resources.deployment333819758aa4cdb9d204502b959c4903f4d5d29f, `new resource deployment333819758aa4cdb9d204502b959c4903f4d5d29f is not created, instead found ${Object.keys(template.Resources)}`);
        // tokens supported, and are resolved upon synthesis
        const value = 'hello hello';
        deployment.addToLogicalId({ foo: core_1.Lazy.stringValue({ produce: () => value }) });
        const template2 = synthesize();
        test.ok(template2.Resources.deployment333819758d91bed959c6bd6268ba84f6d33e888e, `resource deployment333819758d91bed959c6bd6268ba84f6d33e888e not found, instead found ${Object.keys(template2.Resources)}`);
        test.done();
        function synthesize() {
            return assert_1.SynthUtils.synthesize(stack).template;
        }
    },
    '"addDependency" can be used to add a resource as a dependency'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigateway.RestApi(stack, 'api', { deploy: false, cloudWatchRole: false });
        const deployment = new apigateway.Deployment(stack, 'deployment', { api });
        api.root.addMethod('GET');
        const dep = new core_1.CfnResource(stack, 'MyResource', { type: 'foo' });
        // WHEN
        deployment.node.addDependency(dep);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Deployment', {
            DependsOn: [
                'apiGETECF0BD67',
                'MyResource',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'integration change invalidates deployment'(test) {
        // GIVEN
        const stack1 = new core_1.Stack();
        const stack2 = new core_1.Stack();
        const handler1 = new lambda.Function(stack1, 'handler1', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'lambda')),
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
        });
        const handler2 = new lambda.Function(stack2, 'handler2', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'lambda')),
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
        });
        // WHEN
        const api1 = new apigateway.RestApi(stack1, 'myapi', {
            defaultIntegration: new apigateway.LambdaIntegration(handler1),
        });
        const api2 = new apigateway.RestApi(stack2, 'myapi', {
            defaultIntegration: new apigateway.LambdaIntegration(handler2),
        });
        api1.root.addMethod('GET');
        api2.root.addMethod('GET');
        // THEN
        assert_1.expect(stack1).to(assert_1.haveResource('AWS::ApiGateway::Stage', {
            DeploymentId: { Ref: 'myapiDeploymentB7EF8EB74c5295c27fa87ff13f4d04e13f67662d' },
        }));
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::ApiGateway::Stage', {
            DeploymentId: { Ref: 'myapiDeploymentB7EF8EB7b50d305057ba109c118e4aafd4509355' },
        }));
        test.done();
    },
    'deployment resource depends on all restapi methods defined'(test) {
        const stack = new core_1.Stack();
        const restapi = new apigateway.RestApi(stack, 'myapi', {
            deploy: false,
        });
        restapi.root.addMethod('GET');
        const deployment = new apigateway.Deployment(stack, 'mydeployment', {
            api: restapi,
        });
        const stage = new apigateway.Stage(stack, 'mystage', {
            deployment,
        });
        restapi.deploymentStage = stage;
        restapi.root.addMethod('POST');
        const resource = restapi.root.addResource('myresource');
        resource.addMethod('GET');
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Deployment', {
            DependsOn: [
                'myapiGET9B7CD29E',
                'myapimyresourceGET732851A5',
                'myapiPOST23417BD2',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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