"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const apigw = require("../lib");
const DUMMY_AUTHORIZER = {
    authorizerId: 'dummyauthorizer',
    authorizationType: apigw.AuthorizationType.CUSTOM,
};
module.exports = {
    'default setup'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'POST',
            AuthorizationType: 'NONE',
            Integration: {
                Type: 'MOCK',
            },
        }));
        test.done();
    },
    'method options can be specified'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
            options: {
                apiKeyRequired: true,
                operationName: 'MyOperation',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            ApiKeyRequired: true,
            OperationName: 'MyOperation',
        }));
        test.done();
    },
    'integration can be set via a property'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
            integration: new apigw.AwsIntegration({ service: 's3', path: 'bucket/key' }),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            Integration: {
                IntegrationHttpMethod: 'POST',
                Type: 'AWS',
                Uri: {
                    'Fn::Join': [
                        '',
                        [
                            'arn:', { Ref: 'AWS::Partition' }, ':apigateway:',
                            { Ref: 'AWS::Region' }, ':s3:path/bucket/key',
                        ],
                    ],
                },
            },
        }));
        test.done();
    },
    'integration with a custom http method can be set via a property'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
            integration: new apigw.AwsIntegration({ service: 's3', path: 'bucket/key', integrationHttpMethod: 'GET' }),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            Integration: {
                IntegrationHttpMethod: 'GET',
            },
        }));
        test.done();
    },
    'use default integration from api'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const defaultIntegration = new apigw.Integration({ type: apigw.IntegrationType.HTTP_PROXY, uri: 'https://amazon.com' });
        const api = new apigw.RestApi(stack, 'test-api', {
            cloudWatchRole: false,
            deploy: false,
            defaultIntegration,
        });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            Integration: {
                Type: 'HTTP_PROXY',
                Uri: 'https://amazon.com',
            },
        }));
        test.done();
    },
    '"methodArn" returns the ARN execute-api ARN for this method in the current stage'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api');
        // WHEN
        const method = new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
        });
        // THEN
        test.deepEqual(stack.resolve(method.methodArn), {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':execute-api:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'testapiD6451F70' },
                    '/',
                    { Ref: 'testapiDeploymentStageprod5C9E92A4' },
                    '/POST/',
                ],
            ],
        });
        test.done();
    },
    '"testMethodArn" returns the ARN of the "test-invoke-stage" stage (console UI)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api');
        // WHEN
        const method = new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
        });
        // THEN
        test.deepEqual(stack.resolve(method.testMethodArn), {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':execute-api:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'testapiD6451F70' },
                    '/test-invoke-stage/POST/',
                ],
            ],
        });
        test.done();
    },
    '"methodArn" returns an arn with "*" as its stage when deploymentStage is not set'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        const method = new apigw.Method(stack, 'my-method', { httpMethod: 'POST', resource: api.root });
        // THEN
        test.deepEqual(stack.resolve(method.methodArn), {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':execute-api:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'testapiD6451F70' },
                    '/*/POST/',
                ],
            ],
        });
        test.done();
    },
    '"methodArn" and "testMethodArn" replace path parameters with asterisks'(test) {
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api');
        const petId = api.root.addResource('pets').addResource('{petId}');
        const commentId = petId.addResource('comments').addResource('{commentId}');
        const method = commentId.addMethod('GET');
        test.deepEqual(stack.resolve(method.methodArn), {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':execute-api:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'testapiD6451F70' },
                    '/',
                    { Ref: 'testapiDeploymentStageprod5C9E92A4' },
                    '/GET/pets/*/comments/*',
                ],
            ],
        });
        test.deepEqual(stack.resolve(method.testMethodArn), {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':execute-api:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'testapiD6451F70' },
                    '/test-invoke-stage/GET/pets/*/comments/*',
                ],
            ],
        });
        test.done();
    },
    'integration "credentialsRole" can be used to assume a role when calling backend'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const role = new iam.Role(stack, 'MyRole', { assumedBy: new iam.ServicePrincipal('foo') });
        // WHEN
        api.root.addMethod('GET', new apigw.Integration({
            type: apigw.IntegrationType.AWS_PROXY,
            options: {
                credentialsRole: role,
            },
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            Integration: {
                Credentials: { 'Fn::GetAtt': ['MyRoleF48FFE04', 'Arn'] },
            },
        }));
        test.done();
    },
    'integration "credentialsPassthrough" can be used to passthrough user credentials to backend'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        api.root.addMethod('GET', new apigw.Integration({
            type: apigw.IntegrationType.AWS_PROXY,
            options: {
                credentialsPassthrough: true,
            },
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            Integration: {
                Credentials: { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::*:user/*']] },
            },
        }));
        test.done();
    },
    'methodResponse set one or more method responses via options'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                methodResponses: [{
                        statusCode: '200',
                    }, {
                        statusCode: '400',
                        responseParameters: {
                            'method.response.header.killerbees': false,
                        },
                    }, {
                        statusCode: '500',
                        responseParameters: {
                            'method.response.header.errthing': true,
                        },
                        responseModels: {
                            'application/json': apigw.Model.EMPTY_MODEL,
                            'text/plain': apigw.Model.ERROR_MODEL,
                        },
                    },
                ],
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'GET',
            MethodResponses: [{
                    StatusCode: '200',
                }, {
                    StatusCode: '400',
                    ResponseParameters: {
                        'method.response.header.killerbees': false,
                    },
                }, {
                    StatusCode: '500',
                    ResponseParameters: {
                        'method.response.header.errthing': true,
                    },
                    ResponseModels: {
                        'application/json': 'Empty',
                        'text/plain': 'Error',
                    },
                },
            ],
        }));
        test.done();
    },
    'multiple integration responses can be used'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        api.root.addMethod('GET', new apigw.AwsIntegration({
            service: 'foo-service',
            action: 'BarAction',
            options: {
                integrationResponses: [
                    {
                        statusCode: '200',
                        responseTemplates: { 'application/json': JSON.stringify({ success: true }) },
                    },
                    {
                        selectionPattern: 'Invalid',
                        statusCode: '503',
                        responseTemplates: { 'application/json': JSON.stringify({ success: false, message: 'Invalid Request' }) },
                    },
                ],
            },
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            Integration: {
                IntegrationHttpMethod: 'POST',
                IntegrationResponses: [
                    {
                        ResponseTemplates: { 'application/json': '{"success":true}' },
                        StatusCode: '200',
                    },
                    {
                        ResponseTemplates: { 'application/json': '{"success":false,"message":"Invalid Request"}' },
                        SelectionPattern: 'Invalid',
                        StatusCode: '503',
                    },
                ],
                Type: 'AWS',
                Uri: { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':apigateway:', { Ref: 'AWS::Region' }, ':foo-service:action/BarAction']] },
            },
        }));
        test.done();
    },
    'method is always set as uppercase'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'api');
        // WHEN
        api.root.addMethod('get');
        api.root.addMethod('PoSt');
        api.root.addMethod('PUT');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', { HttpMethod: 'POST' }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', { HttpMethod: 'GET' }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', { HttpMethod: 'PUT' }));
        test.done();
    },
    'requestModel can be set'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const model = api.addModel('test-model', {
            contentType: 'application/json',
            modelName: 'test-model',
            schema: {
                title: 'test',
                type: apigw.JsonSchemaType.OBJECT,
                properties: { message: { type: apigw.JsonSchemaType.STRING } },
            },
        });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                requestModels: {
                    'application/json': model,
                },
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'GET',
            RequestModels: {
                'application/json': { Ref: stack.getLogicalId(model.node.findChild('Resource')) },
            },
        }));
        test.done();
    },
    'methodResponse has a mix of response modes'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const htmlModel = api.addModel('my-model', {
            schema: {
                schema: apigw.JsonSchemaVersion.DRAFT4,
                title: 'test',
                type: apigw.JsonSchemaType.OBJECT,
                properties: { message: { type: apigw.JsonSchemaType.STRING } },
            },
        });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                methodResponses: [{
                        statusCode: '200',
                    }, {
                        statusCode: '400',
                        responseParameters: {
                            'method.response.header.killerbees': false,
                        },
                    }, {
                        statusCode: '500',
                        responseParameters: {
                            'method.response.header.errthing': true,
                        },
                        responseModels: {
                            'application/json': apigw.Model.EMPTY_MODEL,
                            'text/plain': apigw.Model.ERROR_MODEL,
                            'text/html': htmlModel,
                        },
                    },
                ],
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'GET',
            MethodResponses: [{
                    StatusCode: '200',
                }, {
                    StatusCode: '400',
                    ResponseParameters: {
                        'method.response.header.killerbees': false,
                    },
                }, {
                    StatusCode: '500',
                    ResponseParameters: {
                        'method.response.header.errthing': true,
                    },
                    ResponseModels: {
                        'application/json': 'Empty',
                        'text/plain': 'Error',
                        'text/html': { Ref: stack.getLogicalId(htmlModel.node.findChild('Resource')) },
                    },
                },
            ],
        }));
        test.done();
    },
    'method has a request validator'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const validator = api.addRequestValidator('validator', {
            validateRequestBody: true,
            validateRequestParameters: false,
        });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                requestValidator: validator,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            RequestValidatorId: { Ref: stack.getLogicalId(validator.node.findChild('Resource')) },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RequestValidator', {
            RestApiId: { Ref: stack.getLogicalId(api.node.findChild('Resource')) },
            ValidateRequestBody: true,
            ValidateRequestParameters: false,
        }));
        test.done();
    },
    'use default requestParameters'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', {
            cloudWatchRole: false,
            deploy: false,
            defaultMethodOptions: {
                requestParameters: { 'method.request.path.proxy': true },
            },
        });
        // WHEN
        new apigw.Method(stack, 'defaultRequestParameters', {
            httpMethod: 'POST',
            resource: api.root,
            options: {
                operationName: 'defaultRequestParameters',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            OperationName: 'defaultRequestParameters',
            RequestParameters: {
                'method.request.path.proxy': true,
            },
        }));
        test.done();
    },
    'authorizer is bound correctly'(test) {
        const stack = new cdk.Stack();
        const restApi = new apigw.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: DUMMY_AUTHORIZER,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'ANY',
            AuthorizationType: 'CUSTOM',
            AuthorizerId: DUMMY_AUTHORIZER.authorizerId,
        }));
        test.done();
    },
    'authorizer via default method options'(test) {
        const stack = new cdk.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const auth = new apigw.TokenAuthorizer(stack, 'myauthorizer1', {
            authorizerName: 'myauthorizer1',
            handler: func,
        });
        const restApi = new apigw.RestApi(stack, 'myrestapi', {
            defaultMethodOptions: {
                authorizer: auth,
            },
        });
        restApi.root.addMethod('ANY');
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Name: 'myauthorizer1',
            Type: 'TOKEN',
            RestApiId: stack.resolve(restApi.restApiId),
        }));
        test.done();
    },
    'fails when authorization type does not match the authorizer'(test) {
        const stack = new cdk.Stack();
        const restApi = new apigw.RestApi(stack, 'myrestapi');
        test.throws(() => {
            restApi.root.addMethod('ANY', undefined, {
                authorizationType: apigw.AuthorizationType.IAM,
                authorizer: DUMMY_AUTHORIZER,
            });
        }, /Authorization type is set to AWS_IAM which is different from what is required by the authorizer/);
        test.done();
    },
    'fails when authorization type does not match the authorizer in default method options'(test) {
        const stack = new cdk.Stack();
        const restApi = new apigw.RestApi(stack, 'myrestapi', {
            defaultMethodOptions: {
                authorizer: DUMMY_AUTHORIZER,
            },
        });
        test.throws(() => {
            restApi.root.addMethod('ANY', undefined, {
                authorizationType: apigw.AuthorizationType.NONE,
            });
        }, /Authorization type is set to NONE which is different from what is required by the authorizer/);
        test.done();
    },
    'method has Auth Scopes'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
            options: {
                apiKeyRequired: true,
                authorizationScopes: ['AuthScope1', 'AuthScope2'],
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            ApiKeyRequired: true,
            AuthorizationScopes: ['AuthScope1', 'AuthScope2'],
        }));
        test.done();
    },
    'use default Auth Scopes'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', {
            cloudWatchRole: false,
            deploy: false,
            defaultMethodOptions: {
                authorizationScopes: ['DefaultAuth'],
            },
        });
        // WHEN
        new apigw.Method(stack, 'defaultAuthScopes', {
            httpMethod: 'POST',
            resource: api.root,
            options: {
                operationName: 'defaultAuthScopes',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            OperationName: 'defaultAuthScopes',
            AuthorizationScopes: ['DefaultAuth'],
        }));
        test.done();
    },
    'Method options Auth Scopes is picked up'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', {
            cloudWatchRole: false,
            deploy: false,
            defaultMethodOptions: {
                authorizationScopes: ['DefaultAuth'],
            },
        });
        // WHEN
        new apigw.Method(stack, 'MethodAuthScopeUsed', {
            httpMethod: 'POST',
            resource: api.root,
            options: {
                apiKeyRequired: true,
                authorizationScopes: ['MethodAuthScope'],
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            ApiKeyRequired: true,
            AuthorizationScopes: ['MethodAuthScope'],
        }));
        test.done();
    },
    'Auth Scopes absent'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', {
            cloudWatchRole: false,
            deploy: false,
        });
        // WHEN
        new apigw.Method(stack, 'authScopesAbsent', {
            httpMethod: 'POST',
            resource: api.root,
            options: {
                operationName: 'authScopesAbsent',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            OperationName: 'authScopesAbsent',
            AuthorizationScopes: assert_1.ABSENT,
        }));
        test.done();
    },
    'method has a request validator with provided properties'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                requestValidatorOptions: {
                    requestValidatorName: 'test-validator',
                    validateRequestBody: true,
                    validateRequestParameters: false,
                },
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RequestValidator', {
            RestApiId: stack.resolve(api.restApiId),
            ValidateRequestBody: true,
            ValidateRequestParameters: false,
            Name: 'test-validator',
        }));
        test.done();
    },
    'method does not have a request validator'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            RequestValidatorId: assert_1.ABSENT,
        }));
        test.done();
    },
    'method does not support both request validator and request validator options'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const validator = api.addRequestValidator('test-validator1', {
            validateRequestBody: true,
            validateRequestParameters: false,
        });
        // WHEN
        const methodProps = {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                requestValidatorOptions: {
                    requestValidatorName: 'test-validator2',
                    validateRequestBody: true,
                    validateRequestParameters: false,
                },
                requestValidator: validator,
            },
        };
        // THEN
        test.throws(() => new apigw.Method(stack, 'method', methodProps), /Only one of 'requestValidator' or 'requestValidatorOptions' must be specified./);
        test.done();
    },
    '"restApi" and "api" properties return the RestApi correctly'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'test-api');
        const method = api.root.addResource('pets').addMethod('GET');
        // THEN
        test.ok(method.restApi);
        test.ok(method.api);
        test.deepEqual(stack.resolve(method.api.restApiId), stack.resolve(method.restApi.restApiId));
        test.done();
    },
    '"restApi" throws an error on imported while "api" returns correctly'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const api = apigw.RestApi.fromRestApiAttributes(stack, 'test-api', {
            restApiId: 'test-rest-api-id',
            rootResourceId: 'test-root-resource-id',
        });
        const method = api.root.addResource('pets').addMethod('GET');
        // THEN
        test.throws(() => method.restApi, /not available on Resource not connected to an instance of RestApi/);
        test.ok(method.api);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5tZXRob2QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0Lm1ldGhvZC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEsNENBQWlGO0FBQ2pGLHdDQUF3QztBQUN4Qyw4Q0FBOEM7QUFDOUMscUNBQXFDO0FBRXJDLGdDQUFnQztBQUVoQyxNQUFNLGdCQUFnQixHQUFzQjtJQUMxQyxZQUFZLEVBQUUsaUJBQWlCO0lBQy9CLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxNQUFNO0NBQ2xELENBQUM7QUFFRixpQkFBUztJQUNQLGVBQWUsQ0FBQyxJQUFVO1FBQ3hCLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLEdBQUcsR0FBRyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxFQUFFLGNBQWMsRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7UUFFM0YsT0FBTztRQUNQLElBQUksS0FBSyxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO1lBQ25DLFVBQVUsRUFBRSxNQUFNO1lBQ2xCLFFBQVEsRUFBRSxHQUFHLENBQUMsSUFBSTtTQUNuQixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHlCQUF5QixFQUFFO1lBQ3ZELFVBQVUsRUFBRSxNQUFNO1lBQ2xCLGlCQUFpQixFQUFFLE1BQU07WUFDekIsV0FBVyxFQUFFO2dCQUNYLElBQUksRUFBRSxNQUFNO2FBQ2I7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxpQ0FBaUMsQ0FBQyxJQUFVO1FBQzFDLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLEdBQUcsR0FBRyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxFQUFFLGNBQWMsRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7UUFFM0YsT0FBTztRQUNQLElBQUksS0FBSyxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO1lBQ25DLFVBQVUsRUFBRSxNQUFNO1lBQ2xCLFFBQVEsRUFBRSxHQUFHLENBQUMsSUFBSTtZQUNsQixPQUFPLEVBQUU7Z0JBQ1AsY0FBYyxFQUFFLElBQUk7Z0JBQ3BCLGFBQWEsRUFBRSxhQUFhO2FBQzdCO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx5QkFBeUIsRUFBRTtZQUN2RCxjQUFjLEVBQUUsSUFBSTtZQUNwQixhQUFhLEVBQUUsYUFBYTtTQUM3QixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx1Q0FBdUMsQ0FBQyxJQUFVO1FBQ2hELFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLEdBQUcsR0FBRyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxFQUFFLGNBQWMsRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7UUFFM0YsT0FBTztRQUNQLElBQUksS0FBSyxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO1lBQ25DLFVBQVUsRUFBRSxNQUFNO1lBQ2xCLFFBQVEsRUFBRSxHQUFHLENBQUMsSUFBSTtZQUNsQixXQUFXLEVBQUUsSUFBSSxLQUFLLENBQUMsY0FBYyxDQUFDLEVBQUUsT0FBTyxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsWUFBWSxFQUFFLENBQUM7U0FDN0UsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx5QkFBeUIsRUFBRTtZQUN2RCxXQUFXLEVBQUU7Z0JBQ1gscUJBQXFCLEVBQUUsTUFBTTtnQkFDN0IsSUFBSSxFQUFFLEtBQUs7Z0JBQ1gsR0FBRyxFQUFFO29CQUNILFVBQVUsRUFBRTt3QkFDVixFQUFFO3dCQUNGOzRCQUNFLE1BQU0sRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxFQUFFLGNBQWM7NEJBQ2pELEVBQUUsR0FBRyxFQUFFLGFBQWEsRUFBRSxFQUFFLHFCQUFxQjt5QkFDOUM7cUJBQ0Y7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGlFQUFpRSxDQUFDLElBQVU7UUFDMUUsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sR0FBRyxHQUFHLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEVBQUUsY0FBYyxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQztRQUUzRixPQUFPO1FBQ1AsSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUU7WUFDbkMsVUFBVSxFQUFFLE1BQU07WUFDbEIsUUFBUSxFQUFFLEdBQUcsQ0FBQyxJQUFJO1lBQ2xCLFdBQVcsRUFBRSxJQUFJLEtBQUssQ0FBQyxjQUFjLENBQUMsRUFBRSxPQUFPLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxZQUFZLEVBQUUscUJBQXFCLEVBQUUsS0FBSyxFQUFFLENBQUM7U0FDM0csQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMseUJBQXlCLEVBQUU7WUFDM0QsV0FBVyxFQUFFO2dCQUNYLHFCQUFxQixFQUFFLEtBQUs7YUFDN0I7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxrQ0FBa0MsQ0FBQyxJQUFVO1FBQzNDLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLGtCQUFrQixHQUFHLElBQUksS0FBSyxDQUFDLFdBQVcsQ0FBQyxFQUFFLElBQUksRUFBRSxLQUFLLENBQUMsZUFBZSxDQUFDLFVBQVUsRUFBRSxHQUFHLEVBQUUsb0JBQW9CLEVBQUUsQ0FBQyxDQUFDO1FBQ3hILE1BQU0sR0FBRyxHQUFHLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQy9DLGNBQWMsRUFBRSxLQUFLO1lBQ3JCLE1BQU0sRUFBRSxLQUFLO1lBQ2Isa0JBQWtCO1NBQ25CLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRTtZQUNuQyxVQUFVLEVBQUUsTUFBTTtZQUNsQixRQUFRLEVBQUUsR0FBRyxDQUFDLElBQUk7U0FDbkIsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx5QkFBeUIsRUFBRTtZQUN2RCxXQUFXLEVBQUU7Z0JBQ1gsSUFBSSxFQUFFLFlBQVk7Z0JBQ2xCLEdBQUcsRUFBRSxvQkFBb0I7YUFDMUI7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxrRkFBa0YsQ0FBQyxJQUFVO1FBQzNGLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLEdBQUcsR0FBRyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBRWpELE9BQU87UUFDUCxNQUFNLE1BQU0sR0FBRyxJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRTtZQUNsRCxVQUFVLEVBQUUsTUFBTTtZQUNsQixRQUFRLEVBQUUsR0FBRyxDQUFDLElBQUk7U0FDbkIsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLEVBQUU7WUFDOUMsVUFBVSxFQUFFO2dCQUNWLEVBQUU7Z0JBQ0Y7b0JBQ0UsTUFBTTtvQkFDTixFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRTtvQkFDekIsZUFBZTtvQkFDZixFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUU7b0JBQ3RCLEdBQUc7b0JBQ0gsRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUU7b0JBQ3pCLEdBQUc7b0JBQ0gsRUFBRSxHQUFHLEVBQUUsaUJBQWlCLEVBQUU7b0JBQzFCLEdBQUc7b0JBQ0gsRUFBRSxHQUFHLEVBQUUsb0NBQW9DLEVBQUU7b0JBQzdDLFFBQVE7aUJBQ1Q7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwrRUFBK0UsQ0FBQyxJQUFVO1FBQ3hGLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLEdBQUcsR0FBRyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBRWpELE9BQU87UUFDUCxNQUFNLE1BQU0sR0FBRyxJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRTtZQUNsRCxVQUFVLEVBQUUsTUFBTTtZQUNsQixRQUFRLEVBQUUsR0FBRyxDQUFDLElBQUk7U0FDbkIsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLEVBQUU7WUFDbEQsVUFBVSxFQUFFO2dCQUNWLEVBQUU7Z0JBQ0Y7b0JBQ0UsTUFBTTtvQkFDTixFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRTtvQkFDekIsZUFBZTtvQkFDZixFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUU7b0JBQ3RCLEdBQUc7b0JBQ0gsRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUU7b0JBQ3pCLEdBQUc7b0JBQ0gsRUFBRSxHQUFHLEVBQUUsaUJBQWlCLEVBQUU7b0JBQzFCLDBCQUEwQjtpQkFDM0I7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxrRkFBa0YsQ0FBQyxJQUFVO1FBQzNGLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLEdBQUcsR0FBRyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxFQUFFLE1BQU0sRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDO1FBRXBFLE9BQU87UUFDUCxNQUFNLE1BQU0sR0FBRyxJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRSxFQUFFLFVBQVUsRUFBRSxNQUFNLEVBQUUsUUFBUSxFQUFFLEdBQUcsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBRWhHLE9BQU87UUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxFQUFFO1lBQzlDLFVBQVUsRUFBRTtnQkFDVixFQUFFO2dCQUNGO29CQUNFLE1BQU07b0JBQ04sRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUU7b0JBQ3pCLGVBQWU7b0JBQ2YsRUFBRSxHQUFHLEVBQUUsYUFBYSxFQUFFO29CQUN0QixHQUFHO29CQUNILEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFO29CQUN6QixHQUFHO29CQUNILEVBQUUsR0FBRyxFQUFFLGlCQUFpQixFQUFFO29CQUMxQixVQUFVO2lCQUNYO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsd0VBQXdFLENBQUMsSUFBVTtRQUNqRixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLEdBQUcsR0FBRyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQ2pELE1BQU0sS0FBSyxHQUFHLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNsRSxNQUFNLFNBQVMsR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDLFVBQVUsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUMzRSxNQUFNLE1BQU0sR0FBRyxTQUFTLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRTFDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLEVBQUU7WUFDOUMsVUFBVSxFQUFFO2dCQUNWLEVBQUU7Z0JBQ0Y7b0JBQ0UsTUFBTTtvQkFDTixFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRTtvQkFDekIsZUFBZTtvQkFDZixFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUU7b0JBQ3RCLEdBQUc7b0JBQ0gsRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUU7b0JBQ3pCLEdBQUc7b0JBQ0gsRUFBRSxHQUFHLEVBQUUsaUJBQWlCLEVBQUU7b0JBQzFCLEdBQUc7b0JBQ0gsRUFBRSxHQUFHLEVBQUUsb0NBQW9DLEVBQUU7b0JBQzdDLHdCQUF3QjtpQkFDekI7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLEVBQUU7WUFDbEQsVUFBVSxFQUFFO2dCQUNWLEVBQUU7Z0JBQ0Y7b0JBQ0UsTUFBTTtvQkFDTixFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRTtvQkFDekIsZUFBZTtvQkFDZixFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUU7b0JBQ3RCLEdBQUc7b0JBQ0gsRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUU7b0JBQ3pCLEdBQUc7b0JBQ0gsRUFBRSxHQUFHLEVBQUUsaUJBQWlCLEVBQUU7b0JBQzFCLDBDQUEwQztpQkFDM0M7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxpRkFBaUYsQ0FBQyxJQUFVO1FBQzFGLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLEdBQUcsR0FBRyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxFQUFFLE1BQU0sRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDO1FBQ3BFLE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFLEVBQUUsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUUzRixPQUFPO1FBQ1AsR0FBRyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFLElBQUksS0FBSyxDQUFDLFdBQVcsQ0FBQztZQUM5QyxJQUFJLEVBQUUsS0FBSyxDQUFDLGVBQWUsQ0FBQyxTQUFTO1lBQ3JDLE9BQU8sRUFBRTtnQkFDUCxlQUFlLEVBQUUsSUFBSTthQUN0QjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosT0FBTztRQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMseUJBQXlCLEVBQUU7WUFDM0QsV0FBVyxFQUFFO2dCQUNYLFdBQVcsRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFFLGdCQUFnQixFQUFFLEtBQUssQ0FBRSxFQUFFO2FBQzNEO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFDSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNkZBQTZGLENBQUMsSUFBVTtRQUN0RyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsRUFBRSxNQUFNLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQztRQUVwRSxPQUFPO1FBQ1AsR0FBRyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFLElBQUksS0FBSyxDQUFDLFdBQVcsQ0FBQztZQUM5QyxJQUFJLEVBQUUsS0FBSyxDQUFDLGVBQWUsQ0FBQyxTQUFTO1lBQ3JDLE9BQU8sRUFBRTtnQkFDUCxzQkFBc0IsRUFBRSxJQUFJO2FBQzdCO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx5QkFBeUIsRUFBRTtZQUMzRCxXQUFXLEVBQUU7Z0JBQ1gsV0FBVyxFQUFFLEVBQUUsVUFBVSxFQUFFLENBQUUsRUFBRSxFQUFFLENBQUUsTUFBTSxFQUFFLEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFLEVBQUUsZ0JBQWdCLENBQUUsQ0FBRSxFQUFFO2FBQzdGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFDSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNkRBQTZELENBQUMsSUFBVTtRQUN0RSxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsRUFBRSxNQUFNLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQztRQUVwRSxPQUFPO1FBQ1AsSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7WUFDcEMsVUFBVSxFQUFFLEtBQUs7WUFDakIsUUFBUSxFQUFFLEdBQUcsQ0FBQyxJQUFJO1lBQ2xCLE9BQU8sRUFBRTtnQkFDUCxlQUFlLEVBQUUsQ0FBQzt3QkFDaEIsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCLEVBQUU7d0JBQ0QsVUFBVSxFQUFFLEtBQUs7d0JBQ2pCLGtCQUFrQixFQUFFOzRCQUNsQixtQ0FBbUMsRUFBRSxLQUFLO3lCQUMzQztxQkFDRixFQUFFO3dCQUNELFVBQVUsRUFBRSxLQUFLO3dCQUNqQixrQkFBa0IsRUFBRTs0QkFDbEIsaUNBQWlDLEVBQUUsSUFBSTt5QkFDeEM7d0JBQ0QsY0FBYyxFQUFFOzRCQUNkLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxLQUFLLENBQUMsV0FBVzs0QkFDM0MsWUFBWSxFQUFFLEtBQUssQ0FBQyxLQUFLLENBQUMsV0FBVzt5QkFDdEM7cUJBQ0Y7aUJBQ0E7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMseUJBQXlCLEVBQUU7WUFDdkQsVUFBVSxFQUFFLEtBQUs7WUFDakIsZUFBZSxFQUFFLENBQUM7b0JBQ2hCLFVBQVUsRUFBRSxLQUFLO2lCQUNsQixFQUFFO29CQUNELFVBQVUsRUFBRSxLQUFLO29CQUNqQixrQkFBa0IsRUFBRTt3QkFDbEIsbUNBQW1DLEVBQUUsS0FBSztxQkFDM0M7aUJBQ0YsRUFBRTtvQkFDRCxVQUFVLEVBQUUsS0FBSztvQkFDakIsa0JBQWtCLEVBQUU7d0JBQ2xCLGlDQUFpQyxFQUFFLElBQUk7cUJBQ3hDO29CQUNELGNBQWMsRUFBRTt3QkFDZCxrQkFBa0IsRUFBRSxPQUFPO3dCQUMzQixZQUFZLEVBQUUsT0FBTztxQkFDdEI7aUJBQ0Y7YUFDQTtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDRDQUE0QyxDQUFDLElBQVU7UUFDckQsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sR0FBRyxHQUFHLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEVBQUUsTUFBTSxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7UUFFcEUsT0FBTztRQUNQLEdBQUcsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssRUFBRSxJQUFJLEtBQUssQ0FBQyxjQUFjLENBQUM7WUFDakQsT0FBTyxFQUFFLGFBQWE7WUFDdEIsTUFBTSxFQUFFLFdBQVc7WUFDbkIsT0FBTyxFQUFFO2dCQUNQLG9CQUFvQixFQUFFO29CQUNwQjt3QkFDRSxVQUFVLEVBQUUsS0FBSzt3QkFDakIsaUJBQWlCLEVBQUUsRUFBRSxrQkFBa0IsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLEVBQUUsT0FBTyxFQUFFLElBQUksRUFBRSxDQUFDLEVBQUU7cUJBQzdFO29CQUNEO3dCQUNFLGdCQUFnQixFQUFFLFNBQVM7d0JBQzNCLFVBQVUsRUFBRSxLQUFLO3dCQUNqQixpQkFBaUIsRUFBRSxFQUFFLGtCQUFrQixFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLE9BQU8sRUFBRSxpQkFBaUIsRUFBRSxDQUFDLEVBQUU7cUJBQzFHO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMseUJBQXlCLEVBQUU7WUFDdkQsV0FBVyxFQUFFO2dCQUNYLHFCQUFxQixFQUFFLE1BQU07Z0JBQzdCLG9CQUFvQixFQUFFO29CQUNwQjt3QkFDRSxpQkFBaUIsRUFBRSxFQUFFLGtCQUFrQixFQUFFLGtCQUFrQixFQUFFO3dCQUM3RCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0Q7d0JBQ0UsaUJBQWlCLEVBQUUsRUFBRSxrQkFBa0IsRUFBRSwrQ0FBK0MsRUFBRTt3QkFDMUYsZ0JBQWdCLEVBQUUsU0FBUzt3QkFDM0IsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2dCQUNELElBQUksRUFBRSxLQUFLO2dCQUNYLEdBQUcsRUFBRSxFQUFFLFVBQVUsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLE1BQU0sRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxFQUFFLGNBQWMsRUFBRSxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUUsRUFBRSwrQkFBK0IsQ0FBQyxDQUFDLEVBQUM7YUFDdkk7U0FDRixDQUFDLENBQUMsQ0FBQztRQUNKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtQ0FBbUMsQ0FBQyxJQUFVO1FBQzVDLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLEdBQUcsR0FBRyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRTVDLE9BQU87UUFDUCxHQUFHLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUMxQixHQUFHLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUMzQixHQUFHLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUUxQixPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHlCQUF5QixFQUFFLEVBQUUsVUFBVSxFQUFFLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQztRQUNsRixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMseUJBQXlCLEVBQUUsRUFBRSxVQUFVLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQ2pGLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx5QkFBeUIsRUFBRSxFQUFFLFVBQVUsRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDakYsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHlCQUF5QixDQUFDLElBQVU7UUFDbEMsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sR0FBRyxHQUFHLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEVBQUUsTUFBTSxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7UUFDcEUsTUFBTSxLQUFLLEdBQUcsR0FBRyxDQUFDLFFBQVEsQ0FBQyxZQUFZLEVBQUU7WUFDdkMsV0FBVyxFQUFFLGtCQUFrQjtZQUMvQixTQUFTLEVBQUUsWUFBWTtZQUN2QixNQUFNLEVBQUU7Z0JBQ04sS0FBSyxFQUFFLE1BQU07Z0JBQ2IsSUFBSSxFQUFFLEtBQUssQ0FBQyxjQUFjLENBQUMsTUFBTTtnQkFDakMsVUFBVSxFQUFFLEVBQUUsT0FBTyxFQUFFLEVBQUUsSUFBSSxFQUFFLEtBQUssQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLEVBQUU7YUFDL0Q7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7WUFDcEMsVUFBVSxFQUFFLEtBQUs7WUFDakIsUUFBUSxFQUFFLEdBQUcsQ0FBQyxJQUFJO1lBQ2xCLE9BQU8sRUFBRTtnQkFDUCxhQUFhLEVBQUU7b0JBQ2Isa0JBQWtCLEVBQUUsS0FBSztpQkFDMUI7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMseUJBQXlCLEVBQUU7WUFDdkQsVUFBVSxFQUFFLEtBQUs7WUFDakIsYUFBYSxFQUFFO2dCQUNiLGtCQUFrQixFQUFFLEVBQUUsR0FBRyxFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFtQixDQUFDLEVBQUU7YUFDcEc7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw0Q0FBNEMsQ0FBQyxJQUFVO1FBQ3JELFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLEdBQUcsR0FBRyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxFQUFFLE1BQU0sRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDO1FBQ3BFLE1BQU0sU0FBUyxHQUFHLEdBQUcsQ0FBQyxRQUFRLENBQUMsVUFBVSxFQUFFO1lBQ3pDLE1BQU0sRUFBRTtnQkFDTixNQUFNLEVBQUUsS0FBSyxDQUFDLGlCQUFpQixDQUFDLE1BQU07Z0JBQ3RDLEtBQUssRUFBRSxNQUFNO2dCQUNiLElBQUksRUFBRSxLQUFLLENBQUMsY0FBYyxDQUFDLE1BQU07Z0JBQ2pDLFVBQVUsRUFBRSxFQUFFLE9BQU8sRUFBRSxFQUFFLElBQUksRUFBRSxLQUFLLENBQUMsY0FBYyxDQUFDLE1BQU0sRUFBRSxFQUFFO2FBQy9EO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLElBQUksS0FBSyxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO1lBQ3BDLFVBQVUsRUFBRSxLQUFLO1lBQ2pCLFFBQVEsRUFBRSxHQUFHLENBQUMsSUFBSTtZQUNsQixPQUFPLEVBQUU7Z0JBQ1AsZUFBZSxFQUFFLENBQUM7d0JBQ2hCLFVBQVUsRUFBRSxLQUFLO3FCQUNsQixFQUFFO3dCQUNELFVBQVUsRUFBRSxLQUFLO3dCQUNqQixrQkFBa0IsRUFBRTs0QkFDbEIsbUNBQW1DLEVBQUUsS0FBSzt5QkFDM0M7cUJBQ0YsRUFBRTt3QkFDRCxVQUFVLEVBQUUsS0FBSzt3QkFDakIsa0JBQWtCLEVBQUU7NEJBQ2xCLGlDQUFpQyxFQUFFLElBQUk7eUJBQ3hDO3dCQUNELGNBQWMsRUFBRTs0QkFDZCxrQkFBa0IsRUFBRSxLQUFLLENBQUMsS0FBSyxDQUFDLFdBQVc7NEJBQzNDLFlBQVksRUFBRSxLQUFLLENBQUMsS0FBSyxDQUFDLFdBQVc7NEJBQ3JDLFdBQVcsRUFBRSxTQUFTO3lCQUN2QjtxQkFDRjtpQkFDQTthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx5QkFBeUIsRUFBRTtZQUN2RCxVQUFVLEVBQUUsS0FBSztZQUNqQixlQUFlLEVBQUUsQ0FBQztvQkFDaEIsVUFBVSxFQUFFLEtBQUs7aUJBQ2xCLEVBQUU7b0JBQ0QsVUFBVSxFQUFFLEtBQUs7b0JBQ2pCLGtCQUFrQixFQUFFO3dCQUNsQixtQ0FBbUMsRUFBRSxLQUFLO3FCQUMzQztpQkFDRixFQUFFO29CQUNELFVBQVUsRUFBRSxLQUFLO29CQUNqQixrQkFBa0IsRUFBRTt3QkFDbEIsaUNBQWlDLEVBQUUsSUFBSTtxQkFDeEM7b0JBQ0QsY0FBYyxFQUFFO3dCQUNkLGtCQUFrQixFQUFFLE9BQU87d0JBQzNCLFlBQVksRUFBRSxPQUFPO3dCQUNyQixXQUFXLEVBQUUsRUFBRSxHQUFHLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQW1CLENBQUMsRUFBRTtxQkFDakc7aUJBQ0Y7YUFDQTtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGdDQUFnQyxDQUFDLElBQVU7UUFDekMsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sR0FBRyxHQUFHLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEVBQUUsTUFBTSxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7UUFDcEUsTUFBTSxTQUFTLEdBQUcsR0FBRyxDQUFDLG1CQUFtQixDQUFDLFdBQVcsRUFBRTtZQUNyRCxtQkFBbUIsRUFBRSxJQUFJO1lBQ3pCLHlCQUF5QixFQUFFLEtBQUs7U0FDakMsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLElBQUksS0FBSyxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO1lBQ3BDLFVBQVUsRUFBRSxLQUFLO1lBQ2pCLFFBQVEsRUFBRSxHQUFHLENBQUMsSUFBSTtZQUNsQixPQUFPLEVBQUU7Z0JBQ1AsZ0JBQWdCLEVBQUUsU0FBUzthQUM1QjtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMseUJBQXlCLEVBQUU7WUFDdkQsa0JBQWtCLEVBQUUsRUFBRSxHQUFHLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQW1CLENBQUMsRUFBRTtTQUN4RyxDQUFDLENBQUMsQ0FBQztRQUNKLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxtQ0FBbUMsRUFBRTtZQUNqRSxTQUFTLEVBQUUsRUFBRSxHQUFHLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQW1CLENBQUMsRUFBRTtZQUN4RixtQkFBbUIsRUFBRSxJQUFJO1lBQ3pCLHlCQUF5QixFQUFFLEtBQUs7U0FDakMsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsK0JBQStCLENBQUMsSUFBVTtRQUN4QyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDL0MsY0FBYyxFQUFFLEtBQUs7WUFDckIsTUFBTSxFQUFFLEtBQUs7WUFDYixvQkFBb0IsRUFBRTtnQkFDcEIsaUJBQWlCLEVBQUUsRUFBQywyQkFBMkIsRUFBRSxJQUFJLEVBQUM7YUFDdkQ7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSwwQkFBMEIsRUFBRTtZQUNsRCxVQUFVLEVBQUUsTUFBTTtZQUNsQixRQUFRLEVBQUUsR0FBRyxDQUFDLElBQUk7WUFDbEIsT0FBTyxFQUFFO2dCQUNQLGFBQWEsRUFBRSwwQkFBMEI7YUFDMUM7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHlCQUF5QixFQUFFO1lBQ3ZELGFBQWEsRUFBRSwwQkFBMEI7WUFDekMsaUJBQWlCLEVBQUU7Z0JBQ2pCLDJCQUEyQixFQUFFLElBQUk7YUFDbEM7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwrQkFBK0IsQ0FBQyxJQUFVO1FBQ3hDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE1BQU0sT0FBTyxHQUFHLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDdEQsT0FBTyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTtZQUN2QyxVQUFVLEVBQUUsZ0JBQWdCO1NBQzdCLENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx5QkFBeUIsRUFBRTtZQUN2RCxVQUFVLEVBQUUsS0FBSztZQUNqQixpQkFBaUIsRUFBRSxRQUFRO1lBQzNCLFlBQVksRUFBRSxnQkFBZ0IsQ0FBQyxZQUFZO1NBQzVDLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHVDQUF1QyxDQUFDLElBQVU7UUFDaEQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFOUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7WUFDcEQsT0FBTyxFQUFFLFNBQVM7WUFDbEIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQztZQUNuQyxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1NBQ3BDLENBQUMsQ0FBQztRQUVILE1BQU0sSUFBSSxHQUFHLElBQUksS0FBSyxDQUFDLGVBQWUsQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFO1lBQzdELGNBQWMsRUFBRSxlQUFlO1lBQy9CLE9BQU8sRUFBRSxJQUFJO1NBQ2QsQ0FBQyxDQUFDO1FBRUgsTUFBTSxPQUFPLEdBQUcsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUU7WUFDcEQsb0JBQW9CLEVBQUU7Z0JBQ3BCLFVBQVUsRUFBRSxJQUFJO2FBQ2pCO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsT0FBTyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFOUIsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLDZCQUE2QixFQUFFO1lBQzNELElBQUksRUFBRSxlQUFlO1lBQ3JCLElBQUksRUFBRSxPQUFPO1lBQ2IsU0FBUyxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQztTQUM1QyxDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw2REFBNkQsQ0FBQyxJQUFVO1FBQ3RFLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE1BQU0sT0FBTyxHQUFHLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFFdEQsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDZixPQUFPLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO2dCQUN2QyxpQkFBaUIsRUFBRSxLQUFLLENBQUMsaUJBQWlCLENBQUMsR0FBRztnQkFDOUMsVUFBVSxFQUFFLGdCQUFnQjthQUM3QixDQUFDLENBQUM7UUFDTCxDQUFDLEVBQUUsaUdBQWlHLENBQUMsQ0FBQztRQUV0RyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsdUZBQXVGLENBQUMsSUFBVTtRQUNoRyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUU5QixNQUFNLE9BQU8sR0FBRyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRTtZQUNwRCxvQkFBb0IsRUFBRTtnQkFDcEIsVUFBVSxFQUFFLGdCQUFnQjthQUM3QjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO1lBQ2YsT0FBTyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTtnQkFDdkMsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLGlCQUFpQixDQUFDLElBQUk7YUFDaEQsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxFQUFFLDhGQUE4RixDQUFDLENBQUM7UUFFbkcsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHdCQUF3QixDQUFDLElBQVU7UUFDakMsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sR0FBRyxHQUFHLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEVBQUUsY0FBYyxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQztRQUUzRixPQUFPO1FBQ1AsSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUU7WUFDbkMsVUFBVSxFQUFFLE1BQU07WUFDbEIsUUFBUSxFQUFFLEdBQUcsQ0FBQyxJQUFJO1lBQ2xCLE9BQU8sRUFBRTtnQkFDUCxjQUFjLEVBQUUsSUFBSTtnQkFDcEIsbUJBQW1CLEVBQUUsQ0FBQyxZQUFZLEVBQUUsWUFBWSxDQUFDO2FBQ2xEO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx5QkFBeUIsRUFBRTtZQUN2RCxjQUFjLEVBQUUsSUFBSTtZQUNwQixtQkFBbUIsRUFBRSxDQUFDLFlBQVksRUFBRSxZQUFZLENBQUM7U0FDbEQsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQseUJBQXlCLENBQUMsSUFBVTtRQUNsQyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDL0MsY0FBYyxFQUFFLEtBQUs7WUFDckIsTUFBTSxFQUFFLEtBQUs7WUFDYixvQkFBb0IsRUFBRTtnQkFDcEIsbUJBQW1CLEVBQUUsQ0FBQyxhQUFhLENBQUM7YUFDckM7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxtQkFBbUIsRUFBRTtZQUMzQyxVQUFVLEVBQUUsTUFBTTtZQUNsQixRQUFRLEVBQUUsR0FBRyxDQUFDLElBQUk7WUFDbEIsT0FBTyxFQUFFO2dCQUNQLGFBQWEsRUFBRSxtQkFBbUI7YUFDbkM7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHlCQUF5QixFQUFFO1lBQ3ZELGFBQWEsRUFBRSxtQkFBbUI7WUFDbEMsbUJBQW1CLEVBQUUsQ0FBQyxhQUFhLENBQUM7U0FDckMsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQseUNBQXlDLENBQUMsSUFBVTtRQUNsRCxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDL0MsY0FBYyxFQUFFLEtBQUs7WUFDckIsTUFBTSxFQUFFLEtBQUs7WUFDYixvQkFBb0IsRUFBRTtnQkFDcEIsbUJBQW1CLEVBQUUsQ0FBQyxhQUFhLENBQUM7YUFDckM7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxxQkFBcUIsRUFBRTtZQUM3QyxVQUFVLEVBQUUsTUFBTTtZQUNsQixRQUFRLEVBQUUsR0FBRyxDQUFDLElBQUk7WUFDbEIsT0FBTyxFQUFFO2dCQUNQLGNBQWMsRUFBRSxJQUFJO2dCQUNwQixtQkFBbUIsRUFBRSxDQUFDLGlCQUFpQixDQUFDO2FBQ3pDO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx5QkFBeUIsRUFBRTtZQUN2RCxjQUFjLEVBQUUsSUFBSTtZQUNwQixtQkFBbUIsRUFBRSxDQUFDLGlCQUFpQixDQUFDO1NBQ3pDLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELG9CQUFvQixDQUFDLElBQVU7UUFDN0IsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sR0FBRyxHQUFHLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQy9DLGNBQWMsRUFBRSxLQUFLO1lBQ3JCLE1BQU0sRUFBRSxLQUFLO1NBQ2QsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLElBQUksS0FBSyxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUU7WUFDMUMsVUFBVSxFQUFFLE1BQU07WUFDbEIsUUFBUSxFQUFFLEdBQUcsQ0FBQyxJQUFJO1lBQ2xCLE9BQU8sRUFBRTtnQkFDUCxhQUFhLEVBQUUsa0JBQWtCO2FBQ2xDO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx5QkFBeUIsRUFBRTtZQUN2RCxhQUFhLEVBQUUsa0JBQWtCO1lBQ2pDLG1CQUFtQixFQUFFLGVBQU07U0FDNUIsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQseURBQXlELENBQUMsSUFBVTtRQUNsRSxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsRUFBRSxNQUFNLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQztRQUVwRSxPQUFPO1FBQ1AsSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7WUFDcEMsVUFBVSxFQUFFLEtBQUs7WUFDakIsUUFBUSxFQUFFLEdBQUcsQ0FBQyxJQUFJO1lBQ2xCLE9BQU8sRUFBRTtnQkFDUCx1QkFBdUIsRUFBRTtvQkFDdkIsb0JBQW9CLEVBQUUsZ0JBQWdCO29CQUN0QyxtQkFBbUIsRUFBRSxJQUFJO29CQUN6Qix5QkFBeUIsRUFBRSxLQUFLO2lCQUNqQzthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxtQ0FBbUMsRUFBRTtZQUNqRSxTQUFTLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsU0FBUyxDQUFDO1lBQ3ZDLG1CQUFtQixFQUFFLElBQUk7WUFDekIseUJBQXlCLEVBQUUsS0FBSztZQUNoQyxJQUFJLEVBQUUsZ0JBQWdCO1NBQ3ZCLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDBDQUEwQyxDQUFDLElBQVU7UUFDbkQsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sR0FBRyxHQUFHLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEVBQUUsTUFBTSxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7UUFFcEUsT0FBTztRQUNQLElBQUksS0FBSyxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO1lBQ3BDLFVBQVUsRUFBRSxLQUFLO1lBQ2pCLFFBQVEsRUFBRSxHQUFHLENBQUMsSUFBSTtTQUNuQixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHlCQUF5QixFQUFFO1lBQ3ZELGtCQUFrQixFQUFFLGVBQU07U0FDM0IsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsOEVBQThFLENBQUMsSUFBVTtRQUN2RixRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsRUFBRSxNQUFNLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQztRQUNwRSxNQUFNLFNBQVMsR0FBRyxHQUFHLENBQUMsbUJBQW1CLENBQUMsaUJBQWlCLEVBQUU7WUFDM0QsbUJBQW1CLEVBQUUsSUFBSTtZQUN6Qix5QkFBeUIsRUFBRSxLQUFLO1NBQ2pDLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLFdBQVcsR0FBRztZQUNsQixVQUFVLEVBQUUsS0FBSztZQUNqQixRQUFRLEVBQUUsR0FBRyxDQUFDLElBQUk7WUFDbEIsT0FBTyxFQUFFO2dCQUNQLHVCQUF1QixFQUFFO29CQUN2QixvQkFBb0IsRUFBRSxpQkFBaUI7b0JBQ3ZDLG1CQUFtQixFQUFFLElBQUk7b0JBQ3pCLHlCQUF5QixFQUFFLEtBQUs7aUJBQ2pDO2dCQUNELGdCQUFnQixFQUFFLFNBQVM7YUFDNUI7U0FDRixDQUFDO1FBRUYsT0FBTztRQUNQLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUUsV0FBVyxDQUFDLEVBQzlELGdGQUFnRixDQUFDLENBQUM7UUFFcEYsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDZEQUE2RCxDQUFDLElBQVU7UUFDdEUsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE9BQU87UUFDUCxNQUFNLEdBQUcsR0FBRyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQ2pELE1BQU0sTUFBTSxHQUFHLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUU3RCxPQUFPO1FBQ1AsSUFBSSxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDeEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDcEIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsU0FBUyxDQUFDLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUM7UUFFN0YsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHFFQUFxRSxDQUFDLElBQVU7UUFDOUUsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE9BQU87UUFDUCxNQUFNLEdBQUcsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLHFCQUFxQixDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDakUsU0FBUyxFQUFFLGtCQUFrQjtZQUM3QixjQUFjLEVBQUUsdUJBQXVCO1NBQ3hDLENBQUMsQ0FBQztRQUNILE1BQU0sTUFBTSxHQUFHLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUU3RCxPQUFPO1FBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxNQUFNLENBQUMsT0FBTyxFQUFFLG1FQUFtRSxDQUFDLENBQUM7UUFDdkcsSUFBSSxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUM7UUFFcEIsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztDQUNGLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBQlNFTlQsIGV4cGVjdCwgaGF2ZVJlc291cmNlLCBoYXZlUmVzb3VyY2VMaWtlIH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IFRlc3QgfSBmcm9tICdub2RldW5pdCc7XG5pbXBvcnQgKiBhcyBhcGlndyBmcm9tICcuLi9saWInO1xuXG5jb25zdCBEVU1NWV9BVVRIT1JJWkVSOiBhcGlndy5JQXV0aG9yaXplciA9IHtcbiAgYXV0aG9yaXplcklkOiAnZHVtbXlhdXRob3JpemVyJyxcbiAgYXV0aG9yaXphdGlvblR5cGU6IGFwaWd3LkF1dGhvcml6YXRpb25UeXBlLkNVU1RPTSxcbn07XG5cbmV4cG9ydCA9IHtcbiAgJ2RlZmF1bHQgc2V0dXAnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IGFwaSA9IG5ldyBhcGlndy5SZXN0QXBpKHN0YWNrLCAndGVzdC1hcGknLCB7IGNsb3VkV2F0Y2hSb2xlOiBmYWxzZSwgZGVwbG95OiBmYWxzZSB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgYXBpZ3cuTWV0aG9kKHN0YWNrLCAnbXktbWV0aG9kJywge1xuICAgICAgaHR0cE1ldGhvZDogJ1BPU1QnLFxuICAgICAgcmVzb3VyY2U6IGFwaS5yb290LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkFwaUdhdGV3YXk6Ok1ldGhvZCcsIHtcbiAgICAgIEh0dHBNZXRob2Q6ICdQT1NUJyxcbiAgICAgIEF1dGhvcml6YXRpb25UeXBlOiAnTk9ORScsXG4gICAgICBJbnRlZ3JhdGlvbjoge1xuICAgICAgICBUeXBlOiAnTU9DSycsXG4gICAgICB9LFxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdtZXRob2Qgb3B0aW9ucyBjYW4gYmUgc3BlY2lmaWVkJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCBhcGkgPSBuZXcgYXBpZ3cuUmVzdEFwaShzdGFjaywgJ3Rlc3QtYXBpJywgeyBjbG91ZFdhdGNoUm9sZTogZmFsc2UsIGRlcGxveTogZmFsc2UgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IGFwaWd3Lk1ldGhvZChzdGFjaywgJ215LW1ldGhvZCcsIHtcbiAgICAgIGh0dHBNZXRob2Q6ICdQT1NUJyxcbiAgICAgIHJlc291cmNlOiBhcGkucm9vdCxcbiAgICAgIG9wdGlvbnM6IHtcbiAgICAgICAgYXBpS2V5UmVxdWlyZWQ6IHRydWUsXG4gICAgICAgIG9wZXJhdGlvbk5hbWU6ICdNeU9wZXJhdGlvbicsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkFwaUdhdGV3YXk6Ok1ldGhvZCcsIHtcbiAgICAgIEFwaUtleVJlcXVpcmVkOiB0cnVlLFxuICAgICAgT3BlcmF0aW9uTmFtZTogJ015T3BlcmF0aW9uJyxcbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnaW50ZWdyYXRpb24gY2FuIGJlIHNldCB2aWEgYSBwcm9wZXJ0eScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgYXBpID0gbmV3IGFwaWd3LlJlc3RBcGkoc3RhY2ssICd0ZXN0LWFwaScsIHsgY2xvdWRXYXRjaFJvbGU6IGZhbHNlLCBkZXBsb3k6IGZhbHNlIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBhcGlndy5NZXRob2Qoc3RhY2ssICdteS1tZXRob2QnLCB7XG4gICAgICBodHRwTWV0aG9kOiAnUE9TVCcsXG4gICAgICByZXNvdXJjZTogYXBpLnJvb3QsXG4gICAgICBpbnRlZ3JhdGlvbjogbmV3IGFwaWd3LkF3c0ludGVncmF0aW9uKHsgc2VydmljZTogJ3MzJywgcGF0aDogJ2J1Y2tldC9rZXknIH0pLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkFwaUdhdGV3YXk6Ok1ldGhvZCcsIHtcbiAgICAgIEludGVncmF0aW9uOiB7XG4gICAgICAgIEludGVncmF0aW9uSHR0cE1ldGhvZDogJ1BPU1QnLFxuICAgICAgICBUeXBlOiAnQVdTJyxcbiAgICAgICAgVXJpOiB7XG4gICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgJycsXG4gICAgICAgICAgICBbXG4gICAgICAgICAgICAgICdhcm46JywgeyBSZWY6ICdBV1M6OlBhcnRpdGlvbicgfSwgJzphcGlnYXRld2F5OicsXG4gICAgICAgICAgICAgIHsgUmVmOiAnQVdTOjpSZWdpb24nIH0sICc6czM6cGF0aC9idWNrZXQva2V5JyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2ludGVncmF0aW9uIHdpdGggYSBjdXN0b20gaHR0cCBtZXRob2QgY2FuIGJlIHNldCB2aWEgYSBwcm9wZXJ0eScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgYXBpID0gbmV3IGFwaWd3LlJlc3RBcGkoc3RhY2ssICd0ZXN0LWFwaScsIHsgY2xvdWRXYXRjaFJvbGU6IGZhbHNlLCBkZXBsb3k6IGZhbHNlIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBhcGlndy5NZXRob2Qoc3RhY2ssICdteS1tZXRob2QnLCB7XG4gICAgICBodHRwTWV0aG9kOiAnUE9TVCcsXG4gICAgICByZXNvdXJjZTogYXBpLnJvb3QsXG4gICAgICBpbnRlZ3JhdGlvbjogbmV3IGFwaWd3LkF3c0ludGVncmF0aW9uKHsgc2VydmljZTogJ3MzJywgcGF0aDogJ2J1Y2tldC9rZXknLCBpbnRlZ3JhdGlvbkh0dHBNZXRob2Q6ICdHRVQnIH0pLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpBcGlHYXRld2F5OjpNZXRob2QnLCB7XG4gICAgICBJbnRlZ3JhdGlvbjoge1xuICAgICAgICBJbnRlZ3JhdGlvbkh0dHBNZXRob2Q6ICdHRVQnLFxuICAgICAgfSxcbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAndXNlIGRlZmF1bHQgaW50ZWdyYXRpb24gZnJvbSBhcGknKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IGRlZmF1bHRJbnRlZ3JhdGlvbiA9IG5ldyBhcGlndy5JbnRlZ3JhdGlvbih7IHR5cGU6IGFwaWd3LkludGVncmF0aW9uVHlwZS5IVFRQX1BST1hZLCB1cmk6ICdodHRwczovL2FtYXpvbi5jb20nIH0pO1xuICAgIGNvbnN0IGFwaSA9IG5ldyBhcGlndy5SZXN0QXBpKHN0YWNrLCAndGVzdC1hcGknLCB7XG4gICAgICBjbG91ZFdhdGNoUm9sZTogZmFsc2UsXG4gICAgICBkZXBsb3k6IGZhbHNlLFxuICAgICAgZGVmYXVsdEludGVncmF0aW9uLFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBhcGlndy5NZXRob2Qoc3RhY2ssICdteS1tZXRob2QnLCB7XG4gICAgICBodHRwTWV0aG9kOiAnUE9TVCcsXG4gICAgICByZXNvdXJjZTogYXBpLnJvb3QsXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6QXBpR2F0ZXdheTo6TWV0aG9kJywge1xuICAgICAgSW50ZWdyYXRpb246IHtcbiAgICAgICAgVHlwZTogJ0hUVFBfUFJPWFknLFxuICAgICAgICBVcmk6ICdodHRwczovL2FtYXpvbi5jb20nLFxuICAgICAgfSxcbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnXCJtZXRob2RBcm5cIiByZXR1cm5zIHRoZSBBUk4gZXhlY3V0ZS1hcGkgQVJOIGZvciB0aGlzIG1ldGhvZCBpbiB0aGUgY3VycmVudCBzdGFnZScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgYXBpID0gbmV3IGFwaWd3LlJlc3RBcGkoc3RhY2ssICd0ZXN0LWFwaScpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IG1ldGhvZCA9IG5ldyBhcGlndy5NZXRob2Qoc3RhY2ssICdteS1tZXRob2QnLCB7XG4gICAgICBodHRwTWV0aG9kOiAnUE9TVCcsXG4gICAgICByZXNvdXJjZTogYXBpLnJvb3QsXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZShtZXRob2QubWV0aG9kQXJuKSwge1xuICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAnJyxcbiAgICAgICAgW1xuICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICB7IFJlZjogJ0FXUzo6UGFydGl0aW9uJyB9LFxuICAgICAgICAgICc6ZXhlY3V0ZS1hcGk6JyxcbiAgICAgICAgICB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LFxuICAgICAgICAgICc6JyxcbiAgICAgICAgICB7IFJlZjogJ0FXUzo6QWNjb3VudElkJyB9LFxuICAgICAgICAgICc6JyxcbiAgICAgICAgICB7IFJlZjogJ3Rlc3RhcGlENjQ1MUY3MCcgfSxcbiAgICAgICAgICAnLycsXG4gICAgICAgICAgeyBSZWY6ICd0ZXN0YXBpRGVwbG95bWVudFN0YWdlcHJvZDVDOUU5MkE0JyB9LFxuICAgICAgICAgICcvUE9TVC8nLFxuICAgICAgICBdLFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdcInRlc3RNZXRob2RBcm5cIiByZXR1cm5zIHRoZSBBUk4gb2YgdGhlIFwidGVzdC1pbnZva2Utc3RhZ2VcIiBzdGFnZSAoY29uc29sZSBVSSknKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IGFwaSA9IG5ldyBhcGlndy5SZXN0QXBpKHN0YWNrLCAndGVzdC1hcGknKTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCBtZXRob2QgPSBuZXcgYXBpZ3cuTWV0aG9kKHN0YWNrLCAnbXktbWV0aG9kJywge1xuICAgICAgaHR0cE1ldGhvZDogJ1BPU1QnLFxuICAgICAgcmVzb3VyY2U6IGFwaS5yb290LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUobWV0aG9kLnRlc3RNZXRob2RBcm4pLCB7XG4gICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICcnLFxuICAgICAgICBbXG4gICAgICAgICAgJ2FybjonLFxuICAgICAgICAgIHsgUmVmOiAnQVdTOjpQYXJ0aXRpb24nIH0sXG4gICAgICAgICAgJzpleGVjdXRlLWFwaTonLFxuICAgICAgICAgIHsgUmVmOiAnQVdTOjpSZWdpb24nIH0sXG4gICAgICAgICAgJzonLFxuICAgICAgICAgIHsgUmVmOiAnQVdTOjpBY2NvdW50SWQnIH0sXG4gICAgICAgICAgJzonLFxuICAgICAgICAgIHsgUmVmOiAndGVzdGFwaUQ2NDUxRjcwJyB9LFxuICAgICAgICAgICcvdGVzdC1pbnZva2Utc3RhZ2UvUE9TVC8nLFxuICAgICAgICBdLFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdcIm1ldGhvZEFyblwiIHJldHVybnMgYW4gYXJuIHdpdGggXCIqXCIgYXMgaXRzIHN0YWdlIHdoZW4gZGVwbG95bWVudFN0YWdlIGlzIG5vdCBzZXQnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IGFwaSA9IG5ldyBhcGlndy5SZXN0QXBpKHN0YWNrLCAndGVzdC1hcGknLCB7IGRlcGxveTogZmFsc2UgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgbWV0aG9kID0gbmV3IGFwaWd3Lk1ldGhvZChzdGFjaywgJ215LW1ldGhvZCcsIHsgaHR0cE1ldGhvZDogJ1BPU1QnLCByZXNvdXJjZTogYXBpLnJvb3QgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZShtZXRob2QubWV0aG9kQXJuKSwge1xuICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAnJyxcbiAgICAgICAgW1xuICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICB7IFJlZjogJ0FXUzo6UGFydGl0aW9uJyB9LFxuICAgICAgICAgICc6ZXhlY3V0ZS1hcGk6JyxcbiAgICAgICAgICB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LFxuICAgICAgICAgICc6JyxcbiAgICAgICAgICB7IFJlZjogJ0FXUzo6QWNjb3VudElkJyB9LFxuICAgICAgICAgICc6JyxcbiAgICAgICAgICB7IFJlZjogJ3Rlc3RhcGlENjQ1MUY3MCcgfSxcbiAgICAgICAgICAnLyovUE9TVC8nLFxuICAgICAgICBdLFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdcIm1ldGhvZEFyblwiIGFuZCBcInRlc3RNZXRob2RBcm5cIiByZXBsYWNlIHBhdGggcGFyYW1ldGVycyB3aXRoIGFzdGVyaXNrcycodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IGFwaSA9IG5ldyBhcGlndy5SZXN0QXBpKHN0YWNrLCAndGVzdC1hcGknKTtcbiAgICBjb25zdCBwZXRJZCA9IGFwaS5yb290LmFkZFJlc291cmNlKCdwZXRzJykuYWRkUmVzb3VyY2UoJ3twZXRJZH0nKTtcbiAgICBjb25zdCBjb21tZW50SWQgPSBwZXRJZC5hZGRSZXNvdXJjZSgnY29tbWVudHMnKS5hZGRSZXNvdXJjZSgne2NvbW1lbnRJZH0nKTtcbiAgICBjb25zdCBtZXRob2QgPSBjb21tZW50SWQuYWRkTWV0aG9kKCdHRVQnKTtcblxuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUobWV0aG9kLm1ldGhvZEFybiksIHtcbiAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgJycsXG4gICAgICAgIFtcbiAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAgeyBSZWY6ICdBV1M6OlBhcnRpdGlvbicgfSxcbiAgICAgICAgICAnOmV4ZWN1dGUtYXBpOicsXG4gICAgICAgICAgeyBSZWY6ICdBV1M6OlJlZ2lvbicgfSxcbiAgICAgICAgICAnOicsXG4gICAgICAgICAgeyBSZWY6ICdBV1M6OkFjY291bnRJZCcgfSxcbiAgICAgICAgICAnOicsXG4gICAgICAgICAgeyBSZWY6ICd0ZXN0YXBpRDY0NTFGNzAnIH0sXG4gICAgICAgICAgJy8nLFxuICAgICAgICAgIHsgUmVmOiAndGVzdGFwaURlcGxveW1lbnRTdGFnZXByb2Q1QzlFOTJBNCcgfSxcbiAgICAgICAgICAnL0dFVC9wZXRzLyovY29tbWVudHMvKicsXG4gICAgICAgIF0sXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZShtZXRob2QudGVzdE1ldGhvZEFybiksIHtcbiAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgJycsXG4gICAgICAgIFtcbiAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAgeyBSZWY6ICdBV1M6OlBhcnRpdGlvbicgfSxcbiAgICAgICAgICAnOmV4ZWN1dGUtYXBpOicsXG4gICAgICAgICAgeyBSZWY6ICdBV1M6OlJlZ2lvbicgfSxcbiAgICAgICAgICAnOicsXG4gICAgICAgICAgeyBSZWY6ICdBV1M6OkFjY291bnRJZCcgfSxcbiAgICAgICAgICAnOicsXG4gICAgICAgICAgeyBSZWY6ICd0ZXN0YXBpRDY0NTFGNzAnIH0sXG4gICAgICAgICAgJy90ZXN0LWludm9rZS1zdGFnZS9HRVQvcGV0cy8qL2NvbW1lbnRzLyonLFxuICAgICAgICBdLFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdpbnRlZ3JhdGlvbiBcImNyZWRlbnRpYWxzUm9sZVwiIGNhbiBiZSB1c2VkIHRvIGFzc3VtZSBhIHJvbGUgd2hlbiBjYWxsaW5nIGJhY2tlbmQnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IGFwaSA9IG5ldyBhcGlndy5SZXN0QXBpKHN0YWNrLCAndGVzdC1hcGknLCB7IGRlcGxveTogZmFsc2UgfSk7XG4gICAgY29uc3Qgcm9sZSA9IG5ldyBpYW0uUm9sZShzdGFjaywgJ015Um9sZScsIHsgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2ZvbycpIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGFwaS5yb290LmFkZE1ldGhvZCgnR0VUJywgbmV3IGFwaWd3LkludGVncmF0aW9uKHtcbiAgICAgIHR5cGU6IGFwaWd3LkludGVncmF0aW9uVHlwZS5BV1NfUFJPWFksXG4gICAgICBvcHRpb25zOiB7XG4gICAgICAgIGNyZWRlbnRpYWxzUm9sZTogcm9sZSxcbiAgICAgIH0sXG4gICAgfSkpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpBcGlHYXRld2F5OjpNZXRob2QnLCB7XG4gICAgICBJbnRlZ3JhdGlvbjoge1xuICAgICAgICBDcmVkZW50aWFsczogeyAnRm46OkdldEF0dCc6IFsgJ015Um9sZUY0OEZGRTA0JywgJ0FybicgXSB9LFxuICAgICAgfSxcbiAgICB9KSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2ludGVncmF0aW9uIFwiY3JlZGVudGlhbHNQYXNzdGhyb3VnaFwiIGNhbiBiZSB1c2VkIHRvIHBhc3N0aHJvdWdoIHVzZXIgY3JlZGVudGlhbHMgdG8gYmFja2VuZCcodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgYXBpID0gbmV3IGFwaWd3LlJlc3RBcGkoc3RhY2ssICd0ZXN0LWFwaScsIHsgZGVwbG95OiBmYWxzZSB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBhcGkucm9vdC5hZGRNZXRob2QoJ0dFVCcsIG5ldyBhcGlndy5JbnRlZ3JhdGlvbih7XG4gICAgICB0eXBlOiBhcGlndy5JbnRlZ3JhdGlvblR5cGUuQVdTX1BST1hZLFxuICAgICAgb3B0aW9uczoge1xuICAgICAgICBjcmVkZW50aWFsc1Bhc3N0aHJvdWdoOiB0cnVlLFxuICAgICAgfSxcbiAgICB9KSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkFwaUdhdGV3YXk6Ok1ldGhvZCcsIHtcbiAgICAgIEludGVncmF0aW9uOiB7XG4gICAgICAgIENyZWRlbnRpYWxzOiB7ICdGbjo6Sm9pbic6IFsgJycsIFsgJ2FybjonLCB7IFJlZjogJ0FXUzo6UGFydGl0aW9uJyB9LCAnOmlhbTo6Kjp1c2VyLyonIF0gXSB9LFxuICAgICAgfSxcbiAgICB9KSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ21ldGhvZFJlc3BvbnNlIHNldCBvbmUgb3IgbW9yZSBtZXRob2QgcmVzcG9uc2VzIHZpYSBvcHRpb25zJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCBhcGkgPSBuZXcgYXBpZ3cuUmVzdEFwaShzdGFjaywgJ3Rlc3QtYXBpJywgeyBkZXBsb3k6IGZhbHNlIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBhcGlndy5NZXRob2Qoc3RhY2ssICdtZXRob2QtbWFuJywge1xuICAgICAgaHR0cE1ldGhvZDogJ0dFVCcsXG4gICAgICByZXNvdXJjZTogYXBpLnJvb3QsXG4gICAgICBvcHRpb25zOiB7XG4gICAgICAgIG1ldGhvZFJlc3BvbnNlczogW3tcbiAgICAgICAgICBzdGF0dXNDb2RlOiAnMjAwJyxcbiAgICAgICAgfSwge1xuICAgICAgICAgIHN0YXR1c0NvZGU6ICc0MDAnLFxuICAgICAgICAgIHJlc3BvbnNlUGFyYW1ldGVyczoge1xuICAgICAgICAgICAgJ21ldGhvZC5yZXNwb25zZS5oZWFkZXIua2lsbGVyYmVlcyc6IGZhbHNlLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sIHtcbiAgICAgICAgICBzdGF0dXNDb2RlOiAnNTAwJyxcbiAgICAgICAgICByZXNwb25zZVBhcmFtZXRlcnM6IHtcbiAgICAgICAgICAgICdtZXRob2QucmVzcG9uc2UuaGVhZGVyLmVycnRoaW5nJzogdHJ1ZSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHJlc3BvbnNlTW9kZWxzOiB7XG4gICAgICAgICAgICAnYXBwbGljYXRpb24vanNvbic6IGFwaWd3Lk1vZGVsLkVNUFRZX01PREVMLFxuICAgICAgICAgICAgJ3RleHQvcGxhaW4nOiBhcGlndy5Nb2RlbC5FUlJPUl9NT0RFTCxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpBcGlHYXRld2F5OjpNZXRob2QnLCB7XG4gICAgICBIdHRwTWV0aG9kOiAnR0VUJyxcbiAgICAgIE1ldGhvZFJlc3BvbnNlczogW3tcbiAgICAgICAgU3RhdHVzQ29kZTogJzIwMCcsXG4gICAgICB9LCB7XG4gICAgICAgIFN0YXR1c0NvZGU6ICc0MDAnLFxuICAgICAgICBSZXNwb25zZVBhcmFtZXRlcnM6IHtcbiAgICAgICAgICAnbWV0aG9kLnJlc3BvbnNlLmhlYWRlci5raWxsZXJiZWVzJzogZmFsc2UsXG4gICAgICAgIH0sXG4gICAgICB9LCB7XG4gICAgICAgIFN0YXR1c0NvZGU6ICc1MDAnLFxuICAgICAgICBSZXNwb25zZVBhcmFtZXRlcnM6IHtcbiAgICAgICAgICAnbWV0aG9kLnJlc3BvbnNlLmhlYWRlci5lcnJ0aGluZyc6IHRydWUsXG4gICAgICAgIH0sXG4gICAgICAgIFJlc3BvbnNlTW9kZWxzOiB7XG4gICAgICAgICAgJ2FwcGxpY2F0aW9uL2pzb24nOiAnRW1wdHknLFxuICAgICAgICAgICd0ZXh0L3BsYWluJzogJ0Vycm9yJyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgICBdLFxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdtdWx0aXBsZSBpbnRlZ3JhdGlvbiByZXNwb25zZXMgY2FuIGJlIHVzZWQnKHRlc3Q6IFRlc3QpIHsgLy8gQHNlZSBodHRwczovL2dpdGh1Yi5jb20vYXdzL2F3cy1jZGsvaXNzdWVzLzE2MDhcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IGFwaSA9IG5ldyBhcGlndy5SZXN0QXBpKHN0YWNrLCAndGVzdC1hcGknLCB7IGRlcGxveTogZmFsc2UgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgYXBpLnJvb3QuYWRkTWV0aG9kKCdHRVQnLCBuZXcgYXBpZ3cuQXdzSW50ZWdyYXRpb24oe1xuICAgICAgc2VydmljZTogJ2Zvby1zZXJ2aWNlJyxcbiAgICAgIGFjdGlvbjogJ0JhckFjdGlvbicsXG4gICAgICBvcHRpb25zOiB7XG4gICAgICAgIGludGVncmF0aW9uUmVzcG9uc2VzOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgc3RhdHVzQ29kZTogJzIwMCcsXG4gICAgICAgICAgICByZXNwb25zZVRlbXBsYXRlczogeyAnYXBwbGljYXRpb24vanNvbic6IEpTT04uc3RyaW5naWZ5KHsgc3VjY2VzczogdHJ1ZSB9KSB9LFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgc2VsZWN0aW9uUGF0dGVybjogJ0ludmFsaWQnLFxuICAgICAgICAgICAgc3RhdHVzQ29kZTogJzUwMycsXG4gICAgICAgICAgICByZXNwb25zZVRlbXBsYXRlczogeyAnYXBwbGljYXRpb24vanNvbic6IEpTT04uc3RyaW5naWZ5KHsgc3VjY2VzczogZmFsc2UsIG1lc3NhZ2U6ICdJbnZhbGlkIFJlcXVlc3QnIH0pIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgfSkpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkFwaUdhdGV3YXk6Ok1ldGhvZCcsIHtcbiAgICAgIEludGVncmF0aW9uOiB7XG4gICAgICAgIEludGVncmF0aW9uSHR0cE1ldGhvZDogJ1BPU1QnLFxuICAgICAgICBJbnRlZ3JhdGlvblJlc3BvbnNlczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIFJlc3BvbnNlVGVtcGxhdGVzOiB7ICdhcHBsaWNhdGlvbi9qc29uJzogJ3tcInN1Y2Nlc3NcIjp0cnVlfScgfSxcbiAgICAgICAgICAgIFN0YXR1c0NvZGU6ICcyMDAnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVzcG9uc2VUZW1wbGF0ZXM6IHsgJ2FwcGxpY2F0aW9uL2pzb24nOiAne1wic3VjY2Vzc1wiOmZhbHNlLFwibWVzc2FnZVwiOlwiSW52YWxpZCBSZXF1ZXN0XCJ9JyB9LFxuICAgICAgICAgICAgU2VsZWN0aW9uUGF0dGVybjogJ0ludmFsaWQnLFxuICAgICAgICAgICAgU3RhdHVzQ29kZTogJzUwMycsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgICAgVHlwZTogJ0FXUycsXG4gICAgICAgIFVyaTogeyAnRm46OkpvaW4nOiBbJycsIFsnYXJuOicsIHsgUmVmOiAnQVdTOjpQYXJ0aXRpb24nIH0sICc6YXBpZ2F0ZXdheTonLCB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LCAnOmZvby1zZXJ2aWNlOmFjdGlvbi9CYXJBY3Rpb24nXV19LFxuICAgICAgfSxcbiAgICB9KSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ21ldGhvZCBpcyBhbHdheXMgc2V0IGFzIHVwcGVyY2FzZScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgYXBpID0gbmV3IGFwaWd3LlJlc3RBcGkoc3RhY2ssICdhcGknKTtcblxuICAgIC8vIFdIRU5cbiAgICBhcGkucm9vdC5hZGRNZXRob2QoJ2dldCcpO1xuICAgIGFwaS5yb290LmFkZE1ldGhvZCgnUG9TdCcpO1xuICAgIGFwaS5yb290LmFkZE1ldGhvZCgnUFVUJyk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6QXBpR2F0ZXdheTo6TWV0aG9kJywgeyBIdHRwTWV0aG9kOiAnUE9TVCcgfSkpO1xuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkFwaUdhdGV3YXk6Ok1ldGhvZCcsIHsgSHR0cE1ldGhvZDogJ0dFVCcgfSkpO1xuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkFwaUdhdGV3YXk6Ok1ldGhvZCcsIHsgSHR0cE1ldGhvZDogJ1BVVCcgfSkpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdyZXF1ZXN0TW9kZWwgY2FuIGJlIHNldCcodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgYXBpID0gbmV3IGFwaWd3LlJlc3RBcGkoc3RhY2ssICd0ZXN0LWFwaScsIHsgZGVwbG95OiBmYWxzZSB9KTtcbiAgICBjb25zdCBtb2RlbCA9IGFwaS5hZGRNb2RlbCgndGVzdC1tb2RlbCcsIHtcbiAgICAgIGNvbnRlbnRUeXBlOiAnYXBwbGljYXRpb24vanNvbicsXG4gICAgICBtb2RlbE5hbWU6ICd0ZXN0LW1vZGVsJyxcbiAgICAgIHNjaGVtYToge1xuICAgICAgICB0aXRsZTogJ3Rlc3QnLFxuICAgICAgICB0eXBlOiBhcGlndy5Kc29uU2NoZW1hVHlwZS5PQkpFQ1QsXG4gICAgICAgIHByb3BlcnRpZXM6IHsgbWVzc2FnZTogeyB0eXBlOiBhcGlndy5Kc29uU2NoZW1hVHlwZS5TVFJJTkcgfSB9LFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgYXBpZ3cuTWV0aG9kKHN0YWNrLCAnbWV0aG9kLW1hbicsIHtcbiAgICAgIGh0dHBNZXRob2Q6ICdHRVQnLFxuICAgICAgcmVzb3VyY2U6IGFwaS5yb290LFxuICAgICAgb3B0aW9uczoge1xuICAgICAgICByZXF1ZXN0TW9kZWxzOiB7XG4gICAgICAgICAgJ2FwcGxpY2F0aW9uL2pzb24nOiBtb2RlbCxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6QXBpR2F0ZXdheTo6TWV0aG9kJywge1xuICAgICAgSHR0cE1ldGhvZDogJ0dFVCcsXG4gICAgICBSZXF1ZXN0TW9kZWxzOiB7XG4gICAgICAgICdhcHBsaWNhdGlvbi9qc29uJzogeyBSZWY6IHN0YWNrLmdldExvZ2ljYWxJZChtb2RlbC5ub2RlLmZpbmRDaGlsZCgnUmVzb3VyY2UnKSBhcyBjZGsuQ2ZuRWxlbWVudCkgfSxcbiAgICAgIH0sXG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ21ldGhvZFJlc3BvbnNlIGhhcyBhIG1peCBvZiByZXNwb25zZSBtb2RlcycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgYXBpID0gbmV3IGFwaWd3LlJlc3RBcGkoc3RhY2ssICd0ZXN0LWFwaScsIHsgZGVwbG95OiBmYWxzZSB9KTtcbiAgICBjb25zdCBodG1sTW9kZWwgPSBhcGkuYWRkTW9kZWwoJ215LW1vZGVsJywge1xuICAgICAgc2NoZW1hOiB7XG4gICAgICAgIHNjaGVtYTogYXBpZ3cuSnNvblNjaGVtYVZlcnNpb24uRFJBRlQ0LFxuICAgICAgICB0aXRsZTogJ3Rlc3QnLFxuICAgICAgICB0eXBlOiBhcGlndy5Kc29uU2NoZW1hVHlwZS5PQkpFQ1QsXG4gICAgICAgIHByb3BlcnRpZXM6IHsgbWVzc2FnZTogeyB0eXBlOiBhcGlndy5Kc29uU2NoZW1hVHlwZS5TVFJJTkcgfSB9LFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgYXBpZ3cuTWV0aG9kKHN0YWNrLCAnbWV0aG9kLW1hbicsIHtcbiAgICAgIGh0dHBNZXRob2Q6ICdHRVQnLFxuICAgICAgcmVzb3VyY2U6IGFwaS5yb290LFxuICAgICAgb3B0aW9uczoge1xuICAgICAgICBtZXRob2RSZXNwb25zZXM6IFt7XG4gICAgICAgICAgc3RhdHVzQ29kZTogJzIwMCcsXG4gICAgICAgIH0sIHtcbiAgICAgICAgICBzdGF0dXNDb2RlOiAnNDAwJyxcbiAgICAgICAgICByZXNwb25zZVBhcmFtZXRlcnM6IHtcbiAgICAgICAgICAgICdtZXRob2QucmVzcG9uc2UuaGVhZGVyLmtpbGxlcmJlZXMnOiBmYWxzZSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LCB7XG4gICAgICAgICAgc3RhdHVzQ29kZTogJzUwMCcsXG4gICAgICAgICAgcmVzcG9uc2VQYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgICAnbWV0aG9kLnJlc3BvbnNlLmhlYWRlci5lcnJ0aGluZyc6IHRydWUsXG4gICAgICAgICAgfSxcbiAgICAgICAgICByZXNwb25zZU1vZGVsczoge1xuICAgICAgICAgICAgJ2FwcGxpY2F0aW9uL2pzb24nOiBhcGlndy5Nb2RlbC5FTVBUWV9NT0RFTCxcbiAgICAgICAgICAgICd0ZXh0L3BsYWluJzogYXBpZ3cuTW9kZWwuRVJST1JfTU9ERUwsXG4gICAgICAgICAgICAndGV4dC9odG1sJzogaHRtbE1vZGVsLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkFwaUdhdGV3YXk6Ok1ldGhvZCcsIHtcbiAgICAgIEh0dHBNZXRob2Q6ICdHRVQnLFxuICAgICAgTWV0aG9kUmVzcG9uc2VzOiBbe1xuICAgICAgICBTdGF0dXNDb2RlOiAnMjAwJyxcbiAgICAgIH0sIHtcbiAgICAgICAgU3RhdHVzQ29kZTogJzQwMCcsXG4gICAgICAgIFJlc3BvbnNlUGFyYW1ldGVyczoge1xuICAgICAgICAgICdtZXRob2QucmVzcG9uc2UuaGVhZGVyLmtpbGxlcmJlZXMnOiBmYWxzZSxcbiAgICAgICAgfSxcbiAgICAgIH0sIHtcbiAgICAgICAgU3RhdHVzQ29kZTogJzUwMCcsXG4gICAgICAgIFJlc3BvbnNlUGFyYW1ldGVyczoge1xuICAgICAgICAgICdtZXRob2QucmVzcG9uc2UuaGVhZGVyLmVycnRoaW5nJzogdHJ1ZSxcbiAgICAgICAgfSxcbiAgICAgICAgUmVzcG9uc2VNb2RlbHM6IHtcbiAgICAgICAgICAnYXBwbGljYXRpb24vanNvbic6ICdFbXB0eScsXG4gICAgICAgICAgJ3RleHQvcGxhaW4nOiAnRXJyb3InLFxuICAgICAgICAgICd0ZXh0L2h0bWwnOiB7IFJlZjogc3RhY2suZ2V0TG9naWNhbElkKGh0bWxNb2RlbC5ub2RlLmZpbmRDaGlsZCgnUmVzb3VyY2UnKSBhcyBjZGsuQ2ZuRWxlbWVudCkgfSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgICBdLFxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdtZXRob2QgaGFzIGEgcmVxdWVzdCB2YWxpZGF0b3InKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IGFwaSA9IG5ldyBhcGlndy5SZXN0QXBpKHN0YWNrLCAndGVzdC1hcGknLCB7IGRlcGxveTogZmFsc2UgfSk7XG4gICAgY29uc3QgdmFsaWRhdG9yID0gYXBpLmFkZFJlcXVlc3RWYWxpZGF0b3IoJ3ZhbGlkYXRvcicsIHtcbiAgICAgIHZhbGlkYXRlUmVxdWVzdEJvZHk6IHRydWUsXG4gICAgICB2YWxpZGF0ZVJlcXVlc3RQYXJhbWV0ZXJzOiBmYWxzZSxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgYXBpZ3cuTWV0aG9kKHN0YWNrLCAnbWV0aG9kLW1hbicsIHtcbiAgICAgIGh0dHBNZXRob2Q6ICdHRVQnLFxuICAgICAgcmVzb3VyY2U6IGFwaS5yb290LFxuICAgICAgb3B0aW9uczoge1xuICAgICAgICByZXF1ZXN0VmFsaWRhdG9yOiB2YWxpZGF0b3IsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkFwaUdhdGV3YXk6Ok1ldGhvZCcsIHtcbiAgICAgIFJlcXVlc3RWYWxpZGF0b3JJZDogeyBSZWY6IHN0YWNrLmdldExvZ2ljYWxJZCh2YWxpZGF0b3Iubm9kZS5maW5kQ2hpbGQoJ1Jlc291cmNlJykgYXMgY2RrLkNmbkVsZW1lbnQpIH0sXG4gICAgfSkpO1xuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkFwaUdhdGV3YXk6OlJlcXVlc3RWYWxpZGF0b3InLCB7XG4gICAgICBSZXN0QXBpSWQ6IHsgUmVmOiBzdGFjay5nZXRMb2dpY2FsSWQoYXBpLm5vZGUuZmluZENoaWxkKCdSZXNvdXJjZScpIGFzIGNkay5DZm5FbGVtZW50KSB9LFxuICAgICAgVmFsaWRhdGVSZXF1ZXN0Qm9keTogdHJ1ZSxcbiAgICAgIFZhbGlkYXRlUmVxdWVzdFBhcmFtZXRlcnM6IGZhbHNlLFxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICd1c2UgZGVmYXVsdCByZXF1ZXN0UGFyYW1ldGVycycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgYXBpID0gbmV3IGFwaWd3LlJlc3RBcGkoc3RhY2ssICd0ZXN0LWFwaScsIHtcbiAgICAgIGNsb3VkV2F0Y2hSb2xlOiBmYWxzZSxcbiAgICAgIGRlcGxveTogZmFsc2UsXG4gICAgICBkZWZhdWx0TWV0aG9kT3B0aW9uczoge1xuICAgICAgICByZXF1ZXN0UGFyYW1ldGVyczogeydtZXRob2QucmVxdWVzdC5wYXRoLnByb3h5JzogdHJ1ZX0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBhcGlndy5NZXRob2Qoc3RhY2ssICdkZWZhdWx0UmVxdWVzdFBhcmFtZXRlcnMnLCB7XG4gICAgICBodHRwTWV0aG9kOiAnUE9TVCcsXG4gICAgICByZXNvdXJjZTogYXBpLnJvb3QsXG4gICAgICBvcHRpb25zOiB7XG4gICAgICAgIG9wZXJhdGlvbk5hbWU6ICdkZWZhdWx0UmVxdWVzdFBhcmFtZXRlcnMnLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpBcGlHYXRld2F5OjpNZXRob2QnLCB7XG4gICAgICBPcGVyYXRpb25OYW1lOiAnZGVmYXVsdFJlcXVlc3RQYXJhbWV0ZXJzJyxcbiAgICAgIFJlcXVlc3RQYXJhbWV0ZXJzOiB7XG4gICAgICAgICdtZXRob2QucmVxdWVzdC5wYXRoLnByb3h5JzogdHJ1ZSxcbiAgICAgIH0sXG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2F1dGhvcml6ZXIgaXMgYm91bmQgY29ycmVjdGx5Jyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICBjb25zdCByZXN0QXBpID0gbmV3IGFwaWd3LlJlc3RBcGkoc3RhY2ssICdteXJlc3RhcGknKTtcbiAgICByZXN0QXBpLnJvb3QuYWRkTWV0aG9kKCdBTlknLCB1bmRlZmluZWQsIHtcbiAgICAgIGF1dGhvcml6ZXI6IERVTU1ZX0FVVEhPUklaRVIsXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpBcGlHYXRld2F5OjpNZXRob2QnLCB7XG4gICAgICBIdHRwTWV0aG9kOiAnQU5ZJyxcbiAgICAgIEF1dGhvcml6YXRpb25UeXBlOiAnQ1VTVE9NJyxcbiAgICAgIEF1dGhvcml6ZXJJZDogRFVNTVlfQVVUSE9SSVpFUi5hdXRob3JpemVySWQsXG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2F1dGhvcml6ZXIgdmlhIGRlZmF1bHQgbWV0aG9kIG9wdGlvbnMnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIGNvbnN0IGZ1bmMgPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnbXlmdW5jdGlvbicsIHtcbiAgICAgIGhhbmRsZXI6ICdoYW5kbGVyJyxcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21JbmxpbmUoJ2ZvbycpLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEwX1gsXG4gICAgfSk7XG5cbiAgICBjb25zdCBhdXRoID0gbmV3IGFwaWd3LlRva2VuQXV0aG9yaXplcihzdGFjaywgJ215YXV0aG9yaXplcjEnLCB7XG4gICAgICBhdXRob3JpemVyTmFtZTogJ215YXV0aG9yaXplcjEnLFxuICAgICAgaGFuZGxlcjogZnVuYyxcbiAgICB9KTtcblxuICAgIGNvbnN0IHJlc3RBcGkgPSBuZXcgYXBpZ3cuUmVzdEFwaShzdGFjaywgJ215cmVzdGFwaScsIHtcbiAgICAgIGRlZmF1bHRNZXRob2RPcHRpb25zOiB7XG4gICAgICAgIGF1dGhvcml6ZXI6IGF1dGgsXG4gICAgICB9LFxuICAgIH0pO1xuICAgIHJlc3RBcGkucm9vdC5hZGRNZXRob2QoJ0FOWScpO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6QXBpR2F0ZXdheTo6QXV0aG9yaXplcicsIHtcbiAgICAgIE5hbWU6ICdteWF1dGhvcml6ZXIxJyxcbiAgICAgIFR5cGU6ICdUT0tFTicsXG4gICAgICBSZXN0QXBpSWQ6IHN0YWNrLnJlc29sdmUocmVzdEFwaS5yZXN0QXBpSWQpLFxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdmYWlscyB3aGVuIGF1dGhvcml6YXRpb24gdHlwZSBkb2VzIG5vdCBtYXRjaCB0aGUgYXV0aG9yaXplcicodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgY29uc3QgcmVzdEFwaSA9IG5ldyBhcGlndy5SZXN0QXBpKHN0YWNrLCAnbXlyZXN0YXBpJyk7XG5cbiAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICByZXN0QXBpLnJvb3QuYWRkTWV0aG9kKCdBTlknLCB1bmRlZmluZWQsIHtcbiAgICAgICAgYXV0aG9yaXphdGlvblR5cGU6IGFwaWd3LkF1dGhvcml6YXRpb25UeXBlLklBTSxcbiAgICAgICAgYXV0aG9yaXplcjogRFVNTVlfQVVUSE9SSVpFUixcbiAgICAgIH0pO1xuICAgIH0sIC9BdXRob3JpemF0aW9uIHR5cGUgaXMgc2V0IHRvIEFXU19JQU0gd2hpY2ggaXMgZGlmZmVyZW50IGZyb20gd2hhdCBpcyByZXF1aXJlZCBieSB0aGUgYXV0aG9yaXplci8pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2ZhaWxzIHdoZW4gYXV0aG9yaXphdGlvbiB0eXBlIGRvZXMgbm90IG1hdGNoIHRoZSBhdXRob3JpemVyIGluIGRlZmF1bHQgbWV0aG9kIG9wdGlvbnMnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIGNvbnN0IHJlc3RBcGkgPSBuZXcgYXBpZ3cuUmVzdEFwaShzdGFjaywgJ215cmVzdGFwaScsIHtcbiAgICAgIGRlZmF1bHRNZXRob2RPcHRpb25zOiB7XG4gICAgICAgIGF1dGhvcml6ZXI6IERVTU1ZX0FVVEhPUklaRVIsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGVzdC50aHJvd3MoKCkgPT4ge1xuICAgICAgcmVzdEFwaS5yb290LmFkZE1ldGhvZCgnQU5ZJywgdW5kZWZpbmVkLCB7XG4gICAgICAgIGF1dGhvcml6YXRpb25UeXBlOiBhcGlndy5BdXRob3JpemF0aW9uVHlwZS5OT05FLFxuICAgICAgfSk7XG4gICAgfSwgL0F1dGhvcml6YXRpb24gdHlwZSBpcyBzZXQgdG8gTk9ORSB3aGljaCBpcyBkaWZmZXJlbnQgZnJvbSB3aGF0IGlzIHJlcXVpcmVkIGJ5IHRoZSBhdXRob3JpemVyLyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnbWV0aG9kIGhhcyBBdXRoIFNjb3BlcycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgYXBpID0gbmV3IGFwaWd3LlJlc3RBcGkoc3RhY2ssICd0ZXN0LWFwaScsIHsgY2xvdWRXYXRjaFJvbGU6IGZhbHNlLCBkZXBsb3k6IGZhbHNlIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBhcGlndy5NZXRob2Qoc3RhY2ssICdteS1tZXRob2QnLCB7XG4gICAgICBodHRwTWV0aG9kOiAnUE9TVCcsXG4gICAgICByZXNvdXJjZTogYXBpLnJvb3QsXG4gICAgICBvcHRpb25zOiB7XG4gICAgICAgIGFwaUtleVJlcXVpcmVkOiB0cnVlLFxuICAgICAgICBhdXRob3JpemF0aW9uU2NvcGVzOiBbJ0F1dGhTY29wZTEnLCAnQXV0aFNjb3BlMiddLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpBcGlHYXRld2F5OjpNZXRob2QnLCB7XG4gICAgICBBcGlLZXlSZXF1aXJlZDogdHJ1ZSxcbiAgICAgIEF1dGhvcml6YXRpb25TY29wZXM6IFsnQXV0aFNjb3BlMScsICdBdXRoU2NvcGUyJ10sXG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3VzZSBkZWZhdWx0IEF1dGggU2NvcGVzJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCBhcGkgPSBuZXcgYXBpZ3cuUmVzdEFwaShzdGFjaywgJ3Rlc3QtYXBpJywge1xuICAgICAgY2xvdWRXYXRjaFJvbGU6IGZhbHNlLFxuICAgICAgZGVwbG95OiBmYWxzZSxcbiAgICAgIGRlZmF1bHRNZXRob2RPcHRpb25zOiB7XG4gICAgICAgIGF1dGhvcml6YXRpb25TY29wZXM6IFsnRGVmYXVsdEF1dGgnXSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IGFwaWd3Lk1ldGhvZChzdGFjaywgJ2RlZmF1bHRBdXRoU2NvcGVzJywge1xuICAgICAgaHR0cE1ldGhvZDogJ1BPU1QnLFxuICAgICAgcmVzb3VyY2U6IGFwaS5yb290LFxuICAgICAgb3B0aW9uczoge1xuICAgICAgICBvcGVyYXRpb25OYW1lOiAnZGVmYXVsdEF1dGhTY29wZXMnLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpBcGlHYXRld2F5OjpNZXRob2QnLCB7XG4gICAgICBPcGVyYXRpb25OYW1lOiAnZGVmYXVsdEF1dGhTY29wZXMnLFxuICAgICAgQXV0aG9yaXphdGlvblNjb3BlczogWydEZWZhdWx0QXV0aCddLFxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdNZXRob2Qgb3B0aW9ucyBBdXRoIFNjb3BlcyBpcyBwaWNrZWQgdXAnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IGFwaSA9IG5ldyBhcGlndy5SZXN0QXBpKHN0YWNrLCAndGVzdC1hcGknLCB7XG4gICAgICBjbG91ZFdhdGNoUm9sZTogZmFsc2UsXG4gICAgICBkZXBsb3k6IGZhbHNlLFxuICAgICAgZGVmYXVsdE1ldGhvZE9wdGlvbnM6IHtcbiAgICAgICAgYXV0aG9yaXphdGlvblNjb3BlczogWydEZWZhdWx0QXV0aCddLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgYXBpZ3cuTWV0aG9kKHN0YWNrLCAnTWV0aG9kQXV0aFNjb3BlVXNlZCcsIHtcbiAgICAgIGh0dHBNZXRob2Q6ICdQT1NUJyxcbiAgICAgIHJlc291cmNlOiBhcGkucm9vdCxcbiAgICAgIG9wdGlvbnM6IHtcbiAgICAgICAgYXBpS2V5UmVxdWlyZWQ6IHRydWUsXG4gICAgICAgIGF1dGhvcml6YXRpb25TY29wZXM6IFsnTWV0aG9kQXV0aFNjb3BlJ10sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkFwaUdhdGV3YXk6Ok1ldGhvZCcsIHtcbiAgICAgIEFwaUtleVJlcXVpcmVkOiB0cnVlLFxuICAgICAgQXV0aG9yaXphdGlvblNjb3BlczogWydNZXRob2RBdXRoU2NvcGUnXSxcbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnQXV0aCBTY29wZXMgYWJzZW50Jyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCBhcGkgPSBuZXcgYXBpZ3cuUmVzdEFwaShzdGFjaywgJ3Rlc3QtYXBpJywge1xuICAgICAgY2xvdWRXYXRjaFJvbGU6IGZhbHNlLFxuICAgICAgZGVwbG95OiBmYWxzZSxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgYXBpZ3cuTWV0aG9kKHN0YWNrLCAnYXV0aFNjb3Blc0Fic2VudCcsIHtcbiAgICAgIGh0dHBNZXRob2Q6ICdQT1NUJyxcbiAgICAgIHJlc291cmNlOiBhcGkucm9vdCxcbiAgICAgIG9wdGlvbnM6IHtcbiAgICAgICAgb3BlcmF0aW9uTmFtZTogJ2F1dGhTY29wZXNBYnNlbnQnLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpBcGlHYXRld2F5OjpNZXRob2QnLCB7XG4gICAgICBPcGVyYXRpb25OYW1lOiAnYXV0aFNjb3Blc0Fic2VudCcsXG4gICAgICBBdXRob3JpemF0aW9uU2NvcGVzOiBBQlNFTlQsXG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ21ldGhvZCBoYXMgYSByZXF1ZXN0IHZhbGlkYXRvciB3aXRoIHByb3ZpZGVkIHByb3BlcnRpZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IGFwaSA9IG5ldyBhcGlndy5SZXN0QXBpKHN0YWNrLCAndGVzdC1hcGknLCB7IGRlcGxveTogZmFsc2UgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IGFwaWd3Lk1ldGhvZChzdGFjaywgJ21ldGhvZC1tYW4nLCB7XG4gICAgICBodHRwTWV0aG9kOiAnR0VUJyxcbiAgICAgIHJlc291cmNlOiBhcGkucm9vdCxcbiAgICAgIG9wdGlvbnM6IHtcbiAgICAgICAgcmVxdWVzdFZhbGlkYXRvck9wdGlvbnM6IHtcbiAgICAgICAgICByZXF1ZXN0VmFsaWRhdG9yTmFtZTogJ3Rlc3QtdmFsaWRhdG9yJyxcbiAgICAgICAgICB2YWxpZGF0ZVJlcXVlc3RCb2R5OiB0cnVlLFxuICAgICAgICAgIHZhbGlkYXRlUmVxdWVzdFBhcmFtZXRlcnM6IGZhbHNlLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpBcGlHYXRld2F5OjpSZXF1ZXN0VmFsaWRhdG9yJywge1xuICAgICAgUmVzdEFwaUlkOiBzdGFjay5yZXNvbHZlKGFwaS5yZXN0QXBpSWQpLFxuICAgICAgVmFsaWRhdGVSZXF1ZXN0Qm9keTogdHJ1ZSxcbiAgICAgIFZhbGlkYXRlUmVxdWVzdFBhcmFtZXRlcnM6IGZhbHNlLFxuICAgICAgTmFtZTogJ3Rlc3QtdmFsaWRhdG9yJyxcbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnbWV0aG9kIGRvZXMgbm90IGhhdmUgYSByZXF1ZXN0IHZhbGlkYXRvcicodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgYXBpID0gbmV3IGFwaWd3LlJlc3RBcGkoc3RhY2ssICd0ZXN0LWFwaScsIHsgZGVwbG95OiBmYWxzZSB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgYXBpZ3cuTWV0aG9kKHN0YWNrLCAnbWV0aG9kLW1hbicsIHtcbiAgICAgIGh0dHBNZXRob2Q6ICdHRVQnLFxuICAgICAgcmVzb3VyY2U6IGFwaS5yb290LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkFwaUdhdGV3YXk6Ok1ldGhvZCcsIHtcbiAgICAgIFJlcXVlc3RWYWxpZGF0b3JJZDogQUJTRU5ULFxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdtZXRob2QgZG9lcyBub3Qgc3VwcG9ydCBib3RoIHJlcXVlc3QgdmFsaWRhdG9yIGFuZCByZXF1ZXN0IHZhbGlkYXRvciBvcHRpb25zJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCBhcGkgPSBuZXcgYXBpZ3cuUmVzdEFwaShzdGFjaywgJ3Rlc3QtYXBpJywgeyBkZXBsb3k6IGZhbHNlIH0pO1xuICAgIGNvbnN0IHZhbGlkYXRvciA9IGFwaS5hZGRSZXF1ZXN0VmFsaWRhdG9yKCd0ZXN0LXZhbGlkYXRvcjEnLCB7XG4gICAgICB2YWxpZGF0ZVJlcXVlc3RCb2R5OiB0cnVlLFxuICAgICAgdmFsaWRhdGVSZXF1ZXN0UGFyYW1ldGVyczogZmFsc2UsXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgbWV0aG9kUHJvcHMgPSB7XG4gICAgICBodHRwTWV0aG9kOiAnR0VUJyxcbiAgICAgIHJlc291cmNlOiBhcGkucm9vdCxcbiAgICAgIG9wdGlvbnM6IHtcbiAgICAgICAgcmVxdWVzdFZhbGlkYXRvck9wdGlvbnM6IHtcbiAgICAgICAgICByZXF1ZXN0VmFsaWRhdG9yTmFtZTogJ3Rlc3QtdmFsaWRhdG9yMicsXG4gICAgICAgICAgdmFsaWRhdGVSZXF1ZXN0Qm9keTogdHJ1ZSxcbiAgICAgICAgICB2YWxpZGF0ZVJlcXVlc3RQYXJhbWV0ZXJzOiBmYWxzZSxcbiAgICAgICAgfSxcbiAgICAgICAgcmVxdWVzdFZhbGlkYXRvcjogdmFsaWRhdG9yLFxuICAgICAgfSxcbiAgICB9O1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QudGhyb3dzKCgpID0+IG5ldyBhcGlndy5NZXRob2Qoc3RhY2ssICdtZXRob2QnLCBtZXRob2RQcm9wcyksXG4gICAgICAvT25seSBvbmUgb2YgJ3JlcXVlc3RWYWxpZGF0b3InIG9yICdyZXF1ZXN0VmFsaWRhdG9yT3B0aW9ucycgbXVzdCBiZSBzcGVjaWZpZWQuLyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnXCJyZXN0QXBpXCIgYW5kIFwiYXBpXCIgcHJvcGVydGllcyByZXR1cm4gdGhlIFJlc3RBcGkgY29ycmVjdGx5Jyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCBhcGkgPSBuZXcgYXBpZ3cuUmVzdEFwaShzdGFjaywgJ3Rlc3QtYXBpJyk7XG4gICAgY29uc3QgbWV0aG9kID0gYXBpLnJvb3QuYWRkUmVzb3VyY2UoJ3BldHMnKS5hZGRNZXRob2QoJ0dFVCcpO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3Qub2sobWV0aG9kLnJlc3RBcGkpO1xuICAgIHRlc3Qub2sobWV0aG9kLmFwaSk7XG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZShtZXRob2QuYXBpLnJlc3RBcGlJZCksIHN0YWNrLnJlc29sdmUobWV0aG9kLnJlc3RBcGkucmVzdEFwaUlkKSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnXCJyZXN0QXBpXCIgdGhyb3dzIGFuIGVycm9yIG9uIGltcG9ydGVkIHdoaWxlIFwiYXBpXCIgcmV0dXJucyBjb3JyZWN0bHknKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IGFwaSA9IGFwaWd3LlJlc3RBcGkuZnJvbVJlc3RBcGlBdHRyaWJ1dGVzKHN0YWNrLCAndGVzdC1hcGknLCB7XG4gICAgICByZXN0QXBpSWQ6ICd0ZXN0LXJlc3QtYXBpLWlkJyxcbiAgICAgIHJvb3RSZXNvdXJjZUlkOiAndGVzdC1yb290LXJlc291cmNlLWlkJyxcbiAgICB9KTtcbiAgICBjb25zdCBtZXRob2QgPSBhcGkucm9vdC5hZGRSZXNvdXJjZSgncGV0cycpLmFkZE1ldGhvZCgnR0VUJyk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC50aHJvd3MoKCkgPT4gbWV0aG9kLnJlc3RBcGksIC9ub3QgYXZhaWxhYmxlIG9uIFJlc291cmNlIG5vdCBjb25uZWN0ZWQgdG8gYW4gaW5zdGFuY2Ugb2YgUmVzdEFwaS8pO1xuICAgIHRlc3Qub2sobWV0aG9kLmFwaSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbn07XG4iXX0=