"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnParameter = void 0;
const cfn_element_1 = require("./cfn-element");
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
/**
 * A CloudFormation parameter.
 *
 * Use the optional Parameters section to customize your templates.
 * Parameters enable you to input custom values to your template each time you create or
 * update a stack.
 */
class CfnParameter extends cfn_element_1.CfnElement {
    /**
     * Creates a parameter construct.
     * Note that the name (logical ID) of the parameter will derive from it's `coname` and location
     * within the stack. Therefore, it is recommended that parameters are defined at the stack level.
     *
     * @param scope The parent construct.
     * @param props The parameter properties.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.props = props;
        this.type = props.type || 'String';
    }
    /**
     * Indicates if this parameter is configured with "NoEcho" enabled.
     */
    get noEcho() {
        return !!this.props.noEcho;
    }
    /**
     * The parameter value as a Token
     */
    get value() {
        return cfn_reference_1.CfnReference.for(this, 'Ref');
    }
    /**
     * The parameter value, if it represents a string.
     */
    get valueAsString() {
        if (!isStringType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a string type`);
        }
        return token_1.Token.asString(this.value);
    }
    /**
     * The parameter value, if it represents a string list.
     */
    get valueAsList() {
        if (!isListType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a string list type`);
        }
        return token_1.Token.asList(this.value);
    }
    /**
     * The parameter value, if it represents a number.
     */
    get valueAsNumber() {
        if (!isNumberType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a number type`);
        }
        return token_1.Token.asNumber(this.value);
    }
    /**
     * @internal
     */
    _toCloudFormation() {
        return {
            Parameters: {
                [this.logicalId]: {
                    Type: this.type,
                    Default: this.props.default,
                    AllowedPattern: this.props.allowedPattern,
                    AllowedValues: this.props.allowedValues,
                    ConstraintDescription: this.props.constraintDescription,
                    Description: this.props.description,
                    MaxLength: this.props.maxLength,
                    MaxValue: this.props.maxValue,
                    MinLength: this.props.minLength,
                    MinValue: this.props.minValue,
                    NoEcho: this.props.noEcho,
                },
            },
        };
    }
    resolve(_context) {
        return this.value;
    }
}
exports.CfnParameter = CfnParameter;
/**
 * Whether the given parameter type looks like a list type
 */
function isListType(type) {
    return type.indexOf('List<') >= 0 || type.indexOf('CommaDelimitedList') >= 0;
}
/**
 * Whether the given parameter type looks like a number type
 */
function isNumberType(type) {
    return type === 'Number';
}
/**
 * Whether the given parameter type looks like a string type
 */
function isStringType(type) {
    return !isListType(type) && !isNumberType(type);
}
//# sourceMappingURL=data:application/json;base64,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