"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretValue = void 0;
const cfn_dynamic_reference_1 = require("./cfn-dynamic-reference");
const intrinsic_1 = require("./private/intrinsic");
/**
 * Work with secret values in the CDK
 *
 * Secret values in the CDK (such as those retrieved from SecretsManager) are
 * represented as regular strings, just like other values that are only
 * available at deployment time.
 *
 * To help you avoid accidental mistakes which would lead to you putting your
 * secret values directly into a CloudFormation template, constructs that take
 * secret values will not allow you to pass in a literal secret value. They do
 * so by calling `Secret.assertSafeSecret()`.
 *
 * You can escape the check by calling `Secret.plainText()`, but doing
 * so is highly discouraged.
 */
class SecretValue extends intrinsic_1.Intrinsic {
    /**
     * Construct a literal secret value for use with secret-aware constructs
     *
     * *Do not use this method for any secrets that you care about.*
     *
     * The only reasonable use case for using this method is when you are testing.
     */
    static plainText(secret) {
        return new SecretValue(secret);
    }
    /**
     * Creates a `SecretValue` with a value which is dynamically loaded from AWS Secrets Manager.
     * @param secretId The ID or ARN of the secret
     * @param options Options
     */
    static secretsManager(secretId, options = {}) {
        if (!secretId) {
            throw new Error('secretId cannot be empty');
        }
        if (!secretId.startsWith('arn:') && secretId.includes(':')) {
            throw new Error(`secret id "${secretId}" is not an ARN but contains ":"`);
        }
        const parts = [
            secretId,
            'SecretString',
            options.jsonField || '',
            options.versionStage || '',
            options.versionId || '',
        ];
        const dyref = new cfn_dynamic_reference_1.CfnDynamicReference(cfn_dynamic_reference_1.CfnDynamicReferenceService.SECRETS_MANAGER, parts.join(':'));
        return this.cfnDynamicReference(dyref);
    }
    /**
     * Use a secret value stored from a Systems Manager (SSM) parameter.
     *
     * @param parameterName The name of the parameter in the Systems Manager
     * Parameter Store. The parameter name is case-sensitive.
     *
     * @param version An integer that specifies the version of the parameter to
     * use. You must specify the exact version. You cannot currently specify that
     * AWS CloudFormation use the latest version of a parameter.
     */
    static ssmSecure(parameterName, version) {
        const parts = [parameterName, version];
        return this.cfnDynamicReference(new cfn_dynamic_reference_1.CfnDynamicReference(cfn_dynamic_reference_1.CfnDynamicReferenceService.SSM_SECURE, parts.join(':')));
    }
    /**
     * Obtain the secret value through a CloudFormation dynamic reference.
     *
     * If possible, use `SecretValue.ssmSecure` or `SecretValue.secretsManager` directly.
     *
     * @param ref The dynamic reference to use.
     */
    static cfnDynamicReference(ref) {
        return new SecretValue(ref);
    }
    /**
     * Obtain the secret value through a CloudFormation parameter.
     *
     * Generally, this is not a recommended approach. AWS Secrets Manager is the
     * recommended way to reference secrets.
     *
     * @param param The CloudFormation parameter to use.
     */
    static cfnParameter(param) {
        if (!param.noEcho) {
            throw new Error('CloudFormation parameter must be configured with "NoEcho"');
        }
        return new SecretValue(param.value);
    }
}
exports.SecretValue = SecretValue;
//# sourceMappingURL=data:application/json;base64,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