"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.withResolved = exports.isResolvableObject = exports.Tokenization = exports.Token = void 0;
const lazy_1 = require("./lazy");
const encoding_1 = require("./private/encoding");
const intrinsic_1 = require("./private/intrinsic");
const resolve_1 = require("./private/resolve");
const token_map_1 = require("./private/token-map");
/**
 * Represents a special or lazily-evaluated value.
 *
 * Can be used to delay evaluation of a certain value in case, for example,
 * that it requires some context or late-bound data. Can also be used to
 * mark values that need special processing at document rendering time.
 *
 * Tokens can be embedded into strings while retaining their original
 * semantics.
 */
class Token {
    /**
     * Returns true if obj represents an unresolved value
     *
     * One of these must be true:
     *
     * - `obj` is an IResolvable
     * - `obj` is a string containing at least one encoded `IResolvable`
     * - `obj` is either an encoded number or list
     *
     * This does NOT recurse into lists or objects to see if they
     * containing resolvables.
     *
     * @param obj The object to test.
     */
    static isUnresolved(obj) {
        return encoding_1.unresolved(obj);
    }
    /**
     * Return a reversible string representation of this token
     *
     * If the Token is initialized with a literal, the stringified value of the
     * literal is returned. Otherwise, a special quoted string representation
     * of the Token is returned that can be embedded into other strings.
     *
     * Strings with quoted Tokens in them can be restored back into
     * complex values with the Tokens restored by calling `resolve()`
     * on the string.
     */
    static asString(value, options = {}) {
        if (typeof value === 'string') {
            return value;
        }
        return token_map_1.TokenMap.instance().registerString(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a reversible number representation of this token
     */
    static asNumber(value) {
        if (typeof value === 'number') {
            return value;
        }
        return token_map_1.TokenMap.instance().registerNumber(Token.asAny(value));
    }
    /**
     * Return a reversible list representation of this token
     */
    static asList(value, options = {}) {
        if (Array.isArray(value) && value.every(x => typeof x === 'string')) {
            return value;
        }
        return token_map_1.TokenMap.instance().registerList(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a resolvable representation of the given value
     */
    static asAny(value) {
        return isResolvableObject(value) ? value : new intrinsic_1.Intrinsic(value);
    }
    constructor() {
    }
}
exports.Token = Token;
/**
 * Less oft-needed functions to manipulate Tokens
 */
class Tokenization {
    /**
     * Un-encode a string potentially containing encoded tokens
     */
    static reverseString(s) {
        return token_map_1.TokenMap.instance().splitString(s);
    }
    /**
     * Un-encode a Tokenized value from a number
     */
    static reverseNumber(n) {
        return token_map_1.TokenMap.instance().lookupNumberToken(n);
    }
    /**
     * Un-encode a Tokenized value from a list
     */
    static reverseList(l) {
        return token_map_1.TokenMap.instance().lookupList(l);
    }
    /**
     * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
     * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
     *
     * @param obj The object to resolve.
     * @param options Prefix key path components for diagnostics.
     */
    static resolve(obj, options) {
        return resolve_1.resolve(obj, {
            scope: options.scope,
            resolver: options.resolver,
            preparing: (options.preparing !== undefined ? options.preparing : false),
        });
    }
    /**
     * Return whether the given object is an IResolvable object
     *
     * This is different from Token.isUnresolved() which will also check for
     * encoded Tokens, whereas this method will only do a type check on the given
     * object.
     */
    static isResolvable(obj) {
        return isResolvableObject(obj);
    }
    /**
     * Stringify a number directly or lazily if it's a Token. If it is an object (i.e., { Ref: 'SomeLogicalId' }), return it as-is.
     */
    static stringifyNumber(x) {
        // only convert numbers to strings so that Refs, conditions, and other things don't end up synthesizing as [object object]
        if (Token.isUnresolved(x)) {
            return lazy_1.Lazy.stringValue({ produce: context => {
                    const resolved = context.resolve(x);
                    return typeof resolved !== 'number' ? resolved : `${resolved}`;
                } });
        }
        else {
            return typeof x !== 'number' ? x : `${x}`;
        }
    }
    constructor() {
    }
}
exports.Tokenization = Tokenization;
function isResolvableObject(x) {
    return typeof (x) === 'object' && x !== null && typeof x.resolve === 'function';
}
exports.isResolvableObject = isResolvableObject;
function withResolved(...args) {
    if (args.length < 2) {
        return;
    }
    const argArray = args.slice(0, args.length - 1);
    if (argArray.some(Token.isUnresolved)) {
        return;
    }
    args[args.length - 1].apply(arguments, argArray);
}
exports.withResolved = withResolved;
//# sourceMappingURL=data:application/json;base64,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