"use strict";
const fs = require("fs");
const path = require("path");
const lib_1 = require("../../lib");
const util_1 = require("../util");
const TEST_HANDLER = `${__dirname}/mock-provider`;
module.exports = {
    'minimal configuration'(test) {
        var _a;
        // GIVEN
        const stack = new lib_1.Stack();
        // WHEN
        lib_1.CustomResourceProvider.getOrCreate(stack, 'Custom:MyResourceType', {
            codeDirectory: TEST_HANDLER,
            runtime: lib_1.CustomResourceProviderRuntime.NODEJS_12,
        });
        // THEN
        test.ok(fs.existsSync(path.join(TEST_HANDLER, '__entrypoint__.js')), 'expecting entrypoint to be copied to the handler directory');
        const cfn = util_1.toCloudFormation(stack);
        // The asset hash constantly changes, so in order to not have to chase it, just look
        // it up from the output.
        const staging = (_a = stack.node.tryFindChild('Custom:MyResourceTypeCustomResourceProvider')) === null || _a === void 0 ? void 0 : _a.node.tryFindChild('Staging');
        const assetHash = staging.sourceHash;
        const paramNames = Object.keys(cfn.Parameters);
        const bucketParam = paramNames[0];
        const keyParam = paramNames[1];
        const hashParam = paramNames[2];
        test.deepEqual(cfn, {
            Resources: {
                CustomMyResourceTypeCustomResourceProviderRoleBD5E655F: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Version: '2012-10-17',
                            Statement: [
                                {
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: {
                                        Service: 'lambda.amazonaws.com',
                                    },
                                },
                            ],
                        },
                        ManagedPolicyArns: [
                            {
                                'Fn::Sub': 'arn:${AWS::Partition}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole',
                            },
                        ],
                    },
                },
                CustomMyResourceTypeCustomResourceProviderHandler29FBDD2A: {
                    Type: 'AWS::Lambda::Function',
                    Properties: {
                        Code: {
                            S3Bucket: { Ref: bucketParam },
                            S3Key: {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::Select': [
                                                0,
                                                {
                                                    'Fn::Split': [
                                                        '||',
                                                        { Ref: keyParam },
                                                    ],
                                                },
                                            ],
                                        },
                                        {
                                            'Fn::Select': [
                                                1,
                                                {
                                                    'Fn::Split': [
                                                        '||',
                                                        { Ref: keyParam },
                                                    ],
                                                },
                                            ],
                                        },
                                    ],
                                ],
                            },
                        },
                        Timeout: 900,
                        MemorySize: 128,
                        Handler: '__entrypoint__.handler',
                        Role: {
                            'Fn::GetAtt': [
                                'CustomMyResourceTypeCustomResourceProviderRoleBD5E655F',
                                'Arn',
                            ],
                        },
                        Runtime: 'nodejs12.x',
                    },
                    DependsOn: [
                        'CustomMyResourceTypeCustomResourceProviderRoleBD5E655F',
                    ],
                },
            },
            Parameters: {
                [bucketParam]: {
                    Type: 'String',
                    Description: `S3 bucket for asset "${assetHash}"`,
                },
                [keyParam]: {
                    Type: 'String',
                    Description: `S3 key for asset version "${assetHash}"`,
                },
                [hashParam]: {
                    Type: 'String',
                    Description: `Artifact hash for asset "${assetHash}"`,
                },
            },
        });
        test.done();
    },
    'policyStatements can be used to add statements to the inline policy'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        // WHEN
        lib_1.CustomResourceProvider.getOrCreate(stack, 'Custom:MyResourceType', {
            codeDirectory: TEST_HANDLER,
            runtime: lib_1.CustomResourceProviderRuntime.NODEJS_12,
            policyStatements: [
                { statement1: 123 },
                { statement2: { foo: 111 } },
            ],
        });
        // THEN
        const template = util_1.toCloudFormation(stack);
        const role = template.Resources.CustomMyResourceTypeCustomResourceProviderRoleBD5E655F;
        test.deepEqual(role.Properties.Policies, [{
                PolicyName: 'Inline',
                PolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [{ statement1: 123 }, { statement2: { foo: 111 } }],
                },
            }]);
        test.done();
    },
    'memorySize and timeout'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        // WHEN
        lib_1.CustomResourceProvider.getOrCreate(stack, 'Custom:MyResourceType', {
            codeDirectory: TEST_HANDLER,
            runtime: lib_1.CustomResourceProviderRuntime.NODEJS_12,
            memorySize: lib_1.Size.gibibytes(2),
            timeout: lib_1.Duration.minutes(5),
        });
        // THEN
        const template = util_1.toCloudFormation(stack);
        const lambda = template.Resources.CustomMyResourceTypeCustomResourceProviderHandler29FBDD2A;
        test.deepEqual(lambda.Properties.MemorySize, 2048);
        test.deepEqual(lambda.Properties.Timeout, 300);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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