"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
const rule_1 = require("../lib/rule");
class SomeTarget {
    constructor(id, resource) {
        this.id = id;
        this.resource = resource;
    }
    bind() {
        return {
            id: this.id || '',
            arn: 'ARN1', kinesisParameters: { partitionKeyPath: 'partitionKeyPath' },
            targetResource: this.resource,
        };
    }
}
module.exports = {
    'default rule'(test) {
        const stack = new cdk.Stack();
        new rule_1.Rule(stack, 'MyRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(10)),
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyRuleA44AB831': {
                    'Type': 'AWS::Events::Rule',
                    'Properties': {
                        'ScheduleExpression': 'rate(10 minutes)',
                        'State': 'ENABLED'
                    }
                }
            }
        });
        test.done();
    },
    'can get rule name'(test) {
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'MyRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(10)),
        });
        new cdk.CfnResource(stack, 'Res', {
            type: 'Test::Resource',
            properties: {
                RuleName: rule.ruleName
            }
        });
        assert_1.expect(stack).to(assert_1.haveResource('Test::Resource', {
            RuleName: { Ref: 'MyRuleA44AB831' }
        }));
        test.done();
    },
    'rule with physical name'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new rule_1.Rule(stack, 'MyRule', {
            ruleName: 'PhysicalName',
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(10)),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            Name: 'PhysicalName'
        }));
        test.done();
    },
    'eventPattern is rendered properly'(test) {
        const stack = new cdk.Stack();
        new rule_1.Rule(stack, 'MyRule', {
            eventPattern: {
                account: ['account1', 'account2'],
                detail: {
                    foo: [1, 2],
                },
                detailType: ['detailType1'],
                id: ['id1', 'id2'],
                region: ['region1', 'region2', 'region3'],
                resources: ['r1'],
                source: ['src1', 'src2'],
                time: ['t1'],
                version: ['0']
            }
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyRuleA44AB831': {
                    'Type': 'AWS::Events::Rule',
                    'Properties': {
                        'EventPattern': {
                            account: ['account1', 'account2'],
                            detail: { foo: [1, 2] },
                            'detail-type': ['detailType1'],
                            id: ['id1', 'id2'],
                            region: ['region1', 'region2', 'region3'],
                            resources: ['r1'],
                            source: ['src1', 'src2'],
                            time: ['t1'],
                            version: ['0']
                        },
                        'State': 'ENABLED'
                    }
                }
            }
        });
        test.done();
    },
    'fails synthesis if neither eventPattern nor scheudleExpression are specified'(test) {
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'MyStack');
        new rule_1.Rule(stack, 'Rule');
        test.throws(() => app.synth(), /Either 'eventPattern' or 'schedule' must be defined/);
        test.done();
    },
    'addEventPattern can be used to add filters'(test) {
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'MyRule');
        rule.addEventPattern({
            account: ['12345'],
            detail: {
                foo: ['hello']
            }
        });
        rule.addEventPattern({
            source: ['aws.source'],
            detail: {
                foo: ['bar'],
                goo: {
                    hello: ['world']
                }
            }
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyRuleA44AB831': {
                    'Type': 'AWS::Events::Rule',
                    'Properties': {
                        'EventPattern': {
                            'account': [
                                '12345'
                            ],
                            'detail': {
                                'foo': [
                                    'hello',
                                    'bar'
                                ],
                                'goo': {
                                    'hello': [
                                        'world'
                                    ]
                                }
                            },
                            'source': [
                                'aws.source'
                            ]
                        },
                        'State': 'ENABLED'
                    }
                }
            }
        });
        test.done();
    },
    'targets can be added via props or addTarget with input transformer'(test) {
        const stack = new cdk.Stack();
        const t1 = {
            bind: () => ({
                id: '',
                arn: 'ARN1',
                kinesisParameters: { partitionKeyPath: 'partitionKeyPath' }
            })
        };
        const t2 = {
            bind: () => ({
                id: '',
                arn: 'ARN2',
                input: lib_1.RuleTargetInput.fromText(`This is ${lib_1.EventField.fromPath('$.detail.bla')}`),
            })
        };
        const rule = new rule_1.Rule(stack, 'EventRule', {
            targets: [t1],
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(5)),
        });
        rule.addTarget(t2);
        assert_1.expect(stack).toMatch({
            'Resources': {
                'EventRule5A491D2C': {
                    'Type': 'AWS::Events::Rule',
                    'Properties': {
                        'ScheduleExpression': 'rate(5 minutes)',
                        'State': 'ENABLED',
                        'Targets': [
                            {
                                'Arn': 'ARN1',
                                'Id': 'Target0',
                                'KinesisParameters': {
                                    'PartitionKeyPath': 'partitionKeyPath'
                                }
                            },
                            {
                                'Arn': 'ARN2',
                                'Id': 'Target1',
                                'InputTransformer': {
                                    'InputPathsMap': {
                                        'detail-bla': '$.detail.bla'
                                    },
                                    'InputTemplate': '"This is <detail-bla>"'
                                },
                            }
                        ]
                    }
                }
            }
        });
        test.done();
    },
    'input template can contain tokens'(test) {
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'EventRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
        });
        // a plain string should just be stringified (i.e. double quotes added and escaped)
        rule.addTarget({
            bind: () => ({
                id: '', arn: 'ARN2', input: lib_1.RuleTargetInput.fromText('Hello, "world"')
            })
        });
        // tokens are used here (FnConcat), but this is a text template so we
        // expect it to be wrapped in double quotes automatically for us.
        rule.addTarget({
            bind: () => ({
                id: '',
                arn: 'ARN1', kinesisParameters: { partitionKeyPath: 'partitionKeyPath' },
                input: lib_1.RuleTargetInput.fromText(cdk.Fn.join('', ['a', 'b']).toString()),
            })
        });
        // jsonTemplate can be used to format JSON documents with replacements
        rule.addTarget({
            bind: () => ({
                id: '',
                arn: 'ARN3',
                input: lib_1.RuleTargetInput.fromObject({ foo: lib_1.EventField.fromPath('$.detail.bar') }),
            })
        });
        // tokens can also used for JSON templates.
        rule.addTarget({
            bind: () => ({
                id: '',
                arn: 'ARN4',
                input: lib_1.RuleTargetInput.fromText(cdk.Fn.join(' ', ['hello', '"world"']).toString()),
            })
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'EventRule5A491D2C': {
                    'Type': 'AWS::Events::Rule',
                    'Properties': {
                        'State': 'ENABLED',
                        'ScheduleExpression': 'rate(1 minute)',
                        'Targets': [
                            {
                                'Arn': 'ARN2',
                                'Id': 'Target0',
                                'Input': '"Hello, \\"world\\""',
                            },
                            {
                                'Arn': 'ARN1',
                                'Id': 'Target1',
                                'Input': '"ab"',
                                'KinesisParameters': {
                                    'PartitionKeyPath': 'partitionKeyPath'
                                }
                            },
                            {
                                'Arn': 'ARN3',
                                'Id': 'Target2',
                                'InputTransformer': {
                                    'InputPathsMap': {
                                        'detail-bar': '$.detail.bar'
                                    },
                                    'InputTemplate': '{"foo":<detail-bar>}'
                                }
                            },
                            {
                                'Arn': 'ARN4',
                                'Id': 'Target3',
                                'Input': '"hello \\"world\\""'
                            }
                        ]
                    }
                }
            }
        });
        test.done();
    },
    'target can declare role which will be used'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'EventRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
        });
        const role = new iam.Role(stack, 'SomeRole', {
            assumedBy: new iam.ServicePrincipal('nobody')
        });
        // a plain string should just be stringified (i.e. double quotes added and escaped)
        rule.addTarget({
            bind: () => ({
                id: '',
                arn: 'ARN2',
                role,
            })
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
            'Targets': [
                {
                    'Arn': 'ARN2',
                    'Id': 'Target0',
                    'RoleArn': { 'Fn::GetAtt': ['SomeRole6DDC54DD', 'Arn'] }
                }
            ]
        }));
        test.done();
    },
    'asEventRuleTarget can use the ruleArn and a uniqueId of the rule'(test) {
        const stack = new cdk.Stack();
        let receivedRuleArn = 'FAIL';
        let receivedRuleId = 'FAIL';
        const t1 = {
            bind: (eventRule) => {
                receivedRuleArn = eventRule.ruleArn;
                receivedRuleId = eventRule.node.uniqueId;
                return {
                    id: '',
                    arn: 'ARN1',
                    kinesisParameters: { partitionKeyPath: 'partitionKeyPath' }
                };
            }
        };
        const rule = new rule_1.Rule(stack, 'EventRule');
        rule.addTarget(t1);
        test.deepEqual(stack.resolve(receivedRuleArn), stack.resolve(rule.ruleArn));
        test.deepEqual(receivedRuleId, rule.node.uniqueId);
        test.done();
    },
    'fromEventRuleArn'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const importedRule = rule_1.Rule.fromEventRuleArn(stack, 'ImportedRule', 'arn:aws:events:us-east-2:123456789012:rule/example');
        // THEN
        test.deepEqual(importedRule.ruleArn, 'arn:aws:events:us-east-2:123456789012:rule/example');
        test.deepEqual(importedRule.ruleName, 'example');
        test.done();
    },
    'rule can be disabled'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new rule_1.Rule(stack, 'Rule', {
            schedule: lib_1.Schedule.expression('foom'),
            enabled: false
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            'State': 'DISABLED'
        }));
        test.done();
    },
    'can add multiple targets with the same id'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'Rule', {
            schedule: lib_1.Schedule.expression('foom'),
            enabled: false
        });
        rule.addTarget(new SomeTarget());
        rule.addTarget(new SomeTarget());
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            Targets: [
                {
                    'Arn': 'ARN1',
                    'Id': 'Target0',
                    'KinesisParameters': {
                        'PartitionKeyPath': 'partitionKeyPath'
                    }
                },
                {
                    'Arn': 'ARN1',
                    'Id': 'Target1',
                    'KinesisParameters': {
                        'PartitionKeyPath': 'partitionKeyPath'
                    }
                }
            ]
        }));
        test.done();
    },
    'rule and target must be in the same region'(test) {
        const app = new cdk.App();
        const sourceStack = new cdk.Stack(app, 'SourceStack');
        const rule = new rule_1.Rule(sourceStack, 'Rule');
        const targetStack = new cdk.Stack(app, 'TargetStack', { env: { region: 'us-west-2' } });
        const resource = new cdk.Construct(targetStack, 'Resource');
        test.throws(() => {
            rule.addTarget(new SomeTarget('T', resource));
        }, /Rule and target must be in the same region/);
        test.done();
    },
    'sqsParameters are generated when they are specified in target props'(test) {
        const stack = new cdk.Stack();
        const t1 = {
            bind: () => ({
                id: '',
                arn: 'ARN1',
                sqsParameters: { messageGroupId: 'messageGroupId' }
            })
        };
        new rule_1.Rule(stack, 'EventRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(5)),
            targets: [t1],
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            Targets: [
                {
                    'Arn': 'ARN1',
                    'Id': 'Target0',
                    'SqsParameters': {
                        'MessageGroupId': 'messageGroupId'
                    }
                }
            ]
        }));
        test.done();
    },
    'associate rule with event bus'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const eventBus = new lib_1.EventBus(stack, 'EventBus');
        // WHEN
        new rule_1.Rule(stack, 'MyRule', {
            eventPattern: {
                detail: ['detail']
            },
            eventBus,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            EventBusName: {
                Ref: 'EventBus7B8748AA'
            }
        }));
        test.done();
    },
    'throws with eventBus and schedule'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'MyStack');
        const eventBus = new lib_1.EventBus(stack, 'EventBus');
        // THEN
        test.throws(() => new rule_1.Rule(stack, 'MyRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(10)),
            eventBus,
        }), /Cannot associate rule with 'eventBus' when using 'schedule'/);
        test.done();
    },
    'for cross-account targets': {
        'requires that the source stack specify a concrete account'(test) {
            const app = new cdk.App();
            const sourceStack = new cdk.Stack(app, 'SourceStack');
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(app, 'TargetStack', { env: { account: targetAccount } });
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /You need to provide a concrete account for the source stack when using cross-account events/);
            test.done();
        },
        'requires that the target stack specify a concrete account'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(app, 'SourceStack', { env: { account: sourceAccount } });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetStack = new cdk.Stack(app, 'TargetStack');
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /You need to provide a concrete account for the target stack when using cross-account events/);
            test.done();
        },
        'requires that the target stack specify a concrete region'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(app, 'SourceStack', { env: { account: sourceAccount } });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(app, 'TargetStack', { env: { account: targetAccount } });
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /You need to provide a concrete region for the target stack when using cross-account events/);
            test.done();
        },
        'requires that the source stack be part of an App'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(undefined, 'SourceStack', { env: { account: sourceAccount, region: 'us-west-2' } });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(app, 'TargetStack', { env: { account: targetAccount, region: 'us-west-2' } });
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /Event stack which uses cross-account targets must be part of a CDK app/);
            test.done();
        },
        'requires that the target stack be part of an App'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(app, 'SourceStack', { env: { account: sourceAccount, region: 'us-west-2' } });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(undefined, 'TargetStack', { env: { account: targetAccount, region: 'us-west-2' } });
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /Target stack which uses cross-account event targets must be part of a CDK app/);
            test.done();
        },
        'requires that the source and target stacks be part of the same App'(test) {
            const sourceApp = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(sourceApp, 'SourceStack', { env: { account: sourceAccount, region: 'us-west-2' } });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetApp = new cdk.App();
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(targetApp, 'TargetStack', { env: { account: targetAccount, region: 'us-west-2' } });
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /Event stack and target stack must belong to the same CDK app/);
            test.done();
        },
        'generates the correct rules in the source and target stacks when eventPattern is passed in the constructor'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(app, 'SourceStack', {
                env: {
                    account: sourceAccount,
                    region: 'us-west-2',
                },
            });
            const rule = new rule_1.Rule(sourceStack, 'Rule', {
                eventPattern: {
                    source: ['some-event'],
                },
            });
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(app, 'TargetStack', {
                env: {
                    account: targetAccount,
                    region: 'us-west-2',
                },
            });
            const resource1 = new cdk.Construct(targetStack, 'Resource1');
            const resource2 = new cdk.Construct(targetStack, 'Resource2');
            rule.addTarget(new SomeTarget('T1', resource1));
            rule.addTarget(new SomeTarget('T2', resource2));
            assert_1.expect(sourceStack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                'EventPattern': {
                    'source': [
                        'some-event',
                    ],
                },
                'State': 'ENABLED',
                'Targets': [
                    {
                        'Id': 'T1',
                        'Arn': {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    { 'Ref': 'AWS::Partition' },
                                    `:events:us-west-2:${targetAccount}:event-bus/default`,
                                ],
                            ],
                        },
                    },
                ],
            }));
            assert_1.expect(targetStack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                'EventPattern': {
                    'source': [
                        'some-event',
                    ],
                },
                'State': 'ENABLED',
                'Targets': [
                    {
                        'Id': 'T1',
                        'Arn': 'ARN1',
                    },
                ],
            }));
            assert_1.expect(targetStack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                'EventPattern': {
                    'source': [
                        'some-event',
                    ],
                },
                'State': 'ENABLED',
                'Targets': [
                    {
                        'Id': 'T2',
                        'Arn': 'ARN1',
                    },
                ],
            }));
            const eventBusPolicyStack = app.node.findChild(`EventBusPolicy-${sourceAccount}-us-west-2-${targetAccount}`);
            assert_1.expect(eventBusPolicyStack).to(assert_1.haveResourceLike('AWS::Events::EventBusPolicy', {
                'Action': 'events:PutEvents',
                'StatementId': 'MySid',
                'Principal': sourceAccount,
            }));
            test.done();
        },
        'generates the correct rule in the target stack when addEventPattern in the source rule is used'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(app, 'SourceStack', {
                env: {
                    account: sourceAccount,
                    region: 'us-west-2',
                },
            });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(app, 'TargetStack', {
                env: {
                    account: targetAccount,
                    region: 'us-west-2',
                },
            });
            const resource = new cdk.Construct(targetStack, 'Resource1');
            rule.addTarget(new SomeTarget('T', resource));
            rule.addEventPattern({
                source: ['some-event'],
            });
            assert_1.expect(targetStack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                'EventPattern': {
                    'source': [
                        'some-event',
                    ],
                },
                'State': 'ENABLED',
                'Targets': [
                    {
                        'Id': 'T',
                        'Arn': 'ARN1',
                    },
                ],
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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