"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationListener = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const base_listener_1 = require("../shared/base-listener");
const enums_1 = require("../shared/enums");
const listener_certificate_1 = require("../shared/listener-certificate");
const util_1 = require("../shared/util");
const application_listener_action_1 = require("./application-listener-action");
const application_listener_certificate_1 = require("./application-listener-certificate");
const application_listener_rule_1 = require("./application-listener-rule");
const application_target_group_1 = require("./application-target-group");
/**
 * Define an ApplicationListener.
 *
 * @stability stable
 * @resource AWS::ElasticLoadBalancingV2::Listener
 */
class ApplicationListener extends base_listener_1.BaseListener {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_ApplicationListenerProps(props);
        const [protocol, port] = util_1.determineProtocolAndPort(props.protocol, props.port);
        if (protocol === undefined || port === undefined) {
            throw new Error('At least one of \'port\' or \'protocol\' is required');
        }
        super(scope, id, {
            loadBalancerArn: props.loadBalancer.loadBalancerArn,
            certificates: core_1.Lazy.any({ produce: () => this.certificateArns.map(certificateArn => ({ certificateArn })) }, { omitEmptyArray: true }),
            protocol,
            port,
            sslPolicy: props.sslPolicy,
        });
        this.loadBalancer = props.loadBalancer;
        this.protocol = protocol;
        this.certificateArns = [];
        // Attach certificates
        if (props.certificateArns && props.certificateArns.length > 0) {
            this.addCertificateArns('ListenerCertificate', props.certificateArns);
        }
        if (props.certificates && props.certificates.length > 0) {
            this.addCertificates('DefaultCertificates', props.certificates);
        }
        // This listener edits the securitygroup of the load balancer,
        // but adds its own default port.
        this.connections = new ec2.Connections({
            securityGroups: props.loadBalancer.connections.securityGroups,
            defaultPort: ec2.Port.tcp(port),
        });
        if (props.defaultAction && props.defaultTargetGroups) {
            throw new Error('Specify at most one of \'defaultAction\' and \'defaultTargetGroups\'');
        }
        if (props.defaultAction) {
            this.setDefaultAction(props.defaultAction);
        }
        if (props.defaultTargetGroups) {
            this.setDefaultAction(application_listener_action_1.ListenerAction.forward(props.defaultTargetGroups));
        }
        if (props.open !== false) {
            this.connections.allowDefaultPortFrom(ec2.Peer.anyIpv4(), `Allow from anyone on port ${port}`);
            if (this.loadBalancer.ipAddressType === enums_1.IpAddressType.DUAL_STACK) {
                this.connections.allowDefaultPortFrom(ec2.Peer.anyIpv6(), `Allow from anyone on port ${port}`);
            }
        }
    }
    /**
     * Look up an ApplicationListener.
     *
     * @stability stable
     */
    static fromLookup(scope, id, options) {
        jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_ApplicationListenerLookupOptions(options);
        if (core_1.Token.isUnresolved(options.listenerArn)) {
            throw new Error('All arguments to look up a load balancer listener must be concrete (no Tokens)');
        }
        let listenerProtocol;
        switch (options.listenerProtocol) {
            case enums_1.ApplicationProtocol.HTTP:
                listenerProtocol = cxschema.LoadBalancerListenerProtocol.HTTP;
                break;
            case enums_1.ApplicationProtocol.HTTPS:
                listenerProtocol = cxschema.LoadBalancerListenerProtocol.HTTPS;
                break;
        }
        const props = base_listener_1.BaseListener._queryContextProvider(scope, {
            userOptions: options,
            loadBalancerType: cxschema.LoadBalancerType.APPLICATION,
            listenerArn: options.listenerArn,
            listenerProtocol,
        });
        return new LookedUpApplicationListener(scope, id, props);
    }
    /**
     * Import an existing listener.
     *
     * @stability stable
     */
    static fromApplicationListenerAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_ApplicationListenerAttributes(attrs);
        return new ImportedApplicationListener(scope, id, attrs);
    }
    /**
     * (deprecated) Add one or more certificates to this listener.
     *
     * After the first certificate, this creates ApplicationListenerCertificates
     * resources since cloudformation requires the certificates array on the
     * listener resource to have a length of 1.
     *
     * @deprecated Use `addCertificates` instead.
     */
    addCertificateArns(id, arns) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-elasticloadbalancingv2.ApplicationListener#addCertificateArns", "Use `addCertificates` instead.");
        this.addCertificates(id, arns.map(listener_certificate_1.ListenerCertificate.fromArn));
    }
    /**
     * Add one or more certificates to this listener.
     *
     * After the first certificate, this creates ApplicationListenerCertificates
     * resources since cloudformation requires the certificates array on the
     * listener resource to have a length of 1.
     *
     * @stability stable
     */
    addCertificates(id, certificates) {
        const additionalCerts = [...certificates];
        if (this.certificateArns.length === 0 && additionalCerts.length > 0) {
            const first = additionalCerts.splice(0, 1)[0];
            this.certificateArns.push(first.certificateArn);
        }
        // Only one certificate can be specified per resource, even though
        // `certificates` is of type Array
        for (let i = 0; i < additionalCerts.length; i++) {
            new application_listener_certificate_1.ApplicationListenerCertificate(this, `${id}${i + 1}`, {
                listener: this,
                certificates: [additionalCerts[i]],
            });
        }
    }
    /**
     * Perform the given default action on incoming requests.
     *
     * This allows full control of the default action of the load balancer,
     * including Action chaining, fixed responses and redirect responses. See
     * the `ListenerAction` class for all options.
     *
     * It's possible to add routing conditions to the Action added in this way.
     * At least one Action must be added without conditions (which becomes the
     * default Action).
     *
     * @stability stable
     */
    addAction(id, props) {
        jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_AddApplicationActionProps(props);
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            //
            // TargetGroup.registerListener is called inside ApplicationListenerRule.
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                priority: props.priority,
                ...props,
            });
        }
        else {
            // New default target with these targetgroups
            this.setDefaultAction(props.action);
        }
    }
    /**
     * Load balance incoming requests to the given target groups.
     *
     * All target groups will be load balanced to with equal weight and without
     * stickiness. For a more complex configuration than that, use `addAction()`.
     *
     * It's possible to add routing conditions to the TargetGroups added in this
     * way. At least one TargetGroup must be added without conditions (which will
     * become the default Action for this listener).
     *
     * @stability stable
     */
    addTargetGroups(id, props) {
        jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_AddApplicationTargetGroupsProps(props);
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            //
            // TargetGroup.registerListener is called inside ApplicationListenerRule.
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                priority: props.priority,
                ...props,
            });
        }
        else {
            // New default target with these targetgroups
            this.setDefaultAction(application_listener_action_1.ListenerAction.forward(props.targetGroups));
        }
    }
    /**
     * Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates an ApplicationTargetGroup for the targets
     * involved, and a 'forward' action to route traffic to the given TargetGroup.
     *
     * If you want more control over the precise setup, create the TargetGroup
     * and use `addAction` yourself.
     *
     * It's possible to add conditions to the targets added in this way. At least
     * one set of targets must be added without conditions.
     *
     * @returns The newly created target group
     * @stability stable
     */
    addTargets(id, props) {
        jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_AddApplicationTargetsProps(props);
        if (!this.loadBalancer.vpc) {
            // eslint-disable-next-line max-len
            throw new Error('Can only call addTargets() when using a constructed Load Balancer or an imported Load Balancer with specified vpc; construct a new TargetGroup and use addTargetGroup');
        }
        const group = new application_target_group_1.ApplicationTargetGroup(this, id + 'Group', {
            vpc: this.loadBalancer.vpc,
            ...props,
        });
        this.addTargetGroups(id, {
            targetGroups: [group],
            ...props,
        });
        return group;
    }
    /**
     * (deprecated) Add a fixed response.
     *
     * @deprecated Use `addAction()` instead
     */
    addFixedResponse(id, props) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-elasticloadbalancingv2.ApplicationListener#addFixedResponse", "Use `addAction()` instead");
        jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_AddFixedResponseProps(props);
        checkAddRuleProps(props);
        const fixedResponse = {
            statusCode: props.statusCode,
            contentType: props.contentType,
            messageBody: props.messageBody,
        };
        application_listener_rule_1.validateFixedResponse(fixedResponse);
        if (props.priority) {
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                priority: props.priority,
                fixedResponse,
                ...props,
            });
        }
        else {
            this.setDefaultAction(application_listener_action_1.ListenerAction.fixedResponse(core_1.Token.asNumber(props.statusCode), {
                contentType: props.contentType,
                messageBody: props.messageBody,
            }));
        }
    }
    /**
     * (deprecated) Add a redirect response.
     *
     * @deprecated Use `addAction()` instead
     */
    addRedirectResponse(id, props) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-elasticloadbalancingv2.ApplicationListener#addRedirectResponse", "Use `addAction()` instead");
        jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_AddRedirectResponseProps(props);
        checkAddRuleProps(props);
        const redirectResponse = {
            host: props.host,
            path: props.path,
            port: props.port,
            protocol: props.protocol,
            query: props.query,
            statusCode: props.statusCode,
        };
        application_listener_rule_1.validateRedirectResponse(redirectResponse);
        if (props.priority) {
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                priority: props.priority,
                redirectResponse,
                ...props,
            });
        }
        else {
            this.setDefaultAction(application_listener_action_1.ListenerAction.redirect({
                host: props.host,
                path: props.path,
                port: props.port,
                protocol: props.protocol,
                query: props.query,
                permanent: props.statusCode === 'HTTP_301',
            }));
        }
    }
    /**
     * Register that a connectable that has been added to this load balancer.
     *
     * Don't call this directly. It is called by ApplicationTargetGroup.
     *
     * @stability stable
     */
    registerConnectable(connectable, portRange) {
        connectable.connections.allowFrom(this.loadBalancer, portRange, 'Load balancer to target');
    }
    /**
     * Validate this listener.
     *
     * @stability stable
     */
    validate() {
        const errors = super.validate();
        if (this.protocol === enums_1.ApplicationProtocol.HTTPS && this.certificateArns.length === 0) {
            errors.push('HTTPS Listener needs at least one certificate (call addCertificates)');
        }
        return errors;
    }
    /**
     * Wrapper for _setDefaultAction which does a type-safe bind
     */
    setDefaultAction(action) {
        action.bind(this, this);
        this._setDefaultAction(action);
    }
}
exports.ApplicationListener = ApplicationListener;
_a = JSII_RTTI_SYMBOL_1;
ApplicationListener[_a] = { fqn: "@aws-cdk/aws-elasticloadbalancingv2.ApplicationListener", version: "1.142.0" };
class ExternalApplicationListener extends core_1.Resource {
    constructor(scope, id) {
        super(scope, id);
    }
    /**
     * Register that a connectable that has been added to this load balancer.
     *
     * Don't call this directly. It is called by ApplicationTargetGroup.
     */
    registerConnectable(connectable, portRange) {
        this.connections.allowTo(connectable, portRange, 'Load balancer to target');
    }
    /**
     * Add one or more certificates to this listener.
     * @deprecated use `addCertificates()`
     */
    addCertificateArns(id, arns) {
        this.addCertificates(id, arns.map(listener_certificate_1.ListenerCertificate.fromArn));
    }
    /**
     * Add one or more certificates to this listener.
     */
    addCertificates(id, certificates) {
        new application_listener_certificate_1.ApplicationListenerCertificate(this, id, {
            listener: this,
            certificates,
        });
    }
    /**
     * Load balance incoming requests to the given target groups.
     *
     * It's possible to add conditions to the TargetGroups added in this way.
     * At least one TargetGroup must be added without conditions.
     */
    addTargetGroups(id, props) {
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            new application_listener_rule_1.ApplicationListenerRule(this, id, {
                listener: this,
                priority: props.priority,
                ...props,
            });
        }
        else {
            throw new Error('Cannot add default Target Groups to imported ApplicationListener');
        }
    }
    /**
     * Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates an ApplicationTargetGroup for the targets
     * involved.
     *
     * It's possible to add conditions to the targets added in this way. At least
     * one set of targets must be added without conditions.
     *
     * @returns The newly created target group
     */
    addTargets(_id, _props) {
        // eslint-disable-next-line max-len
        throw new Error('Can only call addTargets() when using a constructed ApplicationListener; construct a new TargetGroup and use addTargetGroup.');
    }
}
/**
 * An imported application listener.
 */
class ImportedApplicationListener extends ExternalApplicationListener {
    constructor(scope, id, props) {
        super(scope, id);
        this.listenerArn = props.listenerArn;
        const defaultPort = props.defaultPort !== undefined ? ec2.Port.tcp(props.defaultPort) : undefined;
        let securityGroup;
        if (props.securityGroup) {
            securityGroup = props.securityGroup;
        }
        else if (props.securityGroupId) {
            securityGroup = ec2.SecurityGroup.fromSecurityGroupId(this, 'SecurityGroup', props.securityGroupId, {
                allowAllOutbound: props.securityGroupAllowsAllOutbound,
            });
        }
        else {
            throw new Error('Either `securityGroup` or `securityGroupId` must be specified to import an application listener.');
        }
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort,
        });
    }
}
class LookedUpApplicationListener extends ExternalApplicationListener {
    constructor(scope, id, props) {
        super(scope, id);
        this.listenerArn = props.listenerArn;
        this.connections = new ec2.Connections({
            defaultPort: ec2.Port.tcp(props.listenerPort),
        });
        for (const securityGroupId of props.securityGroupIds) {
            const securityGroup = ec2.SecurityGroup.fromLookupById(this, `SecurityGroup-${securityGroupId}`, securityGroupId);
            this.connections.addSecurityGroup(securityGroup);
        }
    }
}
function checkAddRuleProps(props) {
    var _b;
    const conditionsCount = ((_b = props.conditions) === null || _b === void 0 ? void 0 : _b.length) || 0;
    const hasAnyConditions = conditionsCount !== 0 ||
        props.hostHeader !== undefined || props.pathPattern !== undefined || props.pathPatterns !== undefined;
    const hasPriority = props.priority !== undefined;
    if (hasAnyConditions !== hasPriority) {
        throw new Error('Setting \'conditions\', \'pathPattern\' or \'hostHeader\' also requires \'priority\', and vice versa');
    }
}
//# sourceMappingURL=data:application/json;base64,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