"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkTargetGroup = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const cdk = require("@aws-cdk/core");
const base_target_group_1 = require("../shared/base-target-group");
const enums_1 = require("../shared/enums");
const imported_1 = require("../shared/imported");
const util_1 = require("../shared/util");
/**
 * Define a Network Target Group.
 *
 * @stability stable
 */
class NetworkTargetGroup extends base_target_group_1.TargetGroupBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_NetworkTargetGroupProps(props);
        const proto = props.protocol || enums_1.Protocol.TCP;
        util_1.validateNetworkProtocol(proto);
        super(scope, id, props, {
            protocol: proto,
            port: props.port,
        });
        this.listeners = [];
        if (props.proxyProtocolV2 != null) {
            this.setAttribute('proxy_protocol_v2.enabled', props.proxyProtocolV2 ? 'true' : 'false');
        }
        if (props.preserveClientIp !== undefined) {
            this.setAttribute('preserve_client_ip.enabled', props.preserveClientIp ? 'true' : 'false');
        }
        this.addTarget(...(props.targets || []));
    }
    /**
     * Import an existing target group.
     *
     * @stability stable
     */
    static fromTargetGroupAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_TargetGroupAttributes(attrs);
        return new ImportedNetworkTargetGroup(scope, id, attrs);
    }
    /**
     * (deprecated) Import an existing listener.
     *
     * @deprecated Use `fromTargetGroupAttributes` instead
     */
    static import(scope, id, props) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-elasticloadbalancingv2.NetworkTargetGroup#import", "Use `fromTargetGroupAttributes` instead");
        jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_TargetGroupImportProps(props);
        return NetworkTargetGroup.fromTargetGroupAttributes(scope, id, props);
    }
    /**
     * Add a load balancing target to this target group.
     *
     * @stability stable
     */
    addTarget(...targets) {
        jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_INetworkLoadBalancerTarget(targets);
        for (const target of targets) {
            const result = target.attachToNetworkTargetGroup(this);
            this.addLoadBalancerTarget(result);
        }
    }
    /**
     * Register a listener that is load balancing to this target group.
     *
     * Don't call this directly. It will be called by listeners.
     *
     * @stability stable
     */
    registerListener(listener) {
        jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_INetworkListener(listener);
        this.loadBalancerAttachedDependencies.add(listener);
        this.listeners.push(listener);
    }
    /**
     * The number of targets that are considered healthy.
     *
     * @default Average over 5 minutes
     * @stability stable
     */
    metricHealthyHostCount(props) {
        return this.metric('HealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of targets that are considered unhealthy.
     *
     * @default Average over 5 minutes
     * @stability stable
     */
    metricUnHealthyHostCount(props) {
        return this.metric('UnHealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * Full name of first load balancer.
     *
     * @stability stable
     */
    get firstLoadBalancerFullName() {
        if (this.listeners.length === 0) {
            throw new Error('The TargetGroup needs to be attached to a LoadBalancer before you can call this method');
        }
        return base_target_group_1.loadBalancerNameFromListenerArn(this.listeners[0].listenerArn);
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        const ret = super.validate();
        const healthCheck = this.healthCheck || {};
        const allowedIntervals = [10, 30];
        if (healthCheck.interval) {
            const seconds = healthCheck.interval.toSeconds();
            if (!cdk.Token.isUnresolved(seconds) && !allowedIntervals.includes(seconds)) {
                ret.push(`Health check interval '${seconds}' not supported. Must be one of the following values '${allowedIntervals.join(',')}'.`);
            }
        }
        if (healthCheck.healthyThresholdCount) {
            const thresholdCount = healthCheck.healthyThresholdCount;
            if (thresholdCount < 2 || thresholdCount > 10) {
                ret.push(`Healthy Threshold Count '${thresholdCount}' not supported. Must be a number between 2 and 10.`);
            }
        }
        if (healthCheck.unhealthyThresholdCount) {
            const thresholdCount = healthCheck.unhealthyThresholdCount;
            if (thresholdCount < 2 || thresholdCount > 10) {
                ret.push(`Unhealthy Threshold Count '${thresholdCount}' not supported. Must be a number between 2 and 10.`);
            }
        }
        if (healthCheck.healthyThresholdCount && healthCheck.unhealthyThresholdCount &&
            healthCheck.healthyThresholdCount !== healthCheck.unhealthyThresholdCount) {
            ret.push([
                `Healthy and Unhealthy Threshold Counts must be the same: ${healthCheck.healthyThresholdCount}`,
                `is not equal to ${healthCheck.unhealthyThresholdCount}.`,
            ].join(' '));
        }
        if (!healthCheck.protocol) {
            return ret;
        }
        if (!NLB_HEALTH_CHECK_PROTOCOLS.includes(healthCheck.protocol)) {
            ret.push(`Health check protocol '${healthCheck.protocol}' is not supported. Must be one of [${NLB_HEALTH_CHECK_PROTOCOLS.join(', ')}]`);
        }
        if (healthCheck.path && !NLB_PATH_HEALTH_CHECK_PROTOCOLS.includes(healthCheck.protocol)) {
            ret.push([
                `'${healthCheck.protocol}' health checks do not support the path property.`,
                `Must be one of [${NLB_PATH_HEALTH_CHECK_PROTOCOLS.join(', ')}]`,
            ].join(' '));
        }
        if (healthCheck.timeout && healthCheck.timeout.toSeconds() !== NLB_HEALTH_CHECK_TIMEOUTS[healthCheck.protocol]) {
            ret.push([
                'Custom health check timeouts are not supported for Network Load Balancer health checks.',
                `Expected ${NLB_HEALTH_CHECK_TIMEOUTS[healthCheck.protocol]} seconds for ${healthCheck.protocol}, got ${healthCheck.timeout.toSeconds()}`,
            ].join(' '));
        }
        return ret;
    }
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/NetworkELB',
            metricName,
            dimensionsMap: { LoadBalancer: this.firstLoadBalancerFullName, TargetGroup: this.targetGroupFullName },
            ...props,
        }).attachTo(this);
    }
}
exports.NetworkTargetGroup = NetworkTargetGroup;
_a = JSII_RTTI_SYMBOL_1;
NetworkTargetGroup[_a] = { fqn: "@aws-cdk/aws-elasticloadbalancingv2.NetworkTargetGroup", version: "1.142.0" };
/**
 * An imported network target group
 */
class ImportedNetworkTargetGroup extends imported_1.ImportedTargetGroupBase {
    registerListener(_listener) {
    }
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToNetworkTargetGroup(this);
            if (result.targetJson !== undefined) {
                throw new Error('Cannot add a non-self registering target to an imported TargetGroup. Create a new TargetGroup instead.');
            }
        }
    }
}
const NLB_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS, enums_1.Protocol.TCP];
const NLB_PATH_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS];
const NLB_HEALTH_CHECK_TIMEOUTS = {
    [enums_1.Protocol.HTTP]: 6,
    [enums_1.Protocol.HTTPS]: 10,
    [enums_1.Protocol.TCP]: 10,
};
//# sourceMappingURL=data:application/json;base64,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