"use strict";
const assert_1 = require("@aws-cdk/assert");
const codebuild = require("@aws-cdk/aws-codebuild");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const core_1 = require("@aws-cdk/core");
const cpactions = require("../../lib");
function createBitBucketAndCodeBuildPipeline(stack, props) {
    const sourceOutput = new codepipeline.Artifact();
    new codepipeline.Pipeline(stack, 'Pipeline', {
        stages: [
            {
                stageName: 'Source',
                actions: [
                    new cpactions.BitBucketSourceAction({
                        actionName: 'BitBucket',
                        owner: 'aws',
                        repo: 'aws-cdk',
                        output: sourceOutput,
                        connectionArn: 'arn:aws:codestar-connections:us-east-1:123456789012:connection/12345678-abcd-12ab-34cdef5678gh',
                        codeBuildCloneOutput: props.codeBuildCloneOutput,
                    }),
                ],
            },
            {
                stageName: 'Build',
                actions: [
                    new cpactions.CodeBuildAction({
                        actionName: 'CodeBuild',
                        project: new codebuild.PipelineProject(stack, 'MyProject'),
                        input: sourceOutput,
                        outputs: [new codepipeline.Artifact()],
                    }),
                ],
            },
        ],
    });
}
module.exports = {
    'BitBucket source Action': {
        'produces the correct configuration when added to a pipeline'(test) {
            const stack = new core_1.Stack();
            createBitBucketAndCodeBuildPipeline(stack, {
                codeBuildCloneOutput: false,
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source',
                        'Actions': [
                            {
                                'Name': 'BitBucket',
                                'ActionTypeId': {
                                    'Owner': 'AWS',
                                    'Provider': 'CodeStarSourceConnection',
                                },
                                'Configuration': {
                                    'ConnectionArn': 'arn:aws:codestar-connections:us-east-1:123456789012:connection/12345678-abcd-12ab-34cdef5678gh',
                                    'FullRepositoryId': 'aws/aws-cdk',
                                    'BranchName': 'master',
                                },
                            },
                        ],
                    },
                    {
                        'Name': 'Build',
                        'Actions': [
                            {
                                'Name': 'CodeBuild',
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
    },
    'setting codeBuildCloneOutput=true adds permission to use the connection to the following CodeBuild Project'(test) {
        const stack = new core_1.Stack();
        createBitBucketAndCodeBuildPipeline(stack, {
            codeBuildCloneOutput: true,
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:PutLogEvents',
                        ],
                    },
                    {},
                    {},
                    {},
                    {},
                    {
                        'Action': 'codestar-connections:UseConnection',
                        'Effect': 'Allow',
                        'Resource': 'arn:aws:codestar-connections:us-east-1:123456789012:connection/12345678-abcd-12ab-34cdef5678gh',
                    },
                ],
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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