"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_cloudformation_1 = require("@aws-cdk/aws-cloudformation");
const codebuild = require("@aws-cdk/aws-codebuild");
const codecommit = require("@aws-cdk/aws-codecommit");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const cpactions = require("../../lib");
/**
 * A test stack with a half-prepared pipeline ready to add CloudFormation actions to
 */
class TestFixture extends cdk.Stack {
    constructor() {
        super();
        this.pipeline = new codepipeline.Pipeline(this, 'Pipeline');
        this.sourceStage = this.pipeline.addStage({ stageName: 'Source' });
        this.deployStage = this.pipeline.addStage({ stageName: 'Deploy' });
        this.repo = new codecommit.Repository(this, 'MyVeryImportantRepo', {
            repositoryName: 'my-very-important-repo',
        });
        this.sourceOutput = new codepipeline.Artifact('SourceArtifact');
        const source = new cpactions.CodeCommitSourceAction({
            actionName: 'Source',
            output: this.sourceOutput,
            repository: this.repo,
        });
        this.sourceStage.addAction(source);
    }
}
module.exports = {
    'CreateChangeSetAction can be used to make a change set from a CodePipeline'(test) {
        const stack = new cdk.Stack();
        const pipeline = new codepipeline.Pipeline(stack, 'MagicPipeline');
        const changeSetExecRole = new aws_iam_1.Role(stack, 'ChangeSetRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('cloudformation.amazonaws.com'),
        });
        /** Source! */
        const repo = new codecommit.Repository(stack, 'MyVeryImportantRepo', {
            repositoryName: 'my-very-important-repo',
        });
        const sourceOutput = new codepipeline.Artifact('SourceArtifact');
        const source = new cpactions.CodeCommitSourceAction({
            actionName: 'source',
            output: sourceOutput,
            repository: repo,
            trigger: cpactions.CodeCommitTrigger.POLL,
        });
        pipeline.addStage({
            stageName: 'source',
            actions: [source],
        });
        /** Build! */
        const project = new codebuild.PipelineProject(stack, 'MyBuildProject');
        const buildOutput = new codepipeline.Artifact('OutputYo');
        const buildAction = new cpactions.CodeBuildAction({
            actionName: 'build',
            project,
            input: sourceOutput,
            outputs: [buildOutput],
        });
        pipeline.addStage({
            stageName: 'build',
            actions: [buildAction],
        });
        /** Deploy! */
        // To execute a change set - yes, you probably do need *:* 🤷‍♀️
        changeSetExecRole.addToPolicy(new aws_iam_1.PolicyStatement({ resources: ['*'], actions: ['*'] }));
        const stackName = 'BrelandsStack';
        const changeSetName = 'MyMagicalChangeSet';
        pipeline.addStage({
            stageName: 'prod',
            actions: [
                new cpactions.CloudFormationCreateReplaceChangeSetAction({
                    actionName: 'BuildChangeSetProd',
                    stackName,
                    changeSetName,
                    deploymentRole: changeSetExecRole,
                    templatePath: new codepipeline.ArtifactPath(buildOutput, 'template.yaml'),
                    templateConfiguration: new codepipeline.ArtifactPath(buildOutput, 'templateConfig.json'),
                    adminPermissions: false,
                }),
                new cpactions.CloudFormationExecuteChangeSetAction({
                    actionName: 'ExecuteChangeSetProd',
                    stackName,
                    changeSetName,
                }),
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'ArtifactStore': {
                'Location': {
                    'Ref': 'MagicPipelineArtifactsBucket212FE7BF',
                },
                'Type': 'S3',
            },
            'RoleArn': {
                'Fn::GetAtt': ['MagicPipelineRoleFB2BD6DE',
                    'Arn'],
            },
            'Stages': [{
                    'Actions': [
                        {
                            'ActionTypeId': {
                                'Category': 'Source',
                                'Owner': 'AWS',
                                'Provider': 'CodeCommit',
                                'Version': '1',
                            },
                            'Configuration': {
                                'RepositoryName': {
                                    'Fn::GetAtt': [
                                        'MyVeryImportantRepo11BC3EBD',
                                        'Name',
                                    ],
                                },
                                'BranchName': 'master',
                                'PollForSourceChanges': true,
                            },
                            'Name': 'source',
                            'OutputArtifacts': [
                                {
                                    'Name': 'SourceArtifact',
                                },
                            ],
                            'RunOrder': 1,
                        },
                    ],
                    'Name': 'source',
                },
                {
                    'Actions': [
                        {
                            'ActionTypeId': {
                                'Category': 'Build',
                                'Owner': 'AWS',
                                'Provider': 'CodeBuild',
                                'Version': '1',
                            },
                            'Configuration': {
                                'ProjectName': {
                                    'Ref': 'MyBuildProject30DB9D6E',
                                },
                            },
                            'InputArtifacts': [
                                {
                                    'Name': 'SourceArtifact',
                                },
                            ],
                            'Name': 'build',
                            'OutputArtifacts': [
                                {
                                    'Name': 'OutputYo',
                                },
                            ],
                            'RunOrder': 1,
                        },
                    ],
                    'Name': 'build',
                },
                {
                    'Actions': [
                        {
                            'ActionTypeId': {
                                'Category': 'Deploy',
                                'Owner': 'AWS',
                                'Provider': 'CloudFormation',
                                'Version': '1',
                            },
                            'Configuration': {
                                'ActionMode': 'CHANGE_SET_REPLACE',
                                'ChangeSetName': 'MyMagicalChangeSet',
                                'RoleArn': {
                                    'Fn::GetAtt': [
                                        'ChangeSetRole0BCF99E6',
                                        'Arn',
                                    ],
                                },
                                'StackName': 'BrelandsStack',
                                'TemplatePath': 'OutputYo::template.yaml',
                                'TemplateConfiguration': 'OutputYo::templateConfig.json',
                            },
                            'InputArtifacts': [{ 'Name': 'OutputYo' }],
                            'Name': 'BuildChangeSetProd',
                            'RunOrder': 1,
                        },
                        {
                            'ActionTypeId': {
                                'Category': 'Deploy',
                                'Owner': 'AWS',
                                'Provider': 'CloudFormation',
                                'Version': '1',
                            },
                            'Configuration': {
                                'ActionMode': 'CHANGE_SET_EXECUTE',
                                'ChangeSetName': 'MyMagicalChangeSet',
                            },
                            'Name': 'ExecuteChangeSetProd',
                            'RunOrder': 1,
                        },
                    ],
                    'Name': 'prod',
                }],
        }));
        test.done();
    },
    'fullPermissions leads to admin role and full IAM capabilities with pipeline bucket+key read permissions'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            adminPermissions: true,
        }));
        const roleId = 'PipelineDeployCreateUpdateRole515CB7D4';
        // THEN: Action in Pipeline has named IAM capabilities
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                { 'Name': 'Source' /* don't care about the rest */ },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'Configuration': {
                                'Capabilities': 'CAPABILITY_NAMED_IAM',
                                'RoleArn': { 'Fn::GetAtt': [roleId, 'Arn'] },
                                'ActionMode': 'CREATE_UPDATE',
                                'StackName': 'MyStack',
                                'TemplatePath': 'SourceArtifact::template.yaml',
                            },
                            'InputArtifacts': [{ 'Name': 'SourceArtifact' }],
                            'Name': 'CreateUpdate',
                        },
                    ],
                },
            ],
        }));
        // THEN: Role is created with full permissions
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        'Action': [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        'Effect': 'Allow',
                    },
                    {
                        'Action': [
                            'kms:Decrypt',
                            'kms:DescribeKey',
                        ],
                        'Effect': 'Allow',
                    },
                    {
                        Action: '*',
                        Effect: 'Allow',
                        Resource: '*',
                    },
                ],
            },
            Roles: [{ Ref: roleId }],
        }));
        test.done();
    },
    'outputFileName leads to creation of output artifact'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            outputFileName: 'CreateResponse.json',
            adminPermissions: false,
        }));
        // THEN: Action has output artifacts
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                { 'Name': 'Source' /* don't care about the rest */ },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'OutputArtifacts': [{ 'Name': 'CreateUpdate_MyStack_Artifact' }],
                            'Name': 'CreateUpdate',
                        },
                    ],
                },
            ],
        }));
        test.done();
    },
    'replaceOnFailure switches action type'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            replaceOnFailure: true,
            adminPermissions: false,
        }));
        // THEN: Action has output artifacts
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                { 'Name': 'Source' /* don't care about the rest */ },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'Configuration': {
                                'ActionMode': 'REPLACE_ON_FAILURE',
                            },
                            'Name': 'CreateUpdate',
                        },
                    ],
                },
            ],
        }));
        test.done();
    },
    'parameterOverrides are serialized as a string'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            adminPermissions: false,
            parameterOverrides: {
                RepoName: stack.repo.repositoryName,
            },
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                { 'Name': 'Source' /* don't care about the rest */ },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'Configuration': {
                                'ParameterOverrides': {
                                    'Fn::Join': ['', [
                                            '{"RepoName":"',
                                            { 'Fn::GetAtt': ['MyVeryImportantRepo11BC3EBD', 'Name'] },
                                            '"}',
                                        ]],
                                },
                            },
                            'Name': 'CreateUpdate',
                        },
                    ],
                },
            ],
        }));
        test.done();
    },
    'Action service role is passed to template'(test) {
        const stack = new TestFixture();
        const importedRole = aws_iam_1.Role.fromRoleArn(stack, 'ImportedRole', 'arn:aws:iam::000000000000:role/action-role');
        const freshRole = new aws_iam_1.Role(stack, 'FreshRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('magicservice'),
        });
        stack.deployStage.addAction(new cpactions.CloudFormationExecuteChangeSetAction({
            actionName: 'ImportedRoleAction',
            role: importedRole,
            changeSetName: 'magicSet',
            stackName: 'magicStack',
        }));
        stack.deployStage.addAction(new cpactions.CloudFormationExecuteChangeSetAction({
            actionName: 'FreshRoleAction',
            role: freshRole,
            changeSetName: 'magicSet',
            stackName: 'magicStack',
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                {
                    'Name': 'Source',
                },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'Name': 'ImportedRoleAction',
                            'RoleArn': 'arn:aws:iam::000000000000:role/action-role',
                        },
                        {
                            'Name': 'FreshRoleAction',
                            'RoleArn': {
                                'Fn::GetAtt': [
                                    'FreshRole472F6E18',
                                    'Arn',
                                ],
                            },
                        },
                    ],
                },
            ],
        }));
        test.done();
    },
    'Single capability is passed to template'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            adminPermissions: false,
            capabilities: [
                aws_cloudformation_1.CloudFormationCapabilities.NAMED_IAM,
            ],
        }));
        const roleId = 'PipelineDeployCreateUpdateRole515CB7D4';
        // THEN: Action in Pipeline has named IAM capabilities
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                { 'Name': 'Source' /* don't care about the rest */ },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'Configuration': {
                                'Capabilities': 'CAPABILITY_NAMED_IAM',
                                'RoleArn': { 'Fn::GetAtt': [roleId, 'Arn'] },
                                'ActionMode': 'CREATE_UPDATE',
                                'StackName': 'MyStack',
                                'TemplatePath': 'SourceArtifact::template.yaml',
                            },
                            'InputArtifacts': [{ 'Name': 'SourceArtifact' }],
                            'Name': 'CreateUpdate',
                        },
                    ],
                },
            ],
        }));
        test.done();
    },
    'Multiple capabilities are passed to template'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            adminPermissions: false,
            capabilities: [
                aws_cloudformation_1.CloudFormationCapabilities.NAMED_IAM,
                aws_cloudformation_1.CloudFormationCapabilities.AUTO_EXPAND,
            ],
        }));
        const roleId = 'PipelineDeployCreateUpdateRole515CB7D4';
        // THEN: Action in Pipeline has named IAM and AUTOEXPAND capabilities
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                { 'Name': 'Source' /* don't care about the rest */ },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'Configuration': {
                                'Capabilities': 'CAPABILITY_NAMED_IAM,CAPABILITY_AUTO_EXPAND',
                                'RoleArn': { 'Fn::GetAtt': [roleId, 'Arn'] },
                                'ActionMode': 'CREATE_UPDATE',
                                'StackName': 'MyStack',
                                'TemplatePath': 'SourceArtifact::template.yaml',
                            },
                            'InputArtifacts': [{ 'Name': 'SourceArtifact' }],
                            'Name': 'CreateUpdate',
                        },
                    ],
                },
            ],
        }));
        test.done();
    },
    'Empty capabilities is not passed to template'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            adminPermissions: false,
            capabilities: [
                aws_cloudformation_1.CloudFormationCapabilities.NONE,
            ],
        }));
        const roleId = 'PipelineDeployCreateUpdateRole515CB7D4';
        // THEN: Action in Pipeline has no capabilities
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                { 'Name': 'Source' /* don't care about the rest */ },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'Configuration': {
                                'RoleArn': { 'Fn::GetAtt': [roleId, 'Arn'] },
                                'ActionMode': 'CREATE_UPDATE',
                                'StackName': 'MyStack',
                                'TemplatePath': 'SourceArtifact::template.yaml',
                            },
                            'InputArtifacts': [{ 'Name': 'SourceArtifact' }],
                            'Name': 'CreateUpdate',
                        },
                    ],
                },
            ],
        }));
        test.done();
    },
    'cross-account CFN Pipeline': {
        'correctly creates the deployment Role in the other account'(test) {
            const app = new cdk.App();
            const pipelineStack = new cdk.Stack(app, 'PipelineStack', {
                env: {
                    account: '234567890123',
                    region: 'us-west-2',
                },
            });
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.CodeCommitSourceAction({
                                actionName: 'CodeCommit',
                                repository: codecommit.Repository.fromRepositoryName(pipelineStack, 'Repo', 'RepoName'),
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Deploy',
                        actions: [
                            new cpactions.CloudFormationCreateUpdateStackAction({
                                actionName: 'CFN',
                                stackName: 'MyStack',
                                adminPermissions: true,
                                templatePath: sourceOutput.atPath('template.yaml'),
                                account: '123456789012',
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source',
                    },
                    {
                        'Name': 'Deploy',
                        'Actions': [
                            {
                                'Name': 'CFN',
                                'RoleArn': {
                                    'Fn::Join': ['', ['arn:', { 'Ref': 'AWS::Partition' },
                                            ':iam::123456789012:role/pipelinestack-support-123loycfnactionrole56af64af3590f311bc50']],
                                },
                                'Configuration': {
                                    'RoleArn': {
                                        'Fn::Join': ['', ['arn:', { 'Ref': 'AWS::Partition' },
                                                ':iam::123456789012:role/pipelinestack-support-123fndeploymentrole4668d9b5a30ce3dc4508']],
                                    },
                                },
                            },
                        ],
                    },
                ],
            }));
            // the pipeline's BucketPolicy should trust both CFN roles
            assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Action': [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            'Effect': 'Allow',
                            'Principal': {
                                'AWS': {
                                    'Fn::Join': ['', ['arn:', { 'Ref': 'AWS::Partition' },
                                            ':iam::123456789012:role/pipelinestack-support-123fndeploymentrole4668d9b5a30ce3dc4508']],
                                },
                            },
                        },
                        {
                            'Action': [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            'Effect': 'Allow',
                            'Principal': {
                                'AWS': {
                                    'Fn::Join': ['', ['arn:', { 'Ref': 'AWS::Partition' },
                                            ':iam::123456789012:role/pipelinestack-support-123loycfnactionrole56af64af3590f311bc50']],
                                },
                            },
                        },
                    ],
                },
            }));
            const otherStack = app.node.findChild('cross-account-support-stack-123456789012');
            assert_1.expect(otherStack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                'RoleName': 'pipelinestack-support-123loycfnactionrole56af64af3590f311bc50',
            }));
            assert_1.expect(otherStack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                'RoleName': 'pipelinestack-support-123fndeploymentrole4668d9b5a30ce3dc4508',
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5jbG91ZGZvcm1hdGlvbi1waXBlbGluZS1hY3Rpb25zLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGVzdC5jbG91ZGZvcm1hdGlvbi1waXBlbGluZS1hY3Rpb25zLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSw0Q0FBMkQ7QUFDM0Qsb0VBQXlFO0FBQ3pFLG9EQUFvRDtBQUNwRCxzREFBc0Q7QUFDdEQsMERBQTBEO0FBQzFELDhDQUEyRTtBQUMzRSxxQ0FBcUM7QUFFckMsdUNBQXVDO0FBMHBCdkM7O0dBRUc7QUFDSCxNQUFNLFdBQVksU0FBUSxHQUFHLENBQUMsS0FBSztJQU9qQztRQUNFLEtBQUssRUFBRSxDQUFDO1FBRVIsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQzVELElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsRUFBRSxTQUFTLEVBQUUsUUFBUSxFQUFFLENBQUMsQ0FBQztRQUNuRSxJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLEVBQUUsU0FBUyxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUM7UUFDbkUsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLFVBQVUsQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFLHFCQUFxQixFQUFFO1lBQ2pFLGNBQWMsRUFBRSx3QkFBd0I7U0FDekMsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUNoRSxNQUFNLE1BQU0sR0FBRyxJQUFJLFNBQVMsQ0FBQyxzQkFBc0IsQ0FBQztZQUNsRCxVQUFVLEVBQUUsUUFBUTtZQUNwQixNQUFNLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDekIsVUFBVSxFQUFFLElBQUksQ0FBQyxJQUFJO1NBQ3RCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBQ3JDLENBQUM7Q0FDRjtBQWpyQkQsaUJBQVM7SUFDUCw0RUFBNEUsQ0FBQyxJQUFVO1FBQ3JGLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE1BQU0sUUFBUSxHQUFHLElBQUksWUFBWSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsZUFBZSxDQUFDLENBQUM7UUFFbkUsTUFBTSxpQkFBaUIsR0FBRyxJQUFJLGNBQUksQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFO1lBQ3pELFNBQVMsRUFBRSxJQUFJLDBCQUFnQixDQUFDLDhCQUE4QixDQUFDO1NBQ2hFLENBQUMsQ0FBQztRQUVILGNBQWM7UUFDZCxNQUFNLElBQUksR0FBRyxJQUFJLFVBQVUsQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1lBQ25FLGNBQWMsRUFBRSx3QkFBd0I7U0FDekMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxZQUFZLEdBQUcsSUFBSSxZQUFZLENBQUMsUUFBUSxDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFDakUsTUFBTSxNQUFNLEdBQUcsSUFBSSxTQUFTLENBQUMsc0JBQXNCLENBQUM7WUFDbEQsVUFBVSxFQUFFLFFBQVE7WUFDcEIsTUFBTSxFQUFFLFlBQVk7WUFDcEIsVUFBVSxFQUFFLElBQUk7WUFDaEIsT0FBTyxFQUFFLFNBQVMsQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJO1NBQzFDLENBQUMsQ0FBQztRQUNILFFBQVEsQ0FBQyxRQUFRLENBQUM7WUFDaEIsU0FBUyxFQUFFLFFBQVE7WUFDbkIsT0FBTyxFQUFFLENBQUMsTUFBTSxDQUFDO1NBQ2xCLENBQUMsQ0FBQztRQUVILGFBQWE7UUFFYixNQUFNLE9BQU8sR0FBRyxJQUFJLFNBQVMsQ0FBQyxlQUFlLENBQUMsS0FBSyxFQUFFLGdCQUFnQixDQUFDLENBQUM7UUFFdkUsTUFBTSxXQUFXLEdBQUcsSUFBSSxZQUFZLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQzFELE1BQU0sV0FBVyxHQUFHLElBQUksU0FBUyxDQUFDLGVBQWUsQ0FBQztZQUNoRCxVQUFVLEVBQUUsT0FBTztZQUNuQixPQUFPO1lBQ1AsS0FBSyxFQUFFLFlBQVk7WUFDbkIsT0FBTyxFQUFFLENBQUMsV0FBVyxDQUFDO1NBQ3ZCLENBQUMsQ0FBQztRQUNILFFBQVEsQ0FBQyxRQUFRLENBQUM7WUFDaEIsU0FBUyxFQUFFLE9BQU87WUFDbEIsT0FBTyxFQUFFLENBQUMsV0FBVyxDQUFDO1NBQ3ZCLENBQUMsQ0FBQztRQUVILGNBQWM7UUFFZCxnRUFBZ0U7UUFDaEUsaUJBQWlCLENBQUMsV0FBVyxDQUFDLElBQUkseUJBQWUsQ0FBQyxFQUFFLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxFQUFFLE9BQU8sRUFBRSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBRXpGLE1BQU0sU0FBUyxHQUFHLGVBQWUsQ0FBQztRQUNsQyxNQUFNLGFBQWEsR0FBRyxvQkFBb0IsQ0FBQztRQUMzQyxRQUFRLENBQUMsUUFBUSxDQUFDO1lBQ2hCLFNBQVMsRUFBRSxNQUFNO1lBQ2pCLE9BQU8sRUFBRTtnQkFDUCxJQUFJLFNBQVMsQ0FBQywwQ0FBMEMsQ0FBQztvQkFDdkQsVUFBVSxFQUFFLG9CQUFvQjtvQkFDaEMsU0FBUztvQkFDVCxhQUFhO29CQUNiLGNBQWMsRUFBRSxpQkFBaUI7b0JBQ2pDLFlBQVksRUFBRSxJQUFJLFlBQVksQ0FBQyxZQUFZLENBQUMsV0FBVyxFQUFFLGVBQWUsQ0FBQztvQkFDekUscUJBQXFCLEVBQUUsSUFBSSxZQUFZLENBQUMsWUFBWSxDQUFDLFdBQVcsRUFBRSxxQkFBcUIsQ0FBQztvQkFDeEYsZ0JBQWdCLEVBQUUsS0FBSztpQkFDeEIsQ0FBQztnQkFDRixJQUFJLFNBQVMsQ0FBQyxvQ0FBb0MsQ0FBQztvQkFDakQsVUFBVSxFQUFFLHNCQUFzQjtvQkFDbEMsU0FBUztvQkFDVCxhQUFhO2lCQUNkLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsNkJBQTZCLEVBQUU7WUFDL0QsZUFBZSxFQUFFO2dCQUNmLFVBQVUsRUFBRTtvQkFDVixLQUFLLEVBQUUsc0NBQXNDO2lCQUM5QztnQkFDRCxNQUFNLEVBQUUsSUFBSTthQUNiO1lBQ0QsU0FBUyxFQUFFO2dCQUNULFlBQVksRUFBRSxDQUFDLDJCQUEyQjtvQkFDeEMsS0FBSyxDQUFDO2FBQ1Q7WUFDRCxRQUFRLEVBQUUsQ0FBQztvQkFDVCxTQUFTLEVBQUU7d0JBQ1Q7NEJBQ0UsY0FBYyxFQUFFO2dDQUNkLFVBQVUsRUFBRSxRQUFRO2dDQUNwQixPQUFPLEVBQUUsS0FBSztnQ0FDZCxVQUFVLEVBQUUsWUFBWTtnQ0FDeEIsU0FBUyxFQUFFLEdBQUc7NkJBQ2Y7NEJBQ0QsZUFBZSxFQUFFO2dDQUNmLGdCQUFnQixFQUFFO29DQUNoQixZQUFZLEVBQUU7d0NBQ1osNkJBQTZCO3dDQUM3QixNQUFNO3FDQUNQO2lDQUNGO2dDQUNELFlBQVksRUFBRSxRQUFRO2dDQUN0QixzQkFBc0IsRUFBRSxJQUFJOzZCQUM3Qjs0QkFDRCxNQUFNLEVBQUUsUUFBUTs0QkFDaEIsaUJBQWlCLEVBQUU7Z0NBQ2pCO29DQUNFLE1BQU0sRUFBRSxnQkFBZ0I7aUNBQ3pCOzZCQUNGOzRCQUNELFVBQVUsRUFBRSxDQUFDO3lCQUNkO3FCQUNGO29CQUNELE1BQU0sRUFBRSxRQUFRO2lCQUNqQjtnQkFDRDtvQkFDRSxTQUFTLEVBQUU7d0JBQ1Q7NEJBQ0UsY0FBYyxFQUFFO2dDQUNkLFVBQVUsRUFBRSxPQUFPO2dDQUNuQixPQUFPLEVBQUUsS0FBSztnQ0FDZCxVQUFVLEVBQUUsV0FBVztnQ0FDdkIsU0FBUyxFQUFFLEdBQUc7NkJBQ2Y7NEJBQ0QsZUFBZSxFQUFFO2dDQUNmLGFBQWEsRUFBRTtvQ0FDYixLQUFLLEVBQUUsd0JBQXdCO2lDQUNoQzs2QkFDRjs0QkFDRCxnQkFBZ0IsRUFBRTtnQ0FDaEI7b0NBQ0UsTUFBTSxFQUFFLGdCQUFnQjtpQ0FDekI7NkJBQ0Y7NEJBQ0QsTUFBTSxFQUFFLE9BQU87NEJBQ2YsaUJBQWlCLEVBQUU7Z0NBQ2pCO29DQUNFLE1BQU0sRUFBRSxVQUFVO2lDQUNuQjs2QkFDRjs0QkFDRCxVQUFVLEVBQUUsQ0FBQzt5QkFDZDtxQkFDRjtvQkFDRCxNQUFNLEVBQUUsT0FBTztpQkFDaEI7Z0JBQ0Q7b0JBQ0UsU0FBUyxFQUFFO3dCQUNUOzRCQUNFLGNBQWMsRUFBRTtnQ0FDZCxVQUFVLEVBQUUsUUFBUTtnQ0FDcEIsT0FBTyxFQUFFLEtBQUs7Z0NBQ2QsVUFBVSxFQUFFLGdCQUFnQjtnQ0FDNUIsU0FBUyxFQUFFLEdBQUc7NkJBQ2Y7NEJBQ0QsZUFBZSxFQUFFO2dDQUNmLFlBQVksRUFBRSxvQkFBb0I7Z0NBQ2xDLGVBQWUsRUFBRSxvQkFBb0I7Z0NBQ3JDLFNBQVMsRUFBRTtvQ0FDVCxZQUFZLEVBQUU7d0NBQ1osdUJBQXVCO3dDQUN2QixLQUFLO3FDQUNOO2lDQUNGO2dDQUNELFdBQVcsRUFBRSxlQUFlO2dDQUM1QixjQUFjLEVBQUUseUJBQXlCO2dDQUN6Qyx1QkFBdUIsRUFBRSwrQkFBK0I7NkJBQ3pEOzRCQUNELGdCQUFnQixFQUFFLENBQUMsRUFBRSxNQUFNLEVBQUUsVUFBVSxFQUFFLENBQUM7NEJBQzFDLE1BQU0sRUFBRSxvQkFBb0I7NEJBQzVCLFVBQVUsRUFBRSxDQUFDO3lCQUNkO3dCQUNEOzRCQUNFLGNBQWMsRUFBRTtnQ0FDZCxVQUFVLEVBQUUsUUFBUTtnQ0FDcEIsT0FBTyxFQUFFLEtBQUs7Z0NBQ2QsVUFBVSxFQUFFLGdCQUFnQjtnQ0FDNUIsU0FBUyxFQUFFLEdBQUc7NkJBQ2Y7NEJBQ0QsZUFBZSxFQUFFO2dDQUNmLFlBQVksRUFBRSxvQkFBb0I7Z0NBQ2xDLGVBQWUsRUFBRSxvQkFBb0I7NkJBQ3RDOzRCQUNELE1BQU0sRUFBRSxzQkFBc0I7NEJBQzlCLFVBQVUsRUFBRSxDQUFDO3lCQUNkO3FCQUNGO29CQUNELE1BQU0sRUFBRSxNQUFNO2lCQUNmLENBQUM7U0FDSCxDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUVkLENBQUM7SUFFRCx5R0FBeUcsQ0FBQyxJQUFVO1FBQ3BILFFBQVE7UUFDTixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQVcsRUFBRSxDQUFDO1FBRWhDLE9BQU87UUFDUCxLQUFLLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxJQUFJLFNBQVMsQ0FBQyxxQ0FBcUMsQ0FBQztZQUM5RSxVQUFVLEVBQUUsY0FBYztZQUMxQixTQUFTLEVBQUUsU0FBUztZQUNwQixZQUFZLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsZUFBZSxDQUFDO1lBQ3hELGdCQUFnQixFQUFFLElBQUk7U0FDdkIsQ0FBQyxDQUFDLENBQUM7UUFFSixNQUFNLE1BQU0sR0FBRyx3Q0FBd0MsQ0FBQztRQUV4RCxzREFBc0Q7UUFDdEQsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyw2QkFBNkIsRUFBRTtZQUMvRCxRQUFRLEVBQUU7Z0JBQ1IsRUFBRSxNQUFNLEVBQUUsUUFBUSxDQUFDLCtCQUErQixFQUFFO2dCQUNwRDtvQkFDRSxNQUFNLEVBQUUsUUFBUTtvQkFDaEIsU0FBUyxFQUFFO3dCQUNUOzRCQUNFLGVBQWUsRUFBRTtnQ0FDZixjQUFjLEVBQUUsc0JBQXNCO2dDQUN0QyxTQUFTLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDLEVBQUU7Z0NBQzVDLFlBQVksRUFBRSxlQUFlO2dDQUM3QixXQUFXLEVBQUUsU0FBUztnQ0FDdEIsY0FBYyxFQUFFLCtCQUErQjs2QkFDaEQ7NEJBQ0QsZ0JBQWdCLEVBQUUsQ0FBQyxFQUFFLE1BQU0sRUFBRSxnQkFBZ0IsRUFBRSxDQUFDOzRCQUNoRCxNQUFNLEVBQUUsY0FBYzt5QkFDdkI7cUJBQ0Y7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosOENBQThDO1FBQzlDLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsa0JBQWtCLEVBQUU7WUFDcEQsY0FBYyxFQUFFO2dCQUNkLE9BQU8sRUFBRSxZQUFZO2dCQUNyQixTQUFTLEVBQUU7b0JBQ1Q7d0JBQ0UsUUFBUSxFQUFFOzRCQUNSLGVBQWU7NEJBQ2YsZUFBZTs0QkFDZixVQUFVO3lCQUNYO3dCQUNELFFBQVEsRUFBRSxPQUFPO3FCQUNsQjtvQkFDRDt3QkFDRSxRQUFRLEVBQUU7NEJBQ1IsYUFBYTs0QkFDYixpQkFBaUI7eUJBQ2xCO3dCQUNELFFBQVEsRUFBRSxPQUFPO3FCQUNsQjtvQkFDRDt3QkFDRSxNQUFNLEVBQUUsR0FBRzt3QkFDWCxNQUFNLEVBQUUsT0FBTzt3QkFDZixRQUFRLEVBQUUsR0FBRztxQkFDZDtpQkFDRjthQUNGO1lBQ0QsS0FBSyxFQUFFLENBQUMsRUFBRSxHQUFHLEVBQUUsTUFBTSxFQUFFLENBQUM7U0FDekIsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQscURBQXFELENBQUMsSUFBVTtRQUNoRSxRQUFRO1FBQ04sTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFXLEVBQUUsQ0FBQztRQUVoQyxPQUFPO1FBQ1AsS0FBSyxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsSUFBSSxTQUFTLENBQUMscUNBQXFDLENBQUM7WUFDOUUsVUFBVSxFQUFFLGNBQWM7WUFDMUIsU0FBUyxFQUFFLFNBQVM7WUFDcEIsWUFBWSxFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLGVBQWUsQ0FBQztZQUN4RCxjQUFjLEVBQUUscUJBQXFCO1lBQ3JDLGdCQUFnQixFQUFFLEtBQUs7U0FDeEIsQ0FBQyxDQUFDLENBQUM7UUFFSixvQ0FBb0M7UUFDcEMsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyw2QkFBNkIsRUFBRTtZQUMvRCxRQUFRLEVBQUU7Z0JBQ1IsRUFBRSxNQUFNLEVBQUUsUUFBUSxDQUFDLCtCQUErQixFQUFFO2dCQUNwRDtvQkFDRSxNQUFNLEVBQUUsUUFBUTtvQkFDaEIsU0FBUyxFQUFFO3dCQUNUOzRCQUNFLGlCQUFpQixFQUFFLENBQUMsRUFBRSxNQUFNLEVBQUUsK0JBQStCLEVBQUUsQ0FBQzs0QkFDaEUsTUFBTSxFQUFFLGNBQWM7eUJBQ3ZCO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx1Q0FBdUMsQ0FBQyxJQUFVO1FBQ2xELFFBQVE7UUFDTixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQVcsRUFBRSxDQUFDO1FBRWhDLE9BQU87UUFDUCxLQUFLLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxJQUFJLFNBQVMsQ0FBQyxxQ0FBcUMsQ0FBQztZQUM5RSxVQUFVLEVBQUUsY0FBYztZQUMxQixTQUFTLEVBQUUsU0FBUztZQUNwQixZQUFZLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsZUFBZSxDQUFDO1lBQ3hELGdCQUFnQixFQUFFLElBQUk7WUFDdEIsZ0JBQWdCLEVBQUUsS0FBSztTQUN4QixDQUFDLENBQUMsQ0FBQztRQUVKLG9DQUFvQztRQUNwQyxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLDZCQUE2QixFQUFFO1lBQy9ELFFBQVEsRUFBRTtnQkFDUixFQUFFLE1BQU0sRUFBRSxRQUFRLENBQUMsK0JBQStCLEVBQUU7Z0JBQ3BEO29CQUNFLE1BQU0sRUFBRSxRQUFRO29CQUNoQixTQUFTLEVBQUU7d0JBQ1Q7NEJBQ0UsZUFBZSxFQUFFO2dDQUNmLFlBQVksRUFBRSxvQkFBb0I7NkJBQ25DOzRCQUNELE1BQU0sRUFBRSxjQUFjO3lCQUN2QjtxQkFDRjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsK0NBQStDLENBQUMsSUFBVTtRQUMxRCxRQUFRO1FBQ04sTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFXLEVBQUUsQ0FBQztRQUVoQyxPQUFPO1FBQ1AsS0FBSyxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsSUFBSSxTQUFTLENBQUMscUNBQXFDLENBQUM7WUFDOUUsVUFBVSxFQUFFLGNBQWM7WUFDMUIsU0FBUyxFQUFFLFNBQVM7WUFDcEIsWUFBWSxFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLGVBQWUsQ0FBQztZQUN4RCxnQkFBZ0IsRUFBRSxLQUFLO1lBQ3ZCLGtCQUFrQixFQUFFO2dCQUNsQixRQUFRLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxjQUFjO2FBQ3BDO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyw2QkFBNkIsRUFBRTtZQUMvRCxRQUFRLEVBQUU7Z0JBQ1IsRUFBRSxNQUFNLEVBQUUsUUFBUSxDQUFDLCtCQUErQixFQUFFO2dCQUNwRDtvQkFDRSxNQUFNLEVBQUUsUUFBUTtvQkFDaEIsU0FBUyxFQUFFO3dCQUNUOzRCQUNFLGVBQWUsRUFBRTtnQ0FDZixvQkFBb0IsRUFBRTtvQ0FDcEIsVUFBVSxFQUFFLENBQUMsRUFBRSxFQUFFOzRDQUNmLGVBQWU7NENBQ2YsRUFBRSxZQUFZLEVBQUUsQ0FBQyw2QkFBNkIsRUFBRSxNQUFNLENBQUMsRUFBRTs0Q0FDekQsSUFBSTt5Q0FDTCxDQUFDO2lDQUNIOzZCQUNGOzRCQUNELE1BQU0sRUFBRSxjQUFjO3lCQUN2QjtxQkFDRjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsMkNBQTJDLENBQUMsSUFBVTtRQUNwRCxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQVcsRUFBRSxDQUFDO1FBRWhDLE1BQU0sWUFBWSxHQUFHLGNBQUksQ0FBQyxXQUFXLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSw0Q0FBNEMsQ0FBQyxDQUFDO1FBQzNHLE1BQU0sU0FBUyxHQUFHLElBQUksY0FBSSxDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUU7WUFDN0MsU0FBUyxFQUFFLElBQUksMEJBQWdCLENBQUMsY0FBYyxDQUFDO1NBQ2hELENBQUMsQ0FBQztRQUVILEtBQUssQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLElBQUksU0FBUyxDQUFDLG9DQUFvQyxDQUFDO1lBQzdFLFVBQVUsRUFBRSxvQkFBb0I7WUFDaEMsSUFBSSxFQUFFLFlBQVk7WUFDbEIsYUFBYSxFQUFFLFVBQVU7WUFDekIsU0FBUyxFQUFFLFlBQVk7U0FDeEIsQ0FBQyxDQUFDLENBQUM7UUFFSixLQUFLLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxJQUFJLFNBQVMsQ0FBQyxvQ0FBb0MsQ0FBQztZQUM3RSxVQUFVLEVBQUUsaUJBQWlCO1lBQzdCLElBQUksRUFBRSxTQUFTO1lBQ2YsYUFBYSxFQUFFLFVBQVU7WUFDekIsU0FBUyxFQUFFLFlBQVk7U0FDeEIsQ0FBQyxDQUFDLENBQUM7UUFFSixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLDZCQUE2QixFQUFFO1lBQy9ELFFBQVEsRUFBRTtnQkFDUjtvQkFDRSxNQUFNLEVBQUUsUUFBUTtpQkFDakI7Z0JBQ0Q7b0JBQ0UsTUFBTSxFQUFFLFFBQVE7b0JBQ2hCLFNBQVMsRUFBRTt3QkFDVDs0QkFDRSxNQUFNLEVBQUUsb0JBQW9COzRCQUM1QixTQUFTLEVBQUUsNENBQTRDO3lCQUN4RDt3QkFDRDs0QkFDRSxNQUFNLEVBQUUsaUJBQWlCOzRCQUN6QixTQUFTLEVBQUU7Z0NBQ1QsWUFBWSxFQUFFO29DQUNaLG1CQUFtQjtvQ0FDbkIsS0FBSztpQ0FDTjs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQseUNBQXlDLENBQUMsSUFBVTtRQUNwRCxRQUFRO1FBQ04sTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFXLEVBQUUsQ0FBQztRQUVoQyxPQUFPO1FBQ1AsS0FBSyxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsSUFBSSxTQUFTLENBQUMscUNBQXFDLENBQUM7WUFDOUUsVUFBVSxFQUFFLGNBQWM7WUFDMUIsU0FBUyxFQUFFLFNBQVM7WUFDcEIsWUFBWSxFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLGVBQWUsQ0FBQztZQUN4RCxnQkFBZ0IsRUFBRSxLQUFLO1lBQ3ZCLFlBQVksRUFBRTtnQkFDWiwrQ0FBMEIsQ0FBQyxTQUFTO2FBQ3JDO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixNQUFNLE1BQU0sR0FBRyx3Q0FBd0MsQ0FBQztRQUV4RCxzREFBc0Q7UUFDdEQsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyw2QkFBNkIsRUFBRTtZQUMvRCxRQUFRLEVBQUU7Z0JBQ1IsRUFBRSxNQUFNLEVBQUUsUUFBUSxDQUFDLCtCQUErQixFQUFFO2dCQUNwRDtvQkFDRSxNQUFNLEVBQUUsUUFBUTtvQkFDaEIsU0FBUyxFQUFFO3dCQUNUOzRCQUNFLGVBQWUsRUFBRTtnQ0FDZixjQUFjLEVBQUUsc0JBQXNCO2dDQUN0QyxTQUFTLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDLEVBQUU7Z0NBQzVDLFlBQVksRUFBRSxlQUFlO2dDQUM3QixXQUFXLEVBQUUsU0FBUztnQ0FDdEIsY0FBYyxFQUFFLCtCQUErQjs2QkFDaEQ7NEJBQ0QsZ0JBQWdCLEVBQUUsQ0FBQyxFQUFFLE1BQU0sRUFBRSxnQkFBZ0IsRUFBRSxDQUFDOzRCQUNoRCxNQUFNLEVBQUUsY0FBYzt5QkFDdkI7cUJBQ0Y7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDhDQUE4QyxDQUFDLElBQVU7UUFDekQsUUFBUTtRQUNOLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBVyxFQUFFLENBQUM7UUFFaEMsT0FBTztRQUNQLEtBQUssQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLElBQUksU0FBUyxDQUFDLHFDQUFxQyxDQUFDO1lBQzlFLFVBQVUsRUFBRSxjQUFjO1lBQzFCLFNBQVMsRUFBRSxTQUFTO1lBQ3BCLFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxlQUFlLENBQUM7WUFDeEQsZ0JBQWdCLEVBQUUsS0FBSztZQUN2QixZQUFZLEVBQUU7Z0JBQ1osK0NBQTBCLENBQUMsU0FBUztnQkFDcEMsK0NBQTBCLENBQUMsV0FBVzthQUN2QztTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosTUFBTSxNQUFNLEdBQUcsd0NBQXdDLENBQUM7UUFFeEQscUVBQXFFO1FBQ3JFLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsNkJBQTZCLEVBQUU7WUFDL0QsUUFBUSxFQUFFO2dCQUNSLEVBQUUsTUFBTSxFQUFFLFFBQVEsQ0FBQywrQkFBK0IsRUFBRTtnQkFDcEQ7b0JBQ0UsTUFBTSxFQUFFLFFBQVE7b0JBQ2hCLFNBQVMsRUFBRTt3QkFDVDs0QkFDRSxlQUFlLEVBQUU7Z0NBQ2YsY0FBYyxFQUFFLDZDQUE2QztnQ0FDN0QsU0FBUyxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUMsTUFBTSxFQUFFLEtBQUssQ0FBQyxFQUFFO2dDQUM1QyxZQUFZLEVBQUUsZUFBZTtnQ0FDN0IsV0FBVyxFQUFFLFNBQVM7Z0NBQ3RCLGNBQWMsRUFBRSwrQkFBK0I7NkJBQ2hEOzRCQUNELGdCQUFnQixFQUFFLENBQUMsRUFBRSxNQUFNLEVBQUUsZ0JBQWdCLEVBQUUsQ0FBQzs0QkFDaEQsTUFBTSxFQUFFLGNBQWM7eUJBQ3ZCO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw4Q0FBOEMsQ0FBQyxJQUFVO1FBQ3pELFFBQVE7UUFDTixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQVcsRUFBRSxDQUFDO1FBRWhDLE9BQU87UUFDUCxLQUFLLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxJQUFJLFNBQVMsQ0FBQyxxQ0FBcUMsQ0FBQztZQUM5RSxVQUFVLEVBQUUsY0FBYztZQUMxQixTQUFTLEVBQUUsU0FBUztZQUNwQixZQUFZLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsZUFBZSxDQUFDO1lBQ3hELGdCQUFnQixFQUFFLEtBQUs7WUFDdkIsWUFBWSxFQUFFO2dCQUNaLCtDQUEwQixDQUFDLElBQUk7YUFDaEM7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLE1BQU0sTUFBTSxHQUFHLHdDQUF3QyxDQUFDO1FBRXhELCtDQUErQztRQUMvQyxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLDZCQUE2QixFQUFFO1lBQy9ELFFBQVEsRUFBRTtnQkFDUixFQUFFLE1BQU0sRUFBRSxRQUFRLENBQUMsK0JBQStCLEVBQUU7Z0JBQ3BEO29CQUNFLE1BQU0sRUFBRSxRQUFRO29CQUNoQixTQUFTLEVBQUU7d0JBQ1Q7NEJBQ0UsZUFBZSxFQUFFO2dDQUNmLFNBQVMsRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFDLE1BQU0sRUFBRSxLQUFLLENBQUMsRUFBRTtnQ0FDNUMsWUFBWSxFQUFFLGVBQWU7Z0NBQzdCLFdBQVcsRUFBRSxTQUFTO2dDQUN0QixjQUFjLEVBQUUsK0JBQStCOzZCQUNoRDs0QkFDRCxnQkFBZ0IsRUFBRSxDQUFDLEVBQUUsTUFBTSxFQUFFLGdCQUFnQixFQUFFLENBQUM7NEJBQ2hELE1BQU0sRUFBRSxjQUFjO3lCQUN2QjtxQkFDRjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNEJBQTRCLEVBQUU7UUFDNUIsNERBQTRELENBQUMsSUFBVTtZQUNyRSxNQUFNLEdBQUcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLEVBQUUsQ0FBQztZQUUxQixNQUFNLGFBQWEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsR0FBRyxFQUFFLGVBQWUsRUFBRTtnQkFDeEQsR0FBRyxFQUFFO29CQUNILE9BQU8sRUFBRSxjQUFjO29CQUN2QixNQUFNLEVBQUUsV0FBVztpQkFDcEI7YUFDRixDQUFDLENBQUM7WUFFSCxNQUFNLFlBQVksR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLEVBQUUsQ0FBQztZQUNqRCxJQUFJLFlBQVksQ0FBQyxRQUFRLENBQUMsYUFBYSxFQUFFLFVBQVUsRUFBRTtnQkFDbkQsTUFBTSxFQUFFO29CQUNOO3dCQUNFLFNBQVMsRUFBRSxRQUFRO3dCQUNuQixPQUFPLEVBQUU7NEJBQ1AsSUFBSSxTQUFTLENBQUMsc0JBQXNCLENBQUM7Z0NBQ25DLFVBQVUsRUFBRSxZQUFZO2dDQUN4QixVQUFVLEVBQUUsVUFBVSxDQUFDLFVBQVUsQ0FBQyxrQkFBa0IsQ0FBQyxhQUFhLEVBQUUsTUFBTSxFQUFFLFVBQVUsQ0FBQztnQ0FDdkYsTUFBTSxFQUFFLFlBQVk7NkJBQ3JCLENBQUM7eUJBQ0g7cUJBQ0Y7b0JBQ0Q7d0JBQ0UsU0FBUyxFQUFFLFFBQVE7d0JBQ25CLE9BQU8sRUFBRTs0QkFDUCxJQUFJLFNBQVMsQ0FBQyxxQ0FBcUMsQ0FBQztnQ0FDbEQsVUFBVSxFQUFFLEtBQUs7Z0NBQ2pCLFNBQVMsRUFBRSxTQUFTO2dDQUNwQixnQkFBZ0IsRUFBRSxJQUFJO2dDQUN0QixZQUFZLEVBQUUsWUFBWSxDQUFDLE1BQU0sQ0FBQyxlQUFlLENBQUM7Z0NBQ2xELE9BQU8sRUFBRSxjQUFjOzZCQUN4QixDQUFDO3lCQUNIO3FCQUNGO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLGFBQWEsQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyw2QkFBNkIsRUFBRTtnQkFDdkUsUUFBUSxFQUFFO29CQUNSO3dCQUNFLE1BQU0sRUFBRSxRQUFRO3FCQUNqQjtvQkFDRDt3QkFDRSxNQUFNLEVBQUUsUUFBUTt3QkFDaEIsU0FBUyxFQUFFOzRCQUNUO2dDQUNFLE1BQU0sRUFBRSxLQUFLO2dDQUNiLFNBQVMsRUFBRTtvQ0FDVCxVQUFVLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxNQUFNLEVBQUUsRUFBRSxLQUFLLEVBQUUsZ0JBQWdCLEVBQUU7NENBQ25ELHVGQUF1RixDQUFDLENBQUM7aUNBQzVGO2dDQUNELGVBQWUsRUFBRTtvQ0FDZixTQUFTLEVBQUU7d0NBQ1QsVUFBVSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsTUFBTSxFQUFFLEVBQUUsS0FBSyxFQUFFLGdCQUFnQixFQUFFO2dEQUNuRCx1RkFBdUYsQ0FBQyxDQUFDO3FDQUM1RjtpQ0FDRjs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosMERBQTBEO1lBQzFELGVBQU0sQ0FBQyxhQUFhLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsdUJBQXVCLEVBQUU7Z0JBQ2pFLGdCQUFnQixFQUFFO29CQUNoQixXQUFXLEVBQUU7d0JBQ1g7NEJBQ0UsUUFBUSxFQUFFO2dDQUNSLGVBQWU7Z0NBQ2YsZUFBZTtnQ0FDZixVQUFVOzZCQUNYOzRCQUNELFFBQVEsRUFBRSxPQUFPOzRCQUNqQixXQUFXLEVBQUU7Z0NBQ1gsS0FBSyxFQUFFO29DQUNMLFVBQVUsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLE1BQU0sRUFBRSxFQUFFLEtBQUssRUFBRSxnQkFBZ0IsRUFBRTs0Q0FDbkQsdUZBQXVGLENBQUMsQ0FBQztpQ0FDNUY7NkJBQ0Y7eUJBQ0Y7d0JBQ0Q7NEJBQ0UsUUFBUSxFQUFFO2dDQUNSLGVBQWU7Z0NBQ2YsZUFBZTtnQ0FDZixVQUFVOzZCQUNYOzRCQUNELFFBQVEsRUFBRSxPQUFPOzRCQUNqQixXQUFXLEVBQUU7Z0NBQ1gsS0FBSyxFQUFFO29DQUNMLFVBQVUsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLE1BQU0sRUFBRSxFQUFFLEtBQUssRUFBRSxnQkFBZ0IsRUFBRTs0Q0FDbkQsdUZBQXVGLENBQUMsQ0FBQztpQ0FDNUY7NkJBQ0Y7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLE1BQU0sVUFBVSxHQUFHLEdBQUcsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLDBDQUEwQyxDQUFjLENBQUM7WUFDL0YsZUFBTSxDQUFDLFVBQVUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxnQkFBZ0IsRUFBRTtnQkFDdkQsVUFBVSxFQUFFLCtEQUErRDthQUM1RSxDQUFDLENBQUMsQ0FBQztZQUNKLGVBQU0sQ0FBQyxVQUFVLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsZ0JBQWdCLEVBQUU7Z0JBQ3ZELFVBQVUsRUFBRSwrREFBK0Q7YUFDNUUsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBQ0Y7Q0FDRixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgZXhwZWN0LCBoYXZlUmVzb3VyY2VMaWtlIH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCB7IENsb3VkRm9ybWF0aW9uQ2FwYWJpbGl0aWVzIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWNsb3VkZm9ybWF0aW9uJztcbmltcG9ydCAqIGFzIGNvZGVidWlsZCBmcm9tICdAYXdzLWNkay9hd3MtY29kZWJ1aWxkJztcbmltcG9ydCAqIGFzIGNvZGVjb21taXQgZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVjb21taXQnO1xuaW1wb3J0ICogYXMgY29kZXBpcGVsaW5lIGZyb20gJ0Bhd3MtY2RrL2F3cy1jb2RlcGlwZWxpbmUnO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50LCBSb2xlLCBTZXJ2aWNlUHJpbmNpcGFsIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0ICogYXMgY3BhY3Rpb25zIGZyb20gJy4uLy4uL2xpYic7XG5cbi8qIGVzbGludC1kaXNhYmxlIHF1b3RlLXByb3BzICovXG5cbmV4cG9ydCA9IHtcbiAgJ0NyZWF0ZUNoYW5nZVNldEFjdGlvbiBjYW4gYmUgdXNlZCB0byBtYWtlIGEgY2hhbmdlIHNldCBmcm9tIGEgQ29kZVBpcGVsaW5lJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICBjb25zdCBwaXBlbGluZSA9IG5ldyBjb2RlcGlwZWxpbmUuUGlwZWxpbmUoc3RhY2ssICdNYWdpY1BpcGVsaW5lJyk7XG5cbiAgICBjb25zdCBjaGFuZ2VTZXRFeGVjUm9sZSA9IG5ldyBSb2xlKHN0YWNrLCAnQ2hhbmdlU2V0Um9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IFNlcnZpY2VQcmluY2lwYWwoJ2Nsb3VkZm9ybWF0aW9uLmFtYXpvbmF3cy5jb20nKSxcbiAgICB9KTtcblxuICAgIC8qKiBTb3VyY2UhICovXG4gICAgY29uc3QgcmVwbyA9IG5ldyBjb2RlY29tbWl0LlJlcG9zaXRvcnkoc3RhY2ssICdNeVZlcnlJbXBvcnRhbnRSZXBvJywge1xuICAgICAgcmVwb3NpdG9yeU5hbWU6ICdteS12ZXJ5LWltcG9ydGFudC1yZXBvJyxcbiAgICB9KTtcblxuICAgIGNvbnN0IHNvdXJjZU91dHB1dCA9IG5ldyBjb2RlcGlwZWxpbmUuQXJ0aWZhY3QoJ1NvdXJjZUFydGlmYWN0Jyk7XG4gICAgY29uc3Qgc291cmNlID0gbmV3IGNwYWN0aW9ucy5Db2RlQ29tbWl0U291cmNlQWN0aW9uKHtcbiAgICAgIGFjdGlvbk5hbWU6ICdzb3VyY2UnLFxuICAgICAgb3V0cHV0OiBzb3VyY2VPdXRwdXQsXG4gICAgICByZXBvc2l0b3J5OiByZXBvLFxuICAgICAgdHJpZ2dlcjogY3BhY3Rpb25zLkNvZGVDb21taXRUcmlnZ2VyLlBPTEwsXG4gICAgfSk7XG4gICAgcGlwZWxpbmUuYWRkU3RhZ2Uoe1xuICAgICAgc3RhZ2VOYW1lOiAnc291cmNlJyxcbiAgICAgIGFjdGlvbnM6IFtzb3VyY2VdLFxuICAgIH0pO1xuXG4gICAgLyoqIEJ1aWxkISAqL1xuXG4gICAgY29uc3QgcHJvamVjdCA9IG5ldyBjb2RlYnVpbGQuUGlwZWxpbmVQcm9qZWN0KHN0YWNrLCAnTXlCdWlsZFByb2plY3QnKTtcblxuICAgIGNvbnN0IGJ1aWxkT3V0cHV0ID0gbmV3IGNvZGVwaXBlbGluZS5BcnRpZmFjdCgnT3V0cHV0WW8nKTtcbiAgICBjb25zdCBidWlsZEFjdGlvbiA9IG5ldyBjcGFjdGlvbnMuQ29kZUJ1aWxkQWN0aW9uKHtcbiAgICAgIGFjdGlvbk5hbWU6ICdidWlsZCcsXG4gICAgICBwcm9qZWN0LFxuICAgICAgaW5wdXQ6IHNvdXJjZU91dHB1dCxcbiAgICAgIG91dHB1dHM6IFtidWlsZE91dHB1dF0sXG4gICAgfSk7XG4gICAgcGlwZWxpbmUuYWRkU3RhZ2Uoe1xuICAgICAgc3RhZ2VOYW1lOiAnYnVpbGQnLFxuICAgICAgYWN0aW9uczogW2J1aWxkQWN0aW9uXSxcbiAgICB9KTtcblxuICAgIC8qKiBEZXBsb3khICovXG5cbiAgICAvLyBUbyBleGVjdXRlIGEgY2hhbmdlIHNldCAtIHllcywgeW91IHByb2JhYmx5IGRvIG5lZWQgKjoqIPCfpLfigI3imYDvuI9cbiAgICBjaGFuZ2VTZXRFeGVjUm9sZS5hZGRUb1BvbGljeShuZXcgUG9saWN5U3RhdGVtZW50KHsgcmVzb3VyY2VzOiBbJyonXSwgYWN0aW9uczogWycqJ10gfSkpO1xuXG4gICAgY29uc3Qgc3RhY2tOYW1lID0gJ0JyZWxhbmRzU3RhY2snO1xuICAgIGNvbnN0IGNoYW5nZVNldE5hbWUgPSAnTXlNYWdpY2FsQ2hhbmdlU2V0JztcbiAgICBwaXBlbGluZS5hZGRTdGFnZSh7XG4gICAgICBzdGFnZU5hbWU6ICdwcm9kJyxcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgbmV3IGNwYWN0aW9ucy5DbG91ZEZvcm1hdGlvbkNyZWF0ZVJlcGxhY2VDaGFuZ2VTZXRBY3Rpb24oe1xuICAgICAgICAgIGFjdGlvbk5hbWU6ICdCdWlsZENoYW5nZVNldFByb2QnLFxuICAgICAgICAgIHN0YWNrTmFtZSxcbiAgICAgICAgICBjaGFuZ2VTZXROYW1lLFxuICAgICAgICAgIGRlcGxveW1lbnRSb2xlOiBjaGFuZ2VTZXRFeGVjUm9sZSxcbiAgICAgICAgICB0ZW1wbGF0ZVBhdGg6IG5ldyBjb2RlcGlwZWxpbmUuQXJ0aWZhY3RQYXRoKGJ1aWxkT3V0cHV0LCAndGVtcGxhdGUueWFtbCcpLFxuICAgICAgICAgIHRlbXBsYXRlQ29uZmlndXJhdGlvbjogbmV3IGNvZGVwaXBlbGluZS5BcnRpZmFjdFBhdGgoYnVpbGRPdXRwdXQsICd0ZW1wbGF0ZUNvbmZpZy5qc29uJyksXG4gICAgICAgICAgYWRtaW5QZXJtaXNzaW9uczogZmFsc2UsXG4gICAgICAgIH0pLFxuICAgICAgICBuZXcgY3BhY3Rpb25zLkNsb3VkRm9ybWF0aW9uRXhlY3V0ZUNoYW5nZVNldEFjdGlvbih7XG4gICAgICAgICAgYWN0aW9uTmFtZTogJ0V4ZWN1dGVDaGFuZ2VTZXRQcm9kJyxcbiAgICAgICAgICBzdGFja05hbWUsXG4gICAgICAgICAgY2hhbmdlU2V0TmFtZSxcbiAgICAgICAgfSksXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNvZGVQaXBlbGluZTo6UGlwZWxpbmUnLCB7XG4gICAgICAnQXJ0aWZhY3RTdG9yZSc6IHtcbiAgICAgICAgJ0xvY2F0aW9uJzoge1xuICAgICAgICAgICdSZWYnOiAnTWFnaWNQaXBlbGluZUFydGlmYWN0c0J1Y2tldDIxMkZFN0JGJyxcbiAgICAgICAgfSxcbiAgICAgICAgJ1R5cGUnOiAnUzMnLFxuICAgICAgfSxcbiAgICAgICdSb2xlQXJuJzoge1xuICAgICAgICAnRm46OkdldEF0dCc6IFsnTWFnaWNQaXBlbGluZVJvbGVGQjJCRDZERScsXG4gICAgICAgICAgJ0FybiddLFxuICAgICAgfSxcbiAgICAgICdTdGFnZXMnOiBbe1xuICAgICAgICAnQWN0aW9ucyc6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICAnQWN0aW9uVHlwZUlkJzoge1xuICAgICAgICAgICAgICAnQ2F0ZWdvcnknOiAnU291cmNlJyxcbiAgICAgICAgICAgICAgJ093bmVyJzogJ0FXUycsXG4gICAgICAgICAgICAgICdQcm92aWRlcic6ICdDb2RlQ29tbWl0JyxcbiAgICAgICAgICAgICAgJ1ZlcnNpb24nOiAnMScsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgJ0NvbmZpZ3VyYXRpb24nOiB7XG4gICAgICAgICAgICAgICdSZXBvc2l0b3J5TmFtZSc6IHtcbiAgICAgICAgICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAgICAgICAgICdNeVZlcnlJbXBvcnRhbnRSZXBvMTFCQzNFQkQnLFxuICAgICAgICAgICAgICAgICAgJ05hbWUnLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICdCcmFuY2hOYW1lJzogJ21hc3RlcicsXG4gICAgICAgICAgICAgICdQb2xsRm9yU291cmNlQ2hhbmdlcyc6IHRydWUsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgJ05hbWUnOiAnc291cmNlJyxcbiAgICAgICAgICAgICdPdXRwdXRBcnRpZmFjdHMnOiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnTmFtZSc6ICdTb3VyY2VBcnRpZmFjdCcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgJ1J1bk9yZGVyJzogMSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgICAnTmFtZSc6ICdzb3VyY2UnLFxuICAgICAgfSxcbiAgICAgIHtcbiAgICAgICAgJ0FjdGlvbnMnOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgJ0FjdGlvblR5cGVJZCc6IHtcbiAgICAgICAgICAgICAgJ0NhdGVnb3J5JzogJ0J1aWxkJyxcbiAgICAgICAgICAgICAgJ093bmVyJzogJ0FXUycsXG4gICAgICAgICAgICAgICdQcm92aWRlcic6ICdDb2RlQnVpbGQnLFxuICAgICAgICAgICAgICAnVmVyc2lvbic6ICcxJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAnQ29uZmlndXJhdGlvbic6IHtcbiAgICAgICAgICAgICAgJ1Byb2plY3ROYW1lJzoge1xuICAgICAgICAgICAgICAgICdSZWYnOiAnTXlCdWlsZFByb2plY3QzMERCOUQ2RScsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgJ0lucHV0QXJ0aWZhY3RzJzogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgJ05hbWUnOiAnU291cmNlQXJ0aWZhY3QnLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICdOYW1lJzogJ2J1aWxkJyxcbiAgICAgICAgICAgICdPdXRwdXRBcnRpZmFjdHMnOiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnTmFtZSc6ICdPdXRwdXRZbycsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgJ1J1bk9yZGVyJzogMSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgICAnTmFtZSc6ICdidWlsZCcsXG4gICAgICB9LFxuICAgICAge1xuICAgICAgICAnQWN0aW9ucyc6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICAnQWN0aW9uVHlwZUlkJzoge1xuICAgICAgICAgICAgICAnQ2F0ZWdvcnknOiAnRGVwbG95JyxcbiAgICAgICAgICAgICAgJ093bmVyJzogJ0FXUycsXG4gICAgICAgICAgICAgICdQcm92aWRlcic6ICdDbG91ZEZvcm1hdGlvbicsXG4gICAgICAgICAgICAgICdWZXJzaW9uJzogJzEnLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICdDb25maWd1cmF0aW9uJzoge1xuICAgICAgICAgICAgICAnQWN0aW9uTW9kZSc6ICdDSEFOR0VfU0VUX1JFUExBQ0UnLFxuICAgICAgICAgICAgICAnQ2hhbmdlU2V0TmFtZSc6ICdNeU1hZ2ljYWxDaGFuZ2VTZXQnLFxuICAgICAgICAgICAgICAnUm9sZUFybic6IHtcbiAgICAgICAgICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAgICAgICAgICdDaGFuZ2VTZXRSb2xlMEJDRjk5RTYnLFxuICAgICAgICAgICAgICAgICAgJ0FybicsXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgJ1N0YWNrTmFtZSc6ICdCcmVsYW5kc1N0YWNrJyxcbiAgICAgICAgICAgICAgJ1RlbXBsYXRlUGF0aCc6ICdPdXRwdXRZbzo6dGVtcGxhdGUueWFtbCcsXG4gICAgICAgICAgICAgICdUZW1wbGF0ZUNvbmZpZ3VyYXRpb24nOiAnT3V0cHV0WW86OnRlbXBsYXRlQ29uZmlnLmpzb24nLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICdJbnB1dEFydGlmYWN0cyc6IFt7ICdOYW1lJzogJ091dHB1dFlvJyB9XSxcbiAgICAgICAgICAgICdOYW1lJzogJ0J1aWxkQ2hhbmdlU2V0UHJvZCcsXG4gICAgICAgICAgICAnUnVuT3JkZXInOiAxLFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgJ0FjdGlvblR5cGVJZCc6IHtcbiAgICAgICAgICAgICAgJ0NhdGVnb3J5JzogJ0RlcGxveScsXG4gICAgICAgICAgICAgICdPd25lcic6ICdBV1MnLFxuICAgICAgICAgICAgICAnUHJvdmlkZXInOiAnQ2xvdWRGb3JtYXRpb24nLFxuICAgICAgICAgICAgICAnVmVyc2lvbic6ICcxJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAnQ29uZmlndXJhdGlvbic6IHtcbiAgICAgICAgICAgICAgJ0FjdGlvbk1vZGUnOiAnQ0hBTkdFX1NFVF9FWEVDVVRFJyxcbiAgICAgICAgICAgICAgJ0NoYW5nZVNldE5hbWUnOiAnTXlNYWdpY2FsQ2hhbmdlU2V0JyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAnTmFtZSc6ICdFeGVjdXRlQ2hhbmdlU2V0UHJvZCcsXG4gICAgICAgICAgICAnUnVuT3JkZXInOiAxLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICAgICdOYW1lJzogJ3Byb2QnLFxuICAgICAgfV0sXG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG5cbiAgfSxcblxuICAnZnVsbFBlcm1pc3Npb25zIGxlYWRzIHRvIGFkbWluIHJvbGUgYW5kIGZ1bGwgSUFNIGNhcGFiaWxpdGllcyB3aXRoIHBpcGVsaW5lIGJ1Y2tldCtrZXkgcmVhZCBwZXJtaXNzaW9ucycodGVzdDogVGVzdCkge1xuICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFRlc3RGaXh0dXJlKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgc3RhY2suZGVwbG95U3RhZ2UuYWRkQWN0aW9uKG5ldyBjcGFjdGlvbnMuQ2xvdWRGb3JtYXRpb25DcmVhdGVVcGRhdGVTdGFja0FjdGlvbih7XG4gICAgICBhY3Rpb25OYW1lOiAnQ3JlYXRlVXBkYXRlJyxcbiAgICAgIHN0YWNrTmFtZTogJ015U3RhY2snLFxuICAgICAgdGVtcGxhdGVQYXRoOiBzdGFjay5zb3VyY2VPdXRwdXQuYXRQYXRoKCd0ZW1wbGF0ZS55YW1sJyksXG4gICAgICBhZG1pblBlcm1pc3Npb25zOiB0cnVlLFxuICAgIH0pKTtcblxuICAgIGNvbnN0IHJvbGVJZCA9ICdQaXBlbGluZURlcGxveUNyZWF0ZVVwZGF0ZVJvbGU1MTVDQjdENCc7XG5cbiAgICAvLyBUSEVOOiBBY3Rpb24gaW4gUGlwZWxpbmUgaGFzIG5hbWVkIElBTSBjYXBhYmlsaXRpZXNcbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6Q29kZVBpcGVsaW5lOjpQaXBlbGluZScsIHtcbiAgICAgICdTdGFnZXMnOiBbXG4gICAgICAgIHsgJ05hbWUnOiAnU291cmNlJyAvKiBkb24ndCBjYXJlIGFib3V0IHRoZSByZXN0ICovIH0sXG4gICAgICAgIHtcbiAgICAgICAgICAnTmFtZSc6ICdEZXBsb3knLFxuICAgICAgICAgICdBY3Rpb25zJzogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICAnQ29uZmlndXJhdGlvbic6IHtcbiAgICAgICAgICAgICAgICAnQ2FwYWJpbGl0aWVzJzogJ0NBUEFCSUxJVFlfTkFNRURfSUFNJyxcbiAgICAgICAgICAgICAgICAnUm9sZUFybic6IHsgJ0ZuOjpHZXRBdHQnOiBbcm9sZUlkLCAnQXJuJ10gfSxcbiAgICAgICAgICAgICAgICAnQWN0aW9uTW9kZSc6ICdDUkVBVEVfVVBEQVRFJyxcbiAgICAgICAgICAgICAgICAnU3RhY2tOYW1lJzogJ015U3RhY2snLFxuICAgICAgICAgICAgICAgICdUZW1wbGF0ZVBhdGgnOiAnU291cmNlQXJ0aWZhY3Q6OnRlbXBsYXRlLnlhbWwnLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAnSW5wdXRBcnRpZmFjdHMnOiBbeyAnTmFtZSc6ICdTb3VyY2VBcnRpZmFjdCcgfV0sXG4gICAgICAgICAgICAgICdOYW1lJzogJ0NyZWF0ZVVwZGF0ZScsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0pKTtcblxuICAgIC8vIFRIRU46IFJvbGUgaXMgY3JlYXRlZCB3aXRoIGZ1bGwgcGVybWlzc2lvbnNcbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICBWZXJzaW9uOiAnMjAxMi0xMC0xNycsXG4gICAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgICdBY3Rpb24nOiBbXG4gICAgICAgICAgICAgICdzMzpHZXRPYmplY3QqJyxcbiAgICAgICAgICAgICAgJ3MzOkdldEJ1Y2tldConLFxuICAgICAgICAgICAgICAnczM6TGlzdConLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICdFZmZlY3QnOiAnQWxsb3cnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgJ0FjdGlvbic6IFtcbiAgICAgICAgICAgICAgJ2ttczpEZWNyeXB0JyxcbiAgICAgICAgICAgICAgJ2ttczpEZXNjcmliZUtleScsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgJ0VmZmVjdCc6ICdBbGxvdycsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBBY3Rpb246ICcqJyxcbiAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICAgIFJlc291cmNlOiAnKicsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgICBSb2xlczogW3sgUmVmOiByb2xlSWQgfV0sXG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ291dHB1dEZpbGVOYW1lIGxlYWRzIHRvIGNyZWF0aW9uIG9mIG91dHB1dCBhcnRpZmFjdCcodGVzdDogVGVzdCkge1xuICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFRlc3RGaXh0dXJlKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgc3RhY2suZGVwbG95U3RhZ2UuYWRkQWN0aW9uKG5ldyBjcGFjdGlvbnMuQ2xvdWRGb3JtYXRpb25DcmVhdGVVcGRhdGVTdGFja0FjdGlvbih7XG4gICAgICBhY3Rpb25OYW1lOiAnQ3JlYXRlVXBkYXRlJyxcbiAgICAgIHN0YWNrTmFtZTogJ015U3RhY2snLFxuICAgICAgdGVtcGxhdGVQYXRoOiBzdGFjay5zb3VyY2VPdXRwdXQuYXRQYXRoKCd0ZW1wbGF0ZS55YW1sJyksXG4gICAgICBvdXRwdXRGaWxlTmFtZTogJ0NyZWF0ZVJlc3BvbnNlLmpzb24nLFxuICAgICAgYWRtaW5QZXJtaXNzaW9uczogZmFsc2UsXG4gICAgfSkpO1xuXG4gICAgLy8gVEhFTjogQWN0aW9uIGhhcyBvdXRwdXQgYXJ0aWZhY3RzXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNvZGVQaXBlbGluZTo6UGlwZWxpbmUnLCB7XG4gICAgICAnU3RhZ2VzJzogW1xuICAgICAgICB7ICdOYW1lJzogJ1NvdXJjZScgLyogZG9uJ3QgY2FyZSBhYm91dCB0aGUgcmVzdCAqLyB9LFxuICAgICAgICB7XG4gICAgICAgICAgJ05hbWUnOiAnRGVwbG95JyxcbiAgICAgICAgICAnQWN0aW9ucyc6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgJ091dHB1dEFydGlmYWN0cyc6IFt7ICdOYW1lJzogJ0NyZWF0ZVVwZGF0ZV9NeVN0YWNrX0FydGlmYWN0JyB9XSxcbiAgICAgICAgICAgICAgJ05hbWUnOiAnQ3JlYXRlVXBkYXRlJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3JlcGxhY2VPbkZhaWx1cmUgc3dpdGNoZXMgYWN0aW9uIHR5cGUnKHRlc3Q6IFRlc3QpIHtcbiAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBUZXN0Rml4dHVyZSgpO1xuXG4gICAgLy8gV0hFTlxuICAgIHN0YWNrLmRlcGxveVN0YWdlLmFkZEFjdGlvbihuZXcgY3BhY3Rpb25zLkNsb3VkRm9ybWF0aW9uQ3JlYXRlVXBkYXRlU3RhY2tBY3Rpb24oe1xuICAgICAgYWN0aW9uTmFtZTogJ0NyZWF0ZVVwZGF0ZScsXG4gICAgICBzdGFja05hbWU6ICdNeVN0YWNrJyxcbiAgICAgIHRlbXBsYXRlUGF0aDogc3RhY2suc291cmNlT3V0cHV0LmF0UGF0aCgndGVtcGxhdGUueWFtbCcpLFxuICAgICAgcmVwbGFjZU9uRmFpbHVyZTogdHJ1ZSxcbiAgICAgIGFkbWluUGVybWlzc2lvbnM6IGZhbHNlLFxuICAgIH0pKTtcblxuICAgIC8vIFRIRU46IEFjdGlvbiBoYXMgb3V0cHV0IGFydGlmYWN0c1xuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDb2RlUGlwZWxpbmU6OlBpcGVsaW5lJywge1xuICAgICAgJ1N0YWdlcyc6IFtcbiAgICAgICAgeyAnTmFtZSc6ICdTb3VyY2UnIC8qIGRvbid0IGNhcmUgYWJvdXQgdGhlIHJlc3QgKi8gfSxcbiAgICAgICAge1xuICAgICAgICAgICdOYW1lJzogJ0RlcGxveScsXG4gICAgICAgICAgJ0FjdGlvbnMnOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgICdDb25maWd1cmF0aW9uJzoge1xuICAgICAgICAgICAgICAgICdBY3Rpb25Nb2RlJzogJ1JFUExBQ0VfT05fRkFJTFVSRScsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICdOYW1lJzogJ0NyZWF0ZVVwZGF0ZScsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdwYXJhbWV0ZXJPdmVycmlkZXMgYXJlIHNlcmlhbGl6ZWQgYXMgYSBzdHJpbmcnKHRlc3Q6IFRlc3QpIHtcbiAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBUZXN0Rml4dHVyZSgpO1xuXG4gICAgLy8gV0hFTlxuICAgIHN0YWNrLmRlcGxveVN0YWdlLmFkZEFjdGlvbihuZXcgY3BhY3Rpb25zLkNsb3VkRm9ybWF0aW9uQ3JlYXRlVXBkYXRlU3RhY2tBY3Rpb24oe1xuICAgICAgYWN0aW9uTmFtZTogJ0NyZWF0ZVVwZGF0ZScsXG4gICAgICBzdGFja05hbWU6ICdNeVN0YWNrJyxcbiAgICAgIHRlbXBsYXRlUGF0aDogc3RhY2suc291cmNlT3V0cHV0LmF0UGF0aCgndGVtcGxhdGUueWFtbCcpLFxuICAgICAgYWRtaW5QZXJtaXNzaW9uczogZmFsc2UsXG4gICAgICBwYXJhbWV0ZXJPdmVycmlkZXM6IHtcbiAgICAgICAgUmVwb05hbWU6IHN0YWNrLnJlcG8ucmVwb3NpdG9yeU5hbWUsXG4gICAgICB9LFxuICAgIH0pKTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6Q29kZVBpcGVsaW5lOjpQaXBlbGluZScsIHtcbiAgICAgICdTdGFnZXMnOiBbXG4gICAgICAgIHsgJ05hbWUnOiAnU291cmNlJyAvKiBkb24ndCBjYXJlIGFib3V0IHRoZSByZXN0ICovIH0sXG4gICAgICAgIHtcbiAgICAgICAgICAnTmFtZSc6ICdEZXBsb3knLFxuICAgICAgICAgICdBY3Rpb25zJzogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICAnQ29uZmlndXJhdGlvbic6IHtcbiAgICAgICAgICAgICAgICAnUGFyYW1ldGVyT3ZlcnJpZGVzJzoge1xuICAgICAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogWycnLCBbXG4gICAgICAgICAgICAgICAgICAgICd7XCJSZXBvTmFtZVwiOlwiJyxcbiAgICAgICAgICAgICAgICAgICAgeyAnRm46OkdldEF0dCc6IFsnTXlWZXJ5SW1wb3J0YW50UmVwbzExQkMzRUJEJywgJ05hbWUnXSB9LFxuICAgICAgICAgICAgICAgICAgICAnXCJ9JyxcbiAgICAgICAgICAgICAgICAgIF1dLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICdOYW1lJzogJ0NyZWF0ZVVwZGF0ZScsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdBY3Rpb24gc2VydmljZSByb2xlIGlzIHBhc3NlZCB0byB0ZW1wbGF0ZScodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFRlc3RGaXh0dXJlKCk7XG5cbiAgICBjb25zdCBpbXBvcnRlZFJvbGUgPSBSb2xlLmZyb21Sb2xlQXJuKHN0YWNrLCAnSW1wb3J0ZWRSb2xlJywgJ2Fybjphd3M6aWFtOjowMDAwMDAwMDAwMDA6cm9sZS9hY3Rpb24tcm9sZScpO1xuICAgIGNvbnN0IGZyZXNoUm9sZSA9IG5ldyBSb2xlKHN0YWNrLCAnRnJlc2hSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgU2VydmljZVByaW5jaXBhbCgnbWFnaWNzZXJ2aWNlJyksXG4gICAgfSk7XG5cbiAgICBzdGFjay5kZXBsb3lTdGFnZS5hZGRBY3Rpb24obmV3IGNwYWN0aW9ucy5DbG91ZEZvcm1hdGlvbkV4ZWN1dGVDaGFuZ2VTZXRBY3Rpb24oe1xuICAgICAgYWN0aW9uTmFtZTogJ0ltcG9ydGVkUm9sZUFjdGlvbicsXG4gICAgICByb2xlOiBpbXBvcnRlZFJvbGUsXG4gICAgICBjaGFuZ2VTZXROYW1lOiAnbWFnaWNTZXQnLFxuICAgICAgc3RhY2tOYW1lOiAnbWFnaWNTdGFjaycsXG4gICAgfSkpO1xuXG4gICAgc3RhY2suZGVwbG95U3RhZ2UuYWRkQWN0aW9uKG5ldyBjcGFjdGlvbnMuQ2xvdWRGb3JtYXRpb25FeGVjdXRlQ2hhbmdlU2V0QWN0aW9uKHtcbiAgICAgIGFjdGlvbk5hbWU6ICdGcmVzaFJvbGVBY3Rpb24nLFxuICAgICAgcm9sZTogZnJlc2hSb2xlLFxuICAgICAgY2hhbmdlU2V0TmFtZTogJ21hZ2ljU2V0JyxcbiAgICAgIHN0YWNrTmFtZTogJ21hZ2ljU3RhY2snLFxuICAgIH0pKTtcblxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDb2RlUGlwZWxpbmU6OlBpcGVsaW5lJywge1xuICAgICAgJ1N0YWdlcyc6IFtcbiAgICAgICAge1xuICAgICAgICAgICdOYW1lJzogJ1NvdXJjZScsIC8qIGRvbid0IGNhcmUgYWJvdXQgdGhlIHJlc3QgKi9cbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgICdOYW1lJzogJ0RlcGxveScsXG4gICAgICAgICAgJ0FjdGlvbnMnOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgICdOYW1lJzogJ0ltcG9ydGVkUm9sZUFjdGlvbicsXG4gICAgICAgICAgICAgICdSb2xlQXJuJzogJ2Fybjphd3M6aWFtOjowMDAwMDAwMDAwMDA6cm9sZS9hY3Rpb24tcm9sZScsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICAnTmFtZSc6ICdGcmVzaFJvbGVBY3Rpb24nLFxuICAgICAgICAgICAgICAnUm9sZUFybic6IHtcbiAgICAgICAgICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAgICAgICAgICdGcmVzaFJvbGU0NzJGNkUxOCcsXG4gICAgICAgICAgICAgICAgICAnQXJuJyxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnU2luZ2xlIGNhcGFiaWxpdHkgaXMgcGFzc2VkIHRvIHRlbXBsYXRlJyh0ZXN0OiBUZXN0KSB7XG4gIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgVGVzdEZpeHR1cmUoKTtcblxuICAgIC8vIFdIRU5cbiAgICBzdGFjay5kZXBsb3lTdGFnZS5hZGRBY3Rpb24obmV3IGNwYWN0aW9ucy5DbG91ZEZvcm1hdGlvbkNyZWF0ZVVwZGF0ZVN0YWNrQWN0aW9uKHtcbiAgICAgIGFjdGlvbk5hbWU6ICdDcmVhdGVVcGRhdGUnLFxuICAgICAgc3RhY2tOYW1lOiAnTXlTdGFjaycsXG4gICAgICB0ZW1wbGF0ZVBhdGg6IHN0YWNrLnNvdXJjZU91dHB1dC5hdFBhdGgoJ3RlbXBsYXRlLnlhbWwnKSxcbiAgICAgIGFkbWluUGVybWlzc2lvbnM6IGZhbHNlLFxuICAgICAgY2FwYWJpbGl0aWVzOiBbXG4gICAgICAgIENsb3VkRm9ybWF0aW9uQ2FwYWJpbGl0aWVzLk5BTUVEX0lBTSxcbiAgICAgIF0sXG4gICAgfSkpO1xuXG4gICAgY29uc3Qgcm9sZUlkID0gJ1BpcGVsaW5lRGVwbG95Q3JlYXRlVXBkYXRlUm9sZTUxNUNCN0Q0JztcblxuICAgIC8vIFRIRU46IEFjdGlvbiBpbiBQaXBlbGluZSBoYXMgbmFtZWQgSUFNIGNhcGFiaWxpdGllc1xuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDb2RlUGlwZWxpbmU6OlBpcGVsaW5lJywge1xuICAgICAgJ1N0YWdlcyc6IFtcbiAgICAgICAgeyAnTmFtZSc6ICdTb3VyY2UnIC8qIGRvbid0IGNhcmUgYWJvdXQgdGhlIHJlc3QgKi8gfSxcbiAgICAgICAge1xuICAgICAgICAgICdOYW1lJzogJ0RlcGxveScsXG4gICAgICAgICAgJ0FjdGlvbnMnOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgICdDb25maWd1cmF0aW9uJzoge1xuICAgICAgICAgICAgICAgICdDYXBhYmlsaXRpZXMnOiAnQ0FQQUJJTElUWV9OQU1FRF9JQU0nLFxuICAgICAgICAgICAgICAgICdSb2xlQXJuJzogeyAnRm46OkdldEF0dCc6IFtyb2xlSWQsICdBcm4nXSB9LFxuICAgICAgICAgICAgICAgICdBY3Rpb25Nb2RlJzogJ0NSRUFURV9VUERBVEUnLFxuICAgICAgICAgICAgICAgICdTdGFja05hbWUnOiAnTXlTdGFjaycsXG4gICAgICAgICAgICAgICAgJ1RlbXBsYXRlUGF0aCc6ICdTb3VyY2VBcnRpZmFjdDo6dGVtcGxhdGUueWFtbCcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICdJbnB1dEFydGlmYWN0cyc6IFt7ICdOYW1lJzogJ1NvdXJjZUFydGlmYWN0JyB9XSxcbiAgICAgICAgICAgICAgJ05hbWUnOiAnQ3JlYXRlVXBkYXRlJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ011bHRpcGxlIGNhcGFiaWxpdGllcyBhcmUgcGFzc2VkIHRvIHRlbXBsYXRlJyh0ZXN0OiBUZXN0KSB7XG4gIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgVGVzdEZpeHR1cmUoKTtcblxuICAgIC8vIFdIRU5cbiAgICBzdGFjay5kZXBsb3lTdGFnZS5hZGRBY3Rpb24obmV3IGNwYWN0aW9ucy5DbG91ZEZvcm1hdGlvbkNyZWF0ZVVwZGF0ZVN0YWNrQWN0aW9uKHtcbiAgICAgIGFjdGlvbk5hbWU6ICdDcmVhdGVVcGRhdGUnLFxuICAgICAgc3RhY2tOYW1lOiAnTXlTdGFjaycsXG4gICAgICB0ZW1wbGF0ZVBhdGg6IHN0YWNrLnNvdXJjZU91dHB1dC5hdFBhdGgoJ3RlbXBsYXRlLnlhbWwnKSxcbiAgICAgIGFkbWluUGVybWlzc2lvbnM6IGZhbHNlLFxuICAgICAgY2FwYWJpbGl0aWVzOiBbXG4gICAgICAgIENsb3VkRm9ybWF0aW9uQ2FwYWJpbGl0aWVzLk5BTUVEX0lBTSxcbiAgICAgICAgQ2xvdWRGb3JtYXRpb25DYXBhYmlsaXRpZXMuQVVUT19FWFBBTkQsXG4gICAgICBdLFxuICAgIH0pKTtcblxuICAgIGNvbnN0IHJvbGVJZCA9ICdQaXBlbGluZURlcGxveUNyZWF0ZVVwZGF0ZVJvbGU1MTVDQjdENCc7XG5cbiAgICAvLyBUSEVOOiBBY3Rpb24gaW4gUGlwZWxpbmUgaGFzIG5hbWVkIElBTSBhbmQgQVVUT0VYUEFORCBjYXBhYmlsaXRpZXNcbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6Q29kZVBpcGVsaW5lOjpQaXBlbGluZScsIHtcbiAgICAgICdTdGFnZXMnOiBbXG4gICAgICAgIHsgJ05hbWUnOiAnU291cmNlJyAvKiBkb24ndCBjYXJlIGFib3V0IHRoZSByZXN0ICovIH0sXG4gICAgICAgIHtcbiAgICAgICAgICAnTmFtZSc6ICdEZXBsb3knLFxuICAgICAgICAgICdBY3Rpb25zJzogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICAnQ29uZmlndXJhdGlvbic6IHtcbiAgICAgICAgICAgICAgICAnQ2FwYWJpbGl0aWVzJzogJ0NBUEFCSUxJVFlfTkFNRURfSUFNLENBUEFCSUxJVFlfQVVUT19FWFBBTkQnLFxuICAgICAgICAgICAgICAgICdSb2xlQXJuJzogeyAnRm46OkdldEF0dCc6IFtyb2xlSWQsICdBcm4nXSB9LFxuICAgICAgICAgICAgICAgICdBY3Rpb25Nb2RlJzogJ0NSRUFURV9VUERBVEUnLFxuICAgICAgICAgICAgICAgICdTdGFja05hbWUnOiAnTXlTdGFjaycsXG4gICAgICAgICAgICAgICAgJ1RlbXBsYXRlUGF0aCc6ICdTb3VyY2VBcnRpZmFjdDo6dGVtcGxhdGUueWFtbCcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICdJbnB1dEFydGlmYWN0cyc6IFt7ICdOYW1lJzogJ1NvdXJjZUFydGlmYWN0JyB9XSxcbiAgICAgICAgICAgICAgJ05hbWUnOiAnQ3JlYXRlVXBkYXRlJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ0VtcHR5IGNhcGFiaWxpdGllcyBpcyBub3QgcGFzc2VkIHRvIHRlbXBsYXRlJyh0ZXN0OiBUZXN0KSB7XG4gIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgVGVzdEZpeHR1cmUoKTtcblxuICAgIC8vIFdIRU5cbiAgICBzdGFjay5kZXBsb3lTdGFnZS5hZGRBY3Rpb24obmV3IGNwYWN0aW9ucy5DbG91ZEZvcm1hdGlvbkNyZWF0ZVVwZGF0ZVN0YWNrQWN0aW9uKHtcbiAgICAgIGFjdGlvbk5hbWU6ICdDcmVhdGVVcGRhdGUnLFxuICAgICAgc3RhY2tOYW1lOiAnTXlTdGFjaycsXG4gICAgICB0ZW1wbGF0ZVBhdGg6IHN0YWNrLnNvdXJjZU91dHB1dC5hdFBhdGgoJ3RlbXBsYXRlLnlhbWwnKSxcbiAgICAgIGFkbWluUGVybWlzc2lvbnM6IGZhbHNlLFxuICAgICAgY2FwYWJpbGl0aWVzOiBbXG4gICAgICAgIENsb3VkRm9ybWF0aW9uQ2FwYWJpbGl0aWVzLk5PTkUsXG4gICAgICBdLFxuICAgIH0pKTtcblxuICAgIGNvbnN0IHJvbGVJZCA9ICdQaXBlbGluZURlcGxveUNyZWF0ZVVwZGF0ZVJvbGU1MTVDQjdENCc7XG5cbiAgICAvLyBUSEVOOiBBY3Rpb24gaW4gUGlwZWxpbmUgaGFzIG5vIGNhcGFiaWxpdGllc1xuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDb2RlUGlwZWxpbmU6OlBpcGVsaW5lJywge1xuICAgICAgJ1N0YWdlcyc6IFtcbiAgICAgICAgeyAnTmFtZSc6ICdTb3VyY2UnIC8qIGRvbid0IGNhcmUgYWJvdXQgdGhlIHJlc3QgKi8gfSxcbiAgICAgICAge1xuICAgICAgICAgICdOYW1lJzogJ0RlcGxveScsXG4gICAgICAgICAgJ0FjdGlvbnMnOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgICdDb25maWd1cmF0aW9uJzoge1xuICAgICAgICAgICAgICAgICdSb2xlQXJuJzogeyAnRm46OkdldEF0dCc6IFtyb2xlSWQsICdBcm4nXSB9LFxuICAgICAgICAgICAgICAgICdBY3Rpb25Nb2RlJzogJ0NSRUFURV9VUERBVEUnLFxuICAgICAgICAgICAgICAgICdTdGFja05hbWUnOiAnTXlTdGFjaycsXG4gICAgICAgICAgICAgICAgJ1RlbXBsYXRlUGF0aCc6ICdTb3VyY2VBcnRpZmFjdDo6dGVtcGxhdGUueWFtbCcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICdJbnB1dEFydGlmYWN0cyc6IFt7ICdOYW1lJzogJ1NvdXJjZUFydGlmYWN0JyB9XSxcbiAgICAgICAgICAgICAgJ05hbWUnOiAnQ3JlYXRlVXBkYXRlJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2Nyb3NzLWFjY291bnQgQ0ZOIFBpcGVsaW5lJzoge1xuICAgICdjb3JyZWN0bHkgY3JlYXRlcyB0aGUgZGVwbG95bWVudCBSb2xlIGluIHRoZSBvdGhlciBhY2NvdW50Jyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBhcHAgPSBuZXcgY2RrLkFwcCgpO1xuXG4gICAgICBjb25zdCBwaXBlbGluZVN0YWNrID0gbmV3IGNkay5TdGFjayhhcHAsICdQaXBlbGluZVN0YWNrJywge1xuICAgICAgICBlbnY6IHtcbiAgICAgICAgICBhY2NvdW50OiAnMjM0NTY3ODkwMTIzJyxcbiAgICAgICAgICByZWdpb246ICd1cy13ZXN0LTInLFxuICAgICAgICB9LFxuICAgICAgfSk7XG5cbiAgICAgIGNvbnN0IHNvdXJjZU91dHB1dCA9IG5ldyBjb2RlcGlwZWxpbmUuQXJ0aWZhY3QoKTtcbiAgICAgIG5ldyBjb2RlcGlwZWxpbmUuUGlwZWxpbmUocGlwZWxpbmVTdGFjaywgJ1BpcGVsaW5lJywge1xuICAgICAgICBzdGFnZXM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBzdGFnZU5hbWU6ICdTb3VyY2UnLFxuICAgICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgICBuZXcgY3BhY3Rpb25zLkNvZGVDb21taXRTb3VyY2VBY3Rpb24oe1xuICAgICAgICAgICAgICAgIGFjdGlvbk5hbWU6ICdDb2RlQ29tbWl0JyxcbiAgICAgICAgICAgICAgICByZXBvc2l0b3J5OiBjb2RlY29tbWl0LlJlcG9zaXRvcnkuZnJvbVJlcG9zaXRvcnlOYW1lKHBpcGVsaW5lU3RhY2ssICdSZXBvJywgJ1JlcG9OYW1lJyksXG4gICAgICAgICAgICAgICAgb3V0cHV0OiBzb3VyY2VPdXRwdXQsXG4gICAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIHN0YWdlTmFtZTogJ0RlcGxveScsXG4gICAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAgIG5ldyBjcGFjdGlvbnMuQ2xvdWRGb3JtYXRpb25DcmVhdGVVcGRhdGVTdGFja0FjdGlvbih7XG4gICAgICAgICAgICAgICAgYWN0aW9uTmFtZTogJ0NGTicsXG4gICAgICAgICAgICAgICAgc3RhY2tOYW1lOiAnTXlTdGFjaycsXG4gICAgICAgICAgICAgICAgYWRtaW5QZXJtaXNzaW9uczogdHJ1ZSxcbiAgICAgICAgICAgICAgICB0ZW1wbGF0ZVBhdGg6IHNvdXJjZU91dHB1dC5hdFBhdGgoJ3RlbXBsYXRlLnlhbWwnKSxcbiAgICAgICAgICAgICAgICBhY2NvdW50OiAnMTIzNDU2Nzg5MDEyJyxcbiAgICAgICAgICAgICAgfSksXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHBpcGVsaW5lU3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6Q29kZVBpcGVsaW5lOjpQaXBlbGluZScsIHtcbiAgICAgICAgJ1N0YWdlcyc6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICAnTmFtZSc6ICdTb3VyY2UnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgJ05hbWUnOiAnRGVwbG95JyxcbiAgICAgICAgICAgICdBY3Rpb25zJzogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgJ05hbWUnOiAnQ0ZOJyxcbiAgICAgICAgICAgICAgICAnUm9sZUFybic6IHtcbiAgICAgICAgICAgICAgICAgICdGbjo6Sm9pbic6IFsnJywgWydhcm46JywgeyAnUmVmJzogJ0FXUzo6UGFydGl0aW9uJyB9LFxuICAgICAgICAgICAgICAgICAgICAnOmlhbTo6MTIzNDU2Nzg5MDEyOnJvbGUvcGlwZWxpbmVzdGFjay1zdXBwb3J0LTEyM2xveWNmbmFjdGlvbnJvbGU1NmFmNjRhZjM1OTBmMzExYmM1MCddXSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICdDb25maWd1cmF0aW9uJzoge1xuICAgICAgICAgICAgICAgICAgJ1JvbGVBcm4nOiB7XG4gICAgICAgICAgICAgICAgICAgICdGbjo6Sm9pbic6IFsnJywgWydhcm46JywgeyAnUmVmJzogJ0FXUzo6UGFydGl0aW9uJyB9LFxuICAgICAgICAgICAgICAgICAgICAgICc6aWFtOjoxMjM0NTY3ODkwMTI6cm9sZS9waXBlbGluZXN0YWNrLXN1cHBvcnQtMTIzZm5kZXBsb3ltZW50cm9sZTQ2NjhkOWI1YTMwY2UzZGM0NTA4J11dLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSkpO1xuXG4gICAgICAvLyB0aGUgcGlwZWxpbmUncyBCdWNrZXRQb2xpY3kgc2hvdWxkIHRydXN0IGJvdGggQ0ZOIHJvbGVzXG4gICAgICBleHBlY3QocGlwZWxpbmVTdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpTMzo6QnVja2V0UG9saWN5Jywge1xuICAgICAgICAnUG9saWN5RG9jdW1lbnQnOiB7XG4gICAgICAgICAgJ1N0YXRlbWVudCc6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgJ0FjdGlvbic6IFtcbiAgICAgICAgICAgICAgICAnczM6R2V0T2JqZWN0KicsXG4gICAgICAgICAgICAgICAgJ3MzOkdldEJ1Y2tldConLFxuICAgICAgICAgICAgICAgICdzMzpMaXN0KicsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICdFZmZlY3QnOiAnQWxsb3cnLFxuICAgICAgICAgICAgICAnUHJpbmNpcGFsJzoge1xuICAgICAgICAgICAgICAgICdBV1MnOiB7XG4gICAgICAgICAgICAgICAgICAnRm46OkpvaW4nOiBbJycsIFsnYXJuOicsIHsgJ1JlZic6ICdBV1M6OlBhcnRpdGlvbicgfSxcbiAgICAgICAgICAgICAgICAgICAgJzppYW06OjEyMzQ1Njc4OTAxMjpyb2xlL3BpcGVsaW5lc3RhY2stc3VwcG9ydC0xMjNmbmRlcGxveW1lbnRyb2xlNDY2OGQ5YjVhMzBjZTNkYzQ1MDgnXV0sXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgICdBY3Rpb24nOiBbXG4gICAgICAgICAgICAgICAgJ3MzOkdldE9iamVjdConLFxuICAgICAgICAgICAgICAgICdzMzpHZXRCdWNrZXQqJyxcbiAgICAgICAgICAgICAgICAnczM6TGlzdConLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAnRWZmZWN0JzogJ0FsbG93JyxcbiAgICAgICAgICAgICAgJ1ByaW5jaXBhbCc6IHtcbiAgICAgICAgICAgICAgICAnQVdTJzoge1xuICAgICAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogWycnLCBbJ2FybjonLCB7ICdSZWYnOiAnQVdTOjpQYXJ0aXRpb24nIH0sXG4gICAgICAgICAgICAgICAgICAgICc6aWFtOjoxMjM0NTY3ODkwMTI6cm9sZS9waXBlbGluZXN0YWNrLXN1cHBvcnQtMTIzbG95Y2ZuYWN0aW9ucm9sZTU2YWY2NGFmMzU5MGYzMTFiYzUwJ11dLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICB9KSk7XG5cbiAgICAgIGNvbnN0IG90aGVyU3RhY2sgPSBhcHAubm9kZS5maW5kQ2hpbGQoJ2Nyb3NzLWFjY291bnQtc3VwcG9ydC1zdGFjay0xMjM0NTY3ODkwMTInKSBhcyBjZGsuU3RhY2s7XG4gICAgICBleHBlY3Qob3RoZXJTdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpJQU06OlJvbGUnLCB7XG4gICAgICAgICdSb2xlTmFtZSc6ICdwaXBlbGluZXN0YWNrLXN1cHBvcnQtMTIzbG95Y2ZuYWN0aW9ucm9sZTU2YWY2NGFmMzU5MGYzMTFiYzUwJyxcbiAgICAgIH0pKTtcbiAgICAgIGV4cGVjdChvdGhlclN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OklBTTo6Um9sZScsIHtcbiAgICAgICAgJ1JvbGVOYW1lJzogJ3BpcGVsaW5lc3RhY2stc3VwcG9ydC0xMjNmbmRlcGxveW1lbnRyb2xlNDY2OGQ5YjVhMzBjZTNkYzQ1MDgnLFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICB9LFxufTtcblxuLyoqXG4gKiBBIHRlc3Qgc3RhY2sgd2l0aCBhIGhhbGYtcHJlcGFyZWQgcGlwZWxpbmUgcmVhZHkgdG8gYWRkIENsb3VkRm9ybWF0aW9uIGFjdGlvbnMgdG9cbiAqL1xuY2xhc3MgVGVzdEZpeHR1cmUgZXh0ZW5kcyBjZGsuU3RhY2sge1xuICBwdWJsaWMgcmVhZG9ubHkgcGlwZWxpbmU6IGNvZGVwaXBlbGluZS5QaXBlbGluZTtcbiAgcHVibGljIHJlYWRvbmx5IHNvdXJjZVN0YWdlOiBjb2RlcGlwZWxpbmUuSVN0YWdlO1xuICBwdWJsaWMgcmVhZG9ubHkgZGVwbG95U3RhZ2U6IGNvZGVwaXBlbGluZS5JU3RhZ2U7XG4gIHB1YmxpYyByZWFkb25seSByZXBvOiBjb2RlY29tbWl0LlJlcG9zaXRvcnk7XG4gIHB1YmxpYyByZWFkb25seSBzb3VyY2VPdXRwdXQ6IGNvZGVwaXBlbGluZS5BcnRpZmFjdDtcblxuICBjb25zdHJ1Y3RvcigpIHtcbiAgICBzdXBlcigpO1xuXG4gICAgdGhpcy5waXBlbGluZSA9IG5ldyBjb2RlcGlwZWxpbmUuUGlwZWxpbmUodGhpcywgJ1BpcGVsaW5lJyk7XG4gICAgdGhpcy5zb3VyY2VTdGFnZSA9IHRoaXMucGlwZWxpbmUuYWRkU3RhZ2UoeyBzdGFnZU5hbWU6ICdTb3VyY2UnIH0pO1xuICAgIHRoaXMuZGVwbG95U3RhZ2UgPSB0aGlzLnBpcGVsaW5lLmFkZFN0YWdlKHsgc3RhZ2VOYW1lOiAnRGVwbG95JyB9KTtcbiAgICB0aGlzLnJlcG8gPSBuZXcgY29kZWNvbW1pdC5SZXBvc2l0b3J5KHRoaXMsICdNeVZlcnlJbXBvcnRhbnRSZXBvJywge1xuICAgICAgcmVwb3NpdG9yeU5hbWU6ICdteS12ZXJ5LWltcG9ydGFudC1yZXBvJyxcbiAgICB9KTtcbiAgICB0aGlzLnNvdXJjZU91dHB1dCA9IG5ldyBjb2RlcGlwZWxpbmUuQXJ0aWZhY3QoJ1NvdXJjZUFydGlmYWN0Jyk7XG4gICAgY29uc3Qgc291cmNlID0gbmV3IGNwYWN0aW9ucy5Db2RlQ29tbWl0U291cmNlQWN0aW9uKHtcbiAgICAgIGFjdGlvbk5hbWU6ICdTb3VyY2UnLFxuICAgICAgb3V0cHV0OiB0aGlzLnNvdXJjZU91dHB1dCxcbiAgICAgIHJlcG9zaXRvcnk6IHRoaXMucmVwbyxcbiAgICB9KTtcbiAgICB0aGlzLnNvdXJjZVN0YWdlLmFkZEFjdGlvbihzb3VyY2UpO1xuICB9XG59XG4iXX0=