"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LowCostECS = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const lib = require("aws-cdk-lib");
const ec2 = require("aws-cdk-lib/aws-ec2");
const ecs = require("aws-cdk-lib/aws-ecs");
const aws_efs_1 = require("aws-cdk-lib/aws-efs");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const route53 = require("aws-cdk-lib/aws-route53");
const aws_sns_1 = require("aws-cdk-lib/aws-sns");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const sfn_tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
const constructs_1 = require("constructs");
class LowCostECS extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.vpc =
            props.vpc ??
                new ec2.Vpc(scope, 'Vpc', {
                    natGateways: 0,
                    subnetConfiguration: [
                        {
                            name: 'PublicSubnet',
                            subnetType: ec2.SubnetType.PUBLIC,
                        },
                    ],
                });
        this.cluster = new ecs.Cluster(scope, 'Cluster', {
            vpc: this.vpc,
            containerInsights: props.containerInsights,
        });
        this.hostAutoScalingGroup = this.cluster.addCapacity('HostInstanceCapacity', {
            machineImage: ecs.EcsOptimizedImage.amazonLinux2(ecs.AmiHardwareType.STANDARD, {
                cachedInContext: true,
            }),
            instanceType: new ec2.InstanceType(props.hostInstanceType ?? 't2.micro'),
            spotPrice: props.hostInstanceSpotPrice,
            vpcSubnets: { subnetType: ec2.SubnetType.PUBLIC },
            associatePublicIpAddress: true,
            minCapacity: 1,
            maxCapacity: 1,
        });
        if (props.securityGroup) {
            this.hostAutoScalingGroup.node.tryRemoveChild('InstanceSecurityGroup');
            this.hostAutoScalingGroup.addSecurityGroup(props.securityGroup);
        }
        else {
            this.hostAutoScalingGroup.connections.allowFromAnyIpv4(ec2.Port.tcp(80));
            this.hostAutoScalingGroup.connections.allowFromAnyIpv4(ec2.Port.tcp(443));
            this.hostAutoScalingGroup.connections.allowFrom(ec2.Peer.anyIpv6(), ec2.Port.tcp(80));
            this.hostAutoScalingGroup.connections.allowFrom(ec2.Peer.anyIpv6(), ec2.Port.tcp(443));
        }
        /**
         * Add managed policy to allow ssh through ssm manager
         */
        this.hostAutoScalingGroup.role.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        /**
         * Add policy to associate elastic ip on startup
         */
        this.hostAutoScalingGroup.role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['ec2:DescribeAddresses', 'ec2:AssociateAddress'],
            resources: ['*'],
        }));
        const hostInstanceIp = new ec2.CfnEIP(this, 'HostInstanceIp');
        const tagUniqueId = lib.Names.uniqueId(hostInstanceIp);
        hostInstanceIp.tags.setTag('Name', tagUniqueId);
        const awsCliTag = props.awsCliDockerTag ?? 'latest';
        this.hostAutoScalingGroup.addUserData('INSTANCE_ID=$(curl --silent http://169.254.169.254/latest/meta-data/instance-id)', `ALLOCATION_ID=$(docker run --net=host amazon/aws-cli:${awsCliTag} ec2 describe-addresses --region ${this.hostAutoScalingGroup.env.region} --filter Name=tag:Name,Values=${tagUniqueId} --query 'Addresses[].AllocationId' --output text | head)`, `docker run --net=host amazon/aws-cli:${awsCliTag} ec2 associate-address --region ${this.hostAutoScalingGroup.env.region} --instance-id "$INSTANCE_ID" --allocation-id "$ALLOCATION_ID" --allow-reassociation`);
        this.certFileSystem = new aws_efs_1.FileSystem(this, 'FileSystem', {
            vpc: this.vpc,
            encrypted: true,
            securityGroup: new ec2.SecurityGroup(this, 'FileSystemSecurityGroup', {
                vpc: this.vpc,
                allowAllOutbound: false,
            }),
            removalPolicy: props.removalPolicy ?? lib.RemovalPolicy.DESTROY,
        });
        this.certFileSystem.connections.allowDefaultPortTo(this.hostAutoScalingGroup);
        this.certFileSystem.connections.allowDefaultPortFrom(this.hostAutoScalingGroup);
        /**
         * ARecord to Elastic ip
         */
        const hostedZone = route53.HostedZone.fromLookup(this, 'HostedZone', {
            domainName: props.hostedZoneDomain,
        });
        const records = props.recordDomainNames ?? [hostedZone.zoneName];
        records.forEach((record) => new route53.ARecord(this, `ARecord${record}`, {
            zone: hostedZone,
            recordName: record,
            target: route53.RecordTarget.fromIpAddresses(hostInstanceIp.ref),
        }));
        /**
         * Certbot Task Definition
         * Mounts generated certificate to EFS
         */
        const logGroup = props.logGroup ??
            new aws_logs_1.LogGroup(this, 'LogGroup', {
                retention: aws_logs_1.RetentionDays.TWO_YEARS,
                removalPolicy: props.removalPolicy ?? lib.RemovalPolicy.DESTROY,
            });
        const certbotTaskDefinition = new ecs.Ec2TaskDefinition(this, 'CertbotTaskDefinition');
        certbotTaskDefinition.addToTaskRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['route53:ListHostedZones', 'route53:GetChange'],
            resources: ['*'],
        }));
        certbotTaskDefinition.addToTaskRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['route53:ChangeResourceRecordSets'],
            resources: [hostedZone.hostedZoneArn],
        }));
        const certbotTag = props.certbotDockerTag ?? 'v1.29.0';
        const certbotContainer = certbotTaskDefinition.addContainer('CertbotContainer', {
            image: ecs.ContainerImage.fromRegistry(`certbot/dns-route53:${certbotTag}`),
            containerName: 'certbot',
            memoryReservationMiB: 64,
            command: [
                'certonly',
                '--verbose',
                '--preferred-challenges=dns-01',
                '--dns-route53',
                '--dns-route53-propagation-seconds=300',
                '--non-interactive',
                '--agree-tos',
                '--expand',
                '-m',
                props.email,
                '--cert-name',
                records[0],
                ...records.flatMap((domain) => ['-d', domain]),
            ],
            logging: ecs.LogDriver.awsLogs({
                logGroup,
                streamPrefix: certbotTag,
            }),
        });
        this.certFileSystem.grant(certbotTaskDefinition.taskRole, 'elasticfilesystem:ClientWrite');
        certbotTaskDefinition.addVolume({
            name: 'certVolume',
            efsVolumeConfiguration: {
                fileSystemId: this.certFileSystem.fileSystemId,
            },
        });
        certbotContainer.addMountPoints({
            sourceVolume: 'certVolume',
            containerPath: '/etc/letsencrypt',
            readOnly: false,
        });
        /**
         * Schedule Certbot certificate create/renew on Step Functions
         * Sends email notification on certbot failure
         */
        this.topic = new aws_sns_1.Topic(this, 'Topic');
        new aws_sns_1.Subscription(this, 'EmailSubscription', {
            topic: this.topic,
            protocol: aws_sns_1.SubscriptionProtocol.EMAIL,
            endpoint: props.email,
        });
        const certbotRunTask = new sfn_tasks.EcsRunTask(this, 'CreateCertificate', {
            cluster: this.cluster,
            taskDefinition: certbotTaskDefinition,
            launchTarget: new sfn_tasks.EcsEc2LaunchTarget(),
            integrationPattern: sfn.IntegrationPattern.RUN_JOB,
        });
        certbotRunTask.addCatch(new sfn_tasks.SnsPublish(this, 'SendEmailOnFailure', {
            topic: this.topic,
            message: sfn.TaskInput.fromJsonPathAt('$'),
        }).next(new sfn.Fail(this, 'Fail')));
        certbotRunTask.addRetry({
            interval: lib.Duration.seconds(20),
        });
        const certbotStateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: certbotRunTask,
        });
        new aws_events_1.Rule(this, 'CertbotScheduleRule', {
            schedule: aws_events_1.Schedule.rate(lib.Duration.days(props.certbotScheduleInterval ?? 60)),
            targets: [new aws_events_targets_1.SfnStateMachine(certbotStateMachine)],
        });
        /**
         * Server ECS task
         */
        this.serverTaskDefinition = this.createTaskDefinition(props.serverTaskDefinition ?? this.sampleTaskDefinition(records, logGroup));
        if (!this.serverTaskDefinition.defaultContainer) {
            throw new Error('defaultContainer is required for serverTaskDefinition. Add at least one essential container.');
        }
        this.certFileSystem.grant(this.serverTaskDefinition.taskRole, 'elasticfilesystem:ClientMount');
        this.serverTaskDefinition.addVolume({
            name: 'certVolume',
            efsVolumeConfiguration: {
                fileSystemId: this.certFileSystem.fileSystemId,
            },
        });
        this.serverTaskDefinition.defaultContainer.addMountPoints({
            sourceVolume: 'certVolume',
            containerPath: '/etc/letsencrypt',
            readOnly: true,
        });
        /**
         * AWS cli container to execute certbot sfn before the default container startup.
         */
        this.serverTaskDefinition.defaultContainer.addContainerDependencies({
            container: this.serverTaskDefinition.addContainer('AWSCliContainer', {
                image: ecs.ContainerImage.fromRegistry(`amazon/aws-cli:${awsCliTag}`),
                containerName: 'aws-cli',
                memoryReservationMiB: 64,
                entryPoint: ['/bin/bash', '-c'],
                command: [
                    `set -eux
          aws configure set region ${certbotStateMachine.env.region} && \\
          aws configure set output text && \\
          EXECUTION_ARN=$(aws stepfunctions start-execution --state-machine-arn ${certbotStateMachine.stateMachineArn} --query executionArn) && \\
          until [ $(aws stepfunctions describe-execution --execution-arn "$EXECUTION_ARN" --query status) != RUNNING ];
          do
            echo "Waiting for $EXECUTION_ARN"
            sleep 10
          done`,
                ],
                essential: false,
                logging: ecs.LogDriver.awsLogs({
                    logGroup: logGroup,
                    streamPrefix: awsCliTag,
                }),
            }),
            condition: ecs.ContainerDependencyCondition.COMPLETE,
        });
        certbotStateMachine.grantExecution(this.serverTaskDefinition.taskRole, 'states:DescribeExecution');
        certbotStateMachine.grantStartExecution(this.serverTaskDefinition.taskRole);
        this.service = new ecs.Ec2Service(this, 'Service', {
            cluster: this.cluster,
            taskDefinition: this.serverTaskDefinition,
            desiredCount: 1,
            minHealthyPercent: 0,
            maxHealthyPercent: 100,
            circuitBreaker: {
                rollback: true,
            },
            enableExecuteCommand: true,
        });
        new lib.CfnOutput(this, 'PublicIpAddress', { value: hostInstanceIp.ref });
        new lib.CfnOutput(this, 'CertbotStateMachineName', {
            value: certbotStateMachine.stateMachineName,
        });
        new lib.CfnOutput(this, 'ClusterName', { value: this.cluster.clusterName });
        new lib.CfnOutput(this, 'ServiceName', { value: this.service.serviceName });
    }
    createTaskDefinition(taskDefinitionOptions) {
        const serverTaskDefinition = new ecs.Ec2TaskDefinition(this, 'ServerTaskDefinition', taskDefinitionOptions.taskDefinition);
        taskDefinitionOptions.containers.forEach((containerDefinition, index) => {
            serverTaskDefinition.addContainer(containerDefinition.containerName ?? `container${index}`, containerDefinition);
        });
        taskDefinitionOptions.volumes?.forEach((volume) => serverTaskDefinition.addVolume(volume));
        return serverTaskDefinition;
    }
    sampleTaskDefinition(records, logGroup) {
        return {
            containers: [
                {
                    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../examples/containers/nginx')),
                    containerName: 'nginx',
                    memoryReservationMiB: 64,
                    essential: true,
                    environment: {
                        SERVER_NAME: records.join(' '),
                        CERT_NAME: records[0],
                    },
                    logging: ecs.LogDrivers.awsLogs({
                        logGroup: logGroup,
                        streamPrefix: 'sample',
                    }),
                    portMappings: [
                        {
                            hostPort: 80,
                            containerPort: 80,
                            protocol: ecs.Protocol.TCP,
                        },
                        {
                            hostPort: 443,
                            containerPort: 443,
                            protocol: ecs.Protocol.TCP,
                        },
                    ],
                },
            ],
        };
    }
}
exports.LowCostECS = LowCostECS;
_a = JSII_RTTI_SYMBOL_1;
LowCostECS[_a] = { fqn: "low-cost-ecs.LowCostECS", version: "0.0.24" };
//# sourceMappingURL=data:application/json;base64,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