"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const resource_arn_suffix_1 = require("./resource-arn-suffix");
/**
 * Determines the level of detail about provisioned throughput consumption that is returned.
 */
var DynamoConsumedCapacity;
(function (DynamoConsumedCapacity) {
    /**
     * The response includes the aggregate ConsumedCapacity for the operation,
     * together with ConsumedCapacity for each table and secondary index that was accessed
     */
    DynamoConsumedCapacity["INDEXES"] = "INDEXES";
    /**
     * The response includes only the aggregate ConsumedCapacity for the operation.
     */
    DynamoConsumedCapacity["TOTAL"] = "TOTAL";
    /**
     * No ConsumedCapacity details are included in the response.
     */
    DynamoConsumedCapacity["NONE"] = "NONE";
})(DynamoConsumedCapacity = exports.DynamoConsumedCapacity || (exports.DynamoConsumedCapacity = {}));
/**
 * Determines whether item collection metrics are returned.
 */
var DynamoItemCollectionMetrics;
(function (DynamoItemCollectionMetrics) {
    /**
     * If set to SIZE, the response includes statistics about item collections,
     * if any, that were modified during the operation.
     */
    DynamoItemCollectionMetrics["SIZE"] = "SIZE";
    /**
     * If set to NONE, no statistics are returned.
     */
    DynamoItemCollectionMetrics["NONE"] = "NONE";
})(DynamoItemCollectionMetrics = exports.DynamoItemCollectionMetrics || (exports.DynamoItemCollectionMetrics = {}));
/**
 * Use ReturnValues if you want to get the item attributes as they appear before or after they are changed
 */
var DynamoReturnValues;
(function (DynamoReturnValues) {
    /**
     * Nothing is returned
     */
    DynamoReturnValues["NONE"] = "NONE";
    /**
     * Returns all of the attributes of the item
     */
    DynamoReturnValues["ALL_OLD"] = "ALL_OLD";
    /**
     * Returns only the updated attributes
     */
    DynamoReturnValues["UPDATED_OLD"] = "UPDATED_OLD";
    /**
     * Returns all of the attributes of the item
     */
    DynamoReturnValues["ALL_NEW"] = "ALL_NEW";
    /**
     * Returns only the updated attributes
     */
    DynamoReturnValues["UPDATED_NEW"] = "UPDATED_NEW";
})(DynamoReturnValues = exports.DynamoReturnValues || (exports.DynamoReturnValues = {}));
/**
 * Class to generate AttributeValue
 */
class DynamoAttributeValue {
    constructor() {
        this.attributeValue = {};
    }
    /**
     * Sets an attribute of type String. For example:  "S": "Hello"
     */
    withS(value) {
        this.attributeValue.S = value;
        return this;
    }
    /**
     * Sets an attribute of type Number. For example:  "N": "123.45"
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    withN(value) {
        this.attributeValue.N = value;
        return this;
    }
    /**
     * Sets an attribute of type Binary. For example:  "B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"
     */
    withB(value) {
        this.attributeValue.B = value;
        return this;
    }
    /**
     * Sets an attribute of type String Set. For example:  "SS": ["Giraffe", "Hippo" ,"Zebra"]
     */
    withSS(value) {
        this.attributeValue.SS = value;
        return this;
    }
    /**
     * Sets an attribute of type Number Set. For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    withNS(value) {
        this.attributeValue.NS = value;
        return this;
    }
    /**
     * Sets an attribute of type Binary Set. For example:  "BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]
     */
    withBS(value) {
        this.attributeValue.BS = value;
        return this;
    }
    /**
     * Sets an attribute of type Map. For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}
     */
    withM(value) {
        this.attributeValue.M = transformAttributeValueMap(value);
        return this;
    }
    /**
     * Sets an attribute of type List. For example:  "L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N", "3.14159"}]
     */
    withL(value) {
        this.attributeValue.L = value.map(val => val.toObject());
        return this;
    }
    /**
     * Sets an attribute of type Null. For example:  "NULL": true
     */
    withNULL(value) {
        this.attributeValue.NULL = value;
        return this;
    }
    /**
     * Sets an attribute of type Boolean. For example:  "BOOL": true
     */
    withBOOL(value) {
        this.attributeValue.BOOL = value;
        return this;
    }
    /**
     * Return the attributeValue object
     */
    toObject() {
        return this.attributeValue;
    }
}
exports.DynamoAttributeValue = DynamoAttributeValue;
/**
 * Class to generate projection expression
 */
class DynamoProjectionExpression {
    constructor() {
        this.expression = [];
    }
    /**
     * Adds the passed attribute to the chain
     *
     * @param attr Attribute name
     */
    withAttribute(attr) {
        if (this.expression.length) {
            this.expression.push(`.${attr}`);
        }
        else {
            this.expression.push(attr);
        }
        return this;
    }
    /**
     * Adds the array literal access for passed index
     *
     * @param index array index
     */
    atIndex(index) {
        if (!this.expression.length) {
            throw new Error('Expression must start with an attribute');
        }
        this.expression.push(`[${index}]`);
        return this;
    }
    /**
     * converts and return the string expression
     */
    toString() {
        return this.expression.join('');
    }
}
exports.DynamoProjectionExpression = DynamoProjectionExpression;
/**
 * A StepFunctions task to call DynamoGetItem
 */
class DynamoGetItem {
    constructor(props) {
        this.props = props;
        core_1.withResolved(props.tableName, validateTableName);
    }
    bind(_task) {
        var _a;
        return {
            resourceArn: getDynamoResourceArn(DynamoMethod.GET),
            policyStatements: getDynamoPolicyStatements(_task, this.props.tableName, DynamoMethod.GET),
            parameters: {
                Key: configurePrimaryKey(this.props.partitionKey, this.props.sortKey),
                TableName: this.props.tableName,
                ConsistentRead: (_a = this.props.consistentRead) !== null && _a !== void 0 ? _a : false,
                ExpressionAttributeNames: this.props.expressionAttributeNames,
                ProjectionExpression: this.configureProjectionExpression(this.props.projectionExpression),
                ReturnConsumedCapacity: this.props.returnConsumedCapacity
            }
        };
    }
    configureProjectionExpression(expressions) {
        return expressions
            ? expressions.map(expression => expression.toString()).join(',')
            : undefined;
    }
}
exports.DynamoGetItem = DynamoGetItem;
/**
 * A StepFunctions task to call DynamoPutItem
 */
class DynamoPutItem {
    constructor(props) {
        this.props = props;
        core_1.withResolved(props.tableName, validateTableName);
    }
    bind(_task) {
        return {
            resourceArn: getDynamoResourceArn(DynamoMethod.PUT),
            policyStatements: getDynamoPolicyStatements(_task, this.props.tableName, DynamoMethod.PUT),
            parameters: {
                Item: transformAttributeValueMap(this.props.item),
                TableName: this.props.tableName,
                ConditionExpression: this.props.conditionExpression,
                ExpressionAttributeNames: this.props.expressionAttributeNames,
                ExpressionAttributeValues: transformAttributeValueMap(this.props.expressionAttributeValues),
                ReturnConsumedCapacity: this.props.returnConsumedCapacity,
                ReturnItemCollectionMetrics: this.props.returnItemCollectionMetrics,
                ReturnValues: this.props.returnValues
            }
        };
    }
}
exports.DynamoPutItem = DynamoPutItem;
/**
 * A StepFunctions task to call DynamoDeleteItem
 */
class DynamoDeleteItem {
    constructor(props) {
        this.props = props;
        core_1.withResolved(props.tableName, validateTableName);
    }
    bind(_task) {
        return {
            resourceArn: getDynamoResourceArn(DynamoMethod.DELETE),
            policyStatements: getDynamoPolicyStatements(_task, this.props.tableName, DynamoMethod.DELETE),
            parameters: {
                Key: configurePrimaryKey(this.props.partitionKey, this.props.sortKey),
                TableName: this.props.tableName,
                ConditionExpression: this.props.conditionExpression,
                ExpressionAttributeNames: this.props.expressionAttributeNames,
                ExpressionAttributeValues: transformAttributeValueMap(this.props.expressionAttributeValues),
                ReturnConsumedCapacity: this.props.returnConsumedCapacity,
                ReturnItemCollectionMetrics: this.props.returnItemCollectionMetrics,
                ReturnValues: this.props.returnValues
            }
        };
    }
}
exports.DynamoDeleteItem = DynamoDeleteItem;
/**
 * A StepFunctions task to call DynamoUpdateItem
 */
class DynamoUpdateItem {
    constructor(props) {
        this.props = props;
        core_1.withResolved(props.tableName, validateTableName);
    }
    bind(_task) {
        return {
            resourceArn: getDynamoResourceArn(DynamoMethod.UPDATE),
            policyStatements: getDynamoPolicyStatements(_task, this.props.tableName, DynamoMethod.UPDATE),
            parameters: {
                Key: configurePrimaryKey(this.props.partitionKey, this.props.sortKey),
                TableName: this.props.tableName,
                ConditionExpression: this.props.conditionExpression,
                ExpressionAttributeNames: this.props.expressionAttributeNames,
                ExpressionAttributeValues: transformAttributeValueMap(this.props.expressionAttributeValues),
                ReturnConsumedCapacity: this.props.returnConsumedCapacity,
                ReturnItemCollectionMetrics: this.props.returnItemCollectionMetrics,
                ReturnValues: this.props.returnValues,
                UpdateExpression: this.props.updateExpression
            }
        };
    }
}
exports.DynamoUpdateItem = DynamoUpdateItem;
/**
 * A helper wrapper class to call all DynamoDB APIs
 */
class CallDynamoDB {
    /**
     * Method to get DynamoGetItem task
     *
     * @param props DynamoGetItemProps
     */
    static getItem(props) {
        return new DynamoGetItem(props);
    }
    /**
     * Method to get DynamoPutItem task
     *
     * @param props DynamoPutItemProps
     */
    static putItem(props) {
        return new DynamoPutItem(props);
    }
    /**
     * Method to get DynamoDeleteItem task
     *
     * @param props DynamoDeleteItemProps
     */
    static deleteItem(props) {
        return new DynamoDeleteItem(props);
    }
    /**
     * Method to get DynamoUpdateItem task
     *
     * @param props DynamoUpdateItemProps
     */
    static updateItem(props) {
        return new DynamoUpdateItem(props);
    }
}
exports.CallDynamoDB = CallDynamoDB;
var DynamoMethod;
(function (DynamoMethod) {
    DynamoMethod["GET"] = "Get";
    DynamoMethod["PUT"] = "Put";
    DynamoMethod["DELETE"] = "Delete";
    DynamoMethod["UPDATE"] = "Update";
})(DynamoMethod || (DynamoMethod = {}));
function validateTableName(tableName) {
    if (tableName.length < 3 ||
        tableName.length > 255 ||
        !new RegExp(/[a-zA-Z0-9_.-]+$/).test(tableName)) {
        throw new Error(`TableName should not contain alphanumeric characters and should be between 3-255 characters long. Received: ${tableName}`);
    }
}
function getDynamoResourceArn(method) {
    return resource_arn_suffix_1.getResourceArn('dynamodb', `${method.toLowerCase()}Item`, sfn.ServiceIntegrationPattern.FIRE_AND_FORGET);
}
function getDynamoPolicyStatements(task, tableName, method) {
    return [
        new iam.PolicyStatement({
            resources: [
                core_1.Stack.of(task).formatArn({
                    service: 'dynamodb',
                    resource: 'table',
                    resourceName: tableName
                })
            ],
            actions: [`dynamodb:${method}Item`]
        })
    ];
}
function configurePrimaryKey(partitionKey, sortKey) {
    const key = {
        [partitionKey.name]: partitionKey.value.toObject()
    };
    if (sortKey) {
        key[sortKey.name] = sortKey.value.toObject();
    }
    return key;
}
function transformAttributeValueMap(attrMap) {
    const transformedValue = {};
    for (const key in attrMap) {
        if (key) {
            transformedValue[key] = attrMap[key].toObject();
        }
    }
    return attrMap ? transformedValue : undefined;
}
//# sourceMappingURL=data:application/json;base64,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