"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const resource_arn_suffix_1 = require("./resource-arn-suffix");
/**
 * A Step Functions Task to create an EMR Cluster.
 *
 * The ClusterConfiguration is defined as Parameters in the state machine definition.
 *
 * OUTPUT: the ClusterId.
 *
 * @experimental
 */
class EmrCreateCluster {
    constructor(props) {
        this.props = props;
        this.visibleToAllUsers = (this.props.visibleToAllUsers !== undefined) ? this.props.visibleToAllUsers : true;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.SYNC;
        this._serviceRole = this.props.serviceRole;
        this._clusterRole = this.props.clusterRole;
        this._autoScalingRole = this.props.autoScalingRole;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.SYNC
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call CreateCluster.`);
        }
    }
    /**
     * The service role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get serviceRole() {
        if (this._serviceRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._serviceRole;
    }
    /**
     * The instance role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get clusterRole() {
        if (this._clusterRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._clusterRole;
    }
    /**
     * The autoscaling role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get autoScalingRole() {
        if (this._autoScalingRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._autoScalingRole;
    }
    bind(task) {
        var _a, _b;
        // If the Roles are undefined then they weren't provided, so create them
        this._serviceRole = this._serviceRole || this.createServiceRole(task);
        this._clusterRole = this._clusterRole || this.createClusterRole(task);
        // AutoScaling roles are not valid with InstanceFleet clusters.
        // Attempt to create only if .instances.instanceFleets is undefined or empty
        if (this.props.instances.instanceFleets === undefined || this.props.instances.instanceFleets.length === 0) {
            this._autoScalingRole = this._autoScalingRole || this.createAutoScalingRole(task);
            // If InstanceFleets are used and an AutoScaling Role is specified, throw an error
        }
        else if (this._autoScalingRole !== undefined) {
            throw new Error('Auto Scaling roles can not be specified with instance fleets.');
        }
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('elasticmapreduce', 'createCluster', this.integrationPattern),
            policyStatements: this.createPolicyStatements(task, this._serviceRole, this._clusterRole, this._autoScalingRole),
            parameters: {
                Instances: EmrCreateCluster.InstancesConfigPropertyToJson(this.props.instances),
                JobFlowRole: cdk.stringToCloudFormation(this._clusterRole.roleName),
                Name: cdk.stringToCloudFormation(this.props.name),
                ServiceRole: cdk.stringToCloudFormation(this._serviceRole.roleName),
                AdditionalInfo: cdk.stringToCloudFormation(this.props.additionalInfo),
                Applications: cdk.listMapper(EmrCreateCluster.ApplicationConfigPropertyToJson)(this.props.applications),
                AutoScalingRole: cdk.stringToCloudFormation((_a = this._autoScalingRole) === null || _a === void 0 ? void 0 : _a.roleName),
                BootstrapActions: cdk.listMapper(EmrCreateCluster.BootstrapActionConfigToJson)(this.props.bootstrapActions),
                Configurations: cdk.listMapper(EmrCreateCluster.ConfigurationPropertyToJson)(this.props.configurations),
                CustomAmiId: cdk.stringToCloudFormation(this.props.customAmiId),
                EbsRootVolumeSize: cdk.numberToCloudFormation(this.props.ebsRootVolumeSize),
                KerberosAttributes: (this.props.kerberosAttributes === undefined) ?
                    this.props.kerberosAttributes :
                    EmrCreateCluster.KerberosAttributesPropertyToJson(this.props.kerberosAttributes),
                LogUri: cdk.stringToCloudFormation(this.props.logUri),
                ReleaseLabel: cdk.stringToCloudFormation(this.props.releaseLabel),
                ScaleDownBehavior: cdk.stringToCloudFormation((_b = this.props.scaleDownBehavior) === null || _b === void 0 ? void 0 : _b.valueOf()),
                SecurityConfiguration: cdk.stringToCloudFormation(this.props.securityConfiguration),
                Tags: cdk.listMapper(cdk.cfnTagToCloudFormation)(this.props.tags),
                VisibleToAllUsers: cdk.booleanToCloudFormation(this.visibleToAllUsers)
            }
        };
    }
    /**
     * This generates the PolicyStatements required by the Task to call CreateCluster.
     */
    createPolicyStatements(task, serviceRole, clusterRole, autoScalingRole) {
        const stack = cdk.Stack.of(task);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: [
                    'elasticmapreduce:RunJobFlow',
                    'elasticmapreduce:DescribeCluster',
                    'elasticmapreduce:TerminateJobFlows'
                ],
                resources: ['*']
            })
        ];
        // Allow the StateMachine to PassRole to Cluster roles
        policyStatements.push(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [
                serviceRole.roleArn,
                clusterRole.roleArn
            ]
        }));
        if (autoScalingRole !== undefined) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [autoScalingRole.roleArn]
            }));
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.SYNC) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventForEMRRunJobFlowRule'
                    })]
            }));
        }
        return policyStatements;
    }
    /**
     * Generate the Role used by the EMR Service
     */
    createServiceRole(task) {
        return new iam.Role(task, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceRole')
            ]
        });
    }
    /**
     * Generate the Role and Instance Profile used by the EC2 instances
     *
     * Data access permissions will need to be updated by the user
     */
    createClusterRole(task) {
        const role = new iam.Role(task, 'InstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com')
        });
        new iam.CfnInstanceProfile(task, 'InstanceProfile', {
            roles: [role.roleName],
            instanceProfileName: role.roleName
        });
        return role;
    }
    /**
     * Generate the Role used to AutoScale the Cluster
     */
    createAutoScalingRole(task) {
        var _a;
        const role = new iam.Role(task, 'AutoScalingRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceforAutoScalingRole')
            ]
        });
        (_a = role.assumeRolePolicy) === null || _a === void 0 ? void 0 : _a.addStatements(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            principals: [
                new iam.ServicePrincipal('application-autoscaling.amazonaws.com')
            ],
            actions: [
                'sts:AssumeRole'
            ]
        }));
        return role;
    }
}
exports.EmrCreateCluster = EmrCreateCluster;
(function (EmrCreateCluster) {
    /**
     * Valid valus for the Cluster ScaleDownBehavior
     *
     * @experimental
     */
    let EmrClusterScaleDownBehavior;
    (function (EmrClusterScaleDownBehavior) {
        /**
         * Indicates that Amazon EMR terminates nodes at the instance-hour boundary, regardless of when the request to terminate the instance was
         * submitted. This option is only available with Amazon EMR 5.1.0 and later and is the default for clusters created using that version
         */
        EmrClusterScaleDownBehavior["TERMINATE_AT_INSTANCE_HOUR"] = "TERMINATE_AT_INSTANCE_HOUR";
        /**
         * Indicates that Amazon EMR blacklists and drains tasks from nodes before terminating the Amazon EC2 instances, regardless of the
         * instance-hour boundary.
         */
        EmrClusterScaleDownBehavior["TERMINATE_AT_TASK_COMPLETION"] = "TERMINATE_AT_TASK_COMPLETION";
    })(EmrClusterScaleDownBehavior = EmrCreateCluster.EmrClusterScaleDownBehavior || (EmrCreateCluster.EmrClusterScaleDownBehavior = {}));
    /**
     * Instance Role Types
     *
     * @experimental
     */
    let InstanceRoleType;
    (function (InstanceRoleType) {
        /**
         * Master Node
         */
        InstanceRoleType["MASTER"] = "MASTER";
        /**
         * Core Node
         */
        InstanceRoleType["CORE"] = "CORE";
        /**
         * Task Node
         */
        InstanceRoleType["TASK"] = "TASK";
    })(InstanceRoleType = EmrCreateCluster.InstanceRoleType || (EmrCreateCluster.InstanceRoleType = {}));
    /**
     * EBS Volume Types
     *
     * @experimental
     */
    let EbsBlockDeviceVolumeType;
    (function (EbsBlockDeviceVolumeType) {
        /**
         * gp2 Volume Type
         */
        EbsBlockDeviceVolumeType["GP2"] = "gp2";
        /**
         * io1 Volume Type
         */
        EbsBlockDeviceVolumeType["IO1"] = "io1";
        /**
         * Standard Volume Type
         */
        EbsBlockDeviceVolumeType["STANDARD"] = "standard";
    })(EbsBlockDeviceVolumeType = EmrCreateCluster.EbsBlockDeviceVolumeType || (EmrCreateCluster.EbsBlockDeviceVolumeType = {}));
    /**
     * Render the EbsBlockDeviceConfigProperty as JSON
     *
     * @param property
     */
    function EbsBlockDeviceConfigPropertyToJson(property) {
        var _a;
        return {
            VolumeSpecification: {
                Iops: cdk.numberToCloudFormation(property.volumeSpecification.iops),
                SizeInGB: cdk.numberToCloudFormation(property.volumeSpecification.sizeInGB),
                VolumeType: cdk.stringToCloudFormation((_a = property.volumeSpecification.volumeType) === null || _a === void 0 ? void 0 : _a.valueOf())
            },
            VolumesPerInstance: cdk.numberToCloudFormation(property.volumesPerInstance)
        };
    }
    EmrCreateCluster.EbsBlockDeviceConfigPropertyToJson = EbsBlockDeviceConfigPropertyToJson;
    /**
     * Render the EbsConfigurationProperty to JSON
     *
     * @param property
     */
    function EbsConfigurationPropertyToJson(property) {
        return {
            EbsBlockDeviceConfigs: cdk.listMapper(EbsBlockDeviceConfigPropertyToJson)(property.ebsBlockDeviceConfigs),
            EbsOptimized: cdk.booleanToCloudFormation(property.ebsOptimized)
        };
    }
    EmrCreateCluster.EbsConfigurationPropertyToJson = EbsConfigurationPropertyToJson;
    /**
     * Render the InstanceTypeConfigProperty to JSON]
     *
     * @param property
     */
    function InstanceTypeConfigPropertyToJson(property) {
        var _a;
        return {
            BidPrice: cdk.stringToCloudFormation(property.bidPrice),
            BidPriceAsPercentageOfOnDemandPrice: cdk.numberToCloudFormation(property.bidPriceAsPercentageOfOnDemandPrice),
            Configurations: cdk.listMapper(ConfigurationPropertyToJson)(property.configurations),
            EbsConfiguration: (property.ebsConfiguration === undefined) ?
                property.ebsConfiguration :
                EbsConfigurationPropertyToJson(property.ebsConfiguration),
            InstanceType: cdk.stringToCloudFormation((_a = property.instanceType) === null || _a === void 0 ? void 0 : _a.valueOf()),
            WeightedCapacity: cdk.numberToCloudFormation(property.weightedCapacity)
        };
    }
    EmrCreateCluster.InstanceTypeConfigPropertyToJson = InstanceTypeConfigPropertyToJson;
    /**
     * Spot Timeout Actions
     *
     * @experimental
     */
    let SpotTimeoutAction;
    (function (SpotTimeoutAction) {
        /**\
         * SWITCH_TO_ON_DEMAND
         */
        SpotTimeoutAction["SWITCH_TO_ON_DEMAND"] = "SWITCH_TO_ON_DEMAND";
        /**
         * TERMINATE_CLUSTER
         */
        SpotTimeoutAction["TERMINATE_CLUSTER"] = "TERMINATE_CLUSTER";
    })(SpotTimeoutAction = EmrCreateCluster.SpotTimeoutAction || (EmrCreateCluster.SpotTimeoutAction = {}));
    /**
     * Render the InstanceFleetProvisioningSpecificationsProperty to JSON
     *
     * @param property
     */
    function InstanceFleetProvisioningSpecificationsPropertyToJson(property) {
        var _a;
        return {
            SpotSpecification: {
                BlockDurationMinutes: cdk.numberToCloudFormation(property.spotSpecification.blockDurationMinutes),
                TimeoutAction: cdk.stringToCloudFormation((_a = property.spotSpecification.timeoutAction) === null || _a === void 0 ? void 0 : _a.valueOf()),
                TimeoutDurationMinutes: cdk.numberToCloudFormation(property.spotSpecification.timeoutDurationMinutes)
            }
        };
    }
    EmrCreateCluster.InstanceFleetProvisioningSpecificationsPropertyToJson = InstanceFleetProvisioningSpecificationsPropertyToJson;
    /**
     * Render the InstanceFleetConfigProperty as JSON
     *
     * @param property
     */
    function InstanceFleetConfigPropertyToJson(property) {
        var _a;
        return {
            InstanceFleetType: cdk.stringToCloudFormation((_a = property.instanceFleetType) === null || _a === void 0 ? void 0 : _a.valueOf()),
            InstanceTypeConfigs: cdk.listMapper(InstanceTypeConfigPropertyToJson)(property.instanceTypeConfigs),
            LaunchSpecifications: (property.launchSpecifications === undefined) ?
                property.launchSpecifications :
                InstanceFleetProvisioningSpecificationsPropertyToJson(property.launchSpecifications),
            Name: cdk.stringToCloudFormation(property.name),
            TargetOnDemandCapacity: cdk.numberToCloudFormation(property.targetOnDemandCapacity),
            TargetSpotCapacity: cdk.numberToCloudFormation(property.targetSpotCapacity)
        };
    }
    EmrCreateCluster.InstanceFleetConfigPropertyToJson = InstanceFleetConfigPropertyToJson;
    /**
     * CloudWatch Alarm Comparison Operators
     *
     * @experimental
     */
    let CloudWatchAlarmComparisonOperator;
    (function (CloudWatchAlarmComparisonOperator) {
        /**
         * GREATER_THAN_OR_EQUAL
         */
        CloudWatchAlarmComparisonOperator["GREATER_THAN_OR_EQUAL"] = "GREATER_THAN_OR_EQUAL";
        /**
         * GREATER_THAN
         */
        CloudWatchAlarmComparisonOperator["GREATER_THAN"] = "GREATER_THAN";
        /**
         * LESS_THAN
         */
        CloudWatchAlarmComparisonOperator["LESS_THAN"] = "LESS_THAN";
        /**
         * LESS_THAN_OR_EQUAL
         */
        CloudWatchAlarmComparisonOperator["LESS_THAN_OR_EQUAL"] = "LESS_THAN_OR_EQUAL";
    })(CloudWatchAlarmComparisonOperator = EmrCreateCluster.CloudWatchAlarmComparisonOperator || (EmrCreateCluster.CloudWatchAlarmComparisonOperator = {}));
    /**
     * CloudWatch Alarm Statistics
     *
     * @experimental
     */
    let CloudWatchAlarmStatistic;
    (function (CloudWatchAlarmStatistic) {
        /**
         * SAMPLE_COUNT
         */
        CloudWatchAlarmStatistic["SAMPLE_COUNT"] = "SAMPLE_COUNT";
        /**
         * AVERAGE
         */
        CloudWatchAlarmStatistic["AVERAGE"] = "AVERAGE";
        /**
         * SUM
         */
        CloudWatchAlarmStatistic["SUM"] = "SUM";
        /**
         * MINIMUM
         */
        CloudWatchAlarmStatistic["MINIMUM"] = "MINIMUM";
        /**
         * MAXIMUM
         */
        CloudWatchAlarmStatistic["MAXIMUM"] = "MAXIMUM";
    })(CloudWatchAlarmStatistic = EmrCreateCluster.CloudWatchAlarmStatistic || (EmrCreateCluster.CloudWatchAlarmStatistic = {}));
    /**
     * CloudWatch Alarm Units
     *
     * @experimental
     */
    let CloudWatchAlarmUnit;
    (function (CloudWatchAlarmUnit) {
        /**
         * NONE
         */
        CloudWatchAlarmUnit["NONE"] = "NONE";
        /**
         * SECONDS
         */
        CloudWatchAlarmUnit["SECONDS"] = "SECONDS";
        /**
         * MICRO_SECONDS
         */
        CloudWatchAlarmUnit["MICRO_SECONDS"] = "MICRO_SECONDS";
        /**
         * MILLI_SECONDS
         */
        CloudWatchAlarmUnit["MILLI_SECONDS"] = "MILLI_SECONDS";
        /**
         * BYTES
         */
        CloudWatchAlarmUnit["BYTES"] = "BYTES";
        /**
         * KILO_BYTES
         */
        CloudWatchAlarmUnit["KILO_BYTES"] = "KILO_BYTES";
        /**
         * MEGA_BYTES
         */
        CloudWatchAlarmUnit["MEGA_BYTES"] = "MEGA_BYTES";
        /**
         * GIGA_BYTES
         */
        CloudWatchAlarmUnit["GIGA_BYTES"] = "GIGA_BYTES";
        /**
         * TERA_BYTES
         */
        CloudWatchAlarmUnit["TERA_BYTES"] = "TERA_BYTES";
        /**
         * BITS
         */
        CloudWatchAlarmUnit["BITS"] = "BITS";
        /**
         * KILO_BITS
         */
        CloudWatchAlarmUnit["KILO_BITS"] = "KILO_BITS";
        /**
         * MEGA_BITS
         */
        CloudWatchAlarmUnit["MEGA_BITS"] = "MEGA_BITS";
        /**
         * GIGA_BITS
         */
        CloudWatchAlarmUnit["GIGA_BITS"] = "GIGA_BITS";
        /**
         * TERA_BITS
         */
        CloudWatchAlarmUnit["TERA_BITS"] = "TERA_BITS";
        /**
         * PERCENT
         */
        CloudWatchAlarmUnit["PERCENT"] = "PERCENT";
        /**
         * COUNT
         */
        CloudWatchAlarmUnit["COUNT"] = "COUNT";
        /**
         * BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["BYTES_PER_SECOND"] = "BYTES_PER_SECOND";
        /**
         * KILO_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["KILO_BYTES_PER_SECOND"] = "KILO_BYTES_PER_SECOND";
        /**
         * MEGA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["MEGA_BYTES_PER_SECOND"] = "MEGA_BYTES_PER_SECOND";
        /**
         * GIGA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["GIGA_BYTES_PER_SECOND"] = "GIGA_BYTES_PER_SECOND";
        /**
         * TERA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["TERA_BYTES_PER_SECOND"] = "TERA_BYTES_PER_SECOND";
        /**
         * BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["BITS_PER_SECOND"] = "BITS_PER_SECOND";
        /**
         * KILO_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["KILO_BITS_PER_SECOND"] = "KILO_BITS_PER_SECOND";
        /**
         * MEGA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["MEGA_BITS_PER_SECOND"] = "MEGA_BITS_PER_SECOND";
        /**
         * GIGA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["GIGA_BITS_PER_SECOND"] = "GIGA_BITS_PER_SECOND";
        /**
         * TERA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["TERA_BITS_PER_SECOND"] = "TERA_BITS_PER_SECOND";
        /**
         * COUNT_PER_SECOND
         */
        CloudWatchAlarmUnit["COUNT_PER_SECOND"] = "COUNT_PER_SECOND";
    })(CloudWatchAlarmUnit = EmrCreateCluster.CloudWatchAlarmUnit || (EmrCreateCluster.CloudWatchAlarmUnit = {}));
    /**
     * Render the MetricDimensionProperty as JSON
     *
     * @param property
     */
    function MetricDimensionPropertyToJson(property) {
        return {
            Key: cdk.stringToCloudFormation(property.key),
            Value: cdk.stringToCloudFormation(property.value)
        };
    }
    EmrCreateCluster.MetricDimensionPropertyToJson = MetricDimensionPropertyToJson;
    /**
     * Render the ScalingTriggerProperty to JSON
     *
     * @param property
     */
    function ScalingTriggerPropertyToJson(property) {
        var _a, _b, _c;
        return {
            CloudWatchAlarmDefinition: {
                ComparisonOperator: cdk.stringToCloudFormation((_a = property.cloudWatchAlarmDefinition.comparisonOperator) === null || _a === void 0 ? void 0 : _a.valueOf()),
                Dimensions: cdk.listMapper(MetricDimensionPropertyToJson)(property.cloudWatchAlarmDefinition.dimensions),
                EvaluationPeriods: cdk.numberToCloudFormation(property.cloudWatchAlarmDefinition.evalutionPeriods),
                MetricName: cdk.stringToCloudFormation(property.cloudWatchAlarmDefinition.metricName),
                Namespace: cdk.stringToCloudFormation(property.cloudWatchAlarmDefinition.namespace),
                Period: cdk.numberToCloudFormation(property.cloudWatchAlarmDefinition.period.toSeconds()),
                Statistic: cdk.stringToCloudFormation((_b = property.cloudWatchAlarmDefinition.statistic) === null || _b === void 0 ? void 0 : _b.valueOf()),
                Threshold: cdk.numberToCloudFormation(property.cloudWatchAlarmDefinition.threshold),
                Unit: cdk.stringToCloudFormation((_c = property.cloudWatchAlarmDefinition.unit) === null || _c === void 0 ? void 0 : _c.valueOf())
            }
        };
    }
    EmrCreateCluster.ScalingTriggerPropertyToJson = ScalingTriggerPropertyToJson;
    /**
     * EC2 Instance Market
     *
     * @experimental
     */
    let InstanceMarket;
    (function (InstanceMarket) {
        /**
         * On Demand Instance
         */
        InstanceMarket["ON_DEMAND"] = "ON_DEMAND";
        /**
         * Spot Instance
         */
        InstanceMarket["SPOT"] = "SPOT";
    })(InstanceMarket = EmrCreateCluster.InstanceMarket || (EmrCreateCluster.InstanceMarket = {}));
    /**
     * AutoScaling Adjustment Type
     *
     * @experimental
     */
    let ScalingAdjustmentType;
    (function (ScalingAdjustmentType) {
        /**
         * CHANGE_IN_CAPACITY
         */
        ScalingAdjustmentType["CHANGE_IN_CAPACITY"] = "CHANGE_IN_CAPACITY";
        /**
         * PERCENT_CHANGE_IN_CAPACITY
         */
        ScalingAdjustmentType["PERCENT_CHANGE_IN_CAPACITY"] = "PERCENT_CHANGE_IN_CAPACITY";
        /**
         * EXACT_CAPACITY
         */
        ScalingAdjustmentType["EXACT_CAPACITY"] = "EXACT_CAPACITY";
    })(ScalingAdjustmentType = EmrCreateCluster.ScalingAdjustmentType || (EmrCreateCluster.ScalingAdjustmentType = {}));
    /**
     * Render the ScalingActionPropety to JSON
     *
     * @param property
     */
    function ScalingActionPropertyToJson(property) {
        var _a;
        return {
            Market: cdk.stringToCloudFormation((_a = property.market) === null || _a === void 0 ? void 0 : _a.valueOf()),
            SimpleScalingPolicyConfiguration: {
                AdjustmentType: cdk.stringToCloudFormation(property.simpleScalingPolicyConfiguration.adjustmentType),
                CoolDown: cdk.numberToCloudFormation(property.simpleScalingPolicyConfiguration.coolDown),
                ScalingAdjustment: cdk.numberToCloudFormation(property.simpleScalingPolicyConfiguration.scalingAdjustment)
            }
        };
    }
    EmrCreateCluster.ScalingActionPropertyToJson = ScalingActionPropertyToJson;
    /**
     * Render the ScalingRuleProperty to JSON
     *
     * @param property
     */
    function ScalingRulePropertyToJson(property) {
        return {
            Action: ScalingActionPropertyToJson(property.action),
            Description: cdk.stringToCloudFormation(property.description),
            Name: cdk.stringToCloudFormation(property.name),
            Trigger: ScalingTriggerPropertyToJson(property.trigger)
        };
    }
    EmrCreateCluster.ScalingRulePropertyToJson = ScalingRulePropertyToJson;
    /**
     * Render the AutoScalingPolicyProperty to JSON
     *
     * @param property
     */
    function AutoScalingPolicyPropertyToJson(property) {
        return {
            Constraints: {
                MaxCapacity: cdk.numberToCloudFormation(property.constraints.maxCapacity),
                MinCapacity: cdk.numberToCloudFormation(property.constraints.minCapacity)
            },
            Rules: cdk.listMapper(ScalingRulePropertyToJson)(property.rules)
        };
    }
    EmrCreateCluster.AutoScalingPolicyPropertyToJson = AutoScalingPolicyPropertyToJson;
    /**
     * Render the InstanceGroupConfigProperty to JSON
     *
     * @param property
     */
    function InstanceGroupConfigPropertyToJson(property) {
        var _a, _b;
        return {
            AutoScalingPolicy: (property.autoScalingPolicy === undefined) ?
                property.autoScalingPolicy :
                AutoScalingPolicyPropertyToJson(property.autoScalingPolicy),
            BidPrice: cdk.numberToCloudFormation(property.bidPrice),
            Configurations: cdk.listMapper(ConfigurationPropertyToJson)(property.configurations),
            EbsConfiguration: (property.ebsConfiguration === undefined) ?
                property.ebsConfiguration :
                EbsConfigurationPropertyToJson(property.ebsConfiguration),
            InstanceCount: cdk.numberToCloudFormation(property.instanceCount),
            InstanceRole: cdk.stringToCloudFormation((_a = property.instanceRole) === null || _a === void 0 ? void 0 : _a.valueOf()),
            InstanceType: cdk.stringToCloudFormation(property.instanceType),
            Market: cdk.stringToCloudFormation((_b = property.market) === null || _b === void 0 ? void 0 : _b.valueOf()),
            Name: cdk.stringToCloudFormation(property.name)
        };
    }
    EmrCreateCluster.InstanceGroupConfigPropertyToJson = InstanceGroupConfigPropertyToJson;
    /**
     * Render the PlacementTypeProperty to JSON
     *
     * @param property
     */
    function PlacementTypePropertyToJson(property) {
        return {
            AvailabilityZone: cdk.stringToCloudFormation(property.availabilityZone),
            AvailabilityZones: cdk.listMapper(cdk.stringToCloudFormation)(property.availabilityZones)
        };
    }
    EmrCreateCluster.PlacementTypePropertyToJson = PlacementTypePropertyToJson;
    /**
     * Render the InstancesConfigProperty to JSON
     *
     * @param property
     */
    function InstancesConfigPropertyToJson(property) {
        return {
            AdditionalMasterSecurityGroups: cdk.listMapper(cdk.stringToCloudFormation)(property.additionalMasterSecurityGroups),
            AdditionalSlaveSecurityGroups: cdk.listMapper(cdk.stringToCloudFormation)(property.additionalSlaveSecurityGroups),
            Ec2KeyName: cdk.stringToCloudFormation(property.ec2KeyName),
            Ec2SubnetId: cdk.stringToCloudFormation(property.ec2SubnetId),
            Ec2SubnetIds: cdk.listMapper(cdk.stringToCloudFormation)(property.ec2SubnetIds),
            EmrManagedMasterSecurityGroup: cdk.stringToCloudFormation(property.emrManagedMasterSecurityGroup),
            EmrManagedSlaveSecurityGroup: cdk.stringToCloudFormation(property.emrManagedSlaveSecurityGroup),
            HadoopVersion: cdk.stringToCloudFormation(property.hadoopVersion),
            InstanceCount: cdk.numberToCloudFormation(property.instanceCount),
            InstanceFleets: cdk.listMapper(InstanceFleetConfigPropertyToJson)(property.instanceFleets),
            InstanceGroups: cdk.listMapper(InstanceGroupConfigPropertyToJson)(property.instanceGroups),
            KeepJobFlowAliveWhenNoSteps: true,
            MasterInstanceType: cdk.stringToCloudFormation(property.masterInstanceType),
            Placement: (property.placement === undefined) ?
                property.placement :
                PlacementTypePropertyToJson(property.placement),
            ServiceAccessSecurityGroup: cdk.stringToCloudFormation(property.serviceAccessSecurityGroup),
            SlaveInstanceType: cdk.stringToCloudFormation(property.slaveInstanceType),
            TerminationProtected: cdk.booleanToCloudFormation(property.terminationProtected)
        };
    }
    EmrCreateCluster.InstancesConfigPropertyToJson = InstancesConfigPropertyToJson;
    /**
     * Render the ApplicationConfigProperty as JSON
     *
     * @param property
     */
    function ApplicationConfigPropertyToJson(property) {
        return {
            Name: cdk.stringToCloudFormation(property.name),
            Args: cdk.listMapper(cdk.stringToCloudFormation)(property.args),
            Version: cdk.stringToCloudFormation(property.version),
            AdditionalInfo: cdk.objectToCloudFormation(property.additionalInfo)
        };
    }
    EmrCreateCluster.ApplicationConfigPropertyToJson = ApplicationConfigPropertyToJson;
    /**
     * Render the BootstrapActionProperty as JSON
     *
     * @param property
     */
    function BootstrapActionConfigToJson(property) {
        return {
            Name: cdk.stringToCloudFormation(property.name),
            ScriptBootstrapAction: {
                Path: cdk.stringToCloudFormation(property.scriptBootstrapAction.path),
                Args: cdk.listMapper(cdk.stringToCloudFormation)(property.scriptBootstrapAction.args)
            }
        };
    }
    EmrCreateCluster.BootstrapActionConfigToJson = BootstrapActionConfigToJson;
    /**
     * Render the ConfigurationProperty as JSON
     *
     * @param property
     */
    function ConfigurationPropertyToJson(property) {
        return {
            Classification: cdk.stringToCloudFormation(property.classification),
            Properties: cdk.objectToCloudFormation(property.properties),
            Configurations: cdk.listMapper(ConfigurationPropertyToJson)(property.configurations)
        };
    }
    EmrCreateCluster.ConfigurationPropertyToJson = ConfigurationPropertyToJson;
    /**
     * Render the KerberosAttributesProperty as JSON
     *
     * @param property
     */
    function KerberosAttributesPropertyToJson(property) {
        return {
            ADDomainJoinPassword: cdk.stringToCloudFormation(property.adDomainJoinPassword),
            ADDomainJoinUser: cdk.stringToCloudFormation(property.adDomainJoinUser),
            CrossRealmTrustPrincipalPassword: cdk.stringToCloudFormation(property.crossRealmTrustPrincipalPassword),
            KdcAdminPassword: cdk.stringToCloudFormation(property.kdcAdminPassword),
            Realm: cdk.stringToCloudFormation(property.realm)
        };
    }
    EmrCreateCluster.KerberosAttributesPropertyToJson = KerberosAttributesPropertyToJson;
})(EmrCreateCluster = exports.EmrCreateCluster || (exports.EmrCreateCluster = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZW1yLWNyZWF0ZS1jbHVzdGVyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiZW1yLWNyZWF0ZS1jbHVzdGVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsd0NBQXdDO0FBQ3hDLGtEQUFrRDtBQUNsRCxxQ0FBcUM7QUFDckMsK0RBQXVEO0FBb0p2RDs7Ozs7Ozs7R0FRRztBQUNILE1BQWEsZ0JBQWdCO0lBUzNCLFlBQTZCLEtBQTRCO1FBQTVCLFVBQUssR0FBTCxLQUFLLENBQXVCO1FBQ3ZELElBQUksQ0FBQyxpQkFBaUIsR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsaUJBQWlCLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQztRQUM1RyxJQUFJLENBQUMsa0JBQWtCLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixJQUFJLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxJQUFJLENBQUM7UUFFekYsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQztRQUMzQyxJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDO1FBQzNDLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQztRQUVuRCxNQUFNLGlCQUFpQixHQUFHO1lBQ3hCLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxlQUFlO1lBQzdDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxJQUFJO1NBQ25DLENBQUM7UUFFRixJQUFJLENBQUMsaUJBQWlCLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFO1lBQ3hELE1BQU0sSUFBSSxLQUFLLENBQUMsd0NBQXdDLElBQUksQ0FBQyxrQkFBa0IsMENBQTBDLENBQUMsQ0FBQztTQUM1SDtJQUNILENBQUM7SUFFRDs7OztPQUlHO0lBQ0gsSUFBVyxXQUFXO1FBQ3BCLElBQUksSUFBSSxDQUFDLFlBQVksS0FBSyxTQUFTLEVBQUU7WUFDbkMsTUFBTSxJQUFJLEtBQUssQ0FBQyx3REFBd0QsQ0FBQyxDQUFDO1NBQzNFO1FBQ0QsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDO0lBQzNCLENBQUM7SUFFRDs7OztPQUlHO0lBQ0gsSUFBVyxXQUFXO1FBQ3BCLElBQUksSUFBSSxDQUFDLFlBQVksS0FBSyxTQUFTLEVBQUU7WUFDbkMsTUFBTSxJQUFJLEtBQUssQ0FBQyx3REFBd0QsQ0FBQyxDQUFDO1NBQzNFO1FBQ0QsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDO0lBQzNCLENBQUM7SUFFRDs7OztPQUlHO0lBQ0gsSUFBVyxlQUFlO1FBQ3hCLElBQUksSUFBSSxDQUFDLGdCQUFnQixLQUFLLFNBQVMsRUFBRTtZQUN2QyxNQUFNLElBQUksS0FBSyxDQUFDLHdEQUF3RCxDQUFDLENBQUM7U0FDM0U7UUFDRCxPQUFPLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQztJQUMvQixDQUFDO0lBRU0sSUFBSSxDQUFDLElBQWM7O1FBQ3hCLHdFQUF3RTtRQUN4RSxJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksQ0FBQyxZQUFZLElBQUksSUFBSSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3RFLElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxDQUFDLFlBQVksSUFBSSxJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFdEUsK0RBQStEO1FBQy9ELDRFQUE0RTtRQUM1RSxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLGNBQWMsS0FBSyxTQUFTLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsY0FBYyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDekcsSUFBSSxDQUFDLGdCQUFnQixHQUFHLElBQUksQ0FBQyxnQkFBZ0IsSUFBSSxJQUFJLENBQUMscUJBQXFCLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDcEYsa0ZBQWtGO1NBQ2pGO2FBQU0sSUFBSSxJQUFJLENBQUMsZ0JBQWdCLEtBQUssU0FBUyxFQUFFO1lBQzlDLE1BQU0sSUFBSSxLQUFLLENBQUMsK0RBQStELENBQUMsQ0FBQztTQUNsRjtRQUVELE9BQU87WUFDTCxXQUFXLEVBQUUsb0NBQWMsQ0FBQyxrQkFBa0IsRUFBRSxlQUFlLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDO1lBQ3pGLGdCQUFnQixFQUFFLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWSxFQUFFLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQztZQUNoSCxVQUFVLEVBQUU7Z0JBQ1YsU0FBUyxFQUFFLGdCQUFnQixDQUFDLDZCQUE2QixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDO2dCQUMvRSxXQUFXLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDO2dCQUNuRSxJQUFJLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDO2dCQUNqRCxXQUFXLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDO2dCQUNuRSxjQUFjLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDO2dCQUNyRSxZQUFZLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxnQkFBZ0IsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDO2dCQUN2RyxlQUFlLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixPQUFDLElBQUksQ0FBQyxnQkFBZ0IsMENBQUUsUUFBUSxDQUFDO2dCQUM1RSxnQkFBZ0IsRUFBRSxHQUFHLENBQUMsVUFBVSxDQUFDLGdCQUFnQixDQUFDLDJCQUEyQixDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztnQkFDM0csY0FBYyxFQUFFLEdBQUcsQ0FBQyxVQUFVLENBQUMsZ0JBQWdCLENBQUMsMkJBQTJCLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQztnQkFDdkcsV0FBVyxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQztnQkFDL0QsaUJBQWlCLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUM7Z0JBQzNFLGtCQUFrQixFQUFFLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDO29CQUNqRSxJQUFJLENBQUMsS0FBSyxDQUFDLGtCQUFrQixDQUFDLENBQUM7b0JBQy9CLGdCQUFnQixDQUFDLGdDQUFnQyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsa0JBQWtCLENBQUM7Z0JBQ2xGLE1BQU0sRUFBRSxHQUFHLENBQUMsc0JBQXNCLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUM7Z0JBQ3JELFlBQVksRUFBRSxHQUFHLENBQUMsc0JBQXNCLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUM7Z0JBQ2pFLGlCQUFpQixFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsT0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGlCQUFpQiwwQ0FBRSxPQUFPLEdBQUc7Z0JBQ3RGLHFCQUFxQixFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLHFCQUFxQixDQUFDO2dCQUNuRixJQUFJLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQztnQkFDakUsaUJBQWlCLEVBQUUsR0FBRyxDQUFDLHVCQUF1QixDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQzthQUN2RTtTQUNGLENBQUM7SUFDSixDQUFDO0lBRUQ7O09BRUc7SUFDSyxzQkFBc0IsQ0FDNUIsSUFBYyxFQUFFLFdBQXNCLEVBQUUsV0FBc0IsRUFDOUQsZUFBMkI7UUFDM0IsTUFBTSxLQUFLLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFakMsTUFBTSxnQkFBZ0IsR0FBRztZQUN2QixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7Z0JBQ3RCLE9BQU8sRUFBRTtvQkFDUCw2QkFBNkI7b0JBQzdCLGtDQUFrQztvQkFDbEMsb0NBQW9DO2lCQUNyQztnQkFDRCxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7YUFDakIsQ0FBQztTQUNILENBQUM7UUFFRixzREFBc0Q7UUFDdEQsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUM1QyxPQUFPLEVBQUUsQ0FBQyxjQUFjLENBQUM7WUFDekIsU0FBUyxFQUFFO2dCQUNULFdBQVcsQ0FBQyxPQUFPO2dCQUNuQixXQUFXLENBQUMsT0FBTzthQUNwQjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBQ0osSUFBSSxlQUFlLEtBQUssU0FBUyxFQUFFO1lBQ2pDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7Z0JBQzVDLE9BQU8sRUFBRSxDQUFDLGNBQWMsQ0FBQztnQkFDekIsU0FBUyxFQUFFLENBQUUsZUFBZSxDQUFDLE9BQU8sQ0FBRTthQUN2QyxDQUFDLENBQUMsQ0FBQztTQUNMO1FBRUQsSUFBSSxJQUFJLENBQUMsa0JBQWtCLEtBQUssR0FBRyxDQUFDLHlCQUF5QixDQUFDLElBQUksRUFBRTtZQUNsRSxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUM1QyxPQUFPLEVBQUUsQ0FBQyxtQkFBbUIsRUFBRSxnQkFBZ0IsRUFBRSxxQkFBcUIsQ0FBQztnQkFDdkUsU0FBUyxFQUFFLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQzt3QkFDMUIsT0FBTyxFQUFFLFFBQVE7d0JBQ2pCLFFBQVEsRUFBRSxNQUFNO3dCQUNoQixZQUFZLEVBQUUsMkNBQTJDO3FCQUMxRCxDQUFDLENBQUM7YUFDSixDQUFDLENBQUMsQ0FBQztTQUNMO1FBRUQsT0FBTyxnQkFBZ0IsQ0FBQztJQUMxQixDQUFDO0lBRUQ7O09BRUc7SUFDSyxpQkFBaUIsQ0FBQyxJQUFjO1FBQ3RDLE9BQU8sSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7WUFDdkMsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLGdDQUFnQyxDQUFDO1lBQ3JFLGVBQWUsRUFBRTtnQkFDZixHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLHlDQUF5QyxDQUFDO2FBQ3RGO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxpQkFBaUIsQ0FBQyxJQUFjO1FBQ3RDLE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQzlDLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxtQkFBbUIsQ0FBQztTQUN6RCxDQUFDLENBQUM7UUFFSCxJQUFJLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLEVBQUU7WUFDbEQsS0FBSyxFQUFFLENBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBRTtZQUN4QixtQkFBbUIsRUFBRSxJQUFJLENBQUMsUUFBUTtTQUNuQyxDQUFDLENBQUM7UUFFSCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7T0FFRztJQUNLLHFCQUFxQixDQUFDLElBQWM7O1FBQzFDLE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLEVBQUU7WUFDakQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLGdDQUFnQyxDQUFDO1lBQ3JFLGVBQWUsRUFBRTtnQkFDZixHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLHVEQUF1RCxDQUFDO2FBQ3BHO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBQSxJQUFJLENBQUMsZ0JBQWdCLDBDQUFFLGFBQWEsQ0FDbEMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3RCLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsVUFBVSxFQUFFO2dCQUNWLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHVDQUF1QyxDQUFDO2FBQ2xFO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLGdCQUFnQjthQUNqQjtTQUNGLENBQUMsRUFDRjtRQUVGLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztDQUNGO0FBaE5ELDRDQWdOQztBQUVELFdBQWlCLGdCQUFnQjtJQUMvQjs7OztPQUlHO0lBQ0gsSUFBWSwyQkFZWDtJQVpELFdBQVksMkJBQTJCO1FBQ3JDOzs7V0FHRztRQUNILHdGQUF5RCxDQUFBO1FBRXpEOzs7V0FHRztRQUNILDRGQUE2RCxDQUFBO0lBQy9ELENBQUMsRUFaVywyQkFBMkIsR0FBM0IsNENBQTJCLEtBQTNCLDRDQUEyQixRQVl0QztJQUVEOzs7O09BSUc7SUFDSCxJQUFZLGdCQWFYO0lBYkQsV0FBWSxnQkFBZ0I7UUFDMUI7O1dBRUc7UUFDSCxxQ0FBaUIsQ0FBQTtRQUNqQjs7V0FFRztRQUNILGlDQUFhLENBQUE7UUFDYjs7V0FFRztRQUNILGlDQUFhLENBQUE7SUFDZixDQUFDLEVBYlcsZ0JBQWdCLEdBQWhCLGlDQUFnQixLQUFoQixpQ0FBZ0IsUUFhM0I7SUFFRDs7OztPQUlHO0lBQ0gsSUFBWSx3QkFhWDtJQWJELFdBQVksd0JBQXdCO1FBQ2xDOztXQUVHO1FBQ0gsdUNBQVcsQ0FBQTtRQUNYOztXQUVHO1FBQ0gsdUNBQVcsQ0FBQTtRQUNYOztXQUVHO1FBQ0gsaURBQXFCLENBQUE7SUFDdkIsQ0FBQyxFQWJXLHdCQUF3QixHQUF4Qix5Q0FBd0IsS0FBeEIseUNBQXdCLFFBYW5DO0lBb0REOzs7O09BSUc7SUFDSCxTQUFnQixrQ0FBa0MsQ0FBQyxRQUFzQzs7UUFDdkYsT0FBTztZQUNMLG1CQUFtQixFQUFFO2dCQUNuQixJQUFJLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLENBQUM7Z0JBQ25FLFFBQVEsRUFBRSxHQUFHLENBQUMsc0JBQXNCLENBQUMsUUFBUSxDQUFDLG1CQUFtQixDQUFDLFFBQVEsQ0FBQztnQkFDM0UsVUFBVSxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsT0FBQyxRQUFRLENBQUMsbUJBQW1CLENBQUMsVUFBVSwwQ0FBRSxPQUFPLEdBQUc7YUFDM0Y7WUFDRCxrQkFBa0IsRUFBRSxHQUFHLENBQUMsc0JBQXNCLENBQUMsUUFBUSxDQUFDLGtCQUFrQixDQUFDO1NBQzVFLENBQUM7SUFDSixDQUFDO0lBVGUsbURBQWtDLHFDQVNqRCxDQUFBO0lBeUJEOzs7O09BSUc7SUFDSCxTQUFnQiw4QkFBOEIsQ0FBQyxRQUFrQztRQUMvRSxPQUFPO1lBQ0wscUJBQXFCLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQztZQUN6RyxZQUFZLEVBQUUsR0FBRyxDQUFDLHVCQUF1QixDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUM7U0FDakUsQ0FBQztJQUNKLENBQUM7SUFMZSwrQ0FBOEIsaUNBSzdDLENBQUE7SUFzREQ7Ozs7T0FJRztJQUNILFNBQWdCLGdDQUFnQyxDQUFDLFFBQW9DOztRQUNuRixPQUFPO1lBQ0wsUUFBUSxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDO1lBQ3ZELG1DQUFtQyxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxRQUFRLENBQUMsbUNBQW1DLENBQUM7WUFDN0csY0FBYyxFQUFFLEdBQUcsQ0FBQyxVQUFVLENBQUMsMkJBQTJCLENBQUMsQ0FBQyxRQUFRLENBQUMsY0FBYyxDQUFDO1lBQ3BGLGdCQUFnQixFQUFFLENBQUMsUUFBUSxDQUFDLGdCQUFnQixLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUM7Z0JBQzNELFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO2dCQUMzQiw4QkFBOEIsQ0FBQyxRQUFRLENBQUMsZ0JBQWdCLENBQUM7WUFDM0QsWUFBWSxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsT0FBQyxRQUFRLENBQUMsWUFBWSwwQ0FBRSxPQUFPLEdBQUc7WUFDMUUsZ0JBQWdCLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQztTQUN4RSxDQUFDO0lBQ0osQ0FBQztJQVhlLGlEQUFnQyxtQ0FXL0MsQ0FBQTtJQUVEOzs7O09BSUc7SUFDSCxJQUFZLGlCQVNYO0lBVEQsV0FBWSxpQkFBaUI7UUFDM0I7O1dBRUc7UUFDSCxnRUFBMkMsQ0FBQTtRQUMzQzs7V0FFRztRQUNILDREQUF1QyxDQUFBO0lBQ3pDLENBQUMsRUFUVyxpQkFBaUIsR0FBakIsa0NBQWlCLEtBQWpCLGtDQUFpQixRQVM1QjtJQTBDRDs7OztPQUlHO0lBQ0gsU0FBZ0IscURBQXFELENBQUMsUUFBeUQ7O1FBQzdILE9BQU87WUFDTCxpQkFBaUIsRUFBRTtnQkFDakIsb0JBQW9CLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxpQkFBaUIsQ0FBQyxvQkFBb0IsQ0FBQztnQkFDakcsYUFBYSxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsT0FBQyxRQUFRLENBQUMsaUJBQWlCLENBQUMsYUFBYSwwQ0FBRSxPQUFPLEdBQUc7Z0JBQzlGLHNCQUFzQixFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxRQUFRLENBQUMsaUJBQWlCLENBQUMsc0JBQXNCLENBQUM7YUFDdEc7U0FDRixDQUFDO0lBQ0osQ0FBQztJQVJlLHNFQUFxRCx3REFRcEUsQ0FBQTtJQW1ERDs7OztPQUlHO0lBQ0gsU0FBZ0IsaUNBQWlDLENBQUMsUUFBcUM7O1FBQ3JGLE9BQU87WUFDTCxpQkFBaUIsRUFBRSxHQUFHLENBQUMsc0JBQXNCLE9BQUMsUUFBUSxDQUFDLGlCQUFpQiwwQ0FBRSxPQUFPLEdBQUc7WUFDcEYsbUJBQW1CLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQztZQUNuRyxvQkFBb0IsRUFBRSxDQUFDLFFBQVEsQ0FBQyxvQkFBb0IsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDO2dCQUNuRSxRQUFRLENBQUMsb0JBQW9CLENBQUMsQ0FBQztnQkFDL0IscURBQXFELENBQUMsUUFBUSxDQUFDLG9CQUFvQixDQUFDO1lBQ3RGLElBQUksRUFBRSxHQUFHLENBQUMsc0JBQXNCLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQztZQUMvQyxzQkFBc0IsRUFBRSxHQUFHLENBQUMsc0JBQXNCLENBQUMsUUFBUSxDQUFDLHNCQUFzQixDQUFDO1lBQ25GLGtCQUFrQixFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxRQUFRLENBQUMsa0JBQWtCLENBQUM7U0FDNUUsQ0FBQztJQUNKLENBQUM7SUFYZSxrREFBaUMsb0NBV2hELENBQUE7SUFFRDs7OztPQUlHO0lBQ0gsSUFBWSxpQ0FpQlg7SUFqQkQsV0FBWSxpQ0FBaUM7UUFDM0M7O1dBRUc7UUFDSCxvRkFBK0MsQ0FBQTtRQUMvQzs7V0FFRztRQUNILGtFQUE2QixDQUFBO1FBQzdCOztXQUVHO1FBQ0gsNERBQXVCLENBQUE7UUFDdkI7O1dBRUc7UUFDSCw4RUFBeUMsQ0FBQTtJQUMzQyxDQUFDLEVBakJXLGlDQUFpQyxHQUFqQyxrREFBaUMsS0FBakMsa0RBQWlDLFFBaUI1QztJQUVEOzs7O09BSUc7SUFDSCxJQUFZLHdCQXFCWDtJQXJCRCxXQUFZLHdCQUF3QjtRQUNsQzs7V0FFRztRQUNILHlEQUE2QixDQUFBO1FBQzdCOztXQUVHO1FBQ0gsK0NBQW1CLENBQUE7UUFDbkI7O1dBRUc7UUFDSCx1Q0FBVyxDQUFBO1FBQ1g7O1dBRUc7UUFDSCwrQ0FBbUIsQ0FBQTtRQUNuQjs7V0FFRztRQUNILCtDQUFtQixDQUFBO0lBQ3JCLENBQUMsRUFyQlcsd0JBQXdCLEdBQXhCLHlDQUF3QixLQUF4Qix5Q0FBd0IsUUFxQm5DO0lBRUQ7Ozs7T0FJRztJQUNILElBQVksbUJBNkdYO0lBN0dELFdBQVksbUJBQW1CO1FBQzdCOztXQUVHO1FBQ0gsb0NBQWEsQ0FBQTtRQUNiOztXQUVHO1FBQ0gsMENBQW1CLENBQUE7UUFDbkI7O1dBRUc7UUFDSCxzREFBK0IsQ0FBQTtRQUMvQjs7V0FFRztRQUNILHNEQUErQixDQUFBO1FBQy9COztXQUVHO1FBQ0gsc0NBQWUsQ0FBQTtRQUNmOztXQUVHO1FBQ0gsZ0RBQXlCLENBQUE7UUFDekI7O1dBRUc7UUFDSCxnREFBeUIsQ0FBQTtRQUN6Qjs7V0FFRztRQUNILGdEQUF5QixDQUFBO1FBQ3pCOztXQUVHO1FBQ0gsZ0RBQXlCLENBQUE7UUFDekI7O1dBRUc7UUFDSCxvQ0FBYSxDQUFBO1FBQ2I7O1dBRUc7UUFDSCw4Q0FBdUIsQ0FBQTtRQUN2Qjs7V0FFRztRQUNILDhDQUF1QixDQUFBO1FBQ3ZCOztXQUVHO1FBQ0gsOENBQXVCLENBQUE7UUFDdkI7O1dBRUc7UUFDSCw4Q0FBdUIsQ0FBQTtRQUN2Qjs7V0FFRztRQUNILDBDQUFtQixDQUFBO1FBQ25COztXQUVHO1FBQ0gsc0NBQWUsQ0FBQTtRQUNmOztXQUVHO1FBQ0gsNERBQXFDLENBQUE7UUFDckM7O1dBRUc7UUFDSCxzRUFBK0MsQ0FBQTtRQUMvQzs7V0FFRztRQUNILHNFQUErQyxDQUFBO1FBQy9DOztXQUVHO1FBQ0gsc0VBQStDLENBQUE7UUFDL0M7O1dBRUc7UUFDSCxzRUFBK0MsQ0FBQTtRQUMvQzs7V0FFRztRQUNILDBEQUFtQyxDQUFBO1FBQ25DOztXQUVHO1FBQ0gsb0VBQTZDLENBQUE7UUFDN0M7O1dBRUc7UUFDSCxvRUFBNkMsQ0FBQTtRQUM3Qzs7V0FFRztRQUNILG9FQUE2QyxDQUFBO1FBQzdDOztXQUVHO1FBQ0gsb0VBQTZDLENBQUE7UUFDN0M7O1dBRUc7UUFDSCw0REFBcUMsQ0FBQTtJQUN2QyxDQUFDLEVBN0dXLG1CQUFtQixHQUFuQixvQ0FBbUIsS0FBbkIsb0NBQW1CLFFBNkc5QjtJQXVCRDs7OztPQUlHO0lBQ0gsU0FBZ0IsNkJBQTZCLENBQUMsUUFBaUM7UUFDN0UsT0FBTztZQUNMLEdBQUcsRUFBRSxHQUFHLENBQUMsc0JBQXNCLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQztZQUM3QyxLQUFLLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUM7U0FDbEQsQ0FBQztJQUNKLENBQUM7SUFMZSw4Q0FBNkIsZ0NBSzVDLENBQUE7SUF3RkQ7Ozs7T0FJRztJQUNILFNBQWdCLDRCQUE0QixDQUFDLFFBQWdDOztRQUMzRSxPQUFPO1lBQ0wseUJBQXlCLEVBQUU7Z0JBQ3pCLGtCQUFrQixFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsT0FBQyxRQUFRLENBQUMseUJBQXlCLENBQUMsa0JBQWtCLDBDQUFFLE9BQU8sR0FBRztnQkFDaEgsVUFBVSxFQUFFLEdBQUcsQ0FBQyxVQUFVLENBQUMsNkJBQTZCLENBQUMsQ0FBQyxRQUFRLENBQUMseUJBQXlCLENBQUMsVUFBVSxDQUFDO2dCQUN4RyxpQkFBaUIsRUFBRSxHQUFHLENBQUMsc0JBQXNCLENBQUMsUUFBUSxDQUFDLHlCQUF5QixDQUFDLGdCQUFnQixDQUFDO2dCQUNsRyxVQUFVLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyx5QkFBeUIsQ0FBQyxVQUFVLENBQUM7Z0JBQ3JGLFNBQVMsRUFBRSxHQUFHLENBQUMsc0JBQXNCLENBQUMsUUFBUSxDQUFDLHlCQUF5QixDQUFDLFNBQVMsQ0FBQztnQkFDbkYsTUFBTSxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxRQUFRLENBQUMseUJBQXlCLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBRSxDQUFDO2dCQUN6RixTQUFTLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixPQUFDLFFBQVEsQ0FBQyx5QkFBeUIsQ0FBQyxTQUFTLDBDQUFFLE9BQU8sR0FBRztnQkFDOUYsU0FBUyxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxRQUFRLENBQUMseUJBQXlCLENBQUMsU0FBUyxDQUFDO2dCQUNuRixJQUFJLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixPQUFDLFFBQVEsQ0FBQyx5QkFBeUIsQ0FBQyxJQUFJLDBDQUFFLE9BQU8sR0FBRzthQUNyRjtTQUNGLENBQUM7SUFDSixDQUFDO0lBZGUsNkNBQTRCLCtCQWMzQyxDQUFBO0lBRUQ7Ozs7T0FJRztJQUNILElBQVksY0FTWDtJQVRELFdBQVksY0FBYztRQUN4Qjs7V0FFRztRQUNILHlDQUF1QixDQUFBO1FBQ3ZCOztXQUVHO1FBQ0gsK0JBQWEsQ0FBQTtJQUNmLENBQUMsRUFUVyxjQUFjLEdBQWQsK0JBQWMsS0FBZCwrQkFBYyxRQVN6QjtJQUVEOzs7O09BSUc7SUFDSCxJQUFZLHFCQWFYO0lBYkQsV0FBWSxxQkFBcUI7UUFDL0I7O1dBRUc7UUFDSCxrRUFBeUMsQ0FBQTtRQUN6Qzs7V0FFRztRQUNILGtGQUF5RCxDQUFBO1FBQ3pEOztXQUVHO1FBQ0gsMERBQWlDLENBQUE7SUFDbkMsQ0FBQyxFQWJXLHFCQUFxQixHQUFyQixzQ0FBcUIsS0FBckIsc0NBQXFCLFFBYWhDO0lBMEREOzs7O09BSUc7SUFDSCxTQUFnQiwyQkFBMkIsQ0FBQyxRQUErQjs7UUFDekUsT0FBTztZQUNMLE1BQU0sRUFBRSxHQUFHLENBQUMsc0JBQXNCLE9BQUMsUUFBUSxDQUFDLE1BQU0sMENBQUUsT0FBTyxHQUFHO1lBQzlELGdDQUFnQyxFQUFFO2dCQUNoQyxjQUFjLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxnQ0FBZ0MsQ0FBQyxjQUFjLENBQUM7Z0JBQ3BHLFFBQVEsRUFBRSxHQUFHLENBQUMsc0JBQXNCLENBQUMsUUFBUSxDQUFDLGdDQUFnQyxDQUFDLFFBQVEsQ0FBQztnQkFDeEYsaUJBQWlCLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxnQ0FBZ0MsQ0FBQyxpQkFBaUIsQ0FBQzthQUMzRztTQUNGLENBQUM7SUFDSixDQUFDO0lBVGUsNENBQTJCLDhCQVMxQyxDQUFBO0lBa0NEOzs7O09BSUc7SUFDSCxTQUFnQix5QkFBeUIsQ0FBQyxRQUE2QjtRQUNyRSxPQUFPO1lBQ0wsTUFBTSxFQUFFLDJCQUEyQixDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUM7WUFDcEQsV0FBVyxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDO1lBQzdELElBQUksRUFBRSxHQUFHLENBQUMsc0JBQXNCLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQztZQUMvQyxPQUFPLEVBQUUsNEJBQTRCLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQztTQUN4RCxDQUFDO0lBQ0osQ0FBQztJQVBlLDBDQUF5Qiw0QkFPeEMsQ0FBQTtJQTRDRDs7OztPQUlHO0lBQ0gsU0FBZ0IsK0JBQStCLENBQUMsUUFBbUM7UUFDakYsT0FBTztZQUNMLFdBQVcsRUFBRTtnQkFDWCxXQUFXLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsV0FBVyxDQUFDO2dCQUN6RSxXQUFXLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsV0FBVyxDQUFDO2FBQzFFO1lBQ0QsS0FBSyxFQUFFLEdBQUcsQ0FBQyxVQUFVLENBQUMseUJBQXlCLENBQUMsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDO1NBQ2pFLENBQUM7SUFDSixDQUFDO0lBUmUsZ0RBQStCLGtDQVE5QyxDQUFBO0lBb0VEOzs7O09BSUc7SUFDSCxTQUFnQixpQ0FBaUMsQ0FBQyxRQUFxQzs7UUFDckYsT0FBTztZQUNMLGlCQUFpQixFQUFFLENBQUMsUUFBUSxDQUFDLGlCQUFpQixLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUM7Z0JBQzdELFFBQVEsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO2dCQUM1QiwrQkFBK0IsQ0FBQyxRQUFRLENBQUMsaUJBQWlCLENBQUM7WUFDN0QsUUFBUSxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDO1lBQ3ZELGNBQWMsRUFBRSxHQUFHLENBQUMsVUFBVSxDQUFDLDJCQUEyQixDQUFDLENBQUMsUUFBUSxDQUFDLGNBQWMsQ0FBQztZQUNwRixnQkFBZ0IsRUFBRSxDQUFDLFFBQVEsQ0FBQyxnQkFBZ0IsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDO2dCQUMzRCxRQUFRLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztnQkFDM0IsOEJBQThCLENBQUMsUUFBUSxDQUFDLGdCQUFnQixDQUFDO1lBQzNELGFBQWEsRUFBRSxHQUFHLENBQUMsc0JBQXNCLENBQUMsUUFBUSxDQUFDLGFBQWEsQ0FBQztZQUNqRSxZQUFZLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixPQUFDLFFBQVEsQ0FBQyxZQUFZLDBDQUFFLE9BQU8sR0FBRztZQUMxRSxZQUFZLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUM7WUFDL0QsTUFBTSxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsT0FBQyxRQUFRLENBQUMsTUFBTSwwQ0FBRSxPQUFPLEdBQUc7WUFDOUQsSUFBSSxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDO1NBQ2hELENBQUM7SUFDSixDQUFDO0lBaEJlLGtEQUFpQyxvQ0FnQmhELENBQUE7SUEyQkQ7Ozs7T0FJRztJQUNILFNBQWdCLDJCQUEyQixDQUFDLFFBQStCO1FBQ3pFLE9BQU87WUFDTCxnQkFBZ0IsRUFBRSxHQUFHLENBQUMsc0JBQXNCLENBQUMsUUFBUSxDQUFDLGdCQUFnQixDQUFDO1lBQ3ZFLGlCQUFpQixFQUFFLEdBQUcsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUMsUUFBUSxDQUFDLGlCQUFpQixDQUFDO1NBQzFGLENBQUM7SUFDSixDQUFDO0lBTGUsNENBQTJCLDhCQUsxQyxDQUFBO0lBZ0lEOzs7O09BSUc7SUFDSCxTQUFnQiw2QkFBNkIsQ0FBQyxRQUFpQztRQUM3RSxPQUFPO1lBQ0wsOEJBQThCLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQyxRQUFRLENBQUMsOEJBQThCLENBQUM7WUFDbkgsNkJBQTZCLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQyxRQUFRLENBQUMsNkJBQTZCLENBQUM7WUFDakgsVUFBVSxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDO1lBQzNELFdBQVcsRUFBRSxHQUFHLENBQUMsc0JBQXNCLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQztZQUM3RCxZQUFZLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDO1lBQy9FLDZCQUE2QixFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxRQUFRLENBQUMsNkJBQTZCLENBQUM7WUFDakcsNEJBQTRCLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyw0QkFBNEIsQ0FBQztZQUMvRixhQUFhLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxhQUFhLENBQUM7WUFDakUsYUFBYSxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxRQUFRLENBQUMsYUFBYSxDQUFDO1lBQ2pFLGNBQWMsRUFBRSxHQUFHLENBQUMsVUFBVSxDQUFDLGlDQUFpQyxDQUFDLENBQUMsUUFBUSxDQUFDLGNBQWMsQ0FBQztZQUMxRixjQUFjLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxjQUFjLENBQUM7WUFDMUYsMkJBQTJCLEVBQUUsSUFBSTtZQUNqQyxrQkFBa0IsRUFBRSxHQUFHLENBQUMsc0JBQXNCLENBQUMsUUFBUSxDQUFDLGtCQUFrQixDQUFDO1lBQzNFLFNBQVMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxTQUFTLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQztnQkFDN0MsUUFBUSxDQUFDLFNBQVMsQ0FBQyxDQUFDO2dCQUNwQiwyQkFBMkIsQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDO1lBQ2pELDBCQUEwQixFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxRQUFRLENBQUMsMEJBQTBCLENBQUM7WUFDM0YsaUJBQWlCLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxpQkFBaUIsQ0FBQztZQUN6RSxvQkFBb0IsRUFBRSxHQUFHLENBQUMsdUJBQXVCLENBQUMsUUFBUSxDQUFDLG9CQUFvQixDQUFDO1NBQ2pGLENBQUM7SUFDSixDQUFDO0lBdEJlLDhDQUE2QixnQ0FzQjVDLENBQUE7SUEyQ0Q7Ozs7T0FJRztJQUNILFNBQWdCLCtCQUErQixDQUFDLFFBQW1DO1FBQ2pGLE9BQU87WUFDTCxJQUFJLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUM7WUFDL0MsSUFBSSxFQUFFLEdBQUcsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQztZQUMvRCxPQUFPLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUM7WUFDckQsY0FBYyxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxRQUFRLENBQUMsY0FBYyxDQUFDO1NBQ3BFLENBQUM7SUFDSixDQUFDO0lBUGUsZ0RBQStCLGtDQU85QyxDQUFBO0lBNENEOzs7O09BSUc7SUFDSCxTQUFnQiwyQkFBMkIsQ0FBQyxRQUF1QztRQUNqRixPQUFPO1lBQ0wsSUFBSSxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDO1lBQy9DLHFCQUFxQixFQUFFO2dCQUNyQixJQUFJLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLENBQUM7Z0JBQ3JFLElBQUksRUFBRSxHQUFHLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLENBQUM7YUFDdEY7U0FDRixDQUFDO0lBQ0osQ0FBQztJQVJlLDRDQUEyQiw4QkFRMUMsQ0FBQTtJQW1DRDs7OztPQUlHO0lBQ0gsU0FBZ0IsMkJBQTJCLENBQUMsUUFBK0I7UUFDekUsT0FBTztZQUNMLGNBQWMsRUFBRSxHQUFHLENBQUMsc0JBQXNCLENBQUMsUUFBUSxDQUFDLGNBQWMsQ0FBQztZQUNuRSxVQUFVLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUM7WUFDM0QsY0FBYyxFQUFFLEdBQUcsQ0FBQyxVQUFVLENBQUMsMkJBQTJCLENBQUMsQ0FBQyxRQUFRLENBQUMsY0FBYyxDQUFDO1NBQ3JGLENBQUM7SUFDSixDQUFDO0lBTmUsNENBQTJCLDhCQU0xQyxDQUFBO0lBaUREOzs7O09BSUc7SUFDSCxTQUFnQixnQ0FBZ0MsQ0FBQyxRQUFvQztRQUNuRixPQUFPO1lBQ0wsb0JBQW9CLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQztZQUMvRSxnQkFBZ0IsRUFBRSxHQUFHLENBQUMsc0JBQXNCLENBQUMsUUFBUSxDQUFDLGdCQUFnQixDQUFDO1lBQ3ZFLGdDQUFnQyxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxRQUFRLENBQUMsZ0NBQWdDLENBQUM7WUFDdkcsZ0JBQWdCLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQztZQUN2RSxLQUFLLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUM7U0FDbEQsQ0FBQztJQUNKLENBQUM7SUFSZSxpREFBZ0MsbUNBUS9DLENBQUE7QUFDSCxDQUFDLEVBNTJDZ0IsZ0JBQWdCLEdBQWhCLHdCQUFnQixLQUFoQix3QkFBZ0IsUUE0MkNoQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIHNmbiBmcm9tICdAYXdzLWNkay9hd3Mtc3RlcGZ1bmN0aW9ucyc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBnZXRSZXNvdXJjZUFybiB9IGZyb20gJy4vcmVzb3VyY2UtYXJuLXN1ZmZpeCc7XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgRW1yQ3JlYXRlQ2x1c3RlclxuICpcbiAqIFNlZSB0aGUgUnVuSm9iRmxvdyBBUEkgZm9yIGNvbXBsZXRlIGRvY3VtZW50YXRpb24gb24gaW5wdXQgcGFyYW1ldGVyc1xuICpcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SdW5Kb2JGbG93Lmh0bWxcbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRW1yQ3JlYXRlQ2x1c3RlclByb3BzIHtcbiAgLyoqXG4gICAqIEEgc3BlY2lmaWNhdGlvbiBvZiB0aGUgbnVtYmVyIGFuZCB0eXBlIG9mIEFtYXpvbiBFQzIgaW5zdGFuY2VzLlxuICAgKi9cbiAgcmVhZG9ubHkgaW5zdGFuY2VzOiBFbXJDcmVhdGVDbHVzdGVyLkluc3RhbmNlc0NvbmZpZ1Byb3BlcnR5O1xuXG4gIC8qKlxuICAgKiBBbHNvIGNhbGxlZCBpbnN0YW5jZSBwcm9maWxlIGFuZCBFQzIgcm9sZS4gQW4gSUFNIHJvbGUgZm9yIGFuIEVNUiBjbHVzdGVyLiBUaGUgRUMyIGluc3RhbmNlcyBvZiB0aGUgY2x1c3RlciBhc3N1bWUgdGhpcyByb2xlLlxuICAgKlxuICAgKiBUaGlzIGF0dHJpYnV0ZSBoYXMgYmVlbiByZW5hbWVkIGZyb20gam9iRmxvd1JvbGUgdG8gY2x1c3RlclJvbGUgdG8gYWxpZ24gd2l0aCBvdGhlciBFUk0vU3RlcEZ1bmN0aW9uIGludGVncmF0aW9uIHBhcmFtZXRlcnMuXG4gICAqIEEgUm9sZSB3aWxsIGJlIGNyZWF0ZWQgaWYgb25lIGlzIG5vdCBwcm92aWRlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgTm8gY2x1c3RlclJvbGVcbiAgICovXG4gIHJlYWRvbmx5IGNsdXN0ZXJSb2xlPzogaWFtLklSb2xlO1xuXG4gIC8qKlxuICAgKiBUaGUgTmFtZSBvZiB0aGUgQ2x1c3RlclxuICAgKi9cbiAgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgSUFNIHJvbGUgdGhhdCB3aWxsIGJlIGFzc3VtZWQgYnkgdGhlIEFtYXpvbiBFTVIgc2VydmljZSB0byBhY2Nlc3MgQVdTIHJlc291cmNlcyBvbiB5b3VyIGJlaGFsZi4gQSBSb2xlIHdpbGwgYmUgY3JlYXRlZCBpZlxuICAgKiBvbmUgaXMgbm90IHByb3ZpZGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBObyBzZXJ2aWNlUm9sZVxuICAgKi9cbiAgcmVhZG9ubHkgc2VydmljZVJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgLyoqXG4gICAqIEEgSlNPTiBzdHJpbmcgZm9yIHNlbGVjdGluZyBhZGRpdGlvbmFsIGZlYXR1cmVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBObyBhZGRpdGlvbmFsSW5mb1xuICAgKi9cbiAgcmVhZG9ubHkgYWRkaXRpb25hbEluZm8/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEEgY2FzZS1pbnNlbnNpdGl2ZSBsaXN0IG9mIGFwcGxpY2F0aW9ucyBmb3IgQW1hem9uIEVNUiB0byBpbnN0YWxsIGFuZCBjb25maWd1cmUgd2hlbiBsYXVuY2hpbmcgdGhlIGNsdXN0ZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IEVNUiBzZWxlY3RlZCBkZWZhdWx0XG4gICAqL1xuICByZWFkb25seSBhcHBsaWNhdGlvbnM/OiBFbXJDcmVhdGVDbHVzdGVyLkFwcGxpY2F0aW9uQ29uZmlnUHJvcGVydHlbXTtcblxuICAvKipcbiAgICogQW4gSUFNIHJvbGUgZm9yIGF1dG9tYXRpYyBzY2FsaW5nIHBvbGljaWVzLiBBIFJvbGUgd2lsbCBiZSBjcmVhdGVkIGlmIG9uZSBpcyBub3QgcHJvdmlkZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IE5vIGF1dG9TY2FsaW5nUm9sZVxuICAgKi9cbiAgcmVhZG9ubHkgYXV0b1NjYWxpbmdSb2xlPzogaWFtLklSb2xlO1xuXG4gIC8qKlxuICAgKiBBIGxpc3Qgb2YgYm9vdHN0cmFwIGFjdGlvbnMgdG8gcnVuIGJlZm9yZSBIYWRvb3Agc3RhcnRzIG9uIHRoZSBjbHVzdGVyIG5vZGVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBObyBib290c3RyYXBBY3Rpb25zXG4gICAqL1xuICByZWFkb25seSBib290c3RyYXBBY3Rpb25zPzogRW1yQ3JlYXRlQ2x1c3Rlci5Cb290c3RyYXBBY3Rpb25Db25maWdQcm9wZXJ0eVtdO1xuXG4gIC8qKlxuICAgKiBUaGUgbGlzdCBvZiBjb25maWd1cmF0aW9ucyBzdXBwbGllZCBmb3IgdGhlIEVNUiBjbHVzdGVyIHlvdSBhcmUgY3JlYXRpbmcuXG4gICAqXG4gICAqIEBkZWZhdWx0IE5vIGNvbmZpZ3VyYXRpb25zXG4gICAqL1xuICByZWFkb25seSBjb25maWd1cmF0aW9ucz86IEVtckNyZWF0ZUNsdXN0ZXIuQ29uZmlndXJhdGlvblByb3BlcnR5W107XG5cbiAgLyoqXG4gICAqIFRoZSBJRCBvZiBhIGN1c3RvbSBBbWF6b24gRUJTLWJhY2tlZCBMaW51eCBBTUkuXG4gICAqXG4gICAqIEBkZWZhdWx0IE5vIGN1c3RvbUFtaUlkXG4gICAqL1xuICByZWFkb25seSBjdXN0b21BbWlJZD86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHNpemUsIGluIEdpQiwgb2YgdGhlIEVCUyByb290IGRldmljZSB2b2x1bWUgb2YgdGhlIExpbnV4IEFNSSB0aGF0IGlzIHVzZWQgZm9yIGVhY2ggRUMyIGluc3RhbmNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBFTVIgc2VsZWN0ZWQgZGVmYXVsdFxuICAgKi9cbiAgcmVhZG9ubHkgZWJzUm9vdFZvbHVtZVNpemU/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIEF0dHJpYnV0ZXMgZm9yIEtlcmJlcm9zIGNvbmZpZ3VyYXRpb24gd2hlbiBLZXJiZXJvcyBhdXRoZW50aWNhdGlvbiBpcyBlbmFibGVkIHVzaW5nIGEgc2VjdXJpdHkgY29uZmlndXJhdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgTm8ga2VyYmVyb3NBdHRyaWJ1dGVzXG4gICAqL1xuICByZWFkb25seSBrZXJiZXJvc0F0dHJpYnV0ZXM/OiBFbXJDcmVhdGVDbHVzdGVyLktlcmJlcm9zQXR0cmlidXRlc1Byb3BlcnR5O1xuXG4gIC8qKlxuICAgKiBUaGUgbG9jYXRpb24gaW4gQW1hem9uIFMzIHRvIHdyaXRlIHRoZSBsb2cgZmlsZXMgb2YgdGhlIGpvYiBmbG93LlxuICAgKlxuICAgKiBAZGVmYXVsdCBObyBsb2dVcmlcbiAgICovXG4gIHJlYWRvbmx5IGxvZ1VyaT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIEFtYXpvbiBFTVIgcmVsZWFzZSBsYWJlbCwgd2hpY2ggZGV0ZXJtaW5lcyB0aGUgdmVyc2lvbiBvZiBvcGVuLXNvdXJjZSBhcHBsaWNhdGlvbiBwYWNrYWdlcyBpbnN0YWxsZWQgb24gdGhlIGNsdXN0ZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IEVNUiBzZWxlY3RlZCBkZWZhdWx0XG4gICAqL1xuICByZWFkb25seSByZWxlYXNlTGFiZWw/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFNwZWNpZmllcyB0aGUgd2F5IHRoYXQgaW5kaXZpZHVhbCBBbWF6b24gRUMyIGluc3RhbmNlcyB0ZXJtaW5hdGUgd2hlbiBhbiBhdXRvbWF0aWMgc2NhbGUtaW4gYWN0aXZpdHkgb2NjdXJzIG9yIGFuIGluc3RhbmNlIGdyb3VwIGlzIHJlc2l6ZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IEVNUiBzZWxlY3RlZCBkZWZhdWx0XG4gICAqL1xuICByZWFkb25seSBzY2FsZURvd25CZWhhdmlvcj86IEVtckNyZWF0ZUNsdXN0ZXIuRW1yQ2x1c3RlclNjYWxlRG93bkJlaGF2aW9yO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiBhIHNlY3VyaXR5IGNvbmZpZ3VyYXRpb24gdG8gYXBwbHkgdG8gdGhlIGNsdXN0ZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IE5vIHNlY3VyaXR5Q29uZmlndXJhdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgc2VjdXJpdHlDb25maWd1cmF0aW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBIGxpc3Qgb2YgdGFncyB0byBhc3NvY2lhdGUgd2l0aCBhIGNsdXN0ZXIgYW5kIHByb3BhZ2F0ZSB0byBBbWF6b24gRUMyIGluc3RhbmNlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgTm8gVGFnc1xuICAgKi9cbiAgcmVhZG9ubHkgdGFncz86IGNkay5DZm5UYWdbXTtcblxuICAvKipcbiAgICogQSB2YWx1ZSBvZiB0cnVlIGluZGljYXRlcyB0aGF0IGFsbCBJQU0gdXNlcnMgaW4gdGhlIEFXUyBhY2NvdW50IGNhbiBwZXJmb3JtIGNsdXN0ZXIgYWN0aW9ucyBpZiB0aGV5IGhhdmUgdGhlIHByb3BlciBJQU0gcG9saWN5IHBlcm1pc3Npb25zLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSB2aXNpYmxlVG9BbGxVc2Vycz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBzZXJ2aWNlIGludGVncmF0aW9uIHBhdHRlcm4gaW5kaWNhdGVzIGRpZmZlcmVudCB3YXlzIHRvIGNhbGwgQ3JlYXRlQ2x1c3Rlci5cbiAgICpcbiAgICogVGhlIHZhbGlkIHZhbHVlIGlzIGVpdGhlciBGSVJFX0FORF9GT1JHRVQgb3IgU1lOQy5cbiAgICpcbiAgICogQGRlZmF1bHQgU1lOQ1xuICAgKi9cbiAgcmVhZG9ubHkgaW50ZWdyYXRpb25QYXR0ZXJuPzogc2ZuLlNlcnZpY2VJbnRlZ3JhdGlvblBhdHRlcm47XG59XG5cbi8qKlxuICogQSBTdGVwIEZ1bmN0aW9ucyBUYXNrIHRvIGNyZWF0ZSBhbiBFTVIgQ2x1c3Rlci5cbiAqXG4gKiBUaGUgQ2x1c3RlckNvbmZpZ3VyYXRpb24gaXMgZGVmaW5lZCBhcyBQYXJhbWV0ZXJzIGluIHRoZSBzdGF0ZSBtYWNoaW5lIGRlZmluaXRpb24uXG4gKlxuICogT1VUUFVUOiB0aGUgQ2x1c3RlcklkLlxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGNsYXNzIEVtckNyZWF0ZUNsdXN0ZXIgaW1wbGVtZW50cyBzZm4uSVN0ZXBGdW5jdGlvbnNUYXNrIHtcblxuICBwcml2YXRlIHJlYWRvbmx5IHZpc2libGVUb0FsbFVzZXJzOiBib29sZWFuO1xuICBwcml2YXRlIHJlYWRvbmx5IGludGVncmF0aW9uUGF0dGVybjogc2ZuLlNlcnZpY2VJbnRlZ3JhdGlvblBhdHRlcm47XG5cbiAgcHJpdmF0ZSBfc2VydmljZVJvbGU/OiBpYW0uSVJvbGU7XG4gIHByaXZhdGUgX2NsdXN0ZXJSb2xlPzogaWFtLklSb2xlO1xuICBwcml2YXRlIF9hdXRvU2NhbGluZ1JvbGU/OiBpYW0uSVJvbGU7XG5cbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBwcm9wczogRW1yQ3JlYXRlQ2x1c3RlclByb3BzKSB7XG4gICAgdGhpcy52aXNpYmxlVG9BbGxVc2VycyA9ICh0aGlzLnByb3BzLnZpc2libGVUb0FsbFVzZXJzICE9PSB1bmRlZmluZWQpID8gdGhpcy5wcm9wcy52aXNpYmxlVG9BbGxVc2VycyA6IHRydWU7XG4gICAgdGhpcy5pbnRlZ3JhdGlvblBhdHRlcm4gPSBwcm9wcy5pbnRlZ3JhdGlvblBhdHRlcm4gfHwgc2ZuLlNlcnZpY2VJbnRlZ3JhdGlvblBhdHRlcm4uU1lOQztcblxuICAgIHRoaXMuX3NlcnZpY2VSb2xlID0gdGhpcy5wcm9wcy5zZXJ2aWNlUm9sZTtcbiAgICB0aGlzLl9jbHVzdGVyUm9sZSA9IHRoaXMucHJvcHMuY2x1c3RlclJvbGU7XG4gICAgdGhpcy5fYXV0b1NjYWxpbmdSb2xlID0gdGhpcy5wcm9wcy5hdXRvU2NhbGluZ1JvbGU7XG5cbiAgICBjb25zdCBzdXBwb3J0ZWRQYXR0ZXJucyA9IFtcbiAgICAgIHNmbi5TZXJ2aWNlSW50ZWdyYXRpb25QYXR0ZXJuLkZJUkVfQU5EX0ZPUkdFVCxcbiAgICAgIHNmbi5TZXJ2aWNlSW50ZWdyYXRpb25QYXR0ZXJuLlNZTkNcbiAgICBdO1xuXG4gICAgaWYgKCFzdXBwb3J0ZWRQYXR0ZXJucy5pbmNsdWRlcyh0aGlzLmludGVncmF0aW9uUGF0dGVybikpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgSW52YWxpZCBTZXJ2aWNlIEludGVncmF0aW9uIFBhdHRlcm46ICR7dGhpcy5pbnRlZ3JhdGlvblBhdHRlcm59IGlzIG5vdCBzdXBwb3J0ZWQgdG8gY2FsbCBDcmVhdGVDbHVzdGVyLmApO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgc2VydmljZSByb2xlIGZvciB0aGUgRU1SIENsdXN0ZXIuXG4gICAqXG4gICAqIE9ubHkgYXZhaWxhYmxlIGFmdGVyIHRhc2sgaGFzIGJlZW4gYWRkZWQgdG8gYSBzdGF0ZSBtYWNoaW5lLlxuICAgKi9cbiAgcHVibGljIGdldCBzZXJ2aWNlUm9sZSgpOiBpYW0uSVJvbGUge1xuICAgIGlmICh0aGlzLl9zZXJ2aWNlUm9sZSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3JvbGUgbm90IGF2YWlsYWJsZSB5ZXQtLXVzZSB0aGUgb2JqZWN0IGluIGEgVGFzayBmaXJzdCcpO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcy5fc2VydmljZVJvbGU7XG4gIH1cblxuICAvKipcbiAgICogVGhlIGluc3RhbmNlIHJvbGUgZm9yIHRoZSBFTVIgQ2x1c3Rlci5cbiAgICpcbiAgICogT25seSBhdmFpbGFibGUgYWZ0ZXIgdGFzayBoYXMgYmVlbiBhZGRlZCB0byBhIHN0YXRlIG1hY2hpbmUuXG4gICAqL1xuICBwdWJsaWMgZ2V0IGNsdXN0ZXJSb2xlKCk6IGlhbS5JUm9sZSB7XG4gICAgaWYgKHRoaXMuX2NsdXN0ZXJSb2xlID09PSB1bmRlZmluZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcigncm9sZSBub3QgYXZhaWxhYmxlIHlldC0tdXNlIHRoZSBvYmplY3QgaW4gYSBUYXNrIGZpcnN0Jyk7XG4gICAgfVxuICAgIHJldHVybiB0aGlzLl9jbHVzdGVyUm9sZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgYXV0b3NjYWxpbmcgcm9sZSBmb3IgdGhlIEVNUiBDbHVzdGVyLlxuICAgKlxuICAgKiBPbmx5IGF2YWlsYWJsZSBhZnRlciB0YXNrIGhhcyBiZWVuIGFkZGVkIHRvIGEgc3RhdGUgbWFjaGluZS5cbiAgICovXG4gIHB1YmxpYyBnZXQgYXV0b1NjYWxpbmdSb2xlKCk6IGlhbS5JUm9sZSB7XG4gICAgaWYgKHRoaXMuX2F1dG9TY2FsaW5nUm9sZSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3JvbGUgbm90IGF2YWlsYWJsZSB5ZXQtLXVzZSB0aGUgb2JqZWN0IGluIGEgVGFzayBmaXJzdCcpO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcy5fYXV0b1NjYWxpbmdSb2xlO1xuICB9XG5cbiAgcHVibGljIGJpbmQodGFzazogc2ZuLlRhc2spOiBzZm4uU3RlcEZ1bmN0aW9uc1Rhc2tDb25maWcge1xuICAgIC8vIElmIHRoZSBSb2xlcyBhcmUgdW5kZWZpbmVkIHRoZW4gdGhleSB3ZXJlbid0IHByb3ZpZGVkLCBzbyBjcmVhdGUgdGhlbVxuICAgIHRoaXMuX3NlcnZpY2VSb2xlID0gdGhpcy5fc2VydmljZVJvbGUgfHwgdGhpcy5jcmVhdGVTZXJ2aWNlUm9sZSh0YXNrKTtcbiAgICB0aGlzLl9jbHVzdGVyUm9sZSA9IHRoaXMuX2NsdXN0ZXJSb2xlIHx8IHRoaXMuY3JlYXRlQ2x1c3RlclJvbGUodGFzayk7XG5cbiAgICAvLyBBdXRvU2NhbGluZyByb2xlcyBhcmUgbm90IHZhbGlkIHdpdGggSW5zdGFuY2VGbGVldCBjbHVzdGVycy5cbiAgICAvLyBBdHRlbXB0IHRvIGNyZWF0ZSBvbmx5IGlmIC5pbnN0YW5jZXMuaW5zdGFuY2VGbGVldHMgaXMgdW5kZWZpbmVkIG9yIGVtcHR5XG4gICAgaWYgKHRoaXMucHJvcHMuaW5zdGFuY2VzLmluc3RhbmNlRmxlZXRzID09PSB1bmRlZmluZWQgfHwgdGhpcy5wcm9wcy5pbnN0YW5jZXMuaW5zdGFuY2VGbGVldHMubGVuZ3RoID09PSAwKSB7XG4gICAgICB0aGlzLl9hdXRvU2NhbGluZ1JvbGUgPSB0aGlzLl9hdXRvU2NhbGluZ1JvbGUgfHwgdGhpcy5jcmVhdGVBdXRvU2NhbGluZ1JvbGUodGFzayk7XG4gICAgLy8gSWYgSW5zdGFuY2VGbGVldHMgYXJlIHVzZWQgYW5kIGFuIEF1dG9TY2FsaW5nIFJvbGUgaXMgc3BlY2lmaWVkLCB0aHJvdyBhbiBlcnJvclxuICAgIH0gZWxzZSBpZiAodGhpcy5fYXV0b1NjYWxpbmdSb2xlICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQXV0byBTY2FsaW5nIHJvbGVzIGNhbiBub3QgYmUgc3BlY2lmaWVkIHdpdGggaW5zdGFuY2UgZmxlZXRzLicpO1xuICAgIH1cblxuICAgIHJldHVybiB7XG4gICAgICByZXNvdXJjZUFybjogZ2V0UmVzb3VyY2VBcm4oJ2VsYXN0aWNtYXByZWR1Y2UnLCAnY3JlYXRlQ2x1c3RlcicsIHRoaXMuaW50ZWdyYXRpb25QYXR0ZXJuKSxcbiAgICAgIHBvbGljeVN0YXRlbWVudHM6IHRoaXMuY3JlYXRlUG9saWN5U3RhdGVtZW50cyh0YXNrLCB0aGlzLl9zZXJ2aWNlUm9sZSwgdGhpcy5fY2x1c3RlclJvbGUsIHRoaXMuX2F1dG9TY2FsaW5nUm9sZSksXG4gICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgIEluc3RhbmNlczogRW1yQ3JlYXRlQ2x1c3Rlci5JbnN0YW5jZXNDb25maWdQcm9wZXJ0eVRvSnNvbih0aGlzLnByb3BzLmluc3RhbmNlcyksXG4gICAgICAgIEpvYkZsb3dSb2xlOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbih0aGlzLl9jbHVzdGVyUm9sZS5yb2xlTmFtZSksXG4gICAgICAgIE5hbWU6IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHRoaXMucHJvcHMubmFtZSksXG4gICAgICAgIFNlcnZpY2VSb2xlOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbih0aGlzLl9zZXJ2aWNlUm9sZS5yb2xlTmFtZSksXG4gICAgICAgIEFkZGl0aW9uYWxJbmZvOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbih0aGlzLnByb3BzLmFkZGl0aW9uYWxJbmZvKSxcbiAgICAgICAgQXBwbGljYXRpb25zOiBjZGsubGlzdE1hcHBlcihFbXJDcmVhdGVDbHVzdGVyLkFwcGxpY2F0aW9uQ29uZmlnUHJvcGVydHlUb0pzb24pKHRoaXMucHJvcHMuYXBwbGljYXRpb25zKSxcbiAgICAgICAgQXV0b1NjYWxpbmdSb2xlOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbih0aGlzLl9hdXRvU2NhbGluZ1JvbGU/LnJvbGVOYW1lKSxcbiAgICAgICAgQm9vdHN0cmFwQWN0aW9uczogY2RrLmxpc3RNYXBwZXIoRW1yQ3JlYXRlQ2x1c3Rlci5Cb290c3RyYXBBY3Rpb25Db25maWdUb0pzb24pKHRoaXMucHJvcHMuYm9vdHN0cmFwQWN0aW9ucyksXG4gICAgICAgIENvbmZpZ3VyYXRpb25zOiBjZGsubGlzdE1hcHBlcihFbXJDcmVhdGVDbHVzdGVyLkNvbmZpZ3VyYXRpb25Qcm9wZXJ0eVRvSnNvbikodGhpcy5wcm9wcy5jb25maWd1cmF0aW9ucyksXG4gICAgICAgIEN1c3RvbUFtaUlkOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbih0aGlzLnByb3BzLmN1c3RvbUFtaUlkKSxcbiAgICAgICAgRWJzUm9vdFZvbHVtZVNpemU6IGNkay5udW1iZXJUb0Nsb3VkRm9ybWF0aW9uKHRoaXMucHJvcHMuZWJzUm9vdFZvbHVtZVNpemUpLFxuICAgICAgICBLZXJiZXJvc0F0dHJpYnV0ZXM6ICh0aGlzLnByb3BzLmtlcmJlcm9zQXR0cmlidXRlcyA9PT0gdW5kZWZpbmVkKSA/XG4gICAgICAgICAgdGhpcy5wcm9wcy5rZXJiZXJvc0F0dHJpYnV0ZXMgOlxuICAgICAgICAgIEVtckNyZWF0ZUNsdXN0ZXIuS2VyYmVyb3NBdHRyaWJ1dGVzUHJvcGVydHlUb0pzb24odGhpcy5wcm9wcy5rZXJiZXJvc0F0dHJpYnV0ZXMpLFxuICAgICAgICBMb2dVcmk6IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHRoaXMucHJvcHMubG9nVXJpKSxcbiAgICAgICAgUmVsZWFzZUxhYmVsOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbih0aGlzLnByb3BzLnJlbGVhc2VMYWJlbCksXG4gICAgICAgIFNjYWxlRG93bkJlaGF2aW9yOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbih0aGlzLnByb3BzLnNjYWxlRG93bkJlaGF2aW9yPy52YWx1ZU9mKCkpLFxuICAgICAgICBTZWN1cml0eUNvbmZpZ3VyYXRpb246IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHRoaXMucHJvcHMuc2VjdXJpdHlDb25maWd1cmF0aW9uKSxcbiAgICAgICAgVGFnczogY2RrLmxpc3RNYXBwZXIoY2RrLmNmblRhZ1RvQ2xvdWRGb3JtYXRpb24pKHRoaXMucHJvcHMudGFncyksXG4gICAgICAgIFZpc2libGVUb0FsbFVzZXJzOiBjZGsuYm9vbGVhblRvQ2xvdWRGb3JtYXRpb24odGhpcy52aXNpYmxlVG9BbGxVc2VycylcbiAgICAgIH1cbiAgICB9O1xuICB9XG5cbiAgLyoqXG4gICAqIFRoaXMgZ2VuZXJhdGVzIHRoZSBQb2xpY3lTdGF0ZW1lbnRzIHJlcXVpcmVkIGJ5IHRoZSBUYXNrIHRvIGNhbGwgQ3JlYXRlQ2x1c3Rlci5cbiAgICovXG4gIHByaXZhdGUgY3JlYXRlUG9saWN5U3RhdGVtZW50cyhcbiAgICB0YXNrOiBzZm4uVGFzaywgc2VydmljZVJvbGU6IGlhbS5JUm9sZSwgY2x1c3RlclJvbGU6IGlhbS5JUm9sZSxcbiAgICBhdXRvU2NhbGluZ1JvbGU/OiBpYW0uSVJvbGUpOiBpYW0uUG9saWN5U3RhdGVtZW50W10ge1xuICAgIGNvbnN0IHN0YWNrID0gY2RrLlN0YWNrLm9mKHRhc2spO1xuXG4gICAgY29uc3QgcG9saWN5U3RhdGVtZW50cyA9IFtcbiAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICdlbGFzdGljbWFwcmVkdWNlOlJ1bkpvYkZsb3cnLFxuICAgICAgICAgICdlbGFzdGljbWFwcmVkdWNlOkRlc2NyaWJlQ2x1c3RlcicsXG4gICAgICAgICAgJ2VsYXN0aWNtYXByZWR1Y2U6VGVybWluYXRlSm9iRmxvd3MnXG4gICAgICAgIF0sXG4gICAgICAgIHJlc291cmNlczogWycqJ11cbiAgICAgIH0pXG4gICAgXTtcblxuICAgIC8vIEFsbG93IHRoZSBTdGF0ZU1hY2hpbmUgdG8gUGFzc1JvbGUgdG8gQ2x1c3RlciByb2xlc1xuICAgIHBvbGljeVN0YXRlbWVudHMucHVzaChuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBhY3Rpb25zOiBbJ2lhbTpQYXNzUm9sZSddLFxuICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgIHNlcnZpY2VSb2xlLnJvbGVBcm4sXG4gICAgICAgIGNsdXN0ZXJSb2xlLnJvbGVBcm5cbiAgICAgIF1cbiAgICB9KSk7XG4gICAgaWYgKGF1dG9TY2FsaW5nUm9sZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICBwb2xpY3lTdGF0ZW1lbnRzLnB1c2gobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBhY3Rpb25zOiBbJ2lhbTpQYXNzUm9sZSddLFxuICAgICAgICByZXNvdXJjZXM6IFsgYXV0b1NjYWxpbmdSb2xlLnJvbGVBcm4gXVxuICAgICAgfSkpO1xuICAgIH1cblxuICAgIGlmICh0aGlzLmludGVncmF0aW9uUGF0dGVybiA9PT0gc2ZuLlNlcnZpY2VJbnRlZ3JhdGlvblBhdHRlcm4uU1lOQykge1xuICAgICAgcG9saWN5U3RhdGVtZW50cy5wdXNoKG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgYWN0aW9uczogWydldmVudHM6UHV0VGFyZ2V0cycsICdldmVudHM6UHV0UnVsZScsICdldmVudHM6RGVzY3JpYmVSdWxlJ10sXG4gICAgICAgIHJlc291cmNlczogW3N0YWNrLmZvcm1hdEFybih7XG4gICAgICAgICAgc2VydmljZTogJ2V2ZW50cycsXG4gICAgICAgICAgcmVzb3VyY2U6ICdydWxlJyxcbiAgICAgICAgICByZXNvdXJjZU5hbWU6ICdTdGVwRnVuY3Rpb25zR2V0RXZlbnRGb3JFTVJSdW5Kb2JGbG93UnVsZSdcbiAgICAgICAgfSldXG4gICAgICB9KSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHBvbGljeVN0YXRlbWVudHM7XG4gIH1cblxuICAvKipcbiAgICogR2VuZXJhdGUgdGhlIFJvbGUgdXNlZCBieSB0aGUgRU1SIFNlcnZpY2VcbiAgICovXG4gIHByaXZhdGUgY3JlYXRlU2VydmljZVJvbGUodGFzazogc2ZuLlRhc2spOiBpYW0uSVJvbGUge1xuICAgIHJldHVybiBuZXcgaWFtLlJvbGUodGFzaywgJ1NlcnZpY2VSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2VsYXN0aWNtYXByZWR1Y2UuYW1hem9uYXdzLmNvbScpLFxuICAgICAgbWFuYWdlZFBvbGljaWVzOiBbXG4gICAgICAgIGlhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnc2VydmljZS1yb2xlL0FtYXpvbkVsYXN0aWNNYXBSZWR1Y2VSb2xlJylcbiAgICAgIF1cbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHZW5lcmF0ZSB0aGUgUm9sZSBhbmQgSW5zdGFuY2UgUHJvZmlsZSB1c2VkIGJ5IHRoZSBFQzIgaW5zdGFuY2VzXG4gICAqXG4gICAqIERhdGEgYWNjZXNzIHBlcm1pc3Npb25zIHdpbGwgbmVlZCB0byBiZSB1cGRhdGVkIGJ5IHRoZSB1c2VyXG4gICAqL1xuICBwcml2YXRlIGNyZWF0ZUNsdXN0ZXJSb2xlKHRhc2s6IHNmbi5UYXNrKTogaWFtLklSb2xlIHtcbiAgICBjb25zdCByb2xlID0gbmV3IGlhbS5Sb2xlKHRhc2ssICdJbnN0YW5jZVJvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWMyLmFtYXpvbmF3cy5jb20nKVxuICAgIH0pO1xuXG4gICAgbmV3IGlhbS5DZm5JbnN0YW5jZVByb2ZpbGUodGFzaywgJ0luc3RhbmNlUHJvZmlsZScsIHtcbiAgICAgIHJvbGVzOiBbIHJvbGUucm9sZU5hbWUgXSxcbiAgICAgIGluc3RhbmNlUHJvZmlsZU5hbWU6IHJvbGUucm9sZU5hbWVcbiAgICB9KTtcblxuICAgIHJldHVybiByb2xlO1xuICB9XG5cbiAgLyoqXG4gICAqIEdlbmVyYXRlIHRoZSBSb2xlIHVzZWQgdG8gQXV0b1NjYWxlIHRoZSBDbHVzdGVyXG4gICAqL1xuICBwcml2YXRlIGNyZWF0ZUF1dG9TY2FsaW5nUm9sZSh0YXNrOiBzZm4uVGFzayk6IGlhbS5JUm9sZSB7XG4gICAgY29uc3Qgcm9sZSA9IG5ldyBpYW0uUm9sZSh0YXNrLCAnQXV0b1NjYWxpbmdSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2VsYXN0aWNtYXByZWR1Y2UuYW1hem9uYXdzLmNvbScpLFxuICAgICAgbWFuYWdlZFBvbGljaWVzOiBbXG4gICAgICAgIGlhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnc2VydmljZS1yb2xlL0FtYXpvbkVsYXN0aWNNYXBSZWR1Y2Vmb3JBdXRvU2NhbGluZ1JvbGUnKVxuICAgICAgXVxuICAgIH0pO1xuXG4gICAgcm9sZS5hc3N1bWVSb2xlUG9saWN5Py5hZGRTdGF0ZW1lbnRzKFxuICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICAgIHByaW5jaXBhbHM6IFtcbiAgICAgICAgICBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2FwcGxpY2F0aW9uLWF1dG9zY2FsaW5nLmFtYXpvbmF3cy5jb20nKVxuICAgICAgICBdLFxuICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgJ3N0czpBc3N1bWVSb2xlJ1xuICAgICAgICBdXG4gICAgICB9KVxuICAgICk7XG5cbiAgICByZXR1cm4gcm9sZTtcbiAgfVxufVxuXG5leHBvcnQgbmFtZXNwYWNlIEVtckNyZWF0ZUNsdXN0ZXIge1xuICAvKipcbiAgICogVmFsaWQgdmFsdXMgZm9yIHRoZSBDbHVzdGVyIFNjYWxlRG93bkJlaGF2aW9yXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBlbnVtIEVtckNsdXN0ZXJTY2FsZURvd25CZWhhdmlvciB7XG4gICAgLyoqXG4gICAgICogSW5kaWNhdGVzIHRoYXQgQW1hem9uIEVNUiB0ZXJtaW5hdGVzIG5vZGVzIGF0IHRoZSBpbnN0YW5jZS1ob3VyIGJvdW5kYXJ5LCByZWdhcmRsZXNzIG9mIHdoZW4gdGhlIHJlcXVlc3QgdG8gdGVybWluYXRlIHRoZSBpbnN0YW5jZSB3YXNcbiAgICAgKiBzdWJtaXR0ZWQuIFRoaXMgb3B0aW9uIGlzIG9ubHkgYXZhaWxhYmxlIHdpdGggQW1hem9uIEVNUiA1LjEuMCBhbmQgbGF0ZXIgYW5kIGlzIHRoZSBkZWZhdWx0IGZvciBjbHVzdGVycyBjcmVhdGVkIHVzaW5nIHRoYXQgdmVyc2lvblxuICAgICAqL1xuICAgIFRFUk1JTkFURV9BVF9JTlNUQU5DRV9IT1VSID0gJ1RFUk1JTkFURV9BVF9JTlNUQU5DRV9IT1VSJyxcblxuICAgIC8qKlxuICAgICAqIEluZGljYXRlcyB0aGF0IEFtYXpvbiBFTVIgYmxhY2tsaXN0cyBhbmQgZHJhaW5zIHRhc2tzIGZyb20gbm9kZXMgYmVmb3JlIHRlcm1pbmF0aW5nIHRoZSBBbWF6b24gRUMyIGluc3RhbmNlcywgcmVnYXJkbGVzcyBvZiB0aGVcbiAgICAgKiBpbnN0YW5jZS1ob3VyIGJvdW5kYXJ5LlxuICAgICAqL1xuICAgIFRFUk1JTkFURV9BVF9UQVNLX0NPTVBMRVRJT04gPSAnVEVSTUlOQVRFX0FUX1RBU0tfQ09NUExFVElPTidcbiAgfVxuXG4gIC8qKlxuICAgKiBJbnN0YW5jZSBSb2xlIFR5cGVzXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBlbnVtIEluc3RhbmNlUm9sZVR5cGUge1xuICAgIC8qKlxuICAgICAqIE1hc3RlciBOb2RlXG4gICAgICovXG4gICAgTUFTVEVSID0gJ01BU1RFUicsXG4gICAgLyoqXG4gICAgICogQ29yZSBOb2RlXG4gICAgICovXG4gICAgQ09SRSA9ICdDT1JFJyxcbiAgICAvKipcbiAgICAgKiBUYXNrIE5vZGVcbiAgICAgKi9cbiAgICBUQVNLID0gJ1RBU0snXG4gIH1cblxuICAvKipcbiAgICogRUJTIFZvbHVtZSBUeXBlc1xuICAgKlxuICAgKiBAZXhwZXJpbWVudGFsXG4gICAqL1xuICBleHBvcnQgZW51bSBFYnNCbG9ja0RldmljZVZvbHVtZVR5cGUge1xuICAgIC8qKlxuICAgICAqIGdwMiBWb2x1bWUgVHlwZVxuICAgICAqL1xuICAgIEdQMiA9ICdncDInLFxuICAgIC8qKlxuICAgICAqIGlvMSBWb2x1bWUgVHlwZVxuICAgICAqL1xuICAgIElPMSA9ICdpbzEnLFxuICAgIC8qKlxuICAgICAqIFN0YW5kYXJkIFZvbHVtZSBUeXBlXG4gICAgICovXG4gICAgU1RBTkRBUkQgPSAnc3RhbmRhcmQnXG4gIH1cblxuICAvKipcbiAgICogRUJTIHZvbHVtZSBzcGVjaWZpY2F0aW9ucyBzdWNoIGFzIHZvbHVtZSB0eXBlLCBJT1BTLCBhbmQgc2l6ZSAoR2lCKSB0aGF0IHdpbGwgYmUgcmVxdWVzdGVkIGZvciB0aGUgRUJTIHZvbHVtZSBhdHRhY2hlZFxuICAgKiB0byBhbiBFQzIgaW5zdGFuY2UgaW4gdGhlIGNsdXN0ZXIuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Wb2x1bWVTcGVjaWZpY2F0aW9uLmh0bWxcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGludGVyZmFjZSBWb2x1bWVTcGVjaWZpY2F0aW9uUHJvcGVydHkge1xuICAgIC8qKlxuICAgICAqIFRoZSBudW1iZXIgb2YgSS9PIG9wZXJhdGlvbnMgcGVyIHNlY29uZCAoSU9QUykgdGhhdCB0aGUgdm9sdW1lIHN1cHBvcnRzLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgRU1SIHNlbGVjdGVkIGRlZmF1bHRcbiAgICAgKi9cbiAgICByZWFkb25seSBpb3BzPzogbnVtYmVyO1xuXG4gICAgLyoqXG4gICAgICogVGhlIHZvbHVtZSBzaXplLCBpbiBnaWJpYnl0ZXMgKEdpQikuIFRoaXMgY2FuIGJlIGEgbnVtYmVyIGZyb20gMSAtIDEwMjQuIElmIHRoZSB2b2x1bWUgdHlwZSBpcyBFQlMtb3B0aW1pemVkLCB0aGUgbWluaW11bSB2YWx1ZSBpcyAxMC5cbiAgICAgKi9cbiAgICByZWFkb25seSBzaXplSW5HQjogbnVtYmVyO1xuXG4gICAgLyoqXG4gICAgICogVGhlIHZvbHVtZSB0eXBlLiBWb2x1bWUgdHlwZXMgc3VwcG9ydGVkIGFyZSBncDIsIGlvMSwgc3RhbmRhcmQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgdm9sdW1lVHlwZTogRWJzQmxvY2tEZXZpY2VWb2x1bWVUeXBlXG4gIH1cblxuICAvKipcbiAgICogQ29uZmlndXJhdGlvbiBvZiByZXF1ZXN0ZWQgRUJTIGJsb2NrIGRldmljZSBhc3NvY2lhdGVkIHdpdGggdGhlIGluc3RhbmNlIGdyb3VwIHdpdGggY291bnQgb2Ygdm9sdW1lcyB0aGF0IHdpbGwgYmVcbiAgICogYXNzb2NpYXRlZCB0byBldmVyeSBpbnN0YW5jZS5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Vic0Jsb2NrRGV2aWNlQ29uZmlnLmh0bWxcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGludGVyZmFjZSBFYnNCbG9ja0RldmljZUNvbmZpZ1Byb3BlcnR5IHtcbiAgICAvKipcbiAgICAgKiBFQlMgdm9sdW1lIHNwZWNpZmljYXRpb25zIHN1Y2ggYXMgdm9sdW1lIHR5cGUsIElPUFMsIGFuZCBzaXplIChHaUIpIHRoYXQgd2lsbCBiZSByZXF1ZXN0ZWQgZm9yIHRoZSBFQlMgdm9sdW1lIGF0dGFjaGVkIHRvIGFuIEVDMlxuICAgICAqIGluc3RhbmNlIGluIHRoZSBjbHVzdGVyLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHZvbHVtZVNwZWNpZmljYXRpb246IFZvbHVtZVNwZWNpZmljYXRpb25Qcm9wZXJ0eTtcblxuICAgIC8qKlxuICAgICAqIE51bWJlciBvZiBFQlMgdm9sdW1lcyB3aXRoIGEgc3BlY2lmaWMgdm9sdW1lIGNvbmZpZ3VyYXRpb24gdGhhdCB3aWxsIGJlIGFzc29jaWF0ZWQgd2l0aCBldmVyeSBpbnN0YW5jZSBpbiB0aGUgaW5zdGFuY2UgZ3JvdXBcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IEVNUiBzZWxlY3RlZCBkZWZhdWx0XG4gICAgICovXG4gICAgcmVhZG9ubHkgdm9sdW1lc1Blckluc3RhbmNlPzogbnVtYmVyO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlbmRlciB0aGUgRWJzQmxvY2tEZXZpY2VDb25maWdQcm9wZXJ0eSBhcyBKU09OXG4gICAqXG4gICAqIEBwYXJhbSBwcm9wZXJ0eVxuICAgKi9cbiAgZXhwb3J0IGZ1bmN0aW9uIEVic0Jsb2NrRGV2aWNlQ29uZmlnUHJvcGVydHlUb0pzb24ocHJvcGVydHk6IEVic0Jsb2NrRGV2aWNlQ29uZmlnUHJvcGVydHkpIHtcbiAgICByZXR1cm4ge1xuICAgICAgVm9sdW1lU3BlY2lmaWNhdGlvbjoge1xuICAgICAgICBJb3BzOiBjZGsubnVtYmVyVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS52b2x1bWVTcGVjaWZpY2F0aW9uLmlvcHMpLFxuICAgICAgICBTaXplSW5HQjogY2RrLm51bWJlclRvQ2xvdWRGb3JtYXRpb24ocHJvcGVydHkudm9sdW1lU3BlY2lmaWNhdGlvbi5zaXplSW5HQiksXG4gICAgICAgIFZvbHVtZVR5cGU6IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5LnZvbHVtZVNwZWNpZmljYXRpb24udm9sdW1lVHlwZT8udmFsdWVPZigpKVxuICAgICAgfSxcbiAgICAgIFZvbHVtZXNQZXJJbnN0YW5jZTogY2RrLm51bWJlclRvQ2xvdWRGb3JtYXRpb24ocHJvcGVydHkudm9sdW1lc1Blckluc3RhbmNlKVxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogVGhlIEFtYXpvbiBFQlMgY29uZmlndXJhdGlvbiBvZiBhIGNsdXN0ZXIgaW5zdGFuY2UuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9FYnNDb25maWd1cmF0aW9uLmh0bWxcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGludGVyZmFjZSBFYnNDb25maWd1cmF0aW9uUHJvcGVydHkge1xuICAgIC8qKlxuICAgICAqIEFuIGFycmF5IG9mIEFtYXpvbiBFQlMgdm9sdW1lIHNwZWNpZmljYXRpb25zIGF0dGFjaGVkIHRvIGEgY2x1c3RlciBpbnN0YW5jZS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIGVic0Jsb2NrRGV2aWNlQ29uZmlnc1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGVic0Jsb2NrRGV2aWNlQ29uZmlncz86IEVic0Jsb2NrRGV2aWNlQ29uZmlnUHJvcGVydHlbXTtcblxuICAgIC8qKlxuICAgICAqIEluZGljYXRlcyB3aGV0aGVyIGFuIEFtYXpvbiBFQlMgdm9sdW1lIGlzIEVCUy1vcHRpbWl6ZWQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBFTVIgc2VsZWN0ZWQgZGVmYXVsdFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGVic09wdGltaXplZD86IGJvb2xlYW47XG4gIH1cblxuICAvKipcbiAgICogUmVuZGVyIHRoZSBFYnNDb25maWd1cmF0aW9uUHJvcGVydHkgdG8gSlNPTlxuICAgKlxuICAgKiBAcGFyYW0gcHJvcGVydHlcbiAgICovXG4gIGV4cG9ydCBmdW5jdGlvbiBFYnNDb25maWd1cmF0aW9uUHJvcGVydHlUb0pzb24ocHJvcGVydHk6IEVic0NvbmZpZ3VyYXRpb25Qcm9wZXJ0eSkge1xuICAgIHJldHVybiB7XG4gICAgICBFYnNCbG9ja0RldmljZUNvbmZpZ3M6IGNkay5saXN0TWFwcGVyKEVic0Jsb2NrRGV2aWNlQ29uZmlnUHJvcGVydHlUb0pzb24pKHByb3BlcnR5LmVic0Jsb2NrRGV2aWNlQ29uZmlncyksXG4gICAgICBFYnNPcHRpbWl6ZWQ6IGNkay5ib29sZWFuVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5lYnNPcHRpbWl6ZWQpXG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBbiBpbnN0YW5jZSB0eXBlIGNvbmZpZ3VyYXRpb24gZm9yIGVhY2ggaW5zdGFuY2UgdHlwZSBpbiBhbiBpbnN0YW5jZSBmbGVldCwgd2hpY2ggZGV0ZXJtaW5lcyB0aGUgRUMyIGluc3RhbmNlcyBBbWF6b24gRU1SIGF0dGVtcHRzIHRvXG4gICAqIHByb3Zpc2lvbiB0byBmdWxmaWxsIE9uLURlbWFuZCBhbmQgU3BvdCB0YXJnZXQgY2FwYWNpdGllcy5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0luc3RhbmNlVHlwZUNvbmZpZy5odG1sXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBpbnRlcmZhY2UgSW5zdGFuY2VUeXBlQ29uZmlnUHJvcGVydHkge1xuICAgIC8qKlxuICAgICAqIFRoZSBiaWQgcHJpY2UgZm9yIGVhY2ggRUMyIFNwb3QgaW5zdGFuY2UgdHlwZSBhcyBkZWZpbmVkIGJ5IEluc3RhbmNlVHlwZS4gRXhwcmVzc2VkIGluIFVTRC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIGJpZFByaWNlXG4gICAgICovXG4gICAgcmVhZG9ubHkgYmlkUHJpY2U/OiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgYmlkIHByaWNlLCBhcyBhIHBlcmNlbnRhZ2Ugb2YgT24tRGVtYW5kIHByaWNlLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gYmlkUHJpY2VBc1BlcmNlbnRhZ2VPZk9uRGVtYW5kUHJpY2VcbiAgICAgKi9cbiAgICByZWFkb25seSBiaWRQcmljZUFzUGVyY2VudGFnZU9mT25EZW1hbmRQcmljZT86IG51bWJlcjtcblxuICAgIC8qKlxuICAgICAqIEEgY29uZmlndXJhdGlvbiBjbGFzc2lmaWNhdGlvbiB0aGF0IGFwcGxpZXMgd2hlbiBwcm92aXNpb25pbmcgY2x1c3RlciBpbnN0YW5jZXMsIHdoaWNoIGNhbiBpbmNsdWRlIGNvbmZpZ3VyYXRpb25zIGZvciBhcHBsaWNhdGlvbnNcbiAgICAgKiBhbmQgc29mdHdhcmUgdGhhdCBydW4gb24gdGhlIGNsdXN0ZXIuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBjb25maWd1cmF0aW9uc1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGNvbmZpZ3VyYXRpb25zPzogQ29uZmlndXJhdGlvblByb3BlcnR5W107XG5cbiAgICAvKipcbiAgICAgKiBUaGUgY29uZmlndXJhdGlvbiBvZiBBbWF6b24gRWxhc3RpYyBCbG9jayBTdG9yYWdlIChFQlMpIGF0dGFjaGVkIHRvIGVhY2ggaW5zdGFuY2UgYXMgZGVmaW5lZCBieSBJbnN0YW5jZVR5cGUuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBlYnNDb25maWd1cmF0aW9uXG4gICAgICovXG4gICAgcmVhZG9ubHkgZWJzQ29uZmlndXJhdGlvbj86IEVic0NvbmZpZ3VyYXRpb25Qcm9wZXJ0eTtcblxuICAgIC8qKlxuICAgICAqIEFuIEVDMiBpbnN0YW5jZSB0eXBlXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW5zdGFuY2VUeXBlOiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgbnVtYmVyIG9mIHVuaXRzIHRoYXQgYSBwcm92aXNpb25lZCBpbnN0YW5jZSBvZiB0aGlzIHR5cGUgcHJvdmlkZXMgdG93YXJkIGZ1bGZpbGxpbmcgdGhlIHRhcmdldCBjYXBhY2l0aWVzIGRlZmluZWRcbiAgICAgKiBpbiB0aGUgSW5zdGFuY2VGbGVldENvbmZpZy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIHdlaWdodGVkQ2FwYWNpdHlcbiAgICAgKi9cbiAgICByZWFkb25seSB3ZWlnaHRlZENhcGFjaXR5PzogbnVtYmVyO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlbmRlciB0aGUgSW5zdGFuY2VUeXBlQ29uZmlnUHJvcGVydHkgdG8gSlNPTl1cbiAgICpcbiAgICogQHBhcmFtIHByb3BlcnR5XG4gICAqL1xuICBleHBvcnQgZnVuY3Rpb24gSW5zdGFuY2VUeXBlQ29uZmlnUHJvcGVydHlUb0pzb24ocHJvcGVydHk6IEluc3RhbmNlVHlwZUNvbmZpZ1Byb3BlcnR5KSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIEJpZFByaWNlOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5iaWRQcmljZSksXG4gICAgICBCaWRQcmljZUFzUGVyY2VudGFnZU9mT25EZW1hbmRQcmljZTogY2RrLm51bWJlclRvQ2xvdWRGb3JtYXRpb24ocHJvcGVydHkuYmlkUHJpY2VBc1BlcmNlbnRhZ2VPZk9uRGVtYW5kUHJpY2UpLFxuICAgICAgQ29uZmlndXJhdGlvbnM6IGNkay5saXN0TWFwcGVyKENvbmZpZ3VyYXRpb25Qcm9wZXJ0eVRvSnNvbikocHJvcGVydHkuY29uZmlndXJhdGlvbnMpLFxuICAgICAgRWJzQ29uZmlndXJhdGlvbjogKHByb3BlcnR5LmVic0NvbmZpZ3VyYXRpb24gPT09IHVuZGVmaW5lZCkgP1xuICAgICAgICBwcm9wZXJ0eS5lYnNDb25maWd1cmF0aW9uIDpcbiAgICAgICAgRWJzQ29uZmlndXJhdGlvblByb3BlcnR5VG9Kc29uKHByb3BlcnR5LmVic0NvbmZpZ3VyYXRpb24pLFxuICAgICAgSW5zdGFuY2VUeXBlOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5pbnN0YW5jZVR5cGU/LnZhbHVlT2YoKSksXG4gICAgICBXZWlnaHRlZENhcGFjaXR5OiBjZGsubnVtYmVyVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS53ZWlnaHRlZENhcGFjaXR5KVxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogU3BvdCBUaW1lb3V0IEFjdGlvbnNcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGVudW0gU3BvdFRpbWVvdXRBY3Rpb24ge1xuICAgIC8qKlxcXG4gICAgICogU1dJVENIX1RPX09OX0RFTUFORFxuICAgICAqL1xuICAgIFNXSVRDSF9UT19PTl9ERU1BTkQgPSAnU1dJVENIX1RPX09OX0RFTUFORCcsXG4gICAgLyoqXG4gICAgICogVEVSTUlOQVRFX0NMVVNURVJcbiAgICAgKi9cbiAgICBURVJNSU5BVEVfQ0xVU1RFUiA9ICdURVJNSU5BVEVfQ0xVU1RFUidcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgbGF1bmNoIHNwZWNpZmljYXRpb24gZm9yIFNwb3QgaW5zdGFuY2VzIGluIHRoZSBpbnN0YW5jZSBmbGVldCwgd2hpY2ggZGV0ZXJtaW5lcyB0aGUgZGVmaW5lZCBkdXJhdGlvbiBhbmQgcHJvdmlzaW9uaW5nIHRpbWVvdXQgYmVoYXZpb3IuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TcG90UHJvdmlzaW9uaW5nU3BlY2lmaWNhdGlvbi5odG1sXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBpbnRlcmZhY2UgU3BvdFByb3Zpc2lvbmluZ1NwZWNpZmljYXRpb25Qcm9wZXJ0eSB7XG4gICAgLyoqXG4gICAgICogVGhlIGRlZmluZWQgZHVyYXRpb24gZm9yIFNwb3QgaW5zdGFuY2VzIChhbHNvIGtub3duIGFzIFNwb3QgYmxvY2tzKSBpbiBtaW51dGVzLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gYmxvY2tEdXJhdGlvbk1pbnV0ZXNcbiAgICAgKi9cbiAgICByZWFkb25seSBibG9ja0R1cmF0aW9uTWludXRlcz86IG51bWJlcjtcblxuICAgIC8qKlxuICAgICAqIFRoZSBhY3Rpb24gdG8gdGFrZSB3aGVuIFRhcmdldFNwb3RDYXBhY2l0eSBoYXMgbm90IGJlZW4gZnVsZmlsbGVkIHdoZW4gdGhlIFRpbWVvdXREdXJhdGlvbk1pbnV0ZXMgaGFzIGV4cGlyZWRcbiAgICAgKi9cbiAgICByZWFkb25seSB0aW1lb3V0QWN0aW9uOiBTcG90VGltZW91dEFjdGlvbjtcblxuICAgIC8qKlxuICAgICAqIFRoZSBzcG90IHByb3Zpc2lvbmluZyB0aW1lb3V0IHBlcmlvZCBpbiBtaW51dGVzLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHRpbWVvdXREdXJhdGlvbk1pbnV0ZXM6IG51bWJlcjtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgbGF1bmNoIHNwZWNpZmljYXRpb24gZm9yIFNwb3QgaW5zdGFuY2VzIGluIHRoZSBmbGVldCwgd2hpY2ggZGV0ZXJtaW5lcyB0aGUgZGVmaW5lZCBkdXJhdGlvbiBhbmQgcHJvdmlzaW9uaW5nIHRpbWVvdXQgYmVoYXZpb3IuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9JbnN0YW5jZUZsZWV0UHJvdmlzaW9uaW5nU3BlY2lmaWNhdGlvbnMuaHRtbFxuICAgKlxuICAgKiBAZXhwZXJpbWVudGFsXG4gICAqL1xuICBleHBvcnQgaW50ZXJmYWNlIEluc3RhbmNlRmxlZXRQcm92aXNpb25pbmdTcGVjaWZpY2F0aW9uc1Byb3BlcnR5IHtcbiAgICAvKipcbiAgICAgKiBUaGUgbGF1bmNoIHNwZWNpZmljYXRpb24gZm9yIFNwb3QgaW5zdGFuY2VzIGluIHRoZSBmbGVldCwgd2hpY2ggZGV0ZXJtaW5lcyB0aGUgZGVmaW5lZCBkdXJhdGlvbiBhbmQgcHJvdmlzaW9uaW5nIHRpbWVvdXQgYmVoYXZpb3IuXG4gICAgICovXG4gICAgcmVhZG9ubHkgc3BvdFNwZWNpZmljYXRpb246IFNwb3RQcm92aXNpb25pbmdTcGVjaWZpY2F0aW9uUHJvcGVydHk7XG4gIH1cblxuICAvKipcbiAgICogUmVuZGVyIHRoZSBJbnN0YW5jZUZsZWV0UHJvdmlzaW9uaW5nU3BlY2lmaWNhdGlvbnNQcm9wZXJ0eSB0byBKU09OXG4gICAqXG4gICAqIEBwYXJhbSBwcm9wZXJ0eVxuICAgKi9cbiAgZXhwb3J0IGZ1bmN0aW9uIEluc3RhbmNlRmxlZXRQcm92aXNpb25pbmdTcGVjaWZpY2F0aW9uc1Byb3BlcnR5VG9Kc29uKHByb3BlcnR5OiBJbnN0YW5jZUZsZWV0UHJvdmlzaW9uaW5nU3BlY2lmaWNhdGlvbnNQcm9wZXJ0eSkge1xuICAgIHJldHVybiB7XG4gICAgICBTcG90U3BlY2lmaWNhdGlvbjoge1xuICAgICAgICBCbG9ja0R1cmF0aW9uTWludXRlczogY2RrLm51bWJlclRvQ2xvdWRGb3JtYXRpb24ocHJvcGVydHkuc3BvdFNwZWNpZmljYXRpb24uYmxvY2tEdXJhdGlvbk1pbnV0ZXMpLFxuICAgICAgICBUaW1lb3V0QWN0aW9uOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5zcG90U3BlY2lmaWNhdGlvbi50aW1lb3V0QWN0aW9uPy52YWx1ZU9mKCkpLFxuICAgICAgICBUaW1lb3V0RHVyYXRpb25NaW51dGVzOiBjZGsubnVtYmVyVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5zcG90U3BlY2lmaWNhdGlvbi50aW1lb3V0RHVyYXRpb25NaW51dGVzKVxuICAgICAgfVxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogVGhlIGNvbmZpZ3VyYXRpb24gdGhhdCBkZWZpbmVzIGFuIGluc3RhbmNlIGZsZWV0LlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfSW5zdGFuY2VGbGVldENvbmZpZy5odG1sXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBpbnRlcmZhY2UgSW5zdGFuY2VGbGVldENvbmZpZ1Byb3BlcnR5IHtcbiAgICAvKipcbiAgICAgKiBUaGUgbm9kZSB0eXBlIHRoYXQgdGhlIGluc3RhbmNlIGZsZWV0IGhvc3RzLiBWYWxpZCB2YWx1ZXMgYXJlIE1BU1RFUixDT1JFLGFuZCBUQVNLLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGluc3RhbmNlRmxlZXRUeXBlOiBJbnN0YW5jZVJvbGVUeXBlO1xuXG4gICAgLyoqXG4gICAgICogVGhlIGluc3RhbmNlIHR5cGUgY29uZmlndXJhdGlvbnMgdGhhdCBkZWZpbmUgdGhlIEVDMiBpbnN0YW5jZXMgaW4gdGhlIGluc3RhbmNlIGZsZWV0LlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gaW5zdGFuY2VUcGVDb25maWdzXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW5zdGFuY2VUeXBlQ29uZmlncz86IEluc3RhbmNlVHlwZUNvbmZpZ1Byb3BlcnR5W107XG5cbiAgICAvKipcbiAgICAgKiBUaGUgbGF1bmNoIHNwZWNpZmljYXRpb24gZm9yIHRoZSBpbnN0YW5jZSBmbGVldC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIGxhdW5jaFNwZWNpZmljYXRpb25zXG4gICAgICovXG4gICAgcmVhZG9ubHkgbGF1bmNoU3BlY2lmaWNhdGlvbnM/OiBJbnN0YW5jZUZsZWV0UHJvdmlzaW9uaW5nU3BlY2lmaWNhdGlvbnNQcm9wZXJ0eTtcblxuICAgIC8qKlxuICAgICAqIFRoZSBmcmllbmRseSBuYW1lIG9mIHRoZSBpbnN0YW5jZSBmbGVldC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIG5hbWVcbiAgICAgKi9cbiAgICByZWFkb25seSBuYW1lPzogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgICogVGhlIHRhcmdldCBjYXBhY2l0eSBvZiBPbi1EZW1hbmQgdW5pdHMgZm9yIHRoZSBpbnN0YW5jZSBmbGVldCwgd2hpY2ggZGV0ZXJtaW5lcyBob3cgbWFueSBPbi1EZW1hbmQgaW5zdGFuY2VzIHRvIHByb3Zpc2lvbi5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIHRhcmdldE9uRGVtYW5kQ2FwYWNpdHlcbiAgICAgKi9cbiAgICByZWFkb25seSB0YXJnZXRPbkRlbWFuZENhcGFjaXR5PzogbnVtYmVyO1xuXG4gICAgLyoqXG4gICAgICogVGhlIHRhcmdldCBjYXBhY2l0eSBvZiBTcG90IHVuaXRzIGZvciB0aGUgaW5zdGFuY2UgZmxlZXQsIHdoaWNoIGRldGVybWluZXMgaG93IG1hbnkgU3BvdCBpbnN0YW5jZXMgdG8gcHJvdmlzaW9uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyB0YXJnZXRTcG90Q2FwYWNpdHlcbiAgICAgKi9cbiAgICByZWFkb25seSB0YXJnZXRTcG90Q2FwYWNpdHk/OiBudW1iZXI7XG4gIH1cblxuICAvKipcbiAgICogUmVuZGVyIHRoZSBJbnN0YW5jZUZsZWV0Q29uZmlnUHJvcGVydHkgYXMgSlNPTlxuICAgKlxuICAgKiBAcGFyYW0gcHJvcGVydHlcbiAgICovXG4gIGV4cG9ydCBmdW5jdGlvbiBJbnN0YW5jZUZsZWV0Q29uZmlnUHJvcGVydHlUb0pzb24ocHJvcGVydHk6IEluc3RhbmNlRmxlZXRDb25maWdQcm9wZXJ0eSkge1xuICAgIHJldHVybiB7XG4gICAgICBJbnN0YW5jZUZsZWV0VHlwZTogY2RrLnN0cmluZ1RvQ2xvdWRGb3JtYXRpb24ocHJvcGVydHkuaW5zdGFuY2VGbGVldFR5cGU/LnZhbHVlT2YoKSksXG4gICAgICBJbnN0YW5jZVR5cGVDb25maWdzOiBjZGsubGlzdE1hcHBlcihJbnN0YW5jZVR5cGVDb25maWdQcm9wZXJ0eVRvSnNvbikocHJvcGVydHkuaW5zdGFuY2VUeXBlQ29uZmlncyksXG4gICAgICBMYXVuY2hTcGVjaWZpY2F0aW9uczogKHByb3BlcnR5LmxhdW5jaFNwZWNpZmljYXRpb25zID09PSB1bmRlZmluZWQpID9cbiAgICAgICAgcHJvcGVydHkubGF1bmNoU3BlY2lmaWNhdGlvbnMgOlxuICAgICAgICBJbnN0YW5jZUZsZWV0UHJvdmlzaW9uaW5nU3BlY2lmaWNhdGlvbnNQcm9wZXJ0eVRvSnNvbihwcm9wZXJ0eS5sYXVuY2hTcGVjaWZpY2F0aW9ucyksXG4gICAgICBOYW1lOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5uYW1lKSxcbiAgICAgIFRhcmdldE9uRGVtYW5kQ2FwYWNpdHk6IGNkay5udW1iZXJUb0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5LnRhcmdldE9uRGVtYW5kQ2FwYWNpdHkpLFxuICAgICAgVGFyZ2V0U3BvdENhcGFjaXR5OiBjZGsubnVtYmVyVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS50YXJnZXRTcG90Q2FwYWNpdHkpXG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDbG91ZFdhdGNoIEFsYXJtIENvbXBhcmlzb24gT3BlcmF0b3JzXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBlbnVtIENsb3VkV2F0Y2hBbGFybUNvbXBhcmlzb25PcGVyYXRvciB7XG4gICAgLyoqXG4gICAgICogR1JFQVRFUl9USEFOX09SX0VRVUFMXG4gICAgICovXG4gICAgR1JFQVRFUl9USEFOX09SX0VRVUFMID0gJ0dSRUFURVJfVEhBTl9PUl9FUVVBTCcsXG4gICAgLyoqXG4gICAgICogR1JFQVRFUl9USEFOXG4gICAgICovXG4gICAgR1JFQVRFUl9USEFOID0gJ0dSRUFURVJfVEhBTicsXG4gICAgLyoqXG4gICAgICogTEVTU19USEFOXG4gICAgICovXG4gICAgTEVTU19USEFOID0gJ0xFU1NfVEhBTicsXG4gICAgLyoqXG4gICAgICogTEVTU19USEFOX09SX0VRVUFMXG4gICAgICovXG4gICAgTEVTU19USEFOX09SX0VRVUFMID0gJ0xFU1NfVEhBTl9PUl9FUVVBTCdcbiAgfVxuXG4gIC8qKlxuICAgKiBDbG91ZFdhdGNoIEFsYXJtIFN0YXRpc3RpY3NcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGVudW0gQ2xvdWRXYXRjaEFsYXJtU3RhdGlzdGljIHtcbiAgICAvKipcbiAgICAgKiBTQU1QTEVfQ09VTlRcbiAgICAgKi9cbiAgICBTQU1QTEVfQ09VTlQgPSAnU0FNUExFX0NPVU5UJyxcbiAgICAvKipcbiAgICAgKiBBVkVSQUdFXG4gICAgICovXG4gICAgQVZFUkFHRSA9ICdBVkVSQUdFJyxcbiAgICAvKipcbiAgICAgKiBTVU1cbiAgICAgKi9cbiAgICBTVU0gPSAnU1VNJyxcbiAgICAvKipcbiAgICAgKiBNSU5JTVVNXG4gICAgICovXG4gICAgTUlOSU1VTSA9ICdNSU5JTVVNJyxcbiAgICAvKipcbiAgICAgKiBNQVhJTVVNXG4gICAgICovXG4gICAgTUFYSU1VTSA9ICdNQVhJTVVNJ1xuICB9XG5cbiAgLyoqXG4gICAqIENsb3VkV2F0Y2ggQWxhcm0gVW5pdHNcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGVudW0gQ2xvdWRXYXRjaEFsYXJtVW5pdCB7XG4gICAgLyoqXG4gICAgICogTk9ORVxuICAgICAqL1xuICAgIE5PTkUgPSAnTk9ORScsXG4gICAgLyoqXG4gICAgICogU0VDT05EU1xuICAgICAqL1xuICAgIFNFQ09ORFMgPSAnU0VDT05EUycsXG4gICAgLyoqXG4gICAgICogTUlDUk9fU0VDT05EU1xuICAgICAqL1xuICAgIE1JQ1JPX1NFQ09ORFMgPSAnTUlDUk9fU0VDT05EUycsXG4gICAgLyoqXG4gICAgICogTUlMTElfU0VDT05EU1xuICAgICAqL1xuICAgIE1JTExJX1NFQ09ORFMgPSAnTUlMTElfU0VDT05EUycsXG4gICAgLyoqXG4gICAgICogQllURVNcbiAgICAgKi9cbiAgICBCWVRFUyA9ICdCWVRFUycsXG4gICAgLyoqXG4gICAgICogS0lMT19CWVRFU1xuICAgICAqL1xuICAgIEtJTE9fQllURVMgPSAnS0lMT19CWVRFUycsXG4gICAgLyoqXG4gICAgICogTUVHQV9CWVRFU1xuICAgICAqL1xuICAgIE1FR0FfQllURVMgPSAnTUVHQV9CWVRFUycsXG4gICAgLyoqXG4gICAgICogR0lHQV9CWVRFU1xuICAgICAqL1xuICAgIEdJR0FfQllURVMgPSAnR0lHQV9CWVRFUycsXG4gICAgLyoqXG4gICAgICogVEVSQV9CWVRFU1xuICAgICAqL1xuICAgIFRFUkFfQllURVMgPSAnVEVSQV9CWVRFUycsXG4gICAgLyoqXG4gICAgICogQklUU1xuICAgICAqL1xuICAgIEJJVFMgPSAnQklUUycsXG4gICAgLyoqXG4gICAgICogS0lMT19CSVRTXG4gICAgICovXG4gICAgS0lMT19CSVRTID0gJ0tJTE9fQklUUycsXG4gICAgLyoqXG4gICAgICogTUVHQV9CSVRTXG4gICAgICovXG4gICAgTUVHQV9CSVRTID0gJ01FR0FfQklUUycsXG4gICAgLyoqXG4gICAgICogR0lHQV9CSVRTXG4gICAgICovXG4gICAgR0lHQV9CSVRTID0gJ0dJR0FfQklUUycsXG4gICAgLyoqXG4gICAgICogVEVSQV9CSVRTXG4gICAgICovXG4gICAgVEVSQV9CSVRTID0gJ1RFUkFfQklUUycsXG4gICAgLyoqXG4gICAgICogUEVSQ0VOVFxuICAgICAqL1xuICAgIFBFUkNFTlQgPSAnUEVSQ0VOVCcsXG4gICAgLyoqXG4gICAgICogQ09VTlRcbiAgICAgKi9cbiAgICBDT1VOVCA9ICdDT1VOVCcsXG4gICAgLyoqXG4gICAgICogQllURVNfUEVSX1NFQ09ORFxuICAgICAqL1xuICAgIEJZVEVTX1BFUl9TRUNPTkQgPSAnQllURVNfUEVSX1NFQ09ORCcsXG4gICAgLyoqXG4gICAgICogS0lMT19CWVRFU19QRVJfU0VDT05EXG4gICAgICovXG4gICAgS0lMT19CWVRFU19QRVJfU0VDT05EID0gJ0tJTE9fQllURVNfUEVSX1NFQ09ORCcsXG4gICAgLyoqXG4gICAgICogTUVHQV9CWVRFU19QRVJfU0VDT05EXG4gICAgICovXG4gICAgTUVHQV9CWVRFU19QRVJfU0VDT05EID0gJ01FR0FfQllURVNfUEVSX1NFQ09ORCcsXG4gICAgLyoqXG4gICAgICogR0lHQV9CWVRFU19QRVJfU0VDT05EXG4gICAgICovXG4gICAgR0lHQV9CWVRFU19QRVJfU0VDT05EID0gJ0dJR0FfQllURVNfUEVSX1NFQ09ORCcsXG4gICAgLyoqXG4gICAgICogVEVSQV9CWVRFU19QRVJfU0VDT05EXG4gICAgICovXG4gICAgVEVSQV9CWVRFU19QRVJfU0VDT05EID0gJ1RFUkFfQllURVNfUEVSX1NFQ09ORCcsXG4gICAgLyoqXG4gICAgICogQklUU19QRVJfU0VDT05EXG4gICAgICovXG4gICAgQklUU19QRVJfU0VDT05EID0gJ0JJVFNfUEVSX1NFQ09ORCcsXG4gICAgLyoqXG4gICAgICogS0lMT19CSVRTX1BFUl9TRUNPTkRcbiAgICAgKi9cbiAgICBLSUxPX0JJVFNfUEVSX1NFQ09ORCA9ICdLSUxPX0JJVFNfUEVSX1NFQ09ORCcsXG4gICAgLyoqXG4gICAgICogTUVHQV9CSVRTX1BFUl9TRUNPTkRcbiAgICAgKi9cbiAgICBNRUdBX0JJVFNfUEVSX1NFQ09ORCA9ICdNRUdBX0JJVFNfUEVSX1NFQ09ORCcsXG4gICAgLyoqXG4gICAgICogR0lHQV9CSVRTX1BFUl9TRUNPTkRcbiAgICAgKi9cbiAgICBHSUdBX0JJVFNfUEVSX1NFQ09ORCA9ICdHSUdBX0JJVFNfUEVSX1NFQ09ORCcsXG4gICAgLyoqXG4gICAgICogVEVSQV9CSVRTX1BFUl9TRUNPTkRcbiAgICAgKi9cbiAgICBURVJBX0JJVFNfUEVSX1NFQ09ORCA9ICdURVJBX0JJVFNfUEVSX1NFQ09ORCcsXG4gICAgLyoqXG4gICAgICogQ09VTlRfUEVSX1NFQ09ORFxuICAgICAqL1xuICAgIENPVU5UX1BFUl9TRUNPTkQgPSAnQ09VTlRfUEVSX1NFQ09ORCdcbiAgfVxuXG4gIC8qKlxuICAgKiBBIENsb3VkV2F0Y2ggZGltZW5zaW9uLCB3aGljaCBpcyBzcGVjaWZpZWQgdXNpbmcgYSBLZXkgKGtub3duIGFzIGEgTmFtZSBpbiBDbG91ZFdhdGNoKSwgVmFsdWUgcGFpci4gQnkgZGVmYXVsdCwgQW1hem9uIEVNUiB1c2VzXG4gICAqIG9uZSBkaW1lbnNpb24gd2hvc2UgS2V5IGlzIEpvYkZsb3dJRCBhbmQgVmFsdWUgaXMgYSB2YXJpYWJsZSByZXByZXNlbnRpbmcgdGhlIGNsdXN0ZXIgSUQsIHdoaWNoIGlzICR7ZW1yLmNsdXN0ZXJJZH0uIFRoaXMgZW5hYmxlc1xuICAgKiB0aGUgcnVsZSB0byBib290c3RyYXAgd2hlbiB0aGUgY2x1c3RlciBJRCBiZWNvbWVzIGF2YWlsYWJsZVxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTWV0cmljRGltZW5zaW9uLmh0bWxcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGludGVyZmFjZSBNZXRyaWNEaW1lbnNpb25Qcm9wZXJ0eSB7XG4gICAgLyoqXG4gICAgICogVGhlIGRpbWVuc2lvbiBuYW1lXG4gICAgICovXG4gICAgcmVhZG9ubHkga2V5OiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgZGltZW5zaW9uIHZhbHVlXG4gICAgICovXG4gICAgcmVhZG9ubHkgdmFsdWU6IHN0cmluZztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZW5kZXIgdGhlIE1ldHJpY0RpbWVuc2lvblByb3BlcnR5IGFzIEpTT05cbiAgICpcbiAgICogQHBhcmFtIHByb3BlcnR5XG4gICAqL1xuICBleHBvcnQgZnVuY3Rpb24gTWV0cmljRGltZW5zaW9uUHJvcGVydHlUb0pzb24ocHJvcGVydHk6IE1ldHJpY0RpbWVuc2lvblByb3BlcnR5KSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIEtleTogY2RrLnN0cmluZ1RvQ2xvdWRGb3JtYXRpb24ocHJvcGVydHkua2V5KSxcbiAgICAgIFZhbHVlOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS52YWx1ZSlcbiAgICB9O1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBkZWZpbml0aW9uIG9mIGEgQ2xvdWRXYXRjaCBtZXRyaWMgYWxhcm0sIHdoaWNoIGRldGVybWluZXMgd2hlbiBhbiBhdXRvbWF0aWMgc2NhbGluZyBhY3Rpdml0eSBpcyB0cmlnZ2VyZWQuIFdoZW4gdGhlIGRlZmluZWQgYWxhcm0gY29uZGl0aW9uc1xuICAgKiBhcmUgc2F0aXNmaWVkLCBzY2FsaW5nIGFjdGl2aXR5IGJlZ2lucy5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Nsb3VkV2F0Y2hBbGFybURlZmluaXRpb24uaHRtbFxuICAgKlxuICAgKiBAZXhwZXJpbWVudGFsXG4gICAqL1xuICBleHBvcnQgaW50ZXJmYWNlIENsb3VkV2F0Y2hBbGFybURlZmluaXRpb25Qcm9wZXJ0eSB7XG4gICAgLyoqXG4gICAgICogRGV0ZXJtaW5lcyBob3cgdGhlIG1ldHJpYyBzcGVjaWZpZWQgYnkgTWV0cmljTmFtZSBpcyBjb21wYXJlZCB0byB0aGUgdmFsdWUgc3BlY2lmaWVkIGJ5IFRocmVzaG9sZFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNvbXBhcmlzb25PcGVyYXRvcjogQ2xvdWRXYXRjaEFsYXJtQ29tcGFyaXNvbk9wZXJhdG9yO1xuXG4gICAgLyoqXG4gICAgICogQSBDbG91ZFdhdGNoIG1ldHJpYyBkaW1lbnNpb25cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIGRpbWVuc2lvbnNcbiAgICAgKi9cbiAgICByZWFkb25seSBkaW1lbnNpb25zPzogTWV0cmljRGltZW5zaW9uUHJvcGVydHlbXTtcblxuICAgIC8qKlxuICAgICAqIFRoZSBudW1iZXIgb2YgcGVyaW9kcywgaW4gZml2ZS1taW51dGUgaW5jcmVtZW50cywgZHVyaW5nIHdoaWNoIHRoZSBhbGFybSBjb25kaXRpb24gbXVzdCBleGlzdCBiZWZvcmUgdGhlIGFsYXJtIHRyaWdnZXJzIGF1dG9tYXRpY1xuICAgICAqIHNjYWxpbmcgYWN0aXZpdHkuIFRoZSBkZWZhdWx0IHZhbHVlIGlzIDEuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBldmFsdWF0aW9uUGVyaW9kc1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGV2YWx1dGlvblBlcmlvZHM/OiBudW1iZXI7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgbmFtZSBvZiB0aGUgQ2xvdWRXYXRjaCBtZXRyaWMgdGhhdCBpcyB3YXRjaGVkIHRvIGRldGVybWluZSBhbiBhbGFybSBjb25kaXRpb24uXG4gICAgICovXG4gICAgcmVhZG9ubHkgbWV0cmljTmFtZTogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgICogVGhlIG5hbWVzcGFjZSBmb3IgdGhlIENsb3VkV2F0Y2ggbWV0cmljLiBUaGUgZGVmYXVsdCBpcyBBV1MvRWxhc3RpY01hcFJlZHVjZS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIG5hbXBlc3BhY2VcbiAgICAgKi9cbiAgICByZWFkb25seSBuYW1lc3BhY2U/OiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgcGVyaW9kLCBpbiBzZWNvbmRzLCBvdmVyIHdoaWNoIHRoZSBzdGF0aXN0aWMgaXMgYXBwbGllZC4gRU1SIENsb3VkV2F0Y2ggbWV0cmljcyBhcmUgZW1pdHRlZCBldmVyeSBmaXZlIG1pbnV0ZXMgKDMwMCBzZWNvbmRzKSwgc28gaWZcbiAgICAgKiBhbiBFTVIgQ2xvdWRXYXRjaCBtZXRyaWMgaXMgc3BlY2lmaWVkLCBzcGVjaWZ5IDMwMC5cbiAgICAgKi9cbiAgICByZWFkb25seSBwZXJpb2Q6IGNkay5EdXJhdGlvbjtcblxuICAgIC8qKlxuICAgICAqIFRoZSBzdGF0aXN0aWMgdG8gYXBwbHkgdG8gdGhlIG1ldHJpYyBhc3NvY2lhdGVkIHdpdGggdGhlIGFsYXJtLiBUaGUgZGVmYXVsdCBpcyBBVkVSQUdFLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gc3RhdGlzdGljXG4gICAgICovXG4gICAgcmVhZG9ubHkgc3RhdGlzdGljPzogQ2xvdWRXYXRjaEFsYXJtU3RhdGlzdGljO1xuXG4gICAgLyoqXG4gICAgICogVGhlIHZhbHVlIGFnYWluc3Qgd2hpY2ggdGhlIHNwZWNpZmllZCBzdGF0aXN0aWMgaXMgY29tcGFyZWQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyB0aHJlc2hvbGRcbiAgICAgKi9cbiAgICByZWFkb25seSB0aHJlc2hvbGQ/OiBudW1iZXI7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgdW5pdCBvZiBtZWFzdXJlIGFzc29jaWF0ZWQgd2l0aCB0aGUgQ2xvdWRXYXRjaCBtZXRyaWMgYmVpbmcgd2F0Y2hlZC4gVGhlIHZhbHVlIHNwZWNpZmllZCBmb3IgVW5pdCBtdXN0IGNvcnJlc3BvbmQgdG8gdGhlIHVuaXRzXG4gICAgICogc3BlY2lmaWVkIGluIHRoZSBDbG91ZFdhdGNoIG1ldHJpYy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIHVuaXRcbiAgICAgKi9cbiAgICByZWFkb25seSB1bml0PzogQ2xvdWRXYXRjaEFsYXJtVW5pdDtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgY29uZGl0aW9ucyB0aGF0IHRyaWdnZXIgYW4gYXV0b21hdGljIHNjYWxpbmcgYWN0aXZpdHkgYW5kIHRoZSBkZWZpbml0aW9uIG9mIGEgQ2xvdWRXYXRjaCBtZXRyaWMgYWxhcm0uXG4gICAqIFdoZW4gdGhlIGRlZmluZWQgYWxhcm0gY29uZGl0aW9ucyBhcmUgbWV0IGFsb25nIHdpdGggb3RoZXIgdHJpZ2dlciBwYXJhbWV0ZXJzLCBzY2FsaW5nIGFjdGl2aXR5IGJlZ2lucy5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1NjYWxpbmdUcmlnZ2VyLmh0bWxcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGludGVyZmFjZSBTY2FsaW5nVHJpZ2dlclByb3BlcnR5IHtcbiAgICAvKipcbiAgICAgKiBUaGUgZGVmaW5pdGlvbiBvZiBhIENsb3VkV2F0Y2ggbWV0cmljIGFsYXJtLiBXaGVuIHRoZSBkZWZpbmVkIGFsYXJtIGNvbmRpdGlvbnMgYXJlIG1ldCBhbG9uZyB3aXRoIG90aGVyIHRyaWdnZXIgcGFyYW1ldGVycyxcbiAgICAgKiBzY2FsaW5nIGFjdGl2aXR5IGJlZ2lucy5cbiAgICAgKi9cbiAgICByZWFkb25seSBjbG91ZFdhdGNoQWxhcm1EZWZpbml0aW9uOiBDbG91ZFdhdGNoQWxhcm1EZWZpbml0aW9uUHJvcGVydHk7XG4gIH1cblxuICAvKipcbiAgICogUmVuZGVyIHRoZSBTY2FsaW5nVHJpZ2dlclByb3BlcnR5IHRvIEpTT05cbiAgICpcbiAgICogQHBhcmFtIHByb3BlcnR5XG4gICAqL1xuICBleHBvcnQgZnVuY3Rpb24gU2NhbGluZ1RyaWdnZXJQcm9wZXJ0eVRvSnNvbihwcm9wZXJ0eTogU2NhbGluZ1RyaWdnZXJQcm9wZXJ0eSkge1xuICAgIHJldHVybiB7XG4gICAgICBDbG91ZFdhdGNoQWxhcm1EZWZpbml0aW9uOiB7XG4gICAgICAgIENvbXBhcmlzb25PcGVyYXRvcjogY2RrLnN0cmluZ1RvQ2xvdWRGb3JtYXRpb24ocHJvcGVydHkuY2xvdWRXYXRjaEFsYXJtRGVmaW5pdGlvbi5jb21wYXJpc29uT3BlcmF0b3I/LnZhbHVlT2YoKSksXG4gICAgICAgIERpbWVuc2lvbnM6IGNkay5saXN0TWFwcGVyKE1ldHJpY0RpbWVuc2lvblByb3BlcnR5VG9Kc29uKShwcm9wZXJ0eS5jbG91ZFdhdGNoQWxhcm1EZWZpbml0aW9uLmRpbWVuc2lvbnMpLFxuICAgICAgICBFdmFsdWF0aW9uUGVyaW9kczogY2RrLm51bWJlclRvQ2xvdWRGb3JtYXRpb24ocHJvcGVydHkuY2xvdWRXYXRjaEFsYXJtRGVmaW5pdGlvbi5ldmFsdXRpb25QZXJpb2RzKSxcbiAgICAgICAgTWV0cmljTmFtZTogY2RrLnN0cmluZ1RvQ2xvdWRGb3JtYXRpb24ocHJvcGVydHkuY2xvdWRXYXRjaEFsYXJtRGVmaW5pdGlvbi5tZXRyaWNOYW1lKSxcbiAgICAgICAgTmFtZXNwYWNlOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5jbG91ZFdhdGNoQWxhcm1EZWZpbml0aW9uLm5hbWVzcGFjZSksXG4gICAgICAgIFBlcmlvZDogY2RrLm51bWJlclRvQ2xvdWRGb3JtYXRpb24ocHJvcGVydHkuY2xvdWRXYXRjaEFsYXJtRGVmaW5pdGlvbi5wZXJpb2QudG9TZWNvbmRzKCkpLFxuICAgICAgICBTdGF0aXN0aWM6IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5LmNsb3VkV2F0Y2hBbGFybURlZmluaXRpb24uc3RhdGlzdGljPy52YWx1ZU9mKCkpLFxuICAgICAgICBUaHJlc2hvbGQ6IGNkay5udW1iZXJUb0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5LmNsb3VkV2F0Y2hBbGFybURlZmluaXRpb24udGhyZXNob2xkKSxcbiAgICAgICAgVW5pdDogY2RrLnN0cmluZ1RvQ2xvdWRGb3JtYXRpb24ocHJvcGVydHkuY2xvdWRXYXRjaEFsYXJtRGVmaW5pdGlvbi51bml0Py52YWx1ZU9mKCkpXG4gICAgICB9XG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBFQzIgSW5zdGFuY2UgTWFya2V0XG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBlbnVtIEluc3RhbmNlTWFya2V0IHtcbiAgICAvKipcbiAgICAgKiBPbiBEZW1hbmQgSW5zdGFuY2VcbiAgICAgKi9cbiAgICBPTl9ERU1BTkQgPSAnT05fREVNQU5EJyxcbiAgICAvKipcbiAgICAgKiBTcG90IEluc3RhbmNlXG4gICAgICovXG4gICAgU1BPVCA9ICdTUE9UJ1xuICB9XG5cbiAgLyoqXG4gICAqIEF1dG9TY2FsaW5nIEFkanVzdG1lbnQgVHlwZVxuICAgKlxuICAgKiBAZXhwZXJpbWVudGFsXG4gICAqL1xuICBleHBvcnQgZW51bSBTY2FsaW5nQWRqdXN0bWVudFR5cGUge1xuICAgIC8qKlxuICAgICAqIENIQU5HRV9JTl9DQVBBQ0lUWVxuICAgICAqL1xuICAgIENIQU5HRV9JTl9DQVBBQ0lUWSA9ICdDSEFOR0VfSU5fQ0FQQUNJVFknLFxuICAgIC8qKlxuICAgICAqIFBFUkNFTlRfQ0hBTkdFX0lOX0NBUEFDSVRZXG4gICAgICovXG4gICAgUEVSQ0VOVF9DSEFOR0VfSU5fQ0FQQUNJVFkgPSAnUEVSQ0VOVF9DSEFOR0VfSU5fQ0FQQUNJVFknLFxuICAgIC8qKlxuICAgICAqIEVYQUNUX0NBUEFDSVRZXG4gICAgICovXG4gICAgRVhBQ1RfQ0FQQUNJVFkgPSAnRVhBQ1RfQ0FQQUNJVFknXG4gIH1cblxuICAvKipcbiAgICogQW4gYXV0b21hdGljIHNjYWxpbmcgY29uZmlndXJhdGlvbiwgd2hpY2ggZGVzY3JpYmVzIGhvdyB0aGUgcG9saWN5IGFkZHMgb3IgcmVtb3ZlcyBpbnN0YW5jZXMsIHRoZSBjb29sZG93biBwZXJpb2QsIGFuZCB0aGUgbnVtYmVyIG9mIEVDMlxuICAgKiBpbnN0YW5jZXMgdGhhdCB3aWxsIGJlIGFkZGVkIGVhY2ggdGltZSB0aGUgQ2xvdWRXYXRjaCBtZXRyaWMgYWxhcm0gY29uZGl0aW9uIGlzIHNhdGlzZmllZC5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1NpbXBsZVNjYWxpbmdQb2xpY3lDb25maWd1cmF0aW9uLmh0bWxcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGludGVyZmFjZSBTaW1wbGVTY2FsaW5nUG9saWN5Q29uZmlndXJhdGlvblByb3BlcnR5IHtcbiAgICAvKipcbiAgICAgKiBUaGUgd2F5IGluIHdoaWNoIEVDMiBpbnN0YW5jZXMgYXJlIGFkZGVkIChpZiBTY2FsaW5nQWRqdXN0bWVudCBpcyBhIHBvc2l0aXZlIG51bWJlcikgb3IgdGVybWluYXRlZCAoaWYgU2NhbGluZ0FkanVzdG1lbnQgaXMgYSBuZWdhdGl2ZVxuICAgICAqIG51bWJlcikgZWFjaCB0aW1lIHRoZSBzY2FsaW5nIGFjdGl2aXR5IGlzIHRyaWdnZXJlZC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIGFkanVzdG1lbnRUeXBlXG4gICAgICovXG4gICAgcmVhZG9ubHkgYWRqdXN0bWVudFR5cGU/OiBTY2FsaW5nQWRqdXN0bWVudFR5cGU7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgYW1vdW50IG9mIHRpbWUsIGluIHNlY29uZHMsIGFmdGVyIGEgc2NhbGluZyBhY3Rpdml0eSBjb21wbGV0ZXMgYmVmb3JlIGFueSBmdXJ0aGVyIHRyaWdnZXItcmVsYXRlZCBzY2FsaW5nIGFjdGl2aXRpZXMgY2FuIHN0YXJ0LlxuICAgICAqIFRoZSBkZWZhdWx0IHZhbHVlIGlzIDAuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBjb29sRG93blxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNvb2xEb3duPzogbnVtYmVyO1xuXG4gICAgLyoqXG4gICAgICogVGhlIGFtb3VudCBieSB3aGljaCB0byBzY2FsZSBpbiBvciBzY2FsZSBvdXQsIGJhc2VkIG9uIHRoZSBzcGVjaWZpZWQgQWRqdXN0bWVudFR5cGUuIEEgcG9zaXRpdmUgdmFsdWUgYWRkcyB0byB0aGUgaW5zdGFuY2UgZ3JvdXAnc1xuICAgICAqIEVDMiBpbnN0YW5jZSBjb3VudCB3aGlsZSBhIG5lZ2F0aXZlIG51bWJlciByZW1vdmVzIGluc3RhbmNlcy4gSWYgQWRqdXN0bWVudFR5cGUgaXMgc2V0IHRvIEVYQUNUX0NBUEFDSVRZLCB0aGUgbnVtYmVyIHNob3VsZCBvbmx5IGJlXG4gICAgICogYSBwb3NpdGl2ZSBpbnRlZ2VyLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHNjYWxpbmdBZGp1c3RtZW50OiBudW1iZXI7XG4gIH1cblxuICAvKipcbiAgICogVGhlIHR5cGUgb2YgYWRqdXN0bWVudCB0aGUgYXV0b21hdGljIHNjYWxpbmcgYWN0aXZpdHkgbWFrZXMgd2hlbiB0cmlnZ2VyZWQsIGFuZCB0aGUgcGVyaW9kaWNpdHkgb2YgdGhlIGFkanVzdG1lbnQuXG4gICAqIEFuZCBhbiBhdXRvbWF0aWMgc2NhbGluZyBjb25maWd1cmF0aW9uLCB3aGljaCBkZXNjcmliZXMgaG93IHRoZSBwb2xpY3kgYWRkcyBvciByZW1vdmVzIGluc3RhbmNlcywgdGhlIGNvb2xkb3duIHBlcmlvZCxcbiAgICogYW5kIHRoZSBudW1iZXIgb2YgRUMyIGluc3RhbmNlcyB0aGF0IHdpbGwgYmUgYWRkZWQgZWFjaCB0aW1lIHRoZSBDbG91ZFdhdGNoIG1ldHJpYyBhbGFybSBjb25kaXRpb24gaXMgc2F0aXNmaWVkLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU2NhbGluZ0FjdGlvbi5odG1sXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBpbnRlcmZhY2UgU2NhbGluZ0FjdGlvblByb3BlcnR5IHtcbiAgICAvKipcbiAgICAgKiBOb3QgYXZhaWxhYmxlIGZvciBpbnN0YW5jZSBncm91cHMuIEluc3RhbmNlIGdyb3VwcyB1c2UgdGhlIG1hcmtldCB0eXBlIHNwZWNpZmllZCBmb3IgdGhlIGdyb3VwLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgRU1SIHNlbGVjdGVkIGRlZmF1bHRcbiAgICAgKi9cbiAgICByZWFkb25seSBtYXJrZXQ/OiBJbnN0YW5jZU1hcmtldDtcblxuICAgIC8qKlxuICAgICAqIFRoZSB0eXBlIG9mIGFkanVzdG1lbnQgdGhlIGF1dG9tYXRpYyBzY2FsaW5nIGFjdGl2aXR5IG1ha2VzIHdoZW4gdHJpZ2dlcmVkLCBhbmQgdGhlIHBlcmlvZGljaXR5IG9mIHRoZSBhZGp1c3RtZW50LlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHNpbXBsZVNjYWxpbmdQb2xpY3lDb25maWd1cmF0aW9uOiBTaW1wbGVTY2FsaW5nUG9saWN5Q29uZmlndXJhdGlvblByb3BlcnR5O1xuICB9XG5cbiAgLyoqXG4gICAqIFJlbmRlciB0aGUgU2NhbGluZ0FjdGlvblByb3BldHkgdG8gSlNPTlxuICAgKlxuICAgKiBAcGFyYW0gcHJvcGVydHlcbiAgICovXG4gIGV4cG9ydCBmdW5jdGlvbiBTY2FsaW5nQWN0aW9uUHJvcGVydHlUb0pzb24ocHJvcGVydHk6IFNjYWxpbmdBY3Rpb25Qcm9wZXJ0eSkge1xuICAgIHJldHVybiB7XG4gICAgICBNYXJrZXQ6IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5Lm1hcmtldD8udmFsdWVPZigpKSxcbiAgICAgIFNpbXBsZVNjYWxpbmdQb2xpY3lDb25maWd1cmF0aW9uOiB7XG4gICAgICAgIEFkanVzdG1lbnRUeXBlOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5zaW1wbGVTY2FsaW5nUG9saWN5Q29uZmlndXJhdGlvbi5hZGp1c3RtZW50VHlwZSksXG4gICAgICAgIENvb2xEb3duOiBjZGsubnVtYmVyVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5zaW1wbGVTY2FsaW5nUG9saWN5Q29uZmlndXJhdGlvbi5jb29sRG93biksXG4gICAgICAgIFNjYWxpbmdBZGp1c3RtZW50OiBjZGsubnVtYmVyVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5zaW1wbGVTY2FsaW5nUG9saWN5Q29uZmlndXJhdGlvbi5zY2FsaW5nQWRqdXN0bWVudClcbiAgICAgIH1cbiAgICB9O1xuICB9XG5cbiAgLyoqXG4gICAqIEEgc2NhbGUtaW4gb3Igc2NhbGUtb3V0IHJ1bGUgdGhhdCBkZWZpbmVzIHNjYWxpbmcgYWN0aXZpdHksIGluY2x1ZGluZyB0aGUgQ2xvdWRXYXRjaCBtZXRyaWMgYWxhcm0gdGhhdCB0cmlnZ2VycyBhY3Rpdml0eSwgaG93IEVDMlxuICAgKiBpbnN0YW5jZXMgYXJlIGFkZGVkIG9yIHJlbW92ZWQsIGFuZCB0aGUgcGVyaW9kaWNpdHkgb2YgYWRqdXN0bWVudHMuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TY2FsaW5nUnVsZS5odG1sXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBpbnRlcmZhY2UgU2NhbGluZ1J1bGVQcm9wZXJ0eSB7XG4gICAgLyoqXG4gICAgICogVGhlIGNvbmRpdGlvbnMgdGhhdCB0cmlnZ2VyIGFuIGF1dG9tYXRpYyBzY2FsaW5nIGFjdGl2aXR5LlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFjdGlvbjogU2NhbGluZ0FjdGlvblByb3BlcnR5O1xuXG4gICAgLyoqXG4gICAgICogQSBmcmllbmRseSwgbW9yZSB2ZXJib3NlIGRlc2NyaXB0aW9uIG9mIHRoZSBhdXRvbWF0aWMgc2NhbGluZyBydWxlLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gZGVzY3JpcHRpb25cbiAgICAgKi9cbiAgICByZWFkb25seSBkZXNjcmlwdGlvbj86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFRoZSBuYW1lIHVzZWQgdG8gaWRlbnRpZnkgYW4gYXV0b21hdGljIHNjYWxpbmcgcnVsZS4gUnVsZSBuYW1lcyBtdXN0IGJlIHVuaXF1ZSB3aXRoaW4gYSBzY2FsaW5nIHBvbGljeS5cbiAgICAgKi9cbiAgICByZWFkb25seSBuYW1lOiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgQ2xvdWRXYXRjaCBhbGFybSBkZWZpbml0aW9uIHRoYXQgZGV0ZXJtaW5lcyB3aGVuIGF1dG9tYXRpYyBzY2FsaW5nIGFjdGl2aXR5IGlzIHRyaWdnZXJlZC5cbiAgICAgKi9cbiAgICByZWFkb25seSB0cmlnZ2VyOiBTY2FsaW5nVHJpZ2dlclByb3BlcnR5O1xuICB9XG5cbiAgLyoqXG4gICAqIFJlbmRlciB0aGUgU2NhbGluZ1J1bGVQcm9wZXJ0eSB0byBKU09OXG4gICAqXG4gICAqIEBwYXJhbSBwcm9wZXJ0eVxuICAgKi9cbiAgZXhwb3J0IGZ1bmN0aW9uIFNjYWxpbmdSdWxlUHJvcGVydHlUb0pzb24ocHJvcGVydHk6IFNjYWxpbmdSdWxlUHJvcGVydHkpIHtcbiAgICByZXR1cm4ge1xuICAgICAgQWN0aW9uOiBTY2FsaW5nQWN0aW9uUHJvcGVydHlUb0pzb24ocHJvcGVydHkuYWN0aW9uKSxcbiAgICAgIERlc2NyaXB0aW9uOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5kZXNjcmlwdGlvbiksXG4gICAgICBOYW1lOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5uYW1lKSxcbiAgICAgIFRyaWdnZXI6IFNjYWxpbmdUcmlnZ2VyUHJvcGVydHlUb0pzb24ocHJvcGVydHkudHJpZ2dlcilcbiAgICB9O1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSB1cHBlciBhbmQgbG93ZXIgRUMyIGluc3RhbmNlIGxpbWl0cyBmb3IgYW4gYXV0b21hdGljIHNjYWxpbmcgcG9saWN5LiBBdXRvbWF0aWMgc2NhbGluZyBhY3Rpdml0aWVzIHRyaWdnZXJlZCBieSBhdXRvbWF0aWMgc2NhbGluZ1xuICAgKiBydWxlcyB3aWxsIG5vdCBjYXVzZSBhbiBpbnN0YW5jZSBncm91cCB0byBncm93IGFib3ZlIG9yIGJlbG93IHRoZXNlIGxpbWl0cy5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1NjYWxpbmdDb25zdHJhaW50cy5odG1sXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBpbnRlcmZhY2UgU2NhbGluZ0NvbnN0cmFpbnRzUHJvcGVydHkge1xuICAgIC8qKlxuICAgICAqIFRoZSB1cHBlciBib3VuZGFyeSBvZiBFQzIgaW5zdGFuY2VzIGluIGFuIGluc3RhbmNlIGdyb3VwIGJleW9uZCB3aGljaCBzY2FsaW5nIGFjdGl2aXRpZXMgYXJlIG5vdCBhbGxvd2VkIHRvIGdyb3cuIFNjYWxlLW91dFxuICAgICAqIGFjdGl2aXRpZXMgd2lsbCBub3QgYWRkIGluc3RhbmNlcyBiZXlvbmQgdGhpcyBib3VuZGFyeS5cbiAgICAgKi9cbiAgICByZWFkb25seSBtYXhDYXBhY2l0eTogbnVtYmVyO1xuXG4gICAgLyoqXG4gICAgICogVGhlIGxvd2VyIGJvdW5kYXJ5IG9mIEVDMiBpbnN0YW5jZXMgaW4gYW4gaW5zdGFuY2UgZ3JvdXAgYmVsb3cgd2hpY2ggc2NhbGluZyBhY3Rpdml0aWVzIGFyZSBub3QgYWxsb3dlZCB0byBzaHJpbmsuIFNjYWxlLWluXG4gICAgICogYWN0aXZpdGllcyB3aWxsIG5vdCB0ZXJtaW5hdGUgaW5zdGFuY2VzIGJlbG93IHRoaXMgYm91bmRhcnkuXG4gICAgICovXG4gICAgcmVhZG9ubHkgbWluQ2FwYWNpdHk6IG51bWJlcjtcbiAgfVxuXG4gIC8qKlxuICAgKiBBbiBhdXRvbWF0aWMgc2NhbGluZyBwb2xpY3kgZm9yIGEgY29yZSBpbnN0YW5jZSBncm91cCBvciB0YXNrIGluc3RhbmNlIGdyb3VwIGluIGFuIEFtYXpvbiBFTVIgY2x1c3Rlci5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0F1dG9TY2FsaW5nUG9saWN5Lmh0bWxcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGludGVyZmFjZSBBdXRvU2NhbGluZ1BvbGljeVByb3BlcnR5IHtcbiAgICAvKipcbiAgICAgKiBUaGUgdXBwZXIgYW5kIGxvd2VyIEVDMiBpbnN0YW5jZSBsaW1pdHMgZm9yIGFuIGF1dG9tYXRpYyBzY2FsaW5nIHBvbGljeS4gQXV0b21hdGljIHNjYWxpbmcgYWN0aXZpdHkgd2lsbCBub3QgY2F1c2UgYW4gaW5zdGFuY2VcbiAgICAgKiBncm91cCB0byBncm93IGFib3ZlIG9yIGJlbG93IHRoZXNlIGxpbWl0cy5cbiAgICAgKi9cbiAgICByZWFkb25seSBjb25zdHJhaW50czogU2NhbGluZ0NvbnN0cmFpbnRzUHJvcGVydHk7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgc2NhbGUtaW4gYW5kIHNjYWxlLW91dCBydWxlcyB0aGF0IGNvbXByaXNlIHRoZSBhdXRvbWF0aWMgc2NhbGluZyBwb2xpY3kuXG4gICAgICovXG4gICAgcmVhZG9ubHkgcnVsZXM6IFNjYWxpbmdSdWxlUHJvcGVydHlbXTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZW5kZXIgdGhlIEF1dG9TY2FsaW5nUG9saWN5UHJvcGVydHkgdG8gSlNPTlxuICAgKlxuICAgKiBAcGFyYW0gcHJvcGVydHlcbiAgICovXG4gIGV4cG9ydCBmdW5jdGlvbiBBdXRvU2NhbGluZ1BvbGljeVByb3BlcnR5VG9Kc29uKHByb3BlcnR5OiBBdXRvU2NhbGluZ1BvbGljeVByb3BlcnR5KSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIENvbnN0cmFpbnRzOiB7XG4gICAgICAgIE1heENhcGFjaXR5OiBjZGsubnVtYmVyVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5jb25zdHJhaW50cy5tYXhDYXBhY2l0eSksXG4gICAgICAgIE1pbkNhcGFjaXR5OiBjZGsubnVtYmVyVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5jb25zdHJhaW50cy5taW5DYXBhY2l0eSlcbiAgICAgIH0sXG4gICAgICBSdWxlczogY2RrLmxpc3RNYXBwZXIoU2NhbGluZ1J1bGVQcm9wZXJ0eVRvSnNvbikocHJvcGVydHkucnVsZXMpXG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb25maWd1cmF0aW9uIGRlZmluaW5nIGEgbmV3IGluc3RhbmNlIGdyb3VwLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfSW5zdGFuY2VHcm91cENvbmZpZy5odG1sXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBpbnRlcmZhY2UgSW5zdGFuY2VHcm91cENvbmZpZ1Byb3BlcnR5IHtcbiAgICAvKipcbiAgICAgKiBBbiBhdXRvbWF0aWMgc2NhbGluZyBwb2xpY3kgZm9yIGEgY29yZSBpbnN0YW5jZSBncm91cCBvciB0YXNrIGluc3RhbmNlIGdyb3VwIGluIGFuIEFtYXpvbiBFTVIgY2x1c3Rlci5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIGF1dG9TY2FsaW5nUG9saWN5XG4gICAgICovXG4gICAgcmVhZG9ubHkgYXV0b1NjYWxpbmdQb2xpY3k/OiBBdXRvU2NhbGluZ1BvbGljeVByb3BlcnR5O1xuXG4gICAgLyoqXG4gICAgICogVGhlIGJpZCBwcmljZSBmb3IgZWFjaCBFQzIgU3BvdCBpbnN0YW5jZSB0eXBlIGFzIGRlZmluZWQgYnkgSW5zdGFuY2VUeXBlLiBFeHByZXNzZWQgaW4gVVNELlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gYmlkUHJpY2VcbiAgICAgKi9cbiAgICByZWFkb25seSBiaWRQcmljZT86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFRoZSBsaXN0IG9mIGNvbmZpZ3VyYXRpb25zIHN1cHBsaWVkIGZvciBhbiBFTVIgY2x1c3RlciBpbnN0YW5jZSBncm91cC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIGNvbmZpZ3VyYXRpb25zXG4gICAgICovXG4gICAgcmVhZG9ubHkgY29uZmlndXJhdGlvbnM/OiBDb25maWd1cmF0aW9uUHJvcGVydHlbXTtcblxuICAgIC8qKlxuICAgICAqIEVCUyBjb25maWd1cmF0aW9ucyB0aGF0IHdpbGwgYmUgYXR0YWNoZWQgdG8gZWFjaCBFQzIgaW5zdGFuY2UgaW4gdGhlIGluc3RhbmNlIGdyb3VwLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gZWJzQ29uZmlndXJhdGlvblxuICAgICAqL1xuICAgIHJlYWRvbmx5IGVic0NvbmZpZ3VyYXRpb24/OiBFYnNDb25maWd1cmF0aW9uUHJvcGVydHk7XG5cbiAgICAvKipcbiAgICAgKiBUYXJnZXQgbnVtYmVyIG9mIGluc3RhbmNlcyBmb3IgdGhlIGluc3RhbmNlIGdyb3VwLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGluc3RhbmNlQ291bnQ6IG51bWJlcjtcblxuICAgIC8qKlxuICAgICAqIFRoZSByb2xlIG9mIHRoZSBpbnN0YW5jZSBncm91cCBpbiB0aGUgY2x1c3Rlci5cbiAgICAgKi9cbiAgICByZWFkb25seSBpbnN0YW5jZVJvbGU6IEluc3RhbmNlUm9sZVR5cGU7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgRUMyIGluc3RhbmNlIHR5cGUgZm9yIGFsbCBpbnN0YW5jZXMgaW4gdGhlIGluc3RhbmNlIGdyb3VwLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGluc3RhbmNlVHlwZTogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgICogTWFya2V0IHR5cGUgb2YgdGhlIEVDMiBpbnN0YW5jZXMgdXNlZCB0byBjcmVhdGUgYSBjbHVzdGVyIG5vZGUuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBFTVIgc2VsZWN0ZWQgZGVmYXVsdFxuICAgICAqL1xuICAgIHJlYWRvbmx5IG1hcmtldD86IEluc3RhbmNlTWFya2V0O1xuXG4gICAgLyoqXG4gICAgICogRnJpZW5kbHkgbmFtZSBnaXZlbiB0byB0aGUgaW5zdGFuY2UgZ3JvdXAuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBuYW1lXG4gICAgICovXG4gICAgcmVhZG9ubHkgbmFtZT86IHN0cmluZztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZW5kZXIgdGhlIEluc3RhbmNlR3JvdXBDb25maWdQcm9wZXJ0eSB0byBKU09OXG4gICAqXG4gICAqIEBwYXJhbSBwcm9wZXJ0eVxuICAgKi9cbiAgZXhwb3J0IGZ1bmN0aW9uIEluc3RhbmNlR3JvdXBDb25maWdQcm9wZXJ0eVRvSnNvbihwcm9wZXJ0eTogSW5zdGFuY2VHcm91cENvbmZpZ1Byb3BlcnR5KSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIEF1dG9TY2FsaW5nUG9saWN5OiAocHJvcGVydHkuYXV0b1NjYWxpbmdQb2xpY3kgPT09IHVuZGVmaW5lZCkgP1xuICAgICAgICBwcm9wZXJ0eS5hdXRvU2NhbGluZ1BvbGljeSA6XG4gICAgICAgIEF1dG9TY2FsaW5nUG9saWN5UHJvcGVydHlUb0pzb24ocHJvcGVydHkuYXV0b1NjYWxpbmdQb2xpY3kpLFxuICAgICAgQmlkUHJpY2U6IGNkay5udW1iZXJUb0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5LmJpZFByaWNlKSxcbiAgICAgIENvbmZpZ3VyYXRpb25zOiBjZGsubGlzdE1hcHBlcihDb25maWd1cmF0aW9uUHJvcGVydHlUb0pzb24pKHByb3BlcnR5LmNvbmZpZ3VyYXRpb25zKSxcbiAgICAgIEVic0NvbmZpZ3VyYXRpb246IChwcm9wZXJ0eS5lYnNDb25maWd1cmF0aW9uID09PSB1bmRlZmluZWQpID9cbiAgICAgICAgcHJvcGVydHkuZWJzQ29uZmlndXJhdGlvbiA6XG4gICAgICAgIEVic0NvbmZpZ3VyYXRpb25Qcm9wZXJ0eVRvSnNvbihwcm9wZXJ0eS5lYnNDb25maWd1cmF0aW9uKSxcbiAgICAgIEluc3RhbmNlQ291bnQ6IGNkay5udW1iZXJUb0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5Lmluc3RhbmNlQ291bnQpLFxuICAgICAgSW5zdGFuY2VSb2xlOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5pbnN0YW5jZVJvbGU/LnZhbHVlT2YoKSksXG4gICAgICBJbnN0YW5jZVR5cGU6IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5Lmluc3RhbmNlVHlwZSksXG4gICAgICBNYXJrZXQ6IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5Lm1hcmtldD8udmFsdWVPZigpKSxcbiAgICAgIE5hbWU6IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5Lm5hbWUpXG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgQW1hem9uIEVDMiBBdmFpbGFiaWxpdHkgWm9uZSBjb25maWd1cmF0aW9uIG9mIHRoZSBjbHVzdGVyIChqb2IgZmxvdykuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QbGFjZW1lbnRUeXBlLmh0bWxcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGludGVyZmFjZSBQbGFjZW1lbnRUeXBlUHJvcGVydHkge1xuICAgIC8qKlxuICAgICAqIFRoZSBBbWF6b24gRUMyIEF2YWlsYWJpbGl0eSBab25lIGZvciB0aGUgY2x1c3Rlci4gQXZhaWxhYmlsaXR5Wm9uZSBpcyB1c2VkIGZvciB1bmlmb3JtIGluc3RhbmNlIGdyb3Vwcywgd2hpbGUgQXZhaWxhYmlsaXR5Wm9uZXNcbiAgICAgKiAocGx1cmFsKSBpcyB1c2VkIGZvciBpbnN0YW5jZSBmbGVldHMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBFTVIgc2VsZWN0ZWQgZGVmYXVsdFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGF2YWlsYWJpbGl0eVpvbmU/OiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBXaGVuIG11bHRpcGxlIEF2YWlsYWJpbGl0eSBab25lcyBhcmUgc3BlY2lmaWVkLCBBbWF6b24gRU1SIGV2YWx1YXRlcyB0aGVtIGFuZCBsYXVuY2hlcyBpbnN0YW5jZXMgaW4gdGhlIG9wdGltYWwgQXZhaWxhYmlsaXR5IFpvbmUuXG4gICAgICogQXZhaWxhYmlsaXR5Wm9uZXMgaXMgdXNlZCBmb3IgaW5zdGFuY2UgZmxlZXRzLCB3aGlsZSBBdmFpbGFiaWxpdHlab25lIChzaW5ndWxhcikgaXMgdXNlZCBmb3IgdW5pZm9ybSBpbnN0YW5jZSBncm91cHMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBFTVIgc2VsZWN0ZWQgZGVmYXVsdFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGF2YWlsYWJpbGl0eVpvbmVzPzogc3RyaW5nW107XG4gIH1cblxuICAvKipcbiAgICogUmVuZGVyIHRoZSBQbGFjZW1lbnRUeXBlUHJvcGVydHkgdG8gSlNPTlxuICAgKlxuICAgKiBAcGFyYW0gcHJvcGVydHlcbiAgICovXG4gIGV4cG9ydCBmdW5jdGlvbiBQbGFjZW1lbnRUeXBlUHJvcGVydHlUb0pzb24ocHJvcGVydHk6IFBsYWNlbWVudFR5cGVQcm9wZXJ0eSkge1xuICAgIHJldHVybiB7XG4gICAgICBBdmFpbGFiaWxpdHlab25lOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5hdmFpbGFiaWxpdHlab25lKSxcbiAgICAgIEF2YWlsYWJpbGl0eVpvbmVzOiBjZGsubGlzdE1hcHBlcihjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbikocHJvcGVydHkuYXZhaWxhYmlsaXR5Wm9uZXMpXG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBIHNwZWNpZmljYXRpb24gb2YgdGhlIG51bWJlciBhbmQgdHlwZSBvZiBBbWF6b24gRUMyIGluc3RhbmNlcy5cbiAgICpcbiAgICogU2VlIHRoZSBSdW5Kb2JGbG93IEFQSSBmb3IgY29tcGxldGUgZG9jdW1lbnRhdGlvbiBvbiBpbnB1dCBwYXJhbWV0ZXJzXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Kb2JGbG93SW5zdGFuY2VzQ29uZmlnLmh0bWxcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGludGVyZmFjZSBJbnN0YW5jZXNDb25maWdQcm9wZXJ0eSB7XG4gICAgLyoqXG4gICAgICogQSBsaXN0IG9mIGFkZGl0aW9uYWwgQW1hem9uIEVDMiBzZWN1cml0eSBncm91cCBJRHMgZm9yIHRoZSBtYXN0ZXIgbm9kZS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIGFkZGl0aW9uYWxNYXN0ZXJTZWN1cml0eUdyb3Vwc1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGFkZGl0aW9uYWxNYXN0ZXJTZWN1cml0eUdyb3Vwcz86IHN0cmluZ1tdO1xuXG4gICAgLyoqXG4gICAgICogQSBsaXN0IG9mIGFkZGl0aW9uYWwgQW1hem9uIEVDMiBzZWN1cml0eSBncm91cCBJRHMgZm9yIHRoZSBjb3JlIGFuZCB0YXNrIG5vZGVzLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gYWRkaXRpb25hbFNsYXZlU2VjdXJpdHlHcm91cHNcbiAgICAgKi9cbiAgICByZWFkb25seSBhZGRpdGlvbmFsU2xhdmVTZWN1cml0eUdyb3Vwcz86IHN0cmluZ1tdO1xuXG4gICAgLyoqXG4gICAgICogVGhlIG5hbWUgb2YgdGhlIEVDMiBrZXkgcGFpciB0aGF0IGNhbiBiZSB1c2VkIHRvIHNzaCB0byB0aGUgbWFzdGVyIG5vZGUgYXMgdGhlIHVzZXIgY2FsbGVkIFwiaGFkb29wLlwiXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBlYzJLZXlOYW1lXG4gICAgICovXG4gICAgcmVhZG9ubHkgZWMyS2V5TmFtZT86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIEFwcGxpZXMgdG8gY2x1c3RlcnMgdGhhdCB1c2UgdGhlIHVuaWZvcm0gaW5zdGFuY2UgZ3JvdXAgY29uZmlndXJhdGlvbi4gVG8gbGF1bmNoIHRoZSBjbHVzdGVyIGluIEFtYXpvbiBWaXJ0dWFsIFByaXZhdGUgQ2xvdWQgKEFtYXpvbiBWUEMpLFxuICAgICAqIHNldCB0aGlzIHBhcmFtZXRlciB0byB0aGUgaWRlbnRpZmllciBvZiB0aGUgQW1hem9uIFZQQyBzdWJuZXQgd2hlcmUgeW91IHdhbnQgdGhlIGNsdXN0ZXIgdG8gbGF1bmNoLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgRU1SIHNlbGVjdGVkIGRlZmF1bHRcbiAgICAgKi9cbiAgICByZWFkb25seSBlYzJTdWJuZXRJZD86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIEFwcGxpZXMgdG8gY2x1c3RlcnMgdGhhdCB1c2UgdGhlIGluc3RhbmNlIGZsZWV0IGNvbmZpZ3VyYXRpb24uIFdoZW4gbXVsdGlwbGUgRUMyIHN1Ym5ldCBJRHMgYXJlIHNwZWNpZmllZCwgQW1hem9uIEVNUiBldmFsdWF0ZXMgdGhlbSBhbmRcbiAgICAgKiBsYXVuY2hlcyBpbnN0YW5jZXMgaW4gdGhlIG9wdGltYWwgc3VibmV0LlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgRU1SIHNlbGVjdGVkIGRlZmF1bHRcbiAgICAgKi9cbiAgICByZWFkb25seSBlYzJTdWJuZXRJZHM/OiBzdHJpbmdbXTtcblxuICAgIC8qKlxuICAgICAqIFRoZSBpZGVudGlmaWVyIG9mIHRoZSBBbWF6b24gRUMyIHNlY3VyaXR5IGdyb3VwIGZvciB0aGUgbWFzdGVyIG5vZGUuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBlbXJNYW5hZ2VkTWFzdGVyU2VjdXJpdHlHcm91cFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGVtck1hbmFnZWRNYXN0ZXJTZWN1cml0eUdyb3VwPzogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgICogVGhlIGlkZW50aWZpZXIgb2YgdGhlIEFtYXpvbiBFQzIgc2VjdXJpdHkgZ3JvdXAgZm9yIHRoZSBjb3JlIGFuZCB0YXNrIG5vZGVzLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gZW1yTWFuYWdlZFNsYXZlU2VjdXJpdHlHcm91cFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGVtck1hbmFnZWRTbGF2ZVNlY3VyaXR5R3JvdXA/OiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBBcHBsaWVzIG9ubHkgdG8gQW1hem9uIEVNUiByZWxlYXNlIHZlcnNpb25zIGVhcmxpZXIgdGhhbiA0LjAuIFRoZSBIYWRvb3AgdmVyc2lvbiBmb3IgdGhlIGNsdXN0ZXIuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBoYWRvb3BWZXJzaW9uXG4gICAgICovXG4gICAgcmVhZG9ubHkgaGFkb29wVmVyc2lvbj86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFRoZSBudW1iZXIgb2YgRUMyIGluc3RhbmNlcyBpbiB0aGUgY2x1c3Rlci5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIGluc3RhbmNlQ291bnRcbiAgICAgKi9cbiAgICByZWFkb25seSBpbnN0YW5jZUNvdW50PzogbnVtYmVyO1xuXG4gICAgLyoqXG4gICAgICogRGVzY3JpYmVzIHRoZSBFQzIgaW5zdGFuY2VzIGFuZCBpbnN0YW5jZSBjb25maWd1cmF0aW9ucyBmb3IgY2x1c3RlcnMgdGhhdCB1c2UgdGhlIGluc3RhbmNlIGZsZWV0IGNvbmZpZ3VyYXRpb24uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBpbnN0YW5jZUZsZWV0c1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGluc3RhbmNlRmxlZXRzPzogSW5zdGFuY2VGbGVldENvbmZpZ1Byb3BlcnR5W107XG5cbiAgICAvKipcbiAgICAgKiBDb25maWd1cmF0aW9uIGZvciB0aGUgaW5zdGFuY2UgZ3JvdXBzIGluIGEgY2x1c3Rlci5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIGluc3RhbmNlR3JvdXBzXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW5zdGFuY2VHcm91cHM/OiBJbnN0YW5jZUdyb3VwQ29uZmlnUHJvcGVydHlbXTtcblxuICAgIC8qKlxuICAgICAqIFRoZSBFQzIgaW5zdGFuY2UgdHlwZSBvZiB0aGUgbWFzdGVyIG5vZGUuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBtYXN0ZXJJbnN0YW5jZVR5cGVcbiAgICAgKi9cbiAgICByZWFkb25seSBtYXN0ZXJJbnN0YW5jZVR5cGU/OiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgQXZhaWxhYmlsaXR5IFpvbmUgaW4gd2hpY2ggdGhlIGNsdXN0ZXIgcnVucy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IEVNUiBzZWxlY3RlZCBkZWZhdWx0XG4gICAgICovXG4gICAgcmVhZG9ubHkgcGxhY2VtZW50PzogUGxhY2VtZW50VHlwZVByb3BlcnR5O1xuXG4gICAgLyoqXG4gICAgICogVGhlIGlkZW50aWZpZXIgb2YgdGhlIEFtYXpvbiBFQzIgc2VjdXJpdHkgZ3JvdXAgZm9yIHRoZSBBbWF6b24gRU1SIHNlcnZpY2UgdG8gYWNjZXNzIGNsdXN0ZXJzIGluIFZQQyBwcml2YXRlIHN1Ym5ldHMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBzZXJ2aWNlQWNjZXNzU2VjdXJpdHlHcm91cFxuICAgICAqL1xuICAgIHJlYWRvbmx5IHNlcnZpY2VBY2Nlc3NTZWN1cml0eUdyb3VwPzogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgICogVGhlIEVDMiBpbnN0YW5jZSB0eXBlIG9mIHRoZSBjb3JlIGFuZCB0YXNrIG5vZGVzLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gc2xhdmVJbnN0YW5jZVRocGVcbiAgICAgKi9cbiAgICByZWFkb25seSBzbGF2ZUluc3RhbmNlVHlwZT86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFNwZWNpZmllcyB3aGV0aGVyIHRvIGxvY2sgdGhlIGNsdXN0ZXIgdG8gcHJldmVudCB0aGUgQW1hem9uIEVDMiBpbnN0YW5jZXMgZnJvbSBiZWluZyB0ZXJtaW5hdGVkIGJ5IEFQSSBjYWxsLCB1c2VyIGludGVydmVudGlvbixcbiAgICAgKiBvciBpbiB0aGUgZXZlbnQgb2YgYSBqb2ItZmxvdyBlcnJvci5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IEVNUiBzZWxlY3RlZCBkZWZhdWx0IChmYWxzZSlcbiAgICAgKi9cbiAgICByZWFkb25seSB0ZXJtaW5hdGlvblByb3RlY3RlZD86IGJvb2xlYW47XG4gIH1cblxuICAvKipcbiAgICogUmVuZGVyIHRoZSBJbnN0YW5jZXNDb25maWdQcm9wZXJ0eSB0byBKU09OXG4gICAqXG4gICAqIEBwYXJhbSBwcm9wZXJ0eVxuICAgKi9cbiAgZXhwb3J0IGZ1bmN0aW9uIEluc3RhbmNlc0NvbmZpZ1Byb3BlcnR5VG9Kc29uKHByb3BlcnR5OiBJbnN0YW5jZXNDb25maWdQcm9wZXJ0eSkge1xuICAgIHJldHVybiB7XG4gICAgICBBZGRpdGlvbmFsTWFzdGVyU2VjdXJpdHlHcm91cHM6IGNkay5saXN0TWFwcGVyKGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKShwcm9wZXJ0eS5hZGRpdGlvbmFsTWFzdGVyU2VjdXJpdHlHcm91cHMpLFxuICAgICAgQWRkaXRpb25hbFNsYXZlU2VjdXJpdHlHcm91cHM6IGNkay5saXN0TWFwcGVyKGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKShwcm9wZXJ0eS5hZGRpdGlvbmFsU2xhdmVTZWN1cml0eUdyb3VwcyksXG4gICAgICBFYzJLZXlOYW1lOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5lYzJLZXlOYW1lKSxcbiAgICAgIEVjMlN1Ym5ldElkOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5lYzJTdWJuZXRJZCksXG4gICAgICBFYzJTdWJuZXRJZHM6IGNkay5saXN0TWFwcGVyKGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKShwcm9wZXJ0eS5lYzJTdWJuZXRJZHMpLFxuICAgICAgRW1yTWFuYWdlZE1hc3RlclNlY3VyaXR5R3JvdXA6IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5LmVtck1hbmFnZWRNYXN0ZXJTZWN1cml0eUdyb3VwKSxcbiAgICAgIEVtck1hbmFnZWRTbGF2ZVNlY3VyaXR5R3JvdXA6IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5LmVtck1hbmFnZWRTbGF2ZVNlY3VyaXR5R3JvdXApLFxuICAgICAgSGFkb29wVmVyc2lvbjogY2RrLnN0cmluZ1RvQ2xvdWRGb3JtYXRpb24ocHJvcGVydHkuaGFkb29wVmVyc2lvbiksXG4gICAgICBJbnN0YW5jZUNvdW50OiBjZGsubnVtYmVyVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5pbnN0YW5jZUNvdW50KSxcbiAgICAgIEluc3RhbmNlRmxlZXRzOiBjZGsubGlzdE1hcHBlcihJbnN0YW5jZUZsZWV0Q29uZmlnUHJvcGVydHlUb0pzb24pKHByb3BlcnR5Lmluc3RhbmNlRmxlZXRzKSxcbiAgICAgIEluc3RhbmNlR3JvdXBzOiBjZGsubGlzdE1hcHBlcihJbnN0YW5jZUdyb3VwQ29uZmlnUHJvcGVydHlUb0pzb24pKHByb3BlcnR5Lmluc3RhbmNlR3JvdXBzKSxcbiAgICAgIEtlZXBKb2JGbG93QWxpdmVXaGVuTm9TdGVwczogdHJ1ZSxcbiAgICAgIE1hc3Rlckluc3RhbmNlVHlwZTogY2RrLnN0cmluZ1RvQ2xvdWRGb3JtYXRpb24ocHJvcGVydHkubWFzdGVySW5zdGFuY2VUeXBlKSxcbiAgICAgIFBsYWNlbWVudDogKHByb3BlcnR5LnBsYWNlbWVudCA9PT0gdW5kZWZpbmVkKSA/XG4gICAgICAgIHByb3BlcnR5LnBsYWNlbWVudCA6XG4gICAgICAgIFBsYWNlbWVudFR5cGVQcm9wZXJ0eVRvSnNvbihwcm9wZXJ0eS5wbGFjZW1lbnQpLFxuICAgICAgU2VydmljZUFjY2Vzc1NlY3VyaXR5R3JvdXA6IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5LnNlcnZpY2VBY2Nlc3NTZWN1cml0eUdyb3VwKSxcbiAgICAgIFNsYXZlSW5zdGFuY2VUeXBlOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5zbGF2ZUluc3RhbmNlVHlwZSksXG4gICAgICBUZXJtaW5hdGlvblByb3RlY3RlZDogY2RrLmJvb2xlYW5Ub0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5LnRlcm1pbmF0aW9uUHJvdGVjdGVkKVxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogUHJvcGVydGllcyBmb3IgdGhlIEVNUiBDbHVzdGVyIEFwcGxpY2F0aW9uc1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIEFtYXpvbiBFTVIgcmVsZWFzZXMgNC4wIGFuZCBsYXRlci4gQSBjYXNlLWluc2Vuc2l0aXZlIGxpc3Qgb2YgYXBwbGljYXRpb25zIGZvciBBbWF6b24gRU1SIHRvIGluc3RhbGwgYW5kIGNvbmZpZ3VyZSB3aGVuIGxhdW5jaGluZ1xuICAgKiB0aGUgY2x1c3Rlci5cbiAgICpcbiAgICogU2VlIHRoZSBSdW5Kb2JGbG93IEFQSSBmb3IgY29tcGxldGUgZG9jdW1lbnRhdGlvbiBvbiBpbnB1dCBwYXJhbWV0ZXJzXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9BcHBsaWNhdGlvbi5odG1sXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBpbnRlcmZhY2UgQXBwbGljYXRpb25Db25maWdQcm9wZXJ0eSB7XG4gICAgLyoqXG4gICAgICogVGhpcyBvcHRpb24gaXMgZm9yIGFkdmFuY2VkIHVzZXJzIG9ubHkuIFRoaXMgaXMgbWV0YSBpbmZvcm1hdGlvbiBhYm91dCB0aGlyZC1wYXJ0eSBhcHBsaWNhdGlvbnMgdGhhdCB0aGlyZC1wYXJ0eSB2ZW5kb3JzIHVzZVxuICAgICAqIGZvciB0ZXN0aW5nIHB1cnBvc2VzLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gYWRkaXRpb25hbEluZm9cbiAgICAgKi9cbiAgICByZWFkb25seSBhZGRpdGlvbmFsSW5mbz86IHtba2V5OiBzdHJpbmddOiBzdHJpbmd9O1xuXG4gICAgLyoqXG4gICAgICogQXJndW1lbnRzIGZvciBBbWF6b24gRU1SIHRvIHBhc3MgdG8gdGhlIGFwcGxpY2F0aW9uLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gYXJnc1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGFyZ3M/OiBzdHJpbmdbXTtcblxuICAgIC8qKlxuICAgICAqIFRoZSBuYW1lIG9mIHRoZSBhcHBsaWNhdGlvbi5cbiAgICAgKi9cbiAgICByZWFkb25seSBuYW1lOiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgdmVyc2lvbiBvZiB0aGUgYXBwbGljYXRpb24uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyB2ZXJzaW9uXG4gICAgICovXG4gICAgcmVhZG9ubHkgdmVyc2lvbj86IHN0cmluZztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZW5kZXIgdGhlIEFwcGxpY2F0aW9uQ29uZmlnUHJvcGVydHkgYXMgSlNPTlxuICAgKlxuICAgKiBAcGFyYW0gcHJvcGVydHlcbiAgICovXG4gIGV4cG9ydCBmdW5jdGlvbiBBcHBsaWNhdGlvbkNvbmZpZ1Byb3BlcnR5VG9Kc29uKHByb3BlcnR5OiBBcHBsaWNhdGlvbkNvbmZpZ1Byb3BlcnR5KSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIE5hbWU6IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5Lm5hbWUpLFxuICAgICAgQXJnczogY2RrLmxpc3RNYXBwZXIoY2RrLnN0cmluZ1RvQ2xvdWRGb3JtYXRpb24pKHByb3BlcnR5LmFyZ3MpLFxuICAgICAgVmVyc2lvbjogY2RrLnN0cmluZ1RvQ2xvdWRGb3JtYXRpb24ocHJvcGVydHkudmVyc2lvbiksXG4gICAgICBBZGRpdGlvbmFsSW5mbzogY2RrLm9iamVjdFRvQ2xvdWRGb3JtYXRpb24ocHJvcGVydHkuYWRkaXRpb25hbEluZm8pXG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb25maWd1cmF0aW9uIG9mIHRoZSBzY3JpcHQgdG8gcnVuIGR1cmluZyBhIGJvb3RzdHJhcCBhY3Rpb24uXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TY3JpcHRCb290c3RyYXBBY3Rpb25Db25maWcuaHRtbFxuICAgKlxuICAgKiBAZXhwZXJpbWVudGFsXG4gICAqL1xuICBleHBvcnQgaW50ZXJmYWNlIFNjcmlwdEJvb3RzdHJhcEFjdGlvbkNvbmZpZ1Byb3BlcnR5IHtcbiAgICAvKipcbiAgICAgKiBMb2NhdGlvbiBvZiB0aGUgc2NyaXB0IHRvIHJ1biBkdXJpbmcgYSBib290c3RyYXAgYWN0aW9uLiBDYW4gYmUgZWl0aGVyIGEgbG9jYXRpb24gaW4gQW1hem9uIFMzIG9yIG9uIGEgbG9jYWwgZmlsZSBzeXN0ZW0uXG4gICAgICovXG4gICAgcmVhZG9ubHkgcGF0aDogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgICogQSBsaXN0IG9mIGNvbW1hbmQgbGluZSBhcmd1bWVudHMgdG8gcGFzcyB0byB0aGUgYm9vdHN0cmFwIGFjdGlvbiBzY3JpcHQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBhcmdzXG4gICAgICovXG4gICAgcmVhZG9ubHkgYXJncz86IHN0cmluZ1tdO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbmZpZ3VyYXRpb24gb2YgYSBib290c3RyYXAgYWN0aW9uLlxuICAgKlxuICAgKiBTZWUgdGhlIFJ1bkpvYkZsb3cgQVBJIGZvciBjb21wbGV0ZSBkb2N1bWVudGF0aW9uIG9uIGlucHV0IHBhcmFtZXRlcnNcbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Jvb3RzdHJhcEFjdGlvbkNvbmZpZy5odG1sXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBpbnRlcmZhY2UgQm9vdHN0cmFwQWN0aW9uQ29uZmlnUHJvcGVydHkge1xuICAgIC8qKlxuICAgICAqIFRoZSBuYW1lIG9mIHRoZSBib290c3RyYXAgYWN0aW9uLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFRoZSBzY3JpcHQgcnVuIGJ5IHRoZSBib290c3RyYXAgYWN0aW9uXG4gICAgICovXG4gICAgcmVhZG9ubHkgc2NyaXB0Qm9vdHN0cmFwQWN0aW9uOiBTY3JpcHRCb290c3RyYXBBY3Rpb25Db25maWdQcm9wZXJ0eVxuICB9XG5cbiAgLyoqXG4gICAqIFJlbmRlciB0aGUgQm9vdHN0cmFwQWN0aW9uUHJvcGVydHkgYXMgSlNPTlxuICAgKlxuICAgKiBAcGFyYW0gcHJvcGVydHlcbiAgICovXG4gIGV4cG9ydCBmdW5jdGlvbiBCb290c3RyYXBBY3Rpb25Db25maWdUb0pzb24ocHJvcGVydHk6IEJvb3RzdHJhcEFjdGlvbkNvbmZpZ1Byb3BlcnR5KSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIE5hbWU6IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5Lm5hbWUpLFxuICAgICAgU2NyaXB0Qm9vdHN0cmFwQWN0aW9uOiB7XG4gICAgICAgIFBhdGg6IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5LnNjcmlwdEJvb3RzdHJhcEFjdGlvbi5wYXRoKSxcbiAgICAgICAgQXJnczogY2RrLmxpc3RNYXBwZXIoY2RrLnN0cmluZ1RvQ2xvdWRGb3JtYXRpb24pKHByb3BlcnR5LnNjcmlwdEJvb3RzdHJhcEFjdGlvbi5hcmdzKVxuICAgICAgfVxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogQW4gb3B0aW9uYWwgY29uZmlndXJhdGlvbiBzcGVjaWZpY2F0aW9uIHRvIGJlIHVzZWQgd2hlbiBwcm92aXNpb25pbmcgY2x1c3RlciBpbnN0YW5jZXMsIHdoaWNoIGNhbiBpbmNsdWRlIGNvbmZpZ3VyYXRpb25zIGZvclxuICAgKiBhcHBsaWNhdGlvbnMgYW5kIHNvZnR3YXJlIGJ1bmRsZWQgd2l0aCBBbWF6b24gRU1SLlxuICAgKlxuICAgKiBTZWUgdGhlIFJ1bkpvYkZsb3cgQVBJIGZvciBjb21wbGV0ZSBkb2N1bWVudGF0aW9uIG9uIGlucHV0IHBhcmFtZXRlcnNcbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NvbmZpZ3VyYXRpb24uaHRtbFxuICAgKlxuICAgKiBAZXhwZXJpbWVudGFsXG4gICAqL1xuICBleHBvcnQgaW50ZXJmYWNlIENvbmZpZ3VyYXRpb25Qcm9wZXJ0eSB7XG4gICAgLyoqXG4gICAgICogVGhlIGNsYXNzaWZpY2F0aW9uIHdpdGhpbiBhIGNvbmZpZ3VyYXRpb24uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBjbGFzc2lmaWNhdGlvblxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNsYXNzaWZpY2F0aW9uPzogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgICogQSBzZXQgb2YgcHJvcGVydGllcyBzcGVjaWZpZWQgd2l0aGluIGEgY29uZmlndXJhdGlvbiBjbGFzc2lmaWNhdGlvbi5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIHByb3BlcnRpZXNcbiAgICAgKi9cbiAgICByZWFkb25seSBwcm9wZXJ0aWVzPzoge1trZXk6IHN0cmluZ106IHN0cmluZ307XG5cbiAgICAvKipcbiAgICAgKiBBIGxpc3Qgb2YgYWRkaXRpb25hbCBjb25maWd1cmF0aW9ucyB0byBhcHBseSB3aXRoaW4gYSBjb25maWd1cmF0aW9uIG9iamVjdC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIGNvbmZpZ3VyYXRpb25zXG4gICAgICovXG4gICAgcmVhZG9ubHkgY29uZmlndXJhdGlvbnM/OiBDb25maWd1cmF0aW9uUHJvcGVydHlbXTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZW5kZXIgdGhlIENvbmZpZ3VyYXRpb25Qcm9wZXJ0eSBhcyBKU09OXG4gICAqXG4gICAqIEBwYXJhbSBwcm9wZXJ0eVxuICAgKi9cbiAgZXhwb3J0IGZ1bmN0aW9uIENvbmZpZ3VyYXRpb25Qcm9wZXJ0eVRvSnNvbihwcm9wZXJ0eTogQ29uZmlndXJhdGlvblByb3BlcnR5KSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIENsYXNzaWZpY2F0aW9uOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5jbGFzc2lmaWNhdGlvbiksXG4gICAgICBQcm9wZXJ0aWVzOiBjZGsub2JqZWN0VG9DbG91ZEZvcm1hdGlvbihwcm9wZXJ0eS5wcm9wZXJ0aWVzKSxcbiAgICAgIENvbmZpZ3VyYXRpb25zOiBjZGsubGlzdE1hcHBlcihDb25maWd1cmF0aW9uUHJvcGVydHlUb0pzb24pKHByb3BlcnR5LmNvbmZpZ3VyYXRpb25zKVxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogQXR0cmlidXRlcyBmb3IgS2VyYmVyb3MgY29uZmlndXJhdGlvbiB3aGVuIEtlcmJlcm9zIGF1dGhlbnRpY2F0aW9uIGlzIGVuYWJsZWQgdXNpbmcgYSBzZWN1cml0eSBjb25maWd1cmF0aW9uLlxuICAgKlxuICAgKiBTZWUgdGhlIFJ1bkpvYkZsb3cgQVBJIGZvciBjb21wbGV0ZSBkb2N1bWVudGF0aW9uIG9uIGlucHV0IHBhcmFtZXRlcnNcbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0tlcmJlcm9zQXR0cmlidXRlcy5odG1sXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBpbnRlcmZhY2UgS2VyYmVyb3NBdHRyaWJ1dGVzUHJvcGVydHkge1xuICAgIC8qKlxuICAgICAqIFRoZSBBY3RpdmUgRGlyZWN0b3J5IHBhc3N3b3JkIGZvciBBRERvbWFpbkpvaW5Vc2VyLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gYWREb21haW5Kb2luUGFzc3dvcmRcbiAgICAgKi9cbiAgICByZWFkb25seSBhZERvbWFpbkpvaW5QYXNzd29yZD86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFJlcXVpcmVkIG9ubHkgd2hlbiBlc3RhYmxpc2hpbmcgYSBjcm9zcy1yZWFsbSB0cnVzdCB3aXRoIGFuIEFjdGl2ZSBEaXJlY3RvcnkgZG9tYWluLiBBIHVzZXIgd2l0aCBzdWZmaWNpZW50IHByaXZpbGVnZXMgdG8gam9pblxuICAgICAqIHJlc291cmNlcyB0byB0aGUgZG9tYWluLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gYWREb21haW5Kb2luVXNlclxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFkRG9tYWluSm9pblVzZXI/OiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBSZXF1aXJlZCBvbmx5IHdoZW4gZXN0YWJsaXNoaW5nIGEgY3Jvc3MtcmVhbG0gdHJ1c3Qgd2l0aCBhIEtEQyBpbiBhIGRpZmZlcmVudCByZWFsbS4gVGhlIGNyb3NzLXJlYWxtIHByaW5jaXBhbCBwYXNzd29yZCwgd2hpY2hcbiAgICAgKiBtdXN0IGJlIGlkZW50aWNhbCBhY3Jvc3MgcmVhbG1zLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gY3Jvc3NSZWFsbVRydXN0UHJpbmNpcGFsUGFzc3dvcmRcbiAgICAgKi9cbiAgICByZWFkb25seSBjcm9zc1JlYWxtVHJ1c3RQcmluY2lwYWxQYXNzd29yZD86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFRoZSBwYXNzd29yZCB1c2VkIHdpdGhpbiB0aGUgY2x1c3RlciBmb3IgdGhlIGthZG1pbiBzZXJ2aWNlIG9uIHRoZSBjbHVzdGVyLWRlZGljYXRlZCBLREMsIHdoaWNoIG1haW50YWlucyBLZXJiZXJvcyBwcmluY2lwYWxzLFxuICAgICAqIHBhc3N3b3JkIHBvbGljaWVzLCBhbmQga2V5dGFicyBmb3IgdGhlIGNsdXN0ZXIuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBrZGNBZG1pblBhc3N3b3JkXG4gICAgICovXG4gICAgcmVhZG9ubHkga2RjQWRtaW5QYXNzd29yZD86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFRoZSBuYW1lIG9mIHRoZSBLZXJiZXJvcyByZWFsbSB0byB3aGljaCBhbGwgbm9kZXMgaW4gYSBjbHVzdGVyIGJlbG9uZy4gRm9yIGV4YW1wbGUsIEVDMi5JTlRFUk5BTC5cbiAgICAgKi9cbiAgICByZWFkb25seSByZWFsbTogc3RyaW5nO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlbmRlciB0aGUgS2VyYmVyb3NBdHRyaWJ1dGVzUHJvcGVydHkgYXMgSlNPTlxuICAgKlxuICAgKiBAcGFyYW0gcHJvcGVydHlcbiAgICovXG4gIGV4cG9ydCBmdW5jdGlvbiBLZXJiZXJvc0F0dHJpYnV0ZXNQcm9wZXJ0eVRvSnNvbihwcm9wZXJ0eTogS2VyYmVyb3NBdHRyaWJ1dGVzUHJvcGVydHkpIHtcbiAgICByZXR1cm4ge1xuICAgICAgQUREb21haW5Kb2luUGFzc3dvcmQ6IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5LmFkRG9tYWluSm9pblBhc3N3b3JkKSxcbiAgICAgIEFERG9tYWluSm9pblVzZXI6IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5LmFkRG9tYWluSm9pblVzZXIpLFxuICAgICAgQ3Jvc3NSZWFsbVRydXN0UHJpbmNpcGFsUGFzc3dvcmQ6IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5LmNyb3NzUmVhbG1UcnVzdFByaW5jaXBhbFBhc3N3b3JkKSxcbiAgICAgIEtkY0FkbWluUGFzc3dvcmQ6IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5LmtkY0FkbWluUGFzc3dvcmQpLFxuICAgICAgUmVhbG06IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHByb3BlcnR5LnJlYWxtKVxuICAgIH07XG4gIH1cbn0iXX0=