"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const resource_arn_suffix_1 = require("./resource-arn-suffix");
/**
 * A StepFunctions Task to run a Task on ECS or Fargate
 */
class EcsRunTaskBase {
    constructor(props) {
        this.props = props;
        /**
         * Manage allowed network traffic for this service
         */
        this.connections = new ec2.Connections();
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
            sfn.ServiceIntegrationPattern.SYNC
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call ECS.`);
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.containerOverrides)) {
            throw new Error('Task Token is missing in containerOverrides (pass Context.taskToken somewhere in containerOverrides)');
        }
        for (const override of this.props.containerOverrides || []) {
            const name = override.containerName;
            if (!cdk.Token.isUnresolved(name)) {
                const cont = this.props.taskDefinition.node.tryFindChild(name);
                if (!cont) {
                    throw new Error(`Overrides mention container with name '${name}', but no such container in task definition`);
                }
            }
        }
    }
    bind(task) {
        if (this.networkConfiguration !== undefined) {
            // Make sure we have a security group if we're using AWSVPC networking
            if (this.securityGroup === undefined) {
                this.securityGroup = new ec2.SecurityGroup(task, 'SecurityGroup', { vpc: this.props.cluster.vpc });
            }
            this.connections.addSecurityGroup(this.securityGroup);
        }
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('ecs', 'runTask', this.integrationPattern),
            parameters: {
                Cluster: this.props.cluster.clusterArn,
                TaskDefinition: this.props.taskDefinition.taskDefinitionArn,
                NetworkConfiguration: this.networkConfiguration,
                Overrides: renderOverrides(this.props.containerOverrides),
                ...this.props.parameters,
            },
            policyStatements: this.makePolicyStatements(task),
        };
    }
    configureAwsVpcNetworking(vpc, assignPublicIp, subnetSelection, securityGroup) {
        if (subnetSelection === undefined) {
            subnetSelection = { subnetType: assignPublicIp ? ec2.SubnetType.PUBLIC : ec2.SubnetType.PRIVATE };
        }
        // If none is given here, one will be created later on during bind()
        this.securityGroup = securityGroup;
        this.networkConfiguration = {
            AwsvpcConfiguration: {
                AssignPublicIp: assignPublicIp !== undefined ? (assignPublicIp ? 'ENABLED' : 'DISABLED') : undefined,
                Subnets: vpc.selectSubnets(subnetSelection).subnetIds,
                SecurityGroups: cdk.Lazy.listValue({ produce: () => [this.securityGroup.securityGroupId] }),
            }
        };
    }
    makePolicyStatements(task) {
        const stack = cdk.Stack.of(task);
        // https://docs.aws.amazon.com/step-functions/latest/dg/ecs-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['ecs:RunTask'],
                resources: [this.props.taskDefinition.taskDefinitionArn],
            }),
            new iam.PolicyStatement({
                actions: ['ecs:StopTask', 'ecs:DescribeTasks'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: cdk.Lazy.listValue({ produce: () => this.taskExecutionRoles().map(r => r.roleArn) })
            }),
        ];
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.SYNC) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForECSTaskRule'
                    })]
            }));
        }
        return policyStatements;
    }
    taskExecutionRoles() {
        // Need to be able to pass both Task and Execution role, apparently
        const ret = new Array();
        ret.push(this.props.taskDefinition.taskRole);
        if (this.props.taskDefinition.executionRole) {
            ret.push(this.props.taskDefinition.executionRole);
        }
        return ret;
    }
}
exports.EcsRunTaskBase = EcsRunTaskBase;
function renderOverrides(containerOverrides) {
    if (!containerOverrides) {
        return undefined;
    }
    const ret = new Array();
    for (const override of containerOverrides) {
        ret.push({
            Name: override.containerName,
            Command: override.command,
            Cpu: override.cpu,
            Memory: override.memoryLimit,
            MemoryReservation: override.memoryReservation,
            Environment: override.environment && override.environment.map(e => ({
                Name: e.name,
                Value: e.value,
            }))
        });
    }
    return { ContainerOverrides: ret };
}
//# sourceMappingURL=data:application/json;base64,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