"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const resource_arn_suffix_1 = require("./resource-arn-suffix");
/**
 * Invoke a Glue job as a Task
 *
 * OUTPUT: the output of this task is a JobRun structure, for details consult
 * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-JobRun
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-glue.html
 */
class RunGlueJobTask {
    constructor(glueJobName, props = {}) {
        this.glueJobName = glueJobName;
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.SYNC
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call Glue.`);
        }
    }
    bind(task) {
        var _a;
        const notificationProperty = this.props.notifyDelayAfter ? { NotifyDelayAfter: this.props.notifyDelayAfter.toMinutes() } : null;
        let iamActions;
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.FIRE_AND_FORGET) {
            iamActions = ['glue:StartJobRun'];
        }
        else if (this.integrationPattern === sfn.ServiceIntegrationPattern.SYNC) {
            iamActions = [
                'glue:StartJobRun',
                'glue:GetJobRun',
                'glue:GetJobRuns',
                'glue:BatchStopJobRun'
            ];
        }
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('glue', 'startJobRun', this.integrationPattern),
            policyStatements: [new iam.PolicyStatement({
                    resources: [
                        core_1.Stack.of(task).formatArn({
                            service: 'glue',
                            resource: 'job',
                            resourceName: this.glueJobName
                        })
                    ],
                    actions: iamActions
                })],
            metricPrefixSingular: 'GlueJob',
            metricPrefixPlural: 'GlueJobs',
            metricDimensions: { GlueJobName: this.glueJobName },
            parameters: {
                JobName: this.glueJobName,
                Arguments: this.props.arguments,
                Timeout: (_a = this.props.timeout) === null || _a === void 0 ? void 0 : _a.toMinutes(),
                SecurityConfiguration: this.props.securityConfiguration,
                NotificationProperty: notificationProperty
            }
        };
    }
}
exports.RunGlueJobTask = RunGlueJobTask;
//# sourceMappingURL=data:application/json;base64,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