"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../lib");
let stack;
const TABLE_NAME = 'SOME_TABLE';
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
});
test('GetItem task', () => {
    // WHEN
    const task = new sfn.Task(stack, 'GetItem', {
        task: tasks.CallDynamoDB.getItem({
            partitionKey: {
                name: 'SOME_KEY',
                value: new tasks.DynamoAttributeValue().withS('1234')
            },
            sortKey: {
                name: 'OTHER_KEY',
                value: new tasks.DynamoAttributeValue().withN('4321')
            },
            tableName: TABLE_NAME,
            consistentRead: true,
            expressionAttributeNames: { OTHER_KEY: '#OK' },
            projectionExpression: [
                new tasks.DynamoProjectionExpression()
                    .withAttribute('Messages')
                    .atIndex(1)
                    .withAttribute('Tags'),
                new tasks.DynamoProjectionExpression().withAttribute('ID')
            ],
            returnConsumedCapacity: tasks.DynamoConsumedCapacity.TOTAL
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition'
                    },
                    ':states:::dynamodb:getItem'
                ]
            ]
        },
        End: true,
        Parameters: {
            Key: { SOME_KEY: { S: '1234' }, OTHER_KEY: { N: '4321' } },
            TableName: TABLE_NAME,
            ConsistentRead: true,
            ExpressionAttributeNames: { OTHER_KEY: '#OK' },
            ProjectionExpression: 'Messages[1].Tags,ID',
            ReturnConsumedCapacity: 'TOTAL'
        }
    });
});
test('PutItem task', () => {
    // WHEN
    const task = new sfn.Task(stack, 'PutItem', {
        task: tasks.CallDynamoDB.putItem({
            item: { SOME_KEY: new tasks.DynamoAttributeValue().withS('1234') },
            tableName: TABLE_NAME,
            conditionExpression: 'ForumName <> :f and Subject <> :s',
            expressionAttributeNames: { OTHER_KEY: '#OK' },
            expressionAttributeValues: {
                ':val': new tasks.DynamoAttributeValue().withN(sfn.Data.stringAt('$.Item.TotalCount.N'))
            },
            returnConsumedCapacity: tasks.DynamoConsumedCapacity.TOTAL,
            returnItemCollectionMetrics: tasks.DynamoItemCollectionMetrics.SIZE,
            returnValues: tasks.DynamoReturnValues.ALL_NEW
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition'
                    },
                    ':states:::dynamodb:putItem'
                ]
            ]
        },
        End: true,
        Parameters: {
            Item: { SOME_KEY: { S: '1234' } },
            TableName: TABLE_NAME,
            ConditionExpression: 'ForumName <> :f and Subject <> :s',
            ExpressionAttributeNames: { OTHER_KEY: '#OK' },
            ExpressionAttributeValues: { ':val': { 'N.$': '$.Item.TotalCount.N' } },
            ReturnConsumedCapacity: 'TOTAL',
            ReturnItemCollectionMetrics: 'SIZE',
            ReturnValues: 'ALL_NEW'
        }
    });
});
test('DeleteItem task', () => {
    // WHEN
    const task = new sfn.Task(stack, 'DeleteItem', {
        task: tasks.CallDynamoDB.deleteItem({
            partitionKey: {
                name: 'SOME_KEY',
                value: new tasks.DynamoAttributeValue().withS('1234')
            },
            tableName: TABLE_NAME,
            conditionExpression: 'ForumName <> :f and Subject <> :s',
            expressionAttributeNames: { OTHER_KEY: '#OK' },
            expressionAttributeValues: {
                ':val': new tasks.DynamoAttributeValue().withN(sfn.Data.stringAt('$.Item.TotalCount.N'))
            },
            returnConsumedCapacity: tasks.DynamoConsumedCapacity.TOTAL,
            returnItemCollectionMetrics: tasks.DynamoItemCollectionMetrics.SIZE,
            returnValues: tasks.DynamoReturnValues.ALL_NEW
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition'
                    },
                    ':states:::dynamodb:deleteItem'
                ]
            ]
        },
        End: true,
        Parameters: {
            Key: { SOME_KEY: { S: '1234' } },
            TableName: TABLE_NAME,
            ConditionExpression: 'ForumName <> :f and Subject <> :s',
            ExpressionAttributeNames: { OTHER_KEY: '#OK' },
            ExpressionAttributeValues: { ':val': { 'N.$': '$.Item.TotalCount.N' } },
            ReturnConsumedCapacity: 'TOTAL',
            ReturnItemCollectionMetrics: 'SIZE',
            ReturnValues: 'ALL_NEW'
        }
    });
});
test('UpdateItem task', () => {
    // WHEN
    const task = new sfn.Task(stack, 'UpdateItem', {
        task: tasks.CallDynamoDB.updateItem({
            partitionKey: {
                name: 'SOME_KEY',
                value: new tasks.DynamoAttributeValue().withS('1234')
            },
            tableName: TABLE_NAME,
            conditionExpression: 'ForumName <> :f and Subject <> :s',
            expressionAttributeNames: { OTHER_KEY: '#OK' },
            expressionAttributeValues: {
                ':val': new tasks.DynamoAttributeValue().withN(sfn.Data.stringAt('$.Item.TotalCount.N'))
            },
            returnConsumedCapacity: tasks.DynamoConsumedCapacity.TOTAL,
            returnItemCollectionMetrics: tasks.DynamoItemCollectionMetrics.SIZE,
            returnValues: tasks.DynamoReturnValues.ALL_NEW,
            updateExpression: 'SET TotalCount = TotalCount + :val'
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition'
                    },
                    ':states:::dynamodb:updateItem'
                ]
            ]
        },
        End: true,
        Parameters: {
            Key: { SOME_KEY: { S: '1234' } },
            TableName: TABLE_NAME,
            ConditionExpression: 'ForumName <> :f and Subject <> :s',
            ExpressionAttributeNames: { OTHER_KEY: '#OK' },
            ExpressionAttributeValues: { ':val': { 'N.$': '$.Item.TotalCount.N' } },
            ReturnConsumedCapacity: 'TOTAL',
            ReturnItemCollectionMetrics: 'SIZE',
            ReturnValues: 'ALL_NEW',
            UpdateExpression: 'SET TotalCount = TotalCount + :val'
        }
    });
});
test('supports tokens', () => {
    // WHEN
    const task = new sfn.Task(stack, 'GetItem', {
        task: tasks.CallDynamoDB.getItem({
            partitionKey: {
                name: 'SOME_KEY',
                value: new tasks.DynamoAttributeValue().withS(sfn.Data.stringAt('$.partitionKey'))
            },
            sortKey: {
                name: 'OTHER_KEY',
                value: new tasks.DynamoAttributeValue().withN(sfn.Data.stringAt('$.sortKey'))
            },
            tableName: sfn.Data.stringAt('$.tableName'),
            consistentRead: true,
            expressionAttributeNames: { OTHER_KEY: sfn.Data.stringAt('$.otherKey') },
            projectionExpression: [
                new tasks.DynamoProjectionExpression()
                    .withAttribute('Messages')
                    .atIndex(1)
                    .withAttribute('Tags'),
                new tasks.DynamoProjectionExpression().withAttribute('ID')
            ],
            returnConsumedCapacity: tasks.DynamoConsumedCapacity.TOTAL
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition'
                    },
                    ':states:::dynamodb:getItem'
                ]
            ]
        },
        End: true,
        Parameters: {
            // tslint:disable:object-literal-key-quotes
            Key: {
                SOME_KEY: { 'S.$': '$.partitionKey' },
                OTHER_KEY: { 'N.$': '$.sortKey' }
            },
            'TableName.$': '$.tableName',
            ConsistentRead: true,
            ExpressionAttributeNames: { 'OTHER_KEY.$': '$.otherKey' },
            ProjectionExpression: 'Messages[1].Tags,ID',
            ReturnConsumedCapacity: 'TOTAL'
        }
    });
});
test('Invalid value of TableName should throw', () => {
    expect(() => {
        new sfn.Task(stack, 'GetItem', {
            task: tasks.CallDynamoDB.getItem({
                partitionKey: {
                    name: 'SOME_KEY',
                    value: new tasks.DynamoAttributeValue().withS('1234')
                },
                tableName: 'ab'
            })
        });
    }).toThrow(/TableName should not contain alphanumeric characters and should be between 3-255 characters long. Received: ab/);
    expect(() => {
        new sfn.Task(stack, 'GetItem', {
            task: tasks.CallDynamoDB.getItem({
                partitionKey: {
                    name: 'SOME_KEY',
                    value: new tasks.DynamoAttributeValue().withS('1234')
                },
                tableName: 'abU93s5MTZDv6TYLk3Q3BE3Hj3AMca3NOb5ypSNZv1JZIONg7p8L8LNxuAStavPxYZKcoG36KwXktkuFHf0jJvt7SKofEqwYHmmK0tNJSkGoPe3MofnB7IWu3V48HbrqNGZqW005CMmDHESQWf40JK8qK0CSQtM8Z64zqysB7SZZazDRm7kKr062RXQKL82nvTxnKxTPfCHiG2YJEhuFdUywHCTN2Rjinl3P7TpwyIuPWyYHm6nZodRKLMmWpgUftZ'
            })
        });
    }).toThrow(/TableName should not contain alphanumeric characters and should be between 3-255 characters long. Received: abU93s5MTZDv6TYLk3Q3BE3Hj3AMca3NOb5ypSNZv1JZIONg7p8L8LNxuAStavPxYZKcoG36KwXktkuFHf0jJvt7SKofEqwYHmmK0tNJSkGoPe3MofnB7IWu3V48HbrqNGZqW005CMmDHESQWf40JK8qK0CSQtM8Z64zqysB7SZZazDRm7kKr062RXQKL82nvTxnKxTPfCHiG2YJEhuFdUywHCTN2Rjinl3P7TpwyIuPWyYHm6nZodRKLMmWpgUftZ/);
    expect(() => {
        new sfn.Task(stack, 'GetItem', {
            task: tasks.CallDynamoDB.getItem({
                partitionKey: {
                    name: 'SOME_KEY',
                    value: new tasks.DynamoAttributeValue().withS('1234')
                },
                tableName: 'abcd@'
            })
        });
    }).toThrow(/TableName should not contain alphanumeric characters and should be between 3-255 characters long. Received: abcd@/);
});
describe('DynamoProjectionExpression', () => {
    test('should correctly configure projectionExpression', () => {
        expect(new tasks.DynamoProjectionExpression()
            .withAttribute('Messages')
            .atIndex(1)
            .atIndex(10)
            .withAttribute('Tags')
            .withAttribute('Items')
            .atIndex(0)
            .toString()).toEqual('Messages[1][10].Tags.Items[0]');
    });
    test('should throw if expression starts with atIndex', () => {
        expect(() => new tasks.DynamoProjectionExpression()
            .atIndex(1)
            .withAttribute('Messages')
            .toString()).toThrow(/Expression must start with an attribute/);
    });
});
//# sourceMappingURL=data:application/json;base64,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