"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../lib");
/**
 * Pre verification steps:
 * * aws dynamodb create-table --table-name Messages --key-schema AttributeName=MessageId,KeyType=HASH \
 * * --attribute-definitions AttributeName=MessageId,AttributeType=S \
 * * --provisioned-throughput ReadCapacityUnits=10,WriteCapacityUnits=5
 */
/*
 * Stack verification steps:
 * * aws stepfunctions start-execution --state-machine-arn <deployed state machine arn> : should return execution arn
 * *
 * * aws stepfunctions describe-execution --execution-arn <exection-arn generated before> --query 'status': should return status as SUCCEEDED
 * * aws stepfunctions describe-execution --execution-arn <exection-arn generated before> --query 'output': should return the number 42
 */
/**
 * Post verification steps:
 * * aws dynamodb delete-table --table-name Messages
 */
class CallDynamoDBStack extends cdk.Stack {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        const TABLE_NAME = 'Messages';
        const MESSAGE_ID = '1234';
        const firstNumber = 18;
        const secondNumber = 24;
        const putItemTask = new sfn.Task(this, 'PutItem', {
            task: tasks.CallDynamoDB.putItem({
                item: {
                    MessageId: new tasks.DynamoAttributeValue().withS(MESSAGE_ID),
                    Text: new tasks.DynamoAttributeValue().withS(sfn.Data.stringAt('$.bar')),
                    TotalCount: new tasks.DynamoAttributeValue().withN(`${firstNumber}`)
                },
                tableName: TABLE_NAME
            })
        });
        const getItemTaskAfterPut = new sfn.Task(this, 'GetItemAfterPut', {
            task: tasks.CallDynamoDB.getItem({
                partitionKey: {
                    name: 'MessageId',
                    value: new tasks.DynamoAttributeValue().withS(MESSAGE_ID)
                },
                tableName: TABLE_NAME
            })
        });
        const updateItemTask = new sfn.Task(this, 'UpdateItem', {
            task: tasks.CallDynamoDB.updateItem({
                partitionKey: {
                    name: 'MessageId',
                    value: new tasks.DynamoAttributeValue().withS(MESSAGE_ID)
                },
                tableName: TABLE_NAME,
                expressionAttributeValues: {
                    ':val': new tasks.DynamoAttributeValue().withN(sfn.Data.stringAt('$.Item.TotalCount.N')),
                    ':rand': new tasks.DynamoAttributeValue().withN(`${secondNumber}`)
                },
                updateExpression: 'SET TotalCount = :val + :rand'
            })
        });
        const getItemTaskAfterUpdate = new sfn.Task(this, 'GetItemAfterUpdate', {
            task: tasks.CallDynamoDB.getItem({
                partitionKey: {
                    name: 'MessageId',
                    value: new tasks.DynamoAttributeValue().withS(MESSAGE_ID)
                },
                tableName: TABLE_NAME
            }),
            outputPath: sfn.Data.stringAt('$.Item.TotalCount.N')
        });
        const deleteItemTask = new sfn.Task(this, 'DeleteItem', {
            task: tasks.CallDynamoDB.deleteItem({
                partitionKey: {
                    name: 'MessageId',
                    value: new tasks.DynamoAttributeValue().withS(MESSAGE_ID)
                },
                tableName: TABLE_NAME
            }),
            resultPath: 'DISCARD'
        });
        const definition = new sfn.Pass(this, 'Start', {
            result: sfn.Result.fromObject({ bar: 'SomeValue' })
        })
            .next(putItemTask)
            .next(getItemTaskAfterPut)
            .next(updateItemTask)
            .next(getItemTaskAfterUpdate)
            .next(deleteItemTask);
        const stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition
        });
        new cdk.CfnOutput(this, 'StateMachineArn', {
            value: stateMachine.stateMachineArn
        });
    }
}
const app = new cdk.App();
new CallDynamoDBStack(app, 'aws-stepfunctions-integ');
app.synth();
//# sourceMappingURL=data:application/json;base64,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