"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const glue = require("@aws-cdk/aws-glue");
const iam = require("@aws-cdk/aws-iam");
const assets = require("@aws-cdk/aws-s3-assets");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const path = require("path");
const tasks = require("../lib");
/*
 * Stack verification steps:
 * * aws stepfunctions start-execution --state-machine-arn <deployed state machine arn>
 * * aws stepfunctions describe-execution --execution-arn <execution arn created above>
 * The "describe-execution" call should eventually return status "SUCCEEDED".
 * NOTE: It will take up to 15 minutes for the step function to completem due to the cold start time
 * for AWS Glue, which as of 02/2020, is around 10-15 minutes.
 */
const app = new cdk.App();
const stack = new cdk.Stack(app, 'aws-stepfunctions-integ');
const codeAsset = new assets.Asset(stack, 'Glue Job Script', {
    path: path.join(__dirname, 'my-glue-script/job.py')
});
const jobRole = new iam.Role(stack, 'Glue Job Role', {
    assumedBy: new iam.ServicePrincipal('glue'),
    managedPolicies: [
        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSGlueServiceRole')
    ]
});
codeAsset.grantRead(jobRole);
const job = new glue.CfnJob(stack, 'Glue Job', {
    name: 'My Glue Job',
    glueVersion: '1.0',
    command: {
        name: 'glueetl',
        pythonVersion: '3',
        scriptLocation: `s3://${codeAsset.s3BucketName}/${codeAsset.s3ObjectKey}`
    },
    role: jobRole.roleArn
});
const jobTask = new sfn.Task(stack, 'Glue Job Task', {
    task: new tasks.RunGlueJobTask(job.name, {
        integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
        arguments: {
            '--enable-metrics': 'true'
        }
    })
});
const startTask = new sfn.Pass(stack, 'Start Task');
const endTask = new sfn.Pass(stack, 'End Task');
const stateMachine = new sfn.StateMachine(stack, 'State Machine', {
    definition: sfn.Chain.start(startTask).next(jobTask).next(endTask)
});
new cdk.CfnOutput(stack, 'State Machine ARN Output', {
    value: stateMachine.stateMachineArn
});
app.synth();
//# sourceMappingURL=data:application/json;base64,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