"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../lib");
let stack;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
});
test('create basic training job', () => {
    // WHEN
    const task = new sfn.Task(stack, 'TrainSagemaker', { task: new tasks.SagemakerTrainTask({
            trainingJobName: 'MyTrainJob',
            algorithmSpecification: {
                algorithmName: 'BlazingText',
            },
            inputDataConfig: [
                {
                    channelName: 'train',
                    dataSource: {
                        s3DataSource: {
                            s3Location: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'InputBucket', 'mybucket'), 'mytrainpath')
                        }
                    }
                }
            ],
            outputDataConfig: {
                s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'OutputBucket', 'mybucket'), 'myoutputpath')
            },
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createTrainingJob',
                ],
            ],
        },
        End: true,
        Parameters: {
            AlgorithmSpecification: {
                AlgorithmName: 'BlazingText',
                TrainingInputMode: 'File',
            },
            InputDataConfig: [
                {
                    ChannelName: 'train',
                    DataSource: {
                        S3DataSource: {
                            S3DataType: 'S3Prefix',
                            S3Uri: {
                                'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/mytrainpath']]
                            }
                        }
                    }
                }
            ],
            OutputDataConfig: {
                S3OutputPath: {
                    'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/myoutputpath']]
                }
            },
            ResourceConfig: {
                InstanceCount: 1,
                InstanceType: 'ml.m4.xlarge',
                VolumeSizeInGB: 10
            },
            RoleArn: { 'Fn::GetAtt': ['TrainSagemakerSagemakerRole89E8C593', 'Arn'] },
            StoppingCondition: {
                MaxRuntimeInSeconds: 3600
            },
            TrainingJobName: 'MyTrainJob',
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new sfn.Task(stack, 'TrainSagemaker', { task: new tasks.SagemakerTrainTask({
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
                trainingJobName: 'MyTrainJob',
                algorithmSpecification: {
                    algorithmName: 'BlazingText',
                },
                inputDataConfig: [
                    {
                        channelName: 'train',
                        dataSource: {
                            s3DataSource: {
                                s3Location: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'InputBucket', 'mybucket'), 'mytrainpath')
                            }
                        }
                    }
                ],
                outputDataConfig: {
                    s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'OutputBucket', 'mybucket'), 'myoutputpath')
                },
            }) });
    }).toThrow(/Invalid Service Integration Pattern: WAIT_FOR_TASK_TOKEN is not supported to call SageMaker./i);
});
test('create complex training job', () => {
    // WHEN
    const kmsKey = new kms.Key(stack, 'Key');
    const vpc = new ec2.Vpc(stack, 'VPC');
    const securityGroup = new ec2.SecurityGroup(stack, 'SecurityGroup', { vpc, description: 'My SG' });
    securityGroup.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'allow ssh access from the world');
    const role = new iam.Role(stack, 'Role', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
        managedPolicies: [
            iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess')
        ],
    });
    const trainTask = new tasks.SagemakerTrainTask({
        trainingJobName: 'MyTrainJob',
        integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
        role,
        algorithmSpecification: {
            algorithmName: 'BlazingText',
            trainingInputMode: tasks.InputMode.FILE,
            metricDefinitions: [
                {
                    name: 'mymetric', regex: 'regex_pattern'
                }
            ]
        },
        hyperparameters: {
            lr: '0.1'
        },
        inputDataConfig: [
            {
                channelName: 'train',
                contentType: 'image/jpeg',
                compressionType: tasks.CompressionType.NONE,
                recordWrapperType: tasks.RecordWrapperType.RECORD_IO,
                dataSource: {
                    s3DataSource: {
                        s3DataType: tasks.S3DataType.S3_PREFIX,
                        s3Location: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'InputBucketA', 'mybucket'), 'mytrainpath'),
                    }
                }
            },
            {
                channelName: 'test',
                contentType: 'image/jpeg',
                compressionType: tasks.CompressionType.GZIP,
                recordWrapperType: tasks.RecordWrapperType.RECORD_IO,
                dataSource: {
                    s3DataSource: {
                        s3DataType: tasks.S3DataType.S3_PREFIX,
                        s3Location: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'InputBucketB', 'mybucket'), 'mytestpath'),
                    }
                }
            }
        ],
        outputDataConfig: {
            s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'OutputBucket', 'mybucket'), 'myoutputpath'),
            encryptionKey: kmsKey
        },
        resourceConfig: {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.P3, ec2.InstanceSize.XLARGE2),
            volumeSizeInGB: 50,
            volumeEncryptionKey: kmsKey,
        },
        stoppingCondition: {
            maxRuntime: cdk.Duration.hours(1)
        },
        tags: {
            Project: 'MyProject'
        },
        vpcConfig: {
            vpc,
        }
    });
    trainTask.addSecurityGroup(securityGroup);
    const task = new sfn.Task(stack, 'TrainSagemaker', { task: trainTask });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createTrainingJob.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            TrainingJobName: 'MyTrainJob',
            RoleArn: { 'Fn::GetAtt': ['Role1ABCC5F0', 'Arn'] },
            AlgorithmSpecification: {
                TrainingInputMode: 'File',
                AlgorithmName: 'BlazingText',
                MetricDefinitions: [
                    { Name: 'mymetric', Regex: 'regex_pattern' }
                ]
            },
            HyperParameters: {
                lr: '0.1'
            },
            InputDataConfig: [
                {
                    ChannelName: 'train',
                    CompressionType: 'None',
                    RecordWrapperType: 'RecordIO',
                    ContentType: 'image/jpeg',
                    DataSource: {
                        S3DataSource: {
                            S3DataType: 'S3Prefix',
                            S3Uri: {
                                'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/mytrainpath']]
                            }
                        }
                    }
                },
                {
                    ChannelName: 'test',
                    CompressionType: 'Gzip',
                    RecordWrapperType: 'RecordIO',
                    ContentType: 'image/jpeg',
                    DataSource: {
                        S3DataSource: {
                            S3DataType: 'S3Prefix',
                            S3Uri: {
                                'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/mytestpath']]
                            }
                        }
                    }
                }
            ],
            OutputDataConfig: {
                S3OutputPath: {
                    'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/myoutputpath']]
                },
                KmsKeyId: { 'Fn::GetAtt': ['Key961B73FD', 'Arn'] },
            },
            ResourceConfig: {
                InstanceCount: 1,
                InstanceType: 'ml.p3.2xlarge',
                VolumeSizeInGB: 50,
                VolumeKmsKeyId: { 'Fn::GetAtt': ['Key961B73FD', 'Arn'] },
            },
            StoppingCondition: {
                MaxRuntimeInSeconds: 3600
            },
            Tags: [
                { Key: 'Project', Value: 'MyProject' }
            ],
            VpcConfig: {
                SecurityGroupIds: [
                    { 'Fn::GetAtt': ['SecurityGroupDD263621', 'GroupId'] },
                    { 'Fn::GetAtt': ['TrainSagemakerTrainJobSecurityGroup7C858EB9', 'GroupId'] },
                ],
                Subnets: [
                    { Ref: 'VPCPrivateSubnet1Subnet8BCA10E0' },
                    { Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A' },
                ]
            }
        },
    });
});
test('pass param to training job', () => {
    // WHEN
    const role = new iam.Role(stack, 'Role', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
        managedPolicies: [
            iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess'),
        ],
    });
    const task = new sfn.Task(stack, 'TrainSagemaker', { task: new tasks.SagemakerTrainTask({
            trainingJobName: sfn.Data.stringAt('$.JobName'),
            role,
            algorithmSpecification: {
                algorithmName: 'BlazingText',
                trainingInputMode: tasks.InputMode.FILE
            },
            inputDataConfig: [
                {
                    channelName: 'train',
                    dataSource: {
                        s3DataSource: {
                            s3DataType: tasks.S3DataType.S3_PREFIX,
                            s3Location: tasks.S3Location.fromJsonExpression('$.S3Bucket')
                        }
                    }
                }
            ],
            outputDataConfig: {
                s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'Bucket', 'mybucket'), 'myoutputpath'),
            },
            resourceConfig: {
                instanceCount: 1,
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.P3, ec2.InstanceSize.XLARGE2),
                volumeSizeInGB: 50
            },
            stoppingCondition: {
                maxRuntime: cdk.Duration.hours(1)
            }
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createTrainingJob',
                ],
            ],
        },
        End: true,
        Parameters: {
            'TrainingJobName.$': '$.JobName',
            'RoleArn': { 'Fn::GetAtt': ['Role1ABCC5F0', 'Arn'] },
            'AlgorithmSpecification': {
                TrainingInputMode: 'File',
                AlgorithmName: 'BlazingText',
            },
            'InputDataConfig': [
                {
                    ChannelName: 'train',
                    DataSource: {
                        S3DataSource: {
                            'S3DataType': 'S3Prefix',
                            'S3Uri.$': '$.S3Bucket'
                        }
                    }
                }
            ],
            'OutputDataConfig': {
                S3OutputPath: {
                    'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/myoutputpath']]
                }
            },
            'ResourceConfig': {
                InstanceCount: 1,
                InstanceType: 'ml.p3.2xlarge',
                VolumeSizeInGB: 50
            },
            'StoppingCondition': {
                MaxRuntimeInSeconds: 3600
            }
        },
    });
});
test('Cannot create a SageMaker train task with both algorithm name and image name missing', () => {
    expect(() => new tasks.SagemakerTrainTask({
        trainingJobName: 'myTrainJob',
        algorithmSpecification: {},
        inputDataConfig: [
            {
                channelName: 'train',
                dataSource: {
                    s3DataSource: {
                        s3DataType: tasks.S3DataType.S3_PREFIX,
                        s3Location: tasks.S3Location.fromJsonExpression('$.S3Bucket')
                    }
                }
            }
        ],
        outputDataConfig: {
            s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'Bucket', 'mybucket'), 'myoutputpath/')
        },
    }))
        .toThrowError(/Must define either an algorithm name or training image URI in the algorithm specification/);
});
//# sourceMappingURL=data:application/json;base64,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