//
// Programmer:    Craig Stuart Sapp <craig@ccrma.stanford.edu>
// Creation Date: Sat Aug  8 12:24:49 PDT 2015
// Last Modified: Sun Nov 15 23:12:30 PST 2020
// Filename:      /include/humlib.cpp
// URL:           https://github.com/craigsapp/humlib/blob/master/src/humlib.cpp
// Syntax:        C++11
// vim:           ts=3
//
// Description:   Source file for humlib library.
//
/*
Copyright (c) 2015-2020 Craig Stuart Sapp
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
   and the following disclaimer in the documentation and/or other materials
   provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#include "humlib.h"

namespace hum {



//////////////////////////////
//
// Convert::majorScaleBase40 -- Return the base-40 scale degree
//     tonic-intervals for each  note in a major scale.  The input is the
//     base-40 pitch-class of the root.  The default input is 0, which
//     will return a list of the intervals for each scale degree to the
//     tonic of the key.
//

vector<int> Convert::majorScaleBase40(void) {
	return {0, 6, 12, 17, 23, 29, 35};
}



//////////////////////////////
//
// Convert::minorHScaleBase40 -- Return the base-40 scale degree
//     tonic-intervals for each  note in a harmonic minor scale.  The input
//     is the base-40 pitch-class of the root.  The default input is 0, which
//     will return a list of the intervals for each scale degree to the
//     tonic of the key.
//

vector<int> Convert::minorHScaleBase40(void) {
	return {0, 6, 11, 17, 23, 28, 35};
}



//////////////////////////////
//
// Convert::keyToBase40 -- convert a Humdrum **kern key designation into
//    a base-40 integer.  Positive values are for major keys and negative
//    values are for minor keys.  (C-double-flat major is 40 rather than 0).
//    Returns 0 if no legitimate key was found.
//

int Convert::keyToBase40(const string& key) {
	string token;
	auto loc = key.find(":");
	if (loc != std::string::npos) {
		token = key.substr(0, loc);
	} else {
		token = key;
	}

	int base40 = Convert::kernToBase40(token);
	if (base40 < 0)  {
		return 0;
	}

	if (base40 >= 160) {
		base40 = -(base40 % 40);
		if (base40 == 0) {
			base40 = -40;
		}
	} else {
		base40 = base40 % 40;
		if (base40 == 0) {
			base40 = 40;
		}
	}
	return base40;
}



//////////////////////////////
//
// Convert::keyToInversion -- Extract the inversion from a **harm token.
//    Root position is 0, first inversion is 1, etc. up to 6th inversion
//    for 13th chords.
//

int Convert::keyToInversion(const string& harm) {
	for (char ch : harm) {
		if ((ch >= 'a') && (ch <= 'g')) {
			return ch - 'a';
		}
	}
	return 0;
}



//////////////////////////////
//
// Convert::chromaticAlteration -- Return the sum of "#" minus "-" in the string.
//

int Convert::chromaticAlteration(const string& content) {
	int sum = 0;
	for (char ch : content) {
		switch (ch) {
			case '#': sum++; break;
			case '-': sum--; break;
		}
	}
	return sum;
}



//////////////////////////////
//
// Convert::makeAdjustedKeyRootAndMode --
//

void Convert::makeAdjustedKeyRootAndMode(const string& secondary, int& keyroot,
		int& keymode) {

	vector<int> majorkey = Convert::majorScaleBase40();
	vector<int> minorkey = Convert::minorHScaleBase40();

	vector<string> roots;
	HumRegex hre;
	hre.split(roots, secondary, "/");

	for (int i=0; i<(int)roots.size(); i++) {
		string piece = roots[(int)roots.size() - i - 1];
		int number = Convert::romanNumeralToInteger(piece);
		if (number == 0) {
			continue;
		} else if (number > 7) {
			number = (number - 1) % 7;
		} else {
			number -= 1;
		}
		if (keymode == 0) { // major key
			keyroot += majorkey[number];
		} else {
			keyroot += minorkey[number];
		}
		int alteration = chromaticAlteration(piece);
		keyroot += alteration;
		if ((!piece.empty()) && isupper(piece[0])) {
			keymode = 0; // major
		} else {
			keymode = 1; // minor
		}
	}

	keyroot = keyroot % 40;
}



//////////////////////////////
//
// Convert::harmToBase40 -- Convert a **harm chord into a list of
//   pitch classes contained in the chord.  The output is a vector
//   that contains the root pitch class in the first slot, then
//   the successive chord tones after that.  If the vector is empty
//   then there was some sort of syntax error in the **harm token.
//   The bass note is placed in the 3rd octave and other pitch classes
//   in the chord are placed in the 4th octave.
//

vector<int> Convert::harmToBase40(const string& harm, const string& key) {
	int keyroot = Convert::keyToBase40(key);
	int keymode = 0; // major key
	if (keyroot < 0) {
		keyroot = -keyroot;
		keymode = 1; // minor key
	}
	return harmToBase40(harm, keyroot, keymode);
}


vector<int> Convert::harmToBase40(const string& harm, int keyroot, int keymode) {
	// Create a tonic-interval list of the scale-degrees:
	vector<int> degrees;
	if (keymode == 1) {
		degrees = Convert::minorHScaleBase40();
	} else {
		degrees = Convert::majorScaleBase40();
	}

	// Remove any **recip prefixed to token:
	string newharm = harm;
	HumRegex hre;
	if (hre.search(harm, R"(^[{}\d%._\][]+(.*))")) {
		newharm = hre.getMatch(1);
	}

	// Remove alternate chord labels:
	string single;
	auto loc = newharm.find('[');
	if (loc != string::npos) {
		single = newharm.substr(0, loc);
	} else {
		single = newharm;
	}

	// Split off secondary dominant qualifications
	string cbase;     // base chord
	string secondary; // secondary chord qualifiers
	loc = single.find("/");
	if (loc != string::npos) {
		cbase = single.substr(0, loc);
		secondary = single.substr(loc+1, string::npos);
	} else {
		cbase = single;
	}

	// Calculate interval offset for secondary dominants:
	int newkeyroot = keyroot;
	int newkeymode = keymode;
	if (!secondary.empty()) {
		makeAdjustedKeyRootAndMode(secondary, newkeyroot, newkeymode);
	}

	int rootdeg = -1; // chord root scale degree in key
	int degalt = 0;   // degree alteration

	vector<char> chars(256, 0);
	for (auto ch : cbase) {
		chars[ch]++;
	}

	rootdeg = -1; // invalid scale degree
	degalt = chars['#'] - chars['-'];

	int vcount = chars['V'] + chars['v'];
	int icount = chars['I'] + chars['i'];

	if (vcount == 1) {
		switch (icount) {
			case 0: rootdeg = 4; break; // V
			case 1:
				if (cbase.find("IV") != string::npos) {
					rootdeg = 3; break; // IV
				} else if (cbase.find("iv") != string::npos) {
					rootdeg = 3; break; // iv
				} else {
					rootdeg = 5; break; // VI/vi
				}
			case 2: rootdeg = 6; break; // VII
			case 3: rootdeg = 0; break; // VIII (I)
		}
	} else {
		switch (icount) {
			case 0:  // N, Fr, Gn, Lt, Tr
				if (chars['N']) {
					// Neapolitan (flat-second scale degree)
					rootdeg = 1; // -II
					degalt += -1; // -II
				} else if (chars['L'] || chars['F'] || chars['G']) {
					// augmented 6th chord on -VII
					rootdeg = 5;
					// fixed to -VI of major scale:
					if (newkeymode == 0) { // major
						degalt += -1;
					} else { // minor
						// already at -VI in minor
						degalt += 0;
					}
				}
				break;
			case 1: rootdeg = 0; break; // I
			case 2: rootdeg = 1; break; // II
			case 3: rootdeg = 2; break; // III
		}
	}

	int inversion = Convert::keyToInversion(single);
	vector<int> output;

	if (rootdeg < 0) {
		return output;
	}

	int root = degrees.at(rootdeg) + newkeyroot;
	output.push_back(root);

	int int3  = -1;
	int int5  = 23;  // assume a perfect 5th
	int int7  = -1;
	int int9  = -1;
	// int int11 = -1;
	// int int13 = -1;

	// determine the third's interval
	if (chars['i'] || chars['v']) {
		// minor third
		int3 = 11;
	} else if (chars['I'] || chars['V']) {
		// major third
		int3 = 12;
	} else if (chars['N']) {
		// neapolitan (major triad)
		int3 = 12;
		int5 = 23;
	} else if (chars['G']) {
		// german aug. 6th chord
		int3 = 12;
		int5 = 23;
		int7 = 30; // technically on 6th
	} else if (chars['L']) {
		// Italian aug. 6th chord
		int3 = 12;
		int5 = -1;
		int7 = 30; // technically on 6th
	} else if (chars['F']) {
		// French aug. 6th chord
		int3 = 12;
		int5 = 18; // technically on 4th
		int7 = 30; // technically on 6th
	}

	// determine the fifth's interval
	if (chars['o']) { // diminished
		int5 = 22;
	}
	if (chars['+']) { // augmented
		int5 = 24;
	}

	if (int3 > 0) {
		output.push_back(int3 + output[0]);
	}
	if (int5 > 0) {
		output.push_back(int5 + output[0]);
	}


	///// determine higher chord notes

	// determine the seventh
	if (chars['7']) {
		int7 = degrees.at((rootdeg + 6) % 7) - degrees.at(rootdeg);
		if (int7 < 0) {
			int7 += 40;
		}
		if (hre.search(cbase, "(A+|D+|M|m)7")) {
			string quality = hre.getMatch(1);
			if (quality == "M") {
				int7 = 35;
			} else if (quality == "m") {
				int7 = 34;
			} else if (quality[0] == 'D') {
				int7 = 34 - (int)quality.size();
			} else if (quality[0] == 'A') {
				int7 = 35 + (int)quality.size();
			}
		}
		output.push_back(int7 % 40 + output[0]);
	}

	// determine the 9th
	if (chars['9']) {
		HumRegex hre;
		int9 = degrees.at((rootdeg + 1) % 7) - degrees.at(rootdeg);
		if (int9 < 0) {
			int9 += 40;
		}
		if (hre.search(cbase, "(A+|D+|M|m)9")) {
			string quality = hre.getMatch(1);
			if (quality == "M") {
				int9 = 46;
			} else if (quality == "m") {
				int9 = 45;
			} else if (quality[0] == 'D') {
				int9 = 45 - (int)quality.size();
			} else if (quality[0] == 'A') {
				int9 = 46 + (int)quality.size();
			}
		}
		output.push_back(int9 + output[0]);
	}


	// add inverion
	if (inversion < (int)output.size()) {
		output[inversion] = output[inversion] % 40 + 3 * 40;
	}

	int oct = 4;
	int lastvalue = -1;
	for (int i=0; i<(int)output.size(); i++) {
		if (i != inversion) {
			output[i] = output[i] % 40 + oct * 40;
			if (output[i] < lastvalue) {
				output[i] += 40;
			}
			if (output[i] < lastvalue) {
				output[i] += 40;
			}
			lastvalue = output[i];
		} else {
		}
	}

	return output;

}





//////////////////////////////
//
// Convert::isKernRest -- Returns true if the input string represents
//   a **kern rest.
//

bool Convert::isKernRest(const string& kerndata) {
	if (kerndata.find("r") != string::npos) {
		return true;
	} else {
		return false;
	}
}



//////////////////////////////
//
// Convert::isKernNote -- Returns true if the input string represents
//   a **kern note (i.e., token with a pitch, not a null token or a rest).
//

bool Convert::isKernNote(const string& kerndata) {
	char ch;
	for (int i=0; i < (int)kerndata.size(); i++) {
		ch = std::tolower(kerndata[i]);
		if ((ch >= 'a') && (ch <= 'g')) {
			return true;
		}
	}
	return false;
}



//////////////////////////////
//
// Convert::isKernSecondaryTiedNote -- Returns true if the input string
//   represents a **kern note (i.e., token with a pitch,
//   not a null token or a rest) and has a '_' or ']' character.
//

bool Convert::isKernSecondaryTiedNote(const string& kerndata) {
	char ch;
	if (!Convert::isKernNote(kerndata)) {
		return false;
	}
	for (int i=0; i < (int)kerndata.size(); i++) {
		ch = std::tolower(kerndata[i]);
		if ((ch == '_') || (ch == ']')) {
			return true;
		}
	}
	return false;
}



//////////////////////////////
//
// Convert::isKernNoteAttack -- Returns true if the input string
//   represents a **kern note (not null or rest) and is not a
//   secondary tied note.
//

bool Convert::isKernNoteAttack(const string& kerndata) {
	char ch;
	if (!Convert::isKernNote(kerndata)) {
		return false;
	}
	for (int i=0; i < (int)kerndata.size(); i++) {
		ch = std::tolower(kerndata[i]);
		if ((ch == '_') || (ch == ']')) {
			return false;
		}
	}
	return true;
}



//////////////////////////////
//
// Convert::hasKernSlurStart -- Returns true if the input string
//   has a '('.
//

bool Convert::hasKernSlurStart(const string& kerndata) {
	return kerndata.find('(') != string::npos;
}



//////////////////////////////
//
// Convert::hasKernSlurEnd -- Returns true if the input string
//   has a ')'.
//

bool Convert::hasKernSlurEnd(const string& kerndata) {
	return kerndata.find(')') != string::npos;
}



//////////////////////////////
//
// Convert::hasKernPhraseStart -- Returns true if the input string
//   has a '{'.
//

bool Convert::hasKernPhraseStart(const string& kerndata) {
	return kerndata.find('{') != string::npos;
}



//////////////////////////////
//
// Convert::hasKernPhraseEnd -- Returns true if the input string
//   has a '}'.
//

bool Convert::hasKernPhraseEnd(const string& kerndata) {
	return kerndata.find('}') != string::npos;
}



//////////////////////////////
//
// Convert::getKernSlurStartElisionLevel -- Returns the number of
//   '&' characters before the given '(' character in a kern token.
//   Returns -1 if no '(' character in string.
//

int Convert::getKernSlurStartElisionLevel(const string& kerndata, int index) {
	bool foundSlurStart = false;
	int output = 0;
	int count = 0;
	int target = index + 1;
	for (int i=0; i<(int)kerndata.size(); i++) {
		char ch = kerndata[i];
		if (ch == '(') {
			count++;
		}
		if (count == target) {
			foundSlurStart = true;
			for (int j=i-1; j>=0; j--) {
				ch = kerndata[j];
				if (ch == '&') {
					output++;
				} else {
					break;
				}
			}
			break;
		}
	}
	if (!foundSlurStart) {
		return -1;
	} else {
		return output;
	}
}



//////////////////////////////
//
// Convert::getKernSlurEndElisionLevel -- Returns the number of
//   '&' characters before the last ')' character in a kern token.
//   Returns -1 if no ')' character in string.
//

int Convert::getKernSlurEndElisionLevel(const string& kerndata, int index) {
	bool foundSlurEnd = false;
	int output = 0;
	int count = 0;
	int target = index + 1;
	for (int i=0; i<(int)kerndata.size(); i++) {
		char ch = kerndata[i];
		if (ch == ')') {
			count++;
		}
		if (count == target) {
			foundSlurEnd = true;
			for (int j=i-1; j>=0; j--) {
				ch = kerndata[j];
				if (ch == '&') {
					output++;
				} else {
					break;
				}
			}
			break;
		}
	}
	if (!foundSlurEnd) {
		return -1;
	} else {
		return output;
	}
}



//////////////////////////////
//
// Convert::getKernPhraseStartElisionLevel -- Returns the number of
//   '&' characters before the given '{' character in a kern token.
//   Returns -1 if no '{' character in string.
//

int Convert::getKernPhraseStartElisionLevel(const string& kerndata, int index) {
	bool foundPhraseStart = false;
	int output = 0;
	int count = 0;
	int target = index + 1;
	for (int i=0; i<(int)kerndata.size(); i++) {
		char ch = kerndata[i];
		if (ch == '{') {
			count++;
		}
		if (count == target) {
			foundPhraseStart = true;
			for (int j=i-1; j>=0; j--) {
				ch = kerndata[j];
				if (ch == '&') {
					output++;
				} else {
					break;
				}
			}
			break;
		}
	}
	if (!foundPhraseStart) {
		return -1;
	} else {
		return output;
	}
}



//////////////////////////////
//
// Convert::getKernPhraseEndElisionLevel -- Returns the number of
//   '&' characters before the last '}' character in a kern token.
//   Returns -1 if no '}' character in string.
//

int Convert::getKernPhraseEndElisionLevel(const string& kerndata, int index) {
	bool foundPhraseEnd = false;
	int output = 0;
	int count = 0;
	int target = index + 1;
	for (int i=0; i<(int)kerndata.size(); i++) {
		char ch = kerndata[i];
		if (ch == '}') {
			count++;
		}
		if (count == target) {
			foundPhraseEnd = true;
			for (int j=i-1; j>=0; j--) {
				ch = kerndata[j];
				if (ch == '&') {
					output++;
				} else {
					break;
				}
			}
			break;
		}
	}
	if (!foundPhraseEnd) {
		return -1;
	} else {
		return output;
	}
}



//////////////////////////////
//
// Convert::getKernPitchAttributes --
//    pc         = pitch class
//    numacc     = numeric accidental (-1=flat, 0=natural, 1=sharp)
//    explicit   = force showing of accidental
//    oct        = octave number (middle C = 4)
//    base40     = base-40 enumeration of pitch (valid if abs(numacc) <= 2)
//

string Convert::getKernPitchAttributes(const string& kerndata) {
	int accid = kernToAccidentalCount(kerndata);
	string output = "";

	output += " dpc=\"";
	output += kernToDiatonicUC(kerndata);
	output += "\"";

	output += " numacc=\"";
	output += to_string(accid);
	output += "\"";

	if (kerndata.find('n') != string::npos) {
		output += " explicit =\"true\"";
	}

	output += " oct=\"";
	output += to_string(kernToOctaveNumber(kerndata));
	output += "\"";

	if (abs(accid) <= 2) {
		output += " base40=\"";
		output += to_string(kernToBase40(kerndata));
		output += "\"";
	}

	return output;
}



//////////////////////////////
//
// Convert::hasKernStemDirection -- Returns true if a stem direction in data; otherwise,
//    return false.  If true, then '/' means stem up, and '\\' means stem down.
//

char Convert::hasKernStemDirection(const string& kerndata) {
	for (int i=0; i<(int)kerndata.size(); i++) {
		if (kerndata[i] == '/') {
			return '/';
		}
		if (kerndata[i] == '\\') {
			return '\\';
		}
	}
	return '\0';
}



//////////////////////////////
//
// Convert::kernToRecip -- Extract only the **recip data from **kern data.
//

string Convert::kernToRecip(const std::string& kerndata) {
	string output;
	output.reserve(kerndata.size());
	for (int i=0; i<(int)kerndata.size(); i++) {
		if (kerndata.at(i) == ' ') {
			// only process the first subtoken
			break;
		}
		switch (kerndata.at(i)) {
			case '0':
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case '7':
			case '8':
			case '9':
			case '.':   // augmentation dot
			case '%':   // rational rhythms
			case 'q':   // grace note (zero duration)
				output += kerndata.at(i);
		}
	}
	return output;
}


string Convert::kernToRecip(HTp token) {
	return Convert::kernToRecip((string)*token);
}






//////////////////////////////
//
// Convert::getLcm -- Return the Least Common Multiple of a list of numbers.
//

int Convert::getLcm(const vector<int>& numbers) {
	if (numbers.size() == 0) {
		return 1;
	}
	int output = numbers[0];
	for (int i=1; i<(int)numbers.size(); i++) {
		output = (output * numbers[i]) / getGcd(output, numbers[i]);
	}
	return output;
}



//////////////////////////////
//
// Convert::getGcd -- Return the Greatest Common Divisor of two numbers.
//

int Convert::getGcd(int a, int b) {
	if (b == 0) {
		return a;
	}
	int c = a % b;
	a = b;
	int output = getGcd(a, c);
	return output;
}



//////////////////////////////
//
// Convert::primeFactors -- Return a list of prime factors of a number.
//

void Convert::primeFactors(vector<int>& output, int n) {
	output.clear();
	while (n%2 == 0) {
		output.push_back(2);
		n = n >> 1;
	}
	for (int i=3; i <= sqrt(n); i += 2) {
		while (n%i == 0) {
			output.push_back(i);
			n = n/i;
		}
	}
	if (n > 2) {
		output.push_back(n);
	}
}



//////////////////////////////
//
// Convert::nearIntQuantize -- avoid small deviations from integer values.
//    devault value: delta = 0.00001
//

double Convert::nearIntQuantize(double value, double delta) {
	if ((value + delta) - int(value+delta)  < delta*2) {
		value = (int)(value+delta);
	}
	return value;
}



//////////////////////////////
//
// Convert::significantDigits --
//

double Convert::significantDigits(double value, int digits) {
	double scale = pow(10, digits);
	return (int(value * scale + 0.5))/scale;
}



//////////////////////////////
//
// Convert::isNaN -- needed due to compiler differences.
//

bool Convert::isNaN(double value) {
	union { uint64_t u; double f; } ieee754;
	ieee754.f = value;
	return ( (unsigned)(ieee754.u >> 32) & 0x7fffffff ) +
           ( (unsigned)ieee754.u != 0 ) > 0x7ff00000;
}



//////////////////////////////
//
// Convert::isPowerOfTwo --
//

bool Convert::isPowerOfTwo(int value) {
	if (value < 0) {
		return (-value & (-value - 1)) == 0;
	} else if (value > 0) {
		return (value & (value - 1)) == 0;
	} else {
		return false;
	}
}



//////////////////////////////
//
// Tool_transpose::pearsonCorrelation --
//

double Convert::pearsonCorrelation(const vector<double>& x, const vector<double>& y) {
	double sumx  = 0.0;
	double sumy  = 0.0;
	double sumco = 0.0;
	double meanx = x[0];
	double meany = y[0];

	int size = (int)x.size();
	if ((int)y.size() < size) {
		size = (int)y.size();
	}

	for (int i=2; i<=size; i++) {
		double sweep = (i-1.0) / i;
		double deltax = x[i-1] - meanx;
		double deltay = y[i-1] - meany;
		sumx  += deltax * deltax * sweep;
		sumy  += deltay * deltay * sweep;
		sumco += deltax * deltay * sweep;
		meanx += deltax / i;
		meany += deltay / i;
	}

	double popsdx = sqrt(sumx / size);
	double popsdy = sqrt(sumy / size);
	double covxy  = sumco / size;

	return covxy / (popsdx * popsdy);
}



//////////////////////////////
//
// Convert::standardDeviation --
//

double Convert::standardDeviation(const vector<double>& x) {
	double sum = 0.0;
	for (int i=0; i<(int)x.size(); i++) {
		sum += x[i];
	}
	double mean = sum / x.size();
	double variance = 0.0;
	for (int i=0; i<(int)x.size(); i++) {
		variance += pow(x[i] - mean, 2);
	}
	variance = variance / x.size();
	return sqrt(variance);
}



//////////////////////////////
//
// Convert::standardDeviationSample -- Similar to Convert::standardDeviation,
//     but divide by (size-1) rather than (size).
//

double Convert::standardDeviationSample(const vector<double>& x) {
	double sum = 0.0;
	for (int i=0; i<(int)x.size(); i++) {
		sum += x[i];
	}
	double mean = sum / x.size();
	double variance = 0.0;
	for (int i=0; i<(int)x.size(); i++) {
		variance += pow(x[i] - mean, 2);
	}
	variance = variance / ((int)x.size()-1);
	return sqrt(variance);
}



//////////////////////////////
//
// Convert::mean -- calculate the mean (average) of a list of numbers.
//

double Convert::mean(const std::vector<double>& x) {
	double output = 0.0;
	for (int i=0; i<(int)x.size(); i++) {
		output += x[i];
	}
	return output / (int)x.size();
}



//////////////////////////////
//
// Convert::coefficientOfVariationPopulation -- Standard deviation divided by
//    mean.  From: Patel, Iversen & Rosenberg (2006): Comparing the
//    rhythm and melody of speech and music: The case of British
//    English and French.  JASA 119(5), May 2006, pp. 3034-3047.
//

double Convert::coefficientOfVariationPopulation(const std::vector<double>& x) {
	double sd = Convert::standardDeviation(x);
	double mean = Convert::mean(x);
	return sd / mean;
}



//////////////////////////////
//
// Convert::coefficientOfVariationSample -- Standard deviation divided by
//    mean.  From: Patel, Iversen & Rosenberg (2006): Comparing the
//    rhythm and melody of speech and music: The case of British
//    English and French.  JASA 119(5), May 2006, pp. 3034-3047.
//

double Convert::coefficientOfVariationSample(const std::vector<double>& x) {
	double sd = Convert::standardDeviationSample(x);
	double mean = Convert::mean(x);
	return sd / mean;
}



//////////////////////////////
//
// Convert::nPvi -- normalized pairwise variablity index.
//    See: Linguistic: Grabe & Lowe 2002.
//    See: Daniele & Patel 2004.
//    See: Patel, Iversen & Rosenberg (2006): Comparing the
//    rhythm and melody of speech and music: The case of British
//    English and French.  JASA 119(5), May 2006, pp. 3034-3047.
//

double Convert::nPvi(const std::vector<double>& x) {
	double output = 0.0;
	for (int i=0; i<(int)x.size() - 1; i++) {
		output += fabs((x[i] - x[i+1]) / (x[i] + x[i+1]));
	}
	output *= 200.0 / ((int)x.size() - 1);
	return output;
}



//////////////////////////////
//
// Convert::romanNumeralToInteger -- Convert a roman numeral into an integer.
//

int Convert::romanNumeralToInteger(const string& roman) {
	int rdigit;
	int sum = 0;
	char previous='_';
	for (int i=(int)roman.length()-1; i>=0; i--) {
		switch (roman[i]) {
			case 'I': case 'i': rdigit =    1; break;
			case 'V': case 'v': rdigit =    5; break;
			case 'X': case 'x': rdigit =   10; break;
			case 'L': case 'l': rdigit =   50; break;
			case 'C': case 'c': rdigit =  100; break;
			case 'D': case 'd': rdigit =  500; break;
			case 'M': case 'm': rdigit = 1000; break;
			default:  rdigit =   -1;
		}
		if (rdigit < 0) {
			continue;
		} else if (rdigit < sum && (roman[i] != previous)) {
			sum -= rdigit;
		} else {
			sum += rdigit;
		}
		previous = roman[i];
	}

	return sum;
}





//////////////////////////////
//
// Convert::isMensRest -- Returns true if the input string represents
//   a **mens rest.
//

bool Convert::isMensRest(const string& mensdata) {
	return mensdata.find('r') != std::string::npos;
}



//////////////////////////////
//
// Convert::isMensNote -- Returns true if the input string represents
//   a **mens note (i.e., token with a pitch, not a null token or a rest).
//

bool Convert::isMensNote(const string& mensdata) {
	char ch;
	for (int i=0; i < (int)mensdata.size(); i++) {
		ch = std::tolower(mensdata[i]);
		if ((ch >= 'a') && (ch <= 'g')) {
			return true;
		}
	}
	return false;
}



//////////////////////////////
//
// Convert::hasLigatureBegin -- Returns true if the input string
//   has a '<' or '[' character.
//

bool Convert::hasLigatureBegin(const string& mensdata) {
	return hasRectaLigatureBegin(mensdata) || hasObliquaLigatureBegin(mensdata);
}



//////////////////////////////
//
// Convert::hasRectaLigatureBegin --
//

bool Convert::hasRectaLigatureBegin(const string& mensdata) {
	return mensdata.find('[') != std::string::npos;
}



//////////////////////////////
//
// Convert::hasObliquaLigatureBegin --
//

bool Convert::hasObliquaLigatureBegin(const string& mensdata) {
	return mensdata.find('<') != std::string::npos;
}



//////////////////////////////
//
// Convert::hasLigatureEnd --
//

bool Convert::hasLigatureEnd(const string& mensdata) {
	return hasRectaLigatureEnd(mensdata) || hasObliquaLigatureEnd(mensdata);
}



//////////////////////////////
//
// Convert::hasRectaLigatureEnd -- Returns true if the input string
//   has a ']'.
//

bool Convert::hasRectaLigatureEnd(const string& mensdata) {
	return mensdata.find(']') != std::string::npos;
}



//////////////////////////////
//
// Convert::hasObliquaLigatureEnd -- Returns true if the input string
//   has a '>'.
//

bool Convert::hasObliquaLigatureEnd(const string& mensdata) {
	return mensdata.find('>') != std::string::npos;
}



//////////////////////////////
//
// Convert::getMensStemDir -- Returns the stem direction of **mens
//   notes.  These are the same as **kern pitches.
//      / = stem up, return +1
//      \ = stemp down, return -1
//      otherwise return 0 for no stem information
//

bool Convert::getMensStemDirection(const string& mensdata) {
	if (mensdata.find('/') != std::string::npos) {
		return +1;
	} else if (mensdata.find('\\') != std::string::npos) {
		return +1;
	} else {
		return 0;
	}
}



//////////////////////////////
//
// Convert::mensToDuration --
//   X = maxima (octuple whole note)
//   L = long  (quadruple whole note)
//   S = breve (double whole note)
//   s = semi-breve (whole note)
//   M = minim (half note)
//   m = semi-minim (quarter note)
//   U = fusa (eighth note)
//   u = semifusa (sixteenth note)
//
//   p = perfect (dotted)
//   i = imperfect (not-dotted)
//
// Still have to deal with coloration (triplets)
//
// Default value: scale = 4 (convert to quarter note units)
//                separator = " " (space between chord notes)
//

HumNum Convert::mensToDuration(const string& mensdata, HumNum scale,
		const string& separator) {
	HumNum output(0);
   bool perfect = false;
   // bool imperfect = true;
	for (int i=0; i<(int)mensdata.size(); i++) {
		if (mensdata[i] == 'p') {
			perfect = true;
			// imperfect = false;
		}
		if (mensdata[i] == 'i') {
			perfect = false;
			// imperfect = true;
		}

		// units are in whole notes, but scaling will probably
		// convert to quarter notes (default
		switch (mensdata[i]) {
			case 'X': output = 8; break;              // octuple whole note
			case 'L': output = 4; break;              // quadruple whole note
			case 'S': output = 2; break;              // double whole note
			case 's': output = 1; break;              // whole note
			case 'M': output.setValue(1, 2);  break;  // half note
			case 'm': output.setValue(1, 4);  break;  // quarter note
			case 'U': output.setValue(1, 8);  break;  // eighth note
			case 'u': output.setValue(1, 16); break;  // sixteenth note
		}

		if (mensdata.compare(i, separator.size(), separator) == 0) {
			// only get duration of first note in chord
			break;
		}
	}

	if (perfect) {
		output *= 3;
		output /= 2;
	}
	output *= scale;
	return output;
}



//////////////////////////////
//
// Convert::mensToDurationNoDots -- The imperfect duration of the **mens rhythm.
//

HumNum Convert::mensToDurationNoDots(const string& mensdata, HumNum scale,
		const string& separator) {
	HumNum output(0);

	for (int i=0; i<(int)mensdata.size(); i++) {
		switch (mensdata[i]) {
			case 'X': output = 8; break;              // octuple whole note
			case 'L': output = 4; break;              // quadruple whole note
			case 'S': output = 2; break;              // double whole note
			case 's': output = 1; break;              // whole note
			case 'M': output.setValue(1, 2);  break;  // half note
			case 'm': output.setValue(1, 4);  break;  // quarter note
			case 'U': output.setValue(1, 8);  break;  // eighth note
			case 'u': output.setValue(1, 16); break;  // sixteenth note
		}
		if (mensdata.compare(i, separator.size(), separator) == 0) {
			// only get duration of first note in chord
			break;
		}
	}
	output *= scale;

	return output;
}



//////////////////////////////
//
// Convert::mensToRecip --
//

string Convert::mensToRecip(const string& mensdata, HumNum scale,
		const string& separator) {
	HumNum duration = Convert::mensToDuration(mensdata, scale, separator);
	return Convert::durationToRecip(duration);
}






//////////////////////////////
//
// Convert::museToBase40 -- Convert a MuseData pitch into base-40 representation.
//

int Convert::museToBase40(const string& pitchString) {
   string temp = pitchString;
   int octave;
   int i = (int)temp.size() - 1;
   while (i >= 0 && !isdigit(temp[i])) {
      i--;
   }

   if (i <= 0) {
      cerr << "Error: could not find octave in string: " << pitchString << endl;
		cerr << "Assigning to octave 4" << endl;
      octave = 4;
   } else {
   	octave = temp[i] - '0';
	}
   temp.resize(i);

	for (int i=0; i<(int)temp.size(); i++) {
		if (temp[i] == 'f') {
			temp[i] = '-';
		}
	}
	int kb40 = Convert::kernToBase40(temp);
	if (kb40 < 0) {
		return kb40;
	}
   return kb40 % 40 + 40 * octave;
}



//////////////////////////////
//
// Convert::musePitchToKernPitch -- 
//

string Convert::musePitchToKernPitch(const string& museInput) {
   return base40ToKern(museToBase40(museInput));
}



//////////////////////////////
//
// Convert::museClefToKernClef --
//

string Convert::museClefToKernClef(const string& mclef) {
	if (mclef == "4") {         // treble clef
		return "*clefG2";
	} else if (mclef == "22") { // bass clef
		return "*clefF4";
	} else if (mclef == "13") { // alto clef
		return "*clefC3";
	} else if (mclef == "12") { // tenor clef
		return "*clefC4";
	} else if (mclef == "15") { // soprano clef
		return "*clefC1";
	} else if (mclef == "14") { // mezzo-soprano clef
		return "*clefC2";
	} else if (mclef == "14") { // baritone clef
		return "*clefC5";
	} else if (mclef == "5") { // French violin clef
		return "*clefG1";
	} else if (mclef == "3") {
		return "*clefG3";
	} else if (mclef == "2") {
		return "*clefG4";
	} else if (mclef == "1") {
		return "*clefG5";
	} else if (mclef == "25") {
		return "*clefF1";
	} else if (mclef == "24") {
		return "*clefF2";
	} else if (mclef == "23") {
		return "*clefF3";
	} else if (mclef == "21") {
		return "*clefF5";
	} else if (mclef == "35") {
		return "*clefGv1";
	} else if (mclef == "34") {  // vocal tenor clef
		return "*clefGv2";
	} else if (mclef == "33") {
		return "*clefGv3";
	} else if (mclef == "32") {
		return "*clefGv3";
	} else if (mclef == "31") {
		return "*clefGv5";
	}
	// percussion clef?
	// return unknown clef:
	return "*";
}



//////////////////////////////
//
// Convert::museKeySigToKernKeySig --
//

string Convert::museKeySigToKernKeySig(const string& mkeysig) {
	if (mkeysig == "0") {
		return "*k[]";
	} else if (mkeysig == "1") {
		return "*k[f#]";
	} else if (mkeysig == "-1") {
		return "*k[b-]";
	} else if (mkeysig == "2") {
		return "*k[f#c#]";
	} else if (mkeysig == "-2") {
		return "*k[b-e-]";
	} else if (mkeysig == "3") {
		return "*k[f#c#g#]";
	} else if (mkeysig == "-3") {
		return "*k[b-e-a-]";
	} else if (mkeysig == "4") {
		return "*k[f#c#g#d#]";
	} else if (mkeysig == "-4") {
		return "*k[b-e-a-d-]";
	} else if (mkeysig == "5") {
		return "*k[f#c#g#d#a#]";
	} else if (mkeysig == "-5") {
		return "*k[b-e-a-d-g-]";
	} else if (mkeysig == "6") {
		return "*k[f#c#g#d#a#e#]";
	} else if (mkeysig == "-6") {
		return "*k[b-e-a-d-g-c-]";
	} else if (mkeysig == "7") {
		return "*k[f#c#g#d#a#e#b#]";
	} else if (mkeysig == "-7") {
		return "*k[b-e-a-d-g-c-f-]";
	}
	return "*";
}



//////////////////////////////
//
// Convert::museTimeSigToKernTimeSig --
//

string Convert::museTimeSigToKernTimeSig(const string& mtimesig) {
	if (mtimesig == "11/0") {
		return "*M3/1";    // "*met(O)
	} else if (mtimesig == "1/1") {
		return "*M4/4";    // "*met(c)
	} else if (mtimesig == "0/0") {
		return "*M2/2";    // "*met(c)
	} else if (mtimesig == "12/0") {
		return "";    // "*met(O:)
	} else if (mtimesig == "21/0") {
		return "";    // "*met(O.)
	} else if (mtimesig == "22/0") {
		return "";    // "*met(O;)
	} else if (mtimesig == "31/0") {
		return "*M2/1";    // "*met(C)
	} else if (mtimesig == "41/0") {
		return "";    // "*met(C.)
	} else if (mtimesig == "42/0") {
		return "";    // "*met(C.3/2)
	} else if (mtimesig == "43/0") {
		return "";    // "*met(C.3/8)
	} else if (mtimesig == "51/0") {
		return "";    // "*met(Cr)
	} else if (mtimesig == "52/0") {
		return "";    // "*met(Cr|)
	} else if (mtimesig == "61/0") {
		return "*M2/1"; // "*met(C|)
	} else if (mtimesig == "62/0") {
		return "";     // "*met(C|/2)
	} else if (mtimesig == "63/0") {
		return "";     // "*met(C|.)
	} else if (mtimesig == "71/0") {
		return "";     // "*met(C2)
	} else if (mtimesig == "72/0") {
		return "";     // "*met(C2/3)
	} else if (mtimesig == "81/0") {
		return "";     // "*met(O2)
	} else if (mtimesig == "82/0") {
		return "";     // "*met(O3/2)
	} else if (mtimesig == "91/0") {
		return "*M3/1"; // "*met(O|)
	} else if (mtimesig == "92/0") {
		return ""; // "*met(O|3)
	} else if (mtimesig == "93/0") {
		return ""; // "*met(O|3/2)
	} else if (mtimesig == "101/0") {
		return ""; // "*met(C|3)
	} else if (mtimesig == "102/0") {
		return ""; // "*met(3)
	} else if (mtimesig == "103/0") {
		return ""; // "*met(3/2)
	} else if (mtimesig == "104/0") {
		return ""; // "*met(C|/3)
	} else if (mtimesig == "105/0") {
		return ""; // "*met(C3)
	} else if (mtimesig == "106/0") {
		return ""; // "*met(O/3)
	} else if (mtimesig == "111/0") {
		return ""; // "*met(C|2)
	} else if (mtimesig == "112/0") {
		return ""; // "*met(2)
	} else if (mtimesig == "121/0") {
		return ""; // "*met(Oo)
	}
	string output = "*M" + mtimesig;
	return output;
}



//////////////////////////////
//
// Convert::museMeterSigToKernMeterSig --
//

string Convert::museMeterSigToKernMeterSig(const string& mtimesig) {
	if (mtimesig == "11/0") {
		return "*met(O)";
	} else if (mtimesig == "1/1") {
		return "*met(c)";
	} else if (mtimesig == "0/0") {
		return "*met(c)";
	} else if (mtimesig == "12/0") {
		return "*met(O:)";
	} else if (mtimesig == "21/0") {
		return "*met(O.)";
	} else if (mtimesig == "22/0") {
		return "*met(O;)";
	} else if (mtimesig == "31/0") {
		return "*met(C)";
	} else if (mtimesig == "41/0") {
		return "*met(C.)";
	} else if (mtimesig == "42/0") {
		return "*met(C.3/2)";
	} else if (mtimesig == "43/0") {
		return "*met(C.3/8)";
	} else if (mtimesig == "51/0") {
		return "*met(Cr)";
	} else if (mtimesig == "52/0") {
		return "*met(Cr|)";
	} else if (mtimesig == "61/0") {
		return "*met(C|)";
	} else if (mtimesig == "62/0") {
		return "*met(C|/2)";
	} else if (mtimesig == "63/0") {
		return "*met(C|.)";
	} else if (mtimesig == "71/0") {
		return "*met(C2)";
	} else if (mtimesig == "72/0") {
		return "*met(C2/3)";
	} else if (mtimesig == "81/0") {
		return "*met(O2)";
	} else if (mtimesig == "82/0") {
		return "*met(O3/2)";
	} else if (mtimesig == "91/0") {
		return "*met(O|)";
	} else if (mtimesig == "92/0") {
		return "*met(O|3)";
	} else if (mtimesig == "93/0") {
		return "*met(O|3/2)";
	} else if (mtimesig == "101/0") {
		return "*met(C|3)";
	} else if (mtimesig == "102/0") {
		return "*met(3)";
	} else if (mtimesig == "103/0") {
		return "*met(3/2)";
	} else if (mtimesig == "104/0") {
		return "*met(C|/3)";
	} else if (mtimesig == "105/0") {
		return "*met(C3)";
	} else if (mtimesig == "106/0") {
		return "*met(O/3)";
	} else if (mtimesig == "111/0") {
		return "*met(C|2)";
	} else if (mtimesig == "112/0") {
		return "*met(2)";
	} else if (mtimesig == "121/0") {
		return "*met(Oo)";
	}
	return "";
}



//////////////////////////////
//
// Convert::museFiguredBassToKernFiguredBass --
//

string Convert::museFiguredBassToKernFiguredBass(const string& mfb) {
	string output;
	for (int i=0; i<(int)mfb.size(); i++) {
		if (mfb[i] == 'b') { // blank spot in figure stack
			output += 'X';
		} else if (mfb[i] == 'f') { // flat
			output += '-';
		} else if ((mfb[i] == '&') && (i < (int)mfb.size()-1) && (mfb[i+1] == '0')) {
			output += ":";
			i++;
		} else if ((mfb[i] == '/')) {  // assuming slash means flat
			output += "-/";
		} else if ((mfb[i] == '\\')) { // assuming slash means sharp
			output += "#/";
		} else if ((mfb[i] == '+')) {  // assuming slash means sharp
			output += "#|";
		} else if (isdigit(mfb[i]) && (i < (int)mfb.size() - 1) && (mfb[i+1] == '#')) {
			output += mfb[i];
			output += mfb[i+1];
			output += 'r';
			i++;
		} else if (isdigit(mfb[i]) && (i < (int)mfb.size() - 1) && (mfb[i+1] == 'f')) {
			output += mfb[i];
			output += '-';
			output += 'r';
			i++;
		} else if (isdigit(mfb[i]) && (i < (int)mfb.size() - 1) && (mfb[i+1] == 'n')) {
			output += mfb[i];
			output += mfb[i+1];
			output += 'r';
			i++;
		} else {
			output += mfb[i];
		}
	}
	return output;
}





//////////////////////////////
//
// Convert::kernToScientificPitch -- Convert a **kern pitch to
//   ScientificPitch notation, which is the diatonic letter name,
//   followed by a possible accidental, then an optional separator
//   string, and finally the octave number.  A string representing a
//   chord can be given to this function, and the output will return
//   a list of the pitches in the chord, separated by a space.
// default value: flat      = "b"
// default value: sharp     = "#"
// default value: separator = ""
//

string Convert::kernToScientificPitch(const string& kerndata,
		string flat, string sharp, string separator) {
	vector<string> subtokens = Convert::splitString(kerndata);
	string output;

	for (int i=0; i<(int)subtokens.size(); i++) {
		char diatonic   = Convert::kernToDiatonicUC(subtokens[i]);
		int accidental = Convert::kernToAccidentalCount(subtokens[i]);
		int octave     = Convert::kernToOctaveNumber(subtokens[i]);
		if ((i > 0) && (i < (int)subtokens.size()-1)) {
			output += " ";
		}
		output += diatonic;
		for (int j=0; j<abs(accidental); j++) {
			output += (accidental < 0 ? flat : sharp);
		}
		output += separator;
		output += to_string(octave);
	}

	return output;
}



//////////////////////////////
//
// Convert::kernToDiatonicPC -- Convert a kern token into a diatonic
//    note pitch-class where 0="C", 1="D", ..., 6="B".  -1000 is returned
//    if the note is rest, and -2000 if there is no pitch information in the
//    input string. Only the first subtoken in the string is considered.
//

int Convert::kernToDiatonicPC(const string& kerndata) {
	for (int i=0; i<(int)kerndata.size(); i++) {
		if (kerndata[i] == ' ') {
			break;
		}
		if (kerndata[i] == 'r') {
			return -1000;
		}
		switch (kerndata[i]) {
			case 'A': case 'a': return 5;
			case 'B': case 'b': return 6;
			case 'C': case 'c': return 0;
			case 'D': case 'd': return 1;
			case 'E': case 'e': return 2;
			case 'F': case 'f': return 3;
			case 'G': case 'g': return 4;
		}
	}
	return -2000;
}



//////////////////////////////
//
// Convert::kernToDiatonicUC -- Convert a kern token into a diatonic
//    note pitch-class.  "R" is returned if the note is rest, and
//    "X" is returned if there is no pitch name in the string.
//    Only the first subtoken in the string is considered.
//

char Convert::kernToDiatonicUC(const string& kerndata) {
	for (int i=0; i<(int)kerndata.size(); i++) {
		if (kerndata[i] == ' ') {
			break;
		}
		if (kerndata[i] == 'r') {
			return 'R';
		}
		if (('A' <= kerndata[i]) && (kerndata[i] <= 'G')) {
			return kerndata[i];
		}
		if (('a' <= kerndata[i]) && (kerndata[i] <= 'g')) {
			return toupper(kerndata[i]);
		}
	}
	return 'X';
}



//////////////////////////////
//
// Convert::kernToDiatonicLC -- Similar to kernToDiatonicUC, but
//    the returned pitch name is lower case.
//

char Convert::kernToDiatonicLC(const string& kerndata) {
	return tolower(Convert::kernToDiatonicUC(kerndata));
}



//////////////////////////////
//
// Convert::kernToAccidentalCount -- Convert a kern token into a count
//    of accidentals in the first subtoken.  Sharps are assigned to the
//    value +1 and flats to -1.  So a double sharp is +2 and a double
//    flat is -2.  Only the first subtoken in the string is considered.
//    Cases such as "#-" should not exist, but in this case the return
//    value will be 0.
//

int Convert::kernToAccidentalCount(const string& kerndata) {
	int output = 0;
	for (int i=0; i<(int)kerndata.size(); i++) {
		if (kerndata[i] == ' ') {
			break;
		}
		if (kerndata[i] == '-') {
			output--;
		}
		if (kerndata[i] == '#') {
			output++;
		}
	}
	return output;
}



//////////////////////////////
//
// Convert::kernToOctaveNumber -- Convert a kern token into an octave number.
//    Middle C is the start of the 4th octave. -1000 is returned if there
//    is not pitch in the string.  Only the first subtoken in the string is
//    considered.
//

int Convert::kernToOctaveNumber(const string& kerndata) {
	int uc = 0;
	int lc = 0;
	if (kerndata == ".") {
		return -1000;
	}
	for (int i=0; i<(int)kerndata.size(); i++) {
		if (kerndata[i] == ' ') {
			break;
		}
		if (kerndata[i] == 'r') {
			return -1000;
		}
		uc += ('A' <= kerndata[i]) && (kerndata[i] <= 'G') ? 1 : 0;
		lc += ('a' <= kerndata[i]) && (kerndata[i] <= 'g') ? 1 : 0;
	}
	if ((uc > 0) && (lc > 0)) {
		// invalid pitch description
		return -1000;
	}
	if (uc > 0) {
		return 4 - uc;
	} else if (lc > 0) {
		return 3 + lc;
	} else {
		return -1000;
	}
}



//////////////////////////////
//
// Convert::kernToBase40PC -- Convert **kern pitch to a base-40 pitch class.
//    Will ignore subsequent pitches in a chord.
//

int Convert::kernToBase40PC(const string& kerndata) {
	int diatonic = Convert::kernToDiatonicPC(kerndata);
	if (diatonic < 0) {
		return diatonic;
	}
	int accid  = Convert::kernToAccidentalCount(kerndata);
	int output = -1000;
	switch (diatonic) {
		case 0: output =  0; break;
		case 1: output =  6; break;
		case 2: output = 12; break;
		case 3: output = 17; break;
		case 4: output = 23; break;
		case 5: output = 29; break;
		case 6: output = 35; break;
	}
	output += accid;
	return output + 2;     // +2 to make c-flat-flat bottom of octave.
}



//////////////////////////////
//
// Convert::kernToBase40 -- Convert **kern pitch to a base-40 integer.
//    Will ignore subsequent pitches in a chord.
//

int Convert::kernToBase40(const string& kerndata) {
	int pc = Convert::kernToBase40PC(kerndata);
	if (pc < 0) {
		return pc;
	}
	int octave   = Convert::kernToOctaveNumber(kerndata);
	return pc + 40 * octave;
}



//////////////////////////////
//
// Convert::kernToBase12PC -- Convert **kern pitch to a base-12 pitch-class.
//   C=0, C#/D-flat=1, D=2, etc.  Will return -1 instead of 11 for C-, and
//   will return 12 instead of 0 for B#.
//

int Convert::kernToBase12PC(const string& kerndata) {
	int diatonic = Convert::kernToDiatonicPC(kerndata);
	if (diatonic < 0) {
		return diatonic;
	}
	int accid    = Convert::kernToAccidentalCount(kerndata);
	int output = -1000;
	switch (diatonic) {
		case 0: output =  0; break;
		case 1: output =  2; break;
		case 2: output =  4; break;
		case 3: output =  5; break;
		case 4: output =  7; break;
		case 5: output =  9; break;
		case 6: output = 11; break;
	}
	output += accid;
	return output;
}



//////////////////////////////
//
// Convert::kernToBase12 -- Convert **kern pitch to a base-12 integer.
//     (middle C = 48).
//

int Convert::kernToBase12(const string& kerndata) {
	int pc = Convert::kernToBase12PC(kerndata);
	int octave = Convert::kernToOctaveNumber(kerndata);
	return pc + 12 * octave;
}



//////////////////////////////
//
// Convert::base40ToKern -- Convert Base-40 integer pitches into
//   **kern pitch representation.
//

string Convert::base40ToKern(int b40) {
	int octave     = b40 / 40;
	int accidental = Convert::base40ToAccidental(b40);
	int diatonic   = Convert::base40ToDiatonic(b40) % 7;
	char base = 'a';
	switch (diatonic) {
		case 0: base = 'c'; break;
		case 1: base = 'd'; break;
		case 2: base = 'e'; break;
		case 3: base = 'f'; break;
		case 4: base = 'g'; break;
		case 5: base = 'a'; break;
		case 6: base = 'b'; break;
	}
	if (octave < 4) {
		base = toupper(base);
	}
	int repeat = 0;
	if (octave > 4) {
		repeat = octave - 4;
	} else if (octave < 3) {
		repeat = 3 - octave;
	}
	if (repeat > 12) {
		cerr << "Error: unreasonable octave value: " << octave << " for " << b40 << endl;
		exit(1);
	}
	string output;
	output += base;
	for (int i=0; i<repeat; i++) {
		output += base;
	}
	if (accidental == 0) {
		return output;
	}
	if (accidental > 0) {
		for (int i=0; i<accidental; i++) {
			output += '#';
		}
	} else if (accidental < 0) {
		for (int i=0; i<-accidental; i++) {
			output += '-';
		}
	}

	return output;
}



//////////////////////////////
//
// Convert::base40ToDiatonic -- find the diatonic pitch of the
//   given base-40 pitch.  Output pitch classes: 0=C, 1=D, 2=E,
//   3=F, 4=G, 5=A, 6=B.  To this the diatonic octave is added.
//   To get only the diatonic pitch class, mod by 7: (% 7).
//   Base-40 pitches are not allowed, and the algorithm will have
//   to be adjusted to allow them.  Currently any negative base-40
//   value is presumed to be a rest and not processed.
//

int Convert::base40ToDiatonic(int b40) {
	int chroma = b40 % 40;
	int octaveoffset = (b40 / 40) * 7;
	if (b40 < 0) {
		return -1;   // rest;
	}
	switch (chroma) {
		case 0: case 1: case 2: case 3: case 4:      // C-- to C##
			return 0 + octaveoffset;
		case 6: case 7: case 8: case 9: case 10:     // D-- to D##
			return 1 + octaveoffset;
		case 12: case 13: case 14: case 15: case 16: // E-- to E##
			return 2 + octaveoffset;
		case 17: case 18: case 19: case 20: case 21: // F-- to F##
			return 3 + octaveoffset;
		case 23: case 24: case 25: case 26: case 27: // G-- to G##
			return 4 + octaveoffset;
		case 29: case 30: case 31: case 32: case 33: // A-- to A##
			return 5 + octaveoffset;
		case 35: case 36: case 37: case 38: case 39: // B-- to B##
			return 6 + octaveoffset;
	}

	// found an empty slot, so return rest:
	return -1;
}



//////////////////////////////
//
// Convert::base40ToMidiNoteNumber --
//

int Convert::base40ToMidiNoteNumber(int b40) {
	// +1 since middle-C octave is 5 in MIDI:
	int octave     = b40 / 40 + 1;
	int accidental = Convert::base40ToAccidental(b40);
	int diatonicpc = Convert::base40ToDiatonic(b40) % 7;
	switch (diatonicpc) {
		case 0: return octave * 12 +  0 + accidental;
		case 1: return octave * 12 +  2 + accidental;
		case 2: return octave * 12 +  4 + accidental;
		case 3: return octave * 12 +  5 + accidental;
		case 4: return octave * 12 +  7 + accidental;
		case 5: return octave * 12 +  9 + accidental;
		case 6: return octave * 12 + 11 + accidental;
		default: return -1000; // can't deal with negative pitches
	}
}



//////////////////////////////
//
// Convert::base40ToAccidental -- +1 = 1 sharp, +2 = double sharp, 0 = natural
//	-1 = 1 flat, -2 = double flat
//

int Convert::base40ToAccidental(int b40) {
	if (b40 < 0) {
		// not considering low pitches.  If so then the mod operator
		// below whould need fixing.
		return 0;
	}

	switch (b40 % 40) {
		case 0:	return -2;      // C-double-flat
		case 1:	return -1;      // C-flat
		case 2:	return  0;      // C
		case 3:	return  1;      // C-sharp
		case 4:	return  2;      // C-double-sharp
		case 5:	return 1000;
		case 6:	return -2;
		case 7:	return -1;
		case 8:	return  0;      // D
		case 9:	return  1;
		case 10:	return  2;
		case 11:	return 1000;
		case 12:	return -2;
		case 13:	return -1;
		case 14:	return  0;      // E
		case 15:	return  1;
		case 16:	return  2;
		case 17:	return -2;
		case 18:	return -1;
		case 19:	return  0;      // F
		case 20:	return  1;
		case 21:	return  2;
		case 22:	return 1000;
		case 23:	return -2;
		case 24:	return -1;
		case 25:	return  0;      // G
		case 26:	return  1;
		case 27:	return  2;
		case 28:	return 1000;
		case 29:	return -2;
		case 30:	return -1;
		case 31:	return  0;      // A
		case 32:	return  1;
		case 33:	return  2;
		case 34:	return 1000;
		case 35:	return -2;
		case 36:	return -1;
		case 37:	return  0;      // B
		case 38:	return  1;
		case 39:	return  2;
	}

	return 0;
}



///////////////////////////////
//
// Convert::kernToMidiNoteNumber -- Convert **kern to MIDI note number
//    (middle C = 60).  Middle C is assigned to octave 5 rather than
//    octave 4 for the kernToBase12() function.
//

int Convert::kernToMidiNoteNumber(const string& kerndata) {
	int pc = Convert::kernToBase12PC(kerndata);
	int octave = Convert::kernToOctaveNumber(kerndata);
	return pc + 12 * (octave + 1);
}



//////////////////////////////
//
// Convert::kernToBase7 -- Convert **kern pitch to a base-7 integer.
//    This is a diatonic pitch class with C=0, D=1, ..., B=6.
//

int Convert::kernToBase7(const string& kerndata) {
	int diatonic = Convert::kernToDiatonicPC(kerndata);
	if (diatonic < 0) {
		return diatonic;
	}
	int octave = Convert::kernToOctaveNumber(kerndata);
	return diatonic + 7 * octave;;
}



//////////////////////////////
//
// Convert::pitchToWbh -- Convert a given diatonic pitch class and
//   accidental adjustment into an integer.  The diatonic pitch class
//   is C=0, D=1, E=2, F=3, G=4, A=5, B=6. "acc" is the accidental
//   count: -2=double flat, -1=double flat, 0 natural, +1=sharp, etc.
//   "octave" is the octave number, with middle-C being the start of
//   octave 4.  //   "maxacc" is the maximum accidental which defines
//    the base:
//    maxacc = 2 -> Base-40.
//    maxacc = n -> Base (n*2+1)*7 + 5.
//

int Convert::pitchToWbh(int dpc, int acc, int octave, int maxacc) {
	if (dpc > 6) {
		// allow for pitch-classes expressed as ASCII characters:
		dpc = tolower(dpc) - 'a' + 5;
		dpc = dpc % 7;
	}
	int output = -1000;
	switch (dpc) {
		case 0: output = maxacc;            break;
		case 1: output =  3  * maxacc + 2;  break;
		case 2: output =  5  * maxacc + 4;  break;
		case 3: output =  7  * maxacc + 5;  break;
		case 4: output =  9  * maxacc + 7;  break;
		case 5: output =  11 * maxacc + 9;  break;
		case 6: output =  13 * maxacc + 11; break;
	}
	if (output < 0) {
		return output;
	}
	return (output + acc) + (7 * (maxacc * 2 + 1) + 5) * octave;
}



//////////////////////////////
//
// Convert::wbhToPitch -- Convert an integer-based pitch into
//    a diatonic pitch class, accidental alteration and octave number
//   The output diatonic pitch classes are 0=C, 1=D, 2=E, 3=F, 4=G, 5=A, 6=B.
//   "acc" is the accidental count: -2=double flat, -1=double flat,
//   0 natural, +1=sharp, etc.
//   "octave" is the octave number, with middle-C being the start of
//   octave 4.
//   "maxacc" is the maximum accidental which defines
//    the base:
//    maxacc = 2 -> Base-40.
//    maxacc = n -> Base (n*2+1)*7 + 5.
//    This valus must match the the analogous value used in PitchToWbh().
//

void Convert::wbhToPitch(int& dpc, int& acc, int& octave, int maxacc,
		int wbh) {
	int cwidth = maxacc * 2 + 1;
	int base = 7 * cwidth + 5;
	octave = wbh / base;
	int pc = wbh % base;

	// test for C diatonic pitch:
	int pctest = cwidth;
	if (pc < pctest) {
		dpc = 0;
		acc = pc - pctest + maxacc + 1;
		return;
	}

	// test for D diatonic pitch
	pctest += 1 + cwidth;
	if (pc < pctest) {
		dpc = 1;
		acc = pc - pctest + maxacc + 1;
		return;
	}

	// test for E diatonic pitch
	pctest += 1 + cwidth;
	if (pc < pctest) {
		dpc = 2;
		acc = pc - pctest + maxacc + 1;
		return;
	}

	// test for F diatonic pitch
	pctest += cwidth;
	if (pc < pctest) {
		dpc = 3;
		acc = pc - pctest + maxacc + 1;
		return;
	}

	// test for G diatonic pitch
	pctest += 1 + cwidth;
	if (pc < pctest) {
		dpc = 4;
		acc = pc - pctest + maxacc + 1;
		return;
	}

	// test for A diatonic pitch
	pctest += 1 + cwidth;
	if (pc < pctest) {
		dpc = 5;
		acc = pc - pctest + maxacc + 1;
		return;
	}

	// test for B diatonic pitch
	pctest += 1 + cwidth;
	if (pc < pctest) {
		dpc = 6;
		acc = pc - pctest + maxacc + 1;
		return;
	}

	// if acc in any of the above tests is +3/-3, then there was an
	// accidental overflow (overflow of the accidental).
}



//////////////////////////////
//
// Convert::kernClefToBaseline -- returns the diatonic pitch
//    of the bottom line on the staff.
//

int Convert::kernClefToBaseline(HTp input) {
	return kernClefToBaseline((string)*input);
}


int Convert::kernClefToBaseline(const string& input) {
	string clefname;
	if (input.compare(0, 5, "*clef") == 0) {
		clefname = input.substr(5);
	} else if (input.compare(0, 4, "clef") == 0) {
		clefname = input.substr(4);
	} else {
		cerr << "Error in Convert::kernClefToBaseline: " << input << endl;
		return -1000;
	}

	if (clefname == "G2") {                        // treble clef
		return Convert::kernToBase7("e");
	} else if (clefname == "F4") {                 // bass clef
		return Convert::kernToBase7("GG");
	} else if (clefname == "C3") {                 // alto clef
		return Convert::kernToBase7("F");
	} else if (clefname == "C4") {                 // tenor clef
		return Convert::kernToBase7("D");
	} else if (clefname == "Gv2") {                // vocal tenor clef
		return Convert::kernToBase7("E");

	// rest of C clef possibilities:
	} else if (clefname == "C1") {                 // soprano clef
		return Convert::kernToBase7("c");
	} else if (clefname == "C2") {                 // mezzo-soprano clef
		return Convert::kernToBase7("A");
	} else if (clefname == "C5") {                 // baritone clef
		return Convert::kernToBase7("BB");

	// rest of G clef possibilities:
	} else if (clefname == "G1") {                 // French-violin clef
		return Convert::kernToBase7("g");
	} else if (clefname == "G3") {
		return Convert::kernToBase7("c");
	} else if (clefname == "G4") {
		return Convert::kernToBase7("A");
	} else if (clefname == "G5") {
		return Convert::kernToBase7("F");

	// rest of F clef possibilities:
	} else if (clefname == "F1") {
		return Convert::kernToBase7("F");
	} else if (clefname == "F2") {
		return Convert::kernToBase7("D");
	} else if (clefname == "F3") {
		return Convert::kernToBase7("BB");
	} else if (clefname == "F5") {
		return Convert::kernToBase7("EE");

	// rest of G clef down an octave possibilities:
	} else if (clefname == "Gv1") {
		return Convert::kernToBase7("G");
	} else if (clefname == "Gv3") {
		return Convert::kernToBase7("C");
	} else if (clefname == "Gv4") {
		return Convert::kernToBase7("AA");
	} else if (clefname == "Gv5") {
		return Convert::kernToBase7("FF");

	// F clef down an octave possibilities:
	} else if (clefname == "Fv1") {
		return Convert::kernToBase7("FF");
	} else if (clefname == "Fv2") {
		return Convert::kernToBase7("DD");
	} else if (clefname == "Fv3") {
		return Convert::kernToBase7("BBB");
	} else if (clefname == "Fv4") {
		return Convert::kernToBase7("GGG");
	} else if (clefname == "Fv5") {
		return Convert::kernToBase7("EEE");

	// C clef down an octave possibilities:
	} else if (clefname == "Cv1") {
		return Convert::kernToBase7("C");
	} else if (clefname == "Cv2") {
		return Convert::kernToBase7("AA");
	} else if (clefname == "Cv3") {
		return Convert::kernToBase7("FF");
	} else if (clefname == "Cv4") {
		return Convert::kernToBase7("DD");
	} else if (clefname == "Cv5") {
		return Convert::kernToBase7("BBB");

	// G clef up an octave possibilities:
	} else if (clefname == "G^1") {
		return Convert::kernToBase7("gg");
	} else if (clefname == "G^2") {
		return Convert::kernToBase7("ee");
	} else if (clefname == "G^3") {
		return Convert::kernToBase7("cc");
	} else if (clefname == "G^4") {
		return Convert::kernToBase7("a");
	} else if (clefname == "G^5") {
		return Convert::kernToBase7("f");

	// F clef up an octave possibilities:
	} else if (clefname == "F^1") {
		return Convert::kernToBase7("f");
	} else if (clefname == "F^2") {
		return Convert::kernToBase7("d");
	} else if (clefname == "F^3") {
		return Convert::kernToBase7("B");
	} else if (clefname == "F^4") {
		return Convert::kernToBase7("G");
	} else if (clefname == "F^5") {
		return Convert::kernToBase7("E");

	// C clef up an octave possibilities:
	} else if (clefname == "C^1") {
		return Convert::kernToBase7("cc");
	} else if (clefname == "C^2") {
		return Convert::kernToBase7("a");
	} else if (clefname == "C^3") {
		return Convert::kernToBase7("f");
	} else if (clefname == "C^4") {
		return Convert::kernToBase7("d");
	} else if (clefname == "C^5") {
		return Convert::kernToBase7("B");

	// there are also two octaves down (*clefGvv2) and two octaves up (*clefG^^2)
	} else {
		// but just use treble clef if don't know what the clef it by this point
		return Convert::kernToBase7("e");
	}
}



//////////////////////////////
//
// Convert::base40ToTrans -- convert a base-40 interval into
//    a trans program's diatonic/chromatic alteration marker
//

string Convert::base40ToTrans(int base40) {
	int sign = 1;
	int chroma;
	int octave;
	if (base40 < 0) {
		sign = -1;
		chroma = -base40 % 40;
		octave = -base40 / 40;
	} else {
		sign = +1;
		chroma = base40 % 40;
		octave = base40 / 40;
	}

	int cval = 0;
	int dval = 0;

	switch (chroma * sign) {
		case   0: dval=0;  cval=0;   break; // C -> C
		case   1: dval=0;  cval=1;   break; // C -> C#
		case   2: dval=0;  cval=2;   break; // C -> C##
		case   4: dval=1;  cval=0;   break; // C -> D--
		case   5: dval=1;  cval=1;   break; // C -> D-
		case   6: dval=1;  cval=2;   break; // C -> D
		case   7: dval=1;  cval=3;   break; // C -> D#
		case   8: dval=1;  cval=4;   break; // C -> D##
		case  10: dval=2;  cval=2;   break; // C -> E--
		case  11: dval=2;  cval=3;   break; // C -> E-
		case  12: dval=2;  cval=4;   break; // C -> E
		case  13: dval=2;  cval=5;   break; // C -> E#
		case  14: dval=2;  cval=6;   break; // C -> E##
		case  15: dval=3;  cval=3;   break; // C -> F--
		case  16: dval=3;  cval=4;   break; // C -> F-
		case  17: dval=3;  cval=5;   break; // C -> F
		case  18: dval=3;  cval=6;   break; // C -> F#
		case  19: dval=3;  cval=7;   break; // C -> F##
		case  21: dval=4;  cval=5;   break; // C -> G--
		case  22: dval=4;  cval=6;   break; // C -> G-
		case  23: dval=4;  cval=7;   break; // C -> G
		case  24: dval=4;  cval=8;   break; // C -> G#
		case  25: dval=4;  cval=9;   break; // C -> G##
		case  27: dval=5;  cval=7;   break; // C -> A--
		case  28: dval=5;  cval=8;   break; // C -> A-
		case  29: dval=5;  cval=9;   break; // C -> A
		case  30: dval=5;  cval=10;  break; // C -> A#
		case  31: dval=5;  cval=11;  break; // C -> A##
		case  33: dval=6;  cval=9;   break; // C -> B--
		case  34: dval=6;  cval=10;  break; // C -> B-
		case  35: dval=6;  cval=11;  break; // C -> B
		case  36: dval=6;  cval=12;  break; // C -> B#
		case  37: dval=6;  cval=13;  break; // C -> B##
		case  38: dval=7;  cval=10;  break; // C -> c--
		case  39: dval=7;  cval=11;  break; // C -> c-
		case  -1: dval=-0; cval=-1;  break; // c -> c-
		case  -2: dval=-0; cval=-2;  break; // c -> c--
		case  -3: dval=-1; cval=1;   break; // c -> B##
		case  -4: dval=-1; cval=-0;  break; // c -> B#
		case  -5: dval=-1; cval=-1;  break; // c -> B
		case  -6: dval=-1; cval=-2;  break; // c -> B-
		case  -7: dval=-1; cval=-3;  break; // c -> B--
		case  -9: dval=-2; cval=-1;  break; // c -> A##
		case -10: dval=-2; cval=-2;  break; // c -> A#
		case -11: dval=-2; cval=-3;  break; // c -> A
		case -12: dval=-2; cval=-4;  break; // c -> A-
		case -13: dval=-2; cval=-5;  break; // c -> A-
		case -15: dval=-3; cval=-3;  break; // c -> G##
		case -16: dval=-3; cval=-4;  break; // c -> G#
		case -17: dval=-3; cval=-5;  break; // c -> G
		case -18: dval=-3; cval=-6;  break; // c -> G-
		case -19: dval=-3; cval=-7;  break; // c -> G--
		case -21: dval=-4; cval=-5;  break; // c -> F##
		case -22: dval=-4; cval=-6;  break; // c -> F#
		case -23: dval=-4; cval=-7;  break; // c -> F
		case -24: dval=-4; cval=-8;  break; // c -> F-
		case -25: dval=-4; cval=-9;  break; // c -> F--
		case -26: dval=-5; cval=-6;  break; // c -> E##
		case -27: dval=-5; cval=-7;  break; // c -> E#
		case -28: dval=-5; cval=-8;  break; // c -> E
		case -29: dval=-5; cval=-9;  break; // c -> E-
		case -30: dval=-5; cval=-10; break; // c -> E--
		case -32: dval=-6; cval=-8;  break; // c -> D##
		case -33: dval=-6; cval=-9;  break; // c -> D#
		case -34: dval=-6; cval=-10; break; // c -> D
		case -35: dval=-6; cval=-11; break; // c -> D-
		case -36: dval=-6; cval=-12; break; // c -> D--
		case -38: dval=-7; cval=-10; break; // c -> C##
		case -39: dval=-7; cval=-11; break; // c -> C#
		default:
			dval=0; cval=0;
	}

	if (octave > 0) {
		dval = dval + sign * octave * 7;
		cval = cval + sign * octave * 12;
	}

	string output = "d";
	output += to_string(dval);
	output += "c";
	output += to_string(cval);

	return output;
}



//////////////////////////////
//
// Convert::base40ToIntervalAbbr --
//

string Convert::base40ToIntervalAbbr(int base40interval) {
	if (base40interval < -1000) {
		return "r";
	}

	string output;
	if (base40interval < 0) {
		output = "-";
		base40interval = -base40interval;
	}

	// Add chromatic prefix
	switch (base40interval % 40) {
		case  0: output += "p"   ; break;  // C
		case  1: output += "a"   ; break;  // C#
		case  2: output += "aa"  ; break;  // C##
		case  3: output += "X"   ; break;  // X
		case  4: output += "d"   ; break;  // D--
		case  5: output += "m"   ; break;  // D-
		case  6: output += "M"   ; break;  // D
		case  7: output += "a"   ; break;  // D#
		case  8: output += "aa"  ; break;  // D##
		case  9: output += "X"   ; break;  // X
		case 10: output += "d"   ; break;  // E--
		case 11: output += "m"   ; break;  // E-
		case 12: output += "M"   ; break;  // E
		case 13: output += "a"   ; break;  // E#
		case 14: output += "aa"  ; break;  // E##
		case 15: output += "dd"  ; break;  // F--
		case 16: output += "d"   ; break;  // F-
		case 17: output += "p"   ; break;  // F
		case 18: output += "a"   ; break;  // F#
		case 19: output += "aa"  ; break;  // F##
		case 20: output += "X"   ; break;  // X
		case 21: output += "dd"  ; break;  // G--
		case 22: output += "d"   ; break;  // G-
		case 23: output += "p"   ; break;  // G
		case 24: output += "a"   ; break;  // G#
		case 25: output += "aa"  ; break;  // G##
		case 26: output += "X"   ; break;  // X
		case 27: output += "d"   ; break;  // A--
		case 28: output += "m"   ; break;  // A-
		case 29: output += "M"   ; break;  // A
		case 30: output += "a"   ; break;  // A#
		case 31: output += "aa"  ; break;  // A##
		case 32: output += "X"   ; break;  // X
		case 33: output += "d"   ; break;  // B--
		case 34: output += "m"   ; break;  // B-
		case 35: output += "M"   ; break;  // B
		case 36: output += "a"   ; break;  // B#
		case 37: output += "aa"  ; break;  // B##
		case 38: output += "dd"  ; break;  // C--
		case 39: output += "d"   ; break;  // C-
	}

	// Add base-7 number
	char buffer2[32] = {0};
	int diatonic = Convert::base40IntervalToDiatonic(base40interval)+1;
	sprintf(buffer2, "%d", diatonic);
	output += buffer2;

	return output;
}



//////////////////////////////
//
// Convert::base40IntervalToDiatonic -- convert a base40 interval
//    into a diatonic interval (excluding the chromatic alteration)
//

int Convert::base40IntervalToDiatonic(int base40interval) {
   int sign = 1;
   if (base40interval < 0) {
      sign = -1;
      base40interval = -base40interval;
   }
   int octave = base40interval / 40;
   base40interval = base40interval % 40;

   int diatonic = 0;
   switch (base40interval) {
      case  0: diatonic = 0; break;  // C
      case  1: diatonic = 0; break;  // C#
      case  2: diatonic = 0; break;  // C##

      case  3: diatonic = 1000; break;  // blank

      case  4: diatonic = 1; break;  // D--
      case  5: diatonic = 1; break;  // D-
      case  6: diatonic = 1; break;  // D
      case  7: diatonic = 1; break;  // D#
      case  8: diatonic = 1; break;  // D##

      case  9: diatonic = 1000; break;  // blank

      case 10: diatonic = 2; break;  // E--
      case 11: diatonic = 2; break;  // E-
      case 12: diatonic = 2; break;  // E
      case 13: diatonic = 2; break;  // E#
      case 14: diatonic = 2; break;  // E##

      case 15: diatonic = 3; break;  // F--
      case 16: diatonic = 3; break;  // F-
      case 17: diatonic = 3; break;  // F
      case 18: diatonic = 3; break;  // F#
      case 19: diatonic = 3; break;  // F##

      case 20: diatonic = 1000; break;  // blank

      case 21: diatonic = 4; break;  // G--
      case 22: diatonic = 4; break;  // G-
      case 23: diatonic = 4; break;  // G
      case 24: diatonic = 4; break;  // G#
      case 25: diatonic = 4; break;  // G##

      case 26: diatonic = 1000; break;  // blank

      case 27: diatonic = 5; break;  // A--
      case 28: diatonic = 5; break;  // A-
      case 29: diatonic = 5; break;  // A
      case 30: diatonic = 5; break;  // A#
      case 31: diatonic = 5; break;  // A##

      case 32: diatonic = 1000; break;  // blank

      case 33: diatonic = 6; break;  // B--
      case 34: diatonic = 6; break;  // B-
      case 35: diatonic = 6; break;  // B
      case 36: diatonic = 6; break;  // B#
      case 37: diatonic = 6; break;  // B##

      case 38: diatonic = 0; break;  // C--
      case 39: diatonic = 0; break;  // C-
   }

   return sign * (diatonic + octave * 7);
}



//////////////////////////////
//
// Convert::transToBase40 -- convert the Humdrum Toolkit program
//     trans's binomial notation for intervals into base-40.
//  The input can be in three formats:
//     d1c2      == no prepended text on information
//     *Trd1c2   == Transposition interpretation marker prefixed
//     *ITrd1c2  == Instrumental transposition marker prefixed
//

int Convert::transToBase40(const string& input) {
	int dval = 0;
	int cval = 0;
	if (sscanf(input.c_str(), "d%dc%d", &dval, &cval) != 2) {
		if (sscanf(input.c_str(), "*Trd%dc%d", &dval, &cval) != 2) {
			if (sscanf(input.c_str(), "*ITrd%dc%d", &dval, &cval) != 2) {
			   // cerr << "Cannot find correct information" << endl;
			   return 0;
			}
		}
	}

	int dsign = 1;
	// int csign = 1;
	if (dval < 0) {
		dsign = -1;
	}
	// if (cval < 0) {
	//    csign = -1;
	// }

	int doctave = dsign * dval / 7;
	// int coctave = csign * cval / 12;

	int base = 0;

		  if ((dval==0)  && (cval==0))   { base =	 0; }
	else if ((dval==0)  && (cval==1))   { base =	 1; }
	else if ((dval==0)  && (cval==2))   { base =	 2; }
	else if ((dval==1)  && (cval==0))   { base =	 4; }
	else if ((dval==1)  && (cval==1))   { base =	 5; }
	else if ((dval==1)  && (cval==2))   { base =	 6; }
	else if ((dval==1)  && (cval==3))   { base =	 7; }
	else if ((dval==1)  && (cval==4))   { base =	 8; }
	else if ((dval==2)  && (cval==2))   { base =	 10; }
	else if ((dval==2)  && (cval==3))   { base =	 11; }
	else if ((dval==2)  && (cval==4))   { base =	 12; }
	else if ((dval==2)  && (cval==5))   { base =	 13; }
	else if ((dval==2)  && (cval==6))   { base =	 14; }
	else if ((dval==3)  && (cval==3))   { base =	 15; }
	else if ((dval==3)  && (cval==4))   { base =	 16; }
	else if ((dval==3)  && (cval==5))   { base =	 17; }
	else if ((dval==3)  && (cval==6))   { base =	 18; }
	else if ((dval==3)  && (cval==7))   { base =	 19; }
	else if ((dval==4)  && (cval==5))   { base =	 21; }
	else if ((dval==4)  && (cval==6))   { base =	 22; }
	else if ((dval==4)  && (cval==7))   { base =	 23; }
	else if ((dval==4)  && (cval==8))   { base =	 24; }
	else if ((dval==4)  && (cval==9))   { base =	 25; }
	else if ((dval==5)  && (cval==7))   { base =	 27; }
	else if ((dval==5)  && (cval==8))   { base =	 28; }
	else if ((dval==5)  && (cval==9))   { base =	 29; }
	else if ((dval==5)  && (cval==10))  { base =	 30; }
	else if ((dval==5)  && (cval==11))  { base =	 31; }
	else if ((dval==6)  && (cval==9))   { base =	 33; }
	else if ((dval==6)  && (cval==10))  { base =	 34; }
	else if ((dval==6)  && (cval==11))  { base =	 35; }
	else if ((dval==6)  && (cval==12))  { base =	 36; }
	else if ((dval==6)  && (cval==13))  { base =	 37; }
	else if ((dval==7)  && (cval==10))  { base =	 38; }
	else if ((dval==7)  && (cval==11))  { base =	 38; }
	else if ((dval==-0) && (cval==-0))  { base =	 -0; }
	else if ((dval==-0) && (cval==-1))  { base =	 -1; }
	else if ((dval==-0) && (cval==-2))  { base =	 -2; }
	else if ((dval==-1) && (cval==1))   { base =	 -3; }
	else if ((dval==-1) && (cval==-0))  { base =	 -4; }
	else if ((dval==-1) && (cval==-1))  { base =	 -5; }
	else if ((dval==-1) && (cval==-2))  { base =	 -6; }
	else if ((dval==-1) && (cval==-3))  { base =	 -7; }
	else if ((dval==-2) && (cval==-1))  { base =	 -9; }
	else if ((dval==-2) && (cval==-2))  { base =	-10; }
	else if ((dval==-2) && (cval==-3))  { base =	-11; }
	else if ((dval==-2) && (cval==-4))  { base =	-12; }
	else if ((dval==-2) && (cval==-5))  { base =	-13; }
	else if ((dval==-3) && (cval==-3))  { base =	-15; }
	else if ((dval==-3) && (cval==-4))  { base =	-16; }
	else if ((dval==-3) && (cval==-5))  { base =	-17; }
	else if ((dval==-3) && (cval==-6))  { base =	-18; }
	else if ((dval==-3) && (cval==-7))  { base =	-19; }
	else if ((dval==-4) && (cval==-5))  { base =	-21; }
	else if ((dval==-4) && (cval==-6))  { base =	-22; }
	else if ((dval==-4) && (cval==-7))  { base =	-23; }
	else if ((dval==-4) && (cval==-8))  { base =	-24; }
	else if ((dval==-4) && (cval==-9))  { base =	-25; }
	else if ((dval==-5) && (cval==-6))  { base =	-26; }
	else if ((dval==-5) && (cval==-7))  { base =	-27; }
	else if ((dval==-5) && (cval==-8))  { base =	-28; }
	else if ((dval==-5) && (cval==-9))  { base =	-29; }
	else if ((dval==-5) && (cval==-10)) { base =	-30; }
	else if ((dval==-6) && (cval==-8))  { base =	-32; }
	else if ((dval==-6) && (cval==-9))  { base =	-33; }
	else if ((dval==-6) && (cval==-10)) { base =	-34; }
	else if ((dval==-6) && (cval==-11)) { base =	-35; }
	else if ((dval==-6) && (cval==-12)) { base =	-36; }
	else if ((dval==-7) && (cval==-10)) { base =	-38; }
	else if ((dval==-7) && (cval==-11)) { base =	-39; }
	else { // some error occurred or accidentals out of range
		// cerr << "Problem occured in transToBase40()" << endl;
		base = 0;
	}

	base += 40 * doctave * dsign;

	return base;
}



//////////////////////////////
//
// Convert::base40IntervalToLineOfFifths -- 0 => 0 (unison),
//    Perfect Fifth => 1, Major second => 2 (two fifths up), etc.
//

int Convert::base40IntervalToLineOfFifths(int base40interval) {
	base40interval += 4000;
	base40interval = base40interval % 40;

	switch (base40interval) {
		case 0:    return   0;     // C
		case 1:    return   7;     // C#
		case 2:    return  14;     // C##
		case 3:    return 100;     // X
		case 4:    return -12;     // D--
		case 5:    return  -5;     // D-
		case 6:    return   2;     // D
		case 7:    return   9;     // D#
		case 8:    return  16;     // D##
		case 9:    return 100;     // X
		case 10:   return -10;     // E--
		case 11:   return  -3;     // E-
		case 12:   return   4;     // E
		case 13:   return  11;     // E#
		case 14:   return  18;     // E##
		case 15:   return -15;     // F--
		case 16:   return  -8;     // F-
		case 17:   return  -1;     // F
		case 18:   return   6;     // F#
		case 19:   return  13;     // F##
		case 20:   return 100;     // X
		case 21:   return -13;     // G--
		case 22:   return  -6;     // G-
		case 23:   return   1;     // G
		case 24:   return   8;     // G#
		case 25:   return  15;     // G##
		case 26:   return 100;     // X
		case 27:   return -11;     // A--
		case 28:   return  -4;     // A-
		case 29:   return   3;     // A
		case 30:   return  10;     // A#
		case 31:   return  17;     // A##
		case 32:   return 100;     // X
		case 33:   return  -9;     // B--
		case 34:   return  -2;     // B-
		case 35:   return   5;     // B
		case 36:   return  12;     // B#
		case 37:   return  19;     // B##
		case 38:   return -14;     // C--
		case 39:   return  -7;     // C-
		default:   return 100;     // X
	}

	return 100;
}



//////////////////////////////
//
// Convert::keyNumberToKern -- reverse of kernKeyToNumber.
//

string Convert::keyNumberToKern(int number) {
	switch (number) {
		case -7: return "*k[b-e-a-d-g-c-f-]";
		case -6: return "*k[b-e-a-d-g-c-]";
		case -5: return "*k[b-e-a-d-g-]";
		case -4: return "*k[b-e-a-d-]";
		case -3: return "*k[b-e-a-]";
		case -2: return "*k[b-e-]";
		case -1: return "*k[b-]";
		case  0: return "*k[]";
		case +1: return "*k[f#]";
		case +2: return "*k[f#c#]";
		case +3: return "*k[f#c#g#]";
		case +4: return "*k[f#c#g#d#]";
		case +5: return "*k[f#c#g#d#a#]";
		case +6: return "*k[f#c#g#d#a#e#]";
		case +7: return "*k[f#c#g#d#a#e#b#]";
		default: return "*k[]";
	}
}



//////////////////////////////
//
// Convert::base7ToBase40 -- Convert a base7 value to a base-40 value
//   (without accidentals).  Negative values are not allowed, but not
//   checked for.
//

int Convert::base7ToBase40(int base7) {
	int octave = base7 / 7;
	int b7pc = base7 % 7;
	int b40pc = 0;
	switch (b7pc) {
		case 0: b40pc =  0; break; // C
		case 1: b40pc =  6; break; // D
		case 2: b40pc = 12; break; // E
		case 3: b40pc = 17; break; // F
		case 4: b40pc = 23; break; // G
		case 5: b40pc = 29; break; // A
		case 6: b40pc = 35; break; // B
	}
	return octave * 40 + 2 + b40pc;
}



//////////////////////////////
//
// Convert::kernToStaffLocation -- 0 = bottom line of staff, 1 = next space higher,
//     2 = second line of staff, etc.  -1 = space below bottom line.
//

int Convert::kernToStaffLocation(HTp token, HTp clef) {
	if (clef == NULL) {
		return Convert::kernToStaffLocation(*token, "");
	} else {
		return Convert::kernToStaffLocation(*token, *clef);
	}
}


int Convert::kernToStaffLocation(HTp token, const string& clef) {
	return Convert::kernToStaffLocation(*token, clef);
}


int Convert::kernToStaffLocation(const string& token, const string& clef) {
	int offset = 0;
	HumRegex hre;
	if (hre.search(clef, "clef([GFC])([v^]*)(\\d+)")) {
		string letter = hre.getMatch(1);
		string vcaret = hre.getMatch(2);
		int line = hre.getMatchInt(3);
		int octadj = 0;
		if (!vcaret.empty()) {
			for (int i=0; i<(int)vcaret.size(); i++) {
				if (vcaret[i] == '^') {
					octadj--;
				} else if (vcaret[i] == 'v') {
					octadj++;
				}
			}
		}
		if (letter == "F") {
			offset = 14 + 4;
		} else if (letter == "C") {
			offset = 28;
		} else {
			offset = 28 + 4;
		}
		offset += (line - 1)  * 2;
		offset += octadj * 7;
	} else {
		// pretend clefG2:
		offset = 28 + 2;
	}

	int diatonic = Convert::kernToBase7(token);
	return diatonic - offset;
}






//////////////////////////////
//
// Convert::getReferenceKeyMeaning -- 
//

string Convert::getReferenceKeyMeaning(HTp token) {
	string text = token->getText();
	return Convert::getReferenceKeyMeaning(text);
}

string Convert::getReferenceKeyMeaning(const string& token) {
	string key;
	string keybase;
	string translation;
	string language;
	string number;
	HumRegex hre;
	if (hre.search(token, "^!!!+\\s*([^:]+)\\s*:")) {
		key = hre.getMatch(1);
	}
	if (key.empty()) {
		return "";
	}
	if (islower(key[0])) {
		// non-standard reference record.
		return "";
	}

	// extract language information
	if (hre.search(key, "^([^@])+@@([^@]+)$")) {
		key      = hre.getMatch(1);
		language = hre.getMatch(2);
	} else if (hre.search(key, "^([^@])+@([^@]+)$")) {
		key         = hre.getMatch(1);
		translation = hre.getMatch(2);
	}
		
	// extract number qualifier
	if (hre.search(key, "^(.*)(\\d+)$")) {
		key     = hre.getMatch(1);
		number  = hre.getMatch(2);
	}

	if (key.empty()) {
		return "";
	}

	string meaning;
	switch (key[0]) {
		case 'A':	// analytic information
			if      (key == "ACO") { meaning = "Collection designation"; }
			else if (key == "AFR") { meaning = "Form designation"; }
			else if (key == "AGN") { meaning = "genre designation"; }
			else if (key == "AST") { meaning = "Syle/period"; }
			else if (key == "AMD") { meaning = "Mode classification"; }
			else if (key == "AMT") { meaning = "Meter classification"; }
			else if (key == "AIN") { meaning = "Instrumentation"; }
			else if (key == "ARE") { meaning = "Geographical region of origin"; }
			else if (key == "ARL") { meaning = "Origin coordinates"; }
			break;

		case 'C':	// composer-base reference records
			if      (key == "COM") { meaning = "Composer"; }
			else if (key == "CDT") { meaning = "Composer's dates"; }
			else if (key == "CNT") { meaning = "Composer's nationality"; }
			else if (key == "COA") { meaning = "Attributed composer"; }
			else if (key == "COS") { meaning = "Suspected composer"; }
			else if (key == "COL") { meaning = "Composer's stage name"; }
			else if (key == "COC") { meaning = "Composer's corporate name"; }
			else if (key == "CBL") { meaning = "Composer's birth location"; }
			else if (key == "CDL") { meaning = "Composer's death location"; }
			break;

		case 'E':	// electronic editor
			if      (key == "EED") { meaning = "Electronic editor"; }
			else if (key == "ENC") { meaning = "Electronic encoder"; }
			else if (key == "END") { meaning = "Electronic encoding date"; }
			else if (key == "EMD") { meaning = "Modification description"; }
			else if (key == "EEV") { meaning = "Electronic edition version"; }
			else if (key == "EFL") { meaning = "Electronic file number"; }
			else if (key == "EST") { meaning = "Encoding status"; }
			break;

		case 'G':	// group information
			if      (key == "GTL") { meaning = "Group title"; }
			else if (key == "GAW") { meaning = "Associated work"; }
			else if (key == "GCO") { meaning = "Collection designation"; }
			break;

		case 'H':	// historical information
			if      (key == "HAO") { meaning = "Aural history"; }
			else if (key == "HTX") { meaning = "Vocal text translation"; }
			break;

		case 'L':	// lyricist/librettis/arranger/orchestrator reference records
			if      (key == "LYR") { meaning = "Lyricist"; }
			else if (key == "LIB") { meaning = "Librettist"; }
			else if (key == "LOR") { meaning = "Orchestrator"; }
			break;

		case 'M':	// performance information
			if      (key == "MPN") { meaning = "Performer"; }
			else if (key == "MPS") { meaning = "Suspected performer"; }
			else if (key == "MGN") { meaning = "Performance group name"; }
			else if (key == "MRD") { meaning = "Performance date"; }
			else if (key == "MLC") { meaning = "Performance location"; }
			else if (key == "MCN") { meaning = "Conductor"; }
			else if (key == "MPD") { meaning = "Premier date"; }
			break;

		case 'O':	// work (opus) information
			if      (key == "OTL") { meaning = "Work title"; }
			else if (key == "OTP") { meaning = "Popular title"; }
			else if (key == "OTA") { meaning = "Alternative title"; }
			else if (key == "OPR") { meaning = "Parent-work title"; }
			else if (key == "OAC") { meaning = "Act number"; }
			else if (key == "OSC") { meaning = "Scene number"; }
			else if (key == "OMV") { meaning = "Movement number"; }
			else if (key == "OMD") { meaning = "Movement designation"; }
			else if (key == "OPS") { meaning = "Opus number"; }
			else if (key == "ONM") { meaning = "Work number in opus"; }
			else if (key == "OVM") { meaning = "Volume number"; }
			else if (key == "ODE") { meaning = "Dedicatee"; }
			else if (key == "OCO") { meaning = "Commission"; }
			else if (key == "OCL") { meaning = "Collector"; }
			else if (key == "OCL") { meaning = "Free-form note"; }
			else if (key == "OCY") { meaning = "Composition country"; }
			else if (key == "OPC") { meaning = "Composition city"; }
			break;

		case 'P':	// publication information
			if      (key == "PUB") { meaning = "Publication status"; }
			else if (key == "PPR") { meaning = "First publisher"; }
			else if (key == "PTL") { meaning = "Publication title"; }
			else if (key == "PDT") { meaning = "Publication date"; }
			else if (key == "PPP") { meaning = "Publication location"; }
			else if (key == "PC#") { meaning = "Publication catalog number"; }
			else if (key == "SCT") { meaning = "Scholarly catalog abbreviation and number"; }
			else if (key == "SCA") { meaning = "Scholarly catalog unabbreviated name"; }
			else if (key == "SMS") { meaning = "Manuscript source name"; }
			else if (key == "SML") { meaning = "Manuscript location"; }
			else if (key == "SMA") { meaning = "Manuscript access"; }
			break;

		case 'R':
			// recording information
			if      (key == "RTL") { meaning = "Recording Title"; }
			else if (key == "RMM") { meaning = "Manufacturer"; }
			else if (key == "RC#") { meaning = "Catalog number"; }
			else if (key == "RRD") { meaning = "Recording release date"; }
			else if (key == "RLC") { meaning = "Recording location"; }
			else if (key == "RNP") { meaning = "Record producer"; }
			else if (key == "RDT") { meaning = "Recording date"; }
			else if (key == "RT#") { meaning = "Recording track number"; }
			// representation information 
			else if (key == "RLN") { meaning = "ASCII language setting"; }
			else if (key == "RDF") { meaning = "User-defined signifiers"; }
			else if (key == "RDT") { meaning = "Encoding date"; }
			else if (key == "RNB") { meaning = "Encoding note"; }
			else if (key == "RWG") { meaning = "Encoding warning"; }
			break;
		
		case 'T':	// translator
			if      (key == "TRN") { meaning = "Translator"; }
			break;

		case 'V':	// version
			if      (key == "VTS") { meaning = "Data checksum"; }
			break;

		case 'Y':	// copyright information
			if      (key == "YEP") { meaning = "Publisher of electronic edition"; }
			else if (key == "YEC") { meaning = "Electronic edition copyright"; }
			else if (key == "YER") { meaning = "Electronic edition release year"; }
			else if (key == "YEM") { meaning = "Copyright message"; }
			else if (key == "YEC") { meaning = "Country of copyright"; }
			else if (key == "YOR") { meaning = "Original document"; }
			else if (key == "YOO") { meaning = "Original edition owner"; }
			else if (key == "YOY") { meaning = "Original edition copyright year"; }
			else if (key == "YOE") { meaning = "Original edition editor"; }
			break;
	}

	if (!number.empty()) {
		meaning += " #" + number;
	}

	if (!language.empty()) {
		meaning += ", original language " + Convert::getLanguageName(language);
	} else if (!translation.empty()) {
		meaning += ", translated into " + Convert::getLanguageName(language);
	}

	return meaning;
}



//////////////////////////////
//
// Convert::getLanguageName --
//

string Convert::getLanguageName(const string& abbreviation) {
	string code;
	for (int i=0; i<(int)abbreviation.size(); i++) {
		if (abbreviation[i] == '@') {
			continue;
		}
		code.push_back(tolower(abbreviation[i]));
	}

	if (code.size() == 2) {
		// ISO 639-1 language codes

		if (code == "aa") { return "Afar"; }
		if (code == "ab") { return "Abkhazian"; }
		if (code == "ae") { return "Avestan"; }
		if (code == "af") { return "Afrikaans"; }
		if (code == "ak") { return "Akan"; }
		if (code == "am") { return "Amharic"; }
		if (code == "an") { return "Aragonese"; }
		if (code == "ar") { return "Arabic"; }
		if (code == "as") { return "Assamese"; }
		if (code == "av") { return "Avaric"; }
		if (code == "ay") { return "Aymara"; }
		if (code == "az") { return "Azerbaijani"; }
		if (code == "ba") { return "Bashkir"; }
		if (code == "be") { return "Belarusian"; }
		if (code == "bg") { return "Bulgarian"; }
		if (code == "bh") { return "Bihari languages"; }
		if (code == "bi") { return "Bislama"; }
		if (code == "bm") { return "Bambara"; }
		if (code == "bn") { return "Bengali"; }
		if (code == "bo") { return "Tibetan"; }
		if (code == "br") { return "Breton"; }
		if (code == "bs") { return "Bosnian"; }
		if (code == "ca") { return "Catalan"; }
		if (code == "ce") { return "Chechen"; }
		if (code == "ch") { return "Chamorro"; }
		if (code == "co") { return "Corsican"; }
		if (code == "cr") { return "Cree"; }
		if (code == "cs") { return "Czech"; }
		if (code == "cs") { return "Czech"; }
		if (code == "cu") { return "Church Slavic"; }
		if (code == "cv") { return "Chuvash"; }
		if (code == "cy") { return "Welsh"; }
		if (code == "cy") { return "Welsh"; }
		if (code == "da") { return "Danish"; }
		if (code == "de") { return "German"; }
		if (code == "dv") { return "Divehi"; }
		if (code == "dz") { return "Dzongkha"; }
		if (code == "ee") { return "Ewe"; }
		if (code == "el") { return "Greek, Modern (1453-)"; }
		if (code == "en") { return "English"; }
		if (code == "eo") { return "Esperanto"; }
		if (code == "es") { return "Spanish"; }
		if (code == "et") { return "Estonian"; }
		if (code == "eu") { return "Basque"; }
		if (code == "eu") { return "Basque"; }
		if (code == "fa") { return "Persian"; }
		if (code == "ff") { return "Fulah"; }
		if (code == "fi") { return "Finnish"; }
		if (code == "fj") { return "Fijian"; }
		if (code == "fo") { return "Faroese"; }
		if (code == "fr") { return "French"; }
		if (code == "fy") { return "Western Frisian"; }
		if (code == "ga") { return "Irish"; }
		if (code == "gd") { return "Gaelic"; }
		if (code == "gl") { return "Galician"; }
		if (code == "gn") { return "Guarani"; }
		if (code == "gu") { return "Gujarati"; }
		if (code == "gv") { return "Manx"; }
		if (code == "ha") { return "Hausa"; }
		if (code == "he") { return "Hebrew"; }
		if (code == "hi") { return "Hindi"; }
		if (code == "ho") { return "Hiri Motu"; }
		if (code == "hr") { return "Croatian"; }
		if (code == "ht") { return "Haitian"; }
		if (code == "hu") { return "Hungarian"; }
		if (code == "hy") { return "Armenian"; }
		if (code == "hz") { return "Herero"; }
		if (code == "ia") { return "Interlingua"; }
		if (code == "id") { return "Indonesian"; }
		if (code == "ie") { return "Interlingue"; }
		if (code == "ig") { return "Igbo"; }
		if (code == "ii") { return "Sichuan Yi"; }
		if (code == "ik") { return "Inupiaq"; }
		if (code == "io") { return "Ido"; }
		if (code == "is") { return "Icelandic"; }
		if (code == "it") { return "Italian"; }
		if (code == "iu") { return "Inuktitut"; }
		if (code == "ja") { return "Japanese"; }
		if (code == "jv") { return "Javanese"; }
		if (code == "ka") { return "Georgian"; }
		if (code == "kg") { return "Kongo"; }
		if (code == "ki") { return "Kikuyu"; }
		if (code == "kj") { return "Kuanyama"; }
		if (code == "kk") { return "Kazakh"; }
		if (code == "kl") { return "Greenlandic"; }
		if (code == "km") { return "Central Khmer"; }
		if (code == "kn") { return "Kannada"; }
		if (code == "ko") { return "Korean"; }
		if (code == "kr") { return "Kanuri"; }
		if (code == "ks") { return "Kashmiri"; }
		if (code == "ku") { return "Kurdish"; }
		if (code == "kv") { return "Komi"; }
		if (code == "kw") { return "Cornish"; }
		if (code == "ky") { return "Kirghiz"; }
		if (code == "la") { return "Latin"; }
		if (code == "lb") { return "Luxembourgish"; }
		if (code == "lg") { return "Ganda"; }
		if (code == "li") { return "Limburgan"; }
		if (code == "ln") { return "Lingala"; }
		if (code == "lo") { return "Lao"; }
		if (code == "lt") { return "Lithuanian"; }
		if (code == "lu") { return "Luba-Katanga"; }
		if (code == "lv") { return "Latvian"; }
		if (code == "mg") { return "Malagasy"; }
		if (code == "mh") { return "Marshallese"; }
		if (code == "mi") { return "Maori"; }
		if (code == "mk") { return "Macedonian"; }
		if (code == "mk") { return "Macedonian"; }
		if (code == "ml") { return "Malayalam"; }
		if (code == "mn") { return "Mongolian"; }
		if (code == "mr") { return "Marathi"; }
		if (code == "ms") { return "Malay"; }
		if (code == "mt") { return "Maltese"; }
		if (code == "my") { return "Burmese"; }
		if (code == "my") { return "Burmese"; }
		if (code == "na") { return "Nauru"; }
		if (code == "nb") { return "Bokmål, Norwegian"; }
		if (code == "nd") { return "Ndebele, North"; }
		if (code == "ne") { return "Nepali"; }
		if (code == "ng") { return "Ndonga"; }
		if (code == "nl") { return "Dutch"; }
		if (code == "nl") { return "Dutch"; }
		if (code == "nn") { return "Norwegian Nynorsk"; }
		if (code == "no") { return "Norwegian"; }
		if (code == "nr") { return "Ndebele, South"; }
		if (code == "nv") { return "Navajo"; }
		if (code == "ny") { return "Chichewa"; }
		if (code == "oc") { return "Occitan (post 1500)"; }
		if (code == "oj") { return "Ojibwa"; }
		if (code == "om") { return "Oromo"; }
		if (code == "or") { return "Oriya"; }
		if (code == "os") { return "Ossetian"; }
		if (code == "pa") { return "Panjabi"; }
		if (code == "pi") { return "Pali"; }
		if (code == "pl") { return "Polish"; }
		if (code == "ps") { return "Pushto"; }
		if (code == "pt") { return "Portuguese"; }
		if (code == "qu") { return "Quechua"; }
		if (code == "rm") { return "Romansh"; }
		if (code == "rn") { return "Rundi"; }
		if (code == "ro") { return "Romanian"; }
		if (code == "ru") { return "Russian"; }
		if (code == "rw") { return "Kinyarwanda"; }
		if (code == "sa") { return "Sanskrit"; }
		if (code == "sc") { return "Sardinian"; }
		if (code == "sd") { return "Sindhi"; }
		if (code == "se") { return "Northern Sami"; }
		if (code == "sg") { return "Sango"; }
		if (code == "si") { return "Sinhala"; }
		if (code == "sl") { return "Slovenian"; }
		if (code == "sm") { return "Samoan"; }
		if (code == "sn") { return "Shona"; }
		if (code == "so") { return "Somali"; }
		if (code == "sq") { return "Albanian"; }
		if (code == "sr") { return "Serbian"; }
		if (code == "ss") { return "Swati"; }
		if (code == "st") { return "Sotho, Southern"; }
		if (code == "su") { return "Sundanese"; }
		if (code == "sv") { return "Swedish"; }
		if (code == "sw") { return "Swahili"; }
		if (code == "ta") { return "Tamil"; }
		if (code == "te") { return "Telugu"; }
		if (code == "tg") { return "Tajik"; }
		if (code == "th") { return "Thai"; }
		if (code == "ti") { return "Tigrinya"; }
		if (code == "tk") { return "Turkmen"; }
		if (code == "tl") { return "Tagalog"; }
		if (code == "tn") { return "Tswana"; }
		if (code == "to") { return "Tonga (Tonga Islands)"; }
		if (code == "tr") { return "Turkish"; }
		if (code == "ts") { return "Tsonga"; }
		if (code == "tt") { return "Tatar"; }
		if (code == "tw") { return "Twi"; }
		if (code == "ty") { return "Tahitian"; }
		if (code == "ug") { return "Uighur"; }
		if (code == "uk") { return "Ukrainian"; }
		if (code == "ur") { return "Urdu"; }
		if (code == "uz") { return "Uzbek"; }
		if (code == "ve") { return "Venda"; }
		if (code == "vi") { return "Vietnamese"; }
		if (code == "vo") { return "Volapük"; }
		if (code == "wa") { return "Walloon"; }
		if (code == "wo") { return "Wolof"; }
		if (code == "xh") { return "Xhosa"; }
		if (code == "yi") { return "Yiddish"; }
		if (code == "yo") { return "Yoruba"; }
		if (code == "za") { return "Zhuang"; }
		if (code == "zh") { return "Chinese"; }
		if (code == "zu") { return "Zulu"; }

	} else if (code.size() == 3) {
		// ISO 639-2 language codes

		if (code == "aar") { return "Afar"; }
		if (code == "abk") { return "Abkhazian"; }
		if (code == "ace") { return "Achinese"; }
		if (code == "ach") { return "Acoli"; }
		if (code == "ada") { return "Adangme"; }
		if (code == "ady") { return "Adyghe"; }
		if (code == "afa") { return "Afro-Asiatic languages"; }
		if (code == "afh") { return "Afrihili"; }
		if (code == "afr") { return "Afrikaans"; }
		if (code == "ain") { return "Ainu"; }
		if (code == "aka") { return "Akan"; }
		if (code == "akk") { return "Akkadian"; }
		if (code == "alb") { return "Albanian"; }
		if (code == "ale") { return "Aleut"; }
		if (code == "alg") { return "Algonquian languages"; }
		if (code == "alt") { return "Southern Altai"; }
		if (code == "amh") { return "Amharic"; }
		if (code == "ang") { return "English, Old (ca.450-1100)"; }
		if (code == "anp") { return "Angika"; }
		if (code == "apa") { return "Apache languages"; }
		if (code == "ara") { return "Arabic"; }
		if (code == "arc") { return "Aramaic (700-300 BCE)"; }
		if (code == "arg") { return "Aragonese"; }
		if (code == "arm") { return "Armenian"; }
		if (code == "arn") { return "Mapudungun"; }
		if (code == "arp") { return "Arapaho"; }
		if (code == "art") { return "Artificial languages"; }
		if (code == "arw") { return "Arawak"; }
		if (code == "asm") { return "Assamese"; }
		if (code == "ast") { return "Asturian"; }
		if (code == "ath") { return "Athapascan languages"; }
		if (code == "aus") { return "Australian languages"; }
		if (code == "ava") { return "Avaric"; }
		if (code == "ave") { return "Avestan"; }
		if (code == "awa") { return "Awadhi"; }
		if (code == "aym") { return "Aymara"; }
		if (code == "aze") { return "Azerbaijani"; }
		if (code == "bad") { return "Banda languages"; }
		if (code == "bai") { return "Bamileke languages"; }
		if (code == "bak") { return "Bashkir"; }
		if (code == "bal") { return "Baluchi"; }
		if (code == "bam") { return "Bambara"; }
		if (code == "ban") { return "Balinese"; }
		if (code == "baq") { return "Basque"; }
		if (code == "baq") { return "Basque"; }
		if (code == "bas") { return "Basa"; }
		if (code == "bat") { return "Baltic languages"; }
		if (code == "bej") { return "Beja"; }
		if (code == "bel") { return "Belarusian"; }
		if (code == "bem") { return "Bemba"; }
		if (code == "ben") { return "Bengali"; }
		if (code == "ber") { return "Berber languages"; }
		if (code == "bho") { return "Bhojpuri"; }
		if (code == "bih") { return "Bihari languages"; }
		if (code == "bik") { return "Bikol"; }
		if (code == "bin") { return "Bini"; }
		if (code == "bis") { return "Bislama"; }
		if (code == "bla") { return "Siksika"; }
		if (code == "bnt") { return "Bantu languages"; }
		if (code == "bod") { return "Tibetan"; }
		if (code == "bos") { return "Bosnian"; }
		if (code == "bra") { return "Braj"; }
		if (code == "bre") { return "Breton"; }
		if (code == "btk") { return "Batak languages"; }
		if (code == "bua") { return "Buriat"; }
		if (code == "bug") { return "Buginese"; }
		if (code == "bul") { return "Bulgarian"; }
		if (code == "bur") { return "Burmese"; }
		if (code == "bur") { return "Burmese"; }
		if (code == "byn") { return "Blin"; }
		if (code == "cad") { return "Caddo"; }
		if (code == "cai") { return "Central American Indian languages"; }
		if (code == "car") { return "Galibi Carib"; }
		if (code == "cat") { return "Catalan"; }
		if (code == "cau") { return "Caucasian languages"; }
		if (code == "ceb") { return "Cebuano"; }
		if (code == "cel") { return "Celtic languages"; }
		if (code == "ces") { return "Czech"; }
		if (code == "ces") { return "Czech"; }
		if (code == "cha") { return "Chamorro"; }
		if (code == "chb") { return "Chibcha"; }
		if (code == "che") { return "Chechen"; }
		if (code == "chg") { return "Chagatai"; }
		if (code == "chi") { return "Chinese"; }
		if (code == "chk") { return "Chuukese"; }
		if (code == "chm") { return "Mari"; }
		if (code == "chn") { return "Chinook jargon"; }
		if (code == "cho") { return "Choctaw"; }
		if (code == "chp") { return "Chipewyan"; }
		if (code == "chr") { return "Cherokee"; }
		if (code == "chu") { return "Church Slavic"; }
		if (code == "chv") { return "Chuvash"; }
		if (code == "chy") { return "Cheyenne"; }
		if (code == "cmc") { return "Chamic languages"; }
		if (code == "cnr") { return "Montenegrin"; }
		if (code == "cop") { return "Coptic"; }
		if (code == "cor") { return "Cornish"; }
		if (code == "cos") { return "Corsican"; }
		if (code == "cpe") { return "Creoles and pidgins, English based"; }
		if (code == "cpf") { return "Creoles and pidgins, French-based"; }
		if (code == "cpp") { return "Creoles and pidgins, Portuguese-based"; }
		if (code == "cre") { return "Cree"; }
		if (code == "crh") { return "Crimean Tatar"; }
		if (code == "crp") { return "Creoles and pidgins"; }
		if (code == "csb") { return "Kashubian"; }
		if (code == "cus") { return "Cushitic languages"; }
		if (code == "cym") { return "Welsh"; }
		if (code == "cym") { return "Welsh"; }
		if (code == "cze") { return "Czech"; }
		if (code == "cze") { return "Czech"; }
		if (code == "dak") { return "Dakota"; }
		if (code == "dan") { return "Danish"; }
		if (code == "dar") { return "Dargwa"; }
		if (code == "day") { return "Land Dayak languages"; }
		if (code == "del") { return "Delaware"; }
		if (code == "den") { return "Slave (Athapascan)"; }
		if (code == "deu") { return "German"; }
		if (code == "dgr") { return "Dogrib"; }
		if (code == "din") { return "Dinka"; }
		if (code == "div") { return "Divehi"; }
		if (code == "doi") { return "Dogri"; }
		if (code == "dra") { return "Dravidian languages"; }
		if (code == "dsb") { return "Lower Sorbian"; }
		if (code == "dua") { return "Duala"; }
		if (code == "dum") { return "Dutch, Middle (ca.1050-1350)"; }
		if (code == "dut") { return "Dutch"; }
		if (code == "dut") { return "Dutch"; }
		if (code == "dyu") { return "Dyula"; }
		if (code == "dzo") { return "Dzongkha"; }
		if (code == "efi") { return "Efik"; }
		if (code == "egy") { return "Egyptian (Ancient)"; }
		if (code == "eka") { return "Ekajuk"; }
		if (code == "ell") { return "Greek, Modern (1453-)"; }
		if (code == "elx") { return "Elamite"; }
		if (code == "eng") { return "English"; }
		if (code == "enm") { return "English, Middle (1100-1500)"; }
		if (code == "epo") { return "Esperanto"; }
		if (code == "est") { return "Estonian"; }
		if (code == "eus") { return "Basque"; }
		if (code == "eus") { return "Basque"; }
		if (code == "ewe") { return "Ewe"; }
		if (code == "ewo") { return "Ewondo"; }
		if (code == "fan") { return "Fang"; }
		if (code == "fao") { return "Faroese"; }
		if (code == "fas") { return "Persian"; }
		if (code == "fat") { return "Fanti"; }
		if (code == "fij") { return "Fijian"; }
		if (code == "fil") { return "Filipino"; }
		if (code == "fin") { return "Finnish"; }
		if (code == "fiu") { return "Finno-Ugrian languages"; }
		if (code == "fon") { return "Fon"; }
		if (code == "fra") { return "French"; }
		if (code == "fre") { return "French"; }
		if (code == "frm") { return "French, Middle (ca.1400-1600)"; }
		if (code == "fro") { return "French, Old (842-ca.1400)"; }
		if (code == "frr") { return "Northern Frisian"; }
		if (code == "frs") { return "Eastern Frisian"; }
		if (code == "fry") { return "Western Frisian"; }
		if (code == "ful") { return "Fulah"; }
		if (code == "fur") { return "Friulian"; }
		if (code == "gaa") { return "Ga"; }
		if (code == "gay") { return "Gayo"; }
		if (code == "gba") { return "Gbaya"; }
		if (code == "gem") { return "Germanic languages"; }
		if (code == "geo") { return "Georgin"; }
		if (code == "ger") { return "German"; }
		if (code == "gez") { return "Geez"; }
		if (code == "gil") { return "Gilbertese"; }
		if (code == "gla") { return "Gaelic"; }
		if (code == "gle") { return "Irish"; }
		if (code == "glg") { return "Galician"; }
		if (code == "glv") { return "Manx"; }
		if (code == "gmh") { return "German, Middle High (ca.1050-1500)"; }
		if (code == "goh") { return "German, Old High (ca.750-1050)"; }
		if (code == "gon") { return "Gondi"; }
		if (code == "gor") { return "Gorontalo"; }
		if (code == "got") { return "Gothic"; }
		if (code == "grb") { return "Grebo"; }
		if (code == "grc") { return "Greek, Ancient (to 1453)"; }
		if (code == "gre") { return "Greek"; }
		if (code == "grn") { return "Guarani"; }
		if (code == "gsw") { return "Swiss German"; }
		if (code == "guj") { return "Gujarati"; }
		if (code == "gwi") { return "Gwich'in"; }
		if (code == "hai") { return "Haida"; }
		if (code == "hat") { return "Haitian"; }
		if (code == "hau") { return "Hausa"; }
		if (code == "haw") { return "Hawaiian"; }
		if (code == "heb") { return "Hebrew"; }
		if (code == "her") { return "Herero"; }
		if (code == "hil") { return "Hiligaynon"; }
		if (code == "him") { return "Himachali languages"; }
		if (code == "hin") { return "Hindi"; }
		if (code == "hit") { return "Hittite"; }
		if (code == "hmn") { return "Hmong"; }
		if (code == "hmo") { return "Hiri Motu"; }
		if (code == "hrv") { return "Croatian"; }
		if (code == "hsb") { return "Upper Sorbian"; }
		if (code == "hun") { return "Hungarian"; }
		if (code == "hup") { return "Hupa"; }
		if (code == "hye") { return "Armenian"; }
		if (code == "iba") { return "Iban"; }
		if (code == "ibo") { return "Igbo"; }
		if (code == "ice") { return "Icelandic"; }
		if (code == "ido") { return "Ido"; }
		if (code == "iii") { return "Sichuan Yi"; }
		if (code == "ijo") { return "Ijo languages"; }
		if (code == "iku") { return "Inuktitut"; }
		if (code == "ile") { return "Interlingue"; }
		if (code == "ilo") { return "Iloko"; }
		if (code == "ina") { return "Interlingua)"; }
		if (code == "inc") { return "Indic languages"; }
		if (code == "ind") { return "Indonesian"; }
		if (code == "ine") { return "Indo-European languages"; }
		if (code == "inh") { return "Ingush"; }
		if (code == "ipk") { return "Inupiaq"; }
		if (code == "ira") { return "Iranian languages"; }
		if (code == "iro") { return "Iroquoian languages"; }
		if (code == "isl") { return "Icelandic"; }
		if (code == "ita") { return "Italian"; }
		if (code == "jav") { return "Javanese"; }
		if (code == "jbo") { return "Lojban"; }
		if (code == "jpn") { return "Japanese"; }
		if (code == "jpr") { return "Judeo-Persian"; }
		if (code == "jrb") { return "Judeo-Arabic"; }
		if (code == "kaa") { return "Kara-Kalpak"; }
		if (code == "kab") { return "Kabyle"; }
		if (code == "kac") { return "Kachin"; }
		if (code == "kal") { return "Greenlandic"; }
		if (code == "kam") { return "Kamba"; }
		if (code == "kan") { return "Kannada"; }
		if (code == "kar") { return "Karen languages"; }
		if (code == "kas") { return "Kashmiri"; }
		if (code == "kat") { return "Georgian"; }
		if (code == "kau") { return "Kanuri"; }
		if (code == "kaw") { return "Kawi"; }
		if (code == "kaz") { return "Kazakh"; }
		if (code == "kbd") { return "Kabardian"; }
		if (code == "kha") { return "Khasi"; }
		if (code == "khi") { return "Khoisan languages"; }
		if (code == "khm") { return "Central Khmer"; }
		if (code == "kho") { return "Khotanese"; }
		if (code == "kik") { return "Kikuyu"; }
		if (code == "kin") { return "Kinyarwanda"; }
		if (code == "kir") { return "Kirghiz"; }
		if (code == "kmb") { return "Kimbundu"; }
		if (code == "kok") { return "Konkani"; }
		if (code == "kom") { return "Komi"; }
		if (code == "kon") { return "Kongo"; }
		if (code == "kor") { return "Korean"; }
		if (code == "kos") { return "Kosraean"; }
		if (code == "kpe") { return "Kpelle"; }
		if (code == "krc") { return "Karachay-Balkar"; }
		if (code == "krl") { return "Karelian"; }
		if (code == "kro") { return "Kru languages"; }
		if (code == "kru") { return "Kurukh"; }
		if (code == "kua") { return "Kuanyama"; }
		if (code == "kum") { return "Kumyk"; }
		if (code == "kur") { return "Kurdish"; }
		if (code == "kut") { return "Kutenai"; }
		if (code == "lad") { return "Ladino"; }
		if (code == "lah") { return "Lahnda"; }
		if (code == "lam") { return "Lamba"; }
		if (code == "lao") { return "Lao"; }
		if (code == "lat") { return "Latin"; }
		if (code == "lav") { return "Latvian"; }
		if (code == "lez") { return "Lezghian"; }
		if (code == "lim") { return "Limburgan"; }
		if (code == "lin") { return "Lingala"; }
		if (code == "lit") { return "Lithuanian"; }
		if (code == "lol") { return "Mongo"; }
		if (code == "loz") { return "Lozi"; }
		if (code == "ltz") { return "Luxembourgish"; }
		if (code == "lua") { return "Luba-Lulua"; }
		if (code == "lub") { return "Luba-Katanga"; }
		if (code == "lug") { return "Ganda"; }
		if (code == "lui") { return "Luiseno"; }
		if (code == "lun") { return "Lunda"; }
		if (code == "luo") { return "Luo (Kenya and Tanzania)"; }
		if (code == "lus") { return "Lushai"; }
		if (code == "mac") { return "Macedonian"; }
		if (code == "mac") { return "Macedonian"; }
		if (code == "mad") { return "Madurese"; }
		if (code == "mag") { return "Magahi"; }
		if (code == "mah") { return "Marshallese"; }
		if (code == "mai") { return "Maithili"; }
		if (code == "mak") { return "Makasar"; }
		if (code == "mal") { return "Malayalam"; }
		if (code == "man") { return "Mandingo"; }
		if (code == "mao") { return "Maori"; }
		if (code == "map") { return "Austronesian languages"; }
		if (code == "mar") { return "Marathi"; }
		if (code == "mas") { return "Masai"; }
		if (code == "may") { return "Malay"; }
		if (code == "mdf") { return "Moksha"; }
		if (code == "mdr") { return "Mandar"; }
		if (code == "men") { return "Mende"; }
		if (code == "mga") { return "Irish, Middle (900-1200)"; }
		if (code == "mic") { return "Mi'kmaq"; }
		if (code == "min") { return "Minangkabau"; }
		if (code == "mis") { return "Uncoded languages"; }
		if (code == "mkd") { return "Macedonian"; }
		if (code == "mkd") { return "Macedonian"; }
		if (code == "mkh") { return "Mon-Khmer languages"; }
		if (code == "mlg") { return "Malagasy"; }
		if (code == "mlt") { return "Maltese"; }
		if (code == "mnc") { return "Manchu"; }
		if (code == "mni") { return "Manipuri"; }
		if (code == "mno") { return "Manobo languages"; }
		if (code == "moh") { return "Mohawk"; }
		if (code == "mon") { return "Mongolian"; }
		if (code == "mos") { return "Mossi"; }
		if (code == "mri") { return "Maori"; }
		if (code == "msa") { return "Malay"; }
		if (code == "mul") { return "Multiple languages"; }
		if (code == "mun") { return "Munda languages"; }
		if (code == "mus") { return "Creek"; }
		if (code == "mwl") { return "Mirandese"; }
		if (code == "mwr") { return "Marwari"; }
		if (code == "mya") { return "Burmese"; }
		if (code == "mya") { return "Burmese"; }
		if (code == "myn") { return "Mayan languages"; }
		if (code == "myv") { return "Erzya"; }
		if (code == "nah") { return "Nahuatl languages"; }
		if (code == "nai") { return "North American Indian languages"; }
		if (code == "nap") { return "Neapolitan"; }
		if (code == "nau") { return "Nauru"; }
		if (code == "nav") { return "Navajo"; }
		if (code == "nbl") { return "Ndebele, South"; }
		if (code == "nde") { return "Ndebele, North"; }
		if (code == "ndo") { return "Ndonga"; }
		if (code == "nds") { return "Low German"; }
		if (code == "nep") { return "Nepali"; }
		if (code == "new") { return "Nepal Bhasa"; }
		if (code == "nia") { return "Nias"; }
		if (code == "nic") { return "Niger-Kordofanian languages"; }
		if (code == "niu") { return "Niuean"; }
		if (code == "nld") { return "Dutch"; }
		if (code == "nld") { return "Dutch"; }
		if (code == "nno") { return "Norwegian Nynorsk"; }
		if (code == "nob") { return "Bokmål, Norwegian"; }
		if (code == "nog") { return "Nogai"; }
		if (code == "non") { return "Norse, Old"; }
		if (code == "nor") { return "Norwegian"; }
		if (code == "nqo") { return "N'Ko"; }
		if (code == "nso") { return "Pedi"; }
		if (code == "nub") { return "Nubian languages"; }
		if (code == "nwc") { return "Classical Newari"; }
		if (code == "nya") { return "Chichewa"; }
		if (code == "nym") { return "Nyamwezi"; }
		if (code == "nyn") { return "Nyankole"; }
		if (code == "nyo") { return "Nyoro"; }
		if (code == "nzi") { return "Nzima"; }
		if (code == "oci") { return "Occitan (post 1500)"; }
		if (code == "oji") { return "Ojibwa"; }
		if (code == "ori") { return "Oriya"; }
		if (code == "orm") { return "Oromo"; }
		if (code == "osa") { return "Osage"; }
		if (code == "oss") { return "Ossetian"; }
		if (code == "ota") { return "Turkish, Ottoman (1500-1928)"; }
		if (code == "oto") { return "Otomian languages"; }
		if (code == "paa") { return "Papuan languages"; }
		if (code == "pag") { return "Pangasinan"; }
		if (code == "pal") { return "Pahlavi"; }
		if (code == "pam") { return "Pampanga"; }
		if (code == "pan") { return "Panjabi"; }
		if (code == "pap") { return "Papiamento"; }
		if (code == "pau") { return "Palauan"; }
		if (code == "peo") { return "Persian, Old (ca.600-400 B.C.)"; }
		if (code == "per") { return "Persian"; }
		if (code == "phi") { return "Philippine languages"; }
		if (code == "phn") { return "Phoenician"; }
		if (code == "pli") { return "Pali"; }
		if (code == "pol") { return "Polish"; }
		if (code == "pon") { return "Pohnpeian"; }
		if (code == "por") { return "Portuguese"; }
		if (code == "pra") { return "Prakrit languages"; }
		if (code == "pro") { return "Provençal, Old (to 1500)"; }
		if (code == "pus") { return "Pushto"; }
		if (code == "que") { return "Quechua"; }
		if (code == "raj") { return "Rajasthani"; }
		if (code == "rap") { return "Rapanui"; }
		if (code == "rar") { return "Rarotongan"; }
		if (code == "roa") { return "Romance languages"; }
		if (code == "roh") { return "Romansh"; }
		if (code == "rom") { return "Romany"; }
		if (code == "ron") { return "Romanian"; }
		if (code == "rum") { return "Romanian"; }
		if (code == "run") { return "Rundi"; }
		if (code == "rup") { return "Aromanian"; }
		if (code == "rus") { return "Russian"; }
		if (code == "sad") { return "Sandawe"; }
		if (code == "sag") { return "Sango"; }
		if (code == "sah") { return "Yakut"; }
		if (code == "sai") { return "South American Indian languages"; }
		if (code == "sal") { return "Salishan languages"; }
		if (code == "sam") { return "Samaritan Aramaic"; }
		if (code == "san") { return "Sanskrit"; }
		if (code == "sas") { return "Sasak"; }
		if (code == "sat") { return "Santali"; }
		if (code == "scn") { return "Sicilian"; }
		if (code == "sco") { return "Scots"; }
		if (code == "sel") { return "Selkup"; }
		if (code == "sem") { return "Semitic languages"; }
		if (code == "sga") { return "Irish, Old (to 900)"; }
		if (code == "sgn") { return "Sign Languages"; }
		if (code == "shn") { return "Shan"; }
		if (code == "sid") { return "Sidamo"; }
		if (code == "sin") { return "Sinhala"; }
		if (code == "sio") { return "Siouan languages"; }
		if (code == "sit") { return "Sino-Tibetan languages"; }
		if (code == "sla") { return "Slavic languages"; }
		if (code == "slo") { return "Slovak"; }
		if (code == "slv") { return "Slovenian"; }
		if (code == "sma") { return "Southern Sami"; }
		if (code == "sme") { return "Northern Sami"; }
		if (code == "smi") { return "Sami languages"; }
		if (code == "smj") { return "Lule Sami"; }
		if (code == "smn") { return "Inari Sami"; }
		if (code == "smo") { return "Samoan"; }
		if (code == "sms") { return "Skolt Sami"; }
		if (code == "sna") { return "Shona"; }
		if (code == "snd") { return "Sindhi"; }
		if (code == "snk") { return "Soninke"; }
		if (code == "sog") { return "Sogdian"; }
		if (code == "som") { return "Somali"; }
		if (code == "son") { return "Songhai languages"; }
		if (code == "sot") { return "Sotho, Southern"; }
		if (code == "spa") { return "Spanish"; }
		if (code == "sqi") { return "Albanian"; }
		if (code == "srd") { return "Sardinian"; }
		if (code == "srn") { return "Sranan Tongo"; }
		if (code == "srp") { return "Serbian"; }
		if (code == "srr") { return "Serer"; }
		if (code == "ssa") { return "Nilo-Saharan languages"; }
		if (code == "ssw") { return "Swati"; }
		if (code == "suk") { return "Sukuma"; }
		if (code == "sun") { return "Sundanese"; }
		if (code == "sus") { return "Susu"; }
		if (code == "sux") { return "Sumerian"; }
		if (code == "swa") { return "Swahili"; }
		if (code == "swe") { return "Swedish"; }
		if (code == "syc") { return "Classical Syriac"; }
		if (code == "syr") { return "Syriac"; }
		if (code == "tah") { return "Tahitian"; }
		if (code == "tai") { return "Tai languages"; }
		if (code == "tam") { return "Tamil"; }
		if (code == "tat") { return "Tatar"; }
		if (code == "tel") { return "Telugu"; }
		if (code == "tem") { return "Timne"; }
		if (code == "ter") { return "Tereno"; }
		if (code == "tet") { return "Tetum"; }
		if (code == "tgk") { return "Tajik"; }
		if (code == "tgl") { return "Tagalog"; }
		if (code == "tha") { return "Thai"; }
		if (code == "tib") { return "Tibetian"; }
		if (code == "tig") { return "Tigre"; }
		if (code == "tir") { return "Tigrinya"; }
		if (code == "tiv") { return "Tiv"; }
		if (code == "tkl") { return "Tokelau"; }
		if (code == "tlh") { return "Klingon"; }
		if (code == "tli") { return "Tlingit"; }
		if (code == "tmh") { return "Tamashek"; }
		if (code == "tog") { return "Tonga (Nyasa)"; }
		if (code == "ton") { return "Tonga (Tonga Islands)"; }
		if (code == "tpi") { return "Tok Pisin"; }
		if (code == "tsi") { return "Tsimshian"; }
		if (code == "tsn") { return "Tswana"; }
		if (code == "tso") { return "Tsonga"; }
		if (code == "tuk") { return "Turkmen"; }
		if (code == "tum") { return "Tumbuka"; }
		if (code == "tup") { return "Tupi languages"; }
		if (code == "tur") { return "Turkish"; }
		if (code == "tut") { return "Altaic languages"; }
		if (code == "tvl") { return "Tuvalu"; }
		if (code == "twi") { return "Twi"; }
		if (code == "tyv") { return "Tuvinian"; }
		if (code == "udm") { return "Udmurt"; }
		if (code == "uga") { return "Ugaritic"; }
		if (code == "uig") { return "Uighur"; }
		if (code == "ukr") { return "Ukrainian"; }
		if (code == "umb") { return "Umbundu"; }
		if (code == "und") { return "Undetermined"; }
		if (code == "urd") { return "Urdu"; }
		if (code == "uzb") { return "Uzbek"; }
		if (code == "vai") { return "Vai"; }
		if (code == "ven") { return "Venda"; }
		if (code == "vie") { return "Vietnamese"; }
		if (code == "vol") { return "Volapük"; }
		if (code == "vot") { return "Votic"; }
		if (code == "wak") { return "Wakashan languages"; }
		if (code == "wal") { return "Wolaitta"; }
		if (code == "war") { return "Waray"; }
		if (code == "was") { return "Washo"; }
		if (code == "wel") { return "Welsh"; }
		if (code == "wel") { return "Welsh"; }
		if (code == "wen") { return "Sorbian languages"; }
		if (code == "wln") { return "Walloon"; }
		if (code == "wol") { return "Wolof"; }
		if (code == "xal") { return "Kalmyk"; }
		if (code == "xho") { return "Xhosa"; }
		if (code == "yao") { return "Yao"; }
		if (code == "yap") { return "Yapese"; }
		if (code == "yid") { return "Yiddish"; }
		if (code == "yor") { return "Yoruba"; }
		if (code == "ypk") { return "Yupik languages"; }
		if (code == "zap") { return "Zapotec"; }
		if (code == "zbl") { return "Blissymbols"; }
		if (code == "zen") { return "Zenaga"; }
		if (code == "zgh") { return "Moroccan"; }
		if (code == "zha") { return "Zhuang"; }
		if (code == "zho") { return "Chinese"; }
		if (code == "znd") { return "Zande languages"; }
		if (code == "zul") { return "Zulu"; }
		if (code == "zun") { return "Zuni"; }
		if (code == "zza") { return "Zaza"; }
	}
	return code;
}




//////////////////////////////
//
// Convert::recipToDuration -- Convert **recip rhythmic values into
//     rational number durations in terms of quarter notes.  For example "4"
//     will be converted to 1, "4." to 3/2 (1+1/2).  The second parameter
//     is a scaling factor which can change the rhythmic value's base duration.
//     Giving a scale of 1 will return the duration in whole note units, so
//     "4" will return a value of 1/4 (one quarter of a whole note).  Using
//     3/2 will give the duration in terms of dotted-quarter note units.
//     The third parameter is the sub-token separate.  For example if the input
//     string contains a space, anything after the first space will be ignored
//     when extracting the string.  **kern data which also includes the pitch
//     along with the rhythm can also be given and will be ignored.
// default value: scale = 4 (duration in terms of quarter notes)
// default value: separator = " " (sub-token separator)
//

HumNum Convert::recipToDuration(string* recip, HumNum scale,
		const string& separator) {
	return Convert::recipToDuration(*recip, scale, separator);
}


HumNum Convert::recipToDuration(const string& recip, HumNum scale,
		const string& separator) {
	size_t loc;
	loc = recip.find(separator);
	string subtok;
	if (loc != string::npos) {
		subtok = recip.substr(0, loc);
	} else {
		subtok = recip;
	}

	loc = recip.find('q');
	if (loc != string::npos) {
		// grace note, ignore printed rhythm
		HumNum zero(0);
		return zero;
	}

	int dotcount = 0;
	int i;
	int numi = -1;
	for (i=0; i<(int)subtok.size(); i++) {
		if (subtok[i] == '.') {
			dotcount++;
		}
		if ((numi < 0) && isdigit(subtok[i])) {
			numi = i;
		}
	}
	loc = subtok.find("%");
	int numerator = 1;
	int denominator = 1;
	HumNum output;
	if (loc != string::npos) {
		// reciprocal rhythm
		numerator = 1;
		denominator = subtok[numi++] - '0';
		while ((numi<(int)subtok.size()) && isdigit(subtok[numi])) {
			denominator = denominator * 10 + (subtok[numi++] - '0');
		}
		if ((loc + 1 < subtok.size()) && isdigit(subtok[loc+1])) {
			int xi = (int)loc + 1;
			numerator = subtok[xi++] - '0';
			while ((xi<(int)subtok.size()) && isdigit(subtok[xi])) {
				numerator = numerator * 10 + (subtok[xi++] - '0');
			}
		}
		output.setValue(numerator, denominator);
	} else if (numi < 0) {
		// no rhythm found
		HumNum zero(0);
		return zero;
	} else if (subtok[numi] == '0') {
		// 0-symbol
		int zerocount = 1;
		for (i=numi+1; i<(int)subtok.size(); i++) {
			if (subtok[i] == '0') {
				zerocount++;
			} else {
				break;
			}
		}
		numerator = (int)pow(2, zerocount);
		output.setValue(numerator, 1);
	} else {
		// plain rhythm
		denominator = subtok[numi++] - '0';
		while ((numi<(int)subtok.size()) && isdigit(subtok[numi])) {
			denominator = denominator * 10 + (subtok[numi++] - '0');
		}
		output.setValue(1, denominator);
	}

	if (dotcount <= 0) {
		return output * scale;
	}

	int bot = (int)pow(2.0, dotcount);
	int top = (int)pow(2.0, dotcount + 1) - 1;
	HumNum factor(top, bot);
	return output * factor * scale;
}



//////////////////////////////
//
// Convert::recipToDurationIgnoreGrace -- Similar to recipToDuration(), but 
//     do not set grace notes to a zero duration, but rather give their
//     visual duration.
// default value: scale = 4 (duration in terms of quarter notes)
// default value: separator = " " (sub-token separator)
//

HumNum Convert::recipToDurationIgnoreGrace(string* recip, HumNum scale,
		const string& separator) {
	return Convert::recipToDurationIgnoreGrace(*recip, scale, separator);
}


HumNum Convert::recipToDurationIgnoreGrace(const string& recip, HumNum scale,
		const string& separator) {
	size_t loc;
	loc = recip.find(separator);
	string subtok;
	if (loc != string::npos) {
		subtok = recip.substr(0, loc);
	} else {
		subtok = recip;
	}

	int dotcount = 0;
	int i;
	int numi = -1;
	for (i=0; i<(int)subtok.size(); i++) {
		if (subtok[i] == '.') {
			dotcount++;
		}
		if ((numi < 0) && isdigit(subtok[i])) {
			numi = i;
		}
	}
	loc = subtok.find("%");
	int numerator = 1;
	int denominator = 1;
	HumNum output;
	if (loc != string::npos) {
		// reciprocal rhythm
		numerator = 1;
		denominator = subtok[numi++] - '0';
		while ((numi<(int)subtok.size()) && isdigit(subtok[numi])) {
			denominator = denominator * 10 + (subtok[numi++] - '0');
		}
		if ((loc + 1 < subtok.size()) && isdigit(subtok[loc+1])) {
			int xi = (int)loc + 1;
			numerator = subtok[xi++] - '0';
			while ((xi<(int)subtok.size()) && isdigit(subtok[xi])) {
				numerator = numerator * 10 + (subtok[xi++] - '0');
			}
		}
		output.setValue(numerator, denominator);
	} else if (numi < 0) {
		// no rhythm found
		HumNum zero(0);
		return zero;
	} else if (subtok[numi] == '0') {
		// 0-symbol
		int zerocount = 1;
		for (i=numi+1; i<(int)subtok.size(); i++) {
			if (subtok[i] == '0') {
				zerocount++;
			} else {
				break;
			}
		}
		numerator = (int)pow(2, zerocount);
		output.setValue(numerator, 1);
	} else {
		// plain rhythm
		denominator = subtok[numi++] - '0';
		while ((numi<(int)subtok.size()) && isdigit(subtok[numi])) {
			denominator = denominator * 10 + (subtok[numi++] - '0');
		}
		output.setValue(1, denominator);
	}

	if (dotcount <= 0) {
		return output * scale;
	}

	int bot = (int)pow(2.0, dotcount);
	int top = (int)pow(2.0, dotcount + 1) - 1;
	HumNum factor(top, bot);
	return output * factor * scale;
}



//////////////////////////////
//
// Convert::recipToDurationNoDots -- Same as recipToDuration(), but ignore
//   any augmentation dots.
//

HumNum Convert::recipToDurationNoDots(string* recip, HumNum scale,
		const string& separator) {
	return Convert::recipToDurationNoDots(*recip, scale, separator);
}


HumNum Convert::recipToDurationNoDots(const string& recip, HumNum scale,
		const string& separator) {
	string temp = recip;
	std::replace(temp.begin(), temp.end(), '.', 'Z');
	return Convert::recipToDuration(temp, scale, separator);
}


//////////////////////////////
//
// Convert::durationToRecip -- Duration input is in units of quarter notes,
//     since the default value for scale is 1/4.
//

string Convert::durationToRecip(HumNum duration, HumNum scale) {
	duration *= scale;
	if (duration.getNumerator() == 1) {
		// simple rhythm (integer divisions of the whole note)
		return to_string(duration.getDenominator());
	}
	if (duration.getDenominator() == 1) {
		if (duration.getNumerator() == 2) {
			return "0";  // breve
		} else if (duration.getNumerator() == 3) {
			return "0."; // dotted breve
		} else if (duration.getNumerator() == 4) {
			return "00";  // long
		} else if (duration.getNumerator() == 6) {
			return "00."; // dotted long
		} else if (duration.getNumerator() == 8) {
			return "000";  // maxima
		} else if (duration.getNumerator() == 12) {
			return "000."; // dotted maxima
		}
	}
	if (duration.getNumerator() == 0) {
		// grace note
		return "q";
	}

	// now decide if the rhythm can be represented simply with one dot.
	HumNum test1dot = (duration * 2) / 3;
	if (test1dot.getNumerator() == 1) {
		// single dot works
		// string output = to_string(test1dot.getDenominator() * 2);
		string output = to_string(test1dot.getDenominator());
		output += ".";
		return output;
	}

	// now decide if the rhythm can be represented simply with two dots.
	HumNum test2dot = (duration * 4) / 7;
	if (test2dot.getNumerator() == 1) {
		// double dot works
		string output = to_string(test2dot.getDenominator() * 2);
		output += "..";
		return output;
	}

	// now decide if the rhythm can be represented simply with three dots.
	HumNum test3dot = (duration * 8) / 15;
	if (test3dot.getNumerator() == 1) {
		// single dot works
		string output = to_string(test3dot.getDenominator() * 4);
		output += "...";
		return output;
	}

	// duration required more than three dots or is not simple,
	// so assume that it is not simple:
	string output = to_string(duration.getDenominator());
	output += "%";
	output += to_string(duration.getNumerator());
	return output;
}



//////////////////////////////
//
// Convert::durationFloatToRecip -- not allowed to have more than
//	three rhythmic dots
//	default value: timebase = 1;
//

string Convert::durationFloatToRecip(double input, HumNum timebase) {
	string output;

   double testinput = input;
   double basic = 4.0 / input * timebase.getFloat();
   double diff = basic - (int)basic;

   if (diff > 0.998) {
      diff = 1.0 - diff;
      basic += diff;
   }

	// do power of two checks instead
   if (input == 0.0625)  { output = "64"; return output; }
   if (input == 0.125)   { output = "32"; return output; }
   if (input == 0.25)    { output = "16"; return output; }
   if (input == 0.5)  { output = "8";    return output; }
   if (input == 1.0)  { output = "4";    return output; }
   if (input == 2.0)  { output = "2";    return output; }
   if (input == 4.0)  { output = "1";    return output; }
   if (input == 8.0)  { output = "0";    return output; }
   if (input == 12.0) { output = "0.";   return output; }
   if (input == 16.0) { output = "00";   return output; }
   if (input == 24.0) { output = "00.";  return output; }
   if (input == 32.0) { output = "000";  return output; }
   if (input == 48.0) { output = "000."; return output; }

   // special case for triplet whole notes:
   if (fabs(input - (4.0 * 2.0 / 3.0)) < 0.0001) {
		return "3%2";
   }

   // special case for triplet breve notes:
   if (fabs(input - (4.0 * 4.0 / 3.0)) < 0.0001) {
		return "3%4";
   }

   // special case for 9/8 full rests
   if (fabs(input - (4.0 * 9.0 / 8.0)) < 0.0001) {
		return "8%9";
   }

   // special case for 9/2 full-measure rest
   if (fabs(input - 18.0) < 0.0001) {
		return "2%9";
   }

   // handle special rounding cases primarily for SCORE which
   // only stores 4 digits for a duration
   if (input == 0.0833) {
      // triplet 32nd note, which has a real duration of 0.0833333 etc.
		return "48";
   }

   if (diff < 0.002) {
		output += to_string((int)basic);
   } else {
      testinput = input / 3.0 * 2.0;
      basic = 4.0 / testinput;
      diff = basic - (int)basic;
      if (diff < 0.002) {
			output += to_string((int)basic);
			output += ".";
      } else {
         testinput = input / 7.0 * 4.0;
         basic = 4.0 / testinput;
         diff = basic - (int)basic;
         if (diff < 0.002) {
				output += to_string((int)basic);
            output += "..";
         } else {
            testinput = input / 15.0 * 4.0;
            basic = 2.0 / testinput;
            diff = basic - (int)basic;
            if (diff < 0.002) {
					output += to_string((int)basic);
               output += "...";
            } else {
					// Don't know what it could be so echo as a grace note.
					output += "q";
					output += to_string(input);
            }
         }
      }
   }

   return output;
}



//////////////////////////////
//
// Convert::timeSigToDurationInQuarters -- Convert a **kern time signature 
//   into the duration of the measure for that time signature.
//   output units are in quarter notes.
//   Example: 6/8 => 3 quarters
//   Example: 3/4 => 3 quarters
//   Example: 3/8 => 3/2 quarters
//

HumNum Convert::timeSigToDurationInQuarter(HTp token) {
	HumRegex hre;
	if (!token->isTimeSignature()) {
		return 0;
	}
	// Handle extended **recip for denominator later...
	if (!hre.search(token, "^\\*M(\\d+)/(\\d+)")) {
		return 0;
	}
	int top = hre.getMatchInt(1);
	int bot = hre.getMatchInt(2);
	HumNum output = 4;
	output /= bot;
	output *= top;
	return output;
}





//////////////////////////////
//
// Convert::replaceOccurrences -- Similar to s// regular expressions
//    operator.  This function replaces the search string in the source
//    string with the replace string.
//

void Convert::replaceOccurrences(string& source, const string& search,
		const string& replace) {
	for (int loc=0; ; loc += (int)replace.size()) {
		loc = (int)source.find(search, loc);
		if (loc == (int)string::npos) {
			break;
		}
		source.erase(loc, search.length());
		source.insert(loc, replace);
	}
}



//////////////////////////////
//
// Convert::splitString -- Splits a string into a list of strings
//   separated by the given character.  Empty strings will be generated
//   if the separator occurs at the start/end of the input string, and
//   if two or more separates are adjacent to each other.
// default value: separator = ' ';
//

vector<string> Convert::splitString(const string& data, char separator) {
	stringstream ss(data);
	string key;
	vector<string> output;
	while (getline(ss, key, separator)) {
		output.push_back(key);
	}
	if (output.size() == 0) {
		output.push_back(data);
	}
	return output;
}



//////////////////////////////
//
// Convert::repeatString -- Returns a string which repeats the given
//   pattern by the given count.
//

string Convert::repeatString(const string& pattern, int count) {
	string output;
	for (int i=0; i<count; i++) {
		output += pattern;
	}
	return output;
}


//////////////////////////////
//
// Convert::encodeXml -- Encode a string for XML printing.  Ampersands
//    get converted to &amp;, < to &lt; > to &gt;, " to &quot; and
//    ' to &apos;.
//

string Convert::encodeXml(const string& input) {
	string output;
	output.reserve(input.size()*2);
	for (int i=0; i<(int)input.size(); i++) {
		switch (input[i]) {
			case '&':  output += "&amp;";   break;
			case '<':  output += "&lt;";    break;
			case '>':  output += "&gt;";    break;
			case '"':  output += "&quot;";  break;
			case '\'': output += "&apos;";  break;
			default:   output += input[i];
		}
	}
	return output;
}



//////////////////////////////
//
// Convert::getHumNumAttributes -- Returns XML attributes for a HumNum
//   number.  First @float which gives the floating-point representation.
//   If the number has a fractional part, then also add @ratfrac with the
//   fractional representation of the non-integer portion number.
//

string Convert::getHumNumAttributes(const HumNum& num) {
	string output;
	if (num.isInteger()) {
		output += " float=\"" + to_string(num.getNumerator()) + "\"";
	} else {
		stringstream sstr;
		sstr << num.toFloat();
		output += " float=\"" + sstr.str() + "\"";
	}
	if (!num.isInteger()) {
		HumNum rem = num.getRemainder();
		output += " ratfrac=\"" + to_string(rem.getNumerator()) +
				+ "/" + to_string(rem.getDenominator()) + "\"";
	}
	return output;
}



//////////////////////////////
//
// Convert::trimWhiteSpace -- remove spaces, tabs and/or newlines
//     from the beginning and end of input string.
//

string Convert::trimWhiteSpace(const string& input) {
	string s = input;
	s.erase(s.begin(), std::find_if(s.begin(), s.end(),
			[](int c) {return !std::isspace(c);}));
	s.erase(std::find_if(s.rbegin(), s.rend(),
			[](int c) {return !std::isspace(c);}).base(), s.end());
	return s;
}



//////////////////////////////
//
// Convert::startsWith --
//

bool Convert::startsWith(const string& input, const string& searchstring) {
	return input.compare(0, searchstring.size(), searchstring) == 0;
}


/////////////////////////////
//
// Convert::contains -- Returns true if the character or string
//    is found in the string.
//

bool Convert::contains(const string& input, const string& pattern) {
	return input.find(pattern) != string::npos;
}

bool Convert::contains(const string& input, char pattern) {
	return input.find(pattern) != string::npos;
}

bool Convert::contains(string* input, const string& pattern) {
	return Convert::contains(*input, pattern);
}

bool Convert::contains(string* input, char pattern) {
	return Convert::contains(*input, pattern);
}


//////////////////////////////
//
// Convert::makeBooleanTrackList -- Given a string
//   such as "1,2,3" and a max track of 5, then
//   create a vector with contents:
//      0:false, 1:true, 2:true, 3:true, 4:false, 5:false.
//   The 0 track is not used, and the two tracks not specified
//   in the string are set to false.  Special abbreviations:
//     $ = maxtrack
//     $1 = maxtrack - 1
//     $2 = maxtrack - 2
//     etc.
//   Ranges can be given, such as 1-3 instead of 1,2,3
//

void Convert::makeBooleanTrackList(vector<bool>& spinelist,
		 const string& spinestring, int maxtrack) {
   spinelist.resize(maxtrack+1);

	if (spinestring.size() == 0) {
		fill(spinelist.begin()+1, spinelist.end(), true);
		return;
	}
	fill(spinelist.begin(), spinelist.end(), false);

   string buffer = spinestring;;
	vector<string> entries;
	string separator = "[^\\d\\$-]+";
   HumRegex hre;

	// create an initial list of values:
	hre.split(entries, buffer, separator);

	// Now process each token in the extracted list:
	int val = -1;
	int val2 = -1;
	bool range = false;
	string tbuff;
	for (int i=0; i<(int)entries.size(); i++) {

		if (hre.search(entries[i], "\\$(\\d*)")) {
			if (hre.getMatch(1).size() == 0) {
				tbuff = to_string(maxtrack);
			} else {
				val = hre.getMatchInt(1);
				tbuff = to_string(maxtrack - val);
			}
			hre.replaceDestructive(entries[i], tbuff, "\\$\\d+");
		}

		range = false;
		if (entries[i].find('-') != string::npos) {
			range = true;
			// check for second $ abbreviation at end of range:
			if (hre.search(entries[i], "\\$(\\d*)")) {
				if (hre.getMatch(1).size() == 0) {
					tbuff = to_string(maxtrack);
				} else {
					val = hre.getMatchInt(1);
					tbuff = to_string(maxtrack - val);
				}
				hre.replaceDestructive(entries[i], tbuff, "\\$\\d+");
			}
			if (entries[i].back() == '$') {
				entries[i].pop_back();
				entries[i] += to_string(maxtrack);
			}
			// extract second vlaue
			if (hre.search(entries[i], "-(\\d+)")) {
				val2 = hre.getMatchInt(1);
			} else {
				range = false;
			}
		}


		// get first value:
		if (hre.search(entries[i], "(\\d+)")) {
			val = stoi(hre.getMatch(1));
		}
		if (range) {
			int direction = 1;
			if (val > val2) {
				direction = -1;
			}
			for (int j=val; j != val2; j += direction) {
				if ((j > 0) && (j < maxtrack + 1)) {
					spinelist[j] = true;
				}
			}
			if ((val2 > 0) && (val2 < maxtrack + 1)) {
				spinelist[val2] = true;
			}
		} else {
			// not a range
			if ((val > 0) && (val < maxtrack+1)) {
				spinelist[val] = true;
			}
		}
	}
}



//////////////////////////////
//
// Convert::extractIntegerList -- Convert a list such as 1-4 into the vector 1,2,3,4.
//   $ (or %) can be used to represent the maximum value, so if the input
//   is 1-$ (or 1-%), and the maximum should be 5, then the output will be a
//   vector 1,2,3,4,5.  In addition commas can be used to generate non-consecutive
//   sequences, and adding a number after the $/% sign means to subtract that
//   value from the maximum.  So if the string is 1,$-$2 and the maximum is 5,
//   then the vector will be 1,5,4,3.  Notice that ranges can be reversed to
//   place the sequence in reverse order, such as $-1 with a maximum of 5 will
//   result in the vector 5,4,3,2,1.  This function does not expect negative
//   values.
//

std::vector<int> Convert::extractIntegerList(const std::string& input, int maximum) {
	std::vector<int> output;
	if (maximum < 0) {
		maximum = 0;
	}
	if (maximum < 1000) {
		output.reserve(maximum);
	} else {
		output.reserve(1000);
	}
	HumRegex hre;
	string buffer = input;
	hre.replaceDestructive(buffer, "", "\\s", "gs");
	int start = 0;
	string tempstr;
	vector<int> tempdata;
	while (hre.search(buffer,  start, "^([^,]+,?)")) {
		tempdata.clear();
		processSegmentEntry(tempdata, hre.getMatch(1), maximum);
		start += hre.getMatchEndIndex(1);
		output.insert(output.end(), tempdata.begin(), tempdata.end());
	}
	return output;
}



//////////////////////////////
//
// Convert::processSegmentEntry --
//   3-6 expands to 3 4 5 6
//   $   expands to maximum file number
//   $-1 expands to maximum file number minus 1, etc.
//

void Convert::processSegmentEntry(vector<int>& field,
		const string& astring, int maximum) {

	HumRegex hre;
	string buffer = astring;

	// remove any comma left at end of input astring (or anywhere else)
	hre.replaceDestructive(buffer, "", ",", "g");

	// first remove $ symbols and replace with the correct values
	removeDollarsFromString(buffer, maximum);

	if (hre.search(buffer, "^(\\d+)-(\\d+)$")) {
		int firstone = hre.getMatchInt(1);
		int lastone  = hre.getMatchInt(2);

		if ((firstone < 1) && (firstone != 0)) {
			cerr << "Error: range token: \"" << astring << "\""
				  << " contains too small a number at start: " << firstone << endl;
			cerr << "Minimum number allowed is " << 1 << endl;
			return;
		}
		if ((lastone < 1) && (lastone != 0)) {
			cerr << "Error: range token: \"" << astring << "\""
				  << " contains too small a number at end: " << lastone << endl;
			cerr << "Minimum number allowed is " << 1 << endl;
			return;
		}
		if (firstone > maximum) {
			cerr << "Error: range token: \"" << astring << "\""
				  << " contains number too large at start: " << firstone << endl;
			cerr << "Maximum number allowed is " << maximum << endl;
			return;
		}
		if (lastone > maximum) {
			cerr << "Error: range token: \"" << astring << "\""
				  << " contains number too large at end: " << lastone << endl;
			cerr << "Maximum number allowed is " << maximum << endl;
			return;
		}

		if (firstone > lastone) {
			for (int i=firstone; i>=lastone; i--) {
				field.push_back(i);
			}
		} else {
			for (int i=firstone; i<=lastone; i++) {
				field.push_back(i);
			}
		}
	} else if (hre.search(buffer, "^(\\d+)")) {
		int value = hre.getMatchInt(1);
		if ((value < 1) && (value != 0)) {
			cerr << "Error: range token: \"" << astring << "\""
				  << " contains too small a number at end: " << value << endl;
			cerr << "Minimum number allowed is " << 1 << endl;
			return;
		}
		if (value > maximum) {
			cerr << "Error: range token: \"" << astring << "\""
				  << " contains number too large at start: " << value << endl;
			cerr << "Maximum number allowed is " << maximum << endl;
			return;
		}
		field.push_back(value);
	}
}



//////////////////////////////
//
// Convert::removeDollarsFromString -- substitute $ sign for maximum file count.
//

void Convert::removeDollarsFromString(string& buffer, int maximum) {
	HumRegex hre;
	string buf2 = to_string(maximum);
	if (hre.search(buffer, "[%$]$")) {
		hre.replaceDestructive(buffer, buf2, "[$%]$");
	} else if (hre.search(buffer, "[%$](?![\\d-])")) {
		// don't know how this case could happen, however...
		hre.replaceDestructive(buffer, buf2, "[%$](?![\\d-])", "g");
	} else if (hre.search(buffer, "[%$]$0")) {
		// replace $0 with maximum (used for reverse orderings)
		hre.replaceDestructive(buffer, buf2, "[%$]0", "g");
	} else if (hre.search(buffer, "^[%$]-")) {
		// replace $ with maximum at start of string
		hre.replaceDestructive(buffer, buf2, "^[%$]", "");
	}

	while (hre.search(buffer, "[%$](\\d+)")) {
		int value2 = maximum - abs(hre.getMatchInt(1));
		buf2 = to_string(value2);
		hre.replaceDestructive(buffer, buf2, "[%$]\\d+");
	}
}




//////////////////////////////
//
// GridMeasure::GridMeasure -- Constructor.
//

GridMeasure::GridMeasure(HumGrid* owner) {
	m_owner = owner;
	m_style = MeasureStyle::Plain;
}



//////////////////////////////
//
// GridMeasure::~GridMeasure -- Deconstructor.
//

GridMeasure::~GridMeasure(void) {
	for (auto it = this->begin(); it != this->end(); it++) {
		if (*it) {
			delete *it;
			*it = NULL;
		}
	}
}



//////////////////////////////
//
// GridMeasure::appendGlobalLayout --
//

GridSlice* GridMeasure::appendGlobalLayout(const string& tok, HumNum timestamp) {
	GridSlice* gs = new GridSlice(this, timestamp, SliceType::GlobalLayouts, 1);
	gs->addToken(tok, 0, 0, 0);
	gs->setDuration(0);
	this->push_back(gs);
	return gs;
}



//////////////////////////////
//
// GridSlice::addGraceToken -- Add a grace note token at the given
//   gracenumber grace note line before the data line at the given
//   timestamp.
//

GridSlice* GridMeasure::addGraceToken(const string& tok, HumNum timestamp,
	int part, int staff, int voice, int maxstaff, int gracenumber) {
	if (gracenumber < 1) {
		cerr << "ERROR: gracenumber " << gracenumber << " has to be larger than 0" << endl;
		return NULL;
	}

	GridSlice* gs = NULL;
	// GridSlice* datatarget = NULL;
	auto iterator = this->begin();
	if (this->empty()) {
		// add a new GridSlice to an empty list or at end of list if timestamp
		// is after last entry in list.
		gs = new GridSlice(this, timestamp, SliceType::GraceNotes, maxstaff);
		gs->addToken(tok, part, staff, voice);
		this->push_back(gs);
	} else if (timestamp > this->back()->getTimestamp()) {

		// Grace note needs to be added at the end of a measure:
		auto it2 = this->end();
		it2--;
		int counter = 0;
		while (it2 != this->end()) {
			if ((*it2)->isGraceSlice()) {
				counter++;
				if (counter == gracenumber) {
					// insert grace note into this slice
					(*it2)->addToken(tok, part, staff, voice);
					return *it2;
				}
			} else if ((*it2)->isLayoutSlice()) {
				// skip over any layout paramter lines.
				it2--;
				continue;
			} else if ((*it2)->isDataSlice()) {
				// insert grace note after this note
				gs = new GridSlice(this, timestamp, SliceType::GraceNotes, maxstaff);
				gs->addToken(tok, part, staff, voice);
				it2++;
				this->insert(it2, gs);
				return gs;
			}
			it2--;
		}
		return NULL;

	} else {
		// search for existing line with same timestamp on a data slice:

		while (iterator != this->end()) {
			if (timestamp < (*iterator)->getTimestamp()) {
				cerr << "STRANGE CASE 2 IN GRIDMEASURE::ADDGRACETOKEN" << endl;
				cerr << "\tGRACE TIMESTAMP: " << timestamp << endl;
				cerr << "\tTEST  TIMESTAMP: " << (*iterator)->getTimestamp() << endl;
				return NULL;
			}
			if ((*iterator)->isDataSlice()) {
				if ((*iterator)->getTimestamp() == timestamp) {
					// found dataslice just before graceslice(s)
					// datatarget = *iterator;
					break;
				}
			}
			iterator++;
		}

		auto it2 = iterator;
		it2--;
		int counter = 0;
		while (it2 != this->end()) {
			if ((*it2)->isGraceSlice()) {
				counter++;
				if (counter == gracenumber) {
					// insert grace note into this slice
					(*it2)->addToken(tok, part, staff, voice);
					return *it2;
				}
			} else if ((*it2)->isLayoutSlice()) {
				// skip over any layout paramter lines.
				it2--;
				continue;
			} else if ((*it2)->isDataSlice()) {
				// insert grace note after this note
				gs = new GridSlice(this, timestamp, SliceType::GraceNotes, maxstaff);
				gs->addToken(tok, part, staff, voice);
				it2++;
				this->insert(it2, gs);
				return gs;
			}
			it2--;
		}

		// grace note should be added at start of measure
		gs = new GridSlice(this, timestamp, SliceType::GraceNotes, maxstaff);
		gs->addToken(tok, part, staff, voice);
		this->insert(this->begin(), gs);

	}

	return NULL;
}



//////////////////////////////
//
// GridMeasure::addDataToken -- Add a data token in the data slice at the given
//    timestamp (or create a new data slice at that timestamp), placing the
//    token at the specified part, staff, and voice index.
//

GridSlice* GridMeasure::addDataToken(const string& tok, HumNum timestamp,
		int part, int staff, int voice, int maxstaff) {
	GridSlice* gs = NULL;
	if (this->empty() || (this->back()->getTimestamp() < timestamp)) {
		// add a new GridSlice to an empty list or at end of list if timestamp
		// is after last entry in list.
		gs = new GridSlice(this, timestamp, SliceType::Notes, maxstaff);
		gs->addToken(tok, part, staff, voice);
		this->push_back(gs);
	} else {
		// search for existing line with same timestamp and the same slice type
		GridSlice* target = NULL;
		auto iterator = this->begin();
		while (iterator != this->end()) {
			if ((timestamp == (*iterator)->getTimestamp()) && ((*iterator)->isGraceSlice())) {
				iterator++;
				continue;
			}
			if (!(*iterator)->isDataSlice()) {
				iterator++;
				continue;
			} else if ((*iterator)->getTimestamp() == timestamp) {
				target = *iterator;
				target->addToken(tok, part, staff, voice);
				gs = target;
				break;
			} else if ((*iterator)->getTimestamp() > timestamp) {
				gs = new GridSlice(this, timestamp, SliceType::Notes, maxstaff);
				gs->addToken(tok, part, staff, voice);
				this->insert(iterator, gs);
				break;
			}
			iterator++;
		}

		if (iterator == this->end()) {
			// Couldn't find a place for the lef, so place at end of measure.
			gs = new GridSlice(this, timestamp, SliceType::Notes, maxstaff);
			gs->addToken(tok, part, staff, voice);
			this->insert(iterator, gs);
		}
	}

	return gs;
}



//////////////////////////////
//
// GridMeasure::addTempoToken -- Add a tempo token in the data slice at
//    the given timestamp (or create a new tempo slice at that timestamp), placing the
//    token at the specified part, staff, and voice index.
//

GridSlice* GridMeasure::addTempoToken(const string& tok, HumNum timestamp,
		int part, int staff, int voice, int maxstaff) {
	GridSlice* gs = NULL;
	if (this->empty() || (this->back()->getTimestamp() < timestamp)) {
		// add a new GridSlice to an empty list or at end of list if timestamp
		// is after last entry in list.
		gs = new GridSlice(this, timestamp, SliceType::Tempos, maxstaff);
		gs->addToken(tok, part, staff, voice);
		this->push_back(gs);
	} else {
		// search for existing line with same timestamp and the same slice type
		GridSlice* target = NULL;
		auto iterator = this->begin();
		while (iterator != this->end()) {
			if (((*iterator)->getTimestamp() == timestamp) && (*iterator)->isTempoSlice()) {
				target = *iterator;
				target->addToken(tok, part, staff, voice);
				break;
			} else if (((*iterator)->getTimestamp() == timestamp) && (*iterator)->isDataSlice()) {
				// found the correct timestamp, but no clef slice at the timestamp
				// so add the clef slice before the data slice (eventually keeping
				// track of the order in which the other non-data slices should be placed).
				gs = new GridSlice(this, timestamp, SliceType::Tempos, maxstaff);
				gs->addToken(tok, part, staff, voice);
				this->insert(iterator, gs);
				break;
			} else if ((*iterator)->getTimestamp() > timestamp) {
				gs = new GridSlice(this, timestamp, SliceType::Tempos, maxstaff);
				gs->addToken(tok, part, staff, voice);
				this->insert(iterator, gs);
				break;
			}
			iterator++;
		}

		if (iterator == this->end()) {
			// Couldn't find a place for the key signature, so place at end of measure.
			gs = new GridSlice(this, timestamp, SliceType::Tempos, maxstaff);
			gs->addToken(tok, part, staff, voice);
			this->insert(iterator, gs);
		}

	}
	return gs;
}



//////////////////////////////
//
// GridMeasure::addTempoToken --
//

GridSlice* GridMeasure::addTempoToken(GridSlice* slice, int partindex,
		const string& tempo) {
	auto iter = this->rbegin();
	if (iter == this->rend()) {
		// something strange happened: expecting at least one item in measure.
		return slice;
	}

	auto previous = iter;
	previous++;
	while (previous != this->rend()) {
		if ((*previous)->isLayoutSlice()) {
			GridPart* gp = (*previous)->at(partindex);
			GridStaff* gs = gp->at(0);
			GridVoice* gv;
			if (gs->size() == 0) {
				gv = new GridVoice;
				gs->push_back(gv);
			}
			gv = gs->at(0);
			if (gv) {
				if (gv->getToken() == NULL) {
					// create a token with text
					HTp newtoken = new HumdrumToken(tempo);
					gv->setToken(newtoken);
					return slice;
				} else if (*gv->getToken() == "*") {
					// replace token with text
					HTp newtoken = new HumdrumToken(tempo);
					gv->setToken(newtoken);
					return slice;
				}
			} else {
				previous++;
				continue;
			}
		} else {
			break;
		}
		previous++;
	}

	auto insertpoint = previous.base();
	GridSlice* newslice = new GridSlice(this, (*iter)->getTimestamp(), SliceType::Layouts);
	newslice->initializeBySlice(*iter);
	this->insert(insertpoint, newslice);
	HTp newtoken = new HumdrumToken(tempo);
	if (newslice->at(partindex)->at(0)->size() == 0) {
		GridVoice* v = new GridVoice;
		newslice->at(partindex)->at(0)->push_back(v);
	}
	newslice->at(partindex)->at(0)->at(0)->setToken(newtoken);
	return newslice;
}



//////////////////////////////
//
// GridMeasure::addTimeSigToken -- Add a time signature token in the data slice at
//    the given timestamp (or create a new timesig slice at that timestamp), placing the
//    token at the specified part, staff, and voice index.
//

GridSlice* GridMeasure::addTimeSigToken(const string& tok, HumNum timestamp,
		int part, int staff, int voice, int maxstaff) {
	GridSlice* gs = NULL;
	if (this->empty() || (this->back()->getTimestamp() < timestamp)) {
		// add a new GridSlice to an empty list or at end of list if timestamp
		// is after last entry in list.
		gs = new GridSlice(this, timestamp, SliceType::TimeSigs, maxstaff);
		gs->addToken(tok, part, staff, voice);
		this->push_back(gs);
	} else {
		// search for existing line with same timestamp and the same slice type
		GridSlice* target = NULL;
		auto iterator = this->begin();
		while (iterator != this->end()) {
			if (((*iterator)->getTimestamp() == timestamp) && (*iterator)->isTimeSigSlice()) {
				target = *iterator;
				target->addToken(tok, part, staff, voice);
				break;
			} else if (((*iterator)->getTimestamp() == timestamp) && (*iterator)->isDataSlice()) {
				// found the correct timestamp, but no clef slice at the timestamp
				// so add the clef slice before the data slice (eventually keeping
				// track of the order in which the other non-data slices should be placed).
				gs = new GridSlice(this, timestamp, SliceType::TimeSigs, maxstaff);
				gs->addToken(tok, part, staff, voice);
				this->insert(iterator, gs);
				break;
			} else if ((*iterator)->getTimestamp() > timestamp) {
				gs = new GridSlice(this, timestamp, SliceType::TimeSigs, maxstaff);
				gs->addToken(tok, part, staff, voice);
				this->insert(iterator, gs);
				break;
			}
			iterator++;
		}

		if (iterator == this->end()) {
			// Couldn't find a place for the key signature, so place at end of measure.
			gs = new GridSlice(this, timestamp, SliceType::TimeSigs, maxstaff);
			gs->addToken(tok, part, staff, voice);
			this->insert(iterator, gs);
		}

	}
	return gs;
}



//////////////////////////////
//
// GridMeasure::addMeterSigToken -- Add a meter signature token in the data slice at
//    the given timestamp (or create a new timesig slice at that timestamp), placing the
//    token at the specified part, staff, and voice index.
//
//    To do:
//      The meter signtature should occur immediately after a time signature line.
//

GridSlice* GridMeasure::addMeterSigToken(const string& tok, HumNum timestamp,
		int part, int staff, int voice, int maxstaff) {
	GridSlice* gs = NULL;
	if (this->empty() || (this->back()->getTimestamp() < timestamp)) {
		// add a new GridSlice to an empty list or at end of list if timestamp
		// is after last entry in list.
		gs = new GridSlice(this, timestamp, SliceType::MeterSigs, maxstaff);
		gs->addToken(tok, part, staff, voice);
		this->push_back(gs);
	} else {
		// search for existing line with same timestamp and the same slice type
		GridSlice* target = NULL;
		auto iterator = this->begin();
		while (iterator != this->end()) {
			if (((*iterator)->getTimestamp() == timestamp) && (*iterator)->isMeterSigSlice()) {
				target = *iterator;
				target->addToken(tok, part, staff, voice);
				break;
			} else if (((*iterator)->getTimestamp() == timestamp) && (*iterator)->isDataSlice()) {
				// found the correct timestamp, but no clef slice at the timestamp
				// so add the clef slice before the data slice (eventually keeping
				// track of the order in which the other non-data slices should be placed).
				gs = new GridSlice(this, timestamp, SliceType::MeterSigs, maxstaff);
				gs->addToken(tok, part, staff, voice);
				this->insert(iterator, gs);
				break;
			} else if ((*iterator)->getTimestamp() > timestamp) {
				gs = new GridSlice(this, timestamp, SliceType::MeterSigs, maxstaff);
				gs->addToken(tok, part, staff, voice);
				this->insert(iterator, gs);
				break;
			}
			iterator++;
		}

		if (iterator == this->end()) {
			// Couldn't find a place for the key signature, so place at end of measure.
			gs = new GridSlice(this, timestamp, SliceType::MeterSigs, maxstaff);
			gs->addToken(tok, part, staff, voice);
			this->insert(iterator, gs);
		}

	}
	return gs;
}


//////////////////////////////
//
// GridMeasure::addKeySigToken -- Add a key signature  token in a key sig slice at
//    the given timestamp (or create a new keysig slice at that timestamp), placing the
//    token at the specified part, staff, and voice index.
//

GridSlice* GridMeasure::addKeySigToken(const string& tok, HumNum timestamp,
		int part, int staff, int voice, int maxstaff) {
	GridSlice* gs = NULL;
	if (this->empty() || (this->back()->getTimestamp() < timestamp)) {
		// add a new GridSlice to an empty list or at end of list if timestamp
		// is after last entry in list.
		gs = new GridSlice(this, timestamp, SliceType::KeySigs, maxstaff);
		gs->addToken(tok, part, staff, voice);
		this->push_back(gs);
	} else {
		// search for existing line with same timestamp and the same slice type
		GridSlice* target = NULL;
		auto iterator = this->begin();
		while (iterator != this->end()) {
			if (((*iterator)->getTimestamp() == timestamp) && (*iterator)->isKeySigSlice()) {
				target = *iterator;
				target->addToken(tok, part, staff, voice);
				break;
			} else if (((*iterator)->getTimestamp() == timestamp) && (*iterator)->isDataSlice()) {
				// found the correct timestamp, but no clef slice at the timestamp
				// so add the clef slice before the data slice (eventually keeping
				// track of the order in which the other non-data slices should be placed).
				gs = new GridSlice(this, timestamp, SliceType::KeySigs, maxstaff);
				gs->addToken(tok, part, staff, voice);
				this->insert(iterator, gs);
				break;
			} else if ((*iterator)->getTimestamp() > timestamp) {
				gs = new GridSlice(this, timestamp, SliceType::KeySigs, maxstaff);
				gs->addToken(tok, part, staff, voice);
				this->insert(iterator, gs);
				break;
			}
			iterator++;
		}

		if (iterator == this->end()) {
			// Couldn't find a place for the key signature, so place at end of measure.
			gs = new GridSlice(this, timestamp, SliceType::KeySigs, maxstaff);
			gs->addToken(tok, part, staff, voice);
			this->insert(iterator, gs);
		}

	}
	return gs;
}




//////////////////////////////
//
// GridMeasure::addLabelToken -- Add an instrument label token in a label slice at
//    the given timestamp (or create a new label slice at that timestamp), placing the
//    token at the specified part, staff, and voice index.
//

GridSlice* GridMeasure::addLabelToken(const string& tok, HumNum timestamp,
		int part, int staff, int voice, int maxpart, int maxstaff) {
	GridSlice* gs = NULL;
	if (this->empty() || (this->back()->getTimestamp() < timestamp)) {
		// add a new GridSlice to an empty list or at end of list if timestamp
		// is after last entry in list.
		gs = new GridSlice(this, timestamp, SliceType::Labels, maxpart);
		gs->addToken(tok, part, maxstaff-1, voice);
		this->push_back(gs);
	} else {
		// search for existing line with same timestamp and the same slice type
		GridSlice* target = NULL;
		auto iterator = this->begin();
		while (iterator != this->end()) {
			if (((*iterator)->getTimestamp() == timestamp) && (*iterator)->isLabelSlice()) {
				target = *iterator;
				target->addToken(tok, part, maxstaff-1, voice);
				break;
			}
			iterator++;
		}
		if (iterator == this->end()) {
			// Couldn't find a place for the label abbreviation line, so place at end of measure.
			gs = new GridSlice(this, timestamp, SliceType::Labels, maxpart);
			gs->addToken(tok, part, maxstaff-1, voice);
			this->insert(this->begin(), gs);
		}
	}
	return gs;
}



//////////////////////////////
//
// GridMeasure::addLabelAbbrToken -- Add an instrument label token in a label slice at
//    the given timestamp (or create a new label slice at that timestamp), placing the
//    token at the specified part, staff, and voice index.
//

GridSlice* GridMeasure::addLabelAbbrToken(const string& tok, HumNum timestamp,
		int part, int staff, int voice, int maxpart, int maxstaff) {
	GridSlice* gs = NULL;
	if (this->empty() || (this->back()->getTimestamp() < timestamp)) {
		// add a new GridSlice to an empty list or at end of list if timestamp
		// is after last entry in list.
		gs = new GridSlice(this, timestamp, SliceType::LabelAbbrs, maxpart);
		gs->addToken(tok, part, maxstaff-1, voice);
		this->push_back(gs);
	} else {
		// search for existing line with same timestamp and the same slice type
		GridSlice* target = NULL;
		auto iterator = this->begin();
		while (iterator != this->end()) {
			if (((*iterator)->getTimestamp() == timestamp) && (*iterator)->isLabelAbbrSlice()) {
				target = *iterator;
				target->addToken(tok, part, maxstaff-1, voice);
				break;
			}
			iterator++;
		}
		if (iterator == this->end()) {
			// Couldn't find a place for the label abbreviation line, so place at end of measure.
			gs = new GridSlice(this, timestamp, SliceType::LabelAbbrs, maxpart);
			gs->addToken(tok, part, maxstaff-1, voice);
			this->insert(this->begin(), gs);
		}
	}
	return gs;
}



//////////////////////////////
//
// GridMeasure::addTransposeToken -- Add a transposition token in the data slice at
//    the given timestamp (or create a new transposition slice at that timestamp), placing
//    the token at the specified part, staff, and voice index.
//
//    Note: should placed after clef if present and no other transpose slice at
//    same time.
//

GridSlice* GridMeasure::addTransposeToken(const string& tok, HumNum timestamp,
		int part, int staff, int voice, int maxstaff) {
	GridSlice* gs = NULL;
	if (this->empty() || (this->back()->getTimestamp() < timestamp)) {
		// add a new GridSlice to an empty list or at end of list if timestamp
		// is after last entry in list.
		gs = new GridSlice(this, timestamp, SliceType::Transpositions, maxstaff);
		gs->addToken(tok, part, staff, voice);
		this->push_back(gs);
	} else {
		// search for existing line with same timestamp and the same slice type
		GridSlice* target = NULL;
		auto iterator = this->begin();
		while (iterator != this->end()) {
			if (((*iterator)->getTimestamp() == timestamp) && (*iterator)->isTransposeSlice()) {
				target = *iterator;
				target->addToken(tok, part, staff, voice);
				break;
			} else if (((*iterator)->getTimestamp() == timestamp) && (*iterator)->isDataSlice()) {
				// found the correct timestamp, but no clef slice at the timestamp
				// so add the clef slice before the data slice (eventually keeping
				// track of the order in which the other non-data slices should be placed).
				gs = new GridSlice(this, timestamp, SliceType::Transpositions, maxstaff);
				gs->addToken(tok, part, staff, voice);
				this->insert(iterator, gs);
				break;
			} else if ((*iterator)->getTimestamp() > timestamp) {
				gs = new GridSlice(this, timestamp, SliceType::Transpositions, maxstaff);
				gs->addToken(tok, part, staff, voice);
				this->insert(iterator, gs);
				break;
			}
			iterator++;
		}

		if (iterator == this->end()) {
			// Couldn't find a place for the key signature, so place at end of measure.
			gs = new GridSlice(this, timestamp, SliceType::Transpositions, maxstaff);
			gs->addToken(tok, part, staff, voice);
			this->insert(iterator, gs);
		}

	}
	return gs;
}



//////////////////////////////
//
// GridMeasure::addClefToken -- Add a clef token in the data slice at the given
//    timestamp (or create a new clef slice at that timestamp), placing the
//    token at the specified part, staff, and voice index.
//

GridSlice* GridMeasure::addClefToken(const string& tok, HumNum timestamp,
		int part, int staff, int voice, int maxstaff) {
	GridSlice* gs = NULL;
	if (this->empty() || (this->back()->getTimestamp() < timestamp)) {
		// add a new GridSlice to an empty list or at end of list if timestamp
		// is after last entry in list.
		gs = new GridSlice(this, timestamp, SliceType::Clefs, maxstaff);
		gs->addToken(tok, part, staff, voice);
		this->push_back(gs);
	} else {
		// search for existing line with same timestamp and the same slice type
		GridSlice* target = NULL;
		auto iterator = this->begin();
		while (iterator != this->end()) {
			if (((*iterator)->getTimestamp() == timestamp) && (*iterator)->isClefSlice()) {
				target = *iterator;
				target->addToken(tok, part, staff, voice);
				break;
			} else if (((*iterator)->getTimestamp() == timestamp) && (*iterator)->isDataSlice()) {
				// found the correct timestamp, but no clef slice at the timestamp
				// so add the clef slice before the data slice (eventually keeping
				// track of the order in which the other non-data slices should be placed).
				gs = new GridSlice(this, timestamp, SliceType::Clefs, maxstaff);
				gs->addToken(tok, part, staff, voice);
				this->insert(iterator, gs);
				break;
			} else if ((*iterator)->getTimestamp() > timestamp) {
				gs = new GridSlice(this, timestamp, SliceType::Clefs, maxstaff);
				gs->addToken(tok, part, staff, voice);
				this->insert(iterator, gs);
				break;
			}
			iterator++;
		}

		if (iterator == this->end()) {
			// Couldn't find a place for the key signature, so place at end of measure.
			gs = new GridSlice(this, timestamp, SliceType::Clefs, maxstaff);
			gs->addToken(tok, part, staff, voice);
			this->insert(iterator, gs);
		}
	}

	return gs;
}



//////////////////////////////
//
// GridMeasure::addFiguredBass --
//
GridSlice* GridMeasure::addFiguredBass(HTp token, HumNum timestamp, int part, int maxstaff) {
	GridSlice* gs = NULL;
	bool processed = false;

	if (this->empty() || (this->back()->getTimestamp() < timestamp)) {
		// add a new GridSlice to an empty list or at end of list if timestamp
		// is after last entry in list.
		gs = new GridSlice(this, timestamp, SliceType::Notes, maxstaff);
		int staff = 0;
		int voice = 0;
		string null = ".";
		gs->addToken(null, part, staff, voice);
		gs->at(part)->setFiguredBass(token);
		this->push_back(gs);
		processed = true;
	} else {
		// search for existing line with same timestamp and the same slice type
		GridSlice* target = NULL;
		auto iterator = this->begin();
		while (iterator != this->end()) {
			if (((*iterator)->getTimestamp() == timestamp) && (*iterator)->isDataSlice()) {
				target = *iterator;
				target->at(part)->setFiguredBass(token);
				processed = true;
				break;
			} else if ((*iterator)->getTimestamp() > timestamp) {
				// Need to add figured bass data where there are note notes,
				// so add an emtpy data line and add the figure bass contnet.
				gs = new GridSlice(this, timestamp, SliceType::Notes, maxstaff);
				int staff = 0;
				int voice = 0;
				string null = ".";
				gs->addToken(null, part, staff, voice);
				gs->at(part)->setFiguredBass(token);
				this->insert(iterator, gs);
				processed = true;
				break;
			}
			iterator++;
		}
	}


	if ((!processed) && (!this->empty()) && (this->back()->getTimestamp() == timestamp)) {
		// This case is related to putting figures on the first note in a measure
		// but the note is not yet there, but the key signature/meter/clef etc. have already
		// been added.
		gs = new GridSlice(this, timestamp, SliceType::Notes, maxstaff);
		int staff = 0;
		int voice = 0;
		string null = ".";
		gs->addToken(null, part, staff, voice);
		gs->at(part)->setFiguredBass(token);
		this->push_back(gs);
		processed = true;
	}

	if (!processed) {
		cerr << "Error: could not insert figured bass: " << token << endl;
	} else {
		HumGrid* hg = getOwner();
		if (hg) {
			hg->setFiguredBassPresent(part);
		}
	}

	return gs;
}



//////////////////////////////
//
// GridMeasure::addFiguredBass --
//
GridSlice* GridMeasure::addFiguredBass(const string& tok, HumNum timestamp, int part, int maxstaff) {
	GridSlice* gs = NULL;
	bool processed = false;

	if (this->empty() || (this->back()->getTimestamp() < timestamp)) {

		// add a new GridSlice to an empty list or at end of list if timestamp
		// is after last entry in list.
		gs = new GridSlice(this, timestamp, SliceType::Notes, maxstaff);
		int staff = 0;
		int voice = 0;
		string null = ".";
		gs->addToken(null, part, staff, voice);
		gs->at(part)->setFiguredBass(tok);
		this->push_back(gs);
		processed = true;
	} else {
		// search for existing line with same timestamp and the same slice type
		GridSlice* target = NULL;
		auto iterator = this->begin();
		while (iterator != this->end()) {
			if (((*iterator)->getTimestamp() == timestamp) && (*iterator)->isDataSlice()) {
				target = *iterator;
				target->at(part)->setFiguredBass(tok);
				processed = true;
				break;
			} else if ((*iterator)->getTimestamp() > timestamp) {
				// Need to add figured bass data where there are note notes,
				// so add an emtpy data line and add the figure bass contnet.
				gs = new GridSlice(this, timestamp, SliceType::Notes, maxstaff);
				int staff = 0;
				int voice = 0;
				string null = ".";
				gs->addToken(null, part, staff, voice);
				gs->at(part)->setFiguredBass(tok);
				this->insert(iterator, gs);
				processed = true;
				break;
			}
			iterator++;
		}
	}

	if ((!processed) && (!this->empty()) && (this->back()->getTimestamp() == timestamp)) {
		// This case is related to putting figures on the first note in a measure
		// but the note is not yet there, but the key signature/meter/clef etc. have already
		// been added.
		gs = new GridSlice(this, timestamp, SliceType::Notes, maxstaff);
		int staff = 0;
		int voice = 0;
		string null = ".";
		gs->addToken(null, part, staff, voice);
		gs->at(part)->setFiguredBass(tok);
		this->push_back(gs);
		processed = true;
	}

	if (!processed) {
		cerr << "Error: could not inser figured bass: " << tok << endl;
	} else {
		HumGrid* hg = getOwner();
		if (hg) {
			hg->setFiguredBassPresent(part);
		}
	}

	return gs;
}



//////////////////////////////
//
// GridMeasure::addGlobalComment -- Add a global comment at the given
//    timestamp (before any data line at the same timestamp).  Suppress
//    adding the comment if it matches to another global comment at the
//    same timestamp with the same text.
//

GridSlice* GridMeasure::addGlobalComment(const string& tok, HumNum timestamp) {
	GridSlice* gs = NULL;
	if (this->empty() || (this->back()->getTimestamp() < timestamp)) {
		// add a new GridSlice to an empty list or at end of list if timestamp
		// is after last entry in list.
		gs = new GridSlice(this, timestamp, SliceType::GlobalComments, 1);
		gs->addToken(tok, 0, 0, 0);
		this->push_back(gs);
	} else {
		// search for existing data line (or any other type)  with same timestamp
		auto iterator = this->begin();
		while (iterator != this->end()) {
			// does it need to be before data slice or any slice?
			// if (((*iterator)->getTimestamp() == timestamp) && (*iterator)->isDataSlice()) {
			if ((*iterator)->getTimestamp() == timestamp) {
				// found the correct timestamp on a data slice, so add the global comment
				// before the data slice.  But don't add if the previous
				// grid slice is a global comment with the same text.
				if ((iterator != this->end()) && (*iterator)->isGlobalComment()) {
					if (tok == (*iterator)->at(0)->at(0)->at(0)->getToken()->getText()) {
						// do not insert duplicate global comment
						gs = *iterator;
						break;
					}
				}
				gs = new GridSlice(this, timestamp, SliceType::GlobalComments, 1);
				gs->addToken(tok, 0, 0, 0);
				this->insert(iterator, gs);
				break;
			} else if ((*iterator)->getTimestamp() > timestamp) {
				gs = new GridSlice(this, timestamp, SliceType::GlobalComments, 1);
				gs->addToken(tok, 0, 0, 0);
				this->insert(iterator, gs);
				break;
			}
			iterator++;
		}
	}
	return gs;
}



//////////////////////////////
//
// GridMeasure::transferTokens --
//    default value: startbarnum = 0
//

bool GridMeasure::transferTokens(HumdrumFile& outfile, bool recip,
		bool addbar, int startbarnum) {

	// If the last data slice duration is zero, then calculate
	// the true duration from the duration of the measure.
	if (this->size() > 0) {
		GridSlice* slice = back();
		if (slice->isMeasureSlice() && (this->size() >= 2)) {
			auto ending = this->end();
			--ending;
			--ending;
			while ((ending != this->begin()) && (!(*ending)->isDataSlice())) {
				--ending;
			}
			slice = *ending;
		} else {
			slice = NULL;
		}
		if ((slice != NULL) && slice->isDataSlice()
				&& (slice->getDuration() == 0)) {
			HumNum mts  = getTimestamp();
			HumNum mdur = getDuration();
			HumNum sts  = slice->getTimestamp();
			HumNum slicedur = (mts + mdur) - sts;
			slice->setDuration(slicedur);
		}
	}

	bool founddata = false;
	bool addedbar = false;

	for (auto it : *this) {
		if (it->isInvalidSlice()) {
			// ignore slices to be removed from output (used for
			// removing redundant clef slices).
			continue;
		}
		if (it->isDataSlice()) {
			founddata = true;
		}
		if (it->isLayoutSlice()) {
			// didn't actually find data, but barline should
			// not cross this line.
			founddata = true;
		}
		if (it->isManipulatorSlice()) {
			// didn't acutally find data, but the barline should
			// be placed before any manipulator (a spine split), since
			// that is more a property of the data than of the header
			// interpretations.
			founddata = true;
		}
		if (founddata && addbar && !addedbar) {
			if (getDuration() == 0) {
				// do nothing
			} else {
				if (startbarnum) {
					appendInitialBarline(outfile, startbarnum);
				} else {
					appendInitialBarline(outfile);
				}
				addedbar = true;
			}
		}
		it->transferTokens(outfile, recip);
	}
	return true;
}



//////////////////////////////
//
// GridMeasure::appendInitialBarline -- The barline will be
//    duplicated to all spines later.
//

void GridMeasure::appendInitialBarline(HumdrumFile& infile, int startbarline) {
	(void)startbarline; // suppress compiler warnings about variable not being used
	if (infile.getLineCount() == 0) {
		// strange case which should never happen.
		return;
	}
	if (getMeasureNumber() > 0) {
		startbarline = getMeasureNumber();
	}
	int fieldcount = infile.back()->getFieldCount();
	HLp line = new HumdrumLine;
	string tstring = "=";
//	if (startbarline) {
//		tstring += to_string(startbarline);
//	} else {
//		tstring += "1";
//	}
	// probably best not to start with an invisible barline since
	// a plain barline would not be shown before the first measure anyway.
	// tstring += "-";
	HTp token;
	for (int i=0; i<fieldcount; i++) {
		token = new HumdrumToken(tstring);
		line->appendToken(token);
	}
	infile.push_back(line);
}



//////////////////////////////
//
// GridMeasure::getOwner --
//

HumGrid* GridMeasure::getOwner(void) {
	return m_owner;
}



//////////////////////////////
//
// GridMeasure::setOwner --
//

void GridMeasure::setOwner(HumGrid* owner) {
	m_owner = owner;
}



//////////////////////////////
//
// GridMeasure::setDuration --
//

void GridMeasure::setDuration(HumNum duration) {
	m_duration = duration;
}



//////////////////////////////
//
// GridMeasure::getDuration --
//

HumNum GridMeasure::getDuration(void) {
	return m_duration;
}



//////////////////////////////
//
// GridMeasure::getTimestamp --
//

HumNum GridMeasure::getTimestamp(void) {
	return m_timestamp;
}



//////////////////////////////
//
// GridMeasure::setTimestamp --
//

void GridMeasure::setTimestamp(HumNum timestamp) {
	m_timestamp = timestamp;
}



//////////////////////////////
//
// GridMeasure::getTimeSigDur --
//

HumNum GridMeasure::getTimeSigDur(void) {
	return m_timesigdur;
}



//////////////////////////////
//
// GridMeasure::setTimeSigDur --
//

void GridMeasure::setTimeSigDur(HumNum duration) {
	m_timesigdur = duration;
}



//////////////////////////////
//
// GridMeasure::addLayoutParameter -- Add a layout line at a particular timestamp
//

void GridMeasure::addLayoutParameter(HumNum timestamp, int partindex,
		int staffindex, const string& locomment) {
	auto iter = this->rbegin();
	if (iter == this->rend()) {
		// no items in measure yet, so add
		cerr << "DEAL WITH THIS LAYOUT COMMAND" << endl;
		return;
	}
	GridPart* part;
	GridStaff* staff;
	GridVoice* voice;

	auto previous = iter;
	previous++;
	while (previous != this->rend()) {
		if ((*previous)->isLayoutSlice()) {
			part = (*previous)->at(partindex);
			staff = part->at(0);
			if (staff->size() == 0) {
				GridVoice* v = new GridVoice;
				staff->push_back(v);
			}
			voice = staff->at(0);
			if (voice) {
				if (voice->getToken() == NULL) {
					// create a token with text
					HTp newtoken = new HumdrumToken(locomment);
					voice->setToken(newtoken);
					return;
				} else if (*voice->getToken() == "!") {
					// replace token with text
					HTp newtoken = new HumdrumToken(locomment);
					voice->setToken(newtoken);
					return;
				}
			} else {
				previous++;
				continue;
			}
		} else {
			break;
		}
		previous++;
	}

	auto insertpoint = previous.base();
	GridSlice* newslice = new GridSlice(this, (*iter)->getTimestamp(), SliceType::Layouts);
	newslice->initializeBySlice(*iter);
	this->insert(insertpoint, newslice);
	HTp newtoken = new HumdrumToken(locomment);
	if (newslice->at(partindex)->at(0)->size() == 0) {
		GridVoice* v = new GridVoice;
		newslice->at(partindex)->at(0)->push_back(v);
	}
	newslice->at(partindex)->at(0)->at(0)->setToken(newtoken);
}



//////////////////////////////
//
// GridMeasure::addLayoutParameter --
//

void GridMeasure::addLayoutParameter(GridSlice* slice, int partindex,
		const string& locomment) {
	auto iter = this->rbegin();
	if (iter == this->rend()) {
		// something strange happened: expecting at least one item in measure.
		return;
	}
	GridPart* part;
	GridStaff* staff;
	GridVoice* voice;

	auto previous = iter;
	previous++;
	while (previous != this->rend()) {
		if ((*previous)->isLayoutSlice()) {
			part = (*previous)->at(partindex);
			staff = part->at(0);
			if (staff->size() == 0) {
				GridVoice* v = new GridVoice;
				staff->push_back(v);
			}
			voice = staff->at(0);
			if (voice) {
				if (voice->getToken() == NULL) {
					// create a token with text
					HTp newtoken = new HumdrumToken(locomment);
					voice->setToken(newtoken);
					return;
				} else if (*voice->getToken() == "!") {
					// replace token with text
					HTp newtoken = new HumdrumToken(locomment);
					voice->setToken(newtoken);
					return;
				}
			} else {
				previous++;
				continue;
			}
		} else {
			break;
		}
		previous++;
	}

	auto insertpoint = previous.base();
	GridSlice* newslice = new GridSlice(this, (*iter)->getTimestamp(), SliceType::Layouts);
	newslice->initializeBySlice(*iter);
	this->insert(insertpoint, newslice);
	HTp newtoken = new HumdrumToken(locomment);
	if (newslice->at(partindex)->at(0)->size() == 0) {
		GridVoice* v = new GridVoice;
		newslice->at(partindex)->at(0)->push_back(v);
	}
	newslice->at(partindex)->at(0)->at(0)->setToken(newtoken);
}



//////////////////////////////
//
// GridMeasure::addInterpretationAfter -- Add an interpretation line after the
//      given slice, (which is probably a note slice).
//

void GridMeasure::addInterpretationAfter(GridSlice* slice, int partindex,
		int staffindex, int voiceindex, const string& interpretation, HumNum timestamp) {
	HumNum targettime = slice->getTimestamp();

	auto iter = this->rbegin();
	if (iter == this->rend()) {
		// something strange happened: expecting at least one item in measure.
		return;
	}
	// GridPart* part;
	// GridStaff* staff;
	// GridVoice* voice;

	auto previous = iter;
	// auto last = previous;
	previous++;
	HumNum ptime;
	HumNum newtargettime;
	if (previous != this->rend()) {
		ptime = (*previous)->getTimestamp();
		newtargettime = ptime;
	} else {
		ptime = targettime;
		newtargettime = targettime;
	}

	if (ptime == targettime) {

		auto nextone = previous;
		nextone--;
		if ((nextone != this->rend()) && ((*nextone)->isInterpretationSlice())) {
			GridPart* gp = (*nextone)->at(partindex);
			GridStaff* gs = gp->at(staffindex);
			// only place in voice=0 for now:
			if (gs->size() == 0) {
				GridVoice* gv = new GridVoice;
				gs->push_back(gv);
			}
			HTp token = gs->at(0)->getToken();
			if (token == NULL) {
				gs->at(0)->setToken(interpretation);
				return;
			} else if (*token == "*") {
				gs->at(0)->setToken(interpretation);
				return;
			}
		}
	}

	if (ptime <= targettime) {
		// Insert slice at end of measure.
		GridSlice* newslice = new GridSlice(this, timestamp, SliceType::_Interpretation);
		newslice->initializeBySlice(slice);
		this->push_back(newslice);
		HTp newtoken = new HumdrumToken(interpretation);
		if (newslice->at(partindex)->at(0)->size() == 0) {
			GridVoice* v = new GridVoice;
			newslice->at(partindex)->at(0)->push_back(v);
		}
		newslice->at(partindex)->at(0)->at(0)->setToken(newtoken);
		return;
	}
}



//////////////////////////////
//
// GridMeasure::addInterpretationBefore -- Add an interpretation line before the
//      given slice, (which is probably a note slice).
//

void GridMeasure::addInterpretationBefore(GridSlice* slice, int partindex, int staffindex, int voiceindex,
		const string& interpretation) {
	auto iter = this->rbegin();
	if (iter == this->rend()) {
		// something strange happened: expecting at least one item in measure.
		return;
	}

	auto previous = iter;
	previous++;

	// Try to insert new token in the current interpretation
	// line if there is a spot that is empty.  This should usually
	// work, but there could be times when the interpretation line
	// has a specific submeaning that will not match the inserted
	// interpretation.
	if ((previous != this->rend()) && (*previous)->isInterpretationSlice()) {
		GridPart* gp = (*previous)->at(partindex);
		GridStaff* gs = gp->at(0);
		GridVoice* gv = NULL;
		if (gs->empty()) {
			gv = new GridVoice;
			gs->push_back(gv);
		}
		// only allowing at index 0 voice for now:
		// And is asuumed now to be non-null.
		gv = gs->at(0);
		HTp token = gv->getToken();
		if (!token) {
			gv->setToken(interpretation);
			return;
		} else if (token->isNull()) {
			gv->setToken(interpretation);
			return;
		}
	}

	// Could not insert interpretation on interpretation line immediatly before
	// the assumed starting data line, so insert a new interpretation slice just before
	// the data slice.
	auto insertpoint = previous.base();
	GridSlice* newslice = new GridSlice(this, (*iter)->getTimestamp(), SliceType::_Interpretation);
	newslice->initializeBySlice(*iter);
	this->insert(insertpoint, newslice);
	HTp newtoken = new HumdrumToken(interpretation);
	if (newslice->at(partindex)->at(0)->size() == 0) {
		GridVoice* v = new GridVoice;
		newslice->at(partindex)->at(0)->push_back(v);
	}
	newslice->at(partindex)->at(0)->at(0)->setToken(newtoken);
}




//////////////////////////////
//
// GridMeasure::addDynamicsLayoutParameters --
//

void GridMeasure::addDynamicsLayoutParameters(GridSlice* slice, int partindex,
		const string& locomment) {
	auto iter = this->rbegin();
	if (iter == this->rend()) {
		// something strange happened: expecting at least one item in measure.
		return;
	}
	GridPart* part;

	while ((iter != this->rend()) && (*iter != slice)) {
		iter++;
	}

	if (*iter != slice) {
		// cannot find owning line.
		return;
	}

	auto previous = iter;
	previous++;
	while (previous != this->rend()) {
		if ((*previous)->isLayoutSlice()) {
			part = (*previous)->at(partindex);
			if ((part->getDynamics() == NULL) || (*part->getDynamics() == "!")) {
				HTp token = new HumdrumToken(locomment);
				part->setDynamics(token);
				return;
			} else {
				previous++;
				continue;
			}
		} else {
			break;
		}
	}

	auto insertpoint = previous.base();
	GridSlice* newslice = new GridSlice(this, (*iter)->getTimestamp(), SliceType::Layouts);
	newslice->initializeBySlice(*iter);
	this->insert(insertpoint, newslice);

	HTp newtoken = new HumdrumToken(locomment);
	newslice->at(partindex)->setDynamics(newtoken);
}



//////////////////////////////
//
// GridMeasure::addFiguredBassLayoutParameters --
//

void GridMeasure::addFiguredBassLayoutParameters(GridSlice* slice, int partindex,
		const string& locomment) {
	auto iter = this->rbegin();
	if (iter == this->rend()) {
		// something strange happened: expecting at least one item in measure.
		return;
	}
	GridPart* part;

	while ((iter != this->rend()) && (*iter != slice)) {
		iter++;
	}

	if (*iter != slice) {
		// cannot find owning line.
		return;
	}

	auto previous = iter;
	previous++;
	while (previous != this->rend()) {
		if ((*previous)->isLayoutSlice()) {
			part = (*previous)->at(partindex);
			if ((part->getFiguredBass() == NULL) || (*part->getFiguredBass() == "!")) {
				HTp token = new HumdrumToken(locomment);
				part->setFiguredBass(token);
				return;
			} else {
				previous++;
				continue;
			}
		} else {
			break;
		}
	}

	auto insertpoint = previous.base();
	GridSlice* newslice = new GridSlice(this, (*iter)->getTimestamp(), SliceType::Layouts);
	newslice->initializeBySlice(*iter);
	this->insert(insertpoint, newslice);

	HTp newtoken = new HumdrumToken(locomment);
	newslice->at(partindex)->setFiguredBass(newtoken);
}



//////////////////////////////
//
// GridMeasure::isMonophonicMeasure --  One part starts with note/rest, the others
//     with invisible rest.
//

bool GridMeasure::isMonophonicMeasure(void) {
	int inviscount = 0;
	int viscount = 0;

	for (auto slice : *this) {
		if (!slice->isDataSlice()) {
			continue;
		}
		for (int p=0; p<(int)slice->size(); p++) {
			GridPart* part = slice->at(p);
			for (int s=0; s<(int)part->size(); s++) {
				GridStaff* staff = part->at(s);
				for (int v=0; v<(int)staff->size(); v++) {
					GridVoice* voice = staff->at(v);
					HTp token = voice->getToken();
					if (!token) {
						return false;
					}
					if (token->find("yy")) {
						inviscount++;
					} else {
						viscount++;
					}
				}
				if (inviscount + viscount) {
					break;
				}
			}
			if (inviscount + viscount) {
				break;
			}
		}
		if (inviscount + viscount) {
			break;
		}
	}
	if ((viscount = 1) && (inviscount > 0)) {
		return true;
	} else {
		return false;
	}
}



//////////////////////////////
//
// GridMeasure::isSingleChordMeasure --
//

bool GridMeasure::isSingleChordMeasure(void) {

	for (auto slice : *this) {
		if (!slice->isDataSlice()) {
			continue;
		}
		for (int p=0; p<(int)slice->size(); p++) {
			GridPart* part = slice->at(p);
			for (int s=0; s<(int)part->size(); s++) {
				GridStaff* staff = part->at(s);
				for (int v=0; v<(int)staff->size(); v++) {
					GridVoice* voice = staff->at(v);
					HTp token = voice->getToken();
					if (!token) {
						return false;
					}
					if (!token->isChord()) {
						return false;
					}
				}
			}
		}
	}
	return true;

}



//////////////////////////////
//
// GridMeasure::isInvisible --  Return true if all contents is invisible.
//

bool GridMeasure::isInvisible(void) {

	for (auto slice : *this) {
		if (!slice->isDataSlice()) {
			continue;
		}
		for (int p=0; p<(int)slice->size(); p++) {
			GridPart* part = slice->at(p);
			for (int s=0; s<(int)part->size(); s++) {
				GridStaff* staff = part->at(s);
				for (int v=0; v<(int)staff->size(); v++) {
					GridVoice* voice = staff->at(v);
					HTp token = voice->getToken();
					if (!token) {
						return false;
					}
					if (token->find("yy") == string::npos) {
						return false;
					}
				}
			}
		}
	}
	return true;

}



//////////////////////////////
//
// GridMeasure::getFirstSpinedSlice --
//

GridSlice* GridMeasure::getFirstSpinedSlice(void) {
	GridSlice* output = NULL;
	for (auto tslice : *this) {
		if (!tslice->hasSpines()) {
			continue;
		}
		output = tslice;
		break;
	}
	return output;
}



//////////////////////////////
//
// GridMeasure::getLastSpinedSlice --
//

GridSlice* GridMeasure::getLastSpinedSlice(void) {
	for (auto rit = this->rbegin(); rit != this->rend(); rit++) {
		GridSlice* slice = *rit;
		if (!slice) {
			continue;
		}
		if (slice->isGlobalLayout()) {
			continue;
		}
		if (slice->isGlobalComment()) {
			continue;
		}
		if (slice->isReferenceRecord()) {
			continue;
		}
		return slice;
	}
	return NULL;
}



//////////////////////////////
//
// GridMeasure::setMeasureNumber --
//

void GridMeasure::setMeasureNumber(int value) {
	m_barnum = value;
}



//////////////////////////////
//
// GridMeasure::getMeasureNumber --
//

int GridMeasure::getMeasureNumber(void) {
	return m_barnum;
}



//////////////////////////////
//
// operator<< --
//

ostream& operator<<(ostream& output, GridMeasure* measure) {
	output << *measure;
	return output;
}

ostream& operator<<(ostream& output, GridMeasure& measure) {
	for (auto item : measure) {
		output << item << endl;
	}
	return output;
}




//////////////////////////////
//
// GridPart::GridPart -- Constructor.
//

GridPart::GridPart(void) : GridSide() {
	// do nothing;
}

//////////////////////////////
//
// GridPart::~GridPart -- Deconstructor: delete any GridStaff items
//     being stored.
//

GridPart::~GridPart(void) {
	for (int i=0; i<(int)this->size(); i++) {
		if (this->at(i)) {
			delete this->at(i);
			this->at(i) = NULL;
		}
	}
}


//////////////////////////////
//
// operator<< -- print the contents of a GridPart data structure --
//

ostream& operator<<(ostream& output, GridPart* part) {
	if (part == NULL) {
		output << "{n}";
		return output;
	}
	for (int s=0; s<(int)part->size(); s++) {
		GridStaff* staff = part->at(s);
		output << "(s" << s << ":)";
		if (staff == NULL) {
			output << "{n}";
			continue;
		}
		for (int t=0; t<(int)staff->size(); t++) {
			GridVoice* gt = staff->at(t);
			output << "(v" << t << ":)";
			if (gt == NULL) {
				output << "{n}";
				continue;
			} else {
				HTp token = gt->getToken();
				if (token == NULL) {
					output << "{n}";
				} else {
					output << " \"" << *token << "\" ";
				}
			}
		}
	}
	output << " ppp " << (GridSide*) part;
	return output;
}


ostream& operator<<(ostream& output, GridPart& part) {
	output << &part;
	return output;
}




//////////////////////////////
//
// GridSide::GridSide -- Constructor.
//

GridSide::GridSide(void) {
	// do nothing
}



//////////////////////////////
//
// GridSide::~GridSide -- Deconstructor.
//

GridSide::~GridSide(void) {

	for (int i=0; i<(int)m_verses.size(); i++) {
		if (m_verses[i]) {
			delete m_verses[i];
			m_verses[i] = NULL;
		}
	}
	m_verses.resize(0);

	if (m_dynamics) {
		delete m_dynamics;
		m_dynamics = NULL;
	}

	if (m_harmony) {
		delete m_harmony;
		m_harmony = NULL;
	}
}



//////////////////////////////
//
// GridSide::setVerse --
//

void GridSide::setVerse(int index, HTp token) {
	if (token == NULL) {
		// null tokens are written in the transfer process when responsibility
		// for deleting the pointer is given to another object (HumdrumFile class).
	}
   if (index == (int)m_verses.size()) {
		// Append to the end of the verse list.
		m_verses.push_back(token);
	} else if (index < (int)m_verses.size()) {
		// Insert in a slot which might already have a verse token
		if ((token != NULL) && (m_verses.at(index) != NULL)) {
			// don't delete a previous non-NULL token if a NULL
			// token is being stored, as it is assumed that the
			// token has been transferred to a HumdrumFile object.
			delete m_verses[index];
		}
		m_verses[index] = token;
	} else {
		// Add more than one verse spot and insert verse:
		int oldsize = (int)m_verses.size();
		int newsize = index + 1;
		m_verses.resize(newsize);
		for (int i=oldsize; i<newsize; i++) {
			m_verses.at(i) = NULL;
		}
		m_verses.at(index) = token;
	}
}


void GridSide::setVerse(int index, const string& token) {
	HTp newtoken = new HumdrumToken(token);
	setVerse(index, newtoken);
}


//////////////////////////////
//
// GridSide::getVerse --
//

HTp GridSide::getVerse(int index) {
	if (index < 0 || index >= getVerseCount()) {
		return NULL;
	}
	return m_verses[index];
}



//////////////////////////////
//
// GridSide::getVerseCount --
//

int GridSide::getVerseCount(void) {
 	return (int)m_verses.size();
}



//////////////////////////////
//
// GridSide::getHarmonyCount --
//

int GridSide::getHarmonyCount(void) {
	if (m_harmony == NULL) {
		return 0;
	} else {
		return 1;
	}
}



//////////////////////////////
//
// GridSide::setHarmony --
//

void GridSide::setHarmony(HTp token) {
	if (m_harmony) {
		delete m_harmony;
		m_harmony = NULL;
	}
	m_harmony = token;
}


void GridSide::setHarmony(const string& token) {
	HTp newtoken = new HumdrumToken(token);
	setHarmony(newtoken);
}



//////////////////////////////
//
// GridSide::setDynamics --
//

void GridSide::setDynamics(HTp token) {
	if (m_dynamics) {
		delete m_dynamics;
		m_dynamics = NULL;
	}
	m_dynamics = token;
}


void GridSide::setDynamics(const string& token) {
	HTp newtoken = new HumdrumToken(token);
	setDynamics(newtoken);
}



//////////////////////////////
//
// GridSide::setFiguredBass --
//

void GridSide::setFiguredBass(HTp token) {
	if (m_figured_bass) {
		delete m_figured_bass;
		m_figured_bass = NULL;
	}
	m_figured_bass = token;
}


void GridSide::setFiguredBass(const string& token) {
	HTp newtoken = new HumdrumToken(token);
	setFiguredBass(newtoken);
}



///////////////////////////
//
// GridSide::detachHarmony --
//

void GridSide::detachHarmony(void) {
	m_harmony = NULL;
}



///////////////////////////
//
// GridSide::detachDynamics --
//

void GridSide::detachDynamics(void) {
	m_dynamics = NULL;
}



///////////////////////////
//
// GridSide::detachFiguredBass --
//

void GridSide::detachFiguredBass(void) {
	m_figured_bass = NULL;
}



//////////////////////////////
//
// GridSide::getHarmony --
//

HTp GridSide::getHarmony(void) {
	return m_harmony;
}



//////////////////////////////
//
// GridSide::getDynamics --
//

HTp GridSide::getDynamics(void) {
	return m_dynamics;
}



//////////////////////////////
//
// GridSide::getDynamicsCount --
//

int GridSide::getDynamicsCount(void) {
	if (m_dynamics == NULL) {
		return 0;
	} else {
		return 1;
	}
}



//////////////////////////////
//
// GridSide::getFiguredBass --
//

HTp GridSide::getFiguredBass(void) {
	return m_figured_bass;
}



//////////////////////////////
//
// GridSide::getFiguredBassCount --
//

int GridSide::getFiguredBassCount(void) {
	if (m_figured_bass == NULL) {
		return 0;
	} else {
		return 1;
	}
}



//////////////////////////////
//
// operator<< --
//

ostream& operator<<(ostream& output, GridSide* side) {
	output << " [";

	if (side->getVerseCount() > 0) {
		output << " verse:";
	}
	for (int i=0; i<(int)side->getVerseCount(); i++) {
		output << side->getVerse(i);
		if (i < (int)side->getVerseCount() - 1) {
			output << "; ";
		}

	}

	if (side->getDynamicsCount() > 0) {
		output << "dyn:" << side->getDynamics();
	}

	if (side->getHarmonyCount() > 0) {
		output << "harm:" << side->getHarmony();
	}

	output << "] ";
	return output;
}




//////////////////////////////
//
// GridSlice::GridSlice -- Constructor.  If partcount is positive, then
//    allocate the desired number of parts (still have to allocate staves
//    in part before using).
// default value: partcount = 0
//

GridSlice::GridSlice(GridMeasure* measure, HumNum timestamp, SliceType type,
		int partcount) {
	m_timestamp = timestamp;
	m_type      = type;
	m_owner     = NULL;
	m_measure   = measure;
	if (m_measure) {
		m_owner = measure->getOwner();
		m_measure = measure;
	}
	if (partcount > 0) {
		// create primary part/staff/voice structures
		this->resize(partcount);
		for (int p=0; p<partcount; p++) {
			this->at(p) = new GridPart;
			this->at(p)->resize(1);
			this->at(p)->at(0) = new GridStaff;
			this->at(p)->at(0)->resize(1);
			this->at(p)->at(0)->at(0) = new GridVoice;
		}
	}
}


//
// This constructor allocates the matching part and staff count of the
// input slice parameter.  There will be no GridVoices allocated inside the
// GridStaffs (they will be required to have at least one).
//

GridSlice::GridSlice(GridMeasure* measure, HumNum timestamp, SliceType type,
		const GridSlice& slice) {
	m_timestamp = timestamp;
	m_type = type;
	m_owner = measure->getOwner();
	m_measure = measure;
	int partcount = (int)slice.size();
	int staffcount;
	int voicecount;
	if (partcount > 0) {
		this->resize(partcount);
		for (int p=0; p<partcount; p++) {
			this->at(p) = new GridPart;
			GridPart* part = this->at(p);
			staffcount = (int)slice.at(p)->size();
			part->resize(staffcount);
			for (int s=0; s<staffcount; s++) {
				part->at(s) = new GridStaff;
				// voicecount = (int)slice.at(p)->at(s)->size();
				voicecount = 0;
				GridStaff* staff = part->at(s);
				staff->resize(voicecount);
				for (int v=0; v<voicecount; v++) {
					staff->at(v) = new GridVoice;
				}
			}
		}
	}
}


GridSlice::GridSlice(GridMeasure* measure, HumNum timestamp, SliceType type,
		GridSlice* slice) {
	m_timestamp = timestamp;
	m_type = type;
	m_owner = measure->getOwner();
	m_measure = measure;
	int partcount = (int)slice->size();
	int staffcount;
	int voicecount;
	if (partcount > 0) {
		this->resize(partcount);
		for (int p=0; p<partcount; p++) {
			this->at(p) = new GridPart;
			GridPart* part = this->at(p);
			staffcount = (int)slice->at(p)->size();
			part->resize(staffcount);
			for (int s=0; s<staffcount; s++) {
				part->at(s) = new GridStaff;
				// voicecount = (int)slice->at(p)->at(s)->size();
				voicecount = 0;
				GridStaff* staff = part->at(s);
				staff->resize(voicecount);
				for (int v=0; v<voicecount; v++) {
					staff->at(v) = new GridVoice;
				}
			}
		}
	}
}



//////////////////////////////
//
// GridSlice::~GridSlice -- Deconstructor.
//

GridSlice::~GridSlice(void) {
	for (int i=0; i<(int)this->size(); i++) {
		if (this->at(i)) {
			delete this->at(i);
			this->at(i) = NULL;
		}
	}
}



//////////////////////////////
//
// GridSlice::addToken -- Will not allocate part array, but will
//     grow staff or voice array if needed.
//

void GridSlice::addToken(const string& tok, int parti, int staffi, int voicei) {
	if ((parti < 0) || (parti >= (int)this->size())) {
		cerr << "Error: part index " << parti << " is out of range: size is ";
		cerr << this->size() << endl;
		return;
	}
	if (staffi < 0) {
		cerr << "Error: staff index " << staffi << " is out of range: size is ";
		cerr << this->at(parti)->size() << endl;
		return;
	}

	if (staffi >= (int)this->at(parti)->size()) {
		int ssize = (int)this->at(parti)->size();
		for (int i=ssize; i<=staffi; i++) {
			GridStaff* gs = new GridStaff;
			this->at(parti)->push_back(gs);
		}
	}

	if (voicei >= (int)this->at(parti)->at(staffi)->size()) {
		int oldsize = (int)this->at(parti)->at(staffi)->size();
		this->at(parti)->at(staffi)->resize(voicei+1);
		for (int j=oldsize; j<=voicei; j++) {
			this->at(parti)->at(staffi)->at(j) = new GridVoice;
		}
	}
	this->at(parti)->at(staffi)->at(voicei)->setToken(tok);
}



//////////////////////////////
//
// GridSlice::createRecipTokenFromDuration --  Will not be able to
//   distinguish between triplet notes and dotted normal equivalents,
//   this can be changed later by checking neighboring durations in the
//   list for the presence of triplets.
//

HTp GridSlice::createRecipTokenFromDuration(HumNum duration) {
	duration /= 4;  // convert to quarter note units.
	HTp token;
	string str;
	HumNum dotdur;
	if (duration.getNumerator() == 0) {
		// if the GridSlice is at the end of a measure, the
		// time between the starttime/endtime of the GridSlice should
		// be subtracted from the endtime of the current GridMeasure.
		token = new HumdrumToken("g");
		return token;
	} else if (duration.getNumerator() == 1) {
		token = new HumdrumToken(to_string(duration.getDenominator()));
		return token;
	} else if (duration.getNumerator() % 3 == 0) {
		dotdur = ((duration * 2) / 3);
		if (dotdur.getNumerator() == 1) {
			token = new HumdrumToken(to_string(dotdur.getDenominator()) + ".");
			return token;
		}
	}

	// try to fit to two dots here

	// try to fit to three dots here

	str = to_string(duration.getDenominator()) + "%" +
	         to_string(duration.getNumerator());
	token = new HumdrumToken(str);
	return token;
}



//////////////////////////////
//
// GridSlice::isInterpretationSlice --
//

bool GridSlice::isInterpretationSlice(void) {
	SliceType type = getType();
	if (type < SliceType::_Measure) {
		return false;
	}
	if (type > SliceType::_Interpretation) {
		return false;
	}
	return true;
}



//////////////////////////////
//
// GridSlice::isDataSlice --
//

bool GridSlice::isDataSlice(void) {
	SliceType type = getType();
	if (type <= SliceType::_Data) {
		return true;
	} else {
		return false;
	}
}



//////////////////////////////
//
// GridSlice::transferTokens -- Create a HumdrumLine and append it to
//    the data.
//

void GridSlice::transferTokens(HumdrumFile& outfile, bool recip) {
	HTp token = NULL;
	HLp line = new HumdrumLine;
	GridVoice* voice;
	string empty = ".";
	if (isMeasureSlice()) {
		if (this->size() > 0) {
			if (this->at(0)->at(0)->size() > 0) {
				voice = this->at(0)->at(0)->at(0);
				empty = (string)*voice->getToken();
			} else {
				empty = "=YYYYYY";
			}
		}
	} else if (isInterpretationSlice()) {
		empty = "*";
	} else if (isLayoutSlice()) {
		empty = "!";
	} else if (!hasSpines()) {
		empty = "???";
	}

	if (recip) {
		if (isNoteSlice()) {
			token = createRecipTokenFromDuration(getDuration());
		} else if (isClefSlice()) {
			token = new HumdrumToken("*");
			empty = "*";
		} else if (isMeasureSlice()) {
			if (this->at(0)->at(0)->size() > 0) {
				voice = this->at(0)->at(0)->at(0);
				token = new HumdrumToken((string)*voice->getToken());
			} else {
				token = new HumdrumToken("=XXXXX");
			}
			empty = (string)*token;
		} else if (isInterpretationSlice()) {
			token = new HumdrumToken("*");
			empty = "*";
		} else if (isGraceSlice()) {
			token = new HumdrumToken("q");
			empty = ".";
		} else if (hasSpines()) {
			token = new HumdrumToken("55");
			empty = "!";
		}
		if (token != NULL) {
			if (hasSpines()) {
				line->appendToken(token);
			} else {
				delete token;
				token = NULL;
			}
		}
	}

	// extract the Tokens from each part/staff
	int p; // part index
	int s; // staff index
	int v; // voice index

	for (p=(int)size()-1; p>=0; p--) {
		if ((!hasSpines()) && (p != 0)) {
			continue;
		}
		GridPart& part = *this->at(p);
		for (s=(int)part.size()-1; s>=0; s--) {
			if ((!hasSpines()) && (s != 0)) {
				continue;
			}
			GridStaff& staff = *part.at(s);
			if (staff.size() == 0) {
				// fix this later.  For now if there are no notes
				// on the staff, add a null token.  Fix so that
				// all open voices are given null tokens.
				token = new HumdrumToken(empty);
				line->appendToken(token);
			} else {
				for (v=0; v<(int)staff.size(); v++) {
					if (staff.at(v) && staff.at(v)->getToken()) {
						line->appendToken(staff.at(v)->getToken());
						staff.at(v)->forgetToken();
					} else if (!staff.at(v)) {
						token = new HumdrumToken(empty);
						line->appendToken(token);
					} else {
						token = new HumdrumToken(empty);
						line->appendToken(token);
					}
				}
			}

			if (!this->hasSpines()) {
				// Don't add sides to non-spined lines
				continue;
			}

			int maxvcount = getVerseCount(p, s);
			int maxhcount = getHarmonyCount(p, s);
			int maxfcount = getFiguredBassCount(p, s);
			if (hasSpines()) {
				transferSides(*line, staff, empty, maxvcount, maxhcount, maxfcount);
			}
		}

		// Transfer the sides at the part level
		int maxhcount = getHarmonyCount(p);
		int maxvcount = getVerseCount(p, -1);
		int maxdcount = getDynamicsCount(p);
		int maxfcount = getFiguredBassCount(p);

		if (hasSpines()) {
			transferSides(*line, part, p, empty, maxvcount, maxhcount, maxdcount, maxfcount);
		}
	}

	outfile.appendLine(line);
}



//////////////////////////////
//
// GridSlice::getMeasureDuration --
//

HumNum GridSlice::getMeasureDuration(void) {
	GridMeasure* measure = getMeasure();
	if (!measure) {
		return -1;
	} else {
		return measure->getDuration();
	}
}



//////////////////////////////
//
// GridSlice::getMeasureTimestamp -- Return the start time of the measure.
//

HumNum GridSlice::getMeasureTimestamp(void) {
	GridMeasure* measure = getMeasure();
	if (!measure) {
		return -1;
	} else {
		return measure->getTimestamp();
	}
}



//////////////////////////////
//
// GridSlice::getVerseCount --
//

int GridSlice::getVerseCount(int partindex, int staffindex) {
	HumGrid* grid = getOwner();
	if (!grid) {
		return 0;
	}
	return grid->getVerseCount(partindex, staffindex);
}



//////////////////////////////
//
// GridSlice::getHarmonyCount --
//    default value: staffindex = -1; (currently not looking for
//        harmony data attached directly to staff (only to part.)
//

int GridSlice::getHarmonyCount(int partindex, int staffindex) {
	HumGrid* grid = getOwner();
	if (!grid) {
		return 0;
	}
	if (staffindex >= 0) {
		// ignoring staff-level harmony
		return 0;
	} else {
		return grid->getHarmonyCount(partindex);
	}
}



//////////////////////////////
//
// GridSlice::getDynamicsCount -- Return 0 if no dynamics, otherwise typically returns 1.
//

int GridSlice::getDynamicsCount(int partindex, int staffindex) {
	HumGrid* grid = getOwner();
	if (!grid) {
		return 0;
	}
	if (staffindex >= 0) {
		// ignoring staff-level harmony
		return 0;
	} else {
		return grid->getDynamicsCount(partindex);
	}
}



//////////////////////////////
//
// GridSlice::getFiguredBassCount -- Return 0 if no figured bass; otherwise,
//     typically returns 1.
//

int GridSlice::getFiguredBassCount(int partindex, int staffindex) {
	HumGrid* grid = getOwner();
	if (!grid) {
		return 0;
	}
	if (staffindex >= 0) {
		// ignoring staff-level figured bass
		return 0;
	} else {
		return grid->getFiguredBassCount(partindex);
	}
}



//////////////////////////////
//
// GridSlice::transferSides --
//

// this version is used to transfer Sides from the Part
void GridSlice::transferSides(HumdrumLine& line, GridPart& sides,
		int partindex, const string& empty, int maxvcount, int maxhcount,
		int maxdcount, int maxfcount) {

	int hcount = sides.getHarmonyCount();
	int vcount = sides.getVerseCount();

	HTp newtoken;

	for (int i=0; i<vcount; i++) {
		HTp verse = sides.getVerse(i);
		if (verse) {
			line.appendToken(verse);
			sides.detachHarmony();
		} else {
			newtoken = new HumdrumToken(empty);
			line.appendToken(newtoken);
		}
	}

	for (int i=vcount; i<maxvcount; i++) {
		newtoken = new HumdrumToken(empty);
		line.appendToken(newtoken);
	}

	if (maxdcount > 0) {
		HTp dynamics = sides.getDynamics();
		if (dynamics) {
			line.appendToken(dynamics);
			sides.detachDynamics();
		} else {
			newtoken = new HumdrumToken(empty);
			line.appendToken(newtoken);
		}
	}

	if (maxfcount > 0) {
		HTp figuredbass = sides.getFiguredBass();
		if (figuredbass) {
			line.appendToken(figuredbass);
			sides.detachFiguredBass();
		} else {
			newtoken = new HumdrumToken(empty);
			line.appendToken(newtoken);
		}
	}

	for (int i=0; i<hcount; i++) {
		HTp harmony = sides.getHarmony();
		if (harmony) {
			line.appendToken(harmony);
			sides.detachHarmony();
		} else {
			newtoken = new HumdrumToken(empty);
			line.appendToken(newtoken);
		}
	}

	for (int i=hcount; i<maxhcount; i++) {
		newtoken = new HumdrumToken(empty);
		line.appendToken(newtoken);
	}
}


// this version is used to transfer Sides from the Staff
void GridSlice::transferSides(HumdrumLine& line, GridStaff& sides,
		const string& empty, int maxvcount, int maxhcount, int maxfcount) {

	// existing verses:
	int vcount = sides.getVerseCount();

	int fcount = sides.getFiguredBassCount();

	// there should not be any harony attached to staves
	// (only to parts, so hcount should only be zero):
	int hcount = sides.getHarmonyCount();
	HTp newtoken;

	for (int i=0; i<vcount; i++) {
		HTp verse = sides.getVerse(i);
		if (verse) {
			line.appendToken(verse);
			sides.setVerse(i, NULL); // needed to avoid double delete
		} else {
			newtoken = new HumdrumToken(empty);
			line.appendToken(newtoken);
		}
	}

	if (vcount < maxvcount) {
		for (int i=vcount; i<maxvcount; i++) {
			newtoken = new HumdrumToken(empty);
			line.appendToken(newtoken);
		}
	}

	for (int i=0; i<hcount; i++) {
		HTp harmony = sides.getHarmony();
		if (harmony) {
			line.appendToken(harmony);
			sides.detachHarmony();
		} else {
			newtoken = new HumdrumToken(empty);
			line.appendToken(newtoken);
		}
	}

	for (int i=0; i<fcount; i++) {
		HTp figuredbass = sides.getFiguredBass();
		if (figuredbass) {
			line.appendToken(figuredbass);
			sides.detachFiguredBass();
		} else {
			newtoken = new HumdrumToken(empty);
			line.appendToken(newtoken);
		}
	}

	if (hcount < maxhcount) {
		for (int i=hcount; i<maxhcount; i++) {
			newtoken = new HumdrumToken(empty);
			line.appendToken(newtoken);
		}
	}

	if (fcount < maxfcount) {
		for (int i=fcount; i<maxfcount; i++) {
			newtoken = new HumdrumToken(empty);
			line.appendToken(newtoken);
		}
	}

}



//////////////////////////////
//
// GridSlice::initializePartStaves -- Also initialize sides
//

void GridSlice::initializePartStaves(vector<MxmlPart>& partdata) {
	int i, j;
	if (this->size() > 0) {
		// strange that this should happen, but presume the data
		// needs to be deleted.
		for (int i=0; i<(int)this->size(); i++) {
			if (this->at(i)) {
				delete this->at(i);
				this->at(i) = NULL;
			}
		}
	}
	this->resize(partdata.size());

	for (i=0; i<(int)partdata.size(); i++) {
		this->at(i) = new GridPart;
		this->at(i)->resize(partdata[i].getStaffCount());
		for (j=0; j<(int)partdata[i].getStaffCount(); j++) {
			this->at(i)->at(j) = new GridStaff;
		}
	}
}



//////////////////////////////
//
// GridSlice::initializeByStaffCount -- Initialize with parts containing a single staff.
//

void GridSlice::initializeByStaffCount(int staffcount) {
	if (this->size() > 0) {
		// strange that this should happen, but presume the data
		// needs to be deleted.
		for (int i=0; i<(int)this->size(); i++) {
			if (this->at(i)) {
				delete this->at(i);
				this->at(i) = NULL;
			}
		}
	}
	this->clear();
	this->resize(staffcount);

	for (int i=0; i<staffcount; i++) {
		this->at(i) = new GridPart;
		this->at(i)->resize(1);
		this->at(i)->at(0) = new GridStaff;
		this->at(i)->at(0)->resize(1);
		this->at(i)->at(0)->at(0) = new GridVoice;
	}
}



//////////////////////////////
//
// GridSlice::initializeBySlice -- Allocate parts/staves/voices counts by an existing slice.
//   Presuming that the slice is not already initialize with content.
//

void GridSlice::initializeBySlice(GridSlice* slice) {
	int partcount = (int)slice->size();
	this->resize(partcount);
	for (int p = 0; p < partcount; p++) {
		this->at(p) = new GridPart;
		int staffcount = (int)slice->at(p)->size();
		this->at(p)->resize(staffcount);
		for (int s = 0; s < staffcount; s++) {
			this->at(p)->at(s) = new GridStaff;
			int voicecount = (int)slice->at(p)->at(s)->size();
			this->at(p)->at(s)->resize(voicecount);
			for (int v=0; v < voicecount; v++) {
				this->at(p)->at(s)->at(v) = new GridVoice;
			}
		}
	}
}



//////////////////////////////
//
// GridSlice::getDuration -- Return the duration of the slice in
//      quarter notes.
//

HumNum GridSlice::getDuration(void) {
	return m_duration;
}



//////////////////////////////
//
// GridSlice::setDuration --
//

void GridSlice::setDuration(HumNum duration) {
	m_duration = duration;
}



//////////////////////////////
//
// GridSlice::getTimestamp --
//

HumNum GridSlice::getTimestamp(void) {
	return m_timestamp;
}



//////////////////////////////
//
// GridSlice::setTimestamp --
//

void GridSlice::setTimestamp(HumNum timestamp) {
	m_timestamp = timestamp;
}



//////////////////////////////
//
// GridSlice::setOwner --
//

void GridSlice::setOwner(HumGrid* owner) {
	m_owner = owner;
}



//////////////////////////////
//
// GridSlice::getOwner --
//

HumGrid* GridSlice::getOwner(void) {
	return m_owner;
}



//////////////////////////////
//
// GridSlice::getMeasure --
//

GridMeasure* GridSlice::getMeasure(void) {
	return m_measure;
}



//////////////////////////////
//
// operator<< -- print token content of a slice
//

ostream& operator<<(ostream& output, GridSlice& slice) {
	return output << &slice;
}


ostream& operator<<(ostream& output, GridSlice* slice) {
	if (slice == NULL) {
		output << "{n}";
		return output;
	}
	output << "TS=" << slice->getTimestamp() << " ";
	for (int p=0; p<(int)slice->size(); p++) {
		GridPart* part = slice->at(p);
		output << "(p" << p << ":)";
		if (part == NULL) {
			output << "{n}";
			continue;
		}
		for (int s=0; s<(int)part->size(); s++) {
			GridStaff* staff = part->at(s);
			output << "(s" << s << ":)";
			if (staff == NULL) {
				output << "{n}";
				continue;
			}
			for (int t=0; t<(int)staff->size(); t++) {
				GridVoice* gt = staff->at(t);
				output << "(v" << t << ":)";
				if (gt == NULL) {
					output << "{n}";
					continue;
				} else {
					HTp token = gt->getToken();
					if (token == NULL) {
						output << "{n}";
					} else {
						output << " \"" << *token << "\" ";
					}
				}
			}
			output << " sside:" << (GridSide*)staff;
		}
		output << " pside:" << (GridSide*)part;
	}
	return output;
}



//////////////////////////////
//
// GridSlice::hasSpines -- true if not a global comment or similar.
//

bool GridSlice::hasSpines(void) {
	SliceType type = getType();
	if (type < SliceType::_Spined) {
		return true;
	} else {
		return false;
	}
}



//////////////////////////////
//
// GridSlice::invalidate -- Mark the slice as invalid, which means that
//    it should not be transferred to the output Humdrum file in HumGrid.
//    Tokens stored in the GridSlice will be deleted by GridSlice when it
//    is destroyed.
//

void GridSlice::invalidate(void) {
		m_type = SliceType::Invalid;
		// should only do with 0 duration slices, but force to 0 if not already.
		setDuration(0);
}



//////////////////////////////
//
// GridSlice::reportVerseCount --
//

void GridSlice::reportVerseCount(int partindex, int staffindex, int count) {
	if (!m_owner) {
		return;
	}
	m_owner->reportVerseCount(partindex, staffindex, count);
}



//////////////////////////////
//
// GridSlice::getNullTokenForSlice --
//

string GridSlice::getNullTokenForSlice(void) {
	if (isDataSlice()) {
		return ".";
	} else if (isInterpretationSlice()) {
		return "*";
	} else if (isMeasureSlice()) {
		return "=";
	} else if (!hasSpines()) {
		return "!!";
	} else {
		return "!";
	}
}




//////////////////////////////
//
// GridStaff::GridStaff -- Constructor.
//

GridStaff::GridStaff(void) : vector<GridVoice*>(0), GridSide() {
	// do nothing;
}



//////////////////////////////
//
// GridStaff::~GridStaff -- Deconstructor.
//

GridStaff::~GridStaff(void) {
	for (int i=0; i<(int)this->size(); i++) {
		if (this->at(i)) {
			delete this->at(i);
			this->at(i) = NULL;
		}
	}
}



//////////////////////////////
//
// GridStaff::setTokenLayer -- Insert a token at the given voice/layer index.
//    If there is another token already there, then delete it.  If there
//    is no slot for the given voice, then create one and fill in all of the
//    other new ones with NULLs.
//

GridVoice* GridStaff::setTokenLayer(int layerindex, HTp token, HumNum duration) {
	if (layerindex < 0) {
		cerr << "Error: layer index is " << layerindex
		     << " for " << token << endl;
		return NULL;
	}
	if (layerindex > (int)this->size()-1) {
		int oldsize = (int)this->size();
		this->resize(layerindex+1);
		for (int i=oldsize; i<(int)this->size(); i++) {
			this->at(i) = NULL;
		}
	}
	if (this->at(layerindex) != NULL) {
		delete this->at(layerindex);
	}
	GridVoice* gv = new GridVoice(token, duration);
	this->at(layerindex) = gv;
	return gv;
}



////////////////////////////
//
// GridStaff::setNullTokenLayer --
//

void GridStaff::setNullTokenLayer(int layerindex, SliceType type,
		HumNum nextdur) {

	if (type == SliceType::Invalid) {
		return;
	}
	if (type == SliceType::GlobalLayouts) {
		return;
	}
	if (type == SliceType::GlobalComments) {
		return;
	}
	if (type == SliceType::ReferenceRecords) {
		return;
	}

	string nulltoken;
	if (type < SliceType::_Data) {
		nulltoken = ".";
	} else if (type <= SliceType::_Measure) {
		nulltoken = "=";
	} else if (type <= SliceType::_Interpretation) {
		nulltoken = "*";
	} else if (type <= SliceType::_Spined) {
		nulltoken = "!";
	} else {
		cerr << "!!STRANGE ERROR: " << this << endl;
		cerr << "!!SLICE TYPE: " << (int)type << endl;
	}

	if (layerindex < (int)this->size()) {
		if ((at(layerindex) != NULL) && (at(layerindex)->getToken() != NULL)) {
			if ((string)*at(layerindex)->getToken() == nulltoken) {
				// there is already a null data token here, so don't
				// replace it.
				return;
			}
			cerr << "Warning, replacing existing token: "
			     << *this->at(layerindex)->getToken()
			     << " with a null token"
			     << endl;
		}
	}
	HumdrumToken* token = new  HumdrumToken(nulltoken);
	setTokenLayer(layerindex, token, nextdur);

}



//////////////////////////////
//
// GridStaff::appendTokenLayer -- concatenate the string content
//   of a token onto the current token stored in the slot (or just
//   place this one in the slot if none there yet).  This is used for
//   chords normally.
//

void GridStaff::appendTokenLayer(int layerindex, HTp token, HumNum duration,
		const string& spacer) {

	GridVoice* gt;
	if (layerindex > (int)this->size()-1) {
		int oldsize = (int)this->size();
		this->resize(layerindex+1);
		for (int i=oldsize; i<(int)this->size(); i++) {
			this->at(i) = NULL;
		}
	}
	if (this->at(layerindex) != NULL) {
		string newtoken;
		newtoken = (string)*this->at(layerindex)->getToken();
		newtoken += spacer;
		newtoken += (string)*token;
		(string)*(this->at(layerindex)->getToken()) = newtoken;
	} else {
		gt = new GridVoice(token, duration);
		this->at(layerindex) = gt;
	}
}



//////////////////////////////
//
// GridStaff::getMaxVerseCount --
//

int GridStaff::getMaxVerseCount(void) {
	return 5;
}



//////////////////////////////
//
// GridStaff::getString --
//

string GridStaff::getString(void) {
	string output;
	for (int v=0; v<(int)size(); v++) {
		GridVoice* gv = at(v);
		if (gv == NULL) {
			output += "{nv}";
		} else {
			HTp token = gv->getToken();
			if (token == NULL) {
				output += "{n}";
			} else {
				output += *token;
			}
		}
		if (v < (int)size() - 1) {
			output += "\t";
		}
	}
	return output;
}



//////////////////////////////
//
// operator<< --
//

ostream& operator<<(ostream& output, GridStaff* staff) {
	if (staff == NULL) {
		output << "{n}";
		return output;
	}
	for (int t=0; t<(int)staff->size(); t++) {
		GridVoice* gt = staff->at(t);
		cout << "(v" << t << ":)";
		if (gt == NULL) {
			cout << "{gt:n}";
			continue;
		} else {
			HTp token = gt->getToken();
			if (token == NULL) {
				cout << "{n}";
			} else {
				cout << " \"" << *token << "\" ";
			}
		}
	}
	output << (GridSide*) staff;
	return output;
}



//////////////////////////////
//
// GridVoice::GridVoice -- Constructor.
//

GridVoice::GridVoice(void) {
	m_token      = NULL;
	m_transfered = false;
}

GridVoice::GridVoice(HTp token, HumNum duration) {
	m_token      = token;
	m_nextdur    = duration;
	m_transfered = false;
}


GridVoice::GridVoice(const char* token, HumNum duration) {
	m_token      = new HumdrumToken(token);
	m_nextdur    = duration;
	m_transfered = false;
}


GridVoice::GridVoice(const string& token, HumNum duration) {
	m_token      = new HumdrumToken(token);
	m_nextdur    = duration;
	m_transfered = false;
}



//////////////////////////////
//
// GridVoice::~GridVoice -- Deconstructor: delete the token only if it
//     has not been transfered to a HumdrumFile object.
//

GridVoice::~GridVoice() {
	if (m_token && !m_transfered) {
		delete m_token;
	}
	m_token = NULL;
}



//////////////////////////////
//
// GridVoice::isTransfered -- True if token was copied to a HumdrumFile
//      object.
//

bool GridVoice::isTransfered(void) {
	return m_transfered;
}



//////////////////////////////
//
// GridVoice::setTransfered -- True if the object should not be
//    deleted with the object is destroyed.  False if the token
//    is not NULL and should be deleted when object is destroyed.
//

void GridVoice::setTransfered(bool state) {
	m_transfered = state;
}



//////////////////////////////
//
// GridVoice::getToken --
//

HTp GridVoice::getToken(void) const {
	return m_token;
}



//////////////////////////////
//
// GridVoice::setToken --
//

void GridVoice::setToken(HTp token) {
	if (!m_transfered && m_token) {
		delete m_token;
	}
	m_token = token;
	m_transfered = false;
}


void GridVoice::setToken(const string& token) {
	HTp realtoken = new HumdrumToken(token);
	setToken(realtoken);
}


void GridVoice::setToken(const char* token) {
	HTp realtoken = new HumdrumToken(token);
	setToken(realtoken);
}



//////////////////////////////
//
// GridVoice::isNull -- returns true if token is NULL or ".".
//

bool GridVoice::isNull(void) const {
	if (getToken() == NULL) {
		return true;
	} else if (getToken()->isNull()) {
		return true;
	} else {
		return false;
	}
}



//////////////////////////////
//
// GridVoice::setDuration --
//

void GridVoice::setDuration(HumNum duration) {
	m_nextdur = duration;
	m_prevdur = 0;
}



//////////////////////////////
//
// GridVoice::setDurationToPrev --
//

void GridVoice::setDurationToPrev(HumNum dur) {
	m_prevdur = dur;
}



//////////////////////////////
//
// GridVoice::getDurationToNext --
//

HumNum GridVoice::getDurationToNext(void) const {
	return m_nextdur;
}



//////////////////////////////
//
// GridVoice::getDurationToPrev --
//

HumNum GridVoice::getDurationToPrev(void) const {
	return m_nextdur;
}



//////////////////////////////
//
// GridVoice::incrementDuration --
//

void GridVoice::incrementDuration(HumNum duration) {
	m_nextdur -= duration;
	m_prevdur += duration;
}



//////////////////////////////
//
// GridVoice::forgetToken -- The HumdrumToken was passed off
//      to some other object which is now responsible for
//      deleting it.
//

void GridVoice::forgetToken(void) {
	setTransfered(true);
	m_token = NULL;
}



//////////////////////////////
//
// GridVoice::getDuration -- Return the total duration of the
//   durational item, the sum of the nextdur and prevdur.
//

HumNum GridVoice::getDuration(void) const {
	return m_nextdur + m_prevdur;
}



//////////////////////////////
//
// GridVoice::getString --
//

string GridVoice::getString(void) {
	string output;
	HTp token = getToken();
	if (token == NULL) {
		cout << "{n}";
	} else {
		cout << *token;
	}
	return output;
}



//////////////////////////////
//
// operator<< -- print token content of a voice
//

ostream& operator<<(ostream& output, GridVoice* voice) {
	if (voice == NULL) {
		output << "{n}";
		return output;
	}

	HTp token = voice->getToken();
	if (token == NULL) {
		cout << "{n}";
	} else {
		cout << " \"" << *token << "\" ";
	}
	return output;
}

ostream& operator<<(ostream& output, GridVoice& voice) {
	output << &voice;
	return output;
}




//////////////////////////////
//
// HumAddress::HumAddress -- HumAddress constructor.
//

HumAddress::HumAddress(void) {
	m_track         = -1;
	m_subtrack      = -1;
	m_subtrackcount = 0;
	m_fieldindex    = -1;
	m_owner         = NULL;
}


HumAddress::HumAddress(HumAddress& address) {
	m_fieldindex    = address.m_fieldindex;
	m_track         = address.m_track;
	m_subtrack      = address.m_subtrack;
	m_subtrackcount = address.m_subtrackcount;
	m_spining       = address.m_spining;
	m_owner         = address.m_owner;
}



//////////////////////////////
//
// HumAddress::~HumAddress -- HumAddress deconstructor.
//

HumAddress::~HumAddress() {
	m_track         = -1;
	m_subtrack      = -1;
	m_fieldindex    = -1;
	m_subtrackcount = 0;
	m_owner         = NULL;
}



//////////////////////////////
//
// HumAddress::operator= -- Copy humdrum address to another object.
//

HumAddress& HumAddress::operator=(const HumAddress& address) {
	m_fieldindex    = address.m_fieldindex;
	m_track         = address.m_track;
	m_subtrack      = address.m_subtrack;
	m_subtrackcount = address.m_subtrackcount;
	m_spining       = address.m_spining;
	m_owner         = address.m_owner;
	return *this;
}


//////////////////////////////
//
// HumAddress::getLineIndex -- Returns the line index in the owning HumdrumFile
//    for the token associated with the address.  Returns -1 if not owned by a
//    HumdrumLine (or line assignments have not been made for tokens in the
//    file).
//

int  HumAddress::getLineIndex(void) const {
	if (m_owner == NULL) {
		return -1;
	} else {
		return m_owner->getLineIndex();
	}
}



//////////////////////////////
//
// HumAddress::getLineNumber --  Similar to getLineIndex() but adds one.
//

int HumAddress::getLineNumber(void) const {
	return getLineIndex() + 1;
}



//////////////////////////////
//
// HumAddress::getFieldIndex -- Returns the field index on the line of the
//     token associated with the address.
//

int HumAddress::getFieldIndex(void) const {
	return m_fieldindex;
}



//////////////////////////////
//
// HumAddress::getDataType -- Return the exclusive interpretation string of the
//    token associated with the address.
//

const HumdrumToken& HumAddress::getDataType(void) const {
	static HumdrumToken null("");
	if (m_owner == NULL) {
		return null;
	}
	HumdrumToken* tok = m_owner->getTrackStart(getTrack());
	if (tok == NULL) {
		return null;
	}
	return *tok;
}



//////////////////////////////
//
// HumAddress::getSpineInfo -- Return the spine information for the token
//     associated with the address.  Examples: "1" the token is in the first
//     (left-most) spine, and there are no active sub-spines for the spine.
//     "(1)a"/"(1)b" are the spine descriptions of the two sub-spines after
//     a split manipulator (*^).  "((1)a)b" is the second sub-spines of the
//     first sub-spine for spine 1.
//
//

const string& HumAddress::getSpineInfo(void) const {
	return m_spining;
}



//////////////////////////////
//
// HumAddress::getTrack -- The track number of the given spine.  This is the
//   first number in the spine info string.  The track number is the same
//   as a spine number.
//

int HumAddress::getTrack(void) const {
	return m_track;
}



//////////////////////////////
//
// HumAddress::getSubtrack -- The subtrack number of the given spine.  This
//   functions in a similar manner to layer numbers in MEI data.  The first
//   sub-spine of a spine is always subtrack 1, regardless of whether or not
//   an exchange manipulator (*x) was used to switch the left-to-right ordering
//   of the spines in the file.  All sub-spines regardless of their splitting
//   origin are given sequential subtrack numbers.  For example if the spine
//   info is "(1)a"/"((1)b)a"/"((1)b)b" -- the spine is split, then the second
//   sub-spine only is split--then the sub-spines are labeled as sub-tracks "1",
//   "2", "3" respectively.  When a track has only one sub-spine (i.e., it has
//   been split), the subtrack value will be "0".
//

int HumAddress::getSubtrack(void) const {
	if (m_subtrackcount == 1) {
		return 0;
	} else {
		// return m_subtrack + 1;
		// Should already be indexed from one.
		return m_subtrack;
	}
}



//////////////////////////////
//
// HumAddress::getSubtrackCount -- The number of subtrack spines for a
//   given spine on the owning HumdurmLine.  Returns 0 if spine analysis
//   has not been done, or if the line does not have spines (i.e., reference
//   records, global comments and empty lines).
//

int HumAddress::getSubtrackCount(void) const {
	return m_subtrackcount;
}



//////////////////////////////
//
// HumAddress::getTrackString --  Return the track and subtrack as a string.
//      The returned string will have the track number if the sub-spine value
//      is zero.  The optional separator parameter is used to separate the
//      track number from the subtrack number.
// default value: separator = "."
//

string HumAddress::getTrackString(string separator) const {
	string output;
	int thetrack    = getTrack();
	int thesubtrack = getSubtrack();
	output += to_string(thetrack);
	if (thesubtrack > 0) {
		output += separator + to_string(thesubtrack);
	}
	return output;
}



//////////////////////////////
//
// HumAddress::setOwner -- Stores a pointer to the HumdrumLine on which
//   the token associated with this address belongs.  When not owned by
//   a HumdrumLine, the parameter's value should be NULL.
//

void HumAddress::setOwner(HLp aLine) {
	m_owner = aLine;
}



//////////////////////////////
//
// HumAddress::getLine -- return the HumdrumLine which owns the token
//    associated with this address.  Returns NULL if it does not belong
//    to a HumdrumLine object.
//

HLp HumAddress::getLine(void) const {
	return m_owner;
}



//////////////////////////////
//
// HumAddress::hasOwner -- Returns true if a HumdrumLine owns the token
//    associated with the address.
//

bool HumAddress::hasOwner(void) const {
	return m_owner == NULL ? 0 : 1;
}



//////////////////////////////
//
// HumAddress::setFieldIndex -- Set the field index of associated token
//   in the HumdrumLine owner.  If the token is now owned by a HumdrumLine,
//   then the input parameter should be -1.
//

void HumAddress::setFieldIndex(int index) {
	m_fieldindex = index;
}



//////////////////////////////
//
// HumAddress::setSpineInfo -- Set the spine description of the associated
//     token.  For example "2" for the second spine (from the left), or
//     "((2)a)b" for a sub-spine created as the left sub-spine of the main
//     spine and then as the right sub-spine of that sub-spine.  This function
//     is used by the HumdrumFileStructure class.
//

void HumAddress::setSpineInfo(const string& spineinfo) {
	m_spining = spineinfo;
}



//////////////////////////////
//
// HumAddress::setTrack -- Set the track number of the associated token.
//   This should always be the first number in the spine information string,
//   or -1 if the spine info is empty.  Tracks are limited to an arbitrary
//   count of 1000 (could be increased in the future if needed).  This function
//   is used by the HumdrumFileStructure class.
//

void HumAddress::setTrack(int aTrack, int aSubtrack) {
	setTrack(aTrack);
	setSubtrack(aSubtrack);
}


void HumAddress::setTrack(int aTrack) {
	if (aTrack < 0) {
		aTrack = -1;
	}
	if (aTrack > 1000) {
		aTrack = 1000;
	}
	m_track = aTrack;
}



//////////////////////////////
//
// HumAddress::setSubtrack -- Set the subtrack of the spine.
//   If the token is the only one active for a spine, the subtrack should
//   be set to zero.  If there are more than one sub-tracks for the spine, this
//   is the one-offset index of the spine (be careful if a sub-spine column
//   is exchanged with another spine other than the one from which it was
//   created.  In this case the subtrack number is not useful to calculate
//   the field index of other sub-tracks for the given track.
//   This function is used by the HumdrumFileStructure class.
//

void HumAddress::setSubtrack(int aSubtrack) {
	if (aSubtrack < 0) {
		aSubtrack = -1;
	}
	if (aSubtrack > 1000) {
		aSubtrack = 1000;
	}
	m_subtrack = aSubtrack;
}



//////////////////////////////
//
// HumAddress::setSubtrackCount --
//

void HumAddress::setSubtrackCount(int count) {
	m_subtrackcount = count;
}



//////////////////////////////
//
// HumGrid::HumGrid -- Constructor.
//

HumGrid::HumGrid(void) {
	// Limited to 100 parts:
	m_verseCount.resize(100);
	m_harmonyCount.resize(100);
	m_dynamics.resize(100);
	m_figured_bass.resize(100);
	fill(m_dynamics.begin(), m_dynamics.end(), false);
	fill(m_figured_bass.begin(), m_figured_bass.end(), false);
	fill(m_harmonyCount.begin(), m_harmonyCount.end(), 0);

	// default options
	m_musicxmlbarlines = false;
	m_recip = false;
	m_pickup = false;
}



//////////////////////////////
//
// HumGrid::~HumGrid -- Deconstructor.
//

HumGrid::~HumGrid(void) {
	for (int i=0; i<(int)this->size(); i++) {
		if (this->at(i)) {
			delete this->at(i);
		}
	}
}



//////////////////////////////
//
// HumGrid::addMeasureToBack -- Allocate a GridMeasure at the end of the
//     measure list.
//

GridMeasure* HumGrid::addMeasureToBack(void) {
	GridMeasure* gm = new GridMeasure(this);
	this->push_back(gm);
	return this->back();
}



//////////////////////////////
//
// HumGrid::enableRecipSpine --
//

void HumGrid::enableRecipSpine(void) {
	m_recip = true;
}



//////////////////////////////
//
// HumGrid::getPartCount -- Return the number of parts in the Grid
//   by looking at the number of parts in the first spined GridSlice.
//

int  HumGrid::getPartCount(void) {
	if (!m_allslices.empty()) {
		return (int)m_allslices[0]->size();
	}

	if (this->empty()) {
		return 0;
	}

	if (this->at(0)->empty()) {
		return 0;
	}

	return (int)this->at(0)->back()->size();
}



//////////////////////////////
//
// HumGrid::getStaffCount --
//

int HumGrid::getStaffCount(int partindex) {
	if (this->empty()) {
		return 0;
	}

	if (this->at(0)->empty()) {
		return 0;
	}

	return (int)this->at(0)->back()->at(partindex)->size();
}



//////////////////////////////
//
// HumGrid::getHarmonyCount --
//

int HumGrid::getHarmonyCount(int partindex) {
	if ((partindex < 0) || (partindex >= (int)m_harmonyCount.size())) {
		return 0;
	}
	return m_harmonyCount.at(partindex);
}



//////////////////////////////
//
// HumGrid::getDynamicsCount --
//

int HumGrid::getDynamicsCount(int partindex) {
	if ((partindex < 0) || (partindex >= (int)m_dynamics.size())) {
		return 0;
	}
	return m_dynamics[partindex];
}



//////////////////////////////
//
// HumGrid::getFiguredBassCount --
//

int HumGrid::getFiguredBassCount(int partindex) {
	if ((partindex < 0) || (partindex >= (int)m_figured_bass.size())) {
		return 0;
	}
	return m_figured_bass[partindex];
}



//////////////////////////////
//
// HumGrid::getVerseCount --
//

int HumGrid::getVerseCount(int partindex, int staffindex) {
	if ((partindex < 0) || (partindex >= (int)m_verseCount.size())) {
		return 0;
	}
	int staffnumber = staffindex + 1;
	if ((staffnumber < 1) ||
			(staffnumber >= (int)m_verseCount.at(partindex).size())) {
		return 0;
	}
	int value = m_verseCount.at(partindex).at(staffnumber);
	return value;
}



//////////////////////////////
//
// HumGrid::hasDynamics -- Return true if there are any dyanmics for the part.
//

bool HumGrid::hasDynamics(int partindex) {
	if ((partindex < 0) || (partindex >= (int)m_dynamics.size())) {
		return false;
	}
	return m_dynamics[partindex];
}



//////////////////////////////
//
// HumGrid::hasFiguredBass -- Return true if there is any figured bass for the part.
//

bool HumGrid::hasFiguredBass(int partindex) {
	if ((partindex < 0) || (partindex >= (int)m_figured_bass.size())) {
		return false;
	}
	return m_figured_bass[partindex];
}



//////////////////////////////
//
// HumGrid::setDynamicsPresent -- Indicate that part needs a **dynam spine.
//

void HumGrid::setDynamicsPresent(int partindex) {
	if ((partindex < 0) || (partindex >= (int)m_dynamics.size())) {
		return;
	}
	m_dynamics[partindex] = true;
}



//////////////////////////////
//
// HumGrid::setFiguredBassPresent -- Indicate that part needs a **fb spine.
//

void HumGrid::setFiguredBassPresent(int partindex) {
	if ((partindex < 0) || (partindex >= (int)m_figured_bass.size())) {
		return;
	}
	m_figured_bass[partindex] = true;
}



//////////////////////////////
//
// HumGrid::setHarmonyPresent -- Indicate that part needs a **harm spine.
//

void HumGrid::setHarmonyPresent(int partindex) {
	if ((partindex < 0) || (partindex >= (int)m_harmony.size())) {
		return;
	}
	m_harmony[partindex] = true;
}



//////////////////////////////
//
// HumGrid::setHarmonyCount -- part size hardwired to 100 for now.
//

void HumGrid::setHarmonyCount(int partindex, int count) {
	if ((partindex < 0) || (partindex > (int)m_harmonyCount.size())) {
		return;
	}
	m_harmonyCount[partindex] = count;
}



//////////////////////////////
//
// HumGrid::reportVerseCount --
//

void HumGrid::reportVerseCount(int partindex, int staffindex, int count) {
	if (count <= 0) {
		return;
	}
	int staffnumber = staffindex + 1;
	int partsize = (int)m_verseCount.size();
	if (partindex >= partsize) {
		m_verseCount.resize(partindex+1);
	}
	int staffcount = (int)m_verseCount.at(partindex).size();
	if (staffnumber >= staffcount) {
		m_verseCount.at(partindex).resize(staffnumber+1);
		for (int i=staffcount; i<=staffnumber; i++) {
			m_verseCount.at(partindex).at(i) = 0;
		}
	}
	if (count > m_verseCount.at(partindex).at(staffnumber)) {
		m_verseCount.at(partindex).at(staffnumber) = count;
	}
}



//////////////////////////////
//
// HumGrid::setVerseCount --
//

void HumGrid::setVerseCount(int partindex, int staffindex, int count) {
	if ((partindex < 0) || (partindex > (int)m_verseCount.size())) {
		return;
	}
	int staffnumber = staffindex + 1;
	if (staffnumber < 0) {
		return;
	}
	if (staffnumber < (int)m_verseCount.at(partindex).size()) {
		m_verseCount.at(partindex).at(staffnumber) = count;
	} else {
		int oldsize = (int)m_verseCount.at(partindex).size();
		int newsize = staffnumber + 1;
		m_verseCount.at(partindex).resize(newsize);
		for (int i=oldsize; i<newsize; i++) {
			m_verseCount.at(partindex).at(i) = 0;
		}
		m_verseCount.at(partindex).at(staffnumber) = count;
	}
}



//////////////////////////////
//
// HumGrid::transferTokens --
//   default value: startbarnum = 0.
//

bool HumGrid::transferTokens(HumdrumFile& outfile, int startbarnum) {
	bool status = buildSingleList();
	if (!status) {
		return false;
	}
	calculateGridDurations();
	addNullTokens();
	addInvisibleRestsInFirstTrack();
	addMeasureLines();
	buildSingleList();  // is this needed a second time?
	cleanTempos();
	addLastMeasure();
	if (manipulatorCheck()) {
		cleanupManipulators();
	}

	insertPartNames(outfile);
	insertStaffIndications(outfile);
	insertPartIndications(outfile);
	insertExclusiveInterpretationLine(outfile);
	bool addstartbar = (!hasPickup()) && (!m_musicxmlbarlines);
	for (int m=0; m<(int)this->size(); m++) {
		if (addstartbar && m == 0) {
			status &= at(m)->transferTokens(outfile, m_recip, addstartbar, startbarnum);
		} else {
			status &= at(m)->transferTokens(outfile, m_recip, false);
		}
		if (!status) {
			break;
		}
	}
	insertDataTerminationLine(outfile);
	return true;
}



//////////////////////////////
//
// HumGrid::cleanupManipulators --
//

void HumGrid::cleanupManipulators(void) {
	GridSlice* current = NULL;
	GridSlice* last = NULL;
	vector<GridSlice*> newslices;
	for (int m=0; m<(int)this->size(); m++) {
		for (auto it = this->at(m)->begin(); it != this->at(m)->end(); it++) {
			last = current;
			current = *it;
			if ((*it)->getType() != SliceType::Manipulators) {
				if (last && (last->getType() != SliceType::Manipulators)) {
					matchVoices(current, last);
				}
				continue;
			}
			if (last && (last->getType() != SliceType::Manipulators)) {
				matchVoices(current, last);
			}
			// check to see if manipulator needs to be split into
			// multiple lines.
			newslices.resize(0);
			cleanManipulator(newslices, *it);
			if (newslices.size()) {
				for (int j=0; j<(int)newslices.size(); j++) {
					this->at(m)->insert(it, newslices.at(j));
				}
			}
		}
	}
}



//////////////////////////////
//
// HumGrid::cleanManipulator --
//

void HumGrid::cleanManipulator(vector<GridSlice*>& newslices, GridSlice* curr) {
	newslices.resize(0);
	GridSlice* output;

	// deal with *^ manipulators:
	while ((output = checkManipulatorExpand(curr))) {
		newslices.push_back(output);
	}

	// deal with *v manipulators:
	while ((output = checkManipulatorContract(curr))) {
		newslices.push_back(output);
	}
}



//////////////////////////////
//
// HumGrid::checkManipulatorExpand -- Check for cases where a spine expands
//    into sub-spines.
//

GridSlice* HumGrid::checkManipulatorExpand(GridSlice* curr) {
	GridStaff* staff     = NULL;
	GridPart*  part      = NULL;
	GridVoice* voice     = NULL;
	HTp        token     = NULL;
	bool       neednew   = false;

	int p, s, v;
	int partcount = (int)curr->size();
	int staffcount;

	for (p=0; p<partcount; p++) {
		part = curr->at(p);
		staffcount = (int)part->size();
		for (s=0; s<staffcount; s++) {
			staff = part->at(s);
			for (v=0; v<(int)staff->size(); v++) {
				voice = staff->at(v);
				token = voice->getToken();
				if (token->compare(0, 2, "*^") == 0) {
					if ((token->size() > 2) && isdigit((*token)[2])) {
						neednew = true;
						break;
					}
				}
			}
			if (neednew) {
				break;
			}
		}
		if (neednew) {
			break;
		}
	}

	if (neednew == false) {
		return NULL;
	}

	// need to split *^#'s into separate *^

	GridSlice* newmanip = new GridSlice(curr->getMeasure(), curr->getTimestamp(),
	curr->getType(), curr);

	for (p=0; p<partcount; p++) {
		part = curr->at(p);
		staffcount = (int)part->size();
		for (s=0; s<staffcount; s++) {
			staff = part->at(s);
			adjustExpansionsInStaff(newmanip, curr, p, s);
		}
	}
	return newmanip;
}



//////////////////////////////
//
// HumGrid::adjustExpansionsInStaff -- duplicate null
//   manipulators, and expand large-expansions, such as *^3 into
//   *^ and *^ on the next line, or *^4 into *^ and *^3 on the
//   next line.  The "newmanip" will be placed before curr, so
//

void HumGrid::adjustExpansionsInStaff(GridSlice* newmanip, GridSlice* curr, int p, int s) {
	HTp token = NULL;
	GridVoice* newvoice  = NULL;
	GridVoice* curvoice  = NULL;
	GridStaff* newstaff  = newmanip->at(p)->at(s);
	GridStaff* curstaff  = curr->at(p)->at(s);

	int originalsize = (int)curstaff->size();
	int cv = 0;

	for (int v=0; v<originalsize; v++) {
		curvoice = curstaff->at(cv);
		token = curvoice->getToken();

		if (token->compare(0, 2, "*^") == 0) {
			if ((token->size() > 2) && isdigit((*token)[2])) {
				// transfer *^ to newmanip and replace with * and *^(n-1) in curr
				// Convert *^3 to *^ and add ^* to next line, for example
				// Convert *^4 to *^ and add ^*3 to next line, for example
				int count = 0;
				if (!sscanf(token->c_str(), "*^%d", &count)) {
					cerr << "Error finding expansion number" << endl;
				}
				newstaff->push_back(curvoice);
				curvoice->getToken()->setText("*^");
				newvoice = createVoice("*", "B", 0, p, s);
				curstaff->at(cv) = newvoice;
				if (count <= 3) {
					newvoice = new GridVoice("*^", 0);
				} else {
					newvoice = new GridVoice("*^" + to_string(count-1), 0);
				}
				curstaff->insert(curstaff->begin()+cv+1, newvoice);
				cv++;
				continue;
			} else {
				// transfer *^ to newmanip and replace with two * in curr
				newstaff->push_back(curvoice);
				newvoice = createVoice("*", "C", 0, p, s);
				curstaff->at(cv) = newvoice;
				newvoice = createVoice("*", "D", 0, p, s);
				curstaff->insert(curstaff->begin()+cv, newvoice);
				cv++;
				continue;
			}
		} else {
			// insert * in newmanip
			newvoice = createVoice("*", "E", 0, p, s);
			newstaff->push_back(newvoice);
			cv++;
			continue;
		}

	}
}



//////////////////////////////
//
// HumGrid::checkManipulatorContract -- Will only check for adjacent
//    *v records across adjacent staves, which should be good enough.
//    Will not check within a staff, but this should not occur within
//    MusicXML input data due to the way it is being processed.
//    The return value is a newly created GridSlice pointer which contains
//    a new manipulator to add to the file (and the current manipultor
//    slice will also be modified if the return value is not NULL).
//

GridSlice* HumGrid::checkManipulatorContract(GridSlice* curr) {
	GridVoice* lastvoice = NULL;
	GridVoice* voice     = NULL;
	GridStaff* staff     = NULL;
	GridPart*  part      = NULL;
	bool       neednew   = false;

	int p, s;
	int partcount = (int)curr->size();
	int staffcount;
	bool init = false;
	for (p=partcount-1; p>=0; p--) {
		part  = curr->at(p);
		staffcount = (int)part->size();
		for (s=staffcount-1; s>=0; s--) {
			staff = part->at(s);
			if (staff->empty()) {
				continue;
			}
			voice = staff->back();
			if (!init) {
				lastvoice = staff->back();
				init = true;
				continue;
			}
			if (lastvoice != NULL) {
           	if ((*voice->getToken() == "*v") &&
						(*lastvoice->getToken() == "*v")) {
					neednew = true;
					break;
				}
			}
			lastvoice = staff->back();
		}
		if (neednew) {
			break;
		}
	}

	if (neednew == false) {
		return NULL;
	}

	// need to split *v's from different adjacent staves onto separate lines.
	GridSlice* newmanip = new GridSlice(curr->getMeasure(), curr->getTimestamp(),
		curr->getType(), curr);
	lastvoice = NULL;
	GridStaff* laststaff    = NULL;
	GridStaff* newstaff     = NULL;
	GridStaff* newlaststaff = NULL;
	bool foundnew = false;
	partcount = (int)curr->size();
	int lastp = 0;
	int lasts = 0;
	int partsplit = -1;
	int voicecount;

	for (p=partcount-1; p>=0; p--) {
		part  = curr->at(p);
		staffcount = (int)part->size();
		for (s=staffcount-1; s>=0; s--) {
			staff = part->at(s);
			voicecount = (int)staff->size();
			voice = staff->back();
			newstaff = newmanip->at(p)->at(s);
			if (lastvoice != NULL) {
           	if ((*voice->getToken() == "*v") &&
						(*lastvoice->getToken() == "*v")) {
               // splitting the slices at this staff boundary

					newlaststaff = newmanip->at(lastp)->at(lasts);
					transferMerges(staff, laststaff, newstaff, newlaststaff, p, s);
					foundnew = true;
					partsplit = p;
					break;
				}
			} else {
				if (voicecount > 1) {
					for (int j=(int)newstaff->size(); j<voicecount; j++) {
						// GridVoice* vdata = createVoice("*", "F", 0, p, s);
						// newstaff->push_back(vdata);
					}
				}
			}
			laststaff = staff;
			lastvoice = laststaff->back();
			lastp = p;
			lasts = s;
		}

		if (foundnew) {
			// transfer all of the subsequent manipulators forward
			// after the staff/newstaff point in the slice
			if (partsplit > 0) {
				transferOtherParts(curr, newmanip, partsplit);
			}
			break;
		}
	}

	// fill in any missing voice null interpretation tokens
	adjustVoices(curr, newmanip, partsplit);

	return newmanip;
}



//////////////////////////////
//
// HumGrid::adjustVoices --
//

void HumGrid::adjustVoices(GridSlice* curr, GridSlice* newmanip, int partsplit) {
	int p1count = (int)curr->size();
	// int p2count = (int)newmanip->size();
	//cerr << "PARTSPLIT " << partsplit << endl;
	for (int p=0; p<p1count; p++) {
		int s1count = (int)curr->at(p)->size();
		// int s2count = (int)curr->at(p)->size();
		// cerr << "\tCURR STAVES " << s1count << "\tNEWM STAVES " << s2count << endl;
		// cerr << "\t\tCURR SCOUNT = " << curr->at(p)->size() << "\tNEWM SCOUNT = " << newmanip->at(p)->size() << endl;
		for (int s=0; s<s1count; s++) {
			GridStaff* s1 = curr->at(p)->at(s);
			GridStaff* s2 = newmanip->at(p)->at(s);
			if ((s1->size() == 0) && (s2->size() > 0)) {
				createMatchedVoiceCount(s1, s2, p, s);
			} else if ((s2->size() == 0) && (s1->size() > 0)) {
				createMatchedVoiceCount(s2, s1, p, s);
			}
			// cerr << "\t\t\tCURR VCOUNT = " << curr->at(p)->at(s)->size() << "\t(" << curr->at(p)->at(s)->getString() << ")" << "\t";
			// cerr << "\tNEWM VCOUNT = " << newmanip->at(p)->at(s)->size() << "\t(" << newmanip->at(p)->at(s)->getString() << ")" << endl;
		}
	}
}



//////////////////////////////
//
// HumGrid::createMatchedVoiceCount --
//

void HumGrid::createMatchedVoiceCount(GridStaff* snew, GridStaff* sold, int p, int s) {
	if (snew->size() != 0) {
		// this function is only for creating a totally new
		return;
	}
	int count = (int)sold->size();
	snew->resize(count);
	for (int i=0; i<count; i++) {
		GridVoice* gv = createVoice("*", "N", p, s, i);
		snew->at(i) = gv;
	}
}



//////////////////////////////
//
// HumGrid::matchVoices --
//

void HumGrid::matchVoices(GridSlice* current, GridSlice* last) {
	if (current == NULL) {
		return;
	}
	if (last == NULL) {
		return;
	}
	int pcount1 = (int)current->size();
	int pcount2 = (int)current->size();
	if (pcount1 != pcount2) {
		return;
	}
	for (int i=0; i<pcount1; i++) {
		GridPart* part1 = current->at(i);
		GridPart* part2 = current->at(i);
		int scount1 = (int)part1->size();
		int scount2 = (int)part2->size();
		if (scount1 != scount2) {
			continue;
		}
		for (int j=0; j<scount1; j++) {
			GridStaff* staff1 = part1->at(j);
			GridStaff* staff2 = part2->at(j);
			int vcount1 = (int)staff1->size();
			int vcount2 = (int)staff2->size();
			if (vcount1 == vcount2) {
				continue;
			}
			if (vcount2 > vcount1) {
				// strange if it happens
				continue;
			}
			int difference = vcount1 - vcount2;
			for (int k=0; k<difference; k++) {
				GridVoice* gv = createVoice("*", "A", 0, i, j);
				staff2->push_back(gv);
			}
		}
	}
}



//////////////////////////////
//
// HumGrid::transferOtherParts -- after a line split due to merges
//    occurring at the same time.
//

void HumGrid::transferOtherParts(GridSlice* oldline, GridSlice* newline, int maxpart) {
	GridPart* temp;
	int partcount = (int)oldline->size();
	if (maxpart >= partcount) {
		return;
	}
	for (int i=0; i<maxpart; i++) {
		temp = oldline->at(i);
		oldline->at(i) = newline->at(i);
		newline->at(i) = temp;
		// duplicate the voice counts on the old line (needed if there are more
		// than one voices in a staff when splitting a line due to *v merging.
		for (int j=0; j<(int)oldline->at(i)->size(); j++) {
			int voices = (int)newline->at(i)->at(j)->size();
			int adjustment = 0;
			for (int k=0; k<voices; k++) {
				if (!newline->at(i)->at(j)->at(k)) {
					continue;
				}
				HTp tok = newline->at(i)->at(j)->at(k)->getToken();
				if (*tok == "*v") {
					adjustment++;
				}
			}
			if (adjustment > 0) {
				adjustment--;
			}
			voices -= adjustment;
			oldline->at(i)->at(j)->resize(voices);
			for (int k=0; k<voices; k++) {
				oldline->at(i)->at(j)->at(k) = createVoice("*", "Z", 0, i, j);
			}
		}
	}

	for (int p=0; p<(int)newline->size(); p++) {
			GridPart* newpart = newline->at(p);
			GridPart* oldpart = oldline->at(p);
		for (int s=0; s<(int)newpart->size(); s++) {
			GridStaff* newstaff = newpart->at(s);
			GridStaff* oldstaff = oldpart->at(s);
			if (newstaff->size() >= oldstaff->size()) {
				continue;
			}
			int diff = (int)(oldstaff->size() - newstaff->size());

			for (int v=0; v<diff; v++) {
				GridVoice* voice = createVoice("*", "G", 0, p, s);
				newstaff->push_back(voice);
			}

		}
	}
}



//////////////////////////////
//
// HumGrid::transferMerges -- Move *v spines from one staff to last staff,
//   and re-adjust staff "*v" tokens to a single "*" token.
// Example:
//                 laststaff      staff
// old:            *v   *v        *v   *v
// converts to:
// new:            *v   *v        *    *
// old:            *              *v   *v
//
//
//

void HumGrid::transferMerges(GridStaff* oldstaff, GridStaff* oldlaststaff,
		GridStaff* newstaff, GridStaff* newlaststaff, int pindex, int sindex) {
	if ((oldstaff == NULL) || (oldlaststaff == NULL)) {
		cerr << "Weird error in HumGrid::transferMerges()" << endl;
		return;
	}
	// New staves are presumed to be totally empty.

	GridVoice* gv;

	// First create "*" tokens for newstaff slice where there are
	// "*v" in old staff.  All other tokens should be set to "*".
	int tcount = (int)oldstaff->size();
	int t;
	for (t=0; t<tcount; t++) {
		if (*oldstaff->at(t)->getToken() == "*v") {
			gv = createVoice("*", "H", 0, pindex, sindex);
			newstaff->push_back(gv);
		} else {
			gv = createVoice("*", "I", 0, pindex, sindex);
			newstaff->push_back(gv);
		}
	}

	// Next, all "*v" tokens at end of old previous staff should be
	// transferred to the new previous staff and replaced with
	// a single "*" token.  Non "*v" tokens in the old last staff should
	// be converted to "*" tokens in the new last staff.
	//
	// It may be possible for *v tokens to not be only at the end of
	// the list of oldlaststaff tokens, but does not seem possible.

	tcount = (int)oldlaststaff->size();
	bool addednull = false;
	for (t=0; t<tcount; t++) {
		if (*oldlaststaff->at(t)->getToken() == "*v") {
			newlaststaff->push_back(oldlaststaff->at(t));
			if (addednull == false) {
				gv = createVoice("*", "J", 0, pindex, sindex);
				oldlaststaff->at(t) = gv;
				addednull = true;
			} else {
				oldlaststaff->at(t) = NULL;
			}
		} else {
			gv = createVoice("*", "K", 0, pindex, sindex);
			newlaststaff->push_back(gv);
		}
	}

	// Go back to the oldlaststaff and chop off all ending NULLs
	// * it should never get to zero (there should be at least one "*" left.
	// In theory intermediate NULLs should be checked for, and if they
	// exist, then something bad will happen.  But it does not seem
	// possible to have intermediate NULLs.
	tcount = (int)oldlaststaff->size();
	for (t=tcount-1; t>=0; t--) {
		if (oldlaststaff->at(t) == NULL) {
			int newsize = (int)oldlaststaff->size() - 1;
			oldlaststaff->resize(newsize);
		}
	}
}



//////////////////////////////
//
// HumGrid::createVoice -- create voice with given token contents.
//

GridVoice* HumGrid::createVoice(const string& tok, const string& post, HumNum duration, int pindex, int sindex) {
	//std::string token = tok;
	//token += ":" + post + ":" + to_string(pindex) + "," + to_string(sindex);
	GridVoice* gv = new GridVoice(tok.c_str(), 0);
	return gv;
}



//////////////////////////////
//
// HumGrid::getNextSpinedLine -- Find next spined GridSlice.
//

GridSlice* HumGrid::getNextSpinedLine(const GridMeasure::iterator& it, int measureindex) {
	auto nextone = it;
	nextone++;
	while (nextone != this->at(measureindex)->end()) {
		if ((*nextone)->hasSpines()) {
			break;
		}
		nextone++;
	}

	if (nextone != this->at(measureindex)->end()) {
		return *nextone;
	}

	measureindex++;
	if (measureindex >= (int)this->size()) {
		// end of data, so nothing to adjust with
		// but this should never happen in general.
		return NULL;
	}
	nextone = this->at(measureindex)->begin();
	while (nextone != this->at(measureindex)->end()) {
		if ((*nextone)->hasSpines()) {
			return *nextone;
		}
		nextone++;
	}

	return NULL;
}



//////////////////////////////
//
// HumGrid::manipulatorCheck --
//

bool HumGrid::manipulatorCheck(void) {
	GridSlice* manipulator;
	int m;
	GridSlice* s1;
	GridSlice* s2;
	bool output = false;
	for (m=0; m<(int)this->size(); m++) {
		if (this->at(m)->size() == 0) {
			continue;
		}
		for (auto it = this->at(m)->begin(); it != this->at(m)->end(); it++) {
			if (!(*it)->hasSpines()) {
				// Don't monitor manipulators on no-spined lines.
				continue;
			}
			s1 = *it;
			s2 = getNextSpinedLine(it, m);

			manipulator = manipulatorCheck(s1, s2);
			if (manipulator == NULL) {
				continue;
			}
			output = true;
			auto inserter = it;
			inserter++;
			this->at(m)->insert(inserter, manipulator);
			it++; // skip over the new manipulator line (expand it later)
		}
	}
	return output;
}


//
// HumGrid::manipulatorCheck -- Look for differences in voice/layer count
//   for each part/staff pairing between adjacent lines.  If they do not match,
//   then add spine manipulator line to Grid between the two lines.
//

GridSlice* HumGrid::manipulatorCheck(GridSlice* ice1, GridSlice* ice2) {
	int p1count;
	int p2count;
	int s1count;
	int s2count;
	int v1count;
	int v2count;
	int p;
	int s;
	int v;
	bool needmanip = false;

	if (ice1 == NULL) {
		return NULL;
	}
	if (ice2 == NULL) {
		return NULL;
	}
	if (!ice1->hasSpines()) {
		return NULL;
	}
	if (!ice2->hasSpines()) {
		return NULL;
	}
	p1count = (int)ice1->size();
	p2count = (int)ice2->size();
	if (p1count != p2count) {
		cerr << "Warning: Something weird happend here" << endl;
		cerr << "p1count = " << p1count << endl;
		cerr << "p2count = " << p2count << endl;
		cerr << "ICE1: " << ice1 << endl;
		cerr << "ICE2: " << ice2 << endl;
		cerr << "The above two values should be the same." << endl;
		return NULL;
	}
	for (p=0; p<p1count; p++) {
		s1count = (int)ice1->at(p)->size();
		s2count = (int)ice2->at(p)->size();
		if (s1count != s2count) {
			cerr << "Warning: Something weird happend here with staff" << endl;
			return NULL;
		}
		for (s=0; s<s1count; s++) {
			v1count = (int)ice1->at(p)->at(s)->size();
			// the voice count always must be at least 1.  This case
			// is related to inserting clefs in other parts.
			if (v1count < 1) {
				v1count = 1;
			}
			v2count = (int)ice2->at(p)->at(s)->size();
			if (v2count < 1) {
				v2count = 1;
			}
			if (v1count == v2count) {
				continue;
			}
			needmanip = true;
			break;
		}
		if (needmanip) {
			break;
		}
	}

	if (!needmanip) {
		return NULL;
	}

	// build manipulator line (which will be expanded further if adjacent
	// staves have *v manipulators.

	GridSlice* mslice;
	mslice = new GridSlice(ice1->getMeasure(), ice2->getTimestamp(),
			SliceType::Manipulators);

	int z;
	HTp token;
	GridVoice* gv;
	p1count = (int)ice1->size();
	mslice->resize(p1count);
	for (p=0; p<p1count; p++) {
		mslice->at(p) = new GridPart;
		s1count = (int)ice1->at(p)->size();
		mslice->at(p)->resize(s1count);
		for (s=0; s<s1count; s++) {
			mslice->at(p)->at(s) = new GridStaff;
			v1count = (int)ice1->at(p)->at(s)->size();
			v2count = (int)ice2->at(p)->at(s)->size();
			if (v2count < 1) {
				// empty spines will be filled in with at least one null token.
				v2count = 1;
			}
			if (v1count < 1) {
				// empty spines will be filled in with at least one null token.
				v1count = 1;
			}
			if ((v1count == 0) && (v2count == 1)) {
				// grace note at the start of the measure in another voice
				// no longer can get here due to v1count min being 1.
				token = createHumdrumToken("*", p, s);
				gv = new GridVoice(token, 0);
				mslice->at(p)->at(s)->push_back(gv);
			} else if (v1count == v2count) {
				for (v=0; v<v1count; v++) {
					token = createHumdrumToken("*", p, s);
					gv = new GridVoice(token, 0);
					mslice->at(p)->at(s)->push_back(gv);
				}
			} else if (v1count < v2count) {
				// need to grow
				int grow = v2count - v1count;
				// if (grow == 2 * v1count) {
				if (v2count == 2 * v1count) {
					// all subspines split
					for (z=0; z<v1count; z++) {
						token = new HumdrumToken("*^");
						gv = new GridVoice(token, 0);
						mslice->at(p)->at(s)->push_back(gv);
					}
				} else if ((v1count > 0) && (grow > 2 * v1count)) {
					// too large to split all at the same time, deal with later
					for (z=0; z<v1count-1; z++) {
						token = new HumdrumToken("*^");
						gv = new GridVoice(token, 0);
						mslice->at(p)->at(s)->push_back(gv);
					}
					int extra = v2count - (v1count - 1) * 2;
					if (extra > 2) {
						token = new HumdrumToken("*^" + to_string(extra));
					} else {
						token = new HumdrumToken("*^");
					}
					gv = new GridVoice(token, 0);
					mslice->at(p)->at(s)->push_back(gv);
				} else {
					// only split spines at end of list
					int doubled = v2count - v1count;
					int notdoubled = v1count - doubled;
					for (z=0; z<notdoubled; z++) {
						token = createHumdrumToken("*", p, s);
						gv = new GridVoice(token, 0);
						mslice->at(p)->at(s)->push_back(gv);
					}
					//for (z=0; z<doubled; z++) {
						if (doubled > 1) {
							token = new HumdrumToken("*^" + to_string(doubled+1));
						} else {
							token = new HumdrumToken("*^");
						}
						// token = new HumdrumToken("*^");
						gv = new GridVoice(token, 0);
						mslice->at(p)->at(s)->push_back(gv);
					//}
				}
			} else if (v1count > v2count) {
				// need to shrink
				int shrink = v1count - v2count + 1;
				int notshrink = v1count - shrink;
				for (z=0; z<notshrink; z++) {
					token = createHumdrumToken("*", p, s);
					gv = new GridVoice(token, 0);
					mslice->at(p)->at(s)->push_back(gv);
				}
				for (z=0; z<shrink; z++) {
					token = new HumdrumToken("*v");
					gv = new GridVoice(token, 0);
					mslice->at(p)->at(s)->push_back(gv);
				}
			}
		}
	}
	return mslice;
}



//////////////////////////////
//
// HumGrid::createHumdrumToken --
//

HTp HumGrid::createHumdrumToken(const string& tok, int pindex, int sindex) {
	std::string token = tok;
	// token += ":" + to_string(pindex) + "," + to_string(sindex);
	HTp output = new HumdrumToken(token.c_str());
	return output;
}



//////////////////////////////
//
// HumGrid::addMeasureLines --
//

void HumGrid::addMeasureLines(void) {
	HumNum timestamp;
	GridSlice* mslice;
	GridSlice* endslice;
	GridPart* part;
	GridStaff* staff;
	GridVoice* gv;
	string token;
	int staffcount, partcount, vcount, nextvcount, lcount;
	GridMeasure* measure = NULL;
	GridMeasure* nextmeasure = NULL;

	vector<int> barnums;
	if (!m_musicxmlbarlines) {
		getMetricBarNumbers(barnums);
	}

	for (int m=0; m<(int)this->size()-1; m++) {
		measure = this->at(m);
		nextmeasure = this->at(m+1);
		if (nextmeasure->size() == 0) {
			// next measure is empty for some reason so give up
			continue;
		}
		GridSlice* firstspined = nextmeasure->getFirstSpinedSlice();
		timestamp = firstspined->getTimestamp();
		if (measure->size() == 0) {
			continue;
		}

		if (measure->getDuration() == 0) {
			continue;
		}
		mslice = new GridSlice(measure, timestamp, SliceType::Measures);
		// what to do when endslice is NULL?
		endslice = measure->getLastSpinedSlice(); // this has to come before next line
		measure->push_back(mslice); // this has to come after the previous line
		partcount = (int)firstspined->size();
		mslice->resize(partcount);

		for (int p=0; p<partcount; p++) {
			part = new GridPart();
			mslice->at(p) = part;
			staffcount = (int)firstspined->at(p)->size();
			mslice->at(p)->resize(staffcount);
			for (int s=0; s<(int)staffcount; s++) {
				staff = new GridStaff;
				mslice->at(p)->at(s) = staff;

				// insert the minimum number of barlines based on the
				// voices in the current and next measure.
				vcount = (int)endslice->at(p)->at(s)->size();
				if (firstspined) {
					nextvcount = (int)firstspined->at(p)->at(s)->size();
				} else {
					// perhaps an empty measure?  This will cause problems.
					nextvcount = 0;
				}
				lcount = vcount;
				if (lcount > nextvcount) {
					lcount = nextvcount;
				}
				if (lcount == 0) {
					lcount = 1;
				}
				for (int v=0; v<lcount; v++) {
					int num = measure->getMeasureNumber();
					if (m < (int)barnums.size() - 1) {
						num = barnums[m+1];
					}
					token = createBarToken(m, num, measure);
					gv = new GridVoice(token, 0);
					mslice->at(p)->at(s)->push_back(gv);
				}
			}
		}
	}
}



//////////////////////////////
//
// HumGrid::createBarToken --
//

string HumGrid::createBarToken(int m, int barnum, GridMeasure* measure) {
	string token;
	string barstyle = getBarStyle(measure);
	string number = "";
	if (barnum > 0) {
		number = to_string(barnum);
	}
	if (m_musicxmlbarlines) {
		// m+1 because of the measure number
		// comes from the previous measure.
		if (barstyle == "=") {
			token = "==";
			token += to_string(m+1);
		} else {
			token = "=";
			token += to_string(m+1);
			token += barstyle;
		}
	} else {
		if (barnum > 0) {
			if (barstyle == "=") {
				token = "==";
				token += number;
			} else {
				token = "=";
				token += number;
				token += barstyle;
			}
		} else {
			if (barstyle == "=") {
				token = "==";
			} else {
				token = "=";
				token += barstyle;
			}
		}
	}
	return token;
}



//////////////////////////////
//
// HumGrid::getMetricBarNumbers --
//

void HumGrid::getMetricBarNumbers(vector<int>& barnums) {
	int mcount = (int)this->size();
	barnums.resize(mcount);

	if (mcount == 0) {
		return;
	}

	vector<HumNum> mdur(mcount);
	vector<HumNum> tsdur(mcount); // time signature duration

	for (int m=0; m<(int)this->size(); m++) {
		mdur[m]   = this->at(m)->getDuration();
		tsdur[m] = this->at(m)->getTimeSigDur();
		if (tsdur[m] <= 0) {
			tsdur[m] = mdur[m];
		}
	}

	int start = 0;
	if (!mdur.empty()) {
		if (mdur[0] == 0) {
			start = 1;
		}
	}

	// int counter = 1;  // this was causing a problem https://github.com/humdrum-tools/verovio-humdrum-viewer/issues/254
	int counter = 0;
	if (mdur[start] == tsdur[start]) {
		m_pickup = false;
		counter++;
		// add the initial barline later when creating HumdrumFile.
	} else {
		m_pickup = true;
	}

	for (int m=start; m<(int)this->size(); m++) {
		if ((m == start) && (mdur[m] == 0)) {
			barnums[m] = counter-1;
			continue;
		} else if (mdur[m] == 0) {
			barnums[m] = -1;
			continue;
		}
		if ((m < mcount-1) && (tsdur[m] == tsdur[m+1])) {
			if (mdur[m] + mdur[m+1] == tsdur[m]) {
				barnums[m] = -1;
			} else {
				barnums[m] = counter++;
			}
		} else {
			barnums[m] = counter++;
		}
	}
}



//////////////////////////////
//
// HumGrid::getBarStyle --
//

string HumGrid::getBarStyle(GridMeasure* measure) {
	string output = "";
	if (measure->isDouble()) {
		output = "||";
	} else if (measure->isFinal()) {
		output = "=";
	} else if (measure->isInvisibleBarline()) {
		output = "-";
	} else if (measure->isRepeatBoth()) {
		output = ":|!|:";
	} else if (measure->isRepeatBackward()) {
		output = ":|!";
	} else if (measure->isRepeatForward()) {
		output = "!|:";
	}
	return output;
}



//////////////////////////////
//
// HumGrid::addLastMeasure --
//

void HumGrid::addLastMeasure(void) {
   // add the last measure, which will be only one voice
	// for each part/staff.
	GridSlice* model = this->back()->back();
	if (model == NULL) {
		return;
	}

	// probably not the correct timestamp, but probably not important
	// to get correct:
	HumNum timestamp = model->getTimestamp();

	if (this->empty()) {
		return;
	}
	GridMeasure* measure = this->back();

	string barstyle = getBarStyle(measure);

	GridSlice* mslice = new GridSlice(model->getMeasure(), timestamp,
			SliceType::Measures);
	this->back()->push_back(mslice);
	mslice->setTimestamp(timestamp);
	int partcount = (int)model->size();
	mslice->resize(partcount);
	for (int p=0; p<partcount; p++) {
		GridPart* part = new GridPart();
		mslice->at(p) = part;
		int staffcount = (int)model->at(p)->size();
		mslice->at(p)->resize(staffcount);
		for (int s=0; s<staffcount; s++) {
			GridStaff* staff = new GridStaff;
			mslice->at(p)->at(s) = staff;
			HTp token = new HumdrumToken("=" + barstyle);
			GridVoice* gv = new GridVoice(token, 0);
			mslice->at(p)->at(s)->push_back(gv);
		}
	}
}



//////////////////////////////
//
// HumGrid::buildSingleList --
//

bool HumGrid::buildSingleList(void) {
	m_allslices.resize(0);

	int gridcount = 0;
	for (auto it : (vector<GridMeasure*>)*this) {
		gridcount += (int)it->size();
	}
	m_allslices.reserve(gridcount + 100);
	for (int m=0; m<(int)this->size(); m++) {
		for (auto it : (list<GridSlice*>)*this->at(m)) {
			m_allslices.push_back(it);
		}
	}

	HumNum ts1;
	HumNum ts2;
	HumNum dur;
	for (int i=0; i<(int)m_allslices.size() - 1; i++) {
		ts1 = m_allslices[i]->getTimestamp();
		ts2 = m_allslices[i+1]->getTimestamp();
		dur = (ts2 - ts1); // whole-note units
		m_allslices[i]->setDuration(dur);
	}
	return !m_allslices.empty();
}



//////////////////////////////
//
// HumGrid::addNullTokensForGraceNotes -- Avoid grace notes at
//     starts of measures from contracting the subspine count.
//

void HumGrid::addNullTokensForGraceNotes(void) {
	// add null tokens for grace notes in other voices
	GridSlice *lastnote = NULL;
	GridSlice *nextnote = NULL;
	for (int i=0; i<(int)m_allslices.size(); i++) {
		if (!m_allslices[i]->isGraceSlice()) {
			continue;
		}
		// cerr << "PROCESSING " << m_allslices[i] << endl;
		lastnote = NULL;
		nextnote = NULL;

		for (int j=i+1; j<(int)m_allslices.size(); j++) {
			if (m_allslices[j]->isNoteSlice()) {
				nextnote = m_allslices[j];
				break;
			}
		}
		if (nextnote == NULL) {
			continue;
		}

		for (int j=i-1; j>=0; j--) {
			if (m_allslices[j]->isNoteSlice()) {
				lastnote = m_allslices[j];
				break;
			}
		}
		if (lastnote == NULL) {
			continue;
		}

		fillInNullTokensForGraceNotes(m_allslices[i], lastnote, nextnote);
	}
}



//////////////////////////////
//
// HumGrid::addNullTokensForLayoutComments -- Avoid layout in multi-subspine
//     regions from contracting to a single spine.
//

void HumGrid::addNullTokensForLayoutComments(void) {
	// add null tokens for key changes in other voices
	GridSlice *lastnote = NULL;
	GridSlice *nextnote = NULL;
	for (int i=0; i<(int)m_allslices.size(); i++) {
		if (!m_allslices[i]->isLocalLayoutSlice()) {
			continue;
		}
		// cerr << "PROCESSING " << m_allslices[i] << endl;
		lastnote = NULL;
		nextnote = NULL;

		for (int j=i+1; j<(int)m_allslices.size(); j++) {
			if (m_allslices[j]->isNoteSlice()) {
				nextnote = m_allslices[j];
				break;
			}
		}
		if (nextnote == NULL) {
			continue;
		}

		for (int j=i-1; j>=0; j--) {
			if (m_allslices[j]->isNoteSlice()) {
				lastnote = m_allslices[j];
				break;
			}
		}
		if (lastnote == NULL) {
			continue;
		}

		fillInNullTokensForLayoutComments(m_allslices[i], lastnote, nextnote);
	}
}



//////////////////////////////
//
// HumGrid::addNullTokensForClefChanges -- Avoid clef in multi-subspine
//     regions from contracting to a single spine.
//

void HumGrid::addNullTokensForClefChanges(void) {
	// add null tokens for clef changes in other voices
	GridSlice *lastnote = NULL;
	GridSlice *nextnote = NULL;
	for (int i=0; i<(int)m_allslices.size(); i++) {
		if (!m_allslices[i]->isClefSlice()) {
			continue;
		}
		// cerr << "PROCESSING " << m_allslices[i] << endl;
		lastnote = NULL;
		nextnote = NULL;

		for (int j=i+1; j<(int)m_allslices.size(); j++) {
			if (m_allslices[j]->isNoteSlice()) {
				nextnote = m_allslices[j];
				break;
			}
		}
		if (nextnote == NULL) {
			continue;
		}

		for (int j=i-1; j>=0; j--) {
			if (m_allslices[j]->isNoteSlice()) {
				lastnote = m_allslices[j];
				break;
			}
		}
		if (lastnote == NULL) {
			continue;
		}

		fillInNullTokensForClefChanges(m_allslices[i], lastnote, nextnote);
	}
}



//////////////////////////////
//
// HumGrid::fillInNullTokensForClefChanges --
//

void HumGrid::fillInNullTokensForClefChanges(GridSlice* clefslice,
		GridSlice* lastnote, GridSlice* nextnote) {

	if (clefslice == NULL) { return; }
	if (lastnote == NULL)  { return; }
	if (nextnote == NULL)  { return; }

	// cerr << "CHECKING CLEF SLICE: " << endl;
	// cerr << "\tclef\t" << clefslice << endl;
	// cerr << "\tlast\t" << lastnote << endl;
	// cerr << "\tnext\t" << nextnote << endl;

	int partcount = (int)clefslice->size();

	for (int p=0; p<partcount; p++) {
		int staffcount = (int)lastnote->at(p)->size();
		for (int s=0; s<staffcount; s++) {
			int v1count = (int)lastnote->at(p)->at(s)->size();
			int v2count = (int)nextnote->at(p)->at(s)->size();
			int vgcount = (int)clefslice->at(p)->at(s)->size();
			// if (vgcount < 1) {
			// 	vgcount = 1;
			// }
			if (v1count < 1) {
				v1count = 1;
			}
			if (v2count < 1) {
				v2count = 1;
			}
			// cerr << "p=" << p << "\ts=" << s << "\tv1count = " << v1count;
			// cerr << "\tv2count = " << v2count;
			// cerr << "\tvgcount = " << vgcount << endl;
			if (v1count != v2count) {
				// Note slices are expanding or contracting so do
				// not try to adjust clef slice between them.
				continue;
			}
			if (vgcount == v1count) {
				// Grace note slice does not need to be adjusted.
			}
			int diff = v1count - vgcount;
			// fill in a null for each empty slot in voice
			for (int i=0; i<diff; i++) {
				GridVoice* gv = createVoice("*", "P", 0, p, s);
				clefslice->at(p)->at(s)->push_back(gv);
			}
		}
	}
}



//////////////////////////////
//
// HumGrid::fillInNullTokensForLayoutComments --
//

void HumGrid::fillInNullTokensForLayoutComments(GridSlice* layoutslice,
		GridSlice* lastnote, GridSlice* nextnote) {

	if (layoutslice == NULL) { return; }
	if (lastnote == NULL)    { return; }
	if (nextnote == NULL)    { return; }

	// cerr << "CHECKING CLEF SLICE: " << endl;
	// cerr << "\tclef\t" << layoutslice << endl;
	// cerr << "\tlast\t" << lastnote << endl;
	// cerr << "\tnext\t" << nextnote << endl;

	int partcount = (int)layoutslice->size();
	int staffcount;
	int vgcount;
	int v1count;
	int v2count;

	for (int p=0; p<partcount; p++) {
		staffcount = (int)lastnote->at(p)->size();
		for (int s=0; s<staffcount; s++) {
			v1count = (int)lastnote->at(p)->at(s)->size();
			v2count = (int)nextnote->at(p)->at(s)->size();
			vgcount = (int)layoutslice->at(p)->at(s)->size();
			// if (vgcount < 1) {
			// 	vgcount = 1;
			// }
			if (v1count < 1) {
				v1count = 1;
			}
			if (v2count < 1) {
				v2count = 1;
			}
			// cerr << "p=" << p << "\ts=" << s << "\tv1count = " << v1count;
			// cerr << "\tv2count = " << v2count;
			// cerr << "\tvgcount = " << vgcount << endl;
			if (v1count != v2count) {
				// Note slices are expanding or contracting so do
				// not try to adjust clef slice between them.
				continue;
			}
			if (vgcount == v1count) {
				// Grace note slice does not need to be adjusted.
			}
			int diff = v1count - vgcount;
			// fill in a null for each empty slot in voice
			for (int i=0; i<diff; i++) {
				GridVoice* gv = new GridVoice("!", 0);
				layoutslice->at(p)->at(s)->push_back(gv);
			}
		}
	}
}



//////////////////////////////
//
// HumGrid::fillInNullTokensForGraceNotes --
//

void HumGrid::fillInNullTokensForGraceNotes(GridSlice* graceslice, GridSlice* lastnote,
		GridSlice* nextnote) {

	if (graceslice == NULL) {
		return;
	}
	if (lastnote == NULL) {
		return;
	}
	if (nextnote == NULL) {
		return;
	}

	// cerr << "CHECKING GRACE SLICE: " << endl;
	// cerr << "\tgrace\t" << graceslice << endl;
	// cerr << "\tlast\t" << lastnote << endl;
	// cerr << "\tnext\t" << nextnote << endl;

	int partcount = (int)graceslice->size();
	int staffcount;
	int vgcount;
	int v1count;
	int v2count;

	for (int p=0; p<partcount; p++) {
		staffcount = (int)lastnote->at(p)->size();
		for (int s=0; s<staffcount; s++) {
			v1count = (int)lastnote->at(p)->at(s)->size();
			v2count = (int)nextnote->at(p)->at(s)->size();
			vgcount = (int)graceslice->at(p)->at(s)->size();
			// if (vgcount < 1) {
			// 	vgcount = 1;
			// }
			if (v1count < 1) {
				v1count = 1;
			}
			if (v2count < 1) {
				v2count = 1;
			}
			// cerr << "p=" << p << "\ts=" << s << "\tv1count = " << v1count;
			// cerr << "\tv2count = " << v2count;
			// cerr << "\tvgcount = " << vgcount << endl;
			if (v1count != v2count) {
				// Note slices are expanding or contracting so do
				// not try to adjust grace slice between them.
				continue;
			}
			if (vgcount == v1count) {
				// Grace note slice does not need to be adjusted.
			}
			int diff = v1count - vgcount;
			// fill in a null for each empty slot in voice
			for (int i=0; i<diff; i++) {
				GridVoice* gv = new GridVoice(".", 0);
				graceslice->at(p)->at(s)->push_back(gv);
			}
		}
	}
}



//////////////////////////////
//
// HumGrid::addNullTokens --
//

void HumGrid::addNullTokens(void) {
	int i; // slice index
	int p; // part index
	int s; // staff index
	int v; // voice index

	if ((0)) {
		cerr << "SLICE TIMESTAMPS: " << endl;
		for (int x=0; x<(int)m_allslices.size(); x++) {
			cerr << "\tTIMESTAMP " << x << "= "
			     << m_allslices[x]->getTimestamp()
			     << "\tDUR=" << m_allslices[x]->getDuration()
			     << "\t"
			     << m_allslices[x]
			     << endl;
		}
	}

	for (i=0; i<(int)m_allslices.size(); i++) {

		GridSlice& slice = *m_allslices.at(i);
		if (!slice.isNoteSlice()) {
			// probably need to deal with grace note slices here
			continue;
		}
      for (p=0; p<(int)slice.size(); p++) {
			GridPart& part = *slice.at(p);
      	for (s=0; s<(int)part.size(); s++) {
				GridStaff& staff = *part.at(s);
      		for (v=0; v<(int)staff.size(); v++) {
					if (!staff.at(v)) {
						// in theory should not happen
						continue;
					}
					GridVoice& gv = *staff.at(v);
					if (gv.isNull()) {
						continue;
					}
					// found a note/rest which should have a non-zero
					// duration that needs to be extended to the next
					// duration in the
					extendDurationToken(i, p, s, v);
				}
			}
		}

	}

	addNullTokensForGraceNotes();
	adjustClefChanges();
	addNullTokensForClefChanges();
	addNullTokensForLayoutComments();
	checkForNullDataHoles();
}



//////////////////////////////
//
// HumGrid::checkForNullData -- identify any spots in the grid which are NULL
//     pointers and allocate invisible rests for them by finding the next
//     durational item in the particular staff/layer.
//

void HumGrid::checkForNullDataHoles(void) {
	for (int i=0; i<(int)m_allslices.size(); i++) {
		GridSlice& slice = *m_allslices.at(i);
		if (!slice.isNoteSlice()) {
			continue;
		}
      for (int p=0; p<(int)slice.size(); p++) {
			GridPart& part = *slice.at(p);
      	for (int s=0; s<(int)part.size(); s++) {
				GridStaff& staff = *part.at(s);
      		for (int v=0; v<(int)staff.size(); v++) {
					if (!staff.at(v)) {
						staff.at(v) = new GridVoice();
						// Calculate duration of void by searching
						// for the next non-null voice in the current part/staff/voice
						HumNum duration = slice.getDuration();
						GridPart *pp;
						GridStaff *sp;
						GridVoice *vp;
						for (int q=i+1; q<(int)m_allslices.size(); q++) {
							GridSlice *slicep = m_allslices.at(q);
							if (!slicep->isNoteSlice()) {
								// or isDataSlice()?
								continue;
							}
							if (p >= (int)slicep->size() - 1) {
								continue;
							}
							pp = slicep->at(p);
							if (s >= (int)pp->size() - 1) {
								continue;
							}
							sp = pp->at(s);
							if (v >= (int)sp->size() - 1) {
								// Found a data line with no data at given voice, so
								// add slice duration to cumulative duration.
								duration += slicep->getDuration();
								continue;
							}
							vp = sp->at(v);
							if (!vp) {
								// found another null spot which should be dealt with later.
								break;
							} else {
								// there is a token at the same part/staff/voice position.
								// Maybe check if a null token, but if not a null token,
								// then break here also.
								break;
							}
						}
						string recip = Convert::durationToRecip(duration);
						// ggg @ marker is added to keep track of them for more debugging.
						recip += "ryy@";
						staff.at(v)->setToken(recip);
						continue;
					}
				}
			}
		}
	}
}



//////////////////////////////
//
// HumGrid::setPartStaffDimensions --
//

void HumGrid::setPartStaffDimensions(vector<vector<GridSlice*>>& nextevent,
		GridSlice* startslice) {
	nextevent.clear();
	for (int i=0; i<(int)m_allslices.size(); i++) {
		if (!m_allslices[i]->isNoteSlice()) {
			continue;
		}
		GridSlice* slice = m_allslices[i];
		nextevent.resize(slice->size());
		for (int p=0; p<(int)slice->size(); p++) {
			nextevent.at(p).resize(slice->at(p)->size());
			for (int j=0; j<(int)nextevent.at(p).size(); j++) {
				nextevent.at(p).at(j) = startslice;
			}
		}
		break;
	}
}



//////////////////////////////
//
// HumGrid::addInvisibleRestsInFirstTrack --  If there are any
//    timing gaps in the first track of a **kern spine, then
//    fill in with invisible rests.
//
// ggg

void HumGrid::addInvisibleRestsInFirstTrack(void) {
	int i; // slice index
	int p; // part index
	int s; // staff index
	int v = 0; // only looking at first voice

	vector<vector<GridSlice*>> nextevent;
	GridSlice* lastslice = m_allslices.back();
	setPartStaffDimensions(nextevent, lastslice);

	for (i=(int)m_allslices.size()-1; i>=0; i--) {
		GridSlice& slice = *m_allslices.at(i);
		if (!slice.isNoteSlice()) {
			continue;
		}
      for (p=0; p<(int)slice.size(); p++) {
			GridPart& part = *slice.at(p);
      	for (s=0; s<(int)part.size(); s++) {
				GridStaff& staff = *part.at(s);
				if (staff.size() == 0) {
					// cerr << "EMPTY STAFF VOICE WILL BE FILLED IN LATER!!!!" << endl;
					continue;
				}
				if (!staff.at(v)) {
					// in theory should not happen
					continue;
				}
				GridVoice& gv = *staff.at(v);
				if (gv.isNull()) {
					continue;
				}

				// Found a note/rest.  Check if its duration matches
				// the next non-null data token.  If not, then add
				// an invisible rest somewhere between the two

				// first check to see if the previous item is a
				// NULL.  If so, then store and continue.
				if (nextevent[p][s] == NULL) {
					nextevent[p][s] = &slice;
					continue;
				}
				addInvisibleRest(nextevent, i, p, s);
			}
		}
	}
}



//////////////////////////////
//
// HumGrid::addInvisibleRest --
//
// ggg

void HumGrid::addInvisibleRest(vector<vector<GridSlice*>>& nextevent,
		int index, int p, int s) {
	GridSlice *ending = nextevent.at(p).at(s);
	if (ending == NULL) {
		cerr << "Not handling this case yet at end of data." << endl;
		return;
	}
	HumNum endtime = ending->getTimestamp();

	GridSlice* starting = m_allslices.at(index);
	HumNum starttime = starting->getTimestamp();
	HTp token = starting->at(p)->at(s)->at(0)->getToken();
	HumNum duration = Convert::recipToDuration(token);
	HumNum difference = endtime - starttime;
	HumNum gap = difference - duration;
	if (gap == 0) {
		// nothing to do
		nextevent.at(p).at(s) = starting;
		return;
	}
	HumNum target = starttime + duration;

	string kern = Convert::durationToRecip(gap);
	kern += "ryy";

	for (int i=index+1; i<(int)m_allslices.size(); i++) {
		GridSlice* slice = m_allslices[i];
		if (!slice->isNoteSlice()) {
			continue;
		}
		HumNum timestamp = slice->getTimestamp();
		if (timestamp < target) {
			continue;
		}
		if (timestamp > target) {
			cerr << "Cannot deal with this slice addition case yet..." << endl;
			nextevent[p][s] = starting;
			return;
		}
		// At timestamp for adding new token.
		if ((m_allslices.at(i)->at(p)->at(s)->size() > 0) && !m_allslices.at(i)->at(p)->at(s)->at(0)) {
			// Element is null where an invisible rest should be
			// so allocate space for it.
			m_allslices.at(i)->at(p)->at(s)->at(0) = new GridVoice();
		}
		if (m_allslices.at(i)->at(p)->at(s)->size() > 0) {
			m_allslices.at(i)->at(p)->at(s)->at(0)->setToken(kern);
		}
		break;
	}

	// Store the current event in the buffer
	nextevent.at(p).at(s) = starting;
}



//////////////////////////////
//
// HumGrid::adjustClefChanges -- If a clef change starts at the
//     beginning of a meausre, move it to before the measure (unless
//     the measure has zero duration).
//

void HumGrid::adjustClefChanges(void) {
	vector<GridMeasure*>& measures = *this;
	for (int i=1; i<(int)measures.size(); i++) {
		auto it = measures[i]->begin();
		if ((*it) == NULL) {
			cerr << "Warning: GridSlice is null in GridMeasure " << i << endl;
			continue;
		}
		if ((*it)->empty()) {
			cerr << "Warning: GridSlice is empty in GridMeasure "  << i << endl;
			continue;
		}
		if (!(*it)->isClefSlice()) {
			continue;
		}
		// move clef to end of previous measure
		GridSlice* tempslice = *it;
		measures[i]->pop_front();
		measures[i-1]->push_back(tempslice);
	}
}



//////////////////////////////
//
// HumGrid::extendDurationToken --
//

void HumGrid::extendDurationToken(int slicei, int parti, int staffi,
		int voicei) {
	if ((slicei < 0) || (slicei >= ((int)m_allslices.size()) - 1)) {
		// nothing after this line, so can extend further.
		return;
	}

	if (!m_allslices.at(slicei)->hasSpines()) {
		// no extensions needed in non-spined slices.
		return;
	}

	if (m_allslices.at(slicei)->isGraceSlice()) {
		cerr << "THIS IS A GRACE SLICE SO DO NOT FILL" << endl;
		return;
	}

	GridVoice* gv = m_allslices.at(slicei)->at(parti)->at(staffi)->at(voicei);
 	HTp token = gv->getToken();
	if (!token) {
		cerr << "STRANGE: token should not be null" << endl;
		return;
	}
	if (*token == ".") {
		// null data token so ignore;
		// change this later to add a duration for the null token below.
		return;
	}

	HumNum tokendur = Convert::recipToDuration((string)*token);
	HumNum currts   = m_allslices.at(slicei)->getTimestamp();
	HumNum nextts   = m_allslices.at(slicei+1)->getTimestamp();
	HumNum slicedur = nextts - currts;
	HumNum timeleft = tokendur - slicedur;

	if ((0)) {
		cerr << "===================" << endl;
		cerr << "EXTENDING TOKEN    " << token      << endl;
		cerr << "\tTOKEN DUR:       " << tokendur   << endl;
		cerr << "\tTOKEN START:     " << currts     << endl;
		cerr << "\tSLICE DUR:       " << slicedur   << endl;
		cerr << "\tNEXT SLICE START:" << nextts     << endl;
		cerr << "\tTIME LEFT:       " << timeleft   << endl;
		cerr << "\t-----------------" << endl;
	}

	if (timeleft != 0) {
		// fill in null tokens for the required duration.
		if (timeleft < 0) {
			cerr << "ERROR: Negative duration: " << timeleft << endl;
			cerr << "\ttokendur = " << tokendur << endl;
			cerr << "\tslicedur = " << slicedur << endl;
			cerr << "\ttoken    = " << token << endl;
			cerr << "\tCURRENT SLICE = " << m_allslices.at(slicei) << endl;
			cerr << "\tTIMESTAMP " << currts << endl;
			cerr << "\tNEXT SLICE = " << m_allslices.at(slicei) << endl;
			cerr << "\tNEXT TIMESTAMP " << nextts << endl;
			return;
		}

		SliceType type;
		GridStaff* gs;
		int s = slicei+1;

		while ((s < (int)m_allslices.size()) && (timeleft > 0)) {
			if (!m_allslices.at(s)->hasSpines()) {
				s++;
				continue;
			}
			currts = nextts;
			int nexts = 1;
			while (s < (int)m_allslices.size() - nexts) {
				if (!m_allslices.at(s+nexts)->hasSpines()) {
					nexts++;
					continue;
				}
				break;
			}
			if (s < (int)m_allslices.size() - nexts) {
				nextts = m_allslices.at(s+nexts)->getTimestamp();
			} else {
				nextts = currts + m_allslices.at(s)->getDuration();
			}
			slicedur = nextts - currts;
			type = m_allslices[s]->getType();

			if (staffi == (int)m_allslices.at(s)->at(parti)->size()) {
					cerr << "WARNING: staff index " << staffi << " is probably incorrect: increasing staff count for part to " << staffi + 1 << endl;
					m_allslices.at(s)->at(parti)->resize(m_allslices.at(s)->at(parti)->size() + 1);
					m_allslices.at(s)->at(parti)->at(staffi) = new GridStaff();
			}
			gs = m_allslices.at(s)->at(parti)->at(staffi);
			if (gs == NULL) {
				cerr << "Strange error6 in extendDurationToken()" << endl;
				return;
			}

			if (m_allslices.at(s)->isGraceSlice()) {
				m_allslices[s]->setDuration(0);
			} else if (m_allslices.at(s)->isDataSlice()) {
				gs->setNullTokenLayer(voicei, type, slicedur);
				timeleft = timeleft - slicedur;
			} else if (m_allslices.at(s)->isInvalidSlice()) {
				cerr << "THIS IS AN INVALID SLICE" << m_allslices.at(s) << endl;
			} else {
				// store a null token for the non-data slice, but probably skip
				// if there is a token already there (such as a clef-change).
				if ((voicei < (int)gs->size()) && (gs->at(voicei) != NULL)) {
					// there is already a token here, so do not replace it.
					// cerr << "Not replacing token: "  << gs->at(voicei)->getToken() << endl;
				} else {
					gs->setNullTokenLayer(voicei, type, slicedur);
				}
			}
			s++;
			if (s == (int)m_allslices.size() - 1) {
				m_allslices[s]->setDuration(timeleft);
			}
		}
	}
	// walk through zero-dur items and fill them in, but stop at
	// a token (likely a grace note which should not be erased).
}



//////////////////////////////
//
// HumGrid::getGridVoice -- Check to see if GridVoice exists, returns
//    NULL otherwise. Requires HumGrid::buildSingleList() being run first.
//

GridVoice* HumGrid::getGridVoice(int slicei, int parti, int staffi,
		int voicei) {
	if (slicei >= (int)m_allslices.size()) {
		cerr << "Strange error 1a" << endl;
		return NULL;
	}
	GridSlice* gsl = m_allslices.at(slicei);
	if (gsl == NULL) {
		cerr << "Strange error 1b" << endl;
		return NULL;
	}

	if (parti >= (int)gsl->size()) {
		cerr << "Strange error 2a" << endl;
		return NULL;
	}
	GridPart* gp = gsl->at(parti);
	if (gp == NULL) {
		cerr << "Strange error 2" << endl;
		return NULL;
	}

	if (staffi >= (int)gp->size()) {
		cerr << "Strange error 3a" << endl;
		return NULL;
	}
	GridStaff* gst = gp->at(staffi);
	if (gst == NULL) {
		cerr << "Strange error 3b" << endl;
		return NULL;
	}

	if (voicei >= (int)gst->size()) {
		cerr << "Strange error 4a" << endl;
		return NULL;
	}
	GridVoice* gv = gst->at(voicei);
	if (gv == NULL) {
		cerr << "Strange error 4b" << endl;
		return NULL;
	}
	return gv;
}



//////////////////////////////
//
// HumGrid::calculateGridDurations --
//

void HumGrid::calculateGridDurations(void) {

	// the last line has to be calculated from the shortest or
   // longest duration on the line.  Acutally all durations
	// starting on this line must be the same, so just search for
	// the first duration.

	auto last = m_allslices.back();

	// set to zero in case not a duration type of line:
	last->setDuration(0);

	bool finished = false;
	if (last->isNoteSlice()) {
		for (auto part : *last) {
			for (auto staff : *part) {
				for (auto voice : *staff) {
					if (!voice) {
						continue;
					}
					if (voice->getDuration() > 0) {
						last->setDuration(voice->getDuration());
						finished = true;
						break;
					}
				}
				if (finished) {
					break;
				}
			}
			if (finished) {
				break;
			}
		}
	}
}



//////////////////////////////
//
// HumGrid::insertExclusiveInterpretationLine -- Currently presumes
//    that the first entry contains spines.  And the first measure
//    in the HumGrid object must contain a slice.
//

void HumGrid::insertExclusiveInterpretationLine(HumdrumFile& outfile) {
	if (this->size() == 0) {
		return;
	}
	if (this->at(0)->empty()) {
		return;
	}

	HLp line = new HumdrumLine;
	HTp token;

	if (m_recip) {
		token = new HumdrumToken("**recip");
		line->appendToken(token);
	}

	GridSlice& slice = *this->at(0)->front();
	int p; // part index
	int s; // staff index
	for (p=(int)slice.size()-1; p>=0; p--) {
		GridPart& part = *slice[p];
		for (s=(int)part.size()-1; s>=0; s--) {
			token = new HumdrumToken("**kern");
			line->appendToken(token);
			insertExInterpSides(line, p, s); // insert staff sides
		}
		insertExInterpSides(line, p, -1);   // insert part sides
	}
	outfile.insertLine(0, line);
}



//////////////////////////////
//
// HumGrid::insertExInterpSides --
//

void HumGrid::insertExInterpSides(HLp line, int part, int staff) {

	if (staff >= 0) {
		int versecount = getVerseCount(part, staff); // verses related to staff
		for (int i=0; i<versecount; i++) {
			HTp token = new HumdrumToken("**text");
			line->appendToken(token);
		}
	}

	if ((staff < 0) && hasDynamics(part)) {
		HTp token = new HumdrumToken("**dynam");
		line->appendToken(token);
	}

	if ((staff < 0) && hasFiguredBass(part)) {
		HTp token = new HumdrumToken("**fb");
		line->appendToken(token);
	}

	if (staff < 0) {
		int harmonyCount = getHarmonyCount(part);
		for (int i=0; i<harmonyCount; i++) {
			HTp token = new HumdrumToken("**mxhm");
			line->appendToken(token);
		}

	}
}



//////////////////////////////
//
// HumGrid::insertPartNames --
//

void HumGrid::insertPartNames(HumdrumFile& outfile) {
	if (m_partnames.size() == 0) {
		return;
	}
	HLp line = new HumdrumLine;
	HTp token;

	if (m_recip) {
		token = new HumdrumToken("*");
		line->appendToken(token);
	}

	string text;
	GridSlice& slice = *this->at(0)->front();
	int p; // part index
	int s; // staff index
	for (p=(int)slice.size()-1; p>=0; p--) {
		GridPart& part = *slice[p];
		for (s=(int)part.size()-1; s>=0; s--) {
			text = "*";
			string pname = m_partnames[p];
			if (!pname.empty()) {
				text += "I\"";
				text += pname;
			}
			token = new HumdrumToken(text);
			line->appendToken(token);
			insertSideNullInterpretations(line, p, s);
		}
		insertSideNullInterpretations(line, p, -1);
	}
	outfile.insertLine(0, line);
}



//////////////////////////////
//
// HumGrid::insertPartIndications -- Currently presumes
//    that the first entry contains spines.  And the first measure
//    in the HumGrid object must contain a slice.  This is the
//    MusicXML Part number. (Some parts will contain more than one
//    staff).
//

void HumGrid::insertPartIndications(HumdrumFile& outfile) {

	if (this->size() == 0) {
		return;
	}
	if (this->at(0)->empty()) {
		return;
	}
	HLp line = new HumdrumLine;
	HTp token;

	if (m_recip) {
		token = new HumdrumToken("*");
		line->appendToken(token);
	}

	string text;
	GridSlice& slice = *this->at(0)->front();
	int p; // part index
	int s; // staff index
	for (p=(int)slice.size()-1; p>=0; p--) {
		GridPart& part = *slice[p];
		for (s=(int)part.size()-1; s>=0; s--) {
			text = "*part" + to_string(p+1);
			token = new HumdrumToken(text);
			line->appendToken(token);
			insertSidePartInfo(line, p, s);
		}
		insertSidePartInfo(line, p, -1);   // insert part sides
	}
	outfile.insertLine(0, line);

}



//////////////////////////////
//
// HumGrid::insertSideNullInterpretations --
//

void HumGrid::insertSideNullInterpretations(HLp line,
		int part, int staff) {
	HTp token;
	string text;

	if (staff < 0) {

		if (hasDynamics(part)) {
			token = new HumdrumToken("*");
			line->appendToken(token);
		}

		if (hasFiguredBass(part)) {
			token = new HumdrumToken("*");
			line->appendToken(token);
		}

		int harmcount = getHarmonyCount(part);
		for (int i=0; i<harmcount; i++) {
			token = new HumdrumToken("*");
			line->appendToken(token);
		}

	} else {
		int versecount = getVerseCount(part, staff);
		for (int i=0; i<versecount; i++) {
			token = new HumdrumToken("*");
			line->appendToken(token);
		}
	}
}



//////////////////////////////
//
// HumGrid::insertSidePartInfo --
//

void HumGrid::insertSidePartInfo(HLp line, int part, int staff) {
	HTp token;
	string text;

	if (staff < 0) {

		if (hasDynamics(part)) {
			text = "*part" + to_string(part+1);
			token = new HumdrumToken(text);
			line->appendToken(token);
		}

		if (hasFiguredBass(part)) {
			text = "*part" + to_string(part+1);
			token = new HumdrumToken(text);
			line->appendToken(token);
		}

		int harmcount = getHarmonyCount(part);
		for (int i=0; i<harmcount; i++) {
			text = "*part" + to_string(part+1);
			token = new HumdrumToken(text);
			line->appendToken(token);
		}

	} else {
		int versecount = getVerseCount(part, staff);
		for (int i=0; i<versecount; i++) {
			text = "*part" + to_string(part+1);
			token = new HumdrumToken(text);
			line->appendToken(token);
		}
	}
}



//////////////////////////////
//
// HumGrid::insertStaffIndications -- Currently presumes
//    that the first entry contains spines.  And the first measure
//    in the HumGrid object must contain a slice.  This is the
//    MusicXML Part number. (Some parts will contain more than one
//    staff).
//

void HumGrid::insertStaffIndications(HumdrumFile& outfile) {
	if (this->size() == 0) {
		return;
	}
	if (this->at(0)->empty()) {
		return;
	}

	HLp line = new HumdrumLine;
	HTp token;

	if (m_recip) {
		token = new HumdrumToken("*");
		line->appendToken(token);
	}

	string text;
	GridSlice& slice = *this->at(0)->front();
	int p; // part index
	int s; // staff index

	int staffcount = 0;
	for (p=0; p<(int)slice.size(); p++) {
		GridPart& part = *slice[p];
		staffcount += (int)part.size();
	}

	for (p=(int)slice.size()-1; p>=0; p--) {
		GridPart& part = *slice[p];
		for (s=(int)part.size()-1; s>=0; s--) {
			text = "*staff" + to_string(staffcount--);
			token = new HumdrumToken(text);
			line->appendToken(token);
			insertSideStaffInfo(line, p, s, staffcount+1);
		}
		insertSideStaffInfo(line, p, -1, -1);  // insert part sides
	}
	outfile.insertLine(0, line);
}



//////////////////////////////
//
// HumGrid::insertSideStaffInfo --
//

void HumGrid::insertSideStaffInfo(HLp line, int part, int staff,
		int staffnum) {
	HTp token;
	string text;

	// part-specific sides (no staff markers)
	if (staffnum < 0) {

		if (hasDynamics(part)) {
			token = new HumdrumToken("*");
			line->appendToken(token);
		}

		if (hasFiguredBass(part)) {
			token = new HumdrumToken("*");
			line->appendToken(token);
		}

		int harmcount = getHarmonyCount(part);
		for (int i=0; i<harmcount; i++) {
			token = new HumdrumToken("*");
			line->appendToken(token);
		}

		return;
	}

	int versecount = getVerseCount(part, staff);
	for (int i=0; i<versecount; i++) {
		if (staffnum > 0) {
			text = "*staff" + to_string(staffnum);
			token = new HumdrumToken(text);
		} else {
			token = new HumdrumToken("*");
		}
		line->appendToken(token);
	}


}



//////////////////////////////
//
// HumGrid::insertDataTerminationLine -- Currently presumes
//    that the last entry contains spines.  And the first
//    measure in the HumGrid object must contain a slice.
//    Also need to compensate for *v on previous line.
//

void HumGrid::insertDataTerminationLine(HumdrumFile& outfile) {
	if (this->size() == 0) {
		return;
	}
	if (this->at(0)->empty()) {
		return;
	}
	HLp line = new HumdrumLine;
	HTp token;

	if (m_recip) {
		token = new HumdrumToken("*-");
		line->appendToken(token);
	}

	GridSlice& slice = *this->at(0)->back();
	int p; // part index
	int s; // staff index
	for (p=(int)slice.size()-1; p>=0; p--) {
		GridPart& part = *slice[p];
		for (s=(int)part.size()-1; s>=0; s--) {
			token = new HumdrumToken("*-");
			line->appendToken(token);
			insertSideTerminals(line, p, s);
		}
		insertSideTerminals(line, p, -1);   // insert part sides
	}
	outfile.appendLine(line);
}



//////////////////////////////
//
// HumGrid::insertSideTerminals --
//

void HumGrid::insertSideTerminals(HLp line, int part, int staff) {
	HTp token;

	if (staff < 0) {

		if (hasDynamics(part)) {
			token = new HumdrumToken("*-");
			line->appendToken(token);
		}

		if (hasFiguredBass(part)) {
			token = new HumdrumToken("*-");
			line->appendToken(token);
		}

		int harmcount = getHarmonyCount(part);
		for (int i=0; i<harmcount; i++) {
			token = new HumdrumToken("*-");
			line->appendToken(token);
		}

	} else {
		int versecount = getVerseCount(part, staff);
		for (int i=0; i<versecount; i++) {
			token = new HumdrumToken("*-");
			line->appendToken(token);
		}
	}
}



//////////////////////////////
//
// HumGrid::transferNonDataSlices --
//

void HumGrid::transferNonDataSlices(GridMeasure* output, GridMeasure* input) {
	for (auto it = input->begin(); it != input->end(); it++) {
		GridSlice* slice = *it;
		if (slice->isDataSlice()) {
			continue;
		}
		output->push_front(slice);
		input->erase(it);
		it--;
	}
}



//////////////////////////////
//
// HumGrid::removeSibeliusIncipit --
//

void HumGrid::removeSibeliusIncipit(void) {

	if (this->size() == 0) {
		return;
	}
	GridMeasure* measure = this->at(0);
	bool invisible = measure->isInvisible();
	if (!invisible) {
		return;
	}

	this->erase(this->begin());
	if (this->size() > 0) {
		// [20171012] remove this for now since it is crashing
		// emscripten version of code.
		// transferNonDataSlices(this->at(0), measure);
	}
	delete measure;
	measure = NULL;

	// remove vocal ranges, if present
	if (this->size() == 0) {
		return;
	}

	measure = this->at(0);
	bool singlechord = measure->isSingleChordMeasure();
	if (!singlechord) {
		return;
	}

	this->erase(this->begin());
	if (this->size() > 0) {
		transferNonDataSlices(this->at(0), measure);
	}
	delete measure;
	measure = NULL;

	measure = this->at(0);
	bool monophonic = measure->isMonophonicMeasure();
	if (!monophonic) {
		return;
	}

	string melody = extractMelody(measure);

	this->erase(this->begin());
	if (this->size() > 0) {
		transferNonDataSlices(this->at(0), measure);
	}
	delete measure;
	measure = NULL;

	if (this->size() > 0) {
		insertMelodyString(this->at(0), melody);
	}

}



//////////////////////////////
//
// HumGrid::insertMelodyString -- Insert a global comment before first data line.
//

void HumGrid::insertMelodyString(GridMeasure* measure, const string& melody) {
	for (auto it = measure->begin(); it != measure->end(); it++) {
		GridSlice* slice = *it;
		if (!slice->isDataSlice()) {
			continue;
		}

		// insert a new GridSlice
		// first need to implement global commands in GridSlice object...
		break;
	}
}



//////////////////////////////
//
// HumGrid::extractMelody --
//

string HumGrid::extractMelody(GridMeasure* measure) {
	string output = "!!";

	int parti  = -1;
	int staffi = -1;
	int voicei = -1;

	// First find the part which has the melody:
	for (auto slice : *measure) {
		if (!slice->isDataSlice()) {
			continue;
		}
		for (int p=0; p<(int)slice->size(); p++) {
			GridPart* part = slice->at(p);
			for (int s=0; s<(int)part->size(); s++) {
				GridStaff* staff = part->at(s);
				for (int v=0; v<(int)staff->size(); v++) {
					GridVoice* voice = staff->at(v);
					HTp token = voice->getToken();
					if (!token) {
						continue;
					}
					if (token->find("yy") == string::npos) {
						parti  = p;
						staffi = s;
						voicei = v;
						goto loop_end;
					}
				}
			}
		}
	}

	loop_end:

	if (parti < 0) {
		return output;
	}

	// First find the part which has the melody:
	for (auto slice : *measure) {
		if (!slice->isDataSlice()) {
			continue;
		}
		HTp token = slice->at(parti)->at(staffi)->at(voicei)->getToken();
		if (!token) {
			continue;
		}
		if (*token == ".") {
			continue;
		}
		output += " ";
		output += *token;
	}

	return output;
}




//////////////////////////////
//
// HumGrid::removeRedundantClefChanges -- Will also have to consider
//		the meter signature.
//

void HumGrid::removeRedundantClefChanges(void) {
	// curclef is a list of the current staff on the part:staff.
	vector<vector<string> > curclef;

	bool hasduplicate = false;
	for (int m=0; m<(int)this->size(); m++) {
		GridMeasure* measure = this->at(m);
		for (auto slice : *measure) {
			if (!slice->isClefSlice()) {
				continue;
			}
			bool allempty = true;
			for (int p=0; p<(int)slice->size(); p++) {
				for (int s=0; s<(int)slice->at(p)->size(); s++) {
					if (slice->at(p)->at(s)->size() < 1) {
						continue;
					}
					GridVoice* voice = slice->at(p)->at(s)->at(0);
					HTp token = voice->getToken();
					if (!token) {
						continue;
					}
					if (string(*token) == "*") {
						continue;
					}
					if (token->find("clef") == string::npos) {
						// something (probably invalid) which is not a clef change
						allempty = false;
						continue;
					}
					if (p >= (int)curclef.size()) {
						curclef.resize(p+1);
					}
					if (s >= (int)curclef[p].size()) {
						// first clef on the staff, so can't be a duplicate
						curclef[p].resize(s+1);
						curclef[p][s] = *token;
						allempty = false;
						continue;
					} else {
						if (curclef[p][s] == (string)*token) {
							// clef is already active, so remove this one
							hasduplicate = true;
							voice->setToken("*");
						} else {
							// new clef change
							curclef[p][s] = *token;
							allempty = false;
						}
					}
				}
			}
			if (!hasduplicate) {
				continue;
			}
			// Check the slice to see if it empty, and delete if so.
			// This algorithm does not consider GridSide content.
			if (allempty) {
				slice->invalidate();
			}

		}
	}
}



//////////////////////////////
//
// HumGrid::cleanTempos --
//

void HumGrid::cleanTempos(void) {
//		std::vector<GridSlice*>       m_allslices;
// ggg
	for (int i=0; i<(int)m_allslices.size(); i++) {
		if (!m_allslices[i]->isTempoSlice()) {
			continue;
		}
		cleanTempos(m_allslices[i]);
	}
}


void HumGrid::cleanTempos(GridSlice* slice) {
	if (!slice->isTempoSlice()) {
		return;
	}
	HTp token = NULL;

	for (int part=0; part<(int)slice->size(); part++) {
		GridPart* gp = slice->at(part);
		for (int staff=0; staff<(int)gp->size(); staff++) {
			GridStaff* gs = gp->at(staff);
			for (int voice=0; voice<(int)gs->size(); voice++) {
				GridVoice* gv = gs->at(voice);
				token = gv->getToken();
				if (token) {
					break;
				}
			}
			if (token) {
				break;
			}
		}
		if (token) {
			break;
		}
	}

	if (!token) {
		return;
	}

	for (int part=0; part<(int)slice->size(); part++) {
		GridPart* gp = slice->at(part);
		for (int staff=0; staff<(int)gp->size(); staff++) {
			GridStaff* gs = gp->at(staff);
			for (int voice=0; voice<(int)gs->size(); voice++) {
				GridVoice* gv = gs->at(voice);
				if (gv->getToken()) {
					continue;
				}
				gv->setToken(*token);
			}
		}
	}
}



//////////////////////////////
//
// HumGrid::hasPickup --
//

bool HumGrid::hasPickup(void) {
	return m_pickup;
}



//////////////////////////////
//
// HumGrid::deleteMeasure --
//

void HumGrid::deleteMeasure(int index) {
	delete this->at(index);
	this->at(index) = NULL;
	this->erase(this->begin() + index);
}



//////////////////////////////
//
// HumGrid::expandLocalCommentLayers -- Walk backwards in the
//   data list, and match the layer count for local comments
//   to have them match to the next data line.  This is needed
//   to attach layout parameters properly to data tokens.  Layout
//   parameters cannot pass through spine manipulator lines, so
//   this function is necessary to prevent spine manipulators
//   from orphaning local layout parameter lines.
//
//   For now just adjust local layout parameter slices, but maybe
//   later do all types of local comments.
//

void HumGrid::expandLocalCommentLayers(void) {
	GridSlice *dataslice = NULL;
	GridSlice *localslice = NULL;
	for (int i=(int)m_allslices.size() - 1; i>=0; i--) {
		if (m_allslices[i]->isDataSlice()) {
			dataslice = m_allslices[i];
		} else if (m_allslices[i]->isMeasureSlice()) {
			dataslice = m_allslices[i];
		}
		// Other slice types should be considered as well,
		// but definitely not manipulator slices:
		if (m_allslices[i]->isManipulatorSlice()) {
			dataslice = m_allslices[i];
		}

		if (!m_allslices[i]->isLocalLayoutSlice()) {
			continue;
		}
		localslice = m_allslices[i];
		if (!dataslice) {
			continue;
		}
		matchLayers(localslice, dataslice);
	}
}


//////////////////////////////
//
// HumGrid::matchLayers -- Make sure every staff in both inputs
//   have the same number of voices.
//

void HumGrid::matchLayers(GridSlice* output, GridSlice* input) {
	if (output->size() != input->size()) {
		// something wrong or one of the slices
		// could be a non-spined line.
		return;
	}
	int partcount = (int)input->size();
	for (int part=0; part<partcount; part++) {
		GridPart* ipart = input->at(part);
		GridPart* opart = output->at(part);
		if (ipart->size() != opart->size()) {
			// something string that should never happen
			continue;
		}
		int scount = (int)ipart->size();
		for (int staff=0; staff<scount; staff++) {
			GridStaff* istaff = ipart->at(staff);
			GridStaff* ostaff = opart->at(staff);
			matchLayers(ostaff, istaff);
		}
	}
}


void HumGrid::matchLayers(GridStaff* output, GridStaff* input) {
	if (input->size() == output->size()) {
		// The voice counts match so nothing to do.
		return;
	}
	if (input->size() < output->size()) {
		// Ignore potentially strange case.
	}

	int diff = (int)input->size() - (int)output->size();
	for (int i=0; i<diff; i++) {
		GridVoice* voice = new GridVoice("!", 0);
		output->push_back(voice);
	}
}



//////////////////////////////
//
// HumGrid::setPartName --
//

void HumGrid::setPartName(int index, const string& name) {
	if (index < 0) {
		return;
	} else if (index < (int)m_partnames.size()) {
		m_partnames[index] = name;
	} else if (index < 100) {
		// grow the array and then store name
		m_partnames.resize(index+1);
		m_partnames.back() = name;
	}
}



//////////////////////////////
//
// HumGrid::getPartName --
//

std::string HumGrid::getPartName(int index) {
	if (index < 0) {
		return "";
	} else if (index < (int)m_partnames.size()) {
		return m_partnames[index];
	} else {
		return "";
	}
}



//////////////////////////////
//
// operator<< -- Debugging printing of Humgrid Contents.
//

ostream& operator<<(ostream& out, HumGrid& grid) {
	for (int i=0; i<(int)grid.size(); i++) {
		out << "\nMEASURE " << i << " =========================" << endl;
		out << grid[i];
	}
	return out;
}




////////////////////////////////
//
// HumParameter::HumParameter -- HumParameter constructor.
//

HumParameter::HumParameter(void) {
	origin = NULL;
}


HumParameter::HumParameter(const string& str) : string(str) {
	origin = NULL;
}



//////////////////////////////
//
// HumHash::HumHash -- HumHash constructor.  The data storage is empty
//    until the first parameter in the Hash is set.
//

HumHash::HumHash(void) {
	parameters = NULL;
}



//////////////////////////////
//
// HumHash::~HumHash -- The HumHash deconstructor, which removed any
//    allocated storage before the object dies.
//

HumHash::~HumHash() {
	if (parameters != NULL) {
		delete parameters;
		parameters = NULL;
	}
}



//////////////////////////////
//
// HumHash::getValue -- Returns the value specified by the given key.
//    If there is no colon in the key then return the value for the key
//    in the default namespaces (NS1="" and NS2="").  If there is one colon,
//    then the two pieces of the string as NS2 and the key, with NS1="".
//    If there are two colons, then that specified the complete namespaces/key
//    address of the value.  The namespaces and key can be specified as
//    separate parameters in a similar manner to the single-string version.
//    But in these cases colon concatenation of the namespaces and/or key
//    are not allowed.
//

string HumHash::getValue(const string& key) const {
	if (parameters == NULL) {
		return "";
	} else {
		vector<string> keys = getKeyList(key);
		if (keys.size() == 1) {
			return getValue("", "", keys[0]);
		} else if (keys.size() == 2) {
			return getValue("", keys[0], keys[1]);
		} else {
			return getValue(keys[0], keys[1], keys[2]);
		}
	}
}


string HumHash::getValue(const string& ns2, const string& key) const {
	if (parameters == NULL) {
		return "";
	} else {
		return getValue("", ns2, key);
	}
}


string HumHash::getValue(const string& ns1, const string& ns2,
		const string& key) const {
	if (parameters == NULL) {
		return "";
	}
	MapNNKV& p = *parameters;
	auto it1 = p.find(ns1);
	if (it1 == p.end()) {
		return "";
	}
	auto it2 = it1->second.find(ns2);
	if (it2 == it1->second.end()) {
		return "";
	}
	auto it3 = it2->second.find(key);
	if (it3 == it2->second.end()) {
		return "";
	}
	return it3->second;
}



//////////////////////////////
//
// HumHash::getValueHTp -- Return an address of a HumdrumToken.
//   Presumes 64-bit pointers (or at least not 128-bit pointers).
//

HTp HumHash::getValueHTp(const string& key) const {
	if (parameters == NULL) {
		return NULL;
	}
	vector<string> keys = getKeyList(key);
	if (keys.size() == 1) {
		return getValueHTp("", "", keys[2]);
	} else if (keys.size() == 2) {
		return getValueHTp(keys[0], keys[1]);
	} else {
		return getValueHTp(keys[0], keys[1], keys[2]);
	}
}


HTp HumHash::getValueHTp(const string& ns2, const string& key) const {
	if (parameters == NULL) {
		return NULL;
	}
	return getValueHTp("", ns2, key);
}


HTp HumHash::getValueHTp(const string& ns1, const string& ns2,
		const string& key) const {
	if (parameters == NULL) {
		return NULL;
	}
	string value = getValue(ns1, ns2, key);
	if (value.find("HT_") != 0) {
		return NULL;
	} else {
		HTp pointer = NULL;
		try {
			pointer = (HTp)(stoll(value.substr(3)));
		} catch (invalid_argument& e) {
         std::cerr << e.what() << std::endl;
			pointer = NULL;
		}
		return pointer;
	}
}



//////////////////////////////
//
// HumHash::getValueInt -- Return the value as an integer.  The value must
//   start with a number and have no text before it; otherwise the
//   returned value will be "0".  The HumHash class is aware of fractional
//   values, so the integer form of the fraction will be returned.  For
//   example if the value is "12/7", then the return value will be "1"
//   since the integer part of 12/7 is 1 with a remainder of 5/7ths
//   which will be chopped off.
//

int HumHash::getValueInt(const string& key) const {
	if (parameters == NULL) {
		return 0;
	}
	vector<string> keys = getKeyList(key);
	if (keys.size() == 1) {
		return getValueInt("", "", keys[2]);
	} else if (keys.size() == 2) {
		return getValueInt(keys[0], keys[1]);
	} else {
		return getValueInt(keys[0], keys[1], keys[2]);
	}
}


int HumHash::getValueInt(const string& ns2, const string& key) const {
	if (parameters == NULL) {
		return 0;
	}
	return getValueInt("", ns2, key);
}


int HumHash::getValueInt(const string& ns1, const string& ns2,
		const string& key) const {
	if (parameters == NULL) {
		return 0;
	}
	string value = getValue(ns1, ns2, key);
	if (value.find("/") != string::npos) {
		HumNum nvalue(value);
		return  nvalue.getInteger();
	} else {
		int intvalue;
		try {
			// problem with emscripten with stoi:
			// intvalue = stoi(value);
			stringstream converter(value);
			if (!(converter >> intvalue)) {
				intvalue = 0;
			}
		} catch (invalid_argument& e) {
         std::cerr << e.what() << std::endl;
			intvalue = 0;
		}
		return intvalue;
	}
}



//////////////////////////////
//
// HumHash::getValueFraction -- Return the value as a HumNum fraction.
//    If the string represents an integer, it will be preserved in the
//    HumNum return value.  For floating-point values, the fractional
//    part will be ignored.  For example "1.52" will be returned as "1".
//

HumNum HumHash::getValueFraction(const string& key) const {
	if (parameters == NULL) {
		return 0;
	}
	vector<string> keys = getKeyList(key);
	if (keys.size() == 1) {
		return getValueFraction("", "", keys[0]);
	} else if (keys.size() == 2) {
		return getValueFraction(keys[0], keys[1]);
	} else {
		return getValueFraction(keys[0], keys[1], keys[2]);
	}
}


HumNum HumHash::getValueFraction(const string& ns2, const string& key) const {
	if (parameters == NULL) {
		return 0;
	}
	return getValueFraction("", ns2, key);
}


HumNum HumHash::getValueFraction(const string& ns1, const string& ns2,
		const string& key) const {
	if (!isDefined(ns1, ns2, key)) {
		return 0;
	}
	string value = getValue(ns1, ns2, key);
	HumNum fractionvalue(value);
	return fractionvalue;
}



//////////////////////////////
//
// HumHash::getValueFloat --  Return the floating-point interpretation
//   of the value string.  If the string can represent a HumNum fraction,
//   then convert the HumNum interpretation as a floating point number.
//   For example "1.25" and "5/4" will both return 1.25.  The value
//   cannot contain a slash unless it is part of the first fraction
//   on in the value string (this may be changed when regular expressions
//   are used to implement this function).
//

double HumHash::getValueFloat(const string& key) const {
	if (parameters == NULL) {
		return 0.0;
	}
	vector<string> keys = getKeyList(key);
	if (keys.size() == 1) {
		return getValueFloat("", "", keys[2]);
	} else if (keys.size() == 2) {
		return getValueFloat(keys[0], keys[1]);
	} else {
		return getValueFloat(keys[0], keys[1], keys[2]);
	}
}


double HumHash::getValueFloat(const string& ns2, const string& key) const {
	if (parameters == NULL) {
		return 0.0;
	}
	return getValueInt("", ns2, key);
}


double HumHash::getValueFloat(const string& ns1, const string& ns2,
		const string& key) const {
	if (parameters == NULL) {
		return 0.0;
	}
	string value = getValue(ns1, ns2, key);
	if (value.find("/") != string::npos) {
		HumNum nvalue(value);
		return nvalue.getFloat();
	} else {
		double floatvalue;
		try {
			floatvalue = stod(value);
		} catch (invalid_argument& e) {
         std::cerr << e.what() << std::endl;
			floatvalue = 0;
		}
		return floatvalue;
	}
}



//////////////////////////////
//
// HumHash::getValueBool -- Return true or false based on the
//   value.  If the value is "0" or false, then the function
//   will return false.  If the value is anything else, then
//   true will be returned.  If the parameter is not defined
//   in the HumHash, then false will also be defined.
//   See also hasParameter() if you do not like this last
//   behavior.
//

bool HumHash::getValueBool(const string& key) const {
	vector<string> keys = getKeyList(key);
	if (keys.size() == 1) {
		return getValueBool("", "", keys[2]);
	} else if (keys.size() == 2) {
		return getValueBool(keys[0], keys[1]);
	} else {
		return getValueBool(keys[0], keys[1], keys[2]);
	}
}


bool HumHash::getValueBool(const string& ns2, const string& key) const {
	return getValueBool("", ns2, key);
}


bool HumHash::getValueBool(const string& ns1, const string& ns2,
		const string& key) const {
	if (parameters == NULL) {
		return false;
	}
	if (!isDefined(ns1, ns2, key)) {
		return false;
	}
	if ((*parameters)[ns1][ns2][key] == "false") {
		return false;
	} else if ((*parameters)[ns1][ns2][key] == "0") {
		return false;
	} else {
		return true;
	}
}



//////////////////////////////
//
// HumHash::setValue -- Set the parameter to the given value,
//     over-writing any previous value for the parameter.  The
//     value is any arbitrary string, but preferably does not
//     include tabs or colons.  If a colon is needed, then specify
//     as "&colon;" without the quotes.  Values such as integers
//     fractions and floats can be specified, and these wil be converted
//     internally into strings (use getValueInt() or getValueFloat()
//     to recover the original type).
//

void HumHash::setValue(const string& key, const string& value) {
	vector<string> keys = getKeyList(key);
	if (keys.size() == 1) {
		setValue("", "", keys[0], value);
	} else if (keys.size() == 2) {
		setValue("", keys[0], keys[1], value);
	} else {
		setValue(keys[0], keys[1], keys[2], value);
	}
}


void HumHash::setValue(const string& ns2, const string& key,
		const string& value) {
		setValue("", ns2, key, value);
}


void HumHash::setValue(const string& ns1, const string& ns2,
		const string& key, const string& value) {
	initializeParameters();
	(*parameters)[ns1][ns2][key] = value;
}


void HumHash::setValue(const string& key, const char* value) {
	setValue(key, (string)value);
}


void HumHash::setValue(const string& ns2, const string& key,
		const char* value) {
	setValue(ns2, key, (string)value);
}


void HumHash::setValue(const string& ns1, const string& ns2, const string& key,
		const char* value) {
	setValue(ns1, ns2, key, (string)value);
}


void HumHash::setValue(const string& key, int value) {
	vector<string> keys = getKeyList(key);
	if (keys.size() == 1) {
		setValue("", "", keys[0], value);
	} else if (keys.size() == 2) {
		setValue("", keys[0], keys[1], value);
	} else {
		setValue(keys[0], keys[1], keys[2], value);
	}
}


void HumHash::setValue(const string& ns2, const string& key, int value) {
		setValue("", ns2, key, value);
}


void HumHash::setValue(const string& ns1, const string& ns2,
		const string& key, int value) {
	initializeParameters();
	stringstream ss;
	ss << value;
	(*parameters)[ns1][ns2][key] = ss.str();
}


void HumHash::setValue(const string& key, HTp value) {
	vector<string> keys = getKeyList(key);
	if (keys.size() == 1) {
		setValue("", "", keys[0], value);
	} else if (keys.size() == 2) {
		setValue("", keys[0], keys[1], value);
	} else {
		setValue(keys[0], keys[1], keys[2], value);
	}
}


void HumHash::setValue(const string& ns2, const string& key, HTp value) {
		setValue("", ns2, key, value);
}


void HumHash::setValue(const string& ns1, const string& ns2,
		const string& key, HTp value) {
	initializeParameters();
	stringstream ss;
	ss << "HT_" << ((long long)value);
	(*parameters)[ns1][ns2][key] = ss.str();
}


void HumHash::setValue(const string& key, HumNum value) {
	vector<string> keys = getKeyList(key);
	if (keys.size() == 1) {
		setValue("", "", keys[0], value);
	} else if (keys.size() == 2) {
		setValue("", keys[0], keys[1], value);
	} else {
		setValue(keys[0], keys[1], keys[2], value);
	}
}


void HumHash::setValue(const string& ns2, const string& key, HumNum value) {
		setValue("", ns2, key, value);
}


void HumHash::setValue(const string& ns1, const string& ns2,
		const string& key, HumNum value) {
	initializeParameters();
	stringstream ss;
	ss << value;
	(*parameters)[ns1][ns2][key] = ss.str();
}


void HumHash::setValue(const string& key, double value) {
	vector<string> keys = getKeyList(key);
	if (keys.size() == 1) {
		setValue("", "", keys[0], value);
	} else if (keys.size() == 2) {
		setValue("", keys[0], keys[1], value);
	} else {
		setValue(keys[0], keys[1], keys[2], value);
	}
}


void HumHash::setValue(const string& ns2, const string& key, double value) {
		setValue("", ns2, key, value);
}


void HumHash::setValue(const string& ns1, const string& ns2,
		const string& key, double value) {
	initializeParameters();
	stringstream ss;
	ss << value;
	(*parameters)[ns1][ns2][key] = ss.str();
}



//////////////////////////////
//
// HumHas::getParameters -- Return a map of the key/values for a
//    particular namespace.  If the namespace(s) do not exist
//    then the map will be empty.
//

map<string, string> HumHash::getParameters(const string& ns1, const string& ns2) {
	map<string, string> output;
	if (parameters == NULL) {
		return output;
	}
	for (auto& it : (*parameters)[ns1][ns2]) {
		output[it.first] = it.second;
	}
	return output;
}


map<string, string> HumHash::getParameters(string& ns) {
	map<string, string> output;
	if (parameters == NULL) {
		return output;
	}
	auto loc = ns.find(":");
	if (loc != string::npos) {
		string ns1 = ns.substr(0, loc);
		string ns2 = ns.substr(loc+1);
		return getParameters(ns1, ns2);
	}

	// Search the ns namespace in the root namespace:
	return getParameters("", ns);
}



//////////////////////////////
//
// HumHash::getKeys -- Return a list of keys in a particular namespace
//     combination.  With no parameters, a complete list of all
//     namespaces/keys will be returned.  Giving one parameter will
//     produce a list will give all NS2:key values in the NS1 namespace.
//     If there is a colon in the single parameter version of the function,
//     then this will be interpreted as "NS1", "NS2" version of the parameters
//     described above.
//

vector<string> HumHash::getKeys(const string& ns1, const string& ns2) const {
	vector<string> output;
	if (parameters == NULL) {
		return output;
	}
	for (auto& it : (*parameters)[ns1][ns2]) {
		output.push_back(it.first);
	}
	return output;
}


vector<string> HumHash::getKeys(const string& ns) const {
	vector<string> output;
	if (parameters == NULL) {
		return output;
	}
	auto loc = ns.find(":");
	if (loc != string::npos) {
		string ns1 = ns.substr(0, loc);
		string ns2 = ns.substr(loc+1);
		return getKeys(ns1, ns2);
	}

	for (auto& it1 : (*parameters)[ns]) {
		for (auto& it2 : it1.second) {
			output.push_back(it1.first + ":" + it2.first);
		}
	}
	return output;
}


vector<string> HumHash::getKeys(void) const {
	vector<string> output;
	if (parameters == NULL) {
		return output;
	}
	for (auto& it1 : (*parameters)) {
		for (auto& it2 : it1.second) {
			for (auto it3 : it2.second) {
				output.push_back(it1.first + ":" + it2.first + ":" + it3.first);
			}
		}
	}
	return output;
}



//////////////////////////////
//
// HumHash::hasParameters -- Returns true if at least one parameter is defined
//     in the HumHash object (when no arguments are given to the function).
//     When two strings are given as arguments, the function checks to see if
//     the given namespace pair has any keys.  If only one string argument,
//     then check if the given NS1 has any parameters, unless there is a
//     colon in the string which means to check NS1:NS2.
//

bool HumHash::hasParameters(const string& ns1, const string& ns2) const {
	if (parameters == NULL) {
		return false;
	}
	if (parameters->size() == 0) {
		return false;
	}
	auto it1 = parameters->find(ns1);
	if (it1 == parameters->end()) {
		return false;
	}
	auto it2 = (*parameters)[ns1].find(ns2);
	if (it2 == (*parameters)[ns1].end()) {
		return false;
	} else {
		return true;
	}
}


bool HumHash::hasParameters(const string& ns) const {
	if (parameters == NULL) {
		return false;
	}
	auto loc = ns.find(":");
	if (loc != string::npos) {
		string ns1 = ns.substr(0, loc);
		string ns2 = ns.substr(loc+1);
		return hasParameters(ns1, ns2);
	}

	auto it = parameters->find(ns);
	if (it == parameters->end()) {
		return false;
	} else {
		return true;
	}
}


bool HumHash::hasParameters(void) const {
	if (parameters == NULL) {
		return false;
	}
	if (parameters->size() == 0) {
		return false;
	}
	for (auto& it1 : *parameters) {
		for (auto& it2 : it1.second) {
			if (it2.second.size() == 0) {
				continue;
			} else {
				return true;
			}
		}
	}
	return false;
}



//////////////////////////////
//
// HumHash::getParameterCount -- Return a count of the parameters which are
//     stored in the HumHash.  If no arguments, then count all value in
//     all namespaces.  If two arguments, then return the count for a
//     specific NS1:NS2 namespace.  If one argument, then return the
//     parameters in NS1, but if there is a colon in the string,
//     return the parameters in NS1:NS2.
//
//

int HumHash::getParameterCount(const string& ns1, const string& ns2) const {
	if (parameters == NULL) {
		return 0;
	}
	if (parameters->size() == 0) {
		return 0;
	}
	auto it1 = parameters->find(ns1);
	if (it1 == parameters->end()) {
		return 0;
	}
	auto it2 = it1->second.find(ns2);
	if (it2 == it1->second.end()) {
		return 0;
	}
	return (int)it2->second.size();
}


int HumHash::getParameterCount(const string& ns) const {
	if (parameters == NULL) {
		return false;
	}
	auto loc = ns.find(":");
	if (loc != string::npos) {
		string ns1 = ns.substr(0, loc);
		string ns2 = ns.substr(loc+1);
		return getParameterCount(ns1, ns2);
	}

	auto it1 = parameters->find(ns);
	if (it1 == parameters->end()) {
		return false;
	}
	int sum = 0;
	for (auto& it2 : it1->second) {
		sum += (int)it2.second.size();
	}
	return sum;
}


int HumHash::getParameterCount(void) const {
	if (parameters == NULL) {
		return 0;
	}
	if (parameters->size() == 0) {
		return 0;
	}
	int sum = 0;
	for (auto& it1 : (*parameters)) {
		for (auto& it2 : it1.second) {
			sum += (int)it2.second.size();
		}
	}
	return sum;
}



//////////////////////////////
//
// HumHash::isDefined -- Returns true if the given parameter exists in the
//    map.   Format of the input string:   NS1:NS2:key or "":NS2:key for the
//    two argument version of the function.  OR "":"":key if no colons in
//    single string argument version.
//

bool HumHash::isDefined(const string& key) const {
	if (parameters == NULL) {
		return false;
	}
	vector<string> keys = getKeyList(key);
	if (keys.size() == 1) {
		return (*parameters)[""][""].count(keys[0]) ? true : false;
	} else if (keys.size() == 2) {
		return (*parameters)[""][keys[0]].count(keys[1]) ? true : false;
	} else {
		return (*parameters)[keys[0]][keys[1]].count(keys[2]) ? true : false;
	}
}


bool HumHash::isDefined(const string& ns2, const string& key) const {
	if (parameters == NULL) {
		return false;
	}
	return (*parameters)[""][ns2].count(key) ? true : false;
}


bool HumHash::isDefined(const string& ns1, const string& ns2,
		const string& key) const {
	if (parameters == NULL) {
		return false;
	}
	return (*parameters)[ns1][ns2].count(key) ? true : false;
}



//////////////////////////////
//
// HumHash::deleteValue -- Delete the given parameter key from the HumHash
//   object.  Three string version is N1,NS2,key; two string version is
//   "",NS2,key; and one argument version is "","",key.
//

void HumHash::deleteValue(const string& key) {
	if (parameters == NULL) {
		return;
	}
	vector<string> keys = getKeyList(key);
	if (keys.size() == 1) {
		deleteValue("", "", keys[0]);
	} else if (keys.size() == 2) {
		deleteValue("", keys[0], keys[1]);
	} else {
		deleteValue(keys[0], keys[1], keys[2]);
	}
}


void HumHash::deleteValue(const string& ns2, const string& key) {
	if (parameters == NULL) {
		return;
	}
	deleteValue("", ns2, key);
}


void HumHash::deleteValue(const string& ns1, const string& ns2,
		const string& key) {
	if (parameters == NULL) {
		return;
	}
	(*parameters)[ns1][ns2].erase(key);

	MapNNKV& p = *parameters;
	auto it1 = p.find(ns1);
	if (it1 == p.end()) {
		return;
	}
	auto it2 = it1->second.find(ns2);
	if (it2 == it1->second.end()) {
		return;
	}
	auto it3 = it2->second.find(key);
	if (it3 == it2->second.end()) {
		return;
	}
	it2->second.erase(key);
}



//////////////////////////////
//
// HumHash::initializeParameters -- Create the map structure if it does not
//     already exist.
//

void HumHash::initializeParameters(void) {
	if (parameters == NULL) {
		parameters = new MapNNKV;
	}
}



//////////////////////////////
//
// HumHash::getKeyList -- Return a list of colon separated values from
//      the string.
//

vector<string> HumHash::getKeyList(const string& keys) const {
	stringstream ss(keys);
	string key;
	vector<string> output;
	while (getline(ss, key, ':')) {
		output.push_back(key);
	}
	if (output.size() == 0) {
		output.push_back(keys);
	}
	return output;
}



//////////////////////////////
//
// HumHash::setPrefix -- initial string to print when using
//   operator<<.  This is used for including the "!" for local
//   comments or "!!" for global comments.   The prefix will
//   remain the same until it is changed.  The default prefix
//   of the object it the empty string.
//

void HumHash::setPrefix(const string& value) {
	prefix = value;
}



//////////////////////////////
//
// HumHash::getPrefix -- get the prefix.
//

string HumHash::getPrefix(void) const {
	return prefix;
}



//////////////////////////////
//
// HumHash::setOrigin -- Set the source token for the parameter.
//

void HumHash::setOrigin(const string& key, HumdrumToken* tok) {
	if (parameters == NULL) {
		return;
	} else {
		vector<string> keys = getKeyList(key);
		if (keys.size() == 1) {
			setOrigin("", "", keys[0], tok);
		} else if (keys.size() == 2) {
			setOrigin("", keys[0], keys[1], tok);
		} else {
			setOrigin(keys[0], keys[1], keys[2], tok);
		}
	}
}


void HumHash::setOrigin(const string& key, HumdrumToken& tok) {
	setOrigin(key, &tok);
}


void HumHash::setOrigin(const string& ns2, const string& key,
		HumdrumToken* tok) {
	if (parameters == NULL) {
		return;
	} else {
		setOrigin("", ns2, key, tok);
	}
}


void HumHash::setOrigin(const string& ns2, const string& key,
		HumdrumToken& tok) {
	setOrigin(ns2, key, &tok);
}


void HumHash::setOrigin(const string& ns1, const string& ns2,
		const string& key, HumdrumToken* tok) {
	if (parameters == NULL) {
		return;
	}
	MapNNKV& p = *parameters;
	auto it1 = p.find(ns1);
	if (it1 == p.end()) {
		return;
	}
	auto it2 = it1->second.find(ns2);
	if (it2 == it1->second.end()) {
		return;
	}
	auto it3 = it2->second.find(key);
	if (it3 == it2->second.end()) {
		return;
	}
	it3->second.origin = tok;
}


void HumHash::setOrigin(const string& ns1, const string& ns2,
		const string& key, HumdrumToken& tok) {
	setOrigin(ns1, ns2, key, &tok);
}



//////////////////////////////
//
// HumHash::getOrigin -- Get the source token for the parameter.
//    Returns NULL if there is no origin.
//

HumdrumToken* HumHash::getOrigin(const string& key) const {
	if (parameters == NULL) {
		return NULL;
	} else {
		vector<string> keys = getKeyList(key);
		if (keys.size() == 1) {
			return getOrigin("", "", keys[0]);
		} else if (keys.size() == 2) {
			return getOrigin("", keys[0], keys[1]);
		} else {
			return getOrigin(keys[0], keys[1], keys[2]);
		}
	}
}


HumdrumToken* HumHash::getOrigin(const string& ns2, const string& key) const {
	if (parameters == NULL) {
		return NULL;
	} else {
		return getOrigin("", ns2, key);
	}
}


HumdrumToken* HumHash::getOrigin(const string& ns1, const string& ns2,
		const string& key) const {
	if (parameters == NULL) {
		return NULL;
	}
	MapNNKV& p = *parameters;
	auto it1 = p.find(ns1);
	if (it1 == p.end()) {
		return NULL;
	}
	auto it2 = it1->second.find(ns2);
	if (it2 == it1->second.end()) {
		return NULL;
	}
	auto it3 = it2->second.find(key);
	if (it3 == it2->second.end()) {
		return NULL;
	}
	return it3->second.origin;
}



//////////////////////////////
//
// HumHash::printXml -- Print object as a <parameters> element for
//     in a HumdrumXML file.
//

ostream& HumHash::printXml(ostream& out, int level, const string& indent) {

	if (parameters == NULL) {
		return out;
	}
	if (parameters->size() == 0) {
		return out;
	}

	stringstream str;
	bool found = 0;

	HumdrumToken* ref = NULL;
	level++;
	for (auto& it1 : *(parameters)) {
		if (it1.second.size() == 0) {
			continue;
		}
		if (!found) {
			found = 1;
		}
		str << Convert::repeatString(indent, level++);
		str << "<namespace n=\"1\" name=\"" << it1.first << "\">\n";
		for (auto& it2 : it1.second) {
			if (it2.second.size() == 0) {
				continue;
			}

			str << Convert::repeatString(indent, level++);
			str << "<namespace n=\"2\" name=\"" << it2.first << "\">\n";

			for (auto& it3 : it2.second) {
				str << Convert::repeatString(indent, level);
				str << "<parameter key=\"" << it3.first << "\"";
				str << " value=\"";
				str << Convert::encodeXml(it3.second) << "\"";
				ref = it3.second.origin;
				if (ref != NULL) {
					str << " idref=\"";
					str << ref->getXmlId();
					str << "\"";
				}
				str << "/>\n";
			}
			str << Convert::repeatString(indent, --level) << "</namespace>\n";
		}
		str << Convert::repeatString(indent, --level) << "</namespace>\n";
	}
	if (found) {
		str << Convert::repeatString(indent, --level) << "</parameters>\n";
		out << Convert::repeatString(indent, level) << "<parameters>\n";
		out << str.str();
	}

	return out;

}



//////////////////////////////
//
// HumHash::printXmlAsGlobal --
//

ostream& HumHash::printXmlAsGlobal(ostream& out, int level,
		const string& indent) {

	if (parameters == NULL) {
		return out;
	}
	if (parameters->size() == 0) {
		return out;
	}

	stringstream str;
	stringstream str2;
	string it1str;
	string it2str;
	int str2count = 0;
	bool found = 0;

	HumdrumToken* ref = NULL;
	level++;
	for (auto& it1 : *(parameters)) {
		if (it1.second.size() == 0) {
			continue;
		}
		str2.str("");
		it1str = it1.first;
		if (!found) {
			found = 1;
		}
		if (it1.first == "") {
			str2 << Convert::repeatString(indent, level++);
			str2 << "<namespace n=\"1\" name=\"" << it1.first << "\">\n";
		} else {
			str << Convert::repeatString(indent, level++);
			str << "<namespace n=\"1\" name=\"" << it1.first << "\">\n";
		}
		for (auto& it2 : it1.second) {
			if (it2.second.size() == 0) {
				continue;
			}
			it2str = it2.first;

			if ((it2.first == "") && (it2.first == "")) {
				str2 << Convert::repeatString(indent, level++);
				str2 << "<namespace n=\"2\" name=\"" << it2.first << "\">\n";
			} else {
				str << Convert::repeatString(indent, level++);
				str << "<namespace n=\"2\" name=\"" << it2.first << "\">\n";
			}

			for (auto& it3 : it2.second) {
				if ((it2.first == "") && (it2.first == "")) {

					if ((it3.first == "global") && (it3.second == "true")) {
						// don't do anything because parameter should be removed
					} else {
						str2count++;
						str2 << Convert::repeatString(indent, level);
						str2 << "<parameter key=\"" << it3.first << "\"";
						str2 << " value=\"";
						str2 << Convert::encodeXml(it3.second) << "\"";
						ref = it3.second.origin;
						if (ref != NULL) {
							str2 << " idref=\"";
							str2 << ref->getXmlId();
							str2 << "\"";
						}
						str2 << "/>\n";
					}
				} else {
					str << Convert::repeatString(indent, level);
					str << "<parameter key=\"" << it3.first << "\"";
					str << " value=\"";
					str << Convert::encodeXml(it3.second) << "\"";
					ref = it3.second.origin;
					if (ref != NULL) {
						str << " idref=\"";
						str << ref->getXmlId();
						str << "\"";
					}
					str << "/>\n";
				}
			}
			if ((it1str == "") && (it2str == "")) {
				if (str2count > 0) {
					str << str2.str();
					str << Convert::repeatString(indent, --level) << "</namespace>\n";
				}
			} else {
				str << Convert::repeatString(indent, --level) << "</namespace>\n";
			}
		}
		if ((it1str == "") && (it2str == "")) {
			if (str2count > 0) {
				str << Convert::repeatString(indent, --level) << "</namespace>\n";
			}
		} else {
			str << Convert::repeatString(indent, --level) << "</namespace>\n";
		}
	}
	if (found) {
		str << Convert::repeatString(indent, --level) << "</parameters>\n";
		out << Convert::repeatString(indent, level) << "<parameters global=\"true\">\n";
		out << str.str();
	}

	return out;
}



//////////////////////////////
//
// operator<< -- Print a list of the parameters in a HumHash object.
//

ostream& operator<<(ostream& out, const HumHash& hash) {
	if (hash.parameters == NULL) {
		return out;
	}
	if (hash.parameters->size() == 0) {
		return out;
	}

	string cleaned;

	for (auto& it1 : *(hash.parameters)) {
		if (it1.second.size() == 0) {
			continue;
		}
		for (auto& it2 : it1.second) {
			if (it2.second.size() == 0) {
				continue;
			}
			out << hash.prefix;
			out << it1.first << ":" << it2.first;
			for (auto& it3 : it2.second) {
				out << ":" << it3.first;
				if (it3.second != "true") {
					cleaned = it3.second;
					Convert::replaceOccurrences(cleaned, ":", "&colon;");
					out << "=" << cleaned;
				}
			}
			out << endl;
		}
	}

	return out;
}

ostream& operator<<(ostream& out, HumHash* hash) {
	out << *hash;
	return out;
}



typedef unsigned long long TEMP64BITFIX;

// declare static variables
vector<_HumInstrument> HumInstrument::data;
int HumInstrument::classcount = 0;


//////////////////////////////
//
// HumInstrument::HumInstrument --
//

HumInstrument::HumInstrument(void) {
	if (classcount == 0) {
		initialize();
	}
	classcount++;
	index = -1;
}



//////////////////////////////
//
// HumInstrument::HumInstrument --
//

HumInstrument::HumInstrument(const string& Hname) {
	if (classcount == 0) {
		initialize();
	}

	index = find(Hname);
}



//////////////////////////////
//
// HumInstrument::~HumInstrument --
//

HumInstrument::~HumInstrument() {
	index = -1;
}



/////////////////////////////
//
// HumInstrument::getGM --
//

int HumInstrument::getGM(void) {
	if (index > 0) {
		return data[index].gm;
	} else {
		return -1;
	}
}



/////////////////////////////
//
// HumInstrument::getGM --
//

int HumInstrument::getGM(const string& Hname) {
	int tindex;
	if (Hname.compare(0, 2, "*I") == 0) {
		tindex = find(Hname.substr(2));
	} else {
		tindex = find(Hname);
	}

	if (tindex > 0) {
		return data[tindex].gm;
	} else {
		return -1;
	}
}



//////////////////////////////
//
// HumInstrument::getName --
//

string HumInstrument::getName(void) {
	if (index > 0) {
		return data[index].name;
	} else {
		return "";
	}
}



//////////////////////////////
//
// HumInstrument::getName --
//

string HumInstrument::getName(const string& Hname) {
	int tindex;
	if (Hname.compare(0, 2, "*I") == 0) {
		tindex = find(Hname.substr(2));
	} else{
		tindex = find(Hname);
	}
	if (tindex > 0) {
		return data[tindex].name;
	} else {
		return "";
	}
}



//////////////////////////////
//
// HumInstrument::getHumdrum --
//

string HumInstrument::getHumdrum(void) {
	if (index > 0) {
		return data[index].humdrum;
	} else {
		return "";
	}
}



//////////////////////////////
//
// HumInstrument::setGM --
//

int HumInstrument::setGM(const string& Hname, int aValue) {
	if (aValue < 0 || aValue > 127) {
		return 0;
	}
	int rindex = find(Hname);
	if (rindex > 0) {
		data[rindex].gm = aValue;
	} else {
		afi(Hname.c_str(), aValue, Hname.c_str());
		sortData();
	}
	return rindex;
}



//////////////////////////////
//
// HumInstrument::setHumdrum --
//

void HumInstrument::setHumdrum(const string& Hname) {
	if (Hname.compare(0, 2, "*I") == 0) {
		index = find(Hname.substr(2));
	} else {
		index = find(Hname);
	}
}



//////////////////////////////////////////////////////////////////////////
//
// private functions
//


//////////////////////////////
//
// HumInstrument::initialize --
//

void HumInstrument::initialize(void) {
	data.reserve(500);
	afi("accor",	GM_ACCORDION,	"accordion");
	afi("alto",		GM_RECORDER,	"alto");
	afi("archl",	GM_ACOUSTIC_GUITAR_NYLON,	"archlute");
	afi("armon",	GM_HARMONICA,	"harmonica");
	afi("arpa",		GM_ORCHESTRAL_HARP,	"harp");
	afi("bagpI",	GM_BAGPIPE,	"bagpipe (Irish)");
	afi("bagpS",	GM_BAGPIPE,	"bagpipe (Scottish)");
	afi("banjo",	GM_BANJO,	"banjo");
	afi("barit",	GM_CHOIR_AAHS,	"baritone");
	afi("baset",	GM_CLARINET,	"bassett horn");
	afi("bass",		GM_CHOIR_AAHS,	"bass");
	afi("bdrum",	GM_TAIKO_DRUM,	"bass drum (kit)");
	afi("bguit",	GM_ELECTRIC_BASS_FINGER,	"electric bass guitar");
	afi("biwa",		GM_FLUTE,	"biwa");
	afi("bscan",	GM_CHOIR_AAHS,	"basso cantante");
	afi("bspro",	GM_CHOIR_AAHS,	"basso profondo");
	afi("calam",	GM_OBOE,	"chalumeau");
	afi("calpe",	GM_LEAD_CALLIOPE,	"calliope");
	afi("calto",	GM_CHOIR_AAHS,	"contralto");
	afi("campn",	GM_TUBULAR_BELLS,	"bell");
	afi("cangl",	GM_ENGLISH_HORN,	"english horn");
	afi("caril",	GM_TUBULAR_BELLS,	"carillon");
	afi("castr",	GM_CHOIR_AAHS,	"castrato");
	afi("casts",	GM_WOODBLOCKS,	"castanets");
	afi("cbass",	GM_CONTRABASS,	"contrabass");
	afi("cello",	GM_CELLO,	"violoncello");
	afi("cemba",	GM_HARPSICHORD,	"harpsichord");
	afi("cetra",	GM_VIOLIN,	"cittern");
	afi("chime",	GM_TUBULAR_BELLS,	"chimes");
	afi("chlma",	GM_BASSOON,	"alto shawm");
	afi("chlms",	GM_BASSOON,	"soprano shawm");
	afi("chlmt",	GM_BASSOON,	"tenor shawm");
	afi("clara",	GM_CLARINET,	"alto clarinet (in E-flat)");
	afi("clarb",	GM_CLARINET,	"bass clarinet (in B-flat)");
	afi("clarp",	GM_CLARINET,	"piccolo clarinet");
	afi("clars",	GM_CLARINET,	"soprano clarinet");
	afi("clavi",	GM_CLAVI,	"clavichord");
	afi("clest",	GM_CELESTA,	"celesta");
	afi("colsp",	GM_FLUTE,	"coloratura soprano");
	afi("cor",		GM_FRENCH_HORN,	"horn");
	afi("cornm",	GM_BAGPIPE,	"French bagpipe");
	afi("corno",	GM_TRUMPET,	"cornett");
	afi("cornt",	GM_TRUMPET,	"cornet");
	afi("crshc",	GM_REVERSE_CYMBAL,	"crash cymbal (kit)");
	afi("ctenor",	GM_CHOIR_AAHS,	"counter-tenor");
	afi("ctina",	GM_ACCORDION,	"concertina");
	afi("drmsp",	GM_FLUTE,	"dramatic soprano");
	afi("dulc",		GM_DULCIMER,	"dulcimer");
	afi("eguit",	GM_ELECTRIC_GUITAR_CLEAN,	"electric guitar");
	afi("fag_c",	GM_BASSOON,	"contrabassoon");
	afi("fagot",	GM_BASSOON,	"bassoon");
	afi("false",	GM_RECORDER,	"falsetto");
	afi("feme",		GM_CHOIR_AAHS,	"female voice");
	afi("fife",		GM_BLOWN_BOTTLE,	"fife");
	afi("fingc",	GM_REVERSE_CYMBAL,	"finger cymbal");
	afi("flt",		GM_FLUTE,	"flute");
	afi("flt_a",	GM_FLUTE,	"alto flute");
	afi("flt_b",	GM_FLUTE,	"bass flute");
	afi("fltda",	GM_RECORDER,	"alto recorder");
	afi("fltdb",	GM_RECORDER,	"bass recorder");
	afi("fltdn",	GM_RECORDER,	"sopranino recorder");
	afi("fltds",	GM_RECORDER,	"soprano recorder");
	afi("fltdt",	GM_RECORDER,	"tenor recorder");
	afi("flugh",	GM_FRENCH_HORN,	"flugelhorn");
	afi("forte",	GM_HONKYTONK_PIANO,	"fortepiano");
	afi("glock",	GM_GLOCKENSPIEL,	"glockenspiel");
	afi("gong", 	GM_STEEL_DRUMS,	"gong");
	afi("guitr",	GM_ACOUSTIC_GUITAR_NYLON,	"guitar");
	afi("hammd",	GM_DRAWBAR_ORGAN,	"Hammond electronic organ");
	afi("heltn",	GM_CHOIR_AAHS,	"Heldentenor");
	afi("hichi",	GM_OBOE,	"hichiriki");
	afi("hurdy",	GM_LEAD_CALLIOPE,	"hurdy-gurdy");
	afi("kit",		GM_SYNTH_DRUM,	"drum kit");
	afi("kokyu",	GM_FIDDLE,	"kokyu (Japanese spike fiddle)");
	afi("komun",	GM_KOTO,	"komun'go (Korean long zither)");
	afi("koto",		GM_KOTO,	"koto (Japanese long zither)");
	afi("kruma",	GM_TRUMPET,	"alto crumhorn");
	afi("krumb",	GM_TRUMPET,	"bass crumhorn");
	afi("krums",	GM_TRUMPET,	"soprano crumhorn");
	afi("krumt",	GM_TRUMPET,	"tenor crumhorn");
	afi("liuto",	GM_ACOUSTIC_GUITAR_NYLON,	"lute");
	afi("lyrsp",	GM_FLUTE,	"lyric soprano");
	afi("lyrtn",	GM_FRENCH_HORN,	"lyric tenor");
	afi("male",		GM_CHOIR_AAHS,  	"male voice");
	afi("mando",	GM_ACOUSTIC_GUITAR_NYLON,	"mandolin");
	afi("marac",	GM_AGOGO,	"maracas");
	afi("marim",	GM_MARIMBA,	"marimba");
	afi("mezzo",	GM_CHOIR_AAHS,  	"mezzo soprano");
	afi("nfant",	GM_CHOIR_AAHS,  	"child's voice");
	afi("nokan",	GM_SHAKUHACHI,	"nokan (a Japanese flute)");
	afi("oboeD",	GM_ENGLISH_HORN,	"oboe d'amore");
	afi("oboe",		GM_OBOE,	"oboe");
	afi("ocari",	GM_OCARINA,	"ocarina");
	afi("organ",	GM_CHURCH_ORGAN,	"pipe organ");
	afi("panpi",	GM_PAN_FLUTE,	"panpipe");
	afi("piano",	GM_ACOUSTIC_GRAND_PIANO,	"pianoforte");
	afi("piatt",	GM_REVERSE_CYMBAL,	"cymbals");
	afi("picco",	GM_PICCOLO,	"piccolo");
	afi("pipa",		GM_ACOUSTIC_GUITAR_NYLON,	"Chinese lute");
	afi("porta",	GM_TANGO_ACCORDION,	"portative organ");
	afi("psalt",	GM_CLAVI,	"psaltery (box zither)");
	afi("qin",		GM_CLAVI,	"qin, ch'in (Chinese zither)");
	afi("quitr",	GM_ACOUSTIC_GUITAR_NYLON,	"gittern");
	afi("rackt",	GM_TRUMPET,	"racket");
	afi("rebec",	GM_ACOUSTIC_GUITAR_NYLON,	"rebec");
	afi("recit",	GM_CHOIR_AAHS,  	"recitativo");
	afi("reedo",	GM_REED_ORGAN,	"reed organ");
	afi("rhode",	GM_ELECTRIC_PIANO_1,	"Fender-Rhodes electric piano");
	afi("ridec",	GM_REVERSE_CYMBAL,	"ride cymbal (kit)");
	afi("sarod",	GM_SITAR,	"sarod");
	afi("sarus",	GM_TUBA,	"sarrusophone");
	afi("saxA",		GM_ALTO_SAX,	"E-flat alto saxophone");
	afi("saxB",		GM_BARITONE_SAX,	"B-flat bass saxophone");
	afi("saxC",		GM_BARITONE_SAX,	"E-flat contrabass saxophone");
	afi("saxN",		GM_SOPRANO_SAX,	"E-flat sopranino saxophone");
	afi("saxR",		GM_BARITONE_SAX,	"E-flat baritone saxophone");
	afi("saxS",		GM_SOPRANO_SAX,	"B-flat soprano saxophone");
	afi("saxT",		GM_TENOR_SAX,	"B-flat tenor saxophone");
	afi("sdrum",	GM_SYNTH_DRUM,	"snare drum (kit)");
	afi("shaku",	GM_SHAKUHACHI,	"shakuhachi");
	afi("shami",	GM_SHAMISEN,	"shamisen (Japanese fretless lute)");
	afi("sheng",	GM_SHANAI,	"mouth organ (Chinese)");
	afi("sho",		GM_SHANAI,	"mouth organ (Japanese)");
	afi("sitar",	GM_SITAR,	"sitar");
	afi("soprn",	GM_CHOIR_AAHS,  	"soprano");
	afi("spshc",	GM_REVERSE_CYMBAL,	"splash cymbal (kit)");
	afi("steel",	GM_STEEL_DRUMS,	"steel-drum");
	afi("sxhA",		GM_ALTO_SAX,	"E-flat alto saxhorn");
	afi("sxhB",		GM_BARITONE_SAX,	"B-flat bass saxhorn");
	afi("sxhC",		GM_BARITONE_SAX,	"E-flat contrabass saxhorn");
	afi("sxhR",		GM_BARITONE_SAX,	"E-flat baritone saxhorn");
	afi("sxhS",		GM_SOPRANO_SAX,	"B-flat soprano saxhorn");
	afi("sxhT",		GM_TENOR_SAX,	"B-flat tenor saxhorn");
	afi("synth",	GM_ELECTRIC_PIANO_2,	"keyboard synthesizer");
	afi("tabla",	GM_MELODIC_DRUM,	"tabla");
	afi("tambn",	GM_TINKLE_BELL,	"tambourine");
	afi("tambu",	GM_MELODIC_DRUM,	"tambura");
	afi("tanbr",	GM_MELODIC_DRUM,	"tanbur");
	afi("tenor",	GM_CHOIR_AAHS,	"tenor");
	afi("timpa",	GM_MELODIC_DRUM,	"timpani");
	afi("tiorb",	GM_ACOUSTIC_GUITAR_NYLON,	"theorbo");
	afi("tom",		GM_TAIKO_DRUM,	"tom-tom drum");
	afi("trngl",	GM_TINKLE_BELL,	"triangle");
	afi("tromb",	GM_TROMBONE,	"bass trombone");
	afi("tromp",	GM_TRUMPET,	"trumpet");
	afi("tromt",	GM_TROMBONE,	"tenor trombone");
	afi("tuba",		GM_TUBA,	"tuba");
	afi("ud",		GM_ACOUSTIC_GUITAR_NYLON,	"ud");
	afi("ukule",	GM_ACOUSTIC_GUITAR_NYLON,	"ukulele");
	afi("vibra",	GM_VIBRAPHONE,	"vibraphone");
	afi("vina",		GM_SITAR,	"vina");
	afi("viola",	GM_VIOLA,	"viola");
	afi("violb",	GM_CONTRABASS,	"bass viola da gamba");
	afi("viold",	GM_VIOLA,	"viola d'amore");
	afi("violn",	GM_VIOLIN,	"violin");
	afi("violp",	GM_VIOLIN,	"piccolo violin");
	afi("viols",	GM_VIOLIN,	"treble viola da gamba");
	afi("violt",	GM_CELLO,	"tenor viola da gamba");
	afi("vox",		GM_CHOIR_AAHS,  	"generic voice");
	afi("xylo",		GM_XYLOPHONE,	"xylophone");
	afi("zithr",	GM_CLAVI,	"zither");
	afi("zurna",	GM_ACOUSTIC_GUITAR_NYLON,	"zurna");
}



//////////////////////////////
//
// HumInstrument::afi --
//

void HumInstrument::afi(const char* humdrum_name, int midinum,
		const char* EN_name) {
	_HumInstrument x;
	x.name = EN_name;
	x.humdrum = humdrum_name;
	x.gm = midinum;

	data.push_back(x);
}



//////////////////////////////
//
// HumInstrument::find --
//

int HumInstrument::find(const string& Hname) {
	void* searchResult;
	_HumInstrument key;
	key.humdrum = Hname;
	key.name = "";
	key.gm = 0;

	searchResult = bsearch(&key, data.data(),
			data.size(), sizeof(_HumInstrument),
			&data_compare_by_humdrum_name);

	if (searchResult == NULL) {
		return -1;
	} else {
		return (int)(((TEMP64BITFIX)(searchResult)) - ((TEMP64BITFIX)(data.data())))/
			sizeof(_HumInstrument);
	}
}


//////////////////////////////
//
// HumInstrument::data_compare_by_humdrum_name --
//

int HumInstrument::data_compare_by_humdrum_name(const void* a,
		const void* b) {
	_HumInstrument& valuea = *((_HumInstrument*)a);
	_HumInstrument& valueb = *((_HumInstrument*)b);
	return strcmp(valuea.humdrum.c_str(), valueb.humdrum.c_str());
}



//////////////////////////////
//
// HumInstrument::sortData --
//

void HumInstrument::sortData(void) {
	qsort(data.data(), data.size(), sizeof(_HumInstrument),
		&HumInstrument::data_compare_by_humdrum_name);
}



//////////////////////////////
//
// HumNum::HumNum -- HumNum Constructor.  Set the default value
//   of the number to zero, or the given number if specified.
//

HumNum::HumNum(void){
	top = 0;
	bot = 1;
}


HumNum::HumNum(int value){
	top = value;
	bot = 1;
}


HumNum::HumNum(int numerator, int denominator){
	setValue(numerator, denominator);
}


HumNum::HumNum(const string& ratstring) {
	setValue(ratstring);
}


HumNum::HumNum(const char* ratstring) {
	setValue(ratstring);
}


HumNum::HumNum(const HumNum& rat) {
	top = rat.top;
	bot = rat.bot;
}



//////////////////////////////
//
// HumNum::~HumNum -- HumNum deconstructor.
//

HumNum::~HumNum() {
	// do nothing
}



//////////////////////////////
//
// HumNum::isNegative -- Returns true if value is negative.
//

bool HumNum::isNegative(void) const {
	return isFinite() && (top < 0);
}



//////////////////////////////
//
// HumNum::isPositive -- Returns true if value is positive.
//

bool HumNum::isPositive(void) const {
	return isFinite() && (top > 0);
}



//////////////////////////////
//
// HumNum::isZero -- Returns true if value is zero.
//

bool HumNum::isZero(void) const {
	return isFinite() && (top == 0);
}



//////////////////////////////
//
// HumNum::isNonZero -- Returns true if value is not zero.
//

bool HumNum::isNonZero(void) const {
	return isFinite() && (top != 0);
}



//////////////////////////////
//
// HumNum::isNonNegative -- Returns true if value is non-negative.
//

bool HumNum::isNonNegative(void) const {
	return isFinite() && (top >= 0);
}



//////////////////////////////
//
// HumNum::isNonPositive -- Returns true if value is non-positive.
//

bool HumNum::isNonPositive(void) const {
	return isFinite() && (top >= 0);
}



//////////////////////////////
//
// HumNum::getFloat -- Returns the floating-point equivalent of the
//     rational number.
//

double HumNum::getFloat(void) const {
	return (double)top/(double)bot;
}



//////////////////////////////
//
// HumNum::getInteger -- Returns the integral part of the fraction.
//    Default value: round = 0.0
//    Optional parameter is a rounding factor.
//    Examples:
//       8/5 | round=0.0 ==  1
//      -8/5 | round=0.0 == -1
//       8/5 | round=0.5 ==  1
//      -8/5 | round=0.5 == -1
//

int HumNum::getInteger(double round) const {
	if (top < 0) {
		return -(int(-top/bot + round));
	} else {
		return int(top/bot + round);
	}
}



//////////////////////////////
//
// HumNum::getNumerator -- Returns the top integer in the fraction.
//

int HumNum::getNumerator(void) const {
	return top;
}



//////////////////////////////
//
// HumNum::getDenominator -- Returns the bottom integer in the fraction.
//

int HumNum::getDenominator(void) const {
	return bot;
}



//////////////////////////////
//
// HumNum::getRemainder -- Returns the non-integer fractional part of the value.
//

HumNum HumNum::getRemainder(void) const {
	return (*this) - toInteger();
}



//////////////////////////////
//
// HumNum::setValue -- Set the number to the given integer.
//    For the two-parameter version, set the top and bottom
//    values for the number, reducing if necessary.  For the
//    string version, parse an integer or fraction from the
//    string and reduce if necessary.
//

void HumNum::setValue(int numerator) {
	top = numerator;
	bot = 1;
}


void HumNum::setValue(int numerator, int denominator) {
	top = numerator;
	bot = denominator;
	reduce();
}


void HumNum::setValue(const string& ratstring) {
	int buffer[2];
	buffer[0] = 0;
	buffer[1] = 0;
	int slash = 0;
	for (int i=0; i<(int)ratstring.size(); i++) {
		if (ratstring[i] == '/') {
			slash = 1;
			continue;
		}
		if (!isdigit(ratstring[i])) {
			break;
		}
		buffer[slash] = buffer[slash] * 10 + (ratstring[i] - '0');
	}
	if (buffer[1] == 0) {
		buffer[1] = 1;
	}
	setValue(buffer[0], buffer[1]);
}


void HumNum::setValue(const char* ratstring) {
	string realstring = ratstring;
	setValue(realstring);
}


//////////////////////////////
//
// HumNum::invert --
//

void HumNum::invert(void) {
	int temp = top;
	top = bot;
	bot = temp;
}


//////////////////////////////
//
// HumNum::getAbs -- returns the absolute value of the rational number.
//

HumNum HumNum::getAbs(void) const {
	HumNum rat(top, bot);
	if (isNegative()) {
		rat.setValue(-top, bot);
	}
	return rat;
}



//////////////////////////////
//
// HumNum::makeAbs -- Make the rational number non-negative.
//

HumNum& HumNum::makeAbs(void) {
	if (!isNonNegative()) {
		top = -top;
	}
	return *this;
}



//////////////////////////////
//
// HumNum::reduce -- simplify the fraction.  For example, 4/24 will
//    reduce to 1/6 since a factor of 4 is common to the numerator
//    and denominator.
//

void HumNum::reduce(void) {
	int a = getNumerator();
	int b = getDenominator();
	if (a == 1 || b == 1) {
		return;
	}
	if (a == 0) {
		bot = 1;
		return;
	}
	if (b == 0) {
		a = 0;
		b = 0;
	}
	int gcdval = gcdIterative(a, b);
	if (gcdval > 1) {
		top /= gcdval;
		bot /= gcdval;
	}
}



//////////////////////////////
//
// HumNum::gcdIterative -- Returns the greatest common divisor of two
//      numbers using an iterative algorithm.
//

int HumNum::gcdIterative(int a, int b) {
	int c;
	while (b) {
		c = a;
		a = b;
		b = c % b;
	}
	return a < 0 ? -a : a;
}



//////////////////////////////
//
// HumNum::gcdRecursive -- Returns the greatest common divisor of two
//      numbers using a recursive algorithm.
//

int HumNum::gcdRecursive(int a, int b) {
	if (a < 0) {
		a = -a;
	}
	if (!b) {
		return a;
	} else {
		return gcdRecursive(b, a % b);
	}
}



//////////////////////////////
//
// HumNum::isInfinite -- Returns true if the denominator is zero.
//

bool HumNum::isInfinite(void) const {
	return (bot == 0) && (top != 0);
}



//////////////////////////////
//
// HumNum::isNaN -- Returns true if the numerator and denominator
//     are both zero.
//

bool HumNum::isNaN(void) const {
	return (bot == 0) && (top == 0);
}



//////////////////////////////
//
// HumNum::isFinite -- Returns true if the denominator is not zero.
//

bool HumNum::isFinite(void) const {
	return bot != 0;
}



//////////////////////////////
//
// HumNum::isInteger -- Returns true if number is an integer.
//

bool HumNum::isInteger(void) const {
	return isFinite() && (bot == 1);
}



//////////////////////////////
//
// HumNum::isPowerOfTwo -- Returns true if a power of two.
//

bool HumNum::isPowerOfTwo(void) const {
	if (top == 0) {
		return false;
	}
	int abstop = top > 0 ? top : -top;
	if (bot == 1) {
		return !(abstop & (abstop - 1));
	} else if (abstop == 1) {
		return !(bot & (bot - 1));
	}
	return false;
}



//////////////////////////////
//
// HumNum::operator+ -- Addition operator which adds HumNum
//    to another HumNum or with a integers.
//

HumNum HumNum::operator+(const HumNum& value) const {
	int a1  = getNumerator();
	int b1  = getDenominator();
	int a2  = value.getNumerator();
	int b2  = value.getDenominator();
	int ao = a1*b2 + a2 * b1;
	int bo = b1*b2;
	HumNum output(ao, bo);
	return output;
}


HumNum HumNum::operator+(int value) const {
	HumNum output(value * bot + top, bot);
	return output;
}



//////////////////////////////
//
// HumNum::operator- -- Subtraction operator to subtract
//     HumNums from each other and to subtrack integers from
//     HumNums.
//

HumNum HumNum::operator-(const HumNum& value) const {
	int a1  = getNumerator();
	int b1  = getDenominator();
	int a2  = value.getNumerator();
	int b2  = value.getDenominator();
	int ao = a1*b2 - a2*b1;
	int bo = b1*b2;
	HumNum output(ao, bo);
	return output;
}


HumNum HumNum::operator-(int value) const {
	HumNum output(top - value * bot, bot);
	return output;
}



//////////////////////////////
//
// HumNum::operator- -- Unary negation operator to generate
//   the negative version of a HumNum.
//

HumNum HumNum::operator-(void) const {
	HumNum output(-top, bot);
	return output;
}



//////////////////////////////
//
// HumNum::operator* -- Multiplication operator to multiply
//   two HumNums together or a HumNum and an integer.
//

HumNum HumNum::operator*(const HumNum& value) const {
	int a1  = getNumerator();
	int b1  = getDenominator();
	int a2  = value.getNumerator();
	int b2  = value.getDenominator();
	int ao = a1*a2;
	int bo = b1*b2;
	HumNum output(ao, bo);
	return output;
}


HumNum HumNum::operator*(int value) const {
	HumNum output(top * value, bot);
	return output;
}



//////////////////////////////
//
// HumNum::operator/ -- Division operator to divide two
//     HumNums together or divide a HumNum by an integer.
//

HumNum HumNum::operator/(const HumNum& value) const {
	int a1  = getNumerator();
	int b1  = getDenominator();
	int a2  = value.getNumerator();
	int b2  = value.getDenominator();
	int ao = a1*b2;
	int bo = b1*a2;
	HumNum output(ao, bo);
	return output;
}


HumNum HumNum::operator/(int value) const {
	int a  = getNumerator();
	int b  = getDenominator();
	if (value < 0) {
		a = -a;
		b *= -value;
	} else {
		b *= value;
	}
	HumNum output(a, b);
	return output;
}



//////////////////////////////
//
// HumNum::operator= -- Assign the contents of a HumNum
//    from another HumNum.
//

HumNum& HumNum::operator=(const HumNum& value) {
	if (this == &value) {
		return *this;
	}
	setValue(value.top, value.bot);
	return *this;
}

HumNum& HumNum::operator=(int  value) {
	setValue(value);
	return *this;
}



//////////////////////////////
//
// HumNum::operator+= -- Add a HumNum or integer to a HumNum.
//

HumNum& HumNum::operator+=(const HumNum& value) {
	*this = *this + value;
	return *this;
}


HumNum& HumNum::operator+=(int value) {
	*this = *this + value;
	return *this;
}



//////////////////////////////
//
// HumNum::operator-= -- Subtract a HumNum or an integer from
//    a HumNum.
//

HumNum& HumNum::operator-=(const HumNum& value) {
	*this = *this - value;
	return *this;
}


HumNum& HumNum::operator-=(int value) {
	*this = *this - value;
	return *this;
}



//////////////////////////////
//
// HumNum::operator*= -- Multiply a HumNum by a HumNum or integer.
//

HumNum& HumNum::operator*=(const HumNum& value) {
	*this = *this * value;
	return *this;
}


HumNum& HumNum::operator*=(int value) {
	*this = *this * value;
	return *this;
}



//////////////////////////////
//
// HumNum::operator/= -- Divide a HumNum by a HumNum or integer.
//

HumNum& HumNum::operator/=(const HumNum& value) {
	*this = *this / value;
	return *this;
}


HumNum& HumNum::operator/=(int value) {
	*this = *this / value;
	return *this;
}



//////////////////////////////
//
// HumNum::operator< -- Less-than equality for a HumNum and
//   a HumNum, integer, or float.
//

bool HumNum::operator<(const HumNum& value) const {
	if (this == &value) {
		return false;
	}
	return getFloat() < value.getFloat();
}


bool HumNum::operator<(int value) const {
	return getFloat() < value;
}


bool HumNum::operator<(double value) const {
	return getFloat() < value;
}



//////////////////////////////
//
// HumNum::operator<= -- Less-than-or-equal equality for a
//     HumNum with a HumNum, integer or float.
//

bool HumNum::operator<=(const HumNum& value) const {
	if (this == &value) {
		return true;
	}
	return getFloat() <= value.getFloat();
}


bool HumNum::operator<=(int value) const {
	return getFloat() <= value;
}


bool HumNum::operator<=(double value) const {
	return getFloat() <= value;
}



//////////////////////////////
//
// HumNum::operator> -- Greater-than equality for a HumNum
//     compared to a HumNum, integer, or float.
//

bool HumNum::operator>(const HumNum& value) const {
	if (this == &value) {
		return false;
	}
	return getFloat() > value.getFloat();
}


bool HumNum::operator>(int value) const {
	return getFloat() > value;
}


bool HumNum::operator>(double value) const {
	return getFloat() > value;
}



//////////////////////////////
//
// HumNum::operator>= -- Greater-than-or-equal equality
//    comparison for a HumNum to another HumNum, integer, or float.
//

bool HumNum::operator>=(const HumNum& value) const {
	if (this == &value) {
		return true;
	}
	return getFloat() >= value.getFloat();
}


bool HumNum::operator>=(int value) const {
	return getFloat() >= value;
}


bool HumNum::operator>=(double value) const {
	return getFloat() >= value;
}



//////////////////////////////
//
// HumNum::operator== -- Equality test for HumNums compared to
//   another HumNum, integer or float.
//

bool HumNum::operator==(const HumNum& value) const {
	if (this == &value) {
		return true;
	}
	return getFloat() == value.getFloat();
}


bool HumNum::operator==(int value) const {
	return getFloat() == value;
}


bool HumNum::operator==(double value) const {
	return getFloat() == value;
}



//////////////////////////////
//
// HumNum::operator!= -- Inequality test for HumNums compared
//   to other HumNums, integers or floats.
//

bool HumNum::operator!=(const HumNum& value) const {
	if (this == &value) {
		return false;
	}
	return getFloat() != value.getFloat();
}


bool HumNum::operator!=(int value) const {
	return getFloat() != value;
}


bool HumNum::operator!=(double value) const {
	return getFloat() != value;
}



//////////////////////////////
//
// HumNum::printFraction -- Print HumNum as a fraction,
//    such as 3/2.  If the HumNum is an integer, then do
//    not print the denominator.
//      default parameter: out = cout;
//

ostream& HumNum::printFraction(ostream& out) const {
	if (this->isInteger()) {
		out << getNumerator();
	} else {
		out << getNumerator() << '/' << getDenominator();
	}
	return out;
}



//////////////////////////////
//
// HumNum::printMixedFraction -- Print as an integer plus fractional
//     remainder.  If absolute value is less than one, will only
//     print the fraction.  The second parameter is the output stream
//     for printing, and the third parameter is a separation string
//     between the integer and remainder fraction.
//        default parameter: out = cout;
//        default parameter: separator = "_"
//

ostream& HumNum::printMixedFraction(ostream& out,
		string separator) const {
	if (this->isInteger()) {
		out << getNumerator();
	} else if (top > bot) {
		int intval = this->getInteger();
		int remainder = top - intval * bot;
		out << intval << separator << remainder << '/' << bot;
	} else {
		printFraction(out);
	}
	return out;
}



//////////////////////////////
//
// HumNum::printTwoPart --
//     default value: spacer = "+"
//

ostream& HumNum::printTwoPart(ostream& out, const string& spacer) const {
   int tnum = top;
   int tden = bot;
   int sign = 1;
   if (tnum < 0) {
      tnum = -tnum;
      sign = -sign;
   }
   if (tden < 0) {
      tden = -tden;
      sign = -sign;
   }

   if (tnum < tden) {
      out << *this;
      return out;
   }

   int integ = tnum / tden;
   tnum = tnum - tden * integ;

   if (sign < 0) {
      out << '-';
   }
   if (integ > 0) {
      out << integ;
      if (tnum > 0) {
         out << spacer;
         HumNum newone(tnum, tden);
         out << newone;
      }
   } else {
      HumNum newone(tnum, tden);
      out << newone;
   }

   return out;
}



//////////////////////////////
//
// HumNum::printList -- Print as a list of two numbers, such as
//    "(1, 2)" for 1/2.
// default value: out = cout;
//

ostream& HumNum::printList(ostream& out) const {
	out << '(' << top << ", " << bot << ')';
	return out;
}



//////////////////////////////
//
// operator<< -- Default printing behavior for HumNums.
//

ostream& operator<<(ostream& out, const HumNum& number) {
	number.printFraction(out);
	return out;
}




//////////////////////////////
//
// HumParamSet::HumParamSet --
//

HumParamSet::HumParamSet(void) {
	// do nothing
}

HumParamSet::HumParamSet(const string& token) {
	readString(token);
}

HumParamSet::HumParamSet(HTp token) {
	readString(token);
}



//////////////////////////////
//
// HumParamSet::~HumParamSet --
//

HumParamSet::~HumParamSet() {
	clear();
}


//////////////////////////////
//
// HumParamSet::getNamespace1 --
//

const string& HumParamSet::getNamespace1(void) {
	return m_ns1;
}



//////////////////////////////
//
// HumParamSet::getNamespace2 --
//

const string& HumParamSet::getNamespace2(void) {
	return m_ns2;
}



//////////////////////////////
//
// HumParamSet::getNamespace --
//

string HumParamSet::getNamespace(void) {
	return m_ns1 + ":" + m_ns2;
}



//////////////////////////////
//
// HumParamSet::setNamespace1 --
//

void HumParamSet::setNamespace1(const string& name) {
	m_ns1 = name;
}



//////////////////////////////
//
// HumParamSet::setNamespace2 --
//

void HumParamSet::setNamespace2(const string& name) {
	m_ns2 = name;
}



//////////////////////////////
//
// HumParamSet::setNamespace --
//

void HumParamSet::setNamespace(const string& name) {
	auto loc = name.find(':');
	if (loc == string::npos) {
		m_ns1 = "";
		m_ns2 = name;
	} else {
		m_ns1 = name.substr(0, loc);
		m_ns2 = name.substr(loc+1, string::npos);
	}
}



//////////////////////////////
//
// HumParamSet::setNamespace --
//

void HumParamSet::setNamespace(const string& name1, const string& name2) {
	m_ns1 = name1;
	m_ns2 = name2;
}



//////////////////////////////
//
// HumParamSet::getCount --
//

int HumParamSet::getCount(void) {
	return (int)m_parameters.size();
}



//////////////////////////////
//
// HumParamSet::getParameterName --
//

const string& HumParamSet::getParameterName(int index) {
	return m_parameters.at(index).first;
}



//////////////////////////////
//
// HumParamSet::getParameterValue --
//

const string& HumParamSet::getParameterValue(int index) {
	return m_parameters.at(index).second;
}



//////////////////////////////
//
// HumParamSet::addParameter --
//

int HumParamSet::addParameter(const string& name, const string& value) {
	m_parameters.push_back(make_pair(name, value));
	return (int)m_parameters.size() - 1;
}



//////////////////////////////
//
// HumParamSet::setParameter --
//

int HumParamSet::setParameter(const string& name, const string& value) {
	for (int i=0; i<(int)m_parameters.size(); i++) {
		if (m_parameters[i].first == name) {
			m_parameters[i].second = value;
			return i;
		}
	}
	// Parameter does not exist so create at end of list.
	m_parameters.push_back(make_pair(name, value));
	return (int)m_parameters.size() - 1;
}



//////////////////////////////
//
// HumParamSet::clear --
//

void HumParamSet::clear(void) {
	m_ns1.clear();
	m_ns2.clear();
	m_parameters.clear();
}



//////////////////////////////
//
// HumParamSet::readString --
//


void HumParamSet::readString(HTp token) {
	m_token = token;
	readString(*token);
}


void HumParamSet::readString(const string& text) {
	vector<string> pieces(1);
	bool bangs = true;
	for (int i=0; i<(int)text.size(); i++) {
		if (bangs && text[i] == '!') {
			continue;
		}
		bangs = false;
		if (text[i] == ':') {
			pieces.resize(pieces.size() + 1);
			continue;
		}
		pieces.back() += text[i];
	}

	if (pieces.size() < 3) {
		// not enough information
		return;
	}

	m_ns1 = pieces[0];
	m_ns2 = pieces[1];

	string key;
	string value;
	int loc;
	for (int i=2; i<(int)pieces.size(); i++) {
		Convert::replaceOccurrences(pieces[i], "&colon;", ":");
		loc = (int)pieces[i].find("=");
		if (loc != (int)string::npos) {
			key   = pieces[i].substr(0, loc);
			value = pieces[i].substr(loc+1, pieces[i].size());
		} else {
			key   = pieces[i];
			value = "true";
		}
		addParameter(key, value);
	}
}



//////////////////////////////
//
// HumParamSet::printXml --
//

ostream& HumParamSet::printXml(ostream& out, int level,
		const string& indent) {

	if (getCount() == 0) {
		return out;
	}

	out << Convert::repeatString(indent, level++) << "<linked-parameter-set>\n";
	out << Convert::repeatString(indent, level++);
	out << "<namespace n=\"1\" name=\"" << getNamespace1() << "\">\n";
	out << Convert::repeatString(indent, level++);
	out << "<namespace n=\"2\" name=\"" << getNamespace2() << "\">\n";

	for (int i=0; i<getCount(); i++) {
		out << Convert::repeatString(indent, level);
		out << "<parameter key=\"" << getParameterName(i) << "\"";
		out << " value=\"";
		out << Convert::encodeXml(getParameterValue(i)) << "\"";
		out << "/>\n";
	}

	out << Convert::repeatString(indent, --level) << "</namespace>\n";
	out << Convert::repeatString(indent, --level) << "</namespace>\n";
	out << Convert::repeatString(indent, --level) << "<linked-parameter-set>\n";
	return out;
}



//////////////////////////////
//
// operator<< -- print HumParamSetData as a layout command
//

ostream& operator<<(ostream& out, HumParamSet* hps) {
	out << *hps;
	return out;
}


ostream& operator<<(ostream& out, HumParamSet& hps) {
	out << hps.getNamespace();
	int count = hps.getCount();
	for (int i=0; i<count; i++) {
		out << ":" << hps.getParameterName(i) << "=";
		// should colon-escape the following line's output:
		out << "=" << hps.getParameterValue(i);
	}
	return out;
}




const std::vector<char> HumPitch::m_diatonicPC2letterLC({ 'c', 'd', 'e', 'f', 'g', 'a', 'b' });
const std::vector<char> HumPitch::m_diatonicPC2letterUC({ 'C', 'D', 'E', 'F', 'G', 'A', 'B' });

////////////////////////////////////////////////////////////////////////////
//
// The HumPitch class is an interface for storing information about notes which
// will be used in the HumTransposer class.  The diatonic pitch class, chromatic alteration
// of the diatonic pitch and the octave are store in the class.  Names given to the
// parameters are analogous to MEI note attributes.  Note that note@accid can be also
// note/accid in MEI data, and other complications that need to be resolved into
// storing the correct pitch information in HumPitch.
//


//////////////////////////////
//
// HumPitch::HumPitch -- HumPitch constructor.
//

HumPitch::HumPitch(int aDiatonic, int anAccid, int anOct) {
	setPitch(aDiatonic, anAccid, anOct);
}


HumPitch::HumPitch(const HumPitch &pitch) {
	m_diatonicpc = pitch.m_diatonicpc;
	m_accid = pitch.m_accid;
	m_oct = pitch.m_oct;
}



//////////////////////////////
//
// operator= HumPitch -- copy operator for pitches.
//

HumPitch &HumPitch::operator=(const HumPitch &pitch) {
	if (this != &pitch) {
		m_diatonicpc = pitch.m_diatonicpc;
		m_accid = pitch.m_accid;
		m_oct = pitch.m_oct;
	}
	return *this;
}



//////////////////////////////
//
// HumPitch::isValid -- returns true if the absolute value of the accidental
//     is less than or equal to the max accidental value.

bool HumPitch::isValid(int maxAccid) {
	return abs(m_accid) <= abs(maxAccid);
}



//////////////////////////////
//
// HumPitch::setPitch -- Set the attributes for a pitch all at once.
//

void HumPitch::setPitch(int aDiatonic, int anAccid, int anOct) {
	m_diatonicpc = aDiatonic;
	m_accid = anAccid;
	m_oct = anOct;
}



//////////////////////////////
//
// HumPitch::isRest -- returns true if a rest, which means that m_diatonicpc is negative.
//

bool HumPitch::isRest(void) const {
	return m_diatonicpc < 0 ? true : false;
}



//////////////////////////////
//
// HumPitch::makeRest -- 
//

void HumPitch::makeRest(void) {
	m_diatonicpc = -1;
	m_accid = 0;
	m_oct = 0;
}



//////////////////////////////
//
// HumPitch::getOctave -- Middle C is the start of the 4th octave.
//

int HumPitch::getOctave(void) const {
	return m_oct;
}



//////////////////////////////
//
// HumPitch::getAccid -- +1=sharp, -1=flat, +2=double-sharp, etc.
//   Maybe expand to a double later for quarter tones, etc.
//

int HumPitch::getAccid(void) const {
	return m_accid;
}



//////////////////////////////
//
// HumPitch::getDiatonicPitchClass --  Return the diatonic pitch class:
//   0 = C, 1 = D, 2 = E, 3 = F, 4 = G, 5 = A, 6 = B, -1 = rest.
//

int HumPitch::getDiatonicPitchClass(void) const {
	return m_diatonicpc;
}


int HumPitch::getDiatonicPC(void) const {
	return getDiatonicPitchClass();
}



//////////////////////////////
//
// HumPitch::setOctave -- Set the octave number of the pitch, with 4 meaning
//   the middle-C octave.
//

void HumPitch::setOctave(int anOct) {
	m_oct = anOct;
}



//////////////////////////////
//
// HumPitch::setAccid -- +1 = sharp, -1 = flat, +2 = double-sharp.
//
void HumPitch::setAccid(int anAccid) {
	m_accid = anAccid;
}



//////////////////////////////
//
// HumPitch::makeSharp -- Set the accidental to +1.
//

void HumPitch::makeSharp(void) {
	m_accid = 1;
}



//////////////////////////////
//
// HumPitch::makeFlat -- Set the accidental to -1.
//

void HumPitch::makeFlat(void) {
	m_accid = -1;
}



//////////////////////////////
//
// HumPitch::makeNatural -- Set the accidental to -1.
//

void HumPitch::makeNatural(void) {
	m_accid = 0;
}



//////////////////////////////
//
// HumPitch::setDiatonicPitchClass --
//

void HumPitch::setDiatonicPitchClass(int aDiatonicPC) {
	if (aDiatonicPC < 0) {
		m_diatonicpc = -1;
	} else if (aDiatonicPC < 7) {
		m_diatonicpc = aDiatonicPC;
	} else if (aDiatonicPC >= 'A' && aDiatonicPC <= 'G') {
		m_diatonicpc = (aDiatonicPC - 'A' + 5) % 7;
	} else if (aDiatonicPC >= 'a' && aDiatonicPC <= 'g') {
		m_diatonicpc = (aDiatonicPC - 'a' + 5) % 7;
	} else {
		m_diatonicpc = -1;
	}
}



//////////////////////////////
//
// HumPitch::setDiatonicPC --
//

void HumPitch::setDiatonicPC(int aDiatonicPC) {
	setDiatonicPitchClass(aDiatonicPC);
}



//////////////////////////////
//
// operator<< HumPitch -- Print pitch data as string for debugging.
//

ostream &operator<<(ostream &out, const HumPitch &pitch) {
	switch (pitch.getDiatonicPC()) {
		case dpc_C: out << "C"; break;
		case dpc_D: out << "D"; break;
		case dpc_E: out << "E"; break;
		case dpc_F: out << "F"; break;
		case dpc_G: out << "G"; break;
		case dpc_A: out << "A"; break;
		case dpc_B: out << "B"; break;
		default: out << "R";
	}
	if (pitch.getAccid() > 0) {
		for (int i = 0; i < pitch.getAccid(); i++) {
			out << "#";
		}
	} else if (pitch.getAccid() < 0) {
		for (int i = 0; i < abs(pitch.getAccid()); i++) {
			out << "b";
		}
	}
	out << pitch.getOctave();
	return out;
}



//////////////////////////////
//
// HumPitch::getKernPitch -- Return the pitch as a **kern pitch name.
//

string HumPitch::getKernPitch(void) const {
	if (m_diatonicpc < 0) {
		return "r";
	}

	int count;
	char diatonic;
	if (m_oct < 4) {
		diatonic = m_diatonicPC2letterUC.at(m_diatonicpc);
		count = 4 - m_oct;
	} else {
		count = m_oct - 4 + 1;
		diatonic = m_diatonicPC2letterLC.at(m_diatonicpc);
	}
	string output;
	output = diatonic;
	for (int i=1; i<count; i++) {
		output += diatonic;
	}
	if (m_accid != 0) {
		if (m_accid < 0) {
			for (int i=0; i<-m_accid; i++) {
				output += '-';
			}
		} else {
			for (int i=0; i<m_accid; i++) {
				output += '#';
			}
		}
	}
	return output;
}



//////////////////////////////
//
// HumPitch::setKernPitch -- Set the pitch from a **kern pitch name.
//

bool HumPitch::setKernPitch(const string& kern) {
	makeRest();
	HumRegex hre;
	if (kern.find('r') != string::npos) {
		// rests can have pitch information, but ignore.
		return true;
	}
	if (!hre.search(kern, "(A+|B+|C+|D+|E+|F+|G+|a+|b+|c+|d+|e+|f+|g+)(-+|#+)?")) {
		return false;
	}
	string letters = hre.getMatch(1);
	string accidentals = hre.getMatch(2);

	if (!accidentals.empty()) {
		m_accid = (int)accidentals.size();
		if (accidentals[0] == '-') {
			m_accid = -m_accid;
		}
	}
	int lcount = (int)letters.size();
	m_oct = islower(letters[0]) ? 3 + lcount : 4 - lcount;
	m_diatonicpc = (tolower(letters[0]) - 'a' + 5) % 7;
	return true;
}



//////////////////////////////
//
// HumPitch::getScientificPitch -- Returns the **pitch representation of the pitch.
//    Examples: Eb4, F#3, C-2.
//    Format:    [A-G](b+|#+)?-?\d+
//

std::string HumPitch::getScientificPitch(void) const {
	if (m_diatonicpc < 0) {
		return "R";
	}
	string output;
	output = m_diatonicPC2letterUC.at(m_diatonicpc);
	if (m_accid < 0) {
		for (int i=0; i<-m_accid; i++) {
			output += 'b';
		}
	} else if (m_accid > 0) {
		for (int i=0; i<m_accid; i++) {
			output += '#';
		}
	}
	output = to_string(m_oct);
	return output;
}



//////////////////////////////
//
// HumPitch::setScientificPitch --
//

bool HumPitch::setScientificPitch(const std::string& pitch) {
	makeRest();

	HumRegex hre;
	if (!hre.search(pitch, "([A-Ga-g])(b+|#+)?(-?\\d+)")) {
		return false;
	}
	string diatonic = hre.getMatch(1);
	string accidental = hre.getMatch(2);
	m_oct = hre.getMatchInt(3);
	if (!accidental.empty()) {
		m_accid = (int)accidental.size();
		if (accidental[0] == 'f') {
			m_accid = -m_accid;
		}
	}
	m_diatonicpc = (toupper(diatonic[0]) - 'A' + 5) % 7;
	return true;
}





//////////////////////////////
//
// HumRegex::HumRegex -- Constructor.
//

HumRegex::HumRegex(void) {
	// by default use ECMAScript regular expression syntax:
	m_regexflags  = std::regex_constants::ECMAScript;

	m_searchflags = std::regex_constants::format_first_only;
}


HumRegex::HumRegex(const string& exp, const string& options) {
	// initialize a regular expression for the object
	m_regexflags = (std::regex_constants::syntax_option_type)0;
	m_regexflags = getTemporaryRegexFlags(options);
	if (m_regexflags == 0) {
		// explicitly set the default syntax
		m_regexflags = std::regex_constants::ECMAScript;
	}
	m_regex = regex(exp, getTemporaryRegexFlags(options));
	m_searchflags = (std::regex_constants::match_flag_type)0;
	m_searchflags = getTemporarySearchFlags(options);
}



//////////////////////////////
//
// HumRegex::HumRegex -- Destructor.
//

HumRegex::~HumRegex() {
	// do nothing
}


///////////////////////////////////////////////////////////////////////////
//
// option setting
//

//////////////////////////////
//
// HumRegex::setIgnoreCase --
//

void HumRegex::setIgnoreCase(void) {
	m_regexflags |= std::regex_constants::icase;
}



//////////////////////////////
//
// HumRegex::getIgnoreCase --
//

bool HumRegex::getIgnoreCase(void) {
	return (m_regexflags & std::regex_constants::icase) ? true : false;
}



//////////////////////////////
//
// HumRegex::unsetIgnoreCase --
//

void HumRegex::unsetIgnoreCase(void) {
	m_regexflags &= ~std::regex_constants::icase;
}



//////////////////////////////
//
// HumRegex::setGlobal --
//

void HumRegex::setGlobal(void) {
	m_searchflags &= ~std::regex_constants::format_first_only;
}



//////////////////////////////
//
// HumRegex::getGlobal --
//

bool HumRegex::getGlobal(void) {
	auto value = m_searchflags & std::regex_constants::format_first_only;
	// return value.none();
	return !value;
}



//////////////////////////////
//
// HumRegex::unsetGlobal --
//

void HumRegex::unsetGlobal(void) {
	m_searchflags |= std::regex_constants::format_first_only;
}


///////////////////////////////////////////////////////////////////////////
//
// Searching functions
//

//////////////////////////////
//
// HumRegex::search -- Search for the regular expression in the
//    input string.  Returns the character position + 1 of the first match if any found.
//    Search results can be accessed with .getSubmatchCount() and .getSubmatch(index).
//
//    Warning: a temporary string cannot be used as input to the search function
//    if you want to call getMatch() later.  If you do a memory leak will occur.
//    If you have a temporary string, first save it to a variable which remains
//    in scope while accesssing a match with getMatch().
//

int HumRegex::search(const string& input, const string& exp) {
	m_regex = regex(exp, m_regexflags);
	bool result = regex_search(input, m_matches, m_regex, m_searchflags);
	if (!result) {
		return 0;
	} else if (m_matches.size() < 1) {
		return 0;
	} else {
		// return the char+1 position of the first match
		return (int)m_matches.position(0) + 1;
	}
}


int HumRegex::search(const string& input, int startindex,
		const string& exp) {
	m_regex = regex(exp, m_regexflags);
	auto startit = input.begin() + startindex;
	auto endit   = input.end();
	bool result = regex_search(startit, endit, m_matches, m_regex, m_searchflags);
	if (!result) {
		return 0;
	} else if (m_matches.size() < 1) {
		return 0;
	} else {
		return (int)m_matches.position(0) + 1;
	}
}


int HumRegex::search(string* input, const string& exp) {
	return HumRegex::search(*input, exp);
}


int HumRegex::search(string* input, int startindex, const string& exp) {
	return HumRegex::search(*input, startindex, exp);
}

//
// This version of HumRegex allows for setting the options temporarily.
//

int HumRegex::search(const string& input, const string& exp,
		const string& options) {
	m_regex = regex(exp, getTemporaryRegexFlags(options));
	bool result = regex_search(input, m_matches, m_regex, getTemporarySearchFlags(options));
	if (!result) {
		return 0;
	} else if (m_matches.size() < 1) {
		return 0;
	} else {
		return (int)m_matches.position(0) + 1;
	}
}


int HumRegex::search(const string& input, int startindex, const string& exp,
		const string& options) {
	m_regex = regex(exp, getTemporaryRegexFlags(options));
	auto startit = input.begin() + startindex;
	auto endit   = input.end();
	bool result = regex_search(startit, endit, m_matches, m_regex, getTemporarySearchFlags(options));
	if (!result) {
		return 0;
	} else if (m_matches.size() < 1) {
		return 0;
	} else {
		return (int)m_matches.position(0) + 1;
	}
}


int HumRegex::search(string* input, const string& exp,
		const string& options) {
	return HumRegex::search(*input, exp, options);
}


int HumRegex::search(string* input, int startindex, const string& exp,
		const string& options) {
	return HumRegex::search(*input, startindex, exp, options);
}


///////////////////////////////////////////////////////////////////////////
//
// match-related functions
//

/////////////////////////////
//
// HumRegex::getMatchCount -- Return the number of submatches that a
//   previous call to HumRegex::search generated.
//

int HumRegex::getMatchCount(void) {
	return (int)m_matches.size();
}



//////////////////////////////
//
// HumRegex::getMatch -- Returns the given match.  The first match
//   at "0" is the complete match.  The matches with a larger index
//   are the submatches.
//

string HumRegex::getMatch(int index) {
	if (index < 0) {
		return "";
	} if (index >= (int)m_matches.size()) {
		return "";
	}
	string output = m_matches.str(index);
	return output;
}



//////////////////////////////
//
// HumRegex::getMatchInt -- Get the match interpreted as a integer.
//     returns 0 if match does not start with a valid number.
//

int HumRegex::getMatchInt(int index) {
	string value = m_matches.str(index);
	int output = 0;
	if (value.size() > 0) {
		if (isdigit(value[0])) {
			output = std::stoi(value);
		} else if (value[0] == '-') {
			output = std::stoi(value);
		} else if (value[0] == '+') {
			output = std::stoi(value);
		}
	}
	return output;
}



//////////////////////////////
//
// HumRegex::getMatchDouble -- Get the match interpreted as a double.
//

double HumRegex::getMatchDouble(int index) {
	string value = m_matches.str(index);
	if (value.size() > 0) {
		return stod(value);
	} else {
		return 0.0;
	}
}



//////////////////////////////
//
// HumRegex::getPrefix -- Return the input string text which
//    occurs before the match;
//

string HumRegex::getPrefix(void) {
	return m_matches.prefix().str();
}



//////////////////////////////
//
// HumRegex::getSuffix -- Return the input string text which
//    occurs after the match;
//

string HumRegex::getSuffix(void) {
	return m_matches.suffix().str();
}



//////////////////////////////
//
// HumRegex::getMatchStartIndex -- Get starting index of match in input
//     search string.
//

int HumRegex::getMatchStartIndex(int index) {
	return (int)m_matches.position(index);
}



//////////////////////////////
//
// HumRegex::getMatchEndIndex -- Get ending index of match in input
//     search string.  The index is one larger than the index of the
//     end of the matched position.
//

int HumRegex::getMatchEndIndex(int index) {
	return getMatchStartIndex(index) + getMatchLength(index);
}



//////////////////////////////
//
// HumRegex::getMatchLength -- Get starting character length of match.
//

int HumRegex::getMatchLength(int index) {
	return (int)m_matches.length(index);
}


///////////////////////////////////////////////////////////////////////////
//
// match functions (a "match" is a search that matches a regular
//    expression to the entire string").
//

//////////////////////////////
//
// HumRegex::match --
//

bool HumRegex::match(const string& input, const string& exp) {
	m_regex = regex(exp, m_regexflags);
	return regex_match(input, m_regex, m_searchflags);
}


bool HumRegex::match(const string& input, const string& exp,
		const string& options) {
	m_regex = regex(exp, getTemporaryRegexFlags(options));
	return regex_match(input, m_regex, getTemporarySearchFlags(options));
}


bool HumRegex::match(const string* input, const string& exp) {
	return HumRegex::match(*input, exp);

}


bool HumRegex::match(const string* input, const string& exp,
		const string& options) {
	return HumRegex::match(*input, exp, options);
}



///////////////////////////////////////////////////////////////////////////
//
// search and replace functions.  Default behavior is to only match
// the first match.  use the "g" option or .setGlobal() to do global
// replacing.
//

//////////////////////////////
//
// HumRegex::replaceDestructive -- Replace in input string.
//

string& HumRegex::replaceDestructive(string& input, const string& replacement,
		const string& exp) {
	m_regex = regex(exp, m_regexflags);
	input = regex_replace(input, m_regex, replacement, m_searchflags);
	return input;
}


string& HumRegex::replaceDestructive(string* input, const string& replacement,
		const string& exp) {
	return HumRegex::replaceDestructive(*input, replacement, exp);
}

//
// This version allows for temporary match flag options.
//

string& HumRegex::replaceDestructive(string& input, const string& replacement,
		const string& exp, const string& options) {
	m_regex = regex(exp, getTemporaryRegexFlags(options));
	input = regex_replace(input, m_regex, replacement, getTemporarySearchFlags(options));
	return input;
}


string& HumRegex::replaceDestructive (string* input, const string& replacement,
		const string& exp, const string& options) {
	return HumRegex::replaceDestructive(*input, replacement, exp, options);
}



//////////////////////////////
//
// HumRegex::replaceCopy --  Keep input string the same, return replacement
//    string as output
//

string HumRegex::replaceCopy(const string& input, const string& replacement,
		const string& exp) {
	m_regex = regex(exp, m_regexflags);
	string output;
	regex_replace(std::back_inserter(output), input.begin(),
			input.end(), m_regex, replacement);
	return output;
}


string HumRegex::replaceCopy(string* input, const string& replacement,
		const string& exp) {
	return HumRegex::replaceCopy(*input, replacement, exp);
}

//
// This version allows for temporary match flag options.
//

string HumRegex::replaceCopy(const string& input, const string& exp,
		const string& replacement, const string& options) {
	m_regex = regex(exp, getTemporaryRegexFlags(options));
	string output;
	regex_replace(std::back_inserter(output), input.begin(),
			input.end(), m_regex, replacement, getTemporarySearchFlags(options));
	return output;
}


string HumRegex::replaceCopy(string* input, const string& exp,
		const string& replacement, const string& options) {
	return HumRegex::replaceCopy(*input, replacement, exp, options);
}



//////////////////////////////
//
// HumRegex::tr --
//

string& HumRegex::tr(string& input, const string& from, const string& to) {
	vector<char> trans;
	trans.resize(256);
	for (int i=0; i<(int)trans.size(); i++) {
		trans[i] = (char)i;
	}
	int minmax = (int)from.size();
	if (to.size() < from.size()) {
		minmax = (int)to.size();
	}

	for (int i=0; i<minmax; i++) {
		trans[from[i]] = to[i];
	}

	for (int i=0; i<(int)input.size(); i++) {
		input[i] = trans[input[i]];
	}

	return input;
}



//////////////////////////////
//
// HumRegex::split --
//

bool HumRegex::split(vector<string>& entries, const string& buffer,
		const string& separator) {
	entries.clear();
	string newsep = "(";
	newsep += separator;
	newsep += ")";
	int status = search(buffer, newsep);
	if (!status) {
		if (buffer.size() == 0) {
			return false;
		} else {
			entries.push_back(buffer);
			return true;
		}
	}
	int start = 0;
	while (status) {
		entries.push_back(getPrefix());
		start += getMatchEndIndex(1);
		status = search(buffer, start, newsep);
	}
	// add last token:
	entries.push_back(buffer.substr(start));
	return true;
}



//////////////////////////////
//
// HumRegex::getTemporaryRegexFlags --
//

std::regex_constants::syntax_option_type HumRegex::getTemporaryRegexFlags(
		const string& sflags) {
	if (sflags.empty()) {
		return m_regexflags;
	}
	std::regex_constants::syntax_option_type temp_flags = m_regexflags;
	for (auto it : sflags) {
		switch (it) {
			case 'i':
				temp_flags = (std::regex_constants::syntax_option_type)
						(temp_flags | std::regex_constants::icase);
				break;
			case 'I':
				temp_flags = (std::regex_constants::syntax_option_type)
						(temp_flags & ~std::regex_constants::icase);
				break;
		}
	}
	return temp_flags;
}



//////////////////////////////
//
// HumRegex::getTemporarySearchFlags --
//

std::regex_constants::match_flag_type HumRegex::getTemporarySearchFlags(
		const string& sflags) {
	if (sflags.empty()) {
		return m_searchflags;
	}
	std::regex_constants::match_flag_type temp_flags = m_searchflags;
	for (auto it : sflags) {
		switch (it) {
			case 'g':
				temp_flags = (std::regex_constants::match_flag_type)
						(temp_flags & ~std::regex_constants::format_first_only);
				break;
			case 'G':
				temp_flags = (std::regex_constants::match_flag_type)
						(temp_flags | std::regex_constants::format_first_only);
				break;
		}
	}
	return temp_flags;
}



//////////////////////////////
//
// HumSignifier::HumSignifier --
//

HumSignifier::HumSignifier (void) {
	// do nothing
}



HumSignifier::HumSignifier (const string& rdfline) {
	parseSignifier(rdfline);
}



//////////////////////////////
//
// HumSignifier::~HumSignifier --
//

HumSignifier::~HumSignifier() {
	clear();
}



//////////////////////////////
//
// HumSignifier::clear -- Clear contents of object.
//

void HumSignifier::clear(void) {
	m_exinterp.clear();
	m_signifier.clear();
	m_definition.clear();
	m_parameters.clear();
	m_sigtype = signifier_type::signifier_unknown;
}



//////////////////////////////
//
// HumSignifier::parseSignifier --
//
bool HumSignifier::parseSignifier(const string& rdfline) {
	clear();
	HumRegex hre;
	if (!hre.search(rdfline, "!!!RDF(\\*\\*[^\\s:]+)\\s*:\\s*(.*)\\s*$")) {
		return false;
	}
	m_exinterp   = hre.getMatch(1);
	string value = hre.getMatch(2);

	if (!hre.search(value, "\\s*([^\\s=]+)\\s*=\\s*(.*)\\s*$")) {
		clear();
		return false;
	}
	m_signifier  = hre.getMatch(1);
	m_definition = hre.getMatch(2);

	// identify signifier category

	if (m_exinterp == "**kern") {
		if (m_definition.find("link") != std::string::npos) {
			m_sigtype = signifier_type::signifier_link;
		} else if (m_definition.find("above") != std::string::npos) {
			m_sigtype = signifier_type::signifier_above;
		} else if (m_definition.find("below") != std::string::npos) {
			m_sigtype = signifier_type::signifier_below;
		}
	}

	// parse parameters here

	return true;
}



//////////////////////////////
//
// HumSignifier::getSignifier --
//

std::string HumSignifier::getSignifier(void) {
	return m_signifier;
}



//////////////////////////////
//
// HumSignifier::getDefinition --
//

std::string HumSignifier::getDefinition(void) {
	return m_definition;
}



//////////////////////////////
//
// HumSignifier::getParameter --
//

std::string HumSignifier::getParameter(const std::string& key) {
	auto value = m_parameters.find(key);
	if (value == m_parameters.end()) {
		return "";
	} else {
		return value->second;
	}
}



//////////////////////////////
//
// HumSignifier::isKernLink -- Is a linking signifier
//

bool HumSignifier::isKernLink(void) {
	return (m_sigtype == signifier_type::signifier_link);
}



//////////////////////////////
//
// HumSignifier::isKernAbove -- Is an above signifier.
//

bool HumSignifier::isKernAbove(void) {
	return (m_sigtype == signifier_type::signifier_above);
}



//////////////////////////////
//
// HumSignifier::isKernBelow -- Is a below signifier.
//

bool HumSignifier::isKernBelow(void) {
	return (m_sigtype == signifier_type::signifier_below);
}




//////////////////////////////
//
// HumSignifiers::HumSignifier --
//

HumSignifiers::HumSignifiers(void) {
	// do nothing
}



//////////////////////////////
//
// HumSignifiers::~HumSignifier --
//

HumSignifiers::~HumSignifiers() {
	clear();
}



//////////////////////////////
//
// HumSignifiers::clear --
//

void HumSignifiers::clear(void) {
	m_kernLinkIndex = -1;

	for (int i=0; i<(int)m_signifiers.size(); i++) {
		delete m_signifiers[i];
		m_signifiers[i] = NULL;
	}
	m_signifiers.clear();
}



//////////////////////////////
//
// HumSignifiers::addSignifier --
//

bool HumSignifiers::addSignifier(const std::string& rdfline) {
	HumSignifier *humsig = new HumSignifier;
	if (!humsig->parseSignifier(rdfline)) {
		// ignore malformed RDF reference record.
		return false;
	}
	m_signifiers.push_back(humsig);

	if (m_signifiers.back()->isKernLink()) {
		m_kernLinkIndex = (int)m_signifiers.size() - 1;
	} else if (m_signifiers.back()->isKernAbove()) {
		m_kernAboveIndex = (int)m_signifiers.size() - 1;
	} else if (m_signifiers.back()->isKernBelow()) {
		m_kernBelowIndex = (int)m_signifiers.size() - 1;
	}
	return true;
}



//////////////////////////////
//
// HumSignifiers::hasKernLinkSignifier --
//

bool HumSignifiers::hasKernLinkSignifier(void) {
	return (m_kernLinkIndex >= 0);
}



//////////////////////////////
//
// HumSignifiers::getKernLinkSignifier --
//

std::string HumSignifiers::getKernLinkSignifier(void) {
	if (m_kernLinkIndex < 0) {
		return "";
	}
	return m_signifiers[m_kernLinkIndex]->getSignifier();
}



//////////////////////////////
//
// HumSignifiers::hasKernAboveSignifier --
//

bool HumSignifiers::hasKernAboveSignifier(void) {
	return (m_kernAboveIndex >= 0);
}



//////////////////////////////
//
// HumSignifiers::getKernAboveSignifier --
//

std::string HumSignifiers::getKernAboveSignifier(void) {
	if (m_kernAboveIndex < 0) {
		return "";
	}
	return m_signifiers[m_kernAboveIndex]->getSignifier();
}



//////////////////////////////
//
// HumSignifiers::hasKernBelowSignifier --
//

bool HumSignifiers::hasKernBelowSignifier(void) {
	return (m_kernBelowIndex >= 0);
}



//////////////////////////////
//
// HumSignifiers::getKernBelowSignifier --
//

std::string HumSignifiers::getKernBelowSignifier(void) {
	if (m_kernBelowIndex < 0) {
		return "";
	}
	return m_signifiers[m_kernBelowIndex]->getSignifier();
}



//////////////////////////////
//
// HumSignifiers::getSignifierCount --
//

int HumSignifiers::getSignifierCount(void) {
	return (int)m_signifiers.size();
}



//////////////////////////////
//
// HumSignifiers::getSignifier --
//

HumSignifier* HumSignifiers::getSignifier(int index) {
	if (index < 0) {
		return NULL;
	}
	if (index >= (int)m_signifiers.size()) {
		return NULL;
	}
	return m_signifiers.at(index);
}




//////////////////////////////
//
// HumTool::HumTool --
//

HumTool::HumTool(void) {
	// do nothing
}



//////////////////////////////
//
// HumTool::~HumTool --
//

HumTool::~HumTool() {
	// do nothing
}



//////////////////////////////
//
// HumTool::hasAnyText -- Returns true if the output contains
//    text content in Humdrum syntax.
//

bool HumTool::hasAnyText(void) {
	if (m_suppress) {
		return true;
	}
	return ((!m_humdrum_text.str().empty())
			|| (!m_free_text.str().empty())
			|| (!m_json_text.str().empty()));
}



//////////////////////////////
//
// HumTool::getAllText -- Get the text content from any output
//     streams except warnings and errors.
//

string HumTool::getAllText(void) {
	return  m_humdrum_text.str()
	      + m_json_text.str()
	      + m_free_text.str();
}

//
// ostream version:
//

ostream& HumTool::getAllText(ostream& out) {
	out << m_humdrum_text.str();
	out << m_json_text.str();
	out << m_free_text.str();
	return out;
}



//////////////////////////////
//
// HumTool::suppressHumdrumFileOutput --
//

void HumTool::suppressHumdrumFileOutput(void) {
	m_suppress = true;
}



//////////////////////////////
//
// HumTool::hasHumdrumText -- Returns true if the output contains
//    text content in Humdrum syntax.
//

bool HumTool::hasHumdrumText(void) {
	return m_humdrum_text.str().empty() ? false : true;
}



//////////////////////////////
//
// HumTool::getHumdrumText -- Get the text content which represents
//     Humdrum syntax.
//

string HumTool::getHumdrumText(void) {
	return m_humdrum_text.str();
}

//
// ostream version:
//

ostream& HumTool::getHumdrumText(ostream& out) {
	out << m_humdrum_text.str();
	return out;
}



//////////////////////////////
//
// HumTool::hasFreeText --
//

bool HumTool::hasFreeText(void) {
	return m_free_text.str().empty() ? false : true;
}



//////////////////////////////
//
// HumTool::getFreeText -- Return any free-form text output from the
//     tool.
//

string HumTool::getFreeText(void) {
	return m_free_text.str();
}

//
// ostream version:
//

ostream& HumTool::getFreeText(ostream& out) {
	out << m_free_text.str();
	return out;
}



//////////////////////////////
//
// HumTool::hasJsonText --
//

bool HumTool::hasJsonText(void) {
	return m_json_text.str().empty() ? false : true;
}



//////////////////////////////
//
// HumTool::getFreeText -- Return any JSON text output from the
//     tool.
//

string HumTool::getJsonText(void) {
	return m_json_text.str();
}

//
// ostream version:
//

ostream& HumTool::getJsonText(ostream& out) {
	out << m_json_text.str();
	return out;
}



//////////////////////////////
//
// HumTool::hasWarning --
//

bool HumTool::hasWarning(void) {
	return m_warning_text.str().empty() ? false : true;
}



//////////////////////////////
//
// HumTool::getWarning -- Return any warning messages generated when
//     running the tool.
//

string HumTool::getWarning(void) {
	return m_warning_text.str();
}

//
// ostream version:
//

ostream& HumTool::getWarning(ostream& out) {
	out << m_warning_text.str();
	return out;
}



//////////////////////////////
//
// HumTool::hasError -- Return true if there is an error in processing
//    the options or input file(s).
//

bool HumTool::hasError(void) {
	if (hasParseError()) {
		return true;
	}
	return m_error_text.str().empty() ? false : true;
}



//////////////////////////////
//
// HumTool::getError -- Return any error messages generated when
//     running the tool.   This includes option parsing errors as
//     well.
//

string HumTool::getError(void) {
	string output = getParseError();
	output += m_error_text.str();
	return output;
}

//
// ostream version:
//

ostream& HumTool::getError(ostream& out) {
	out << getParseError();
	out << m_error_text.str();
	return out;
}


//////////////////////////////
//
// HumTool::clearOutput -- clear write buffers to get ready to
//     process another file.
//

void HumTool::clearOutput(void) {
	m_humdrum_text.str("");
	m_json_text.str("");
	m_free_text.str("");
  	m_warning_text.str("");
  	m_error_text.str("");
}



///////////////////////////////
//
// HumTool::setError --
//

void HumTool::setError(const string& message) {
	m_error_text << message << endl;
}






const std::vector<int> HumTransposer::m_diatonic2semitone({ 0, 2, 4, 5, 7, 9, 11 });


//////////////////////////////
//
// HumTransposer::HumTransposer -- HumTransposer constructor.
//

HumTransposer::HumTransposer() {
	// Initialize with base-600 system by default:
	setMaxAccid(42);
}



//////////////////////////////
//
// HumTransposer::~HumTransposer -- HumTransposer deconstructor.
//

HumTransposer::~HumTransposer() {
	// do nothing;
}



//////////////////////////////
//
// HumTransposer::setTransposition -- Set the transposition value which is an
//   interval class in the current base system.  When HumTransposer::setMaxAccid()
//   or HumTransposer::setBase*() are called, the transposition value will be set
//   to 0 (a perfect unison).  The integer is a base-40 class of number.  If you
//   want to transpose by semitone, do not use this option but rather the
//   setHumTransposer(int keyFifths, string semitones) function or the
//   setHumTransposer(int keyFifths, int semitones) function that are defined
//   further below.
//

bool HumTransposer::setTransposition(int transVal) {
	m_transpose = transVal;
	return true;
}

// Use a string to set the interval class in the current base system.  For example,
//  "+M2" means up a major second, which is the integer 6 in base-40.

bool HumTransposer::setTransposition(const string &transString) {
	m_transpose = getInterval(transString);
	return m_transpose != INVALID_INTERVAL_CLASS;
}

// Set transposition interval based on two pitches that represent the source data
// key tonic and the target key tonic.

bool HumTransposer::setTransposition(const HumPitch &fromPitch, const string &toString) {
	HumPitch toPitch;
	if (getKeyTonic(toString, toPitch)) {
		// Determine proper octave offset.
		int numSigns = toPitch.getOctave();
		m_transpose = getInterval(fromPitch, toPitch);
		// A transposition with n plus or minus signs should never be more than n octaves away.
		if (numSigns > 0 && m_transpose > perfectOctaveClass() * numSigns) {
			m_transpose -= perfectOctaveClass();
		}
		else if (numSigns < 0 && m_transpose < perfectOctaveClass() * numSigns) {
			m_transpose += perfectOctaveClass();
		}
		// A transposition with 0 plus or minus signs should never be more than 1/2 an octave away.
		else if (numSigns == 0 && m_transpose > perfectOctaveClass() / 2) {
			m_transpose -= perfectOctaveClass();
		}
		else if (numSigns == 0 && m_transpose < -1 * perfectOctaveClass() / 2) {
			m_transpose += perfectOctaveClass();
		}
		return true;
	}
	return false;
}

// Set the transposition based on the key signature (or inferred key signature coming from
// the keySig@diatonic/keySig@accid/keySig@mode information) and a string containing the
// semitone transposition.

bool HumTransposer::setTransposition(int keyFifths, const string &semitones) {
	if (!isValidSemitones(semitones)) {
		return false;
	}
	int semis = stoi(semitones);
	return setTransposition(keyFifths, semis);
}

// Note the order of the variables (key signature information is first in all
// cases where there are two input parametrs to setTransposition().

bool HumTransposer::setTransposition(int keyFifths, int semitones) {
	int intervalClass = semitonesToIntervalClass(keyFifths, semitones);
	return setTransposition(intervalClass);
}



//////////////////////////////
//
// HumTransposer::setTranspositionDC --
//

bool HumTransposer::setTranspositionDC(int diatonic, int chromatic) {
	int interval = HumTransposer::diatonicChromaticToIntervalClass(diatonic, chromatic);
	return setTransposition(interval);
}



//////////////////////////////
//
// HumTransposer::semitonesToIntervalClass -- convert semitones plus key
//     signature information into an integer interval class.
//

int HumTransposer::semitonesToIntervalClass(int keyFifths, int semitones) {
	int sign = semitones < 0 ? -1 : +1;
	semitones = semitones < 0 ? -semitones : semitones;
	int octave = semitones / 12;
	semitones = semitones - octave * 12;
	int sum1, sum2;
	string interval = "P1";
	switch (semitones) {
		case 0: interval = "P1"; break;

		case 1:
			sum1 = keyFifths - 5 * sign;
			sum2 = keyFifths + 7 * sign;
			interval = abs(sum1) < abs(sum2) ? "m2" : "A1";
			break;

		case 2:
			sum1 = keyFifths + 2 * sign;
			sum2 = keyFifths - 10 * sign;
			interval = abs(sum1) < abs(sum2) ? "M2" : "d3";
			break;

		case 3:
			sum1 = keyFifths - 3 * sign;
			sum2 = keyFifths + 9 * sign;
			interval = abs(sum1) < abs(sum2) ? "m3" : "A2";
			break;

		case 4:
			sum1 = keyFifths + 4 * sign;
			sum2 = keyFifths - 8 * sign;
			interval = abs(sum1) < abs(sum2) ? "M3" : "d4";
			break;

		case 5:
			sum1 = keyFifths - 1 * sign;
			sum2 = keyFifths + 11 * sign;
			interval = abs(sum1) < abs(sum2) ? "P4" : "A3";
			break;

		case 6:
			sum1 = keyFifths + 6 * sign;
			sum2 = keyFifths - 6 * sign;
			interval = abs(sum1) < abs(sum2) ? "A4" : "d5";
			break;

		case 7:
			sum1 = keyFifths + 1 * sign;
			sum2 = keyFifths - 11 * sign;
			interval = abs(sum1) < abs(sum2) ? "P5" : "d6";
			break;

		case 8:
			sum1 = keyFifths - 4 * sign;
			sum2 = keyFifths + 8 * sign;
			interval = abs(sum1) < abs(sum2) ? "m6" : "A5";
			break;

		case 9:
			sum1 = keyFifths + 3 * sign;
			sum2 = keyFifths - 9 * sign;
			interval = abs(sum1) < abs(sum2) ? "M6" : "d7";
			break;

		case 10:
			sum1 = keyFifths - 2 * sign;
			sum2 = keyFifths + 10 * sign;
			interval = abs(sum1) < abs(sum2) ? "m7" : "A6";
			break;

		case 11:
			sum1 = keyFifths + 5 * sign;
			sum2 = keyFifths - 7 * sign;
			interval = abs(sum1) < abs(sum2) ? "M7" : "d8";
			break;
	}

	interval = sign < 0 ? "-" + interval : "+" + interval;
	int intint = getInterval(interval);
	intint += sign * octave * m_base;
	return intint;
}



//////////////////////////////
//
// HumTransposer::semitonesToIntervalName -- convert semitones plus key
//     signature information into an interval name string.
//

string HumTransposer::semitonesToIntervalName(int keyFifths, int semitones) {
	int intervalClass = semitonesToIntervalClass(keyFifths, semitones);
	return getIntervalName(intervalClass);
}



//////////////////////////////
//
// HumTransposer::intervalToSemitones --  Convert a base interval class into
//   semitones.  Multiple enharmonic equivalent interval classes will collapse into
//   a single semitone value, so the process is not completely reversable
//   by calling HumTransposer::semitonesToIntervalClass(), but for simple
//   intervals it will be reversable.
//

int HumTransposer::intervalToSemitones(int interval) {
	int sign = interval < 0 ? -1 : +1;
	interval = interval < 0 ? -interval : interval;
	int octave = interval / m_base;
	int intervalClass = interval - octave * m_base;
	int diatonic = 0;
	int chromatic = 0;
	intervalToDiatonicChromatic(diatonic, chromatic, intervalClass);
	if ((diatonic != INVALID_INTERVAL_CLASS) && (chromatic != INVALID_INTERVAL_CLASS)) {
		return (m_diatonic2semitone.at(diatonic) + chromatic) * sign + 12 * octave;
	}
	else {
		return INVALID_INTERVAL_CLASS;
	}
}

//  Conversion from an interval name string into semitones:

int HumTransposer::intervalToSemitones(const string &intervalName) {
	int interval = getInterval(intervalName);
	return intervalToSemitones(interval);
}



//////////////////////////////
//
// HumTransposer::getTranspositionIntervalClass -- return the interval class integer
//   that was set for use with HumTransposer::HumTransposer.
//

int HumTransposer::getTranspositionIntervalClass() {
	return m_transpose;
}



//////////////////////////////
//
// HumTransposer::getTranspositionIntervalClass -- return the interval integer
//   as a string name that was set for use with HumTransposer::HumTransposer.
//
string HumTransposer::getTranspositionIntervalName() {
	return getIntervalName(m_transpose);
}



//////////////////////////////
//
// HumTransposer::transpose -- Do a transposition at the stored transposition interval, or
//   with a temporary provided integer interval class, or a temporary interval name.
//

void HumTransposer::transpose(HumPitch &pitch) {
	int ipitch = humHumPitchToIntegerPitch(pitch);
	ipitch += m_transpose;
	pitch = integerPitchToHumPitch(ipitch);
}

int HumTransposer::transpose(int ipitch) {
	return ipitch + m_transpose;
}

// Use a temporary transposition value in the following
// two functions. To save for later use of HumTransposer::HumTransposer
// without specifying the transposition interval, store
// transposition value with HumTransposer::setTransposition() first.

void HumTransposer::transpose(HumPitch &pitch, int transVal) {
	int ipitch = humHumPitchToIntegerPitch(pitch);
	ipitch += transVal;
	pitch = integerPitchToHumPitch(ipitch);
}

void HumTransposer::transpose(HumPitch &pitch, const string &transString) {
	int transVal = getInterval(transString);
	int ipitch = humHumPitchToIntegerPitch(pitch);
	ipitch += transVal;
	pitch = integerPitchToHumPitch(ipitch);
}



//////////////////////////////
//
// HumTransposer::getBase -- Return the integer interval class representing an octave.
//

int HumTransposer::getBase() {
	return m_base;
}



//////////////////////////////
//
// HumTransposer::getMaxAccid -- Return the maximum possible absolute accidental value
//     that can be represented by the current transposition base.
//

int HumTransposer::getMaxAccid() {
	return m_maxAccid;
}



//////////////////////////////
//
// HumTransposer::setMaxAccid -- Calculate variables related to a specific base system.
//

void HumTransposer::setMaxAccid(int maxAccid) {
	m_maxAccid = abs(maxAccid);
	m_base = 7 * (2 * m_maxAccid + 1) + 5;
	calculateDiatonicMapping();
	m_transpose = 0;
}



//////////////////////////////
//
// HumTransposer::calculateDiatonicMaping -- Calculate the integer values for the
//    natural diatonic pitch classes: C, D, E, F, G, A, and B in the current
//    base system.
//

void HumTransposer::calculateDiatonicMapping() {
	int m2 = m_maxAccid * 2 + 1;
	int M2 = m2 + 1;
	m_diatonicMapping.resize(7);
	m_diatonicMapping[dpc_C] = m_maxAccid;
	m_diatonicMapping[dpc_D] = m_diatonicMapping[dpc_C] + M2;
	m_diatonicMapping[dpc_E] = m_diatonicMapping[dpc_D] + M2;
	m_diatonicMapping[dpc_F] = m_diatonicMapping[dpc_E] + m2;
	m_diatonicMapping[dpc_G] = m_diatonicMapping[dpc_F] + M2;
	m_diatonicMapping[dpc_A] = m_diatonicMapping[dpc_G] + M2;
	m_diatonicMapping[dpc_B] = m_diatonicMapping[dpc_A] + M2;
}



//////////////////////////////
//
// HumTransposer::getKeyTonic -- Convert a key tonic string into a HumPitch
//      where the octave is the direction it should go.
//      Should conform to the following regular expression:
//          ([+]*|[-]*)([A-Ga-g])([Ss#]*|[Ffb]*)

bool HumTransposer::getKeyTonic(const string &keyTonic, HumPitch &tonic) {
	int octave = 0;
	int pitch = 0;
	int accid = 0;
	int state = 0;
	for (unsigned int i = 0; i < (unsigned int)keyTonic.size(); i++) {
		switch (state) {
			case 0:
				switch (keyTonic[i]) {
					case '-': octave--; break;
					case '+': octave++; break;
					default:
						state++;
						i--;
						break;
				}
				break;
			case 1:
				state++;
				switch (keyTonic[i]) {
					case 'C':
					case 'c': pitch = 0; break;
					case 'D':
					case 'd': pitch = 1; break;
					case 'E':
					case 'e': pitch = 2; break;
					case 'F':
					case 'f': pitch = 3; break;
					case 'G':
					case 'g': pitch = 4; break;
					case 'A':
					case 'a': pitch = 5; break;
					case 'B':
					case 'b': pitch = 6; break;
					default:
						cerr << "Invalid keytonic pitch character: " << keyTonic[i] << endl;
						return false;
				}
				break;
			case 2:
				switch (keyTonic[i]) {
					case 'F':
					case 'f':
					case 'b': accid--; break;
					case 'S':
					case 's':
					case '#': accid++; break;
					default:
						cerr << "Invalid keytonic accid character: " << keyTonic[i] << endl;
						return false;
				}
				break;
		}
	}

	tonic = HumPitch(pitch, accid, octave);
	return true;
}



//////////////////////////////
//
// HumTransposer::getInterval -- Convert a diatonic interval with chromatic
//     quality and direction into an integer interval class.   Input string
//     is in the format: direction + quality + diatonic interval.
//     Such as +M2 for up a major second, -P5 is down a perfect fifth.
//     Regular expression that the string should conform to:
//            (-|\+?)([Pp]|M|m|[aA]+|[dD]+)(\d+)
//

int HumTransposer::getInterval(const string &intervalName) {
	string direction;
	string quality;
	string number;
	int state = 0;

	for (int i = 0; i < (int)intervalName.size(); i++) {
		switch (state) {
			case 0: // direction or quality expected
				switch (intervalName[i]) {
					case '-': // interval is down
						direction = "-";
						state++;
						break;
					case '+': // interval is up
						direction += "";
						state++;
						break;
					default: // interval is up by default
						direction += "";
						state++;
						i--;
						break;
				}
				break;

			case 1: // quality expected
				if (isdigit(intervalName[i])) {
					state++;
					i--;
				}
				else {
					switch (intervalName[i]) {
						case 'M': // major
							quality = "M";
							break;
						case 'm': // minor
							quality = "m";
							break;
						case 'P': // perfect
						case 'p': quality = "P"; break;
						case 'D': // diminished
						case 'd': quality += "d"; break;
						case 'A': // augmented
						case 'a': quality += "A"; break;
					}
				}
				break;

			case 2: // digit expected
				if (isdigit(intervalName[i])) {
					number += intervalName[i];
				}
				break;
		}
	}

	if (quality.empty()) {
		cerr << "Interval name requires a chromatic quality: " << intervalName << endl;
		return INVALID_INTERVAL_CLASS;
	}

	if (number.empty()) {
		cerr << "Interval name requires a diatonic interval number: " << intervalName << endl;
		return INVALID_INTERVAL_CLASS;
	}

	int dnum = stoi(number);
	if (dnum == 0) {
		cerr << "Integer interval number cannot be zero: " << intervalName << endl;
		return INVALID_INTERVAL_CLASS;
	}
	dnum--;
	int octave = dnum / 7;
	dnum = dnum - octave * 7;

	int base = 0;
	int adjust = 0;

	switch (dnum) {
		case 0: // unison
			base = perfectUnisonClass();
			if (quality[0] == 'A') {
				adjust = (int)quality.size();
			}
			else if (quality[0] == 'd') {
				adjust = -(int)quality.size();
			}
			else if (quality != "P") {
				cerr << "Error in interval quality: " << intervalName << endl;
				return INVALID_INTERVAL_CLASS;
			}
			break;
		case 1: // second
			if (quality == "M") {
				base = majorSecondClass();
			}
			else if (quality == "m") {
				base = minorSecondClass();
			}
			else if (quality[0] == 'A') {
				base = majorSecondClass();
				adjust = (int)quality.size();
			}
			else if (quality[0] == 'd') {
				base = minorSecondClass();
				adjust = -(int)quality.size();
			}
			else {
				cerr << "Error in interval quality: " << intervalName << endl;
				return INVALID_INTERVAL_CLASS;
			}
			break;
		case 2: // third
			if (quality == "M") {
				base = majorThirdClass();
			}
			else if (quality == "m") {
				base = minorThirdClass();
			}
			else if (quality[0] == 'A') {
				base = majorThirdClass();
				adjust = (int)quality.size();
			}
			else if (quality[0] == 'd') {
				base = minorThirdClass();
				adjust = -(int)quality.size();
			}
			else {
				cerr << "Error in interval quality: " << intervalName << endl;
				return INVALID_INTERVAL_CLASS;
			}
			break;
		case 3: // fourth
			base = perfectFourthClass();
			if (quality[0] == 'A') {
				adjust = (int)quality.size();
			}
			else if (quality[0] == 'd') {
				adjust = -(int)quality.size();
			}
			else if (quality != "P") {
				cerr << "Error in interval quality: " << intervalName << endl;
				return INVALID_INTERVAL_CLASS;
			}
			break;
		case 4: // fifth
			base = perfectFifthClass();
			if (quality[0] == 'A') {
				adjust = (int)quality.size();
			}
			else if (quality[0] == 'd') {
				adjust = -(int)quality.size();
			}
			else if (quality != "P") {
				cerr << "Error in interval quality: " << intervalName << endl;
				return INVALID_INTERVAL_CLASS;
			}
			break;
		case 5: // sixth
			if (quality == "M") {
				base = majorSixthClass();
			}
			else if (quality == "m") {
				base = minorSixthClass();
			}
			else if (quality[0] == 'A') {
				base = majorSixthClass();
				adjust = (int)quality.size();
			}
			else if (quality[0] == 'd') {
				base = minorSixthClass();
				adjust = -(int)quality.size();
			}
			else {
				cerr << "Error in interval quality: " << intervalName << endl;
				return INVALID_INTERVAL_CLASS;
			}
			break;
		case 6: // seventh
			if (quality == "M") {
				base = majorSeventhClass();
			}
			else if (quality == "m") {
				base = minorSeventhClass();
			}
			else if (quality[0] == 'A') {
				base = majorSeventhClass();
				adjust = (int)quality.size();
			}
			else if (quality[0] == 'd') {
				base = minorSeventhClass();
				adjust = -(int)quality.size();
			}
			else {
				cerr << "Error in interval quality: " << intervalName << endl;
				return INVALID_INTERVAL_CLASS;
			}
			break;
	}

	if (direction == "-") {
		return -((octave * m_base) + base + adjust);
	}
	else {
		return (octave * m_base) + base + adjust;
	}
}



//////////////////////////////
//
// HumTransposer::perfectUnisonClass -- Return the integer interval class
//     representing a perfect unison.
//

int HumTransposer::perfectUnisonClass() {
	return 0;
}



//////////////////////////////
//
// HumTransposer::minorSecondClass -- Return the integer interval class
//     representing a minor second.
//

int HumTransposer::minorSecondClass() {
	return m_diatonicMapping[3] - m_diatonicMapping[2]; // F - E
}



//////////////////////////////
//
// HumTransposer::majorSecondClass -- Return the integer interval class
//    representing a major second.
//

int HumTransposer::majorSecondClass() {
	return m_diatonicMapping[1] - m_diatonicMapping[0]; // D - C
}



//////////////////////////////
//
// HumTransposer::minorThirdClass -- Return the integer interval class
//    representing a minor third.
//

int HumTransposer::minorThirdClass() {
	return m_diatonicMapping[3] - m_diatonicMapping[1]; // F - D
}



//////////////////////////////
//
// HumTransposer::majorThirdClass -- Return the integer interval class
//    representing a major third.
//

int HumTransposer::majorThirdClass() {
	return m_diatonicMapping[2] - m_diatonicMapping[0]; // E - C
}



//////////////////////////////
//
// HumTransposer::perfectFourthClass -- Return the integer interval class
//    representing a perfect fourth.
//

int HumTransposer::perfectFourthClass() {
	return m_diatonicMapping[3] - m_diatonicMapping[0]; // F - C
}



//////////////////////////////
//
// HumTransposer::perfectFifthClass -- Return the integer interval class
//    representing a perfect fifth.
//

int HumTransposer::perfectFifthClass() {
	return m_diatonicMapping[4] - m_diatonicMapping[0]; // G - C
}



//////////////////////////////
//
// HumTransposer::minorSixthClass -- Return the integer interval class
//    representing a minor sixth.
//

int HumTransposer::minorSixthClass() {
	return m_diatonicMapping[5] - m_diatonicMapping[0] - 1; // A - C - 1;
}



//////////////////////////////
//
// HumTransposer::majorSixthClass -- Return the integer interval class
//    representing a major sixth.
//

int HumTransposer::majorSixthClass() {
	return m_diatonicMapping[5] - m_diatonicMapping[0]; // A - C
}



//////////////////////////////
//
// HumTransposer::minorSeventhClass -- Return the integer interval class
//    representing a minor sixth.
//

int HumTransposer::minorSeventhClass() {
	return m_diatonicMapping[6] - m_diatonicMapping[0] - 1; // B - C - 1
}



//////////////////////////////
//
// HumTransposer::majorSeventhClass -- Return the integer interval class
//    representing a major sixth.
//

int HumTransposer::majorSeventhClass() {
	return m_diatonicMapping[6] - m_diatonicMapping[0]; // B - C
}



//////////////////////////////
//
// HumTransposer::octaveClass -- Return the integer interval class
//    representing a major second.
//

int HumTransposer::perfectOctaveClass() {
	return m_base;
}



//////////////////////////////
//
// HumTransposer::humHumPitchToIntegerPitch -- Convert a pitch (octave/diatonic pitch class/chromatic
//     alteration) into an integer value according to the current base.
//

int HumTransposer::humHumPitchToIntegerPitch(const HumPitch &pitch) {
	return pitch.getOctave() * m_base + m_diatonicMapping[pitch.getDiatonicPC()] + pitch.getAccid();
}



//////////////////////////////
//
// HumTransposer::integerPitchToHumPitch -- Convert an integer within the current base
//    into a pitch (octave/diatonic pitch class/chromatic alteration).  Pitches
//    with negative octaves will have to be tested.
//

HumPitch HumTransposer::integerPitchToHumPitch(int ipitch) {
	HumPitch pitch;
	pitch.setOctave(ipitch / m_base);
	int chroma = ipitch - pitch.getOctave() * m_base;
	int mindiff = -1000;
	int mini = -1;

	int targetdiff = m_maxAccid;

	if (chroma > m_base / 2) {
		// search from B downwards
		mindiff = chroma - m_diatonicMapping.back();
		mini = (int)m_diatonicMapping.size() - 1;
		for (int i = (int)m_diatonicMapping.size() - 2; i >= 0; i--) {
			int diff = chroma - m_diatonicMapping[i];
			if (abs(diff) < abs(mindiff)) {
				mindiff = diff;
				mini = i;
			}
			if (abs(mindiff) <= targetdiff) {
				break;
			}
		}
	}
	else {
		// search from C upwards
		mindiff = chroma - m_diatonicMapping[0];
		mini = 0;
		for (int i = 1; i < (int)m_diatonicMapping.size(); i++) {
			int diff = chroma - m_diatonicMapping[i];
			if (abs(diff) < abs(mindiff)) {
				mindiff = diff;
				mini = i;
			}
			if (abs(mindiff) <= targetdiff) {
				break;
			}
		}
	}
	pitch.setDiatonicPC(mini);
	pitch.setAccid(mindiff);
	return pitch;
}



//////////////////////////////
//
// HumTransposer::setBase40 -- Standard chromatic alteration mode, allowing up to double sharp/flats.
//

void HumTransposer::setBase40() {
	setMaxAccid(2);
}



//////////////////////////////
//
// HumTransposer::setBase600 -- Extended chromatic alteration mode, allowing up to 42 sharp/flats.
//

void HumTransposer::setBase600() {
	setMaxAccid(42);
}



//////////////////////////////
//
// HumTransposer::getInterval -- Return the interval between two pitches.
//    If the second pitch is higher than the first, then the interval will be
//    positive; otherwise, the interval will be negative.
//

int HumTransposer::getInterval(const HumPitch &p1, const HumPitch &p2) {
	return humHumPitchToIntegerPitch(p2) - humHumPitchToIntegerPitch(p1);
}

// Similar function to getInterval, but the integer interval class is converted
// into a string that is not dependent on a base:

string HumTransposer::getIntervalName(const HumPitch &p1, const HumPitch &p2) {
	int iclass = getInterval(p1, p2);
	return getIntervalName(iclass);
}

string HumTransposer::getIntervalName(int intervalClass) {
	string direction;
	if (intervalClass < 0) {
		direction = "-";
		intervalClass = -intervalClass;
	}

	int octave = intervalClass / m_base;
	int chroma = intervalClass - octave * m_base;

	int mindiff = chroma;
	int mini = 0;
	for (int i = 1; i < (int)m_diatonicMapping.size(); i++) {
		int diff = chroma - (m_diatonicMapping[i] - m_diatonicMapping[0]);
		if (abs(diff) < abs(mindiff)) {
			mindiff = diff;
			mini = i;
		}
		if (abs(mindiff) <= m_maxAccid) {
			break;
		}
	}

	int number = INVALID_INTERVAL_CLASS;
	int diminished = 0;
	int augmented = 0;
	string quality;

	switch (mini) {
		case 0: // unison
			number = 1;
			if (mindiff == 0) {
				quality = "P";
			}
			else if (mindiff < 0) {
				diminished = -mindiff;
			}
			else if (mindiff > 0) {
				augmented = mindiff;
			}
			break;
		case 1: // second
			number = 2;
			if (mindiff == 0) {
				quality = "M";
			}
			else if (mindiff == -1) {
				quality = "m";
			}
			else if (mindiff < 0) {
				diminished = -mindiff - 1;
			}
			else if (mindiff > 0) {
				augmented = mindiff;
			}
			break;
		case 2: // third
			number = 3;
			if (mindiff == 0) {
				quality = "M";
			}
			else if (mindiff == -1) {
				quality = "m";
			}
			else if (mindiff < 0) {
				diminished = -mindiff - 1;
			}
			else if (mindiff > 0) {
				augmented = mindiff;
			}
			break;
		case 3: // fourth
			number = 4;
			if (mindiff == 0) {
				quality = "P";
			}
			else if (mindiff < 0) {
				diminished = -mindiff;
			}
			else if (mindiff > 0) {
				augmented = mindiff;
			}
			break;
		case 4: // fifth
			number = 5;
			if (mindiff == 0) {
				quality = "P";
			}
			else if (mindiff < 0) {
				diminished = -mindiff;
			}
			else if (mindiff > 0) {
				augmented = mindiff;
			}
			break;
		case 5: // sixth
			number = 6;
			if (mindiff == 0) {
				quality = "M";
			}
			else if (mindiff == -1) {
				quality = "m";
			}
			else if (mindiff < 0) {
				diminished = -mindiff - 1;
			}
			else if (mindiff > 0) {
				augmented = mindiff;
			}
			break;
		case 6: // seventh
			number = 7;
			if (mindiff == 0) {
				quality = "M";
			}
			else if (mindiff == -1) {
				quality = "m";
			}
			else if (mindiff < 0) {
				diminished = -mindiff - 1;
			}
			else if (mindiff > 0) {
				augmented = mindiff;
			}
			break;
	}

	if (quality.empty()) {
		if (augmented) {
			for (int i = 0; i < augmented; i++) {
				quality += "A";
			}
		}
		else if (diminished) {
			for (int i = 0; i < diminished; i++) {
				quality += "d";
			}
		}
		else {
			quality = "?";
		}
	}

	number += octave * 7;

	string output = direction;
	output += quality;
	output += to_string(number);

	return output;
}



//////////////////////////////
//
// HumTransposer::intervalToCircleOfFifths -- Returns the circle-of-fiths count
//    that is represented by the given interval class or interval string.
//    Examples:  "P5"  => +1      "-P5" => -1
//               "P4"  => -1      "-P4" => +1
//               "M2"  => +2      "m7"  => -2
//               "M6"  => +3      "m3"  => -3
//               "M3"  => +4      "m6"  => -4
//               "M7"  => +5      "m2"  => -5
//               "A4"  => +6      "d5"  => -6
//               "A1"  => +7      "d1"  => -7
//
// If a key-signature plus the transposition interval in circle-of-fifths format
// is greater than +/-7, Then the -/+ 7 should be added to the key signature to
// avoid double sharp/flats in the key signature (and the transposition interval
// should be adjusted accordingly).
//

int HumTransposer::intervalToCircleOfFifths(const string &transstring) {
	int intervalClass = getInterval(transstring);
	return intervalToCircleOfFifths(intervalClass);
}

int HumTransposer::intervalToCircleOfFifths(int transval) {
	if (transval < 0) {
		transval = (m_base * 100 + transval) % m_base;
	}
	else if (transval == 0) {
		return 0;
	}
	else {
		transval %= m_base;
	}

	int p5 = perfectFifthClass();
	int p4 = perfectFourthClass();
	for (int i = 1; i < m_base; i++) {
		if ((p5 * i) % m_base == transval) {
			return i;
		}
		if ((p4 * i) % m_base == transval) {
			return -i;
		}
	}
	return INVALID_INTERVAL_CLASS;
}



//////////////////////////////
//
// HumTransposer::circleOfFifthsToIntervalClass -- Inputs a circle-of-fifths value and
//   returns the interval class as an integer in the current base.
//

int HumTransposer::circleOfFifthsToIntervalClass(int fifths) {
	if (fifths == 0) {
		return 0;
	}
	else if (fifths > 0) {
		return (perfectFifthClass() * fifths) % m_base;
	}
	else {
		return (perfectFourthClass() * (-fifths)) % m_base;
	}
}



//////////////////////////////
//
// HumTransposer::circleOfFifthsToIntervalName -- Convert a circle-of-fifths position
//    into an interval string.
//

string HumTransposer::circleOfFifthsToIntervalName(int fifths) {
	int intervalClass = circleOfFifthsToIntervalClass(fifths);
	return getIntervalName(intervalClass);
}



//////////////////////////////
//
// HumTransposer::circleOfFifthsTomajorTonic -- Return the tonic
//    of the major key that has the given key signature.  Return
//    value is in the 0th octave.
//

HumPitch HumTransposer::circleOfFifthsToMajorTonic(int fifths) {
	int intervalClass = circleOfFifthsToIntervalClass(fifths);
	return integerPitchToHumPitch((getCPitchClass() + intervalClass) % getBase());
}



//////////////////////////////
//
// HumTransposer::circleOfFifthsToMinorTonic -- Return the tonic
//    of the minor key that has the given key signature.  Return
//    value is in the 0th octave.
//

HumPitch HumTransposer::circleOfFifthsToMinorTonic(int fifths) {
	int intervalClass = circleOfFifthsToIntervalClass(fifths);
	return integerPitchToHumPitch((getAPitchClass() + intervalClass) % getBase());
}



//////////////////////////////
//
// HumTransposer::circleOfFifthsToDorianTonic -- Return the tonic
//    of the dorian key that has the given key signature.  Return
//    value is in the 0th octave.
//

HumPitch HumTransposer::circleOfFifthsToDorianTonic(int fifths) {
	int intervalClass = circleOfFifthsToIntervalClass(fifths);
	return integerPitchToHumPitch((getDPitchClass() + intervalClass) % getBase());
}



//////////////////////////////
//
// HumTransposer::circleOfFifthsToPhrygianTonic -- Return the tonic
//    of the phrygian key that has the given key signature.  Return
//    value is in the 0th octave.
//

HumPitch HumTransposer::circleOfFifthsToPhrygianTonic(int fifths) {
	int intervalClass = circleOfFifthsToIntervalClass(fifths);
	return integerPitchToHumPitch((getEPitchClass() + intervalClass) % getBase());
}



//////////////////////////////
//
// HumTransposer::circleOfFifthsToLydianTonic -- Return the tonic
//    of the lydian key that has the given key signature.  Return
//    value is in the 0th octave.
//

HumPitch HumTransposer::circleOfFifthsToLydianTonic(int fifths) {
	int intervalClass = circleOfFifthsToIntervalClass(fifths);
	return integerPitchToHumPitch((getFPitchClass() + intervalClass) % getBase());
}



//////////////////////////////
//
// HumTransposer::circleOfFifthsToMixolydianTonic -- Return the tonic
//    of the mixolydian key that has the given key signature.  Return
//    value is in the 0th octave.
//

HumPitch HumTransposer::circleOfFifthsToMixolydianTonic(int fifths) {
	int intervalClass = circleOfFifthsToIntervalClass(fifths);
	return integerPitchToHumPitch((getGPitchClass() + intervalClass) % getBase());
}



//////////////////////////////
//
// HumTransposer::circleOfFifthsToLocrianTonic -- Return the tonic
//    of the locrian key that has the given key signature.  Return
//    value is in the 0th octave.
//

HumPitch HumTransposer::circleOfFifthsToLocrianTonic(int fifths) {
	int intervalClass = circleOfFifthsToIntervalClass(fifths);
	return integerPitchToHumPitch((getBPitchClass() + intervalClass) % getBase());
}



//////////////////////////////
//
// HumTransposer::diatonicChromaticToIntervalClass -- Convert a diatonic/chromatic interval
//    into a base-n interval class integer.
//      +1D +1C = m2
//      +1D +2C = M2
//      +1D +3C = A2
//      +2D +4C = M3
//      +2D +3C = m3
//      +2D +2C = m3
//      +2D +1C = d3
//      +3D +5C = P4
//      +3D +6C = A4
//      +3D +4C = d4
//
//

string HumTransposer::diatonicChromaticToIntervalName(int diatonic, int chromatic) {
	if (diatonic == 0) {
		string output;
		if (chromatic == 0) {
			output += "P";
		}
		else if (chromatic > 0) {
			for (int i = 0; i < chromatic; i++) {
				output += "A";
			}
		}
		else {
			for (int i = 0; i < -chromatic; i++) {
				output += "d";
			}
		}
		output += "1";
		return output;
	}

	int octave = 0;
	string direction;
	if (diatonic < 0) {
		direction = "-";
		octave = -diatonic / 7;
		diatonic = (-diatonic - octave * 7);
		chromatic = -chromatic;
	}
	else {
		octave = diatonic / 7;
		diatonic = diatonic - octave * 7;
	}

	int augmented = 0;
	int diminished = 0;
	string quality;

	switch (abs(diatonic)) {
		case 0: // unsion
			if (chromatic == 0) {
				quality = "P";
			}
			else if (chromatic > 0) {
				augmented = chromatic;
			}
			else {
				diminished = chromatic;
			}
			break;
		case 1: // second
			if (chromatic == 2) {
				quality = "M";
			}
			else if (chromatic == 1) {
				quality = "m";
			}
			else if (chromatic > 2) {
				augmented = chromatic - 2;
			}
			else {
				diminished = chromatic - 1;
			}
			break;
		case 2: // third
			if (chromatic == 4) {
				quality = "M";
			}
			else if (chromatic == 3) {
				quality = "m";
			}
			else if (chromatic > 4) {
				augmented = chromatic - 4;
			}
			else {
				diminished = chromatic - 3;
			}
			break;
		case 3: // fourth
			if (chromatic == 5) {
				quality = "P";
			}
			else if (chromatic > 5) {
				augmented = chromatic - 5;
			}
			else {
				diminished = chromatic - 5;
			}
			break;
		case 4: // fifth
			if (chromatic == 7) {
				quality = "P";
			}
			else if (chromatic > 7) {
				augmented = chromatic - 7;
			}
			else {
				diminished = chromatic - 7;
			}
			break;
		case 5: // sixth
			if (chromatic == 9) {
				quality = "M";
			}
			else if (chromatic == 8) {
				quality = "m";
			}
			else if (chromatic > 9) {
				augmented = chromatic - 9;
			}
			else {
				diminished = chromatic - 8;
			}
			break;
		case 6: // seventh
			if (chromatic == 11) {
				quality = "M";
			}
			else if (chromatic == 10) {
				quality = "m";
			}
			else if (chromatic > 11) {
				augmented = chromatic - 11;
			}
			else {
				diminished = chromatic - 10;
			}
			break;
	}

	augmented = abs(augmented);
	diminished = abs(diminished);

	if (quality.empty()) {
		if (augmented) {
			for (int i = 0; i < augmented; i++) {
				quality += "A";
			}
		}
		else if (diminished) {
			for (int i = 0; i < diminished; i++) {
				quality += "d";
			}
		}
	}

	return direction + quality + to_string(octave * 7 + diatonic + 1);
}



//////////////////////////////
//
// HumTransposer::diatonicChromaticToIntervalClass --
//

int HumTransposer::diatonicChromaticToIntervalClass(int diatonic, int chromatic) {
	string intervalName = diatonicChromaticToIntervalName(diatonic, chromatic);
	return getInterval(intervalName);
}



//////////////////////////////
//
// HumTransposer::intervalToDiatonicChromatic --
//

void HumTransposer::intervalToDiatonicChromatic(int &diatonic, int &chromatic, int intervalClass) {
	string intervalName = getIntervalName(intervalClass);
	intervalToDiatonicChromatic(diatonic, chromatic, intervalName);
}

void HumTransposer::intervalToDiatonicChromatic(int &diatonic, int &chromatic, const string &intervalName) {
	int direction = 1;
	string quality;
	string number;
	int state = 0;

	for (int i = 0; i < (int)intervalName.size(); i++) {
		switch (state) {
			case 0: // direction or quality expected
				switch (intervalName[i]) {
					case '-': // interval is down
						direction = -1;
						state++;
						break;
					case '+': // interval is up
						direction = 1;
						state++;
						break;
					default: // interval is up by default
						direction = 1;
						state++;
						i--;
						break;
				}
				break;

			case 1: // quality expected
				if (isdigit(intervalName[i])) {
					state++;
					i--;
				}
				else {
					switch (intervalName[i]) {
						case 'M': // major
							quality = "M";
							break;
						case 'm': // minor
							quality = "m";
							break;
						case 'P': // perfect
						case 'p': quality = "P"; break;
						case 'D': // diminished
						case 'd': quality += "d"; break;
						case 'A': // augmented
						case 'a': quality += "A"; break;
					}
				}
				break;

			case 2: // digit expected
				if (isdigit(intervalName[i])) {
					number += intervalName[i];
				}
				break;
		}
	}

	if (quality.empty()) {
		cerr << "Interval requires a chromatic quality: " << intervalName << endl;
		chromatic = INVALID_INTERVAL_CLASS;
		diatonic = INVALID_INTERVAL_CLASS;
		return;
	}

	if (number.empty()) {
		cerr << "Interval requires a diatonic interval number: " << intervalName << endl;
		chromatic = INVALID_INTERVAL_CLASS;
		diatonic = INVALID_INTERVAL_CLASS;
		return;
	}

	int dnum = stoi(number);
	if (dnum == 0) {
		cerr << "Integer interval number cannot be zero: " << intervalName << endl;
		chromatic = INVALID_INTERVAL_CLASS;
		diatonic = INVALID_INTERVAL_CLASS;
		return;
	}
	dnum--;
	int octave = dnum / 7;
	dnum = dnum - octave * 7;

	diatonic = direction * (octave * 7 + dnum);
	chromatic = 0;

	switch (dnum) {
		case 0: // unison
			if (quality[0] == 'A') {
				chromatic = (int)quality.size();
			}
			else if (quality[0] == 'd') {
				chromatic = -(int)quality.size();
			}
			else if (quality == "P") {
				chromatic = 0;
			}
			else {
				cerr << "Error in Interval quality: " << intervalName << endl;
				chromatic = INVALID_INTERVAL_CLASS;
				diatonic = INVALID_INTERVAL_CLASS;
				return;
			}
			break;
		case 1: // second
			if (quality == "M") {
				chromatic = 2;
			}
			else if (quality == "m") {
				chromatic = 1;
			}
			else if (quality[0] == 'A') {
				chromatic = 2 + (int)quality.size();
			}
			else if (quality[0] == 'd') {
				chromatic = 1 - (int)quality.size();
			}
			else {
				cerr << "Error in Interval quality: " << intervalName << endl;
				chromatic = INVALID_INTERVAL_CLASS;
				diatonic = INVALID_INTERVAL_CLASS;
				return;
			}
			break;
		case 2: // third
			if (quality == "M") {
				chromatic = 4;
			}
			else if (quality == "m") {
				chromatic = 3;
			}
			else if (quality[0] == 'A') {
				chromatic = 4 + (int)quality.size();
			}
			else if (quality[0] == 'd') {
				chromatic = 3 - (int)quality.size();
			}
			else {
				cerr << "Error in Interval quality: " << intervalName << endl;
				chromatic = INVALID_INTERVAL_CLASS;
				diatonic = INVALID_INTERVAL_CLASS;
				return;
			}
			break;
		case 3: // fourth
			if (quality[0] == 'A') {
				chromatic = 5 + (int)quality.size();
			}
			else if (quality[0] == 'd') {
				chromatic = 5 - (int)quality.size();
			}
			else if (quality == "P") {
				chromatic = 5;
			}
			else {
				cerr << "Error in Interval quality: " << intervalName << endl;
				chromatic = INVALID_INTERVAL_CLASS;
				diatonic = INVALID_INTERVAL_CLASS;
				return;
			}
			break;
		case 4: // fifth
			if (quality[0] == 'A') {
				chromatic = 7 + (int)quality.size();
			}
			else if (quality[0] == 'd') {
				chromatic = 7 - (int)quality.size();
			}
			else if (quality == "P") {
				chromatic = 7;
			}
			else {
				cerr << "Error in Interval quality: " << intervalName << endl;
				chromatic = INVALID_INTERVAL_CLASS;
				diatonic = INVALID_INTERVAL_CLASS;
				return;
			}
			break;
		case 5: // sixth
			if (quality == "M") {
				chromatic = 9;
			}
			else if (quality == "m") {
				chromatic = 8;
			}
			else if (quality[0] == 'A') {
				chromatic = 9 + (int)quality.size();
			}
			else if (quality[0] == 'd') {
				chromatic = 8 - (int)quality.size();
			}
			else {
				cerr << "Error in Interval quality: " << intervalName << endl;
				chromatic = INVALID_INTERVAL_CLASS;
				diatonic = INVALID_INTERVAL_CLASS;
				return;
			}
			break;
		case 6: // seventh
			if (quality == "M") {
				chromatic = 11;
			}
			else if (quality == "m") {
				chromatic = 10;
			}
			else if (quality[0] == 'A') {
				chromatic = 11 + (int)quality.size();
			}
			else if (quality[0] == 'd') {
				chromatic = 10 - (int)quality.size();
			}
			else {
				cerr << "Error in Interval quality: " << intervalName << endl;
				chromatic = INVALID_INTERVAL_CLASS;
				diatonic = INVALID_INTERVAL_CLASS;
				return;
			}
			break;
	}
	chromatic *= direction;
}



//////////////////////////////
//
// HumTransposer::isValidIntervalName -- Returns true if the input string
//    is a valid chromatic interval string.  A valid interval name will match
//    this regular expression:
//          (-|\+?)([Pp]|M|m|[aA]+|[dD]+)([1-9][0-9]*)
//
//    Components of the regular expression:
//
//    1:  (-|\+?) == an optional direction for the interval.  When there
//                   is no sign, a + sign is implied.  A sign on a unison (P1)
//                   will be ignored.
//    2:  ([Pp]|M|m|[aA]+|[dD]+) == The chromatic quality of the following
//                   diatonic interval number.  Meanings of the letters:
//                      P or p = perfect
//                      M      = major
//                      m      = minor
//                      A or a = augmented
//                      d or D = diminished
//                   unisons (1), fourths, fifths and octaves (8) and octave multiples
//                   of these intervals can be prefixed by P but not by M or m.  Seconds,
//                   thirds, sixths, sevenths and octave transpositions of those intervals
//                   can be prefixed by M and m but not by P.  All intervals can be prefixed
//                   with A or d (or AA/dd for doubly augmented/diminished, etc.).  M and m
//                   are case sensitive, but P, A, and d are case insensitive.  This function
//                   does not check the correct pairing of M/m and P for the diatonic intervals
//                   (such as the invalid interval construct P3 for a perfect third).
//                   HumTransposer::getInterval(const string &intervalName) will do a
//                   more thorough check for invalid pairings.  This function is used mainly to
//                   determine whether an interval or a key tonic is being used in the --transpose
//                   option for verovio.
//     3: ([1-9][0-9]*) == a positive integer representing the diatonic interval.  1 = unison,
//                   2 = second, 3 = third, and so on.  Compound intervals are allowed, such as
//                   9 for a nineth (2nd plus a perfect octave), 15 for two perfect octaves.
//
//

bool HumTransposer::isValidIntervalName(const string &name) {
	string pattern = "(-|\\+?)([Pp]|M|m|[aA]+|[dD]+)([1-9][0-9]*)";
	if (regex_search(name, regex(pattern))) {
		return true;
	}
	else {
		return false;
	}
}



//////////////////////////////
//
// HumTransposer::isValidSemitones -- Returns true if the input string
//    is a valid semitone interval string.  A valid interval name will match
//    this regular expression:
//          ^(-|\+?)(\d+)$
//
//    Components of the regular expression:
//
//    1:  (-|\+?) == an optional direction for the interval.  When there
//                   is no sign, a + sign is implied.  A sign on 0
//                   will be ignored.
//    2:  (\d+)   == The number of semitones.  0 means transpose at the
//                   unison (i.e., transpose to same pitch as input).
//                   12 means an octave, 14 means a second plus an octave,
//                   24 means two octaves.
//

bool HumTransposer::isValidSemitones(const string &name) {
	string pattern = "^(-|\\+?)(\\d+)$";
	if (regex_search(name, regex(pattern))) {
		return true;
	}
	else {
		return false;
	}
}



//////////////////////////////
//
// HumTransposer::isValidKeyTonicName -- Returns true if the input string
//    is a valid key tonic which can be used to calculate a transposition
//    interval based on the current key.  A valid key tonic will match
//    this regular expression:
//          ([+]*|[-]*)([A-Ga-g])([Ss#]*|[Ffb]*)
//
//    Components of the regular expression:
//
//    1: ([+]*|[-]*) == An optional sign for the direction of the
//                      transposition.  If there is no sign, then
//                      the closest tonic pitch class to the tonic
//                      of the data will be selected.  When The
//                      sign is double/tripled/etc., additional
//                      octaves will be added to the transposition.
//    2: ([A-Ga-g]) ==  The diatonic letter of the tonic key.  The letter
//                      is case insensitive, so "g" and "G" have the
//                      same meaning.
//    3: ([Ss#]*|[Ffb]*) == An optional accidental alteration of the
//                      diatonic letter, such as eF, ef, eb, EF, Ef, or Eb,
//                      all meaning e-flat, and aS, as, a#, AS, As, or
//                      A# all meaning a-sharp.
//

bool HumTransposer::isValidKeyTonic(const string &name) {
	string pattern = "([+]*|[-]*)([A-Ga-g])([Ss#]*|[Ffb]*)";
	if (regex_search(name, regex(pattern))) {
		return true;
	}
	else {
		return false;
	}
}

/*



/////////////////////////////////////////////////////
//
// Test program for HumTransposer class:
//

int main(void) {
	HumPitch pitch(dpc_C, 0, 4); // middle C

	HumTransposer transpose;

	// transpose.setBase40() is the default system.
	transpose.setTransposition(transpose.perfectFifthClass());
	cout << "Starting pitch:\t\t\t\t" << pitch << endl;
	transpose.HumTransposer(pitch);
	cout << "HumTransposerd up a perfect fifth:\t\t" << pitch << endl;

	// testing use of a different base for transposition:
	transpose.setBase600(); // allows up to 42 sharps or flats
	// Note that transpose value is cleared when setAccid() or setBase*() is called.
	transpose.setTransposition(-transpose.perfectFifthClass());
	transpose.HumTransposer(pitch);
	cout << "HumTransposerd back down a perfect fifth:\t" << pitch << endl;

	// testing use of interval string
	transpose.setTransposition("-m3");
	transpose.HumTransposer(pitch);
	cout << "HumTransposerd down a minor third:\t\t" << pitch << endl;

	// testing validation system for under/overflows:
	cout << endl;
	pitch.setPitch(dpc_C, 2, 4); // C##4
	cout << "Initial pitch:\t\t" << pitch << endl;
	transpose.HumTransposer(pitch, "A4"); // now F###4
	bool valid = pitch.isValid(2);
	cout << "Up an aug. 4th:\t\t" << pitch;
	if (!valid) {
		cout << "\t(not valid in base-40 system)";
	}
	cout << endl;

	// calculate interval between two pitches:
	cout << endl;
	cout << "TESTING INTERVAL NAMES IN BASE-40:" << endl;
	transpose.setBase40();
	HumPitch p1(dpc_C, 0, 4);
	HumPitch p2(dpc_F, 2, 4);
	cout << "\tInterval between " << p1 << " and " << p2;
	cout << " is " << transpose.getIntervalName(p1, p2) << endl;
	HumPitch p3(dpc_G, -2, 3);
	cout << "\tInterval between " << p1 << " and " << p3;
	cout << " is " << transpose.getIntervalName(p1, p3) << endl;

	cout << "TESTING INTERVAL NAMES IN BASE-600:" << endl;
	transpose.setBase600();
	cout << "\tInterval between " << p1 << " and " << p2;
	cout << " is " << transpose.getIntervalName(p1, p2) << endl;
	cout << "\tInterval between " << p1 << " and " << p3;
	cout << " is " << transpose.getIntervalName(p1, p3) << endl;
	cout << endl;

	cout << "TESTING INTERVAL NAME TO CIRCLE-OF-FIFTHS:" << endl;
	cout << "\tM6 should be 3:  " << transpose.intervalTocircleOfFifths("M6") << endl;
	cout << "\tm6 should be -4: " << transpose.intervalTocircleOfFifths("m6") << endl;

	cout << "TESTING CIRCLE-OF-FIFTHS TO INTERVAL NAME:" << endl;
	cout << "\t3 should be M6:  " << transpose.circleOfFifthsToIntervalName(3) << endl;
	cout << "\t-4 should be m6: " << transpose.circleOfFifthsToIntervalName(-4) << endl;
	cout << endl;

	cout << "TESTING INTERVAL NAME TO DIATONIC/CHROMATIC:" << endl;
	cout << "\tD-1,C-2 should be -M2:  " << transpose.diatonicChromaticToIntervalName(-1, -2) << endl;
	cout << "\tD3,C6 should be A4:     " << transpose.diatonicChromaticToIntervalName(3, 6) << endl;

	int chromatic;
	int diatonic;

	cout << "TESTING DIATONIC/CHROMATIC TO INTERVAL NAME:" << endl;
	cout << "\t-M2 should be D-1,C-2:  ";
	transpose.intervalToDiatonicChromatic(diatonic, chromatic, "-M2");
	cout << "D" << diatonic << ",C" << chromatic << endl;
	cout << "\tA4 should be D3,C6:     ";
	transpose.intervalToDiatonicChromatic(diatonic, chromatic, "A4");
	cout << "D" << diatonic << ",C" << chromatic << endl;

	return 0;
}

*/

/* Example output from test program:

	Starting pitch:                        C4
	Transposed up a perfect fifth:         G4
	Transposed back down a perfect fifth:  C4
	Transposed down a minor third:         A3

	Initial pitch:   C##4
	Up an aug. 4th:  F###4 (not valid in base-40 system)

	TESTING INTERVAL NAMES IN BASE-40:
		Interval between C4 and F##4 is AA4
		Interval between C4 and Gbb3 is -AA4
	TESTING INTERVAL NAMES IN BASE-600:
		Interval between C4 and F##4 is AA4
		Interval between C4 and Gbb3 is -AA4

	TESTING INTERVAL NAME TO CIRCLE-OF-FIFTHS:
		M6 should be 3:  3
		m6 should be -4: -4
	TESTING CIRCLE-OF-FIFTHS TO INTERVAL NAME:
		3 should be M6:  M6
		-4 should be m6: m6

	TESTING INTERVAL NAME TO DIATONIC/CHROMATIC:
		D-1,C-2 should be -M2:  -M2
		D3,C6 should be A4:     A4
	TESTING DIATONIC/CHROMATIC TO INTERVAL NAME:
		-M2 should be D-1,C-2:  D-1,C-2
		A4 should be D3,C6:     D3,C6

 */


//////////////////////////////
//
// HumdrumFile::HumdrumFile -- HumdrumFile constructor.
//

HumdrumFile::HumdrumFile(void) {
	// do nothing
}

HumdrumFile::HumdrumFile(const string& filename) :
		HUMDRUMFILE_PARENT() {
	read(filename);
}


HumdrumFile::HumdrumFile(istream& contents) :
		HUMDRUMFILE_PARENT() {
	read(contents);
}



//////////////////////////////
//
// HumdrumFile::~HumdrumFile -- HumdrumFile deconstructor.
//

HumdrumFile::~HumdrumFile() {
	// do nothing
}



//////////////////////////////
//
// HumdrumFile::printXml -- Print a HumdrumFile object in XML format.
// default value: level = 0
// default value: indent = tab character
//

ostream& HumdrumFile::printXml(ostream& out, int level,
		const string& indent) {
	out << Convert::repeatString(indent, level) << "<sequence>\n";
	level++;

	out << Convert::repeatString(indent, level) << "<sequenceInfo>\n";
	level++;

	out << Convert::repeatString(indent, level) << "<frameCount>";
	out << getLineCount() << "</frameCount>\n";


	out << Convert::repeatString(indent, level) << "<tpq>";
	out << tpq() << "</tpq>\n";

	// Starting at 0 by default (single segment only).  Eventually
	// add parameter to set the starting time of the sequence, which
	// would be the duration of all previous segments before this one.
	out << Convert::repeatString(indent, level) << "<sequenceStart";
	out << Convert::getHumNumAttributes(0);
	out << "/>\n";

	out << Convert::repeatString(indent, level) << "<sequenceDuration";
	out << Convert::getHumNumAttributes(getScoreDuration());
	out << "/>\n";

	out << Convert::repeatString(indent, level) << "<trackInfo>\n";
	level++;

	out << Convert::repeatString(indent, level) << "<trackCount>";
	out << getMaxTrack() << "</trackCount>\n";

	for (int i=1; i<=getMaxTrack(); i++) {
		out << Convert::repeatString(indent, level) << "<track";
		out << " n=\"" << i << "\"";
		HumdrumToken* trackstart = getTrackStart(i);
		if (trackstart != NULL) {
			out << " dataType=\"" <<  trackstart->getDataType().substr(2) << "\"";
			out << " startId=\"" <<  trackstart->getXmlId() << "\"";
		}
		HumdrumToken* trackend = getTrackEnd(i, 0);
		if (trackend != NULL) {
			out << " endId =\"" <<  trackend->getXmlId() << "\"";
		}
		out << "/>\n";
	}

	level--;
	out << Convert::repeatString(indent, level) << "</trackInfo>\n";

	printXmlParameterInfo(out, level, "\t");

	level--;
	out << Convert::repeatString(indent, level) << "</sequenceInfo>\n";

	out << Convert::repeatString(indent, level) << "<frames>\n";
	level++;
	for (int i=0; i<getLineCount(); i++) {
		m_lines[i]->printXml(out, level, indent);
	}
	level--;
	out << Convert::repeatString(indent, level) << "</frames>\n";

	level--;
	out << Convert::repeatString(indent, level) << "</sequence>\n";

	return out;
}



//////////////////////////////
//
// HumdrumFile::printXmlParameterInfo -- Print contents of HumHash for HumdrumFile.
// default value: out = cout
// default value: level = 0
// default value: indent = "\t"
//

ostream& HumdrumFile::printXmlParameterInfo(ostream& out, int level,
		const string& indent) {
	((HumHash*)this)->printXml(out, level, indent);
	return out;
}




//////////////////////////////
//
// HumdrumFileBase::getUriToUrlMapping --
//

string HumdrumFileBase::getUriToUrlMapping(const string& uri) {
	auto css = uri.find("://");
	if (css == string::npos) {
		// this is not a URI, so just return input:
		return string(uri);
	}

	if (Convert::startsWith(uri, "http://")) {
		// The URI is a URL, so just copy:
		return string(uri);
	}

	string tag  = uri.substr(0, css);
	string rest = uri.substr(css+3);
	if (rest.empty()) {
		rest = "/";
	}

	// getting a repertory:
	// http://kern.humdrum.org/data?l=osu/classical/bach/inventions
	// getting a single file:
	// http://kern.humdrum.org/data?s=http://kern.humdrum.org/data?s=osu/classical/bach/inventions&file=inven15.krn
	// (Should allow repertory from &s...)
	if ((tag == "humdrum") || (tag == "hum") || (tag == "h")) {
		string testlocation;
		string testfilename;
		int repertoryQ = false;
		auto slash = rest.rfind('/');
		if (slash != string::npos) {
			testlocation = rest.substr(0, slash);
			testfilename = rest.substr(slash+1);
			if (testfilename.find('.') == string::npos) {
				repertoryQ = true;
			}
		} if (slash == string::npos) {
			// no files in root directory, but no reperoties either
			repertoryQ = true;
		}
		string output = "http://";;
		output += "kern.ccarh.org";
		output += "/data?";
		if (repertoryQ) {
			output += "l=";
		} else {
			output += "s=";
		}
		output += rest;
		// probably not needed:
		//output += "&format=kern";
		return output;
	}

	if (tag == "jrp") {
		string output = "http://";
		output += "jrp.ccarh.org";
		output += "/cgi-bin/jrp?a=humdrum";
		output += "&f=";
		output += rest;
		return output;
	}

	// not familiar with the URI, just assume that it is a URL,
	// such as "https://".
	return uri;
}


#ifdef USING_URI

//////////////////////////////
//
// HumdrumFileBase::readFromHumdrumUri -- Read a Humdrum file from an
//      humdrum:// web address
//
// Example:
//    maps: humdrum://osu/classical/haydn/london/sym099a.krn
// into:
//    http://kern.ccarh.org/cgi-bin/ksdata?file=sym099a.krn&l=/osu/classical/haydn/london&format=kern
//

void HumdrumFileBase::readFromHumdrumUri(const string& humaddress) {
	string url = HumdrumFileBase::getUriToUrlMapping(humaddress);
	readFromHttpUri(url);
}



//////////////////////////////
//
// readFromJrpUri -- Read a Humdrum file from a jrp:// web-style address
//
// Example:
// maps:
//    jrp://Jos2721-La_Bernardina
// into:
//    http://jrp.ccarh.org/cgi-bin/jrp?a=humdrum&f=Jos2721-La_Bernardina
//

void HumdrumFileBase::readFromJrpUri(const string& jrpaddress) {
	string url = HumdrumFileBase::getUriToUrlMapping(jrpaddress);
	readFromHttpUri(url);
}


//////////////////////////////
//
// HumdrumFileBase::readFromHttpUri -- download content from the web.
//

void HumdrumFileBase::readFromHttpUri(const string& webaddress) {
	stringstream inputdata;
	readStringFromHttpUri(inputdata, webaddress);
	HumdrumFileBase::readString(inputdata.str());
}



//////////////////////////////
//
// readStringFromHttpUri -- Read a Humdrum file from an http:// web address
//

void HumdrumFileBase::readStringFromHttpUri(stringstream& inputdata,
		const string& webaddress) {
	auto css = webaddress.find("://");
	if (css == string::npos) {
		// give up since URI was not in correct format
	}
	string rest = webaddress.substr(css+3);
	string hostname;
	string location;
	css = rest.find("/");
	if (css != string::npos) {
		hostname = rest.substr(0, css);
		location = rest.substr(css);
	} else {
		hostname = rest;
		location = "/";
	}
	if (location.empty()) {
		location = "/";
	}

	string newline({0x0d, 0x0a});

	stringstream request;
	request << "GET "   << location << " HTTP/1.1" << newline;
	request << "Host: " << hostname << newline;
	request << "User-Agent: HumdrumFile Downloader 2.0 ("
		     << __DATE__ << ")" << newline;
	request << "Connection: close" << newline;  // this line is necessary
	request << newline;

	unsigned short int port = 80;
	int socket_id = open_network_socket(hostname, port);
	if (::write(socket_id, request.str().c_str(), request.str().size()) == -1) {
		exit(-1);
	}

	#define URI_BUFFER_SIZE (10000)
	char buffer[URI_BUFFER_SIZE];
	unsigned int message_len;
	stringstream header;
	int foundcontent   = 0;
	int newlinecounter = 0;
	int i;

	// read the response header:
	while ((message_len = ::read(socket_id, buffer, 1)) != 0) {
		header << buffer[0];
		if ((buffer[0] == 0x0a) || (buffer[0] == 0x0d)) {
					newlinecounter++;
		} else {
					newlinecounter = 0;
		}
		if (newlinecounter == 4) {
			foundcontent = 1;
			break;
		}
	}
	if (foundcontent == 0) {
		cerr << "Funny error trying to read server response" << endl;
		exit(1);
	}

	// now read the size of the rest of the data which is expected
	int datalength = -1;

	// also, check for chunked transfer encoding:

	int chunked = 0;

	header << ends; // necessary?
	while (header.getline(buffer, URI_BUFFER_SIZE)) {
		int len = (int)strlen(buffer);
		for (i=0; i<len; i++) {
			buffer[i] = std::tolower(buffer[i]);
		}
		if (strstr(buffer, "content-length") != NULL) {
			for (i=14; i<len; i++) {
				if (std::isdigit(buffer[i])) {
					sscanf(&buffer[i], "%d", &datalength);
					if (datalength == 0) {
						cerr << "Error: no data found for URI, probably invalid\n";
						cerr << "URL:   " << webaddress << endl;
						exit(1);
					}
					break;
				}
			}
		} else if ((strstr(buffer, "transfer-encoding") != NULL) &&
			(strstr(buffer, "chunked") != NULL)) {
			chunked = 1;
		}
	}

	// once the length of the remaining data is known (or not), read it:
	if (datalength > 0) {
		getFixedDataSize(socket_id, datalength, inputdata, buffer,
				URI_BUFFER_SIZE);

	} else if (chunked) {
		int chunksize;
		int totalsize = 0;
		do {
			chunksize = getChunk(socket_id, inputdata, buffer, URI_BUFFER_SIZE);
			totalsize += chunksize;
		} while (chunksize > 0);
		if (totalsize == 0) {
			cerr << "Error: no data found for URI (probably invalid)\n";
			exit(1);
		}
	} else {
		// if the size of the rest of the data cannot be found in the
		// header, then just keep reading until done (but this will
		// probably cause a 5 second delay at the last read).
		while ((message_len = ::read(socket_id, buffer, URI_BUFFER_SIZE)) != 0) {
			if (foundcontent) {
				inputdata.write(buffer, message_len);
			} else {
				for (i=0; i<(int)message_len; i++) {
					if (foundcontent) {
						inputdata << buffer[i];
					} else {
						header << buffer[i];
						if ((buffer[i] == 0x0a) || (buffer[i] == 0x0d)) {
							newlinecounter++;
						} else {
							newlinecounter = 0;
						}
						if (newlinecounter == 4) {
							foundcontent = 1;
							continue;
						}
					}

				}
			}
		}
	}

	close(socket_id);
}



//////////////////////////////
//
//  HumdrumFileBase::getChunk --
//
// http://en.wikipedia.org/wiki/Chunked_transfer_encoding
// http://tools.ietf.org/html/rfc2616
//
// Chunk Format
//
// If a Transfer-Encoding header with a value of chunked is specified in
// an HTTP message (either a request sent by a client or the response from
// the server), the body of the message is made of an unspecified number
// of chunks ending with a last, zero-sized, chunk.
//
// Each non-empty chunk starts with the number of octets of the data it
// embeds (size written in hexadecimal) followed by a CRLF (carriage
// return and linefeed), and the data itself. The chunk is then closed
// with a CRLF. In some implementations, white space chars (0x20) are
// padded between chunk-size and the CRLF.
//
// The last chunk is a single line, simply made of the chunk-size (0),
// some optional padding white spaces and the terminating CRLF. It is not
// followed by any data, but optional trailers can be sent using the same
// syntax as the message headers.
//
// The message is finally closed by a last CRLF combination.

int HumdrumFileBase::getChunk(int socket_id, stringstream& inputdata,
		char* buffer, int bufsize) {
	int chunksize = 0;
	unsigned int message_len;
	char digit[2] = {0};
	int founddigit = 0;

	// first read the chunk size:
	while ((message_len = ::read(socket_id, buffer, 1)) != 0) {
		if (isxdigit(buffer[0])) {
			digit[0] = buffer[0];
			chunksize = (chunksize << 4) | (int)strtol(digit, NULL, 16);
			founddigit = 1;
		} else if (founddigit) {
			break;
		} // else skipping whitespace before chunksize
	}
	if ((chunksize <= 0) || (message_len == 0)) {
		// next chunk is zero, so no more primary data0:w
		return 0;
	}

	// read the 0x0d and 0x0a characters which are expected (required)
	// after the size of chunk size:
	if (buffer[0] != 0x0d) {
		cerr << "Strange error occurred right after reading a chunk size" << endl;
		exit(1);
	}

	// now expect 0x0a:
	message_len = ::read(socket_id, buffer, 1);
	if ((message_len == 0) || (buffer[0] != 0x0a)) {
		cerr << "Strange error after reading newline at end of chunk size"<< endl;
		exit(1);
	}

	return getFixedDataSize(socket_id, chunksize, inputdata, buffer, bufsize);
}



//////////////////////////////
//
// getFixedDataSize -- read a know amount of data from a socket.
//

int HumdrumFileBase::getFixedDataSize(int socket_id, int datalength,
		stringstream& inputdata, char* buffer, int bufsize) {
	int readcount = 0;
	int readsize;
	int message_len;

	while (readcount < datalength) {
		readsize = bufsize;
		if (readcount + readsize > datalength) {
			readsize = datalength - readcount;
		}
		message_len = (int)::read(socket_id, buffer, readsize);
		if (message_len == 0) {
			// shouldn't happen, but who knows...
			break;
		}
		inputdata.write(buffer, message_len);
		readcount += message_len;
	}

	return readcount;
}



//////////////////////////////
//
// HumdrumFileBase::prepare_address -- Store a computer name, such as
//    www.google.com into a sockaddr_in structure for later use in
//    open_network_socket.
//

void HumdrumFileBase::prepare_address(struct sockaddr_in *address,
		const string& hostname, unsigned short int port) {

	memset(address, 0, sizeof(struct sockaddr_in));
	struct hostent *host_entry;
	host_entry = gethostbyname(hostname.c_str());

	if (host_entry == NULL) {
		cerr << "Could not find address for " << hostname << endl;
		exit(1);
	}

	// copy the address to the sockaddr_in struct.
	memcpy(&address->sin_addr.s_addr, host_entry->h_addr_list[0],
			host_entry->h_length);

	// set the family type (PF_INET)
	address->sin_family = host_entry->h_addrtype;
	address->sin_port = htons(port);
}



//////////////////////////////
//
// open_network_socket -- Open a connection to a computer on the internet.
//    Intended for downloading a Humdrum file from a website.
//

int HumdrumFileBase::open_network_socket(const string& hostname,
		unsigned short int port) {
	int inet_socket;                 // socket descriptor
	struct sockaddr_in servaddr;     // IP/port of the remote host

	prepare_address(&servaddr, hostname, port);

	// socket(domain, type, protocol)
	//    domain   = PF_INET(internet/IPv4 domain)
	//    type     = SOCK_STREAM(tcp) *
	//    protocol = 0 (only one SOCK_STREAM type in the PF_INET domain
	inet_socket = socket(PF_INET, SOCK_STREAM, 0);

	if (inet_socket < 0) {
		// socket returns -1 on error
		cerr << "Error opening socket to computer " << hostname << endl;
		exit(1);
	}
	if (connect(inet_socket, (struct sockaddr *)&servaddr,
			sizeof(struct sockaddr_in)) < 0) {
		// connect returns -1 on error
		cerr << "Error opening connection to coputer: " << hostname << endl;
		exit(1);
	}

	return inet_socket;
}

#endif




//////////////////////////////
//
// HumdrumFileBase::HumdrumFileBase -- HumdrumFileBase constructor.
//

HumdrumFileBase::HumdrumFileBase(void) : HumHash() {
	addToTrackStarts(NULL);
	m_ticksperquarternote = -1;
	m_quietParse = false;
	m_segmentlevel = 0;
}

HumdrumFileBase::HumdrumFileBase(const string& filename) : HumHash() {
	addToTrackStarts(NULL);
	m_ticksperquarternote = -1;
	m_quietParse = false;
	m_segmentlevel = 0;
	read(filename);
}

HumdrumFileBase::HumdrumFileBase(istream& contents) : HumHash() {
	addToTrackStarts(NULL);
	m_ticksperquarternote = -1;
	m_quietParse = false;
	m_segmentlevel = 0;
	read(contents);
}


//
// HumdrumFileStructure::analyzeStructure() needs to be called after
// using the following constructor:
//

HumdrumFileBase::HumdrumFileBase(HumdrumFileBase& infile) {

	m_filename = infile.m_filename;
	m_segmentlevel = infile.m_segmentlevel;
	m_trackstarts.clear();
	m_trackends.clear();
	m_barlines.clear();
	m_ticksperquarternote = infile.m_ticksperquarternote;
	m_idprefix = infile.m_idprefix;
	m_strand1d.clear();
	m_strand2d.clear();
	m_strophes1d.clear();
	m_strophes2d.clear();
	m_quietParse = infile.m_quietParse;
	m_parseError = infile.m_parseError;
	m_displayError = infile.m_displayError;

	m_lines.resize(infile.m_lines.size());
	for (int i=0; i<(int)m_lines.size(); i++) {
		m_lines[i] = new HumdrumLine(infile.m_lines[i]->getText());
		m_lines[i]->setOwner(this);
	}

	analyzeBaseFromLines();
}



//////////////////////////////
//
// HumdrumFileBase::operator = -- HumdrumFileStructure::analyzeStructure()
// needs to be called after copying from another HumdrumFile.
//
//

HumdrumFileBase& HumdrumFileBase::operator=(HumdrumFileBase& infile) {
	if (this == &infile) {
		return *this;
	}

	m_filename = infile.m_filename;
	m_segmentlevel = infile.m_segmentlevel;
	m_trackstarts.clear();
	m_trackends.clear();
	m_barlines.clear();
	m_ticksperquarternote = infile.m_ticksperquarternote;
	m_idprefix = infile.m_idprefix;
	m_strand1d.clear();
	m_strand2d.clear();
	m_strophes1d.clear();
	m_strophes2d.clear();
	m_quietParse = infile.m_quietParse;
	m_parseError = infile.m_parseError;
	m_displayError = infile.m_displayError;

	m_lines.resize(infile.m_lines.size());
	for (int i=0; i<(int)m_lines.size(); i++) {
		m_lines[i] = new HumdrumLine(infile.m_lines[i]->getText());
		m_lines[i]->setOwner(this);
	}

	analyzeBaseFromLines();
	return *this;
}



//////////////////////////////
//
// HumdrumFileBase::~HumdrumFileBase -- HumdrumFileBase deconstructor.
//

HumdrumFileBase::~HumdrumFileBase() {
	clear();
}



//////////////////////////////
//
// HumdrumFileBase::clear -- Reset the contents of a file to be empty.
//

void HumdrumFileBase::clear(void) {
	// delete memory allocation:
	for (int i=0; i<(int)m_lines.size(); i++) {
		if (m_lines[i] != NULL) {
			delete m_lines[i];
			m_lines[i] = NULL;
		}
	}
	m_lines.clear();

	// clear state variables which are now invalid:
	m_trackstarts.clear();
	m_trackends.clear();
	m_barlines.clear();
	m_ticksperquarternote = -1;
	m_idprefix.clear();
	m_strand1d.clear();
	m_strand2d.clear();
	m_strophes1d.clear();
	m_strophes2d.clear();
	m_filename.clear();
	m_segmentlevel = 0;
	m_analyses.clear();
}



//////////////////////////////
//
// HumdrumFileBase::isStructureAnalyzed --
//

bool HumdrumFileBase::isStructureAnalyzed(void) {
	return m_analyses.m_structure_analyzed;
}



//////////////////////////////
//
// HumdrumFileBase::isRhythmAnalyzed --
//

bool HumdrumFileBase::isRhythmAnalyzed(void) {
	return m_analyses.m_rhythm_analyzed;
}



//////////////////////////////
//
// HumdrumFileBase::areStrandsAnalyzed --
//

bool HumdrumFileBase::areStrandsAnalyzed(void) {
	return m_analyses.m_strands_analyzed;
}



//////////////////////////////
//
// HumdrumFileBase::areStrandsAnalyzed --
//

bool HumdrumFileBase::areStrophesAnalyzed(void) {
	return m_analyses.m_strophes_analyzed;
}



//////////////////////////////
//
// HumdrumFileBase::setXmlIdPrefix -- Set the prefix for a HumdrumXML ID
//     atrribute.  The prefix should not start with a digit, nor have
//     spaces in it.
//

void HumdrumFileBase::setXmlIdPrefix(const string& value) {
	m_idprefix = value;
}



//////////////////////////////
//
// HumdrumFileBase::getXmlIdPrefix -- Return the HumdrumXML ID attribute prefix.
//

string HumdrumFileBase::getXmlIdPrefix(void) {
	return m_idprefix;
}



//////////////////////////////
//
// HumdrumFileBase::operator[] -- Access a Humdrum file line by and index.
//    Negative values reference the end of the list of lines.
//

HumdrumLine& HumdrumFileBase::operator[](int index) {
	if (index < 0) {
		index = (int)m_lines.size() - index;
	}
	if ((index < 0) || (index >= (int)m_lines.size())) {
		cerr << "Error: invalid index: " << index << endl;
		cerr << "Max index is " << m_lines.size() - 1 << endl;
		index = (int)m_lines.size()-1;
	}
	return *m_lines[index];
}



//////////////////////////////
//
// HumdrumFileBase::setParseError -- Set an error message from parsing
//     input data.  The size of the message will keep track of whether
//     or not an error was generated.  If no error message is generated
//     when reading data, then the parsing of the data is assumed to be
//     good.
//

bool HumdrumFileBase::setParseError(const string& err) {
	m_parseError = err;
	return !m_parseError.size();
}


bool HumdrumFileBase::setParseError(stringstream& err) {
	m_parseError = err.str();
	return !m_parseError.size();
}


bool HumdrumFileBase::setParseError(const char* format, ...) {
	char buffer[1024] = {0};
	va_list ap;
	va_start(ap, format);
	snprintf(buffer, 1024, format, ap);
	va_end(ap);
	m_parseError = buffer;
	return !m_parseError.size();
}



//////////////////////////////
//
// HumdrumFileBase::read -- Load file contents from an input stream or file.
//

bool HumdrumFileBase::read(const string& filename) {
	m_displayError = true;
	return HumdrumFileBase::read(filename.c_str());
}


bool HumdrumFileBase::read(const char* filename) {
	string fname = filename;
	m_displayError = true;

#ifdef USING_URI
	if (fname.find("://") != string::npos) {
		if (Convert::startsWith(fname, "http://")) {
			readFromHttpUri(fname);
			return isValid();
		}
		if (Convert::startsWith(fname, "jrp://")) {
			readFromJrpUri(fname);
			return isValid();
		}
		if (Convert::startsWith(fname, "h://") ||
			Convert::startsWith(fname, "hum://") ||
			Convert::startsWith(fname, "humdrum://")) {
			readFromHumdrumUri(fname);
			return isValid();
		}
	}
#endif

	ifstream infile;
	if (fname.empty() || (fname ==  "-")) {
		return HumdrumFileBase::read(cin);
	} else {
		infile.open(filename);
		if (!infile.is_open()) {
			return setParseError("Cannot open file >>%s<< for reading. A", filename);
		}
	}
	HumdrumFileBase::read(infile);
	infile.close();
	return isValid();
}


bool HumdrumFileBase::read(istream& contents) {
	clear();
	m_displayError = true;
	char buffer[123123] = {0};
	HLp s;
	while (contents.getline(buffer, sizeof(buffer), '\n')) {
		s = new HumdrumLine(buffer);
		s->setOwner(this);
		m_lines.push_back(s);
	}
	return analyzeBaseFromLines();
/*
	if (!analyzeTokens()) { return isValid(); }
	if (!analyzeLines() ) { return isValid(); }
	if (!analyzeSpines()) { return isValid(); }
	if (!analyzeLinks() ) { return isValid(); }
	if (!analyzeTracks()) { return isValid(); }
	return isValid();
*/
}



//////////////////////////////
//
// HumdrumFileBase::readCsv -- Read a Humdrum file in CSV format
//    (rather than TSV format).
// default value: separator = ","
//

bool HumdrumFileBase::readCsv(const string& filename, const string& separator) {
	return HumdrumFileBase::readCsv(filename.c_str());
}


bool HumdrumFileBase::readCsv(const char* filename, const string& separator) {
	ifstream infile;
	if ((strlen(filename) == 0) || (strcmp(filename, "-") == 0)) {
		return HumdrumFileBase::readCsv(cin, separator);
	} else {
		infile.open(filename);
		if (!infile.is_open()) {
			return setParseError("Cannot open file %s for reading. B", filename);
		}
	}
	HumdrumFileBase::readCsv(infile, separator);
	infile.close();
	return isValid();
}


bool HumdrumFileBase::readCsv(istream& contents, const string& separator) {
	m_displayError = true;
	char buffer[123123] = {0};
	HLp s;
	while (contents.getline(buffer, sizeof(buffer), '\n')) {
		s = new HumdrumLine;
		s->setLineFromCsv(buffer);
		s->setOwner(this);
		m_lines.push_back(s);
	}
	return analyzeBaseFromLines();
}



//////////////////////////////
//
// HumdrumFileBase::analyzeBaseFromLines --
//

bool HumdrumFileBase::analyzeBaseFromLines(void)  {
	if (!analyzeTokens()) { return isValid(); }
	if (!analyzeLines() ) { return isValid(); }
	if (!analyzeSpines()) { return isValid(); }
	if (!analyzeLinks() ) { return isValid(); }
	if (!analyzeTracks()) { return isValid(); }
	return isValid();
}



//////////////////////////////
//
// HumdrumFileBase::analyzeBaseFromTokens --
//

bool HumdrumFileBase::analyzeBaseFromTokens(void) {
	// if (!analyzeTokens()) { return isValid(); } // this creates tokens from lines
	if (!analyzeLines() ) { return isValid(); }
	if (!analyzeSpines()) { return isValid(); }
	if (!analyzeLinks() ) { return isValid(); }
	if (!analyzeTracks()) { return isValid(); }
	return isValid();
}



//////////////////////////////
//
// HumdrumFileBase::readString -- Read contents from a string rather than
//    an istream or filename.
//

bool HumdrumFileBase::readString(const string& contents) {
	stringstream infile;
	infile << contents;
	bool status = read(infile);
	return status;
}


bool HumdrumFileBase::readString(const char* contents) {
	stringstream infile;
	infile << contents;
	return read(infile);
}



//////////////////////////////
//
// HumdrumFileBase::readStringCsv -- Reads Humdrum data in CSV format.
//

bool HumdrumFileBase::readStringCsv(const char* contents,
		const string& separator) {
	stringstream infile;
	infile << contents;
	return readCsv(infile, separator);
}


bool HumdrumFileBase::readStringCsv(const string& contents,
		const string& separator) {
	stringstream infile;
	infile << contents;
	return readCsv(infile, separator);
}



//////////////////////////////
//
// HumdrumFileBase::getParseError -- Return parse fail reason.
//

string HumdrumFileBase::getParseError(void) const {
	return m_parseError;
}



//////////////////////////////
//
// HumdrumFileBase::isValid -- Returns true if last read was
//     successful.
//

bool HumdrumFileBase::isValid(void) {
	if (m_displayError && (m_parseError.size() > 0)&& !isQuiet()) {
		cerr << m_parseError << endl;
		m_displayError = false;
	}
	return m_parseError.empty();
}



//////////////////////////////
//
// HumdrumFileBase::setQuietParsing -- Prevent error messages from
//   being displayed when reading data.
// @SEEALSO: setNoisyParsing
// @SEEALSO: isQuiet
//

void HumdrumFileBase::setQuietParsing(void) {
	m_quietParse = true;
}



//////////////////////////////
//
// HumdrumFileBase::setFilename --
//

void HumdrumFileBase::setFilename(const string& filename) {
	m_filename = filename;
}



//////////////////////////////
//
// HumdrumFileBase::getFilename --
//

string HumdrumFileBase::getFilename(void) {
	return m_filename;
}



//////////////////////////////
//
// HumdrumFileBase::getFilenameBase -- Remove any path and any
//    dot followed by non-dots.
//

string HumdrumFileBase::getFilenameBase(void) {
	string output;
	auto pos = m_filename.rfind('/');
	if (pos != string::npos) {
		output = m_filename.substr(pos+1);
	} else {
		output = m_filename;
	}
	pos = output.rfind('.');
	if (pos != string::npos) {
		output = output.substr(0,pos);
	}
	return output;
}



//////////////////////////////
//
// HumdrumFileBase::printSegmentLabel --
//

ostream& HumdrumFileBase::printSegmentLabel(ostream& out) {
	out << "!!!!SEGMENT";
	string filename = getFilename();
	int segment = getSegmentLevel();
	if (segment != 0) {
		if (segment < 0) {
			out << segment;
		} else {
			out << "+" << segment;
		}
	}
	out << ": " << filename << endl;
	return out;
}



//////////////////////////////
//
// HumdrumFileBase::printNonemptySegmentLabel --
//

ostream& HumdrumFileBase::printNonemptySegmentLabel(ostream& out) {
	if (getFilename().size() > 0) {
		printSegmentLabel(out);
	}
	return out;
}



//////////////////////////////
//
// HumdrumFileBase::getSegmentLevel -- return the segment level
//

int HumdrumFileBase::getSegmentLevel(void) {
	return m_segmentlevel;
}



//////////////////////////////
//
// HumdrumFileBase::setSegmentLevel -- return the segment level
//

void HumdrumFileBase::setSegmentLevel(int level) {
	m_segmentlevel = level;
}

//////////////////////////////
//
// HumdrumFileBase::setNoisyParsing -- Display error messages
//   on console when reading data.
// @SEEALSO: setQuietParsing
// @SEEALSO: isQuiet
//

void HumdrumFileBase::setNoisyParsing(void) {
	m_quietParse = false;
}



//////////////////////////////
//
// HumdrmFileBase::isQuiet -- Returns true if parsing errors
//    messages should be suppressed. By default the parsing
//    is "noisy" and the error messages will be printed to
//    standard error.
// @SEEALSO: setQuietParsing
// @SEEALSO: setNoisyParsing
//

bool HumdrumFileBase::isQuiet(void) const{
	return m_quietParse;
}



//////////////////////////////
//
// HumdrumFileBase::printCsv -- print Humdrum file content in
//     CSV format.
// default value: out = std::cout
// default value: separator = ","
//

ostream& HumdrumFileBase::printCsv(ostream& out,
		const string& separator) {
	for (int i=0; i<getLineCount(); i++) {
		((*this)[i]).printCsv(out, separator);
	}
	return out;
}



//////////////////////////////
//
// HumdrumFileBase::printFieldNumber --
//

ostream& HumdrumFileBase::printFieldNumber(int fieldnum, ostream& out) {
	return printFieldIndex(fieldnum - 1, out);
}



//////////////////////////////
//
// HumdrumFileBase::printFieldIndex --
//

ostream& HumdrumFileBase::printFieldIndex(int fieldind, ostream& out) {
	if (fieldind < 0) {
		return out;
	}
	HumdrumFileBase& infile = *this;
	for (int i=0; i<infile.getLineCount(); i++) {
		if (!infile[i].hasSpines()) {
			out << infile[i] << endl;
			continue;
		}
		cout << infile.token(i,fieldind) << endl;
	}
	return out;
}



//////////////////////////////
//
// HumdrumFileBase::getLine -- Return a pointer to the line at a
//     given index in the data storage.
//

HLp HumdrumFileBase::getLine(int index) {
	if (index < 0) {
		return NULL;
	} else if (index >= (int)m_lines.size()) {
		return NULL;
	} else {
		return m_lines[index];
	}
}



//////////////////////////////
//
// HumdrumFileBase::analyzeTokens -- Generate token array from
//    current contents of the lines.  If either tokens or the line
//    is changed, then the other state becomes invalid.
//    See createLinesFromTokens for regeneration of lines from tokens.
//

bool HumdrumFileBase::analyzeTokens(void) {
	for (int i=0; i<(int)m_lines.size(); i++) {
		m_lines[i]->createTokensFromLine();
	}
	return isValid();
}


//////////////////////////////
//
// HumdrumFileBase::removeExtraTabs --
//

void HumdrumFileBase::removeExtraTabs(void) {
	for (int i=0; i<(int)m_lines.size(); i++) {
		m_lines[i]->removeExtraTabs();
	}
}



//////////////////////////////
//
// HumdrumFileBase::addExtraTabs --
//

void HumdrumFileBase::addExtraTabs(void) {
	vector<int> trackWidths = getTrackWidths();

	HumdrumFileBase& infile = *this;
	vector<int> local(trackWidths.size());
	for (int i=0; i<infile.getLineCount(); i++) {
		infile[i].addExtraTabs(trackWidths);
	}
}



//////////////////////////////
//
// HumdrumFileBase::getTrackWidths -- Return a list of the maximum
//    subspine count for each track in the file.  The 0th track is
//    not used, so it will be zero and ignored.
//

vector<int> HumdrumFileBase::getTrackWidths(void) {
	HumdrumFileBase& infile = *this;
	vector<int> output(infile.getTrackCount() + 1, 1);
	output[0] = 0;
	vector<int> local(infile.getTrackCount() + 1);
	for (int i=0; i<infile.getLineCount(); i++) {
		if (!infile[i].hasSpines()) {
			continue;
		}
		fill(local.begin(), local.end(), 0);
		for (int j=0; j<infile[i].getFieldCount(); j++) {
			HTp token = infile.token(i, j);
			int track = token->getTrack();
			local[track]++;
		}
		for (int j=1; j<(int)local.size(); j++) {
			if (local[j] > output[j]) {
				output[j] = local[j];
			}
		}
	}
	return output;
}



//////////////////////////////
//
// HumdrumFileBase::createLinesFromTokens -- Generate Humdrum lines strings
//   from the stored list of tokens.
//

void HumdrumFileBase::createLinesFromTokens(void) {
	for (int i=0; i<(int)m_lines.size(); i++) {
		m_lines[i]->createLineFromTokens();
	}
}



////////////////////////////
//
// HumdrumFileBase::appendLine -- Add a line to the file's contents.  The file's
//    spine and rhythmic structure should be recalculated after an append.
//

void HumdrumFileBase::appendLine(const string& line) {
	HLp s = new HumdrumLine(line);
	m_lines.push_back(s);
}


void HumdrumFileBase::appendLine(HLp line) {
	// deletion will be handled by class.
	m_lines.push_back(line);
}



////////////////////////////
//
// HumdrumFileBase::insertLine -- Add a line to the file's contents.  The file's
//    spine and rhythmic structure should be recalculated after an append.
//

void HumdrumFileBase::insertLine(int index, const string& line) {
	HLp s = new HumdrumLine(line);
	m_lines.insert(m_lines.begin() + index, s);

	// Update the line indexes for this line and the following ones:
	for (int i=index; i<(int)m_lines.size(); i++) {
		m_lines[i]->setLineIndex(i);
	}
}


void HumdrumFileBase::insertLine(int index, HLp line) {
	// deletion will be handled by class.
	m_lines.insert(m_lines.begin() + index, line);

	// Update the line indexes for this line and the following ones:
	for (int i=index; i<(int)m_lines.size(); i++) {
		m_lines[i]->setLineIndex(i);
	}
}



//////////////////////////////
//
// HumdrumFileBase::deleteLine -- remove a line from the Humdrum file.
//    Is best used for global comments and reference records for now.
//    Other line types will cause parsing problems untill further
//    generalized to stitch previous next lines together.
//

void HumdrumFileBase::deleteLine(int index) {
	if (index >= (int)m_lines.size()) {
		return;
	}
	if (index < 0) {
		return;
	}
	delete m_lines[index];
	for (int i=index+1; i<(int)m_lines.size(); i++) {
		m_lines[i-1] = m_lines[i];
	}
	m_lines.resize(m_lines.size() - 1);
}


//////////////////////////////
//
// HumdrumFileBase::back --
//

HLp HumdrumFileBase::back(void) {
	return m_lines.back();
}



//////////////////////////////
//
// HumdrumFileBase::getReferenceRecords --
//

vector<HLp> HumdrumFileBase::getReferenceRecords(void) {
	vector<HLp> hlps;
	hlps.reserve(32);
	HLp hlp;
	auto& infile = *this;
	for (int i=0; i<infile.getLineCount(); i++) {
		if (infile[i].isReference()) {
			hlp = &infile[i];
			hlps.push_back(hlp);
		}
	}
	return hlps;
}



//////////////////////////////
//
// HumdrumFileBase::getGlobalReferenceRecords --
//

vector<HLp> HumdrumFileBase::getGlobalReferenceRecords(void) {
	vector<HLp> hlps;
	hlps.reserve(32);
	HLp hlp;
	auto& infile = *this;
	for (int i=0; i<infile.getLineCount(); i++) {
		if (infile[i].isGlobalReference()) {
			hlp = &infile[i];
			hlps.push_back(hlp);
		}
	}
	return hlps;
}



//////////////////////////////
//
// HumdrumFileBase::getUniversalReferenceRecords --
//

vector<HLp> HumdrumFileBase::getUniversalReferenceRecords(void) {
	vector<HLp> hlps;
	hlps.reserve(32);
	HLp hlp;
	HumdrumFileBase& infile = *this;
	for (int i=0; i<infile.getLineCount(); i++) {
		if (infile[i].isUniversalReference()) {
			hlp = &infile[i];
			hlps.push_back(hlp);
		}
	}
	return hlps;
}



////////////////////////////
//
// HumdrumFileBase::getLineCount -- Returns the number of lines.
//

int HumdrumFileBase::getLineCount(void) const {
	return (int)m_lines.size();
}



//////////////////////////////
//
// HumdrumFileBase::token -- Return the token at the given line/field index.
//

HTp HumdrumFileBase::token(int lineindex, int fieldindex) {
	if (lineindex < 0) {
		lineindex += getLineCount();
	}
	return m_lines[lineindex]->token(fieldindex);
}


//
// Special case that returns a subtoken string:
//   default value separator = " "
//

string HumdrumFileBase::token(int lineindex, int fieldindex,
		int subtokenindex, const string& separator) {
	return token(lineindex, fieldindex)->getSubtoken(subtokenindex, separator);
}



//////////////////////////////
//
// HumdrumFileBase::getMaxTrack -- Returns the number of primary
//     spines in the data.
//

int HumdrumFileBase::getMaxTrack(void) const {
	return (int)m_trackstarts.size() - 1;
}



//////////////////////////////
//
// HumdrumFileBase::printSpineInfo -- Print the spine information for all
//    lines/tokens in file (for debugging).
//

ostream& HumdrumFileBase::printSpineInfo(ostream& out) {
	for (int i=0; i<getLineCount(); i++) {
		m_lines[i]->printSpineInfo(out) << '\n';
	}
	return out;
}



//////////////////////////////
//
// HumdrumFileBase::printDataTypeInfo -- Print the data type for all
//     spines in the file (for debugging).
//

ostream& HumdrumFileBase::printDataTypeInfo(ostream& out) {
	for (int i=0; i<getLineCount(); i++) {
		m_lines[i]->printDataTypeInfo(out) << '\n';
	}
	return out;
}



//////////////////////////////
//
// HumdrumFileBase::printTrackInfo -- Print the track numbers for all
//     tokens in the file (for debugging).
//

ostream& HumdrumFileBase::printTrackInfo(ostream& out) {
	for (int i=0; i<getLineCount(); i++) {
		m_lines[i]->printTrackInfo(out) << '\n';
	}
	return out;
}


//////////////////////////////
//
// HumdrumFileBase::getExinterpCount -- return the number of spines in
//    the file that are of the given exinterp type.  The input string
//    may optionally include the ** exinterp prefix.
//

int HumdrumFileBase::getExinterpCount(const string& exinterp) {
	vector<HTp> spinestarts;
	getSpineStartList(spinestarts, exinterp);
	return (int)spinestarts.size();
}



//////////////////////////////
//
// HumdrumFileBase::getSpineStopList -- Return a list of the ending
//     points of spine strands.
//

void HumdrumFileBase::getSpineStopList(vector<HTp>& spinestops) {
	spinestops.reserve(m_trackends.size());
	spinestops.resize(0);
	for (int i=0; i<(int)m_trackends.size(); i++) {
		for (int j=0; j<(int)m_trackends[i].size(); j++) {
			spinestops.push_back(m_trackends[i][j]);
		}
	}
}



//////////////////////////////
//
// HumdrumFileBase::getSpineStartList -- Return a list of the exclustive
//     interpretations starting spines in the data.  The single parameter
//     version of the fuction returns all starting exclusive interpretations.
//     The two-parameter version will result all exclusive interpretations
//     of a given datatype, and the three-parameter version where the third
//     parameter is a vector of string, will selectively include all starting
//     tokens which match one of the data types in the input list.  The
//     trackstarts class variable contains an empty slot at index 0;
//     this is removed in the return vector.
//

void HumdrumFileBase::getSpineStartList(vector<HTp>& spinestarts) {
	spinestarts.reserve(m_trackstarts.size());
	spinestarts.resize(0);
	for (int i=1; i<(int)m_trackstarts.size(); i++) {
		spinestarts.push_back(m_trackstarts[i]);
	}
}


void HumdrumFileBase::getSpineStartList(vector<HTp>& spinestarts,
		const string& exinterp) {
	string newexinterp;
	if (exinterp.compare(0, 2, "**") == 0) {
		newexinterp = exinterp;
	} else {
		newexinterp = "**";
		newexinterp += exinterp;
	}
	spinestarts.reserve(m_trackstarts.size());
	spinestarts.resize(0);
	for (int i=1; i<(int)m_trackstarts.size(); i++) {
		if (newexinterp == *m_trackstarts[i]) {
			spinestarts.push_back(m_trackstarts[i]);
		}
	}
}


void HumdrumFileBase::getSpineStartList(vector<HTp>& spinestarts,
		const vector<string>& exinterps) {
	vector<string> newexinterps(exinterps.size());
	for (int i=0; i<(int)exinterps.size(); i++) {
		if (exinterps[i].compare(0, 2, "**") == 0) {
			newexinterps[i] = exinterps[i];
		} else {
			newexinterps[i] = "**";
			newexinterps[i] += exinterps[i];
		}
	}
	spinestarts.reserve(m_trackstarts.size());
	spinestarts.resize(0);
	for (int i=1; i<(int)m_trackstarts.size(); i++) {
		for (int j=0; j<(int)newexinterps.size(); j++) {
			if (newexinterps[j] == *m_trackstarts[i]) {
				spinestarts.push_back(m_trackstarts[i]);
			}
		}
	}
}


void HumdrumFileBase::getKernSpineStartList(vector<HTp>& spinestarts) {
	getSpineStartList(spinestarts, "**kern");
}

vector<HTp> HumdrumFileBase::getKernSpineStartList(void) {
	vector<HTp> starts;
	HumdrumFileBase::getKernSpineStartList(starts);
	return starts;
}



//////////////////////////////
//
// getPrimaryspineSequence -- Return a list of the HumdrumTokens in a spine,
//    but not any secondary spine content if the spine splits.
//


void HumdrumFileBase::getPrimarySpineSequence(vector<HTp>& sequence, int spine,
		int options) {
	getPrimaryTrackSequence(sequence, spine+1, options);
}



//////////////////////////////
//
// getPrimaryspineSequence -- Return a list of the HumdrumTokens in a spine,
//    but not any secondary spine content if the spine splits.
//


void HumdrumFileBase::getSpineSequence(vector<vector<HTp> >& sequence,
		HTp starttoken, int options) {
	getTrackSequence(sequence, starttoken, options);

}


void HumdrumFileBase::getSpineSequence(vector<vector<HTp> >& sequence,
		int spine, int options) {
	getTrackSequence(sequence, spine+1, options);
}



//////////////////////////////
//
// HumdrumFileBase::getPrimaryTrackSequence -- Return a list of the
//     given primary spine tokens for a given track (indexed starting at
//     one and going through getMaxTrack().
//

void HumdrumFileBase::getPrimaryTrackSequence(vector<HTp>& sequence, int track,
		int options) {
	vector<vector<HTp> > tempseq;
	getTrackSequence(tempseq, track, options | OPT_PRIMARY);
	sequence.resize(tempseq.size());
	for (int i=0; i<(int)tempseq.size(); i++) {
		sequence[i] = tempseq[i][0];
	}
}



/////////////////////////////
//
// HumdrumFileBase::getTrackSequence -- Extract a sequence of tokens
//    for the given spine.  All subspine tokens will be included.
//    See getPrimaryTrackSequence() if you only want the first subspine for
//    a track on all lines.
//
// The following options are used for the getPrimaryTrackTokens:
// * OPT_PRIMARY    => only extract primary subspine/subtrack.
// * OPT_NOEMPTY    => don't include null tokens in extracted list if all
//                        extracted subspines contains null tokens.
//                        Includes null interpretations and comments as well.
// * OPT_NONULL     => don't include any null tokens in extracted list.
// * OPT_NOINTERP   => don't include interprtation tokens.
// * OPT_NOMANIP    => don't include spine manipulators (*^, *v, *x, *+,
//                        but still keep ** and *0).
// * OPT_NOCOMMENT  => don't include comment tokens.
// * OPT_NOGLOBAL   => don't include global records (global comments, reference
//                        records, and empty lines). In other words, only return
//                        a list of tokens from lines which hasSpines() it true.
// * OPT_NOREST     => don't include **kern rests.
// * OPT_NOTIE      => don't include **kern secondary tied notes.
// Compound options:
// * OPT_DATA      (OPT_NOMANIP | OPT_NOCOMMENT | OPT_NOGLOBAL)
//     Only data tokens (including barlines)
// * OPT_ATTACKS   (OPT_DATA | OPT_NOREST | OPT_NOTIE | OPT_NONULL)
//     Only note-attack tokens (when etracting **kern data)
//

void HumdrumFileBase::getTrackSequence(vector<vector<HTp> >& sequence,
		HTp starttoken, int options) {
	int track = starttoken->getTrack();
	getTrackSequence(sequence, track, options);
}


void HumdrumFileBase::getTrackSequence(vector<vector<HTp> >& sequence,
		int track, int options) {
	bool primaryQ   = (options & OPT_PRIMARY) ? true : false;
	bool nonullQ    = (options & OPT_NONULL) ? true : false;
	bool noemptyQ   = (options & OPT_NOEMPTY) ? true : false;
	bool nointerpQ  = (options & OPT_NOINTERP) ? true : false;
	bool nomanipQ   = (options & OPT_NOMANIP) ? true : false;
	bool nocommentQ = (options & OPT_NOCOMMENT) ? true : false;
	bool noglobalQ  = (options & OPT_NOGLOBAL) ? true : false;
	bool norestQ    = (options & OPT_NOREST) ? true : false;
	bool notieQ     = (options & OPT_NOTIE) ? true : false;

	vector<vector<HTp> >& output = sequence;
	output.reserve(getLineCount());
	output.resize(0);

	vector<HTp> tempout;
	auto& infile = *this;
	int i, j;
	bool allNull;
	HTp token;
	bool foundTrack;

	for (i=0; i<infile.getLineCount(); i++) {
		if (infile[i].isEmpty()) {
			continue;
		}
		tempout.resize(0);
		if (!noglobalQ && (infile[i].isGlobal())) {
			tempout.push_back(infile[i].token(0));
			output.push_back(tempout);
			continue;
		}
		if (noemptyQ) {
			allNull = true;
			for (j=0; j<infile[i].getFieldCount(); j++) {
				if (infile[i].token(j)->getTrack() != track) {
					continue;
				}
				if (!infile[i].token(j)->isNull()) {
					allNull = false;
					break;
				}
			}
			if (allNull) {
				continue;
			}
		}

		foundTrack = false;
		for (j=0; j<infile[i].getFieldCount(); j++) {
			token = infile[i].token(j);
			if (token->getTrack() != track) {
				continue;
			}
			if (primaryQ && foundTrack) {
				continue;
			}
			foundTrack = true;
			if (nointerpQ && (infile[i].token(j)->isManipulator() ||
					infile[i].token(j)->isTerminator() ||
					infile[i].token(j)->isExclusive())) {
				continue;
			}
			if (nomanipQ && infile[i].token(j)->isManipulator()) {
				continue;
			}
			if (nonullQ && infile[i].token(j)->isNull()) {
				continue;
			}
			if (nocommentQ && infile[i].token(j)->isComment()) {
				continue;
			}
			if (norestQ && infile[i].token(j)->isRest()) {
				continue;
			}
			if (notieQ && infile[i].token(j)->isSecondaryTiedNote()) {
				continue;
			}

			tempout.push_back(infile[i].token(j));
		}
		if (tempout.size() > 0) {
			output.push_back(tempout);
		}
	}
}



//////////////////////////////
//
// HumdrumFileBase::getTrackStart -- Return the starting exclusive
//     interpretation for the given track.  Returns NULL if the track
//     number is out of range.
//

HTp HumdrumFileBase::getTrackStart(int track) const {
	if ((track > 0) && (track < (int)m_trackstarts.size())) {
		return m_trackstarts[track];
	} else {
		return NULL;
	}
}



//////////////////////////////
//
// HumdrumFileBase::getTrackEndCount -- Return the number of ending tokens
//    for the given track.  Spines must start as a single exclusive
//    interpretation token.  However, since spines may split and merge,
//    it is possible that there are more than one termination points for a
//    track.  This function returns the number of terminations which are
//    present in a file for any given spine/track.
//

int HumdrumFileBase::getTrackEndCount(int track) const {
	if (track < 0) {
		track += (int)m_trackends.size();
	}
	if (track < 0) {
		return 0;
	}
	if (track >= (int)m_trackends.size()) {
		return 0;
	}
	return (int)m_trackends[track].size();
}



//////////////////////////////
//
// HumdrumFileBase::getTrackEnd -- Returns a pointer to the terminal manipulator
//    token for the given track and subtrack.  Sub-tracks are indexed from 0 up
//    to but not including getTrackEndCount.
//

HTp HumdrumFileBase::getTrackEnd(int track, int subtrack) const {
	if (track < 0) {
		track += (int)m_trackends.size();
	}
	if (track < 0) {
		return NULL;
	}
	if (track >= (int)m_trackends.size()) {
		return NULL;
	}
	if (subtrack < 0) {
		subtrack += (int)m_trackends[track].size();
	}
	if (subtrack < 0) {
		return NULL;
	}
	if (subtrack >= (int)m_trackends[track].size()) {
		return NULL;
	}
	return m_trackends[track][subtrack];
}



//////////////////////////////
//
// HumdrumFileBase::analyzeLines -- Store a line's index number in the
//    HumdrumFile within the HumdrumLine object at that index.
//    Returns false if there was an error.
//

bool HumdrumFileBase::analyzeLines(void) {
	for (int i=0; i<(int)m_lines.size(); i++) {
		m_lines[i]->setLineIndex(i);
	}
	return isValid();
}



//////////////////////////////
//
// HumdrumFileBase::analyzeTracks -- Analyze the track structure of the
//     data.  Returns false if there was a parse error.
//

bool HumdrumFileBase::analyzeTracks(void) {
	for (int i=0; i<(int)m_lines.size(); i++) {
		int status = m_lines[i]->analyzeTracks(m_parseError);
		if (!status) {
			return false;
		}
	}
	return isValid();
}



//////////////////////////////
//
// HumdrumFileBase::analyzeLinks -- Generate forward and backwards spine links
//    for each token.
//

bool HumdrumFileBase::analyzeLinks(void) {
	HLp next     = NULL;
	HLp previous = NULL;

	for (int i=0; i<(int)m_lines.size(); i++) {
		if (!m_lines[i]->hasSpines()) {
			continue;
		}
		previous = next;
		next = m_lines[i];
		if (previous != NULL) {
			if (!stitchLinesTogether(*previous, *next)) {
				return isValid();
			}
		}
	}
	return isValid();;
}



//////////////////////////////
//
// HumdrumFileBase::stitchLinesTogether -- Make forward/backward links for
//    tokens on each line.
//

bool HumdrumFileBase::stitchLinesTogether(HumdrumLine& previous,
		HumdrumLine& next) {
	int i;

	// first handle simple cases where the spine assignments are one-to-one:
	if (!previous.isInterpretation() && !next.isInterpretation()) {
		if (previous.getTokenCount() != next.getTokenCount()) {
			stringstream err;
			err << "Error lines " << (previous.getLineNumber())
			    << " and " << (next.getLineNumber()) << " not same length\n";
			err << "Line " << (previous.getLineNumber()) << ": " << previous << endl;
			err << "Line " << (next.getLineNumber()) << ": " << next << endl;
			return setParseError(err);
		}
		for (i=0; i<previous.getTokenCount(); i++) {
			if (next.token(i)) {
				previous.token(i)->makeForwardLink(*next.token(i));
			} else {
				cerr << "Strange error 1" << endl;
			}
		}
		return true;
	}
	int ii = 0;
	for (i=0; i<previous.getTokenCount(); i++) {
		if (!previous.token(i)->isManipulator()) {
			if (next.token(ii) != NULL) {
				previous.token(i)->makeForwardLink(*next.token(ii++));
			} else {
				cerr << "Strange error 2" << endl;
			}
		} else if (previous.token(i)->isSplitInterpretation()) {
			// connect the previous token to the next two tokens.
			if (next.token(ii) != NULL) {
				previous.token(i)->makeForwardLink(*next.token(ii++));
			} else {
				cerr << "Strange error 3" << endl;
			}
			if (next.token(ii) != NULL) {
				previous.token(i)->makeForwardLink(*next.token(ii++));
			} else {
				cerr << "Strange error 4" << endl;
			}
		} else if (previous.token(i)->isMergeInterpretation()) {
			// connect multiple previous tokens which are adjacent *v
			// spine manipulators to the current next token.
			while ((i<previous.getTokenCount()) &&
					previous.token(i)->isMergeInterpretation()) {
				if (next.token(ii) != NULL) {
					previous.token(i)->makeForwardLink(*next.token(ii));
				} else {
					cerr << "Strange error 5" << endl;
				}
				i++;
			}
			i--;
			ii++;
		} else if (previous.token(i)->isExchangeInterpretation()) {
			// swapping the order of two spines.
			if ((i<previous.getTokenCount()) &&
					previous.token(i+1)->isExchangeInterpretation()) {
				if (next.token(ii) != NULL) {
					previous.token(i+1)->makeForwardLink(*next.token(ii++));
				} else {
					cerr << "Strange error 6" << endl;
				}
				if (next.token(ii) != NULL) {
					previous.token(i)->makeForwardLink(*next.token(ii++));
				} else {
					cerr << "Strange error 7" << endl;
				}
			}
			i++;
		} else if (previous.token(i)->isTerminateInterpretation()) {
			// No link should be made.  There may be a problem if a
			// new segment is given (this should be handled by a
			// HumdrumSet class, not HumdrumFileBase.
		} else if (previous.token(i)->isAddInterpretation()) {
			// A new data stream is being added, the next linked token
			// should be an exclusive interpretation.
			if (!next.token(ii+1)->isExclusiveInterpretation()) {
				stringstream err;
				err << "Error: expecting exclusive interpretation on line "
				    << next.getLineNumber() << " at token " << i << " but got "
				    << next.token(i);
				return setParseError(err);
			}
			if (next.token(ii) != NULL) {
				previous.token(i)->makeForwardLink(*next.token(ii++));
			} else {
				cerr << "Strange error 8" << endl;
			}
			ii++;
		} else if (previous.token(i)->isExclusiveInterpretation()) {
			if (next.token(ii) != NULL) {
				if (previous.token(i) != NULL) {
					previous.token(i)->makeForwardLink(*next.token(ii++));
				} else {
					cerr << "Strange error 10" << endl;
				}
			} else {
				cerr << "Strange error 9" << endl;
			}
		} else {
			return setParseError("Error: should not get here");
		}
	}

	if ((i != previous.getTokenCount()) || (ii != next.getTokenCount())) {
		stringstream err;
		err << "Error: cannot stitch lines together due to alignment problem\n";
		err << "Line " << previous.getLineNumber() << ": "
		    << previous << endl;
		err << "Line " << next.getLineNumber() << ": "
		    << next << endl;
		err << "I = " <<i<< " token count " << previous.getTokenCount() << endl;
		err << "II = " <<ii<< " token count " << next.getTokenCount();
		return setParseError(err);
	}

	return isValid();
}



//////////////////////////////
//
// HumdrumFileBase::analyzeSpines -- Analyze the spine structure of the
//     data.  Returns false if there was a parse error.
//

bool HumdrumFileBase::analyzeSpines(void) {
	vector<string> datatype;
	vector<string> sinfo;
	vector<vector<HTp> > lastspine;
	m_trackstarts.resize(0);
	m_trackends.resize(0);
	addToTrackStarts(NULL);

	bool init = false;
	int i, j;
	for (i=0; i<getLineCount(); i++) {
		if (!m_lines[i]->hasSpines()) {
			m_lines[i]->token(0)->setFieldIndex(0);
			continue;
		}
		if ((init == false) && !m_lines[i]->isExclusive()) {
			stringstream err;
			err << "Error on line: " << (i+1) << ':' << endl;
			err << "   Data found before exclusive interpretation" << endl;
			err << "   LINE: " << *m_lines[i];
			return setParseError(err);
		}
		if ((init == false) && m_lines[i]->isExclusive()) {
			// first line of data in file.
			init = true;
			datatype.resize(m_lines[i]->getTokenCount());
			sinfo.resize(m_lines[i]->getTokenCount());
			lastspine.resize(m_lines[i]->getTokenCount());
			for (j=0; j<m_lines[i]->getTokenCount(); j++) {
				datatype[j] = m_lines[i]->getTokenString(j);
				addToTrackStarts(m_lines[i]->token(j));
				sinfo[j]    = to_string(j+1);
				m_lines[i]->token(j)->setSpineInfo(sinfo[j]);
				m_lines[i]->token(j)->setFieldIndex(j);
				lastspine[j].push_back(m_lines[i]->token(j));
			}
			continue;
		}
		if ((int)datatype.size() != m_lines[i]->getTokenCount()) {
			stringstream err;
			err << "Error on line " << (i+1) << ':' << endl;
			err << "   Expected " << datatype.size() << " fields,"
			    << "    but found " << m_lines[i]->getTokenCount();
			err << "\nLine is: " << m_lines[i] << endl;
			if (i > 0) {
				cerr << "Previous line is: " << m_lines[i-1] << endl;
			}
			return setParseError(err);
		}
		for (j=0; j<m_lines[i]->getTokenCount(); j++) {
			m_lines[i]->token(j)->setSpineInfo(sinfo[j]);
			m_lines[i]->token(j)->setFieldIndex(j);
		}
		if (!m_lines[i]->isManipulator()) {
			continue;
		}
		if (!adjustSpines(*m_lines[i], datatype, sinfo)) { return isValid(); }
	}
	return isValid();
}



//////////////////////////////
//
// HumdrumFileBase::addToTrackStarts -- A starting exclusive interpretation was
//    found, so store in the list of track starts.  The first index position
//    in trackstarts is reserve for non-spine usage.
//

void HumdrumFileBase::addToTrackStarts(HTp token) {
	if (token == NULL) {
		m_trackstarts.push_back(NULL);
		m_trackends.resize(m_trackends.size()+1);
	} else if ((m_trackstarts.size() > 1) && (m_trackstarts.back() == NULL)) {
		m_trackstarts.back() = token;
	} else {
		m_trackstarts.push_back(token);
		m_trackends.resize(m_trackends.size()+1);
	}
}



//////////////////////////////
//
// HumdrumFileBase::adjustSpines -- adjust datatype and spineinfo values based
//   on manipulators found in the data.
//

bool HumdrumFileBase::adjustSpines(HumdrumLine& line, vector<string>& datatype,
		vector<string>& sinfo) {
	vector<string> newtype;
	vector<string> newinfo;
	int mergecount = 0;
	int i, j;
	for (i=0; i<line.getTokenCount(); i++) {
		if (line.token(i)->isSplitInterpretation()) {
			newtype.resize(newtype.size() + 1);
			newtype.back() = datatype[i];
			newtype.resize(newtype.size() + 1);
			newtype.back() = datatype[i];
			newinfo.resize(newinfo.size() + 2);
			newinfo[newinfo.size()-2] = '(' + sinfo[i] + ")a";
			newinfo[newinfo.size()-1] = '(' + sinfo[i] + ")b";
		} else if (line.token(i)->isMergeInterpretation()) {
			mergecount = 0;
			for (j=i+1; j<line.getTokenCount(); j++) {
				if (line.token(j)->isMergeInterpretation()) {
					mergecount++;
				} else {
					break;
				}
			}
			newinfo.emplace_back(getMergedSpineInfo(sinfo, i, mergecount));
			newtype.push_back(datatype[i]);
			i += mergecount;
		} else if (line.token(i)->isAddInterpretation()) {
			newtype.resize(newtype.size() + 1);
			newtype.back() = datatype[i];
			newtype.resize(newtype.size() + 1);
			newtype.back() = "";
			newinfo.resize(newinfo.size() + 1);
			newinfo.back() = sinfo[i];
			newinfo.resize(newinfo.size() + 1);
			addToTrackStarts(NULL);
			newinfo.back() = to_string(getMaxTrack());
		} else if (line.token(i)->isExchangeInterpretation()) {
			if (i < line.getTokenCount() - 1) {
				if (line.token(i)->isExchangeInterpretation()) {
					// exchange spine information
					newtype.resize(newtype.size() + 1);
					newtype.back() = datatype[i+1];
					newtype.resize(newtype.size() + 1);
					newtype.back() = datatype[i];
					newinfo.resize(newinfo.size() + 1);
					newinfo.back() = sinfo[i+1];
					newinfo.resize(newinfo.size() + 1);
					newinfo.back() = sinfo[i];
				} else {
					return setParseError("ERROR1 in *x calculation");
				}
				i++;
			} else {
				stringstream err;
				err << "ERROR2 in *x calculation" << endl;
				err << "Index " << i << " larger than allowed: "
				     << line.getTokenCount() - 1;
				return setParseError(err);
			}
		} else if (line.token(i)->isTerminateInterpretation()) {
			// store pointer to terminate token in trackends
			m_trackends[m_trackstarts.size()-1].push_back(line.token(i));
		} else if (((string*)line.token(i))->substr(0, 2) == "**") {
			newtype.resize(newtype.size() + 1);
			newtype.back() = line.getTokenString(i);
			newinfo.resize(newinfo.size() + 1);
			newinfo.back() = sinfo[i];
			if (!((m_trackstarts.size() > 1) && (m_trackstarts.back() == NULL))) {
				stringstream err;
				err << "Error: Exclusive interpretation with no preparation "
				     << "on line " << line.getLineIndex()
				     << " spine index " << i << endl;
				err << "Line: " << line;
				return setParseError(err);
			}
			if (m_trackstarts.back() == NULL) {
				addToTrackStarts(line.token(i));
			}
		} else {
			// should only be null interpretation, but doesn't matter
			newtype.resize(newtype.size() + 1);
			newtype.back() = datatype[i];
			newinfo.resize(newinfo.size() + 1);
			newinfo.back() = sinfo[i];
		}
	}

	datatype.resize(newtype.size());
	sinfo.resize(newinfo.size());
	for (i=0; i<(int)newtype.size(); i++) {
		datatype[i] = newtype[i];
		sinfo[i]    = newinfo[i];
	}

	return true;
}



//////////////////////////////
//
// HumdrumFileBase::getMergedSpineInfo -- Will only simplify a two-spine
//   merge.  Should be expanded to larger spine mergers in the future.
//   In other words, it is best to currently merge spines in the order
//   in which they were split, so that the original spine label can
//   be produced.
//

string HumdrumFileBase::getMergedSpineInfo(vector<string>& info, int starti,
		int extra) {
	string output;
	int len1;
	int len2;
	if (extra < 1) {
		// Strange if get here.
		return info[starti];
	} else if (extra == 1) {
		len1 = (int)info[starti].size();
		len2 = (int)info[starti+1].size();
		if (len1 == len2) {
			if (info[starti].substr(0, len1-1) ==
					info[starti+1].substr(0,len2-1)) {
				output = info[starti].substr(1, len1-3);
				return output;
			}
		}
		output = info[starti] + " " + info[starti+1];
		return output;
	}
	// Generalized code for simplifying up to 4 subspines at once.
	// Not fully generlized so that the subspines will always be
	// simplified if not merged in a simple way, though.
	vector<string> newinfo;
	int i;
	for (i=0; i<=extra; i++) {
		newinfo.push_back(info.at(starti+i));
	}
	for (i=1; i<(int)newinfo.size(); i++) {
		int len1 = (int)newinfo[i-1].size();
		int len2 = (int)newinfo[i].size();
		if (len1 != len2) {
			continue;
		}
		if (newinfo[i-1].compare(0, len1-1, newinfo[i], 0, len2-1) == 0) {
			newinfo[i-1] = "";
			newinfo[i] = newinfo[i].substr(1, len2-3);
		}
	}
	vector<string> newinfo2;
	for (i=0; i<(int)newinfo.size(); i++) {
		if (newinfo[i].empty()) {
			continue;
		}
		newinfo2.push_back(newinfo[i]);
	}
	for (i=1; i<(int)newinfo2.size(); i++) {
		int len1 = (int)newinfo2[i-1].size();
		int len2 = (int)newinfo2[i].size();
		if (len1 != len2) {
			continue;
		}
		if (newinfo2[i-1].compare(0, len1-1, newinfo2[i], 0, len2-1) == 0) {
			newinfo2[i-1] = "";
			newinfo2[i] = newinfo2[i].substr(1, len2-3);
		}
	}
	newinfo.resize(0);
	for (i=0; i<(int)newinfo2.size(); i++) {
		if (newinfo2[i].empty()) {
			continue;
		}
		newinfo.push_back(newinfo2[i]);
	}
	output = newinfo[0];
	for (int i=1; i<(int)newinfo.size(); i++) {
		output += " " + info.at(i);
	}
	return output;
}



//////////////////////////////
//
// HumdrumFileBase::analyzeNonNullDataTokens -- For null data tokens, indicate
//    the previous non-null token which the null token refers to.  After
//    a spine merger, there may be multiple previous tokens, so you would
//		have to decide on the actual source token on based on subtrack or
//    sub-spine information.  The function also gives links to the previous/next
//    non-null tokens, skipping over intervening null data tokens.
//

bool HumdrumFileBase::analyzeNonNullDataTokens(void) {
	vector<HTp> ptokens;

	// analyze forward tokens:
	for (int i=1; i<=getMaxTrack(); i++) {
		if (!processNonNullDataTokensForTrackForward(getTrackStart(i),
				ptokens)) {
			return false;
		}
	}

	ptokens.resize(0);

	// analyze backward tokens:
	for (int i=1; i<=getMaxTrack(); i++) {
		for (int j=0; j<getTrackEndCount(i); j++) {
			if (!processNonNullDataTokensForTrackBackward(getTrackEnd(i, j),
					ptokens)) {
				return false;
			}
		}
	}

	// Eventually set the foward and backward non-null data token for
	// tokens in spines for all types of line types  For now specify
	// the next non-null data token for the exclusive interpretation token.
	// Also this implementation does not consider that the first
	// non-null data tokens may be from multiple split tokens (fix later).

	// This algorithm is probably not right, but good enough for now.
	// There may be missing portions of the file for the analysis,
	// and/or the algorithm is probably retracking tokens in the case
	// of spine splits.

	vector<HTp> stops;
	getSpineStopList(stops);
	HTp nexts = NULL;

	for (int i=0; i<(int)stops.size(); i++) {
		if (stops[i] == NULL) {
			continue;
		}
		HTp token = stops[i];
		if (token->isData() && !token->isNull()) {
			nexts = token;
		}
		token = token->getPreviousToken();

		while (token) {
			if (nexts) {
				token->addNextNonNullToken(nexts);
			}
			if (token->isData() && !token->isNull()) {
				nexts = token;
			}
			token = token->getPreviousToken();
		}
	}

	return true;
}



//////////////////////////////
//
// HumdurmFile::processNonNullDataTokensForTrackBackward -- Helper function
//    for analyzeNonNullDataTokens.  Given any token, this function tells
//    you what is the next non-null data token(s) in the spine after the given
//    token.
//

bool HumdrumFileBase::processNonNullDataTokensForTrackBackward(
		HTp endtoken, vector<HTp> ptokens) {

	HTp token = endtoken;
	int tcount = token->getPreviousTokenCount();

	while (tcount > 0) {
		for (int i=1; i<tcount; i++) {
			if (!processNonNullDataTokensForTrackBackward(
					token->getPreviousToken(i), ptokens)) {
				return false;
			}
		}
		HTp prevtoken = token->getPreviousToken();
		if (prevtoken->isSplitInterpretation()) {
			addUniqueTokens(prevtoken->m_nextNonNullTokens, ptokens);
			if (token != prevtoken->m_nextTokens[0]) {
				// terminate if not most primary subspine
				return true;
			}
		} else if (token->isData()) {
			addUniqueTokens(token->m_nextNonNullTokens, ptokens);
			if (!token->isNull()) {
				ptokens.resize(0);
				ptokens.push_back(token);
			}
		}

		// Follow previous data token 0 since 1 and higher are handled above.
		token = token->getPreviousToken(0);
		tcount = token->getPreviousTokenCount();
	}

	return true;
}



//////////////////////////////
//
// HumdurmFile::processNonNullDataTokensForTrackForward -- Helper function
//    for analyzeNonNullDataTokens.  Given any token, this function tells
//    you what are the previous non-null data token(s) in the spine before
//    the given token.
//

bool HumdrumFileBase::processNonNullDataTokensForTrackForward(HTp starttoken,
		vector<HTp> ptokens) {

	HTp token = starttoken;
	int tcount = token->getNextTokenCount();
	while (tcount > 0) {
		if (token->isSplitInterpretation()) {
			for (int i=1; i<tcount; i++) {
				if (!processNonNullDataTokensForTrackForward(
						token->getNextToken(i), ptokens)) {
					return false;
				}
			}
		} else if (token->isMergeInterpretation()) {
			HTp nexttoken = token->getNextToken();
			addUniqueTokens(nexttoken->m_previousNonNullTokens, ptokens);
			if (token != nexttoken->m_previousTokens[0]) {
				// terminate if not most primary subspine
				return true;
			}
		} else {
			addUniqueTokens(token->m_previousNonNullTokens, ptokens);
			if (token->isData() && !token->isNull()) {
				ptokens.resize(0);
				ptokens.push_back(token);

			}
		}
		// Data tokens can only be followed by up to one next token,
		// so no need to check for more than one next token.
		token = token->getNextToken(0);
		tcount = token->getNextTokenCount();
	}

	return true;
}



//////////////////////////////
//
// HumdrumFileBase::addUniqueTokens -- Used for non-null token analysis.  The
//    analysis is recursive like rhythmic analysis in the HumdrumFileStructure
//    class, but this algorithm does not terminate secondary traversals when
//    recursing.  Perhaps that should be fixed (utilizing the "rhycheck"
//    variable in HumdrumTokens)
//

void HumdrumFileBase::addUniqueTokens(vector<HTp>& target,
		vector<HTp>& source) {
	int i, j;
	bool found;
	for (i=0; i<(int)source.size(); i++) {
		found = false;
		for (j=0; j<(int)target.size(); j++) {
			if (source[i] == target[i]) {
				found = true;
			}
		}
		if (!found) {
			target.push_back(source[i]);
		}
	}
}



//////////////////////////////
//
// HumdrumFileBase::adjustMergeSpineLines -- fix *v lines to that adjacent
//     tracks do not merge at the same time.  In other words, split the line
//     into two or more merge lines.
//

/* still to be implemented

void HumdrumFileBase::adjustMergeSpineLines(void) {
	HumdrumFileBase& infile = *this;
	// going backwards to not have to deal with line number updates
	// at the moment...
	for (int i=infile.getLineCount()-1; i>= 0; i--) {
		if (!infile[i].isManipulator()) {
			continue;
		}
		bool hasbadmerge = false;
		int track1;
		int track2;
		for (int j=1; j<infile[i].getFieldCount(); j++) {
			if (!infile[i].token(j)->equalTo("*v")) {
				continue;
			}
			if (!infile[i].token(j-1)->equalTo("*v")) {
				continue;
			}
			track1 = infile.token(i, j-1)->getTrack();
			track2 = infile.token(i, j)->getTrack();
			if (track1 != track2) {
				hasbadmerge = true;
				break;
			}
		}
		if (hasbadmerge) {
			cerr << "!! BADMERGE on line " << i + 1 << endl;
			fixMerges(i);
		}
	}
}

*/



//////////////////////////////
//
// HumdrumFileBase::fixMerges -- Split a line with merges into two
//    lines.  The line is presumed to have a bad merge which
//    means that two adjacent tracks have adjacent *v tokens.
//    This algorithm will create a new lines where everything
//    after the bad merge is placed on the newline.   Example:
//
// track:    1    2    2    3    3    4    5    5
//           *    *v   *v   *v   *v   *    *v   *v
//
// This is invalid because track 2 and track 3 have adjacent *v tokens.
// This function will create a new line and move everything after
// the bad position to a new line:
//
// track:    1    2    2    3    3    4    5    5
//           *    *v   *v   *v   *v   *    *v   *v
//           *    *    *v   *v   *    *    *
// track:    1    2    3    3    4    5    5
//
// This algorithm only fixes one bad boundary.  The calling function
// will presumably fix any bad boundaries on the newly created line.
//

/* Still to be implemented...
void HumdrumFileBase::fixMerges(int linei) {
	HumdrumFileBase& infile = *this;

	vector<vector<HTp> > linetoks;
	HTp tok;

	// linetoks: collect tokens on the current line by track groups.
	int track1 = -1;
	int track2 = -1;
	for (int j=0; j<infile[linei].getFieldCount(); j++) {
		tok = infile[linei].token(j);
		track2 = tok->getTrack();
		if (track2 != track1) {
			linetoks.resize(linetoks.size()+1);
			linetoks.back().push_back(tok);
		}
		track1 = track2;
	}

	// ptoks: collect the tokens on the previous line for stiching tokens
	// together after adding new line.
	vector<vector<HTp> > ptoks;
	track1 = -1;
	track2 = -1;
	for (int j=0; j<infile[linei-1].getFieldCount(); j++) {
		tok = infile[linei-1].token(j);
		track2 = tok->getTrack();
		if (track2 != track1) {
			ptoks.resize(ptoks.size()+1);
			ptoks.back().push_back(tok);
		}
		track1 = track2;
	}

	// ntoks: collect the tokens on the next line for stiching tokens
	// together after adding new line.
	vector<vector<HTp> > ntoks;
	track1 = -1;
	track2 = -1;
	for (int j=0; j<infile[linei+1].getFieldCount(); j++) {
		tok = infile[linei+1].token(j);
		track2 = tok->getTrack();
		if (track2 != track1) {
			ntoks.resize(ntoks.size()+1);
			ntoks.back().push_back(tok);
		}
		track1 = track2;
	}

	int maxt = infile.getMaxTrack();
	vector<vector<HTp> > newtokbytrack(maxt+1);

// track:    1    2    2    3    3    4    5    5
//           *    *v   *v   *v   *v   *    *v   *v
//
// o = new null tokens.
//
// original     *    *v   *v   o    o    o    o    o
// new          o    o         *v   *v   *    *v   *v
// track:       1    2         3    3    4    5    5

	HLp newline = new HumdrumLine;
	newline->setOwner(this);
	bool foundboundary = false;
	HTp token;
	int findex;
	// int swaptrack = -1;
	int difference = 0;  // decrease in token count on new line
	for (int i=0; i<linetoks.size()-1; i++) {
		if (foundboundary) {
			// transfer the track tokens to the new line, and put
			// new null tokens in their place on the old line.
			for (int j=0; j<(int)linetoks[i].size(); j++) {
				track1 = linetoks[i][j]->getTrack();
				findex = linetoks[i][j]->getFieldIndex();
            // move the token to the next line:
				newline->m_tokens.push_back(linetoks[i][j]);
				// put it in the list for later processing:
				newtokbytrack[track1].push_back(linetoks[i][j]);
				// replace the moved token with a null token:
				token = new HumdrumToken("*");
				infile[linei].m_tokens[findex] = token;
				// probably need to update the HumAddress of both tokens.
			}
		} else if ((!foundboundary) && linetoks[i].back()->equalTo("*v") &&
				linetoks[i+1][0]->equalTo("*v")) {
			// This is the bad boundary.  Keep track fields in the
			// original line, and create one new null token in
			// the newline.
// original     *    *v   *v   o    o    o    o    o
// new          o    o         *v   *v   *    *v   *v
// track:       1    2         3    3    4    5    5
			difference = linetoks[i].size() - 1;

			track1 = linetoks[i][0]->getTrack();
			token = new HumdrumToken("*");
			token->setTrack(track1);
			token->setSubtrack(track1);
			newline->m_tokens.push_back(token);
			// put new token in list for later processing:
			newtokbytrack[track1].push_back(token);

			foundboundary = true;
		} else {
			// add null tokens to the new line, and keep the
			// tokens on the original line as they were
			for (int j=0; j<(int)linetoks[i].size(); j++) {
				track1 = linetoks[i][j]->getTrack();
				token = new HumdrumToken("*");
				token->setTrack(track1);
				token->setSubtrack(track1);
				newline->m_tokens.push_back(token);
				// put new token in list for later processing:
				newtokbytrack[track1].push_back(token);
			}
		}
	}

	// for now the links between the tokens on successive lines
	// will not be updated.  For the most part it will not be
	// important.  Probably more important is to update line numbers
	// for HumdrumLines occurring on new lines.  Maybe need to set
	// the line type for the new line.

	// add the new line to the file:
	m_lines.insert(m_lines.begin() + linei + 1, newline);

}

*/


//////////////////////////////
//
// operator<< -- Default method of printing HumdrumFiles.  This printing method
//    assumes that the HumdrumLine string is correct.  If a token is changed
//    in the file, the HumdrumFileBase::createLinesFromTokens() before printing
//    the contents of the line.
//

ostream& operator<<(ostream& out, HumdrumFileBase& infile) {
	for (int i=0; i<infile.getLineCount(); i++) {
		out << infile[i] << '\n';
	}
	return out;
}


//////////////////////////////
//
// sortTokenParisByLineIndex -- Sort two tokens so that the one
//    on the smaller line is first.  If both are on the same line then
//    sort the left-most token first.
//

bool sortTokenPairsByLineIndex(const TokenPair& a, const TokenPair& b) {
	if (a.first->getLineIndex() < b.first->getLineIndex()) {
		return true;
	}
	if (a.first->getLineIndex() == b.first->getLineIndex()) {
		if (a.first->getFieldIndex() < b.first->getFieldIndex()) {
			return true;
		}
	}
	return false;
}



//////////////////////////////
//
// HumdrumFileBase::makeBooleanTrackList --
//

void HumdrumFileBase::makeBooleanTrackList(vector<bool>& spinelist,
		const string& spinestring) {
	Convert::makeBooleanTrackList(spinelist, spinestring, getMaxTrack());
}



//////////////////////////////
//
// HumdrumFileBase::getMeasureNumbers -- Return a list of measures numbers
//    in the file indexed by line. A value of -1 means no measure number.
//

vector<int> HumdrumFileBase::getMeasureNumbers(void) {
	HumdrumFileBase& infile = *this;
	vector<int> output(infile.getLineCount(), -1);
	int lastmeasure = -1;
	int current;
	for (int i=0; i<infile.getLineCount(); i++) {
		if (infile[i].isBarline()) {
			current = infile.getMeasureNumber(i);
			if (current >= 0) {
				lastmeasure = current;
			}
		}
		output[i] = lastmeasure;
	}
	return output;
}



//////////////////////////////
//
// HumdrumFileBase::getMeasureNumber -- If the current line is a
//      barline, then read the first integer found in the fields on the line.
//

int HumdrumFileBase::getMeasureNumber(int line) {
   HumdrumFileBase& infile = *this;
   int j;
   if (!infile[line].isBarline()) {
      // Return -1 if not a barline.  May be changed in the future
      // to return the measure number of the previous barline.
      return -1;
   }
   HumRegex hre;
   int measurenumber = -1;
   for (j=0; j<infile[line].getFieldCount(); j++) {
      if (hre.search(*infile.token(line, j), "^=[^\\d]*(\\d+)")) {
         measurenumber = hre.getMatchInt(1);
         return measurenumber;
      }
   }
	return -1;
}



//////////////////////////////
//
// HumdrumFileBase::initializeArray -- adjust the size of the input array
//     to the same dimensions as the HumdrumFile, filling in each cell of the
//     array with the given value as a default.
//

template <class TYPE>
void HumdrumFileBase::initializeArray(vector<vector<TYPE>>& array, TYPE value) {
	HumdrumFileBase& infile = *this;
	array.clear();
	array.resize(infile.getLineCount());
	for (int i=0; i<infile.getLineCount(); i++) {
		array[i].resize(infile[i].getFieldCount());
		fill(array[i].begin(), array[i].end(), value);
	}
}



//////////////////////////////
//
// HumdrumFileBase::getReferenceRecord --
//

std::string HumdrumFileBase::getReferenceRecord(const std::string& key) {
	HumdrumFileBase& infile = *this;
	for (int i=0; i<infile.getLineCount(); i++) {
		if (!infile[i].isReference()) {
			continue;
		}
		string refkey = infile[i].getReferenceKey();
		if (refkey == key) {
			string value = infile[i].getReferenceValue();
			return value;
		}
	}
	return "";
}



//////////////////////////////
//
// HumdrumFileBase::insertNullDataLine -- Add a null data line at
//     the given absolute quarter-note timestamp in the file.  If there
//     is already a data line at the given timestamp, then do not create
//     a line and instead return a pointer to the existing line.  Returns
//     NULL if there was a problem.
//

HLp HumdrumFileBase::insertNullDataLine(HumNum timestamp) {
	// for now do a linear search for the insertion point, but later
	// do something more efficient.
	HumdrumFileBase& infile = *this;
	HumNum beforet(-1);
	HumNum aftert(-1);
	int beforei = -1;
	// int afteri = -1;
	HumNum current;
	for (int i=0; i<infile.getLineCount(); i++) {
		if (!infile[i].isData()) {
			continue;
		}
		current = infile[i].getDurationFromStart();
		if (current == timestamp) {
			return &infile[i];
		} else if (current < timestamp) {
			beforet = current;
			beforei = i;
		} else if (current > timestamp) {
			aftert = current;
			// afteri = i;
			break;
		}
	}

	if (beforei < 0) {
		return NULL;
	}
	HLp newline = new HumdrumLine;
	// copyStructure will add null tokens automatically
	newline->copyStructure(&infile[beforei], ".");

	infile.insertLine(beforei+1, newline);

	// Set the timestamp information for inserted line:
	HumNum delta = timestamp - beforet;
	HumNum durationFromStart = infile[beforei].getDurationFromStart() + delta;
	HumNum durationFromBarline = infile[beforei].getDurationFromBarline() + delta;
	HumNum durationToBarline = infile[beforei].getDurationToBarline() - delta;

	newline->m_durationFromStart = durationFromStart;
	newline->m_durationFromBarline = durationFromBarline;
	newline->m_durationToBarline = durationToBarline;

	newline->m_duration = infile[beforei].m_duration - delta;
	infile[beforei].m_duration = delta;

	for (int i=0; i<infile[beforei].getFieldCount(); i++) {
		HTp token = infile.token(beforei, i);
		HTp newtoken = newline->token(i);
		token->insertTokenAfter(newtoken);
	}

	return newline;
}



//////////////////////////////
//
// HumdrumFileBase::insertNullInterpretationLine -- Add a null interpretation
//     line at the given absolute quarter-note timestamp in the file.  The line will
//     be added after any other interpretation lines at that timestamp, but before any
//     local comments that appear immediately before the data line(s) at that timestamp.
//     Returns NULL if there was a problem.
//

HLp HumdrumFileBase::insertNullInterpretationLine(HumNum timestamp) {
	// for now do a linear search for the insertion point, but later
	// do something more efficient.
	HumdrumFileBase& infile = *this;
	HumNum beforet(-1);
	HumNum aftert(-1);
	int beforei = -1;
	// int afteri = -1;
	HumNum current;
	for (int i=0; i<infile.getLineCount(); i++) {
		if (!infile[i].isData()) {
			continue;
		}
		current = infile[i].getDurationFromStart();
		if (current == timestamp) {
			beforei = i;
			break;
		} else if (current < timestamp) {
			beforet = current;
			beforei = i;
		} else if (current > timestamp) {
			aftert = current;
			// afteri = i;
			break;
		}
	}

	if (beforei < 0) {
		return NULL;
	}

	HLp target = getLineForInterpretationInsertion(beforei);

	HLp newline = new HumdrumLine;
	// copyStructure will add null tokens automatically
	newline->copyStructure(target, "*");

	int targeti = target->getLineIndex();

	// There will be problems with linking to previous line if it is
	// a manipulator.
	// infile.insertLine(targeti-1, newline);
	infile.insertLine(targeti, newline);

	// inserted line will increment beforei by one:
	beforei++;

	// Set the timestamp information for inserted line:
	HumNum durationFromStart = infile[beforei].getDurationFromStart();
	HumNum durationFromBarline = infile[beforei].getDurationFromBarline();
	HumNum durationToBarline = infile[beforei].getDurationToBarline();

	newline->m_durationFromStart = durationFromStart;
	newline->m_durationFromBarline = durationFromBarline;
	newline->m_durationToBarline = durationToBarline;

	newline->m_duration = 0;

	// Problems here if targeti line is a manipulator.
	for (int i=0; i<infile[targeti].getFieldCount(); i++) {
		HTp token = infile.token(targeti, i);
		HTp newtoken = newline->token(i);
		token->insertTokenAfter(newtoken);
	}

	return newline;
}



//////////////////////////////
//
// HumdrumFileBase::insertNullInterpretationLineAbove -- Add a null interpretation
//     line at the given absolute quarter-note timestamp in the file.  The line will
//     be added before any other lines at that timestamp.
//     Returns NULL if there was a problem.
//

HLp HumdrumFileBase::insertNullInterpretationLineAbove(HumNum timestamp) {
	// for now do a linear search for the insertion point, but later
	// do something more efficient.
	HumdrumFileBase& infile = *this;
	HumNum beforet(-1);
	HumNum aftert(-1);
	int beforei = -1;
	// int afteri = -1;
	HumNum current;
	for (int i=0; i<infile.getLineCount(); i++) {
		current = infile[i].getDurationFromStart();
		if (current == timestamp) {
			beforei = i;
			break;
		} else if (current < timestamp) {
			beforet = current;
			beforei = i;
		} else if (current > timestamp) {
			aftert = current;
			// afteri = i;
			break;
		}
	}

	if (beforei < 0) {
		return NULL;
	}

	HLp target = getLineForInterpretationInsertionAbove(beforei);

	HLp newline = new HumdrumLine;
	// copyStructure will add null tokens automatically
	newline->copyStructure(target, "*");

	int targeti = target->getLineIndex();

	// There will be problems with linking to previous line if it is
	// a manipulator.
	// infile.insertLine(targeti-1, newline);
	infile.insertLine(targeti, newline);

	// inserted line will increment beforei by one:
	beforei++;

	// Set the timestamp information for inserted line:
	HumNum durationFromStart = infile[beforei].getDurationFromStart();
	HumNum durationFromBarline = infile[beforei].getDurationFromBarline();
	HumNum durationToBarline = infile[beforei].getDurationToBarline();

	newline->m_durationFromStart = durationFromStart;
	newline->m_durationFromBarline = durationFromBarline;
	newline->m_durationToBarline = durationToBarline;

	newline->m_duration = 0;

	// Problems here if targeti line is a manipulator.
	for (int i=0; i<infile[targeti].getFieldCount(); i++) {
		HTp token = infile.token(targeti, i);
		HTp newtoken = newline->token(i);
		token->insertTokenAfter(newtoken);
	}

	return newline;
}



//////////////////////////////
//
// HumdrumFileBase::getLineForInterpretationInsertion --  Search backwards
//    in the file for the first local comment immediately before a data line
//    index given as input.  If there are no local comments, then return the
//    data line.  If there are local comment lines immediately before the data
//    line, then keep searching for the first local comment.  Non-spined lines
//    (global or empty lines) are ignored.  This function is used to insert
//    an empty interpretation before a data line at a specific data line.
//

HLp HumdrumFileBase::getLineForInterpretationInsertion(int index) {
	HumdrumFileBase& infile = *this;
	int current = index - 1;
	int previous = index;
	while (current > 0) {
		if (!infile[current].hasSpines()) {
			current--;
			continue;
		}
		if (infile[current].isCommentLocal()) {
			previous = current;
			current--;
			continue;
		}
		return &infile[previous];
	}
	return &infile[index];
}



//////////////////////////////
//
// HumdrumFileBase::getLineForInterpretationInsertionAbove --  Search backwards
//    in the file for the first line at the same timestamp as the starting line.
//

HLp HumdrumFileBase::getLineForInterpretationInsertionAbove(int index) {
	HumdrumFileBase& infile = *this;
	HumNum timestamp = infile[index].getDurationFromStart();
	HumNum teststamp;
	int current = index - 1;
	int previous = index;
	while (current > 0) {
		if (!infile[current].hasSpines()) {
			current--;
			continue;
		}
		teststamp = infile[current].getDurationFromStart();
		if (teststamp == timestamp) {
			previous = current;
			current--;
			continue;
		}
		return &infile[previous];
	}
	return &infile[index];
}






//////////////////////////////
//
// HumdrumFileContent::analyzeKernAccidentals -- Identify accidentals that
//    should be printed (only in **kern spines) as well as cautionary
//    accidentals (accidentals which are forced to be displayed but otherwise
//    would not be printed.  Algorithm assumes that all secondary tied notes
//    will not display their accidental across a system break.  Consideration
//    about grace-note accidental display still needs to be done.
//

bool HumdrumFileContent::analyzeKernAccidentals(void) {

	// ottava marks must be analyzed first:
	this->analyzeOttavas();

	HumdrumFileContent& infile = *this;
	int i, j, k;
	int kindex;
	int track;

	// ktracks == List of **kern spines in data.
	// rtracks == Reverse mapping from track to ktrack index (part/staff index).
	vector<HTp> ktracks = getKernSpineStartList();
	vector<int> rtracks(getMaxTrack()+1, -1);
	for (i=0; i<(int)ktracks.size(); i++) {
		track = ktracks[i]->getTrack();
		rtracks[track] = i;
	}
	int kcount = (int)ktracks.size();

	// keysigs == key signature spellings of diatonic pitch classes.  This array
	// is duplicated into dstates after each barline.
	vector<vector<int> > keysigs;
	keysigs.resize(kcount);
	for (i=0; i<kcount; i++) {
		keysigs[i].resize(7);
		std::fill(keysigs[i].begin(), keysigs[i].end(), 0);
	}

	// dstates == diatonic states for every pitch in a spine.
	// sub-spines are considered as a single unit, although there are
	// score conventions which would keep a separate voices on a staff
	// with different accidental states (i.e., two parts superimposed
	// on the same staff, but treated as if on separate staves).
	// Eventually this algorithm should be adjusted for dealing with
	// cross-staff notes, where the cross-staff notes should be following
	// the accidentals of a different spine...
	vector<vector<int> > dstates; // diatonic states
	dstates.resize(kcount);
	for (i=0; i<kcount; i++) {
		dstates[i].resize(70);     // 10 octave limit for analysis
			                        // may cause problems; fix later.
		std::fill(dstates[i].begin(), dstates[i].end(), 0);
	}

	// gdstates == grace note diatonic states for every pitch in a spine.
	vector<vector<int> > gdstates; // grace-note diatonic states
	gdstates.resize(kcount);
	for (i=0; i<kcount; i++) {
		gdstates[i].resize(70);
		std::fill(gdstates[i].begin(), gdstates[i].end(), 0);
	}

	// rhythmstart == keep track of first beat in measure.
	vector<int> firstinbar(kcount, 0);

	int lasttrack = -1;
	vector<int> concurrentstate(70, 0);

	for (i=0; i<infile.getLineCount(); i++) {
		if (!infile[i].hasSpines()) {
			continue;
		}
		if (infile[i].isInterpretation()) {
			for (j=0; j<infile[i].getFieldCount(); j++) {
				if (!infile[i].token(j)->isKern()) {
					continue;
				}
				if (infile[i].token(j)->compare(0, 3, "*k[") == 0) {
					track = infile[i].token(j)->getTrack();
					kindex = rtracks[track];
					fillKeySignature(keysigs[kindex], *infile[i].token(j));
					// resetting key states of current measure.  What to do if this
					// key signature is in the middle of a measure?
					resetDiatonicStatesWithKeySignature(dstates[kindex],
							keysigs[kindex]);
					resetDiatonicStatesWithKeySignature(gdstates[kindex],
							keysigs[kindex]);
				}
			}
		} else if (infile[i].isBarline()) {
			for (j=0; j<infile[i].getFieldCount(); j++) {
				if (!infile[i].token(j)->isKern()) {
					continue;
				}
				if (infile[i].token(j)->isInvisible()) {
					continue;
				}
				std::fill(firstinbar.begin(), firstinbar.end(), 1);
				track = infile[i].token(j)->getTrack();
				kindex = rtracks[track];
				// reset the accidental states in dstates to match keysigs.
				resetDiatonicStatesWithKeySignature(dstates[kindex],
						keysigs[kindex]);
				resetDiatonicStatesWithKeySignature(gdstates[kindex],
						keysigs[kindex]);
			}
		}

		if (!infile[i].isData()) {
			continue;
		}

		fill(concurrentstate.begin(), concurrentstate.end(), 0);
		lasttrack = -1;

		for (j=0; j<infile[i].getFieldCount(); j++) {
			if (!infile[i].token(j)->isKern()) {
				continue;
			}
			if (infile[i].token(j)->isNull()) {
				continue;
			}
			if (infile[i].token(j)->isRest()) {
				continue;
			}

			int subcount = infile[i].token(j)->getSubtokenCount();
			track = infile[i].token(j)->getTrack();

			if (lasttrack != track) {
				fill(concurrentstate.begin(), concurrentstate.end(), 0);
			}
			lasttrack = track;

			int rindex = rtracks[track];
			for (k=0; k<subcount; k++) {
				HTp token = infile[i].token(j);
				string subtok = token->getSubtoken(k);
				int b40 = Convert::kernToBase40(subtok);
				int diatonic = Convert::kernToBase7(subtok);
				int octaveadjust = token->getValueInt("auto", "ottava");
				diatonic -= octaveadjust * 7;
				if (diatonic < 0) {
					// Deal with extra-low notes later.
					continue;
				}
				int graceQ = infile[i].token(j)->isGrace();
				int accid = Convert::kernToAccidentalCount(subtok);
				int hiddenQ = 0;
				if (subtok.find("yy") == string::npos) {
					if ((subtok.find("ny") != string::npos) ||
					    (subtok.find("#y") != string::npos) ||
					    (subtok.find("-y") != string::npos)) {
						hiddenQ = 1;
					}
				}

				if (((subtok.find("_") != string::npos) ||
						(subtok.find("]") != string::npos))) {
					// tied notes do not have slurs, so skip them
					if ((accid != keysigs[rindex][diatonic % 7]) &&
							firstinbar[rindex]) {
						// But first, prepare to force an accidental to be shown on
						// the note immediately following the end of a tied group
						// if the tied group crosses a barline.
						dstates[rindex][diatonic] = -1000 + accid;
						gdstates[rindex][diatonic] = -1000 + accid;
					}
					auto loc = subtok.find('X');
					if (loc == string::npos) {
						continue;
					} else if (loc == 0) {
						continue;
					} else {
						if (!((subtok[loc-1] == '#') || (subtok[loc-1] == '-') ||
								(subtok[loc-1] == 'n'))) {
							continue;
						} else {
							// an accidental should be fored at end of tie
						}
					}
				}

				size_t loc;
				// check for accidentals on trills, mordents and turns.
				if (subtok.find("t") != string::npos) {
					// minor second trill
					int trillnote     = b40 + 5;
					int trilldiatonic = Convert::base40ToDiatonic(trillnote);
					int trillaccid    = Convert::base40ToAccidental(trillnote);
					if (dstates[rindex][trilldiatonic] != trillaccid) {
						infile[i].token(j)->setValue("auto", to_string(k),
								"trillAccidental", to_string(trillaccid));
						dstates[rindex][trilldiatonic] = -1000 + trillaccid;
					}
				} else if (subtok.find("T") != string::npos) {
					// major second trill
					int trillnote     = b40 + 6;
					int trilldiatonic = Convert::base40ToDiatonic(trillnote);
					int trillaccid    = Convert::base40ToAccidental(trillnote);
					if (dstates[rindex][trilldiatonic] != trillaccid) {
						infile[i].token(j)->setValue("auto", to_string(k),
								"trillAccidental", to_string(trillaccid));
						dstates[rindex][trilldiatonic] = -1000 + trillaccid;
					}
				} else if (subtok.find("M") != string::npos) {
					// major second upper mordent
					int auxnote     = b40 + 6;
					int auxdiatonic = Convert::base40ToDiatonic(auxnote);
					int auxaccid    = Convert::base40ToAccidental(auxnote);
					if (dstates[rindex][auxdiatonic] != auxaccid) {
						infile[i].token(j)->setValue("auto", to_string(k),
								"mordentUpperAccidental", to_string(auxaccid));
						dstates[rindex][auxdiatonic] = -1000 + auxaccid;
					}
				} else if (subtok.find("m") != string::npos) {
					// minor second upper mordent
					int auxnote     = b40 + 5;
					int auxdiatonic = Convert::base40ToDiatonic(auxnote);
					int auxaccid    = Convert::base40ToAccidental(auxnote);
					if (dstates[rindex][auxdiatonic] != auxaccid) {
						infile[i].token(j)->setValue("auto", to_string(k),
								"mordentUpperAccidental", to_string(auxaccid));
						dstates[rindex][auxdiatonic] = -1000 + auxaccid;
					}
				} else if (subtok.find("W") != string::npos) {
					// major second upper mordent
					int auxnote     = b40 - 6;
					int auxdiatonic = Convert::base40ToDiatonic(auxnote);
					int auxaccid    = Convert::base40ToAccidental(auxnote);
					if (dstates[rindex][auxdiatonic] != auxaccid) {
						infile[i].token(j)->setValue("auto", to_string(k),
								"mordentLowerAccidental", to_string(auxaccid));
						dstates[rindex][auxdiatonic] = -1000 + auxaccid;
					}
				} else if (subtok.find("w") != string::npos) {
					// minor second upper mordent
					int auxnote     = b40 - 5;
					int auxdiatonic = Convert::base40ToDiatonic(auxnote);
					int auxaccid    = Convert::base40ToAccidental(auxnote);
					if (dstates[rindex][auxdiatonic] != auxaccid) {
						infile[i].token(j)->setValue("auto", to_string(k),
								"mordentLowerAccidental", to_string(auxaccid));
						dstates[rindex][auxdiatonic] = -1000 + auxaccid;
					}

				} else if ((loc = subtok.find("$")) != string::npos) {

					int turndiatonic = Convert::base40ToDiatonic(b40);
					// int turnaccid = Convert::base40ToAccidental(b40);
					// inverted turn
					int lowerint = 0;
					int upperint = 0;
					if (loc < subtok.size()-1) {
						if (subtok[loc+1] == 's') {
							lowerint = -5;
						} else if (subtok[loc+1] == 'S') {
							lowerint = -6;
						}
					}
					if (loc < subtok.size()-2) {
						if (subtok[loc+2] == 's') {
							upperint = +5;
						} else if (subtok[loc+2] == 'S') {
							upperint = +6;
						}
					}
					int lowerdiatonic = turndiatonic - 1;
					// Maybe also need to check for forced accidental state...
					int loweraccid = dstates[rindex][lowerdiatonic];
					int lowerb40 = Convert::base7ToBase40(lowerdiatonic) + loweraccid;
					int upperdiatonic = turndiatonic + 1;
					// Maybe also need to check for forced accidental state...
					int upperaccid = dstates[rindex][upperdiatonic];
					int upperb40 = Convert::base7ToBase40(upperdiatonic) + upperaccid;
					if (lowerint == 0) {
						// need to calculate lower interval (but it will not appear
						// below the inverted turn, just calculating for performance
						// rendering.
						lowerint = lowerb40 - b40;
						lowerb40 = b40 + lowerint;
					}
					if (upperint == 0) {
						// need to calculate upper interval (but it will not appear
						// above the inverted turn, just calculating for performance
						// rendering.
						upperint = upperb40 - b40;
						upperb40 = b40 + upperint;
					}
					int uacc = Convert::base40ToAccidental(b40 + upperint);
					int bacc = Convert::base40ToAccidental(b40 + lowerint);
					if (uacc != upperaccid) {
						infile[i].token(j)->setValue("auto", to_string(k),
								"turnUpperAccidental", to_string(uacc));
						dstates[rindex][upperdiatonic] = -1000 + uacc;
					}
					if (bacc != loweraccid) {
						infile[i].token(j)->setValue("auto", to_string(k),
								"turnLowerAccidental", to_string(bacc));
						dstates[rindex][lowerdiatonic] = -1000 + bacc;
					}

				} else if ((loc = subtok.find("S")) != string::npos) {

					int turndiatonic = Convert::base40ToDiatonic(b40);
					// int turnaccid = Convert::base40ToAccidental(b40);
					// regular turn
					int lowerint = 0;
					int upperint = 0;
					if (loc < subtok.size()-1) {
						if (subtok[loc+1] == 's') {
							upperint = +5;
						} else if (subtok[loc+1] == 'S') {
							upperint = +6;
						}
					}
					if (loc < subtok.size()-2) {
						if (subtok[loc+2] == 's') {
							lowerint = -5;
						} else if (subtok[loc+2] == 'S') {
							lowerint = -6;
						}
					}
					int lowerdiatonic = turndiatonic - 1;
					// Maybe also need to check for forced accidental state...
					int loweraccid = dstates[rindex][lowerdiatonic];
					int lowerb40 = Convert::base7ToBase40(lowerdiatonic) + loweraccid;
					int upperdiatonic = turndiatonic + 1;
					// Maybe also need to check for forced accidental state...
					int upperaccid = dstates[rindex][upperdiatonic];
					int upperb40 = Convert::base7ToBase40(upperdiatonic) + upperaccid;
					if (lowerint == 0) {
						// need to calculate lower interval (but it will not appear
						// below the inverted turn, just calculating for performance
						// rendering.
						lowerint = lowerb40 - b40;
						lowerb40 = b40 + lowerint;
					}
					if (upperint == 0) {
						// need to calculate upper interval (but it will not appear
						// above the inverted turn, just calculating for performance
						// rendering.
						upperint = upperb40 - b40;
						upperb40 = b40 + upperint;
					}
					int uacc = Convert::base40ToAccidental(b40 + upperint);
					int bacc = Convert::base40ToAccidental(b40 + lowerint);
					if (uacc != upperaccid) {
						infile[i].token(j)->setValue("auto", to_string(k),
								"turnUpperAccidental", to_string(uacc));
						dstates[rindex][upperdiatonic] = -1000 + uacc;
					}
					if (bacc != loweraccid) {
						infile[i].token(j)->setValue("auto", to_string(k),
								"turnLowerAccidental", to_string(bacc));
						dstates[rindex][lowerdiatonic] = -1000 + bacc;
					}
				}

				if (graceQ && (accid != gdstates[rindex][diatonic])) {
					// accidental is different from the previous state so should be
					// printed
					if (!hiddenQ) {
						infile[i].token(j)->setValue("auto", to_string(k),
								"visualAccidental", "true");
						if (gdstates[rindex][diatonic] < -900) {
							// this is an obligatory cautionary accidental
							// or at least half the time it is (figure that out later)
							infile[i].token(j)->setValue("auto", to_string(k),
									"obligatoryAccidental", "true");
							infile[i].token(j)->setValue("auto", to_string(k),
									"cautionaryAccidental", "true");
						}
					}
					gdstates[rindex][diatonic] = accid;
					// regular notes are not affected by grace notes accidental
					// changes, but should have an obligatory cautionary accidental,
					// displayed for clarification.
					dstates[rindex][diatonic] = -1000 + accid;

				} else if (!graceQ && ((concurrentstate[diatonic] && (concurrentstate[diatonic] == accid))
						|| (accid != dstates[rindex][diatonic]))) {
					// accidental is different from the previous state so should be
					// printed, but only print if not supposed to be hidden.
					if (!hiddenQ) {
						infile[i].token(j)->setValue("auto", to_string(k),
								"visualAccidental", "true");
						concurrentstate[diatonic] = accid;
						if (dstates[rindex][diatonic] < -900) {
							// this is an obligatory cautionary accidental
							// or at least half the time it is (figure that out later)
							infile[i].token(j)->setValue("auto", to_string(k),
									"obligatoryAccidental", "true");
							infile[i].token(j)->setValue("auto", to_string(k),
									"cautionaryAccidental", "true");
						}
					}
					dstates[rindex][diatonic] = accid;
					gdstates[rindex][diatonic] = accid;

				} else if ((accid == 0) && (subtok.find("n") != string::npos) &&
							!hiddenQ) {
					infile[i].token(j)->setValue("auto", to_string(k),
							"cautionaryAccidental", "true");
					infile[i].token(j)->setValue("auto", to_string(k),
							"visualAccidental", "true");
				} else if (subtok.find("XX") == string::npos) {
					// The accidental is not necessary. See if there is a single "X"
					// immediately after the accidental which means to force it to
					// display.
					auto loc = subtok.find("X");
					if ((loc != string::npos) && (loc > 0)) {
						if (subtok[loc-1] == '#') {
							infile[i].token(j)->setValue("auto", to_string(k),
									"cautionaryAccidental", "true");
									infile[i].token(j)->setValue("auto", to_string(k),
											"visualAccidental", "true");
						} else if (subtok[loc-1] == '-') {
							infile[i].token(j)->setValue("auto", to_string(k),
									"cautionaryAccidental", "true");
									infile[i].token(j)->setValue("auto", to_string(k),
											"visualAccidental", "true");
						} else if (subtok[loc-1] == 'n') {
							infile[i].token(j)->setValue("auto", to_string(k),
									"cautionaryAccidental", "true");
							infile[i].token(j)->setValue("auto", to_string(k),
									"visualAccidental", "true");
						}
					}
				}
			}
		}
		std::fill(firstinbar.begin(), firstinbar.end(), 0);
	}

	// Indicate that the accidental analysis has been done:
	infile.setValue("auto", "accidentalAnalysis", "true");

	return true;
}



//////////////////////////////
//
// HumdrumFileContent::fillKeySignature -- Read key signature notes and
//    assign +1 to sharps, -1 to flats in the diatonic input array.  Used
//    only by HumdrumFileContent::analyzeKernAccidentals().
//

void HumdrumFileContent::fillKeySignature(vector<int>& states,
		const string& keysig) {
	std::fill(states.begin(), states.end(), 0);
	if (keysig.find("f#") != string::npos) { states[3] = +1; }
	if (keysig.find("c#") != string::npos) { states[0] = +1; }
	if (keysig.find("g#") != string::npos) { states[4] = +1; }
	if (keysig.find("d#") != string::npos) { states[1] = +1; }
	if (keysig.find("a#") != string::npos) { states[5] = +1; }
	if (keysig.find("e#") != string::npos) { states[2] = +1; }
	if (keysig.find("b#") != string::npos) { states[6] = +1; }
	if (keysig.find("b-") != string::npos) { states[6] = -1; }
	if (keysig.find("e-") != string::npos) { states[2] = -1; }
	if (keysig.find("a-") != string::npos) { states[5] = -1; }
	if (keysig.find("d-") != string::npos) { states[1] = -1; }
	if (keysig.find("g-") != string::npos) { states[4] = -1; }
	if (keysig.find("c-") != string::npos) { states[0] = -1; }
	if (keysig.find("f-") != string::npos) { states[3] = -1; }
}



//////////////////////////////
//
// HumdrumFileContent::resetDiatonicStatesWithKeySignature -- Only used in
//     HumdrumFileContent::analyzeKernAccidentals().  Resets the accidental
//     states for notes
//

void HumdrumFileContent::resetDiatonicStatesWithKeySignature(vector<int>&
		states, vector<int>& signature) {
	for (int i=0; i<(int)states.size(); i++) {
		states[i] = signature[i % 7];
	}
}





//////////////////////////////
//
// HumdrumFileContent::analyzeBarlines -- 
//

void HumdrumFileContent::analyzeBarlines(void) {
	if (m_analyses.m_barlines_analyzed) {
		// Maybe allow forcing reanalysis.
		return;
	}
	m_analyses.m_slurs_analyzed = true;
	m_analyses.m_barlines_different = false;

	string baseline;
	string comparison;
	bool baseQ;

	HumdrumFileContent& infile = *this;
	for (int i=0; i<infile.getLineCount(); i++) {
		if (!infile[i].isBarline()) {
			continue;
		}
		bool allSame = true;
		int fieldcount = infile[i].getFieldCount();
		if (fieldcount <= 1) {
			continue;
		}
		baseQ = false;
		baseline = "";
		comparison = "";
		for (int j=0; j<infile[i].getFieldCount(); j++) {
			HTp token = infile[i].token(j);
			int subtrack = token->getSubtrack();
			if (subtrack > 1) {
				// ignore secondary barlines in subspines.
				continue;
			}
			if (!token->isStaff()) {
				// don't check non-staff barlines
				continue;
			}
			if (!baseQ) {
				baseline = "";
				for (int k=0; k<(int)token->size(); k++) {
					if (isdigit(token->at(k))) {
						// ignore barnumbers
						// maybe ignore fermatas
						continue;
					}
					baseline += token->at(k);
				}
				baseQ = true;
			} else {
				comparison = "";
				for (int k=0; k<(int)token->size(); k++) {
					if (isdigit(token->at(k))) {
						// ignore barnumbers;
						// maybe ignore fermatas
						continue;
					}
					comparison += token->at(k);
				}
				if (comparison != baseline) {
					allSame = false;
					break;
				}
			}
		}

		if (!allSame) {
			infile[i].setValue("auto", "barlinesDifferent", 1);
			m_analyses.m_barlines_different = true;
		}
	}
}



//////////////////////////////
//
// HumdrumFileContent::hasDifferentBarlines --
//

bool HumdrumFileContent::hasDifferentBarlines(void) {
	if (!m_analyses.m_barlines_analyzed) {
		analyzeBarlines();
	}
	return m_analyses.m_barlines_different;
}





//////////////////////////////
//
// HumdrumFileStructure::getMetricLevels -- Each line in the output
//     vector matches to the line of the metric analysis data.
//     undefined is the value to represent undefined analysis data
//     (for non-data spines).
//
//     default value: track = 0: 0 means use the time signature
//         of the first **kern spines in the file; otherwise, use the
//         time signatures found in the given track (indexed from 1
//         for the first spine on a line).
//     default value: undefined = NAN: The value to use for un-analyzed lines.
//

void HumdrumFileContent::getMetricLevels(vector<double>& output,
		int track, double undefined) {

	HumdrumFileStructure& infile = *this;
	int lineCount = infile.getLineCount();
	output.resize(lineCount);
	fill(output.begin(), output.end(), undefined);
	vector<HTp> kernspines = infile.getKernSpineStartList();
	if ((track == 0) && (kernspines.size() > 0)) {
		track = kernspines[0]->getTrack();
	}
	if (track == 0) {
		track = 1;
	}

	int top = 1;                // top number of time signature (0 for no meter)
	int bot = 4;                // bottom number of time signature
	bool compoundQ = false;     // test for compound meters, such as 6/8
	HumNum beatdur(1 * 4, bot); // duration of a beat in the measure
	HumNum measurepos;          // quarter notes from the start of barline
	HumNum combeatdur;          // for adjusting beat level in compound meters
	HumNum commeasurepos;       // for adjusting beat level in compound meters

	for (int i=0; i<lineCount; i++) {
		if (infile[i].isInterpretation()) {
			// check for time signature:
			HumdrumLine& line = *infile.getLine(i);
			for (int j=0; j<line.getFieldCount(); j++) {
				int ttrack = line.token(j)->getTrack();
				if (ttrack != track) {
					continue;
				}
				if (sscanf(infile.token(i,j)->c_str(), "*M%d/%d", &top, &bot)) {
					beatdur.setValue(1*4, bot); // converted to quarter-note units
					if ((top % 3 == 0) && (top != 3)) {
						// if meter top is a multiple of 3 but not 3, then compound
						// such as 6/8, 9/8, 6/4, but not 3/8, 3/4.
						compoundQ = true;
						beatdur *= 3;
					} else {
						compoundQ = false;
					}
					break;
				}
			}
		}
		if (!infile[i].isData()) {
				continue;
		}

		measurepos = infile[i].getDurationFromBarline();
		// Might want to handle cases where the time signature changes in
		// the middle or a measure...
		measurepos /= beatdur;
		int denominator = measurepos.getDenominator();
		if (compoundQ) {
			output[i] = Convert::nearIntQuantize(log(denominator) / log(3.0));
			if ((output[i] != 0.0) && (output[i] != 1.0)) {
				// if not the beat or first level, then calculate
				// levels above level 1.  In 6/8 this means
				// to move the 8th note level to be the "beat"
				// and then use binary levels for rhythmic levels
				// smaller than a beat.
				combeatdur.setValue(4,bot);
				commeasurepos = infile[i].getDurationFromBarline() / combeatdur;
				denominator = commeasurepos.getDenominator();
				output[i] = 1.0 + log(denominator)/log(2.0);
			}
		} else {
			output[i] = Convert::nearIntQuantize(log(denominator) / log(2.0));
		}
	}
}





//////////////////////////////
//
// HumdrumFileContent::analyzeCrossStaffStemDirections -- Calculate stem directions
//    for notes that are cross-staff, and the notes in the presence of cross-staff
//    notes.
//

void HumdrumFileContent::analyzeCrossStaffStemDirections(void) {
	string above = this->getKernAboveSignifier();
	string below = this->getKernBelowSignifier();

	if (above.empty() && below.empty()) {
		// no cross staff notes present in data
		return;
	}

	vector<HTp> kernstarts = getKernSpineStartList();
	for (int i=0; i<(int)kernstarts.size(); i++) {
		analyzeCrossStaffStemDirections(kernstarts[i]);
	}
}



//////////////////////////////
//
// HumdrumFileContent::analyzeCrossStaffStemDirections -- Check for cross-staff
//     notes, and assign stems if they do not have any.  Also assign stems to
//     notes in the target directory if there is only one layer active on that staff.
//

void HumdrumFileContent::analyzeCrossStaffStemDirections(HTp kernstart) {
	if (!kernstart) {
		return;
	}
	if (!kernstart->isKern()) {
		return;
	}
	string above = this->getKernAboveSignifier();
	string below = this->getKernBelowSignifier();
	if (above.empty() && below.empty()) {
		// no cross staff notes present in data
		return;
	}

	HTp current = kernstart;
	while (current) {
		if (current->isData()) {
			checkCrossStaffStems(current, above, below);
		}
		current = current->getNextToken();
	}
}



//////////////////////////////
//
// HumdrumFileContent::checkCrossStaffStems -- Check all notes in all
//     sub-spines of the current token (which should be in the top layer)
//     for cross-staff assignment.
//

void HumdrumFileContent::checkCrossStaffStems(HTp token, string& above, string& below) {
	int track = token->getTrack();

	HTp current = token;
	while (current) {
		int ttrack = current->getTrack();
		if (ttrack != track) {
			break;
		}
		checkDataForCrossStaffStems(current, above, below);
		current = current->getNextFieldToken();
	}
}



//////////////////////////////
//
// HumdrumFileContent::checkDataForCrossStaffStems -- Check a note or chord for
//    cross staff
//

void HumdrumFileContent::checkDataForCrossStaffStems(HTp token, string& above, string& below) {
	if (token->isNull()) {
		return;
	}
	if (token->isRest()) {
		// deal with cross-staff rests later
		return;
	}

	if (token->find('/') != std::string::npos) {
		// has a stem-up marker, so do not try to adjust stems;
		return;
	}

	if (token->find('\\') != std::string::npos) {
		// has a stem-down marker, so do not try to adjust stems;
		return;
	}


	HumRegex hre;
	bool hasaboveQ = false;
	bool hasbelowQ = false;

	if (!above.empty()) {
		string searchstring = "[A-Ga-g]+[#n-]*" + above;
		if (hre.search(*token, searchstring)) {
			// note/chord has staff-above signifier
			hasaboveQ = true;
		}
	}

	if (!below.empty()) {
		string searchstring = "[A-Ga-g]+[#n-]*" + below;
		if (hre.search(*token, searchstring)) {
			// note/chord has staff-below signifier
			hasbelowQ = true;
		}
	}

	if (!(hasaboveQ || hasbelowQ)) {
		// no above/below signifier, so nothing to do
		return;
	}
	if (hasaboveQ && hasbelowQ) {
		// strange complication of above and below, so ignore
		return;
	}

	if (hasaboveQ) {
		prepareStaffAboveNoteStems(token);
	} else if (hasbelowQ) {
		prepareStaffBelowNoteStems(token);
	}
}



//////////////////////////////
//
// HumdrumFileContent::prepareStaffAboveNoteStems --
//

void HumdrumFileContent::prepareStaffAboveNoteStems(HTp token) {
	token->setValue("auto", "stem.dir", "-1");
	int track = token->getTrack();
	HTp curr = token->getNextFieldToken();
	int ttrack;

	// Find the next higher **kern spine (if any):
	while (curr) {
		ttrack = curr->getTrack();
		if (!curr->isKern()) {
			curr = curr->getNextFieldToken();
			continue;
		}
		if (ttrack == track) {
			curr = curr->getNextFieldToken();
			continue;
		}
		// is kern data and in a different spine
		break;
	}
	if (!curr) {
		// no higher staff of **kern data.
		return;
	}
	if (!curr->isKern()) {
		// something strange happened
		return;
	}
	HumNum endtime = token->getDurationFromStart() + token->getDuration();
	HTp curr2 = curr;
	while (curr2) {
		if (curr2->getDurationFromStart() >= endtime) {
			// exceeded the duration of the cross-staff note, so stop looking
			break;
		}
		if (!curr2->isData()) {
			// ignore non-data tokens
			curr2 = curr2->getNextToken();
			continue;
		}
		if (curr2->isNull()) {
			curr2 = curr2->getNextToken();
			continue;
		}
		if (curr2->isRest()) {
			// ignore rests
			curr2 = curr2->getNextToken();
			continue;
		}
		if (!curr2->isNote()) {
			curr2 = curr2->getNextToken();
			continue;
		}
		if ((curr2->find('/') != std::string::npos) || (curr2->find('\\') != std::string::npos)) {
			// the note/chord has a stem direction, so ignore it
			curr2 = curr2->getNextToken();
			continue;
		}
		int layer = curr2->getSubtrack();
		// layer != 0 means there is more than one active layer at this point in the
		// above staff.  If so, then do not assign any stem directions.
		if (layer != 0) {
			curr2 = curr2->getNextToken();
			continue;
		}
		// set the stem to up for the current note/chord
		curr2->setValue("auto", "stem.dir", "1");
		curr2 = curr2->getNextToken();
	}
}



//////////////////////////////
//
// HumdrumFileContent::prepareStaffBelowNoteStems --
//

void HumdrumFileContent::prepareStaffBelowNoteStems(HTp token) {
	token->setValue("auto", "stem.dir", "1");
	int track = token->getTrack();
	HTp curr = token->getPreviousFieldToken();
	int ttrack;

	// Find the next lower **kern spine (if any):
	while (curr) {
		ttrack = curr->getTrack();
		if (!curr->isKern()) {
			curr = curr->getPreviousFieldToken();
			continue;
		}
		if (ttrack == track) {
			curr = curr->getPreviousFieldToken();
			continue;
		}
		// is kern data and in a different spine
		break;
	}
	if (!curr) {
		// no lower staff of **kern data.
		return;
	}
	if (!curr->isKern()) {
		// something strange happened
		return;
	}

	// Find the first subtrack of the identified spine
	int targettrack = curr->getTrack();
	while (curr) {
		HTp ptok = curr->getPreviousFieldToken();
		if (!ptok) {
			break;
		}
		ttrack = ptok->getTrack();
		if (targettrack != ttrack) {
			break;
		}
		curr = ptok;
		ptok = curr->getPreviousToken();
	}
	// Should now be at the first subtrack of the target staff.

	HumNum endtime = token->getDurationFromStart() + token->getDuration();
	HTp curr2 = curr;
	while (curr2) {
		if (curr2->getDurationFromStart() >= endtime) {
			// exceeded the duration of the cross-staff note, so stop looking
			break;
		}
		if (!curr2->isData()) {
			// ignore non-data tokens
			curr2 = curr2->getNextToken();
			continue;
		}
		if (curr2->isNull()) {
			curr2 = curr2->getNextToken();
			continue;
		}
		if (curr2->isRest()) {
			// ignore rests
			curr2 = curr2->getNextToken();
			continue;
		}
		if (!curr2->isNote()) {
			curr2 = curr2->getNextToken();
			continue;
		}
		if ((curr2->find('/') != std::string::npos) || (curr2->find('\\') != std::string::npos)) {
			// the note/chord has a stem direction, so ignore it
			curr2 = curr2->getNextToken();
			continue;
		}
		int layer = curr2->getSubtrack();
		// layer != 0 means there is more than one active layer at this point in the
		// above staff.  If so, then do not assign any stem directions.
		if (layer != 0) {
			curr2 = curr2->getNextToken();
			continue;
		}
		// set the stem to up for the current note/chord
		curr2->setValue("auto", "stem.dir", "-1");
		curr2 = curr2->getNextToken();
	}
}





//////////////////////////////
//
// HumdrumFileContent::analyzeOttavas --
//

void HumdrumFileContent::analyzeOttavas(void) {
	int tcount = getTrackCount();
	vector<int> activeOttava(tcount+1, 0);
	vector<int> octavestate(tcount+1, 0);
	for (int i=0; i<getLineCount(); i++) {
		HLp line = getLine(i);
		if (line->isInterpretation()) {
			int fcount = getLine(i)->getFieldCount();
			for (int j=0; j<fcount; j++) {
				HTp token = line->token(j);
				if (!token->isKern()) {
					continue;
				}
				int track = token->getTrack();
				if (*token == "*8va") {
					octavestate[track] = +1;
					activeOttava[track]++;
				} else if (*token == "*X8va") {
					octavestate[track] = 0;
					activeOttava[track]--;
				} else if (*token == "*8ba") {
					octavestate[track] = -1;
					activeOttava[track]++;
				} else if (*token == "*X8ba") {
					octavestate[track] = 0;
					activeOttava[track]--;
				} else if (*token == "*15ma") {
					octavestate[track] = +2;
					activeOttava[track]++;
				} else if (*token == "*X15ma") {
					octavestate[track] = 0;
					activeOttava[track]--;
				} else if (*token == "*15ba") {
					octavestate[track] = -2;
					activeOttava[track]++;
				} else if (*token == "*X15ba") {
					octavestate[track] = 0;
					activeOttava[track]--;
				}
			}
		}
		else if (line->isData()) {
			int fcount = getLine(i)->getFieldCount();
			for (int j=0; j<fcount; j++) {
				HTp token = line->token(j);
				if (!token->isKern()) {
					continue;
				}
				int track = token->getTrack();
				if (!activeOttava[track]) {
					continue;
				}
				if (octavestate[track] == 0) {
					continue;
				}
				if (token->isNull()) {
					continue;
				}
				if (token->isRest()) {
					// do not exclude rests, since the vertical placement
					// of the staff may need to be updated by the ottava mark.
				}
				token->setValue("auto", "ottava", to_string(octavestate[track]));
			}
		}
	}
}





//////////////////////////////
//
// HumdrumFileContent::analyzePhrasings -- Link start and ends of
//    phrases to each other.
//

bool HumdrumFileContent::analyzePhrasings(void) {
	if (m_analyses.m_phrases_analyzed) {
		return false;
	}
	m_analyses.m_phrases_analyzed = true;
	bool output = true;
	output &= analyzeKernPhrasings();
	return output;
}



//////////////////////////////
//
// HumdrumFileContent::analyzeKernPhrasings -- Link start and ends of
//    phrases to each other.
//

bool HumdrumFileContent::analyzeKernPhrasings(void) {
	vector<HTp> phrasestarts;
	vector<HTp> phraseends;

	vector<HTp> l;
	vector<pair<HTp, HTp>> labels; // first is previous label, second is next label
	HumdrumFileBase& infile = *this;
	labels.resize(infile.getLineCount());
	l.resize(infile.getLineCount());
	for (int i=0; i<infile.getLineCount(); i++) {
		labels[i].first = NULL;
		labels[i].second = NULL;
		l[i] = NULL;
	}
	for (int i=0; i<infile.getLineCount(); i++) {
		if (!infile[i].isInterpretation()) {
			continue;
		}
		HTp token = infile.token(i, 0);
		if ((token->compare(0, 2, "*>") == 0) && (token->find("[") == std::string::npos)) {
			l[i] = token;
		}
	}
	HTp current = NULL;
	for (int i=0; i<infile.getLineCount(); i++) {
		if (l[i] != NULL) {
			current = l[i];
		}
		labels[i].first = current;
	}
	current = NULL;
	for (int i=infile.getLineCount() - 1; i>=0; i--) {
		if (l[i] != NULL) {
			current = l[i];
		}
		labels[i].second = current;
	}

	vector<int> endings(infile.getLineCount(), 0);
	int ending = 0;
	for (int i=0; i<(int)endings.size(); i++) {
		if (l[i]) {
			char lastchar = l[i]->back();
			if (isdigit(lastchar)) {
				ending = lastchar - '0';
			} else {
				ending = 0;
			}
		}
		endings[i] = ending;
	}

	vector<HTp> kernspines;
	getSpineStartList(kernspines, "**kern");
	bool output = true;
	string linkSignifier = m_signifiers.getKernLinkSignifier();
	for (int i=0; i<(int)kernspines.size(); i++) {
		output = output && analyzeKernPhrasings(kernspines[i], phrasestarts, phraseends, labels, endings, linkSignifier);
	}

	createLinkedPhrasings(phrasestarts, phraseends);
	return output;
}


bool HumdrumFileContent::analyzeKernPhrasings(HTp spinestart,
		vector<HTp>& linkstarts, vector<HTp>& linkends, vector<pair<HTp, HTp>>& labels,
		vector<int>& endings, const string& linksig) {

	// linked phrases handled separately, so generate an ignore sequence:
	string ignorebegin = linksig + "{";
	string ignoreend = linksig + "}";

	// tracktokens == the 2-D data list for the track,
	// arranged in layers with the second dimension.
	vector<vector<HTp> > tracktokens;
	this->getTrackSeq(tracktokens, spinestart, OPT_DATA | OPT_NOEMPTY);
	// printSequence(tracktokens);

	// phraseopens == list of phrase openings for each track and elision level
	// first dimension: elision level
	// second dimension: track number
	vector<vector<vector<HTp>>> phraseopens;

	phraseopens.resize(4); // maximum of 4 elision levels
	for (int i=0; i<(int)phraseopens.size(); i++) {
		phraseopens[i].resize(8);  // maximum of 8 layers
	}

	int opencount = 0;
	int closecount = 0;
	int elision = 0;
	HTp token;
	for (int row=0; row<(int)tracktokens.size(); row++) {
		for (int track=0; track<(int)tracktokens[row].size(); track++) {
			token = tracktokens[row][track];
			if (!token->isData()) {
				continue;
			}
			if (token->isNull()) {
				continue;
			}
			opencount = (int)count(token->begin(), token->end(), '{');
			closecount = (int)count(token->begin(), token->end(), '}');

			for (int i=0; i<closecount; i++) {
				bool isLinked = isLinkedPhraseEnd(token, i, ignoreend);
				if (isLinked) {
					linkends.push_back(token);
					continue;
				}
				elision = token->getPhraseEndElisionLevel(i);
				if (elision < 0) {
					continue;
				}
				if (phraseopens[elision][track].size() > 0) {
					linkPhraseEndpoints(phraseopens[elision][track].back(), token);
					// remove phrase opening from buffer
					phraseopens[elision][track].pop_back();
				} else {
					// No starting phrase marker to match to this phrase end in the
					// given track.
					// search for an open phrase in another track:
					bool found = false;
					for (int itrack=0; itrack<(int)phraseopens[elision].size(); itrack++) {
						if (phraseopens[elision][itrack].size() > 0) {
							linkPhraseEndpoints(phraseopens[elision][itrack].back(), token);
							// remove phrase opening from buffer
							phraseopens[elision][itrack].pop_back();
							found = true;
							break;
						}
					}
					if (!found) {
						int lineindex = token->getLineIndex();
						int endnum = endings[lineindex];
						int pindex = -1;
						if (labels[lineindex].first) {
							pindex = labels[lineindex].first->getLineIndex();
							pindex--;
						}
						int endnumpre = -1;
						if (pindex >= 0) {
							endnumpre = endings[pindex];
						}

						if ((endnumpre > 0) && (endnum > 0) && (endnumpre != endnum)) {
							// This is a phrase in an ending that start at the start of an ending.
							HumNum duration = token->getDurationFromStart();
							if (labels[token->getLineIndex()].first) {
								duration -= labels[token->getLineIndex()].first->getDurationFromStart();
							}
							token->setValue("auto", "endingPhraseBack", "true");
							token->setValue("auto", "phraseSide", "stop");
							token->setValue("auto", "phraseDration",
								token->getDurationToEnd());
						} else {
							// This is a phrase closing that does not have a matching opening.
							token->setValue("auto", "hangingPhrase", "true");
							token->setValue("auto", "phraseSide", "stop");
							token->setValue("auto", "phraseOpenIndex", to_string(i));
							token->setValue("auto", "phraseDration",
								token->getDurationToEnd());
						}
					}
				}
			}

			for (int i=0; i<opencount; i++) {
				bool isLinked = isLinkedPhraseBegin(token, i, ignorebegin);
				if (isLinked) {
					linkstarts.push_back(token);
					continue;
				}
				elision = token->getPhraseStartElisionLevel(i);
				if (elision < 0) {
					continue;
				}
				phraseopens[elision][track].push_back(token);
			}
		}
	}

	// Mark un-closed phrase starts:
	for (int i=0; i<(int)phraseopens.size(); i++) {
		for (int j=0; j<(int)phraseopens[i].size(); j++) {
			for (int k=0; k<(int)phraseopens[i][j].size(); k++) {
				phraseopens[i][j][k]->setValue("", "auto", "hangingPhrase", "true");
				phraseopens[i][j][k]->setValue("", "auto", "phraseSide", "start");
				phraseopens[i][j][k]->setValue("", "auto", "phraseDuration",
						phraseopens[i][j][k]->getDurationFromStart());
			}
		}
	}

	return true;
}



//////////////////////////////
//
// HumdrumFileContent::createLinkedPhrasings --  Currently assume that
//    start and ends are matched.
//

void HumdrumFileContent::createLinkedPhrasings(vector<HTp>& linkstarts, vector<HTp>& linkends) {
	int max = (int)linkstarts.size();
	if ((int)linkends.size() < max) {
		max = (int)linkends.size();
	}
	if (max == 0) {
		// nothing to do
		return;
	}

	for (int i=0; i<max; i++) {
		linkPhraseEndpoints(linkstarts[i], linkends[i]);
	}
}



//////////////////////////////
//
// HumdrumFileContent::isLinkedPhraseEnd --
//

bool HumdrumFileContent::isLinkedPhraseEnd(HTp token, int index, const string& pattern) {
	if (pattern.size() <= 1) {
		return false;
	}
	int counter = -1;
	for (int i=0; i<(int)token->size(); i++) {
		if (token->at(i) == ')') {
			counter++;
		}
		if (i == 0) {
			// Can't have linked phrase at starting index in string.
			continue;
		}
		if (counter != index) {
			continue;
		}

		int startindex = i - (int)pattern.size() + 1;
		auto loc = token->find(pattern, startindex);
		if ((loc != std::string::npos) && ((int)loc == startindex)) {
			return true;
		}
		return false;
	}
	return false;
}



//////////////////////////////
//
// HumdrumFileContent::isLinkedPhraseBegin --
//

bool HumdrumFileContent::isLinkedPhraseBegin(HTp token, int index, const string& pattern) {
	if (pattern.size() <= 1) {
		return false;
	}
	int counter = -1;
	for (int i=0; i<(int)token->size(); i++) {
		if (token->at(i) == '(') {
			counter++;
		}
		if (i == 0) {
			continue;
		}
		if (counter != index) {
			continue;
		}
		if (token->find(pattern, i - (int)pattern.size() + 1) != std::string::npos) {
			return true;
		}
		return false;
	}
	return false;
}



//////////////////////////////
//
// HumdrumFileContent::linkPhraseEndpoints --  Allow up to two phrase starts/ends
//      on a note.
//

void HumdrumFileContent::linkPhraseEndpoints(HTp phrasestart, HTp phraseend) {
	string durtag = "phraseDuration";
	string endtag = "phraseEnd";
	int phraseEndCount = phrasestart->getValueInt("auto", "phraseEndCount");
	phraseEndCount++;
	if (phraseEndCount > 1) {
		endtag += to_string(phraseEndCount);
		durtag += to_string(phraseEndCount);
	}
	string starttag = "phraseStart";
	int phraseStartCount = phraseend->getValueInt("auto", "phraseStartCount");
	phraseStartCount++;
	if (phraseStartCount > 1) {
		starttag += to_string(phraseStartCount);
	}

	phrasestart->setValue("auto", endtag, phraseend);
	phrasestart->setValue("auto", "id", phrasestart);
	phraseend->setValue("auto", starttag, phrasestart);
	phraseend->setValue("auto", "id", phraseend);
	HumNum duration = phraseend->getDurationFromStart()
			- phrasestart->getDurationFromStart();
	phrasestart->setValue("auto", durtag, duration);
	phrasestart->setValue("auto", "phraseEndCount", to_string(phraseEndCount));
	phraseend->setValue("auto", "phraseStartCount", to_string(phraseStartCount));
}





//////////////////////////////
//
// HumdrumFileContent::analyzeRestPositions -- Calculate the vertical position
//    of rests on staves with two layers.
//

void HumdrumFileContent::analyzeRestPositions(void) {
	vector<HTp> kernstarts = getKernSpineStartList();
	for (int i=0; i<(int)kernstarts.size(); i++) {
		assignImplicitVerticalRestPositions(kernstarts[i]);
	}

	checkForExplicitVerticalRestPositions();
}



//////////////////////////////
//
// HumdrumFileContent::checkForExplicitVerticalRestPositions -- Starting at the
//     current layer, check all rests in the same track for vertical positioning.
//

void HumdrumFileContent::checkForExplicitVerticalRestPositions(void) {
	HumdrumFileContent& infile = *this;
	vector<int> baselines(infile.getTrackCount() + 1, Convert::kernClefToBaseline("*clefG2"));
	for (int i=0; i<infile.getLineCount(); i++) {
		if (infile[i].isInterpretation()) {
			for (int j=0; j<infile[i].getFieldCount(); j++) {
				HTp tok = infile.token(i, j);
				if (!tok->isKern()) {
					continue;
				}
				if (!tok->isClef()) {
					continue;
				}
				int track = tok->getTrack();
				baselines[track] = Convert::kernClefToBaseline(tok);
			}
		}
		if (!infile[i].isData()) {
			continue;
		}
		for (int j=0; j<infile[i].getFieldCount(); j++) {
			HTp tok = infile.token(i, j);
			if (!tok->isKern()) {
				continue;
			}
			if (!tok->isRest()) {
				continue;
			}
			int track = tok->getTrack();
			checkRestForVerticalPositioning(tok, baselines[track]);
		}
	}
}



//////////////////////////////
//
// HumdrumFileContent::assignImplicitVerticalRestPositions -- Starting at the
//     current layer, check all rests in the same track for vertical positioning.
//     Only checks the first and second layers in a track.
//

void HumdrumFileContent::assignImplicitVerticalRestPositions(HTp kernstart) {
	if (!kernstart) {
		return;
	}

	int baseline = Convert::kernClefToBaseline("*clefG2");
	HTp current = kernstart;
	int track = kernstart->getTrack();

	while (current) {
		if (current->isClef()) {
			baseline = Convert::kernClefToBaseline(current);
			current = current->getNextToken();
			continue;
		}
		if (!current->isData()) {
			current = current->getNextToken();
			continue;
		}
		int strack = -1;
		HTp second = current->getNextFieldToken();
		if (second) {
			strack = second->getTrack();
		}
		if (track != strack) {
         if (current->isRest()) {
				checkRestForVerticalPositioning(current, baseline);
			}
			// only one layer in current spine.
			current = current->getNextToken();
			continue;
		}
		if (current->isNull()) {
			HTp resolve = current->resolveNull();
			if (resolve && resolve->isRest()) {
				if (second && second->isRest()) {
					if (checkRestForVerticalPositioning(second, baseline)) {
						current = current->getNextToken();
						continue;
					}
				}
			}
			current = current->getNextToken();
			continue;
		}
		if (current->isRest()) {
			// assign a default position for the rest, since
			// verovio will try to tweak it when there is
			// more than one layer on the staff.
			setRestOnCenterStaffLine(current, baseline);
		}
		if (current->isRest()) {
			if (checkRestForVerticalPositioning(current, baseline)) {
				if (second && second->isRest()) {
					if (checkRestForVerticalPositioning(second, baseline)) {
						current = current->getNextToken();
						continue;
					}
				}
				current = current->getNextToken();
				continue;
			}
		}
		if (second && second->isRest()) {
			if (checkRestForVerticalPositioning(second, baseline)) {
				current = current->getNextToken();
				continue;
			}
		}
		if (!second) {
			current = current->getNextToken();
			continue;
		}
		if (second->isRest()) {
			// assign a default position for the rest, since
			// verovio will try to tweak it when there is
			// more than one layer on the staff.
			setRestOnCenterStaffLine(current, baseline);
			setRestOnCenterStaffLine(second, baseline);
		}
		if (second->isNull()) {
			current = current->getNextToken();
			continue;
		}
		if (current->isRest() && second->isRest()) {
			// not dealing with rest against rest for now
			current = current->getNextToken();
			// what to do with vertical positions?  The are
			// current collapsed into a single rest
			// with the code above.
			continue;
		}
		if (current->isRest() || second->isRest()) {
			assignVerticalRestPosition(current, second, baseline);
		}
		current = current->getNextToken();
		continue;
	}
}



//////////////////////////////
//
// HumdrumFileContent::checkRestForVerticalPositioning -- Read any pitch information attached to
//     a rest and convert to ploc/oloc values.
//

bool HumdrumFileContent::checkRestForVerticalPositioning(HTp rest, int baseline) {
	HumRegex hre;
	if (!hre.search(rest, "([A-Ga-g]+)")) {
		return false;
	}
	string pitch = hre.getMatch(1);
	int b7 = Convert::kernToBase7(pitch);

	int diff = (b7 - baseline) + 100;
	if (diff % 2) {
		// force to every other diatonic step (stafflines)
		HumNum dur = rest->getDuration();
		if (dur > 1) {
			b7--;
		} else {
			b7++;
		}
	}

	int pc = b7 % 7;
	int oct = b7 / 7;

	string dname;
	switch (pc) {
		case 0: dname = "C"; break;
		case 1: dname = "D"; break;
		case 2: dname = "E"; break;
		case 3: dname = "F"; break;
		case 4: dname = "G"; break;
		case 5: dname = "A"; break;
		case 6: dname = "B"; break;
	}

	if (dname.empty()) {
		return false;
	}

	string oloc = to_string(oct);

	rest->setValue("auto", "ploc", dname);
	rest->setValue("auto", "oloc", oloc);

	return true;
}





//////////////////////////////
//
// HumdrumFileContent::setRestOnCenterStaffLine --
//

void HumdrumFileContent::setRestOnCenterStaffLine(HTp rest, int baseline) {
	int rpos = 4;
	int restdia = rpos + baseline;
	int pc = restdia % 7;
	int oct = restdia / 7;

	string dname;
	switch (pc) {
		case 0: dname = "C"; break;
		case 1: dname = "D"; break;
		case 2: dname = "E"; break;
		case 3: dname = "F"; break;
		case 4: dname = "G"; break;
		case 5: dname = "A"; break;
		case 6: dname = "B"; break;
	}

	if (dname.empty()) {
		return;
	}

	string oloc = to_string(oct);

	rest->setValue("auto", "ploc", dname);
	rest->setValue("auto", "oloc", oloc);
}



//////////////////////////////
//
//	HumdrumFileContents::assignVerticalRestPosition --
//

void HumdrumFileContent::assignVerticalRestPosition(HTp first, HTp second, int baseline) {
	vector<string> tokens;
	vector<int> vpos;

	int notepos = 0;
	HTp rest = NULL;
	HTp notes = NULL;
	if (first->isRest()) {
		rest = first;
		notes = second;
		notepos = -1;
	} else if (second->isRest()) {
		rest = second;
		notes = first;
		notepos = +1;
	} else {
		return;
	}

	int count = notes->getSubtokenCount();
	for (int i=0; i<count; i++) {
		vpos.push_back(Convert::kernToBase7(notes->getSubtoken(i)) - baseline);
	}

	int rpos = 0;
	if (notepos > 0) {
		rpos = getRestPositionBelowNotes(rest, vpos);
	} else if (notepos < 0) {
		rpos = getRestPositionAboveNotes(rest, vpos);
	} else {
		return;
	}

	int restdia = rpos + baseline;
	int pc = restdia % 7;
	int oct = restdia / 7;

	string dname;
	switch (pc) {
		case 0: dname = "C"; break;
		case 1: dname = "D"; break;
		case 2: dname = "E"; break;
		case 3: dname = "F"; break;
		case 4: dname = "G"; break;
		case 5: dname = "A"; break;
		case 6: dname = "B"; break;
	}

	if (dname.empty()) {
		return;
	}

	string oloc = to_string(oct);

	rest->setValue("auto", "ploc", dname);
	rest->setValue("auto", "oloc", oloc);
}



//////////////////////////////
//
// HumdrumFileContent::getRestPositionBelowNotes --
//

int HumdrumFileContent::getRestPositionBelowNotes(HTp rest, vector<int>& vpos) {
	if (vpos.empty()) {
		return 4;
	}
	int lowest = vpos[0];
	for (int i=1; i<(int)vpos.size(); i++) {
		if (lowest > vpos[i]) {
			lowest = vpos[i];
		}
	}
	int restint = 0;
	double resttype = log(rest->getDuration().getFloat()) / log(2.0);
	restint = int(resttype + 1000.0) - 1000;
	int output = 0;

	switch (restint) {

		case 0: // quarter-note rests
			output = 0;
			switch (lowest) {
				case -2: output = -8; break;
				case -1: output = -6; break;
				case  0: output = -6; break;
				case  1: output = -4; break;
				case  2: output = -4; break;
				case  3: output = -2; break;
				case  4: output = -2; break;
				case  5: output =  0; break;
				case  6: output =  0; break;
				case  7: output =  2; break;
				case  8: output =  2; break;
				case  9: output =  4; break;
				case 10: output =  4; break;
			}
			if (lowest > 10) {
				output = 4;
			}
			if (lowest < -2) {
				output = lowest - 6;
				if (lowest % 2) {
					output++;
				}
			}
			return output;
			break;

		case -1: // eighth-note rests
		case -2: // sixteenth-note rests
			output = 0;
			switch (lowest) {
				case -2: output = -6; break;
				case -1: output = -4; break;
				case  0: output = -4; break;
				case  1: output = -2; break;
				case  2: output = -2; break;
				case  3: output =  0; break;
				case  4: output =  0; break;
				case  5: output =  2; break;
				case  6: output =  2; break;
				case  7: output =  4; break;
				case  8: output =  4; break;
				case  9: output =  4; break;
				case 10: output =  4; break;
			}
			if (lowest > 10) {
				output = 4;
			}
			if (lowest < -2) {
				output = lowest - 4;
				if (lowest % 2) {
					output++;
				}
			}
			return output;
			break;

		case -3: // 32nd-note rests
		case -4: // 64h-note rests
			output = 0;
			switch (lowest) {
				case -2: output = -8; break;
				case -1: output = -6; break;
				case  0: output = -6; break;
				case  1: output = -4; break;
				case  2: output = -4; break;
				case  3: output = -2; break;
				case  4: output = -2; break;
				case  5: output =  0; break;
				case  6: output =  0; break;
				case  7: output =  2; break;
				case  8: output =  2; break;
				case  9: output =  4; break;
				case 10: output =  4; break;
			}
			if (lowest > 10) {
				output = 4;
			}
			if (lowest < -2) {
				output = lowest - 6;
				if (lowest % 2) {
					output++;
				}
			}
			return output;
			break;

		case -5: // 128th-note rests
		case -6: // 256th-note rests
			output = 0;
			switch (lowest) {
				case -2: output = -10; break;
				case -1: output = -8;  break;
				case  0: output = -8;  break;
				case  1: output = -6;  break;
				case  2: output = -6;  break;
				case  3: output = -4;  break;
				case  4: output = -4;  break;
				case  5: output = -2;  break;
				case  6: output = -2;  break;
				case  7: output =  0;  break;
				case  8: output =  0;  break;
				case  9: output =  2;  break;
				case 10: output =  2;  break;
			}
			if (lowest > 10) {
				output = 4;
			}
			if (lowest < -2) {
				output = lowest - 8;
				if (lowest % 2) {
					output++;
				}
			}
			return output;
			break;

		case 1: // half-note rests
		case 2: // whole-note rests
		case 3: // breve-note rests
			output = 0;
			switch (lowest) {
				case -2: output = -6; break;
				case -1: output = -6; break;
				case  0: output = -4; break;
				case  1: output = -4; break;
				case  2: output = -2; break;
				case  3: output = -2; break;
				case  4: output =  0; break;
				case  5: output =  0; break;
				case  6: output =  2; break;
				case  7: output =  2; break;
				case  8: output =  4; break;
				case  9: output =  4; break;
				case 10: output =  4; break;
			}
			if (lowest > 10) {
				output = 4;
			}
			if (lowest < -2) {
				output = lowest - 4;
				if (lowest % 2) {
					output--;
				}
			}
			return output;
			break;

	}
	return 0;
}



//////////////////////////////
//
// HumdrumFileContent::getRestPositionAboveNotes --
//

int HumdrumFileContent::getRestPositionAboveNotes(HTp rest, vector<int>& vpos) {
	if (vpos.empty()) {
		return 4;
	}
	int highest = vpos[0];
	for (int i=1; i<(int)vpos.size(); i++) {
		if (highest < vpos[i]) {
			highest = vpos[i];
		}
	}
	int restint = 0;
	double resttype = log(rest->getDuration().getFloat()) / log(2.0);
	restint = int(resttype + 1000.0) - 1000;
	int output = 8;

	switch (restint) {

		case 0: // quarter-note rests
			output = 0;
			switch (highest) {
				case -2: output =  4; break;
				case -1: output =  4; break;
				case  0: output =  4; break;
				case  1: output =  6; break;
				case  2: output =  6; break;
				case  3: output =  8; break;
				case  4: output =  8; break;
				case  5: output = 10; break;
				case  6: output = 10; break;
				case  7: output = 12; break;
				case  8: output = 12; break;
				case  9: output = 14; break;
				case 10: output = 14; break;
			}
			if (highest < -2) {
				output = 4;
			}
			if (highest > 10) {
				output = highest + 4;
				if (highest % 2) {
					output++;
				}
			}
			return output;
			break;

		case -1: // eighth-note rests
			output = 0;
			switch (highest) {
				case -2: output =  4; break;
				case -1: output =  4; break;
				case  0: output =  4; break;
				case  1: output =  4; break;
				case  2: output =  6; break;
				case  3: output =  6; break;
				case  4: output =  8; break;
				case  5: output =  8; break;
				case  6: output = 10; break;
				case  7: output = 10; break;
				case  8: output = 12; break;
				case  9: output = 12; break;
				case 10: output = 14; break;
			}
			if (highest < -2) {
				output = 4;
			}
			if (highest > 10) {
				output = highest + 4;
				if (highest % 2) {
					output--;
				}
			}
			return output;
			break;

		case -2: // sixteenth-note rests
		case -3: // 32nd-note rests
			output = 0;
			switch (highest) {
				case -2: output =  4; break;
				case -1: output =  4; break;
				case  0: output =  6; break;
				case  1: output =  6; break;
				case  2: output =  8; break;
				case  3: output =  8; break;
				case  4: output = 10; break;
				case  5: output = 10; break;
				case  6: output = 12; break;
				case  7: output = 12; break;
				case  8: output = 14; break;
				case  9: output = 14; break;
				case 10: output = 16; break;
			}
			if (highest < -2) {
				output = 4;
			}
			if (highest > 10) {
				output = highest + 6;
				if (highest % 2) {
					output--;
				}
			}
			return output;
			break;

		case -4: // 64th-note rests
		case -5: // 128th-note rests
			output = 0;
			switch (highest) {
				case -2: output =  6; break;
				case -1: output =  6; break;
				case  0: output =  8; break;
				case  1: output =  8; break;
				case  2: output = 10; break;
				case  3: output = 10; break;
				case  4: output = 12; break;
				case  5: output = 12; break;
				case  6: output = 14; break;
				case  7: output = 14; break;
				case  8: output = 16; break;
				case  9: output = 16; break;
				case 10: output = 18; break;
			}
			if (highest < -2) {
				output = 4;
			}
			if (highest > 10) {
				output = highest + 8;
				if (highest % 2) {
					output--;
				}
			}
			return output;
			break;

		case -6: // 256th-note rests
			output = 0;
			switch (highest) {
				case -4: output =  6; break;
				case -3: output =  6; break;
				case -2: output =  8; break;
				case -1: output =  8; break;
				case  0: output = 10; break;
				case  1: output = 10; break;
				case  2: output = 12; break;
				case  3: output = 12; break;
				case  4: output = 14; break;
				case  5: output = 14; break;
				case  6: output = 16; break;
				case  7: output = 16; break;
				case  8: output = 18; break;
				case  9: output = 18; break;
				case 10: output = 20; break;
			}
			if (highest < -4) {
				output = 4;
			}
			if (highest > 10) {
				output = highest + 10;
				if (highest % 2) {
					output--;
				}
			}
			return output;
			break;

		case 1: // half-note rests
		case 2: // whole-note rests
		case 3: // breve-note rests
			output = 0;
			switch (highest) {
				case -2: output =  4; break;
				case -1: output =  4; break;
				case  0: output =  4; break;
				case  1: output =  4; break;
				case  2: output =  6; break;
				case  3: output =  6; break;
				case  4: output =  8; break;
				case  5: output =  8; break;
				case  6: output = 10; break;
				case  7: output = 10; break;
				case  8: output = 12; break;
				case  9: output = 12; break;
				case 10: output = 14; break;
			}
			if (highest < -2) {
				output = 4;
			}
			if (highest > 10) {
				output = highest + 4;
				if (highest % 2) {
					output--;
				}
			}
			return output;
			break;

	}

	return output;
}





//////////////////////////////
//
// HumdrumFileContent::analyzeSlurs -- Link start and ends of
//    slurs to each other.
//

bool HumdrumFileContent::analyzeSlurs(void) {
	if (m_analyses.m_slurs_analyzed) {
		return false;
	}
	m_analyses.m_slurs_analyzed = true;
	bool output = true;
	output &= analyzeKernSlurs();
	output &= analyzeMensSlurs();
	return output;
}



//////////////////////////////
//
// HumdrumFileContent::analyzeMensSlurs -- Link start and ends of
//    slurs to each other.  They are the same as **kern, so borrowing
//    analyzeKernSlurs to do the analysis.
//

bool HumdrumFileContent::analyzeMensSlurs(void) {
	vector<HTp> slurstarts;
	vector<HTp> slurends;

	vector<HTp> l;
	vector<pair<HTp, HTp>> labels; // first is previous label, second is next label
	HumdrumFileBase& infile = *this;
	labels.resize(infile.getLineCount());
	l.resize(infile.getLineCount());
	for (int i=0; i<infile.getLineCount(); i++) {
		labels[i].first = NULL;
		labels[i].second = NULL;
		l[i] = NULL;
	}
	for (int i=0; i<infile.getLineCount(); i++) {
		if (!infile[i].isInterpretation()) {
			continue;
		}
		HTp token = infile.token(i, 0);
		if ((token->compare(0, 2, "*>") == 0) && (token->find("[") == std::string::npos)) {
			l[i] = token;
		}
	}
	HTp current = NULL;
	for (int i=0; i<infile.getLineCount(); i++) {
		if (l[i] != NULL) {
			current = l[i];
		}
		labels[i].first = current;
	}
	current = NULL;
	for (int i=infile.getLineCount() - 1; i>=0; i--) {
		if (l[i] != NULL) {
			current = l[i];
		}
		labels[i].second = current;
	}

	vector<int> endings(infile.getLineCount(), 0);
	int ending = 0;
	for (int i=0; i<(int)endings.size(); i++) {
		if (l[i]) {
			char lastchar = l[i]->back();
			if (isdigit(lastchar)) {
				ending = lastchar - '0';
			} else {
				ending = 0;
			}
		}
		endings[i] = ending;
	}

	vector<HTp> mensspines;
	getSpineStartList(mensspines, "**mens");
	bool output = true;
	string linkSignifier = m_signifiers.getKernLinkSignifier();
	for (int i=0; i<(int)mensspines.size(); i++) {
		output = output && analyzeKernSlurs(mensspines[i], slurstarts, slurends, labels, endings, linkSignifier);
	}
	createLinkedSlurs(slurstarts, slurends);
	return output;
}



//////////////////////////////
//
// HumdrumFileContent::analyzeKernSlurs -- Link start and ends of
//    slurs to each other.
//

bool HumdrumFileContent::analyzeKernSlurs(void) {
	vector<HTp> slurstarts;
	vector<HTp> slurends;

	vector<HTp> l;
	vector<pair<HTp, HTp>> labels; // first is previous label, second is next label
	HumdrumFileBase& infile = *this;
	labels.resize(infile.getLineCount());
	l.resize(infile.getLineCount());
	for (int i=0; i<infile.getLineCount(); i++) {
		labels[i].first = NULL;
		labels[i].second = NULL;
		l[i] = NULL;
	}
	for (int i=0; i<infile.getLineCount(); i++) {
		if (!infile[i].isInterpretation()) {
			continue;
		}
		HTp token = infile.token(i, 0);
		if ((token->compare(0, 2, "*>") == 0) && (token->find("[") == std::string::npos)) {
			l[i] = token;
		}
	}
	HTp current = NULL;
	for (int i=0; i<infile.getLineCount(); i++) {
		if (l[i] != NULL) {
			current = l[i];
		}
		labels[i].first = current;
	}
	current = NULL;
	for (int i=infile.getLineCount() - 1; i>=0; i--) {
		if (l[i] != NULL) {
			current = l[i];
		}
		labels[i].second = current;
	}

	vector<int> endings(infile.getLineCount(), 0);
	int ending = 0;
	for (int i=0; i<(int)endings.size(); i++) {
		if (l[i]) {
			char lastchar = l[i]->back();
			if (isdigit(lastchar)) {
				ending = lastchar - '0';
			} else {
				ending = 0;
			}
		}
		endings[i] = ending;
	}

	vector<HTp> kernspines;
	getSpineStartList(kernspines, "**kern");
	bool output = true;
	string linkSignifier = m_signifiers.getKernLinkSignifier();
	for (int i=0; i<(int)kernspines.size(); i++) {
		output = output && analyzeKernSlurs(kernspines[i], slurstarts, slurends, labels, endings, linkSignifier);
	}

	createLinkedSlurs(slurstarts, slurends);
	return output;
}


bool HumdrumFileContent::analyzeKernSlurs(HTp spinestart,
		vector<HTp>& linkstarts, vector<HTp>& linkends, vector<pair<HTp, HTp>>& labels,
		vector<int>& endings, const string& linksig) {

	// linked slurs handled separately, so generate an ignore sequence:
	string ignorebegin = linksig + "(";
	string ignoreend = linksig + ")";

	// tracktokens == the 2-D data list for the track,
	// arranged in layers with the second dimension.
	vector<vector<HTp> > tracktokens;
	this->getTrackSeq(tracktokens, spinestart, OPT_DATA | OPT_NOEMPTY);
	// printSequence(tracktokens);

	// sluropens == list of slur openings for each track and elision level
	// first dimension: elision level
	// second dimension: track number
	vector<vector<vector<HTp>>> sluropens;

	sluropens.resize(4); // maximum of 4 elision levels
	for (int i=0; i<(int)sluropens.size(); i++) {
		sluropens[i].resize(8);  // maximum of 8 layers
	}

	int opencount = 0;
	int closecount = 0;
	int elision = 0;
	HTp token;
	for (int row=0; row<(int)tracktokens.size(); row++) {
		for (int track=0; track<(int)tracktokens[row].size(); track++) {
			token = tracktokens[row][track];
			if (!token->isData()) {
				continue;
			}
			if (token->isNull()) {
				continue;
			}
			opencount = (int)count(token->begin(), token->end(), '(');
			closecount = (int)count(token->begin(), token->end(), ')');

			for (int i=0; i<closecount; i++) {
				bool isLinked = isLinkedSlurEnd(token, i, ignoreend);
				if (isLinked) {
					linkends.push_back(token);
					continue;
				}
				elision = token->getSlurEndElisionLevel(i);
				if (elision < 0) {
					continue;
				}
				if (sluropens[elision][track].size() > 0) {
					linkSlurEndpoints(sluropens[elision][track].back(), token);
					// remove slur opening from buffer
					sluropens[elision][track].pop_back();
				} else {
					// No starting slur marker to match to this slur end in the
					// given track.
					// search for an open slur in another track:
					bool found = false;
					for (int itrack=0; itrack<(int)sluropens[elision].size(); itrack++) {
						if (sluropens[elision][itrack].size() > 0) {
							linkSlurEndpoints(sluropens[elision][itrack].back(), token);
							// remove slur opening from buffer
							sluropens[elision][itrack].pop_back();
							found = true;
							break;
						}
					}
					if (!found) {
						int lineindex = token->getLineIndex();
						int endnum = endings[lineindex];
						int pindex = -1;
						if (labels[lineindex].first) {
							pindex = labels[lineindex].first->getLineIndex();
							pindex--;
						}
						int endnumpre = -1;
						if (pindex >= 0) {
							endnumpre = endings[pindex];
						}

						if ((endnumpre > 0) && (endnum > 0) && (endnumpre != endnum)) {
							// This is a slur in an ending that start at the start of an ending.
							HumNum duration = token->getDurationFromStart();
							if (labels[token->getLineIndex()].first) {
								duration -= labels[token->getLineIndex()].first->getDurationFromStart();
							}
							token->setValue("auto", "endingSlurBack", "true");
							token->setValue("auto", "slurSide", "stop");
							token->setValue("auto", "slurDration",
								token->getDurationToEnd());
						} else {
							// This is a slur closing that does not have a matching opening.
							token->setValue("auto", "hangingSlur", "true");
							token->setValue("auto", "slurSide", "stop");
							token->setValue("auto", "slurOpenIndex", to_string(i));
							token->setValue("auto", "slurDration",
								token->getDurationToEnd());
						}
					}
				}
			}

			for (int i=0; i<opencount; i++) {
				bool isLinked = isLinkedSlurBegin(token, i, ignorebegin);
				if (isLinked) {
					linkstarts.push_back(token);
					continue;
				}
				elision = token->getSlurStartElisionLevel(i);
				if (elision < 0) {
					continue;
				}
				sluropens[elision][track].push_back(token);
			}
		}
	}

	// Mark un-closed slur starts:
	for (int i=0; i<(int)sluropens.size(); i++) {
		for (int j=0; j<(int)sluropens[i].size(); j++) {
			for (int k=0; k<(int)sluropens[i][j].size(); k++) {
				sluropens[i][j][k]->setValue("", "auto", "hangingSlur", "true");
				sluropens[i][j][k]->setValue("", "auto", "slurSide", "start");
				sluropens[i][j][k]->setValue("", "auto", "slurDuration",
						sluropens[i][j][k]->getDurationFromStart());
			}
		}
	}

	return true;
}



//////////////////////////////
//
// HumdrumFileContent::createLinkedSlurs --  Currently assume that
//    start and ends are matched.
//

void HumdrumFileContent::createLinkedSlurs(vector<HTp>& linkstarts, vector<HTp>& linkends) {
	int max = (int)linkstarts.size();
	if ((int)linkends.size() < max) {
		max = (int)linkends.size();
	}
	if (max == 0) {
		// nothing to do
		return;
	}

	for (int i=0; i<max; i++) {
		linkSlurEndpoints(linkstarts[i], linkends[i]);
	}
}



//////////////////////////////
//
// HumdrumFileContent::isLinkedSlurEnd --
//

bool HumdrumFileContent::isLinkedSlurEnd(HTp token, int index, const string& pattern) {
	if (pattern.size() <= 1) {
		return false;
	}
	int counter = -1;
	for (int i=0; i<(int)token->size(); i++) {
		if (token->at(i) == ')') {
			counter++;
		}
		if (i == 0) {
			// Can't have linked slur at starting index in string.
			continue;
		}
		if (counter != index) {
			continue;
		}

		int startindex = i - (int)pattern.size() + 1;
		auto loc = token->find(pattern, startindex);
		if ((loc != std::string::npos) && ((int)loc == startindex)) {
			return true;
		}
		return false;
	}
	return false;
}



//////////////////////////////
//
// HumdrumFileContent::isLinkedSlurBegin --
//

bool HumdrumFileContent::isLinkedSlurBegin(HTp token, int index, const string& pattern) {
	if (pattern.size() <= 1) {
		return false;
	}
	int counter = -1;
	for (int i=0; i<(int)token->size(); i++) {
		if (token->at(i) == '(') {
			counter++;
		}
		if (i == 0) {
			continue;
		}
		if (counter != index) {
			continue;
		}
		if (token->find(pattern, i - (int)pattern.size() + 1) != std::string::npos) {
			return true;
		}
		return false;
	}
	return false;
}



//////////////////////////////
//
// HumdrumFileContent::linkSlurEndpoints --  Allow up to two slur starts/ends
//      on a note.
//

void HumdrumFileContent::linkSlurEndpoints(HTp slurstart, HTp slurend) {
	string durtag = "slurDuration";
	string endtag = "slurEndId";
	string starttag = "slurStartId";
	string slurstartnumbertag = "slurStartNumber";
	string slurendnumbertag = "slurEndNumber";

	int slurStartCount = slurstart->getValueInt("auto", "slurStartCount");
	int opencount = (int)count(slurstart->begin(), slurstart->end(), '(');
	slurStartCount++;
	int openEnumeration = opencount - slurStartCount + 1;

	if (openEnumeration > 1) {
		endtag += to_string(openEnumeration);
		durtag += to_string(openEnumeration);
		slurendnumbertag += to_string(openEnumeration);
	}

	int slurEndNumber = slurend->getValueInt("auto", "slurEndCount");
	slurEndNumber++;
	int closeEnumeration = slurEndNumber;
	if (closeEnumeration > 1) {
		starttag += to_string(closeEnumeration);
		slurstartnumbertag += to_string(closeEnumeration);
	}

	HumNum duration = slurend->getDurationFromStart()
			- slurstart->getDurationFromStart();

	slurstart->setValue("auto", endtag,            slurend);
	slurstart->setValue("auto", "id",              slurstart);
	slurstart->setValue("auto", slurendnumbertag,  closeEnumeration);
	slurstart->setValue("auto", durtag,            duration);
	slurstart->setValue("auto", "slurStartCount",  slurStartCount);

	slurend->setValue("auto", starttag, slurstart);
	slurend->setValue("auto", "id", slurend);
	slurend->setValue("auto", slurstartnumbertag, openEnumeration);
	slurend->setValue("auto", "slurEndCount",  slurEndNumber);
}





//////////////////////////////
//
// HumdrumFileContent::analyzeKernStemLengths --
//

bool HumdrumFileContent::analyzeKernStemLengths(void) {
	int scount = this->getStrandCount();
	bool output = true;

	vector<vector<int>> centerlines;
	getBaselines(centerlines);
	for (int i=0; i<scount; i++) {
		HTp sstart = this->getStrandStart(i);
		if (!sstart->isKern()) {
			continue;
		}
		HTp send = this->getStrandEnd(i);
		output = output && analyzeKernStemLengths(sstart, send, centerlines);
	}
	return output;
}


bool HumdrumFileContent::analyzeKernStemLengths(HTp stok, HTp etok, vector<vector<int>>& centerlines) {
	HTp tok = stok;
	while (tok && (tok != etok)) {
		if (!tok->isData()) {
			tok = tok->getNextToken();
			continue;
		}
		if (tok->isNull()) {
			tok = tok->getNextToken();
			continue;
		}
		if (tok->isChord()) {
			// don't deal with chords yet
			tok = tok->getNextToken();
			continue;
		}
		if (!tok->isNote()) {
			tok = tok->getNextToken();
			continue;
		}
		int subtrack = tok->getSubtrack();
		if (subtrack == 0) {
			// single voice on staff, so don't process unless it has a stem direction
			// deal with explicit stem direction later.
			tok = tok->getNextToken();
			continue;
		}
		if (subtrack > 2) {
			// 3rd and higher voices will not be processed without stem direction
			// deal with explicit stem direction later.
			tok = tok->getNextToken();
			continue;
		}
		HumNum dur = Convert::recipToDurationNoDots(tok, 8);
		// dur is in units of eighth notes
		if (dur <= 1) {
			// eighth-note or less (could be in beam, so deal with it later)
			tok = tok->getNextToken();
			continue;
		}
		if (dur > 4) {
			// greater than a half-note (no stem)
			tok = tok->getNextToken();
			continue;
		}
		int track = tok->getTrack();
		int b7 = Convert::kernToBase7(tok);
		int diff = b7 - centerlines[track][tok->getLineIndex()];
		if (subtrack == 1) {
			if (diff == 1) { // 0.5 stem length adjustment
				tok->setValue("auto", "stemlen", "6.5");
			} else if (diff == 2) { // 1.0 stem length adjustment
				tok->setValue("auto", "stemlen", "6");
			} else if (diff >= 3) { // 1.5 stem length adjustment
				tok->setValue("auto", "stemlen", "5.5");
			}
		} else if (subtrack == 2) {
			if (diff == -1) { // 0.5 stem length adjustment
				tok->setValue("auto", "stemlen", "6.5");
			} else if (diff == -2) { // 1.0 stem length adjustment
				tok->setValue("auto", "stemlen", "6");
			} else if (diff <= -3) { // 1.5 stem length adjustment
				tok->setValue("auto", "stemlen", "5.5");
			}

		}
		tok = tok->getNextToken();
	}

	return true;
}


//////////////////////////////
//
// HumdrumFileContent::getCenterlines --
//

void HumdrumFileContent::getBaselines(vector<vector<int>>& centerlines) {
	centerlines.resize(this->getTrackCount()+1);

	vector<HTp> kernspines;
	getSpineStartList(kernspines, "**kern");
	int treble = Convert::kernClefToBaseline("*clefG2") + 4;
	int track;

	for (int i=0; i<(int)kernspines.size(); i++) {
		track = kernspines[i]->getTrack();
		centerlines[track].resize(getLineCount());
		for (int j=0; j<getLineCount(); j++) {
			centerlines[track][j] = treble;
		}
	}

	for (int i=0; i<(int)kernspines.size(); i++) {
		HTp tok = kernspines[i];
		int clefcenter = treble;
		while (tok) {
			track = tok->getTrack();
			centerlines[track][tok->getLineIndex()] = clefcenter;
			if (!tok->isClef()) {
				tok = tok->getNextToken();
				continue;
			}
			int centerline = Convert::kernClefToBaseline(tok) + 4;
			centerlines[track][tok->getLineIndex()] = centerline;
			clefcenter = centerline;
			tok = tok->getNextToken();
		}
	}
}




//////////////////////////////
//
// HumdrumFileContent::analyzeTextRepetition -- Look for *ij and *Xij markers
//     that indicate repetition marks.  values added to text:
//      	auto/ij=true: the syllable is in an ij region.
//      	auto/ij-begin=true: the syllable is the first in an ij region.
//      	auto/ij-end=true: the syllable is the last in an ij region.
//
// Returns true if there are any *ij/*Xij markers in the data.
//

bool HumdrumFileContent::analyzeTextRepetition(void) {
	HumdrumFileContent& infile = *this;
	vector<HTp> sstarts;
	infile.getSpineStartList(sstarts);

	bool output = false;
	bool ijstate;
	bool startij;  // true if at the first note in IJ
	HTp lastword;   // non-null if last syllable before *Xij 

	for (int i=0; i<(int)sstarts.size(); i++) {
		ijstate = false;
		startij = false;
		lastword = NULL;
		HTp start = sstarts[i];
		if (!(start->isDataType("**text") || start->isDataType("**sylb"))) {
			continue;
		}
		HTp current = start;
		while (current) {
			if (current->isNull()) {
				current = current->getNextToken();
				continue;;
			}
			if (current->isInterpretation()) {
				if (*current == "*ij") {
					output = true;
					startij = true;
					ijstate = true;
				} else if (*current == "*Xij") {
					output = true;
					startij = false;
					ijstate = false;
					if (lastword) {
						lastword->setValue("auto", "ij-end", "true");
						lastword = NULL;
					}
				}
				current = current->getNextToken();
				continue;
			}
			if (current->isData()) {
				if (ijstate == true) {
					current->setValue("auto", "ij", "true");
					if (startij) {
						current->setValue("auto", "ij-begin", "true");
						startij = false;
					}
					lastword = current;
				}
			}
			current = current->getNextToken();
			continue;
		}
	}

	return output;
}





//////////////////////////////
//
// HumdrumFileContent::analyzeKernTies -- Link start and ends of
//    ties to each other.
//

bool HumdrumFileContent::analyzeKernTies(void) {
	vector<pair<HTp, int>> linkedtiestarts;
	vector<pair<HTp, int>> linkedtieends;

	// vector<HTp> kernspines;
	// getSpineStartList(kernspines, "**kern");
	bool output = true;
	string linkSignifier = m_signifiers.getKernLinkSignifier();
	output = analyzeKernTies(linkedtiestarts, linkedtieends, linkSignifier);
	createLinkedTies(linkedtiestarts, linkedtieends);
	return output;
}


//
// Could be generalized to allow multiple grand-staff pairs by limiting
// the search spines for linking (probably with *part indications).
// Currently all spines are checked for linked ties.
//

bool HumdrumFileContent::analyzeKernTies(
		vector<pair<HTp, int>>& linkedtiestarts,
		vector<pair<HTp, int>>& linkedtieends,
		string& linkSignifier) {

	// Use this in the future to limit to grand-staff search (or 3 staves for organ):
	// vector<vector<HTp> > tracktokens;
	// this->getTrackSeq(tracktokens, spinestart, OPT_DATA | OPT_NOEMPTY);

	// Only analyzing linked ties for now (others ties are handled without analysis in
	// the hum2mei converter, for example.
	if (linkSignifier.empty()) {
		return true;
	}

	string lstart  = linkSignifier + "[";
	string lmiddle = linkSignifier + "_";
	string lend    = linkSignifier + "]";

	vector<pair<HTp, int>> startdatabase(400);

	for (int i=0; i<(int)startdatabase.size(); i++) {
		startdatabase[i].first  = NULL;
		startdatabase[i].second = -1;
	}

	HumdrumFileContent& infile = *this;
	for (int i=0; i<infile.getLineCount(); i++) {
		if (!infile[i].isData()) {
			continue;
		}
		for (int j=0; j<infile[i].getFieldCount(); j++) {
			HTp tok = infile.token(i, j);
			if (!tok->isKern()) {
				continue;
			}
			if (!tok->isData()) {
				continue;
			}
			if (tok->isNull()) {
				continue;
			}
			if (tok->isRest()) {
				continue;
			}
			int scount = tok->getSubtokenCount();
			int b40;
			for (int k=0; k<scount; k++) {
				int index = k;
				if (scount == 1) {
					index = -1;
				}
				std::string tstring = tok->getSubtoken(k);
				if (tstring.find(lstart) != std::string::npos) {
					b40 = Convert::kernToBase40(tstring);
					startdatabase[b40].first  = tok;
					startdatabase[b40].second = index;
					// linkedtiestarts.push_back(std::make_pair(tok, index));
				}
				if (tstring.find(lend) != std::string::npos) {
					b40 = Convert::kernToBase40(tstring);
					if (startdatabase.at(b40).first) {
						linkedtiestarts.push_back(startdatabase[b40]);
						linkedtieends.push_back(std::make_pair(tok, index));
						startdatabase[b40].first  = NULL;
						startdatabase[b40].second = -1;
					}
				}
				if (tstring.find(lmiddle) != std::string::npos) {
					b40 = Convert::kernToBase40(tstring);
					if (startdatabase[b40].first) {
						linkedtiestarts.push_back(startdatabase[b40]);
						linkedtieends.push_back(std::make_pair(tok, index));
					}
					startdatabase[b40].first  = tok;
					startdatabase[b40].second = index;
					// linkedtiestarts.push_back(std::make_pair(tok, index));
					// linkedtieends.push_back(std::make_pair(tok, index));
				}
			}
		}
	}

	return true;
}



//////////////////////////////
//
// HumdrumFileContent::createLinkedTies --
//

void HumdrumFileContent::createLinkedTies(vector<pair<HTp, int>>& linkstarts,
	vector<pair<HTp, int>>& linkends) {
   int max = (int)linkstarts.size();
   if ((int)linkends.size() < max) {
      max = (int)linkends.size();
   }
   if (max == 0) {
      // nothing to do
      return;
   }

   for (int i=0; i<max; i++) {
      linkTieEndpoints(linkstarts[i].first, linkstarts[i].second,
				linkends[i].first, linkends[i].second);
   }

}


//////////////////////////////
//
// HumdrumFileContent::linkTieEndpoints --
//

void HumdrumFileContent::linkTieEndpoints(HTp tiestart,
		int startindex, HTp tieend, int endindex) {

   string durtag   = "tieDuration";
   string starttag = "tieStart";
   string endtag   = "tieEnd";
	string startnum = "tieStartSubtokenNumber";
	string endnum   = "tieEndSubtokenNumber";

	int startnumber = startindex + 1;
	int endnumber   = endindex + 1;

	if (tiestart->isChord()) {
		if (startnumber > 0) {
			durtag   += to_string(startnumber);
			endnum   += to_string(startnumber);
			endtag   += to_string(startnumber);
		}
	}
	if (tieend->isChord()) {
		if (endnumber > 0) {
			starttag += to_string(endnumber);
			startnum += to_string(endnumber);
		}
	}

   tiestart->setValue("auto", endtag, tieend);
   tiestart->setValue("auto", "id", tiestart);
	if (endnumber > 0) {
		tiestart->setValue("auto", endnum, to_string(endnumber));
	}

   tieend->setValue("auto", starttag, tiestart);
   tieend->setValue("auto", "id", tieend);
	if (startnumber > 0) {
		tieend->setValue("auto", startnum, to_string(startnumber));
	}

   HumNum duration = tieend->getDurationFromStart()
         - tiestart->getDurationFromStart();
   tiestart->setValue("auto", durtag, duration);
}




//////////////////////////////
//
// HumdrumFileStructure::getTimeSigs -- Return the prevailing time signature
//     top and bottom for a particular spine for each line in the HumdrumFile.
//     This version does not handle mulimeters such as 2+3/4 or 3/4+6/8.
//     Only checks the primary strand of a spine/track for time signatures.
//
//     default value: track = 0: 0 means use the time signature
//         of the first **kern spine in the file; otherwise, use the
//         time signatures found in the given track (indexed from 1
//         for the first spine on a line).  A value of <0, 0> is used for
//         unassigned time signature lines.
//

void HumdrumFileContent::getTimeSigs(vector<pair<int, HumNum> >& output,
		int track) {
	HumdrumFileStructure& infile = *this;
	int lineCount = infile.getLineCount();
	output.resize(lineCount);
	pair<int, HumNum> current(0, 0);
	fill(output.begin(), output.end(), current);
	if (track == 0) {
		vector<HTp> kernspines = infile.getKernSpineStartList();
		if (kernspines.size() > 0) {
			track = kernspines[0]->getTrack();
		}
	}
	if (track == 0) {
		track = 1;
	}

	int top  = 0;   // top number of time signature (0 for no meter)
	int bot  = 0;   // bottom number of time signature
	int bot2 = 0;   // such as the 2 in 3%2.

	int firstsig  = -1;
	int firstdata = -1;

	HTp token = getTrackStart(track);
	while (token) {
		if (token->isData()) {
			if (firstdata < 0) {
				firstdata = token->getLineIndex();
			}
			token = token->getNextToken();
			continue;
		}
		if (!token->isInterpretation()) {
			token = token->getNextToken();
			continue;
		}
		// check for time signature:
		if (sscanf(token->c_str(), "*M%d/%d%%%d", &top, &bot, &bot2) == 3) {
			current.first = top;
			current.second.setValue(bot, bot2);
			if (firstsig < 0) {
				firstsig = token->getLineIndex();
			}
		} else if (sscanf(token->c_str(), "*M%d/%d", &top, &bot) == 2) {
			current.first = top;
			current.second = bot;
			if (firstsig < 0) {
				firstsig = token->getLineIndex();
			}
		}
		output[token->getLineIndex()] = current;
		token = token->getNextToken();
	}

	// Back-fill the list if the first time signature occurs before
	// the start of the data:
	if ((firstsig > 0) && (firstdata >= firstsig)) {
		current = output[firstsig];
		for (int i=0; i<firstsig; i++) {
			output[i] = current;
		}
	}

	// In-fill the list:
	int starti = firstsig;
	if (starti < 0) {
		starti = 0;
	}
	current = output[starti];
	for (int i=starti+1; i<(int)output.size(); i++) {
		if (output[i].first == 0) {
			output[i] = current;
		} else {
			current = output[i];
		}
	}
}



//////////////////////////////
//
// HumdrumFileContent::HumdrumFileContent --
//

HumdrumFileContent::HumdrumFileContent(void) : HumdrumFileStructure() {
	// do nothing
}


HumdrumFileContent::HumdrumFileContent(const string& filename) :
		HumdrumFileStructure() {
	read(filename);
}


HumdrumFileContent::HumdrumFileContent(istream& contents) :
		HumdrumFileStructure() {
	read(contents);
}



//////////////////////////////
//
// HumdrumFileContent::~HumdrumFileContent --
//

HumdrumFileContent::~HumdrumFileContent() {
	// do nothing
}



//////////////////////////////
//
// HumdrumFileContent::analyzeRScale --
//

bool HumdrumFileContent::analyzeRScale(void) {
	int active = 0; // number of tracks currently having an active rscale parameter
	HumdrumFileBase& infile = *this;
	vector<HumNum> rscales(infile.getMaxTrack() + 1, 1);
	HumRegex hre;
	for (int i=0; i<infile.getLineCount(); i++) {
		if (infile[i].isInterpretation()) {
			int fieldcount = infile[i].getFieldCount();
			for (int j=0; j<fieldcount; j++) {
				HTp token = infile[i].token(j);
				if (token->compare(0, 8, "*rscale:") != 0) {
					continue;
				}
				if (!token->isKern()) {
					continue;
				}
				int track = token->getTrack();
				HumNum value = 1;
				if (hre.search(*token, "\\*rscale:(\\d+)/(\\d+)")) {
					int top = hre.getMatchInt(1);
					int bot = hre.getMatchInt(2);
					value.setValue(top, bot);
				} else if (hre.search(*token, "\\*rscale:(\\d+)")) {
					int top = hre.getMatchInt(1);
					value.setValue(top, 1);
				}
				if (value == 1) {
					if (rscales[track] != 1) {
						rscales[track] = 1;
						active--;
					}
				} else {
					if (rscales[track] == 1) {
						active++;
					}
					rscales[track] = value;
				}
			}
			continue;
		}
		if (!active) {
			continue;
		}
		if (!infile[i].isData()) {
			continue;
		}
		int fieldcount = infile[i].getFieldCount();
		for (int j=0; j<fieldcount; j++) {
			HTp token = infile.token(i, j);
			int track = token->getTrack();
			if (rscales[track] == 1) {
				continue;
			}
			if (!token->isKern()) {
				continue;
			}
			if (token->isNull()) {
				continue;
			}

			int dots = token->getDots();
			HumNum dur = token->getDurationNoDots();
			dur *= rscales[track];
			string vis = Convert::durationToRecip(dur);
			for (int k=0; k<dots; k++) {
				vis += '.';
			}
			token->setValue("LO", "N", "vis", vis);
		}
	}

	return true;
}



//////////////////////////////
//
// HumdrumFileContent::hasPickup -- Return false if there is no pickup measure.
//   Return the barline index number if there is a pickup measure.  A pickup measure
//   is identified when the duration from the start of the file to the first
//   barline is not zero or equal to the duration of the starting time signature.
//   if there is not starting time signature, then there cannot be an identified
//   pickup measure.
//

int HumdrumFileContent::hasPickup(void) {
	HumdrumFileContent& infile = *this;
	int barline = -1;
	HTp tsig = NULL;
	for (int i=0; i<infile.getLineCount(); i++) {
		if (infile[i].isBarline()) {
			if (barline > 0) {
				// second barline found, so stop looking for time signature
				break;
			}
			barline = i;
			continue;
		}
		if (!infile[i].isInterpretation()) {
			continue;
		}
		if (tsig != NULL) {
			continue;
		}
		for (int j=0; j<infile[i].getFieldCount(); j++) {
			HTp token = infile.token(i, j);
			if (token->isTimeSignature()) {
				tsig = token;
				break;
			}
		}
	}
	if (tsig == NULL) {
		// no time signature so return 0
		return 0;
	}
	if (barline < 0) {
		// no barlines in music
		return 0;
	}
	HumNum mdur = infile[barline].getDurationFromStart();
	HumNum tdur = Convert::timeSigToDurationInQuarter(tsig);
	if (mdur == tdur) {
		return 0;
	}
	return barline;
}




//////////////////////////////
//
// HumdrumFileSet::HumdrumFileSet --
//

HumdrumFileSet::HumdrumFileSet(void) {
	// do nothing
}

HumdrumFileSet::HumdrumFileSet(Options& options) {
	read(options);
}

HumdrumFileSet::HumdrumFileSet(const string& contents) {
	readString(contents);
}



//////////////////////////////
//
// HumdrumFileSet::~HumdrumFileSet --
//

HumdrumFileSet::~HumdrumFileSet() {
	clear();
}



//////////////////////////////
//
// HumdrumFileSet::clear -- Remove all Humdrum file content from set.
//

void HumdrumFileSet::clear(void) {
	for (int i=0; i<(int)m_data.size(); i++) {
		delete m_data[i];
		m_data[i] = NULL;
	}
	m_data.resize(0);
}



//////////////////////////////
//
// HumdrumFileSet::clearNoFree -- Remove all Humdrum file content from set
//    but do not delete the contents (since it should be handled externally).
//

void HumdrumFileSet::clearNoFree(void) {
	for (int i=0; i<(int)m_data.size(); i++) {
		m_data[i] = NULL;
	}
	m_data.resize(0);
}



//////////////////////////////
//
// HumdrumFileSet::getSize -- Return the number of Humdrum files in the
//     set.
//

int HumdrumFileSet::getSize(void) {
	return (int)m_data.size();
}



//////////////////////////////
//
// HumdrumFileSet::operator[] -- Return a HumdrumFile.
//

HumdrumFile& HumdrumFileSet::operator[](int index) {
	return *(m_data.at(index));
}



//////////////////////////////
//
// HumdrumFileSet::swap -- Switch position of two scores in the set.
//

bool HumdrumFileSet::swap(int index1, int index2) {
	if (index1 < 0) { return false; }
	if (index2 < 0) { return false; }
	if (index1 >= (int)m_data.size()) { return false; }
	if (index2 >= (int)m_data.size()) { return false; }

	HumdrumFile* temp = m_data[index1];
	m_data[index1] = m_data[index2];
	m_data[index2] = temp;

	return true;
}



//////////////////////////////
//
// HumdrumFileSet::read -- Returns the total number of segments
//

int HumdrumFileSet::readFile(const string& filename) {
	clear();
	return readAppendFile(filename);
}

int HumdrumFileSet::readString(const string& contents) {
	clear();
	return readAppendString(contents);
}

int HumdrumFileSet::readStringCsv(const string& contents) {
	clear();
	return readAppendStringCsv(contents);
}

int HumdrumFileSet::read(istream& inStream) {
	clear();
	return readAppend(inStream);
}

int HumdrumFileSet::read(Options& options) {
	clear();
	return readAppend(options);
}

int HumdrumFileSet::read(HumdrumFileStream& instream) {
	clear();
	return readAppend(instream);
}




//////////////////////////////
//
// HumdrumFileSet::readAppend -- Returns the total number of segments
//    Adds each new HumdrumFile segment to the end of the current data.
//

int HumdrumFileSet::readAppendFile(const string& filename) {
	ifstream indata;
	indata.open(filename);
	string contents((istreambuf_iterator<char>(indata)), istreambuf_iterator<char>());
	HumdrumFileStream instream(contents);
	return readAppend(instream);
}


int HumdrumFileSet::readAppendString(const string& contents) {
	HumdrumFileStream instream(contents);
	return readAppend(instream);
}

int HumdrumFileSet::readAppendStringCsv(const string& contents) {
	cerr << "NOT implemented yet" << endl;
	return 0;
}


int HumdrumFileSet::readAppend(istream& inStream) {
	string contents((istreambuf_iterator<char>(inStream)), istreambuf_iterator<char>());
	HumdrumFileStream instream(contents);
	return readAppend(instream);
}


int HumdrumFileSet::readAppend(Options& options) {
	HumdrumFileStream instream(options);
	return readAppend(instream);
}


int HumdrumFileSet::readAppend(HumdrumFileStream& instream) {
	HumdrumFile* pfile = new HumdrumFile;
	while (instream.read(*pfile)) {
		m_data.push_back(pfile);
		pfile = new HumdrumFile;
	}
	delete pfile;
	return (int)m_data.size();
}


int HumdrumFileSet::readAppendHumdrum(HumdrumFile& infile) {
	stringstream ss;
	ss << infile;
	return readAppendString(ss.str());
}


//////////////////////////////
//
// appendHumdrumPointer --  The infile will be deleted by the object
//    (so do not delete outside of the object or allow it to be inserted
//    from a stack pointer). I.e., this function is dangerous if you do
//    no know what you are doing.
//

int HumdrumFileSet::appendHumdrumPointer(HumdrumFile* infile) {
	m_data.push_back(infile);
	return 1;
}



//////////////////////////////
//
// HumdrumFileSet::hasFilters -- Returns true if has any
//    reference records starting with "!!!!filter:"
//    (universal filters).
//

bool HumdrumFileSet::hasFilters(void) {
	HumdrumFileSet& infiles = *this;
	for (int i=0; i<infiles.getCount(); i++) {
		int lcount = infiles[i].getLineCount();
		for (int j=0; j<lcount; j++) {
			if (!infiles[i][j].isComment()) {
				continue;
			}
			HTp token = infiles[i].token(j, 0);
			if (token->compare(0, 11, "!!!!filter:") == 0) {
				return true;
			}
			if (token->compare(0, 10, "!!!filter:") == 0) {
				return true;
			}
		}
	}
	return false;
}



//////////////////////////////
//
// HumdrumFileSet::hasGlobalFilters -- Returns true if has any
//    reference records starting with "!!!!filter:"
//    (universal filters).
//

bool HumdrumFileSet::hasGlobalFilters(void) {
	HumdrumFileSet& infiles = *this;
	for (int i=0; i<infiles.getCount(); i++) {
		int lcount = infiles[i].getLineCount();
		for (int j=0; j<lcount; j++) {
			if (!infiles[i][j].isComment()) {
				continue;
			}
			HTp token = infiles[i].token(j, 0);
			if (token->compare(0, 10, "!!!filter:") == 0) {
				return true;
			}
		}
	}
	return false;
}



//////////////////////////////
//
// HumdrumFileSet::hasUniversalFilters -- Returns true if has any
//    reference records starting with "!!!!filter:"
//    (universal filters).
//

bool HumdrumFileSet::hasUniversalFilters(void) {
	HumdrumFileSet& infiles = *this;
	for (int i=0; i<infiles.getCount(); i++) {
		int lcount = infiles[i].getLineCount();
		for (int j=0; j<lcount; j++) {
			if (!infiles[i][j].isComment()) {
				continue;
			}
			HTp token = infiles[i].token(j, 0);
			if (token->compare(0, 11, "!!!!filter:") == 0) {
				return true;
			}
		}
	}
	return false;
}



//////////////////////////////
//
// HumdrumFileSet::getUniversalReferenceRecords --
//

vector<HLp> HumdrumFileSet::getUniversalReferenceRecords(void) {
	vector<HLp> hlps;
	hlps.reserve(32);
	HLp hlp;
	HumdrumFileSet& infiles = *this;
	for (int i=0; i<infiles.getCount(); i++) {
		HumdrumFileBase& infile = infiles[i];
		for (int j=0; j<infile.getLineCount(); j++) {
			if (infile[j].isUniversalReference()) {
				hlp = &infile[j];
				hlps.push_back(hlp);
			}
		}
	}
	return hlps;
}





//////////////////////////////
//
// HumdrumFileStream::HumdrumFileStream --
//

HumdrumFileStream::HumdrumFileStream(void) {
	m_curfile = -1;
}

HumdrumFileStream::HumdrumFileStream(char** list) {
	m_curfile = -1;
	setFileList(list);
}

HumdrumFileStream::HumdrumFileStream(const vector<string>& list) {
	m_curfile = -1;
	setFileList(list);
}

HumdrumFileStream::HumdrumFileStream(Options& options) {
	m_curfile = -1;
	vector<string> list;
	options.getArgList(list);
	setFileList(list);
}

HumdrumFileStream::HumdrumFileStream(const string& datastring) {
	m_curfile = -1;
	m_stringbuffer << datastring;
}



//////////////////////////////
//
// HumdrumFileStream::clear -- reset the contents of the class.
//

void HumdrumFileStream::clear(void) {
	m_curfile = 0;
	m_filelist.resize(0);
	m_universals.resize(0);
	m_newfilebuffer.resize(0);
	// m_stringbuffer.clear(0);
	m_stringbuffer.str("");
}



//////////////////////////////
//
// HumdrumFileStream::setFileList --
//

int HumdrumFileStream::setFileList(char** list) {
	m_filelist.reserve(1000);
	m_filelist.resize(0);
	int i = 0;
	while (list[i] != NULL) {
		m_filelist.push_back(list[i]);
		i++;
	}
	return i;
}


int HumdrumFileStream::setFileList(const vector<string>& list) {
	m_filelist = list;
	return (int)list.size();
}



//////////////////////////////
//
// HumdrumFileStream::loadString --
//

void HumdrumFileStream::loadString(const string& data) {
	m_curfile = -1;
	m_stringbuffer << data;
}



//////////////////////////////
//
// HumdrumFileStream::read -- alias for getFile.
//

int HumdrumFileStream::read(HumdrumFile& infile) {
	return getFile(infile);
}


int HumdrumFileStream::read(HumdrumFileSet& infiles) {
	infiles.clear();
	HumdrumFile* infile = new HumdrumFile;
	while (getFile(*infile)) {
		infiles.appendHumdrumPointer(infile);
		infile = new HumdrumFile;
	}
	delete infile;
	return 0;
}



//////////////////////////////
//
// HumdrumFileStream::readSingleSegment -- Get a single file for a set structure.
//

int HumdrumFileStream::readSingleSegment(HumdrumFileSet& infiles) {
	infiles.clear();
	HumdrumFile* infile = new HumdrumFile;
	int status = getFile(*infile);
	if (!status) {
		delete infile;
	} else {
		infiles.appendHumdrumPointer(infile);
	}
	return status;
}



//////////////////////////////
//
// HumdrumFileStream::eof -- returns true if there is no more segements
//     to read from the input source(s).
//

int HumdrumFileStream::eof(void) {
	istream* newinput = NULL;

	// Read HumdrumFile contents from:
	// (1) Current ifstream if open
	// (2) Next filename if ifstream is done
	// (3) cin if no ifstream open and no filenames

	// (1) Is an ifstream open?, then yes, there is more data to read.
	if (m_instream.is_open() && !m_instream.eof()) {
		return 0;
	}

	// (1b) Is the URL data buffer open?
	else if (m_urlbuffer.str() != "") {
		return 0;
	}

	// (2) If ifstream is closed but there is a file to be processed,
	// load it into the ifstream and start processing it immediately.
	else if ((m_filelist.size() > 0) && (m_curfile < (int)m_filelist.size()-1)) {
		return 0;
	} else {
		// no input fstream open and no list of files to process, so
		// start (or continue) reading from standard input.
		if (m_curfile < 0) {
			// but only read from cin if no files have previously been read
			newinput = &cin;
		}
		if ((newinput != NULL) && newinput->eof()) {
			return 1;
		}
	}

	return 1;
}



//////////////////////////////
//
// HumdrumFileStream::getFile -- fills a HumdrumFile class with content
//    from the input stream or next input file in the list.  Returns
//    true if content was extracted, fails if there is no more HumdrumFiles
//    in the input stream.
//

int HumdrumFileStream::getFile(HumdrumFile& infile) {
	infile.clear();
	istream* newinput = NULL;

restarting:

	newinput = NULL;

	if (m_urlbuffer.eof()) {
		// If the URL buffer is at its end, clear the buffer.
		m_urlbuffer.str("");
	}

	// Read HumdrumFile contents from:
	// (1) Read from string buffer
	// (2) Current ifstream if open
	// (3) Next filename if ifstream is done
	// (4) cin if no ifstream open and no filenames

	// (1) Is there content in the string buffer?
	if (!m_stringbuffer.str().empty()) {
		newinput = &m_stringbuffer;
	}

	// (2) Is an ifstream open?
	else if (m_instream.is_open() && !m_instream.eof()) {
		newinput = &m_instream;
	}

	// (2b) Is the URL data buffer open?
	else if (m_urlbuffer.str() != "") {
		m_urlbuffer.clear();
		newinput = &m_urlbuffer;
	}

	// (3) If ifstream is closed but there is a file to be processed,
	// load it into the ifstream and start processing it immediately.
	else if (((int)m_filelist.size() > 0) &&
			(m_curfile < (int)m_filelist.size()-1)) {
		m_curfile++;
		if (m_instream.is_open()) {
			m_instream.close();
		}
		if (strstr(m_filelist[m_curfile].c_str(), "://") != NULL) {
			// The next file to read is a URL/URI, so buffer the
			// data from the internet and start reading that instead
			// of reading from a file on the hard disk.
			fillUrlBuffer(m_urlbuffer, m_filelist[m_curfile].c_str());
			infile.setFilename(m_filelist[m_curfile].c_str());
			goto restarting;
		}
		m_instream.open(m_filelist[m_curfile].c_str());
		infile.setFilename(m_filelist[m_curfile].c_str());
		if (!m_instream.is_open()) {
			// file does not exist or cannot be opened close
			// the file and try luck with next file in the list
			// (perhaps given an error or warning?).
			infile.setFilename("");
			m_instream.close();
			goto restarting;
		}
		newinput = &m_instream;
	} else {
		// no input fstream open and no list of files to process, so
		// start (or continue) reading from standard input.
		if (m_curfile < 0) {
			// but only read from cin if no files have previously been read
			newinput = &cin;
		}
	}

	// At this point the newinput istream is set to read from the given
	// file or from standard input, so start reading Humdrum content.
	// If there is "m_newfilebuffer" content, then set the filename of the
	// HumdrumFile to that value.

	if (m_newfilebuffer.size() > 0) {
		// store the filename for the current HumdrumFile being read:
		HumRegex hre;
		if (hre.search(m_newfilebuffer,
				R"(^!!!!SEGMENT\s*([+-]?\d+)?\s*:\s*(.*)\s*$)")) {
			if (hre.getMatchLength(1) > 0) {
				infile.setSegmentLevel(atoi(hre.getMatch(1).c_str()));
			} else {
				infile.setSegmentLevel(0);
			}
			infile.setFilename(hre.getMatch(2));
		} else if ((m_curfile >=0) && (m_curfile < (int)m_filelist.size())
				&& (m_filelist.size() > 0)) {
			infile.setFilename(m_filelist[m_curfile].c_str());
		} else {
			// reading from standard input, but no name.
		}
	}

	if (newinput == NULL) {
		// something strange happened, or no more files to read.
		return 0;
	}

	stringstream buffer;
	int foundUniversalQ = 0;

	// Start reading the input stream.  If !!!!SEGMENT: universal comment
	// is found, then store that line in m_newfilebuffer and return the
	// newly read HumdrumFile.  If other universal comments are found, then
	// overwrite the old universal comments here.

	int addedFilename = 0;
	//int searchName = 0;
	int dataFoundQ = 0;
	int starstarFoundQ = 0;
	int starminusFoundQ = 0;
	if (m_newfilebuffer.size() < 4) {
		//searchName = 1;
	}
	char templine[123123] = {0};

	if (newinput->eof()) {
		if (m_curfile < (int)m_filelist.size()-1) {
			m_curfile++;
			goto restarting;
		}
		// input stream is close and there is no more files to process.
		return 0;
	}

	istream& input = *newinput;

	// if the previous line from the last read starts with "**"
	// then treat it as part of the current file.
	if ((m_newfilebuffer.size() > 1) &&
		 (strncmp(m_newfilebuffer.c_str(), "**", 2)) == 0) {
		buffer << m_newfilebuffer << "\n";
		m_newfilebuffer = "";
		starstarFoundQ = 1;
	}

	while (!input.eof()) {
		input.getline(templine, 123123, '\n');
		if ((!dataFoundQ) &&
				(strncmp(templine, "!!!!SEGMENT", strlen("!!!!SEGMENT")) == 0)) {
			string tempstring;
			tempstring = templine;
			HumRegex hre;
			if (hre.search(tempstring,
					"^!!!!SEGMENT\\s*([+-]?\\d+)?\\s*:\\s*(.*)\\s*$")) {
				if (hre.getMatchLength(1) > 0) {
					infile.setSegmentLevel(atoi(hre.getMatch(1).c_str()));
				} else {
					infile.setSegmentLevel(0);
				}
				infile.setFilename(hre.getMatch(2));
			}
		}

		if (strncmp(templine, "**", 2) == 0) {
			if (starstarFoundQ == 1) {
				m_newfilebuffer = templine;
				// already found a **, so this one is defined as a file
				// segment.  Exit from the loop and process the previous
				// content, waiting until the next read for to start with
				// this line.
				break;
			}
			starstarFoundQ = 1;
		}

		if (input.eof() && (strcmp(templine, "") == 0)) {
			// No more data coming from current stream, so this is
			// the end of the HumdrumFile.  Break from the while loop
			// and then store the read contents of the stream in the
			// HumdrumFile.
			break;
		}
		// (1) Does the line start with "!!!!SEGMENT"?  If so, then
		// this is either the name of the current or next file to process.
		// (1a) this is the name of the current file to process if no
		// data has yet been found,
		// (1b) or a name is being actively searched for.
		if (strncmp(templine, "!!!!SEGMENT", strlen("!!!!SEGMENT")) == 0) {
			m_newfilebuffer = templine;
			if (dataFoundQ) {
				// this new filename is for the next chunk to process in the
				// current file stream, not this one, so stop reading the
				// HumdrumFile content and send what has already been read back
				// out with new contents.
			}  else {
				// !!!!SEGMENT: came before any real data was read, so
				// it is most likely the name of the current file
				// (i.e., it comes at the start of the file stream and
				// is the name of the first HumdrumFile in the stream).
				HumRegex hre;
				if (hre.search(m_newfilebuffer,
						R"(^!!!!SEGMENT\s*([+-]?\d+)?\s:\s*(.*)\s*$)")) {
					if (hre.getMatchLength(1) > 0) {
						infile.setSegmentLevel(atoi(hre.getMatch(1).c_str()));
					} else {
						infile.setSegmentLevel(0);
					}
					infile.setFilename(hre.getMatch(2));
				}
			}
		}
		int len = (int)strlen(templine);
		if ((len > 4) && (strncmp(templine, "!!!!", 4) == 0) &&
				(templine[4] != '!') &&
				(dataFoundQ == 0) &&
				(strncmp(templine, "!!!!filter:", strlen("!!!!filter:")) != 0) &&
				(strncmp(templine, "!!!!SEGMENT:", strlen("!!!!SEGMENT:")) != 0)) {
			// This is a universal comment.  Should it be appended
			// to the list or should the current list be erased and
			// this record placed into the first entry?
			if (foundUniversalQ) {
				// already found a previous universal, so append.
				m_universals.push_back(templine);
			} else {
				// new universal comment, to delete all previous
				// universal comments and store this one.
				m_universals.reserve(1000);
				m_universals.resize(1);
				m_universals[0] = templine;
				foundUniversalQ = 1;
			}
			continue;
		}

		if (strncmp(templine, "*-", 2) == 0) {
			starminusFoundQ = 1;
		}

		// if before first ** in a data file or after *-, and the line
		// does not start with '!' or '*', then assume that it is a file
		// name which should be added to the file list to read.
		if (((starminusFoundQ == 1) || (starstarFoundQ == 0))
				&& (templine[0] != '*') && (templine[0] != '!')) {
			if ((templine[0] != '\0') && (templine[0] != ' ')) {
				// The file can only be added once in this manner
				// so that infinite loops are prevented.
				int found = 0;
				for (int mm=0; mm<(int)m_filelist.size(); mm++) {
					if (strcmp(m_filelist[mm].c_str(), templine) == 0) {
						found = 1;
					}
				}
				if (!found) {
					m_filelist.push_back(templine);
					addedFilename = 1;
				}
				continue;
			}
		}

		dataFoundQ = 1; // found something other than universal comments
		// should empty lines be treated somewhat as universal comments?

		// store the data line for later parsing into HumdrumFile record:
		buffer << templine << "\n";
	}

	if (dataFoundQ == 0) {
		// never found anything for some strange reason.
		if (addedFilename) {
			goto restarting;
		}
		return 0;
	}

	// Arriving here means that reading of the data stream is complete.
	// The string stream variable "buffer" contains the HumdrumFile
	// content, so send it to the HumdrumFile variable.  Also, prepend
	// Universal comments (demoted into Global comments) at the start
	// of the data stream (maybe allow for postpending Universal comments
	// in the future).
	stringstream contents;
	contents.str(""); // empty any contents in buffer
	contents.clear(); // reset error flags in buffer

	for (int i=0; i<(int)m_universals.size(); i++) {
		// Convert universals reference records to globals, but do not demote !!!!filter:
		if (m_universals[i].compare(0, 11, "!!!!filter:") == 0) {
			continue;
		}
		contents << &(m_universals[i][1]) << "\n";
	}
	contents << buffer.str();
	string filename = infile.getFilename();
	infile.readNoRhythm(contents);
	if (!filename.empty()) {
		infile.setFilename(filename);
	}
	return 1;
}


//////////////////////////////
//
// HumdrumFileStream::fillUrlBuffer --
//


void HumdrumFileStream::fillUrlBuffer(stringstream& uribuffer,
		const string& uriname) {
	#ifdef USING_URI
		uribuffer.str(""); // empty any contents in buffer
		uribuffer.clear(); // reset error flags in buffer
		string webaddress = HumdrumFileBase::getUriToUrlMapping(uriname);
		HumdrumFileBase::readStringFromHttpUri(uribuffer, webaddress);
	#endif
}





//////////////////////////////
//
// HumdrumFileStructure::analyzeStropheMarkers -- Merge this
//    with analyzeStrophes (below) at some point.
//

void HumdrumFileStructure::analyzeStropheMarkers(void) {
	m_analyses.m_strophes_analyzed = true;

	m_strophes1d.clear();
	m_strophes2d.clear();
	int spines = getSpineCount();
	m_strophes2d.resize(spines);

	map<string, HTp> laststrophe;

	HumdrumFileStructure& infile = *this;

	for (int i=0; i<infile.getLineCount(); i++) {
		if (!infile[i].isInterpretation()) {
			continue;
		}
		for (int j=0; j<infile[i].getFieldCount(); j++) {
			HTp token = infile.token(i, j);
			if (*token == "*strophe") {
				string spineinfo = token->getSpineInfo();
	 			HTp lastone = laststrophe[spineinfo];
				if (lastone) {
					// Improperly terminated strophe,
					// ending here and starting a new one.
					TokenPair tp;
					tp.first = lastone;
					tp.last = token;
					m_strophes1d.push_back(tp);
					int spine = token->getTrack() - 1;
					m_strophes2d.at(spine).push_back(tp);
					laststrophe[spineinfo] = token;
				} else {
					// Store the new strophe.
					laststrophe[spineinfo] = token;
				}
			} else if (*token == "*Xstrophe") {
				string spineinfo = token->getSpineInfo();
	 			HTp lastone = laststrophe[spineinfo];
				if (lastone) {
					TokenPair tp;
					tp.first = lastone;
					tp.last = token;
					m_strophes1d.push_back(tp);
					int spine = token->getTrack() - 1;
					m_strophes2d.at(spine).push_back(tp);
					laststrophe[spineinfo] = NULL;
				} else {
					// Improperly placed *Xstrophe, so ignore
					cerr << "WARNING: unmatched strophe end: " << token << " ON LINE " << token->getLineNumber() << endl;
				}
			}
		}
	}

	// Warn about any improperly terminated *strophe:
	for (auto it = laststrophe.begin(); it != laststrophe.end(); ++it) {
		HTp token = it->second;
		if (token != NULL) {
			cerr << "WARNING: unmatched strophe begin: " << token << " ON LINE " << token->getLineNumber() << endl;
		}
	}
}



//////////////////////////////
//
// HumdrumFileStructure::analyzeStrophes --
//

bool HumdrumFileStructure::analyzeStrophes(void) {
	if (!m_analyses.m_strands_analyzed) {
		analyzeStrands();
	}
	analyzeStropheMarkers();

	int scount = (int)m_strand1d.size();
	// bool dataQ;
	vector<HTp> strophestarts;
	strophestarts.reserve(100);
	for (int i=0; i<scount; i++) {
		// dataQ = false;
		HTp current = m_strand1d.at(i).first;
		HTp send = m_strand1d.at(i).last;
		if (!send) {
			continue;
		}
		while (current && (current != send)) {
			if (!current->isInterpretation()) {
				// not a strophe (data not allowed in subspine before strophe marker).
				break;
			}

			if (current->compare(0, 3, "*S/") == 0) {
				int track = current->getTrack();
				HTp first = current->getPreviousFieldToken();
				if (first) {
					int trackp = first->getTrack();
					if (track == trackp) {
						if (first->compare(0, 3, "*S/") == 0) {
							bool found = false;
							for (int j=0; j<(int)strophestarts.size(); j++) {
								if (strophestarts[j] == first) {
									found = true;
									break;
								}
							}
							if (!found) {
								strophestarts.push_back(first);
							}
						}
					}
				}
				bool found = false;
				for (int j=0; j<(int)strophestarts.size(); j++) {
					if (strophestarts[j] == current) {
						found = true;
						break;
					}
				}
				if (!found) {
					strophestarts.push_back(current);
				}
				break;
			}
			current = current->getNextToken();
		}
	}

	// Now store strophe information in tokens.  Currently
	// spine splits are not allowed in strophes.  Spine merges
	// are OK: the first strophe will dominate in a merge.
	for (int i=0; i<(int)strophestarts.size(); i++) {
		HTp current = strophestarts[i];
		if (current->hasStrophe()) {
			continue;
		}
		current->setStrophe(strophestarts[i]);
		current = current->getNextToken();
		while (current) {
			if (current->hasStrophe()) {
				break;
			}
			if (*current == "*Xstrophe") {
				break;
			}
			current->setStrophe(strophestarts[i]);
			current = current->getNextToken();
		}
	}

	return true;
}



//////////////////////////////
//
// HumdrumFileStructure::getStropheCount --  Return the number of
//    strophes in the file (for void input), or return the number
//    of strophes for a particular spine (for int input).
//

int HumdrumFileStructure::getStropheCount(void) {
	return (int)m_strophes1d.size();
}


int HumdrumFileStructure::getStropheCount(int spineindex) {
	if ((spineindex < 0) || (spineindex >= (int)m_strophes2d.size())) {
		return 0;
	}
	return (int)m_strophes2d.at(spineindex).size();
}



//////////////////////////////
//
// HumdrumFileStructure::getStropheStart --
//

HTp HumdrumFileStructure::getStropheStart(int index) {
	if ((index < 0) || (index >= (int)m_strophes1d.size())) {
		return NULL;
	}
	return m_strophes1d.at(index).first;
}

HTp HumdrumFileStructure::getStropheStart(int spine, int index) { 
		if ((spine < 0) || (index < 0)) {
			return NULL;
		}
		if (spine >= (int)m_strophes2d.size()) {
			return NULL;
		}
		if (index >= (int)m_strophes2d.at(spine).size()) {
			return NULL;
		}
		return m_strophes2d.at(spine).at(index).first;
}



//////////////////////////////
//
// HumdrumFileStructure::getStropheEnd --
//

HTp HumdrumFileStructure::getStropheEnd(int index) {
	if ((index < 0) || (index >= (int)m_strophes1d.size())) {
		return NULL;
	}
	return m_strophes1d.at(index).last;
}


HTp HumdrumFileStructure::getStropheEnd(int spine, int index) {
		if ((spine < 0) || (index < 0)) {
			return NULL;
		}
		if (spine >= (int)m_strophes2d.size()) {
			return NULL;
		}
		if (index >= (int)m_strophes2d.at(spine).size()) {
			return NULL;
		}
		return m_strophes2d.at(spine).at(index).last;
}




//////////////////////////////
//
// HumdrumFileStructure::HumdrumFileStructure -- HumdrumFileStructure
//     constructor.
//

HumdrumFileStructure::HumdrumFileStructure(void) : HumdrumFileBase() {
	// do nothing
}

HumdrumFileStructure::HumdrumFileStructure(const string& filename) :
		HumdrumFileBase() {
	read(filename);
}

HumdrumFileStructure::HumdrumFileStructure(istream& contents) :
		HumdrumFileBase() {
	read(contents);
}



//////////////////////////////
//
// HumdrumFileStructure::~HumdrumFileStructure -- HumdrumFileStructure
//     deconstructor.
//

HumdrumFileStructure::~HumdrumFileStructure() {
	// do nothing
}



//////////////////////////////
//
// HumdrumFileStructure::read --  Read the contents of a file from a file or
//   istream.  The file's structure is analyzed, and then the rhythmic structure
//   is calculated.
//


bool HumdrumFileStructure::read(istream& contents) {
	m_displayError = false;
	if (!readNoRhythm(contents)) {
		return isValid();
	}
	return analyzeStructure();
}


bool HumdrumFileStructure::read(const char* filename) {
	m_displayError = false;
	if (!readNoRhythm(filename)) {
		return isValid();
	}
	return analyzeStructure();
}


bool HumdrumFileStructure::read(const string& filename) {
	m_displayError = false;
	if (!readNoRhythm(filename)) {
		return isValid();
	}
	return analyzeStructure();
}



//////////////////////////////
//
// HumdrumFileStructure::readCsv --  Read the contents of a file from a file or
//   istream in CSV format.  The file's structure is analyzed, and then the
//   rhythmic structure is calculated.
// default value: separator = ","
//


bool HumdrumFileStructure::readCsv(istream& contents,
		const string& separator) {
	m_displayError = false;
	if (!readNoRhythmCsv(contents, separator)) {
		return isValid();
	}
	return analyzeStructure();
}


bool HumdrumFileStructure::readCsv(const char* filename,
		const string& separator) {
	m_displayError = false;
	if (!readNoRhythmCsv(filename, separator)) {
		return isValid();
	}
	return analyzeStructure();
}


bool HumdrumFileStructure::readCsv(const string& filename,
		const string& separator) {
	m_displayError = false;
	if (!readNoRhythmCsv(filename, separator)) {
		return isValid();
	}
	return analyzeStructure();
}



//////////////////////////////
//
// HumdrumFileStructure::readString -- Read the contents from a string.
//    Similar to HumdrumFileStructure::read, but for string data.
//

bool HumdrumFileStructure::readString(const char* contents) {
	m_displayError = false;
	if (!HumdrumFileBase::readString(contents)) {
		return isValid();
	}
	return analyzeStructure();
}


bool HumdrumFileStructure::readString(const string& contents) {
	m_displayError = false;
	if (!HumdrumFileBase::readString(contents)) {
		return isValid();
	}
	return analyzeStructure();
}



//////////////////////////////
//
// HumdrumFileStructure::readStringCsv -- Read the contents from a string.
//    Similar to HumdrumFileStructure::read, but for string data.
// default value: separator = ","
//

bool HumdrumFileStructure::readStringCsv(const char* contents,
		const string& separator) {
	m_displayError = false;
	if (!HumdrumFileBase::readStringCsv(contents, separator)) {
		return isValid();
	}
	return analyzeStructure();
}


bool HumdrumFileStructure::readStringCsv(const string& contents,
		const string& separator) {
	m_displayError = false;
	if (!HumdrumFileBase::readStringCsv(contents, separator)) {
		return isValid();
	}
	return analyzeStructure();
}



//////////////////////////////
//
// HumdrumFileStructure::analyzeStructure -- Analyze global/local
//    parameters and rhythmic structure.
//

bool HumdrumFileStructure::analyzeStructure(void) {
	m_analyses.m_structure_analyzed = false;
	if (!m_analyses.m_strands_analyzed) {
		if (!analyzeStrands()       ) { return isValid(); }
	}
	if (!analyzeGlobalParameters() ) { return isValid(); }
	if (!analyzeLocalParameters()  ) { return isValid(); }
	if (!analyzeTokenDurations()   ) { return isValid(); }
	if (!analyzeTokenDurations()   ) { return isValid(); }
	m_analyses.m_structure_analyzed = true;
	if (!analyzeRhythmStructure()  ) { return isValid(); }
	analyzeSignifiers();
	return isValid();
}



//////////////////////////////
//
// HumdrumFileStructure::analyzeStructureNoRhythm -- Analyze global/local
//    parameters but not rhythmic structure.
//

bool HumdrumFileStructure::analyzeStructureNoRhythm(void) {
	m_analyses.m_structure_analyzed = true;
	if (!m_analyses.m_strands_analyzed) {
		if (!analyzeStrands()          ) { return isValid(); }
	}
	if (!analyzeGlobalParameters() ) { return isValid(); }
	if (!analyzeLocalParameters()  ) { return isValid(); }
	if (!analyzeTokenDurations()   ) { return isValid(); }
	analyzeSignifiers();
	return isValid();
}



/////////////////////////////
//
// HumdrumFileStructure::analyzeRhythmStructure --
//

bool HumdrumFileStructure::analyzeRhythmStructure(void) {
	m_analyses.m_rhythm_analyzed = true;
	setLineRhythmAnalyzed();
	if (!isStructureAnalyzed()) {
		if (!analyzeStructureNoRhythm()) { return isValid(); }
	}

	HTp firstspine = getSpineStart(0);
	if (firstspine && firstspine->isDataType("**recip")) {
		assignRhythmFromRecip(firstspine);
	} else {
		if (!analyzeRhythm()           ) { return isValid(); }
		if (!analyzeDurationsOfNonRhythmicSpines()) { return isValid(); }
	}
	return isValid();
}



//////////////////////////////
//
// HumdrumFileStructure::assignRhythmFromRecip --
//

bool HumdrumFileStructure::assignRhythmFromRecip(HTp spinestart) {
	HTp current = spinestart;

	HumNum duration;
	while (current) {
		if (!current->isData()) {
			current = current->getNextToken();
			continue;
		}
		if (current->isNull()) {
			// This should not occur in a well-formed **recip spine, but
			// treat as a zero duration.
			current = current->getNextToken();
			continue;
		}

		if (strchr(current->c_str(), 'q') != NULL) {
			duration = 0;
		} else {
			duration = Convert::recipToDuration((string)*current);
		}
		current->getLine()->setDuration(duration);
		current = current->getNextToken();
	}

	// now go back and set the absolute position from the start of
	// the file.
	HumNum sum = 0;
	HumdrumFileStructure& hfile = *this;
	for (int i=0; i<getLineCount(); i++) {
		hfile[i].setDurationFromStart(sum);
		if (hfile[i].getDuration() < 0) {
			hfile[i].setDuration(0);
		}
		sum += hfile[i].getDuration();
	}

	// Analyze durations to/from barlines:
	if (!analyzeMeter()) { return false; }
	if (!analyzeNonNullDataTokens()) { return false; }
	return true;
}



//////////////////////////////
//
// HumdrumFileStructure::readNoRhythm -- Similar to the read() functions, but
//    does not parse rhythm (or parameters).
//

bool HumdrumFileStructure::readNoRhythm(istream& infile) {
	return HumdrumFileBase::read(infile);
}


bool HumdrumFileStructure::readNoRhythm(const char* filename) {
	return HumdrumFileBase::read(filename);
}


bool HumdrumFileStructure::readNoRhythm(const string& filename) {
	return HumdrumFileBase::read(filename);
}



//////////////////////////////
//
// HumdrumFileStructure::readNoRhythmCsv -- Similar to the readCsv()
//    functions, but does not parse rhythm (or parameters).
// default value: separator = ","
//

bool HumdrumFileStructure::readNoRhythmCsv(istream& infile,
		const string& seperator) {
	return HumdrumFileBase::readCsv(infile);
}


bool HumdrumFileStructure::readNoRhythmCsv(const char* filename,
		const string& seperator) {
	return HumdrumFileBase::readCsv(filename);
}


bool HumdrumFileStructure::readNoRhythmCsv(const string& filename,
		const string& seperator) {
	return HumdrumFileBase::readCsv(filename);
}



//////////////////////////////
//
// HumdrumFileStructure::readStringNoRhythm -- Read a string, but
//   do not analyze the rhythm (or parameters) in the read data.
//


bool HumdrumFileStructure::readStringNoRhythm(const char* contents) {
	return HumdrumFileBase::readString(contents);
}


bool HumdrumFileStructure::readStringNoRhythm(const string& contents) {
	return HumdrumFileBase::readString(contents);
}



//////////////////////////////
//
// HumdrumFileStructure::readStringNoRhythmCsv -- Read a string, but
//   do not analyze the rhythm (or parameters) in the read data.
// default value: separator = ","
//


bool HumdrumFileStructure::readStringNoRhythmCsv(const char* contents,
		const string& separator) {
	return HumdrumFileBase::readStringCsv(contents);
}


bool HumdrumFileStructure::readStringNoRhythmCsv(const string& contents,
		const string& separator) {
	return HumdrumFileBase::readStringCsv(contents);
}



//////////////////////////////
//
// HumdrumFileStructure::getScoreDuration -- Return the total duration
//    of the score in quarter note units.  Returns zero if no lines in the
//    file, or -1 if there are lines, but no rhythmic analysis has been done.
//

HumNum HumdrumFileStructure::getScoreDuration(void) const {
	if (m_lines.size() == 0) {
		return 0;
	}
	return m_lines.back()->getDurationFromStart();
}



//////////////////////////////
//
// HumdrumFileStructure::tpq -- "Ticks per Quarter-note".  Returns the minimal
//    number of integral time units that divide a quarter note into equal
//    subdivisions.  This value is needed to convert Humdrum data into
//    MIDI file data, MuseData, and MusicXML data.  Also useful for timebase
//    type of operations on the data and describing the durations in terms
//    of integers rather than with fractions.  This function will also
//    consider the implicit durations of non-rhythmic spine data.
//

int HumdrumFileStructure::tpq(void) {
	if (m_ticksperquarternote > 0) {
		return m_ticksperquarternote;
	}
	set<HumNum> durlist = getPositiveLineDurations();
	vector<int> dems;
	for (auto& it : durlist) {
		if (it.getDenominator() > 1) {
			dems.push_back(it.getDenominator());
		}
	}
	int lcm = 1;
	if (dems.size() > 0) {
		lcm = Convert::getLcm(dems);
	}
	m_ticksperquarternote = lcm;
	return m_ticksperquarternote;
}



//////////////////////////////
//
// HumdrumFileStructure::getPositiveLineDurations -- Return a list of all
//    unique token durations in the file.  This function could be expanded
//    to limit the search to a range of lines or to a specific track.
//

set<HumNum> HumdrumFileStructure::getPositiveLineDurations(void) {
	set<HumNum> output;
	for (auto& line : m_lines) {
		if (line->getDuration().isPositive()) {
			output.insert(line->getDuration());
		}
	}
	return output;
}



//////////////////////////////
//
// HumdrumFileStructure::printDurationInfo -- Print the assigned duration
//    of each line in a file.  Useful for debugging.
//

ostream& HumdrumFileStructure::printDurationInfo(ostream& out) {
	for (int i=0; i<getLineCount(); i++) {
		m_lines[i]->printDurationInfo(out) << '\n';
	}
	return out;
}



//////////////////////////////
//
// HumdrumFileStructure::getBarline -- Return the given barline from the file
//   based on the index number.  Negative index accesses from the end of the
//   list.  If the first barline is a pickup measure, then the returned
//   HLp will not be an actual barline line.
//

HLp HumdrumFileStructure::getBarline(int index) const {
	if (index < 0) {
		index += (int)m_barlines.size();
	}
	if (index < 0) {
		return NULL;
	}
	if (index >= (int)m_barlines.size()) {
		return NULL;
	}
	return m_barlines[index];
}



//////////////////////////////
//
// HumdrumFileStructure::getBarlineCount -- Return the number of barlines in
//   the file.  If there is a pickup beat, then the count includes an imaginary
//   barline before the first pickup (and the start of the file will be returned
//   for barline(0).
//

int HumdrumFileStructure::getBarlineCount(void) const {
	return (int)m_barlines.size();
}



///////////////////////////////
//
// HumdrumFileStructure::getBarlineDuration --  Return the duration from the
//    current barline to the next barline in the data.  For the last barline,
//    the duration will be calculated from the end of the data;  The final
//    will have a duration of 0 if there are not notes after the barline
//    in the data.
//

HumNum HumdrumFileStructure::getBarlineDuration(int index) const {
	if (index < 0) {
		index += (int)m_barlines.size();
	}
	if (index < 0) {
		return 0;
	}
	if (index >= (int)m_barlines.size()) {
		return 0;
	}
	HumNum startdur = m_barlines[index]->getDurationFromStart();
	HumNum enddur;
	if (index + 1 < (int)m_barlines.size() - 1) {
		enddur = m_barlines[index+1]->getDurationFromStart();
	} else {
		enddur = getScoreDuration();
	}
	return enddur - startdur;
}



///////////////////////////////
//
// HumdrumFileStructure::getBarlineDurationFromStart -- Return the duration
//    between the start of the Humdrum file and the given barline.
//

HumNum HumdrumFileStructure::getBarlineDurationFromStart(int index) const {
	if (index < 0) {
		index += (int)m_barlines.size();
	}
	if (index < 0) {
		return 0;
	}
	if (index >= (int)m_barlines.size()) {
		return getScoreDuration();
	}
	return m_barlines[index]->getDurationFromStart();
}



///////////////////////////////
//
// HumdrumFileStructure::getBarlineDurationToEnd -- Return the duration
//    between barline and the end of the HumdrumFileStructure.
//

HumNum HumdrumFileStructure::getBarlineDurationToEnd(int index) const {
	if (index < 0) {
		index += (int)m_barlines.size();
	}
	if (index < 0) {
		return 0;
	}
	if (index >= (int)m_barlines.size()) {
		return getScoreDuration();
	}
	return m_barlines[index]->getDurationToEnd();
}


//////////////////////////////
//
// HumdrumFileStructure::setLineRhythmAnalyzed --
//

void HumdrumFileStructure::setLineRhythmAnalyzed(void) {
	for (int i=0; i<(int)m_lines.size(); i++) {
		m_lines[i]->m_rhythm_analyzed = true;
	}
}


//////////////////////////////
//
// HumdrumFileStructure::analyzeRhythm -- Analyze the rhythmic structure
//     of the data.  Returns false if there was a parse error.
//

bool HumdrumFileStructure::analyzeRhythm(void) {
	setLineRhythmAnalyzed();
	if (getMaxTrack() == 0) {
		return true;
	}
	int startline = getTrackStart(1)->getLineIndex();
	int testline;
	HumNum zero(0);

	int i;
	for (int i=1; i<=getMaxTrack(); i++) {
		if (!getTrackStart(i)->hasRhythm()) {
			// Can't analyze rhythm of spines that do not have rhythm.
			continue;
		}
		testline = getTrackStart(i)->getLineIndex();
		if (testline == startline) {
			if (!assignDurationsToTrack(getTrackStart(i), zero)) {
				return false;
			}
		} else {
			// Spine does not start at beginning of data, so
			// the starting position of the spine has to be
			// determined before continuing.  Search for a token
			// which is on a line with assigned duration, then work
			// outwards from that position.
			continue;
		}
	}

	// Go back and analyze spines that do not start at the
	// beginning of the data stream.
	for (i=1; i<=getMaxTrack(); i++) {
		if (!getTrackStart(i)->hasRhythm()) {
			// Can't analyze rhythm of spines that do not have rhythm.
			continue;
		}
		testline = getTrackStart(i)->getLineIndex();
		if (testline > startline) {
			if (!analyzeRhythmOfFloatingSpine(getTrackStart(i))) { return false; }
		}
	}

	if (!analyzeNullLineRhythms()) { return false; }
	fillInNegativeStartTimes();
	assignLineDurations();
	if (!analyzeMeter()) { return false; }
	if (!analyzeNonNullDataTokens()) { return false; }

	return true;
}



//////////////////////////////
//
// HumdrumFileStructure::analyzeMeter -- Store the times from the last barline
//     to the current line, as well as the time to the next barline.
//     the sum of these two will be the duration of the barline, except
//     for barlines, where the getDurationToBarline() will store the
//     duration of the measure staring at that barline.  To get the
//     beat, you will have to figure out the current time signature.
//

bool HumdrumFileStructure::analyzeMeter(void) {

	m_barlines.resize(0);

	int i;
	HumNum sum = 0;
	bool foundbarline = false;
	for (i=0; i<getLineCount(); i++) {
		m_lines[i]->setDurationFromBarline(sum);
		sum += m_lines[i]->getDuration();
		if (m_lines[i]->isBarline()) {
			foundbarline = true;
			m_barlines.push_back(m_lines[i]);
			sum = 0;
		}
		if (m_lines[i]->isData() && !foundbarline) {
			// pickup measure, so set the first measure to the start of the file.
			m_barlines.push_back(m_lines[0]);
			foundbarline = 1;
		}
	}

	sum = 0;
	for (i=getLineCount()-1; i>=0; i--) {
		sum += m_lines[i]->getDuration();
		m_lines[i]->setDurationToBarline(sum);
		if (m_lines[i]->isBarline()) {
			sum = 0;
		}
	}

	return true;
}



//////////////////////////////
//
// HumdrumFileStructure::analyzeTokenDurations -- Calculate the duration of
//   all tokens in spines which posses duration in a file.
//

bool HumdrumFileStructure::analyzeTokenDurations (void) {
	for (int i=0; i<getLineCount(); i++) {
		if (!m_lines[i]->analyzeTokenDurations(m_parseError)) {
			return isValid();
		}
	}
	return isValid();
}



///////////////////////////////
//
// HumdrumFileStructure::analyzeGlobalParameters -- only allowing layout
//    parameters at the moment.  Global parameters affect the next
//    line which is either a barline, dataline or an interpretation
//    other than a spine manipulator.  Null lines are also not
//    considered.
//

bool HumdrumFileStructure::analyzeGlobalParameters(void) {
	vector<HLp> globals;

//	for (int i=0; i<(int)m_lines.size(); i++) {
//		if (m_lines[i]->isCommentGlobal()) {
//			m_lines[i]->setLayoutParameters();
//		}
//	}

	for (int i=0; i<(int)m_lines.size(); i++) {
		if (m_lines[i]->isCommentGlobal() && (m_lines[i]->find("!!LO:") != string::npos)) {
			m_lines[i]->storeGlobalLinkedParameters();
			globals.push_back(m_lines[i]);
			continue;
		}
		if (!m_lines[i]->hasSpines()) {
			continue;
		}
		if (m_lines[i]->isAllNull())  {
			continue;
		}
		if (m_lines[i]->isCommentLocal()) {
			continue;
		}
		if (globals.empty()) {
			continue;
		}

		// Filter manipulators or not?  At the moment allow
		// global parameters to pass through manipulators.
		// if (m_lines[i]->isManipulator()) {
		// 	continue;
		// }

		for (int j=0; j<(int)m_lines[i]->getFieldCount(); j++) {
			for (int k=0; k<(int)globals.size(); k++) {
				m_lines[i]->token(j)->addLinkedParameterSet(globals[k]->token(0));
			}
		}
		globals.clear();
	}

	return isValid();
}



///////////////////////////////
//
// HumdrumFileStructure::analyzeLocalParameters -- Parses any
//    local comments before a non-null token.
//

bool HumdrumFileStructure::analyzeLocalParameters(void) {
	// analyze backward tokens:

	for (int i=0; i<getStrandCount(); i++) {
		processLocalParametersForStrand(i);
	}

//	for (int i=1; i<=getMaxTrack(); i++) {
//		for (int j=0; j<getTrackEndCount(i); j++) {
//			if (!processLocalParametersForTrack(getTrackEnd(i, j),
//					getTrackEnd(i, j), getTrackStart(i, j))) {
//				return isValid();
//			}
//		}
//	}

	return isValid();
}



//////////////////////////////
//
// HumdrumFileStructure::analyzeDurationsOfNonRhythmicSpines -- Calculate the
//    duration of non-null data token in non-rhythmic spines.
//

bool HumdrumFileStructure::analyzeDurationsOfNonRhythmicSpines(void) {
	// analyze tokens backwards:
	for (int i=1; i<=getMaxTrack(); i++) {
		for (int j=0; j<getTrackEndCount(i); j++) {
			if (getTrackEnd(i, j)->hasRhythm()) {
				continue;
			}
			if (!assignDurationsToNonRhythmicTrack(getTrackEnd(i, j),
					getTrackEnd(i, j))) {
				return isValid();
			}
		}
	}
	return isValid();
}



//////////////////////////////
//
// HumdrumFileStructure::getMinDur -- Return the smallest duration on the
//   line.  If all durations are zero, then return zero; otherwise,
//   return the smallest positive duration.
//

HumNum HumdrumFileStructure::getMinDur(vector<HumNum>& durs,
		vector<HumNum>& durstate) {
	HumNum mindur = 0;
	bool allzero = true;

	for (int i=0; i<(int)durs.size(); i++) {
		if (durs[i].isPositive()) {
			allzero = false;
			if (mindur.isZero()) {
				mindur = durs[i];
			} else if (mindur > durs[i]) {
				mindur = durs[i];
			}
		}
	}
	if (allzero) {
		return mindur;
	}

	for (int i=0; i<(int)durs.size(); i++) {
		if (durstate[i].isPositive() && mindur.isZero()) {
			if (durstate[i].isZero()) {
				// mindur = durstate[i];
			} else if (mindur > durstate[i]) {
				mindur = durstate[i];
			}
		}
	}
	return mindur;
}



//////////////////////////////
//
// HumdrumFileStructure::getTokenDurations -- Extract the duration of rhythmic
//    tokens on the line.
//

bool HumdrumFileStructure::getTokenDurations(vector<HumNum>& durs, int line) {
	durs.resize(0);
	for (int i=0; i<m_lines[line]->getTokenCount(); i++) {
		HumNum dur = m_lines[line]->token(i)->getDuration();
		durs.push_back(dur);
	}
	if (!cleanDurs(durs, line)) {
		return isValid();
	}
	return isValid();
}



//////////////////////////////
//
// HumdrumFileStructure::cleanDurs -- Check if there are grace note and regular
//    notes on a line (not allowed).  Leaves negative durations which
//    indicate undefined durations (needed for keeping track of null
//    tokens in rhythmic spines.
//

bool HumdrumFileStructure::cleanDurs(vector<HumNum>& durs, int line) {
	bool zero     = false;
	bool positive = false;
	for (int i=0; i<(int)durs.size(); i++) {
		if      (durs[i].isPositive()) { positive = true; }
		else if (durs[i].isZero())     { zero     = true; }
	}
	if (zero && positive) {
		stringstream err;
		err << "Error on line " << (line+1) << " grace note and "
		    << " regular note cannot occur on same line." << endl;
		err << "Line: " << *m_lines[line] << endl;
		return setParseError(err);
	}
	return isValid();
}



//////////////////////////////
//
// HumdrumFileStructure::decrementDurStates -- Subtract the line duration from
//   the current line of running durations.  If any duration is less
//   than zero, then a rhythm error exists in the data.
//

bool HumdrumFileStructure::decrementDurStates(vector<HumNum>& durs,
		HumNum linedur, int line) {
	if (linedur.isZero()) {
		return isValid();
	}
	for (int i=0; i<(int)durs.size(); i++) {
		if (!m_lines[line]->token(i)->hasRhythm()) {
			continue;
		}
		durs[i] -= linedur;
		if (durs[i].isNegative()) {
			stringstream err;
			err << "Error: rhythmic error on line " << (line+1)
			    << " field index " << i << endl;
			err << "Duration state is: " << durs[i] << endl;
			return setParseError(err);
		}
	}
	return isValid();
}



//////////////////////////////
//
// HumdrumFileStructure::assignDurationsToTrack -- Assign duration from starts
//    for each rhythmic spine in the file.  Analysis is done recursively, one
//    sub-spine at a time.  Duplicate analyses are prevented by the state
//    variable in the HumdrumToken (currently called rhycheck because it is only
//    used in this function).  After the durationFromStarts have been assigned
//    for the rhythmic analysis of non-data tokens and non-rhythmic spines is
//    done elsewhere.
//

bool HumdrumFileStructure::assignDurationsToTrack(HTp starttoken,
		HumNum startdur) {
	if (!starttoken->hasRhythm()) {
		return isValid();
	}
	int state = starttoken->getState();
	if (!prepareDurations(starttoken, state, startdur)) {
		return isValid();
	}
	return isValid();
}



//////////////////////////////
//
// HumdrumFileStructure::prepareDurations -- Helper function for
//     HumdrumFileStructure::assignDurationsToTrack() which does all of the
//     work for assigning durationFromStart values.
//

bool HumdrumFileStructure::prepareDurations(HTp token, int state,
		HumNum startdur) {
	if (state != token->getState()) {
		return isValid();
	}

	HumNum dursum = startdur;
	token->incrementState();

	if (!setLineDurationFromStart(token, dursum)) { return isValid(); }
	if (token->getDuration().isPositive()) {
		dursum += token->getDuration();
	}
	int tcount = token->getNextTokenCount();

	vector<HTp> reservoir;
	vector<HumNum> startdurs;

	// Assign line durationFromStarts for primary track first.
	while (tcount > 0) {
		for (int t=1; t<tcount; t++) {
			reservoir.push_back(token->getNextToken(t));
			startdurs.push_back(dursum);
		}
		token = token->getNextToken(0);
		if (state != token->getState()) {
			break;
		}
		token->incrementState();
		if (!setLineDurationFromStart(token, dursum)) { return isValid(); }
		if (token->getDuration().isPositive()) {
			dursum += token->getDuration();
		}
		tcount = token->getNextTokenCount();
	}

	if ((tcount == 0) && (token->isTerminateInterpretation())) {
		if (!setLineDurationFromStart(token, dursum)) { return isValid(); }
	}

	// Process secondary tracks next:
	int newstate = state;

	for (int i=(int)reservoir.size()-1; i>=0; i--) {
		prepareDurations(reservoir[i], newstate, startdurs[i]);
	}

	return isValid();
}



//////////////////////////////
//
// HumdrumFileStructure::setLineDurationFromStart -- Set the duration of
//      a line based on the analysis of tokens in the spine.
//

bool HumdrumFileStructure::setLineDurationFromStart(HTp token,
		HumNum dursum) {
	if ((!token->isTerminateInterpretation()) &&
			token->getDuration().isNegative()) {
		// undefined rhythm, so don't assign line duration information:
		return isValid();
	}
	HLp line = token->getOwner();
	if (line->getDurationFromStart().isNegative()) {
		line->setDurationFromStart(dursum);
	} else if (line->getDurationFromStart() != dursum) {
		stringstream err;
		err << "Error: Inconsistent rhythm analysis occurring near line "
		    << token->getLineNumber() << endl;
		err << "Expected durationFromStart to be: " << dursum
		    << " but found it to be " << line->getDurationFromStart() << endl;
		err << "Line: " << *line << endl;
		return setParseError(err);
	}

	return isValid();
}



//////////////////////////////
//
// HumdrumFileStructure::analyzeRhythmOfFloatingSpine --  This analysis
//    function is used to analyze the rhythm of spines which do not start at
//    the beginning of the data.  The function searches for the first line
//    which has an assigned durationFromStart value, and then uses that
//    as the basis for assigning the initial durationFromStart position
//    for the spine.
//

bool HumdrumFileStructure::analyzeRhythmOfFloatingSpine(
		HTp spinestart) {
	HumNum dursum = 0;
	HumNum founddur = 0;
	HTp token = spinestart;
	int tcount = token->getNextTokenCount();

	// Find a known durationFromStart for a line in the Humdrum file, then
	// use that to calculate the starting duration of the floating spine.
	if (token->getDurationFromStart().isNonNegative()) {
		founddur = token->getLine()->getDurationFromStart();
	} else {
		tcount = token->getNextTokenCount();
		while (tcount > 0) {
			if (token->getDurationFromStart().isNonNegative()) {
				founddur = token->getLine()->getDurationFromStart();
				break;
			}
			if (token->getDuration().isPositive()) {
				dursum += token->getDuration();
			}
			token = token->getNextToken(0);
		}
	}

	if (founddur.isZero()) {
		return setParseError("Error cannot link floating spine to score.");
	}

	if (!assignDurationsToTrack(spinestart, founddur - dursum)) {
		return isValid();
	}

	return isValid();
}



//////////////////////////////
//
// HumdrumFileStructure::analyzeNullLineRhythms -- When a series of null-token
//    data line occur between two data lines possessing a start duration,
//    then split the duration between those two lines amongst the null-token
//    lines.  For example if a data line starts at time 15, and there is one
//    null-token line before another data line at time 16, then the null-token
//    line will be assigned to the position 15.5 in the score.
//

bool HumdrumFileStructure::analyzeNullLineRhythms(void) {
	vector<HLp> nulllines;
	HLp previous = NULL;
	HLp next = NULL;
	HumNum dur;
	HumNum startdur;
	HumNum enddur;
	int i, j;
	for (i=0; i<(int)m_lines.size(); i++) {
		if (!m_lines[i]->hasSpines()) {
			continue;
		}
		if (m_lines[i]->isAllRhythmicNull()) {
			if (m_lines[i]->isData()) {
				nulllines.push_back(m_lines[i]);
			}
			continue;
		}
		dur = m_lines[i]->getDurationFromStart();
		if (dur.isNegative()) {
			if (m_lines[i]->isData()) {
				stringstream err;
				err << "Error: found an unexpected negative duration on line "
			       << m_lines[i]->getDurationFromStart()<< endl;
				err << "Line: " << *m_lines[i] << endl;
				return setParseError(err);
			} else {
				continue;
			}
		}
		next = m_lines[i];
		if (previous == NULL) {
			previous = next;
			nulllines.resize(0);
			continue;
		}
		startdur = previous->getDurationFromStart();
		enddur   = next ->getDurationFromStart();
		HumNum gapdur = enddur - startdur;
		HumNum nulldur = gapdur / ((int)nulllines.size() + 1);
		for (j=0; j<(int)nulllines.size(); j++) {
			nulllines[j]->setDurationFromStart(startdur + (nulldur * (j+1)));
		}
		previous = next;
		nulllines.resize(0);
	}
	return isValid();
}



//////////////////////////////
//
// HumdrumFileStructure::fillInNegativeStartTimes -- Negative line durations
//    after the initial rhythmAnalysis mean that the lines are not data line.
//    Duplicate the duration of the next non-negative duration for all negative
//    durations.
//

void HumdrumFileStructure::fillInNegativeStartTimes(void) {
	int i;
	HumNum lastdur = -1;
	HumNum dur;
	for (i=(int)m_lines.size()-1; i>=0; i--) {
		dur = m_lines[i]->getDurationFromStart();
		if (dur.isNegative() && lastdur.isNonNegative()) {
			m_lines[i]->setDurationFromStart(lastdur);
		}
		if (dur.isNonNegative()) {
			lastdur = dur;
			continue;
		}
	}

	// fill in start times for ending comments
	for (i=0; i<(int)m_lines.size(); i++) {
		dur = m_lines[i]->getDurationFromStart();
		if (dur.isNonNegative()) {
			lastdur = dur;
		} else {
			m_lines[i]->setDurationFromStart(lastdur);
		}
	}
}



//////////////////////////////
//
// HumdrumFileStructure::assignLineDurations --  Calculate the duration of lines
//   based on the durationFromStart of the current line and the next line.
//

void HumdrumFileStructure::assignLineDurations(void) {
	HumNum startdur;
	HumNum enddur;
	HumNum dur;
	for (int i=0; i<(int)m_lines.size()-1; i++) {
		startdur = m_lines[i]->getDurationFromStart();
		enddur = m_lines[i+1]->getDurationFromStart();
		dur = enddur - startdur;
		m_lines[i]->setDuration(dur);
	}
	if (m_lines.size() > 0) {
		m_lines.back()->setDuration(0);
	}
}



//////////////////////////////
//
// HumdrumFileStructure::assignDurationsToNonRhythmicTrack --  After the basic
//   rhythmAnalysis has been done, go back and assign durations to non-rhythmic
//   spine tokens based on the lineFromStart values of the lines that they
//   occur on as well as the distance in the file to the next non-null token for
//   that spine.
//

bool HumdrumFileStructure::assignDurationsToNonRhythmicTrack(
		HTp endtoken, HTp current) {

	string spineinfo = endtoken->getSpineInfo();
	HTp token = endtoken;

	while (token) {
		if (token->getSpineInfo() != spineinfo) {
			if (token->getSpineInfo().find("b") != std::string::npos) {
				break;
			}
			if (spineinfo.find("b") != std::string::npos) {
				break;
			}
		}
		int tcount = token->getPreviousTokenCount();
		if (tcount == 0) {
			break;
		}
		if (tcount > 1) {
			for (int i=1; i<tcount; i++) {
				HTp ptok = token->getPreviousToken(i);
				if (!assignDurationsToNonRhythmicTrack(ptok, current)) {
					return isValid();
				}
			}
		}
		if (token->isData()) {
			if (!token->isNull()) {
				token->setDuration(current->getDurationFromStart() -
					token->getDurationFromStart());
				current = token;
			}
		}
		token = token->getPreviousToken(0);
	}

	return isValid();
}



//////////////////////////////
//
// HumdrumFileStructure::processLocalParametersForStrand --
//

void HumdrumFileStructure::processLocalParametersForStrand(int index) {
	HTp sstart = getStrandStart(index);
	HTp send = getStrandEnd(index);
	HTp tok = send;
	HTp dtok = NULL;
	while (tok) {
		if (tok->isData()) {
			dtok = tok;
		} else if (tok->isBarline()) {
			// layout parameters allowed for barlines
			dtok = tok;
		} else if (tok->isInterpretation() && (*tok != "*")) {
			// layout parameters allowed for non-null interpretations
			dtok = tok;
		} else if (tok->isCommentLocal()) {
			if (tok->find("!LO:") == 0) {
				tok->storeParameterSet();
				if (dtok) {
					dtok->addLinkedParameterSet(tok);
				}
			}
		}
		if (tok == sstart) {
			break;
		}
		tok = tok->getPreviousToken();
	}
}




//////////////////////////////
//
// HumdrumFileStructure::processLocalParametersForTrack --  Search for
//   local parameters backwards in each spine and fill in the HumHash
//   for the token to which the parameter is to be applied.
//
// No longer used.
//

bool HumdrumFileStructure::processLocalParametersForTrack(
		HTp starttok, HTp current) {

	HTp token = starttok;
	int tcount = token->getPreviousTokenCount();

	while (tcount > 0) {
		for (int i=1; i<tcount; i++) {
			if (!processLocalParametersForTrack(
					token->getPreviousToken(i), current)) {
				return isValid();
			}
		}
		HTp prevtoken = token->getPreviousToken();
		if (prevtoken->isSplitInterpretation()) {
			if (token != prevtoken->m_nextTokens[0]) {
				// terminate if not most primary subspine
				return true;
			}
		} else if (!(token->isNull() & token->isManipulator())) {
			if (token->isCommentLocal()) {
				checkForLocalParameters(token, current);
			} else {
				current = token;
			}
		}

		// Follow previous data token 0 since 1 and higher are handled above.
		token = token->getPreviousToken(0);
		tcount = token->getPreviousTokenCount();
	}

	return isValid();
}



//////////////////////////////
//
// HumdrumFileStructure::checkForLocalParameters -- Helper function for
//     HumdrumFileStructure::processLocalParametersForTrack.  Only allowing
//     layout parameters currently.
//

void HumdrumFileStructure::checkForLocalParameters(HTp token,
		HTp current) {
	if (token->size() < 1) {
		return;
	}
	int loc1 = (int)token->find(":");
	if (loc1 == (int)string::npos) {
		return;
	}
	int loc2 = (int)token->substr(loc1).find(":");
	if (loc2 == (int)string::npos) {
		return;
	}
	loc2 += loc1 + 1;
	int sloc = (int)token->find(" ");
	if (sloc != (int)string::npos) {
		if ((sloc < loc1) || (sloc < loc2)) {
			return;
		}
	}
	sloc = (int)token->find("\t");
	if (sloc != (int)string::npos) {
		if ((sloc < loc1) || (sloc < loc2)) {
			return;
		}
	}
	// Looks like a parameter so parse the comment:
	current->setParameters(token);
}



//////////////////////////////
//
// HumdrumFileStructure::analyzeStrands -- Analyze spine strands.
//

bool HumdrumFileStructure::analyzeStrands(void) {
	m_analyses.m_strands_analyzed = true;
	int spines = getSpineCount();
	m_strand1d.clear();
	m_strand2d.clear();
	int i, j;
	for (i=0; i<spines; i++) {
		HTp tok = getSpineStart(i);
		m_strand2d.resize(m_strand2d.size()+1);
		analyzeSpineStrands(m_strand2d.back(), tok);
	}

	for (i=0; i<(int)m_strand2d.size(); i++) {
		std::sort(m_strand2d[i].begin(), m_strand2d[i].end(),
				sortTokenPairsByLineIndex);
		for (j=0; j<(int)m_strand2d[i].size(); j++) {
			m_strand1d.push_back(m_strand2d[i][j]);
		}
	}

	assignStrandsToTokens();

	resolveNullTokens();

	analyzeLocalParameters();

	analyzeStrophes();

	return isValid();
}



///////////////////////////////
//
// HumdrumFileStructure::resolveNullTokens --
//

void HumdrumFileStructure::resolveNullTokens(void) {
	if (m_analyses.m_nulls_analyzed) {
		return;
	}
	m_analyses.m_nulls_analyzed = true;
	if (!areStrandsAnalyzed()) {
		analyzeStrands();
	}

	HTp token;
	HTp data = NULL;
	HTp strandend;
	for (int s=0; s<(int)m_strand1d.size(); s++) {
		token = getStrandStart(s);
		strandend = getStrandEnd(s);
		while (token != strandend) {
			if (!token->isData()) {
				token = token->getNextToken();
				continue;
			}
			if (data == NULL) {
				data = token;
				token->setNullResolution(data);
				token = token->getNextToken();
				continue;
			}
			if (token->isNull()) {
				token->setNullResolution(data);
			} else {
				data = token;
			}
			token = token->getNextToken();
		}
	}
}



//////////////////////////////
//
// HumdrumFileStructure::assignStrandsToTokens -- Store the 1D strand
//    index number for each token in the file.  Global tokens will have
//    strand index set to -1.
//

void HumdrumFileStructure::assignStrandsToTokens(void) {
	HTp tok;
	for (int i=0; i<(int)m_strand1d.size(); i++) {
		tok = m_strand1d[i].first;
		while (tok != NULL) {
			tok->setStrandIndex(i);
			tok = tok->getNextToken();
		}
	}
}



//////////////////////////////
//
// HumdrumFileStructure::analyzeSpineStrands -- Fill in the list of
//   strands in a single spine.
//

void HumdrumFileStructure::analyzeSpineStrands(vector<TokenPair>& ends,
		HTp starttok) {

	ends.resize(ends.size()+1);
	int index = (int)ends.size()-1;
	ends[index].first = starttok;
	HTp tok = starttok;
	while (tok != NULL) {
		if ((tok->getSubtrack() > 1) && (tok->isMerge())) {
			// check to the left: if the left primary/sub spine also has
			// a *v, then this is the end of this strand; otherwise, the
			// strand continues.
			if (tok->getPreviousFieldToken()->isMerge()) {
				ends[index].last = tok;
				return;
			} else {
				tok = tok->getNextToken();
				continue;
			}
		}
		if (tok->isTerminator()) {
			ends[index].last = tok;
			return;
		}
		if (tok->getNextTokenCount() > 1) {
			// should only be 2, but allow for generalizing in the future.
			for (int j=1; j<tok->getNextTokenCount(); j++) {
				analyzeSpineStrands(ends, tok->getNextToken(j));
			}
		}
		tok = tok->getNextToken();
	}

	cerr << "Should not get here in analyzeSpineStrands()\n";
}


//////////////////////////////
//
// HumdrumFileStructure::getStrandCount --
//

int HumdrumFileStructure::getStrandCount(void) {
	if (!areStrandsAnalyzed()) {
		analyzeStrands();
	}
	return (int)m_strand1d.size();
}


int HumdrumFileStructure::getStrandCount(int spineindex) {
	if (!areStrandsAnalyzed()) {
		analyzeStrands();
	}
	if (spineindex < 0) {
		return 0;
	}
	if (spineindex >= (int)m_strand2d.size()) {
		return 0;
	}
	return (int)m_strand2d[spineindex].size();
}



//////////////////////////////
//
// HumdrumFileStructure::getStrandStart -- Return the first token
//    in the a strand.
//

HTp HumdrumFileStructure::getStrandStart(int index) {
	if (!areStrandsAnalyzed()) {
		analyzeStrands();
	}
	return m_strand1d[index].first;
}


HTp HumdrumFileStructure::getStrandEnd(int index) {
	if (!areStrandsAnalyzed()) {
		analyzeStrands();
	}
	return m_strand1d[index].last;
}


HTp HumdrumFileStructure::getStrandStart(int sindex,
		int index) {
	if (!areStrandsAnalyzed()) {
		analyzeStrands();
	}
	return m_strand2d[sindex][index].first;
}


HTp HumdrumFileStructure::getStrandEnd(int sindex, int index) {
	if (!areStrandsAnalyzed()) {
		analyzeStrands();
	}
	return m_strand2d[sindex][index].last;
}



//////////////////////////////
//
// HumdrumFileStructure::hasFilters -- Returns true if has any
//    reference records starting with "!!!filter:" or "!!!!filter:".
//

bool HumdrumFileStructure::hasFilters(void) {
	HumdrumFileBase& infile = *this;
	vector<HLp> refs  = infile.getGlobalReferenceRecords();
	for (int i=0; i<(int)refs.size(); i++) {
		if (refs[i]->getGlobalReferenceKey() == "filter") {
			return true;
		}
	}
	return false;
}



//////////////////////////////
//
// HumdrumFileStructure::hasGlobalFilters -- Returns true if has any
//    reference records starting with "!!!filter:".
//

bool HumdrumFileStructure::hasGlobalFilters(void) {
	HumdrumFileBase& infile = *this;
	for (int i=0; i<infile.getLineCount(); i++) {
		if (!infile[i].isComment()) {
			continue;
		}
		HTp token = infile.token(i, 0);
		if (token->compare(0, 10, "!!!filter:") == 0) {
			return true;
		}
	}
	return false;
}



//////////////////////////////
//
// HumdrumFileStructure::hasUniversalFilters -- Returns true if has any
//    reference records starting with "!!!!filter:".
//

bool HumdrumFileStructure::hasUniversalFilters(void) {
	HumdrumFileBase& infile = *this;
	vector<HLp> refs  = infile.getUniversalReferenceRecords();
	for (int i=0; i<(int)refs.size(); i++) {
		if (refs[i]->getUniversalReferenceKey() == "filter") {
			return true;
		}
	}
	return false;
}



//////////////////////////////
//
// HumdrumFileStructure::analyzeSignifiers --
//

void HumdrumFileStructure::analyzeSignifiers(void) {
	HumdrumFileStructure& infile = *this;
	for (int i=0; i<getLineCount(); i++) {
		if (!infile[i].isSignifier()) {
			continue;
		}
		m_signifiers.addSignifier(infile[i].getText());
	}
}



//////////////////////////////
//
// HumdrumFileStructure::getKernLinkSignifier -- used for linking two
//     non-standard slur/tie ends together.
//

std::string HumdrumFileStructure::getKernLinkSignifier(void) {
	return m_signifiers.getKernLinkSignifier();
}



//////////////////////////////
//
// HumdrumFileStructure::getKernAboveSignifier -- used to place things
//     "above" (note on staff above, slurs/ties with an "above" orientation,
//     etc.
//

std::string HumdrumFileStructure::getKernAboveSignifier(void) {
	return m_signifiers.getKernAboveSignifier();
}



//////////////////////////////
//
// HumdrumFileStructure::getKernBelowSignifier -- Used to place things
//     "below" (note on staff above, slurs/ties with an "below" orientation,
//     etc.
//

std::string HumdrumFileStructure::getKernBelowSignifier(void) {
	return m_signifiers.getKernBelowSignifier();
}




//////////////////////////////
//
// HumdrumLine::HumdrumLine -- HumdrumLine constructor.
//

HumdrumLine::HumdrumLine(void) : string() {
	m_owner = NULL;
	m_duration = -1;
	m_durationFromStart = -1;
	setPrefix("!!");
}


HumdrumLine::HumdrumLine(const string& aString) : string(aString) {
	m_owner = NULL;
	if ((this->size() > 0) && (this->back() == 0x0d)) {
		this->resize(this->size() - 1);
	}
	m_duration = -1;
	m_durationFromStart = -1;
	setPrefix("!!");
	createTokensFromLine();
}


HumdrumLine::HumdrumLine(const char* aString) : string(aString) {
	m_owner = NULL;
	if ((this->size() > 0) && (this->back() == 0x0d)) {
		this->resize(this->size() - 1);
	}
	m_duration = -1;
	m_durationFromStart = -1;
	setPrefix("!!");
	createTokensFromLine();
}


HumdrumLine::HumdrumLine(HumdrumLine& line)  : string((string)line) {
	m_lineindex           = line.m_lineindex;
	m_duration            = line.m_duration;
	m_durationFromStart   = line.m_durationFromStart;
	m_durationFromBarline = line.m_durationFromBarline;
	m_durationToBarline   = line.m_durationToBarline;
	m_tokens.resize(line.m_tokens.size());
	for (int i=0; i<(int)m_tokens.size(); i++) {
		m_tokens[i] = new HumdrumToken(*line.m_tokens[i], this);
	}
	m_tabs.resize(line.m_tabs.size());
	for (int i=0; i<(int)m_tabs.size(); i++) {
		m_tabs.at(i) = line.m_tabs.at(i);
	}
	m_owner = NULL;
}


HumdrumLine::HumdrumLine(HumdrumLine& line, void* owner) : string((string)line) {
	m_lineindex           = line.m_lineindex;
	m_duration            = line.m_duration;
	m_durationFromStart   = line.m_durationFromStart;
	m_durationFromBarline = line.m_durationFromBarline;
	m_durationToBarline   = line.m_durationToBarline;
	m_tokens.resize(line.m_tokens.size());
	for (int i=0; i<(int)m_tokens.size(); i++) {
		m_tokens[i] = new HumdrumToken(*line.m_tokens[i], this);
	}
	m_tabs.resize(line.m_tabs.size());
	for (int i=0; i<(int)m_tabs.size(); i++) {
		m_tabs.at(i) = line.m_tabs.at(i);
	}
	m_owner = owner;
}



//////////////////////////////
//
// HumdrumLine::operator= --
//

HumdrumLine& HumdrumLine::operator=(HumdrumLine& line) {
	m_lineindex           = line.m_lineindex;
	m_duration            = line.m_duration;
	m_durationFromStart   = line.m_durationFromStart;
	m_durationFromBarline = line.m_durationFromBarline;
	m_durationToBarline   = line.m_durationToBarline;
	m_tokens.resize(line.m_tokens.size());
	for (int i=0; i<(int)m_tokens.size(); i++) {
		m_tokens[i] = new HumdrumToken(*line.m_tokens[i], this);
	}
	m_tabs.resize(line.m_tabs.size());
	for (int i=0; i<(int)m_tabs.size(); i++) {
		m_tabs.at(i) = line.m_tabs.at(i);
	}
	m_owner = NULL;
	return *this;
}



//////////////////////////////
//
// HumdrumLine::~HumdrumLine -- HumdrumLine deconstructor.
//

HumdrumLine::~HumdrumLine() {
	// free stored HumdrumTokens:
	for (int i=0; i<(int)m_tokens.size(); i++) {
		if (m_tokens[i] != NULL) {
			delete m_tokens[i];
			m_tokens[i] = NULL;
		}
	}
}



//////////////////////////////
//
// HumdrumLine::setLineFromCsv -- Read a HumdrumLine from a CSV line.
// default value: separator = ","
//

void HumdrumLine::setLineFromCsv(const char* csv, const string& separator) {
	string temp = csv;
	setLineFromCsv(temp);
}



void HumdrumLine::setLineFromCsv(const string& csv, const string& separator) {
	if (csv.size() < 1) {
		return;
	}
	string newcsv = csv;
	if ((newcsv.size() > 0) && (newcsv.back() == 0x0d)) {
		newcsv.resize(newcsv.size() - 1);
	}
	// construct tab-delimited string
	string output;
	bool inquote = false;

	if ((newcsv.size() >= 2) && (newcsv[0] == '!') && (newcsv[1] == '!')) {
		// Global commands and reference records which do not start with a
		// quote are considered to be literal.
		this->setText(newcsv);
		return;
	}

	for (int i=0; i<(int)newcsv.size(); i++) {
		if ((newcsv[i] == '"') && !inquote) {
			inquote = true;
			continue;
		}
		if (inquote && (newcsv[i] == '"') && (newcsv[i+1] == '"')
				&& (i < (int)newcsv.length()-1)) {
			output += '"';
			i++;
			continue;
		}
		if (newcsv[i] == '"') {
			inquote = false;
			continue;
		}
		if ((!inquote) && (newcsv.substr(i, separator.size()) == separator)) {
			output += '\t';
			i += (int)separator.size() - 1;
			continue;
		}
		output += newcsv[i];
	}
	string& value = *this;
	value = output;
}



//////////////////////////////
//
// HumdrumLine::setText -- Get the textual content of the line.  Note that
//    you may need to run HumdrumLine::createLineFromTokens() if the tokens
//    of the line have changed.
//

void HumdrumLine::setText(const string& text) {
	string::assign(text);
}



//////////////////////////////
//
// HumdrumLine::getText --
//

string HumdrumLine::getText(void) {
	return string(*this);
}



//////////////////////////////
//
// HumdrumLine::clear -- Remove stored tokens.
//

void HumdrumLine::clear(void) {
	for (int i=0; i<(int)m_tokens.size(); i++) {
		if (m_tokens[i] != NULL) {
			delete m_tokens[i];
			m_tokens[i] = NULL;
		}
	}
	m_tokens.clear();
	m_tabs.clear();
	m_rhythm_analyzed = false;
}



//////////////////////////////
//
// HumdrumLine::equalChar -- return true if the character at the given
//     index is the given char.
//

bool HumdrumLine::equalChar(int index, char ch) const {
	if ((int)size() <= index) {
		return false;
	}
	if (index < 0) {
		return false;
	}
	if (((string)(*this))[index] == ch) {
		return true;
	} else {
		return false;
	}
}



//////////////////////////////
//
// HumdrumLine::isKernBoundaryStart -- Return true if the
//    line does not have any null tokens in **kern data which
//    refer to data tokens above the line.
//

bool HumdrumLine::isKernBoundaryStart(void) const {
	if (!isData()) {
		return false;
	}
	for (int i=0; i<getFieldCount(); i++) {
		if (!token(i)->isDataType("**kern")) {
			continue;
		}
		if (token(i)->isNull()) {
			return false;
		}
	}
	return true;
}



//////////////////////////////
//
// HumdrumLine::isKernBoundaryEnd -- Return true if the next
//    data line contains no null tokens in the **kern spines.
//    Assuming that a **kern spine split always starts with
//    a non-null token.
//

bool HumdrumLine::isKernBoundaryEnd(void) const {
	if (!isData()) {
		return false;
	}
	HTp ntok;
	for (int i=0; i<getFieldCount(); i++) {
		if (!token(i)->isDataType("**kern")) {
			continue;
		}
		ntok = token(i)->getNextToken();
		if (ntok == NULL) {
			continue;
		}
		while ((ntok != NULL) && !ntok->isData()) {
			ntok = ntok->getNextToken();
		}
		if (ntok == NULL) {
			continue;
		}
		if (ntok->isNull()) {
			return false;
		}
	}
	return true;
}



//////////////////////////////
//
// HumdrumLine::isComment -- Returns true if the first character
//   in the string is '!'. Could be local, global, or a reference record.
//

bool HumdrumLine::isComment(void) const {
	return equalChar(0, '!');
}



//////////////////////////////
//
// HumdrumLine::isCommentLocal -- Returns true if a local comment.
//

bool HumdrumLine::isCommentLocal(void) const {
	return equalChar(0, '!') && !equalChar(1, '!');
}



//////////////////////////////
//
// HumdrumLine::isCommentGlobal -- Returns true if a local comment.
//

bool HumdrumLine::isCommentGlobal(void) const {
	return equalChar(0, '!') && equalChar(1, '!');
}



//////////////////////////////
//
// HumdrumLine::isCommentUniversal -- Returns true if a universal comment.
//

bool HumdrumLine::isCommentUniversal(void) const {
	return equalChar(3, '!') && equalChar(2, '!') && equalChar(1, '!') && equalChar(0, '!');
}



//////////////////////////////
//
// HumdrumLine::isReference -- Returns true if a reference record.
//

bool HumdrumLine::isReference(void) const {
	return isGlobalReference() || isUniversalReference();
}



//////////////////////////////
//
// HumdrumLine::isGlobalReference -- Returns true if a global reference record.
//   Meaning that it is in the form:
//     !!!KEY: VALUE
//

bool HumdrumLine::isGlobalReference(void) const {
	if (this->size() < 5) {
		return false;
	}
	if (this->compare(0, 3, "!!!") != 0) {
		return false;
	}
	if (this->at(3) == '!') {
		return false;
	}
	int spaceloc = (int)this->find(" ");
	int tabloc = (int)this->find("\t");
	int colloc = (int)this->find(":");
	if (colloc == (int)string::npos) {
		return false;
	}
	if ((spaceloc != (int)string::npos) && (spaceloc < colloc)) {
		return false;
	}
	if ((tabloc != (int)string::npos) && (tabloc < colloc)) {
		return false;
	}
	return true;
}



//////////////////////////////
//
// HumdrumLine::isUniversalReference -- Returns true if
//     a universal reference record.
//

bool HumdrumLine::isUniversalReference(void) const {

	if (this->size() < 5) {
		return false;
	}
	if (this->substr(0, 4) != "!!!!") {
		return false;
	}
	if ((*this)[4] == '!') {
		return false;
	}
	int spaceloc = (int)this->find(" ");
	int tabloc = (int)this->find("\t");
	int colloc = (int)this->find(":");
	if (colloc == (int)string::npos) {
		return false;
	}
	if ((spaceloc != (int)string::npos) && (spaceloc < colloc)) {
		return false;
	}
	if ((tabloc != (int)string::npos) && (tabloc < colloc)) {
		return false;
	}
	return true;
}




//////////////////////////////
//
// HumdrumLine::isSignifier -- Returns true if a !!!RDF reference record.
//

bool HumdrumLine::isSignifier(void) const {
	if (this->size() < 9) {
		return false;
	}
	if (this->substr(0, 8) != "!!!RDF**") {
		return false;
	}
	return true;
}



//////////////////////////////
//
// HumdrumLine::getReferenceKey -- Return reference key if a reference
//     record.  Otherwise returns an empty string.
//

string HumdrumLine::getReferenceKey(void) const {
	if (this->size() < 4) {
		return "";
	}
	if (this->substr(0, 3) != "!!!") {
		return "";
	}
	if (this->at(3) != '!') {
		return getGlobalReferenceKey();
	} else {
		return getUniversalReferenceKey();
	}
}



//////////////////////////////
//
// HumdrumLine::getReferenceValue -- Return reference value if a reference
//     record.  Otherwise returns an empty string.
//

string HumdrumLine::getReferenceValue(void) const {
	if (this->size() < 4) {
		return "";
	}
	if (this->substr(0, 3) != "!!!") {
		return "";
	}
	if (this->at(3) != '!') {
		return getGlobalReferenceValue();
	} else {
		return getUniversalReferenceValue();
	}
}



//////////////////////////////
//
// HumdrumLine::getUniversalReferenceKey -- Return reference key if a
//     universal reference record.  Otherwise returns an empty string.
//

string HumdrumLine::getUniversalReferenceKey(void) const {
	if (this->size() < 6) {
		return "";
	}
	if (this->substr(0, 4) != "!!!!") {
		return "";
	}
	if ((*this)[4] == '!') {
		return "";
	}
	int spaceloc = (int)this->find(" ");
	int tabloc = (int)this->find("\t");
	int colloc = (int)this->find(":");
	if (colloc == (int)string::npos) {
		return "";
	}
	if ((spaceloc != (int)string::npos) && (spaceloc < colloc)) {
		return "";
	}
	if ((tabloc != (int)string::npos) && (tabloc < colloc)) {
		return "";
	}
	return this->substr(4, colloc - 4);
}



//////////////////////////////
//
// HumdrumLine::getGlobalReferenceKey -- Return reference key if a
//     universal reference record.  Otherwise returns an empty string.
//

string HumdrumLine::getGlobalReferenceKey(void) const {
	if (this->size() < 6) {
		return "";
	}
	if (this->substr(0, 3) != "!!!") {
		return "";
	}
	if ((*this)[4] == '!') {
		return "";
	}
	int spaceloc = (int)this->find(" ");
	int tabloc = (int)this->find("\t");
	int colloc = (int)this->find(":");
	if (colloc == (int)string::npos) {
		return "";
	}
	if ((spaceloc != (int)string::npos) && (spaceloc < colloc)) {
		return "";
	}
	if ((tabloc != (int)string::npos) && (tabloc < colloc)) {
		return "";
	}
	return this->substr(3, colloc - 3);
}



//////////////////////////////
//
// HumdrumLine::getGlobalReferenceValue -- Return reference value if a reference
//     record.  Otherwise returns an empty string.
//

string HumdrumLine::getGlobalReferenceValue(void) const {
	if (this->size() < 5) {
		return "";
	}
	if (this->substr(0, 3) != "!!!") {
		return "";
	}
	if ((*this)[3] == '!') {
		return "";
	}
	int spaceloc = (int)this->find(" ");
	int tabloc = (int)this->find("\t");
	int colloc = (int)this->find(":");
	if (colloc == (int)string::npos) {
		return "";
	}
	if ((spaceloc != (int)string::npos) && (spaceloc < colloc)) {
		return "";
	}
	if ((tabloc != (int)string::npos) && (tabloc < colloc)) {
		return "";
	}
	return Convert::trimWhiteSpace(this->substr(colloc+1));
}



//////////////////////////////
//
// HumdrumLine::getUniversalReferenceValue -- Return reference value if a reference
//     record.  Otherwise returns an empty string.
//

string HumdrumLine::getUniversalReferenceValue(void) const {
	if (this->size() < 6) {
		return "";
	}
	if (this->substr(0, 4) != "!!!!") {
		return "";
	}
	if ((*this)[4] == '!') {
		return "";
	}
	int spaceloc = (int)this->find(" ");
	int tabloc = (int)this->find("\t");
	int colloc = (int)this->find(":");
	if (colloc == (int)string::npos) {
		return "";
	}
	if ((spaceloc != (int)string::npos) && (spaceloc < colloc)) {
		return "";
	}
	if ((tabloc != (int)string::npos) && (tabloc < colloc)) {
		return "";
	}
	return Convert::trimWhiteSpace(this->substr(colloc+1));
}



//////////////////////////////
//
// HumdrumLine::isExclusive -- Returns true if the first two characters
//     are "**".
//

bool HumdrumLine::isExclusive(void) const {
	return equalChar(1, '*') && equalChar(0, '*');
}



//////////////////////////////
//
// HumdrumLine::isTerminator -- Returns true if all tokens on the line
//    are terminators.
//

bool HumdrumLine::isTerminator(void) const {
	if (getTokenCount() == 0) {
		// if tokens have not been parsed, check line text
		return equalChar(1, '!') && equalChar(0, '*');
	}
	for (int i=0; i<getTokenCount(); i++) {
		if (!token(i)->isTerminator()) {
			return false;
		}
	}
	return true;
}



//////////////////////////////
//
// HumdrumLine::isInterp -- Returns true if starts with '*' character.
//

bool HumdrumLine::isInterp(void) const {
	return equalChar(0, '*');
}



//////////////////////////////
//
// HumdrumLine::isBarline -- Returns true if starts with '=' character.
//

bool HumdrumLine::isBarline(void) const {
	return equalChar(0, '=');
}



//////////////////////////////
//
// HumdrumLine::isData -- Returns true if data (but not measure).
//

bool HumdrumLine::isData(void) const {
	if (isComment() || isInterp() || isBarline() || isEmpty()) {
		return false;
	} else {
		return true;
	}
}



//////////////////////////////
//
// HumdrumLine::isAllNull -- Returns true if all tokens on the line
//    are null ("." if a data line, "*" if an interpretation line, "!"
//    if a local comment line).
//

bool HumdrumLine::isAllNull(void) const {
	if (!hasSpines()) {
		return false;
	}
	for (int i=0; i<getTokenCount(); i++) {
		if (!token(i)->isNull()) {
			return false;
		}
	}
	return true;
}



//////////////////////////////
//
// HumdrumLine::isAllRhythmicNull -- Returns true if all rhythmic
//    data-type tokens on the line are null ("." if a data line,
//    "*" if an interpretation line, "!" if a local comment line).
//

bool HumdrumLine::isAllRhythmicNull(void) const {
	if (!hasSpines()) {
		return false;
	}
	for (int i=0; i<getTokenCount(); i++) {
		if (!token(i)->hasRhythm()) {
			continue;
		}
		if (!token(i)->isNull()) {
			return false;
		}
	}
	return true;
}



//////////////////////////////
//
// HumdrumLine::setLineIndex -- Used by the HumdrumFileBase class to set the
//   index number of the line in the data storage for the file.
//

void HumdrumLine::setLineIndex(int index) {
	m_lineindex = index;
}



//////////////////////////////
//
// HumdrumLine::getLineIndex -- Returns the index number of the line in the
//    HumdrumFileBase storage for the lines.
//

int HumdrumLine::getLineIndex(void) const {
	return m_lineindex;
}



//////////////////////////////
//
// HumdrumLine::getLineNumber -- Returns the line index plus one.
//

int HumdrumLine::getLineNumber(void) const {
	return m_lineindex + 1;
}



//////////////////////////////
//
// HumdrumLine::getDuration -- Get the duration of the line.  The duration will
//    be negative one if rhythmic analysis in HumdrumFileStructure has not been
//    done on the owning HumdrumFile object.  Otherwise this is the duration of
//    the current line in the file.
//

HumNum HumdrumLine::getDuration(void) {
	if (!m_rhythm_analyzed) {
		if (getOwner()) {
			getOwner()->analyzeRhythmStructure();
		}
	}
	return m_duration;
}


HumNum HumdrumLine::getDuration(HumNum scale) {
	if (!m_rhythm_analyzed) {
		if (getOwner()) {
			getOwner()->analyzeRhythmStructure();
		}
	}
	return m_duration * scale;
}



//////////////////////////////
//
// HumdrumLine::getBarlineDuration -- Return the duration following a barline,
//    or the duration of the previous barline in the data.
//

HumNum HumdrumLine::getBarlineDuration(void) {
	if (!m_rhythm_analyzed) {
		if (getOwner()) {
			getOwner()->analyzeRhythmStructure();
		}
	}
	if (isBarline()) {
		return getDurationToBarline();
	} else {
		return getDurationFromBarline() + getDurationToBarline();
	}
}


HumNum HumdrumLine::getBarlineDuration(HumNum scale) {
	if (!m_rhythm_analyzed) {
		if (getOwner()) {
			getOwner()->analyzeRhythmStructure();
		}
	}
	if (isBarline()) {
		return getDurationToBarline(scale);
	} else {
		return getDurationFromBarline(scale) + getDurationToBarline(scale);
	}
}



//////////////////////////////
//
// HumdrumLine::setDurationFromStart -- Sets the duration from the start of the
//    file to the start of the current line.  This is used in rhythmic
//    analysis done in the HumdrumFileStructure class.
//

void HumdrumLine::setDurationFromStart(HumNum dur) {
	m_durationFromStart = dur;
}



//////////////////////////////
//
// HumdrumLine::getDurationFromStart -- Get the duration from the start of the
//    file to the start of the current line.  This will be -1 if rhythmic
//    analysis has not been done in the HumdrumFileStructure class.
//

HumNum HumdrumLine::getDurationFromStart(void) {
	if (!m_rhythm_analyzed) {
		if (getOwner()) {
			getOwner()->analyzeRhythmStructure();
		}
	}
	return m_durationFromStart;
}


HumNum HumdrumLine::getDurationFromStart(HumNum scale) {
	if (!m_rhythm_analyzed) {
		if (getOwner()) {
			getOwner()->analyzeRhythmStructure();
		}
	}
	return m_durationFromStart * scale;
}



//////////////////////////////
//
// HumdrumLine::getDurationToEnd -- Returns the duration from the start of the
//    line to the end of the HumdrumFile which owns this HumdrumLine.  The
//    rhythm of the HumdrumFile must be analyze before using this function;
//    otherwise a 0 will probably be returned.
//

HumNum HumdrumLine::getDurationToEnd(void) {
	if (!m_rhythm_analyzed) {
		if (getOwner()) {
			getOwner()->analyzeRhythmStructure();
		}
	} else {
		return 0;
	}
	return ((HumdrumFile*)getOwner())->getScoreDuration() -  m_durationFromStart;
}


HumNum HumdrumLine::getDurationToEnd(HumNum scale) {
	if (!m_rhythm_analyzed) {
		if (getOwner()) {
			getOwner()->analyzeRhythmStructure();
		}
	} else {
		return 0;
	}
	return scale * (((HumdrumFile*)getOwner())->getScoreDuration() -
		m_durationFromStart);
}



//////////////////////////////
//
// HumdrumLine::getDurationFromBarline -- Returns the duration from the start
//    of the given line to the first barline occurring before the given line.
//    Analysis of this data is found in HumdrumFileStructure::metricAnalysis.
//

HumNum HumdrumLine::getDurationFromBarline(void) {
	if (!m_rhythm_analyzed) {
		if (getOwner()) {
			getOwner()->analyzeRhythmStructure();
		}
	}
	return m_durationFromBarline;
}


HumNum HumdrumLine::getDurationFromBarline(HumNum scale) {
	if (!m_rhythm_analyzed) {
		if (getOwner()) {
			getOwner()->analyzeRhythmStructure();
		}
	}
	return m_durationFromBarline * scale;
}



//////////////////////////////
//
// HumdrumLine::getTrackStart --  Returns the starting exclusive interpretation
//    for the given spine/track.
//

HTp HumdrumLine::getTrackStart(int track) const {
	if (!m_owner) {
		return NULL;
	} else {
		return ((HumdrumFile*)m_owner)->getTrackStart(track);
	}
}



//////////////////////////////
//
// HumdrumLine::getTrackEnd --  Returns the ending exclusive interpretation
//    for the given spine/track.
//

HTp HumdrumLine::getTrackEnd(int track, int subspine) const {
	if (!m_owner) {
		return NULL;
	} else {
		return ((HumdrumFile*)m_owner)->getTrackEnd(track, subspine);
	}
}



//////////////////////////////
//
// HumdrumLine::setDurationFromBarline -- Time from the previous
//    barline to the current line.  This function is used in analyzeMeter in
//    the HumdrumFileStructure class.
//

void HumdrumLine::setDurationFromBarline(HumNum dur) {
	m_durationFromBarline = dur;
}



//////////////////////////////
//
// HumdrumLine::getDurationToBarline -- Time from the starting of the
//   current note to the next barline.
//

HumNum HumdrumLine::getDurationToBarline(void) {
	if (!m_rhythm_analyzed) {
		if (getOwner()) {
			getOwner()->analyzeRhythmStructure();
		}
	}
	return m_durationToBarline;
}


HumNum HumdrumLine::getDurationToBarline(HumNum scale) {
	if (!m_rhythm_analyzed) {
		if (getOwner()) {
			getOwner()->analyzeRhythmStructure();
		}
	}
	return m_durationToBarline * scale;
}



//////////////////////////////
//
// HumdrumLine::getBeat -- Returns the beat number for the data on the
//     current line given the input **recip representation for the duration
//     of a beat.  The beat in a measure is offset from 1 (first beat is
//     1 rather than 0).
//  Default value: beatrecip = "4".
//  Default value: beatdur   = 1.
//

HumNum HumdrumLine::getBeat(HumNum beatdur) {
	if (beatdur.isZero()) {
		return beatdur;
	}
	HumNum beat = (getDurationFromBarline() / beatdur) + 1;
	return beat;
}


HumNum HumdrumLine::getBeatStr(string beatrecip) {
	HumNum beatdur = Convert::recipToDuration(beatrecip);
	if (beatdur.isZero()) {
		return beatdur;
	}
	HumNum beat = (getDurationFromBarline() / beatdur) + 1;
	return beat;
}



//////////////////////////////
//
// HumdrumLine::setDurationToBarline -- Sets the duration from the current
//     line to the next barline in the score.  This function is used by
//     analyzeMeter in the HumdrumFileStructure class.
//

void HumdrumLine::setDurationToBarline(HumNum dur) {
	m_durationToBarline = dur;
}



//////////////////////////////
//
// HumdrumLine::setDuration -- Sets the duration of the line.  This is done
//   in the rhythmic analysis for the HumdurmFileStructure class.
//

void HumdrumLine::setDuration(HumNum aDur) {
	if (aDur.isNonNegative()) {
		m_duration = aDur;
	} else {
		m_duration = 0;
	}
}



//////////////////////////////
//
// HumdrumLine::hasSpines -- Returns true if the line contains spines.  This
//   means the the line is not empty or a global comment (which can include
//   reference records.
//

bool HumdrumLine::hasSpines(void) const {
	return (isEmpty() || isCommentGlobal()) ? false : true;
}



//////////////////////////////
//
// HumdrumLine::isGlobal -- Returns true if the line is a global record: either
//   and empty record, a global comment or a reference record.
//

bool HumdrumLine::isGlobal(void) const {
	return !hasSpines();
}



//////////////////////////////
//
// HumdrumLine::equalFieldsQ --
//

bool HumdrumLine::equalFieldsQ(const string& exinterp, const string& value) {
	for (int i=0; i<getFieldCount(); i++) {
		HTp token = this->token(i);
		if (!token->isDataType(exinterp)) {
			return false;
		}
		if (*token != value) {
			return false;
		}
	}
	return true;
}



//////////////////////////////
//
// HumdrumLine::isManipulator -- Returns true if any tokens on the line are
//   manipulator interpretations.  Only null interpretations are allowed on
//   lines which contain manipulators, but the parser currently does not
//   enforce this rule.
//

bool HumdrumLine::isManipulator(void) const {
	for (int i=0; i<(int)m_tokens.size(); i++) {
		if (m_tokens[i]->isManipulator()) {
			return true;
		}
	}
	return false;
}



//////////////////////////////
//
// HumdrumLine::isEmpty -- Returns true if no characters on line.  A blank line
//   is technically disallowed in the classic Humdrum Toolkit programs, but it
//   is usually tolerated.  In humlib (and HumdrumExtras) empty lines with
//   no content (not even space characters) are allowed and treated as a
//   special class of line.
//

bool HumdrumLine::isEmpty(void) const {
	return (size() == 0) ? true : false;
}



//////////////////////////////
//
// HumdrumLine::getTokenCount --  Returns the number of tokens on the line.
//     This value is set by HumdrumFileBase in analyzeTokens.
//

int HumdrumLine::getTokenCount(void) const {
	return (int)m_tokens.size();
}



//////////////////////////////
//
// HumdrumLine::token -- Returns a reference to the given token on the line.
//    An invalid token index would be bad to give to this function as it
//    returns a reference rather than a pointer (which could be set to
//    NULL if invalid).  Perhaps this function will eventually throw an
//    error if the index is out of bounds.
//

HTp HumdrumLine::token(int index) const {
	return m_tokens[index];
}



//////////////////////////////
//
// HumdrumLine::getTokenString -- Returns a copy of the string component of
//     a token.  This code will return a segmentation fault if index is out of
//     range...
//

string HumdrumLine::getTokenString(int index) const {
	return (string(*m_tokens[index]));
}


//////////////////////////////
//
// HumdrumLine::createTokensFromLine -- Chop up a HumdrumLine string into
//     individual tokens.
//

int HumdrumLine::createTokensFromLine(void) {
	// delete previous tokens (will need to re-analyze structure
	// of file after this).
	for (int i=0; i < (int)m_tokens.size(); i++) {
		delete m_tokens[i];
		m_tokens[i] = NULL;
	}
	m_tokens.clear();
	m_tabs.clear();
	HTp token;
	char ch = 0;
	char lastch = 0;
	string tstring;

	if (this->size() == 0) {
		token = new HumdrumToken();
		token->setOwner(this);
		m_tokens.push_back(token);
		m_tokens.push_back(0);
	} else if (this->compare(0, 2, "!!") == 0) {
		token = new HumdrumToken(this->c_str());
		token->setOwner(this);
		m_tokens.push_back(token);
		m_tabs.push_back(0);
	} else {
		for (int i=0; i<(int)size(); i++) {
			lastch = ch;
			ch = getChar(i);
			if (ch == '\t') {
				// Parser now allows multiple tab characters in a
				// row to represent a single tab.
				if (lastch != '\t') {
					token = new HumdrumToken(tstring);
					token->setOwner(this);
					m_tokens.push_back(token);
					m_tabs.push_back(1);
					tstring.clear();
				} else {
					if (m_tabs.size() > 0) {
						m_tabs.back()++;
					}
				}
			} else {
				tstring += ch;
			}
		}
	}
	if (tstring.size() > 0) {
		token = new HumdrumToken(tstring);
		token->setOwner(this);
		m_tokens.push_back(token);
		m_tabs.push_back(0);
		tstring.clear();
	}

	return (int)m_tokens.size();
}



//////////////////////////////
//
// HumdrumLine::createLineFromTokens --  Re-generate a HumdrumLine string from
//    individual tokens on the line.  This function will be necessary to
//    run before printing a HumdrumFile if you have changed any tokens on the
//    line.  Otherwise, changes in the tokens will not be passed on to the
///   printing of the line.
//

void HumdrumLine::createLineFromTokens(void) {
	string& iline = *this;
	iline = "";
	// needed for empty lines for some reason:
	if (m_tokens.size()) {
		if (m_tokens.back() == NULL) {
			m_tokens.resize(m_tokens.size() - 1);
		}
	}
	for (int i=0; i<(int)m_tokens.size(); i++) {
		iline += (string)(*m_tokens.at(i));
		if (i < (int)m_tokens.size() - 1) {
			if ((int)m_tabs.size() <= i) {
				for (int j=0; j<(int)m_tokens.size() - (int)m_tabs.size(); j++) {
					m_tabs.push_back(1);
				}
			}
			if (m_tabs.at(i) == 0) {
				m_tabs.at(i) = 1;
			}
			for (int j=0; j<m_tabs.at(i); j++) {
				iline += '\t';
			}
		}
	}
}



//////////////////////////////
//
// HumdrumLine::removeExtraTabs -- Allow only for one tab between spine fields.
//    The tab counts are set to 0, but the line creation function knows to use
//    a minimum of one tab between fields.  The HumdrumFile::createLinesFromtTokens()
//    function should be called after running this function and before printing
//    the data so that the tabs in the text line can be updated.
//

void HumdrumLine::removeExtraTabs(void) {
	fill(m_tabs.begin(), m_tabs.end(), 0);
}



//////////////////////////////
//
// HumdrumLine::addExtraTabs -- Adds extra tabs between primary spines so that the
//    first token of a spine is vertically aligned.  The input array to this
//    function is a list of maximum widths.  This is typically caluclated by
//    HumdrumFileBase::getTrackWidths().  The first indexed value is unused,
//    since there is no track 0.
//

void HumdrumLine::addExtraTabs(vector<int>& trackWidths) {
	if (!this->hasSpines()) {
		return;
	}

	fill(m_tabs.begin(), m_tabs.end(), 1);
	vector<int> local(trackWidths.size(), 0);

	int lasttrack = 0;
	int track = 0;
	for (int j=0; j<getFieldCount(); j++) {
		lasttrack = track;
		HTp token = this->token(j);
		track = token->getTrack();
		if ((track != lasttrack) && (lasttrack > 0)) {
			int diff = trackWidths.at(lasttrack) - local.at(lasttrack);
			if ((diff > 0) && (j > 0)) {
				m_tabs.at(j-1) += diff;
			}
		}
		local.at(track)++;
	}
}



//////////////////////////////
//
// HumdrumLine::getTokens -- Returns an array of tokens pointers for a
//   Humdrum line.  This function should not be called on global comments,
//   reference records (which are a sub-cateogry of global comments).  This
//   is because a line's type may contain tabs which are not representing
//   token separators.  Empty lines are ok to input: the output token
//   list will contain one empty string.
//

void HumdrumLine::getTokens(vector<HTp>& list) {
	if (m_tokens.size() == 0) {
		createTokensFromLine();
	}
	list = m_tokens;
}



//////////////////////////////
//
// HumdrumLine::getChar -- Returns character at given index in string, or
//    null if out of range.
//

char HumdrumLine::getChar(int index) const {
	if (index < 0) {
		return '\0';
	}
	if (index >= (int)size()) {
		return '\0';
	}
	return (((string)(*this))[index]);
}



//////////////////////////////
//
// HumdrumLine::printSpineInfo -- Print the spine state information of
//    each token in a file.  Useful for debugging.  The spine info
//    is the track number, such as "1".  When the track splits into
//    subtracks, then there will be two subtracks: "(1)a" and "(1)b".
//    If the second of those subtracks splits again, then its subtracks
//    will be "((1)b)a" and "((1)b)b". If two different tracks merge, such
//    as "1" and "(2)a", then the spine info will be "1 (2)a".
//
// default value: out = cout
//

ostream& HumdrumLine::printSpineInfo(ostream& out) {
	if (isManipulator()) {
		out << *this;
	} else {
		for (int i=0; i<(int)m_tokens.size(); i++) {
			out << m_tokens[i]->getSpineInfo();
			if (i < (int)m_tokens.size() - 1) {
				out << '\t';
			}
		}
	}
	return out;
}



//////////////////////////////
//
// HumdrumLine::printDataTypeInfo -- Print the datatype of each token in
//     the file.  Useful for debugging.  The datatype prefix "**" is removed;
//     otherwise, it is given when you call HumdrumToken::getDataType().
//
// default value: out = cout
//

ostream& HumdrumLine::printDataTypeInfo(ostream& out) {
	if (isManipulator()) {
		out << *this;
	} else {
		for (int i=0; i<(int)m_tokens.size(); i++) {
			out << m_tokens[i]->getDataType().substr(2, string::npos);
			if (i < (int)m_tokens.size() - 1) {
				out << '\t';
			}
		}
	}
	return out;
}



//////////////////////////////
//
// HumdrumLine::analyzeTokenDurations -- Calculate the duration of
//    all tokens on a line.
//

bool HumdrumLine::analyzeTokenDurations(string& err) {
	if (!hasSpines()) {
		return !err.size();
	}
	for (int i=0; i<(int)m_tokens.size(); i++) {
		m_tokens[i]->analyzeDuration();
	}
	return !err.size();
}



//////////////////////////////
//
// HumdrumLine::analyzeTracks -- Calculate the subtrack info for subspines.
//   Subtracks index subspines strictly from left to right on the line.
//   Subspines can be exchanged and be represented left to right out of
//   original order.
//

bool HumdrumLine::analyzeTracks(string& err) {
	if (!hasSpines()) {
		return !err.size();
	}

	string info;
	int track;
	int maxtrack = 0;
	int i, j, k;

	for (i=0; i<(int)m_tokens.size(); i++) {
		info = m_tokens[i]->getSpineInfo();
		track = 0;
		for (j=0; j<(int)info.size(); j++) {
			if (!isdigit(info[j])) {
				continue;
			}
			track = info[j] - '0';
			for (k=j+1; k<(int)info.size(); k++) {
				if (isdigit(info[k])) {
					track = track * 10 + (info[k] - '0');
				} else {
					break;
				}
			}
			break;
		}
		if (maxtrack < track) {
			maxtrack = track;
		}
		m_tokens[i]->setTrack(track);
	}

	int subtrack;
	vector<int> subtracks;
	vector<int> cursub;

	subtracks.resize(maxtrack+1);
	cursub.resize(maxtrack+1);
	fill(subtracks.begin(), subtracks.end(), 0);
	fill(cursub.begin(), cursub.end(), 0);

	for (i=0; i<(int)m_tokens.size(); i++) {
		subtracks[m_tokens[i]->getTrack()]++;
	}
	for (i=0; i<(int)m_tokens.size(); i++) {
		track = m_tokens[i]->getTrack();
		subtrack = subtracks[track];
		if (subtrack > 1) {
			m_tokens[i]->setSubtrack(++cursub[m_tokens[i]->getTrack()]);
		} else {
			m_tokens[i]->setSubtrack(0);
		}
		m_tokens[i]->setSubtrackCount(subtracks[track]);
	}
	return !err.size();
}



//////////////////////////////
//
// HumdrumLine::printDurationInfo -- Print the analyzed duration of each
//     token in a file (for debugging).  If a token has an undefined
//     duration, then its duration is -1.  If a token represents
//     a grace note, then its duration is 0 (regardless of whether it
//     includes a visual duration).
// default value: out = cout
//

ostream& HumdrumLine::printDurationInfo(ostream& out) {
	if (isManipulator()) {
		out << *this;
	} else {
		for (int i=0; i<(int)m_tokens.size(); i++) {
			m_tokens[i]->getDuration().printMixedFraction(out);
			if (i < (int)m_tokens.size() - 1) {
				out << '\t';
			}
		}
	}
	return out;
}


//////////////////////////////
//
// HumdrumLine::printCsv -- print the line as a CSV
//    (comma separate value) line.
// default value: out = std::cout;
// default value: separator = ","
//

ostream& HumdrumLine::printCsv(ostream& out, const string& separator) {
	for (int i=0; i<getFieldCount(); i++) {
		token(i)->printCsv(out);
		if (i<getFieldCount()-1) {
			out << separator;
		}
	}
	out << endl;
	return out;
}



//////////////////////////////
//
// HumdrumLine::printGlobalXmlParameterInfo --
//

ostream& HumdrumLine::printGlobalXmlParameterInfo(ostream& out, int level,
		const string& indent) {
	token(0)->printGlobalXmlParameterInfo(out, level, indent);
	return out;
}



//////////////////////////////
//
// HumdrumLine::printXmlParameterInfo --
//

ostream& HumdrumLine::printXmlParameterInfo(ostream& out, int level,
		const string& indent) {
	((HumHash*)this)->printXml(out, level, indent);
	return out;
}



//////////////////////////////
//
// HumdrumLine::printXmlGlobalLinkedParameterInfo --
//

ostream&	HumdrumLine::printXmlGlobalLinkedParameterInfo(ostream& out, int level,
		const string& indent) {
	return out;
	// return token(0)->printXmlLinkedParameterInfo(out, level, indent);
}


//////////////////////////////
//
// HumdrumLine::printXmlGlobalLinkedParameters --
//

ostream& HumdrumLine::printXmlGlobalLinkedParameters(ostream& out, int level, const string& indent) {
	return out;
	// return token(0)->printXmlLinkedParameters(out, level, indent);
}


//////////////////////////////
//
// HumdrumLine::printXml -- Print the HumdrumLine as a XML element.
//

ostream& HumdrumLine::printXml(ostream& out, int level, const string& indent) {

	if (hasSpines()) {
		out << Convert::repeatString(indent, level) << "<frame";
		out << " n=\"" << getLineIndex() << "\"";
		out << " xml:id=\"" << getXmlId() << "\"";
		out << ">\n";
		level++;

		out << Convert::repeatString(indent, level) << "<frameInfo>\n";
		level++;

		out << Convert::repeatString(indent, level) << "<fieldCount>";
		out << getTokenCount() << "</fieldCount>\n";

		out << Convert::repeatString(indent, level);
		out << "<frameStart";
		out << Convert::getHumNumAttributes(getDurationFromStart());
		out << "/>\n";

		out << Convert::repeatString(indent, level);
		out << "<frameDuration";
		out << Convert::getHumNumAttributes(getDuration());
		out << "/>\n";

		out << Convert::repeatString(indent, level) << "<frameType>";
		if (isData()) {
			out << "data";
		} else if (isBarline()) {
			out << "barline";
		} else if (isInterpretation()) {
			out << "interpretation";
		} else if (isLocalComment()) {
			out << "local-comment";
		}
		out << "</frameType>\n";

		if (isBarline()) {
			// print the duration to the next barline or to the end of the score
			// if there is no barline at the end of the score.
			out << Convert::repeatString(indent, level);
			out << "<barlineDuration";
			out << Convert::getHumNumAttributes(getBarlineDuration());
			out << "/>\n";
		}

		bool bstart = isKernBoundaryStart();
		bool bend   = isKernBoundaryEnd();
		if (bstart || bend) {
			out << Convert::repeatString(indent, level);
			cout << "<kernBoundary";
			cout << " start=\"";
			if (bstart) {
				cout << "true";
			} else {
				cout << "false";
			}
			cout << "\"";
			cout << " end=\"";
			if (bend) {
				cout << "true";
			} else {
				cout << "false";
			}
			cout << "\"";
			cout << "/>\n";
		}

		level--;
		out << Convert::repeatString(indent, level) << "</frameInfo>\n";

		out << Convert::repeatString(indent, level) << "<fields>\n";
		level++;
		for (int i=0; i<getFieldCount(); i++) {
			token(i)->printXml(out, level, indent);
		}
		level--;
		out << Convert::repeatString(indent, level) << "</fields>\n";

		printGlobalXmlParameterInfo(out, level, indent);
		printXmlParameterInfo(out, level, indent);
		printXmlGlobalLinkedParameterInfo(out, level, indent);
		printXmlGlobalLinkedParameters(out, level, indent);

		level--;
		out << Convert::repeatString(indent, level) << "</frame>\n";

	} else {
		// global comments, reference records, or blank lines print here.
		out << Convert::repeatString(indent, level) << "<metaFrame";
		out << " n=\"" << getLineIndex() << "\"";
		out << " token=\"" << Convert::encodeXml(((string)(*this))) << "\"";
		out << " xml:id=\"" << getXmlId() << "\"";
		out << ">\n";
		level++;

		out << Convert::repeatString(indent, level) << "<frameInfo>\n";
		level++;

		out << Convert::repeatString(indent, level);
		out << "<startTime";
		out << Convert::getHumNumAttributes(getDurationFromStart());
		out << "/>\n";

		out << Convert::repeatString(indent, level) << "<frameType>";
		if (isReference()) {
			out << "reference";
		} else if (isBlank()) {
			out << "empty";
		} else {
			out << "global-comment";
		}
		out << "</frameType>\n";

		if (isReference()) {
			out << Convert::repeatString(indent, level);
			string key = getReferenceKey();
			string language;
			string primaryLanguage;
			auto loc = key.find("@@");
			if (loc != string::npos) {
				language = key.substr(loc+2);
				key = key.substr(0, loc);
				primaryLanguage = "true";
			} else {
				loc = key.find("@");
				if (loc != string::npos) {
					language = key.substr(loc+1);
					key = key.substr(0, loc);
				}
			}

			out << "<referenceKey";
			if (language.size() > 0) {
				out << " language=\"" << Convert::encodeXml(language) << "\"";
			}
			if (primaryLanguage.size() > 0) {
				out << " primary=\"" << Convert::encodeXml(primaryLanguage) << "\"";
			}
			out << ">" << Convert::encodeXml(key);
			out << "</referenceKey>\n";

			out << Convert::repeatString(indent, level);
			out << "<referenceValue>" << Convert::encodeXml(getGlobalReferenceValue());
			out << "</referenceValue>\n";
		}

		level--;
		out << Convert::repeatString(indent, level) << "</frameInfo>\n";

		printGlobalXmlParameterInfo(out, level-2, indent);
		printXmlParameterInfo(out, level-2, indent);
		printXmlGlobalLinkedParameterInfo(out, level-2, indent);
		printXmlGlobalLinkedParameters(out, level, indent);

		level--;
		out << Convert::repeatString(indent, level) << "</metaFrame>\n";
	}


	return out;
}



//////////////////////////////
//
// HumdrumLine::getXmlId -- Return a unique ID for the current line.
//

string HumdrumLine::getXmlId(const string& prefix) const {
	string output;
	if (prefix.size() > 0) {
		output = prefix;
	} else {
		output = getXmlIdPrefix();
	}
	output += "L" + to_string(getLineIndex() + 1);
	return output;
}



//////////////////////////////
//
// HumdrumLine::getXmlIdPrefix -- Return the pre-set XML ID attribute
//     prefix from the owning HumdrumFile object.
//

string HumdrumLine::getXmlIdPrefix(void) const {
	if (!m_owner) {
		return "";
	}
	return ((HumdrumFileBase*)m_owner)->getXmlIdPrefix();
}



//////////////////////////////
//
// HumdrumLine::printTrackInfo -- Print the analyzed track information.
//     The first (left-most) spine in a Humdrum file is track 1, the
//     next is track 2, etc.  The track value is shared by all subspines,
//     so there may be duplicate track numbers on a line if the spine
//     has split.  When the spine splits, a subtrack number is given
//     after a "." character in the printed output from this function.
//     Subtrack==0 means that there is only one subtrack.
//     Examples:
//         "1"  == Track 1, subtrack 1 (and there are no more subtracks)
//	        "1.1" == Track 1, subtrack 1 (and there are more subtracks)
//	        "1.2" == Track 1, subtrack 2 (and there may be more subtracks)
//	        "1.10" == Track 1, subtrack 10 (and there may be subtracks)
//     Each starting exclusive interpretation is assigned to a unique
//     track number.  When a *+ manipulator is given, the new exclusive
//     interpretation on the next line is give the next higher track
//     number.
//
// default value: out = cout
//

ostream& HumdrumLine::printTrackInfo(ostream& out) {
	if (isManipulator()) {
		out << *this;
	} else {
		for (int i=0; i<(int)m_tokens.size(); i++) {
			out << m_tokens[i]->getTrackString();
			if (i < (int)m_tokens.size() - 1) {
				out << '\t';
			}
		}
	}
	return out;
}



//////////////////////////////
//
// HumdrumLine::setOwner -- store a pointer to the HumdrumFile which
//    manages (owns) this object.
//

void HumdrumLine::setOwner(void* hfile) {
	m_owner = hfile;
}



//////////////////////////////
//
// HumdrumLine::getOwner -- Return the HumdrumFile which manages
//   (owns) this line.
//

HumdrumFile* HumdrumLine::getOwner(void) {
	return (HumdrumFile*)m_owner;
}



//////////////////////////////
//
// HumdrumLine::addLinkedParameter --
//

int HumdrumLine::addLinkedParameter(HTp token) {
	for (int i=0; i<(int)m_linkedParameters.size(); i++) {
		if (m_linkedParameters[i] == token) {
			return i;
		}
	}

	m_linkedParameters.push_back(token);
	return (int)m_linkedParameters.size() - 1;
}



//////////////////////////////
//
// HumdrumLine::setLayoutParameters -- Takes a global comment with
//     the structure:
//        !!LO:NS2:key1=value1:key2=value2:key3=value3
//     and stores it in the HumHash parent class of the line.
//

void HumdrumLine::setLayoutParameters(void) {
	if (this->find("!!LO:") == string::npos) {
		return;
	}
	string pdata = this->substr(2, string::npos);
	setParameters(pdata);
}


//////////////////////////////
//
// HumdrumLine::setParameters -- Store global parameters in the first token
//    of the line.  Also add a marker at ("","","global","true") to indicate
//    that the parameters are global rather than local.  (Global text directions
//    will behave differently from local text directions, for example).
//

void HumdrumLine::setParameters(const string& pdata) {
	vector<string> pieces = Convert::splitString(pdata, ':');
	if (pieces.size() < 3) {
		return;
	}
	string ns1 = pieces[0];
	string ns2 = pieces[1];
	string key;
	string value;
	int loc;
	for (int i=2; i<(int)pieces.size(); i++) {
		Convert::replaceOccurrences(pieces[i], "&colon;", ":");
		loc = (int)pieces[i].find("=");
		if (loc != (int)string::npos) {
			key   = pieces[i].substr(0, loc);
			value = pieces[i].substr(loc+1, pieces[i].size());
		} else {
			key   = pieces[i];
			value = "true";
		}
		token(0)->setValue(ns1, ns2, key, value);
	}
	token(0)->setValue("global", "true");
}



//////////////////////////////
//
// HumdrumLine::appendToken -- add a token at the end of the current
//      list of tokens in the line.
//

void HumdrumLine::appendToken(HTp token, int tabcount) {
	// deletion will be handled by class.
	m_tokens.push_back(token);
	m_tabs.push_back(tabcount);
}


void HumdrumLine::appendToken(const HumdrumToken& token, int tabcount) {
	HTp newtok = new HumdrumToken(token);
	m_tokens.push_back(newtok);
	m_tabs.push_back(tabcount);
}


void HumdrumLine::appendToken(const string& token, int tabcount) {
	HTp newtok = new HumdrumToken(token);
	m_tokens.push_back(newtok);
	m_tabs.push_back(tabcount);
}


void HumdrumLine::appendToken(const char* token, int tabcount) {
	HTp newtok = new HumdrumToken(token);
	m_tokens.push_back(newtok);
	m_tabs.push_back(tabcount);
}



//////////////////////////////
//
// HumdrumLine::getKernNoteAttacks -- Return the number of kern notes
//    that attack on a line.
//

int HumdrumLine::getKernNoteAttacks(void) {
	int output = 0;
	for (int i=0; i<getFieldCount(); i++) {
		if (!token(i)->isKern()) {
			continue;
		}
		if (token(i)->isNoteAttack()) {
			output++;
		}
	}
	return output;
}



//////////////////////////////
//
// HumdrumLine::insertToken -- Add a token before the given token position.
//

void HumdrumLine::insertToken(int index, HTp token, int tabcount) {
	// Warning: deletion will be handled by class.  Don't insert if it
	// already belongs to another HumdrumLine or HumdrumFile.
	m_tokens.insert(m_tokens.begin() + index, token);
	m_tabs.insert(m_tabs.begin() + index, tabcount);
}


void HumdrumLine::insertToken(int index, const HumdrumToken& token, int tabcount) {
	HTp newtok = new HumdrumToken(token);
	m_tokens.insert(m_tokens.begin() + index, newtok);
	m_tabs.insert(m_tabs.begin() + index, tabcount);
}


void HumdrumLine::insertToken(int index, const string& token, int tabcount) {
	HTp newtok = new HumdrumToken(token);
	m_tokens.insert(m_tokens.begin() + index, newtok);
	m_tabs.insert(m_tabs.begin() + index, tabcount);
}


void HumdrumLine::insertToken(int index, const char* token, int tabcount) {
	HTp newtok = new HumdrumToken(token);
	m_tokens.insert(m_tokens.begin() + index, newtok);
	m_tabs.insert(m_tabs.begin() + index, tabcount);
}



//////////////////////////////
//
// HumdrumLine::appendToken -- Add a token after the given token position.
//

void HumdrumLine::appendToken(int index, HTp token, int tabcount) {
	HumdrumLine::insertToken(index+1, token, tabcount);
}


void HumdrumLine::appendToken(int index, const HumdrumToken& token, int tabcount) {
	HumdrumLine::insertToken(index+1, token, tabcount);
}


void HumdrumLine::appendToken(int index, const string& token, int tabcount) {
	HumdrumLine::insertToken(index+1, token, tabcount);
}


void HumdrumLine::appendToken(int index, const char* token, int tabcount) {
	HumdrumLine::insertToken(index+1, token, tabcount);
}



//////////////////////////////
//
// HumdrumLine::storeGlobalLinkedParameters --
//

void HumdrumLine::storeGlobalLinkedParameters(void) {
	token(0)->storeParameterSet();
}



//////////////////////////////
//
// HumdrumLine::getBarNumber -- return the bar number on the line.
//    If the line is not a bar line, then return -1.  If there is
//    no number at any token position on the line then return -1.
//

int HumdrumLine::getBarNumber(void) {
	if (!isBarline()) {
		return -1;
	}
	int output = -1;
	for (int i=0; i<getFieldCount(); i++) {
		HTp tok = token(i);
		if (tok->size() < 2) {
			return -1;
		}
		if (isdigit(tok->at(1))) {
			sscanf(tok->c_str(), "=%d", &output);
			if (output >= 0) {
				return output;
			}
		}
	}
	return -1;
}



//////////////////////////////
//
// HumdrumLine::copyStructure -- For data lines only at the moment.
//

void HumdrumLine::copyStructure(HLp line, const string& empty) {
		m_tokens.resize(line->m_tokens.size());
		for (int i=0; i<(int)m_tokens.size(); i++) {
			m_tokens[i] = new HumdrumToken(empty);
			m_tokens[i]->setOwner(this);
			m_tokens[i]->copyStructure(line->m_tokens[i]);
		}
		createLineFromTokens();

		m_tabs = line->m_tabs;
		m_linkedParameters.clear();
		m_rhythm_analyzed = line->m_rhythm_analyzed;
		m_owner = line->m_owner;

		// Other information that should be set later:
		//    int m_lineindex;
		//    HumNum m_durationFromStart;
		//    HumNum m_durationFromBarline;
		//    HumNum m_durationToBarline;
}



/////////////////////////////
//
// HumdrumLine::allSameStyle -- return true if barlines through all 
//     staves are the same. Requires HumdrumFile::analyzeBarlines() to be
//     run first.
//

bool HumdrumLine::allSameBarlineStyle(void) {
	return !this->getValueInt("auto", "barlinesDifferent");
}



//////////////////////////////
//
// operator<< -- Print a HumdrumLine. Needed to avoid interaction with
//     HumHash parent class.
//

ostream& operator<<(ostream& out, HumdrumLine& line) {
	out << (string)line;
	return out;
}

ostream& operator<< (ostream& out, HLp line) {
	out << (string)(*line);
	return out;
}



// spine mainipulators:
#define SPLIT_TOKEN       "*^"
#define MERGE_TOKEN       "*v"
#define EXCHANGE_TOKEN    "*x"
#define TERMINATE_TOKEN   "*-"
#define ADD_TOKEN         "*+"
// Also exclusive interpretations which start "**" followed by the data type.

// other special tokens:
#define NULL_DATA            "."
#define NULL_INTERPRETATION  "*"
#define NULL_COMMENT_LOCAL   "!"
#define NULL_COMMENT_GLOBAL  "!!"



//////////////////////////////
//
// HumdrumToken::HumdrumToken -- Constructor for HumdrumToken.
//

HumdrumToken::HumdrumToken(void) : string() {
	m_rhycheck = 0;
	setPrefix("!");
	m_strand = -1;
	m_nullresolve = NULL;
	m_strophe     = NULL;
}


HumdrumToken::HumdrumToken(const string& aString) : string(aString) {
	m_rhycheck = 0;
	setPrefix("!");
	m_strand = -1;
	m_nullresolve = NULL;
	m_strophe     = NULL;
}


HumdrumToken::HumdrumToken(const char* aString) : string(aString) {
	m_rhycheck = 0;
	setPrefix("!");
	m_strand = -1;
	m_nullresolve = NULL;
	m_strophe     = NULL;
}


HumdrumToken::HumdrumToken(const HumdrumToken& token) :
		string((string)token), HumHash((HumHash)token) {
	m_address         = token.m_address;
	m_address.m_owner = NULL;
	m_duration        = token.m_duration;
	m_nextTokens      = token.m_nextTokens;
	m_previousTokens.clear();
	m_nextNonNullTokens.clear();
	m_previousNonNullTokens.clear();
	m_rhycheck        = token.m_rhycheck;
	m_strand          = -1;
	m_nullresolve     = NULL;
	m_strophe         = NULL;
	setPrefix(token.getPrefix());
}


HumdrumToken::HumdrumToken(HumdrumToken* token) :
		string((string)(*token)), HumHash((HumHash)(*token)) {
	m_address         = token->m_address;
	m_address.m_owner = NULL;
	m_duration        = token->m_duration;
	m_nextTokens      = token->m_nextTokens;
	m_previousTokens.clear();
	m_nextNonNullTokens.clear();
	m_previousNonNullTokens.clear();
	m_rhycheck        = token->m_rhycheck;
	m_strand          = -1;
	m_nullresolve     = NULL;
	m_strophe         = NULL;
	setPrefix(token->getPrefix());
}



HumdrumToken::HumdrumToken(const HumdrumToken& token, HLp owner) :
		string((string)token), HumHash((HumHash)token) {
	m_address         = token.m_address;
	m_address.m_owner = owner;
	m_duration        = token.m_duration;
	m_nextTokens      = token.m_nextTokens;
	m_previousTokens.clear();
	m_nextNonNullTokens.clear();
	m_previousNonNullTokens.clear();
	m_rhycheck        = token.m_rhycheck;
	m_strand          = -1;
	m_nullresolve     = NULL;
	m_strophe         = NULL;
	setPrefix(token.getPrefix());
}


HumdrumToken::HumdrumToken(HumdrumToken* token, HLp owner) :
		string((string)(*token)), HumHash((HumHash)(*token)) {
	m_address         = token->m_address;
	m_address.m_owner = owner;
	m_duration        = token->m_duration;
	m_nextTokens      = token->m_nextTokens;
	m_previousTokens.clear();
	m_nextNonNullTokens.clear();
	m_previousNonNullTokens.clear();
	m_rhycheck        = token->m_rhycheck;
	m_strand          = -1;
	m_nullresolve     = NULL;
	m_strophe         = NULL;
	setPrefix(token->getPrefix());
}



//////////////////////////////
//
// HumdrumToken::operator= -- Copy operator.
//

HumdrumToken& HumdrumToken::operator=(HumdrumToken& token) {
	if (this == &token) {
		return *this;
	}
	(string)(*this)   = (string)token;
	(HumHash)(*this)  = (HumHash)token;

	m_address         = token.m_address;
	m_address.m_owner = NULL;
	m_duration        = token.m_duration;
	m_nextTokens      = token.m_nextTokens;
	m_previousTokens.clear();
	m_nextNonNullTokens.clear();
	m_previousNonNullTokens.clear();
	m_rhycheck        = token.m_rhycheck;
	m_strand          = -1;
	m_nullresolve     = NULL;
	m_strophe         = NULL;
	setPrefix(token.getPrefix());

	return *this;
}


HumdrumToken& HumdrumToken::operator=(const string& token) {
	(string)(*this) = token;

	m_address.m_owner = NULL;
	m_duration        = 0;
	m_nextTokens.clear();
	m_previousTokens.clear();
	m_nextNonNullTokens.clear();
	m_previousNonNullTokens.clear();
	m_rhycheck        = -1;
	m_strand          = -1;
	m_nullresolve     = NULL;
	m_strophe         = NULL;
	setPrefix("!");

	return *this;
}


HumdrumToken& HumdrumToken::operator=(const char* token) {
	(string)(*this) = token;

	m_address.m_owner = NULL;
	m_duration        = 0;
	m_nextTokens.clear();
	m_previousTokens.clear();
	m_nextNonNullTokens.clear();
	m_previousNonNullTokens.clear();
	m_rhycheck        = -1;
	m_strand          = -1;
	m_nullresolve     = NULL;
	m_strophe         = NULL;
	setPrefix("!");

	return *this;
}



//////////////////////////////
//
// HumdrumToken::~HumdrumToken -- Deconstructor for HumdrumToken.
//

HumdrumToken::~HumdrumToken() {
	if (m_parameterSet) {
		delete m_parameterSet;
		m_parameterSet = NULL;
	}
}


//////////////////////////////
//
// HumdrumToken::equalChar -- Returns true if the character at the given
//     index is the given char.
//

bool HumdrumToken::equalChar(int index, char ch) const {
	if ((int)size() <= index) {
		return false;
	}
	if (index < 0) {
		return false;
	}
	if (((string)(*this))[index] == ch) {
		return true;
	} else {
		return false;
	}
}



//////////////////////////////
//
// HumdrumToken::getPreviousNonNullDataTokenCount -- Returns the number of
//   previous tokens in the spine which is not a null token.  For null
//   tokens, this will be a count of the number of non-null tokens which
//   the null represents.
// @SEEALSO: getPreviousNonNullDataToken
//

int HumdrumToken::getPreviousNonNullDataTokenCount(void) {
	return (int)m_previousNonNullTokens.size();
}


//////////////////////////////
//
// HumdrumToken::insertTokenAfter -- Insert the given token after the this token.
//    This will sever the link from this token to its next token.  There is only
//    presumed to be one next token, at least for the moment.
//
//

void HumdrumToken::insertTokenAfter(HTp newtoken) {
	if (m_nextTokens.empty()) {
		m_nextTokens.push_back(newtoken);
	} else {
		HTp oldnexttoken = m_nextTokens[0];
		m_nextTokens[0] = newtoken;
		newtoken->m_previousTokens.clear();
		newtoken->m_previousTokens.push_back(this);
		newtoken->m_nextTokens.clear();
		newtoken->m_nextTokens.push_back(oldnexttoken);
		if (oldnexttoken->m_previousTokens.empty()) {
			oldnexttoken->m_previousTokens.push_back(newtoken);
		} else {
			oldnexttoken->m_previousTokens[0] = newtoken;
		}
	}
}



//////////////////////////////
//
// HumdrumToken::getPreviousNonNullDataToken -- Returns the non-null
//    data token which occurs before this token in the data in the same
//    spine.  The default value is index 0, since mostly there will only
//    be one previous token.
//

HumdrumToken* HumdrumToken::getPreviousNonNullDataToken(int index) {
	if (index < 0) {
		index += (int)m_previousNonNullTokens.size();
	}
	if (index < 0) {
		return NULL;
	}
	if (index >= (int)m_previousNonNullTokens.size()) {
		return NULL;
	}
	return m_previousNonNullTokens[index];
}



//////////////////////////////
//
// HumdrumToken::getNextNonNullDataTokenCount -- Returns the number of non-null
//     data tokens which follow this token in the spine.
//

int HumdrumToken::getNextNonNullDataTokenCount(void) {
	return (int)m_nextNonNullTokens.size();
}



//////////////////////////////
//
// HumdrumToken::getNextNonNullDataToken -- Returns the given next non-null token
//    following this one in the spine.  The default value for index is 0 since
//    the next non-null data token count will typically be 1.
// default value: index = 0
//

HumdrumToken* HumdrumToken::getNextNonNullDataToken(int index) {
	if (index < 0) {
		index += (int)m_nextNonNullTokens.size();
	}
	if (index < 0) {
		return NULL;
	}
	if (index >= (int)m_nextNonNullTokens.size()) {
		return NULL;
	}
	return m_nextNonNullTokens[index];
}



//////////////////////////////
//
// HumdrumToken::getSlurDuration -- If the note has a slur start, then
//    returns the duration until the endpoint; otherwise, returns 0;
//    Expand later to handle slur ends and elided slurs.  The function
//    HumdrumFileContent::analyzeSlurs() should be called before accessing
//    this function.  If the slur duruation was already calculated, return
//    that value; otherwise, calculate from the location of a matching
//    slur end.
//

HumNum HumdrumToken::getSlurDuration(HumNum scale) {
	if (!isDataType("**kern")) {
		return 0;
	}
	if (isDefined("auto", "slurDuration")) {
		return getValueFraction("auto", "slurDuration");
	} else if (isDefined("auto", "slurEnd")) {
		HTp slurend = getValueHTp("auto", "slurEnd");
		return slurend->getDurationFromStart(scale) -
				getDurationFromStart(scale);
	} else {
		return 0;
	}
}



//////////////////////////////
//
// HumdrumToken::getDataType -- Get the exclusive interpretation type for
//     the token.
// @SEEALSO: isDataType
//

const string& HumdrumToken::getDataType(void) const {
	return m_address.getDataType();
}



//////////////////////////////
//
// HumdrumToken::isDataType -- Returns true if the data type of the token
//   matches the test data type.
// @SEEALSO: getDataType getKern
//

bool HumdrumToken::isDataType(const string& dtype) const {
	if (dtype.compare(0, 2, "**") == 0) {
		return dtype == getDataType();
	} else {
		return getDataType().compare(2, string::npos, dtype) == 0;
	}
}



//////////////////////////////
//
// HumdrumToken::isKern -- Returns true if the data type of the token
//    is **kern.
// @SEEALSO: isDataType
//

bool HumdrumToken::isKern(void) const {
	return isDataType("**kern");
}



//////////////////////////////
//
// HumdrumToken::isMens -- Returns true if the data type of the token
//    is **mens.
// @SEEALSO: isDataType
//

bool HumdrumToken::isMens(void) const {
	return isDataType("**mens");
}



//////////////////////////////
//
// HumdrumToken::setSpineInfo -- Sets the spine manipulation history string.
// @SEEALTO: getSpineInfo
//

void HumdrumToken::setSpineInfo(const string& spineinfo) {
	m_address.setSpineInfo(spineinfo);
}



//////////////////////////////
//
// HumdrumToken::getSpineInfo -- Returns the spine split/merge history
//    for the token.
// @SEEALTO: setSpineInfo
//

string HumdrumToken::getSpineInfo(void) const {
	return m_address.getSpineInfo();
}



//////////////////////////////
//
// HumdrumToken::getLineIndex -- Returns the line index of the owning
//    HumdrumLine for this token.
// @SEEALTO: getLineNumber
//

int HumdrumToken::getLineIndex(void) const {
	return m_address.getLineIndex();
}



//////////////////////////////
//
// HumdrumToken::getFieldIndex -- Returns the index of the token the line.
// @SEEALSO: getFieldIndex
//

int HumdrumToken::getFieldIndex(void) const {
	return m_address.getFieldIndex();
}



//////////////////////////////
//
// HumdrumToken::getFieldNumber -- Returns the index of the token the line.
// @SEEALSO: getFieldNumber
//

int HumdrumToken::getFieldNumber(void) const {
	return m_address.getFieldIndex() + 1;
}



//////////////////////////////
//
// HumdrumToken::getTokenIndex -- Returns the index of the token the line.
// @SEEALSO: getTokenIndex
//

int HumdrumToken::getTokenIndex(void) const {
	return m_address.getFieldIndex();
}



//////////////////////////////
//
// HumdrumToken::getTokenNumber -- Returns the index of the token the line.
// @SEEALSO: getFieldNumber
//

int HumdrumToken::getTokenNumber(void) const {
	return m_address.getFieldIndex() + 1;
}



//////////////////////////////
//
// HumdrumToken::getLineNumber -- Returns the line index plus 1.
// @SEEALTO: getLineIndex
//

int HumdrumToken::getLineNumber(void) const {
	return m_address.getLineNumber();
}



//////////////////////////////
//
// HumdrumToken::setFieldIndex -- Sets the field index of the token on the
//   owning HumdrumLine object.
// @SEEALSO: getFieldIndex
//

void HumdrumToken::setFieldIndex(int index) {
	m_address.setFieldIndex(index);
}



//////////////////////////////
//
// HumdrumToken::setTrack -- Sets the track number (similar to a staff in MEI).
//     The two-parameter version will set the track and sub-track at the same
//     time (subtrack is similar to a staff and layer in MEI).
//

void HumdrumToken::setTrack(int aTrack) {
	m_address.setTrack(aTrack);
}


void HumdrumToken::setTrack(int aTrack, int aSubtrack) {
	setTrack(aTrack);
	setSubtrack(aSubtrack);
}



//////////////////////////////
//
// HumdrumToken::getTrack -- Get the track (similar to a staff in MEI).
//

int HumdrumToken::getTrack(void) const {
	return m_address.getTrack();
}



//////////////////////////////
//
// HumdrumToken::setSubtrack -- Sets the subtrack (similar to a layer
//    in MEI).
//

void HumdrumToken::setSubtrack(int aSubtrack) {
	m_address.setSubtrack(aSubtrack);
}



//////////////////////////////
//
// HumdrumToken::setSubtrackCount -- Sets the subtrack count in the
//    HumdrumLine for all tokens in the same track as the current
//    token.
//

void HumdrumToken::setSubtrackCount(int count) {
	m_address.setSubtrackCount(count);
}



//////////////////////////////
//
// HumdrumToken::setPreviousToken --
//

void HumdrumToken::setPreviousToken(HumdrumToken* token) {
	m_previousTokens.resize(1);
	m_previousTokens[0] = token;
}



//////////////////////////////
//
// HumdrumToken::setNextToken --
//

void HumdrumToken::setNextToken(HumdrumToken* token) {
	m_nextTokens.resize(1);
	m_nextTokens[0] = token;
}



//////////////////////////////
//
// HumdrumToken::addNextNonNullToken --
//

void HumdrumToken::addNextNonNullToken(HTp token) {
	if (token == NULL) {
		return;
	}
	for (int i=0; i<(int)m_nextNonNullTokens.size(); i++) {
		if (token == m_nextNonNullTokens[i]) {
			return;
		}
	}
	m_nextNonNullTokens.push_back(token);
	// maybe should sort by track/subspine order...
}



//////////////////////////////
//
// HumdrumToken::getNextToken -- Returns the next token in the
//    spine.  Since the next token count is usually one, the default
//    index value is zero.  When there is no next token (when the current
//    token is a spine terminaor), then NULL will be returned.
// default value: index = 0
// @SEEALSO: getNextTokens, getPreviousToken
//

HTp HumdrumToken::getNextToken(int index) const {
	if ((index >= 0) && (index < (int)m_nextTokens.size())) {
		return m_nextTokens[index];
	} else {
		return NULL;
	}
}



//////////////////////////////
//
// HumdrumToken::getNextTokens -- Returns a list of the next
//   tokens in the spine after this token.
// @SEEALSO: getNextToken
//

vector<HumdrumToken*> HumdrumToken::getNextTokens(void) const {
	return m_nextTokens;
}



//////////////////////////////
//
// HumdrumToken::getPreviousTokens -- Returns a list of the previous
//    tokens in the spine before this token.
//

vector<HumdrumToken*> HumdrumToken::getPreviousTokens(void) const {
	return m_previousTokens;
}



//////////////////////////////
//
// HumdrumToken::getPreviousToken -- Returns the previous token in the
//    spine.  Since the previous token count is usually one, the default
//    index value is zero.
// default value: index = 0
//

HumdrumToken* HumdrumToken::getPreviousToken(int index) const {
	if ((index >= 0) && (index < (int)m_previousTokens.size())) {
		return m_previousTokens[index];
	} else {
		return NULL;
	}
}


//////////////////////////////
//
// HumdrumToken::getNextFieldToken --
//

HTp HumdrumToken::getNextFieldToken(void) const {
	HLp line = getLine();
	if (!line) {
		return NULL;
	}
	int field = getFieldIndex();
	if (field >= line->getFieldCount()  - 1) {
		return NULL;
	}
	return line->token(field+1);
}



//////////////////////////////
//
// HumdrumToken::getPreviousFieldToken --
//

HTp HumdrumToken::getPreviousFieldToken(void) const {
	HLp line = getLine();
	if (!line) {
		return NULL;
	}
	int field = getFieldIndex();
	if (field < 1) {
		return NULL;
	}
	return line->token(field-1);
}



//////////////////////////////
//
// HumdrumToken::analyzeDuration -- Currently reads the duration of
//   **kern and **recip data.  Add more data types here such as **koto.
//

bool HumdrumToken::analyzeDuration(void) {
	m_rhythm_analyzed = true;
	if ((*this) == NULL_DATA) {
		m_duration.setValue(-1);
		return true;
	}
	if (equalChar(0 ,'!')) {
		m_duration.setValue(-1);
		return true;
	}
	if (equalChar(0 ,'*')) {
		m_duration.setValue(-1);
		return true;
	}
	if (equalChar(0 ,'=')) {
		m_duration.setValue(-1);
		return true;
	}
	string dtype = getDataType();
	if (hasRhythm()) {
		if (isData()) {
			if (!isNull()) {
				if (isKern()) {
					if (strchr(this->c_str(), 'q') != NULL) {
						m_duration = 0;
					} else {
						m_duration = Convert::recipToDuration((string)(*this));
					}
				} else if (isMens()) {
					m_duration = Convert::mensToDuration((string)(*this));
				}
			} else {
				m_duration.setValue(-1);
			}
		} else {
			m_duration.setValue(-1);
		}

	} else {
		m_duration.setValue(-1);
	}
	return true;
}



///////////////////////////////
//
// HumdrumToken::isManipulator -- Returns true if token is one of:
//    SPLIT_TOKEN     = "*^"  == spine splitter
//    MERGE_TOKEN     = "*v"  == spine merger
//    EXCHANGE_TOKEN  = "*x"  == spine exchanger
//    ADD_TOKEN       = "*+"  == spine adder
//    TERMINATE_TOKEN = "*-"  == spine terminator
//    **...  == exclusive interpretation
//

bool HumdrumToken::isManipulator(void) const {
	if (isSplitInterpretation())     { return true; }
	if (isMergeInterpretation())     { return true; }
	if (isExchangeInterpretation())  { return true; }
	if (isAddInterpretation())       { return true; }
	if (isTerminateInterpretation()) { return true; }
	if (isExclusiveInterpretation()) { return true; }
	return false;
}



//////////////////////////////
//
// HumdrumToken::getDuration -- Returns the duration of the token.  The token
//    does not necessarily need to have any explicit duration, as the returned
//    value will also include implicit duration calculated in analyzeRhythm
//    in the HumdrumFileStructure class.
//

HumNum HumdrumToken::getDuration(void) {
	if (!m_rhythm_analyzed) {
		analyzeDuration();
	}
	return m_duration;
}


HumNum HumdrumToken::getDuration(HumNum scale) {
	if (!m_rhythm_analyzed) {
		analyzeDuration();
	}
	return m_duration * scale;
}



//////////////////////////////
//
// HumdrumToken::getTiedDuration -- Returns the duration of the token and any
//    tied notes attached to it.  Does not work well which chords.  Does
//    not work well with secondary spine splits.
//

HumNum HumdrumToken::getTiedDuration(void) {
	if (!m_rhythm_analyzed) {
		analyzeDuration();
	}
	HumNum output = m_duration;
	// start of a tied group so add the durations of the other notes.
	int b40 = Convert::kernToBase40(this);
	HTp note = this;
	HTp nnote = NULL;
	while (note) {
		nnote = note->getNextNNDT();
		if (!nnote) {
			break;
		}
		if (!nnote->isSecondaryTiedNote()) {
			break;
		}
		int nb40 = Convert::kernToBase40(this);
		if (nb40 != b40) {
			break;
		}
		// note is tied to previous one, so add its curation to output.
		output += nnote->getDuration();
		note = nnote;
	}

	return output;
}


HumNum HumdrumToken::getTiedDuration(HumNum scale) {
	return getTiedDuration() * scale;
}



//////////////////////////////
//
// HumdrumToken::getDots -- Count the number of '.' characters in token string.
//    Terminating the count at the first occurrence of the separator character,
//    which is by default a space character.
//

int HumdrumToken::getDots(char separator) const {
	int count = 0;
	for (int i=0; i<(int)this->size()-1; i++) {
		if (this->at(i) == '.') {
			count++;
		}
		if (this->at(i) == separator) {
			break;
		}
	}
	return count;
}



//////////////////////////////
//
// HumdrumToken::getDurationNoDots -- Return the duration of the
//   note excluding any dots.
//

HumNum HumdrumToken::getDurationNoDots(void) {

	int dots = getDots();
	if (dots == 0) {
		return getDuration();
	}
	int bot = (int)pow(2.0, dots + 1) - 1;
	int top = (int)pow(2.0, dots);
	HumNum factor(top, bot);
	return getDuration() * factor;

}


HumNum HumdrumToken::getDurationNoDots(HumNum scale) {
	int dots = getDots();
	if (dots == 0) {
		return getDuration(scale);
	}
	int top = (int)pow(2.0, dots + 1) - 1;
	int bot = (int)pow(2.0, dots);
	HumNum factor(top, bot);
	return getDuration(scale) * factor;
}



//////////////////////////////
//
// HumdrumToken::setDuration -- Sets the duration of the token.  This is done in
//    HumdrumFileStructure::analyzeTokenDurations().
//

void HumdrumToken::setDuration(const HumNum& dur) {
	m_duration = dur;
}



//////////////////////////////
//
// HumdrumToken::getDurationFromStart -- Returns the duration from the
//   start of the owning HumdrumFile to the starting time of the
//   owning HumdrumLine for the token.  The durationFromStart is
//   in reference to the start of the token, not the end of the token,
//   which may be on another HumdrumLine.
//

HumNum HumdrumToken::getDurationFromStart(void) {
	return getLine()->getDurationFromStart();
}


HumNum HumdrumToken::getDurationFromStart(HumNum scale) {
	return getLine()->getDurationFromStart() * scale;
}



//////////////////////////////
//
// HumdrumToken::getDurationToEnd -- Returns the duration from the
//   start of the current line to the start of the last line
//   (the duration of the last line is always zero, so the duration
//   to end is always the duration to the end of the last non-zero
//   duration line.
//

HumNum HumdrumToken::getDurationToEnd(void) {
	return getLine()->getDurationToEnd();
}


HumNum HumdrumToken::getDurationToEnd(HumNum scale) {
	return getLine()->getDurationToEnd() * scale;
}



//////////////////////////////
//
// HumdrumToken::getBarlineDuration -- Returns the duration between
//   the next and previous barline.  If the token is a barline token,
//   then return the duration to the next barline.  The barline duration data
//   is filled in automatically when reading a file with the
//   HumdrumFileStructure::analyzeMeter() function.  The duration
//   will always be non-positive if the file is read with HumdrumFileBase and
//   analyzeMeter() is not run to analyze the data.
//

HumNum HumdrumToken::getBarlineDuration(void) {
	HLp own = getOwner();
	if (own == NULL) {
		return 0;
	}
	return own->getBarlineDuration();
}


HumNum HumdrumToken::getBarlineDuration(HumNum scale) {
	HLp own = getOwner();
	if (own == NULL) {
		return 0;
	}
	return own->getBarlineDuration(scale);
}



//////////////////////////////
//
// HumdrumToken::getDurationToBarline -- Get duration from start of token to
//      the start of the next barline. Units are quarter notes, unless scale
//      is set to a value other than 1.
//

HumNum HumdrumToken::getDurationToBarline(void) {
	HLp own = getOwner();
	if (own == NULL) {
		return 0;
	}
	return own->getDurationToBarline();
}

HumNum HumdrumToken::getDurationToBarline(HumNum scale) {
	HLp own = getOwner();
	if (own == NULL) {
		return 0;
	}
	return own->getDurationToBarline(scale);
}



//////////////////////////////
//
// HumdrumToken::getDurationFromBarline -- Get duration from start of token to
//      the previous barline. Units are quarter notes, unless scale
//      is set to a value other than 1.
//

HumNum HumdrumToken::getDurationFromBarline(void) {
	HLp own = getOwner();
	if (own == NULL) {
		return 0;
	}
	return own->getDurationFromBarline();
}

HumNum HumdrumToken::getDurationFromBarline(HumNum scale) {
	HLp own = getOwner();
	if (own == NULL) {
		return 0;
	}
	return own->getDurationFromBarline(scale);
}



//////////////////////////////
//
// HumdrumToken::hasRhythm -- Returns true if the exclusive interpretation
//    contains rhythmic data which will be used for analyzing the
//    duration of a HumdrumFile, for example.
//

bool HumdrumToken::hasRhythm(void) const {
	string type = getDataType();
	if (type == "**kern") {
		return true;
	}
	if (type == "**recip") {
		return true;
	}
	if (type == "**mens") {
		return true;
	}
	return false;
}



//////////////////////////////
//
// HumdrumToken::hasBeam -- True if **kern has L, J, K, or k.
//

bool HumdrumToken::hasBeam(void) const {
	for (int i=0; i<(int)this->size(); i++) {
		switch (this->at(i)) {
			case 'L':
			case 'J':
			case 'k':
			case 'K':
				return true;
		}
	}
	return false;
}



//////////////////////////////
//
// HumdrumToken::hasFermata --
//

bool HumdrumToken::hasFermata(void) const {
	if (this->find(';') != string::npos) {
		return true;
	} else {
		return false;
	}
}



//////////////////////////////
//
// HumdrumToken::equalTo --
//

bool HumdrumToken::equalTo(const string& pattern) {
	if ((string)(*this) == pattern) {
		return true;
	} else {
		return false;
	}
}



//////////////////////////////
//
// HumdrumToken::isStaff -- Returns true if the spine type represents
//   a notated staff.
//

bool HumdrumToken::isStaff(void) const {
	if (isKern()) {
		return true;
	}
	if (isMens()) {
		return true;
	}
	return false;
}



//////////////////////////////
//
// HumdrumToken::isRest -- Returns true if the token is a (kern) rest.
//

bool HumdrumToken::isRest(void) {
	if (isKern()) {
		if (isNull() && Convert::isKernRest((string)(*resolveNull()))) {
			return true;
		} else if (Convert::isKernRest((string)(*this))) {
			return true;
		}
	} else if (isMens()) {
		if (isNull() && Convert::isMensRest((string)(*resolveNull()))) {
			return true;
		} else if (Convert::isMensRest((string)(*this))) {
			return true;
		}
	}
	return false;
}



//////////////////////////////
//
// HumdrumToken::isNote -- Returns true if the token is a (kern) note
//     (possessing a pitch).
//

bool HumdrumToken::isNote(void) {
	if (!isData()) {
		return false;
	}
	if (isNull()) {
		return false;
	}
	if (isKern()) {
		if (Convert::isKernNote((string)(*this))) {
			return true;
		}
	} else if (isMens()) {
		if (Convert::isMensNote((string)(*this))) {
			return true;
		}
	}
	return false;
}



//////////////////////////////
//
// HumdrumToken::isPitched -- True if not a rest or an unpitched note.
//

bool HumdrumToken::isPitched(void) { 
	if (this->isKern()) {
		for (int i=0; i<(int)this->size(); i++) {
			if ((this->at(i) == 'r') || (this->at(i) == 'R')) {
				return false;
			}
		}
		return true;
	}
	// Don't know data type so return false for now:
	return false;
}



//////////////////////////////
//
// HumdrumToken::isPitched -- True if has an unpitched marker (could be a rest)
//

bool HumdrumToken::isUnpitched(void) {
	if (this->isKern()) {
		if (this->find('R') != string::npos) {
			return 1;
		} else {
			return 0;
		}
	}
	// Don't know data type so return false for now:
	return false;
}




//////////////////////////////
//
// HumdrumToken::isSustainedNote -- Returns true if the token represents
//     a sounding note, but not the attack portion.  Should only be
//     applied to **kern data.
//

bool HumdrumToken::isSustainedNote(void) {
	HTp token = this;
	if (isNull()) {
		token = resolveNull();
	}
	return token->isSecondaryTiedNote();
}



//////////////////////////////
//
// HumdrumToken::isNoteAttack -- Returns true if the token represents
//     the attack of a note.  Should only be applied to **kern data, but
//     this function does not check for that for efficiency reasons.
//

bool HumdrumToken::isNoteAttack(void) {
	HTp token = this;
	if (isNull()) {
		token = resolveNull();
	}
	if (token->isRest()) {
		return false;
	}
	return !token->isSecondaryTiedNote();
}



//////////////////////////////
//
// HumdrumToken::isInvisible -- True if a barline and is invisible (contains
//     a "-" styling), or a note/rest contains the string "yy" which is
//     interpreted as meaning make it invisible.
//
//

bool HumdrumToken::isInvisible(void) {
	if (!isDataType("**kern")) {
			return false;
	}
	if (isBarline()) {
		if (find("-") != string::npos) {
			return true;
		}
	} else if (isData()) {
		if (find("yy") != string::npos) {
			return true;
		}
	}

	return false;
}



//////////////////////////////
//
// HumdrumToken::isGrace -- True if a **kern note has no duration.
//

bool HumdrumToken::isGrace(void) {
	if (!isDataType("**kern")) {
			return false;
	}
	if (!isData()) {
		return false;
	} else if (this->find("q") != string::npos) {
		return true;
	}

	return false;
}



//////////////////////////////
//
// HumdrumToken::isClef -- True if a **kern clef.
//

bool HumdrumToken::isClef(void) {
	if (!(isDataType("**kern") || isDataType("**mens"))) {
			return false;
	}
	if (!isInterpretation()) {
		return false;
	} else if (this->compare(0, 5, "*clef") == 0) {
		return true;
	}

	return false;
}



//////////////////////////////
//
// HumdrumToken::isKeySignature -- True if a **kern key signature.
//

bool HumdrumToken::isKeySignature(void) {
	if (this->compare(0, 3, "*k[") != 0) {
		return false;
	}
	if (this->back() != ']') {
		return false;
	}
	return true;
}



//////////////////////////////
//
// HumdrumToken::isKeyDesignation -- True if a **kern key designation.
//   *C:
//   *A-:
//   *c#:
//   *d:dor
//

bool HumdrumToken::isKeyDesignation(void) {
	if (this->size() < 3) {
		return false;
	}
	if (this->find(":") == string::npos) {
		return false;
	}
	char diatonic = (*this)[1];

	if ((diatonic >= 'A') && (diatonic <= 'G')) {
		return true;
	}
	if ((diatonic >= 'a') && (diatonic <= 'g')) {
		return true;
	}
	return false;
}



//////////////////////////////
//
// HumdrumToken::isTimeSignature -- True if a **kern time signature.
//

bool HumdrumToken::isTimeSignature(void) {
	if (this->size() < 3) {
		return false;
	}
	if (this->compare(0, 2, "*M") != 0) {
		return false;
	}
	if (!isdigit((*this)[2])) {
		return false;
	}
	if (this->find("/") == string::npos) {
		return false;
	}
	return true;
}



//////////////////////////////
//
// HumdrumToken::isTempo -- True if a **kern tempo.
//

bool HumdrumToken::isTempo(void) {
	if (this->size() < 4) {
		return false;
	}
	if (this->compare(0, 3, "*MM") != 0) {
		return false;
	}
	if (!isdigit((*this)[3])) {
		return false;
	}
	return true;
}



//////////////////////////////
//
// HumdrumToken::isMensurationSymbol -- True if a **kern mensuration Symbol.
//

bool HumdrumToken::isMensurationSymbol(void) {
	if (this->compare(0, 5, "*met(") != 0) {
		return false;
	}
	if ((*this)[this->size()-1] != ')') {
		return false;
	}
	return true;
}



//////////////////////////////
//
// HumdrumToken::isInstrumentDesignation -- Such as *Iclars for B-flat clarinet.
//

bool HumdrumToken::isInstrumentDesignation(void) {
	if (this->compare(0, 2, "*I") != 0) {
		return false;
	}
	for (int i=2; i<(int)this->size(); i++) {
		if (!isalpha(this->at(i))) {
			return false;
		}
		if (!islower(this->at(i))) {
			return false;
		}
	}

	return true;
}



//////////////////////////////
//
// HumdrumToken::isInstrumentName -- True if an instrument name token.
//

bool HumdrumToken::isInstrumentName(void) {
	if (this->compare(0, 3, "*I\"") != 0) {
		return false;
	} else {
		return true;
	}
}



//////////////////////////////
//
// HumdrumToken::isInstrumentAbbreviation -- True if an instrument abbreviation token.
//

bool HumdrumToken::isInstrumentAbbreviation(void) {
	if (this->compare(0, 3, "*I'") != 0) {
		return false;
	} else {
		return true;
	}
}



//////////////////////////////
//
// HumdrumToken::getInstrumentName --
//

string HumdrumToken::getInstrumentName(void) {
	if (this->size() < 3) {
		return "";
	} else if (this->compare(0, 3, "*I\"") != 0) {
		return "";
	} else {
		return this->substr(3);
	}
}



//////////////////////////////
//
// HumdrumToken::getInstrumentAbbreviation --
//

string HumdrumToken::getInstrumentAbbreviation(void) {
	if (this->size() < 3) {
		return "";
	} else if (this->compare(0, 3, "*I'") != 0) {
		return "";
	} else {
		return this->substr(3);
	}
}



//////////////////////////////
//
// HumdrumToken::hasSlurStart -- Returns true if the **kern token has
//     a '(' character.
//

bool HumdrumToken::hasSlurStart(void) {
	if (isDataType("**kern")) {
		if (Convert::hasKernSlurStart((string)(*this))) {
			return true;
		}
	}
	return false;
}



//////////////////////////////
//
// HumdrumToken::hasSlurEnd -- Returns true if the **kern token has
//     a ')' character.
//

bool HumdrumToken::hasSlurEnd(void) {
	if (isDataType("**kern")) {
		if (Convert::hasKernSlurEnd((string)(*this))) {
			return true;
		}
	}
	return false;
}



//////////////////////////////
//
// HumdrumToken::hasVisibleAccidental -- Returns true if the accidental
//    of a **kern note is viewable if rendered to graphical notation.
// 	return values:
//      0  = false;
//      1  = true;
//      -1 = undefined;
//

int HumdrumToken::hasVisibleAccidental(int subtokenIndex) const {
	HLp humrec = getOwner();
	if (humrec == NULL) {
		return -1;
	}
	HumdrumFile* humfile = humrec->getOwner();
	if (humfile == NULL) {
		return -1;
	}
	if (!humfile->getValueBool("auto", "accidentalAnalysis")) {
		int status = humfile->analyzeKernAccidentals();
		if (!status) {
			return -1;
		}
	}
	return getValueBool("auto", to_string(subtokenIndex), "visualAccidental");
}



//////////////////////////////
//
// HumdrumToken::hasVisibleAccidental -- Returns true if the accidental
//    of a **kern note is viewable if rendered to graphical notation.
// 	return values:
//      0  = false;
//      1  = true;
//      -1 = undefined;
//

int HumdrumToken::hasCautionaryAccidental(int subtokenIndex) const {
	HLp humrec = getOwner();
	if (humrec == NULL) {
		return -1;
	}
	HumdrumFile* humfile = humrec->getOwner();
	if (humfile == NULL) {
		return -1;
	}
	if (!humfile->getValueBool("auto", "accidentalAnalysis")) {
		int status = humfile->analyzeKernAccidentals();
		if (!status) {
			return -1;
		}
	}
	return getValueBool("auto", to_string(subtokenIndex), "cautionaryAccidental");
}



//////////////////////////////
//
// HumdrumToken::hasLigatureBegin --
//

bool HumdrumToken::hasLigatureBegin(void) {
	if (isMens()) {
		return Convert::hasLigatureBegin(*this);
	} else {
		return false;
	}
}



//////////////////////////////
//
// HumdrumToken::hasRectaLigatureBegin --
//

bool HumdrumToken::hasRectaLigatureBegin(void) {
	if (isMens()) {
		return Convert::hasRectaLigatureBegin(*this);
	} else {
		return false;
	}
}



//////////////////////////////
//
// HumdrumToken::hasObliquaLigatureBegin --
//

bool HumdrumToken::hasObliquaLigatureBegin(void) {
	if (isMens()) {
		return Convert::hasObliquaLigatureBegin(*this);
	} else {
		return false;
	}
}



//////////////////////////////
//
// HumdrumToken::hasStemDirection --
//

char HumdrumToken::hasStemDirection(void) {
	if (isKern()) {
		return Convert::hasKernStemDirection(*this);
	} else {
		// don't know what a stem in this datatype is
		return '\0';
	}
}



//////////////////////////////
//
// HumdrumToken::allSameBarlineStyle --
//

bool HumdrumToken::allSameBarlineStyle(void) {
	HLp owner = getOwner();
	if (!owner) {
		return true;
	}
	return owner->allSameBarlineStyle();
}



//////////////////////////////
//
// HumdrumToken::hasLigatureEnd --
//

bool HumdrumToken::hasLigatureEnd(void) {
	if (isMens()) {
		return Convert::hasLigatureEnd(*this);
	} else {
		return false;
	}
}



//////////////////////////////
//
// HumdrumToken::hasRectaLigatureEnd --
//

bool HumdrumToken::hasRectaLigatureEnd(void) {
	if (isMens()) {
		return Convert::hasRectaLigatureEnd(*this);
	} else {
		return false;
	}
}



//////////////////////////////
//
// HumdrumToken::hasObliquaLigatureEnd --
//

bool HumdrumToken::hasObliquaLigatureEnd(void) {
	if (isMens()) {
		return Convert::hasObliquaLigatureEnd(*this);
	} else {
		return false;
	}
}



//////////////////////////////
//
// HumdrumToken::isSecondaryTiedNote -- Returns true if the token
//     is a (kern) note (possessing a pitch) and has '_' or ']' characters.
//

bool HumdrumToken::isSecondaryTiedNote(void) {
	if (isDataType("**kern")) {
		if (Convert::isKernSecondaryTiedNote((string)(*this))) {
			return true;
		}
	}
	return false;
}



//////////////////////////////
//
// HumdrumToken::isBarline -- Returns true if the first character is an
//   equals sign.
//

bool HumdrumToken::isBarline(void) const {
	if (size() == 0) {
		return false;
	}
	if ((*this)[0] == '=') {
		return true;
	} else {
		return false;
	}
}



//////////////////////////////
//
// HumdrumToken::isCommentGlobal -- Returns true of the token starts with "!!".
//    Currently confused with reference records.
//

bool HumdrumToken::isCommentGlobal(void) const {
	if (size() == 0) {
		return false;
	}
	if ((*this)[0] == '!') {
		if (size() > 1) {
			if ((*this)[1] == '!') {
				// global comment
				return true;
			}
		}
	}
	return false;
}



//////////////////////////////
//
// HumdrumToken::isCommentLocal -- Returns true of the token start with "!",
//   but not "!!" which is for global comments.
//

bool HumdrumToken::isCommentLocal(void) const {
	if (size() == 0) {
		return false;
	}
	if ((*this)[0] == '!') {
		if (size() > 1) {
			if ((*this)[1] == '!') {
				// global comment
				return false;
			}
		}
		return true;
	} else {
		return false;
	}
}



//////////////////////////////
//
// HumdrumToken::isComment -- Returns true of the token start with "!".
//

bool HumdrumToken::isComment(void) const {
	if (size() == 0) {
		return false;
	}
	if ((*this)[0] == '!') {
		return true;
	} else {
		return false;
	}
}



//////////////////////////////
//
// HumdrumToken::isData -- Returns true if not an interpretation, barline
//      or local comment.  This will not work on synthetic tokens generated
//      from an empty line.  So this function should be called only on tokens
//      in lines which pass the HumdrumLine::hasSpines() test.
//

bool HumdrumToken::isData(void) const {
	if (size() == 0) {
		return false;
	}
	int firstchar = (*this)[0];
	if ((firstchar == '*') || (firstchar == '!') || (firstchar == '=')) {
		return false;
	}
	return true;
}



//////////////////////////////
//
// HumdrumToken::isInterpretation -- Returns true if an interpretation.
//

bool HumdrumToken::isInterpretation(void) const {
	if (size() == 0) {
		return false;
	}
	int firstchar = (*this)[0];
	if (firstchar == '*') {
		return true;
	}
	return false;
}



//////////////////////////////
//
// HumdrumToken::isNonNullData -- Returns true if the token is a data token
//    that is not a null token.
//

bool HumdrumToken::isNonNullData(void) const {
	return isData() && !isNull();
}



//////////////////////////////
//
// HumdrumToken::isNullData -- Returns true if the token is a null
//     data token.
//

bool HumdrumToken::isNullData(void) const {
	return isData() && isNull();
}



//////////////////////////////
//
// HumdrumToken::isLabel -- Returns true if a thru label (such as *>A).
//

bool HumdrumToken::isLabel(void) const {
	if (string::compare(0, 2, "*>") != 0) {
		return false;
	}
	if (string::find("[") != string::npos) {
		return false;
	}
	return true;
}



/////////////////////////////
//
// HumdrumToken::isChord -- True if is a chord.  Presuming you know what
//     data type you are accessing.
//     Default value:
//          separate = " "   (**kern note separator)
//

bool HumdrumToken::isChord(const string& separator) {
	return (this->find(separator) != string::npos) ? true : false;
}



//////////////////////////////
//
// HumdrumToken::isExclusiveInterpretation -- Returns true if first two
//     characters are "**".
//

bool HumdrumToken::isExclusiveInterpretation(void) const {
	const string& tok = (string)(*this);
	return tok.substr(0, 2) == "**";
}



//////////////////////////////
//
// HumdrumToken::isSplitInterpretation -- True if the token is "*^".
//

bool HumdrumToken::isSplitInterpretation(void) const {
	return ((string)(*this)) == SPLIT_TOKEN;
}



//////////////////////////////
//
// HumdrumToken::isMergeInterpretation -- True if the token is "*v".
//

bool HumdrumToken::isMergeInterpretation(void) const {
	// [20200331] GCC 6+ will print a compiler warning when checking this against NULL.
	//if ((void*)this == NULL) {
	//	// This was added perhaps due to a new bug [20100125] that is checking a null pointer
	//	return false;
	//}
	return ((string)(*this)) == MERGE_TOKEN;
}



//////////////////////////////
//
// HumdrumToken::isExchangeInterpretation -- True if the token is "*x".
//

bool HumdrumToken::isExchangeInterpretation(void) const {
	return ((string)(*this)) == EXCHANGE_TOKEN;
}



//////////////////////////////
//
// HumdrumToken::isTerminateInterpretation -- True if the token is "*-".
//

bool HumdrumToken::isTerminateInterpretation(void) const {
	return ((string)(*this)) == TERMINATE_TOKEN;
}



//////////////////////////////
//
// HumdrumToken::isAddInterpretation -- True if the token is "*+".
//

bool HumdrumToken::isAddInterpretation(void) const {
	return ((string)(*this)) == ADD_TOKEN;
}



//////////////////////////////
//
// HumdrumToken::isNull -- Returns true if the token is a null token,
//   either for data, comments, or interpretations.  Does not consider
//   null global comments since they are not part of the spine structure.
//

bool HumdrumToken::isNull(void) const {
	const string& tok = (string)(*this);
	if (tok == NULL_DATA)           { return true; }
	if (tok == NULL_INTERPRETATION) { return true; }
	if (tok == NULL_COMMENT_LOCAL)  { return true; }
	return false;
}



//////////////////////////////
//
// HumdrumToken::getSubtrack -- Get the subtrack (similar to a layer
//    in MEI).
//

int HumdrumToken::getSubtrack(void) const {
	return m_address.getSubtrack();
}



//////////////////////////////
//
// HumdrumToken::noteInLowerSubtrack -- Return true if the note
//     is attacked or sustained with another note in a lower layer.
//     This is for using in hum2mei conversion to avoid a bug in
//     verovio related to lyrics in layers where the notes are a
//     second apart.
//

bool HumdrumToken::noteInLowerSubtrack(void) {
	int subtrack = this->getSubtrack();
	if (subtrack <= 1) {
		return false;
	}
	int field = this->getFieldIndex();
	int track = this->getTrack();

	HLp owner = this->getOwner();
	if (owner == NULL) {
		return false;
	}

	for (int i=field-1; i>=0; i--) {
		HTp xtoken = owner->token(i);
		int xtrack = xtoken->getTrack();
		if (xtrack != track) {
			return false;
		}
		if (xtoken->isNull()) {
			continue;
		}
		if (xtoken->find("r") != string::npos) {
			continue;
		}
		return true;
	}

	return false;
}



//////////////////////////////
//
// HumdrumToken::getTrackString -- Gets "track.subtrack" as a string.  The
//     track and subtrack are integers.  The getTrackString function will
//     return a string with the track and subtrack separated by an dot.  The
//     Dot is not a decimal point, but if the subtrack count does not exceed
//     9, then the returned string can be treated as a floating-point number
//     where the subtrack is the fractional part.
// @SEEALSO: getTrack, getSubtrack
//

string HumdrumToken::getTrackString(void) const {
	return m_address.getTrackString();
}



/////////////////////////////
//
// HumdrumToken::getSubtokenCount -- Returns the number of sub-tokens in
//     a token.  The input parameter is the sub-token separator.  If the
//     separator comes at the start or end of the token, then there will
//     be empty sub-token(s) included in the count.
// default value: separator = " "
// @SEEALSO: getSubtoken
//

int HumdrumToken::getSubtokenCount(const string& separator) const {
	int count = 0;
	string::size_type start = 0;
	while ((start = string::find(separator, start)) != string::npos) {
		count++;
		start += separator.size();
	}
	return count+1;
}



/////////////////////////////
//
// HumdrumToken::getSubtoken -- Extract the specified sub-token from the token.
//    Tokens usually are separated by spaces in Humdrum files, but this will
//    depened on the data type (so therefore, the tokens are not presplit into
//    sub-tokens when reading in the file).
// default value: separator = " "
// @SEEALSO: getSubtokenCount, getTrackString
//

string HumdrumToken::getSubtoken(int index, const string& separator) const {
	if (index < 0) {
		return "";
	}

	string output;
	const string& token = *this;
	if (separator.size() == 0) {
		output = token[index];
		return output;
	}

	int count = 0;
	for (int i=0; i<(int)size(); i++) {
		if (string::compare(i, separator.size(), separator) == 0) {
			count++;
			if (count > index) {
				break;
			}
			i += (int)separator.size() - 1;
		} else if (count == index) {
			output += token[i];
		}
	}
	return output;
}



//////////////////////////////
//
// HumdrumToken::getSubtokens -- Return the list of subtokens as an array
//     of strings.
//     default value: separator = " "
//

std::vector<std::string> HumdrumToken::getSubtokens (const std::string& separator) const {
	std::vector<std::string> output;
	const string& token = *this;
	HumRegex hre;
	hre.split(output, token, separator);
	return output;
}



//////////////////////////////
//
// HumdrumToken::replaceSubtoken --
//     default value: separator = " "
//

void HumdrumToken::replaceSubtoken(int index, const std::string& newsubtok,
		const std::string& separator) {
	if (index < 0) {
		return;
	}
	std::vector<std::string> subtokens = getSubtokens(separator);
	if (index >= (int)subtokens.size()) {
		return;
	}
	subtokens[index] = newsubtok;
	string output;
	for (int i=0; i<(int)subtokens.size(); i++) {
		output += subtokens[i];
		if (i < (int)subtokens.size() - 1) {
			output += separator;
		}
	}
	this->setText(output);
}



//////////////////////////////
//
// HumdrumToken::setParameters -- Process a local comment with
//     the structure:
//        !NS1:NS2:key1=value1:key2=value2:key3=value3
//     and store the parameter in the HumHash parent class component of the
//     HumdrumToken object.
// default value for 2-parameter version: ptok = NULL
//

void HumdrumToken::setParameters(HumdrumToken* ptok) {
	HumdrumToken& pl = *ptok;
	if (pl.size() <= 1) {
		return;
	}
	string pdata = pl.substr(1, pl.size()-1);
	setParameters(pdata, ptok);
}


void HumdrumToken::setParameters(const string& pdata, HumdrumToken* ptok) {
	vector<string> pieces = Convert::splitString(pdata, ':');
	if (pieces.size() < 3) {
		return;
	}
	string ns1 = pieces[0];
	string ns2 = pieces[1];
	string key;
	string value;
	int loc;
	for (int i=2; i<(int)pieces.size(); i++) {
		Convert::replaceOccurrences(pieces[i], "&colon;", ":");
		loc = (int)pieces[i].find("=");
		if (loc != (int)string::npos) {
			key   = pieces[i].substr(0, loc);
			value = pieces[i].substr(loc+1, pieces[i].size());
		} else {
			key   = pieces[i];
			value = "true";
		}
		setValue(ns1, ns2, key, value);
		setOrigin(ns1, ns2, key, ptok);
	}
}



//////////////////////////////
//
// HumdrumToken::setText --
//

void HumdrumToken::setText(const string& text) {
	string::assign(text);
}



//////////////////////////////
//
// HumdrumToken::getText --
//

string HumdrumToken::getText(void) const {
	return string(*this);
}



//////////////////////////////
//
// HumdrumToken::addLinkedParamter --
//

int HumdrumToken::addLinkedParameterSet(HTp token) {
	if (token->find(":ignore") != string::npos) {
		// Ignore layout command (store layout command but
		// do not use it.  This is particularly for adding
		// layout parameters for notation, but the parameters
		// currently cause problems in verovio (so they should
		// be unignored at a future date when the layout
		// parameter is handled better).  Note that any
		// parameter starting with "ignore" such as "ignored"
		// will also be suppressed by this if statement.
		return -1;
	}
	for (int i=0; i<(int)m_linkedParameterTokens.size(); i++) {
		if (m_linkedParameterTokens[i] == token) {
			return i;
		}
	}

	if (m_linkedParameterTokens.empty()) {
		m_linkedParameterTokens.push_back(token);
	} else {
		int lineindex = token->getLineIndex();
		if (lineindex >= m_linkedParameterTokens.back()->getLineIndex()) {
			m_linkedParameterTokens.push_back(token);
		} else {
			// Store sorted by line number
			for (auto it = m_linkedParameterTokens.begin(); it != m_linkedParameterTokens.end(); it++) {
				if (lineindex < (*it)->getLineIndex()) {
					m_linkedParameterTokens.insert(it, token);
					break;
				}
			}
		}

	}

	return (int)m_linkedParameterTokens.size() - 1;
}



//////////////////////////////
//
// HumdrumToken::linkedParameterIsGlobal --
//

bool HumdrumToken::linkedParameterIsGlobal(int index) {
	return m_linkedParameterTokens.at(index)->isCommentGlobal();
}



//////////////////////////////
//
// HumdrumToken::getLinkedParameterSetCount --
//

int HumdrumToken::getLinkedParameterSetCount(void) {
	return (int)m_linkedParameterTokens.size();
}



//////////////////////////////
//
// HumdrumToken::getParameterSet --
//

HumParamSet* HumdrumToken::getParameterSet(void) {
	return m_parameterSet;
}



//////////////////////////////
//
// HumdrumToken::getLinkedParameterSet --
//

HumParamSet* HumdrumToken::getLinkedParameterSet(int index) {
	return m_linkedParameterTokens.at(index)->getParameterSet();
}



//////////////////////////////
//
// HumdrumToken::storeParameterSet -- Store the contents of the token
//    in the linked parameter storage.  Used for layout parameters.
//

void HumdrumToken::storeParameterSet(void) {
	if (m_parameterSet) {
		delete m_parameterSet;
		m_parameterSet = NULL;
	}
	if (this->isCommentLocal() && (this->find(':') != string::npos)) {
		m_parameterSet = new HumParamSet(this);
	} else if (this->isCommentGlobal() && (this->find(':') != string::npos)) {
		m_parameterSet = new HumParamSet(this);
	}
}



//////////////////////////////
//
// HumdrumToken::makeForwardLink -- Line a following spine token to this one.
//    Used by the HumdrumFileBase::analyzeLinks function.
//

void HumdrumToken::makeForwardLink(HumdrumToken& nextToken) {
	m_nextTokens.push_back(&nextToken);
	nextToken.m_previousTokens.push_back(this);
}



//////////////////////////////
//
// HumdrumToken::makeBackwarddLink -- Link a previous spine token to this one.
//    Used by the HumdrumFileBase::analyzeLinks function.
//

void HumdrumToken::makeBackwardLink(HumdrumToken& previousToken) {
	m_previousTokens.push_back(&previousToken);
	previousToken.m_nextTokens.push_back(this);
}



//////////////////////////////
//
// HumdrumToken::getVisualDuration -- Returns LO:N:vis parameter if it is attached
//    to a token directly or indirectly through a linked parameter.  Returns empty string
//    if no explicit visual durtation (so the visual duration is same as the logical duration).
//

string HumdrumToken::getVisualDuration(int subtokenindex) {
	// direct storage of the layout parameter is possible, but currently disabled:
	//string parameter = this->getValue("LO", "N", "vis");
	//if (!parameter.empty()) {
	//	return parameter;
	//}
	return this->getLayoutParameter("N", "vis", subtokenindex);
}



//////////////////////////////
//
// HumdrumToken::getVisualDurationChord -- only return the chord-level visual duration
//    parameter (not if it is specific to certain note(s) in the chord).
//

string HumdrumToken::getVisualDurationChord(void) {
	return this->getLayoutParameterChord("N", "vis");
}



//////////////////////////////
//
// HumdrumToken::getVisualDurationNote -- only return the note-level visual duration
//    parameter (not if it is general to the entire chord.
//

string HumdrumToken::getVisualDurationNote(int subtokenindex) {
	return this->getLayoutParameterNote("N", "vis", subtokenindex);
}



//////////////////////////////
//
// HumdrumToken::getLayoutParameter -- Returns requested layout parameter
//     if it is attached to a token directly or indirectly through a linked
//     parameter.  Returns empty string if no explicit visual durtation (so
//     the visual duration is same as the logical duration).  If subtokenindex
//     is less than -1 (the default value for the paramter), then ignore the
//     @n parameter control for indexing the layout parameter to chord notes.
//     The subtokenindex (0 indexed) is converted to note number (1 indexed)
//     for checking @n.  @n is currently only allowed to be a single integer
//     (eventually allow ranges and multiple values).
//

std::string HumdrumToken::getLayoutParameter(const std::string& category,
		const std::string& keyname, int subtokenindex) {

	// First check for any local layout parameter:
	std::string testoutput = this->getValue("LO", category, keyname);
	if (!testoutput.empty()) {
		if (subtokenindex >= 0) {
			int n = this->getValueInt("LO", category, "n");
			if (n == subtokenindex + 1) {
				return testoutput;
			}
		} else {
			return testoutput;
		}
	}

	std::string output;
	int lcount = this->getLinkedParameterSetCount();
	if (lcount == 0) {
		return output;
	}

	std::string nparam;
	for (int p = 0; p < this->getLinkedParameterSetCount(); ++p) {
		hum::HumParamSet *hps = this->getLinkedParameterSet(p);
		if (hps == NULL) {
			continue;
		}
		if (hps->getNamespace1() != "LO") {
			continue;
		}
		if (hps->getNamespace2() != category) {
			continue;
		}

		output = "";
		for (int q = 0; q < hps->getCount(); ++q) {
			string key = hps->getParameterName(q);
			if (key == keyname) {
				output = hps->getParameterValue(q);
				if (subtokenindex < 0) {
					return output;
				}
			}
			if (key == "n") {
				nparam = hps->getParameterValue(q);
			}
		}
		if (nparam.empty()) {
			// No subtoken selection for this parameter,
			// so return if not empty:
			if (!output.empty()) {
				return output;
			}
		} else if (subtokenindex < 0) {
			// No subtoken selection so return output if not empty
			if (!output.empty()) {
				return output;
			}
		} else {
			// There is a subtoken selection number, so
			// return output if n matches it (minus one)

			// currently @n requires a single value
			// (should allow a range or multiple values
			// later).  Also not checking validity of
			// string first (needs to start with a digit);

			int n = stoi(nparam);
			if (n == subtokenindex + 1) {
				return output;
			} else {
				// not the output that is required,
				// so suppress for end of loop:
				output = "";
			}
		}
	}

	return output;
}



//////////////////////////////
//
// HumdrumToken::getSlurLayoutParameter --
//

std::string HumdrumToken::getSlurLayoutParameter(const std::string& keyname,
		int subtokenindex) {
	std::string category = "S";
	std::string output;

	// First check for any local layout parameter:
	std::string testoutput = this->getValue("LO", category, keyname);
	if (!testoutput.empty()) {
		if (subtokenindex >= 0) {
			int s = this->getValueInt("LO", category, "s");
			if (s == subtokenindex + 1) {
				return testoutput;
			}
		} else {
			return testoutput;
		}
	}

	int lcount = this->getLinkedParameterSetCount();
	if (lcount == 0) {
		return output;
	}

	std::string sparam;
	for (int p = 0; p < this->getLinkedParameterSetCount(); ++p) {
		hum::HumParamSet *hps = this->getLinkedParameterSet(p);
		if (hps == NULL) {
			continue;
		}
		if (hps->getNamespace1() != "LO") {
			continue;
		}
		if (hps->getNamespace2() != category) {
			continue;
		}
		for (int q = 0; q < hps->getCount(); ++q) {
			string key = hps->getParameterName(q);
			if (key == "s") {
				sparam = hps->getParameterValue(q);
			}
			if (key == keyname) {
				output = hps->getParameterValue(q);
			}
		}
	}
	if (subtokenindex < 0) {
		// do not filter by s parameter
		return output;
	} else if (sparam.empty()) {
		// parameter is not qualified by a note number, so applies to whole token
		return output;
	}

	// currently @s requires a single value (should allow a range or multiple values later)
	// also not checking validity of string first (needs to start with a digit);
	int s = stoi(sparam);
	if (s == subtokenindex + 1) {
		return output;
	} else {
		return "";
	}
}



//////////////////////////////
//
// HumdrumToken::getPhraseLayoutParameter --
//

std::string HumdrumToken::getPhraseLayoutParameter(const std::string& keyname,
		int subtokenindex) {
	std::string category = "P";
	std::string output;

	// First check for any local layout parameter:
	std::string testoutput = this->getValue("LO", category, keyname);
	if (!testoutput.empty()) {
		if (subtokenindex >= 0) {
			int s = this->getValueInt("LO", category, "s");
			if (s == subtokenindex + 1) {
				return testoutput;
			}
		} else {
			return testoutput;
		}
	}

	int lcount = this->getLinkedParameterSetCount();
	if (lcount == 0) {
		return output;
	}

	std::string sparam;
	for (int p = 0; p < this->getLinkedParameterSetCount(); ++p) {
		hum::HumParamSet *hps = this->getLinkedParameterSet(p);
		if (hps == NULL) {
			continue;
		}
		if (hps->getNamespace1() != "LO") {
			continue;
		}
		if (hps->getNamespace2() != category) {
			continue;
		}
		for (int q = 0; q < hps->getCount(); ++q) {
			string key = hps->getParameterName(q);
			if (key == "s") {
				sparam = hps->getParameterValue(q);
			}
			if (key == keyname) {
				output = hps->getParameterValue(q);
			}
		}
	}
	if (subtokenindex < 0) {
		// do not filter by s parameter
		return output;
	} else if (sparam.empty()) {
		// parameter is not qualified by a note number, so applies to whole token
		return output;
	}

	// currently @s requires a single value (should allow a range or multiple values later)
	// also not checking validity of string first (needs to start with a digit);
	int s = stoi(sparam);
	if (s == subtokenindex + 1) {
		return output;
	} else {
		return "";
	}
}



//////////////////////////////
//
// HumdrumToken::getLayoutParameterChord -- Returns requested layout
// parameter if it is attached to a token directly or indirectly through
// a linked parameter.  The parameter must apply to the entire chord, so
// no @n qualification parameters can be given (even if they include all
// notes in the chord).

std::string HumdrumToken::getLayoutParameterChord(const std::string& category,
		const std::string& keyname) {

	// First check for any local layout parameter:
	std::string testoutput = this->getValue("LO", category, keyname);
	if (!testoutput.empty()) {
		std::string n = this->getValue("LO", category, "n");
		if (n.empty()) {
			return testoutput;
		}
	}

	std::string output;
	int lcount = this->getLinkedParameterSetCount();
	if (lcount == 0) {
		return output;
	}

	std::string nparam;
	for (int p = 0; p < this->getLinkedParameterSetCount(); ++p) {
		hum::HumParamSet *hps = this->getLinkedParameterSet(p);
		if (hps == NULL) {
			continue;
		}
		if (hps->getNamespace1() != "LO") {
			continue;
		}
		if (hps->getNamespace2() != category) {
			continue;
		}
		for (int q = 0; q < hps->getCount(); ++q) {
			string key = hps->getParameterName(q);
			if (key == "n") {
				nparam = hps->getParameterValue(q);
			}
			if (key == keyname) {
				output = hps->getParameterValue(q);
			}
		}
	}

	if (!nparam.empty()) {
		// parameter is qualified by a note number, so does not apply to whole token
		return "";
	} else {
		return output;
	}
}



//////////////////////////////
//
// HumdrumToken::getLayoutParameterNote -- Returns requested layout
//     parameter if it is attached to a token directly or indirectly through a
//     linked parameter.  The parameter must apply to a single note or specific
//     note in a chord.


std::string HumdrumToken::getLayoutParameterNote(const std::string& category,
		const std::string& keyname, int subtokenindex) {

	// First check for any local layout parameter:
	std::string testoutput = this->getValue("LO", category, keyname);
	if (!testoutput.empty()) {
		if (subtokenindex >= 0) {
			int n = this->getValueInt("LO", category, "n");
			if (n == subtokenindex + 1) {
				return testoutput;
			}
		}
	}

	std::string output;
	int lcount = this->getLinkedParameterSetCount();
	if (lcount == 0) {
		return output;
	}

	std::string nparam;
	for (int p = 0; p < this->getLinkedParameterSetCount(); ++p) {
		hum::HumParamSet *hps = this->getLinkedParameterSet(p);
		if (hps == NULL) {
			continue;
		}
		if (hps->getNamespace1() != "LO") {
			continue;
		}
		if (hps->getNamespace2() != category) {
			continue;
		}
		for (int q = 0; q < hps->getCount(); ++q) {
			string key = hps->getParameterName(q);
			if (key == "n") {
				nparam = hps->getParameterValue(q);
			}
			if (key == keyname) {
				output = hps->getParameterValue(q);
			}
		}
	}

	if (!nparam.empty()) {
		// a number number is specified from the parameter(s)
		int n = stoi(nparam);
		if (n == subtokenindex + 1) {
			return output;
		} else {
			// wrong note
			return "";
		}
	}

	if ((subtokenindex < 0) && isChord()) {
		// in chord, and no specific note is selected by @n.
		return "";
	} else {
		// single note, so return parameter:
		return output;
	}
}



//////////////////////////////
//
// HumdrumToken::setOwner -- Sets the HumdrumLine owner of this token.
//

void HumdrumToken::setOwner(HLp aLine) {
	m_address.setOwner(aLine);
}



//////////////////////////////
//
// HumdrumToken::getOwner -- Returns a pointer to the HumdrumLine that
//    owns this token.
//

HLp HumdrumToken::getOwner(void) const {
	return m_address.getOwner();
}



//////////////////////////////
//
// HumdrumToken::getState -- Returns the rhythm state variable.
//

int HumdrumToken::getState(void) const {
	return m_rhycheck;
}



//////////////////////////////
//
// HumdrumToken::getStrandIndex -- Returns the 1-D strand index
//    that the token belongs to in the owning HumdrumFile.
//    Returns -1 if there is no strand assignment.
//

int  HumdrumToken::getStrandIndex(void) const {
	return m_strand;
}



//////////////////////////////
//
// HumdrumToken::getSlurStartElisionLevel -- Returns the count of
//   elision marks ('&') preceding a slur start character '('.
//   Returns -1 if there is no slur start character.
//   Default value: index = 0
//

int HumdrumToken::getSlurStartElisionLevel(int index) const {
	if (isDataType("**kern") || isDataType("**mens")) {
		return Convert::getKernSlurStartElisionLevel((string)(*this), index);
	} else {
		return -1;
	}
}



//////////////////////////////
//
// HumdrumToken::getPhraseStartElisionLevel -- Returns the count of
//   elision marks ('&') preceding a phrase start character '{'.
//   Returns -1 if there is no phrase start character.
//   Default value: index = 0
//

int HumdrumToken::getPhraseStartElisionLevel(int index) const {
	if (isDataType("**kern") || isDataType("**mens")) {
		return Convert::getKernPhraseStartElisionLevel((string)(*this), index);
	} else {
		return -1;
	}
}



//////////////////////////////
//
// HumdrumToken::getSlurEndElisionLevel -- Returns the count of
//   elision marks ('&') preceding a slur end character ')'.
//   Returns -1 if there is no slur end character.
//   Default value: index = 0
//

int HumdrumToken::getSlurEndElisionLevel(int index) const {
	if (isDataType("**kern") || isDataType("**mens")) {
		return Convert::getKernSlurEndElisionLevel((string)(*this), index);
	} else {
		return -1;
	}
}



//////////////////////////////
//
// HumdrumToken::getPhraseEndElisionLevel -- Returns the count of
//   elision marks ('&') preceding a slur end character '}'.
//   Returns -1 if there is no phrase end character.
//   Default value: index = 0
//

int HumdrumToken::getPhraseEndElisionLevel(int index) const {
	if (isDataType("**kern")) {
		return Convert::getKernPhraseEndElisionLevel((string)(*this), index);
	} else {
		return -1;
	}
}



//////////////////////////////
//
// HumdrumToken::setStrandIndex -- Sets the 1-D strand index
//    that the token belongs to in the owning HumdrumFile.
//    By default the strand index is set to -1 when a HumdrumToken
//    is created.
//

void  HumdrumToken::setStrandIndex(int index) {
	m_strand = index;
}



//////////////////////////////
//
// HumdrumToken::incrementState -- update the rhythm analysis state variable.
//    This will prevent redundant recursive analysis in analyzeRhythm of
//    the HumdrumFileStructure class.
//

void HumdrumToken::incrementState(void) {
	m_rhycheck++;
}



//////////////////////////////
//
// HumdrumToken::getNextTokenCount -- Returns the number of tokens in the
//   spine/sub spine which follow this token.  Typically this will be 1,
//   but will be zero for a terminator interpretation (*-), and will be
//   2 for a split interpretation (*^).
//

int HumdrumToken::getNextTokenCount(void) const {
	return (int)m_nextTokens.size();
}



//////////////////////////////
//
// HumdrumToken::getPreviousTokenCount -- Returns the number of tokens
//   in the spine/sub-spine which precede this token.  Typically this will
//   be 1, but will be zero for an exclusive interpretation (starting with
//   "**"), and will be greater than one for a token which follows a
//   spine merger (using *v interpretations).
//

int HumdrumToken::getPreviousTokenCount(void) const {
	return (int)m_previousTokens.size();
}



//////////////////////////////
//
// HumdrumToken::printCsv -- print token in CSV format.
// default value: out = std::cout
//

ostream& HumdrumToken::printCsv(ostream& out) {
	string& value = *this;
	int loc = (int)this->find(",");
	if (loc == (int)string::npos) {
		out << value;
	} else {
		out << '"';
		for (int i=0; i<(int)value.size(); i++) {
		   if (value[i] == '"') {
				out << '"' << '"';
			} else {
				out << value[i];
			}
		}
		out << '"';
	}
	return out;
}



//////////////////////////////
//
// HumdrumToken::printXml -- Print a HumdrumToken in XML format.
// default value: out = cout
// default value: level = 0
// default value: indent = "\t"
//
//

ostream& HumdrumToken::printXml(ostream& out, int level, const string& indent) {

	out << Convert::repeatString(indent, level);
	out << "<field";
	out << " n=\"" << getTokenIndex() << "\"";

	out << " track=\"" << getTrack() << "\"";
	if (getSubtrack() > 0) {
		out << " subtrack=\"" << getSubtrack() << "\"";
	}
	out << " token=\"" << Convert::encodeXml(((string)(*this))) << "\"";
	out << " xml:id=\"" << getXmlId() << "\"";
	out << ">\n";

	printXmlBaseInfo(out, level+1, indent);
	printXmlStructureInfo(out, level+1, indent);

	if (isData()) {
		if (isNote()) {
			out << Convert::repeatString(indent, level+1) << "<pitch";
			out << Convert::getKernPitchAttributes(((string)(*this)));
			out << "/>\n";
		}
	}

	printXmlContentInfo(out, level+1, indent);
	printXmlParameterInfo(out, level+1, indent);
	printXmlLinkedParameterInfo(out, level+1, indent);
	printXmlLinkedParameters(out, level+1, indent);

	out << Convert::repeatString(indent, level) << "</field>\n";
	return out;
}



//////////////////////////////
//
// HumdrumToken::printXmlLinkedParameters --
//

ostream&	HumdrumToken::printXmlLinkedParameters(ostream& out, int level, const string& indent) {
	if (m_parameterSet) {
		m_parameterSet->printXml(out, level, indent);
	}
	return out;
}



//////////////////////////////
//
// HumdrumToken::printXmlLinkedParameterInfo --
//

ostream& HumdrumToken::printXmlLinkedParameterInfo(ostream& out, int level, const string& indent) {
	if (m_linkedParameterTokens.empty()) {
		return out;
	}

	out << Convert::repeatString(indent, level);
	out << "<parameters-linked>\n";

	level++;
	for (int i=0; i<(int)m_linkedParameterTokens.size(); i++) {
		out << Convert::repeatString(indent, level);
		out << "<linked-parameter";
		out << " idref=\"";
		HLp owner = m_linkedParameterTokens[i]->getOwner();
		if (owner && owner->isGlobalComment()) {
			out << owner->getXmlId();
		} else {
			out << m_linkedParameterTokens[i]->getXmlId();
		}
		out << "\"";
		out << ">\n";
	}
	level--;

	out << Convert::repeatString(indent, level);
	out << "</parameters-linked>\n";

	return out;
}



//////////////////////////////
//
// HumdrumToken::printXmlBaseInfo -- print data type and spine info.
// default value: out = cout
// default value: level = 0
// default value: indent = "\t"
//

ostream& HumdrumToken::printXmlBaseInfo(ostream& out, int level,
		const string& indent) {

	// <dataType> redundant with
	// sequence/sequenceInfo/trackInfo/track@dataType
	out << Convert::repeatString(indent, level);
	out << "<dataType>" << getDataType().substr(2) << "</dataType>\n";

	out << Convert::repeatString(indent, level) << "<tokenType>";
	if (isNull()) {
		out << "null";
	} else if (isManipulator()) {
		out << "manipulator";
	} else if (isCommentLocal()) {
		out << "local-comment";
	} else if (isBarline()) {
		out << "barline";
	} else if (isData()) {
		out << "data";
	} else {
		out << "interpretation";
	}
	out << "</tokenType>\n";

	// <tokenFunction>
	if (isDataType("**kern")) {
		if (isNote()) {
			out << Convert::repeatString(indent, level) << "<tokenFunction>";
			out << "note" << "</tokenFunction>\n";
		} else if (isRest()) {
			out << Convert::repeatString(indent, level) << "<tokenFunction>";
			out << "note" << "</tokenFunction>\n";
		}
	}

	if (isNull()) {
		HumdrumToken* previous = getPreviousNonNullDataToken(0);
		if (previous != NULL) {
			out << Convert::repeatString(indent, level) << "<nullResolve";
			out << " text=\"";
			out << Convert::encodeXml(((string)(*previous))) << "\"";
			out << " idref=\"";
			out << previous->getXmlId();
			out << "\"/>\n";
		}
	}

	return out;
}



//////////////////////////////
//
// HumdrumToken::printXmlStructureInfo -- Prints structural information
//    other than spine analysis.
// default value: out = cout
// default value: level = 0
// default value: indent = "\t"
//

ostream& HumdrumToken::printXmlStructureInfo(ostream& out, int level,
		const string& indent) {

	if (getDuration().isNonNegative()) {
		out << Convert::repeatString(indent, level);
		out << "<duration" << Convert::getHumNumAttributes(getDuration());
		out << "/>\n";
	}

	return out;
}



//////////////////////////////
//
// HumdrumToken::printXmlContentInfo -- Print content analysis information.
// default value: out = cout
// default value: level = 0
// default value: indent = "\t"
//

ostream& HumdrumToken::printXmlContentInfo(ostream& out, int level,
		const string& indent) {
	if (hasSlurStart()) {
		out << Convert::repeatString(indent, level) << "<slur";
		if (isDefined("auto", "hangingSlur")) {
			out << " hanging=\"" << getValue("auto", "hangingSlur") << "\"";
		}
		out << ">" << endl;
		out << Convert::repeatString(indent, level+1);
		out << "<duration" << Convert::getHumNumAttributes(getSlurDuration());
		out << "/>\n";
		out << Convert::repeatString(indent, level) << "</slur>" << endl;
	}
	return out;
}



//////////////////////////////
//
// HumdrumToken::printGlobalXmlParameterInfo --
//

ostream& HumdrumToken::printGlobalXmlParameterInfo(ostream& out, int level, const string& indent) {
	((HumHash*)this)->printXmlAsGlobal(out, level, indent);
	return out;
}



//////////////////////////////
//
// HumdrumToken::printXmlParameterInfo -- Print contents of HumHash for token.
// default value: out = cout
// default value: level = 0
// default value: indent = "\t"
//

ostream& HumdrumToken::printXmlParameterInfo(ostream& out, int level,
		const string& indent) {
	((HumHash*)this)->printXml(out, level, indent);
	return out;
}



//////////////////////////////
//
// HumdrumToken::getXmlId -- Returns an XML id attribute based on the line
//     and field index for the location of the token in the HumdrumFile.
//     An optional parameter for a prefix can be given.  If this parameter
//     is an empty string, then the prefix set in the owning HumdrumFile
//     will instead be used.  The prefix cannot start with a digit, and
//     should not include a space charcter.
//

string HumdrumToken::getXmlId(const string& prefix) const {
	string output;
	if (prefix.size() > 0) {
		output = prefix;
	} else {
		output = getXmlIdPrefix();
	}
	output += "loc" + to_string(getLineIndex()) + "_";
	output += to_string(getFieldIndex());
	// subtoken IDS could be added here.
	return output;
}



//////////////////////////////
//
// HumdrumToken::getXmlIdPrefix -- Returns the XML ID prefix from the HumdrumFile
//   structure via the HumdrumLine on which the token resides.
//

string HumdrumToken::getXmlIdPrefix(void) const {
	auto own = getOwner();
	if (own == NULL) {
		return "";
	}
	return own->getXmlIdPrefix();
}



//////////////////////////////
//
// operator<< -- Needed to avoid interaction with the HumHash parent class.
//

ostream& operator<<(ostream& out, const HumdrumToken& token) {
	out << token.c_str();
	return out;
}


ostream& operator<<(ostream& out, HumdrumToken* token) {
	if (token) {
		out << token->c_str();
	} else {
		out << "{NULL}";
	}
	return out;
}



//////////////////////////////
//
// printSequence --
//    default value: out = cout;
//

ostream& printSequence(vector<vector<HTp> >& sequence, ostream& out) {
	for (int i=0; i<(int)sequence.size(); i++) {
		for (int j=0; j<(int)sequence[i].size(); j++) {
			out << sequence[i][j];
			if (j < (int)sequence[i].size() - 1) {
				out << '\t';
			}
		}
		out << endl;
	}
	return out;
}


ostream& printSequence(vector<HTp>& sequence, ostream& out) {
	for (int i=0; i<(int)sequence.size(); i++) {
		out << sequence[i] << endl;
	}
	return out;
}



//////////////////////////////
//
// HumdrumToken::getSlurStartToken -- Return a pointer to the token
//     which starts the given slur.  Returns NULL if no start.  Assumes that
//     HumdrumFileContent::analyzeKernSlurs() has already been run.
//				<parameter key="slurEnd" value="HT_140366146702320" idref=""/>
//

HTp HumdrumToken::getSlurStartToken(int number) {
	string tag = "slurStartId";
	if (number > 1) {
		tag += to_string(number);
	}
	HTp value = getValueHTp("auto", tag);
	return value;
}


//////////////////////////////
//
// HumdrumToken::getSlurStartNumber -- Given a slur ending number, 
//    return the slur start number that it pairs with.
//

int HumdrumToken::getSlurStartNumber(int endnumber) {
	string tag = "slurStartNumber";
	if (endnumber > 1) {
		tag += to_string(endnumber);
	}
	int value = getValueInt("auto", tag);
	return value;
}



//////////////////////////////
//
// HumdrumToken::getSlurEndToken -- Return a pointer to the token
//     which ends the given slur.  Returns NULL if no end.  Assumes that
//     HumdrumFileContent::analyzeKernSlurs() has already been run.
//				<parameter key="slurStart" value="HT_140366146702320" idref=""/>
//

HTp HumdrumToken::getSlurEndToken(int number) {
	string tag = "slurEnd";
	if (number > 1) {
		tag += to_string(number);
	}
	return getValueHTp("auto", tag);
}




//////////////////////////////
//
// HumdrumToken::getPhraseStartToken -- Return a pointer to the token
//     which starts the given phrase.  Returns NULL if no start.  Assumes that
//     HumdrumFileContent::analyzeKernPhrasings() has already been run.
//				<parameter key="phraseEnd" value="HT_140366146702320" idref=""/>
//

HTp HumdrumToken::getPhraseStartToken(int number) {
	string tag = "phraseStart";
	if (number > 1) {
		tag += to_string(number);
	}
	return getValueHTp("auto", tag);
}



//////////////////////////////
//
// HumdrumToken::getPhraseEndToken -- Return a pointer to the token
//     which ends the given phrase.  Returns NULL if no end.  Assumes that
//     HumdrumFileContent::analyzeKernPhrasings() has already been run.
//				<parameter key="phraseStart" value="HT_140366146702320" idref=""/>
//

HTp HumdrumToken::getPhraseEndToken(int number) {
	string tag = "phraseEnd";
	if (number > 1) {
		tag += to_string(number);
	}
	return getValueHTp("auto", tag);
}



//////////////////////////////
//
// HumdrumToken::resolveNull --
//

HTp HumdrumToken::resolveNull(void) {
	if (m_nullresolve == NULL) {
		HLp hline = getOwner();
		if (hline) {
			HumdrumFile* infile = hline->getOwner();
			infile->resolveNullTokens();
		}
		if (m_nullresolve == NULL) {
			return this;
		} else {
			return m_nullresolve;
		}
		return this;
	} else {
		return m_nullresolve;
	}
}



//////////////////////////////
//
// HumdrumToken::setNullResolution --
//

void HumdrumToken::setNullResolution(HTp resolution) {
	m_nullresolve = resolution;
}



//////////////////////////////
//
// HumdrumToken::copyStucture --
//

void HumdrumToken::copyStructure(HTp token) {
	m_strand = token->m_strand;
	HLp temp_owner = m_address.m_owner;
	m_address = token->m_address;
	m_address.m_owner = NULL;  // This will in general be different, so do not copy.
	m_address.m_owner = temp_owner; // But preserve in case already set.
	// m_nullresolve: set this?
}



//////////////////////////////
//
// HumdrumToken::getStrophe -- return the strophe that the token belongs to,
//    or NULL if it is not in a strophe.
//

HTp HumdrumToken::getStrophe(void) {
	return m_strophe;
}



//////////////////////////////
//
// HumdrumToken::setStrophe -- Set the *S/ line of the strophe
//    or NULL if it is not formatted correctly.
//

void HumdrumToken::setStrophe(HTp strophe) {
	if (!strophe) {
		clearStrophe();
		return;
	}
	if (strophe->compare(0, 3, "*S/") != 0) {
		// invalid strophe marker.
		clearStrophe();
		return;
	}
	m_strophe = strophe;
}



//////////////////////////////
//
// HumdrumToken::hasStrophe -- return true if the token is in a strophe; otherwise,
//    return false.
//

bool HumdrumToken::hasStrophe(void) {
	return m_strophe ? true : false;
}



//////////////////////////////
//
// HumdrumToken::clearStrophe -- return true if the token is in a strophe; otherwise,
//    return false.
//

void HumdrumToken::clearStrophe(void) {
	m_strophe = NULL;
}



//////////////////////////////
//
// HumdrumToken::getStropheStartIndex -- return the starting line of the strophe
//    sequence.  Returns -1 if not in a strophe.
//

int HumdrumToken::getStropheStartIndex(void) {
	if (!m_strophe) {
		return -1;
	}
	return m_strophe->getLineIndex();
}



//////////////////////////////
//
// HumdrumToken::isFirstStrophe -- Returns true if the token is in the first 
//    strophe variant.  Returns true if not in a strophe.
//

bool HumdrumToken::isFirstStrophe(void) {
	if (!m_strophe) {
		return true;
	}
	HTp toleft = m_strophe->getPreviousField();
	if (!toleft) {
		return true;
	}
	int track = m_strophe->getTrack();
	int ltrack = toleft->getTrack();
	return track != ltrack;
}


bool HumdrumToken::isPrimaryStrophe(void) {
	return isFirstStrophe();
}



//////////////////////////////
//
// HumdrumToken::isStrophe -- Return true if the token has the given strophe
//   label.
//

bool HumdrumToken::isStrophe(const string& label) {
	if (!m_strophe) {
		return false;
	}
	if (label.empty()) {
		return *m_strophe == "*S/";
	}
	if (label[0] == '*') {
		return *m_strophe == label;
	}
	return m_strophe->substr(3) == label;
}



//////////////////////////////
//
// HumdrumToken::getStropheLabel -- Return the strophe label after *S/ in the
//    strophe token.  Returns the empty string when not in a strophe.
//

string HumdrumToken::getStropheLabel(void) {
	if (!m_strophe) {
		return "";
	}
	if (*m_strophe == "*S/") {
		return "";
	}
	return m_strophe->substr(3);
}





///////////////////////////////////////////////////////////////////////////
//
// MuseEventSet class functions --
//


//////////////////////////////
//
// MuseEventSet::MuseEventSet --
//

MuseEventSet::MuseEventSet (void) {
	events.reserve(20);
	clear();
}


MuseEventSet::MuseEventSet(HumNum atime) {
	setTime(atime);
	events.reserve(20);
}

MuseEventSet::MuseEventSet(const MuseEventSet& aSet) {
	absbeat = aSet.absbeat;
	events.resize(aSet.events.size());
	for (int i=0; i<(int)aSet.events.size(); i++) {
		events[i] = aSet.events[i];
	}
}



//////////////////////////////
//
// MuseEventSet::operator= --
//

MuseEventSet MuseEventSet::operator=(MuseEventSet& anevent) {
	if (&anevent == this) {
		return *this;
	}
	absbeat = anevent.absbeat;
	events.resize(anevent.events.size());
	for (int i=0; i<(int)events.size(); i++) {
		events[i] = anevent.events[i];
	}
	return *this;
}



//////////////////////////////
//
// MuseEventSet::clear --
//

void MuseEventSet::clear(void) {
	events.clear();
	absbeat.setValue(0,1);
}



//////////////////////////////
//
// MuseEventSet::setTime --
//

void MuseEventSet::setTime(HumNum abstime) {
	absbeat = abstime;
}



//////////////////////////////
//
// MuseEventSet::getTime --
//

HumNum MuseEventSet::getTime(void) {
	return absbeat;
}



//////////////////////////////
//
// MuseEventSet::appendRecord -- still have to sort after insertion...
//   also add a removeEvent function so deleted elements can be removed
//   gracefully.
//

void MuseEventSet::appendRecord(MuseRecord* arecord) {
	events.push_back(arecord);
}



//////////////////////////////
//
// MuseEventSet::operator[] --
//

MuseRecord& MuseEventSet::operator[](int eindex) {
	return *(events[eindex]);
}



//////////////////////////////
//
// MuseEventSet::getEventCount --
//

int MuseEventSet::getEventCount(void) {
	return (int)events.size();
}



///////////////////////////////////////////////////////////////////////////
//
// MuseData class functions --
//


//////////////////////////////
//
// MuseData::MuseData --
//

MuseData::MuseData(void) {
	m_data.reserve(100000);
}

MuseData::MuseData(MuseData& input) {
	m_data.resize(input.m_data.size());
	MuseRecord* temprec;
	int i;
	for (i=0; i<(int)m_data.size(); i++) {
		temprec  = new MuseRecord;
		*temprec = *(input.m_data[i]);
		m_data[i]  = temprec;
	}
	m_sequence.resize(input.m_sequence.size());
	for (i=0; i<(int)input.m_sequence.size(); i++) {
	  m_sequence[i] = new MuseEventSet;
	  *(m_sequence[i]) = *(input.m_sequence[i]);
	}

	m_name = input.m_name;
}



//////////////////////////////
//
// MuseData::~MuseData --
//

MuseData::~MuseData() {
	clear();
}



//////////////////////////////
//
// MuseData::operator= --
//

MuseData& MuseData::operator=(MuseData& input) {
	if (this == &input) {
		return *this;
	}
	m_data.resize(input.m_data.size());
	MuseRecord* temprec;
	int i;
	for (i=0; i<(int)m_data.size(); i++) {
		temprec = new MuseRecord;
		*temprec = *(input.m_data[i]);
		m_data[i] = temprec;
	}
	// do something with m_sequence...
	m_name = input.m_name;
	return *this;
}



//////////////////////////////
//
// MuseData::getLineCount -- return the number of lines in the MuseData file.
//

int MuseData::getLineCount(void) {
	return (int)m_data.size();
}



//////////////////////////////
//
// MuseData::append -- add a MuseRecord to end of file.
//

int MuseData::append(MuseRecord& arecord) {
	MuseRecord* temprec;
	temprec = new MuseRecord;
	*temprec = arecord;
	m_data.push_back(temprec);
	return (int)m_data.size()-1;
}


int MuseData::append(MuseData& musedata) {
	int oldsize = (int)m_data.size();
	int newlinecount = musedata.getLineCount();
	if (newlinecount <= 0) {
		return -1;
	}

	m_data.resize((int)m_data.size()+newlinecount);
	for (int i=0; i<newlinecount; i++) {
		m_data[i+oldsize] = new MuseRecord;
		*(m_data[i+oldsize]) = musedata[i];
	}
	return (int)m_data.size()-1;
}


int MuseData::append(string& charstring) {
	MuseRecord* temprec;
	temprec = new MuseRecord;
	temprec->setString(charstring);
	temprec->setType(E_muserec_unknown);
	temprec->setAbsBeat(0);
	m_data.push_back(temprec);
	return (int)m_data.size()-1;
}



//////////////////////////////
//
// MuseData::insert -- add a MuseRecord to middle of file.  Not the most
//   efficient, but not too bad as long as the file is not too long, the
//   insertion is close to the end of the file, and you don't use this
//   method to add a set of sequential lines (write a different function
//   for that).
//

void MuseData::insert(int lindex, MuseRecord& arecord) {
	MuseRecord* temprec;
	temprec = new MuseRecord;
	*temprec = arecord;

	m_data.resize(m_data.size()+1);
	for (int i=(int)m_data.size()-1; i>lindex; i--) {
		m_data[i] = m_data[i-1];
	}
	m_data[lindex] = temprec;
}



//////////////////////////////
//
// MuseData::clear --
//

void MuseData::clear(void) {
	for (int i=0; i<(int)m_data.size(); i++) {
		if (m_data[i] != NULL) {
			delete m_data[i];
			m_data[i] = NULL;
		}
	}
	for (int i=0; i<(int)m_sequence.size(); i++) {
		m_sequence[i]->clear();
		delete m_sequence[i];
		m_sequence[i] = NULL;
	}
	m_error.clear();
	m_data.clear();
	m_sequence.clear();
	m_name = "";
}



//////////////////////////////
//
// MuseData::operator[] --
//

MuseRecord& MuseData::operator[](int lindex) {
	return *(m_data[lindex]);
}



//////////////////////////////
//
// MuseData::getRecord --
//

MuseRecord& MuseData::getRecord(int lindex) {
	return *(m_data[lindex]);
}


//////////////////////////////
//
// MuseData::getRecord -- This version with two index inputs is
//     used to access a data line based on the event time index (time sorted
//     viewpoint) and the particular record index for that event time.
//

MuseRecord& MuseData::getRecord(int eindex, int erecord) {
	return *(m_data[getEvent(eindex)[erecord].getLineIndex()]);
}



//////////////////////////////
//
// MuseData::read -- read a MuseData file from a file or input stream.
//   0x0a      = unix
//   0x0d      = apple
//   0x0d 0x0a = dos
//

int MuseData::read(istream& input) {
	m_error.clear();
	string dataline;
	dataline.reserve(256);
	int character;
	char value;
	int  isnewline;
	char lastvalue = 0;

	while (!input.eof()) {
		character = input.get();
		if (input.eof()) {
			// end of file found without a newline termination on last line.
			if (dataline.size() > 0) {
				MuseData::append(dataline);
				dataline.clear();
				break;
			}
		}
		value = (char)character;
		if ((value == 0x0d) || (value == 0x0a)) {
			isnewline = 1;
		} else {
			isnewline = 0;
		}

		if (isnewline && (value == 0x0a) && (lastvalue == 0x0d)) {
			// ignore the second newline character in a dos-style newline.
			lastvalue = value;
			continue;
		} else {
			lastvalue = value;
		}

		if (isnewline) {
			MuseData::append(dataline);
			dataline.clear();
		} else {
			dataline.push_back(value);
		}
	}

	for (int i=0; i<(int)m_data.size(); i++) {
		m_data[i]->setLineIndex(i);
	}

	doAnalyses();
	if (hasError()) {
		cerr << m_error << endl;
		return 0;
	} else {
		return 1;
	}
}


int MuseData::readFile(const string& filename) {
	ifstream infile(filename);
	return MuseData::read(infile);
}

int MuseData::readString(const string& data) {
	stringstream ss;
	ss << data;
	return MuseData::read(ss);
}



//////////////////////////////
//
// MuseData::doAnalyses --  perform post-processing analysis of the data file
//    (in the correct order).
//

void MuseData::doAnalyses(void) {
	analyzeType();
	analyzeTpq();
	if (hasError()) { return; }
	assignHeaderBodyState();
   analyzeLayers();
	analyzeRhythm();
	if (hasError()) { return; }
	constructTimeSequence();
	if (hasError()) { return; }
	analyzePitch();
	if (hasError()) { return; }
	analyzeTies();
	if (hasError()) { return; }
}



//////////////////////////////
//
// MuseData::analyzeTpq -- Read $ records for Q: field values and store
//    the ticks-per-quarter note value on each line after that $ record.
//    This is used later to extract the logical duration of notes and rests.
//

void MuseData::analyzeTpq(void) {
	HumRegex hre;
	int ticks = 0;
	for (int i=0; i<getLineCount(); i++) {
		MuseRecord* mr = &getRecord(i);
		if (!mr->isAttributes()) {
			mr->setTpq(ticks);

			continue;
		}
		string line = getLine(i);
		if (hre.search(line, " Q:(\\d+)")) {
			ticks = hre.getMatchInt(1);
		}
		mr->setTpq(ticks);
	}
}



//////////////////////////////
//
// MuseData::analyzePitch -- calculate the pitch of all notes in terms
//    of their base40 value.
//

void MuseData::analyzePitch() {
	for (int i=0; i<(int)m_data.size(); i++) {
		m_data[i]->setMarkupPitch(m_data[i]->getBase40());
	}
}



//////////////////////////////
//
// MuseData::analyzeTies -- identify which notes are tied to each other.
//

void MuseData::analyzeTies(void) {
	for (int i=0; i<(int)m_sequence.size(); i++) {
		for (int j=0; j<m_sequence[i]->getEventCount(); j++) {
			if (!getEvent(i)[j].tieQ()) {
				continue;
			}
			processTie(i, j, -1);
		}
	}
}



//////////////////////////////
//
// MuseData::processTie -- follow a tied note to the last note
//   in the tied m_sequence, filling in the tie information along the way.
//   Hanging ties (particularly ties note at the ends of first repeats, etc.)
//   still need to be considered.
//

void MuseData::processTie(int eindex, int rindex, int lastindex) {
	int& i = eindex;
	int& j = rindex;


	// lineindex = index of line in original file for the current line.
	int lineindex = getEvent(i)[j].getLineIndex();


	if ((lastindex < 0) &&
		 (m_data[lineindex]->getLastTiedNoteLineIndex() >= 0)) {
		// If there previously tied note already marked in the data, then
		// this note has already been processed for ties, so exit function
		// without doing any further processing.
		return;
	}

	// store the location of the note tied to previously:
	m_data[lineindex]->setLastTiedNoteLineIndex(lastindex);

	// If the current note contains a tie marker, then there is
	// another tied note in the future, so go look for it.
	if (!m_data[lineindex]->tieQ()) {
		m_data[lineindex]->setNextTiedNoteLineIndex(-1);
		return;
	}

	// There is another note tied to this one in the future, so
	// first get the absolute time location of the future tied note
	HumNum abstime    = m_data[lineindex]->getAbsBeat();
	HumNum notedur    = m_data[lineindex]->getNoteDuration();
	HumNum searchtime = abstime + notedur;

	// Get the event index which occurs at the search time:
	int nexteindex = getNextEventIndex(eindex, abstime + notedur);

	if (nexteindex < 0) {
		// Couldn't find any data at that absolute time index, so give up:
		m_data[lineindex]->setNextTiedNoteLineIndex(-1);
		return;
	}

	// The pitch of the tied note should match this one; otherwise, it
	// would not be a tied note...
	int base40 = m_data[lineindex]->getPitch();

	// The tied note will preferrably be found in the same track as the
	// current note (but there could be a cross-track tie occurring, so
	// check for that if there is no same-track tie):
	int track = m_data[lineindex]->getTrack();

	int nextrindex = searchForPitch(nexteindex, base40, track);
	if (nextrindex < 0) {
		// Didn't find specified note at the given event index in the given
		// track, so search for the same pitch in any track at the event time:
		 nextrindex = searchForPitch(nexteindex, base40, -1);
	}

	if (nextrindex < 0) {
		// Failed to find a note at the target event time which could be
		// tied to the current note (no pitches at that time which match
		// the pitch of the current note).  This is a haning tie which is
		// either a data error, or is a tie to a note at an earlier time
		// in the music (such as at the beginning of a repeated section).
		// for now just ignore the hanging tie, but probably mark it in
		// some way in the future.
		m_data[lineindex]->setNextTiedNoteLineIndex(-1);
		return;
	}

	// now the specific note to which this one is tied to is known, so
	// go and process that note:

	int nextindex = getEvent(nexteindex)[nextrindex].getLineIndex();

	m_data[lineindex]->setNextTiedNoteLineIndex(nextindex);

	processTie(nexteindex, nextrindex, lineindex);
}



//////////////////////////////
//
// MuseData::searchForPitch -- search for a matching pitch in the given
//   track at the specified event index.  If the track is negative, then
//   find the first matching pitch in any track.
//
//   Will also have to separate by category, so that grace notes are
//   searched for separately from regular notes / chord notes, and
//   cue notes as well.
//

int MuseData::searchForPitch(int eventindex, int b40, int track) {
	int targettrack;
	int targetpitch;
	int targettype;

	for (int j=0; j<m_sequence[eventindex]->getEventCount(); j++) {
		targettype = getEvent(eventindex)[j].getType();
		if ((targettype != E_muserec_note_regular) &&
			 (targettype != E_muserec_note_chord) ) {
			// ignore non-note data (at least ones without durations):
			continue;
		}
		targettrack = getEvent(eventindex)[j].getTrack();
		if ((track >= 0) && (track != targettrack)) {
			continue;
		}
		targetpitch = getEvent(eventindex)[j].getPitch();
		if (targetpitch == b40) {
			return j;
		}
	}

	return -1;
}



//////////////////////////////
//
// MuseData::getNextEventIndex -- return the event index for the given
//   absolute time value.  The starting index is given first, and it
//   is assumed that the target absolute time occurs on or after the
//   starting index value.  Returns -1 if that absolute time is not
//   found in the data (or occurs before the start index.
//

int MuseData::getNextEventIndex(int startindex, HumNum target) {
	int output = -1;
	for (int i=startindex; i<(int)m_sequence.size(); i++) {
		if (m_sequence[i]->getTime() == target) {
			output = i;
			break;
		}
	}
	return output;
}


//////////////////////////////
//
// MuseData::last -- return the last record in the data.  Make sure
// that isEmpty() is not true before calling this function.
//

MuseRecord& MuseData::last(void) {
	return (*this)[getNumLines()-1];
}



//////////////////////////////
//
// MuseData::isEmpty -- return true if there are no MuseRecords in the
//    object; otherwise returns true;
//

int MuseData::isEmpty(void) {
	return m_data.empty();
}



//////////////////////////////
//
// MuseData::analyzeType --
//

void MuseData::analyzeType(void) {
	int commentQ = 0;
	int h = 0;
	MuseData& thing = *this;
	int foundattributes = 0;
	int foundend = 0;

	HumRegex hre;
	int groupmemberships = -1;
	for (int i=0; i<getLineCount(); i++) {
		string line = thing[i].getLine();
		if (hre.search(line, "^Group memberships:")) {
			groupmemberships = i;
			break;
		}
	}
	if (groupmemberships < 0) {
		cerr << "Error cannot parse MuseData content" << endl;
		return;
	}

	thing[groupmemberships].setType(E_muserec_header_11);

	h = 11;
	for (int i=groupmemberships-1; i>=0; i--) {
		if (thing[i].getLength() > 0) {
			if (thing[i][0] == '@') {
				thing[i].setType(E_muserec_comment_line);
				continue;
			}
			if (thing[i][0] == '&') {
				// start or end of multi-line comment;
				commentQ = !commentQ;
				if (!commentQ) {
					thing[i].setType(E_muserec_comment_toggle);
	       continue;
				}
			}
			if (commentQ) {
				thing[i].setType(E_muserec_comment_toggle);
				continue;
			}
		}
		h--;
		if      (h==1)  { thing[i].setType(E_muserec_header_1);  continue; }
		else if (h==2)  { thing[i].setType(E_muserec_header_2);  continue; }
		else if (h==3)  { thing[i].setType(E_muserec_header_3);  continue; }
		else if (h==4)  { thing[i].setType(E_muserec_header_4);  continue; }
		else if (h==5)  { thing[i].setType(E_muserec_header_5);  continue; }
		else if (h==6)  { thing[i].setType(E_muserec_header_6);  continue; }
		else if (h==7)  { thing[i].setType(E_muserec_header_7);  continue; }
		else if (h==8)  { thing[i].setType(E_muserec_header_8);  continue; }
		else if (h==9)  { thing[i].setType(E_muserec_header_9);  continue; }
		else if (h==10) { thing[i].setType(E_muserec_header_10); continue; }
	}

	commentQ = 0;
	h = 11;
	for (int i=groupmemberships+1; i<getNumLines(); i++) {

		if (thing[i].getLength() > 0) {
			if (thing[i][0] == '@') {
				thing[i].setType(E_muserec_comment_line);
				continue;
			}
			if (thing[i][0] == '&') {
				// start or end of multi-line comment;
				commentQ = !commentQ;
				if (!commentQ) {
					thing[i].setType(E_muserec_comment_toggle);
	       continue;
				}
			}
			if (commentQ) {
				thing[i].setType(E_muserec_comment_toggle);
				continue;
			}
		}
		h++;
		if      (h==1)  { thing[i].setType(E_muserec_header_1);  continue; }
		else if (h==2)  { thing[i].setType(E_muserec_header_2);  continue; }
		else if (h==3)  { thing[i].setType(E_muserec_header_3);  continue; }
		else if (h==4)  { thing[i].setType(E_muserec_header_4);  continue; }
		else if (h==5)  { thing[i].setType(E_muserec_header_5);  continue; }
		else if (h==6)  { thing[i].setType(E_muserec_header_6);  continue; }
		else if (h==7)  { thing[i].setType(E_muserec_header_7);  continue; }
		else if (h==8)  { thing[i].setType(E_muserec_header_8);  continue; }
		else if (h==9)  { thing[i].setType(E_muserec_header_9);  continue; }
		else if (h==10) { thing[i].setType(E_muserec_header_10); continue; }

		else if (h==11) { thing[i].setType(E_muserec_header_11); continue; }
		else if (h==12) { thing[i].setType(E_muserec_header_12); continue; }

		if (thing[i].getLength() == 0) {
			thing[i].setType(E_muserec_empty);
			continue;
		}

		if ((h > 12) && (thing[i][0] != '$') && (foundattributes == 0)) {
			thing[i].setType(E_muserec_header_12);
			continue;
		}

		if (foundend && thing[i][0] != '/') {
			thing[i].setType(E_muserec_endtext);
			continue;
		}

		switch (thing[i][0]) {
			case 'A': case 'B': case 'C': case 'D': case 'E': case 'F':
			case 'G': thing[i].setType(E_muserec_note_regular);       break;
			case ' ': thing[i].setType(E_muserec_note_chord);         break;
			case 'c': thing[i].setType(E_muserec_note_cue);           break;
			case 'g': thing[i].setType(E_muserec_note_grace);         break;
			case 'P': thing[i].setType(E_muserec_print_suggestion);   break;
			case 'S': thing[i].setType(E_muserec_sound_directives);   break;
			case '/': thing[i].setType(E_muserec_end);
		   foundend = 1;
		   break;
			case 'a': thing[i].setType(E_muserec_append);             break;
			case 'b': thing[i].setType(E_muserec_backspace);          break;
			case 'f': thing[i].setType(E_muserec_figured_harmony);    break;
			case 'i': thing[i].setType(E_muserec_rest_invisible);     break;
			case 'm': thing[i].setType(E_muserec_measure);            break;
			case 'r': thing[i].setType(E_muserec_rest);               break;
			case '*': thing[i].setType(E_muserec_musical_directions); break;
			case '$': thing[i].setType(E_muserec_musical_attributes);
						 foundattributes = 1;
						 break;
		}
	}
}



//////////////////////////////
//
// MuseData::analyzeRhythm -- calculate the start time in quarter notes
//   for each note/rest in the file.
//
//   Secondary chord notes may or may not have a duration listed.
//   If they do not, then the duration of the note is the same
//   as the primary note of the chord.
//
// char*            getTickDurationField         (char* output);
//

void MuseData::analyzeRhythm(void) {
	HumNum cumulative(0,1);
	HumNum linedur(0,1);
	int tpq = 1;
	HumRegex hre;
	HumNum figadj = 0;   // needed for figured harmony
	HumNum primarychordnoteduration(0,1);  // needed for chord notes

	for (int i=0; i<(int)m_data.size(); i++) {
		if (m_data[i]->isAttributes()) {
			string line = m_data[i]->getLine();
			if (hre.search(line, "Q:(\\d+)", "")) {
				tpq = hre.getMatchInt(1);
			}
		}

		if (m_data[i]->isChordNote()) {
			// insert an automatic back command for chord tones
			// also deal with cue-size note chords?
			m_data[i]->setAbsBeat(cumulative - primarychordnoteduration);

			// Check to see if the secondary chord note has a duration.
			// If so, then set the note duration to that value; otherwise,
			// set the note duration to the duration of the primary chord
			// note (first note before the current note which is not a chord
			// note).
			string buffer = m_data[i]->getTickDurationField();
			if (hre.search(buffer, "\\d", "")) {
				m_data[i]->setNoteDuration(m_data[i]->getNoteTickDuration(), tpq);
			} else {
				m_data[i]->setNoteDuration(primarychordnoteduration);
			}
			m_data[i]->setLineDuration(0);
		} else if (m_data[i]->isFiguredHarmony()) {
			// Tick values on figured harmony lines do not advance the
			// cumulative timestamp; instead they temporarily advance
			// the time placement of the next figure if it occurs
			// during the same note as the previous figure.
			m_data[i]->setAbsBeat(cumulative + figadj);
			HumNum tick = m_data[i]->getLineTickDuration();
			if (tick == 0) {
				figadj = 0;
			} else {
				HumNum dur = tick;
				dur /= tpq;
				figadj += dur;
			}
		} else {
			m_data[i]->setAbsBeat(cumulative);
			m_data[i]->setNoteDuration(m_data[i]->getNoteTickDuration(), tpq);
			m_data[i]->setLineDuration(m_data[i]->getNoteDuration());
			linedur.setValue(m_data[i]->getLineTickDuration(), tpq);
			cumulative += linedur;
		}

		switch (m_data[i]->getType()) {
			case E_muserec_note_regular:
			// should also worry about cue and grace note chords?
			primarychordnoteduration = linedur;
		}
	}

	// adjust Sound and Print records so that they occur at the same
	// absolute time as the note they affect.
	for (int i=1; i<(int)m_data.size(); i++) {
		switch (m_data[i]->getType()) {
			case E_muserec_print_suggestion:
			case E_muserec_sound_directives:
				m_data[i]->setAbsBeat(m_data[i-1]->getAbsBeat());
		}
	}

}



//////////////////////////////
//
// MuseData::getInitialTpq -- return the Q: field in the first $ record
//    at the top of the file.
//

int MuseData::getInitialTpq(void) {
	int output = 0;
	if (m_data.empty()) {
		return output;
	}
	HumRegex hre;
	int i;
	if (m_data[0]->getType() == E_muserec_unknown) {
		// search for first line which starts with '$':
		for (i=0; i<(int)m_data.size(); i++) {
			if (m_data[i]->getLength() <= 0) {
				continue;
			}
			if ((*m_data[i])[0] == '$') {
				string line = m_data[i]->getLine();
				if (hre.search(line, "Q:(\\d+)", "")) {
					output = hre.getMatchInt(1);
				}
				break;
			}
		}
	} else {
		for (int i=0; i<(int)m_data.size(); i++) {
			if (m_data[i]->getType() == E_muserec_musical_attributes) {
				string line = m_data[i]->getLine();
				if (hre.search(line, "Q:(\\d+)", "")) {
					output = hre.getMatchInt(1);
				}
				break;
			}
		}
	}

	return output;
}



//////////////////////////////
//
// constructTimeSequence -- Make a list of the lines in the file
//    sorted by the absolute time at which they occur.
//

void MuseData::constructTimeSequence(void) {
	// * clear old event set
	// * allocate the size to match number of lines (* 2 probably).

	MuseData& thing = *this;
	for (int i=0; i<(int)m_data.size(); i++) {
		insertEventBackwards(thing[i].getAbsBeat(), &thing[i]);
		if (hasError()) {
			return;
		}
	}
}



///////////////////////////////
//
// MuseData::getEventCount -- returns the number of unique times
//    at which data line occur in the file.
//

int MuseData::getEventCount(void) {
	return (int)m_sequence.size();
}



///////////////////////////////
//
// MuseData::getEvent --
//

MuseEventSet& MuseData::getEvent(int eindex) {
	return *(m_sequence[eindex]);
}



///////////////////////////////////////////////////////////////////////////
//
// private functions
//


//////////////////////////////
//
// printSequenceTimes --
//

void printSequenceTimes(vector<MuseEventSet*>& m_sequence) {
	for (int i=0; i<(int)m_sequence.size(); i++) {
		cout << m_sequence[i]->getTime().getFloat() << " ";
	}
	cout << endl;
}



//////////////////////////////
//
// MuseData::insertEventBackwards -- insert an event into a time-sorted
//    organization of the file. Searches for the correct time location to
//    insert event starting at the end of the list (since MuseData files
//    are mostly sorted in time.
//
//

void MuseData::insertEventBackwards(HumNum atime, MuseRecord* arecord) {
	if (m_sequence.empty()) {
		MuseEventSet* anevent = new MuseEventSet;
		anevent->setTime(atime);
		anevent->appendRecord(arecord);
		m_sequence.push_back(anevent);
		return;
	}

	for (int i=(int)m_sequence.size()-1; i>=0; i--) {
		if (m_sequence[i]->getTime() == atime) {
			m_sequence[i]->appendRecord(arecord);
			return;
		} else if (m_sequence[i]->getTime() < atime) {
			// insert new event entry after the current one since it occurs later.
			MuseEventSet* anevent = new MuseEventSet;
			anevent->setTime(atime);
			anevent->appendRecord(arecord);
			if (i == (int)m_sequence.size()-1) {
				// just append the event at the end of the list
				m_sequence.push_back(anevent);
	    return;
			} else {
				// event has to be inserted before end of list, so move
				// later ones up in list.
				m_sequence.resize(m_sequence.size()+1);
				for (int j=(int)m_sequence.size()-1; j>i+1; j--) {
					m_sequence[j] = m_sequence[j-1];
				}
				// store the newly created event entry in m_sequence:
				m_sequence[i+1] = anevent;
				return;
			}
		}
	}
	stringstream ss;
	ss << "Funny error occurred at time " << atime;
	setError(ss.str());
}



//////////////////////////////
//
// MuseData::getTiedDuration -- these version acess the record lines
//    via the time-sorted event index.
//

HumNum MuseData::getTiedDuration(int eindex, int erecord) {
	return getTiedDuration(getLineIndex(eindex, erecord));
}



//////////////////////////////
//
// MuseData:getTiedDuration --
//

HumNum MuseData::getTiedDuration(int index) {
	HumNum output(0,1);

	// if the line is not a regular/chord note with duration, then
	// return the duration field.
	if ((getRecord(index).getType() != E_muserec_note_chord) &&
		 (getRecord(index).getType() != E_muserec_note_regular) ) {
		return output;
	}

	// if the note is tied to a previous note, then return a
	// duration of 0 (behavior may change in the future).
	if (getRecord(index).getLastTiedNoteLineIndex() >= 0) {
		return output;
	}

	// if the note is not tied to anything into the future, then
	// gives it actual duration
	if (getRecord(index).getNextTiedNoteLineIndex() < 0) {
		return getRecord(index).getNoteDuration();
	}

	// this is start of a group of tied notes.  Start figuring out
	// how long the duration of the tied group is.
	output = getRecord(index).getNoteDuration();
	int myindex = index;
	while (getRecord(myindex).getNextTiedNoteLineIndex() >= 0) {
		myindex = getRecord(myindex).getNextTiedNoteLineIndex();
		output += getRecord(myindex).getNoteDuration();
	}

	return output;
}



//////////////////////////////
//
// MuseData::getLineIndex -- return the line number of a particular
//    event/record index pair (the line index is the same as the index
//    into the list of lines).
//

int MuseData::getLineIndex(int eindex, int erecord) {
	return getRecord(eindex, erecord).getLineIndex();
}



//////////////////////////////
//
// MuseData::getLineDuration -- return the duration of an isolated line.
//

HumNum MuseData::getLineDuration(int eindex, int erecord) {
	return getRecord(eindex, erecord).getLineDuration();
}



//////////////////////////////
//
// MuseData::getNoteDuration -- return the duration of an isolated note.
//

HumNum MuseData::getNoteDuration(int eindex, int erecord) {
	return getRecord(eindex, erecord).getNoteDuration();
}



//////////////////////////////
//
// MuseData::getLastTiedNoteLineIndex -- return the line index of the
//     previous note to which this one is tied to.  Returns -1 if there
//     is no previous tied note.
//

int MuseData::getLastTiedNoteLineIndex(int eindex, int erecord) {
	return getRecord(eindex, erecord).getLastTiedNoteLineIndex();
}



//////////////////////////////
//
// MuseData::getNextTiedNoteLineIndex -- returns the line index of the
//    next note to which this one is tied to.  Returns -1 if there
//    is no previous tied note.
//

int MuseData::getNextTiedNoteLineIndex(int eindex, int erecord) {
	return getRecord(eindex, erecord).getNextTiedNoteLineIndex();
}



//////////////////////////////
//
// MuseData::getType -- return the record type of a particular event
//     record.
//

int MuseData::getType(int eindex, int erecord) {
	return getRecord(eindex, erecord).getType();
}



//////////////////////////////
//
// MuseData::getAbsBeat -- return the absolute beat time (quarter
//    note durations from the start of the music until the current
//    object.
//

HumNum MuseData::getAbsBeat(int lindex) {
	return m_data[lindex]->getAbsBeat();
}



//////////////////////////////
//
// MuseData::getLineTickDuration --
//

int MuseData::getLineTickDuration(int lindex) {
	return m_data[lindex]->getLineTickDuration();
}


//////////////////////////////
//
// MuseData::setFilename --
//

void MuseData::setFilename(const string& filename) {
	m_name = filename;
}



//////////////////////////////
//
// MuseData::getFilename --
//

string MuseData::getFilename(void) {
	return m_name;
}


/*

//////////////////////////////
//
// MuseData::getPartName --
//

string MuseData::getPartName(void) {
	string output;
	for (int i=0; i<getLineCount(); i++) {
		if (isPartName(i)) {
			output = getPartName(i);
			break;
		}
	}
	for (int i=(int)output.size() - 1; i>=0; i--) {
		if (isspace(output[i])) {
			output.resize((int)output.size() - 1);
		} else {
			break;
		}
	}
	return output;
}

*/


//////////////////////////////
//
// MuseData::getPartName -- return name of the part
//

string MuseData::getPartName(void) {
	int line = getPartNameIndex();
	if (line < 0) {
		return "";
	}
	HumRegex hre;
	string output = m_data[line]->getLine();
	hre.replaceDestructive(output, "", "^\\s+");
	hre.replaceDestructive(output, "", "\\s+$");
	return output;
}



//////////////////////////////
//
// MuseData::getPartNameIndex -- Search for the part name in the header.
//    search the entire file if it is missing (which it should not be.
//

int MuseData::getPartNameIndex(void) {
	int output = -1;
	for (int i=0; i<(int)m_data.size(); i++) {
		if (m_data[i]->isPartName()) {
			return i;
		}
	}
	return output;
}



//////////////////////////////
//
// MuseData::isMember -- returns true if the file belongs to the
//     given membership string.  Example memberships are "score",
//     "skore", "part", "sound".
//

int MuseData::isMember(const string& mstring) {
	for (int i=0; i<(int)m_data.size(); i++) {
		if (m_data[i]->getType() == E_muserec_group_memberships) {
			if (strstr(m_data[i]->getLine().c_str(), mstring.c_str()) != NULL) {
				return 1;
			} else {
				return 0;
			}
		}
		if (m_data[i]->getType() == E_muserec_musical_attributes) {
			break;
		}
	}
	return 0;
}



//////////////////////////////
//
// MuseData::getMembershipPartNumber -- returns the part number within
//     a given membership for the data.
// Example:
//     sound: part 1 of 4
//  in this case the value returned is 1.
//

int MuseData::getMembershipPartNumber(const string& mstring) {
	string searchstring = "^";
	searchstring += mstring;
	searchstring += ":";

	HumRegex hre;
	for (int i=0; i<(int)m_data.size(); i++) {
		if (m_data[i]->getType() == E_muserec_header_12) {
			string line = m_data[i]->getLine();
			if (hre.search(line, searchstring)) {
				if (hre.search(line, "part\\s*(\\d+)\\s*of\\s*(\\d+)")) {
					string partnum = hre.getMatch(1);
					return hre.getMatchInt(1);
				}
			}
		}
		if (m_data[i]->getType() == E_muserec_musical_attributes) {
			break;
		}
	}
	return 0;
}



////////////////////////////////
//
// MuseData::selectMembership --
//

void MuseData::selectMembership(const string& selectstring) {
	if (!isMember(selectstring)) {
		// not a member of the given membership, so can't select that
		// membership.
		return;
	}
	string buffer;
	buffer = "Group memberships: ";
	buffer += selectstring;

	for (int i=0; i<getNumLines(); i++) {
		if ((*this)[i].getType() == E_muserec_group_memberships) {
			(*this)[i].setLine(buffer);
		} else if ((*this)[i].getType() == E_muserec_header_12) {
			if (strncmp((*this)[i].getLine().c_str(), selectstring.c_str(),
					selectstring.size()) != 0) {
				(*this)[i].setType(E_muserec_deleted);
			}
		}
	}


}



///////////////////////////////
//
// MuseData::cleanLineEndings -- remove spaces at the end of lines
//

void MuseData::cleanLineEndings(void) {
	for (int i=0; i<this->getLineCount(); i++) {
		(*this)[i].cleanLineEnding();
	}
}



//////////////////////////////
//
// MuseData::getError --
//

std::string MuseData::getError(void) {
	return m_error;
}



//////////////////////////////
//
// MuseData::hasError --
//

bool MuseData::hasError(void) {
	return !m_error.empty();
}


//////////////////////////////
//
// MuseData::clearError --
//

void MuseData::clearError(void) {
	m_error.clear();
}



//////////////////////////////
//
// MuseData:::etError --
//

void MuseData::setError(const string& error) {
	m_error = error;
}



//////////////////////////////
//
// MuseData::getFileDuration --
//

HumNum MuseData::getFileDuration(void) {
	return getRecord(getLineCount()-1).getAbsBeat();
}



//////////////////////////////
//
// MuseData::getLine -- return the textual content of the given line index.
//

string MuseData::getLine(int index) {
	if (index < 0) {
		return "";
	}
	if (index >= getLineCount()) {
		return "";
	}
	string output = getRecord(index).getLine();
	return output;
}



//////////////////////////////
//
// MuseData::isPartName -- return true if partname line.
//

bool MuseData::isPartName(int index) {
	return getRecord(index).isPartName();
}



//////////////////////////////
//
// MuseData::isWorkInfo -- return true if a work info line.
//

bool MuseData::isWorkInfo(int index) {
	return getRecord(index).isWorkInfo();
}



//////////////////////////////
//
// MuseData::isWorkTitle -- return true if a work title line.
//

bool MuseData::isWorkTitle(int index) {
	return getRecord(index).isWorkTitle();
}



//////////////////////////////
//
// MuseData::isHeaderRecord -- return true if in the header.
//

bool MuseData::isHeaderRecord(int index) {
	return getRecord(index).isHeaderRecord();
}



//////////////////////////////
//
// MuseData::isBodyRecord -- return true if in the body.
//

bool MuseData::isBodyRecord(int index) {
	return getRecord(index).isBodyRecord();
}



//////////////////////////////
//
// MuseData::isCopyright -- return true if a work title line.
//

bool MuseData::isCopyright(int index) {
	return getRecord(index).isCopyright();
}



//////////////////////////////
//
// MuseData::isMovementTitle -- return true if a movement title line.
//

bool MuseData::isMovementTitle(int index) {
	return getRecord(index).isMovementTitle();
}



//////////////////////////////
//
// MuseData::isRegularNote -- return true if a regular note line.
//     This is either a single note, or the first note in a chord.
//

bool MuseData::isRegularNote(int index) {
	return getRecord(index).isRegularNote();
}



//////////////////////////////
//
// MuseData::isAnyNote -- return true if note line of any time.
//

bool MuseData::isAnyNote(int index) {
	return getRecord(index).isAnyNote();
}



//////////////////////////////
//
// MuseData::isEncoder -- return true if note line.
//

bool MuseData::isEncoder(int index) {
	return getRecord(index).isEncoder();
}



//////////////////////////////
//
// MuseData::isId -- return true if Id line.
//

bool MuseData::isId(int index) {
	return getRecord(index).isId();
}



//////////////////////////////
//
// MuseData::isSource -- return true if note line.
//

bool MuseData::isSource(int index) {
	return getRecord(index).isSource();
}



//////////////////////////////
//
// MuseData::getWorkInfo --
//

std::string MuseData::getWorkInfo(void) {
	for (int i=0; i<getLineCount(); i++) {
		if (isWorkInfo(i)) {
			return trimSpaces(getLine(i));
		} else if (isAnyNote(i)) {
			break;
		}
	}
	return "";
}



//////////////////////////////
//
// MuseData::getOpus --
//     WK#:1,1       MV#:1
//

string MuseData::getOpus(void) {
	string workinfo = getWorkInfo();
	HumRegex hre;
	if (hre.search(workinfo, "^\\s*WK\\s*#\\s*:\\s*(\\d+)")) {
		return hre.getMatch(1);
	}
	return "";
}



//////////////////////////////
//
// MuseData::getNumber -- Return number of work in opus.
//     WK#:1,1       MV#:1
//

string MuseData::getNumber(void) {
	string workinfo = getWorkInfo();
	HumRegex hre;
	if (hre.search(workinfo, "^\\s*WK\\s*#\\s*:\\s*(\\d+)\\s*[,/]\\s*(\\d+)")) {
		return hre.getMatch(2);
	}
	return "";
}



//////////////////////////////
//
// MuseData::getMovementNumber --
//     WK#:1,1       MV#:1
//

string MuseData::getMovementNumber(void) {
	string workinfo = getWorkInfo();
	HumRegex hre;
	if (hre.search(workinfo, "MV\\s*#\\s*:\\s*(\\d+)")) {
		return hre.getMatch(1);
	}
	return "";
}



//////////////////////////////
//
// MuseData::getWorkTitle --
//

std::string MuseData::getWorkTitle(void) {
	for (int i=0; i<getLineCount(); i++) {
		if (isWorkTitle(i)) {
			return trimSpaces(getLine(i));
		} else if (isAnyNote(i)) {
			break;
		}
	}
	return "";
}



//////////////////////////////
//
// MuseData::getCopyright --
//

std::string MuseData::getCopyright(void) {
	for (int i=0; i<getLineCount(); i++) {
		if (isCopyright(i)) {
			return trimSpaces(getLine(i));
		} else if (isAnyNote(i)) {
			break;
		}
	}
	return "";
}



//////////////////////////////
//
// MuseData::getMovementTitle --
//

std::string MuseData::getMovementTitle(void) {
	for (int i=0; i<getLineCount(); i++) {
		if (isMovementTitle(i)) {
			return trimSpaces(getLine(i));
		} else if (isAnyNote(i)) {
			break;
		}
	}
	return "";
}



//////////////////////////////
//
// MuseData::getSource --
//

std::string MuseData::getSource(void) {
	for (int i=0; i<getLineCount(); i++) {
		if (isSource(i)) {
			return trimSpaces(getLine(i));
		} else if (isAnyNote(i)) {
			break;
		}
	}
	return "";
}



//////////////////////////////
//
// MuseData::getEncoder --
//

std::string MuseData::getEncoder(void) {
	for (int i=0; i<getLineCount(); i++) {
		if (isEncoder(i)) {
			return trimSpaces(getLine(i));
		} else if (isAnyNote(i)) {
			break;
		}
	}
	return "";
}



//////////////////////////////
//
// MuseData::getId --
//

std::string MuseData::getId(void) {
	for (int i=0; i<getLineCount(); i++) {
		if (isId(i)) {
			return trimSpaces(getLine(i));
		} else if (isAnyNote(i)) {
			break;
		}
	}
	return "";
}



//////////////////////////////
//
// MuseData::getComposer --  The composer is not indicated in a MuseData file.
//    Infer it from the ID line if a file-location ID is present, since the
//    composers name is abbreviated in the directory name.
//

std::string MuseData::getComposer(void) {
	string id = getId();
	if (id.find("{cor/") != string::npos) {
		return "Corelli, Arcangelo";
	} else if (id.find("{beet/") != string::npos) {
		return "Beethoven, Ludwig van";
	}
	return "";
}



//////////////////////////////
//
// MuseData::getComposerDate --
//

std::string MuseData::getComposerDate(void) {
	string id = getId();
	if (id.find("{cor/") != string::npos) {
		return "1653/02/17-1713/01/08";
	} else if (id.find("{beet/") != string::npos) {
		return "1770/12/17-1827/03/26";
	}
	return "";
}



//////////////////////////////
//
// MuseData::getEncoderName --
//

std::string MuseData::getEncoderName(void) {
	string encoder = getEncoder();
	HumRegex hre;
	if (hre.search(encoder, "^\\s*(\\d+)/(\\d+)/(\\d+)\\s+(.*)\\s*$")) {
		return hre.getMatch(4);
	}
	return "";
}



//////////////////////////////
//
// MuseData::getEncoderDate --
//

std::string MuseData::getEncoderDate(void) {
	string encoder = getEncoder();
	HumRegex hre;
	if (hre.search(encoder, "^\\s*(\\d+)/(\\d+)/(\\d+)\\s+(.*)\\s*$")) {
		string month = hre.getMatch(1);
		string day   = hre.getMatch(2);
		string year  = hre.getMatch(3);
		if (year.size() == 2) {
			int value = stoi(year);
			if (value < 70) {
				value += 2000;
			} else {
				value += 1900;
			}
			year = to_string(value);
		}
		if (month.size() == 1) {
			month = "0" + month;
		}
		if (day.size() == 1) {
			day = "0" + day;
		}
		string value = year + "/" + month + "/" + day;
		return value;
	}
	return "";
}



//////////////////////////////
//
// MuseData::trimSpaces --
//

std::string MuseData::trimSpaces(std::string input) {
	string output;
	int status = 0;
	for (int i=0; i<(int)input.size(); i++) {
		if (!status) {
			if (isspace(input[i])) {
				continue;
			}
			status = 1;
		}
		output += input[i];
	}
	for (int i=(int)output.size()-1; i>=0; i--) {
		if (isspace(output[i])) {
			output.resize((int)output.size() - 1);
		} else {
			break;
		}
	}
	return output;
}



//////////////////////////////
//
// MuseData::analyzeLayers --  When there is a backup command in the
//   measure and no voice information, provide the voice information.
//   Primarily use the stem directions to make this determination.
//   Also, other features can be used:
//      beaming (do not split beamed notes across layers)
//      pitch (higher versus lower pitch).
//   Mostly this is only useful for two-voiced measures.
//   How to deal with voices on multiple staves will be more complex.
//

void MuseData::analyzeLayers(void) {
	int lcount = getLineCount();
	for (int i=0; i<lcount; i++) {
		i = analyzeLayersInMeasure(i);
	}
}



//////////////////////////////
//
// MuseData::analyzeLayersInMeasure -- returns the
//   index of the next barline.
//

int MuseData::analyzeLayersInMeasure(int startindex) {
	int i = startindex;
	int lcount = getLineCount();
	if (i >= lcount) {
		return lcount+1;
	}

	// Not necessarily a barline, but at least the first
	// record for the measure (may be missing at start
	// of music).

	while ((i < lcount) && isHeaderRecord(i)) {
		i++;
	}
	if ((i < lcount) && getRecord(i).isBarline()) {
		i++;
	}
	// Now should be at start of data for a measure.

	if (i >= lcount) {
		return lcount+1;
	}

	vector<vector<MuseRecord*>> segments(1);
	while (i < lcount) {
		MuseRecord *mr = &getRecord(i);
		if (mr->isBarline()) {
			break;
		}
		segments.back().push_back(mr);
		if (mr->isBackup()) {
			segments.resize(segments.size() + 1);
		}
		i++;
	}
	int position = i-1;

	if (segments.size() < 2) {
		// no backup in measure, so single voice/layer
		return position;
	}

	// Assign each backup segment to a successive track
	// if the layer does not have explicit track information.
	int track;

	for (int i=0; i<(int)segments.size(); i++) {
		for (int j=0; j<(int)segments[i].size(); j++) {
			MuseRecord* mr = segments[i][j];
			int trackfield = mr->getTrack();
			if (trackfield == 0) {
				track = i+1;
			} else {
				track = trackfield;
			}
			mr->setLayer(track);
		}
	}


	return position;
}



//////////////////////////////
//
// assignHeaderBodyState --
//

void MuseData::assignHeaderBodyState(void) {
	int state = 1;
	int foundend = 0;
	for (int i=0; i<(int)m_data.size(); i++) {
		if (m_data[i]->isAnyComment()) {
			// Comments inherit state if previous non-comment line
			m_data[i]->setHeaderState(state);
			continue;
		}
		if (state == 0) {
			// no longer in the header
			m_data[i]->setHeaderState(state);
			continue;
		}
		if ((!foundend) && m_data[i]->isGroup()) {
			foundend = 1;
			m_data[i]->setHeaderState(state);
			continue;
		}
		if (foundend && !m_data[i]->isGroup()) {
			state = 0;
			m_data[i]->setHeaderState(state);
			continue;
		}
		// still in header
		m_data[i]->setHeaderState(state);
	}
}



///////////////////////////////////////////////////////////////////////////
//
// friendly functions
//


//////////////////////////////
//
// operator<< --
//

ostream& operator<<(ostream& out, MuseData& musedata) {
	for (int i=0; i<musedata.getLineCount(); i++) {
		if (musedata[i].getType() != E_muserec_deleted) {
			out << musedata[i].getLine() << (char)0x0d << (char)0x0a;
		}
	}
	return out;
}




///////////////////////////////////////////////////////////////////////////
//
// MuseDataSet class functions --
//


//////////////////////////////
//
// MuseDataSet::MuseDataSet --
//

MuseDataSet::MuseDataSet (void) {
	m_part.reserve(100);
}



//////////////////////////////
//
// MuseDataSet::clear -- Remove contents of object.
//

void MuseDataSet::clear(void) {
	int i;
	for (i=0; i<(int)m_part.size(); i++) {
		delete m_part[i];
	}

}



//////////////////////////////
//
// MuseDataSet::operator[] --
//

MuseData& MuseDataSet::operator[](int index) {
	return *m_part[index];
}



//////////////////////////////
//
// MuseDataSet::readPart -- read a single MuseData part, appending it
//      to the current list of parts.
//

int MuseDataSet::readPartFile(const string& filename) {
	MuseData* md = new MuseData;
	md->readFile(filename);
	md->setFilename(filename);
	return appendPart(md);
}

int MuseDataSet::readPartString(const string& data) {
	stringstream ss;
	ss << data;
	return readPart(ss);
}


int MuseDataSet::readPart(istream& input) {
	MuseData* md = new MuseData;
	md->read(input);
	return appendPart(md);
}



//////////////////////////////
//
// MuseDataSet::read -- read potentially Multiple parts from a single file.
//   First clear the contents of any previous part data.
//

int MuseDataSet::readFile(const string& filename) {
	MuseDataSet::clear();
	ifstream infile(filename);
	return MuseDataSet::read(infile);
}

int MuseDataSet::readString(const string& data) {
	stringstream ss;
	ss << data;
	return MuseDataSet::read(ss);
}


int MuseDataSet::read(istream& infile) {
	vector<string> datalines;
	datalines.reserve(100000);
	string thing;

	while (!infile.eof()) {
		getline(infile, thing);
		if (infile.eof() && (thing.length() == 0)) {
			// last line was not terminated by a newline character
			break;
		}
		datalines.push_back(thing);
	}

	vector<int> startindex;
	vector<int> stopindex;
	analyzePartSegments(startindex, stopindex, datalines);

	stringstream *sstream;
	MuseData* md;
	for (int i=0; i<(int)startindex.size(); i++) {
		sstream = new stringstream;
		for (int j=startindex[i]; j<=stopindex[i]; j++) {
			 (*sstream) << datalines[j] << '\n';
		}
		md = new MuseData;
		md->read(*sstream);
		appendPart(md);
		delete sstream;
	}
	return 1;
}



//////////////////////////////
//
// MuseDataSet::appendPart -- append a MuseData pointer to the end of the
//   parts list and return the index number of the new part.
//

int MuseDataSet::appendPart(MuseData* musedata) {
	int index = (int)m_part.size();
	m_part.resize(m_part.size()+1);
	m_part[index] = musedata;
	return index;
}



//////////////////////////////
//
// MuseData::analyzePartSegments -- Calculate the starting line index
//    and the ending line index for each part in the input.
//

void MuseDataSet::analyzePartSegments(vector<int>& startindex,
		vector<int>& stopindex, vector<string>& lines) {

	startindex.clear();
	stopindex.clear();
	startindex.reserve(1000);
	stopindex.reserve(1000);

	vector<int> types;
	// MuseData& thing = *this;

	types.resize(lines.size());
	std::fill(types.begin(), types.end(), E_muserec_unknown);

	// first identify lines which are multi-line comments so they will
	// not cause confusion in the next step
	int commentstate = 0;
	for (int i=0; i<(int)lines.size(); i++) {
		if (lines[i].c_str()[0] == '&') {
			types[i] = E_muserec_comment_toggle;
			commentstate = !commentstate;
			continue;
		}
		if (commentstate) {
			types[i] = E_muserec_comment_line;
		}
	}

	// search the data for "Group memberships:" lines which are required
	// to be in the header of each part.
	vector<int> groupmemberships;
	groupmemberships.reserve(1000);
	int len = strlen("Group memberships:");
	for (int i=0; i<(int)lines.size(); i++) {
		if (strncmp("Group memberships:", lines[i].c_str(), len) == 0) {
			if (types[i] != E_muserec_comment_line) {
				groupmemberships.push_back(i);
			}
		}
	}

	// search backwards from "Group memberships:" until the end of the
	// header, sucking any comments which occurs just before the start
	// of the header. (currently only multi-line comments, but also need
	// to add single-line comments)
	int value;
	int headerline;
	int found = 0;
	for (int ii=0; ii<(int)groupmemberships.size(); ii++) {
		int i = groupmemberships[ii];
		types[i] = E_muserec_group_memberships;
		found = 0;
		headerline = 11;
		for (int j=i-1; j>=0; j--) {
			if (j < 0) {
				break;
			}
			if (lines[j].compare(0, 4, "/eof") == 0) {
				// end of previous file
				found = 1;
				value = j + 1;
				startindex.push_back(value);
				break;
			}
			if ((types[j] == E_muserec_comment_line) ||
				 (types[j] == E_muserec_comment_toggle)) {
//				j--;
				continue;
			}
			if (j < 0) {
				break;
			}
			headerline--;

			if (headerline == 0) {
				while ((j>= 0) && (lines[j][0] == '@')) {
					j--;
				}
				value = j+1;
				//value = j+2;
				found = 1;
				startindex.push_back(value);
				break;
			}

			if ((j >= 0) && (headerline == 0)) {
				value = j+1;
				found = 1;
				startindex.push_back(value);
				break;
			}
			if (j<0) {
				value = 0;
				found = 1;
				startindex.push_back(value);
				continue;
			}
			switch (headerline) {
				case 11: types[j] = E_muserec_header_11; break;
				case 10: types[j] = E_muserec_header_10; break;
				case  9: types[j] = E_muserec_header_9; break;
				case  8: types[j] = E_muserec_header_8; break;
				case  7: types[j] = E_muserec_header_7; break;
				case  6: types[j] = E_muserec_header_6; break;
				case  5: types[j] = E_muserec_header_5; break;
				case  4: types[j] = E_muserec_header_4; break;
				case  3: types[j] = E_muserec_header_3; break;
				case  2: types[j] = E_muserec_header_2; break;
				case  1: types[j] = E_muserec_header_1; break;
			}
		}
		if (!found) {
			value = 0;
			startindex.push_back(value);
		}
	}

	// now calculate the stopindexes:
	stopindex.resize(startindex.size());
	stopindex[(int)stopindex.size()-1] = (int)lines.size()-1;
	for (int i=0; i<(int)startindex.size()-1; i++) {
		stopindex[i] = startindex[i+1]-1;
	}
}



//////////////////////////////
//
// MuseDataSet::getPartCount -- return the number of parts found
//      in the MuseDataSet
//

int MuseDataSet::getPartCount(void) {
	return (int)m_part.size();
}



//////////////////////////////
//
// MuseDataSet::deletePart -- remove a particular part from the data set.
//

void MuseDataSet::deletePart(int index) {
	if (index < 0 || index > (int)m_part.size()-1) {
		cerr << "Trying to delete a non-existent part" << endl;
		return;
	}

	delete m_part[index];
	int i;
	for (i=index+1; i<(int)m_part.size(); i++) {
		m_part[i-1] = m_part[i];
	}
	m_part.resize(m_part.size()-1);
}



//////////////////////////////
//
// MuseDataSet::cleanLineEndings -- remove spaces for ends of lines.
//

void MuseDataSet::cleanLineEndings(void) {
	for (int i=0; i<(int)m_part.size(); i++) {
		m_part[i]->cleanLineEndings();
	}
}



//////////////////////////////
//
// MuseDataSet::clearError --
//

void MuseDataSet::clearError(void) {
	m_error = "";
}



//////////////////////////////
//
// MuseDataSet::hasError --
//

bool MuseDataSet::hasError(void) {
	return !m_error.empty();
}



//////////////////////////////
//
// MuseDataSet::setError --
//

void MuseDataSet::setError(const string& error) {
	m_error = error;
}



//////////////////////////////
//
// MuseDataSet::getError --
//

string MuseDataSet::getError(void) {
	return m_error;
}



///////////////////////////////////////////////////////////////////////////

//////////////////////////////
//
// operator<< -- print out all parts in sequential order
//

ostream& operator<<(ostream& out, MuseDataSet& musedataset) {
	for (int i=0; i<musedataset.getPartCount(); i++) {
		for (int j=0; j<musedataset[i].getNumLines(); j++) {
			out << musedataset[i][j] << '\n';
		}
	}
	return out;
}




#define E_unknown   (0x7fff)

//////////////////////////////
//
// MuseRecord::MuseRecord --
//

MuseRecord::MuseRecord(void) : MuseRecordBasic() { }
MuseRecord::MuseRecord(const string& aLine) : MuseRecordBasic(aLine) { }
MuseRecord::MuseRecord(MuseRecord& aRecord) : MuseRecordBasic(aRecord) { }



//////////////////////////////
//
// MuseRecord::~MuseRecord --
//

MuseRecord::~MuseRecord() {
	// do nothing
}



//////////////////////////////
//
// MuseRecord::operator= -- 
//

MuseRecord& MuseRecord::operator=(MuseRecord& aRecord) {
	// don't copy onto self
	if (&aRecord == this) {
		return *this;
	}

	setLine(aRecord.getLine());
	setType(aRecord.getType());
	m_lineindex = aRecord.m_lineindex;

	m_absbeat = aRecord.m_absbeat;
	m_lineduration = aRecord.m_lineduration;
	m_noteduration = aRecord.m_noteduration;

	m_b40pitch     = aRecord.m_b40pitch;
	m_nexttiednote = aRecord.m_nexttiednote;
	m_lasttiednote = aRecord.m_lasttiednote;

	return *this;
}



//////////////////////////////////////////////////////////////////////////
//
// functions that work with note records
//


//////////////////////////////
//
// MuseRecord::getNoteField -- returns the string containing the pitch,
//	accidental and octave characters.
//

string MuseRecord::getNoteField(void) {
	switch (getType()) {
		case E_muserec_note_regular:
			return extract(1, 4);
			break;
		case E_muserec_note_chord:
		case E_muserec_note_cue:
		case E_muserec_note_grace:
			return extract(2, 5);
			break;
		default:
			cerr << "Error: cannot use getNoteField function on line: "
			   << getLine() << endl;
	}
	return "";
}




//////////////////////////////
//
// MuseRecord::getOctave -- returns the first numeric character
//	in the note field of a MuseData note record
//

int MuseRecord::getOctave(void) {
	string recordInfo = getNoteField();
	int index = 0;
	while ((index < (int)recordInfo.size()) && !std::isdigit(recordInfo[index])) {
		index++;
	}
	if (index >= (int)recordInfo.size()) {
		cerr << "Error: no octave specification in note field: " << recordInfo
			  << endl;
		return 0;
	}
	return recordInfo[index] - '0';
}


string MuseRecord::getOctaveString(void) {
	string recordInfo = getNoteField();
	int index = 0;
	while ((index < (int)recordInfo.size()) && !std::isdigit(recordInfo[index])) {
		index++;
	}
	if (index >= (int)recordInfo.size()) {
		cerr << "Error: no octave specification in note field: " << recordInfo
			  << endl;
		return "";
	}
	string output;
	output += recordInfo[index];
	return output;
}



//////////////////////////////
//
// MuseRecord::getPitch -- int version returns the base40 representation
//

int MuseRecord::getPitch(void) {
	string recordInfo = getNoteField();
	return Convert::museToBase40(recordInfo);
}


string MuseRecord::getPitchString(void) {
	string output = getNoteField();
	int len = (int)output.size();
	int index = len-1;
	while (index >= 0 && output[index] == ' ') {
		output.resize(index);
		index--;
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getPitchClass -- returns the pitch without the octave information
//

int MuseRecord::getPitchClass(void) {
	return getPitch() % 40;
}


string MuseRecord::getPitchClassString(void) {
	string output = getNoteField();
	int index = 0;
	while ((index < (int)output.size()) &&  !std::isdigit(output[index])) {
		index++;
	}
	output.resize(index);
	return output;
}



//////////////////////////////
//
// MuseRecord::getAccidental -- int version return -2 for double flat,
//	-1 for flat, 0 for natural, +1 for sharp, +2 for double sharp
//

int MuseRecord::getAccidental(void) {
	string recordInfo = getNoteField();
	int output = 0;
	int index = 0;
	while ((index < (int)recordInfo.size()) && (index < 16)) {
		if (recordInfo[index] == 'f') {
			output--;
		} else if (recordInfo[index] == '#') {
			output++;
		}
		index++;
	}
	return output;
}


string MuseRecord::getAccidentalString(void) {
	string output;
	int type = getAccidental();
	switch (type) {
		case -2: output = "ff"; break;
		case -1: output =  "f"; break;
		case  0: output =   ""; break;
		case  1: output =  "#"; break;
		case  2: output = "##"; break;
		default:
			output = getNoteField();
			cerr << "Error: unknown type of accidental: " << output << endl;
			return "";
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getBase40 -- return the base40 pitch value of the data
// line.  Middle C set to 40 * 4 + 2;  Returns -100 for non-pitched items.
// (might have to update for note_cur_chord and note_grace_chord which
// do not exist yet.
//

int MuseRecord::getBase40(void) {
	switch (getType()) {
		case E_muserec_note_regular:
		case E_muserec_note_chord:
		case E_muserec_note_cue:
		case E_muserec_note_grace:
			break;
		default:
			return -100;
	}
	return getPitch();
}



//////////////////////////////
//
// MuseRecord::setStemDown --
//

void MuseRecord::setStemDown(void) {
	getColumn(23) = 'd';
}



//////////////////////////////
//
// MuseRecord::setStemUp --
//

void MuseRecord::setStemUp(void) {
	getColumn(23) = 'u';
}



//////////////////////////////
//
// MuseRecord::setPitch -- input is a base40 value which gets converted
// to a diatonic pitch name.
//    Default value: chordnote = 0
//    Default value: gracenote = 0
//

void MuseRecord::setPitch(int base40, int chordnote, int gracenote) {
	string diatonic;
	switch (Convert::base40ToDiatonic(base40) % 7) {
		case 0:  diatonic = 'C'; break;
		case 1:  diatonic = 'D'; break;
		case 2:  diatonic = 'E'; break;
		case 3:  diatonic = 'F'; break;
		case 4:  diatonic = 'G'; break;
		case 5:  diatonic = 'A'; break;
		case 6:  diatonic = 'B'; break;
		default: diatonic = 'X';
	}

	string octave;
	octave  += char('0' + base40 / 40);

	string accidental;
	int acc = Convert::base40ToAccidental(base40);
	switch (acc) {
		case -2:   accidental = "ff"; break;
		case -1:   accidental = "f";  break;
		case +1:   accidental = "#";  break;
		case +2:   accidental = "##"; break;
	}
	string pitchname = diatonic + accidental + octave;

	if (chordnote) {
		if (gracenote) {
			setGraceChordPitch(pitchname);
		} else {
			setChordPitch(pitchname);
		}
	} else {
		setPitch(pitchname);
	}
}


void MuseRecord::setChordPitch(const string& pitchname) {
	getColumn(1) = ' ';
	setPitchAtIndex(1, pitchname);
}

void MuseRecord::setGracePitch(const string& pitchname) {
	getColumn(1) = 'g';
	setPitchAtIndex(1, pitchname);
}

void MuseRecord::setGraceChordPitch(const string& pitchname) {
	getColumn(1) = 'g';
	getColumn(2) = ' ';
	setPitchAtIndex(2, pitchname);
}

void MuseRecord::setCuePitch(const string& pitchname) {
	getColumn(1) = 'c';
	setPitchAtIndex(1, pitchname);
}


void MuseRecord::setPitch(const string& pitchname) {
	int start = 0;
	// If the record is already set to a grace note or a cue note,
	// then place pitch information starting at column 2 (index 1).
	if ((getColumn(1) == 'g') || (getColumn(1) == 'c')) {
		start = 1;
	}
	setPitchAtIndex(start, pitchname);
}


void MuseRecord::setPitchAtIndex(int index, const string& pitchname) {
	int len = (int)pitchname.size();
	if ((len > 4) && (pitchname != "irest")) {
		cerr << "Error in MuseRecord::setPitchAtIndex: " << pitchname << endl;
		return;
	}
	insertString(index+1, pitchname);

	// Clear any text fields not used by current pitch data.
	for (int i=4-len-1; i>=0; i--) {
		(*this)[index + len + i] = ' ';
	}
}



//////////////////////////////
//
// MuseRecord::getTickDurationField -- returns the string containing the
//      duration, and tie information.
//

string MuseRecord::getTickDurationField(void) {
	switch (getType()) {
		case E_muserec_figured_harmony:
		case E_muserec_note_regular:
		case E_muserec_note_chord:
		case E_muserec_rest:
		case E_muserec_backward:
		case E_muserec_forward:
			return extract(6, 9);
			break;
		// these record types do not have duration, per se:
		case E_muserec_note_cue:
		case E_muserec_note_grace:
		default:
			return "    ";
			// cerr << "Error: cannot use getTickDurationField function on line: "
			//      << getLine() << endl;
			// return "";
	}
	return "";
}



//////////////////////////////
//
// MuseRecord::getTickDurationString -- returns the string containing the duration,
//

string MuseRecord::getTickDurationString(void) {
	string output = getTickDurationField();
	int length = (int)output.size();
	int i = length - 1;
	while (i>0 && (output[i] == '-' || output[i] == ' ')) {
		output.resize(i);
		i--;
		length--;
	}

	int start = 0;
	while (output[start] == ' ') {
		start++;
	}

	if (start != 0) {
		for (i=0; i<length-start; i++) {
			output[i] = output[start+i];
		}
	}
	output.resize(length-start);

	return output;
}



//////////////////////////////
//
// MuseRecord::getTickDuration -- return the tick value found
//    in columns 6-8 in some data type, returning 0 if the record
//    type does not have a duration field.
//

int MuseRecord::getTickDuration(void) {
	string recordInfo = getTickDurationString();
	if (recordInfo.empty()) {
		return 0;
	}
	return std::stoi(recordInfo);
}



//////////////////////////////
//
// MuseRecord::getLineTickDuration -- returns the logical duration of the
//      data line.  Supresses the duration field of secondary chord notes.
//

int MuseRecord::getLineTickDuration(void) {
	if (getType() == E_muserec_note_chord) {
		return 0;
	}

	string recordInfo = getTickDurationString();
	if (recordInfo.empty()) {
		return 0;
	}
	int value = std::stoi(recordInfo);
	if (getType() == E_muserec_backspace) {
		return -value;
	}

	return value;
}



//////////////////////////////
//
// MuseRecord::getTicks -- similar to getLineTickDuration, but is non-zero
//    for secondary chord notes.
//

int MuseRecord::getTicks(void) {
	string recordInfo = getTickDurationString();
	if (recordInfo.empty()) {
		return 0;
	}
	int value = std::stoi(recordInfo);
	if (getType() == E_muserec_backspace) {
		return -value;
	}

	return value;
}


//////////////////////////////
//
// MuseRecord::getNoteTickDuration -- Similar to getLineTickDuration,
//    but do not suppress the duration of secondary chord-tones.
//

int MuseRecord::getNoteTickDuration(void) {
	string recordInfo = getTickDurationString();
	int value = 0;
	if (recordInfo.empty()) {
		return value;
	}
	value = std::stoi(recordInfo);
	if (getType() == E_muserec_backspace) {
		return -value;
	}
	return value;
}



//////////////////////////////
//
// MuseRecord::setDots --
//

void MuseRecord::setDots(int value) {
	switch (value) {
		case 0: getColumn(18) = ' ';   break;
		case 1: getColumn(18) = '.';   break;
		case 2: getColumn(18) = ':';   break;
		case 3: getColumn(18) = ';';   break;
		case 4: getColumn(18) = '!';   break;
		default: cerr << "Error in MuseRecord::setDots : " << value << endl;
	}
}



//////////////////////////////
//
// MuseRecord::getDotCount --
//

int MuseRecord::getDotCount(void) {
	char value = getColumn(18);
	switch (value) {
		case ' ': return 0;
		case '.': return 1;
		case ':': return 2;
		case ';': return 3;
		case '!': return 4;
	}
	return 0;
}



//////////////////////////////
//
// MuseRecord::setNoteheadShape -- Duration with augmentation dot component
//      removed.  Duration of 1 is quarter note.
//

void MuseRecord::setNoteheadShape(HumNum duration) {
	HumNum  note8th(1,2);
	HumNum  note16th(1,4);
	HumNum  note32nd(1,8);
	HumNum  note64th(1,16);
	HumNum  note128th(1,32);
	HumNum  note256th(1,64);

	if (duration > 16) {                 // maxima
		setNoteheadMaxima();
	} else if (duration > 8) {           // long
		setNoteheadLong();
	} else if (duration > 4) {           // breve
		if (m_roundBreve) {
			setNoteheadBreveRound();
		} else {
			setNoteheadBreve();
		}
	} else if (duration > 2) {           // whole note
		setNoteheadWhole();
	} else if (duration > 1) {           // half note
		setNoteheadHalf();
	} else if (duration > note8th) {     // quarter note
		setNoteheadQuarter();
	} else if (duration > note16th) {    // eighth note
		setNotehead8th();
	} else if (duration > note32nd) {    // 16th note
		setNotehead16th();
	} else if (duration > note64th) {    // 32nd note
		setNotehead32nd();
	} else if (duration > note128th) {   // 64th note
		setNotehead64th();
	} else if (duration > note256th) {   // 128th note
		setNotehead128th();
	} else if (duration == note256th) {  // 256th note
		// not allowing tuplets on the 256th note level.
		setNotehead256th();
	} else {
		cerr << "Error in duration: " << duration << endl;
		return;
	}
}



//////////////////////////////
//
// MuseRecord::setNoteheadShape -- Duration with augmentation dot component
//      removed.  Duration of 1 is quarter note.
//

void MuseRecord::setNoteheadShapeMensural(HumNum duration) {
	HumNum note8th(1, 2);
	HumNum note16th(1, 4);
	HumNum note32th(1, 8);
	HumNum note64th(1, 16);
	HumNum note128th(1, 32);
	HumNum note256th(1, 64);

	if (duration > 16) {                 // maxima
		setNoteheadMaxima();
	} else if (duration > 8) {           // long
		setNoteheadLong();
	} else if (duration > 4) {           // breve
		setNoteheadBreve();
	} else if (duration > 2) {           // whole note
		setNoteheadWholeMensural();
	} else if (duration > 1) {           // half note
		setNoteheadHalfMensural();
	} else if (duration > note8th) {     // quarter note
		setNoteheadQuarterMensural();
	} else if (duration > note16th) {    // eighth note
		setNotehead8thMensural();
	} else if (duration > note32th) {    // 16th note
		setNotehead16thMensural();
	} else if (duration > note64th) {    // 32nd note
		setNotehead32ndMensural();
	} else if (duration > note128th) {   // 64th note
		setNotehead64thMensural();
	} else if (duration > note256th) {   // 128th note
		setNotehead128thMensural();
	} else if (duration >= note256th) {  // 256th note
		// don't allow tuplets on 256th note level.
		setNotehead256thMensural();
	} else {
		cerr << "Error in duration: " << duration << endl;
		return;
	}
}

void MuseRecord::setNoteheadMaxima(void) {
	if ((*this)[0] == 'c' || ((*this)[0] == 'g')) {
		cerr << "Error: cue/grace notes cannot be maximas in setNoteheadLong"
			  << endl;
		return;
	} else {
		getColumn(17) = 'M';
	}
}

void MuseRecord::setNoteheadLong(void) {
	if ((*this)[0] == 'c' || ((*this)[0] == 'g')) {
		cerr << "Error: cue/grace notes cannot be longs in setNoteheadLong"
			  << endl;
		return;
	} else {
		getColumn(17) = 'L';
	}
}

void MuseRecord::setNoteheadBreve(void) {
	setNoteheadBreveSquare();
}

void MuseRecord::setNoteheadBreveSquare(void) {
	if ((*this)[0] == 'g') {        // grace note
		getColumn(8) = 'A';
	} else if ((*this)[0] == 'c') { // cue-sized note (with duration)
		getColumn(17) = 'A';
	} else {                        // normal note
		getColumn(17) = 'B';
	}
}

void MuseRecord::setNoteheadBreveRound(void) {
	if ((*this)[0] == 'g') {        // grace note
		getColumn(8) = 'A';
	} else if ((*this)[0] == 'c') { // cue-sized note (with duration)
		getColumn(17) = 'A';
	} else {                        // normal note
		getColumn(17) = 'b';
	}
}

void MuseRecord::setNoteheadBreveMensural(void) {
	setNoteheadBreveSquare();
}

void MuseRecord::setNoteheadWhole(void) {
	if ((*this)[0] == 'g') {        // grace note
		getColumn(8) = '9';
	} else if ((*this)[0] == 'c') { // cue-sized note (with duration)
		getColumn(17) = '9';
	} else {                        // normal note
		getColumn(17) = 'w';
	}
}

void MuseRecord::setNoteheadWholeMensural(void) {
	if ((*this)[0] == 'g') {        // grace note
		getColumn(8) = '9';
	} else if ((*this)[0] == 'c') { // cue-sized note (with duration)
		getColumn(17) = '9';
	} else {                        // normal note
		getColumn(17) = 'W';
	}
}

void MuseRecord::setNoteheadHalf(void) {
	if ((*this)[0] == 'g') {        // grace note
		getColumn(8) = '8';
	} else if ((*this)[0] == 'c') { // cue-sized note (with duration)
		getColumn(17) = '8';
	} else {                        // normal note
		getColumn(17) = 'h';
	}
}

void MuseRecord::setNoteheadHalfMensural(void) {
	if ((*this)[0] == 'g') {        // grace note
		getColumn(8) = '8';
	} else if ((*this)[0] == 'c') { // cue-sized note (with duration)
		getColumn(17) = '8';
	} else {                        // normal note
		getColumn(17) = 'H';
	}
}

void MuseRecord::setNoteheadQuarter(void) {
	if ((*this)[0] == 'g') {        // grace note
		getColumn(8) = '7';
	} else if ((*this)[0] == 'c') { // cue-sized note (with duration)
		getColumn(17) = '7';
	} else {                        // normal note
		getColumn(17) = 'q';
	}
}

void MuseRecord::setNoteheadQuarterMensural(void) {
	if ((*this)[0] == 'g') {        // grace note
		getColumn(8) = '7';
	} else if ((*this)[0] == 'c') { // cue-sized note (with duration)
		getColumn(17) = '7';
	} else {                        // normal note
		getColumn(17) = 'Q';
	}
}

void MuseRecord::setNotehead8th(void) {
	if ((*this)[0] == 'g') {        // grace note
		getColumn(8) = '6';
	} else if ((*this)[0] == 'c') { // cue-sized note (with duration)
		getColumn(17) = '6';
	} else {                        // normal note
		getColumn(17) = 'e';
	}
}

void MuseRecord::setNotehead8thMensural(void) {
	if ((*this)[0] == 'g') {        // grace note
		getColumn(8) = '6';
	} else if ((*this)[0] == 'c') { // cue-sized note (with duration)
		getColumn(17) = '6';
	} else {                        // normal note
		getColumn(17) = 'E';
	}
}

void MuseRecord::setNotehead16th(void) {
	if ((*this)[0] == 'g') {        // grace note
		getColumn(8) = '5';
	} else if ((*this)[0] == 'c') { // cue-sized note (with duration)
		getColumn(17) = '5';
	} else {                        // normal note
		getColumn(17) = 's';
	}
}

void MuseRecord::setNotehead16thMensural(void) {
	if ((*this)[0] == 'g') {        // grace note
		getColumn(8) = '5';
	} else if ((*this)[0] == 'c') { // cue-sized note (with duration)
		getColumn(17) = '5';
	} else {                        // normal note
		getColumn(17) = 'S';
	}
}

void MuseRecord::setNotehead32nd(void) {
	if ((*this)[0] == 'g') {        // grace note
		getColumn(8) = '4';
	} else if ((*this)[0] == 'c') { // cue-sized note (with duration)
		getColumn(17) = '4';
	} else {                        // normal note
		getColumn(17) = 't';
	}
}

void MuseRecord::setNotehead32ndMensural(void) {
	if ((*this)[0] == 'g') {        // grace note
		getColumn(8) = '4';
	} else if ((*this)[0] == 'c') { // cue-sized note (with duration)
		getColumn(17) = '4';
	} else {                        // normal note
		getColumn(17) = 'T';
	}
}

void MuseRecord::setNotehead64th(void) {
	if ((*this)[0] == 'g') {        // grace note
		getColumn(8) = '3';
	} else if ((*this)[0] == 'c') { // cue-sized note (with duration)
		getColumn(17) = '3';
	} else {                        // normal note
		getColumn(17) = 'x';
	}
}

void MuseRecord::setNotehead64thMensural(void) {
	if ((*this)[0] == 'g') {        // grace note
		getColumn(8) = '3';
	} else if ((*this)[0] == 'c') { // cue-sized note (with duration)
		getColumn(17) = '3';
	} else {                        // normal note
		getColumn(17) = 'X';
	}
}

void MuseRecord::setNotehead128th(void) {
	if ((*this)[0] == 'g') {        // grace note
		getColumn(8) = '2';
	} else if ((*this)[0] == 'c') { // cue-sized note (with duration)
		getColumn(17) = '2';
	} else {                        // normal note
		getColumn(17) = 'y';
	}
}

void MuseRecord::setNotehead128thMensural(void) {
	if ((*this)[0] == 'g') {        // grace note
		getColumn(8) = '2';
	} else if ((*this)[0] == 'c') { // cue-sized note (with duration)
		getColumn(17) = '2';
	} else {                        // normal note
		getColumn(17) = 'Y';
	}
}

void MuseRecord::setNotehead256th(void) {
	if ((*this)[0] == 'g') {        // grace note
		getColumn(8) = '1';
	} else if ((*this)[0] == 'c') { // cue-sized note (with duration)
		getColumn(17) = '1';
	} else {                        // normal note
		getColumn(17) = 'z';
	}
}

void MuseRecord::setNotehead256thMensural(void) {
	if ((*this)[0] == 'g') {        // grace note
		getColumn(8) = '1';
	} else if ((*this)[0] == 'c') { // cue-sized note (with duration)
		getColumn(17) = '1';
	} else {                        // normal note
		getColumn(17) = 'Z';
	}
}


/////////////////////////////
//
// MuseRecord::setBack --
//

void MuseRecord::setBack(int value) {
	insertString(1, "back");
	setTicks(value);
}



/////////////////////////////
//
// MuseRecord::setTicks -- return the numeric value in columns 6-9.
//

void MuseRecord::setTicks(int value) {
	if ((value < 0) || (value >= 1000)) {
		cerr << "@ Error: ticks out of range in MuseRecord::setTicks" << endl;
	}
	stringstream ss;
	ss << value;
	int len = (int)ss.str().size();
	insertString(5+3-len+1, ss.str());
}



//////////////////////////////
//
// MuseRecord::getTie --
//

string MuseRecord::getTieString(void) {
	string output;
	output += getColumn(9);
	if (output == " ") {
		output = "";
	}
	return output;
}


int MuseRecord::getTie(void) {
	return tieQ();
}


//////////////////////////////
//
// MuseRecord::getTie -- Set a tie marker in column 9.  Currently
// the function does not check the type of data, so will overr-write any
// data found in column 9 (such as if the record is not for a note).
//
// If the input parameter hidden is true, then the visual tie is not
// displayed, but the sounding tie is displayed.
//

int MuseRecord::setTie(int hidden) {
	getColumn(9) = '-';
	if (!hidden) {
		return addAdditionalNotation('-');
	} else {
		return -1;
	}
}



//////////////////////////////
//
// MuseRecord::addAdditionalNotation -- ties, slurs and tuplets.
//    Currently not handling editorial levels.
//

int MuseRecord::addAdditionalNotation(char symbol) {
	// search columns 32 to 43 for the specific symbol.
	// if it is found, then don't add.  If it is not found,
	// then do add.
	int i;
	int blank = -1;
	int nonempty = 0;  // true if a non-space character was found.

	for (i=43; i>=32; i--) {
		if (getColumn(i) == symbol) {
			return i;
		} else if (!nonempty && (getColumn(i) == ' ')) {
			blank = i;
		} else {
			nonempty = i;
		}
	}

	if (symbol == '-') {
	  // give preferential treatment to placing only ties in
	  // column 32
	  if (getColumn(32) == ' ') {
		  getColumn(32) = '-';
		  return 32;
	  }
	}

	if (blank < 0) {
		cerr << "Error in MuseRecord::addAdditionalNotation: "
			  << "no empty space for notation" << endl;
		return 0;
	}

	if ((blank <= 32) && (getColumn(33) == ' ')) {
		// avoid putting non-tie items in column 32.
		blank = 33;
	}

	getColumn(blank) = symbol;
	return blank;
}


// add a multi-character additional notation (such as a dynamic like mf):

int MuseRecord::addAdditionalNotation(const string& symbol) {
	int len = (int)symbol.size();
	// search columns 32 to 43 for the specific symbol.
	// if it is found, then don't add.  If it is not found,
	// then do add.
	int i, j;
	int blank = -1;
	int found = 0;
	int nonempty = 0;  // true if a non-space character was found.

	for (i=43-len; i>=32; i--) {
		found = 1;
		for (j=0; j<len; j++) {
			if (getColumn(i+j) != symbol[j]) {
				found = 0;
				break;
			}
		}
		if (found) {
			return i;
		} else if (!nonempty && (getColumn(i) == ' ')) {
// cout << "@COLUMN " << i << " is blank: " << getColumn(i) << endl;
			blank = i;
			// should check that there are enough blank lines to the right
			// as well...
		} else if (getColumn(i) != ' ') {
			nonempty = i;
		}
	}

	if (blank < 0) {
		cerr << "Error in MuseRecord::addAdditionalNotation2: "
			  << "no empty space for notation" << endl;
		return 0;
	}

// cout << "@ GOT HERE symbol = " << symbol << " and blank = " << blank << endl;
	if ((blank <= 32) && (getColumn(33) == ' ')) {
		// avoid putting non-tie items in column 32.
		blank = 33;
		// not worrying about overwriting something to the right
		// of column 33 since the empty spot was checked starting
		// on the right and moving towards the left.
	}
// cout << "@COLUMN 33 = " << getColumn(33) << endl;
// cout << "@ GOT HERE symbol = " << symbol << " and blank = " << blank << endl;

	for (j=0; j<len; j++) {
		getColumn(blank+j) = symbol[j];
	}
	return blank;
}



//////////////////////////////
//
// MuseRecord::tieQ -- returns true if the current line contains
//   a tie to a note in the future.  Does not check if there is a tie
//   to a note in the past.
//

int MuseRecord::tieQ(void) {
	int output = 0;
	switch (getType()) {
		case E_muserec_note_regular:
		case E_muserec_note_chord:
		case E_muserec_note_cue:
		case E_muserec_note_grace:
			if (getColumn(9) == '-') {
				output = 1;
			} else if (getColumn(9) == ' ') {
				output = 0;
			} else {
				output = -1;
			}
			break;
		default:
			return 0;
	}

	return output;
}


//////////////////////////////////////////////////////////////////////////
//
// graphical and intrepretive information for notes
//

//////////////////////////////
//
// MuseRecord::getFootnoteFlagField -- returns column 13 value
//

string MuseRecord::getFootnoteFlagField(void) {
	allowFigurationAndNotesOnly("getFootnoteField");
	return extract(13, 13);
}



//////////////////////////////
//
// MuseRecord::getFootnoteFlagString --
//

string MuseRecord::getFootnoteFlagString(void) {
	string output = getFootnoteFlagField();
	if (output[0] == ' ') {
		output = "";
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getFootnoteFlag --
//

int MuseRecord::getFootnoteFlag(void) {
	int output = 0;
	string recordInfo = getFootnoteFlagString();
	if (recordInfo[0] == ' ') {
		output = -1;
	} else {
		output = std::strtol(recordInfo.c_str(), NULL, 36);
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::footnoteFlagQ --
//

int MuseRecord::footnoteFlagQ(void) {
	int output = 0;
	string recordInfo = getFootnoteFlagField();
	if (recordInfo[0] == ' ') {
		output = 0;
	} else {
		output = 1;
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getLevelField -- return column 14
//

string MuseRecord::getLevelField(void) {
	allowFigurationAndNotesOnly("getLevelField");
	return extract(14, 14);
}



//////////////////////////////
//
// MuseRecord::getLevel --
//

string MuseRecord::getLevelString(void) {
	string output = getLevelField();
	if (output[0] == ' ') {
		output = "";
	}
	return output;
}


int MuseRecord::getLevel(void) {
	int output = 1;
	string recordInfo = getLevelField();
	if (recordInfo[0] == ' ') {
		output = 1;
	} else {
		output = std::strtol(recordInfo.c_str(), NULL, 36);
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::levelQ --
//

int MuseRecord::levelQ(void) {
	int output = 0;
	string recordInfo = getLevelField();
	if (recordInfo[0] == ' ') {
		output = 0;
	} else {
		output = 1;
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getTrackField -- return column 15
//

string MuseRecord::getTrackField(void) {
	if (!isAnyNoteOrRest()) {
		return extract(15, 15);
	} else {
		return " ";
	}
}



//////////////////////////////
//
// MuseRecord::getTrackString --
//

string MuseRecord::getTrackString(void) {
	string output = getTrackField();
	if (output[0] == ' ') {
		output = "";
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getTrack -- Return 0 if no track information (implicitly track 1,
//     or unlabelled higher track).
//

int MuseRecord::getTrack(void) {
	int output = 1;
	string recordInfo = getTrackField();
	if (recordInfo[0] == ' ') {
		output = 0;
	} else {
		output = std::strtol(recordInfo.c_str(), NULL, 36);
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::trackQ --
//

int MuseRecord::trackQ(void) {
	int output = 0;
	string recordInfo = getTrackField();
	if (recordInfo[0] == ' ') {
		output = 0;
	} else {
		output = 1;
	}

	return output;
}



//////////////////////////////
//
// MuseRecord::getGraphicNoteTypeField -- return column 17
//

string MuseRecord::getGraphicNoteTypeField(void) {
// allowNotesOnly("getGraphicNoteTypefield");
	if (getLength() < 17) {
		return " ";
	} else {
		return extract(17, 17);
	}
}



//////////////////////////////
//
// MuseRecord::getGraphicNoteType --
//

string MuseRecord::getGraphicNoteTypeString(void) {
	string output = getGraphicNoteTypeField();
	if (output[0] == ' ') {
		output = "";
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getGraphicRecip --
//

string MuseRecord::getGraphicRecip(void) {
	int notetype = getGraphicNoteType();
	string output;
	switch (notetype) {
		case -3: output = "0000"; break;  // double-maxima
		case -2: output = "000"; break;   // maxima
		case -1: output = "00"; break;    // long
		default:
			output = to_string(notetype);  // regular **recip number
	}
	int dotcount = getDotCount();
	for (int i=0; i<dotcount; i++) {
		output += '.';
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getGraphicNoteType --
//

int MuseRecord::getGraphicNoteType(void) {
	int output = 0;
	string recordInfo = getGraphicNoteTypeField();
	if (recordInfo[0] == ' ') {
		if (isInvisibleRest()) {
			// invisible rests do not have graphic note types
			// so make one up from the logical note type
			HumNum value = getTickDuration();
			value /= getTpq();
			if (value >= 32) {
				return -2;
			} else if (value >= 16) {
				return -1;
			} else if (value >= 8) {
				return 0;
			} else if (value >= 4) {
				return 1;
			} else if (value >= 2) {
				return 2;
			} else if (value >= 1) {
				return 4;
			} else if (value.getFloat() >= 0.5) {
				return 8;
			} else if (value.getFloat() >= 0.25) {
				return 16;
			} else if (value.getFloat() >= 0.125) {
				return 32;
			} else if (value.getFloat() >= 0.0625) {
				return 64;
			} else if (value.getFloat() >= 1.0/128) {
				return 128;
			} else if (value.getFloat() >= 1.0/256) {
				return 256;
			} else if (value.getFloat() >= 1.0/512) {
				return 512;
			} else {
				return 0;
			}
		} else {
			cerr << "Error: no graphic note type specified: " << getLine() << endl;
			return 0;
		}
	}

	switch (recordInfo[0]) {
		case 'M':                          // Maxima
			output = -2;           break;
		case 'L':   case 'B':              // Longa
			output = -1;           break;
		case 'b':   case 'A':              // Breve
			output = 0;            break;
		case 'w':   case '9':              // Whole
			output = 1;            break;
		case 'h':   case '8':              // Half
			output = 2;            break;
		case 'q':   case '7':              // Quarter
			output = 4;            break;
		case 'e':   case '6':              // Eighth
			output = 8;            break;
		case 's':   case '5':              // Sixteenth
			output = 16;           break;
		case 't':   case '4':              // 32nd note
			output = 32;           break;
		case 'x':   case '3':              // 64th note
			output = 64;           break;
		case 'y':   case '2':              // 128th note
			output = 128;          break;
		case 'z':   case '1':              // 256th note
			output = 256;          break;
		default:
			cerr << "Error: unknown graphical note type in column 17: "
				  << getLine() << endl;
	}

	return output;
}


//////////////////////////////
//
// MuseRecord::graphicNoteTypeQ --
//

int MuseRecord::graphicNoteTypeQ(void) {
	int output = 0;
	string recordInfo = getGraphicNoteTypeField();
	if (recordInfo[0] == ' ') {
		output = 0;
	} else {
		output = 1;
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::graphicNoteTypeSize -- return 0 if cue note size,
//	otherwise, it will return 1 if regular size
//

int MuseRecord::getGraphicNoteTypeSize(void) {
	int output = 1;
	string recordInfo = getGraphicNoteTypeField();
	if (recordInfo[0] == ' ') {
		cerr << "Error: not graphic note specified in column 17: "
			  << getLine() << endl;
		return 0;
	}

	switch (recordInfo[0]) {
		case 'L': case 'b': case 'w': case 'h': case 'q': case 'e':
		case 's': case 't': case 'x': case 'y': case 'z':
			output = 1;
			break;
		case 'B': case 'A': case '9': case '8': case '7': case '6':
		case '5': case '4': case '3': case '2': case '1':
			output = 0;
			break;
		default:
			cerr << "Error: unknown graphical note type in column 17: "
				  << getLine() << endl;
			return 0;
	}

	return output;
}



//////////////////////////////
//
// MuseRecord::getProlongationField -- returns column 18
//

string MuseRecord::getProlongationField(void) {
//   allowNotesOnly("getProlongationField");   ---> rests also
	if (getLength() < 18) {
		return " ";
	} else {
		return extract(18, 18);
	}
}



//////////////////////////////
//
// MuseRecord::getProlongationString --
//

string MuseRecord::getProlongationString(void) {
	string output = getProlongationField();
	if (output[0] == ' ') {
		output = "";
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getProlongation --
//

int MuseRecord::getProlongation(void) {
	int output = 0;
	string recordInfo = getProlongationField();
	switch (recordInfo[0]) {
		case ' ':   output = 0;   break;
		case '.':   output = 1;   break;
		case ':':   output = 2;   break;
		default:
			cerr << "Error: unknon prologation character (column 18): "
				  << getLine() << endl;
			return 0;
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getStringProlongation --
//

string MuseRecord::getStringProlongation(void) {
	switch (getProlongation()) {
		case 0:   return "";     break;
		case 1:   return ".";    break;
		case 2:   return "..";   break;
		case 3:   return "...";  break;
		case 4:   return "...."; break;
		default:
			cerr << "Error: unknown number of prolongation dots (column 18): "
				  << getLine() << endl;
			return "";
	}
	return "";
}



//////////////////////////////
//
// MuseRecord::prolongationQ --
//

int MuseRecord::prolongationQ(void) {
	return getProlongation();
}


//////////////////////////////
//
// MuseRecord::getNotatedAccidentalField -- actual notated accidental is
//	    stored in column 19.
//

string MuseRecord::getNotatedAccidentalField(void) {
	allowNotesOnly("getNotatedAccidentalField");
	if (getLength() < 19) {
		return " ";
	} else {
		string temp;
		temp += getColumn(19);
		return temp;
	}
}



//////////////////////////////
//
// MuseRecord::getNotatedAccidentalString --
//

string MuseRecord::getNotatedAccidentalString(void) {
	string output = getNotatedAccidentalField();
	if (output[0] == ' ') {
		output = "";
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getNotatedAccidental --
//

int MuseRecord::getNotatedAccidental(void) {
	int output = 0;
	string recordInfo = getNotatedAccidentalField();
	switch (recordInfo[0]) {
		case ' ':   output =  0;   break;
		case '#':   output =  1;   break;
		case 'n':   output =  0;   break;
		case 'f':   output = -1;   break;
		case 'x':   output =  2;   break;
		case 'X':   output =  2;   break;
		case '&':   output = -2;   break;
		case 'S':   output =  1;   break;
		case 'F':   output = -1;   break;
		default:
			cerr << "Error: unknown accidental: " << recordInfo[0] << endl;
			return 0;
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::notatedAccidentalQ --
//

int MuseRecord::notatedAccidentalQ(void) {
	int output;
	string recordInfo = getNotatedAccidentalField();
	if (recordInfo[0] == ' ') {
		output = 0;
	} else {
		output = 1;
	}
	return output;
}



///////////////////////////////
//
// MuseRecord::getTimeModificationField -- return columns 20 -- 22.
//

string MuseRecord::getTimeModificationField(void) {
//   allowNotesOnly("getTimeModificationField");   ---> rests also
	if (getLength() < 20) {
		return  "   ";
	} else {
		return extract(20, 22);
	}
}



//////////////////////////////
//
// MuseRecord::getTimeModification --
//

string MuseRecord::getTimeModification(void) {
	string output = getTimeModificationField();
	int index = 2;
	while (index >= 0 && output[index] == ' ') {
		output.resize(index);
		index--;
	}
	if (output.size() > 2) {
		if (output[0] == ' ') {
			output[0] = output[1];
			output[1] = output[2];
			output.resize(2);
		}
	}
	if (output.size() > 1) {
		if (output[0] == ' ') {
			output[0] = output[1];
			output.resize(1);
		}
	}
	if (output[0] == ' ') {
		cerr << "Error: funny error occured in time modification "
			  << "(columns 20-22): " << getLine() << endl;
		return "";
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getTimeModificationLeftField -- return column 20
//

string MuseRecord::getTimeModificationLeftField(void) {
	string output = getTimeModificationField();
	output.resize(1);
	return output;
}



//////////////////////////////
//
// MuseRecord::getTimeModificationLeftString --
//

string MuseRecord::getTimeModificationLeftString(void) {
	string output = getTimeModificationField();
	if (output[0] == ' ') {
		output = "";
	} else {
		output.resize(1);
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getTimeModificationLeft --
//

int MuseRecord::getTimeModificationLeft(void) {
	int output = 0;
	string recordInfo = getTimeModificationLeftString();
	if (recordInfo[0] == ' ') {
		output = 0;
	} else {
		output = std::strtol(recordInfo.c_str(), NULL, 36);
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getTimeModificationRightField -- return column 20
//

string MuseRecord::getTimeModificationRightField(void) {
	string output = getTimeModificationField();
	output = output[2];
	return output;
}



//////////////////////////////
//
// MuseRecord::getTimeModificationRight --
//

string MuseRecord::getTimeModificationRightString(void) {
	string output = getTimeModificationField();
	if (output[2] == ' ') {
		output = "";
	} else {
		output = output[2];
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getTimeModificationRight --
//

int MuseRecord::getTimeModificationRight(void) {
	int output = 0;
	string recordInfo = getTimeModificationRightString();
	if (recordInfo[2] == ' ') {
		output = 0;
	} else {
		string temp = recordInfo.substr(2);
		output = std::strtol(temp.c_str(), NULL, 36);
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::timeModificationQ --
//

int MuseRecord::timeModificationQ(void) {
	int output = 0;
	string recordInfo = getTimeModificationField();
	if (recordInfo[0] != ' ' || recordInfo[1] != ' ' || recordInfo[2] != ' ') {
		output = 1;
	} else {
		output = 0;
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::timeModificationLeftQ --
//

int MuseRecord::timeModificationLeftQ(void) {
	int output = 0;
	string recordInfo = getTimeModificationField();
	if (recordInfo[0] == ' ') {
		output = 0;
	} else {
		output = 1;
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::timeModificationRightQ --
//

int MuseRecord::timeModificationRightQ(void) {
	int output = 0;
	string recordInfo = getTimeModificationField();
	if (recordInfo[2] == ' ') {
		output = 0;
	} else {
		output = 1;
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getStemDirectionField --
//

string MuseRecord::getStemDirectionField(void) {
	allowNotesOnly("getStemDirectionField");
	if (getLength() < 23) {
		return " ";
	} else {
		string temp;
		temp += getColumn(23);
		return temp;
	}
}



//////////////////////////////
//
// MuseRecord::getStemDirectionString --
//

string MuseRecord::getStemDirectionString(void) {
	string output = getStemDirectionField();
	if (output[0] == ' ') {
		output = "";
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getStemDirection --
//

int MuseRecord::getStemDirection(void) {
	int output = 0;
	string recordInfo = getStemDirectionField();
	switch (recordInfo[0]) {
		case 'u':   output = 1;   break;
		case 'd':   output = -1;  break;
		case ' ':   output = 0;   break;
		default:
			cerr << "Error: unknown stem direction: " << recordInfo[0] << endl;
			return 0;
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::stemDirectionQ --
//

int MuseRecord::stemDirectionQ(void) {
	int output = 0;
	string recordInfo = getStemDirectionField();
	if (recordInfo[0] == ' ') {
		output = 0;
	} else {
		output = 1;
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getStaffField  -- returns column 24.
//

string MuseRecord::getStaffField(void) {
	allowNotesOnly("getStaffField");
	if (getLength() < 24) {
		return " ";
	} else {
		string temp;
		temp += getColumn(24);
		return temp;
	}
}



//////////////////////////////
//
// MuseRecord::getStaffString --
//

string MuseRecord::getStaffString(void) {
	string output = getStaffField();
	if (output[0] == ' ') {
		output = "";
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getStaff --
//

int MuseRecord::getStaff(void) {
	int output = 1;
	string recordInfo = getStaffField();
	if (recordInfo[0] == ' ') {
		output = 1;
	} else {
		output = std::strtol(recordInfo.c_str(), NULL, 36);
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::staffQ --
//

int MuseRecord::staffQ(void) {
	int output = 0;
	string recordInfo = getStaffField();
	if (recordInfo[0] == ' ') {
		output = 0;
	} else {
		output = 1;
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getBeamField --
//

string MuseRecord::getBeamField(void) {
	allowNotesOnly("getBeamField");
	if (getLength() < 26) {
		return "      ";
	} else {
		return extract(26, 31);
	}
}



//////////////////////////////
//
// MuseRecord::setBeamInfo --
//

void MuseRecord::setBeamInfo(string& strang) {
	setColumns(strang, 26, 31);
}



//////////////////////////////
//
// MuseRecord::beamQ --
//

int MuseRecord::beamQ(void) {
	int output = 0;
	allowNotesOnly("beamQ");
	if (getLength() < 26) {
		output = 0;
	} else {
		for (int i=26; i<=31; i++) {
			if (getColumn(i) != ' ') {
				output = 1;
				break;
			}
		}
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getBeam8 -- column 26
//

char MuseRecord::getBeam8(void) {
	allowNotesOnly("getBeam8");
	return getColumn(26);
}



//////////////////////////////
//
// MuseRecord::getBeam16 -- column 27
//

char MuseRecord::getBeam16(void) {
	allowNotesOnly("getBeam16");
	return getColumn(27);
}



//////////////////////////////
//
// MuseRecord::getBeam32 -- column 28
//

char MuseRecord::getBeam32(void) {
	allowNotesOnly("getBeam32");
	return getColumn(28);
}



//////////////////////////////
//
// MuseRecord::getBeam64 -- column 29
//

char MuseRecord::getBeam64(void) {
	allowNotesOnly("getBeam64");
	return getColumn(29);
}



//////////////////////////////
//
// MuseRecord::getBeam128 -- column 30
//

char MuseRecord::getBeam128(void) {
	allowNotesOnly("getBeam128");
	return getColumn(30);
}



//////////////////////////////
//
// MuseRecord::getBeam256 -- column 31
//

char MuseRecord::getBeam256(void) {
	allowNotesOnly("getBeam256");
	return getColumn(31);
}



//////////////////////////////
//
// MuseRecord::beam8Q --
//

int MuseRecord::beam8Q(void) {
	int output = 0;
	if (getBeam8() == ' ') {
		output = 0;
	} else {
		output = 1;
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::beam16Q --
//

int MuseRecord::beam16Q(void) {
	int output = 0;
	if (getBeam16() == ' ') {
		output = 0;
	} else {
		output = 1;
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::beam32Q --
//

int MuseRecord::beam32Q(void) {
	int output = 0;
	if (getBeam32() == ' ') {
		output = 0;
	} else {
		output = 1;
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::beam64Q --
//

int MuseRecord::beam64Q(void) {
	int output = 0;
	if (getBeam64() == ' ') {
		output = 0;
	} else {
		output = 1;
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::beam128Q --
//

int MuseRecord::beam128Q(void) {
	int output = 0;
	if (getBeam128() == ' ') {
		output = 0;
	} else {
		output = 1;
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::beam256Q --
//

int MuseRecord::beam256Q(void) {
	int output = 0;
	if (getBeam256() == ' ') {
		output = 0;
	} else {
		output = 1;
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getKernBeamStyle --
//

string MuseRecord::getKernBeamStyle(void) {
	string output;
	string beams = getBeamField();
	for (int i=0; i<(int)beams.size(); i++) {
		switch (beams[i]) {
			case '[':                 // start beam
				output += "L";
				break;
			case '=':                 // continue beam
				// do nothing
				break;
			case ']':                 // end beam
				output += "J";
				break;
			case '/':                 // forward hook
				output += "K";
				break;
			case '\\':                 // backward hook
				output += "k";
				break;
			default:
				;  // do nothing
		}
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getAdditionalNotationsField -- returns the contents
// 	of columns 32-43.
//

string MuseRecord::getAdditionalNotationsField(void) {
	allowNotesOnly("getAdditionalNotationsField");
	return extract(32, 43);
}



//////////////////////////////
//
// MuseRecord::additionalNotationsQ --
//

int MuseRecord::additionalNotationsQ(void) {
	int output = 0;
	if (getLength() < 32) {
		output = 0;
	} else {
		for (int i=32; i<=43; i++) {
			if (getColumn(i) != ' ') {
				output = 1;
				break;
			}
		}
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getAddCount -- returns the number of items
//	in the additional notations field
//

int MuseRecord::getAddCount(void) {
	string addString = getAdditionalNotationsField();
	string addElement;    // element from the notation field

	int count = 0;
	int index = 0;
	while (getAddElementIndex(index, addElement, addString)) {
		count++;
	}

	return count;
}



//////////////////////////////
//
// MuseRecord::getAddItem -- returns the specified item
//	in the additional notations field
//

string MuseRecord::getAddItem(int elementIndex) {
	string output;
	int count = 0;
	int index = 0;
	string addString = getAdditionalNotationsField();

	while (count <= elementIndex) {
		getAddElementIndex(index, output, addString);
		count++;
	}

	return output;
}



//////////////////////////////
//
// MuseRecord::getAddItemLevel -- returns the specified item's
//	editorial level in the additional notations field
//

int MuseRecord::getAddItemLevel(int elementIndex) {
	int count = 0;
	int index = 0;
	string number;
	string addString = getAdditionalNotationsField();
	string elementString; // element field

	while (count < elementIndex) {
		getAddElementIndex(index, elementString, addString);
		count++;
	}

	int output = -1;
repeating:
	while (addString[index] != '&' && index >= 0) {
		index--;
	}
	if (addString[index] == '&' && !isalnum(addString[index+1])) {
		index--;
		goto repeating;
	} else if (addString[index] == '&') {
		number = addString[index+1];
		output = std::strtol(number.c_str(), NULL, 36);
	}

	return output;
}



//////////////////////////////
//
// MuseRecord::getEditorialLevels -- returns a string containing the
//	edit levels given in the additional notation fields
//

string MuseRecord::getEditorialLevels(void) {
	string output;
	string addString = getAdditionalNotationsField();
	for (int index = 0; index < 12-1; index++) {
		if (addString[index] == '&' && isalnum(addString[index+1])) {
			output += addString[index+1];
		}
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::addEditorialLevelQ -- returns true if there are any editorial
//	levels present in the additional notations fields
//

int MuseRecord::addEditorialLevelQ(void) {
	string addString = getAdditionalNotationsField();
	int output = 0;
	for (int i=0; i<12-1; i++) {   // minus one for width 2 (&0)
		if (addString[i] == '&' && isalnum(addString[i+1])) {
			output = 1;
		}
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::findField -- returns true when it finds the first
//	instance of the key in the additional fields record.
//

int MuseRecord::findField(const string& key) {
	int len = (int)key.size();
	string notations = getAdditionalNotationsField();
	int output = 0;
	for (int i=0; i<12-len; i++) {
		if (notations[i] == key[0]) {
			output = 1;
			for (int j=0; j<len; j++) {
				if (notations[i] != key[j]) {
					output = 0;
					goto endofloop;
				}
			}
		}

		if (output == 1) {
			break;
		}
endofloop:
	;
	}

	return output;
}



//////////////////////////////
//
// MuseRecord::findField --
//

int MuseRecord::findField(char key, int mincol, int maxcol) {
	int start = mincol;
	int stop = getLength() - 1;

	if (start > stop) {
		return -1;
	}

	if (maxcol < stop) {
		stop = maxcol;
	}

	int i;
	for (i=start; i<=stop; i++) {
		if (m_recordString[i-1] == key) {
			return i;   // return the column which is offset from 1
		}
	}

	return -1;
}



//////////////////////////////
//
// MuseRecord::getSlurParameterRegion --
//

string MuseRecord::getSlurParameterRegion(void) {
	return getColumns(31, 43);
}



//////////////////////////////
//
// MuseRecord::getSlurStartColumn -- search column 32 to 43 for a slur
//    marker.  Returns the first one found from left to right.
//    returns -1 if a slur character was not found.
//

int MuseRecord::getSlurStartColumn(void) {
	int start = 31;
	int stop = getLength() - 1;
	if (stop >= 43) {
		stop = 42;
	}
	int i;
	for (i=start; i<=stop; i++) {
		switch (m_recordString[i]) {
			case '(':   // slur level 1
			case '[':   // slur level 2
			case '{':   // slur level 3
			case 'z':   // slur level 4
				return i+1;  // column is offset from 1
		}
	}

	return -1;
}



//////////////////////////////
//
// MuseRecord::getTextUnderlayField -- returns the contents
// 	of columns 44-80.
//

string MuseRecord::getTextUnderlayField(void) {
	allowNotesOnly("getTextUnderlayField");
	return extract(44, 80);
}



//////////////////////////////
//
// MuseRecord::textUnderlayQ --
//

int MuseRecord::textUnderlayQ(void) {
	int output = 0;
	if (getLength() < 44) {
		output = 0;
	} else {
		for (int i=44; i<=80; i++) {
			if (getColumn(i) != ' ') {
				output = 1;
				break;
			}
		}
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getVerseCount --
//

int MuseRecord::getVerseCount(void) {
	if (!textUnderlayQ()) {
		return 0;
	}

	int count = 1;
	for (int i=44; i<=getLength() && i <= 80; i++) {
		if (getColumn(i) == '|') {
			count++;
		}
	}

	return count;
}



//////////////////////////////
//
// MuseRecord::getVerse --
//

string MuseRecord::getVerse(int index) {
	string output;
	if (!textUnderlayQ()) {
		return output;
	}
	int verseCount = getVerseCount();
	if (index >= verseCount) {
		return output;
	}

	int tindex = 44;
	int c = 0;
	while (c < index && tindex < 80) {
		if (getColumn(tindex) == '|') {
			c++;
		}
		tindex++;
	}

	while (tindex <= 80 && getColumn(tindex) != '|') {
		output += getColumn(tindex++);
	}

	// remove trailing spaces
	int zindex = (int)output.size() - 1;
	while (output[zindex] == ' ') {
		zindex--;
	}
	zindex++;
	output.resize(zindex);

	// remove leading spaces
	int spacecount = 0;
	while (output[spacecount] == ' ') {
		spacecount++;
	}

	// problem here?
	for (int rr = 0; rr <= zindex-spacecount; rr++) {
		output[rr] = output[rr+spacecount];
	}

	return output;
}



//////////////////////////////
//
// MuseRecord::getVerseUtf8 --
//

string MuseRecord::getVerseUtf8(int index) {
	string tverse = getVerse(index);
	return MuseRecord::musedataToUtf8(tverse);
}



//////////////////////////////
//
// MuseRecord::getKernNoteStyle --
//	    default values: beams = 0, stems = 0
//

string MuseRecord::getKernNoteStyle(int beams, int stems) {
	string output;

	if (!isAnyNote()) {
		// not a note, so return nothing
		return "";
	}

	// place the rhythm
	stringstream tempdur;
	int notetype = getGraphicNoteType();
	if (timeModificationLeftQ()) {
		notetype = notetype / 4 * getTimeModificationLeft();
		if (timeModificationRightQ()) {
			notetype = notetype * getTimeModificationRight();
		} else {
			notetype = notetype * 2;
		}
	}

	// logical duration of the note
	HumNum logicalduration = getTicks();
	logicalduration /= getTpq();
	string durrecip = Convert::durationToRecip(logicalduration);

	// graphical duration of the note
	string graphicrecip = getGraphicRecip();
	HumNum graphicdur = Convert::recipToDuration(graphicrecip);

	string displayrecip;

	if (graphicdur != logicalduration) {
		// switch to the logical duration and store the graphic
		// duration.  The logical duration will be used on the
		// main kern token, and the graphic duration will be stored
		// as a layout parameter, such as !LO:N:vis=4. to display
		// the note as a dotted quarter regardless of the logical
		// duration.

		// Current test file has encoding bug related to triplets, so
		// disable graphic notation dealing with tuplets for now.

		// for now just looking to see if one has a dot and the other does not
		if ((durrecip.find(".") != string::npos) &&
				(graphicrecip.find(".") == string::npos)) {
			m_graphicrecip = graphicrecip;
			displayrecip = durrecip;
		} else if ((durrecip.find(".") == string::npos) &&
				(graphicrecip.find(".") != string::npos)) {
			m_graphicrecip = graphicrecip;
			displayrecip = durrecip;
		}
	}

	if (displayrecip.size() > 0) {
		output = displayrecip;
	} else {
		tempdur << notetype;
		output = tempdur.str();
		// add any dots of prolongation to the output string
		output += getStringProlongation();
	}

	// add the pitch to the output string
	string musepitch = getPitchString();
	string kernpitch = Convert::musePitchToKernPitch(musepitch);
	output += kernpitch;

	string logicalAccidental = getAccidentalString();
	string notatedAccidental = getNotatedAccidentalString();

	if (notatedAccidental.empty() && !logicalAccidental.empty()) {
		// Indicate that the logical accidental should not be
		// displayed (because of key signature or previous
		// note in the measure that alters the accidental
		// state of the current note).
		output += "y";
	} else if ((logicalAccidental == notatedAccidental) && !notatedAccidental.empty()) {
		// Indicate that the accidental should be displayed
		// and is not suppressed by the key signature or a
		// previous note in the measure.
		output += "X";
	}
	// There can be cases where the logical accidental
	// is natural but the notated accidetnal is sharp (but
	// the notated accidental means play a natural accidetnal).
	// Deal with this later.

	// if there is a notated natural sign, then add it now:
	string temp = getNotatedAccidentalField();
	if (temp == "n") {
		output += "n";
	}

	// check if a grace note
	if (getType() == 'g') {
		output += "Q";
	}

	// if stems is true, then show stem directions
	if (stems && stemDirectionQ()) {
		switch (getStemDirection()) {
			case 1:                         // 'u' = up
				output += "/";
				break;
			case -1:                        // 'd' = down
				output += "\\";
			default:
				; // nothing                 // ' ' = no stem (if stage 2)
		}
	}

	// if beams is true, then show any beams
	if (beams && beamQ()) {
		temp = getKernBeamStyle();
		output += temp;
	}

	if (isTied()) {
		string tiestarts;
		string tieends;
		int lasttie = getLastTiedNoteLineIndex();
		int nexttie = getNextTiedNoteLineIndex();
		int state = 0;
		if (lasttie >= 0) {
			state |= 2;
		}
		if (nexttie >= 0) {
			state |= 1;
		}
		switch (state) {
			case 1:
				tiestarts += "[";
				break;
			case 2:
				tieends += "]";
				break;
			case 3:
				tieends += "_";
				break;
		}
		if (state) {
			output = tiestarts + output + tieends;
		}
	}

	string slurstarts;
	string slurends;
	getSlurInfo(slurstarts, slurends);
	if ((!slurstarts.empty()) || (!slurends.empty())) {
		output = slurstarts + output + slurends;
	}

	return output;
}



//////////////////////////////
//
// MuseRecord::getSlurInfo --
//
//   ( ) = regular slur
//   [ ] = second levels slur, convert to &( and &)
//   { } = third level slur, convert to &&( and &&)
//   Z   = fourth level slur (how to close?)
//

void MuseRecord::getSlurInfo(string& slurstarts, string& slurends) {
	slurstarts.clear();
	slurends.clear();

	string data = getSlurParameterRegion();
	for (int i=0; i<(int)data.size(); i++) {
		if (data[i] == '(') {
			slurstarts += '(';
		} else if (data[i] == ')') {
			slurends += ')';
		} else if (data[i] == '[') {
			slurstarts += "&{";
		} else if (data[i] == ']') {
			slurends += "&)";
		} else if (data[i] == '{') {
			slurstarts += "&&(";
		} else if (data[i] == '}') {
			slurends += "&&)";
		}
	}
}



//////////////////////////////
//
// MuseRecord::getKernNoteAccents --
//

string MuseRecord::getKernNoteAccents(void) {
	string output;
	int addnotecount = getAddCount();
	for (int i=0; i<addnotecount; i++) {
		string tempnote = getAddItem(i);
		switch (tempnote[0]) {
			case 'v':   output += "v";   break; // up-bow
			case 'n':   output += "u";   break; // down-bow
			case 'o':   output += "j";   break; // harmonic
			case 'O':   output += "I";   break; // open string (to generic)
			case 'A':   output += "^";   break; // accent up
			case 'V':   output += "^";   break; // accent down
			case '>':   output += "^";   break; // horizontal accent
			case '.':   output += "'";   break; // staccato
			case '_':   output += "~";   break; // tenuto
			case '=':   output += "~'";  break; // detached legato
			case 'i':   output += "s";   break; // spiccato
			case '\'':  output += ",";   break; // breath mark
			case 'F':   output += ";";   break; // fermata up
			case 'E':   output += ";";   break; // fermata down
			case 'S':   output += ":";   break; // staccato
			case 't':   output += "O";   break; // trill (to generic)
			case 'r':   output += "S";   break; // turn
			case 'k':   output += "O";   break; // delayed turn (to generic)
			case 'w':   output += "O";   break; // shake (to generic)
			case 'M':   output += "O";   break; // mordent (to generic)
			case 'j':   output += "H";   break; // glissando (slide)
	  }
	}

	return output;
}



//////////////////////////////
//
// MuseRecord::getKernRestStyle --
//

string MuseRecord::getKernRestStyle(void) {

	string output;
	string rhythmstring;

	// place the rhythm
	stringstream tempdur;

	if (!isAnyRest()) {
		// not a rest, so return nothing
		return "";
	}

	// logical duration of the note
	HumNum logicalduration = getTicks();
	logicalduration /= getTpq();
	string durrecip = Convert::durationToRecip(logicalduration);

	/*
	int notetype;
	if (graphicNoteTypeQ()) {
		notetype = getGraphicNoteType();

		if (timeModificationLeftQ()) {
			notetype = notetype / 4 * getTimeModificationLeft();
		}
		if (timeModificationRightQ()) {
			notetype = notetype * getTimeModificationRight() / 2;
		}
		tempdur << notetype;
		output =  tempdur.str();

		// add any dots of prolongation to the output string
		output += getStringProlongation();
	} else {   // stage 1 data:
		HumNum dnotetype(getTickDuration(), quarter);
		rhythmstring = Convert::durationToRecip(dnotetype);
		output += rhythmstring;
	}
	*/

	output = durrecip;

	// add the pitch to the output string
	output += "r";

	if (isInvisibleRest()) {
		output += "yy";
	}

	return output;
}



//////////////////////////////////////////////////////////////////////////
//
// functions that work with measure records
//


//////////////////////////////
//
// MuseRecord::getMeasureNumberField -- columns 9-12
//

string MuseRecord::getMeasureNumberField(void) {
	allowMeasuresOnly("getMeasureNumberField");
	return extract(9, 12);
}



//////////////////////////////
//
// MuseRecord::getMeasureTypeField -- columns 1 -- 7
//

string MuseRecord::getMeasureTypeField(void) {
	allowMeasuresOnly("getMeasureTypeField");
	return extract(1, 7);
}



//////////////////////////////
//
// MuseRecord::getMeasureNumberString --
//

string MuseRecord::getMeasureNumberString(void) {
	string output = getMeasureNumberField();
	for (int i=3; i>=0; i--) {
		if (output[i] == ' ') {
			output.resize(i);
		}
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getMeasureNumber --
//

int MuseRecord::getMeasureNumber(void) {
	string measureInfo = getMeasureNumberField();
	if (measureInfo.empty()) {
		return 0;
	}
	return std::stoi(measureInfo);
}



//////////////////////////////
//
// MuseRecord::measureNumberQ --
//

int MuseRecord::measureNumberQ(void) {
	string temp = getMeasureNumberString();
	int i = 0;
	int output = 0;
	while (temp[i] != '\0') {
		if (temp[i] != ' ') {
			output = 1;
			break;
		}
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getMeasureFlagsString --  Columns 17 to 80.
//

string MuseRecord::getMeasureFlagsString(void) {
	if (m_recordString.size() < 17) {
		return "";
	} else {
		return trimSpaces(m_recordString.substr(16));
	}
}



//////////////////////////////
//
// MuseRecord::measureFermataQ -- returns true if there is a
//	fermata above or below the measure
//

int MuseRecord::measureFermataQ(void) {
	int output = 0;
	for (int i=17; i<=80 && i<= getLength(); i++) {
		if (getColumn(i) == 'F' || getColumn(i) == 'E') {
			output = 1;
			break;
		}
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::measureFlagQ -- Returns true if there are non-space
//     characters in columns 17 through 80.   A more smarter way of
//     doing this is checking the allocated length of the record, and
//     do not search non-allocated columns for non-space characters...
//

int MuseRecord::measureFlagQ(const string& key) {
	int output = 0;
	int len = (int)key.size();
	for (int i=17; i<=80-len && i<getLength(); i++) {
		if (getColumn(i) == key[0]) {
			output = 1;
			for (int j=0; j<len; j++) {
				if (getColumn(i+j) != key[j]) {
					output = 0;
					break;
				}
			}
			if (output == 1) {
				break;
			}
		}
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::addMeasureFlag -- add the following characters to the
//    Flag region of the measure flag area (columns 17-80).  But only
//    add the flag if it is not already present in the region.  If it is
//    not present, then append it after the last non-space character
//    in that region.  A space will be added between the last item
//    and the newly added parameter.
//

void MuseRecord::addMeasureFlag(const string& strang) {
	string flags = getColumns(17, 80);
	string flag = strang;

	HumRegex hre;
	hre.replaceDestructive(flag, "\\*", "\\*", "g");
	hre.replaceDestructive(flag, "\\|", "\\|", "g");
	if (hre.search(flags, flag)) {
		// flag was already found in flags, so don't do anything
		return;
	}
	hre.replaceDestructive(flags, "", "\\s+$");
	flags += " ";
	flags += strang;
	setColumns(flags, 17, 80);
}



//////////////////////////////
//
// MuseRecord::getKernMeasureStyle --
//

string MuseRecord::getKernMeasureStyle(void) {
	allowMeasuresOnly("getKernMeasureStyle");
	string temp;
	string tempstyle = getMeasureTypeField();

	string output = "=";
	if (tempstyle == "mheavy2") {
		output += "=";
	} else if (tempstyle == "mheavy3") {
		output += "=";
	} else if (tempstyle == "mheavy4") {
		output += "=";
	}

	if (measureNumberQ()) {
		temp = getMeasureNumberString();
		output += temp;
	}

	// what is this for-loop for?
	for (int i=0; i<(int)temp.size(); i++) {
		temp[i] = std::tolower(temp[i]);
	}

	if (tempstyle == "mheavy1") {
		output += "!";
	} else if (tempstyle == "mheavy2") {
		  if (measureFlagQ(":||:")) {
			  output += ":|!:";
			  zerase(output, 1);             // make "==" become "="
		  } else if (measureFlagQ(":|")) {
			  output += ":|!";
			  zerase(output, 1);             // make "==" become "="
		  }
	} else if (tempstyle == "mheavy3") {
		output += "!|";
	} else if (tempstyle == "mheavy4") {
		if (measureFlagQ(":||:")) {
			output += ":!!:";
		} else {
			output += "!!";
		}
	}
	return output;
}


//////////////////////////////////////////////////////////////////////////
//
// functions that work with musical attributes records
//

//////////////////////////////
//
// MuseRecord::getAttributeMap --
//

void MuseRecord::getAttributeMap(map<string, string>& amap) {
	amap.clear();
	// Should be "3" on the next line, but "1" or "2" might catch poorly formatted data.
	string contents = getLine().substr(2);
	if (contents.empty()) {
		return;
	}
	int i = 0;
	string key;
	string value;
	int state = 0;  // 0 outside, 1 = in key, 2 = in value
	while (i < (int)contents.size()) {
		switch (state) {
			case 0: // outside of key or value
				if (!isspace(contents[i])) {
					if (contents[i] == ':') {
						// Strange: should not happen
						key.clear();
						state = 2;
					} else {
						state = 1;
						key += contents[i];
					}
				}
				break;
			case 1: // in key
				if (!isspace(contents[i])) {
					if (contents[i] == ':') {
						value.clear();
						state = 2;
					} else {
						// Add to key, such as "C2" for second staff clef.
						key += contents[i];
					}
				}
				break;
			case 2: // in value
				if (key == "D") {
					value += contents[i];
				} else if (isspace(contents[i])) {
					// store parameter and clear variables
					amap[key] = value;
					state = 0;
					key.clear();
					value.clear();
				} else {
					value += contents[i];
				}
				break;
		}
		i++;
	}

	if ((!key.empty()) && (!value.empty())) {
		amap[key] = value;
	}
}



//////////////////////////////
//
// MuseRecord::getAttributes --
//

string MuseRecord::getAttributes(void) {
	string output;
	switch (getType()) {
		case E_muserec_musical_attributes:
			break;
		default:
			cerr << "Error: cannot use getAttributes function on line: "
				  << getLine() << endl;
			return "";
	}

	int ending = 0;
	int tempcol;
	for (int column=4; column <= getLength(); column++) {
		if (getColumn(column) == ':') {
			tempcol = column - 1;
			while (tempcol > 0 && getColumn(tempcol) != ' ') {
				tempcol--;
			}
			tempcol++;
			while (tempcol <= column) {
				output += getColumn(tempcol);
				if (output.back() == 'D') {
					ending = 1;
				}
				tempcol++;
			}
		}
		if (ending) {
			break;
		}
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::attributeQ --
//

int MuseRecord::attributeQ(const string& attribute) {
	switch (getType()) {
		case E_muserec_musical_attributes:
			break;
		default:
			cerr << "Error: cannot use getAttributes function on line: "
				  << getLine() << endl;
			return 0;
	}


	string attributelist = getAttributes();

	int output = 0;
	int attstrlength = (int)attributelist.size();
	int attlength = (int)attribute.size();

	for (int i=0; i<attstrlength-attlength+1; i++) {
		if (attributelist[i] == attribute[0]) {
			output = 1;
			for (int j=0; j<attlength; j++) {
				if (attributelist[i] != attribute[j]) {
					output = 0;
					break;
				}
			}
			if (output == 1) {
				break;
			}
		}
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getAttributeInt --
//

int MuseRecord::getAttributeInt(char attribute) {
	switch (getType()) {
		case E_muserec_musical_attributes:
			break;
		default:
			cerr << "Error: cannot use getAttributeInt function on line: "
				  << getLine() << endl;
			return 0;
	}

	int output = E_unknown;
	int ending = 0;
	int index = 0;
	int tempcol;
	int column;
	for (column=4; column <= getLength(); column++) {
		if (getColumn(column) == ':') {
			tempcol = column - 1;
			while (tempcol > 0 && getColumn(tempcol) != ' ') {
				tempcol--;
			}
			tempcol++;
			while (tempcol <= column) {
				if (getColumn(tempcol) == attribute) {
					ending = 2;
				} else if (getColumn(tempcol) == 'D') {
					ending = 1;
				}
				tempcol++;
				index++;
			}
		}
		if (ending) {
			break;
		}
	}

	if (ending == 0 || ending == 1) {
		return output;
	} else {
		string value = &getColumn(column+1);
		if (value.empty()) {
			output = std::stoi(value);
			return output;
		} else {
			return 0;
		}
	}
}



//////////////////////////////
//
// MuseRecord::getAttributeField -- returns true if found attribute
//

int MuseRecord::getAttributeField(string& value, const string& key) {
	switch (getType()) {
		case E_muserec_musical_attributes:
			break;
		default:
			cerr << "Error: cannot use getAttributeInt function on line: "
				  << getLine() << endl;
			return 0;
	}

	int returnValue = 0;
	int ending = 0;
	int index = 0;
	int tempcol;
	int column;
	for (column=4; column <= getLength(); column++) {
		if (getColumn(column) == ':') {
			tempcol = column - 1;
			while (tempcol > 0 && getColumn(tempcol) != ' ') {
				tempcol--;
			}
			tempcol++;
			while (tempcol <= column) {
				if (getColumn(tempcol) == key[0]) {
					ending = 2;
				} else if (getColumn(tempcol) == 'D') {
					ending = 1;
				}
				tempcol++;
				index++;
			}
		}
		if (ending) {
			break;
		}
	}

	value.clear();
	if (ending == 0 || ending == 1) {
		return returnValue;
	} else {
		returnValue = 1;
		column++;
		while (getColumn(column) != ' ') {
			value += getColumn(column++);
		}
		return returnValue;
	}
}



///////////////////////////////////////////////////////////////////////////
//
// functions that work with basso continuo figuration records (f):
//


//////////////////////////////
//
// MuseRecord::getFigureCountField -- column 2.
//

string MuseRecord::getFigureCountField(void) {
	allowFigurationOnly("getFigureCountField");
	return extract(2, 2);
}



//////////////////////////////
//
// MuseRecord::getFigurationCountString --
//

string MuseRecord::getFigureCountString(void) {
	allowFigurationOnly("getFigureCount");
	string output = extract(2, 2);
	if (output[0] == ' ') {
		output = "";
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getFigurationCount --
//

int MuseRecord::getFigureCount(void) {
	allowFigurationOnly("getFigureCount");
	string temp = getFigureCountString();
	int output = std::strtol(temp.c_str(), NULL, 36);
	return output;
}



//////////////////////////////
//
// getFigurePointerField -- columns 6 -- 8.
//

string MuseRecord::getFigurePointerField(void) {
	allowFigurationOnly("getFigurePointerField");
	return extract(6, 8);
}


//////////////////////////////
//
// figurePointerQ --
//

int MuseRecord::figurePointerQ(void) {
	allowFigurationOnly("figurePointerQ");
	int output = 0;
	for (int i=6; i<=8; i++) {
		if (getColumn(i) != ' ') {
			output = 1;
			break;
		}
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getFigureString --
//

string MuseRecord::getFigureString(void) {
	string output = getFigureFields();
	for (int i=(int)output.size()-1; i>= 0; i--) {
		if (isspace(output[i])) {
			output.resize((int)output.size() - 1);
		} else {
			break;
		}
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::getFigureFields -- columns 17 -- 80
//

string MuseRecord::getFigureFields(void) {
	allowFigurationOnly("getFigureFields");
	return extract(17, 80);
}


//////////////////////////////
//
// MuseRecord::figureFieldsQ --
//

int MuseRecord::figureFieldsQ(void) {
	allowFigurationOnly("figureFieldsQ");
	int output = 0;
	if (getLength() < 17) {
		output = 0;
	} else {
		for (int i=17; i<=80; i++) {
			if (getColumn(i) != ' ') {
				output = 1;
				break;
			}
		}
	}
	return output;
}



//////////////////////////////
//
// getFigure --
//

string MuseRecord::getFigure(int index) {
	string output;
	allowFigurationOnly("getFigure");
	if (index >= getFigureCount()) {
		return output;
	}
	string temp = getFigureString();
	if (index == 0) {
		return temp;
	}
	HumRegex hre;
	vector<string> pieces;
	hre.split(pieces, temp, " +");
	if (index < (int)pieces.size()) {
	output = pieces[index];
	}
	return output;
}



///////////////////////////////////////////////////////////////////////////
//
// protected functions
//


//////////////////////////////
//
// MuseRecord::allowFigurationOnly --
//

void MuseRecord::allowFigurationOnly(const string& functionName) {
	switch (getType()) {
		case E_muserec_figured_harmony:
		  break;
		default:
			cerr << "Error: can only access " << functionName
				  << " on a figuration record.  Line is: " << getLine() << endl;
			return;
	}
}



//////////////////////////////
//
// MuseRecord::allowFigurationAndNotesOnly --
//

void MuseRecord::allowFigurationAndNotesOnly(const string& functionName) {
	switch (getType()) {
		case E_muserec_figured_harmony:
		case E_muserec_note_regular:
		case E_muserec_note_chord:
		case E_muserec_note_grace:
		case E_muserec_note_cue:
		  break;
		default:
			cerr << "Error: can only access " << functionName
				  << " on a figuration record.  Line is: " << getLine() << endl;
			return;
	}
}



//////////////////////////////
//
// MuseRecord::allowMeasuresOnly --
//

void MuseRecord::allowMeasuresOnly(const string& functionName) {
	switch (getType()) {
		case E_muserec_measure:
		  break;
		default:
			cerr << "Error: can only access " << functionName
				  << " on a measure record.  Line is: " << getLine() << endl;
			return;
	}
}


//////////////////////////////
//
// MuseRecord::allDirectionsOnly --
//

void MuseRecord::allowDirectionsOnly(const std::string& functionName) {
	switch (getType()) {
		case E_muserec_musical_directions:
			break;
		default:
			cerr << "Error: can only access " << functionName
				  << " on a musical direction record.  Line is: " << getLine() << endl;
			return;
	}
}



//////////////////////////////
//
// MuseRecord::allowNotesOnly --
//

void MuseRecord::allowNotesOnly(const string& functionName) {
	switch (getType()) {
		case E_muserec_note_regular:
		case E_muserec_note_chord:
		case E_muserec_note_grace:
		case E_muserec_note_cue:
		  break;
		default:
			cerr << "Error: can only access " << functionName
				  << " on a note record.  Line is: " << getLine() << endl;
			return;
	}
}



//////////////////////////////
//
// MuseRecord::allowNotesAndRestsOnly --
//

void MuseRecord::allowNotesAndRestsOnly(const string& functionName) {
	switch (getType()) {
		case E_muserec_note_regular:
		case E_muserec_note_chord:
		case E_muserec_note_grace:
		case E_muserec_note_cue:
		case E_muserec_rest:
		case E_muserec_rest_invisible:
		  break;
		default:
			cerr << "Error: can only access " << functionName
				  << " on a note or rest records.  Line is: " << getLine() << endl;
			return;
	}
}



//////////////////////////////
//
// MuseRecord::getAddElementIndex -- get the first element pointed
//	to by the specified index in the additional notations field.
//	returns 0 if no element was found, or 1 if an element was found.
//

int MuseRecord::getAddElementIndex(int& index, string& output, const string& input) {
	int finished = 0;
	int count = 0;
	output.clear();

	while (!finished) {
		switch (input[index]) {
			case '&':                     // editorial level marker
				// there is exactly one character following an editorial
				// marker.  neither the '&' nor the following character
				// is counted if the following character is in the set
				// [0..9, A..Z, a..z]
				index++;
				if (isalnum(input[index])) {
					index++;
				} else {
					// count '&' as an element
					count++;
					output += '&';
				}
				break;

			case 'p': case 'f':          // piano and forte
				// any sequence of 'p' and 'f' is considered one element
				count++;
				output += input[index++];
				while (input[index] == 'p' || input[index] == 'f') {
					output += input[index++];
				}
				break;

			case 'Z':                    // sfz, or Zp = sfp
				// elements starting with 'Z':
				//    Z      = sfz
				//    Zp     = sfp
				count++;
				output += input[index++];
				if (input[index] == 'p') {
					output += input[index++];
				}
				break;

			case 'm':                      // mezzo
				// a mezzo marking MUST be followed by a 'p' or an 'f'.
				count++;
				output += input[index++];
				if (input[index] == 'p' || input[index] == 'f') {
					output += input[index++];
				} else {
				  cout << "Error at \'m\' in notation field: " << input << endl;
				  return 0;
				}
				break;

			case 'S':                     // arpeggiation
				// elements starting with 'S':
				//   S     = arpeggiate (up)
				//   Sd    = arpeggiate down)
				count++;
				output += input[index++];
				if (input[index] == 'd') {
					output += input[index++];
				}
				break;

			case '1':                     // fingering
			case '2':                     // fingering
			case '3':                     // fingering
			case '4':                     // fingering
			case '5':                     // fingering
			// case ':':                  // finger substitution
				// keep track of finger substitutions
				count++;
				output += input[index++];
				if (input[index] == ':') {
					output += input[index++];
					output += input[index++];
				}
				break;

			//////////////////////////////
			// Ornaments
			//
			case 't':                     // trill (tr.)
			case 'r':                     // turn
			case 'k':                     // delayed turn
			case 'w':                     // shake
			case '~':                     // trill wavy line extension
			case 'c':                     // continued wavy line
			case 'M':                     // mordent
			case 'j':                     // slide (Schleifer)
			  // ornaments can be modified by accidentals:
			  //    s     = sharp
			  //    ss    = double sharp
			  //    f     = flat
			  //    ff    = double flat
			  //    h     = natural
			  //    u     = next accidental is under the ornament
			  // any combination of these characters following a
			  // ornament is considered one element.
			  //
			  count++;
			  index++;
			  while (input[index] == 's' || input[index] == 'f' ||
						input[index] == 'h' || input[index] == 'u') {
				  output += input[index++];
			  }
			  break;

			//////////////////////////////////////////////////////////////
			// The following chars are uniquely SINGLE letter items:    //
			//                                                          //
			//                                                          //
			case '-':                     // tie                        //
			case '(':                     // open  slur #1              //
			case ')':                     // close slur #1              //
			case '[':                     // open  slur #2              //
			case ']':                     // close slur #2              //
			case '{':                     // open  slur #3              //
			case '}':                     // close slur #3              //
			case 'z':                     // open  slur #4              //
			case 'x':                     // close slur #4              //
			case '*':                     // start written tuplet       //
			case '!':                     // end written tuplet         //
			case 'v':                     // up bow                     //
			case 'n':                     // down bow                   //
			case 'o':                     // harmonic                   //
			case 'O':                     // open string                //
			case 'Q':                     // thumb position             //
			case 'A':                     // accent (^)                 //
			case 'V':                     // accent (v)                 //
			case '>':                     // accent (>)                 //
			case '.':                     // staccatto                  //
			case '_':                     // tenuto                     //
			case '=':                     // detached tenuto            //
			case 'i':                     // spiccato                   //
			case '\'':                    // breath mark                //
			case 'F':                     // upright fermata            //
			case 'E':                     // inverted fermata           //
			case 'R':                     // rfz                        //
			case '^':                     // editorial accidental       //
			case '+':                     // cautionary accidental      //
				count++;                                                 //
				output += input[index++];                                //
				break;                                                   //
			//                                                          //
			//                                                          //
			//////////////////////////////////////////////////////////////
			case ' ':
				// ignore blank spaces and continue counting elements
				index++;
				break;
			default:
				cout << "Error: unknown additional notation: "
					  << input[index] << endl;
				return 0;
		}
		if (count != 0 || index >= 12) {
			finished = 1;
		}
	} // end of while (!finished) loop

	return count;
}



////////////////////
//
// MuseRecord::zerase -- removes specified number of characters from
// 	the beginning of the string.
//

void MuseRecord::zerase(string& inout, int num) {
	int len = (int)inout.size();
	if (num >= len) {
		inout = "";
	} else {
		for (int i=num; i<=len; i++) {
			inout[i-num] = inout[i];
		}
	}
	inout.resize(inout.size() - num);
}



/////////////////////////////////////////
//
// MuseRecord::getDirectionTypeString -- columns 17 and 18.
//    A = segno sign
//    B = right-justified text
//    C = center-justified text
//    D = left-justified text
//    E = dynamics hairpin start
//    F = dynamics hairpin end
//    G = letter dynamics (text given starting in column 25)
//    H = begin dashes (after words)
//    J = end dashes
//    P = pedal start
//    Q = pedal stop
//    R = rehearsal number or letter
//    U = octave up start
//    V = octave down start
//    W = octave stop
//

std::string MuseRecord::getDirectionTypeField(void) {
	allowDirectionsOnly("getDirectionType");
	return extract(17, 18);
}



//////////////////////////////
//
// MuseRecord::getDirectionTypeString -- Same as the field version, but
//    trailing spaces are removed (not leading ones, at least for now).
//

std::string MuseRecord::getDirectionTypeString(void) {
	string output = getDirectionTypeField();
	if (output.back() == ' ') {
		output.resize(output.size() - 1);
	}
	if (output.back() == ' ') {
		output.resize(output.size() - 1);
	}
	return output;
}



//////////////////////////////
//
// MuseRecord::isTextDirection -- Text is stored starting at column 25.
//    B = right justified
//    C = center justified
//    D = left justified
//

bool MuseRecord::isTextDirection(void) {
	string typefield = getDirectionTypeField();
	if (typefield.find('B') != string::npos) {
		return true;
	}
	if (typefield.find('C') != string::npos) {
		return true;
	}
	if (typefield.find('D') != string::npos) {
		return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecord::isHairpin --
//

bool MuseRecord::isHairpin(void) {
	string typefield = getDirectionTypeField();
	if (isHairpinStart()) {
		return true;
	}
	if (isHairpinStop()) {
		return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecord::isHairpinStart --
//

bool MuseRecord::isHairpinStart(void) {
	string typefield = getDirectionTypeField();
	if (typefield.find('E') != string::npos) {
		return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecord::isHairpinStop --
//

bool MuseRecord::isHairpinStop(void) {
	string typefield = getDirectionTypeField();
	if (typefield.find('F') != string::npos) {
		return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecord::isDashStart --
//

bool MuseRecord::isDashStart(void) {
	string typefield = getDirectionTypeField();
	if (typefield.find('H') != string::npos) {
		return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecord::isDashStop --
//

bool MuseRecord::isDashStop(void) {
	string typefield = getDirectionTypeField();
	if (typefield.find('J') != string::npos) {
		return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecord::isPedalStart --
//

bool MuseRecord::isPedalStart(void) {
	string typefield = getDirectionTypeField();
	if (typefield.find('P') != string::npos) {
		return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecord::isPedalEnd --
//

bool MuseRecord::isPedalEnd(void) {
	string typefield = getDirectionTypeField();
	if (typefield.find('Q') != string::npos) {
		return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecord::isRehearsal --
//

bool MuseRecord::isRehearsal(void) {
	string typefield = getDirectionTypeField();
	if (typefield.find('R') != string::npos) {
		return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecord::isOctiveUpStart --
//

bool MuseRecord::isOctaveUpStart(void) {
	string typefield = getDirectionTypeField();
	if (typefield.find('U') != string::npos) {
		return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecord::isOctaveDownStart --
//

bool MuseRecord::isOctaveDownStart(void) {
	string typefield = getDirectionTypeField();
	if (typefield.find('V') != string::npos) {
		return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecord::isOctaveStop --
//

bool MuseRecord::isOctaveStop(void) {
	string typefield = getDirectionTypeField();
	if (typefield.find('W') != string::npos) {
		return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecord::getDirectionText -- Return the text starting in column 25.
//

std::string MuseRecord::getDirectionText(void) {
	int length = (int)m_recordString.size();
	if (length < 25) {
		// no text
		return "";
	}
	return trimSpaces(m_recordString.substr(24));
}






//////////////////////////////
//
// MuseRecordBasic::MuseRecordBasic --
//

MuseRecordBasic::MuseRecordBasic(void) {
	m_recordString.reserve(81);
	setType(E_muserec_unknown);

	m_lineindex    =   -1;
	m_absbeat      =    0;
	m_lineduration =    0;
	m_noteduration =    0;
	m_b40pitch     = -100;
	m_nexttiednote =   -1;
	m_lasttiednote =   -1;
	m_roundBreve   =    0;
	m_header       =   -1;
	m_layer        =    0;
}


// default value: index = -1;
MuseRecordBasic::MuseRecordBasic(const string& aLine, int index) {
	m_recordString.reserve(81);
	setLine(aLine);
	setType(E_muserec_unknown);
	m_lineindex = index;

	m_absbeat      =    0;
	m_lineduration =    0;
	m_noteduration =    0;
	m_b40pitch     = -100;
	m_nexttiednote =   -1;
	m_lasttiednote =   -1;
	m_roundBreve   =    0;
	m_header       =   -1;
	m_layer        =    0;
}


MuseRecordBasic::MuseRecordBasic(MuseRecordBasic& aRecord) {
	*this = aRecord;
}



//////////////////////////////
//
// MuseRecordBasic::~MuseRecordBasic --
//

MuseRecordBasic::~MuseRecordBasic() {
	m_recordString.resize(0);

	m_absbeat      =    0;
	m_lineduration =    0;
	m_noteduration =    0;
	m_b40pitch     = -100;
	m_nexttiednote =   -1;
	m_lasttiednote =   -1;
	m_roundBreve   =    0;
	m_layer        =    0;
}



//////////////////////////////
//
// MuseRecordBasic::clear -- remove content of record.
//

void MuseRecordBasic::clear(void) {
	m_recordString.clear();
	m_lineindex    =   -1;
	m_absbeat      =    0;
	m_lineduration =    0;
	m_noteduration =    0;
	m_b40pitch     = -100;
	m_nexttiednote =   -1;
	m_lasttiednote =   -1;
	m_roundBreve   =    0;
	m_header       =   -1;
	m_layer        =    0;
}



/////////////////////////////
//
// MuseRecordBasic::isEmpty -- returns true if only spaces on line, ignoring
//     non-printable characters (which may no longer be necessary).
//

int MuseRecordBasic::isEmpty(void) {
	for (int i=0; i<(int)m_recordString.size(); i++) {
		if (!std::isprint(m_recordString[i])) {
			continue;
		}
		if (!std::isspace(m_recordString[i])) {
			return 0;
		}
	}
	return 1;
}



//////////////////////////////
//
// MuseRecordBasic::extract -- extracts the character columns from the
//	storage string.  Appends a null character to the end of the
//	copied string.
//

string MuseRecordBasic::extract(int start, int end) {
	string output;
	int count = end - start + 1;
	for (int i=0; i<count; i++) {
		if (i+start <= getLength()) {
			output += getColumn(i+start);
		} else {
			output += ' ';
		}
	}
	return output;
}



//////////////////////////////
//
// MuseRecordBasic::getColumn -- same as operator[] but with an
//	offset of 1 rather than 0.
//

char& MuseRecordBasic::getColumn(int columnNumber) {
	int realindex = columnNumber - 1;
	int length = (int)m_recordString.size();
	// originally the limit for data columns was 80:
	// if (realindex < 0 || realindex >= 80) {
	// the new limit is somewhere above 900, but limit to 180
	if (realindex < 0 || realindex >= 180) {
		cerr << "Error trying to access column: " << columnNumber  << endl;
		cerr << "CURRENT DATA: ===============================" << endl;
		cerr << (*this);
		static char x = ' ';
		return x;
	} else if (realindex >= (int)m_recordString.size()) {
		m_recordString.resize(realindex+1);
		for (int i=length; i<=realindex; i++) {
			m_recordString[i] = ' ';
		}
	}
	return m_recordString[realindex];
}



//////////////////////////////
//
// MuseRecordBasic::getColumns --
//

string MuseRecordBasic::getColumns(int startcol, int endcol) {
	string output;
	int charcount = endcol - startcol + 1;
	if (charcount <= 0) {
		return output;
	}
	for (int i=startcol; i<=endcol; i++) {
		output += getColumn(i);
	}
	return output;
}



//////////////////////////////
//
// MuseRecordBasic::setColumns --
//

void MuseRecordBasic::setColumns(string& data, int startcol, int endcol) {
	if (startcol > endcol) {
		int temp = startcol;
		startcol = endcol;
		endcol = temp;
	}

	int dsize = (int)data.size();
	getColumn(endcol) = ' '; // allocate space if not already done
	int i;
	int ii;
	for (i=startcol; i<=endcol; i++) {
		ii = i - startcol;
		if (ii < dsize) {
			getColumn(i) = data[ii];
		} else {
			break;
		}
	}

}



//////////////////////////////
//
// MuseRecordBasic::getLength -- returns the size of the
//	character string being stored.  A number between
//	0 and 80.
//

int MuseRecordBasic::getLength(void) const {
	return (int)m_recordString.size();
}



//////////////////////////////
//
// MuseRecordBasic::getLine -- returns a pointer to data record
//

string MuseRecordBasic::getLine(void) {
	return m_recordString;
}



//////////////////////////////
//
// MuseRecordBasic::getType -- returns the type of the record.
//

int MuseRecordBasic::getType(void) const {
	return m_type;
}



//////////////////////////////
//
// MuseRecordBasic::operator=
//

MuseRecordBasic& MuseRecordBasic::operator=(MuseRecordBasic& aRecord) {
	// don't copy onto self
	if (&aRecord == this) {
		return *this;
	}

	setLine(aRecord.getLine());
	setType(aRecord.getType());
	m_lineindex = aRecord.m_lineindex;

	m_absbeat = aRecord.m_absbeat;
	m_lineduration = aRecord.m_lineduration;
	m_noteduration = aRecord.m_noteduration;

	m_b40pitch     = aRecord.m_b40pitch;
	m_nexttiednote = aRecord.m_nexttiednote;
	m_lasttiednote = aRecord.m_lasttiednote;

	return *this;
}


MuseRecordBasic& MuseRecordBasic::operator=(MuseRecordBasic* aRecord) {
	*this = *aRecord;
	return *this;
}


MuseRecordBasic& MuseRecordBasic::operator=(const string& aLine) {
	setLine(aLine);
	setType(aLine[0]);

	m_lineindex    =   -1;
	m_absbeat      =    0;
	m_lineduration =    0;
	m_noteduration =    0;
	m_b40pitch     = -100;
	m_nexttiednote =   -1;
	m_lasttiednote =   -1;

	return *this;
}



//////////////////////////////
//
// MuseRecordBasic::operator[] -- character array offset from 0.
//

char& MuseRecordBasic::operator[](int index) {
	return getColumn(index+1);
}



//////////////////////////////
//
// MuseRecordBasic::setLine -- sets the record to a (new) string
//

void MuseRecordBasic::setLine(const string& aLine) {
	m_recordString = aLine;
	// Line lengths should not exceed 80 characters according
	// to MuseData standard, so maybe have a warning or error if exceeded.
}



//////////////////////////////
//
// MuseRecordBasic::setType -- sets the type of the record
//

void MuseRecordBasic::setType(int aType) {
	m_type = aType;
}



//////////////////////////////
//
// MuseRecordBasic::setTypeGraceNote -- put a "g" in the first column.
//    shift pitch information over if it exists?  Maybe later.
//    Currently will destroy any pitch information.
//

void MuseRecordBasic::setTypeGraceNote(void) {
	setType(E_muserec_note_grace);
	(*this)[0] = 'g';
}



//////////////////////////////
//
// MuseRecordBasic::setTypeGraceChordNote -- put a "g" in the first column,
//    and a space in the second column.  Shift pitch information over if
//    it exists?  Maybe later.  Currently will destroy any pitch information.
//

void MuseRecordBasic::setTypeGraceChordNote(void) {
	setType(E_muserec_note_grace_chord);
	(*this)[0] = 'g';
	(*this)[1] = ' ';
}



//////////////////////////////
//
// MuseRecordBasic::shrink -- removes trailing spaces in a MuseData record
//

void MuseRecordBasic::shrink(void) {
	int i = (int)m_recordString.size() - 1;
	while (i >= 0 && m_recordString[i] == ' ') {
		m_recordString.resize((int)m_recordString.size()-1);
		i--;
	}
}



//////////////////////////////
//
// MuseRecordBasic::insertString --
//

void MuseRecordBasic::insertString(int column, const string& strang) {
	int len = (int)strang.size();
	if (len == 0) {
		return;
	}
	int index = column - 1;
	// make sure that record has text data up to the end of sring in
	// final location by preallocating the end location of string:
	(*this)[index+len-1] = ' ';
	int i;
	for (i=0; i<len; i++) {
		(*this)[i+index] = strang[i];
	}
}



//////////////////////////////
//
// MuseRecordBasic::insertStringRight -- Insert string right-justified
//    starting at given index.
//

void MuseRecordBasic::insertStringRight(int column, const string& strang) {
	int len = (int)strang.size();
	int index = column - 1;
	// make sure that record has text data up to the end of sring in
	// final location by preallocating the end location of string:
	(*this)[index] = ' ';
	int i;
	int ii;
	for (i=0; i<len; i++) {
		ii = index - i;
		if (ii < 0) {
			break;
		}
		(*this)[ii] = strang[len-i-1];
	}
}



//////////////////////////////
//
// MuseRecordBasic::appendString -- add a string to the end of the current
//     data in the record.
//

void MuseRecordBasic::appendString(const string& astring) {
	insertString(getLength()+1, astring);
}



//////////////////////////////
//
// MuseRecord::appendInteger -- Insert an integer after the last character
//     in the current line.
//

void MuseRecordBasic::appendInteger(int value) {
	string buffer = to_string(value);
	insertString(getLength()+1, buffer);
}



//////////////////////////////
//
// MuseRecord::appendRational -- Insert a rational after the last character
//     in the current line.
//

void MuseRecordBasic::appendRational(HumNum& value) {
	stringstream tout;
	value.printTwoPart(tout);
	tout << ends;
	insertString(getLength()+1, tout.str());
}



//////////////////////////////
//
// MuseRecord::append -- append multiple objects in sequence
// from left to right onto the record.  The format contains
// characters with two possibilities at the moment:
//    "i": integer value
//    "s": string value
//

void MuseRecordBasic::append(const char* format, ...) {
	va_list valist;
	int     i;

	va_start(valist, format);

	union Format_t {
		int   i;
		char *s;
		int  *r;  // array of two integers for rational number
	} FormatData;

	HumNum rn;

	int len = strlen(format);
	for (i=0; i<len; i++) {
		switch (format[i]) {   // Type to expect.
			case 'i':
				FormatData.i = va_arg(valist, int);
				appendInteger(FormatData.i);
				break;

			case 's':
				FormatData.s = va_arg(valist, char *);
				if (strlen(FormatData.s) > 0) {
					appendString(FormatData.s);
				}
				break;

			case 'r':
				 FormatData.r = va_arg(valist, int *);
				 rn.setValue(FormatData.r[0], FormatData.r[1]);
				 appendRational(rn);
				break;

			default:
				// don't put any character other than "i", "r" or "s"
				// in the format string
				break;
		}
	}

	va_end(valist);
}



//////////////////////////////
//
// MuseRecordBasic::setString --
//

void MuseRecordBasic::setString(string& astring) {
	m_recordString = astring;
}



//////////////////////////////
//
// MuseRecordBasic::setAbsBeat --
//


void MuseRecordBasic::setAbsBeat(HumNum value) {
	m_absbeat = value;
}


// default value botval = 1
void MuseRecordBasic::setAbsBeat(int topval, int botval) {
	m_absbeat.setValue(topval, botval);
}



//////////////////////////////
//
// MuseRecordBasic::getAbsBeat --
//

HumNum MuseRecordBasic::getAbsBeat(void) {
	return m_absbeat;
}



//////////////////////////////
//
// MuseRecordBasic::setLineDuration -- set the duration of the line
//     in terms of quarter notes.
//

void MuseRecordBasic::setLineDuration(HumNum value) {
	m_lineduration = value;
}


// default value botval = 1
void MuseRecordBasic::setLineDuration(int topval, int botval) {
	m_lineduration.setValue(topval, botval);
}



//////////////////////////////
//
// MuseRecordBasic::getLineDuration -- set the duration of the line
//     in terms of quarter notes.
//

HumNum MuseRecordBasic::getLineDuration(void) {
	return m_lineduration;
}



//////////////////////////////
//
// MuseRecordBasic::setNoteDuration -- set the duration of the note
//     in terms of quarter notes.  If the line does not represent a note,
//     then the note duration should probably be 0...
//

void MuseRecordBasic::setNoteDuration(HumNum value) {
	m_noteduration = value;
}


// default value botval = 1
void MuseRecordBasic::setNoteDuration(int topval, int botval) {
	m_noteduration.setValue(topval, botval);
}



//////////////////////////////
//
// MuseRecordBasic::getNoteDuration -- get the duration of the note
//     in terms of quarter notes.  If the line does not represent a note,
//     then the note duration should probably be 0...
//

HumNum MuseRecordBasic::getNoteDuration(void) {
	return m_noteduration;
}



//////////////////////////////
//
// MuseRecordBasic::setLineIndex --
//

void MuseRecordBasic::setLineIndex(int index) {
	m_lineindex = index;
}



//////////////////////////////
//
// MuseRecordBasic::isTied -- True if the note is tied to a note
//    before or after it.
//  0 = no ties
//  1 = tied to previous note
//  2 = tied to future note
//  3 = tied to both previous and future note
//

int MuseRecordBasic::isTied(void) {
	int output = 0;
	if (getLastTiedNoteLineIndex() >= 0) {
		output += 1;
	}

	if (getNextTiedNoteLineIndex() >= 0) {
		output += 2;
	}

	return output;
}



//////////////////////////////
//
// MuseRecordBasic::getLastTiedNoteLineIndex --
//


int MuseRecordBasic::getLastTiedNoteLineIndex(void) {
	return m_lasttiednote;
}



//////////////////////////////
//
// MuseRecordBasic::getNextTiedNoteLineIndex --
//


int MuseRecordBasic::getNextTiedNoteLineIndex(void) {
	return m_nexttiednote;
}



//////////////////////////////
//
// MuseRecordBasic::setLastTiedNoteLineIndex --
//


void MuseRecordBasic::setLastTiedNoteLineIndex(int index) {
	m_lasttiednote = index;
}



//////////////////////////////
//
// MuseRecordBasic::setNextTiedNoteLineIndex --
//


void MuseRecordBasic::setNextTiedNoteLineIndex(int index) {
	m_nexttiednote = index;
}



//////////////////////////////
//
// MuseRecordBasic::setRoundedBreve -- set double whole notes rounded flag.
//

void MuseRecordBasic::setRoundedBreve(void) {
	m_roundBreve = 1;
}



//////////////////////////////
//
// MuseRecordBasic::setMarkupPitch -- set the base-40 pitch information
//   in the markup area.  Does not change the original pitch in
//   the text line of the data.
//

void MuseRecordBasic::setMarkupPitch(int aPitch) {
	m_b40pitch = aPitch;
}



//////////////////////////////
//
// MuseRecordBasic::getMarkupPitch -- get the base-40 pitch information
//   in the markup area.  Does not look at the original pitch in
//   the text line of the data. A negative value is a rest (or invalid).
//

int MuseRecordBasic::getMarkupPitch(void) {
	return m_b40pitch;
}



//////////////////////////////
//
// MuseRecordBasic::cleanLineEnding -- remove spaces at the end of the
//    line;
//

void MuseRecordBasic::cleanLineEnding(void) {
	int i = (int)m_recordString.size() - 1;
	// Don't remove first space on line.
	while ((i > 0) && (m_recordString[i] == ' ')) {
		m_recordString.resize((int)m_recordString.size() - 1);
		i = (int)m_recordString.size() - 1;
	}
}



//////////////////////////////
//
// MuseRecordBasic::isPartName --
//

bool MuseRecordBasic::isPartName(void) {
	return m_type == E_muserec_header_part_name;
}



//////////////////////////////
//
// MuseRecordBasic::isAttributes --
//

bool MuseRecordBasic::isAttributes(void) {
	return m_type == E_muserec_musical_attributes;
}



//////////////////////////////
//
// MuseRecordBasic::isSource --
//

bool MuseRecordBasic::isSource(void) {
	return m_type == E_muserec_source;
}



//////////////////////////////
//
// MuseRecordBasic::isEncoder --
//

bool MuseRecordBasic::isEncoder(void) {
	return m_type == E_muserec_encoder;
}



//////////////////////////////
//
// MuseRecordBasic::isId --
//

bool MuseRecordBasic::isId(void) {
	return m_type == E_muserec_id;
}



//////////////////////////////
//
// MuseRecordBasic::isBarline --
//

bool MuseRecordBasic::isBarline(void) {
	return m_type == E_muserec_measure;
}



//////////////////////////////
//
// MuseRecordBasic::isBackup --
//

bool MuseRecordBasic::isBackup(void) {
	return m_type == E_muserec_back;
}



//////////////////////////////
//
// MuseRecordBasic::isAnyComment --
//

bool MuseRecordBasic::isAnyComment(void) {
	return isLineComment() || isBlockComment();
}



//////////////////////////////
//
// MuseRecordBasic::isLineComment --
//

bool MuseRecordBasic::isLineComment(void) {
	return m_type == E_muserec_comment_line;
}



//////////////////////////////
//
// MuseRecordBasic::isBlockComment --
//

bool MuseRecordBasic::isBlockComment(void) {
	return m_type == E_muserec_comment_toggle;
}



//////////////////////////////
//
// MuseRecordBasic::isChordNote -- Is a regular note that is a seoncdary
//    note in a chord (not the first note in the chord).
//

bool MuseRecordBasic::isChordNote(void) {
	return m_type == E_muserec_note_chord;
}



//////////////////////////////
//
// MuseRecordBasic::isDirection -- Is a musical direction (text)
//     instruction.
//

bool MuseRecordBasic::isDirection(void) {
	return m_type == E_muserec_musical_directions;
}



//////////////////////////////
//
// MuseRecordBasic::isGraceNote -- A grace note, either a single note or
//     the first note in a gracenote chord.
//

bool MuseRecordBasic::isGraceNote(void) {
	return m_type == E_muserec_note_grace;
}



//////////////////////////////
//
// MuseRecordBasic::isCueNote --
//

bool MuseRecordBasic::isCueNote(void) {
	return m_type == E_muserec_note_cue;
}



//////////////////////////////
//
// MuseRecordBasic::isChordNote --
//

bool MuseRecordBasic::isChordGraceNote(void) {
	return m_type == E_muserec_note_grace_chord;
}



//////////////////////////////
//
// MuseRecordBasic::isFiguredHarmony --
//

bool MuseRecordBasic::isFiguredHarmony(void) {
	return m_type == E_muserec_figured_harmony;
}



//////////////////////////////
//
// MuseRecordBasic::isRegularNote --
//

bool MuseRecordBasic::isRegularNote(void) {
	switch (m_type) {
		case E_muserec_note_regular:
			return true;
	}
	return false;
}


//////////////////////////////
//
// MuseRecordBasic::isAnyNote --
//

bool MuseRecordBasic::isAnyNote(void) {
	switch (m_type) {
		case E_muserec_note_regular:
		case E_muserec_note_chord:
		case E_muserec_note_cue:
		case E_muserec_note_grace:
		case E_muserec_note_grace_chord:
			return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecordBasic::isAnyNoteOrRest --
//

bool MuseRecordBasic::isAnyNoteOrRest(void) {
	switch (m_type) {
		case E_muserec_note_regular:
		case E_muserec_note_chord:
		case E_muserec_note_cue:
		case E_muserec_note_grace:
		case E_muserec_note_grace_chord:
		case E_muserec_rest_invisible:
		case E_muserec_rest:
			return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecordBasic::isInvisibleRest --
//

bool MuseRecordBasic::isInvisibleRest(void) {
	switch (m_type) {
		case E_muserec_rest_invisible:
			return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecordBasic::isRegularRest --
//

bool MuseRecordBasic::isRegularRest(void) {
	switch (m_type) {
		case E_muserec_rest:
			return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecordBasic::isAnyRest -- Also cue-sized rests?
//

bool MuseRecordBasic::isAnyRest(void) {
	switch (m_type) {
		case E_muserec_rest_invisible:
		case E_muserec_rest:
			return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecordBasic::isCopyright --
//

bool MuseRecordBasic::isCopyright(void) {
	switch (m_type) {
		case E_muserec_copyright:
			return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecordBasic::isWorkInfo --
//

bool MuseRecordBasic::isWorkInfo(void) {
	switch (m_type) {
		case E_muserec_work_info:
			return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecordBasic::isWorkTitle --
//

bool MuseRecordBasic::isWorkTitle(void) {
	switch (m_type) {
		case E_muserec_work_title:
			return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecordBasic::isMovementTitle --
//

bool MuseRecordBasic::isMovementTitle(void) {
	switch (m_type) {
		case E_muserec_movement_title:
			return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecordBasic::isGroup --
//

bool MuseRecordBasic::isGroup(void) {
	switch (m_type) {
		case E_muserec_group:
			return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecordBasic::isGroupMembership --
//

bool MuseRecordBasic::isGroupMembership(void) {
	switch (m_type) {
		case E_muserec_group_memberships:
			return true;
	}
	return false;
}



//////////////////////////////
//
// MuseRecordBasic::isHeaderRecord -- True if a header, or a comment
//   occurring before the first non-header record.
//

bool MuseRecordBasic::isHeaderRecord(void) {
	return m_header > 0;
}



//////////////////////////////
//
// MuseRecordBasic::isBodyRecord -- True if not a header record.
//

bool MuseRecordBasic::isBodyRecord(void) {
	return m_header == 0;
}



//////////////////////////////
//
// MuseRecordBasic::trimSpaces --
//

string MuseRecordBasic::trimSpaces(std::string input) {
	string output;
	int status = 0;
	for (int i=0; i<(int)input.size(); i++) {
		if (!status) {
			if (isspace(input[i])) {
				continue;
			}
			status = 1;
		}
		output += input[i];
	}
	for (int i=(int)output.size()-1; i>=0; i--) {
		if (isspace(output[i])) {
			output.resize((int)output.size() - 1);
		} else {
			break;
		}
	}
	return output;
}



//////////////////////////////
//
// MuseRecordBasic::setHeaderState -- 1 = in header, 0 = in body, -1 = undefined.
//    Access with isHeaderRecord() and isBodyRecord().
//

void MuseRecordBasic::setHeaderState(int state) {
	if (state > 0) {
		m_header = 1;
	} else if (state < 0) {
		m_header = -1;
	} else {
		m_header = 0;
	}
}



//////////////////////////////
//
// MuseRecordBasic::setLayer -- Set the layer for the record.
//    This information is taken from the track parameter
//    of records, but may be inferred from its position in
//    relation to backup commands.  Zero means implicit layer 1.
//

void MuseRecordBasic::setLayer(int layer) {
	if (layer < 0) {
		m_layer = 0;
	} else {
		m_layer = layer;
	}
}



//////////////////////////////
//
// MuseRecordBasic::getLayer -- Get the layer for the record.
//    This information is taken from the track parameter
//    of records, but may be inferred from its position in
//    relation to backup commands.  Zero means implicit layer 1.
//

int MuseRecordBasic::getLayer(void) {
	return m_layer;
}



//////////////////////////////
//
// MuseRecordBasic:hasTpq --
//

bool MuseRecordBasic::hasTpq(void) {
	if (m_tpq) {
		return true;
	} else {
		return false;
	}
}



//////////////////////////////
//
// MuseRecordBasic:getTpq --
//

int MuseRecordBasic::getTpq(void) {
	return m_tpq;
}



//////////////////////////////
//
// MuseRecordBasic:setTpq --
//

void MuseRecordBasic::setTpq(int value) {
	if (value <= 0) {
		m_tpq = 0;
	} else {
		m_tpq = value;
	}
}



//////////////////////////////
//
// MuseRecordBasic:setVoice --
//

void MuseRecordBasic::setVoice(GridVoice* voice) {
	m_voice = voice;
}



//////////////////////////////
//
// MuseRecordBasic:getVoice --
//

GridVoice* MuseRecordBasic::getVoice(void) {
	return m_voice;
}



//////////////////////////////
//
// MuseRecordBasic:LayoutVis -- Return the graphical display of the
//    rhythm for the note/rest if it was different than the logical version.
//    This is temporary storage for inserting a layout command into the
//    MuseData-to-Humdrum converter (tool-musedata2hum.cpp).
//

std::string MuseRecordBasic::getLayoutVis(void) {
	return m_graphicrecip;
}



//////////////////////////////
//
// MuseRecordBasic::musedataToUtf8 --
//

string MuseRecordBasic::musedataToUtf8(string& input) {
	string output;
	int isize = (int)input.size();
	for (int i=0; i<isize; i++) {
		if (input[i] != '\\') {
			output += input[i];
			continue;
		}
		if (i+2 >= isize) {
			output += input[i];
			continue;
		}
		string st = input.substr(i+1, 2);

		// graves
		if (st == "A8") { output += (char)0xc3; output += (char)0x80; i+=2; continue; }
		if (st == "E8") { output += (char)0xc3; output += (char)0x88; i+=2; continue; }
		if (st == "I8") { output += (char)0xc3; output += (char)0x8c; i+=2; continue; }
		if (st == "O8") { output += (char)0xc3; output += (char)0x92; i+=2; continue; }
		if (st == "U8") { output += (char)0xc3; output += (char)0x99; i+=2; continue; }
		if (st == "a8") { output += (char)0xc3; output += (char)0xa0; i+=2; continue; }
		if (st == "e8") { output += (char)0xc3; output += (char)0xa8; i+=2; continue; }
		if (st == "i8") { output += (char)0xc3; output += (char)0xac; i+=2; continue; }
		if (st == "o8") { output += (char)0xc3; output += (char)0xb2; i+=2; continue; }
		if (st == "u8") { output += (char)0xc3; output += (char)0xb9; i+=2; continue; }

		// acutes
		if (st == "A7") { output += (char)0xc3; output += (char)0x81; i+=2; continue; }
		if (st == "E7") { output += (char)0xc3; output += (char)0x89; i+=2; continue; }
		if (st == "I7") { output += (char)0xc3; output += (char)0x8d; i+=2; continue; }
		if (st == "O7") { output += (char)0xc3; output += (char)0x93; i+=2; continue; }
		if (st == "U7") { output += (char)0xc3; output += (char)0x9a; i+=2; continue; }
		if (st == "a7") { output += (char)0xc3; output += (char)0xa1; i+=2; continue; }
		if (st == "e7") { output += (char)0xc3; output += (char)0xa9; i+=2; continue; }
		if (st == "i7") { output += (char)0xc3; output += (char)0xad; i+=2; continue; }
		if (st == "o7") { output += (char)0xc3; output += (char)0xb3; i+=2; continue; }
		if (st == "u7") { output += (char)0xc3; output += (char)0xba; i+=2; continue; }

		// umlauts
		if (st == "A3") { output += (char)0xc3; output += (char)0x84; i+=2; continue; }
		if (st == "E3") { output += (char)0xc3; output += (char)0x8b; i+=2; continue; }
		if (st == "I3") { output += (char)0xc3; output += (char)0x8f; i+=2; continue; }
		if (st == "O3") { output += (char)0xc3; output += (char)0x96; i+=2; continue; }
		if (st == "U3") { output += (char)0xc3; output += (char)0x9c; i+=2; continue; }
		if (st == "a3") { output += (char)0xc3; output += (char)0xa4; i+=2; continue; }
		if (st == "e3") { output += (char)0xc3; output += (char)0xab; i+=2; continue; }
		if (st == "i3") { output += (char)0xc3; output += (char)0xaf; i+=2; continue; }
		if (st == "o3") { output += (char)0xc3; output += (char)0xb6; i+=2; continue; }
		if (st == "u3") { output += (char)0xc3; output += (char)0xbc; i+=2; continue; }

		// other
		if (st == "s2") { output += (char)0xc3; output += (char)0x9f; i+=2; continue; }  // eszett

		// Older Musedata files reverse the number and letters:

		// graves
		if (st == "8A") { output += (char)0xc3; output += (char)0x80; i+=2; continue; }
		if (st == "8E") { output += (char)0xc3; output += (char)0x88; i+=2; continue; }
		if (st == "8I") { output += (char)0xc3; output += (char)0x8c; i+=2; continue; }
		if (st == "8O") { output += (char)0xc3; output += (char)0x92; i+=2; continue; }
		if (st == "8U") { output += (char)0xc3; output += (char)0x99; i+=2; continue; }
		if (st == "8a") { output += (char)0xc3; output += (char)0xa0; i+=2; continue; }
		if (st == "8e") { output += (char)0xc3; output += (char)0xa8; i+=2; continue; }
		if (st == "8i") { output += (char)0xc3; output += (char)0xac; i+=2; continue; }
		if (st == "8o") { output += (char)0xc3; output += (char)0xb2; i+=2; continue; }
		if (st == "8u") { output += (char)0xc3; output += (char)0xb9; i+=2; continue; }

		// acutes
		if (st == "7A") { output += (char)0xc3; output += (char)0x81; i+=2; continue; }
		if (st == "7E") { output += (char)0xc3; output += (char)0x89; i+=2; continue; }
		if (st == "7I") { output += (char)0xc3; output += (char)0x8d; i+=2; continue; }
		if (st == "7O") { output += (char)0xc3; output += (char)0x93; i+=2; continue; }
		if (st == "7U") { output += (char)0xc3; output += (char)0x9a; i+=2; continue; }
		if (st == "7a") { output += (char)0xc3; output += (char)0xa1; i+=2; continue; }
		if (st == "7e") { output += (char)0xc3; output += (char)0xa9; i+=2; continue; }
		if (st == "7i") { output += (char)0xc3; output += (char)0xad; i+=2; continue; }
		if (st == "7o") { output += (char)0xc3; output += (char)0xb3; i+=2; continue; }
		if (st == "7u") { output += (char)0xc3; output += (char)0xba; i+=2; continue; }

		// umlauts
		if (st == "3A") { output += (char)0xc3; output += (char)0x84; i+=2; continue; }
		if (st == "3E") { output += (char)0xc3; output += (char)0x8b; i+=2; continue; }
		if (st == "3I") { output += (char)0xc3; output += (char)0x8f; i+=2; continue; }
		if (st == "3O") { output += (char)0xc3; output += (char)0x96; i+=2; continue; }
		if (st == "3U") { output += (char)0xc3; output += (char)0x9c; i+=2; continue; }
		if (st == "3a") { output += (char)0xc3; output += (char)0xa4; i+=2; continue; }
		if (st == "3e") { output += (char)0xc3; output += (char)0xab; i+=2; continue; }
		if (st == "3i") { output += (char)0xc3; output += (char)0xaf; i+=2; continue; }
		if (st == "3o") { output += (char)0xc3; output += (char)0xb6; i+=2; continue; }
		if (st == "3u") { output += (char)0xc3; output += (char)0xbc; i+=2; continue; }

		// other
		if (st == "2s") { output += (char)0xc3; output += (char)0x9f; i+=2; continue; }  // eszett

	}

	return output;
}



///////////////////////////////////////////////////////////////////////////


//////////////////////////////
//
// operator<<
//

ostream& operator<<(ostream& out, MuseRecordBasic& aRecord) {
	aRecord.shrink();  // have to shrink automatically because
						    // muse2ps program chokes on line 9 of header
						    // if it has more than one space on a blank line.
	out << aRecord.getLine();
	return out;
}

ostream& operator<<(ostream& out, MuseRecordBasic* aRecord) {
	out << *aRecord;
	return out;
}



class MxmlMeasure;
class MxmlPart;

int MxmlEvent::m_counter = 0;

////////////////////////////////////////////////////////////////////////////


//////////////////////////////
//
// MxmlEvent::MxmlEvent -- Constructor.
//

MxmlEvent::MxmlEvent(MxmlMeasure* measure) {
	clear();
	m_owner = measure;
	m_sequence = m_counter++;
	m_stems = false;
}



//////////////////////////////
//
// MxmlEvent::~MxmlEvent -- Destructor.
//

MxmlEvent::~MxmlEvent() {
	clear();
}



//////////////////////////////
//
// MxmlEvent::clear -- Clear any previous contents of the object.
//

void MxmlEvent::clear(void) {
	m_starttime = m_duration = 0;
	m_modification = 1;
	m_eventtype = mevent_unknown;
	m_owner = NULL;
	m_linked = false;
	m_voice = -1;
	m_staff = 0;
	m_invisible = false;
	m_voiceindex = -1;
	m_sequence = -1;
	for (int i=0; i<(int)m_links.size(); i++) {
		delete m_links[i];
		m_links[i] = NULL;
	}
	m_links.resize(0);
}



///////////////////////////////
//
// MxmlEvent::enableStems --
//

void MxmlEvent::enableStems(void) {
	m_stems = true;
}



///////////////////////////////
//
// MxmlEvent::makeDummyRest --
//   default values:
//     staffindex = 0;
//     voiceindex = 0;
//

void MxmlEvent::makeDummyRest(MxmlMeasure* owner, HumNum starttime,
		HumNum duration, int staffindex, int voiceindex) {
	m_starttime = starttime;
	m_duration = duration;
	m_eventtype = mevent_forward;  // not a real rest (will be invisible)
	// m_node remains null
	// m_links remains empty
	m_linked = false;
	m_sequence = -m_counter;
	m_counter++;
	m_voice = 1;  // don't know what the original voice number is
	m_voiceindex = voiceindex;
	m_staff = staffindex + 1;
	m_maxstaff = m_staff;  // how is this used/set?
	//	m_hnode remains null
}



//////////////////////////////
//
// MxmlEvent::setStartTime -- Set the starting timestamp of the event
//    in terms of quater notes since the start of the music.
//

void MxmlEvent::setStartTime(HumNum value) {
	m_starttime = value;
}



//////////////////////////////
//
// MxmlEvent::setDuration -- Set the duration of the event in terms
//   of quarter note durations.
//

void MxmlEvent::setDuration(HumNum value) {
	m_duration = value;
}



//////////////////////////////
//
// MxmlEvent::getModification -- Get the tuplet scaling of the note's duration.
//

HumNum MxmlEvent::getModification(void) const {
	return m_modification;
}



//////////////////////////////
//
// MxmlEvent::setModification -- Set the tuplet scaling of the note's duration.
//

void MxmlEvent::setModification(HumNum value) {
	m_modification = value;
}



//////////////////////////////
//
// MxmlEvent::getStartTime -- Return the start time of the event in terms
//      of quarter notes since the start of the music.
//

HumNum MxmlEvent::getStartTime(void) const {
	return m_starttime;
}



//////////////////////////////
//
// MxmlEvent::getDuration -- Return the duration of the event in terms
//      of quarter note durations.
//

HumNum MxmlEvent::getDuration(void) const {
	return m_duration;
}



//////////////////////////////
//
// MxmlEvent::setOwner -- Indicate which measure the event belongs to.
//

void MxmlEvent::setOwner(MxmlMeasure* measure) {
	m_owner = measure;
}



//////////////////////////////
//
// MxmlEvent::getOwner -- Return the measure object that contains this
//     event.  If there is no owner, then returns NULL.
//

MxmlMeasure* MxmlEvent::getOwner(void) const {
	return m_owner;
}



//////////////////////////////
//
// MxmlEvent::reportVerseCountToOwner --
//

void MxmlEvent::reportVerseCountToOwner(int count) {
	if (!m_owner) {
		return;
	}
	m_owner->reportVerseCountToOwner(count);
}


void MxmlEvent::reportVerseCountToOwner(int staffindex, int count) {
	if (!m_owner) {
		return;
	}
	m_owner->reportVerseCountToOwner(staffindex, count);
}



//////////////////////////////
//
// MxmlEvent::reportDynamicToOwner -- inform the owner that there is a dynamic
//    that needs a spine to store it in.
//

void MxmlEvent::reportDynamicToOwner(void) {
	m_owner->reportDynamicToOwner();
}



//////////////////////////////
//
// MxmlEvent::reportFiguredBassToOwner -- inform the owner that there is a dynamic
//    that needs a spine to store it in.
//

void MxmlEvent::reportFiguredBassToOwner(void) {
	m_owner->reportFiguredBassToOwner();
}



//////////////////////////////
//
// MxmlEvent::reportCaesuraToOwner -- inform the owner that there is a caesura
//    that needs an RDF marker.
// default value: letter = "Z"
//

void MxmlEvent::reportCaesuraToOwner(const string& letter) const {
	m_owner->reportCaesuraToOwner(letter);
}



//////////////////////////////
//
// MxmlEvent::reportOrnamentToOwner --
//

void MxmlEvent::reportOrnamentToOwner(void) const {
	m_owner->reportOrnamentToOwner();
}



//////////////////////////////
//
// MxmlEvent::reportHarmonyCountToOwner --
//

void MxmlEvent::reportHarmonyCountToOwner(int count) {
	if (!m_owner) {
		return;
	}
	m_owner->reportHarmonyCountToOwner(count);
}



//////////////////////////////
//
// MxmlEvent::reportMeasureStyleToOwner --
//

void MxmlEvent::reportMeasureStyleToOwner (MeasureStyle style) {
	if (!m_owner) {
		return;
	}
	m_owner->receiveMeasureStyleFromChild(style);
}



//////////////////////////////
//
// MxmlEvent::reportEditorialAccidentalToOwner --
//

void MxmlEvent::reportEditorialAccidentalToOwner(void) {
	if (!m_owner) {
		return;
	}
	m_owner->receiveEditorialAccidentalFromChild();
}



//////////////////////////////
//
// MxmlEvent::getPartNumber --
//

int MxmlEvent::getPartNumber(void) const {
	if (!m_owner) {
		return 0;
	}
	return m_owner->getPartNumber();
}



//////////////////////////////
//
// MxmlEvent::getPartIndex --
//

int MxmlEvent::getPartIndex(void) const {
	if (!m_owner) {
		return 0;
	}
	return m_owner->getPartIndex();
}



//////////////////////////////
//
// MxmlEvent::getName --
//

const char* MxmlEvent::getName(void) const {
	return m_node.name();
}



//////////////////////////////
//
// MxmlEvent::setQTicks -- Set the number of ticks per quarter note.
//     Returns the number of times that the ticks has been set.
//     Returns 0 if the tick count is invalid.
//

int MxmlEvent::setQTicks(long value) {
	if (value <= 0) {
		return 0;
	}
	if (m_owner) {
		return m_owner->setQTicks(value);
	} else {
		return 0;
	}
}



//////////////////////////////
//
// MxmlEvent::getQTicks -- Get the number of ticks per quarter note.
//

long MxmlEvent::getQTicks(void) const {
	if (m_owner) {
		return m_owner->getQTicks();
	} else {
		return 0;
	}
}



//////////////////////////////
//
// MxmlEvent::getIntValue -- Convenience function for an XPath query,
//    where the child text of the element should be interpreted as
//    an integer.
//

long MxmlEvent::getIntValue(const char* query) const {
	const char* val = m_node.select_node(query).node().child_value();
	if (strcmp(val, "") == 0) {
		return 0;
	} else {
		return atoi(val);
	}
}



//////////////////////////////
//
// Mxmlvent::setDurationByTicks -- Given a <duration> element tick
//    count, set the duration by dividing by the current quarter-note
//    duration tick count (from a prevailing attribute setting for
//    <divisions>).
//

void MxmlEvent::setDurationByTicks(long value, xml_node el) {
	long ticks = getQTicks();
	if (ticks == 0) {
		setDuration(0);
		return;
	}

	if (isGrace()) {
		setDuration(0);
		return;
	}

	HumNum val = (int)value;
	val /= (int)ticks;

	HumNum modification;
	if (el) {
		HumNum checkval = getEmbeddedDuration(modification, el);
		if ((checkval == 0) && isRest()) {
			// This is a whole rest.
			// val = val
		} else if (checkval != val) {
			// cerr << "WARNING: True duration " << checkval << " does not match";
			// cerr << " tick duration (buggy data: " << val << ")" << endl;
			double difference = fabs(checkval.getFloat() - val.getFloat());
			if (difference < 0.1) {
				// only correct if the duration is small, since some programs
				// will mark rests such as half notes as whole notes (since they
				// are displayed as centered whole notes)
				val = checkval;
			}
		}
	}
	setDuration(val);
	setModification(modification);
}



//////////////////////////////
//
// MxmlEvent::hasChild -- True if the given XPath query resulting
//      element has a child node.
//

bool MxmlEvent::hasChild(const char* query) const {
	xpath_node result = m_node.select_node(query);
	return !result.node().empty();
}



//////////////////////////////
//
// MxmlEvent::attachToLast --
//

void MxmlEvent::attachToLastEvent(void) {
	if (!m_owner) {
		return;
	}
	m_owner->attachLastEventToPrevious();
}



//////////////////////////////
//
// MxmlEvent::link --  This function is used to link secondary
//   elements to a primary one.  Currently only used for chord notes.
//   The first note of a chord will be stored in event lists, and
//   secondary notes will be suppressed from the list and instead
//   accessed through the m_links structure.
//

void MxmlEvent::link(MxmlEvent* event) {
	m_links.push_back(event);
	event->setLinked();
}



//////////////////////////////
//
// MxmlEvent::setLinked -- Indicate that a note is a secondary
//     chord note.
//

void MxmlEvent::setLinked(void) {
	m_linked = true;
}



//////////////////////////////
//
// MxmlEvent::isLinked -- Returns true if the note is a secondary
//     chord note.
//

bool MxmlEvent::isLinked(void) const {
	return m_linked;
}



//////////////////////////////
//
// MxmlEvent::isRest --
//

bool MxmlEvent::isRest(void) {
	if (!m_node) {
		return false;
	}
	xml_node child = m_node.first_child();
	while (child) {
		if (nodeType(child, "rest")) {
			return true;
		}
		child = child.next_sibling();
	}
	return false;
}



//////////////////////////////
//
// MxmlEvent::isChord -- Returns true if the event is the primary note
//    in a chord.
//

bool MxmlEvent::isChord(void) const {
	if ((m_links.size() > 0) && nodeType(m_node, "note")) {
		return true;
	} else {
		return false;
	}
}



//////////////////////////////
//
// MxmlEvent::isGrace -- Returns true if the event is a grace note.
//

bool MxmlEvent::isGrace(void) {
	xml_node child = this->getNode();
	if (!nodeType(child, "note")) {
		return false;
	}
	child = child.first_child();
	while (child) {
		if (nodeType(child, "grace")) {
			return true;
		} else if (nodeType(child, "pitch")) {
			// grace element has to come before pitch
			return false;
		}
		child = child.next_sibling();
	}
	return false;
}



//////////////////////////////
//
// MxmlEvent::hasGraceSlash -- Returns true if the note is a grace note
//    with a slash.
//

bool MxmlEvent::hasGraceSlash(void) {
	xml_node child = this->getNode();
	if (!nodeType(child, "note")) {
		return false;
	}
	child = child.first_child();
	while (child) {
		if (nodeType(child, "grace")) {
			string slash = child.attribute("slash").value();
			if (slash == "yes") {
				return true;
			} else {
				return false;
			}
		} else if (nodeType(child, "pitch")) {
			// grace element has to come before pitch
			return false;
		}
		child = child.next_sibling();
	}
	return false;
}





//////////////////////////////
//
// MxmlEvent::hasSlurStart -- Returns 0 if no slur; otherwise, return the
//  number of slurs attached.  Currently ignoring slur@number.
//
//   number: used to keep track of overlapping slurs. (currently ignored).
//
//   direction: 0=unspecified, 1=positive curvature, -1=negative curvature.
//
//  <note>
//     <notations>
//         <slur type="start" orientation="under" number="1">
//         <slur type="start" orientation="over" number="2">
//
//  And also:
//
//  <note>
//     <notations>
//          <slur number="1" placement="above" type="start"/>
//          <slur number="s" placement="below" type="start"/>
//

int MxmlEvent::hasSlurStart(vector<int>& directions) {
	directions.clear();
	int output = 0;
	xml_node child = this->getNode();
	if (!nodeType(child, "note")) {
		return output;
	}
	child = child.first_child();
	while (child) {
		if (nodeType(child, "notations")) {
			xml_node grandchild = child.first_child();
			while (grandchild) {
				if (nodeType(grandchild, "slur")) {
					xml_attribute slurtype = grandchild.attribute("type");
					if (slurtype) {
						if (strcmp(slurtype.value(), "start") == 0) {
							output++;
						} else {
							grandchild = grandchild.next_sibling();
							continue;
						}
					}
					// for now ignore the slur numbers
					//string number = grandchild.attribute("number").value();
					//if (!number.empty()) {
					//	int num = stoi(number);
					//	if (num != 0) {
					//		xxx = num;
					//	}
					//}
					xml_attribute orientation = grandchild.attribute("orientation");
					int dir = 0;
					if (orientation) {
						if (strcmp(orientation.value(), "over") == 0) {
							dir = 1;
						} else if (strcmp(orientation.value(), "under") == 0) {
							dir = -1;
						}
					}
					xml_attribute placement = grandchild.attribute("placement");
					if (placement) {
						if (strcmp(placement.value(), "above") == 0) {
							dir = 1;
						} else if (strcmp(placement.value(), "below") == 0) {
							dir = -1;
						}
					}

					directions.push_back(dir);
				}
				grandchild = grandchild.next_sibling();
			}
		}
		child = child.next_sibling();
	}
	return output;
}



//////////////////////////////
//
// MxmlEvent::hasSlurStop -- Handles multiple stop on the same note now.
//
//  <note>
//     <notations>
//         <slur type="start" orientation="under" number="1">
//

int MxmlEvent::hasSlurStop(void) {
	int output = 0;
	xml_node child = this->getNode();
	if (!nodeType(child, "note")) {
		// maybe allow for other items such as rests?
		return output;
	}
	child = child.first_child();
	while (child) {
		if (nodeType(child, "notations")) {
			xml_node grandchild = child.first_child();
			while (grandchild) {
				if (nodeType(grandchild, "slur")) {
					xml_attribute slurtype = grandchild.attribute("type");
					if (slurtype) {
						if (strcmp(slurtype.value(), "stop") == 0) {
							output++;
							//string number = grandchild.attribute("number").value();
							//if (!number.empty()) {
							//	int num = stoi(number);
							//	if (num != 0) {
							//		return num;
							//	} else {
							//		return 1;
							//	}
							//}
						}
					}
				}
				grandchild = grandchild.next_sibling();
			}
		}
		child = child.next_sibling();
	}
	return output;
}



//////////////////////////////
//
// MxmlEvent::isFloating -- For a harmony or basso continuo item
//     which is not attached to a note onset.
//

bool MxmlEvent::isFloating(void) {
	xml_node empty = xml_node(NULL);
	if (m_node == empty && (m_hnode != empty)) {
		return true;
	} else {
		return false;
	}
}



//////////////////////////////
//
// MxmlEvent::getLinkedNotes --
//

vector<MxmlEvent*> MxmlEvent::getLinkedNotes(void) {
	return m_links;
}



//////////////////////////////
//
// MxmlEvent::print -- Useful for debugging.
//

ostream& MxmlEvent::print(ostream& out) {
	out << getStartTime() << "\t" << getDuration() << "\t" << m_node.name();
	if (isChord()) {
		out << "\tCHORD";
	}
	out << endl;
	return out;
}



//////////////////////////////
//
// MxmlEvent::getSequenceNumber -- Return the sequence number of the
//   event in the input data file.  Useful for sorting items which
//   occur at the same time.
//

int MxmlEvent::getSequenceNumber(void) const {
	return m_sequence;
}



//////////////////////////////
//
// MxmlEvent::getVoiceNumber -- Return the voice number of the event.
//

int MxmlEvent::getVoiceNumber(void) const {
	if (m_voice) {
		return m_voice;
	} else {
		return 1;
	}
}



//////////////////////////////
//
// MxmlEvent::setVoiceIndex --
//

void MxmlEvent::setVoiceIndex(int index) {
	m_voiceindex = index;
}



//////////////////////////////
//
// MxmlEvent::getVoiceIndex -- Return the voice number of the event.
//    But mod 4 which presumably sets the voice number on a staff.
//    This is not always true: "PrintMusic 2010 for Windows" may
//    use voice 2 for staff 2. In this case the voice index should
//    be calculated by %2 rather than %4.
//    default value: maxvoice = 4.
//
//    This function will replace with a query to MxmlPart
//    as to what the voice on a staff should be.
//

int MxmlEvent::getVoiceIndex(int maxvoice) const {
	if (m_voiceindex >= 0) {
		return m_voiceindex;
	}

	if (m_owner) {
		int voiceindex = m_owner->getVoiceIndex(m_voice);
		if (voiceindex >= 0) {
			return voiceindex;
		}
	}

	// the following case handles notes/rests which do not contain
	// a voice number.  Assume that this item should be placed
	// in the first voice.
	if (m_voiceindex < 0) {
		if (nodeType(m_node, "note")) {
			return 0;
		}
	}


	// don't know what the voice mapping is, so make one up:
	if (maxvoice < 1) {
		maxvoice = 4;
	}
	if (m_voice) {
		return (m_voice - 1) % maxvoice;
	} else {
		return 0;
	}
}



//////////////////////////////
//
// MxmlEvent::forceInvisible --
//

void MxmlEvent::forceInvisible(void) {
	m_invisible = true;
}



//////////////////////////////
//
// MxmlEvent::isInvisible --
//

bool MxmlEvent::isInvisible(void) {
	return m_invisible;
}



//////////////////////////////
//
// MxmlEvent::getStaffIndex --
//

int MxmlEvent::getStaffIndex(void) const {
	if (m_staff > 0) {
		return m_staff - 1;
	}
	if (m_owner) {
		int staffindex = m_owner->getStaffIndex(m_voice);
		if (staffindex >= 0) {
			return staffindex;
		}
	}

	// don't know what the modified staff is, so give the original staff index:
	if (!m_staff) {
		return 0;
	} else {
		return m_staff - 1;
	}
}



//////////////////////////////
//
// MxmlEvent::setVoiceNumber --
//

void MxmlEvent::setVoiceNumber(int value) {
	m_voice = (short)value;
}



//////////////////////////////
//
// MxmlEvent::setStaffNumber --
//

void MxmlEvent::setStaffNumber(int value) {
	m_staff = (short)value;
}



//////////////////////////////
//
// MxmlEvent::getStaffNumber --
//

int MxmlEvent::getStaffNumber(void) const {
	if (!m_staff) {
		return 1;
	} else {
		return m_staff;
	}
}



//////////////////////////////
//
// MxmlEvent::getType --
//

measure_event_type MxmlEvent::getType(void) const {
	return m_eventtype;
}



//////////////////////////////
//
// MxmlEvent::parseEvent --
//

bool MxmlEvent::parseEvent(xpath_node el, HumNum starttime) {
	return parseEvent(el.node(), xml_node(NULL), starttime);
}


bool MxmlEvent::parseEvent(xml_node el, xml_node nextel, HumNum starttime) {
	m_node = el;

	bool floatingharmony = false;
	if (nodeType(m_node, "attributes")) {
		m_eventtype = mevent_attributes;
	} else if (nodeType(m_node, "backup")) {
		m_eventtype = mevent_backup;
	} else if (nodeType(m_node, "barline")) {
		m_eventtype = mevent_barline;
		setBarlineStyle(m_node);
	} else if (nodeType(m_node, "bookmark")) {
		m_eventtype = mevent_bookmark;
	} else if (nodeType(m_node, "direction")) {
		m_eventtype = mevent_direction;
	} else if (nodeType(m_node, "figured-bass")) {
		m_eventtype = mevent_figured_bass;
	} else if (nodeType(m_node, "forward")) {
		m_eventtype = mevent_forward;
		m_staff = -1; // set default staff if not supplied
		m_voice = -1; // set default staff if not supplied
	} else if (nodeType(m_node, "grouping")) {
		m_eventtype = mevent_grouping;
	} else if (nodeType(m_node, "harmony")) {
		m_eventtype = mevent_harmony;
		if (!nodeType(nextel, "note")) {
			// harmony is not attached to a note
			floatingharmony = true;
			m_staff = -1;
			m_voice = -1;
		}
	} else if (nodeType(m_node, "link")) {
		m_eventtype = mevent_link;
	} else if (nodeType(m_node, "note")) {
		m_eventtype = mevent_note;
		m_staff = 1; // set default staff if not supplied
		m_voice = -1; // set default staff if not supplied
	} else if (nodeType(m_node, "print")) {
		m_eventtype = mevent_print;
	} else if (nodeType(m_node, "sound")) {
		m_eventtype = mevent_sound;
	} else {
		m_eventtype = mevent_unknown;
	}

	int tempstaff    = 1;
	int tempvoice    = -1;
	int tempduration = 0;
	for (auto el = m_node.first_child(); el; el = el.next_sibling()) {
		if (nodeType(el, "staff")) {
			tempstaff = atoi(el.child_value());
		} else if (nodeType(el, "voice")) {
			tempvoice = atoi(el.child_value());
		} else if (nodeType(el, "duration")) {
			tempduration = atoi(el.child_value());
			// Duration must be set to 0 for figured bass.  But maybe need
			// duration to create line extensions.  Probably other elements
			// which are not notes should also have their durations set
			// to zero.
			if (nodeType(m_node, "figured-bass")) {
				tempduration = 0;
			}
		}
	}

	bool emptyvoice = false;
	if (!floatingharmony) {
		if (tempvoice < 0) {
			emptyvoice = true;
			if (nodeType(el, "note")) {
				this->setVoiceIndex(0);
			}
		}
	}

	if (m_eventtype == mevent_forward) {
		xml_node pel = el.previous_sibling();
		if (nodeType(pel, "harmony")) {
			// This is a spacer forward which is not in any voice/layer,
			// so invalidate is staff/voice to prevent it from being
			// converted to a rest.
			m_voice = -1;
			tempvoice = -1;
			m_staff = -1;
			tempstaff = -1;
		} else {
			// xml_node nel = el.next_sibling();
			// Need to check if the forward element should be interpreted
			// as an invisible rests.  Check to see if the previous and next
			// element are notes.  If so, then check their voice numbers and
			// if equal, then this forward element should be an invisible rest.
			// But this is true only if there is no other event happening
			// at the current position in the other voice(s) on the staff (or
			// perhaps include other staves on the system and/or part.

			// So this case might need to be addressed at a later stage when
			// the score is assembled, such as when adding null tokens, and a
			// null spot is located in the score.
		}
	}

	if (tempvoice >= 0) {
		m_voice = (short)tempvoice;
	}
	if (tempstaff > 0) {
		m_staff = (short)tempstaff;
	}
	if (!emptyvoice) {
   	reportStaffNumberToOwner(m_staff, m_voice);
	} else {
		// no voice child element, or not a note or rest.
	}
	HumNum timesigdur;
	HumNum difference;
	HumNum dur;
	MxmlMeasure* measure = getOwner();
	HumNum mst;
	if (measure) {
		mst = measure->getStartTime();
	}

	setStartTime(starttime);

	switch (m_eventtype) {
		case mevent_note:
			setDuration(0);
			if (hasChild("./chord")) {
				setDuration(0);
				attachToLastEvent();
			} else {
				setDurationByTicks(tempduration, el);
			}
			break;

		case mevent_forward:
			if (tempduration == 1) {
				// handle errors in SharpEye:
				long ticks = getQTicks();
				if ((double)tempduration / (double)ticks < 0.0001) {
					tempduration = 0;
					m_eventtype = mevent_unknown;
				}
			} else if (tempduration < 4) {
				// Warn about possible other errors:
				double fraction = (double)tempduration / getQTicks();
				if (fraction < 0.01) {
					cerr << "WARNING: FORWARD WITH A SMALL VALUE " << tempduration << endl;
				}
			}
			setDurationByTicks(tempduration);
			break;

		case mevent_backup:
			setDurationByTicks(-tempduration);
			dur = getDuration();
			difference = starttime - mst + dur;
			if (difference < 0) {
				// cerr << "Warning: backup before start of measure " << endl;
				setDuration(dur - difference);
			}
			break;

		case mevent_attributes:
			setQTicks(getIntValue("./divisions"));
			timesigdur = getTimeSigDur();
			if (timesigdur > 0) {
				reportTimeSigDurToOwner(timesigdur);
			}
			break;

		case mevent_figured_bass:
		case mevent_harmony:
		case mevent_barline:
		case mevent_bookmark:
		case mevent_grouping:
		case mevent_link:
		case mevent_direction:
		case mevent_print:
		case mevent_sound:
		case mevent_unknown:
			setDuration(tempduration);
			break;
		case mevent_float:
			// assigned later for floating harmony
			break;
	}

	if (floatingharmony) {
		m_hnode = el;
		m_eventtype = mevent_float;
		m_duration = 0;
		m_node = xml_node(NULL);
		m_voice = 1;
		m_voiceindex = 0;
	} else {
		// if the previous sibling was a <harmony>, then store
		// for later parsing.  May have to che