"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationTargetGroup = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const elasticloadbalancingv2_canned_metrics_generated_1 = require("../elasticloadbalancingv2-canned-metrics.generated");
const base_target_group_1 = require("../shared/base-target-group");
const enums_1 = require("../shared/enums");
const imported_1 = require("../shared/imported");
const util_1 = require("../shared/util");
/**
 * Define an Application Target Group.
 *
 * @stability stable
 */
class ApplicationTargetGroup extends base_target_group_1.TargetGroupBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        const [protocol, port] = util_1.determineProtocolAndPort(props.protocol, props.port);
        super(scope, id, { ...props }, {
            protocol,
            port,
        });
        this.protocol = protocol;
        this.port = port;
        this.connectableMembers = [];
        this.listeners = [];
        if (props) {
            if (props.slowStart !== undefined) {
                if (props.slowStart.toSeconds() < 30 || props.slowStart.toSeconds() > 900) {
                    throw new Error('Slow start duration value must be between 30 and 900 seconds.');
                }
                this.setAttribute('slow_start.duration_seconds', props.slowStart.toSeconds().toString());
            }
            if (props.stickinessCookieDuration) {
                this.enableCookieStickiness(props.stickinessCookieDuration, props.stickinessCookieName);
            }
            this.addTarget(...(props.targets || []));
        }
    }
    /**
     * Import an existing target group.
     *
     * @stability stable
     */
    static fromTargetGroupAttributes(scope, id, attrs) {
        return new ImportedApplicationTargetGroup(scope, id, attrs);
    }
    /**
     * (deprecated) Import an existing target group.
     *
     * @deprecated Use `fromTargetGroupAttributes` instead
     */
    static import(scope, id, props) {
        return ApplicationTargetGroup.fromTargetGroupAttributes(scope, id, props);
    }
    /**
     * Add a load balancing target to this target group.
     *
     * @stability stable
     */
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToApplicationTargetGroup(this);
            this.addLoadBalancerTarget(result);
        }
    }
    /**
     * Enable sticky routing via a cookie to members of this target group.
     *
     * Note: If the `cookieName` parameter is set, application-based stickiness will be applied,
     * otherwise it defaults to duration-based stickiness attributes (`lb_cookie`).
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/sticky-sessions.html
     * @stability stable
     */
    enableCookieStickiness(duration, cookieName) {
        if (duration.toSeconds() < 1 || duration.toSeconds() > 604800) {
            throw new Error('Stickiness cookie duration value must be between 1 second and 7 days (604800 seconds).');
        }
        if (cookieName !== undefined) {
            if (!core_1.Token.isUnresolved(cookieName) && (cookieName.startsWith('AWSALB') || cookieName.startsWith('AWSALBAPP') || cookieName.startsWith('AWSALBTG'))) {
                throw new Error('App cookie names that start with the following prefixes are not allowed: AWSALB, AWSALBAPP, and AWSALBTG; they\'re reserved for use by the load balancer.');
            }
            if (cookieName === '') {
                throw new Error('App cookie name cannot be an empty string.');
            }
        }
        this.setAttribute('stickiness.enabled', 'true');
        if (cookieName) {
            this.setAttribute('stickiness.type', 'app_cookie');
            this.setAttribute('stickiness.app_cookie.cookie_name', cookieName);
            this.setAttribute('stickiness.app_cookie.duration_seconds', duration.toSeconds().toString());
        }
        else {
            this.setAttribute('stickiness.type', 'lb_cookie');
            this.setAttribute('stickiness.lb_cookie.duration_seconds', duration.toSeconds().toString());
        }
    }
    /**
     * Register a connectable as a member of this target group.
     *
     * Don't call this directly. It will be called by load balancing targets.
     *
     * @stability stable
     */
    registerConnectable(connectable, portRange) {
        portRange = portRange || ec2.Port.tcp(this.defaultPort);
        // Notify all listeners that we already know about of this new connectable.
        // Then remember for new listeners that might get added later.
        this.connectableMembers.push({ connectable, portRange });
        for (const listener of this.listeners) {
            listener.registerConnectable(connectable, portRange);
        }
    }
    /**
     * Register a listener that is load balancing to this target group.
     *
     * Don't call this directly. It will be called by listeners.
     *
     * @stability stable
     */
    registerListener(listener, associatingConstruct) {
        // Notify this listener of all connectables that we know about.
        // Then remember for new connectables that might get added later.
        for (const member of this.connectableMembers) {
            listener.registerConnectable(member.connectable, member.portRange);
        }
        this.listeners.push(listener);
        this.loadBalancerAttachedDependencies.add((associatingConstruct || listener));
    }
    /**
     * Full name of first load balancer.
     *
     * @stability stable
     */
    get firstLoadBalancerFullName() {
        if (this.listeners.length === 0) {
            throw new Error('The TargetGroup needs to be attached to a LoadBalancer before you can call this method');
        }
        return base_target_group_1.loadBalancerNameFromListenerArn(this.listeners[0].listenerArn);
    }
    /**
     * Return the given named metric for this Application Load Balancer Target Group.
     *
     * Returns the metric for this target group from the point of view of the first
     * load balancer load balancing to it. If you have multiple load balancers load
     * sending traffic to the same target group, you will have to override the dimensions
     * on this metric.
     *
     * @default Average over 5 minutes
     * @stability stable
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ApplicationELB',
            metricName,
            dimensions: {
                TargetGroup: this.targetGroupFullName,
                LoadBalancer: this.firstLoadBalancerFullName,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * The number of IPv6 requests received by the target group.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricIpv6RequestCount(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.ApplicationELBMetrics.iPv6RequestCountSum, props);
    }
    /**
     * The number of requests processed over IPv4 and IPv6.
     *
     * This count includes only the requests with a response generated by a target of the load balancer.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricRequestCount(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.ApplicationELBMetrics.requestCountSum, props);
    }
    /**
     * The number of healthy hosts in the target group.
     *
     * @default Average over 5 minutes
     * @stability stable
     */
    metricHealthyHostCount(props) {
        return this.metric('HealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of unhealthy hosts in the target group.
     *
     * @default Average over 5 minutes
     * @stability stable
     */
    metricUnhealthyHostCount(props) {
        return this.metric('UnHealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of HTTP 2xx/3xx/4xx/5xx response codes generated by all targets in this target group.
     *
     * This does not include any response codes generated by the load balancer.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricHttpCodeTarget(code, props) {
        return this.metric(code, {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The average number of requests received by each target in a target group.
     *
     * The only valid statistic is Sum. Note that this represents the average not the sum.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricRequestCountPerTarget(props) {
        return this.metric('RequestCountPerTarget', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of connections that were not successfully established between the load balancer and target.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricTargetConnectionErrorCount(props) {
        return this.metric('TargetConnectionErrorCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The time elapsed, in seconds, after the request leaves the load balancer until a response from the target is received.
     *
     * @default Average over 5 minutes
     * @stability stable
     */
    metricTargetResponseTime(props) {
        return this.metric('TargetResponseTime', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of TLS connections initiated by the load balancer that did not establish a session with the target.
     *
     * Possible causes include a mismatch of ciphers or protocols.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricTargetTLSNegotiationErrorCount(props) {
        return this.metric('TargetTLSNegotiationErrorCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        const ret = super.validate();
        if (this.targetType !== undefined && this.targetType !== enums_1.TargetType.LAMBDA
            && (this.protocol === undefined || this.port === undefined)) {
            ret.push('At least one of \'port\' or \'protocol\' is required for a non-Lambda TargetGroup');
        }
        if (this.healthCheck && this.healthCheck.protocol && !ALB_HEALTH_CHECK_PROTOCOLS.includes(this.healthCheck.protocol)) {
            ret.push([
                `Health check protocol '${this.healthCheck.protocol}' is not supported. `,
                `Must be one of [${ALB_HEALTH_CHECK_PROTOCOLS.join(', ')}]`,
            ].join(''));
        }
        return ret;
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({
                LoadBalancer: this.firstLoadBalancerFullName,
                TargetGroup: this.targetGroupFullName,
            }),
            ...props,
        }).attachTo(this);
    }
}
exports.ApplicationTargetGroup = ApplicationTargetGroup;
_a = JSII_RTTI_SYMBOL_1;
ApplicationTargetGroup[_a] = { fqn: "@aws-cdk/aws-elasticloadbalancingv2.ApplicationTargetGroup", version: "1.93.0" };
/**
 * An imported application target group
 */
class ImportedApplicationTargetGroup extends imported_1.ImportedTargetGroupBase {
    registerListener(_listener, _associatingConstruct) {
        // Nothing to do, we know nothing of our members
        core_1.Annotations.of(this).addWarning('Cannot register listener on imported target group -- security groups might need to be updated manually');
    }
    registerConnectable(_connectable, _portRange) {
        core_1.Annotations.of(this).addWarning('Cannot register connectable on imported target group -- security groups might need to be updated manually');
    }
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToApplicationTargetGroup(this);
            if (result.targetJson !== undefined) {
                throw new Error('Cannot add a non-self registering target to an imported TargetGroup. Create a new TargetGroup instead.');
            }
        }
    }
}
const ALB_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS];
//# sourceMappingURL=data:application/json;base64,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