"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LakeStrategyFactory = exports.LakeImplStrategy = void 0;
const aws_glue_alpha_1 = require("@aws-cdk/aws-glue-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_lakeformation_1 = require("aws-cdk-lib/aws-lakeformation");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const data_lake_bucket_1 = require("./data-lake-bucket");
const kinesis_ops_1 = require("./data-streams/kinesis-ops");
const kinesis_stream_1 = require("./data-streams/kinesis-stream");
const s3_delivery_stream_1 = require("./data-streams/s3-delivery-stream");
const glue_crawler_1 = require("./etl/glue-crawler");
const glue_job_1 = require("./etl/glue-job");
const glue_job_ops_1 = require("./etl/glue-job-ops");
const glue_table_1 = require("./etl/glue-table");
const enums_1 = require("./global/enums");
const utils_1 = require("./utils");
class LakeImplStrategy {
    constructor() {
        this.locationRegistry = [];
        this.stageName = enums_1.Stage.ALPHA;
        this.downloadLocations = {}; //used for the Custom Resource to allow downloading of existing datasets into datalake
        this.dataStreams = {};
    }
    getDataSetBucketName(pipe, dataTier) {
        return dataTier == enums_1.DataTier.RAW ? this.downloadLocations[pipe.name].rawBucketName :
            dataTier == enums_1.DataTier.REFINED ? this.downloadLocations[pipe.name].refinedBucketName :
                dataTier == enums_1.DataTier.TRUSTED ? this.downloadLocations[pipe.name].trustedBucketName : undefined;
    }
    createDataProduct(props) {
        const pipelineStack = new aws_cdk_lib_1.NestedStack(props.stack, `${props.pipe.name}-dataset-stack`); // props.product.accountId == Aws.ACCOUNT_ID ? new NestedStack(props.stack, `${props.pipe.name}-dataset-stack`) : props.stack;
        this.logBucket = props.logBucket;
        this.stageName = props.stage;
        this.securityGroup = props.securityGroup;
        this.vpc = props.vpc;
        this.datalakeAdminRoleArn = props.datalakeAdminRoleArn;
        this.datalakeDbCreatorRoleArn = props.datalakeDbCreatorRoleArn;
        // if data to download into a tier create the download locations
        if (props.pipe.dataSetDropTier) {
            this.downloadLocations[props.pipe.name] = {
                destinationPrefix: props.pipe.destinationPrefix,
                sourceBucketName: props.pipe.s3Properties ? props.pipe.s3Properties.sourceBucketName : undefined,
                sourceKeys: props.pipe.s3Properties ? props.pipe.s3Properties.sourceKeys : undefined,
                rawBucketName: utils_1.buildS3BucketName({
                    name: props.pipe.name,
                    accountId: props.product.accountId,
                    resourceUse: 'raw',
                    stage: this.stageName,
                }),
                refinedBucketName: utils_1.buildS3BucketName({
                    name: props.pipe.name,
                    accountId: props.product.accountId,
                    resourceUse: 'refined',
                    stage: this.stageName,
                }),
                trustedBucketName: utils_1.buildS3BucketName({
                    name: props.pipe.name,
                    accountId: props.product.accountId,
                    resourceUse: 'trusted',
                    stage: this.stageName,
                }),
                destinationBucketName: utils_1.buildS3BucketName({
                    name: props.pipe.name,
                    accountId: props.product.accountId,
                    resourceUse: props.pipe.dataSetDropTier == enums_1.DataTier.RAW ? 'raw' : props.pipe.dataSetDropTier == enums_1.DataTier.REFINED ? 'refined' : 'trusted',
                    stage: this.stageName,
                }),
            };
        }
        this.createBuckets(pipelineStack, props.pipe, props.product, props.database);
        const bucketName = this.getDataSetBucketName(props.pipe, props.pipe.dataSetDropTier);
        this.addPipeline(pipelineStack, props.pipe, props.product, bucketName);
    }
    createCrawler(stack, pipe, product, bucketName, s3DataLFResource, database) {
        if (pipe.table !== undefined)
            return;
        const name = bucketName.replace(/\W/g, '');
        // only create a crawler for the drop location of the data in the data product of the pipeline
        const crawler = new glue_crawler_1.GlueCrawler(stack, `data-lake-crawler-${name}`, {
            name: utils_1.buildGlueCrawlerName({
                stage: this.stageName,
                resourceUse: 'crawler',
                name: pipe.name,
            }),
            databaseName: product.databaseName,
            bucketName: bucketName,
            bucketPrefix: pipe.destinationPrefix,
            roleName: utils_1.buildRoleName({
                stage: this.stageName,
                resourceUse: 'crawler-role',
                name: pipe.name,
            }),
            lfS3Resource: s3DataLFResource,
        });
        crawler.node.addDependency(database);
        this.locationRegistry.forEach(r => {
            crawler.node.addDependency(r);
        });
    }
    createGlueTable(stack, pipeline, product, bucketName) {
        if (!pipeline.table)
            return;
        const table = new glue_table_1.GlueTable(stack, `${pipeline.name}-table`, {
            catalogId: pipeline.table.catalogId,
            columns: pipeline.table.columns,
            databaseName: product.databaseName,
            description: pipeline.table.description,
            inputFormat: pipeline.table.inputFormat,
            outputFormat: pipeline.table.outputFormat,
            parameters: pipeline.table.parameters,
            partitionKeys: pipeline.table.partitionKeys,
            s3Location: `s3://${bucketName}/${pipeline.destinationPrefix}`,
            serdeParameters: pipeline.table.serdeParameters,
            serializationLibrary: pipeline.table.serializationLibrary,
            tableName: pipeline.table.tableName,
        });
        table.node.addDependency(product.databaseName);
    }
    // this is a jumbled mess clean up once refecto
    createPipelineResources(stack, pipeline, dataProduct, bucketName) {
        switch (pipeline.type) {
            case enums_1.DataPipelineType.S3: {
                break;
            }
            case enums_1.DataPipelineType.STREAM: {
                this.addDataStream(stack, pipeline, bucketName);
                break;
            }
            case enums_1.DataPipelineType.JDBC: {
                this.createJDBCConnection(stack, pipeline);
                break;
            }
        }
        // rethink this whole section
        if (pipeline.job) {
            const jobScript = utils_1.packageAsset(stack, `${pipeline.name}Script`, pipeline.job.jobScript);
            pipeline.job.jobArgs['--TempDir'] = `s3://${this.logBucket.bucketName}/temp/`;
            pipeline.job.jobArgs['--spark-event-logs-path'] = `s3://${this.logBucket.bucketName}/logs/`;
            let s3Location = this.getDataSetBucketName(pipeline, pipeline.job.destinationLocation);
            if (pipeline.job.destinationLocation && s3Location) {
                pipeline.job.jobArgs['--DESTINATION_BUCKET'] = s3Location;
                const job = new glue_job_1.GlueJob(stack, `${pipeline.name}-etl-job`, {
                    deploymentBucket: jobScript.bucket,
                    jobScript: utils_1.toS3Path(jobScript),
                    name: pipeline.job.name,
                    workerType: pipeline.job.workerType,
                    description: pipeline.job.description,
                    glueVersion: pipeline.job.glueVersion,
                    jobArgs: pipeline.job.jobArgs,
                    maxCapacity: pipeline.job.maxCapacity,
                    maxConcurrentRuns: pipeline.job.maxConcurrentRuns,
                    maxRetries: pipeline.job.maxRetries,
                    numberOfWorkers: pipeline.job.numberOfWorkers,
                    roleName: pipeline.job.roleName,
                    timeout: pipeline.job.timeout,
                    jobType: pipeline.job.jobType,
                    readAccessBuckets: [
                        this.logBucket,
                    ],
                    writeAccessBuckets: [
                        this.logBucket,
                        aws_s3_1.Bucket.fromBucketName(stack, 'raw-bucket-role', s3Location),
                    ],
                });
                new glue_job_ops_1.GlueJobOps(stack, `${pipeline.name}-etl-job-ops`, {
                    job: job,
                });
                if (pipeline.streamProperties) {
                    this.dataStreams[pipeline.name].stream.grantRead(job.role);
                }
                new aws_lakeformation_1.CfnPermissions(stack, `${pipeline.name}-create-table-perm`, {
                    dataLakePrincipal: {
                        dataLakePrincipalIdentifier: job.role.roleArn,
                    },
                    resource: {
                        databaseResource: {
                            name: dataProduct.databaseName,
                        },
                    },
                    permissions: [
                        enums_1.Permissions.ALTER,
                        enums_1.Permissions.CREATE_TABLE,
                        enums_1.Permissions.DESCRIBE,
                    ],
                });
                if (pipeline.table) {
                    new aws_lakeformation_1.CfnPermissions(stack, `${pipeline.name}-access-table-perm`, {
                        dataLakePrincipal: {
                            dataLakePrincipalIdentifier: job.role.roleArn,
                        },
                        resource: {
                            tableResource: {
                                databaseName: dataProduct.databaseName,
                                name: pipeline.table.tableName,
                            },
                        },
                        permissions: [
                            enums_1.Permissions.SELECT,
                            enums_1.Permissions.DESCRIBE,
                        ],
                    });
                }
            }
        }
    }
    addDataStream(stack, pipeline, bucketName) {
        const schemaName = pipeline.name;
        const dataStreamStack = new aws_cdk_lib_1.NestedStack(stack, `${schemaName}-datastream-stack`);
        if (!pipeline.streamProperties) {
            throw Error("Cannot create a stream pipeline without 'streamProperties'");
        }
        this.dataStreams[pipeline.name] = new kinesis_stream_1.KinesisStream(dataStreamStack, 'DataStream', {
            shardCount: 1,
            streamName: pipeline.streamProperties.streamName,
        });
        const deliveryStream = new s3_delivery_stream_1.S3DeliveryStream(dataStreamStack, 'deliveryStream', {
            compression: s3_delivery_stream_1.CompressionType.UNCOMPRESSED,
            kinesisStream: this.dataStreams[pipeline.name].stream,
            s3Bucket: aws_s3_1.Bucket.fromBucketName(stack, 'get-bucket-for-kinesis', bucketName),
            s3Prefix: pipeline.destinationPrefix,
        });
        new kinesis_ops_1.KinesisOps(dataStreamStack, 'kinesis-ops', {
            stream: this.dataStreams[pipeline.name],
            deliveryStream: deliveryStream,
        });
        if (pipeline.streamProperties.lambdaDataGenerator) {
            const dataGeneratorFunction = new aws_lambda_1.Function(dataStreamStack, 'data-generator-function', {
                code: pipeline.streamProperties.lambdaDataGenerator.code,
                handler: pipeline.streamProperties.lambdaDataGenerator.handler,
                timeout: pipeline.streamProperties.lambdaDataGenerator.timeout,
                runtime: pipeline.streamProperties.lambdaDataGenerator.runtime,
                functionName: pipeline.streamProperties.lambdaDataGenerator.functionName,
                environment: {
                    KINESIS_STREAM: this.dataStreams[pipeline.name].stream.streamName,
                },
            });
            this.dataStreams[pipeline.name].stream.grantWrite(dataGeneratorFunction);
            const rule = new aws_events_1.Rule(stack, 'Rule', {
                schedule: pipeline.streamProperties.lambdaDataGenerator.schedule,
                ruleName: pipeline.streamProperties.lambdaDataGenerator.ruleName,
            });
            rule.addTarget(new aws_events_targets_1.LambdaFunction(dataGeneratorFunction));
        }
        return this.dataStreams[pipeline.name];
    }
    createJDBCConnection(stack, pipeline) {
        if (this.vpc && this.securityGroup) {
            new aws_glue_alpha_1.Connection(stack, `${pipeline.name}-glue-connection`, {
                type: aws_glue_alpha_1.ConnectionType.JDBC,
                connectionName: `${pipeline.name}-jdbc`,
                description: `JDBC connection for glue to use on pipeline ${pipeline.name}`,
                subnet: this.vpc.isolatedSubnets[0],
                securityGroups: [this.securityGroup],
                properties: {
                    JDBC_CONNECTION_URL: pipeline.jdbcProperties.jdbc,
                    USERNAME: pipeline.jdbcProperties.username,
                    PASSWORD: pipeline.jdbcProperties.password,
                },
            });
        }
        else {
            throw new Error('VPC required to create a JDBC pipeline.');
        }
    }
    createBuckets(stack, pipe, product, database) {
        /// This is confusing. Find a way to simplify
        const dataCatalogAccountId = product.dataCatalogAccountId ?
            product.dataCatalogAccountId : product.accountId;
        const crossAccount = product.dataCatalogAccountId ?
            product.dataCatalogAccountId != product.accountId ? true : false : false;
        pipe.tiers.forEach(r => {
            const bucketName = this.getDataSetBucketName(pipe, r);
            if (this.lakeKind() === enums_1.LakeKind.DATA_PRODUCT || this.lakeKind() === enums_1.LakeKind.DATA_PRODUCT_AND_CATALOG) {
                new data_lake_bucket_1.DataLakeBucket(stack, `s3-${r}-bucket-${pipe.name}`, {
                    bucketName: bucketName,
                    dataCatalogAccountId: dataCatalogAccountId,
                    logBucket: this.logBucket,
                    crossAccount: crossAccount,
                    s3Properties: product.s3BucketProps,
                }).bucket;
            }
            if (this.lakeKind() === enums_1.LakeKind.CENTRAL_CATALOG || this.lakeKind() === enums_1.LakeKind.DATA_PRODUCT_AND_CATALOG) {
                if (this.datalakeDbCreatorRoleArn == undefined)
                    throw new Error('Cannot have datalake without Data Lake DB Creator role defined.');
                const name = this.getDataSetBucketName(pipe, r).replace(/\W/g, '');
                const lfResource = this.registerDataLakeLocation(stack, this.datalakeDbCreatorRoleArn, name);
                this.locationRegistry.push(lfResource);
                if (this.datalakeAdminRoleArn) {
                    this.createDataLocationAccessPermission(stack, `${name}-admin`, this.datalakeAdminRoleArn, name, lfResource);
                }
                this.createCrawler(stack, pipe, product, bucketName, lfResource, database);
            }
        });
    }
    registerDataLakeLocation(stack, datalakeDbCreatorRoleArn, bucketName) {
        const dlResource = new aws_lakeformation_1.CfnResource(stack, `lf-resource-${bucketName}`, {
            resourceArn: `arn:aws:s3:::${bucketName}`,
            useServiceLinkedRole: false,
            roleArn: datalakeDbCreatorRoleArn,
        });
        this.createDataLocationAccessPermission(stack, `${bucketName}-creator`, datalakeDbCreatorRoleArn, bucketName, dlResource);
        return dlResource;
    }
    createDataLocationAccessPermission(stack, name, roleArn, bucketName, resource) {
        const perm = new aws_lakeformation_1.CfnPermissions(stack, `datalake-creator-perm-${name}`, {
            dataLakePrincipal: {
                dataLakePrincipalIdentifier: roleArn,
            },
            resource: {
                dataLocationResource: {
                    s3Resource: `arn:aws:s3:::${bucketName}`,
                },
            },
            permissions: [
                enums_1.Permissions.DATA_LOCATION_ACCESS,
            ],
        });
        perm.node.addDependency(resource);
        return perm;
    }
}
exports.LakeImplStrategy = LakeImplStrategy;
class DataProductStrategy extends LakeImplStrategy {
    lakeKind() {
        return enums_1.LakeKind.DATA_PRODUCT;
    }
    addPipeline(stack, pipeline, dataProduct, bucketName) {
        this.createPipelineResources(stack, pipeline, dataProduct, bucketName);
    }
}
class CentralCatalogStrategy extends LakeImplStrategy {
    lakeKind() {
        return enums_1.LakeKind.CENTRAL_CATALOG;
    }
    addPipeline(stack, pipeline, dataProduct, bucketName) {
        if (pipeline.table) {
            this.createGlueTable(stack, pipeline, dataProduct, bucketName);
        }
    }
}
class ConsumerStrategy extends LakeImplStrategy {
    lakeKind() {
        return enums_1.LakeKind.CONSUMER;
    }
    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore
    addPipeline(stack, pipeline, dataProduct, bucketName) {
        return;
    }
}
class DataProductAndCatalogStrategy extends LakeImplStrategy {
    lakeKind() {
        return enums_1.LakeKind.DATA_PRODUCT_AND_CATALOG;
    }
    addPipeline(stack, pipeline, dataProduct, bucketName) {
        this.createPipelineResources(stack, pipeline, dataProduct, bucketName);
        if (pipeline.table) {
            this.createGlueTable(stack, pipeline, dataProduct, bucketName);
        }
    }
}
class LakeStrategyFactory {
    static getLakeStrategy(lakeKind) {
        return LakeStrategyFactory.strategies[lakeKind];
    }
}
exports.LakeStrategyFactory = LakeStrategyFactory;
LakeStrategyFactory.strategies = {
    [enums_1.LakeKind.DATA_PRODUCT]: new DataProductStrategy(),
    [enums_1.LakeKind.CENTRAL_CATALOG]: new CentralCatalogStrategy(),
    [enums_1.LakeKind.CONSUMER]: new ConsumerStrategy(),
    [enums_1.LakeKind.DATA_PRODUCT_AND_CATALOG]: new DataProductAndCatalogStrategy(),
};
//# sourceMappingURL=data:application/json;base64,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