"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GlueCrawler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const events = require("aws-cdk-lib/aws-events");
const glue = require("aws-cdk-lib/aws-glue");
const iam = require("aws-cdk-lib/aws-iam");
const lf = require("aws-cdk-lib/aws-lakeformation");
const constructs_1 = require("constructs");
const enums_1 = require("../global/enums");
class GlueCrawler extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.role = this.createGlueCrawlerRole(props);
        this.metricSuccessRule = this.crawlerRule('SuccessRule', props.name, 'Succeeded');
        this.metricFailureRule = this.crawlerRule('FailureRule', props.name, 'Failed');
        let s3TargetPaths = new Array();
        s3TargetPaths.push({
            path: `s3://${props.bucketName}/${props.bucketPrefix}`,
        });
        this.crawler = new glue.CfnCrawler(this, `data-lake-crawler-${props.name}-`, {
            name: props.name,
            role: this.role.roleArn,
            databaseName: props.databaseName,
            targets: {
                s3Targets: s3TargetPaths,
            },
        });
        const dbPerms = new lf.CfnPermissions(this, 'glue-role-database-permission', {
            dataLakePrincipal: {
                dataLakePrincipalIdentifier: this.role.roleArn,
            },
            resource: {
                databaseResource: {
                    name: props.databaseName,
                },
            },
            permissions: [
                'CREATE_TABLE',
                'DESCRIBE',
            ],
        });
        const s3perms = new lf.CfnPermissions(this, 'datalake-creator-permission', {
            dataLakePrincipal: {
                dataLakePrincipalIdentifier: this.role.roleArn,
            },
            resource: {
                dataLocationResource: {
                    s3Resource: `arn:aws:s3:::${props.bucketName}`,
                },
            },
            permissions: [
                enums_1.Permissions.DATA_LOCATION_ACCESS,
            ],
        });
        s3perms.addDependsOn(props.lfS3Resource);
        this.crawler.addDependsOn(dbPerms);
        this.crawler.addDependsOn(s3perms);
    }
    metricSuccess(props) {
        return this.ruleMetric(this.metricSuccessRule, props);
    }
    metricFailure(props) {
        return this.ruleMetric(this.metricFailureRule, props);
    }
    ruleMetric({ ruleName }, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Events',
            metricName: 'TriggeredRules',
            dimensionsMap: { RuleName: ruleName },
            statistic: cloudwatch.Statistic.SUM,
            ...props,
        }).attachTo(this);
    }
    crawlerRule(id, crawlerName, ...states) {
        return new events.Rule(this, id, {
            ruleName: crawlerName + states.join(''),
            description: `Event triggered when Glue Crawler ${crawlerName} is in ${states.join(' or ')} state(s)`,
            eventPattern: {
                source: ['aws.glue'],
                detailType: ['Glue Crawler State Change'],
                detail: {
                    state: states,
                    crawlerName: [crawlerName],
                },
            },
        });
    }
    createGlueCrawlerRole(props) {
        const role = new iam.Role(this, 'Role', {
            roleName: props.roleName || props.name + 'Role',
            assumedBy: new iam.ServicePrincipal('glue.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSGlueServiceRole'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonS3FullAccess'),
            ],
        });
        role.addToPolicy(new iam.PolicyStatement({ actions: ['lakeformation:GetDataAccess'], resources: ['*'] }));
        return role;
    }
}
exports.GlueCrawler = GlueCrawler;
_a = JSII_RTTI_SYMBOL_1;
GlueCrawler[_a] = { fqn: "@randyridgley/cdk-datalake-constructs.GlueCrawler", version: "0.0.56" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZ2x1ZS1jcmF3bGVyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2V0bC9nbHVlLWNyYXdsZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFDQSx5REFBeUQ7QUFDekQsaURBQWlEO0FBQ2pELDZDQUE2QztBQUM3QywyQ0FBMkM7QUFDM0Msb0RBQW9EO0FBQ3BELDJDQUF1QztBQUN2QywyQ0FBOEM7QUFZOUMsTUFBYSxXQUFZLFNBQVEsc0JBQVM7SUFNeEMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUE2QjtRQUNyRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDLHFCQUFxQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzlDLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLGFBQWEsRUFBRSxLQUFLLENBQUMsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQ2xGLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLGFBQWEsRUFBRSxLQUFLLENBQUMsSUFBSSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQy9FLElBQUksYUFBYSxHQUFHLElBQUksS0FBSyxFQUFvQyxDQUFDO1FBRWxFLGFBQWEsQ0FBQyxJQUFJLENBQUM7WUFDakIsSUFBSSxFQUFFLFFBQVEsS0FBSyxDQUFDLFVBQVUsSUFBSSxLQUFLLENBQUMsWUFBWSxFQUFFO1NBQ3ZELENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxPQUFPLEdBQUcsSUFBSSxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksRUFBRSxxQkFBcUIsS0FBSyxDQUFDLElBQUksR0FBRyxFQUFFO1lBQzNFLElBQUksRUFBRSxLQUFLLENBQUMsSUFBSTtZQUNoQixJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPO1lBQ3ZCLFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWTtZQUNoQyxPQUFPLEVBQUU7Z0JBQ1AsU0FBUyxFQUFFLGFBQWE7YUFDekI7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLE9BQU8sR0FBRyxJQUFJLEVBQUUsQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLCtCQUErQixFQUFFO1lBQzNFLGlCQUFpQixFQUFFO2dCQUNqQiwyQkFBMkIsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU87YUFDL0M7WUFDRCxRQUFRLEVBQUU7Z0JBQ1IsZ0JBQWdCLEVBQUU7b0JBQ2hCLElBQUksRUFBRSxLQUFLLENBQUMsWUFBWTtpQkFDekI7YUFDRjtZQUNELFdBQVcsRUFBRTtnQkFDWCxjQUFjO2dCQUNkLFVBQVU7YUFDWDtTQUNGLENBQUMsQ0FBQztRQUVILE1BQU0sT0FBTyxHQUFHLElBQUksRUFBRSxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsNkJBQTZCLEVBQUU7WUFDekUsaUJBQWlCLEVBQUU7Z0JBQ2pCLDJCQUEyQixFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTzthQUMvQztZQUNELFFBQVEsRUFBRTtnQkFDUixvQkFBb0IsRUFBRTtvQkFDcEIsVUFBVSxFQUFFLGdCQUFnQixLQUFLLENBQUMsVUFBVSxFQUFFO2lCQUMvQzthQUNGO1lBQ0QsV0FBVyxFQUFFO2dCQUNYLG1CQUFXLENBQUMsb0JBQW9CO2FBQ2pDO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUM7UUFDekMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDbkMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVELGFBQWEsQ0FBQyxLQUFnQztRQUM1QyxPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ3hELENBQUM7SUFFRCxhQUFhLENBQUMsS0FBZ0M7UUFDNUMsT0FBTyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUN4RCxDQUFDO0lBRU8sVUFBVSxDQUFDLEVBQUUsUUFBUSxFQUFlLEVBQUUsS0FBZ0M7UUFDNUUsT0FBTyxJQUFJLFVBQVUsQ0FBQyxNQUFNLENBQUM7WUFDM0IsU0FBUyxFQUFFLFlBQVk7WUFDdkIsVUFBVSxFQUFFLGdCQUFnQjtZQUM1QixhQUFhLEVBQUUsRUFBRSxRQUFRLEVBQUUsUUFBUSxFQUFFO1lBQ3JDLFNBQVMsRUFBRSxVQUFVLENBQUMsU0FBUyxDQUFDLEdBQUc7WUFDbkMsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNwQixDQUFDO0lBRU8sV0FBVyxDQUFDLEVBQVUsRUFBRSxXQUFtQixFQUFFLEdBQUcsTUFBZ0I7UUFDdEUsT0FBTyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRTtZQUMvQixRQUFRLEVBQUUsV0FBVyxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDO1lBQ3ZDLFdBQVcsRUFBRSxxQ0FBcUMsV0FBVyxVQUFVLE1BQU0sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLFdBQVc7WUFDckcsWUFBWSxFQUFFO2dCQUNaLE1BQU0sRUFBRSxDQUFDLFVBQVUsQ0FBQztnQkFDcEIsVUFBVSxFQUFFLENBQUMsMkJBQTJCLENBQUM7Z0JBQ3pDLE1BQU0sRUFBRTtvQkFDTixLQUFLLEVBQUUsTUFBTTtvQkFDYixXQUFXLEVBQUUsQ0FBQyxXQUFXLENBQUM7aUJBQzNCO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU8scUJBQXFCLENBQUMsS0FBNkI7UUFDekQsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxNQUFNLEVBQUU7WUFDdEMsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRLElBQUksS0FBSyxDQUFDLElBQUksR0FBRyxNQUFNO1lBQy9DLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxvQkFBb0IsQ0FBQztZQUN6RCxlQUFlLEVBQUU7Z0JBQ2YsR0FBRyxDQUFDLGFBQWEsQ0FBQyx3QkFBd0IsQ0FBQyxpQ0FBaUMsQ0FBQztnQkFDN0UsR0FBRyxDQUFDLGFBQWEsQ0FBQyx3QkFBd0IsQ0FBQyxvQkFBb0IsQ0FBQzthQUNqRTtTQUNGLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDLEVBQUUsT0FBTyxFQUFFLENBQUMsNkJBQTZCLENBQUMsRUFBRSxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUMxRyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7O0FBekdILGtDQTBHQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IENmblJlc291cmNlIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0ICogYXMgY2xvdWR3YXRjaCBmcm9tICdhd3MtY2RrLWxpYi9hd3MtY2xvdWR3YXRjaCc7XG5pbXBvcnQgKiBhcyBldmVudHMgZnJvbSAnYXdzLWNkay1saWIvYXdzLWV2ZW50cyc7XG5pbXBvcnQgKiBhcyBnbHVlIGZyb20gJ2F3cy1jZGstbGliL2F3cy1nbHVlJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGxmIGZyb20gJ2F3cy1jZGstbGliL2F3cy1sYWtlZm9ybWF0aW9uJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgUGVybWlzc2lvbnMgfSBmcm9tICcuLi9nbG9iYWwvZW51bXMnO1xuXG5leHBvcnQgaW50ZXJmYWNlIElHbHVlQ3Jhd2xlclByb3BlcnRpZXMge1xuICBuYW1lOiBzdHJpbmc7XG4gIGRhdGFiYXNlTmFtZTogc3RyaW5nO1xuICByb2xlTmFtZT86IHN0cmluZztcbiAgdHJpZ2dlcj86IGdsdWUuQ2ZuVHJpZ2dlcjtcbiAgYnVja2V0TmFtZTogc3RyaW5nO1xuICBidWNrZXRQcmVmaXg/OiBzdHJpbmc7XG4gIGxmUzNSZXNvdXJjZTogQ2ZuUmVzb3VyY2U7XG59XG5cbmV4cG9ydCBjbGFzcyBHbHVlQ3Jhd2xlciBleHRlbmRzIENvbnN0cnVjdCB7XG4gIHB1YmxpYyByZWFkb25seSBjcmF3bGVyOiBnbHVlLkNmbkNyYXdsZXI7XG4gIHB1YmxpYyByZWFkb25seSByb2xlOiBpYW0uSVJvbGU7XG4gIHB1YmxpYyByZWFkb25seSBtZXRyaWNTdWNjZXNzUnVsZTogZXZlbnRzLlJ1bGU7XG4gIHB1YmxpYyByZWFkb25seSBtZXRyaWNGYWlsdXJlUnVsZTogZXZlbnRzLlJ1bGU7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IElHbHVlQ3Jhd2xlclByb3BlcnRpZXMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy5yb2xlID0gdGhpcy5jcmVhdGVHbHVlQ3Jhd2xlclJvbGUocHJvcHMpO1xuICAgIHRoaXMubWV0cmljU3VjY2Vzc1J1bGUgPSB0aGlzLmNyYXdsZXJSdWxlKCdTdWNjZXNzUnVsZScsIHByb3BzLm5hbWUsICdTdWNjZWVkZWQnKTtcbiAgICB0aGlzLm1ldHJpY0ZhaWx1cmVSdWxlID0gdGhpcy5jcmF3bGVyUnVsZSgnRmFpbHVyZVJ1bGUnLCBwcm9wcy5uYW1lLCAnRmFpbGVkJyk7XG4gICAgbGV0IHMzVGFyZ2V0UGF0aHMgPSBuZXcgQXJyYXk8Z2x1ZS5DZm5DcmF3bGVyLlMzVGFyZ2V0UHJvcGVydHk+KCk7XG5cbiAgICBzM1RhcmdldFBhdGhzLnB1c2goe1xuICAgICAgcGF0aDogYHMzOi8vJHtwcm9wcy5idWNrZXROYW1lfS8ke3Byb3BzLmJ1Y2tldFByZWZpeH1gLFxuICAgIH0pO1xuXG4gICAgdGhpcy5jcmF3bGVyID0gbmV3IGdsdWUuQ2ZuQ3Jhd2xlcih0aGlzLCBgZGF0YS1sYWtlLWNyYXdsZXItJHtwcm9wcy5uYW1lfS1gLCB7XG4gICAgICBuYW1lOiBwcm9wcy5uYW1lLFxuICAgICAgcm9sZTogdGhpcy5yb2xlLnJvbGVBcm4sXG4gICAgICBkYXRhYmFzZU5hbWU6IHByb3BzLmRhdGFiYXNlTmFtZSxcbiAgICAgIHRhcmdldHM6IHtcbiAgICAgICAgczNUYXJnZXRzOiBzM1RhcmdldFBhdGhzLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGRiUGVybXMgPSBuZXcgbGYuQ2ZuUGVybWlzc2lvbnModGhpcywgJ2dsdWUtcm9sZS1kYXRhYmFzZS1wZXJtaXNzaW9uJywge1xuICAgICAgZGF0YUxha2VQcmluY2lwYWw6IHtcbiAgICAgICAgZGF0YUxha2VQcmluY2lwYWxJZGVudGlmaWVyOiB0aGlzLnJvbGUucm9sZUFybixcbiAgICAgIH0sXG4gICAgICByZXNvdXJjZToge1xuICAgICAgICBkYXRhYmFzZVJlc291cmNlOiB7XG4gICAgICAgICAgbmFtZTogcHJvcHMuZGF0YWJhc2VOYW1lLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICAgIHBlcm1pc3Npb25zOiBbXG4gICAgICAgICdDUkVBVEVfVEFCTEUnLFxuICAgICAgICAnREVTQ1JJQkUnLFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHMzcGVybXMgPSBuZXcgbGYuQ2ZuUGVybWlzc2lvbnModGhpcywgJ2RhdGFsYWtlLWNyZWF0b3ItcGVybWlzc2lvbicsIHtcbiAgICAgIGRhdGFMYWtlUHJpbmNpcGFsOiB7XG4gICAgICAgIGRhdGFMYWtlUHJpbmNpcGFsSWRlbnRpZmllcjogdGhpcy5yb2xlLnJvbGVBcm4sXG4gICAgICB9LFxuICAgICAgcmVzb3VyY2U6IHtcbiAgICAgICAgZGF0YUxvY2F0aW9uUmVzb3VyY2U6IHtcbiAgICAgICAgICBzM1Jlc291cmNlOiBgYXJuOmF3czpzMzo6OiR7cHJvcHMuYnVja2V0TmFtZX1gLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICAgIHBlcm1pc3Npb25zOiBbXG4gICAgICAgIFBlcm1pc3Npb25zLkRBVEFfTE9DQVRJT05fQUNDRVNTLFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIHMzcGVybXMuYWRkRGVwZW5kc09uKHByb3BzLmxmUzNSZXNvdXJjZSk7XG4gICAgdGhpcy5jcmF3bGVyLmFkZERlcGVuZHNPbihkYlBlcm1zKTtcbiAgICB0aGlzLmNyYXdsZXIuYWRkRGVwZW5kc09uKHMzcGVybXMpO1xuICB9XG5cbiAgbWV0cmljU3VjY2Vzcyhwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICByZXR1cm4gdGhpcy5ydWxlTWV0cmljKHRoaXMubWV0cmljU3VjY2Vzc1J1bGUsIHByb3BzKTtcbiAgfVxuXG4gIG1ldHJpY0ZhaWx1cmUocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgcmV0dXJuIHRoaXMucnVsZU1ldHJpYyh0aGlzLm1ldHJpY0ZhaWx1cmVSdWxlLCBwcm9wcyk7XG4gIH1cblxuICBwcml2YXRlIHJ1bGVNZXRyaWMoeyBydWxlTmFtZSB9OiBldmVudHMuUnVsZSwgcHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgcmV0dXJuIG5ldyBjbG91ZHdhdGNoLk1ldHJpYyh7XG4gICAgICBuYW1lc3BhY2U6ICdBV1MvRXZlbnRzJyxcbiAgICAgIG1ldHJpY05hbWU6ICdUcmlnZ2VyZWRSdWxlcycsXG4gICAgICBkaW1lbnNpb25zTWFwOiB7IFJ1bGVOYW1lOiBydWxlTmFtZSB9LFxuICAgICAgc3RhdGlzdGljOiBjbG91ZHdhdGNoLlN0YXRpc3RpYy5TVU0sXG4gICAgICAuLi5wcm9wcyxcbiAgICB9KS5hdHRhY2hUbyh0aGlzKTtcbiAgfVxuXG4gIHByaXZhdGUgY3Jhd2xlclJ1bGUoaWQ6IHN0cmluZywgY3Jhd2xlck5hbWU6IHN0cmluZywgLi4uc3RhdGVzOiBzdHJpbmdbXSk6IGV2ZW50cy5SdWxlIHtcbiAgICByZXR1cm4gbmV3IGV2ZW50cy5SdWxlKHRoaXMsIGlkLCB7XG4gICAgICBydWxlTmFtZTogY3Jhd2xlck5hbWUgKyBzdGF0ZXMuam9pbignJyksXG4gICAgICBkZXNjcmlwdGlvbjogYEV2ZW50IHRyaWdnZXJlZCB3aGVuIEdsdWUgQ3Jhd2xlciAke2NyYXdsZXJOYW1lfSBpcyBpbiAke3N0YXRlcy5qb2luKCcgb3IgJyl9IHN0YXRlKHMpYCxcbiAgICAgIGV2ZW50UGF0dGVybjoge1xuICAgICAgICBzb3VyY2U6IFsnYXdzLmdsdWUnXSxcbiAgICAgICAgZGV0YWlsVHlwZTogWydHbHVlIENyYXdsZXIgU3RhdGUgQ2hhbmdlJ10sXG4gICAgICAgIGRldGFpbDoge1xuICAgICAgICAgIHN0YXRlOiBzdGF0ZXMsXG4gICAgICAgICAgY3Jhd2xlck5hbWU6IFtjcmF3bGVyTmFtZV0sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVHbHVlQ3Jhd2xlclJvbGUocHJvcHM6IElHbHVlQ3Jhd2xlclByb3BlcnRpZXMpOiBpYW0uUm9sZSB7XG4gICAgY29uc3Qgcm9sZSA9IG5ldyBpYW0uUm9sZSh0aGlzLCAnUm9sZScsIHtcbiAgICAgIHJvbGVOYW1lOiBwcm9wcy5yb2xlTmFtZSB8fCBwcm9wcy5uYW1lICsgJ1JvbGUnLFxuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2dsdWUuYW1hem9uYXdzLmNvbScpLFxuICAgICAgbWFuYWdlZFBvbGljaWVzOiBbXG4gICAgICAgIGlhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnc2VydmljZS1yb2xlL0FXU0dsdWVTZXJ2aWNlUm9sZScpLFxuICAgICAgICBpYW0uTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ0FtYXpvblMzRnVsbEFjY2VzcycpLCAvLyBzbGltIHRoaXMgZG93biBpZiBwb3NzaWJsZVxuICAgICAgXSxcbiAgICB9KTtcbiAgICByb2xlLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHsgYWN0aW9uczogWydsYWtlZm9ybWF0aW9uOkdldERhdGFBY2Nlc3MnXSwgcmVzb3VyY2VzOiBbJyonXSB9KSk7XG4gICAgcmV0dXJuIHJvbGU7XG4gIH1cbn0iXX0=